"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StacIngestor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const utils_1 = require("../utils");
class StacIngestor extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.table = this.buildTable();
        const env = {
            DYNAMODB_TABLE: this.table.tableName,
            ROOT_PATH: `/${props.stage}`,
            NO_PYDANTIC_SSM_SETTINGS: "1",
            STAC_URL: props.stacUrl,
            DATA_ACCESS_ROLE: props.dataAccessRole.roleArn,
            ...props.apiEnv,
        };
        this.handlerRole = new aws_cdk_lib_1.aws_iam.Role(this, "execution-role", {
            description: "Role used by STAC Ingestor. Manually defined so that we can choose a name that is supported by the data access roles trust policy",
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("lambda.amazonaws.com"),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaBasicExecutionRole"),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaVPCAccessExecutionRole"),
            ],
        });
        const handler = this.buildApiLambda({
            table: this.table,
            env,
            dataAccessRole: props.dataAccessRole,
            stage: props.stage,
            dbSecret: props.stacDbSecret,
            dbVpc: props.vpc,
            dbSecurityGroup: props.stacDbSecurityGroup,
            subnetSelection: props.subnetSelection,
            lambdaFunctionOptions: props.apiLambdaFunctionOptions,
            pgstacVersion: props.pgstacVersion,
        });
        this.buildApiEndpoint({
            handler,
            stage: props.stage,
            endpointConfiguration: props.apiEndpointConfiguration,
            policy: props.apiPolicy,
            ingestorDomainNameOptions: props.ingestorDomainNameOptions,
        });
        this.buildIngestor({
            table: this.table,
            env: env,
            dbSecret: props.stacDbSecret,
            dbVpc: props.vpc,
            dbSecurityGroup: props.stacDbSecurityGroup,
            subnetSelection: props.subnetSelection,
            lambdaFunctionOptions: props.ingestorLambdaFunctionOptions,
            pgstacVersion: props.pgstacVersion,
        });
        this.registerSsmParameter({
            name: "dynamodb_table",
            value: this.table.tableName,
            description: "Name of table used to store ingestions",
        });
    }
    buildTable() {
        const table = new aws_cdk_lib_1.aws_dynamodb.Table(this, "ingestions-table", {
            partitionKey: { name: "created_by", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            sortKey: { name: "id", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            billingMode: aws_cdk_lib_1.aws_dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            stream: aws_cdk_lib_1.aws_dynamodb.StreamViewType.NEW_IMAGE,
        });
        table.addGlobalSecondaryIndex({
            indexName: "status",
            partitionKey: { name: "status", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            sortKey: { name: "created_at", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
        });
        return table;
    }
    buildApiLambda(props) {
        const handler = new aws_cdk_lib_1.aws_lambda.Function(this, "api-handler", {
            // defaults
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_12,
            handler: "src.handler.handler",
            memorySize: 2048,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            code: aws_cdk_lib_1.aws_lambda.Code.fromDockerBuild(__dirname, {
                file: "runtime/Dockerfile",
                buildArgs: {
                    PYTHON_VERSION: "3.12",
                    PGSTAC_VERSION: props.pgstacVersion || utils_1.DEFAULT_PGSTAC_VERSION,
                },
            }),
            allowPublicSubnet: true,
            vpc: props.dbVpc,
            vpcSubnets: props.subnetSelection,
            environment: { DB_SECRET_ARN: props.dbSecret.secretArn, ...props.env },
            role: this.handlerRole,
            // overwrites defaults with user-provided configurable properties
            ...props.lambdaFunctionOptions,
        });
        // Allow handler to read DB secret
        props.dbSecret.grantRead(handler);
        // Allow handler to connect to DB
        if (props.dbVpc) {
            props.dbSecurityGroup.addIngressRule(handler.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connections from STAC Ingestor");
        }
        props.table.grantReadWriteData(handler);
        return handler;
    }
    buildIngestor(props) {
        const handler = new aws_cdk_lib_1.aws_lambda.Function(this, "stac-ingestor", {
            // defaults
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_12,
            handler: "src.ingestor.handler",
            memorySize: 2048,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(180),
            code: aws_cdk_lib_1.aws_lambda.Code.fromDockerBuild(__dirname, {
                file: "runtime/Dockerfile",
                buildArgs: {
                    PYTHON_VERSION: "3.12",
                    PGSTAC_VERSION: props.pgstacVersion || utils_1.DEFAULT_PGSTAC_VERSION,
                },
            }),
            vpc: props.dbVpc,
            vpcSubnets: props.subnetSelection,
            allowPublicSubnet: true,
            environment: { DB_SECRET_ARN: props.dbSecret.secretArn, ...props.env },
            role: this.handlerRole,
            // overwrites defaults with user-provided configurable properties
            ...props.lambdaFunctionOptions,
        });
        // Allow handler to read DB secret
        props.dbSecret.grantRead(handler);
        // Allow handler to connect to DB
        if (props.dbVpc) {
            props.dbSecurityGroup.addIngressRule(handler.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connections from STAC Ingestor");
        }
        // Allow handler to write results back to DB
        props.table.grantWriteData(handler);
        // Trigger handler from writes to DynamoDB table
        handler.addEventSource(new aws_cdk_lib_1.aws_lambda_event_sources.DynamoEventSource(props.table, {
            // Read when batches reach size...
            batchSize: 1000,
            // ... or when window is reached.
            maxBatchingWindow: aws_cdk_lib_1.Duration.seconds(10),
            // Read oldest data first.
            startingPosition: aws_cdk_lib_1.aws_lambda.StartingPosition.TRIM_HORIZON,
            retryAttempts: 1,
        }));
        return handler;
    }
    buildApiEndpoint(props) {
        return new aws_cdk_lib_1.aws_apigateway.LambdaRestApi(this, `${aws_cdk_lib_1.Stack.of(this).stackName}-ingestor-api`, {
            handler: props.handler,
            proxy: true,
            cloudWatchRole: true,
            deployOptions: { stageName: props.stage },
            endpointExportName: `${aws_cdk_lib_1.Stack.of(this)}-ingestor-api`,
            endpointConfiguration: props.endpointConfiguration,
            policy: props.policy,
            domainName: props.ingestorDomainNameOptions
                ? {
                    domainName: props.ingestorDomainNameOptions.domainName,
                    certificate: props.ingestorDomainNameOptions.certificate,
                }
                : undefined,
        });
    }
    registerSsmParameter(props) {
        const parameterNamespace = aws_cdk_lib_1.Stack.of(this).stackName;
        return new aws_cdk_lib_1.aws_ssm.StringParameter(this, `${props.name.replace("_", "-")}-parameter`, {
            description: props.description,
            parameterName: `/${parameterNamespace}/${props.name}`,
            stringValue: props.value,
        });
    }
}
exports.StacIngestor = StacIngestor;
_a = JSII_RTTI_SYMBOL_1;
StacIngestor[_a] = { fqn: "eoapi-cdk.StacIngestor", version: "10.2.1" };
//# sourceMappingURL=data:application/json;base64,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