"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StacBrowser = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cdk_lib_2 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const child_process_1 = require("child_process");
const fs = require("fs");
const DEFAULT_CLONE_DIRECTORY = './stac-browser';
class StacBrowser extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const buildPath = this.buildApp(props, props.cloneDirectory || DEFAULT_CLONE_DIRECTORY);
        // import a bucket from props.bucketArn if defined, otherwise create a new bucket
        if (props.bucketArn) {
            this.bucket = aws_cdk_lib_1.aws_s3.Bucket.fromBucketArn(this, 'Bucket', props.bucketArn);
        }
        else {
            this.bucket = new aws_cdk_lib_1.aws_s3.Bucket(this, 'Bucket', {
                accessControl: aws_cdk_lib_1.aws_s3.BucketAccessControl.PRIVATE,
                removalPolicy: aws_cdk_lib_2.RemovalPolicy.DESTROY,
                websiteIndexDocument: props.websiteIndexDocument
            });
        }
        // if props.cloudFrontDistributionArn is defined and props.bucketArn is not defined, add a bucket policy to allow read access from the cloudfront distribution
        if (props.cloudFrontDistributionArn && !props.bucketArn) {
            this.bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                sid: 'AllowCloudFrontServicePrincipal',
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject'],
                principals: [new aws_iam_1.ServicePrincipal('cloudfront.amazonaws.com')],
                resources: [this.bucket.arnForObjects('*')],
                conditions: {
                    'StringEquals': {
                        'aws:SourceArn': props.cloudFrontDistributionArn
                    }
                }
            }));
        }
        // add the compiled code to the bucket as a bucket deployment
        this.bucketDeployment = new aws_cdk_lib_1.aws_s3_deployment.BucketDeployment(this, 'BucketDeployment', {
            destinationBucket: this.bucket,
            sources: [aws_cdk_lib_1.aws_s3_deployment.Source.asset(buildPath)]
        });
        new aws_cdk_lib_2.CfnOutput(this, "bucket-name", {
            exportName: `${aws_cdk_lib_1.Stack.of(this).stackName}-bucket-name`,
            value: this.bucket.bucketName,
        });
    }
    buildApp(props, cloneDirectory) {
        // Define where to clone and build
        const githubRepoUrl = 'https://github.com/radiantearth/stac-browser.git';
        // Maybe the repo already exists in cloneDirectory. Try checking out the desired version and if it fails, delete and reclone.
        try {
            console.log(`Checking if a valid cloned repo exists with version ${props.githubRepoTag}...`);
            (0, child_process_1.execSync)(`git checkout tags/${props.githubRepoTag}`, { cwd: cloneDirectory });
        }
        catch (error) {
            // if directory exists, raise an error
            if (fs.existsSync(cloneDirectory)) {
                throw new Error(`Directory ${cloneDirectory} already exists and is not a valid clone of ${githubRepoUrl}. Please delete this directory or specify a different cloneDirectory.`);
            }
            // else, we clone and check out the version.
            // Clone the repo
            console.log(`Cloning ${githubRepoUrl} into ${cloneDirectory}...`);
            (0, child_process_1.execSync)(`git clone ${githubRepoUrl} ${cloneDirectory}`);
            // Check out the desired version
            console.log(`Checking out version ${props.githubRepoTag}...`);
            (0, child_process_1.execSync)(`git checkout tags/${props.githubRepoTag}`, { cwd: cloneDirectory });
        }
        // Install the dependencies and build the application
        console.log(`Installing dependencies`);
        (0, child_process_1.execSync)('npm install', { cwd: cloneDirectory });
        // If a config file is provided, copy it to the stac-browser directory at "config.js", replaces the default config.js.
        if (props.configFilePath) {
            // check that the file exists at this location. if not, raise an error and print current working directory.
            if (!fs.existsSync(props.configFilePath)) {
                throw new Error(`Config file ${props.configFilePath} does not exist. Current working directory is ${process.cwd()}`);
            }
            console.log(`Copying config file ${props.configFilePath} to ${cloneDirectory}/config.js`);
            fs.copyFileSync(props.configFilePath, `${cloneDirectory}/config.js`);
        }
        // Build the app with catalogUrl
        console.log(`Building app with catalogUrl=${props.stacCatalogUrl} into ${cloneDirectory}`);
        (0, child_process_1.execSync)(`npm run build -- --catalogUrl=${props.stacCatalogUrl}`, { cwd: cloneDirectory });
        return './stac-browser/dist';
    }
}
exports.StacBrowser = StacBrowser;
_a = JSII_RTTI_SYMBOL_1;
StacBrowser[_a] = { fqn: "eoapi-cdk.StacBrowser", version: "10.2.1" };
//# sourceMappingURL=data:application/json;base64,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