"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StacItemLoader = exports.StacLoader = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const path = require("path");
/**
 * AWS CDK Construct for STAC Object Loading Infrastructure
 *
 * The StacLoader creates a serverless, event-driven system for loading
 * STAC (SpatioTemporal Asset Catalog) objects into a PostgreSQL database with
 * the pgstac extension. This construct supports multiple ingestion pathways
 * for flexible STAC object loading.
 *
 * ## Architecture Overview
 *
 * This construct creates the following AWS resources:
 * - **SNS Topic**: Entry point for STAC objects and S3 event notifications
 * - **SQS Queue**: Buffers and batches messages before processing (60-second visibility timeout)
 * - **Dead Letter Queue**: Captures failed loading attempts after 5 retries
 * - **Lambda Function**: Python function that processes batches and inserts objects into pgstac
 *
 * ## Data Flow
 *
 * The loader supports two primary data ingestion patterns:
 *
 * ### Direct STAC Object Publishing
 * 1. STAC objects (JSON) are published directly to the SNS topic in message bodies
 * 2. The SQS queue collects messages and batches them (up to {batchSize} objects or 1 minute window)
 * 3. The Lambda function receives batches, validates objects, and inserts into pgstac
 *
 * ### S3 Event-Driven Loading
 * 1. An S3 bucket is configured to send notifications to the SNS topic when json files are created
 * 2. STAC objects are uploaded to S3 buckets as JSON/GeoJSON files
 * 3. S3 event notifications are sent to the SNS topic when objects are uploaded
 * 4. The Lambda function receives S3 events in the SQS message batch, fetches objects from S3, and loads into pgstac
 *
 * ## Batching Behavior
 *
 * The SQS-to-Lambda integration uses intelligent batching to optimize performance:
 *
 * - **Batch Size**: Lambda waits to receive up to `batchSize` messages (default: 500)
 * - **Batching Window**: If fewer than `batchSize` messages are available, Lambda
 *   triggers after `maxBatchingWindow` minutes (default: 1 minute)
 * - **Trigger Condition**: Lambda executes when EITHER condition is met first
 * - **Concurrency**: Limited to `maxConcurrency` concurrent executions to prevent database overload
 * - **Partial Failures**: Uses `reportBatchItemFailures` to retry only failed objects
 *
 * This approach balances throughput (larger batches = fewer database connections)
 * with latency (time-based triggers prevent indefinite waiting).
 *
 * ## Error Handling and Dead Letter Queue
 *
 * Failed messages are sent to the dead letter queue after 5 processing attempts.
 * **Important**: This construct provides NO automated handling of dead letter queue
 * messages - monitoring, inspection, and reprocessing of failed objects is the
 * responsibility of the implementing application.
 *
 * Consider implementing:
 * - CloudWatch alarms on dead letter queue depth
 * - Manual or automated reprocessing workflows
 * - Logging and alerting for failed objects
 * - Regular cleanup of old dead letter messages (14-day retention)
 *
 * ## Operational Characteristics
 *
 * - **Scalability**: Lambda scales automatically based on queue depth
 * - **Reliability**: Dead letter queue captures failures for debugging
 * - **Efficiency**: Batching optimizes database operations for high throughput
 * - **Security**: Database credentials accessed via AWS Secrets Manager
 * - **Observability**: CloudWatch logs retained for one week
 *
 * ## Prerequisites
 *
 * Before using this construct, ensure:
 * - The pgstac database has collections loaded (objects require existing collection IDs)
 * - Database credentials are stored in AWS Secrets Manager
 * - The pgstac extension is properly installed and configured
 *
 * ## Usage Example
 *
 * ```typescript
 * // Create database first
 * const database = new PgStacDatabase(this, 'Database', {
 *   pgstacVersion: '0.9.5'
 * });
 *
 * // Create Object loader
 * const loader = new StacLoader(this, 'StacLoader', {
 *   pgstacDb: database,
 *   batchSize: 1000,          // Process up to 1000 objects per batch
 *   maxBatchingWindowMinutes: 1, // Wait max 1 minute to fill batch
 *   lambdaTimeoutSeconds: 300     // Allow up to 300 seconds for database operations
 * });
 *
 * // The topic ARN can be used by other services to publish objects
 * new CfnOutput(this, 'LoaderTopicArn', {
 *   value: loader.topic.topicArn
 * });
 * ```
 *
 * ## Direct Object Publishing
 *
 * External services can publish STAC objects directly to the topic:
 *
 * ```bash
 * aws sns publish --topic-arn $STAC_LOAD_TOPIC --message  '{
 *   "id": "example-collection",
 *   "type": "Collection",
 *   "title": "Example Collection",
 *   "description": "An example collection",
 *   "license": "proprietary",
 *   "extent": {
 *       "spatial": {"bbox": [[-180, -90, 180, 90]]},
 *       "temporal": {"interval": [[null, null]]}
 *   },
 *   "stac_version": "1.1.0",
 *   "links": []
 * }'
 *
 * aws sns publish --topic-arn $STAC_LOAD_TOPIC --message '{
 *   "type": "Feature",
 *   "stac_version": "1.0.0",
 *   "id": "example-item",
 *   "properties": {"datetime": "2021-01-01T00:00:00Z"},
 *   "geometry": {"type": "Polygon", "coordinates": [...]},
 *   "collection": "example-collection"
 * }'
 *
 *
 * ```
 *
 * ## S3 Event Configuration
 *
 * To enable S3 event-driven loading, configure S3 bucket notifications to send
 * events to the SNS topic when STAC objects (.json or .geojson files) are uploaded:
 *
 * ```typescript
 * // Configure S3 bucket to send notifications to the loader topic
 * bucket.addEventNotification(
 *   s3.EventType.OBJECT_CREATED,
 *   new s3n.SnsDestination(loader.topic),
 *   { suffix: '.json' }
 * );
 *
 * bucket.addEventNotification(
 *   s3.EventType.OBJECT_CREATED,
 *   new s3n.SnsDestination(loader.topic),
 *   { suffix: '.geojson' }
 * );
 * ```
 *
 * When STAC objects are uploaded to the configured S3 bucket, the loader will:
 * 1. Receive S3 event notifications via SNS
 * 2. Fetch the STAC JSON from S3
 * 3. Validate and load the objects into the pgstac database
 *
 * ## Monitoring and Troubleshooting
 *
 * - Monitor Lambda logs: `/aws/lambda/{FunctionName}`
 * - **Dead Letter Queue**: Check for failed objects - **no automated handling provided**
 * - Use batch objects failure reporting for partial batch processing
 * - CloudWatch metrics available for queue depth and Lambda performance
 *
 * ### Dead Letter Queue Management
 *
 * Applications must implement their own dead letter queue monitoring:
 *
 * ```typescript
 * // Example: CloudWatch alarm for dead letter queue depth
 * new cloudwatch.Alarm(this, 'DeadLetterAlarm', {
 *   metric: loader.deadLetterQueue.metricApproximateNumberOfVisibleMessages(),
 *   threshold: 1,
 *   evaluationPeriods: 1
 * });
 *
 * // Example: Lambda to reprocess dead letter messages
 * const reprocessFunction = new lambda.Function(this, 'Reprocess', {
 *   // Implementation to fetch and republish failed messages
 * });
 * ```
 *
 */
class StacLoader extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const timeoutSeconds = props.lambdaTimeoutSeconds ?? 300;
        const lambdaRuntime = props.lambdaRuntime ?? aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_12;
        const maxConcurrency = props.maxConcurrency ?? 2;
        // Create dead letter queue
        this.deadLetterQueue = new aws_cdk_lib_1.aws_sqs.Queue(this, "DeadLetterQueue", {
            retentionPeriod: aws_cdk_lib_1.Duration.days(14),
        });
        // Create main queue
        this.queue = new aws_cdk_lib_1.aws_sqs.Queue(this, "Queue", {
            visibilityTimeout: aws_cdk_lib_1.Duration.seconds(timeoutSeconds + 10),
            encryption: aws_cdk_lib_1.aws_sqs.QueueEncryption.SQS_MANAGED,
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
        });
        // Create SNS topic
        this.topic = new aws_cdk_lib_1.aws_sns.Topic(this, "Topic", {
            displayName: `${id}-StacLoaderTopic`,
        });
        // Subscribe the queue to the topic
        this.topic.addSubscription(new aws_cdk_lib_1.aws_sns_subscriptions.SqsSubscription(this.queue));
        // Create the lambda function
        this.lambdaFunction = new aws_cdk_lib_1.aws_lambda.Function(this, "Function", {
            runtime: lambdaRuntime,
            handler: "stac_loader.handler.handler",
            vpc: props.vpc,
            vpcSubnets: props.subnetSelection,
            code: aws_cdk_lib_1.aws_lambda.Code.fromDockerBuild(path.join(__dirname, ".."), {
                file: "stac-loader/runtime/Dockerfile",
                platform: "linux/amd64",
                buildArgs: {
                    PYTHON_VERSION: lambdaRuntime.toString().replace("python", ""),
                    PGSTAC_VERSION: props.pgstacDb.pgstacVersion,
                },
            }),
            memorySize: props.memorySize ?? 1024,
            timeout: aws_cdk_lib_1.Duration.seconds(timeoutSeconds),
            reservedConcurrentExecutions: maxConcurrency,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
            environment: {
                PGSTAC_SECRET_ARN: props.pgstacDb.pgstacSecret.secretArn,
                ...props.environment,
            },
            // overwrites defaults with user-provided configurable properties
            ...props.lambdaFunctionOptions,
        });
        // Grant permissions to read the database secret
        props.pgstacDb.pgstacSecret.grantRead(this.lambdaFunction);
        // Add SQS event source to the lambda
        this.lambdaFunction.addEventSource(new aws_cdk_lib_1.aws_lambda_event_sources.SqsEventSource(this.queue, {
            batchSize: props.batchSize ?? 500,
            maxBatchingWindow: aws_cdk_lib_1.Duration.minutes(props.maxBatchingWindowMinutes ?? 1),
            maxConcurrency: maxConcurrency,
            reportBatchItemFailures: true,
        }));
        // Create outputs
        const exportPrefix = aws_cdk_lib_1.Stack.of(this).stackName;
        new aws_cdk_lib_1.CfnOutput(this, "TopicArn", {
            value: this.topic.topicArn,
            description: "ARN of the StacLoader SNS Topic",
            exportName: `${exportPrefix}-stac-loader-topic-arn`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "QueueUrl", {
            value: this.queue.queueUrl,
            description: "URL of the StacLoader SQS Queue",
            exportName: `${exportPrefix}-stac-loader-queue-url`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "DeadLetterQueueUrl", {
            value: this.deadLetterQueue.queueUrl,
            description: "URL of the StacLoader Dead Letter Queue",
            exportName: `${exportPrefix}-stac-loader-deadletter-queue-url`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "FunctionName", {
            value: this.lambdaFunction.functionName,
            description: "Name of the StacLoader Lambda Function",
            exportName: `${exportPrefix}-stac-loader-function-name`,
        });
    }
}
exports.StacLoader = StacLoader;
_a = JSII_RTTI_SYMBOL_1;
StacLoader[_a] = { fqn: "eoapi-cdk.StacLoader", version: "10.2.1" };
/**
 * @deprecated Use StacLoader instead. StacItemLoader will be removed in a future version.
 */
class StacItemLoader extends StacLoader {
    constructor(scope, id, props) {
        console.warn(`StacItemLoader is deprecated. Please use StacLoader instead. ` +
            `StacItemLoader will be removed in a future version.`);
        super(scope, id, props);
    }
}
exports.StacItemLoader = StacItemLoader;
_b = JSII_RTTI_SYMBOL_1;
StacItemLoader[_b] = { fqn: "eoapi-cdk.StacItemLoader", version: "10.2.1" };
//# sourceMappingURL=data:application/json;base64,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