"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StactoolsItemGenerator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_assets_1 = require("aws-cdk-lib/aws-ecr-assets");
const constructs_1 = require("constructs");
const path = require("path");
/**
 * AWS CDK Construct for STAC Item Generation Infrastructure
 *
 * The StactoolsItemGenerator creates a serverless, event-driven system for generating
 * STAC (SpatioTemporal Asset Catalog) items from source data. This construct
 * implements the first phase of a two-stage ingestion pipeline that transforms
 * raw geospatial data into standardized STAC metadata.
 *
 * ## Architecture Overview
 *
 * This construct creates the following AWS resources:
 * - **SNS Topic**: Entry point for triggering item generation workflows
 * - **SQS Queue**: Buffers generation requests (120-second visibility timeout)
 * - **Dead Letter Queue**: Captures failed messages after 5 processing attempts
 * - **Lambda Function**: Containerized function that generates STAC items using stactools
 *
 * ## Data Flow
 *
 * 1. External systems publish ItemRequest messages to the SNS topic with metadata about assets
 * 2. The SQS queue buffers these messages and triggers the Lambda function
 * 3. The Lambda function:
 *    - Uses `uvx` to install the required stactools package
 *    - Executes the `create-item` CLI command with provided arguments
 *    - Publishes generated STAC items to the ItemLoad topic
 * 4. Failed processing attempts are sent to the dead letter queue
 *
 * ## Operational Characteristics
 *
 * - **Scalability**: Lambda scales automatically based on queue depth (up to maxConcurrency)
 * - **Flexibility**: Supports any stactools package through dynamic installation
 * - **Reliability**: Dead letter queue captures failed generation attempts
 * - **Isolation**: Each generation task runs in a fresh container environment
 * - **Observability**: CloudWatch logs retained for one week
 *
 * ## Message Schema
 *
 * The function expects messages matching the ItemRequest model:
 *
 * ```json
 * {
 *   "package_name": "stactools-glad-global-forest-change",
 *   "group_name": "gladglobalforestchange",
 *   "create_item_args": [
 *     "https://example.com/data.tif"
 *   ],
 *   "collection_id": "glad-global-forest-change-1.11"
 * }
 * ```
 *
 * ## Usage Example
 *
 * ```typescript
 * // Create item loader first (or get existing topic ARN)
 * const loader = new StacLoader(this, 'ItemLoader', {
 *   pgstacDb: database
 * });
 *
 * // Create item generator that feeds the loader
 * const generator = new StactoolsItemGenerator(this, 'ItemGenerator', {
 *   itemLoadTopicArn: loader.topic.topicArn,
 *   lambdaTimeoutSeconds: 120,    // Allow time for package installation
 *   maxConcurrency: 100,          // Control parallel processing
 *   batchSize: 10                 // Process 10 requests per invocation
 * });
 *
 * // Grant permission to publish to the loader topic
 * loader.topic.grantPublish(generator.lambdaFunction);
 * ```
 *
 * ## Publishing Generation Requests
 *
 * Send messages to the generator topic to trigger item creation:
 *
 * ```bash
 * aws sns publish --topic-arn $ITEM_GEN_TOPIC --message '{
 *   "package_name": "stactools-glad-global-forest-change",
 *   "group_name": "gladglobalforestchange",
 *   "create_item_args": [
 *     "https://storage.googleapis.com/earthenginepartners-hansen/GFC-2023-v1.11/Hansen_GFC-2023-v1.11_gain_40N_080W.tif"
 *   ],
 *   "collection_id": "glad-global-forest-change-1.11"
 * }'
 * ```
 *
 * ## Batch Processing Example
 *
 * For processing many assets, you can loop through URLs:
 *
 * ```bash
 * while IFS= read -r url; do
 *   aws sns publish --topic-arn "$ITEM_GEN_TOPIC" --message "{
 *     \"package_name\": \"stactools-glad-glclu2020\",
 *     \"group_name\": \"gladglclu2020\",
 *     \"create_item_args\": [\"$url\"]
 *   }"
 * done < urls.txt
 * ```
 *
 * ## Monitoring and Troubleshooting
 *
 * - Monitor Lambda logs: `/aws/lambda/{FunctionName}`
 * - Check dead letter queue for failed generation attempts
 * - Use CloudWatch metrics to track processing rates and errors
 * - Failed items can be replayed from the dead letter queue
 *
 * ## Supported Stactools Packages
 *
 * Any package available on PyPI that follows the stactools plugin pattern
 * can be used. Examples include:
 * - `stactools-glad-global-forest-change`
 * - `stactools-glad-glclu2020`
 * - `stactools-landsat`
 * - `stactools-sentinel2`
 *
 * @see {@link https://github.com/stactools-packages} for available stactools packages
 * @see {@link https://stactools.readthedocs.io/} for stactools documentation
 */
class StactoolsItemGenerator extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const timeoutSeconds = props.lambdaTimeoutSeconds ?? 120;
        const lambdaRuntime = props.lambdaRuntime ?? aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_12;
        // Create dead letter queue
        this.deadLetterQueue = new aws_cdk_lib_1.aws_sqs.Queue(this, "DeadLetterQueue", {
            retentionPeriod: aws_cdk_lib_1.Duration.days(14),
        });
        // Create main queue
        this.queue = new aws_cdk_lib_1.aws_sqs.Queue(this, "Queue", {
            visibilityTimeout: aws_cdk_lib_1.Duration.seconds(timeoutSeconds + 10),
            encryption: aws_cdk_lib_1.aws_sqs.QueueEncryption.SQS_MANAGED,
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
        });
        // Create SNS topic
        this.topic = new aws_cdk_lib_1.aws_sns.Topic(this, "Topic", {
            displayName: `${id}-ItemGenTopic`,
        });
        // Subscribe the queue to the topic
        this.topic.addSubscription(new aws_cdk_lib_1.aws_sns_subscriptions.SqsSubscription(this.queue));
        // Create the lambda function
        this.lambdaFunction = new aws_cdk_lib_1.aws_lambda.DockerImageFunction(this, "Function", {
            code: aws_cdk_lib_1.aws_lambda.DockerImageCode.fromImageAsset(path.join(__dirname, ".."), {
                file: "stactools-item-generator/runtime/Dockerfile",
                platform: aws_ecr_assets_1.Platform.LINUX_AMD64,
                buildArgs: {
                    PYTHON_VERSION: lambdaRuntime.toString().replace("python", ""),
                },
            }),
            memorySize: props.memorySize ?? 1024,
            vpc: props.vpc,
            vpcSubnets: props.subnetSelection,
            timeout: aws_cdk_lib_1.Duration.seconds(timeoutSeconds),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
            environment: {
                ITEM_LOAD_TOPIC_ARN: props.itemLoadTopicArn,
                LOG_LEVEL: "INFO",
                ...props.environment,
            },
            // overwrites defaults with user-provided configurable properties
            ...props.lambdaFunctionOptions,
        });
        // Add SQS event source to the lambda
        this.lambdaFunction.addEventSource(new aws_cdk_lib_1.aws_lambda_event_sources.SqsEventSource(this.queue, {
            batchSize: props.batchSize ?? 10,
            reportBatchItemFailures: true,
            maxConcurrency: props.maxConcurrency ?? 100,
        }));
        // Grant permissions to publish to the item load topic
        // Note: This will be granted externally since we only have the ARN
        // The consuming construct should handle this permission
        // Create outputs
        const exportPrefix = aws_cdk_lib_1.Stack.of(this).stackName;
        new aws_cdk_lib_1.CfnOutput(this, "TopicArn", {
            value: this.topic.topicArn,
            description: "ARN of the StactoolsItemGenerator SNS Topic",
            exportName: `${exportPrefix}-stactools-item-generator-topic-arn`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "QueueUrl", {
            value: this.queue.queueUrl,
            description: "URL of the StactoolsItemGenerator SQS Queue",
            exportName: `${exportPrefix}-stactools-item-generator-queue-url`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "DeadLetterQueueUrl", {
            value: this.deadLetterQueue.queueUrl,
            description: "URL of the StactoolsItemGenerator Dead Letter Queue",
            exportName: `${exportPrefix}-stactools-item-generator-deadletter-queue-url`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "FunctionName", {
            value: this.lambdaFunction.functionName,
            description: "Name of the StactoolsItemGenerator Lambda Function",
            exportName: `${exportPrefix}-stactools-item-generator-function-name`,
        });
    }
}
exports.StactoolsItemGenerator = StactoolsItemGenerator;
_a = JSII_RTTI_SYMBOL_1;
StactoolsItemGenerator[_a] = { fqn: "eoapi-cdk.StactoolsItemGenerator", version: "10.2.1" };
//# sourceMappingURL=data:application/json;base64,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