"""Neo blockchain data provider

This module provides a comprehensive interface for interacting with the Neo blockchain
using neo-mamba library. It supports both mainnet and testnet networks and includes
various utility methods for data processing and validation.

The provider handles:
- Address validation and conversion
- Asset amount formatting
- JSON serialization
- Safe dictionary access
- Network-specific API endpoints
"""

import json
from typing import Dict, Any, List, Optional
from decimal import Decimal
from neo3.api import NeoRpcClient
from neo3.core import types
from neo3 import settings

# RPC URLs for different networks
MAINNET_RPC = "https://mainnet1.neo.org:443"
TESTNET_RPC = "https://testnet1.neo.org:443"

class NeoProvider:
    """Neo blockchain data provider using neo-mamba library

    This class provides a unified interface for querying Neo blockchain data
    including addresses, assets, blocks, transactions, and smart contracts.

    Attributes:
        network (str): The Neo network to connect to ('mainnet' or 'testnet')
        rpc_client (NeoRpcClient): The neo-mamba RPC client for blockchain interaction
    """

    def __init__(self, network: str = "testnet"):
        """Initialize the Neo provider

        Args:
            network (str): The Neo network to connect to. Must be 'mainnet' or 'testnet'

        Raises:
            ValueError: If network is not 'mainnet' or 'testnet'
        """
        if network not in ["mainnet", "testnet"]:
            raise ValueError("Network must be 'mainnet' or 'testnet'")

        self.network = network
        rpc_url = MAINNET_RPC if network == "mainnet" else TESTNET_RPC

        # Initialize neo-mamba RPC client
        self.rpc_client = NeoRpcClient(rpc_url)

    def _validate_address(self, address: str) -> types.UInt160:
        """Validate and convert address format

        Converts Neo addresses to script hash format if they are in standard format.
        If the address is already in script hash format, it returns as is.

        Args:
            address (str): The address to validate and convert

        Returns:
            types.UInt160: The address as UInt160 script hash

        Raises:
            ValueError: If the address is not a valid Neo address
        """
        try:
            # Try to parse as script hash first
            if address.startswith("0x"):
                return types.UInt160.from_string(address[2:])
            else:
                return types.UInt160.from_string(address)
        except:
            # If that fails, try to convert from standard address format
            try:
                return self.rpc_client.validate_address(address)
            except:
                raise ValueError(f"Invalid Neo address: {address}")

    def _handle_response(self, result: Any) -> Any:
        """Handle neo-mamba response and extract result

        Args:
            result: The result from neo-mamba RPC call

        Returns:
            Any: The processed result data

        Raises:
            Exception: If the response contains an error
        """
        if result is None:
            raise Exception("Empty response from Neo RPC")

        return result

    def _convert_asset_amount(self, amount_string: str, decimals: int) -> Decimal:
        """Convert asset amount string to decimal with proper decimal places

        Args:
            amount_string (str): The amount as a string
            decimals (int): The number of decimal places for the asset

        Returns:
            Decimal: The converted amount with proper decimal places

        Raises:
            ValueError: If the amount string is invalid
        """
        try:
            amount = Decimal(amount_string)
            return amount / (10 ** decimals)
        except (ValueError, TypeError):
            raise ValueError(f"Invalid amount string: {amount_string}")

    def _format_amount(self, amount: Decimal, decimals: int = 8) -> str:
        """Format amount with specified decimal places

        Args:
            amount (Decimal): The amount to format
            decimals (int): The number of decimal places to use

        Returns:
            str: The formatted amount string
        """
        return f"{amount:.{decimals}f}"

    def _to_json(self, obj: Any) -> str:
        """Convert object to JSON string

        Args:
            obj: The object to serialize

        Returns:
            str: The JSON string representation
        """
        return json.dumps(obj, default=lambda obj: obj.__dict__)

    # Address-related methods
    async def get_active_addresses(self, days: int) -> List[int]:
        """Get active addresses count for specified days

        Args:
            days (int): Number of days to get active address counts for

        Returns:
            List[int]: List of daily active address counts
        """
        # Note: neo-mamba doesn't have a direct method for active addresses
        # This is a limitation that may need to be addressed differently
        return []

    async def get_address_info(self, address: str) -> Dict[str, Any]:
        """Get address information

        Args:
            address (str): The Neo address to get information for

        Returns:
            Dict[str, Any]: Address information including first use time, last use time, etc.
        """
        try:
            validated_address = self._validate_address(address)
            # Get NEP-17 balances for the address
            balances = await self.rpc_client.get_nep17_balances(validated_address)
            return self._handle_response({
                "address": address,
                "balances": balances,
                "script_hash": str(validated_address)
            })
        except Exception as e:
            raise Exception(f"Failed to get address info: {str(e)}")

    async def get_address_count(self) -> int:
        """Get total address count

        Returns:
            int: Total number of addresses on the network
        """
        # Note: neo-mamba doesn't have a direct method for address count
        # This is a limitation that may need to be addressed differently
        return 0

    # Block-related methods
    async def get_block_info(self, block_hash: str) -> Dict[str, Any]:
        """Get block information by hash

        Args:
            block_hash (str): The block hash to get information for

        Returns:
            Dict[str, Any]: Block information including transactions, timestamp, etc.
        """
        try:
            block_hash_uint = types.UInt256.from_string(block_hash)
            block = await self.rpc_client.get_block(block_hash_uint)
            return self._handle_response(block)
        except Exception as e:
            raise Exception(f"Failed to get block info: {str(e)}")

    async def get_block_by_height(self, block_height: int) -> Dict[str, Any]:
        """Get block information by height

        Args:
            block_height (int): The block height to get information for

        Returns:
            Dict[str, Any]: Block information including transactions, timestamp, etc.
        """
        try:
            block = await self.rpc_client.get_block(block_height)
            return self._handle_response(block)
        except Exception as e:
            raise Exception(f"Failed to get block by height: {str(e)}")

    async def get_block_count(self) -> int:
        """Get total block count

        Returns:
            int: Total number of blocks on the network
        """
        try:
            count = await self.rpc_client.get_block_count()
            return self._handle_response(count)
        except Exception as e:
            raise Exception(f"Failed to get block count: {str(e)}")

    # Transaction-related methods
    async def get_transaction_info(self, tx_hash: str) -> Dict[str, Any]:
        """Get transaction information

        Args:
            tx_hash (str): The transaction hash to get information for

        Returns:
            Dict[str, Any]: Transaction information including inputs, outputs, etc.
        """
        try:
            tx_hash_uint = types.UInt256.from_string(tx_hash)
            transaction = await self.rpc_client.get_transaction(tx_hash_uint)
            return self._handle_response(transaction)
        except Exception as e:
            raise Exception(f"Failed to get transaction info: {str(e)}")

    async def get_transaction_count(self) -> int:
        """Get total transaction count

        Returns:
            int: Total number of transactions on the network
        """
        try:
            # Note: neo-mamba doesn't have a direct transaction count method
            # We'll need to use a different approach or estimate
            block_count = await self.rpc_client.get_block_count()
            # For now, return block count as an approximation
            return self._handle_response(block_count)
        except Exception as e:
            raise Exception(f"Failed to get transaction count: {str(e)}")

    # Asset-related methods
    async def get_asset_info(self, asset_hash: str) -> Dict[str, Any]:
        """Get asset information by hash

        Args:
            asset_hash (str): The asset hash to get information for

        Returns:
            Dict[str, Any]: Asset information including name, symbol, decimals, etc.
        """
        try:
            asset_hash_uint = types.UInt160.from_string(asset_hash)
            contract_state = await self.rpc_client.get_contract_state(asset_hash_uint)
            return self._handle_response(contract_state)
        except Exception as e:
            raise Exception(f"Failed to get asset info: {str(e)}")

    async def get_asset_count(self) -> int:
        """Get total asset count

        Returns:
            int: Total number of assets on the network
        """
        # Note: neo-mamba doesn't have a direct method for asset count
        # This is a limitation that may need to be addressed differently
        return 0

    # Contract-related methods
    async def get_contract_info(self, contract_hash: str) -> Dict[str, Any]:
        """Get contract information by hash

        Args:
            contract_hash (str): The contract hash to get information for

        Returns:
            Dict[str, Any]: Contract information including name, hash, etc.
        """
        try:
            contract_hash_uint = types.UInt160.from_string(contract_hash)
            contract_state = await self.rpc_client.get_contract_state(contract_hash_uint)
            return self._handle_response(contract_state)
        except Exception as e:
            raise Exception(f"Failed to get contract info: {str(e)}")

    async def get_contract_count(self) -> int:
        """Get total contract count

        Returns:
            int: Total number of contracts on the network
        """
        # Note: neo-mamba doesn't have a direct method for contract count
        # This is a limitation that may need to be addressed differently
        return 0

    async def close(self):
        """Close the RPC client connection"""
        await self.rpc_client.close()

    def __enter__(self):
        """Context manager entry point"""
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        """Context manager exit point - closes the RPC client"""
        # Note: async close should be handled by the caller
        pass