import pytest

from blockapi.v2.blockchain_mapping import (
    get_blockchain_from_chain_id,
    get_blockchain_from_coingecko_chain,
    get_blockchain_from_debank_chain,
    get_blockchain_from_rango_chain,
)
from blockapi.v2.models import Blockchain


def test_map_case():
    assert get_blockchain_from_rango_chain('TERRA') == Blockchain.TERRA


def test_map_none():
    assert not get_blockchain_from_rango_chain(None)


@pytest.mark.parametrize(
    'chain,expected',
    [
        ('ShibChain', Blockchain.SHIBA_CHAIN),
        ('ancient8', Blockchain.ANCIENT8),
        ('aptos', Blockchain.APTOS),
        ('archway', Blockchain.ARCHWAY),
        ('areon-network', Blockchain.AREON),
        ('astar-zkevm', Blockchain.ASTAR_ZKEVM),
        ('bahamut', Blockchain.BAHAMUT),
        ('base', Blockchain.BASE),
        ('bifrost-network', Blockchain.BIFROST),
        ('bitcanna', Blockchain.BITCANNA),
        ('bitkub-chain', Blockchain.BITKUB_CHAIN),
        ('bitlayer', Blockchain.BIT_LAYER),
        ('bittorrent', Blockchain.BIT_TORRENT),
        ('blast', Blockchain.BLAST),
        ('bob-network', Blockchain.BOB),
        ('bouncebit', Blockchain.BOUNCE_BIT),
        ('bsquared-network', Blockchain.B2),
        ('callisto', Blockchain.CALLISTO),
        ('casper-network', Blockchain.CASPER),
        ('comdex', Blockchain.COMDEX),
        ('core', Blockchain.CORE_CHAIN),
        ('crescent', Blockchain.CRESCENT),
        ('defi-kingdoms-blockchain', Blockchain.DEFI_KINGDOMS),
        ('defichain', Blockchain.DEFI),
        ('drc-20', Blockchain.DRC_20),
        ('empire', Blockchain.EMPIRE),
        ('endurance', Blockchain.ENDURANCE),
        ('eos-evm', Blockchain.EOS),
        ('exosama', Blockchain.EXOSAMA),
        ('filecoin', Blockchain.FILECOIN),
        ('flare-network', Blockchain.FLARE),
        ('fraxtal', Blockchain.FRAXTAL),
        ('function-x', Blockchain.FUNCTION_X),
        ('galachain', Blockchain.GALA),
        ('genesys-network', Blockchain.GENESYS),
        ('gravity-bridge', Blockchain.GRAVITY_BRIDGE),
        ('hoo', Blockchain.HOO_SMART_CHAIN),
        ('hyperliquid', Blockchain.HYPERLIQUID),
        ('hypra-network', Blockchain.HYPRA_NETWORK),
        ('immutable', Blockchain.IMMUTABLE),
        ('iota-evm', Blockchain.IOTA_EVM),
        ('kadena', Blockchain.KADENA),
        ('kujira', Blockchain.KUJIRA),
        ('linea', Blockchain.LINEA),
        ('lisk', Blockchain.LISK),
        ('mainnetz', Blockchain.MAINNETZ),
        ('mantle', Blockchain.MANTLE),
        ('massa', Blockchain.MASSA),
        ('merlin-chain', Blockchain.MERLIN_CHAIN),
        ('mode', Blockchain.MODE),
        ('neon-evm', Blockchain.NEON_EVM),
        ('oasis', Blockchain.OASIS_CHAIN),
        ('oasis-sapphire', Blockchain.OASIS_SAPPHIRE),
        ('oasys', Blockchain.OASYS),
        ('octaspace', Blockchain.OCTA_SPACE),
        ('omax', Blockchain.OMAX),
        ('omniflix', Blockchain.OMNIFLIX),
        ('onus', Blockchain.ONUS_CHAIN),
        ('opbnb', Blockchain.OPTIMISTIC_BNB),
        ('oraichain', Blockchain.ORAI),
        ('ordinals', Blockchain.ORDINALS),
        ('orenium', Blockchain.ORENIUM),
        ('polygon-zkevm', Blockchain.POLYGON_ZK_EVM),
        ('proof-of-memes', Blockchain.PROOF_OF_MEMES),
        ('pulsechain', Blockchain.PULSE),
        ('quicksilver', Blockchain.QUICKSILVER),
        ('rails-network', Blockchain.RAILS),
        ('re-al', Blockchain.REAL),
        ('rollux', Blockchain.ROLLUX),
        ('saakuru', Blockchain.SAAKURU),
        ('saga', Blockchain.SAGA),
        ('saita-chain', Blockchain.SAITA),
        ('sanko', Blockchain.SANKO),
        ('sei-network', Blockchain.SEI_NETWORK),
        ('sei-v2', Blockchain.SEI_V2),
        ('sge', Blockchain.SGE),
        ('shibarium', Blockchain.SHIBARIUM),
        ('sui', Blockchain.SUI),
        ('tenet', Blockchain.TENET),
        ('terra-2', Blockchain.TERRA_2),
        ('the-open-network', Blockchain.OPEN_NETWORK),
        ('trustless-computer', Blockchain.TRUSTLESS_COMPUTER),
        ('valobit', Blockchain.VALOBIT),
        ('wax', Blockchain.WAX),
        ('wemix-network', Blockchain.WEMIX_NETWORK),
        ('xpla', Blockchain.XPLA),
        ('zedxion', Blockchain.ZEDXION),
        ('zetachain', Blockchain.ZETA_CHAIN),
        ('zkfair', Blockchain.ZKFAIR),
        ('zksync', Blockchain.ZKSYNC_ERA),
        ('zora-network', Blockchain.ZORA),
    ],
)
def test_map_coingecko(chain: str, expected: Blockchain):
    assert get_blockchain_from_coingecko_chain(chain) == expected


def test_map_by_id():
    assert get_blockchain_from_chain_id(1) == Blockchain.ETHEREUM


@pytest.mark.parametrize(
    'chain,expected',
    [
        ('ailayer', Blockchain.AI_LAYER),
        ('alot', Blockchain.DEX_ALOT),
        ('apex', Blockchain.PROOF_OF_PLAY_APEX),
        ('aze', Blockchain.ASTAR_ZKEVM),
        ('b2', Blockchain.B2),
        ('bb', Blockchain.BOUNCE_BIT),
        ('bevm', Blockchain.BEVM),
        ('bfc', Blockchain.BIFROST),
        ('bob', Blockchain.BOB),
        ('brise', Blockchain.BITGERT),
        ('btr', Blockchain.BIT_LAYER),
        ('cfx', Blockchain.CONFLUX),
        ('ckb', Blockchain.GODWOKEN),
        ('cyber', Blockchain.CYBER),
        ('dbk', Blockchain.DBK),
        ('degen', Blockchain.DEGEN),
        ('doge', Blockchain.DOGECHAIN),
        ('dym', Blockchain.DYMENSION),
        ('ela', Blockchain.ELASTOS),
        ('eon', Blockchain.HORIZEN_EON),
        ('era', Blockchain.ZKSYNC_ERA),
        ('fon', Blockchain.FON_CHAIN),
        ('frax', Blockchain.FRAXTAL),
        ('fx', Blockchain.FUNCTION_X),
        ('iota', Blockchain.IOTA_EVM),
        ('karak', Blockchain.KARAK),
        ('kroma', Blockchain.KROMA),
        ('lumio', Blockchain.SUPER_LUMIO),
        ('mada', Blockchain.MILKOMEDA_C1),
        ('manta', Blockchain.MANTA_PACIFIC),
        ('map', Blockchain.MAP_PROTOCOL),
        ('merlin', Blockchain.MERLIN_CHAIN),
        ('mint', Blockchain.MINT),
        ('molten', Blockchain.MOLTEN),
        ('neon', Blockchain.NEON_EVM),
        ('nova', Blockchain.ARBITRUM_NOVA),
        ('platon', Blockchain.PLAT_ON),
        ('pze', Blockchain.POLYGON_ZK_EVM),
        ('reya', Blockchain.REYA),
        ('rsk', Blockchain.RSK),
        ('savm', Blockchain.SATOSHI_VM_ALPHA),
        ('scrl', Blockchain.SCROLL),
        ('sei', Blockchain.SEI),
        ('shib', Blockchain.SHIBARIUM),
        ('smr', Blockchain.SHIMMER_EVM),
        ('step', Blockchain.STEP_NETWORK),
        ('strax', Blockchain.STRATIS),
        ('sx', Blockchain.SX_NETWORK),
        ('taiko', Blockchain.TAIKO),
        ('tomb', Blockchain.TOMBCHAIN),
        ('ulx', Blockchain.ULTRON),
        ('xlayer', Blockchain.X_LAYER),
        ('zeta', Blockchain.ZETA_CHAIN),
        ('zklink', Blockchain.ZKLINK_NOVA),
    ],
)
def test_map_debank(chain: str, expected: Blockchain):
    assert get_blockchain_from_debank_chain(chain) == expected
