import { IResource, RemovalPolicy } from "aws-cdk-lib";
import { BuildEnvironment, BuildSpec, ComputeType, IBuildImage, PipelineProject } from "aws-cdk-lib/aws-codebuild";
import { ISecurityGroup, IVpc, SubnetSelection } from "aws-cdk-lib/aws-ec2";
import { IRepository } from "aws-cdk-lib/aws-ecr";
import { LogGroup, RetentionDays } from "aws-cdk-lib/aws-logs";
import { Construct } from "constructs";
/**
 * Common Docker build properties.
 */
export interface CommonDockerProps {
    /**
     * The ECR repository where images are pushed.
     */
    readonly repository: IRepository;
    /**
     * A description of this CodeBuild project.
     */
    readonly description?: string;
    /**
     * VPC network to place CodeBuild network interfaces.
     */
    readonly vpc?: IVpc;
    /**
     * Where to place the network interfaces within the VPC.
     */
    readonly subnetSelection?: SubnetSelection;
    /**
     * Security groups to associate with the project's network interfaces.
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * The removal policy for this project and its logs.
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * The duration to retain log entries.
     *
     * @default - RetentionDays.THREE_MONTHS
     */
    readonly logRetention?: RetentionDays;
}
/**
 * Constructor properties for BaseDockerProject.
 */
export interface BaseDockerProjectProps extends CommonDockerProps {
    /**
     * Filename or contents of buildspec in JSON format.
     */
    readonly buildSpec: BuildSpec;
    /**
     * Build environment to use for the build.
     */
    readonly buildEnvironment: BuildEnvironment;
}
/**
 * The base for Linux-based Docker build projects.
 */
export declare class BaseDockerProject extends Construct implements IResource {
    /**
     * The log group.
     */
    readonly logGroup: LogGroup;
    /**
     * The CodeBuild project.
     */
    readonly project: PipelineProject;
    /**
     * The CodeBuild build spec supplied.
     */
    readonly buildSpec: BuildSpec;
    /**
     * Creates a new BaseDockerProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope: Construct, id: string, props: BaseDockerProjectProps);
    get stack(): import("aws-cdk-lib").Stack;
    get env(): import("aws-cdk-lib").ResourceEnvironment;
    applyRemovalPolicy(policy: RemovalPolicy): void;
}
/**
 * Constructor properties for LinuxDockerBuildProject.
 */
export interface LinuxDockerBuildProjectProps extends CommonDockerProps {
    /**
     * The filename of the Dockerfile.
     *
     * @default - Dockerfile
     */
    readonly dockerfile?: string;
    /**
     * The build context directory.
     *
     * @default - The current directory (.)
     */
    readonly buildDirectory?: string;
    /**
     * Whether to enable build caching.
     *
     * @default - false
     */
    readonly enableCache?: boolean;
    /**
     * Commands used to test the image once built.
     */
    readonly testCommands?: string[];
    /**
     * Whether to push a "latest" tag.
     *
     * @default - true
     */
    readonly pushLatest?: boolean;
    /**
     * The CodeBuild build image to use.
     *
     * @default - LinuxBuildImage.STANDARD_7_0
     */
    readonly buildImage?: IBuildImage;
    /**
     * The type of compute to use for this build.
     *
     * @default - ComputeType.SMALL
     */
    readonly computeType?: ComputeType;
}
/**
 * Sets up a standardized Docker build project.
 *
 * This project accepts the following optional environment variables:
 *
 * - IMAGE_LABELS: JSON-formatted object of container labels and their values.
 * - BUILD_ARGS: JSON-formatted object of build arguments and their values.
 * - IMAGE_TAG: Optional. The image tag (e.g. Git commit ID) (default: build
 *   number).
 */
export declare class LinuxDockerBuildProject extends BaseDockerProject {
    /**
     * Creates a new LinuxDockerBuildProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope: Construct, id: string, props: LinuxDockerBuildProjectProps);
}
/**
 * Constructor properties for LinuxDockerManifestProject.
 */
export interface LinuxDockerManifestProjectProps extends CommonDockerProps {
    /**
     * The names of environment variables that contain the image hashes to add.
     */
    readonly tagVariableNames: string[];
    /**
     * The CodeBuild build image to use.
     *
     * @default - LinuxBuildImage.STANDARD_7_0
     */
    readonly buildImage?: IBuildImage;
    /**
     * The type of compute to use for this build.
     *
     * @default - ComputeType.SMALL
     */
    readonly computeType?: ComputeType;
}
/**
 * Sets up a standardized Docker manifest build project.
 *
 * This project accepts the following variables:
 *
 * - LATEST_TAG: Optional. The tag to push (default: "latest").
 * - MANIFEST_CUSTOM_TAG: Optional. The tag to push, in addition to $LATEST_TAG.
 */
export declare class LinuxDockerManifestProject extends BaseDockerProject {
    /**
     * Creates a new LinuxDockerManifestProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope: Construct, id: string, props: LinuxDockerManifestProjectProps);
}
