"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ShellCommands = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * A utility class that provides POSIX shell commands for User Data scripts.
 */
class ShellCommands {
    /**
     * Writes the literal contents of a string to a destination file.
     *
     * @param contents - The file contents
     * @param destination - The filename to output
     * @param options - Configuration options
     * @returns The shell commands.
     */
    static outputFile(contents, destination, options) {
        const { delimiter = "END_OF_FILE", substitution } = options || {};
        const firstDelimiter = substitution ? delimiter : `'${delimiter}'`;
        return [
            `cat << ${firstDelimiter} > "${destination}"`,
            ...contents.split("\n"),
            delimiter,
        ];
    }
    /**
     * Gets commands to synchronize objects from an S3 bucket to the filesystem.
     *
     * e.g. `syncFromBucket(bucket, {"nginx-config": "/etc/nginx"})`.
     *
     * Be sure to grant your autoscaling group or EC2 instance read access.
     *
     * @param bucket - The source bucket
     * @param destinations - Record with S3 object keys to filesystem path values
     * @returns The shell commands.
     */
    static syncFromBucket(bucket, destinations) {
        const destinationsLength = Object.values(destinations).length;
        if (destinationsLength === 0) {
            return [];
        }
        const region = bucket.env.region;
        if (destinationsLength === 1) {
            return Object.entries(destinations).map(([key, dest]) => `aws --region ${region} s3 sync --only-show-errors "s3://${bucket.bucketName}/${key}" "${dest}"`);
        }
        else {
            const commands = [`configBucket="s3://${bucket.bucketName}"`];
            Object.entries(destinations).forEach(([key, dest]) => commands.push(`aws --region ${region} s3 sync --only-show-errors "$configBucket/${key}" "${dest}"`));
            return commands;
        }
    }
    /**
     * Uses either `mkdir` or `install` to create a directory.
     *
     * @param name - The name of the directory to create
     * @param options - Configuration options
     * @returns The shell commands.
     */
    static addDirectory(name, options) {
        const { owner, group, mode } = options || {};
        if (!owner && !group) {
            return [mode ? `mkdir -p -m "${mode}" "${name}"` : `mkdir -p "${name}"`];
        }
        const parts = ["install -d"];
        if (mode) {
            parts.push(`-m "${mode}"`);
        }
        if (owner) {
            parts.push(`-o "${owner}"`);
        }
        if (group) {
            parts.push(`-g "${group}"`);
        }
        parts.push(`"${name}"`);
        return [parts.join(" ")];
    }
    /**
     * Gets the command to download a Secrets Manager secret to the filesystem.
     *
     * Be sure to grant your autoscaling group or EC2 instance read access.
     *
     * @param secret - The secret to download.
     * @param destination - The local filesystem path where the secret is stored.
     * @returns The shell commands.
     */
    static downloadSecret(secret, destination) {
        const region = secret.env.region;
        return [
            `aws --region ${region} secretsmanager get-secret-value --secret-id "${secret.secretArn}" --query SecretString --output text > "${destination}"`,
        ];
    }
    /**
     * Gets a command to change the ownership and/or group membership of a file.
     *
     * If both `uid` and `gid` are provided, this method returns a single
     * `chown` command to set both values. If just `uid` is provided, this method
     * returns a single `chown` command that sets the owner. If just `gid` is
     * provided, this method returns a single `chgrp` command. If neither are
     * provided, this method returns an empty array.
     *
     * @param filename - The local filesystem path to the file or directory.
     * @param uid - Optional. The owner username or uid.
     * @param gid - Optional. The group name or gid.
     * @returns The shell commands.
     */
    static changeOwnership(filename, uid, gid) {
        const commands = [];
        if (uid && gid) {
            commands.push(`chown ${uid}:${gid} "${filename}"`);
        }
        else if (uid) {
            commands.push(`chown ${uid} "${filename}"`);
        }
        else if (gid) {
            commands.push(`chgrp ${gid} "${filename}"`);
        }
        return commands;
    }
    /**
     * Gets the command to mount an EFS filesystem to a destination path.
     *
     * Be sure to grant your autoscaling group or EC2 instance network access.
     *
     * @param filesystem - The EFS filesystem
     * @param destination - The local filesystem path for the mount point
     * @returns The shell commands.
     */
    static mountElasticFileSystem(filesystem, destination) {
        const region = filesystem.env.region;
        const filesystemHostname = `${filesystem.fileSystemId}.efs.${region}.amazonaws.com`;
        return [
            `mkdir -p "${destination}"`,
            `mount -t nfs4 -o nfsvers=4.1,rsize=1048576,wsize=1048576,hard,timeo=600,retrans=2,noresvport ${filesystemHostname}:/ "${destination}"`,
        ];
    }
    /**
     * Gets a command to disable unattended package upgrades on Debian/Ubuntu.
     *
     * @returns The shell commands.
     */
    static disableUnattendedUpgrades() {
        return [
            'cp "/usr/share/unattended-upgrades/20auto-upgrades-disabled" "/etc/apt/apt.conf.d/"',
            "snap refresh --hold || true",
        ];
    }
    /**
     * Gets commands to setup APT and install packages.
     *
     * @param packages - The packages to install
     * @param options - Configuration options
     * @returns The shell commands.
     */
    static installAptPackages(packages, options) {
        const { repositories = [], autoRemove = true } = options || {};
        const packagesToInstall = new Set(packages);
        const repositoriesToAdd = new Set(repositories);
        const commands = [];
        if (repositoriesToAdd.size > 0) {
            commands.push(...repositories.map((r) => `add-apt-repository -y "${r}"`));
        }
        commands.push("apt -yq update");
        if (packagesToInstall.size > 0) {
            const packageNames = Array.from(packagesToInstall).join(" ");
            commands.push(`DEBIAN_FRONTEND=noninteractive apt -yq -o Dpkg::Options::="--force-confdef" -o Dpkg::Options::="--force-confold" install ${packageNames}`);
        }
        if (autoRemove) {
            commands.push("apt -yq autoremove");
        }
        return commands;
    }
}
exports.ShellCommands = ShellCommands;
_a = JSII_RTTI_SYMBOL_1;
ShellCommands[_a] = { fqn: "shady-island.configuration.ShellCommands", version: "0.1.68" };
//# sourceMappingURL=data:application/json;base64,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