import { UserData, LinuxUserDataOptions } from "aws-cdk-lib/aws-ec2";
/**
 * A container for lines of a User Data script, sortable by `priority`.
 */
export interface PrioritizedLines {
    /**
     * The command lines.
     */
    readonly lines: string[];
    /**
     * The priority for this set of commands.
     */
    readonly priority: number;
}
/**
 * A utility class to assist with composing instance User Data.
 *
 * This class allows multiple observers in code to add lines to the same end
 * result UserData without clobbering each other. Just like `conf.d` directories
 * with priority number prefixes, you can declare the proper execution order of
 * your UserData commands without having to add them in that order.
 */
export declare abstract class UserDataBuilder {
    /**
     * Returns a user data builder for GNU/Linux operating systems.
     *
     * @param options - The Linux UserData constructor options
     * @returns the new builder object
     */
    static forLinux(options?: LinuxUserDataOptions): UserDataBuilder;
    /**
     * Returns a user data builder for Windows operating systems.
     *
     * @returns the new builder object
     */
    static forWindows(): UserDataBuilder;
    /**
     * The groups of prioritized command line entries.
     */
    protected readonly lines: PrioritizedLines[];
    /**
     * Add one or more commands to the user data with a priority of `0`.
     *
     * @param commands - The lines to add
     */
    addCommands(...commands: string[]): void;
    /**
     * Add one or more commands to the user data at a specific priority.
     *
     * @param priority - The priority of these lines (lower executes earlier)
     * @param commands - The lines to add
     */
    insertCommands(priority: number, ...commands: string[]): void;
    /**
     * Produces the User Data script with all lines sorted in priority order.
     *
     * @returns The assembled User Data
     */
    abstract buildUserData(): UserData;
}
