import { Duration, RemovalPolicy } from "aws-cdk-lib";
import { ICertificate } from "aws-cdk-lib/aws-certificatemanager";
import type { IRole } from "aws-cdk-lib/aws-iam";
import { IHostedZone, IPublicHostedZone } from "aws-cdk-lib/aws-route53";
import { Construct, IConstruct } from "constructs";
/**
 * A DNS domain and its wildcard X.509 certificate.
 */
export interface IDomain extends IConstruct {
    /**
     * The fully-qualified domain name of the hosted zone.
     */
    readonly name: string;
    /**
     * The hosted zone that contains records for this domain.
     */
    readonly hostedZone: IHostedZone;
    /**
     * The wildcard certificate for resources in this domain.
     */
    readonly certificate: ICertificate;
}
/**
 * A DNS domain and its wildcard X.509 certificate.
 */
export declare abstract class BaseDomain extends Construct implements IDomain {
    abstract readonly name: string;
    abstract readonly hostedZone: IHostedZone;
    abstract readonly certificate: ICertificate;
}
/**
 * Constructor properties for CrossAccountDelegationDomain.
 */
export interface CrossAccountDelegationDomainProps {
    /**
     * The subdomain in the parent hosted zone.
     */
    readonly subdomain: string;
    /**
     * The hosted zone name in the parent account.
     *
     * @default - no zone name
     */
    readonly parentHostedZoneName?: string;
    /**
     * The hosted zone id in the parent account.
     *
     * @default - hosted zone ID will be looked up based on the zone name
     */
    readonly parentHostedZoneId?: string;
    /**
     * The delegation role in the parent account.
     */
    readonly delegationRole: IRole;
    /**
     * The resource record cache time to live (TTL).
     *
     * @default Duration.days(2)
     */
    readonly ttl?: Duration;
    /**
     * The removal policy to apply.
     *
     * @default RemovalPolicy.DESTROY
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Region from which to obtain temporary credentials.
     *
     * @default - the Route53 signing region in the current partition
     */
    readonly assumeRoleRegion?: string;
}
/**
 * Provides a domain using delegation from a parent zone in another account.
 *
 * This construct creates a new Route 53 hosted zone for the subdomain, a zone
 * delegation record, and a new wildcard ACM certificate for the subdomain.
 */
export declare class CrossAccountDelegationDomain extends BaseDomain {
    readonly name: string;
    readonly hostedZone: IHostedZone;
    readonly certificate: ICertificate;
    private readonly delegationRecord;
    constructor(scope: Construct, id: string, props: CrossAccountDelegationDomainProps);
}
/**
 * Constructor properties for DelegationDomain.
 */
export interface DelegationDomainProps {
    /**
     * The parent/delegating hosted zone. The "zone name" is needed.
     */
    readonly parentHostedZone: IPublicHostedZone;
    /**
     * The subdomain in the parent hosted zone.
     */
    readonly subdomain: string;
    /**
     * The removal policy to apply.
     *
     * @default RemovalPolicy.DESTROY
     */
    readonly removalPolicy?: RemovalPolicy;
}
/**
 * Provides a domain using delegation from a parent zone in the same account.
 *
 * This construct creates a new Route 53 hosted zone for the subdomain, a zone
 * delegation record, and a new wildcard ACM certificate for the subdomain.
 */
export declare class DelegationDomain extends BaseDomain {
    readonly name: string;
    readonly hostedZone: IHostedZone;
    readonly certificate: ICertificate;
    private readonly delegationRecord;
    constructor(scope: Construct, id: string, props: DelegationDomainProps);
    applyRemovalPolicy(policy: RemovalPolicy): void;
}
/**
 * Constructor properties for ExistingZoneDomain.
 */
export interface ExistingZoneDomainProps {
    /**
     * The hosted zone that contains records for this domain.
     */
    readonly hostedZone: IHostedZone;
}
/**
 * A domain in the Domain Name System
 */
export interface DomainAttributes {
    /**
     * The hosted zone that contains records for this domain.
     */
    readonly hostedZone: IHostedZone;
    /**
     * The wildcard certificate for resources in this domain.
     */
    readonly certificate: ICertificate;
}
/**
 * Provides a domain using an existing hosted zone.
 *
 * This construct will create a new wildcard ACM certificate using the existing
 * hosted zone name.
 */
export declare class ExistingZoneDomain extends BaseDomain {
    /**
     * Returns an ExistingZoneDomain using the provided attributes.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param attrs - The provided attributes
     */
    static fromDomainAttributes(scope: Construct, id: string, attrs: DomainAttributes): IDomain;
    readonly hostedZone: IHostedZone;
    readonly certificate: ICertificate;
    constructor(scope: Construct, id: string, props: ExistingZoneDomainProps);
    get name(): string;
}
