import { Duration } from "aws-cdk-lib";
import { ICertificate } from "aws-cdk-lib/aws-certificatemanager";
import { ISecurityGroup, IVpc } from "aws-cdk-lib/aws-ec2";
import { ApplicationTargetGroupProps, IApplicationListener, IApplicationLoadBalancer, IApplicationTargetGroup, IApplicationLoadBalancerTarget, IpAddressType } from "aws-cdk-lib/aws-elasticloadbalancingv2";
import { Construct } from "constructs";
import { ISecretHttpHeader } from "./secret-http-header";
/**
 * Options for adding a new target group.
 */
export interface TargetOptions extends ApplicationTargetGroupProps {
    /**
     * The hostnames on which traffic is served.
     */
    readonly hostnames?: string[];
    /**
     * The priority of the listener rule.
     *
     * @default - Automatically determined
     */
    readonly priority?: number;
}
/**
 * Constructor properties for WebLoadBalancing.
 */
export interface WebLoadBalancingProps {
    /**
     * The VPC where these resources should be deployed.
     */
    readonly vpc: IVpc;
    /**
     * A security group for the load balancer itself.
     *
     * @default - A new security group will be created
     */
    readonly securityGroup?: ISecurityGroup;
    /**
     * The certificate to attach to the load balancer and CloudFront distribution.
     */
    readonly certificates: ICertificate[];
    /**
     * The load balancer idle timeout, in seconds.
     *
     * If you have a reverse proxy in front of this load balancer, such as
     * CloudFront, this number should be less than the reverse proxy's request
     * timeout.
     *
     * @default - 59 seconds
     */
    readonly idleTimeout?: Duration;
    /**
     * The type of IP addresses to use (IPv4 or Dual Stack).
     *
     * @default - IPv4 only
     */
    readonly ipAddressType?: IpAddressType;
    /**
     * Forbid requests that are missing an HTTP header with a specific value.
     *
     * If this option is set to `true`, this construct will provide a new
     * `SecretHttpHeader` accessible on the `secretHeader` property.
     *
     * Requests without the correct header name and value will receive an HTTP 421
     * status response.
     *
     * @default - false
     */
    readonly requireSecretHeader?: boolean;
    /**
     * The name of the secret HTTP header.
     *
     * Providing this option implies that `requireSecretHeader` is `true`.
     *
     * @default - X-Secret-Passphrase
     */
    readonly secretHeaderName?: string;
    /**
     * Forbid requests that ask for an unknown hostname.
     *
     * Requests for an unknown hostname will receive an HTTP 421 status response.
     *
     * @default - false
     */
    readonly requireKnownHostname?: boolean;
}
/**
 * A utility for creating a public-facing Application Load Balancer.
 */
export declare class WebLoadBalancing extends Construct {
    /**
     * The load balancer itself.
     */
    readonly loadBalancer: IApplicationLoadBalancer;
    /**
     * The HTTPS listener.
     */
    readonly listener: IApplicationListener;
    /**
     * The secret header (if `requireSecretHeader` was set to `true`).
     */
    readonly secretHeader?: ISecretHttpHeader;
    private readonly enableSecretHeader;
    private readonly enableKnownHostnames;
    private readonly idleTimeout;
    private readonly priorities;
    private defaultAction?;
    /**
     * Creates a new WebLoadBalancing.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope: Construct, id: string, props: WebLoadBalancingProps);
    /**
     * Adds a target to the listener.
     *
     * If the following options are left undefined, these defaults will be used.
     * - `port`: 443
     * - `protocol`: HTTPS
     * - `deregistrationDelay`: load balancer idle timeout
     * - `healthCheck.path`: /
     * - `healthCheck.healthyThresholdCount`: 2
     * - `healthCheck.interval`: 30 seconds
     * - `healthCheck.timeout`: 29 seconds
     *
     * @param id - The ID of the new target group.
     * @param target - The load balancing target to receive traffic.
     * @param options - The target group options.
     * @returns The new Application Target Group
     */
    addTarget(id: string, target: IApplicationLoadBalancerTarget, options?: TargetOptions): IApplicationTargetGroup;
}
