import { IAspect } from "aws-cdk-lib";
import type { IConstruct } from "constructs";
/**
 * A deployment environment with a specific purpose and audience.
 *
 * You can create any Tier you like, but we include those explained by DTAP.
 *
 * @see https://en.wikipedia.org/wiki/Development,_testing,_acceptance_and_production
 */
export declare class Tier {
    /**
     * A tier that represents a production environment.
     */
    static readonly PRODUCTION: Tier;
    /**
     * A tier that represents an acceptance environment.
     */
    static readonly ACCEPTANCE: Tier;
    /**
     * A tier that represents a testing environment.
     */
    static readonly TESTING: Tier;
    /**
     * A tier that represents a development environment.
     */
    static readonly DEVELOPMENT: Tier;
    /**
     * Finds the deployment tier of the given construct.
     *
     * @param construct - The construct to inspect
     * @returns The assigned deployment tier if found, otherwise undefined
     */
    static of(construct: IConstruct): Tier | undefined;
    /**
     * Return the deployment tier that corresponds to the provided value.
     *
     * Production: "live", "prod", or "production".
     * Acceptance: "uat", "stage", "staging", or "acceptance".
     * Testing: "qc", "qa", "test", or "testing".
     * Development: anything else.
     *
     * @param value - The value to parse, case-insensitive
     * @returns The matching deployment tier, or else `DEVELOPMENT`.
     */
    static parse(value: string): Tier;
    /**
     * The machine-readable identifier for this tier (e.g. prod)
     */
    readonly id: string;
    /**
     * The human-readable label for this tier (e.g. Production)
     */
    readonly label: string;
    /**
     * Creates a new Tier.
     *
     * @param id - The machine-readable identifier for this tier (e.g. prod)
     * @param label - The human-readable label for this tier (e.g. Production)
     */
    constructor(id: string, label: string);
    /**
     * Compares this tier to the provided value and tests for equality.
     *
     * @param other - The value to compare.
     * @returns Whether the provided value is equal to this tier.
     */
    matches(other: Tier): boolean;
    /**
     * Assigns this tier to a construct.
     *
     * This method will register an error annotation on the construct if any of
     * the constructs in its parent scopes have a different tier assigned.
     *
     * @param construct - The construct to receive the tier assignment
     */
    assignTo(construct: IConstruct): void;
    /**
     * Adds the label of this tier as a tag to the provided construct.
     */
    applyTags(construct: IConstruct): void;
}
/**
 * A CDK Aspect to apply the `DeploymentTier` tag to Stacks.
 */
export declare class TierTagger implements IAspect {
    private readonly tier;
    /**
     * Create a new TierTagger.
     *
     * @param tier - The deployment tier
     */
    constructor(tier: Tier);
    visit(node: IConstruct): void;
}
