"""
This module have classes responsible for registering the signals values and
controlling time in them simulation.
"""
from datetime import datetime

#TODO use api
from .backend.python.core.component import Component


VERSION = '0.3.1'
CODENAME = 'Gambiarra'
VALID_UNITS = ['fs', 'ps', 'ns', 'us', 'ms', 's']


class Signal:
    """
    This class represents a signal in the simulation.
    """
    def __init__(self, id: str, value: str) -> None:
        self.id = id
        self.value = value

    def __repr__(self) -> str:
        return f'Signal {self.id}: {self.value}'


class WaveSample:
    """
    This class represents a sample of the waves generated by the circuit.
    """
    def __init__(self, time, signals):
        self.time = time

        # signals is a tuple of signal name and value
        self.signals: list[Signal] = signals


class TestBench:
    def __init__(self, component: Component) -> None:
        self.s_time = 0
        self.time_unit: str = 'ns'
        self.samples: list[WaveSample] = []
        self.component: Component = component

    def __str__(self) -> str:
        return self.component.__str__()

    def wait(self, time: int) -> None:
        """This method waits for a certain time."""
        self.s_time += time

    def set_time_unit(self, time_unit: str) -> None:
        if time_unit not in VALID_UNITS:
            raise ValueError(
                f'Invalid time unit "{time_unit}". Valid units are: '
                f'{VALID_UNITS}'
            )

        else:
            self.time_unit = time_unit

    def dump_vcd(self):
        header_metadata = '' \
            f'$version Generated by Flote v{VERSION} - {CODENAME} $end\n' \
            f'$date {datetime.now().strftime(r"%Y-%m-%d %H:%M:%S")} $end\n' \
            f'$timescale 1{self.time_unit} $end\n'

        header_metadata += '\n$comment Hello from Theresina. $end\n'

        header_declaration = f'\n$scope module {self.component.id_} $end\n'

        for bit_name, bit_bus in self.component.busses.items():
            header_declaration += (
                f'\t$var wire {len(bit_bus.value.raw_value)} {bit_name} {bit_name} $end\n'
            )

        header_declaration += '$upscope $end\n\n'

        header = (
            header_metadata +
            header_declaration +
            '$enddefinitions $end\n'
        )

        datasec = ''

        for sample in self.samples:
            datasec += f"\n#{sample.time}\n\n"

            for signal in sample.signals:
                bits = f'b{signal.value}'
                datasec += f"{bits} {signal.id}\n"

        return header + datasec + f'\n#{self.s_time}\n'

    def save_vcd(self, file_path: str) -> None:
        """This method saves the vcd file."""
        with open(file_path, 'w') as f:
            f.write(self.dump_vcd())
            f.close()

    def update(self, new_values: dict[str, str]) -> None:
        self.component.update_signals(new_values)

        sample = WaveSample(self.s_time, [])

        for id, bit in self.component.busses.items():
            sample.signals.append(Signal(id, bit.value.get_vcd_repr()))

        self.samples.append(sample)
