## A data cleaning and quality solution for messy data, bug prevention, and leakage control.

Ideal for structured data storage where full database integration is overkill. Uses simple 1D/2D list logic with built-in validation, file management, backup and snapshot support. Dependency-free, designed by a Data Analyst/Engineer.

---

## 🚀 Features

- ✅ Safe read/write with structure validation (1D & 2D lists).
- ➕ Append support with consistency checks.
- ❌ Granular delete with filters (cutoff, keep, reverse, size).
- 💾 Backup and 📸 Snapshot support.
- 🧹 Debug and auto-clean corrupted files.
- 📚 Built-in mini-guide.
- 🎗️ Supports Multiple Console usage.
- 🗂️ Clean folder/files organization.
- 💪 Strong anti-corruption and Tamper mechanism.

---

## Why Use This Package?

**For Data Professionals (Analysts, Scientists, Engineers):**

* Build quick dashboards and visualizations from real-time text data.
* Process and clean scraped or pipeline data without needing a database.
* Track, audit, and version small datasets during experimentation.
* Validate incoming data and auto-correct format issues on the fly (clean from the source).
* Run fast, repeatable data operations locally without SQL overhead.

**For General Developers:**

* Great for automation scripts, bots, or microservices.
* Ideal for prototyping features without setting up a database.
* Use as a flat-file backend for small apps or CLI tools.
* Snapshot and backup data safely without extra libraries.
* Handle concurrent access and reduce race-condition bugs automatically.

---

## 📦 Core Functions

### `w(txt_name: str, write_list: list) -> None`

Write (or reset) the contents of a file, validating structure before saving to all backup locations.

### `r(txt_name: str, set_new: list | None = [], notify_new: bool = True) -> list | None`

Read file contents. If the file is missing, return `set_new` and optionally notify user of new file creation.

### `a(txt_name: str, append_list: list) -> list`

Append new rows to an existing file after validating structure. Returns the updated list.

### `d(txt_name: str, ...) -> tuple[int, list]`

Delete matching rows with flexible options:
- `del_list`: values to delete
- `index`: column index for 2D deletion
- `cutoff`: max deletions per value
- `keep`: retain only N per value
- `reverse`: delete from end
- `size`: trim to max N items

### `backup(txt_name: str, display=True)`

Create a manual backup of a file or all (`txt_name="*"`).

### `snapshot(txt_name: str, unit, gap, begin=0, display=True)`

Take time-based snapshots if eligible. Supports:
- `unit`: `'minute'`, `'hour'`, `'day'`, `'month'`, etc.
- `gap`: how much time must pass
- `begin`: used for daily-based triggers

### `debug(txt_name, is_2D=None, clean=None, length=None, display=True)`

Scan and optionally auto-clean a file that fails validation. Great for corrupted data recovery.

### `help()`

Opens the interactive mini-guide documentation tool.

---

## 📁 File Organization

Each file is saved as a `.txt` in a structured folder. All backups and snapshots are handled automatically.

Other folders:
- `Backup 💾/` – Manual backups
- `Snapshot 📸/` – A timed backup

Validation files:
- `*_validation.txt` – Schema registry per file

---

## 📦 Installation – MaxCleanerDB

pip install MaxCleanerDB

## ♻️ Upgrade

pip install --upgrade MaxCleanerDB

---

**Usage Examples**

**import MaxCleanerDB or import MaxCleanerDB as mdb 

**Write (w): Overwrite data**

* Stores a simple (1D) list of student names:  
  `w("students", ["Alice", "Bob"])`

* Stores a 2D list (like a table of people):  
  Think of it like this:<br>
  Headers = [name, sex, age]<br>
  Rows = ["Eve", "female", 25], ["Adam", "male", 30]<br>
  `w("people", [["Eve", "female", 25], ["Adam", "male", 30]])`

---

**Read (r): Read data or set default if missing**

* `r("students")`
  Reads the data from the "students" file.

* `r("new_file", [], notify_new=True)`
  If "new\_file" doesn’t exist, it creates one with an empty list and optionally shows a notification.

---

**Append (a): Add new entries**

* `a("students", ["Charlie"])`
  Adds "Charlie" to the list of students.

* `a("people", [["Lucy", "female", 22], ["Mark", "male", 27]])`
  Adds new rows to the "people" table.

---

**Delete (d): Remove entries with different options**

**For simple lists (like names):**

* `d("students", ["Bob"])`
  Deletes "Bob" from the student list.

* `d("students", ["Bob", "Charlie"])`
  Deletes both "Bob" and "Charlie" if found.

**For tables (like people list with name, sex, age):**

* `d("people", ["Eve"], index=0)`
  Deletes rows where name is "Eve" (index 0 means the name column).

* `d("people", ["female"], index=1, cutoff=1)`
  Deletes only the first row where sex is "female".

* `d("people", ["female"], index=1, keep=1)`
  Keeps only one row with sex as "female", deletes any others.

* `d("people", size=2)`
  Keeps only the last 2 entries in the list.

* `d("people", ["Lucy"], index=0, reverse=True)`
  Deletes rows with name "Lucy", starting from the bottom.

**Using multiple column conditions (like WHERE in SQL):**

* `d("people", [["Adam", "male"]], index=[0, 1])`
  Deletes the row where name is "Adam" AND sex is "male".

* `d("people", [["Eve", "female", 25]], index="*")`
  Deletes the exact row \["Eve", "female", 25].

* `d("people", [["Adam", 30], ["Mark", 27]], index=[0, 2])`
  Deletes rows where (name is "Adam" AND age is 30) OR (name is "Mark" AND age is 27)

---

**Backup and Snapshot**

* `backup("students")`
  Makes a backup copy of the "students" file.

* `backup("*")`
  Backs up all files.

* `snapshot("students", "day", 1)`
  Takes a snapshot once a day if the last one was over a day ago.

* `snapshot("*", "hour", 6)`
  Snapshots all files every 6 hours if enough time has passed.

---

**Debugging**

* `debug("students", is_2D=False, clean=True)`
  Checks and cleans up simple list data in "students".

* `debug("people", is_2D=True, length=3)`
  Checks if each row in "people" has 3 columns (name, sex, age).

---

**Help**

* `help()`
  Shows this usage guide.

