/**
  ******************************************************************************
  * @file    stm32l4xx_hal.c
  * @author  MCD Application Team
  * @brief   HAL module driver.
  *          This is the common part of the HAL initialization
  *
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2017 STMicroelectronics.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  @verbatim
  ==============================================================================
                     ##### How to use this driver #####
  ==============================================================================
    [..]
    The common HAL driver contains a set of generic and common APIs that can be
    used by the PPP peripheral drivers and the user to start using the HAL.
    [..]
    The HAL contains two APIs' categories:
         (+) Common HAL APIs
         (+) Services HAL APIs

  @endverbatim
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "stm32l4xx_hal.h"

/** @addtogroup STM32L4xx_HAL_Driver
 * @{
 */

/** @defgroup HAL HAL
 * @brief HAL module driver
 * @{
 */

#ifdef HAL_MODULE_ENABLED

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/**
 * @brief STM32L4xx HAL Driver version number
 */
#define STM32L4XX_HAL_VERSION_MAIN (0x01U) /*!< [31:24] main version */
#define STM32L4XX_HAL_VERSION_SUB1 (0x0DU) /*!< [23:16] sub1 version */
#define STM32L4XX_HAL_VERSION_SUB2 (0x05U) /*!< [15:8]  sub2 version */
#define STM32L4XX_HAL_VERSION_RC (0x00U)   /*!< [7:0]  release candidate */
#define STM32L4XX_HAL_VERSION ((STM32L4XX_HAL_VERSION_MAIN << 24U) | (STM32L4XX_HAL_VERSION_SUB1 << 16U) | (STM32L4XX_HAL_VERSION_SUB2 << 8U) | (STM32L4XX_HAL_VERSION_RC))

#if defined(VREFBUF)
#define VREFBUF_TIMEOUT_VALUE 10U /* 10 ms (to be confirmed) */
#endif                            /* VREFBUF */

/* ------------ SYSCFG registers bit address in the alias region ------------ */
#define SYSCFG_OFFSET (SYSCFG_BASE - PERIPH_BASE)
/* ---  MEMRMP Register ---*/
/* Alias word address of FB_MODE bit */
#define MEMRMP_OFFSET SYSCFG_OFFSET
#define FB_MODE_BitNumber 8U
#define FB_MODE_BB (PERIPH_BB_BASE + (MEMRMP_OFFSET * 32U) + (FB_MODE_BitNumber * 4U))

/* --- SCSR Register ---*/
/* Alias word address of SRAM2ER bit */
#define SCSR_OFFSET (SYSCFG_OFFSET + 0x18U)
#define BRER_BitNumber 0U
#define SCSR_SRAM2ER_BB (PERIPH_BB_BASE + (SCSR_OFFSET * 32U) + (BRER_BitNumber * 4U))

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/

/* Exported variables --------------------------------------------------------*/

/** @defgroup HAL_Exported_Variables HAL Exported Variables
 * @{
 */
__IO uint32_t uwTick;
uint32_t uwTickPrio = (1UL << __NVIC_PRIO_BITS);        /* Invalid priority */
HAL_TickFreqTypeDef uwTickFreq = HAL_TICK_FREQ_DEFAULT; /* 1KHz */
/**
 * @}
 */

/* Exported functions --------------------------------------------------------*/

/** @defgroup HAL_Exported_Functions HAL Exported Functions
 * @{
 */

/** @defgroup HAL_Exported_Functions_Group1 Initialization and de-initialization Functions
 *  @brief    Initialization and de-initialization functions
 *
@verbatim
 ===============================================================================
              ##### Initialization and de-initialization functions #####
 ===============================================================================
    [..]  This section provides functions allowing to:
      (+) Initialize the Flash interface, the NVIC allocation and initial time base
          clock configuration.
      (+) De-initialize common part of the HAL.
      (+) Configure the time base source to have 1ms time base with a dedicated
          Tick interrupt priority.
        (++) SysTick timer is used by default as source of time base, but user
             can eventually implement his proper time base source (a general purpose
             timer for example or other time source), keeping in mind that Time base
             duration should be kept 1ms since PPP_TIMEOUT_VALUEs are defined and
             handled in milliseconds basis.
        (++) Time base configuration function (HAL_InitTick ()) is called automatically
             at the beginning of the program after reset by HAL_Init() or at any time
             when clock is configured, by HAL_RCC_ClockConfig().
        (++) Source of time base is configured  to generate interrupts at regular
             time intervals. Care must be taken if HAL_Delay() is called from a
             peripheral ISR process, the Tick interrupt line must have higher priority
            (numerically lower) than the peripheral interrupt. Otherwise the caller
            ISR process will be blocked.
       (++) functions affecting time base configurations are declared as __weak
             to make  override possible  in case of other  implementations in user file.
@endverbatim
  * @{
  */

/**
 * @brief  Configure the Flash prefetch, the Instruction and Data caches,
 *         the time base source, NVIC and any required global low level hardware
 *         by calling the HAL_MspInit() callback function to be optionally defined in user file
 *         stm32l4xx_hal_msp.c.
 *
 * @note   HAL_Init() function is called at the beginning of program after reset and before
 *         the clock configuration.
 *
 * @note   In the default implementation the System Timer (Systick) is used as source of time base.
 *         The Systick configuration is based on MSI clock, as MSI is the clock
 *         used after a system Reset and the NVIC configuration is set to Priority group 4.
 *         Once done, time base tick starts incrementing: the tick variable counter is incremented
 *         each 1ms in the SysTick_Handler() interrupt handler.
 *
 * @retval HAL status
 */
HAL_StatusTypeDef HAL_Init(void)
{
  HAL_StatusTypeDef status = HAL_OK;

  /* Configure Flash prefetch, Instruction cache, Data cache */
  /* Default configuration at reset is:                      */
  /* - Prefetch disabled                                     */
  /* - Instruction cache enabled                             */
  /* - Data cache enabled                                    */
#if (INSTRUCTION_CACHE_ENABLE == 0)
  __HAL_FLASH_INSTRUCTION_CACHE_DISABLE();
#endif /* INSTRUCTION_CACHE_ENABLE */

#if (DATA_CACHE_ENABLE == 0)
  __HAL_FLASH_DATA_CACHE_DISABLE();
#endif /* DATA_CACHE_ENABLE */

#if (PREFETCH_ENABLE != 0)
  __HAL_FLASH_PREFETCH_BUFFER_ENABLE();
#endif /* PREFETCH_ENABLE */

  /* Set Interrupt Group Priority */
  HAL_NVIC_SetPriorityGrouping(NVIC_PRIORITYGROUP_4);

  /* Use SysTick as time base source and configure 1ms tick (default clock after Reset is MSI) */
  if (HAL_InitTick(TICK_INT_PRIORITY) != HAL_OK)
  {
    status = HAL_ERROR;
  }
  else
  {
    /* Init the low level hardware */
    HAL_MspInit();
  }

  /* Return function status */
  return status;
}

/**
 * @brief De-initialize common part of the HAL and stop the source of time base.
 * @note This function is optional.
 * @retval HAL status
 */
HAL_StatusTypeDef HAL_DeInit(void)
{
  /* Reset of all peripherals */
  __HAL_RCC_APB1_FORCE_RESET();
  __HAL_RCC_APB1_RELEASE_RESET();

  __HAL_RCC_APB2_FORCE_RESET();
  __HAL_RCC_APB2_RELEASE_RESET();

  __HAL_RCC_AHB1_FORCE_RESET();
  __HAL_RCC_AHB1_RELEASE_RESET();

  __HAL_RCC_AHB2_FORCE_RESET();
  __HAL_RCC_AHB2_RELEASE_RESET();

  __HAL_RCC_AHB3_FORCE_RESET();
  __HAL_RCC_AHB3_RELEASE_RESET();

  /* De-Init the low level hardware */
  HAL_MspDeInit();

  /* Return function status */
  return HAL_OK;
}

/**
 * @brief  Initialize the MSP.
 * @retval None
 */
__weak void HAL_MspInit(void)
{
  /* NOTE : This function should not be modified, when the callback is needed,
            the HAL_MspInit could be implemented in the user file
   */
}

/**
 * @brief  DeInitialize the MSP.
 * @retval None
 */
__weak void HAL_MspDeInit(void)
{
  /* NOTE : This function should not be modified, when the callback is needed,
            the HAL_MspDeInit could be implemented in the user file
   */
}

/**
 * @brief This function configures the source of the time base:
 *        The time source is configured to have 1ms time base with a dedicated
 *        Tick interrupt priority.
 * @note This function is called  automatically at the beginning of program after
 *       reset by HAL_Init() or at any time when clock is reconfigured  by HAL_RCC_ClockConfig().
 * @note In the default implementation, SysTick timer is the source of time base.
 *       It is used to generate interrupts at regular time intervals.
 *       Care must be taken if HAL_Delay() is called from a peripheral ISR process,
 *       The SysTick interrupt must have higher priority (numerically lower)
 *       than the peripheral interrupt. Otherwise the caller ISR process will be blocked.
 *       The function is declared as __weak  to be overwritten  in case of other
 *       implementation  in user file.
 * @param TickPriority  Tick interrupt priority.
 * @retval HAL status
 */
__weak HAL_StatusTypeDef HAL_InitTick(uint32_t TickPriority)
{
  HAL_StatusTypeDef status = HAL_OK;

  /* Check uwTickFreq for MisraC 2012 (even if uwTickFreq is a enum type that doesn't take the value zero)*/
  if ((uint32_t)uwTickFreq != 0U)
  {
    /*Configure the SysTick to have interrupt in 1ms time basis*/
    if (HAL_SYSTICK_Config(SystemCoreClock / (1000U / (uint32_t)uwTickFreq)) == 0U)
    {
      /* Configure the SysTick IRQ priority */
      if (TickPriority < (1UL << __NVIC_PRIO_BITS))
      {
        HAL_NVIC_SetPriority(SysTick_IRQn, TickPriority, 0U);
        uwTickPrio = TickPriority;
      }
      else
      {
        status = HAL_ERROR;
      }
    }
    else
    {
      status = HAL_ERROR;
    }
  }
  else
  {
    status = HAL_ERROR;
  }

  /* Return function status */
  return status;
}

/**
 * @}
 */

/** @defgroup HAL_Exported_Functions_Group2 HAL Control functions
 *  @brief    HAL Control functions
 *
@verbatim
 ===============================================================================
                      ##### HAL Control functions #####
 ===============================================================================
    [..]  This section provides functions allowing to:
      (+) Provide a tick value in millisecond
      (+) Provide a blocking delay in millisecond
      (+) Suspend the time base source interrupt
      (+) Resume the time base source interrupt
      (+) Get the HAL API driver version
      (+) Get the device identifier
      (+) Get the device revision identifier

@endverbatim
  * @{
  */

/**
 * @brief This function is called to increment a global variable "uwTick"
 *        used as application time base.
 * @note In the default implementation, this variable is incremented each 1ms
 *       in SysTick ISR.
 * @note This function is declared as __weak to be overwritten in case of other
 *      implementations in user file.
 * @retval None
 */
__weak void HAL_IncTick(void)
{
  uwTick += (uint32_t)uwTickFreq;
}

/**
 * @brief Provide a tick value in millisecond.
 * @note This function is declared as __weak to be overwritten in case of other
 *       implementations in user file.
 * @retval tick value
 */
__weak uint32_t HAL_GetTick(void)
{
  return uwTick;
}

/**
 * @brief This function returns a tick priority.
 * @retval tick priority
 */
uint32_t HAL_GetTickPrio(void)
{
  return uwTickPrio;
}

/**
 * @brief Set new tick Freq.
 * @param Freq tick frequency
 * @retval HAL status
 */
HAL_StatusTypeDef HAL_SetTickFreq(HAL_TickFreqTypeDef Freq)
{
  HAL_StatusTypeDef status = HAL_OK;
  HAL_TickFreqTypeDef prevTickFreq;

  if (uwTickFreq != Freq)
  {
    /* Back up uwTickFreq frequency */
    prevTickFreq = uwTickFreq;

    /* Update uwTickFreq global variable used by HAL_InitTick() */
    uwTickFreq = Freq;

    /* Apply the new tick Freq  */
    status = HAL_InitTick(uwTickPrio);
    if (status != HAL_OK)
    {
      /* Restore previous tick frequency */
      uwTickFreq = prevTickFreq;
    }
  }

  return status;
}

/**
 * @brief Return tick frequency.
 * @retval Tick frequency.
 *         Value of @ref HAL_TickFreqTypeDef.
 */
HAL_TickFreqTypeDef HAL_GetTickFreq(void)
{
  return uwTickFreq;
}

/**
 * @brief This function provides minimum delay (in milliseconds) based
 *        on variable incremented.
 * @note In the default implementation , SysTick timer is the source of time base.
 *       It is used to generate interrupts at regular time intervals where uwTick
 *       is incremented.
 * @note This function is declared as __weak to be overwritten in case of other
 *       implementations in user file.
 * @param Delay  specifies the delay time length, in milliseconds.
 * @retval None
 */
__weak void HAL_Delay(uint32_t Delay)
{
  uint32_t tickstart = HAL_GetTick();
  uint32_t wait = Delay;

  /* Add a period to guaranty minimum wait */
  if (wait < HAL_MAX_DELAY)
  {
    wait += (uint32_t)uwTickFreq;
  }

  while ((HAL_GetTick() - tickstart) < wait)
  {
  }
}

/**
 * @brief Suspend Tick increment.
 * @note In the default implementation , SysTick timer is the source of time base. It is
 *       used to generate interrupts at regular time intervals. Once HAL_SuspendTick()
 *       is called, the SysTick interrupt will be disabled and so Tick increment
 *       is suspended.
 * @note This function is declared as __weak to be overwritten in case of other
 *       implementations in user file.
 * @retval None
 */
__weak void HAL_SuspendTick(void)
{
  /* Disable SysTick Interrupt */
  SysTick->CTRL &= ~SysTick_CTRL_TICKINT_Msk;
}

/**
 * @brief Resume Tick increment.
 * @note In the default implementation , SysTick timer is the source of time base. It is
 *       used to generate interrupts at regular time intervals. Once HAL_ResumeTick()
 *       is called, the SysTick interrupt will be enabled and so Tick increment
 *       is resumed.
 * @note This function is declared as __weak to be overwritten in case of other
 *       implementations in user file.
 * @retval None
 */
__weak void HAL_ResumeTick(void)
{
  /* Enable SysTick Interrupt */
  SysTick->CTRL |= SysTick_CTRL_TICKINT_Msk;
}

/**
 * @brief  Return the HAL revision.
 * @retval version : 0xXYZR (8bits for each decimal, R for RC)
 */
uint32_t HAL_GetHalVersion(void)
{
  return STM32L4XX_HAL_VERSION;
}

/**
 * @brief  Return the device revision identifier.
 * @retval Device revision identifier
 */
uint32_t HAL_GetREVID(void)
{
  return ((DBGMCU->IDCODE & DBGMCU_IDCODE_REV_ID) >> 16);
}

/**
 * @brief  Return the device identifier.
 * @retval Device identifier
 */
uint32_t HAL_GetDEVID(void)
{
  return (DBGMCU->IDCODE & DBGMCU_IDCODE_DEV_ID);
}

/**
 * @brief  Return the first word of the unique device identifier (UID based on 96 bits)
 * @retval Device identifier
 */
uint32_t HAL_GetUIDw0(void)
{
  return (READ_REG(*((uint32_t *)UID_BASE)));
}

/**
 * @brief  Return the second word of the unique device identifier (UID based on 96 bits)
 * @retval Device identifier
 */
uint32_t HAL_GetUIDw1(void)
{
  return (READ_REG(*((uint32_t *)(UID_BASE + 4U))));
}

/**
 * @brief  Return the third word of the unique device identifier (UID based on 96 bits)
 * @retval Device identifier
 */
uint32_t HAL_GetUIDw2(void)
{
  return (READ_REG(*((uint32_t *)(UID_BASE + 8U))));
}

/**
 * @}
 */

/** @defgroup HAL_Exported_Functions_Group3 HAL Debug functions
 *  @brief    HAL Debug functions
 *
@verbatim
 ===============================================================================
                      ##### HAL Debug functions #####
 ===============================================================================
    [..]  This section provides functions allowing to:
      (+) Enable/Disable Debug module during SLEEP mode
      (+) Enable/Disable Debug module during STOP0/STOP1/STOP2 modes
      (+) Enable/Disable Debug module during STANDBY mode

@endverbatim
  * @{
  */

/**
 * @brief  Enable the Debug Module during SLEEP mode.
 * @retval None
 */
void HAL_DBGMCU_EnableDBGSleepMode(void)
{
  SET_BIT(DBGMCU->CR, DBGMCU_CR_DBG_SLEEP);
}

/**
 * @brief  Disable the Debug Module during SLEEP mode.
 * @retval None
 */
void HAL_DBGMCU_DisableDBGSleepMode(void)
{
  CLEAR_BIT(DBGMCU->CR, DBGMCU_CR_DBG_SLEEP);
}

/**
 * @brief  Enable the Debug Module during STOP0/STOP1/STOP2 modes.
 * @retval None
 */
void HAL_DBGMCU_EnableDBGStopMode(void)
{
  SET_BIT(DBGMCU->CR, DBGMCU_CR_DBG_STOP);
}

/**
 * @brief  Disable the Debug Module during STOP0/STOP1/STOP2 modes.
 * @retval None
 */
void HAL_DBGMCU_DisableDBGStopMode(void)
{
  CLEAR_BIT(DBGMCU->CR, DBGMCU_CR_DBG_STOP);
}

/**
 * @brief  Enable the Debug Module during STANDBY mode.
 * @retval None
 */
void HAL_DBGMCU_EnableDBGStandbyMode(void)
{
  SET_BIT(DBGMCU->CR, DBGMCU_CR_DBG_STANDBY);
}

/**
 * @brief  Disable the Debug Module during STANDBY mode.
 * @retval None
 */
void HAL_DBGMCU_DisableDBGStandbyMode(void)
{
  CLEAR_BIT(DBGMCU->CR, DBGMCU_CR_DBG_STANDBY);
}

/**
 * @}
 */

/** @defgroup HAL_Exported_Functions_Group4 HAL SYSCFG configuration functions
 *  @brief    HAL SYSCFG configuration functions
 *
@verbatim
 ===============================================================================
                      ##### HAL SYSCFG configuration functions #####
 ===============================================================================
    [..]  This section provides functions allowing to:
      (+) Start a hardware SRAM2 erase operation
      (+) Enable/Disable the Internal FLASH Bank Swapping
      (+) Configure the Voltage reference buffer
      (+) Enable/Disable the Voltage reference buffer
      (+) Enable/Disable the I/O analog switch voltage booster

@endverbatim
  * @{
  */

/**
 * @brief  Start a hardware SRAM2 erase operation.
 * @note   As long as SRAM2 is not erased the SRAM2ER bit will be set.
 *         This bit is automatically reset at the end of the SRAM2 erase operation.
 * @retval None
 */
void HAL_SYSCFG_SRAM2Erase(void)
{
  /* unlock the write protection of the SRAM2ER bit */
  SYSCFG->SKR = 0xCA;
  SYSCFG->SKR = 0x53;
  /* Starts a hardware SRAM2 erase operation*/
  *(__IO uint32_t *)SCSR_SRAM2ER_BB = 0x00000001UL;
}

/**
 * @brief  Enable the Internal FLASH Bank Swapping.
 *
 * @note   This function can be used only for STM32L4xx devices.
 *
 * @note   Flash Bank2 mapped at 0x08000000 (and aliased @0x00000000)
 *         and Flash Bank1 mapped at 0x08100000 (and aliased at 0x00100000)
 *
 * @retval None
 */
void HAL_SYSCFG_EnableMemorySwappingBank(void)
{
  *(__IO uint32_t *)FB_MODE_BB = 0x00000001UL;
}

/**
 * @brief  Disable the Internal FLASH Bank Swapping.
 *
 * @note   This function can be used only for STM32L4xx devices.
 *
 * @note   The default state : Flash Bank1 mapped at 0x08000000 (and aliased @0x0000 0000)
 *         and Flash Bank2 mapped at 0x08100000 (and aliased at 0x00100000)
 *
 * @retval None
 */
void HAL_SYSCFG_DisableMemorySwappingBank(void)
{

  *(__IO uint32_t *)FB_MODE_BB = 0x00000000UL;
}

#if defined(VREFBUF)
/**
 * @brief Configure the internal voltage reference buffer voltage scale.
 * @param VoltageScaling  specifies the output voltage to achieve
 *          This parameter can be one of the following values:
 *            @arg SYSCFG_VREFBUF_VOLTAGE_SCALE0: VREF_OUT1 around 2.048 V.
 *                                                This requires VDDA equal to or higher than 2.4 V.
 *            @arg SYSCFG_VREFBUF_VOLTAGE_SCALE1: VREF_OUT2 around 2.5 V.
 *                                                This requires VDDA equal to or higher than 2.8 V.
 * @retval None
 */
void HAL_SYSCFG_VREFBUF_VoltageScalingConfig(uint32_t VoltageScaling)
{
  /* Check the parameters */
  assert_param(IS_SYSCFG_VREFBUF_VOLTAGE_SCALE(VoltageScaling));

  MODIFY_REG(VREFBUF->CSR, VREFBUF_CSR_VRS, VoltageScaling);
}

/**
 * @brief Configure the internal voltage reference buffer high impedance mode.
 * @param Mode  specifies the high impedance mode
 *          This parameter can be one of the following values:
 *            @arg SYSCFG_VREFBUF_HIGH_IMPEDANCE_DISABLE: VREF+ pin is internally connect to VREFINT output.
 *            @arg SYSCFG_VREFBUF_HIGH_IMPEDANCE_ENABLE: VREF+ pin is high impedance.
 * @retval None
 */
void HAL_SYSCFG_VREFBUF_HighImpedanceConfig(uint32_t Mode)
{
  /* Check the parameters */
  assert_param(IS_SYSCFG_VREFBUF_HIGH_IMPEDANCE(Mode));

  MODIFY_REG(VREFBUF->CSR, VREFBUF_CSR_HIZ, Mode);
}

/**
 * @brief  Tune the Internal Voltage Reference buffer (VREFBUF).
 * @retval None
 */
void HAL_SYSCFG_VREFBUF_TrimmingConfig(uint32_t TrimmingValue)
{
  /* Check the parameters */
  assert_param(IS_SYSCFG_VREFBUF_TRIMMING(TrimmingValue));

  MODIFY_REG(VREFBUF->CCR, VREFBUF_CCR_TRIM, TrimmingValue);
}

/**
 * @brief  Enable the Internal Voltage Reference buffer (VREFBUF).
 * @retval HAL_OK/HAL_TIMEOUT
 */
HAL_StatusTypeDef HAL_SYSCFG_EnableVREFBUF(void)
{
  uint32_t tickstart;

  SET_BIT(VREFBUF->CSR, VREFBUF_CSR_ENVR);

  /* Get Start Tick*/
  tickstart = HAL_GetTick();

  /* Wait for VRR bit  */
  while (READ_BIT(VREFBUF->CSR, VREFBUF_CSR_VRR) == 0U)
  {
    if ((HAL_GetTick() - tickstart) > VREFBUF_TIMEOUT_VALUE)
    {
      return HAL_TIMEOUT;
    }
  }

  return HAL_OK;
}

/**
 * @brief  Disable the Internal Voltage Reference buffer (VREFBUF).
 *
 * @retval None
 */
void HAL_SYSCFG_DisableVREFBUF(void)
{
  CLEAR_BIT(VREFBUF->CSR, VREFBUF_CSR_ENVR);
}
#endif /* VREFBUF */

/**
 * @brief  Enable the I/O analog switch voltage booster
 *
 * @retval None
 */
void HAL_SYSCFG_EnableIOAnalogSwitchBooster(void)
{
  SET_BIT(SYSCFG->CFGR1, SYSCFG_CFGR1_BOOSTEN);
}

/**
 * @brief  Disable the I/O analog switch voltage booster
 *
 * @retval None
 */
void HAL_SYSCFG_DisableIOAnalogSwitchBooster(void)
{
  CLEAR_BIT(SYSCFG->CFGR1, SYSCFG_CFGR1_BOOSTEN);
}

/**
 * @}
 */

/**
 * @}
 */

#endif /* HAL_MODULE_ENABLED */
/**
 * @}
 */

/**
 * @}
 */
