# -*- coding: utf-8 -*-

# Copyright (c) 2011 - 2025 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a Matlab lexer with some additional methods.
"""

from PyQt6.Qsci import QsciLexerMatlab

from .Lexer import Lexer


class LexerMatlab(Lexer, QsciLexerMatlab):
    """
    Subclass to implement some additional lexer dependent methods.
    """

    def __init__(self, parent=None):
        """
        Constructor

        @param parent parent widget of this lexer
        @type QWidget
        """
        QsciLexerMatlab.__init__(self, parent)
        Lexer.__init__(self)

        self.commentString = "%~"

        self.keywordSetDescriptions = [
            self.tr("Keywords"),
        ]

    def isCommentStyle(self, style):
        """
        Public method to check, if a style is a comment style.

        @param style style to check
        @type int
        @return flag indicating a comment style
        @rtype bool
        """
        return style in [QsciLexerMatlab.Comment]

    def isStringStyle(self, style):
        """
        Public method to check, if a style is a string style.

        @param style style to check
        @type int
        @return flag indicating a string style
        @rtype bool
        """
        return style in [
            QsciLexerMatlab.DoubleQuotedString,
            QsciLexerMatlab.SingleQuotedString,
        ]

    def defaultKeywords(self, kwSet):
        """
        Public method to get the default keywords.

        @param kwSet number of the keyword set
        @type int
        @return string giving the keywords or None
        @rtype str
        """
        return QsciLexerMatlab.keywords(self, kwSet)


def createLexer(variant, parent=None):  # noqa: U-100
    """
    Function to instantiate a lexer object.

    @param variant name of the language variant (unused)
    @type str
    @param parent parent widget of this lexer
    @type QObject
    @return instantiated lexer object
    @rtype LexerMatlab
    """
    return LexerMatlab(parent=parent)
