"""Integration tests for the DuCommand."""

from __future__ import annotations

from typing import TYPE_CHECKING

import pytest

import pebble_shell.commands
import pebble_shell.shell

if TYPE_CHECKING:
    import ops


@pytest.fixture
def command(shell: pebble_shell.shell.PebbleShell):
    """Fixture to create a DuCommand instance."""
    yield pebble_shell.commands.DuCommand(shell=shell)


def test_name(command: pebble_shell.commands.DuCommand):
    assert command.name == "du [-h] [-s] [path...]"


def test_category(command: pebble_shell.commands.DuCommand):
    assert command.category == "Filesystem Commands"


def test_help(command: pebble_shell.commands.DuCommand):
    with command.shell.console.capture() as capture:
        command.show_help()
    assert "disk usage" in capture.get()


@pytest.mark.parametrize("args", [["-h"], ["--help"]])
def test_execute_help(
    client: ops.pebble.Client,
    command: pebble_shell.commands.DuCommand,
    args: list[str],
):
    with command.shell.console.capture() as capture:
        result = command.execute(client=client, args=args)
    assert result == 0
    assert "disk usage" in capture.get()


def test_execute_default(
    client: ops.pebble.Client,
    command: pebble_shell.commands.DuCommand,
):
    with command.shell.console.capture() as capture:
        result = command.execute(client=client, args=[])
    assert result == 0
    # Should print disk usage information
    output = capture.get()
    assert len(output.strip()) > 0


def test_execute_specific_path(
    client: ops.pebble.Client,
    command: pebble_shell.commands.DuCommand,
):
    with command.shell.console.capture() as _:
        result = command.execute(client=client, args=["/etc"])
    # Should succeed since /etc directory exists
    assert result == 0


def test_execute_nonexistent_path(
    client: ops.pebble.Client,
    command: pebble_shell.commands.DuCommand,
):
    with command.shell.console.capture() as _:
        result = command.execute(client=client, args=["/nonexistent/path"])
    # Should fail for nonexistent path
    assert result == 1
