"""Integration tests for the TarCommand."""

from __future__ import annotations

from typing import TYPE_CHECKING

import pytest

import pebble_shell.commands
import pebble_shell.shell

if TYPE_CHECKING:
    import ops


@pytest.fixture
def command(shell: pebble_shell.shell.PebbleShell):
    """Fixture to create a TarCommand instance."""
    yield pebble_shell.commands.TarCommand(shell=shell)


def test_name(command: pebble_shell.commands.TarCommand):
    assert command.name == "tar"


def test_category(command: pebble_shell.commands.TarCommand):
    assert command.category == "Compression"


def test_help(command: pebble_shell.commands.TarCommand):
    with command.shell.console.capture() as capture:
        command.show_help()
    assert "Archive files" in capture.get()


@pytest.mark.parametrize("args", [["-h"], ["--help"]])
def test_execute_help(
    client: ops.pebble.Client,
    command: pebble_shell.commands.TarCommand,
    args: list[str],
):
    with command.shell.console.capture() as capture:
        result = command.execute(client=client, args=args)
    assert result == 0
    assert "tar" in capture.get()


def test_execute_no_args(
    client: ops.pebble.Client,
    command: pebble_shell.commands.TarCommand,
):
    with command.shell.console.capture() as capture:
        result = command.execute(client=client, args=[])
    assert result == 1
    # Should show usage information
    output = capture.get()
    assert "Usage" in output or "usage" in output


def test_execute_list_nonexistent(
    client: ops.pebble.Client,
    command: pebble_shell.commands.TarCommand,
):
    with command.shell.console.capture() as _:
        result = command.execute(
            client=client, args=["-tf", "/nonexistent/archive.tar"]
        )
    # Should fail for nonexistent file
    assert result == 1
