# HydroTools :: Caches

This subpackage implements different methods to cache objects generated by `hydrotools` methods. See the [Caches Documentation](https://noaa-owp.github.io/hydrotools/hydrotools.caches.html) for a complete list and description of the currently available caches. To report bugs or request additional features, submit an issue through the [HydroTools Issue Tracker](https://github.com/NOAA-OWP/hydrotools/issues) on GitHub.

## Installation

In accordance with the python community, we support and advise the usage of virtual environments in any workflow using python. In the following installation guide, we use python's built-in `venv` module to create a virtual environment in which the tool will be installed. Note this is just personal preference, any python virtual environment manager should work just fine (`conda`, `pipenv`, etc. ).

```bash
# Create and activate python environment, requires python >= 3.8
$ python3 -m venv venv
$ source venv/bin/activate
$ python3 -m pip install --upgrade pip

# Install caches
$ python3 -m pip install hydrotools.caches
```

## Usage

The following example demonstrates how one might use `hydrotools.caches.hdf` to cache a `pandas.dataframe` generated by a long running process.

### Code
```python
from hydrotools.caches.hdf import HDFCache

import pandas as pd
from time import sleep

# Some long running process that returns a pandas.DataFrame
def long_process(cols, rows):
    sleep(1.0)
    data = {f'col_{i}' : [j for j in range(rows)] for i in range(cols)}
    return pd.DataFrame(data)

# Setup the cache with a context manager
#  Similar to setting up a pandas.HDFStore
with HDFCache(
    path='mycache.h5',
    complevel=1,
    complib='zlib',
    fletcher32=True
    ) as cache:
    # The first call runs long_process and stores the result
    df = cache.get(long_process, 'data/results', cols=10, rows=1000000)


    # The second call retrieves the result from cache without 
    #  running long_process
    df = cache.get(long_process, 'data/results', cols=10, rows=1000000)
```
