# pyright: reportUndefinedVariable=false


import logging
import os
from glob import glob
from typing import Any

from starbash.processing import NotEnoughFilesError
from starbash.tool import tools

siril = tools["siril"]

delete_temps = False

# ('context' and 'logger' are normally injected by the starbash runtime)
context: dict[str, Any] = {}
logger: logging.Logger = None  # type: ignore


# FIXME move this into main starbash
def perhaps_delete_temps(temps: list[str]) -> None:
    if delete_temps:
        for t in temps:
            for path in glob(f"{context['process_dir']}/{t}_*"):
                os.remove(path)


def make_stacked(variant: str | None, output_file: str):
    """
    Registers and stacks all pre-processed light frames for a given filter configuration
    across all sessions.
    """
    # If we are being invoked for simple/non duo filter we won't have variants generated by prior steps in the workflow
    if variant is None:
        input_base = "bkg_pp_light"
    else:
        input_base = f"{variant}_bkg_pp_light"

    # The sequence name for all frames of this variant across all sessions
    # e.g. Ha_bkg_pp_light_cHaOiii
    merged_seq_base = f"all_{input_base}"

    # Absolute path for the output stacked file
    stacked_output_path = glob(f"{context['process_dir']}/{output_file}.fit*")

    if stacked_output_path:
        logger.info(f"Using existing stacked file: {stacked_output_path}")
    else:
        # Merge all frames (from multiple sessions and configs) use those for stacking
        frames = glob(f"{context['process_dir']}/{input_base}_s*.fit*")

        logger.info(
            f"Registering and stacking {len(frames)} frames for {variant} -> {stacked_output_path}"
        )
        if len(frames) < 2:
            raise NotEnoughFilesError("Need at least two frames", frames)

        # Siril commands for registration and stacking. We run this in process_dir.
        commands = f"""
            link {merged_seq_base} -out={context["process_dir"]}
            cd {context["process_dir"]}

            # We use -2pass to select the best possible reference frame for others to register against
            register {merged_seq_base} -2pass

            # because we are using -2pass we must complete the registration here before stacking
            # FIXME make drizzle optional
            seqapplyreg {merged_seq_base} -drizzle

            stack r_{merged_seq_base} rej g 0.3 0.05 -filter-wfwhm=3k -norm=addscale -output_norm -32b -out={output_file}

            # and flip if required
            mirrorx_single {output_file}
            """

        context["input_files"] = frames
        siril.run(commands, context=context)

    perhaps_delete_temps([merged_seq_base, f"r_{merged_seq_base}"])


def make_renormalize(channel_num: int):
    """
    Aligns the stacked images (Sii, Ha, OIII) and renormalizes Sii and OIII
    to match the flux of the Ha channel.
    """
    logger.info("Aligning and renormalizing stacked images.")

    # Define file basenames for the stacked images created in the 'process' directory
    ha_base = "results_00001"
    oiii_base = "results_00002"
    sii_base = "results_00003"

    # Define final output paths. The 'results' directory is a symlink in the work dir.
    results_dir = context["output"]["base_path"]
    os.makedirs(results_dir, exist_ok=True)

    commands = ""

    if channel_num == 1:
        # Only one channel - just copy it - eventually we'll add other metadata
        final_path = f"{results_dir}/stacked.fits"
        commands += f"""
            load results_00001
            save "{final_path}"
            """

    # Basenames for registered files (output of 'register' command)
    r_ha = f"r_{ha_base}"
    r_oiii = f"r_{oiii_base}"

    if channel_num >= 2:
        # Do pixelmath to fixup channel brightness
        logger.info("Doing renormalisation of extra Ha/Oiii channels")

        ha_final_path = f"{results_dir}/stacked_Ha.fits"
        oiii_final_path = f"{results_dir}/stacked_OIII.fits"

        # Pixel math formula for renormalization.
        # It matches the median and spread (MAD) of a channel to a reference channel (Ha).
        # Formula: new = old * (MAD(ref)/MAD(old)) - (MAD(ref)/MAD(old)) * MEDIAN(old) + MEDIAN(ref)
        pm_oiii = f'"${r_oiii}$*mad(${r_ha}$)/mad(${r_oiii}$)-mad(${r_ha}$)/mad(${r_oiii}$)*median(${r_oiii}$)+median(${r_ha}$)"'

        # Siril commands to be executed in the 'process' directory
        commands += f"""
            # -transf=shift fails sometimes, which I guess is possible because we have multiple sessions with possible different camera rotation
            # -interp=none also fails sometimes, so let default interp happen
            # -drizzle is required for success on many images
            register results -drizzle
            pm {pm_oiii}
            update_key FILTER Oiii "OSC Duo filter extracted"
            save "{oiii_final_path}"
            load {r_ha}
            update_key FILTER Ha "OSC Duo filter extracted"
            save "{ha_final_path}"
            """

    if channel_num >= 3:
        logger.info("Doing renormalisation of extra Sii channel")

        sii_final_path = f"{results_dir}/stacked_Sii.fits"
        r_sii = f"r_{sii_base}"
        pm_sii = f'"${r_sii}$*mad(${r_ha}$)/mad(${r_sii}$)-mad(${r_ha}$)/mad(${r_sii}$)*median(${r_sii}$)+median(${r_ha}$)"'
        commands += f"""
            pm {pm_sii}
            update_key FILTER Sii "OSC dual Duo filter extracted"
            save "{sii_final_path}"
            """

    siril.run(commands, context=context, cwd=context["process_dir"])
    logger.info(f"Saved final renormalized images to {results_dir}")


def osc_process(has_ha_oiii: bool, has_sii_oiii: bool):
    """Shared code for use by OSC processing scripts"""

    logger.info(f"Running osc_process(has_ha_oiii={has_ha_oiii}, has_sii_oiii={has_sii_oiii})")
    logger.debug("Using context: %s", context)

    channel_num = 0
    if has_sii_oiii:
        # red output channel - from the SiiOiii filter Sii is on the 672nm red channel (mistakenly called Ha by siril)
        channel_num += 1
        make_stacked("Ha", f"results_{channel_num:05d}")

    if has_ha_oiii:
        # green output channel - from the HaOiii filter Ha is on the 656nm red channel
        channel_num += 1
        make_stacked("Ha", f"results_{channel_num:05d}")

    if has_ha_oiii or has_sii_oiii:
        # blue output channel - both filters have Oiii on the 500nm blue channel.  Note the case here is uppercase to match siril output
        channel_num += 1
        make_stacked("OIII", f"results_{channel_num:05d}")

    # if we haven't already processed some other way - just do a single channel process
    # FIXME in this case we want to use a siril line like "stack r_bkg_pp_light rej g 0.3 0.05 -filter-wfwhm=3k -norm=addscale -output_norm -rgb_equal -32b -out=result"
    if channel_num == 0:
        # single channel - just stack all Ha frames together
        channel_num += 1
        make_stacked(None, f"results_{channel_num:05d}")

    # There might be an old/state autogenerated .seq file, delete it so it doesn't confuse renormalize
    results_seq_path = f"{context['process_dir']}/results_.seq"
    if os.path.exists(results_seq_path):
        os.remove(results_seq_path)

    assert channel_num >= 1, "At least one channel should have been processed"
    make_renormalize(channel_num)
