"""Performance verification tests for FastBlocks optimizations.

This module provides comprehensive performance verification including:
- Baseline vs optimized performance comparison
- Regression testing for performance improvements
- End-to-end performance validation
- Memory usage and resource optimization verification
- Async performance and concurrency testing

Requirements:
- pytest-benchmark>=5.1
- pytest-asyncio>=0.24
- pytest-timeout>=2.4
- memory-profiler>=0.61.0

Author: lesleslie <les@wedgwoodwebworks.com>
Created: 2025-01-12
"""

import asyncio
import time

import pytest

# Import our optimization modules
# NOTE: Performance optimizers have been migrated to ACB Services Layer
# These imports will need to be updated once ACB services are fully available
try:
    from fastblocks.adapters.templates._enhanced_cache import (
        CacheTier,
        EnhancedCacheManager,
    )
    from fastblocks.adapters.templates._performance_optimizer import (
        PerformanceMetrics,
        PerformanceOptimizer,
    )
except ImportError:
    # Performance modules will be migrated to ACB services
    pytest.skip(
        "Performance optimization modules require ACB services layer",
        allow_module_level=True,
    )


class TestTemplatePerformanceOptimization:
    """Test template rendering performance optimizations."""

    @pytest.fixture
    def performance_optimizer(self):
        """Create performance optimizer instance."""
        return PerformanceOptimizer()

    @pytest.fixture
    def sample_metrics(self):
        """Create sample performance metrics."""
        return PerformanceMetrics(
            render_time=0.15,
            cache_hit=False,
            template_size=1024,
            context_size=2048,
            fragment_count=1,
            memory_usage=4096,
            concurrent_renders=1,
        )

    @pytest.mark.benchmark(group="template-optimization")
    def test_performance_metrics_tracking(
        self, benchmark, performance_optimizer, sample_metrics
    ):
        """Benchmark performance metrics tracking overhead."""
        template_name = "test_template.html"

        def record_metrics():
            performance_optimizer.record_render(template_name, sample_metrics)
            return performance_optimizer.get_performance_stats()

        result = benchmark(record_metrics)
        assert result.total_renders > 0
        assert template_name in performance_optimizer.template_stats

    @pytest.mark.benchmark(group="template-optimization")
    async def test_context_optimization_performance(
        self, benchmark, performance_optimizer
    ):
        """Benchmark context optimization performance."""
        template_name = "large_template.html"
        large_context = {
            "items": [
                {"id": i, "name": f"item_{i}", "data": f"data_{i}"} for i in range(200)
            ],
            "metadata": {"total": 200, "page": 1, "per_page": 50},
            "user": {"id": 1, "name": "test_user", "permissions": ["read", "write"]},
        }

        # First record some slow performance to trigger optimization
        slow_metrics = PerformanceMetrics(
            render_time=0.15,  # Above the 0.1 threshold
            cache_hit=False,
            template_size=1000,
            context_size=len(str(large_context)),
            fragment_count=5,
        )
        performance_optimizer.record_render(template_name, slow_metrics)

        async def optimize_context():
            return await performance_optimizer.optimize_render_context(
                template_name, large_context
            )

        result = await benchmark.pedantic(optimize_context, iterations=10, rounds=5)

        # Verify optimization applied
        assert "items_paginated" in result
        assert result["items_total"] == 200
        assert len(result["items"]) == 50  # Paginated to first 50

    def test_cache_ttl_optimization(self, performance_optimizer):
        """Test optimal cache TTL calculation."""
        template_name = "slow_template.html"

        # Simulate slow renders
        slow_metrics = PerformanceMetrics(
            render_time=0.25,  # 250ms
            cache_hit=False,
            template_size=512,
            context_size=1024,
        )

        for _ in range(5):
            performance_optimizer.record_render(template_name, slow_metrics)

        optimal_ttl = performance_optimizer.get_optimal_cache_ttl(template_name)
        assert optimal_ttl > 300  # Should get longer TTL for slow templates

    def test_streaming_decision(self, performance_optimizer):
        """Test streaming enablement decision."""
        template_name = "streaming_candidate.html"
        large_context_size = 60000  # 60KB

        should_stream = performance_optimizer.should_enable_streaming(
            template_name, large_context_size
        )
        assert should_stream is True

        # Test with historical slow performance
        slow_metrics = PerformanceMetrics(
            render_time=0.6,  # 600ms
            cache_hit=False,
            template_size=1024,
            context_size=large_context_size,
        )
        performance_optimizer.record_render(template_name, slow_metrics)

        should_stream_historical = performance_optimizer.should_enable_streaming(
            template_name,
            30000,  # Smaller context but historical slowness
        )
        assert should_stream_historical is True

    def test_optimization_recommendations(self, performance_optimizer, sample_metrics):
        """Test performance optimization recommendations."""
        # Create scenario with poor cache performance
        poor_cache_metrics = PerformanceMetrics(
            render_time=0.05,
            cache_hit=False,  # Always cache miss
            template_size=512,
            context_size=1024,
        )

        # Record many cache misses
        for _ in range(20):
            performance_optimizer.record_render(
                "poor_cache_template.html", poor_cache_metrics
            )

        recommendations = performance_optimizer.get_optimization_recommendations()
        assert len(recommendations) > 0
        assert any("Cache hit ratio" in rec for rec in recommendations)


class TestEnhancedCachePerformance:
    """Test enhanced caching performance optimizations."""

    @pytest.fixture
    async def cache_manager(self):
        """Create enhanced cache manager with test-friendly settings."""
        manager = EnhancedCacheManager(
            max_memory_entries=100,
            promotion_threshold=3,  # Lower threshold for testing
            demotion_idle_time=1,  # 1 second for testing
        )
        await manager.initialize()
        yield manager
        await manager.shutdown()

    @pytest.mark.benchmark(group="cache-optimization")
    async def test_cache_get_performance(self, benchmark, cache_manager):
        """Benchmark cache get operations."""
        # Pre-populate cache
        for i in range(50):
            await cache_manager.set(f"key_{i}", f"value_{i}", tier=CacheTier.HOT)

        async def get_operation():
            return await cache_manager.get("key_25")

        result = await benchmark.pedantic(get_operation, iterations=100, rounds=10)
        assert result == "value_25"

    @pytest.mark.benchmark(group="cache-optimization")
    async def test_cache_set_with_metadata(self, benchmark, cache_manager):
        """Benchmark cache set operations with full metadata."""

        async def set_operation():
            await cache_manager.set(
                key="benchmark_key",
                value={"data": "benchmark_value", "metadata": {"size": 1024}},
                ttl=300,
                dependencies={"user_data", "session_data"},
                tags={"user", "session", "temporary"},
                tier=CacheTier.WARM,
            )

        await benchmark.pedantic(set_operation, iterations=50, rounds=5)

        # Verify entry was created
        result = await cache_manager.get("benchmark_key")
        assert result is not None

    async def test_tier_optimization_performance(self, cache_manager):
        """Test cache tier optimization performance."""
        # Create entries with different access patterns
        hot_key = "hot_accessed"
        cold_key = "cold_accessed"

        await cache_manager.set(hot_key, "hot_value", tier=CacheTier.COLD)
        await cache_manager.set(cold_key, "cold_value", tier=CacheTier.HOT)

        # Access hot key frequently (above promotion threshold of 3)
        for _ in range(5):
            await cache_manager.get(hot_key)

        # Don't access cold key, wait for demotion time (1 second)
        await asyncio.sleep(1.1)  # Wait longer than demotion_idle_time

        # Run tier optimization
        optimization_results = await cache_manager.optimize_tiers()

        assert (
            optimization_results["promotions"] > 0
            or optimization_results["demotions"] > 0
        )

    async def test_dependency_invalidation_performance(self, cache_manager):
        """Test dependency-based cache invalidation."""
        # Create entries with dependencies
        for i in range(20):
            await cache_manager.set(
                f"dependent_key_{i}",
                f"value_{i}",
                dependencies={f"dependency_{i % 5}"},  # 5 different dependencies
            )

        # Measure invalidation performance
        start_time = time.time()
        invalidated = await cache_manager.invalidate_by_dependency("dependency_0")
        invalidation_time = time.time() - start_time

        assert len(invalidated) > 0
        assert invalidation_time < 0.1  # Should be fast

    async def test_cache_statistics_accuracy(self, cache_manager):
        """Test cache statistics accuracy and performance."""
        # Perform various operations
        await cache_manager.set("stat_key_1", "value_1", tier=CacheTier.HOT)
        await cache_manager.set("stat_key_2", "value_2", tier=CacheTier.WARM)
        await cache_manager.set("stat_key_3", "value_3", tier=CacheTier.COLD)

        await cache_manager.get("stat_key_1")
        await cache_manager.get("stat_key_2")
        await cache_manager.get("nonexistent_key")  # Cache miss

        stats = await cache_manager.get_stats()

        assert stats.total_entries == 3
        assert stats.metrics.hits >= 2
        assert stats.metrics.misses >= 1
        assert stats.tier_distribution[CacheTier.HOT] >= 1
        assert stats.tier_distribution[CacheTier.WARM] >= 1
        assert stats.tier_distribution[CacheTier.COLD] >= 1


class TestQueryPerformanceOptimization:
    """Test database query performance optimizations."""

    @pytest.fixture
    def query_optimizer(self):
        """Create query optimizer instance."""
        return QueryPerformanceOptimizer(slow_query_threshold=0.1)

    @pytest.mark.benchmark(group="query-optimization")
    async def test_query_analysis_performance(self, benchmark, query_optimizer):
        """Benchmark query analysis performance."""
        complex_query = """
        SELECT u.id, u.name, u.email, p.title, COUNT(c.id) as comment_count
        FROM users u
        LEFT JOIN posts p ON u.id = p.author_id
        LEFT JOIN comments c ON p.id = c.post_id
        WHERE u.created_at > %s
        AND p.published = true
        GROUP BY u.id, u.name, u.email, p.title
        ORDER BY comment_count DESC
        LIMIT 50
        """

        async def analyze_query():
            return await query_optimizer.analyze_query(
                complex_query,
                params={"created_at": "2024-01-01"},
            )

        result = await benchmark.pedantic(analyze_query, iterations=10, rounds=5)

        assert result["query_type"] == QueryType.SELECT
        assert "users" in result["table_names"]
        assert len(result["suggestions"]) > 0

    async def test_query_execution_with_optimization(self, query_optimizer):
        """Test query execution with optimization monitoring."""

        async def mock_query_func():
            await asyncio.sleep(0.05)  # Simulate query execution
            return [{"id": 1, "name": "test"}]

        query = "SELECT id, name FROM users WHERE active = true"

        result, metrics = await query_optimizer.execute_with_optimization(
            mock_query_func,
            query,
            params={"active": True},
            cache_key="users_active_query",
        )

        assert result == [{"id": 1, "name": "test"}]
        assert metrics.execution_time > 0
        assert metrics.query_type == QueryType.SELECT
        assert not metrics.cache_hit  # First execution

        # Second execution should hit cache
        result2, metrics2 = await query_optimizer.execute_with_optimization(
            mock_query_func,
            query,
            params={"active": True},
            cache_key="users_active_query",
        )

        assert result2 == result
        assert metrics2.cache_hit is True

    async def test_slow_query_detection(self, query_optimizer):
        """Test slow query detection and handling."""

        async def slow_query_func():
            await asyncio.sleep(0.15)  # Slow query (>100ms threshold)
            return []

        query = "SELECT * FROM large_table WHERE complex_condition = true"

        _, metrics = await query_optimizer.execute_with_optimization(
            slow_query_func, query
        )

        assert metrics.execution_time > query_optimizer.slow_query_threshold

        # Check that slow query was recorded
        health = await query_optimizer.get_database_health()
        assert health["query_performance"]["slow_query_ratio"] > 0

    async def test_query_pattern_analysis(self, query_optimizer):
        """Test query pattern analysis and recommendations."""
        # Execute similar queries multiple times
        base_query = "SELECT id, name FROM users WHERE category = %s"

        async def mock_query():
            await asyncio.sleep(0.02)
            return []

        # Execute variations of the query
        for category in ["admin", "user", "guest", "admin", "user"]:
            await query_optimizer.execute_with_optimization(
                mock_query,
                base_query,
                params={"category": category},
            )

        recommendations = await query_optimizer.get_query_recommendations()
        assert len(recommendations) > 0

        # Verify pattern detection
        report = await query_optimizer.generate_performance_report()
        assert "pattern_analysis" in report
        assert report["pattern_analysis"]["total_patterns"] > 0


class TestAsyncPerformanceOptimization:
    """Test async performance optimizations."""

    @pytest.fixture
    async def async_optimizer(self):
        """Create async performance optimizer."""
        optimizer = AsyncPerformanceOptimizer(max_concurrent_tasks=100)
        await optimizer.initialize()
        yield optimizer
        await optimizer.shutdown()

    @pytest.mark.benchmark(group="async-optimization")
    async def test_task_execution_overhead(self, benchmark, async_optimizer):
        """Benchmark task execution overhead."""

        async def simple_task():
            await asyncio.sleep(0.001)
            return "completed"

        async def execute_task():
            return await async_optimizer.execute_task(
                simple_task(),
                priority=TaskPriority.NORMAL,
                task_name="benchmark_task",
            )

        result = await benchmark.pedantic(execute_task, iterations=50, rounds=5)
        assert result == "completed"

    async def test_batch_execution_performance(self, async_optimizer):
        """Test batch execution performance and concurrency."""

        async def task_func(task_id):
            await asyncio.sleep(0.01)
            return f"task_{task_id}_completed"

        tasks = [task_func(i) for i in range(20)]

        start_time = time.time()
        results = await async_optimizer.batch_execute(
            tasks,
            max_concurrency=5,
            priority=TaskPriority.HIGH,
        )
        execution_time = time.time() - start_time

        assert len(results) == 20
        assert all(
            "completed" in str(r) for r in results if not isinstance(r, Exception)
        )
        # Should be faster than sequential execution due to concurrency
        assert execution_time < 0.5  # Should complete in less than 500ms

    async def test_resource_pool_performance(self, async_optimizer):
        """Test async resource pool performance."""
        connection_count = 0

        async def create_connection():
            nonlocal connection_count
            connection_count += 1
            return f"connection_{connection_count}"

        # Test resource pool usage
        async with async_optimizer.resource_pool(
            "test_pool", create_connection, max_size=5
        ) as conn1:
            assert "connection_" in conn1

        async with async_optimizer.resource_pool(
            "test_pool", create_connection, max_size=5
        ) as conn2:
            # Should reuse connection from pool
            assert conn2 == conn1

    @pytest.mark.benchmark(group="async-optimization")
    async def test_stream_processing_performance(self, benchmark, async_optimizer):
        """Benchmark async stream processing."""

        async def data_generator():
            for i in range(100):
                yield f"data_{i}"
                await asyncio.sleep(0.001)  # Small delay

        async def processor(item):
            await asyncio.sleep(0.002)  # Processing time
            return f"processed_{item}"

        async def process_stream():
            results = []
            async for result in async_optimizer.stream_processor(
                data_generator(),
                processor,
                max_concurrency=10,
                buffer_size=20,
            ):
                results.append(result)
            return results

        results = await benchmark.pedantic(process_stream, iterations=1, rounds=3)
        assert len(results) == 100
        assert all("processed_data_" in r for r in results)

    async def test_performance_monitoring_accuracy(self, async_optimizer):
        """Test async performance monitoring accuracy."""

        # Execute various tasks
        async def fast_task():
            return "fast"

        async def slow_task():
            await asyncio.sleep(0.05)
            return "slow"

        async def failing_task():
            raise ValueError("Test failure")

        # Execute different types of tasks
        await async_optimizer.execute_task(fast_task(), task_name="fast_task")
        await async_optimizer.execute_task(slow_task(), task_name="slow_task")

        try:
            await async_optimizer.execute_task(failing_task(), task_name="failing_task")
        except ValueError:
            pass

        metrics = await async_optimizer.get_performance_metrics()

        assert metrics["total_tasks"] >= 3
        assert metrics["completed_tasks"] >= 2
        assert metrics["failed_tasks"] >= 1
        assert 0 <= metrics["success_rate"] <= 1

    async def test_optimization_recommendations(self, async_optimizer):
        """Test async optimization recommendations."""

        # Create conditions that trigger recommendations
        # Simulate high queue congestion
        async def queue_filler():
            await asyncio.sleep(1)  # Long running task

        # Add many tasks to create queue congestion
        tasks = [queue_filler() for _ in range(10)]
        await asyncio.gather(
            *[
                async_optimizer.execute_task(task, priority=TaskPriority.LOW)
                for task in tasks[:3]  # Only execute a few to create backlog
            ],
            return_exceptions=True,
        )

        recommendations = await async_optimizer.get_optimization_recommendations()

        # Should have some recommendations
        assert (
            len(recommendations) >= 0
        )  # May or may not have recommendations based on timing


class TestIntegratedPerformanceVerification:
    """Test integrated performance improvements across all optimizations."""

    @pytest.mark.benchmark(group="integrated-performance")
    async def test_end_to_end_performance(self, benchmark):
        """Benchmark end-to-end performance with all optimizations."""
        # Create all optimizers
        template_optimizer = PerformanceOptimizer()
        cache_manager = EnhancedCacheManager(max_memory_entries=50)
        query_optimizer = QueryPerformanceOptimizer()
        async_optimizer = AsyncPerformanceOptimizer(max_concurrent_tasks=50)

        await cache_manager.initialize()
        await async_optimizer.initialize()

        async def integrated_operation():
            # Simulate a complete request cycle with optimizations

            # 1. Query optimization
            async def mock_db_query():
                await asyncio.sleep(0.01)
                return [{"id": 1, "name": "test_user", "data": "sample_data"}]

            (
                query_result,
                query_metrics,
            ) = await query_optimizer.execute_with_optimization(
                mock_db_query,
                "SELECT id, name, data FROM users WHERE active = true",
                cache_key="active_users",
            )

            # 2. Cache optimization
            await cache_manager.set(
                "processed_data",
                query_result,
                ttl=300,
                tier=CacheTier.WARM,
            )

            # 3. Template rendering optimization
            template_metrics = PerformanceMetrics(
                render_time=0.05,
                cache_hit=False,
                template_size=2048,
                context_size=len(str(query_result)),
            )
            template_optimizer.record_render("user_profile.html", template_metrics)

            # 4. Async optimization
            async def render_task():
                await asyncio.sleep(0.02)
                return "<html>Rendered content</html>"

            rendered_content = await async_optimizer.execute_task(
                render_task(),
                priority=TaskPriority.NORMAL,
                task_name="render_template",
            )

            return {
                "query_result": query_result,
                "cached_data": await cache_manager.get("processed_data"),
                "rendered_content": rendered_content,
                "performance_stats": {
                    "query_time": query_metrics.execution_time,
                    "cache_hit": query_metrics.cache_hit,
                    "template_stats": template_optimizer.get_performance_stats(),
                },
            }

        result = await benchmark.pedantic(integrated_operation, iterations=5, rounds=3)

        # Verify all components worked
        assert result["query_result"] is not None
        assert result["cached_data"] is not None
        assert "html" in result["rendered_content"]
        assert result["performance_stats"]["query_time"] > 0

        # Cleanup
        await cache_manager.shutdown()
        await async_optimizer.shutdown()

    async def test_memory_usage_optimization(self):
        """Test memory usage optimization across all components."""
        import os

        import psutil

        process = psutil.Process(os.getpid())
        initial_memory = process.memory_info().rss

        # Create optimizers with memory-conscious settings
        cache_manager = EnhancedCacheManager(max_memory_entries=20)
        query_optimizer = QueryPerformanceOptimizer(max_cached_queries=50)

        await cache_manager.initialize()

        # Perform memory-intensive operations
        for i in range(100):
            # Add data to cache
            await cache_manager.set(
                f"memory_test_{i}",
                {"data": f"large_data_entry_{i}" * 100},  # ~2KB per entry
                tier=CacheTier.COLD,
            )

            # Simulate query operations
            async def mock_query():
                return [{"id": j, "data": f"row_{j}"} for j in range(10)]

            await query_optimizer.execute_with_optimization(
                mock_query,
                f"SELECT * FROM table_{i % 5}",
                cache_key=f"query_{i}",
            )

        final_memory = process.memory_info().rss
        memory_increase = final_memory - initial_memory

        # Memory increase should be reasonable (less than 50MB for this test)
        assert memory_increase < 50 * 1024 * 1024

        # Test memory cleanup
        await cache_manager.clear()
        await query_optimizer.clear_query_cache()

        cleanup_memory = process.memory_info().rss

        # Memory should be reduced after cleanup
        assert cleanup_memory < final_memory

        await cache_manager.shutdown()

    def test_performance_regression_detection(self):
        """Test that performance hasn't regressed from baseline."""
        # This would typically compare against stored baseline metrics
        # For now, we'll test that our optimizations meet minimum standards

        optimizer = PerformanceOptimizer()

        # Test performance thresholds
        fast_metrics = PerformanceMetrics(
            render_time=0.001,  # 1ms - very fast
            cache_hit=True,
            template_size=512,
            context_size=1024,
        )

        slow_metrics = PerformanceMetrics(
            render_time=0.5,  # 500ms - slow
            cache_hit=False,
            template_size=4096,
            context_size=8192,
        )

        optimizer.record_render("fast_template", fast_metrics)
        optimizer.record_render("slow_template", slow_metrics)

        recommendations = optimizer.get_optimization_recommendations()

        # Should recommend optimization for slow template
        assert any("slow_template" in rec for rec in recommendations)

        # Fast template should not need optimization
        stats = optimizer.get_performance_stats()
        assert stats.fastest_templates is not None


# Performance benchmark configuration
@pytest.fixture(scope="session", autouse=True)
def configure_benchmarks():
    """Configure benchmark settings for all tests."""
    # This fixture runs once per test session and configures benchmarks
    # Benchmark thresholds and settings are configured in pyproject.toml
    pass


if __name__ == "__main__":
    # Run verification tests
    pytest.main(
        [
            __file__,
            "-v",
            "--benchmark-group-by=group",
            "--benchmark-columns=min,max,mean,stddev",
            "--benchmark-sort=mean",
        ]
    )
