"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToOrigins = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
class CloudFrontToOrigins extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.aRecords = [];
        // Private maps for internal use
        this.s3BucketsMap = new Map();
        this.httpOriginsMap = new Map();
        this.s3OriginsMap = new Map();
        // Validate that at least one origin is provided
        if ((!props.s3Origins || props.s3Origins.length === 0) &&
            (!props.httpOrigins || props.httpOrigins.length === 0)) {
            throw new Error("At least one origin (s3Origins or httpOrigins) must be provided");
        }
        // Validate unique origin IDs
        this.validateOriginIds(props);
        // Apply intelligent defaults (unless disabled)
        const enhancedProps = props.disableIntelligentDefaults
            ? props
            : this.applyDefaults(props);
        let certificate;
        let logBucket;
        // Setup all S3 origins
        if (enhancedProps.s3Origins) {
            for (const s3Config of enhancedProps.s3Origins) {
                this.s3BucketsMap.set(s3Config.id, s3Config.bucket);
                const s3Origin = this.setupS3Origin(s3Config);
                this.s3OriginsMap.set(s3Config.id, s3Origin);
            }
        }
        // Setup all HTTP origins
        if (enhancedProps.httpOrigins) {
            for (const httpConfig of enhancedProps.httpOrigins) {
                const httpOrigin = this.setupHttpOrigin(httpConfig);
                this.httpOriginsMap.set(httpConfig.id, httpOrigin);
            }
        }
        // Setup custom domain and certificate
        if (enhancedProps.customDomainName) {
            certificate = this.setupCustomDomain(enhancedProps);
        }
        // Setup logging (enabled by default unless explicitly disabled)
        if (enhancedProps.enableLogging !== false) {
            logBucket = this.setupLogging(enhancedProps);
        }
        // Create CloudFront distribution
        this.distribution = this.createDistribution(enhancedProps, {
            certificate,
            logBucket,
        });
        // Setup Route53 records for all domains
        if (enhancedProps.createRoute53Records !== false &&
            enhancedProps.hostedZone) {
            this.setupRoute53Records(enhancedProps);
        }
        // Assign readonly properties
        this.logBucket = logBucket;
        this.certificate = certificate;
        this.domainNames = this.getAllDomainNames(enhancedProps);
    }
    /**
     * Get all domain names (primary + additional)
     */
    getAllDomainNames(props) {
        const domains = [];
        if (props.customDomainName) {
            domains.push(props.customDomainName);
        }
        if (props.additionalDomainNames) {
            domains.push(...props.additionalDomainNames);
        }
        return domains.length > 0 ? domains : undefined;
    }
    /**
     * Validate that all origin IDs are unique
     */
    validateOriginIds(props) {
        const allIds = new Set();
        // Check S3 origin IDs
        if (props.s3Origins) {
            for (const s3Origin of props.s3Origins) {
                if (allIds.has(s3Origin.id)) {
                    throw new Error(`Duplicate origin ID: ${s3Origin.id}`);
                }
                allIds.add(s3Origin.id);
            }
        }
        // Check HTTP origin IDs
        if (props.httpOrigins) {
            for (const httpOrigin of props.httpOrigins) {
                if (allIds.has(httpOrigin.id)) {
                    throw new Error(`Duplicate origin ID: ${httpOrigin.id}`);
                }
                allIds.add(httpOrigin.id);
            }
        }
        // Validate default origin ID exists
        if (props.defaultOriginId && !allIds.has(props.defaultOriginId)) {
            throw new Error(`Default origin ID '${props.defaultOriginId}' not found in origins`);
        }
        // Validate cache behavior origin IDs
        if (props.cacheBehaviors) {
            for (const behavior of props.cacheBehaviors) {
                if (!allIds.has(behavior.originId)) {
                    throw new Error(`Cache behavior origin ID '${behavior.originId}' not found in origins`);
                }
            }
        }
    }
    /**
     * Apply intelligent defaults based on origin configuration
     */
    applyDefaults(props) {
        const hasS3 = props.s3Origins && props.s3Origins.length > 0;
        const hasHttp = props.httpOrigins && props.httpOrigins.length > 0;
        let errorPages;
        // Only apply error page defaults if not disabled and not provided
        if (!props.disableDefaultErrorPages && !props.errorPages) {
            if (hasS3 && hasHttp) {
                // Hybrid: SPA-friendly error pages
                errorPages = this.getHybridErrorPages();
            }
            else if (hasS3) {
                // S3 only: SPA-friendly error pages
                errorPages = this.getS3ErrorPages();
            }
            else {
                // HTTP only: Basic error pages
                errorPages = this.getHttpErrorPages();
            }
        }
        return {
            // Apply base defaults
            enableLogging: true,
            createRoute53Records: true,
            defaultRootObject: "index.html",
            priceClass: cloudfront.PriceClass.PRICE_CLASS_100,
            enabled: true,
            // Override with user props
            ...props,
            // Apply error pages if generated
            ...(errorPages && { errorPages }),
            // Apply HTTP origin protocol defaults
            httpOrigins: props.httpOrigins?.map((httpOrigin) => ({
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
                ...httpOrigin,
            })),
            // Merge cache behaviors
            cacheBehaviors: [...(props.cacheBehaviors || [])],
        };
    }
    /**
     * Default error pages for S3-only distributions
     */
    getS3ErrorPages() {
        return [
            {
                httpStatus: 404,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
            {
                httpStatus: 403,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
        ];
    }
    /**
     * Default error pages for HTTP-only distributions
     */
    getHttpErrorPages() {
        return [
            {
                httpStatus: 504,
                responseHttpStatus: 504,
                responsePagePath: "/error.html",
                ttl: aws_cdk_lib_1.Duration.seconds(30),
            },
        ];
    }
    /**
     * Default error pages for hybrid distributions
     */
    getHybridErrorPages() {
        return [
            {
                httpStatus: 404,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
        ];
    }
    /**
     * Setup S3 origin
     */
    setupS3Origin(s3Config) {
        if (s3Config.useLegacyOAI) {
            // Use legacy Origin Access Identity
            const originAccessIdentity = s3Config.originAccessIdentity ||
                new cloudfront.OriginAccessIdentity(this, `OAI-${s3Config.id}`, {
                    comment: `OAI for ${s3Config.bucket.bucketName}`,
                });
            // Grant CloudFront access to S3 bucket
            s3Config.bucket.grantRead(originAccessIdentity);
            return new origins.S3Origin(s3Config.bucket, {
                originAccessIdentity,
                originPath: s3Config.originPath,
                ...s3Config.s3OriginProps,
            });
        }
        else {
            // Use modern Origin Access Control (recommended)
            return origins.S3BucketOrigin.withOriginAccessControl(s3Config.bucket, {
                originPath: s3Config.originPath,
                ...s3Config.s3OriginProps,
            });
        }
    }
    /**
     * Setup HTTP origin
     */
    setupHttpOrigin(httpConfig) {
        return new origins.HttpOrigin(httpConfig.domainName, {
            originPath: httpConfig.originPath,
            protocolPolicy: httpConfig.protocolPolicy || cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: httpConfig.httpPort,
            httpsPort: httpConfig.httpsPort,
            ...httpConfig.httpOriginProps,
        });
    }
    /**
     * Setup custom domain and certificate
     */
    setupCustomDomain(props) {
        // If certificate ARN is provided, use it directly
        if (props.certificateArn) {
            // Validate that certificate ARN is from us-east-1
            if (!props.certificateArn.includes("us-east-1")) {
                throw new Error("Certificate must be in us-east-1 region for CloudFront");
            }
            return acm.Certificate.fromCertificateArn(this, "Certificate", props.certificateArn);
        }
        // Auto-create certificate in us-east-1 if hostedZone is provided
        if (props.hostedZone && props.customDomainName) {
            return this.createCrossRegionCertificate(props);
        }
        throw new Error("Either certificateArn or hostedZone must be provided for custom domain. " +
            "If hostedZone is provided, certificate will be automatically created in us-east-1.");
    }
    /**
     * Create certificate in us-east-1 using CDK cross-region support
     */
    createCrossRegionCertificate(props) {
        const allDomains = this.getAllDomainNames(props) || [
            props.customDomainName,
        ];
        const currentRegion = aws_cdk_lib_1.Stack.of(this).region;
        // If already in us-east-1, create certificate directly
        if (currentRegion === "us-east-1") {
            return new acm.Certificate(this, "Certificate", {
                domainName: props.customDomainName,
                subjectAlternativeNames: allDomains.length > 1 ? allDomains.slice(1) : undefined,
                validation: acm.CertificateValidation.fromDns(props.hostedZone),
            });
        }
        // For other regions, use cross-region certificate
        return new acm.Certificate(this, "Certificate", {
            domainName: props.customDomainName,
            subjectAlternativeNames: allDomains.length > 1 ? allDomains.slice(1) : undefined,
            validation: acm.CertificateValidation.fromDns(props.hostedZone),
            // CDK will automatically handle cross-region for CloudFront
        });
    }
    /**
     * Setup CloudFront logging
     */
    setupLogging(props) {
        return (props.logBucket ||
            new s3.Bucket(this, "LogBucket", {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                versioned: false,
                lifecycleRules: [
                    {
                        expiration: aws_cdk_lib_1.Duration.days(90),
                        transitions: [
                            {
                                storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                                transitionAfter: aws_cdk_lib_1.Duration.days(30),
                            },
                        ],
                    },
                ],
            }));
    }
    /**
     * Get origin by ID
     */
    getOriginById(originId) {
        // Check S3 origins first
        const s3Origin = this.s3OriginsMap.get(originId);
        if (s3Origin) {
            return s3Origin;
        }
        // Check HTTP origins
        const httpOrigin = this.httpOriginsMap.get(originId);
        if (httpOrigin) {
            return httpOrigin;
        }
        throw new Error(`Origin with ID '${originId}' not found`);
    }
    /**
     * Determine default origin ID
     */
    getDefaultOriginId(props) {
        if (props.defaultOriginId) {
            return props.defaultOriginId;
        }
        // Default to first S3 origin, then first HTTP origin
        if (props.s3Origins && props.s3Origins.length > 0) {
            return props.s3Origins[0].id;
        }
        if (props.httpOrigins && props.httpOrigins.length > 0) {
            return props.httpOrigins[0].id;
        }
        throw new Error("No origins configured");
    }
    /**
     * Create CloudFront distribution
     */
    createDistribution(props, resources) {
        // Determine default origin
        const defaultOriginId = this.getDefaultOriginId(props);
        const defaultOrigin = this.getOriginById(defaultOriginId);
        const defaultBehavior = {
            origin: defaultOrigin,
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            compress: true,
        };
        // Build additional behaviors
        const additionalBehaviors = {};
        if (props.cacheBehaviors) {
            for (const behavior of props.cacheBehaviors) {
                const origin = this.getOriginById(behavior.originId);
                // Build behavior options object properly
                const behaviorConfig = {
                    origin,
                    viewerProtocolPolicy: behavior.viewerProtocolPolicy ||
                        cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                    allowedMethods: behavior.allowedMethods,
                    cachedMethods: behavior.cachedMethods,
                    compress: behavior.compress ?? true,
                };
                // Add cache policy (prioritize direct policy over ID)
                if (behavior.cachePolicy) {
                    behaviorConfig.cachePolicy = behavior.cachePolicy;
                }
                else if (behavior.cachePolicyId) {
                    behaviorConfig.cachePolicy = cloudfront.CachePolicy.fromCachePolicyId(this, `CachePolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.cachePolicyId);
                }
                // Add origin request policy
                if (behavior.originRequestPolicy) {
                    behaviorConfig.originRequestPolicy = behavior.originRequestPolicy;
                }
                else if (behavior.originRequestPolicyId) {
                    behaviorConfig.originRequestPolicy =
                        cloudfront.OriginRequestPolicy.fromOriginRequestPolicyId(this, `OriginRequestPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.originRequestPolicyId);
                }
                // Add response headers policy
                if (behavior.responseHeadersPolicy) {
                    behaviorConfig.responseHeadersPolicy = behavior.responseHeadersPolicy;
                }
                else if (behavior.responseHeadersPolicyId) {
                    behaviorConfig.responseHeadersPolicy =
                        cloudfront.ResponseHeadersPolicy.fromResponseHeadersPolicyId(this, `ResponseHeadersPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.responseHeadersPolicyId);
                }
                additionalBehaviors[behavior.pathPattern] =
                    behaviorConfig;
            }
        }
        // Prepare distribution props
        const distributionProps = {
            comment: props.comment || `CloudFront distribution for ${this.node.id}`,
            defaultBehavior,
            additionalBehaviors,
            domainNames: this.getAllDomainNames(props),
            certificate: resources.certificate,
            defaultRootObject: props.defaultRootObject,
            enabled: props.enabled,
            priceClass: props.priceClass,
            geoRestriction: props.geoRestriction,
            enableLogging: props.enableLogging !== false,
            logBucket: resources.logBucket,
            logFilePrefix: props.logPrefix,
            logIncludesCookies: props.logIncludeCookies,
            errorResponses: props.errorPages,
            webAclId: props.webAclId,
            enableIpv6: props.enableIpv6,
            httpVersion: props.httpVersion,
        };
        return new cloudfront.Distribution(this, "Distribution", distributionProps);
    }
    /**
     * Setup Route53 alias records for all domains
     */
    setupRoute53Records(props) {
        if (!props.hostedZone) {
            return;
        }
        const domains = this.getAllDomainNames(props);
        if (!domains) {
            return;
        }
        // Create A record for each domain
        domains.forEach((domain, index) => {
            const recordId = index === 0 ? "AliasRecord" : `AliasRecord${index + 1}`;
            const aRecord = new route53.ARecord(this, recordId, {
                zone: props.hostedZone,
                recordName: domain,
                target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
            });
            this.aRecords.push(aRecord);
        });
    }
    /**
     * Get S3 bucket by origin ID
     */
    getS3Bucket(originId) {
        return this.s3BucketsMap.get(originId);
    }
    /**
     * Get HTTP origin by origin ID
     */
    getHttpOrigin(originId) {
        return this.httpOriginsMap.get(originId);
    }
    /**
     * Get all S3 bucket origin IDs
     */
    get s3OriginIds() {
        return Array.from(this.s3BucketsMap.keys());
    }
    /**
     * Get all HTTP origin IDs
     */
    get httpOriginIds() {
        return Array.from(this.httpOriginsMap.keys());
    }
    /**
     * Get all S3 buckets as an array of objects with ID and bucket
     */
    get s3Origins() {
        return Array.from(this.s3BucketsMap.entries()).map(([id, bucket]) => ({
            id,
            bucket,
        }));
    }
    /**
     * Get all HTTP origins as an array of objects with ID and origin
     */
    get httpOrigins() {
        return Array.from(this.httpOriginsMap.entries()).map(([id, origin]) => ({
            id,
            origin,
        }));
    }
    /**
     * Get the CloudFront distribution domain name
     */
    get distributionDomainName() {
        return this.distribution.distributionDomainName;
    }
    /**
     * Get the CloudFront distribution URL with protocol
     */
    get distributionUrl() {
        return `https://${this.distribution.distributionDomainName}`;
    }
    /**
     * Get the custom domain URL (if configured)
     */
    get customDomainUrl() {
        return this.domainNames?.[0] ? `https://${this.domainNames[0]}` : undefined;
    }
}
exports.CloudFrontToOrigins = CloudFrontToOrigins;
_a = JSII_RTTI_SYMBOL_1;
CloudFrontToOrigins[_a] = { fqn: "must-cdk.CloudFrontToOrigins", version: "0.0.53" };
//# sourceMappingURL=data:application/json;base64,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