"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToOrigins = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
class CloudFrontToOrigins extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.aRecords = [];
        // Private maps for internal use
        this.s3BucketsMap = new Map();
        this.httpOriginsMap = new Map();
        this.s3OriginsMap = new Map();
        // Validate that at least one origin is provided
        if ((!props.s3Origins || props.s3Origins.length === 0) &&
            (!props.httpOrigins || props.httpOrigins.length === 0)) {
            throw new Error("At least one origin (s3Origins or httpOrigins) must be provided");
        }
        // Validate unique origin IDs
        this.validateOriginIds(props);
        // Apply intelligent defaults (unless disabled)
        const enhancedProps = props.disableIntelligentDefaults
            ? props
            : this.applyDefaults(props);
        let certificate;
        let logBucket;
        // Setup all S3 origins
        if (enhancedProps.s3Origins) {
            for (const s3Config of enhancedProps.s3Origins) {
                this.s3BucketsMap.set(s3Config.id, s3Config.bucket);
                const s3Origin = this.setupS3Origin(s3Config);
                this.s3OriginsMap.set(s3Config.id, s3Origin);
            }
        }
        // Setup all HTTP origins
        if (enhancedProps.httpOrigins) {
            for (const httpConfig of enhancedProps.httpOrigins) {
                const httpOrigin = this.setupHttpOrigin(httpConfig);
                this.httpOriginsMap.set(httpConfig.id, httpOrigin);
            }
        }
        // Setup custom domain and certificate
        if (enhancedProps.customDomainName) {
            certificate = this.setupCustomDomain(enhancedProps);
        }
        // Setup logging (enabled by default unless explicitly disabled)
        if (enhancedProps.enableLogging !== false) {
            logBucket = this.setupLogging(enhancedProps);
        }
        // Create CloudFront distribution
        this.distribution = this.createDistribution(enhancedProps, {
            certificate,
            logBucket,
        });
        // Setup Route53 records for all domains
        if (enhancedProps.createRoute53Records !== false &&
            enhancedProps.hostedZone) {
            this.setupRoute53Records(enhancedProps);
        }
        // Assign readonly properties
        this.logBucket = logBucket;
        this.certificate = certificate;
        this.domainNames = this.getAllDomainNames(enhancedProps);
    }
    /**
     * Get all domain names (primary + additional)
     */
    getAllDomainNames(props) {
        const domains = [];
        if (props.customDomainName) {
            domains.push(props.customDomainName);
        }
        if (props.additionalDomainNames) {
            domains.push(...props.additionalDomainNames);
        }
        return domains.length > 0 ? domains : undefined;
    }
    /**
     * Validate that all origin IDs are unique
     */
    validateOriginIds(props) {
        const allIds = new Set();
        // Check S3 origin IDs
        if (props.s3Origins) {
            for (const s3Origin of props.s3Origins) {
                if (allIds.has(s3Origin.id)) {
                    throw new Error(`Duplicate origin ID: ${s3Origin.id}`);
                }
                allIds.add(s3Origin.id);
            }
        }
        // Check HTTP origin IDs
        if (props.httpOrigins) {
            for (const httpOrigin of props.httpOrigins) {
                if (allIds.has(httpOrigin.id)) {
                    throw new Error(`Duplicate origin ID: ${httpOrigin.id}`);
                }
                allIds.add(httpOrigin.id);
            }
        }
        // Validate default origin ID exists
        if (props.defaultOriginId && !allIds.has(props.defaultOriginId)) {
            throw new Error(`Default origin ID '${props.defaultOriginId}' not found in origins`);
        }
        // Validate cache behavior origin IDs
        if (props.cacheBehaviors) {
            for (const behavior of props.cacheBehaviors) {
                if (!allIds.has(behavior.originId)) {
                    throw new Error(`Cache behavior origin ID '${behavior.originId}' not found in origins`);
                }
            }
        }
    }
    /**
     * Apply intelligent defaults based on origin configuration
     */
    applyDefaults(props) {
        const hasS3 = props.s3Origins && props.s3Origins.length > 0;
        const hasHttp = props.httpOrigins && props.httpOrigins.length > 0;
        let errorPages;
        // Only apply error page defaults if not disabled and not provided
        if (!props.disableDefaultErrorPages && !props.errorPages) {
            if (hasS3 && hasHttp) {
                // Hybrid: SPA-friendly error pages
                errorPages = this.getHybridErrorPages();
            }
            else if (hasS3) {
                // S3 only: SPA-friendly error pages
                errorPages = this.getS3ErrorPages();
            }
            else {
                // HTTP only: Basic error pages
                errorPages = this.getHttpErrorPages();
            }
        }
        return {
            // Apply base defaults
            enableLogging: true,
            createRoute53Records: true,
            defaultRootObject: "index.html",
            priceClass: cloudfront.PriceClass.PRICE_CLASS_100,
            enabled: true,
            // Override with user props
            ...props,
            // Apply error pages if generated
            ...(errorPages && { errorPages }),
            // Apply HTTP origin protocol defaults
            httpOrigins: props.httpOrigins?.map((httpOrigin) => ({
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
                ...httpOrigin,
            })),
            // Merge cache behaviors
            cacheBehaviors: [...(props.cacheBehaviors || [])],
        };
    }
    /**
     * Default error pages for S3-only distributions
     */
    getS3ErrorPages() {
        return [
            {
                httpStatus: 404,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
            {
                httpStatus: 403,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
        ];
    }
    /**
     * Default error pages for HTTP-only distributions
     */
    getHttpErrorPages() {
        return [
            {
                httpStatus: 504,
                responseHttpStatus: 504,
                responsePagePath: "/error.html",
                ttl: aws_cdk_lib_1.Duration.seconds(30),
            },
        ];
    }
    /**
     * Default error pages for hybrid distributions
     */
    getHybridErrorPages() {
        return [
            {
                httpStatus: 404,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
        ];
    }
    /**
     * Setup S3 origin
     */
    setupS3Origin(s3Config) {
        if (s3Config.useLegacyOAI) {
            // Use legacy Origin Access Identity
            const originAccessIdentity = s3Config.originAccessIdentity ||
                new cloudfront.OriginAccessIdentity(this, `OAI-${s3Config.id}`, {
                    comment: `OAI for ${s3Config.bucket.bucketName}`,
                });
            // Grant CloudFront access to S3 bucket
            s3Config.bucket.grantRead(originAccessIdentity);
            return new origins.S3Origin(s3Config.bucket, {
                originAccessIdentity,
                originPath: s3Config.originPath,
                ...s3Config.s3OriginProps,
            });
        }
        else {
            // Use modern Origin Access Control (recommended)
            return origins.S3BucketOrigin.withOriginAccessControl(s3Config.bucket, {
                originPath: s3Config.originPath,
                ...s3Config.s3OriginProps,
            });
        }
    }
    /**
     * Setup HTTP origin
     */
    setupHttpOrigin(httpConfig) {
        return new origins.HttpOrigin(httpConfig.domainName, {
            originPath: httpConfig.originPath,
            protocolPolicy: httpConfig.protocolPolicy || cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: httpConfig.httpPort,
            httpsPort: httpConfig.httpsPort,
            ...httpConfig.httpOriginProps,
        });
    }
    /**
     * Setup custom domain and certificate
     */
    setupCustomDomain(props) {
        // If certificate ARN is provided, use it directly
        if (props.certificateArn) {
            // Validate that certificate ARN is from us-east-1
            if (!props.certificateArn.includes("us-east-1")) {
                throw new Error("Certificate must be in us-east-1 region for CloudFront");
            }
            return acm.Certificate.fromCertificateArn(this, "Certificate", props.certificateArn);
        }
        // Auto-create certificate in us-east-1 if hostedZone is provided
        if (props.hostedZone && props.customDomainName) {
            return this.createCrossRegionCertificate(props);
        }
        throw new Error("Either certificateArn or hostedZone must be provided for custom domain. " +
            "If hostedZone is provided, certificate will be automatically created in us-east-1.");
    }
    /**
     * Create certificate in us-east-1 using custom resource
     */
    createCrossRegionCertificate(props) {
        const allDomains = this.getAllDomainNames(props) || [
            props.customDomainName,
        ];
        // Create a Lambda function to handle certificate creation in us-east-1
        const certHandler = new lambda.Function(this, "CertificateHandler", {
            runtime: lambda.Runtime.PYTHON_3_9,
            handler: "index.handler",
            code: lambda.Code.fromInline(`
import json
import boto3
import time
import logging
from typing import Dict, Any, Optional

# Configure logging
logger = logging.getLogger()
logger.setLevel(logging.INFO)

def handler(event: Dict[str, Any], context: Any) -> Dict[str, Any]:
    """
    Custom resource handler for creating ACM certificates in us-east-1
    """
    logger.info(f"Event: {json.dumps(event, indent=2)}")
    
    request_type = event.get('RequestType')
    resource_properties = event.get('ResourceProperties', {})
    
    domain_name = resource_properties.get('DomainName')
    subject_alternative_names = resource_properties.get('SubjectAlternativeNames')
    hosted_zone_id = resource_properties.get('HostedZoneId')
    
    # Initialize AWS clients
    acm_client = boto3.client('acm', region_name='us-east-1')
    route53_client = boto3.client('route53')
    
    if request_type in ['Create', 'Update']:
        try:
            # Prepare certificate request parameters
            cert_params = {
                'DomainName': domain_name,
                'ValidationMethod': 'DNS'
            }
            
            # Only add SubjectAlternativeNames if it has values
            if subject_alternative_names and len(subject_alternative_names) > 0:
                cert_params['SubjectAlternativeNames'] = subject_alternative_names
                logger.info(f"Including SANs: {subject_alternative_names}")
            
            logger.info(f"Requesting certificate with params: {cert_params}")
            
            # Request certificate
            cert_response = acm_client.request_certificate(**cert_params)
            certificate_arn = cert_response['CertificateArn']
            
            logger.info(f"Certificate ARN: {certificate_arn}")
            
            # Wait for DNS validation records to be available
            max_attempts = 60
            attempt = 0
            
            while attempt < max_attempts:
                try:
                    cert_details = acm_client.describe_certificate(
                        CertificateArn=certificate_arn
                    )
                    
                    validation_options = cert_details['Certificate'].get('DomainValidationOptions', [])
                    
                    if validation_options:
                        all_records_ready = True
                        
                        for option in validation_options:
                            if 'ResourceRecord' not in option:
                                all_records_ready = False
                                break
                        
                        if all_records_ready:
                            logger.info("All DNS validation records are ready")
                            
                            # Create DNS validation records
                            for option in validation_options:
                                resource_record = option['ResourceRecord']
                                
                                logger.info(f"Creating validation record: {resource_record}")
                                
                                route53_client.change_resource_record_sets(
                                    HostedZoneId=hosted_zone_id,
                                    ChangeBatch={
                                        'Changes': [{
                                            'Action': 'UPSERT',
                                            'ResourceRecordSet': {
                                                'Name': resource_record['Name'],
                                                'Type': resource_record['Type'],
                                                'TTL': 300,
                                                'ResourceRecords': [
                                                    {'Value': resource_record['Value']}
                                                ]
                                            }
                                        }]
                                    }
                                )
                            
                            logger.info("DNS validation records created successfully")
                            break
                    
                    attempt += 1
                    if attempt < max_attempts:
                        logger.info(f"Attempt {attempt}/{max_attempts}: Waiting for validation records...")
                        time.sleep(5)
                    
                except Exception as e:
                    logger.error(f"Error on attempt {attempt}: {str(e)}")
                    attempt += 1
                    if attempt < max_attempts:
                        time.sleep(5)
                    else:
                        raise
            
            if attempt >= max_attempts:
                raise Exception("Timeout waiting for DNS validation records")
            
            return {
                'PhysicalResourceId': certificate_arn,
                'Data': {
                    'CertificateArn': certificate_arn,
                    'DomainName': domain_name
                }
            }
            
        except Exception as e:
            logger.error(f"Error creating certificate: {str(e)}")
            raise
    
    elif request_type == 'Delete':
        # For delete, we just return success
        # ACM will handle certificate deletion automatically if not in use
        physical_resource_id = event.get('PhysicalResourceId', 'certificate-not-created')
        logger.info(f"Deleting certificate resource: {physical_resource_id}")
        
        return {
            'PhysicalResourceId': physical_resource_id
        }
    
    else:
        raise Exception(f"Unsupported request type: {request_type}")
      `),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
        });
        // Grant permissions to the Lambda function
        certHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                "acm:RequestCertificate",
                "acm:DescribeCertificate",
                "acm:DeleteCertificate",
                "route53:ChangeResourceRecordSets",
                "route53:GetChange",
            ],
            resources: ["*"],
        }));
        // Create custom resource
        const customResource = new aws_cdk_lib_1.CustomResource(this, "CertificateCustomResource", {
            serviceToken: certHandler.functionArn,
            properties: {
                DomainName: props.customDomainName,
                SubjectAlternativeNames: allDomains.length > 1 ? allDomains.slice(1) : undefined,
                HostedZoneId: props.hostedZone.hostedZoneId,
            },
        });
        // Return certificate reference
        return acm.Certificate.fromCertificateArn(this, "Certificate", customResource.getAttString("CertificateArn"));
    }
    /**
     * Setup CloudFront logging
     */
    setupLogging(props) {
        return (props.logBucket ||
            new s3.Bucket(this, "LogBucket", {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                versioned: false,
                lifecycleRules: [
                    {
                        expiration: aws_cdk_lib_1.Duration.days(90),
                        transitions: [
                            {
                                storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                                transitionAfter: aws_cdk_lib_1.Duration.days(30),
                            },
                        ],
                    },
                ],
            }));
    }
    /**
     * Get origin by ID
     */
    getOriginById(originId) {
        // Check S3 origins first
        const s3Origin = this.s3OriginsMap.get(originId);
        if (s3Origin) {
            return s3Origin;
        }
        // Check HTTP origins
        const httpOrigin = this.httpOriginsMap.get(originId);
        if (httpOrigin) {
            return httpOrigin;
        }
        throw new Error(`Origin with ID '${originId}' not found`);
    }
    /**
     * Determine default origin ID
     */
    getDefaultOriginId(props) {
        if (props.defaultOriginId) {
            return props.defaultOriginId;
        }
        // Default to first S3 origin, then first HTTP origin
        if (props.s3Origins && props.s3Origins.length > 0) {
            return props.s3Origins[0].id;
        }
        if (props.httpOrigins && props.httpOrigins.length > 0) {
            return props.httpOrigins[0].id;
        }
        throw new Error("No origins configured");
    }
    /**
     * Create CloudFront distribution
     */
    createDistribution(props, resources) {
        // Determine default origin
        const defaultOriginId = this.getDefaultOriginId(props);
        const defaultOrigin = this.getOriginById(defaultOriginId);
        const defaultBehavior = {
            origin: defaultOrigin,
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            compress: true,
        };
        // Build additional behaviors
        const additionalBehaviors = {};
        if (props.cacheBehaviors) {
            for (const behavior of props.cacheBehaviors) {
                const origin = this.getOriginById(behavior.originId);
                // Build behavior options object properly
                const behaviorConfig = {
                    origin,
                    viewerProtocolPolicy: behavior.viewerProtocolPolicy ||
                        cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                    allowedMethods: behavior.allowedMethods,
                    cachedMethods: behavior.cachedMethods,
                    compress: behavior.compress ?? true,
                };
                // Add cache policy (prioritize direct policy over ID)
                if (behavior.cachePolicy) {
                    behaviorConfig.cachePolicy = behavior.cachePolicy;
                }
                else if (behavior.cachePolicyId) {
                    behaviorConfig.cachePolicy = cloudfront.CachePolicy.fromCachePolicyId(this, `CachePolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.cachePolicyId);
                }
                // Add origin request policy
                if (behavior.originRequestPolicy) {
                    behaviorConfig.originRequestPolicy = behavior.originRequestPolicy;
                }
                else if (behavior.originRequestPolicyId) {
                    behaviorConfig.originRequestPolicy =
                        cloudfront.OriginRequestPolicy.fromOriginRequestPolicyId(this, `OriginRequestPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.originRequestPolicyId);
                }
                // Add response headers policy
                if (behavior.responseHeadersPolicy) {
                    behaviorConfig.responseHeadersPolicy = behavior.responseHeadersPolicy;
                }
                else if (behavior.responseHeadersPolicyId) {
                    behaviorConfig.responseHeadersPolicy =
                        cloudfront.ResponseHeadersPolicy.fromResponseHeadersPolicyId(this, `ResponseHeadersPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.responseHeadersPolicyId);
                }
                additionalBehaviors[behavior.pathPattern] =
                    behaviorConfig;
            }
        }
        // Prepare distribution props
        const distributionProps = {
            comment: props.comment || `CloudFront distribution for ${this.node.id}`,
            defaultBehavior,
            additionalBehaviors,
            domainNames: this.getAllDomainNames(props),
            certificate: resources.certificate,
            defaultRootObject: props.defaultRootObject,
            enabled: props.enabled,
            priceClass: props.priceClass,
            geoRestriction: props.geoRestriction,
            enableLogging: props.enableLogging !== false,
            logBucket: resources.logBucket,
            logFilePrefix: props.logPrefix,
            logIncludesCookies: props.logIncludeCookies,
            errorResponses: props.errorPages,
            webAclId: props.webAclId,
            enableIpv6: props.enableIpv6,
            httpVersion: props.httpVersion,
        };
        return new cloudfront.Distribution(this, "Distribution", distributionProps);
    }
    /**
     * Setup Route53 alias records for all domains
     */
    setupRoute53Records(props) {
        if (!props.hostedZone) {
            return;
        }
        const domains = this.getAllDomainNames(props);
        if (!domains) {
            return;
        }
        // Create A record for each domain
        domains.forEach((domain, index) => {
            const recordId = index === 0 ? "AliasRecord" : `AliasRecord${index + 1}`;
            const aRecord = new route53.ARecord(this, recordId, {
                zone: props.hostedZone,
                recordName: domain,
                target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
            });
            this.aRecords.push(aRecord);
        });
    }
    /**
     * Get S3 bucket by origin ID
     */
    getS3Bucket(originId) {
        return this.s3BucketsMap.get(originId);
    }
    /**
     * Get HTTP origin by origin ID
     */
    getHttpOrigin(originId) {
        return this.httpOriginsMap.get(originId);
    }
    /**
     * Get all S3 bucket origin IDs
     */
    get s3OriginIds() {
        return Array.from(this.s3BucketsMap.keys());
    }
    /**
     * Get all HTTP origin IDs
     */
    get httpOriginIds() {
        return Array.from(this.httpOriginsMap.keys());
    }
    /**
     * Get all S3 buckets as an array of objects with ID and bucket
     */
    get s3Origins() {
        return Array.from(this.s3BucketsMap.entries()).map(([id, bucket]) => ({
            id,
            bucket,
        }));
    }
    /**
     * Get all HTTP origins as an array of objects with ID and origin
     */
    get httpOrigins() {
        return Array.from(this.httpOriginsMap.entries()).map(([id, origin]) => ({
            id,
            origin,
        }));
    }
    /**
     * Get the CloudFront distribution domain name
     */
    get distributionDomainName() {
        return this.distribution.distributionDomainName;
    }
    /**
     * Get the CloudFront distribution URL with protocol
     */
    get distributionUrl() {
        return `https://${this.distribution.distributionDomainName}`;
    }
    /**
     * Get the custom domain URL (if configured)
     */
    get customDomainUrl() {
        return this.domainNames?.[0] ? `https://${this.domainNames[0]}` : undefined;
    }
}
exports.CloudFrontToOrigins = CloudFrontToOrigins;
//# sourceMappingURL=data:application/json;base64,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