"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsCodeDeploy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class EcsCodeDeploy extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.logGroup = this.createLogGroup();
        const { taskRole, taskExecRole } = this.createRole();
        this.taskRole = taskRole;
        this.taskExecutionRole = taskExecRole;
        this.taskDef = this.createTaskDefinition();
        this.service = this.createEcsService();
        this.loadBalancer = this.createApplicationLoadBalancer();
        const { blueTargetGroup, greenTargetGroup } = this.createTargetGroups();
        this.blueTargetGroup = blueTargetGroup;
        this.greenTargetGroup = greenTargetGroup;
        this.setupAutoScaling();
        this.codeDeployApp = this.createCodeDeployApplication();
        this.createCodeDeployDeploymentGroup();
    }
    createRole() {
        const taskRole = this.props.taskRole ??
            new aws_cdk_lib_1.aws_iam.Role(this, "TaskRole", {
                roleName: `${this.props.serviceName}-task-role`,
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
                description: `Task role for ${this.props.serviceName}`,
            });
        const taskExecRole = this.props.taskExecRole ??
            new aws_cdk_lib_1.aws_iam.Role(this, "TaskExecRole", {
                roleName: `${this.props.serviceName}-task-exec-role`,
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
                description: `Task execution role for ${this.props.serviceName}`,
                managedPolicies: [
                    aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonECSTaskExecutionRolePolicy"),
                ],
            });
        return { taskRole, taskExecRole };
    }
    createLogGroup() {
        const { serviceName } = this.props;
        return new aws_cdk_lib_1.aws_logs.LogGroup(this, "LogGroup", {
            logGroupName: `/aws/ecs/${serviceName}`,
            retention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
    }
    createTaskDefinition() {
        const { taskCPU = 1024, memoryLimit = 2048, containers } = this.props;
        const taskDef = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, "TaskDefinition", {
            memoryLimitMiB: memoryLimit,
            cpu: taskCPU,
            executionRole: this.taskExecutionRole,
            taskRole: this.taskRole,
        });
        // Add containers to task definition
        this.addContainersToTaskDefinition(taskDef, containers);
        return taskDef;
    }
    addContainersToTaskDefinition(taskDef, containers) {
        containers.forEach((container, index) => {
            taskDef.addContainer(`Container${index}`, {
                image: container.image,
                memoryReservationMiB: container.memoryLimit ?? 1024,
                logging: aws_cdk_lib_1.aws_ecs.LogDriver.awsLogs({
                    streamPrefix: "ecs",
                    logGroup: this.logGroup,
                }),
                portMappings: [{ containerPort: container.containerPort }],
                healthCheck: container.healthCheck ?? {
                    command: [
                        "CMD-SHELL",
                        `curl -f http://localhost:${container.containerPort} || exit 1`,
                    ],
                    interval: aws_cdk_lib_1.Duration.seconds(30),
                    timeout: aws_cdk_lib_1.Duration.seconds(5),
                },
            });
        });
    }
    createEcsService() {
        const { cluster, serviceName, subnets, securityGroups } = this.props;
        return new aws_cdk_lib_1.aws_ecs.FargateService(this, "Service", {
            cluster,
            serviceName: `${serviceName}-service`,
            taskDefinition: this.taskDef,
            assignPublicIp: true,
            desiredCount: 1,
            deploymentController: {
                type: aws_cdk_lib_1.aws_ecs.DeploymentControllerType.CODE_DEPLOY,
            },
            healthCheckGracePeriod: aws_cdk_lib_1.Duration.seconds(300),
            vpcSubnets: subnets,
            securityGroups: securityGroups,
        });
    }
    createApplicationLoadBalancer() {
        const { vpc, subnets, enablePublicLoadBalancer = false } = this.props;
        const alb = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer(this, "LoadBalancer", {
            vpc,
            loadBalancerName: `${this.props.serviceName}-alb`,
            internetFacing: enablePublicLoadBalancer,
            vpcSubnets: subnets,
        });
        // Add HTTP to HTTPS redirect
        alb.addRedirect();
        return alb;
    }
    createTargetGroups() {
        const { certificates, containers } = this.props;
        const primaryContainer = containers[0];
        const targetPort = this.props.albTargetPort ?? primaryContainer.containerPort;
        // Create HTTPS listeners
        const httpsBlueListener = this.loadBalancer.addListener("BlueHttpsListener", {
            port: 443,
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTPS,
            certificates: certificates,
        });
        const httpsGreenListener = this.loadBalancer.addListener("GreenHttpsListener", {
            port: 8080,
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTPS,
            certificates: certificates,
        });
        // Health check configuration
        const healthCheck = {
            path: "/",
            interval: aws_cdk_lib_1.Duration.seconds(30),
            timeout: aws_cdk_lib_1.Duration.seconds(10),
            healthyThresholdCount: 2,
            unhealthyThresholdCount: 5,
        };
        // Create target groups
        const blueTargetGroup = httpsBlueListener.addTargets("BlueTarget", {
            port: targetPort,
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            targets: [this.service],
            healthCheck: healthCheck,
            stickinessCookieDuration: aws_cdk_lib_1.Duration.days(1),
        });
        const greenTargetGroup = httpsGreenListener.addTargets("GreenTarget", {
            port: targetPort,
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            targets: [], // Empty for green deployment
            healthCheck: healthCheck,
            stickinessCookieDuration: aws_cdk_lib_1.Duration.days(1),
        });
        return { blueTargetGroup, greenTargetGroup };
    }
    setupAutoScaling() {
        const { autoScaling } = this.props;
        if (!autoScaling) {
            return;
        }
        const scalableTarget = this.service.autoScaleTaskCount({
            minCapacity: autoScaling.minCapacity,
            maxCapacity: autoScaling.maxCapacity,
        });
        // Setup CPU-based scaling if configured
        if (autoScaling.cpuScale) {
            scalableTarget.scaleOnCpuUtilization("CpuScaling", autoScaling.cpuScale);
        }
        // Setup memory-based scaling if configured
        if (autoScaling.memoryScale) {
            scalableTarget.scaleOnMemoryUtilization("MemoryScaling", autoScaling.memoryScale);
        }
    }
    createCodeDeployApplication() {
        const { serviceName } = this.props;
        return new aws_cdk_lib_1.aws_codedeploy.EcsApplication(this, "CodeDeployApp", {
            applicationName: serviceName,
        });
    }
    createCodeDeployDeploymentGroup() {
        const { serviceName } = this.props;
        return new aws_cdk_lib_1.aws_codedeploy.EcsDeploymentGroup(this, "DeploymentGroup", {
            deploymentGroupName: serviceName,
            application: this.codeDeployApp,
            service: this.service,
            blueGreenDeploymentConfig: {
                blueTargetGroup: this.blueTargetGroup,
                greenTargetGroup: this.greenTargetGroup,
                listener: this.loadBalancer.listeners[0],
            },
        });
    }
    blueListener() {
        return (this.loadBalancer.listeners.find((listener) => listener.listenerArn.includes("BlueHttpsListener")) || this.loadBalancer.listeners[0]);
    }
    greenListener() {
        return (this.loadBalancer.listeners.find((listener) => listener.listenerArn.includes("GreenHttpsListener")) || this.loadBalancer.listeners[1]);
    }
    allListeners() {
        return this.loadBalancer.listeners;
    }
    loadBalancerDnsName() {
        return this.loadBalancer.loadBalancerDnsName;
    }
    serviceArn() {
        return this.service.serviceArn;
    }
}
exports.EcsCodeDeploy = EcsCodeDeploy;
_a = JSII_RTTI_SYMBOL_1;
EcsCodeDeploy[_a] = { fqn: "must-cdk.EcsCodeDeploy", version: "0.0.53" };
//# sourceMappingURL=data:application/json;base64,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