#!/usr/bin/env python3
"""
Python Implementation of OpenFace 2.2 CalcParams Function

Replicates the PDM::CalcParams() function from OpenFace 2.2's PDM.cpp
for calculating optimal 3D pose parameters (6 global + 34 local) from 2D landmarks.

Reference: OpenFace/lib/local/LandmarkDetector/src/PDM.cpp lines 508-705

This implements iterative optimization using Jacobian and Hessian to fit
a 3D Point Distribution Model (PDM) to detected 2D facial landmarks.

Author: Replicated from OpenFace 2.2 C++ implementation
Date: 2025-10-29
"""

import numpy as np
from scipy import linalg
import cv2

# Try to import Cython-optimized rotation update for 99.9% accuracy
try:
    import sys
    from pathlib import Path
    # Add parent directory to path so we can import cython extensions
    sys.path.insert(0, str(Path(__file__).parent.parent))
    from cython_rotation_update import update_rotation_cython
    CYTHON_AVAILABLE = True
    print("Cython rotation update module loaded - targeting 99.9% accuracy")
except ImportError:
    CYTHON_AVAILABLE = False
    print("Warning: Cython rotation update not available - using Python (99.45% accuracy)")

# Try to import Numba JIT-accelerated CalcParams functions for 2-5x speedup
try:
    from pyfaceau.alignment.numba_calcparams_accelerator import (
        compute_jacobian_accelerated,
        project_shape_to_2d_jit,
        euler_to_rotation_matrix_jit
    )
    NUMBA_AVAILABLE = True
except ImportError:
    NUMBA_AVAILABLE = False
    print("Warning: Numba JIT accelerator not available - using standard Python (slower)")


class CalcParams:
    """
    Calculate optimal PDM parameters from 2D landmarks

    Implements the CalcParams algorithm from OpenFace 2.2 which optimizes:
    - 6 global parameters: scale, rx, ry, rz, tx, ty
    - 34 local parameters: PCA shape coefficients

    Uses iterative Gauss-Newton optimization with Jacobian and Hessian.
    """

    def __init__(self, pdm_parser):
        """
        Initialize CalcParams with PDM model

        Args:
            pdm_parser: PDMParser instance containing mean_shape, princ_comp, eigen_values
        """
        self.pdm = pdm_parser
        self.mean_shape = pdm_parser.mean_shape  # (204, 1)
        self.princ_comp = pdm_parser.princ_comp  # (204, 34)
        self.eigen_values = pdm_parser.eigen_values  # (34,)

    @staticmethod
    def euler_to_rotation_matrix(euler_angles):
        """
        Convert Euler angles to 3x3 rotation matrix

        Uses XYZ convention: R = Rx * Ry * Rz (left-handed positive sign)
        Matches Utilities::Euler2RotationMatrix() from RotationHelpers.h

        Args:
            euler_angles: (rx, ry, rz) in radians

        Returns:
            3x3 rotation matrix
        """
        rx, ry, rz = euler_angles

        s1, s2, s3 = np.sin(rx), np.sin(ry), np.sin(rz)
        c1, c2, c3 = np.cos(rx), np.cos(ry), np.cos(rz)

        R = np.array([
            [c2 * c3, -c2 * s3, s2],
            [c1 * s3 + c3 * s1 * s2, c1 * c3 - s1 * s2 * s3, -c2 * s1],
            [s1 * s3 - c1 * c3 * s2, c3 * s1 + c1 * s2 * s3, c1 * c2]
        ], dtype=np.float32)

        return R

    @staticmethod
    def rotation_matrix_to_euler(R):
        """
        Convert 3x3 rotation matrix to Euler angles using robust quaternion extraction

        Matches RotationMatrix2Euler() from RotationHelpers.h
        Uses Shepperd's method for robust quaternion extraction (handles all cases)

        Args:
            R: 3x3 rotation matrix

        Returns:
            (rx, ry, rz) Euler angles in radians
        """
        # Robust quaternion extraction using Shepperd's method
        # This handles all rotation cases without singularities
        trace = R[0,0] + R[1,1] + R[2,2]

        if trace > 0:
            # Standard case: trace is positive
            s = np.sqrt(trace + 1.0) * 2.0  # s = 4*q0
            q0 = 0.25 * s
            q1 = (R[2,1] - R[1,2]) / s
            q2 = (R[0,2] - R[2,0]) / s
            q3 = (R[1,0] - R[0,1]) / s
        elif (R[0,0] > R[1,1]) and (R[0,0] > R[2,2]):
            # q1 is largest component
            s = np.sqrt(1.0 + R[0,0] - R[1,1] - R[2,2]) * 2.0  # s = 4*q1
            q0 = (R[2,1] - R[1,2]) / s
            q1 = 0.25 * s
            q2 = (R[0,1] + R[1,0]) / s
            q3 = (R[0,2] + R[2,0]) / s
        elif R[1,1] > R[2,2]:
            # q2 is largest component
            s = np.sqrt(1.0 + R[1,1] - R[0,0] - R[2,2]) * 2.0  # s = 4*q2
            q0 = (R[0,2] - R[2,0]) / s
            q1 = (R[0,1] + R[1,0]) / s
            q2 = 0.25 * s
            q3 = (R[1,2] + R[2,1]) / s
        else:
            # q3 is largest component
            s = np.sqrt(1.0 + R[2,2] - R[0,0] - R[1,1]) * 2.0  # s = 4*q3
            q0 = (R[1,0] - R[0,1]) / s
            q1 = (R[0,2] + R[2,0]) / s
            q2 = (R[1,2] + R[2,1]) / s
            q3 = 0.25 * s

        # Quaternion to Euler angles
        t1 = 2.0 * (q0*q2 + q1*q3)
        t1 = np.clip(t1, -1.0, 1.0)  # Handle precision issues

        yaw = np.arcsin(t1)
        pitch = np.arctan2(2.0 * (q0*q1 - q2*q3), q0*q0 - q1*q1 - q2*q2 + q3*q3)
        roll = np.arctan2(2.0 * (q0*q3 - q1*q2), q0*q0 + q1*q1 - q2*q2 - q3*q3)

        return np.array([pitch, yaw, roll], dtype=np.float32)

    @staticmethod
    def rotation_matrix_to_axis_angle(R):
        """Convert rotation matrix to axis-angle representation using Rodrigues"""
        axis_angle, _ = cv2.Rodrigues(R)
        return axis_angle.flatten()

    @staticmethod
    def axis_angle_to_euler(axis_angle):
        """Convert axis-angle to Euler angles via rotation matrix"""
        R, _ = cv2.Rodrigues(axis_angle)
        return CalcParams.rotation_matrix_to_euler(R)

    @staticmethod
    def orthonormalise(R):
        """
        Ensure rotation matrix is orthonormal

        Matches PDM::Orthonormalise() from PDM.cpp lines 59-76
        Uses SVD to project matrix onto SO(3)

        Args:
            R: 3x3 matrix (possibly not perfectly orthonormal)

        Returns:
            3x3 orthonormal rotation matrix
        """
        U, S, Vt = np.linalg.svd(R)

        # Ensure no reflection (determinant = 1, not -1)
        W = np.eye(3, dtype=np.float32)
        d = np.linalg.det(U @ Vt)
        W[2, 2] = d

        R_ortho = U @ W @ Vt
        return R_ortho.astype(np.float32)

    def calc_shape_3d(self, params_local):
        """
        Calculate 3D shape from local parameters

        Matches PDM::CalcShape3D()

        Args:
            params_local: (34,) PCA coefficients

        Returns:
            shape_3d: (204,) 3D coordinates [X0...X67, Y0...Y67, Z0...Z67]
        """
        params_local = params_local.reshape(-1, 1)
        shape_3d = self.mean_shape + self.princ_comp @ params_local
        return shape_3d.flatten()

    def extract_bounding_box(self, landmarks_2d):
        """
        Extract bounding box from 2D landmarks

        Args:
            landmarks_2d: (n*2,) array [X0...Xn, Y0...Yn]

        Returns:
            (min_x, max_x, min_y, max_y)
        """
        n = len(landmarks_2d) // 2
        x_coords = landmarks_2d[:n]
        y_coords = landmarks_2d[n:]

        return np.min(x_coords), np.max(x_coords), np.min(y_coords), np.max(y_coords)

    def calc_bounding_box_model(self):
        """
        Calculate bounding box of mean PDM shape at identity pose

        Returns:
            (width, height) of model bounding box
        """
        # Get mean shape at identity pose (scale=1, rotation=0, translation=0)
        mean_3d = self.mean_shape.flatten()  # (204,)
        n = len(mean_3d) // 3

        # Project to 2D with identity transformation (just take X, Y)
        x_coords = mean_3d[:n]
        y_coords = mean_3d[n:2*n]

        width = np.max(x_coords) - np.min(x_coords)
        height = np.max(y_coords) - np.min(y_coords)

        return width, height

    def compute_jacobian(self, params_local, params_global, weight_matrix):
        """
        Compute Jacobian matrix for optimization

        Matches PDM::ComputeJacobian() from PDM.cpp lines 346-449

        Jacobian has shape (n*2, 6+m) where:
        - n = number of landmarks
        - 6 = global params (scale, rx, ry, rz, tx, ty)
        - m = number of local params (34 PCA coefficients)

        Args:
            params_local: (34,) local parameters
            params_global: (6,) global parameters [scale, rx, ry, rz, tx, ty]
            weight_matrix: (n*2, n*2) diagonal weight matrix

        Returns:
            J: (n*2, 6+m) Jacobian matrix
            J_w_t: (6+m, n*2) weighted Jacobian transpose
        """
        # Use Numba JIT-accelerated version if available (2-5x speedup!)
        if NUMBA_AVAILABLE:
            n_vis = self.mean_shape.shape[0] // 3
            m = 34
            return compute_jacobian_accelerated(
                params_local, params_global, self.princ_comp,
                self.mean_shape, weight_matrix, n_vis, m
            )

        # Fall back to Python implementation (slower)
        n = 68  # number of landmarks
        m = 34  # number of PCA modes

        # Extract global parameters
        s = params_global[0]  # scale
        euler = params_global[1:4]  # rotation (rx, ry, rz)

        # Get 3D shape from current local parameters
        shape_3d = self.calc_shape_3d(params_local)  # (204,)
        shape_3d = shape_3d.reshape(3, n).astype(np.float32)  # (3, 68) -> [X, Y, Z] rows

        # Get rotation matrix (force float32 for consistency with C++)
        R = self.euler_to_rotation_matrix(euler).astype(np.float32)  # (3, 3)
        r11, r12, r13 = R[0, :]
        r21, r22, r23 = R[1, :]
        r31, r32, r33 = R[2, :]

        # Initialize Jacobian
        J = np.zeros((n * 2, 6 + m), dtype=np.float32)

        # Iterate over each landmark
        for i in range(n):
            X, Y, Z = shape_3d[:, i]

            # Row for x coordinate
            row_x = i
            # Row for y coordinate
            row_y = i + n

            # --- Global parameter derivatives ---

            # Scaling term (column 0)
            J[row_x, 0] = X * r11 + Y * r12 + Z * r13
            J[row_y, 0] = X * r21 + Y * r22 + Z * r23

            # Rotation terms (columns 1-3: rx, ry, rz)
            # Using small angle approximation for rotation Jacobian
            J[row_x, 1] = s * (Y * r13 - Z * r12)
            J[row_y, 1] = s * (Y * r23 - Z * r22)

            J[row_x, 2] = -s * (X * r13 - Z * r11)
            J[row_y, 2] = -s * (X * r23 - Z * r21)

            J[row_x, 3] = s * (X * r12 - Y * r11)
            J[row_y, 3] = s * (X * r22 - Y * r21)

            # Translation terms (columns 4-5: tx, ty)
            J[row_x, 4] = 1.0
            J[row_y, 4] = 0.0

            J[row_x, 5] = 0.0
            J[row_y, 5] = 1.0

            # --- Local parameter derivatives ---
            # Columns 6 to 6+m-1
            for j in range(m):
                # Get principal component for this mode at this landmark
                Vx = self.princ_comp[i, j]          # princ_comp is (204, 34), row i corresponds to X coordinate
                Vy = self.princ_comp[i + n, j]      # row i+n corresponds to Y coordinate
                Vz = self.princ_comp[i + 2*n, j]    # row i+2*n corresponds to Z coordinate

                # Derivative of 2D projection w.r.t. local parameter j
                J[row_x, 6 + j] = s * (r11*Vx + r12*Vy + r13*Vz)
                J[row_y, 6 + j] = s * (r21*Vx + r22*Vy + r23*Vz)

        # Apply weighting (matches PDM.cpp lines 422-445)
        if not np.allclose(np.diag(weight_matrix), 1.0):
            J_weighted = J.copy()
            for i in range(n):
                w_x = weight_matrix[i, i]
                w_y = weight_matrix[i + n, i + n]
                J_weighted[i, :] *= w_x
                J_weighted[i + n, :] *= w_y
            J_w_t = J_weighted.T
        else:
            J_w_t = J.T

        return J, J_w_t

    def update_model_parameters(self, delta_p, params_local, params_global):
        """
        Update model parameters using computed delta

        Matches PDM::UpdateModelParameters() from PDM.cpp lines 454-505

        Args:
            delta_p: (6+m,) parameter update
            params_local: (34,) current local parameters
            params_global: (6,) current global parameters

        Returns:
            updated_local: (34,) updated local parameters
            updated_global: (6,) updated global parameters
        """
        # Clone parameters
        updated_global = params_global.copy()
        updated_local = params_local.copy()

        # Scaling and translation can be directly added
        updated_global[0] += delta_p[0]  # scale
        updated_global[4] += delta_p[4]  # tx
        updated_global[5] += delta_p[5]  # ty

        # Rotation update is more complex (uses rotation composition)
        if CYTHON_AVAILABLE:
            # Use Cython-optimized rotation update for 99.9% accuracy
            # C-level math guarantees bit-for-bit precision
            euler_current = params_global[1:4].astype(np.float32)
            delta_rotation = delta_p[1:4].astype(np.float32)
            euler_new = update_rotation_cython(euler_current, delta_rotation)
            updated_global[1:4] = euler_new
        else:
            # Fallback to Python implementation (99.45% accuracy)
            # Get current rotation matrix
            euler_current = params_global[1:4]
            R1 = self.euler_to_rotation_matrix(euler_current)

            # Construct incremental rotation matrix R'
            # R' = [1,   -wz,   wy ]
            #      [wz,   1,   -wx ]
            #      [-wy,  wx,   1  ]
            R2 = np.eye(3, dtype=np.float32)
            R2[1, 2] = -delta_p[1]  # -wx
            R2[2, 1] = delta_p[1]   # wx
            R2[2, 0] = -delta_p[2]  # -wy
            R2[0, 2] = delta_p[2]   # wy
            R2[0, 1] = -delta_p[3]  # -wz
            R2[1, 0] = delta_p[3]   # wz

            # Orthonormalise R2
            R2 = self.orthonormalise(R2)

            # Combine rotations
            R3 = R1 @ R2

            # Convert back to Euler angles using quaternion (matching C++ RotationHelpers.h)
            # C++ uses: RotationMatrix2AxisAngle then AxisAngle2Euler (via quaternion)
            # Direct quaternion conversion matches C++ better than axis-angle via OpenCV
            euler_new = self.rotation_matrix_to_euler(R3)

            # Handle numerical instability
            if np.any(np.isnan(euler_new)):
                euler_new = np.zeros(3, dtype=np.float32)

            updated_global[1:4] = euler_new

        # Update local parameters (simple addition)
        if len(delta_p) > 6:
            updated_local += delta_p[6:]

        return updated_local, updated_global

    def calc_params(self, landmarks_2d, rotation_init=None):
        """
        Calculate optimal PDM parameters from 2D landmarks

        Main function matching PDM::CalcParams() from PDM.cpp lines 508-705

        Args:
            landmarks_2d: (136,) array of 2D landmarks [X0...X67, Y0...Y67]
                         OR (68, 2) array of landmarks
            rotation_init: Initial rotation (rx, ry, rz). Defaults to (0, 0, 0)

        Returns:
            params_global: (6,) optimized global parameters [scale, rx, ry, rz, tx, ty]
            params_local: (34,) optimized local parameters
        """
        # Convert landmarks to (136,) format if needed
        if landmarks_2d.shape == (68, 2):
            landmarks_2d = np.concatenate([landmarks_2d[:, 0], landmarks_2d[:, 1]])

        n = 68  # number of landmarks
        m = 34  # number of PCA modes

        if rotation_init is None:
            rotation_init = np.zeros(3, dtype=np.float32)

        # Filter invisible landmarks (x coordinate == 0)
        visi_ind_2d = np.ones(n * 2, dtype=bool)
        visi_ind_3d = np.ones(n * 3, dtype=bool)

        for i in range(n):
            if landmarks_2d[i] == 0:  # invisible landmark
                visi_ind_2d[i] = False
                visi_ind_2d[i + n] = False
                visi_ind_3d[i] = False
                visi_ind_3d[i + n] = False
                visi_ind_3d[i + 2*n] = False

        # Subsample mean shape and principal components for visible landmarks
        # Use .copy() to prevent modifying shared PDM state
        M = self.mean_shape[visi_ind_3d].reshape(-1, 1).copy()
        V = self.princ_comp[visi_ind_3d, :].copy()

        # Extract visible landmarks
        landmarks_vis = landmarks_2d[visi_ind_2d]

        # Number of visible points
        n_vis = np.sum(visi_ind_3d) // 3

        # Compute initial global parameters from bounding box
        min_x, max_x, min_y, max_y = self.extract_bounding_box(landmarks_vis)
        width = abs(max_x - min_x)
        height = abs(max_y - min_y)

        model_width, model_height = self.calc_bounding_box_model()

        scaling = ((width / model_width) + (height / model_height)) / 2.0

        R = self.euler_to_rotation_matrix(rotation_init)
        translation = np.array([(min_x + max_x) / 2.0, (min_y + max_y) / 2.0], dtype=np.float32)

        # Initialize parameters
        params_local = np.zeros(m, dtype=np.float32)
        params_global = np.array([
            scaling,
            rotation_init[0],
            rotation_init[1],
            rotation_init[2],
            translation[0],
            translation[1]
        ], dtype=np.float32)

        # CRITICAL FIX: Use isolated PDM copies instead of modifying shared state
        # This prevents corruption of geometric features that read PDM concurrently
        # Save original PDM state
        mean_shape_original = self.mean_shape
        princ_comp_original = self.princ_comp

        # Temporarily update PDM to use subsampled matrices (isolated copies)
        self.mean_shape = M
        self.princ_comp = V

        try:
            # Initial projection error
            shape_3d = M.flatten() + (V @ params_local.reshape(-1, 1)).flatten()
            shape_3d = shape_3d.reshape(3, n_vis).astype(np.float32)

            # Use Numba JIT for 2D projection if available
            if NUMBA_AVAILABLE:
                curr_shape_2d = project_shape_to_2d_jit(shape_3d, R, scaling, translation[0], translation[1], n_vis)
            else:
                curr_shape_2d = np.zeros(n_vis * 2, dtype=np.float32)
                for i in range(n_vis):
                    X, Y, Z = shape_3d[:, i]
                    curr_shape_2d[i] = scaling * (R[0,0]*X + R[0,1]*Y + R[0,2]*Z) + translation[0]
                    curr_shape_2d[i + n_vis] = scaling * (R[1,0]*X + R[1,1]*Y + R[1,2]*Z) + translation[1]

            curr_error = np.linalg.norm(curr_shape_2d - landmarks_vis)

            # Regularization (inverse of eigenvalues)
            # Using 1.0 to match C++ baseline (was 10.0 - too much!)
            reg_factor = 1.0
            regularisation = np.zeros(6 + m, dtype=np.float32)
            regularisation[6:] = reg_factor / self.eigen_values
            regularisation = np.diag(regularisation)

            # Weight matrix (identity for now)
            weight_matrix = np.eye(n_vis * 2, dtype=np.float32)

            # Iterative optimization (up to 1000 iterations)
            not_improved_in = 0
            max_iterations = 1000

            for iteration in range(max_iterations):
                # Get current 3D shape
                shape_3d = self.calc_shape_3d(params_local)
                shape_3d = shape_3d.reshape(3, n_vis).astype(np.float32)

                # Get current rotation
                R = self.euler_to_rotation_matrix(params_global[1:4]).astype(np.float32)
                s = params_global[0]
                t = params_global[4:6]

                # Project to 2D using Numba JIT if available
                if NUMBA_AVAILABLE:
                    curr_shape_2d = project_shape_to_2d_jit(shape_3d, R, s, t[0], t[1], n_vis)
                else:
                    curr_shape_2d = np.zeros(n_vis * 2, dtype=np.float32)
                    for i in range(n_vis):
                        X, Y, Z = shape_3d[:, i]
                        curr_shape_2d[i] = s * (R[0,0]*X + R[0,1]*Y + R[0,2]*Z) + t[0]
                        curr_shape_2d[i + n_vis] = s * (R[1,0]*X + R[1,1]*Y + R[1,2]*Z) + t[1]

                # Compute error residual
                error_resid = (landmarks_vis - curr_shape_2d).astype(np.float32)

                # Compute Jacobian
                J, J_w_t = self.compute_jacobian(params_local, params_global, weight_matrix)

                # Compute J_w_t * error_resid
                J_w_t_m = J_w_t @ error_resid

                # Add regularization term for local parameters
                J_w_t_m[6:] = J_w_t_m[6:] - (regularisation[6:, 6:] @ params_local.reshape(-1, 1)).flatten()

                # Compute Hessian: J_w_t * J + regularisation
                Hessian = J_w_t @ J + regularisation

                # Adaptive Tikhonov regularization for ill-conditioned matrices
                # Use OpenCV's Cholesky solver to match C++ exactly (cv::solve with DECOMP_CHOLESKY)
                try:
                    # Convert to OpenCV format (needs float64 for numerical stability)
                    Hessian_cv = Hessian.astype(np.float64)
                    J_w_t_m_cv = J_w_t_m.reshape(-1, 1).astype(np.float64)

                    # Try OpenCV's Cholesky first (matches C++ line 657 exactly)
                    success, param_update_cv = cv2.solve(
                        Hessian_cv,
                        J_w_t_m_cv,
                        flags=cv2.DECOMP_CHOLESKY
                    )

                    if success:
                        param_update = param_update_cv.flatten().astype(np.float32)
                    else:
                        # If OpenCV Cholesky fails, add adaptive Tikhonov and retry
                        tikhonov_lambda = 1e-6 * np.mean(np.diag(Hessian_cv))
                        Hessian_stable = Hessian_cv + np.eye(Hessian_cv.shape[0]) * tikhonov_lambda

                        success, param_update_cv = cv2.solve(
                            Hessian_stable,
                            J_w_t_m_cv,
                            flags=cv2.DECOMP_CHOLESKY
                        )

                        if success:
                            param_update = param_update_cv.flatten().astype(np.float32)
                        else:
                            # Last resort: scipy lstsq
                            param_update = np.linalg.lstsq(Hessian, J_w_t_m, rcond=1e-6)[0].astype(np.float32)

                except Exception as e:
                    # Fallback to scipy if OpenCV fails for any reason
                    try:
                        param_update = linalg.solve(Hessian, J_w_t_m, assume_a='pos')
                    except np.linalg.LinAlgError:
                        param_update = np.linalg.lstsq(Hessian, J_w_t_m, rcond=1e-6)[0].astype(np.float32)

                # Reduce step size to avoid overshoot
                param_update *= 0.75

                # Update parameters
                params_local, params_global = self.update_model_parameters(
                    param_update, params_local, params_global
                )

                # Compute new error
                shape_3d = self.calc_shape_3d(params_local)
                shape_3d = shape_3d.reshape(3, n_vis)
                R = self.euler_to_rotation_matrix(params_global[1:4])
                s = params_global[0]
                t = params_global[4:6]

                curr_shape_2d = np.zeros(n_vis * 2, dtype=np.float32)
                for i in range(n_vis):
                    X, Y, Z = shape_3d[:, i]
                    curr_shape_2d[i] = s * (R[0,0]*X + R[0,1]*Y + R[0,2]*Z) + t[0]
                    curr_shape_2d[i + n_vis] = s * (R[1,0]*X + R[1,1]*Y + R[1,2]*Z) + t[1]

                new_error = np.linalg.norm(curr_shape_2d - landmarks_vis)

                # Check for improvement
                if 0.999 * curr_error < new_error:
                    not_improved_in += 1
                    if not_improved_in == 3:
                        break
                else:
                    not_improved_in = 0

                curr_error = new_error

        finally:
            # Restore original PDM matrices (critical for thread safety)
            self.mean_shape = mean_shape_original
            self.princ_comp = princ_comp_original

        # DO NOT normalize - C++ OpenFace outputs unnormalized params!
        # (Previous assumption was wrong - C++ does NOT divide by sqrt(eigenvalue))
        # params_local_normalized = params_local / np.sqrt(self.eigen_values)

        return params_global, params_local


def test_calc_params():
    """Test CalcParams implementation"""
    print("=" * 80)
    print("CalcParams Implementation Test")
    print("=" * 80)

    # This would require loading a PDM and testing
    # For now, just verify the module loads
    print("\nCalcParams module loaded successfully")
    print("Ready to use with PDMParser")

    print("\n" + "=" * 80)


if __name__ == "__main__":
    test_calc_params()
