import asyncio
import tempfile
from pathlib import Path
from typing import Any
from zipfile import ZipFile

import httpx
import polars as pl
import requests
from tqdm import tqdm

from libsms.data_model import EcoliExperiment, SimulationRun

__all__ = ["analysis_manifest", "analysis_output", "ecoli_experiment", "observables_data", "simulation_status", "simulation_log"]


def ecoli_experiment(
    config_id: str, max_retries: int = 20, delay_s: float = 1.0, verbose: bool = False, **body: dict[str, Any]
) -> EcoliExperiment | None:
    """Run a SMS API vEcoli simulation workflow.

    :param config_id: (str) Configuration ID of desired simulation experiment workflow. Defaults to "sms" (single cell)
    :param max_retries: (int) Maximum number of times to retry the workflow before giving up. Defaults to 20.
    :param delay_s: (float) Delay time between retries in seconds. Defaults to 1.0.
    :param verbose: (bool) Verbose mode. If set to ``True``, log print statements. Defaults to False.
    :param body: (kwargs/dict) Key/value pairs including: "overrides": {"config": {...}}, "variants": {"config": {...}}

    :rtype: EcoliExperiment
    :return: EcoliExperiment object with queriable experiment_id and experiment_tag
    """
    return asyncio.run(run_simulation(config_id, max_retries, delay_s, verbose, **body))


def simulation_status(
    experiment: EcoliExperiment, max_retries: int = 20, delay_s: float = 1.0, verbose: bool = False
) -> SimulationRun | None:
    """Run a SMS API vEcoli simulation workflow.

    :param experiment: (EcoliExperiment) Experiment generated by run_simulation.
    :param max_retries: (int) Maximum number of times to retry the workflow before giving up. Defaults to 20.
    :param delay_s: (float) Delay time between retries in seconds. Defaults to 1.0.
    :param verbose: (bool) Verbose mode. If set to ``True``, log print statements. Defaults to False.

    :rtype: SimulationRun
    :return: SimulationRun confirming run status (status will be one of "waiting", "running", "completed", "failed"
    """
    return asyncio.run(check_simulation_status(experiment, max_retries, delay_s, verbose))


def analysis_manifest(
    experiment: EcoliExperiment, max_retries: int = 20, delay_s: float = 1.0, verbose: bool = False
) -> dict[str, Any] | None | Any:
    return asyncio.run(get_analysis_manifest(experiment, max_retries, delay_s, verbose))


def analysis_output(
    experiment: EcoliExperiment,
    filename: str,
    variant: int = 0,
    lineage_seed: int = 0,
    generation: int = 1,
    agent_id: int = 0,
    max_retries: int = 20,
    delay_s: float = 1.0,
    verbose: bool = False,
) -> None | Any:
    return asyncio.run(
        download_analysis_output(
            experiment, filename, variant, lineage_seed, generation, agent_id, max_retries, delay_s, verbose
        )
    )


def observables_data(observables: list[str] | None = None, experiment_id: str | None = None) -> pl.DataFrame:
    """Get the output data from parquet files generated from a given vEcoli simulation as a
    dataframe containing all simulation timepoints.

    :param observables: list of observables(dataframe columns) to include. If None is passed, defaults to all columns.
    :param experiment_id: the experiment ID for the simulation that you wish to query.
        If None is passed, defaults to the example pinned simulation: "sms_single".

    :rtype: polars.DataFrame
    :return: A dataframe containing all simulation timepoints.

    """
    expid = experiment_id or "sms_single"
    tmpdir = tempfile.TemporaryDirectory()
    dirpath = Path(tmpdir.name)
    zippath = download_parquet(dirpath, expid)
    unzip_parquet(zippath, dirpath)
    df = pl.scan_parquet(f"{dirpath!s}/*.pq").select(observables).collect()
    tmpdir.cleanup()
    return df


def simulation_log(experiment: EcoliExperiment) -> str:
    return get_simulation_log(experiment)


def get_simulation_log(experiment: EcoliExperiment) -> str:
    import requests
    import json

    url = "https://sms.cam.uchc.edu/wcm/simulation/run/log"

    payload = json.loads(experiment.model_dump_json())

    headers = {
        "accept": "application/json",
        "Content-Type": "application/json"
    }

    response = requests.post(url, headers=headers, data=json.dumps(payload))

    # Get JSON response
    data = response.json()
    return data


async def run_simulation(
    config_id: str, max_retries: int = 20, delay_s: float = 1.0, verbose: bool = False, **body: dict[str, Any]
) -> EcoliExperiment | None:
    """Run a SMS API vEcoli simulation workflow.

    :param config_id: (str) Configuration ID of desired simulation experiment workflow. Defaults to "sms" (single cell)
    :param max_retries: (int) Maximum number of times to retry the workflow before giving up. Defaults to 20.
    :param delay_s: (float) Delay time between retries in seconds. Defaults to 1.0.
    :param verbose: (bool) Verbose mode. If set to ``True``, log print statements. Defaults to False.
    :param body: (kwargs/dict) Key/value pairs including: "overrides": {"config": {...}}, "variants": {"config": {...}}

    :rtype: EcoliExperiment
    :return: EcoliExperiment object with queriable experiment_id and experiment_tag
    """
    url = f"https://sms.cam.uchc.edu/wcm/simulation/run?config_id={config_id}"
    if not body:
        body = {
            "overrides": {"config": {}},
            "variants": {"config": {}},
        }

    attempt = 0
    pbar = tqdm(total=max_retries)
    async with httpx.AsyncClient() as client:
        print(f"Running a simulation with config id: {config_id}...")
        while attempt < max_retries:
            attempt += 1
            pbar.update(1)
            try:
                if verbose:
                    print(f"Attempt {attempt}...")
                response = await client.post(
                    url,
                    json=body,
                    headers={"Accept": "application/json"},
                    timeout=30.0,  # optional, adjust as needed
                )

                response.raise_for_status()  # raises for 4xx/5xx

                data = response.json()
                if verbose:
                    print("Success on attempt", attempt)
                pbar.total = attempt
                print("Simulation submitted!")
                pbar.close()
                return EcoliExperiment(**data)

            except (httpx.RequestError, httpx.HTTPStatusError) as err:
                if attempt == max_retries:
                    print(f"Attempt {attempt} failed:", err)
                    raise
                await asyncio.sleep(delay_s)
    pbar.close()
    return None


async def check_simulation_status(
    experiment: EcoliExperiment, max_retries: int = 20, delay_s: float = 1.0, verbose: bool = False
) -> SimulationRun | None:
    """Run a SMS API vEcoli simulation workflow.

    :param experiment: (EcoliExperiment) Experiment generated by run_simulation.
    :param max_retries: (int) Maximum number of times to retry the workflow before giving up. Defaults to 20.
    :param delay_s: (float) Delay time between retries in seconds. Defaults to 1.0.
    :param verbose: (bool) Verbose mode. If set to ``True``, log print statements. Defaults to False.
    :rtype: SimulationRun
    :return: SimulationRun confirming run status (status will be one of "waiting", "running", "completed", "failed"
    """
    pbar = tqdm(total=max_retries)
    url = f"https://sms.cam.uchc.edu/wcm/simulation/run/status?experiment_tag={experiment.experiment_tag}"
    attempt = 0
    async with httpx.AsyncClient() as client:
        print(f"Checking simulation status for experiment: {experiment.experiment_tag}...")
        while attempt < max_retries:
            attempt += 1
            pbar.update(1)
            try:
                if verbose:
                    print(f"Attempt {attempt}...")
                response = await client.get(
                    url,
                    headers={"Accept": "application/json"},
                    timeout=30.0,  # optional, adjust as needed
                )

                response.raise_for_status()  # raises for 4xx/5xx

                data = response.json()
                if verbose:
                    print("Success on attempt", attempt)
                pbar.total = attempt
                pbar.close()
                return SimulationRun(**data)

            except (httpx.RequestError, httpx.HTTPStatusError) as err:
                if attempt == max_retries:
                    print(f"Attempt {attempt} failed:", err)
                    raise
                await asyncio.sleep(delay_s)
            pbar.update(1.0)
    pbar.close()
    return None


async def get_analysis_manifest(
    experiment: EcoliExperiment, max_retries: int = 20, delay_s: float = 1.0, verbose: bool = False
) -> dict[str, Any] | None | Any:
    url = f"https://sms.cam.uchc.edu/wcm/analysis/outputs?experiment_id={experiment.experiment_id}"
    pbar = tqdm(total=max_retries)
    attempt = 0
    async with httpx.AsyncClient() as client:
        print(f"Getting analysis manifest for experiment: {experiment.experiment_id}...")
        while attempt < max_retries:
            attempt += 1
            pbar.update(1)
            try:
                if verbose:
                    print(f"Attempt {attempt}...")
                response = await client.get(
                    url,
                    headers={"Accept": "application/json"},
                    timeout=30.0,  # optional, adjust as needed
                )

                response.raise_for_status()  # raises for 4xx/5xx

                data = response.json()
                if verbose:
                    print("Success on attempt", attempt)
                pbar.total = attempt
                pbar.close()
                return data

            except (httpx.RequestError, httpx.HTTPStatusError) as err:
                if attempt == max_retries:
                    print(f"Attempt {attempt} failed:", err)
                    raise
                await asyncio.sleep(delay_s)
    pbar.close()
    return None


async def download_analysis_output(
    experiment: EcoliExperiment,
    filename: str,
    variant: int = 0,
    lineage_seed: int = 0,
    generation: int = 1,
    agent_id: int = 0,
    max_retries: int = 20,
    delay_s: float = 1.0,
    verbose: bool = False,
) -> None | Any:
    url = f"https://sms.cam.uchc.edu/wcm/analysis/download?experiment_id={experiment.experiment_id}&variant_id={variant}&lineage_seed_id={lineage_seed}&generation_id={generation}&agent_id={agent_id}&filename={filename}"
    pbar = tqdm(total=max_retries)
    attempt = 0
    async with httpx.AsyncClient() as client:
        print(f"Fetching analysis output for experiment: {experiment.experiment_id}...")
        while attempt < max_retries:
            attempt += 1
            pbar.update(1)
            try:
                if verbose:
                    print(f"Attempt {attempt}...")
                response = await client.get(
                    url,
                    headers={"Accept": "application/json"},
                    timeout=30.0,  # optional, adjust as needed
                )

                response.raise_for_status()  # raises for 4xx/5xx

                data = response.json()
                if verbose:
                    print("Success on attempt", attempt)
                pbar.total = attempt
                pbar.close()
                return data

            except (httpx.RequestError, httpx.HTTPStatusError) as err:
                if attempt == max_retries:
                    print(f"Attempt {attempt} failed:", err)
                    raise
                await asyncio.sleep(delay_s)
    pbar.close()
    return None


def unzip_parquet(zip_file_path: Path, local_dirpath: Path) -> None:
    extraction_path = local_dirpath
    try:
        with ZipFile(zip_file_path, "r") as zip_ref:
            zip_ref.extractall(extraction_path)
        print(f"Successfully unzipped '{zip_file_path}' to '{extraction_path}'")
    except FileNotFoundError:
        print(f"Error: The file '{zip_file_path}' was not found.")
    except Exception as e:
        print(f"An error occurred: {e}")


def download_parquet(local_dirpath: Path, experiment_id: str) -> Path:
    url = f"https://sms.cam.uchc.edu/core/download/parquet?experiment_id={experiment_id}"

    response = requests.post(url, headers={"Accept": "*/*"})

    if response.status_code != 200:
        raise Exception(f"HTTP error! status: {response.status_code}")

    zippath = local_dirpath / f"{experiment_id}.zip"
    with open(zippath, "wb") as f:
        f.write(response.content)

    return zippath
