from __future__ import annotations

import os
import shutil
import sys
from pathlib import Path
from typing import TYPE_CHECKING

import pytest
from test_helper import LBUG_ROOT

python_build_dir = Path(__file__).parent.parent / "build"
try:
    import real_ladybug as lb
except ModuleNotFoundError:
    sys.path.append(str(python_build_dir))
    import real_ladybug as lb

if TYPE_CHECKING:
    from type_aliases import ConnDB


def init_npy(conn: lb.Connection) -> None:
    conn.execute(
        """
        CREATE NODE TABLE npyoned (
          i64 INT64,
          i32 INT32,
          i16 INT16,
          f64 DOUBLE,
          f32 FLOAT,
          PRIMARY KEY(i64)
        );
        """
    )
    conn.execute(
        f"""
        COPY npyoned from (
          "{LBUG_ROOT}/dataset/npy-1d/one_dim_int64.npy",
          "{LBUG_ROOT}/dataset/npy-1d/one_dim_int32.npy",
          "{LBUG_ROOT}/dataset/npy-1d/one_dim_int16.npy",
          "{LBUG_ROOT}/dataset/npy-1d/one_dim_double.npy",
          "{LBUG_ROOT}/dataset/npy-1d/one_dim_float.npy") BY COLUMN;
        """
    )
    conn.execute(
        """
        CREATE NODE TABLE npytwod (
          id INT64,
          i64 INT64[3],
          i32 INT32[3],
          i16 INT16[3],
          f64 DOUBLE[3],
          f32 FLOAT[3],
          PRIMARY KEY(id)
        );
        """
    )
    conn.execute(
        f"""
        COPY npytwod FROM (
          "{LBUG_ROOT}/dataset/npy-2d/id_int64.npy",
          "{LBUG_ROOT}/dataset/npy-2d/two_dim_int64.npy",
          "{LBUG_ROOT}/dataset/npy-2d/two_dim_int32.npy",
          "{LBUG_ROOT}/dataset/npy-2d/two_dim_int16.npy",
          "{LBUG_ROOT}/dataset/npy-2d/two_dim_double.npy",
          "{LBUG_ROOT}/dataset/npy-2d/two_dim_float.npy") BY COLUMN;
        """
    )


def init_tensor(conn: lb.Connection) -> None:
    conn.execute(
        """
        CREATE NODE TABLE tensor (
          ID INT64,
          boolTensor BOOLEAN[],
          doubleTensor DOUBLE[][],
          intTensor INT64[][][],
          oneDimInt INT64,
          PRIMARY KEY (ID)
        );
        """
    )
    conn.execute(f'COPY tensor FROM "{LBUG_ROOT}/dataset/tensor-list/vTensor.csv" (HEADER=true)')


def init_long_str(conn: lb.Connection) -> None:
    conn.execute("CREATE NODE TABLE personLongString (name STRING, spouse STRING, PRIMARY KEY(name))")
    conn.execute(f'COPY personLongString FROM "{LBUG_ROOT}/dataset/long-string-pk-tests/vPerson.csv"')
    conn.execute("CREATE REL TABLE knowsLongString (FROM personLongString TO personLongString, MANY_MANY)")
    conn.execute(f'COPY knowsLongString FROM "{LBUG_ROOT}/dataset/long-string-pk-tests/eKnows.csv"')


def init_tinysnb(conn: lb.Connection) -> None:
    tiny_snb_path = (Path(__file__).parent / f"{LBUG_ROOT}/dataset/tinysnb").resolve()
    schema_path = tiny_snb_path / "schema.cypher"
    with schema_path.open(mode="r") as f:
        for line in f.readlines():
            line = line.strip()
            if line:
                conn.execute(line)

    copy_path = tiny_snb_path / "copy.cypher"
    with copy_path.open(mode="r") as f:
        for line in f.readlines():
            line = line.strip()
            line = line.replace("dataset/tinysnb", f"{LBUG_ROOT}/dataset/tinysnb")
            if line:
                conn.execute(line)


def init_demo(conn: lb.Connection) -> None:
    tiny_snb_path = (Path(__file__).parent / f"{LBUG_ROOT}/dataset/demo-db/csv").resolve()
    schema_path = tiny_snb_path / "schema.cypher"
    with schema_path.open(mode="r") as f:
        for line in f.readlines():
            line = line.strip()
            if line:
                conn.execute(line)

    copy_path = tiny_snb_path / "copy.cypher"
    with copy_path.open(mode="r") as f:
        for line in f.readlines():
            line = line.strip()
            line = line.replace("dataset/demo-db/csv", f"{LBUG_ROOT}/dataset/demo-db/csv")
            if line:
                conn.execute(line)


def init_movie_serial(conn: lb.Connection) -> None:
    conn.execute(
        """
        CREATE NODE TABLE moviesSerial (
          ID SERIAL,
          name STRING,
          length INT32,
          note STRING,
          PRIMARY KEY (ID)
        );"""
    )
    conn.execute(f'COPY moviesSerial from "{LBUG_ROOT}/dataset/tinysnb-serial/vMovies.csv"')


_POOL_SIZE_: int = 256 * 1024 * 1024


def get_db_file_path(tmp_path: Path) -> Path:
    """Return the path to the database file."""
    return tmp_path / "db.kz"


def init_db(path: Path) -> Path:
    if Path(path).exists():
        shutil.rmtree(path)
        Path.mkdir(path)

    db_path = get_db_file_path(path)
    conn, _ = create_conn_db(db_path, read_only=False)
    init_tinysnb(conn)
    init_demo(conn)
    init_npy(conn)
    init_tensor(conn)
    init_long_str(conn)
    init_movie_serial(conn)
    return db_path


_READONLY_CONN_DB_: ConnDB | None = None
_READONLY_ASYNC_CONNECTION_: lb.AsyncConnection | None = None


def create_conn_db(path: Path, *, read_only: bool) -> ConnDB:
    """Return a new connection and database."""
    db = lb.Database(path, buffer_pool_size=_POOL_SIZE_, read_only=read_only)
    conn = lb.Connection(db, num_threads=4)
    return conn, db


@pytest.fixture
def conn_db_readonly(tmp_path: Path) -> ConnDB:
    """Return a cached read-only connection and database."""
    global _READONLY_CONN_DB_
    if _READONLY_CONN_DB_ is None:
        _READONLY_CONN_DB_ = create_conn_db(init_db(tmp_path), read_only=True)
    return _READONLY_CONN_DB_


@pytest.fixture
def conn_db_readwrite(tmp_path: Path) -> ConnDB:
    """Return a new writable connection and database."""
    return create_conn_db(init_db(tmp_path), read_only=False)


@pytest.fixture
def async_connection_readonly(tmp_path: Path) -> lb.AsyncConnection:
    """Return a cached read-only async connection."""
    global _READONLY_ASYNC_CONNECTION_
    if _READONLY_ASYNC_CONNECTION_ is None:
        conn, db = create_conn_db(init_db(tmp_path), read_only=True)
        conn.close()
        _READONLY_ASYNC_CONNECTION_ = lb.AsyncConnection(db, max_threads_per_query=4)
    return _READONLY_ASYNC_CONNECTION_


@pytest.fixture
def async_connection_readwrite(tmp_path: Path) -> lb.AsyncConnection:
    """Return a writeable async connection."""
    conn, db = create_conn_db(init_db(tmp_path), read_only=False)
    conn.close()
    return lb.AsyncConnection(db, max_threads_per_query=4)


@pytest.fixture
def conn_db_empty(tmp_path: Path) -> ConnDB:
    """Return a new empty connection and database."""
    return create_conn_db(get_db_file_path(tmp_path), read_only=False)


@pytest.fixture
def conn_db_in_mem() -> ConnDB:
    """Return a new in-memory connection and database."""
    db = lb.Database(database_path=":memory:", buffer_pool_size=_POOL_SIZE_, read_only=False)
    conn = lb.Connection(db, num_threads=4)
    return conn, db


@pytest.fixture
def build_dir() -> Path:
    return python_build_dir
