"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cdk8s = require("cdk8s");
const minimatch_1 = require("minimatch");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * ConfigMap holds configuration data for pods to consume.
 */
class ConfigMap extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id);
        this._binaryData = {};
        this._data = {};
        this.apiObject = new k8s.KubeConfigMap(this, 'Resource', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
        });
        for (const [k, v] of Object.entries((_b = props.data) !== null && _b !== void 0 ? _b : {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries((_c = props.binaryData) !== null && _c !== void 0 ? _c : {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     * @param name The name of the config map to import
     */
    static fromConfigMapName(name) {
        return { name };
    }
    /**
     * Adds a data entry to the config map.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map. BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     * @param localFile The path to the local file
     * @param key The ConfigMap key (default to the file name).
     */
    addFile(localFile, key) {
        key = key !== null && key !== void 0 ? key : path.basename(localFile);
        const value = fs.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     * @param localDir A path to a local directory
     * @param options Options
     */
    addDirectory(localDir, options = {}) {
        var _b, _c;
        const exclude = (_b = options.exclude) !== null && _b !== void 0 ? _b : [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                const mm = new minimatch_1.Minimatch(pattern);
                if (mm.match(file)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = (_c = options.keyPrefix) !== null && _c !== void 0 ? _c : '';
        for (const file of fs.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (fs.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
_a = JSII_RTTI_SYMBOL_1;
ConfigMap[_a] = { fqn: "cdk8s-plus-22.ConfigMap", version: "1.0.0-beta.163" };
//# sourceMappingURL=data:application/json;base64,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