"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
const service_1 = require("./service");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id);
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.replicas = (_b = props.replicas) !== null && _b !== void 0 ? _b : 1;
        this._podTemplate = new pod_1.PodTemplate(props);
        this._labelSelector = {};
        if ((_c = props.defaultSelector) !== null && _c !== void 0 ? _c : true) {
            const selector = 'cdk8s.deployment';
            const matcher = cdk8s_1.Names.toLabelValue(this);
            this.podMetadata.addLabel(selector, matcher);
            this.selectByLabel(selector, matcher);
        }
    }
    get podMetadata() {
        return this._podTemplate.podMetadata;
    }
    /**
     * The labels this deployment will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    get containers() {
        return this._podTemplate.containers;
    }
    get initContainers() {
        return this._podTemplate.initContainers;
    }
    get hostAliases() {
        return this._podTemplate.hostAliases;
    }
    get volumes() {
        return this._podTemplate.volumes;
    }
    get restartPolicy() {
        return this._podTemplate.restartPolicy;
    }
    get serviceAccount() {
        return this._podTemplate.serviceAccount;
    }
    get securityContext() {
        return this._podTemplate.securityContext;
    }
    /**
     * Configure a label selector to this deployment.
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        var _b;
        const service = new service_1.Service(this, 'Service', {
            metadata: options.name ? { name: options.name } : undefined,
            type: (_b = options.serviceType) !== null && _b !== void 0 ? _b : service_1.ServiceType.CLUSTER_IP,
        });
        service.addDeployment(this, { protocol: options.protocol, targetPort: options.targetPort, port: options.port });
        return service;
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const service = this.exposeViaService(options);
        return service.exposeViaIngress(path, options);
    }
    addContainer(container) {
        return this._podTemplate.addContainer(container);
    }
    addInitContainer(container) {
        return this._podTemplate.addInitContainer(container);
    }
    addHostAlias(hostAlias) {
        return this._podTemplate.addHostAlias(hostAlias);
    }
    addVolume(volume) {
        return this._podTemplate.addVolume(volume);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            template: this._podTemplate._toPodTemplateSpec(),
            selector: {
                matchLabels: this._labelSelector,
            },
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-22.Deployment", version: "1.0.0-beta.163" };
//# sourceMappingURL=data:application/json;base64,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