"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatefulSet = exports.PodManagementPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
/**
 * Controls how pods are created during initial scale up, when replacing pods on nodes,
 * or when scaling down.
 *
 * The default policy is `OrderedReady`, where pods are created in increasing order
 * (pod-0, then pod-1, etc) and the controller will wait until each pod is ready before
 * continuing. When scaling down, the pods are removed in the opposite order.
 *
 * The alternative policy is `Parallel` which will create pods in parallel to match the
 * desired scale without waiting, and on scale down will delete all pods at once.
 */
var PodManagementPolicy;
(function (PodManagementPolicy) {
    PodManagementPolicy["ORDERED_READY"] = "OrderedReady";
    PodManagementPolicy["PARALLEL"] = "Parallel";
})(PodManagementPolicy = exports.PodManagementPolicy || (exports.PodManagementPolicy = {}));
/**
 * StatefulSet is the workload API object used to manage stateful applications.
 *
 * Manages the deployment and scaling of a set of Pods, and provides guarantees
 * about the ordering and uniqueness of these Pods.
 *
 * Like a Deployment, a StatefulSet manages Pods that are based on an identical
 * container spec. Unlike a Deployment, a StatefulSet maintains a sticky identity
 * for each of their Pods. These pods are created from the same spec, but are not
 * interchangeable: each has a persistent identifier that it maintains across any
 * rescheduling.
 *
 * If you want to use storage volumes to provide persistence for your workload, you
 * can use a StatefulSet as part of the solution. Although individual Pods in a StatefulSet
 * are susceptible to failure, the persistent Pod identifiers make it easier to match existing
 * volumes to the new Pods that replace any that have failed.
 *
 * Using StatefulSets
 * ------------------
 * StatefulSets are valuable for applications that require one or more of the following.
 *
 * - Stable, unique network identifiers.
 * - Stable, persistent storage.
 * - Ordered, graceful deployment and scaling.
 * - Ordered, automated rolling updates.
 */
class StatefulSet extends base_1.Resource {
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        this.apiObject = new k8s.KubeStatefulSet(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this._service = props.service;
        this.apiObject.addDependency(this._service);
        this.replicas = (_b = props.replicas) !== null && _b !== void 0 ? _b : 1;
        this.podManagementPolicy = (_c = props.podManagementPolicy) !== null && _c !== void 0 ? _c : PodManagementPolicy.ORDERED_READY;
        this._podTemplate = new pod_1.PodTemplate(props);
        this._labelSelector = {};
        if ((_d = props.defaultSelector) !== null && _d !== void 0 ? _d : true) {
            const selector = 'cdk8s.statefulset';
            const matcher = cdk8s_1.Names.toLabelValue(this);
            this.podMetadata.addLabel(selector, matcher);
            this.selectByLabel(selector, matcher);
        }
        const selectors = Object.entries(this.labelSelector);
        for (const [k, v] of selectors) {
            this._service.addSelector(k, v);
        }
    }
    get podMetadata() {
        return this._podTemplate.podMetadata;
    }
    /**
      * The labels this statefulset will match against in order to select pods.
      *
      * Returns a a copy. Use `selectByLabel()` to add labels.
      */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    get containers() {
        return this._podTemplate.containers;
    }
    get initContainers() {
        return this._podTemplate.initContainers;
    }
    get hostAliases() {
        return this._podTemplate.hostAliases;
    }
    get volumes() {
        return this._podTemplate.volumes;
    }
    get restartPolicy() {
        return this._podTemplate.restartPolicy;
    }
    get serviceAccount() {
        return this._podTemplate.serviceAccount;
    }
    /**
      * Configure a label selector to this deployment.
      * Pods that have the label will be selected by deployments configured with this spec.
      *
      * @param key - The label key.
      * @param value - The label value.
      */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    addContainer(container) {
        return this._podTemplate.addContainer(container);
    }
    addInitContainer(container) {
        return this._podTemplate.addInitContainer(container);
    }
    addHostAlias(hostAlias) {
        return this._podTemplate.addHostAlias(hostAlias);
    }
    addVolume(volume) {
        return this._podTemplate.addVolume(volume);
    }
    get securityContext() {
        return this._podTemplate.securityContext;
    }
    /**
      * @internal
      */
    _toKube() {
        return {
            serviceName: this._service.name,
            replicas: this.replicas,
            template: this._podTemplate._toPodTemplateSpec(),
            selector: {
                matchLabels: this._labelSelector,
            },
            podManagementPolicy: this.podManagementPolicy,
        };
    }
}
exports.StatefulSet = StatefulSet;
_a = JSII_RTTI_SYMBOL_1;
StatefulSet[_a] = { fqn: "cdk8s-plus-22.StatefulSet", version: "1.0.0-beta.163" };
//# sourceMappingURL=data:application/json;base64,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