"""Constants for OSMWayLoader."""

import numpy as np

MPH_TO_KMH = 1.609344
INCHES_TO_METERS = 0.0254
FEET_TO_METERS = 0.3048

METADATA_COLUMNS = ["from", "to", "key", "osmid", "name", "reversed", "length", "ref"]

OSMNX_TIMEOUT: int = 180
"""The timeout interval in seconds for HTTP requests."""

OSMNX_WAY_KEYS: list[str] = [
    "bridge",
    "tunnel",
    "oneway",
    "lanes",
    "ref",
    "name",
    "highway",
    "maxspeed",
    "service",
    "access",
    "area",
    "landuse",
    "width",
    "est_width",
    "junction",
    # missing in the original OSMnx config
    "surface",
    "footway",
    "bicycle",
    "lit",
]
"""Extended OSMnx way tags."""


OSM_WAY_TAGS: dict[str, list[str]] = {
    "oneway": ["False", "True"],
    "lanes": list(map(str, range(1, 21))),
    "highway": [
        "motorway",
        "motorway_link",
        "trunk",
        "trunk_link",
        "primary",
        "primary_link",
        "secondary",
        "secondary_link",
        "tertiary",
        "tertiary_link",
        "residential",
        "unclassified",
        "living_street",
        "service",
        "pedestrian",
        "track",
        "bus_guideway",
        "escape",
        "raceway",
        "road",
        "busway",
        "footway",
        "bridleway",
        "steps",
        "corridor",
        "path",
    ],
    "maxspeed": [
        "5",
        "7",
        "10",
        "15",
        "20",
        "30",
        "40",
        "50",
        "60",
        "70",
        "80",
        "90",
        "100",
        "110",
        "120",
        "130",
        "140",
    ],
    "bridge": [
        "yes",
        "aqueduct",
        "boardwalk",
        "cantilever",
        "covered",
        "low_water_crossing",
        "movable",
        "trestle",
        "viaduct",
        "simple_brunnel",
    ],
    "access": [
        "yes",
        "no",
        "private",
        "public",
        "permissive",
        "permit",
        "destination",
        "delivery",
        "customers",
        "designated",
        "use_sidepath",
        "dismount",
        "agricultural",
        "forestry",
        "discouraged",
        "unknown",
        "restricted",
        "official",
        "students",
        "licence",
        "license",
        "exclusion_zone",
        "dismount",
    ],
    "junction": [
        "yes",
        "roundabout",
        "circular",
        "spui",
        "jughandle",
        "filter",
        "intersection",
        "crossing",
        "uncontrolled",
    ],
    "width": list(np.arange(0, 30.5, 0.5).astype("str")),
    "tunnel": ["building_passage", "yes", "avalanche_protector"],
    "surface": [
        "paved",
        "asphalt",
        "chipseal",
        "concrete",
        "concrete:lanes",
        "concrete:plates",
        "paving_stones",
        "sett",
        "unhewn_cobblestone",
        "cobblestone",
        "metal",
        "wood",
        "stepping_stones",
        "rubber",
        "unpaved",
        "compacted",
        "fine_gravel",
        "gravel",
        "rock",
        "pebblestone",
        "ground",
        "dirt",
        "earth",
        "grass",
        "grass_paver",
        "mud",
        "sand",
        "woodchips",
        "snow",
        "ice",
        "salt",
    ],
    "bicycle": [
        "yes",
        "no",
        "designated",
        "use_sidepath",
        "dismount",
        "permissive",
        "private",
        "customers",
        "destination",
        "official",
    ],
    "lit": ["yes", "no", "sunset-sunrise", "24/7", "automatic", "disused", "limited", "interval"],
}
"""
Acceptable features assembled based on [1] and [2].

Assembled values that are officially defined in the wiki and are related to the `way` OSM type.

References:
    1. https://taginfo.openstreetmap.org/
    2. https://wiki.openstreetmap.org/wiki/Main_Page
"""


OSM_IMPLICIT_MAXSPEEDS = {
    "AR:urban": "40",
    "AR:rural": "110",
    "AR:urban:primary": "60",
    "AR:urban:secondary": "60",
    "AT:urban": "50",
    "AT:rural": "100",
    "AT:bicycle_road": "30",
    "AT:trunk": "100",
    "AT:motorway": "130",
    "AT:zone:30": "30",
    "AT:walk": "5",
    "walk": "5",
    "AU:urban": "None",
    "AU:rural": "None",
    "BE:living_street": "20",
    "BE:cyclestreet": "30",
    "BE:zone30": "30",
    "BE:trunk": "120",
    "BE:motorway": "120",
    "BG:urban": "50",
    "BG:rural": "90",
    "BG:living_street": "20",
    "BG:trunk": "120",
    "BG:motorway": "140",
    "BY:urban": "60",
    "BY:rural": "90",
    "BY:living_street": "20",
    "BY:motorway": "110",
    "CH:rural": "80",
    "CH:trunk": "100",
    "CH:motorway": "120",
    "CZ:urban": "50",
    "CZ:rural": "90",
    "CZ:pedestrian_zone": "20",
    "CZ:living_street": "20",
    "CZ:urban_motorway": "80",
    "CZ:urban_trunk": "80",
    "CZ:trunk": "110",
    "CZ:motorway": "130",
    "DE:urban": "50",
    "DE:rural": "100",
    "DE:living_street": "7",
    "DE:bicycle_road": "30",
    "DE:motorway": "None",
    "DK:urban": "50",
    "DK:rural": "80",
    "DK:motorway": "130",
    "EE:urban": "50",
    "EE:rural": "90",
    "ES:urban": "50",
    "ES:rural": "90",
    "ES:living_street": "20",
    "ES:zone30": "30",
    "ES:trunk": "90",
    "ES:motorway": "120",
    "FI:urban": "50",
    "FI:rural": "80",
    "FI:trunk": "100",
    "FI:motorway": "120",
    "FR:urban": "50",
    "FR:rural": "80",
    "FR:zone30": "30",
    "FR:motorway": "None",
    "GB:nsl_restricted": "30 mph",
    "GB:motorway": "70 mph",
    "GB:nsl_dual": "70 mph",
    "GB:nsl_single": "60 mph",
    "GR:urban": "50",
    "GR:rural": "90",
    "GR:trunk": "110",
    "GR:motorway": "130",
    "HU:urban": "50",
    "HU:rural": "90",
    "HU:living_street": "20",
    "HU:trunk": "110",
    "HU:motorway": "130",
    "IT:urban": "50",
    "IT:rural": "90",
    "IT:trunk": "110",
    "IT:motorway": "130",
    "JP:nsl": "60",
    "JP:express": "100",
    "LT:urban": "50",
    "LT:rural": "90",
    "NO:urban": "50",
    "NO:rural": "80",
    "PH:urban": "30",
    "PH:rural": "80",
    "PH:express": "100",
    "PL:urban": "50",
    "PT:urban": "50",
    "PT:rural": "90",
    "PT:trunk": "100",
    "PT:motorway": "120",
    "RO:urban": "50",
    "RO:rural": "90",
    "RO:trunk": "100",
    "RO:motorway": "130",
    "RS:urban": "50",
    "RS:rural": "80",
    "RS:living_street": "10",
    "RS:trunk": "100",
    "RS:motorway": "130",
    "RU:urban": "60",
    "RU:rural": "90",
    "RU:living_street": "20",
    "RU:motorway": "110",
    "SE:urban": "50",
    "SE:rural": "70",
    "SE:trunk": "90",
    "SE:motorway": "110",
    "SI:urban": "50",
    "SI:rural": "90",
    "SI:trunk": "110",
    "SI:motorway": "130",
    "SK:urban": "50",
    "SK:rural": "90",
    "SK:living_street": "20",
    "SK:trunk": "90",
    "SK:motorway": "130",
    "SK:motorway_urban": "90",
    "UA:urban": "50",
    "UA:rural": "90",
    "UA:living_street": "20",
    "UA:trunk": "110",
    "UA:motorway": "130",
    "UK:motorway": "70 mph",
    "UK:nsl_dual": "70 mph",
    "UK:nsl_single": "60 mph",
    "UZ:urban": "70",
    "UZ:rural": "100",
    "UZ:living_street": "30",
    "UZ:motorway": "110",
}
"""
Mapping for implicit maxspeeds.

The mapping is incomplete (see Notes section). Mostly taken from [1].

Notes:
    Consider using better (full) table in machine readable format available in [2].

References:
    1. https://wiki.openstreetmap.org/wiki/Key:maxspeed#Implicit_maxspeed_values
    2. https://wiki.openstreetmap.org/wiki/Default_speed_limits
"""
