import re
from typing import Optional

import numpy as np
from shapely import LineString


def var_line_string(
    pi: list,
    pj: list,
    path: list,
    closure: bool = False,
    n_sec: float = 2,
    sec_locs: Optional[list] = None,
    y_degree: int = 1,
    y_sym_plane: str = "j-0",
    z_degree: int = 2,
    z_sym_plane: str = "j-0",
):
    """Returns the varied line string.

    Parameters
    ----------
    pi : list[list[float, float]]
        I end line points.
    pj : list[list[float, float]]
        J end line points.
    path : list
        Coordinate path of section normal,
        such as [(x1, y1, z1), (x2, y2, z2), ... , (xn, yn, zn)].
        The section will be varied along this path.
        It is recommended as a division of the elements.
    closure: bool, default=False
        If True, the line will be a closed loop.
    n_sec : float, optional
        The number of sections within each line segment between two coords in Arg `path`,
        by default 2.
    sec_locs : list or 1D numpy array, optional, default=None
        The section position of each line segment (between two coords in Arg `path`) in the range 0-1.
        If not None, it overwrites n_sec.
        If None, it will be evenly divided according to Arg *n_sec*.
    y_degree : int, optional
        The polynomial order of the y-axis dimension variation of the section,
        1=linear, 2=parabolic, by default 1.
    y_sym_plane : str, optional, by default "j-0"
        When `y_degree`=2, specify the position of the symmetry plane, where the derivative is 0.
        The format is "i-{d}" or "j-{d}", which means that the distance from end i or end j is d.
        For example, "j-0" refers to end j, and "j-1.5" refers to the position 1.5 away from end j.
    z_degree : int, optional
        The polynomial order of the z-axis dimension variation of the section,
        1=linear, 2=parabolic, by default 2.
    z_sym_plane : str, optional, by default "j-0"
        When `z_degree`=2, specify the position of the symmetry plane, where the derivative is 0.
        The format is "i-{d}" or "j-{d}", which means that the distance from end i or end j is d.
        For example, "j-0" refers to end j, and "j-1.5" refers to the position 1.5 away from end j.

    Returns
    --------
    sec_points: list[list[list[float, float]]]
        A list of line points, the number of which is determined by `path` and `n_sec`.
    """
    length, cum_length, _ = _get_path_len(path, n_sec, sec_locs)
    if closure:
        if pi[-1] != pi[0]:
            pi = list(pi)
            pi.append(pi[0])
        if pj[-1] != pj[0]:
            pj = list(pj)
            pj.append(pj[0])

    linesi = LineString(pi)
    linesj = LineString(pj)
    yi, zi = linesi.xy
    yj, zj = linesj.xy
    cum_points = []
    for x in cum_length:
        points_ = []
        for i in range(len(yi)):
            y = _get_coord(0, yi[i], length, yj[i], x, degree=y_degree, sym_plane=y_sym_plane)
            z = _get_coord(0, zi[i], length, zj[i], x, degree=z_degree, sym_plane=z_sym_plane)
            points_.append((y, z))
        cum_points.append(points_)
    return cum_points


# def vis_var_sec(
#     sec_meshes: list,
#     path: list,
#     n_sec: float = 2,
#     loc_sec: list | None = None,
#     on_notebook: bool = False,
#     show_outline: bool = True,
# ):
#     """Visualize varied section meshes.

#     .. warning::
#         Please carefully check your section local axes.
#         For visualization purposes, this function defaults to using vecxz=[-1,0,0] for vertical elements
#         and vecxz=[0,0,1] for non-vertical ones.

#     Parameters
#     ----------
#     sec_meshes : list
#         Section meshes list generated by :class:`opstool.pre.SecMesh`.
#     path : list
#         Coordinate path of section normal,
#         such as [(x1, y1, z1), (x2, y2, z2), ... , (xn, yn, zn)].
#     n_sec : float, optional
#         The number of sections within each line segment between two coords in Arg `path`,
#         by default 2.
#     loc_sec : list or 1D numpy array, optional, default=None
#         The section position of each element in the range 0-1.
#         If not None, it overwrites n_sec.
#         If None, it will be evenly divided according to Arg *n_sec*.
#     show_outline: bool, optional, by default False
#         If True, display bound outline.
#     """
#     _, _, cum_coord = _get_path_len(path, n_sec, loc_sec)
#     cum_coord = np.array(cum_coord, dtype=np.float32)
#     point_plot1 = pv.PolyData(cum_coord)
#     plotter.add_mesh(point_plot1, color="black", point_size=5, render_points_as_spheres=True)
#     path = np.array(path, dtype=np.float32)
#     point_plot2 = pv.PolyData(path)
#     plotter.add_mesh(point_plot2, color="#8f1402", point_size=10, render_points_as_spheres=True)
#     line_cells = []
#     for i in range(len(path) - 1):
#         line_cells.extend([2, i, i + 1])

#     line_plot = _generate_mesh(path, line_cells, kind="line")
#     plotter.add_mesh(
#         line_plot,
#         color="black",
#         render_lines_as_tubes=False,
#         line_width=2,
#     )
#     local_axes = _get_path_local_axes(path, n_sec, loc_sec)
#     for i in range(len(sec_meshes)):
#         sec_mesh = sec_meshes[i]
#         center0 = cum_coord[i]
#         _, vecy, vecz = local_axes[i]
#         if not sec_mesh.is_centring:
#             sec_mesh.centring()
#         points = sec_mesh.points
#         points = points[:, 0].reshape((-1, 1)) @ np.reshape(vecy, (1, 3)) + points[:, 1].reshape((-1, 1)) @ np.reshape(
#             vecz, (1, 3)
#         )
#         points += center0
#         for name, faces in sec_mesh.cells_map.items():
#             faces = np.insert(faces, 0, values=3, axis=1)
#             face_plot = _generate_mesh(points, faces, kind="face")
#             plotter.add_mesh(face_plot, color=sec_mesh.color_map[name], show_edges=True, opacity=1)
#         for rdata in sec_mesh.rebar_data:
#             color = rdata["color"]
#             r = rdata["dia"] / 2
#             rebar_xy = np.array(rdata["rebar_xy"])
#             rebar_xy = rebar_xy[:, 0].reshape((-1, 1)) @ np.reshape(vecy, (1, 3)) + rebar_xy[:, 1].reshape((
#                 -1,
#                 1,
#             )) @ np.reshape(vecz, (1, 3))
#             rebar_xy += center0
#             spheres = []
#             for coord in rebar_xy:
#                 spheres.append(pv.Sphere(radius=r, center=coord))
#             merged = pv.MultiBlock(spheres)
#             plotter.add_mesh(merged, color=color)
#     if show_outline:
#         plotter.show_bounds(
#             grid=False,
#             location="outer",
#             show_zaxis=True,
#             # color="black",
#         )
#     plotter.add_axes()
#     plotter.view_isometric()
#     plotter.show(title="opstool")


def _get_path_len(path, n_sec, loc=None):
    n = len(path)
    xs = loc if loc is not None else np.linspace(0, 1, n_sec)
    length = 0
    cum_length = []
    cum_coord = []
    for i in range(n - 1):
        p1 = np.array(path[i])
        p2 = np.array(path[i + 1])
        seg = np.sqrt(np.sum((p2 - p1) ** 2))
        for li in xs:
            cum_length.append(length + seg * li)
            cum_coord.append(p1 + (p2 - p1) * li)
        length += seg
    cum_length = np.array(cum_length, dtype=np.float32)
    cum_coord = np.array(cum_coord, dtype=np.float32)
    return length, cum_length, cum_coord


def _get_path_local_axes(path, n_sec, loc):
    n = len(path)
    if loc is not None:
        n_sec = len(loc)
    local_axes = []
    for i in range(n - 1):
        p1 = np.array(path[i])
        p2 = np.array(path[i + 1])
        local_x = (p2 - p1) / np.linalg.norm(p2 - p1)
        global_z = np.array([0, 0, 1])
        cos2 = local_x @ global_z / (np.linalg.norm(local_x) * np.linalg.norm(global_z))
        vecxz = [-1, 0, 0] if np.abs(1 - cos2) < 1e-12 else [0, 0, 1]
        local_y = np.cross(vecxz, local_x)
        local_z = np.cross(local_x, local_y)
        local_y /= np.linalg.norm(local_y)
        local_z /= np.linalg.norm(local_z)
        for _j in range(n_sec):
            local_axes.append([local_x, local_y, local_z])
    return local_axes


def _get_coord(x1, y1, x2, y2, x, degree=1, sym_plane="j-0"):
    if degree == 1:
        y = y1 + (y2 - y1) * (x - x1) / (x2 - x1)
    elif degree == 2:
        d = float(re.findall(r"\d+\.?\d*", sym_plane)[0])
        if sym_plane[0].lower() == "j":
            a = (y2 - y1) / (x2**2 - x1**2 - 2 * (x2 + d) * (x2 - x1))
            b = -2 * a * (x2 + d)
            c = y1 - a * x1**2 - b * x1
            y = a * x**2 + b * x + c
        elif sym_plane[0].lower() == "i":
            a = (y2 - y1) / (x2**2 - x1**2 - 2 * (x1 - d) * (x2 - x1))
            b = -2 * a * (x1 - d)
            c = y1 - a * x1**2 - b * x1
            y = a * x**2 + b * x + c
        else:
            raise ValueError(f"Error arg sym_plane={sym_plane}!")  # noqa: TRY003
    else:
        raise ValueError("Currently only support degree=1 or 2!")  # noqa: TRY003
    return y


def _generate_mesh(points, cells, kind="line"):
    """
    generate the mesh from the points and cells
    """
    if kind == "line":
        pltr = pv.PolyData()
        pltr.points = points
        pltr.lines = cells
    elif kind == "face":
        pltr = pv.PolyData()
        pltr.points = points
        pltr.faces = cells
    else:
        raise ValueError(f"not supported {kind}!")  # noqa: TRY003
    return pltr


def get_lobatto_loc(n_sec: int) -> np.ndarray:
    """Get the location of the Lobatto integration sections.

    Paramaters
    -----------
    n_sec: int,
        The number of integration sections.

    Returns
    --------
    1d Numpy Array:
        The position of each beam element in the range 0-1.
    """
    xi = np.zeros(n_sec)
    if n_sec == 2:
        xi[0] = -1.0
        xi[1] = 1.0
    elif n_sec == 3:
        xi[0] = -1.0
        xi[1] = 0.0
        xi[2] = 1.0
    elif n_sec == 4:
        xi[0] = -1.0
        xi[1] = -0.44721360
        xi[2] = 0.44721360
        xi[3] = 1.0
    elif n_sec == 5:
        xi[0] = -1.0
        xi[1] = -0.65465367
        xi[2] = 0.0
        xi[3] = 0.65465367
        xi[4] = 1.0
    elif n_sec == 6:
        xi[0] = -1.0
        xi[1] = -0.7650553239
        xi[2] = -0.2852315164
        xi[3] = 0.2852315164
        xi[4] = 0.7650553239
        xi[5] = 1.0
    elif n_sec == 7:
        xi[0] = -1.0
        xi[1] = -0.8302238962
        xi[2] = -0.4688487934
        xi[3] = 0.0
        xi[4] = 0.4688487934
        xi[5] = 0.8302238962
        xi[6] = 1.0
    elif n_sec == 8:
        xi[0] = -1.0
        xi[1] = -0.8717401485
        xi[2] = -0.5917001814
        xi[3] = -0.2092992179
        xi[4] = 0.2092992179
        xi[5] = 0.5917001814
        xi[6] = 0.8717401485
        xi[7] = 1.0
    elif n_sec == 9:
        xi[0] = -1.0
        xi[1] = -0.8997579954
        xi[2] = -0.6771862795
        xi[3] = -0.3631174638
        xi[4] = 0.0
        xi[5] = 0.3631174638
        xi[6] = 0.6771862795
        xi[7] = 0.8997579954
        xi[8] = 1.0
    elif n_sec == 10:
        xi[0] = -1.0
        xi[1] = -0.9195339082
        xi[2] = -0.7387738651
        xi[3] = -0.4779249498
        xi[4] = -0.1652789577
        xi[5] = 0.1652789577
        xi[6] = 0.4779249498
        xi[7] = 0.7387738651
        xi[8] = 0.9195339082
        xi[9] = 1.0
    loc = 0.5 * (xi + 1.0)
    return loc


def get_legendre_loc(n_sec: int) -> np.ndarray:
    """
    Get the location of Legendre integration points mapped to [0, 1].

    Parameters
    ----------
    n_sec : int
        Number of integration sections (1 ≤ n ≤ 100).

    Returns
    -------
    np.ndarray
        Location of integration points in [0, 1].
    """
    if n_sec < 1:
        raise ValueError("n_sec must be ≥ 1.")  # noqa: TRY003
    xi, _ = np.polynomial.legendre.leggauss(n_sec)
    return 0.5 * (xi + 1.0)
