# coding=utf-8
from abc import (
    abstractmethod,
)
import builtins
from conjure_python_client import (
    BinaryType,
    ConjureBeanType,
    ConjureDecoder,
    ConjureEncoder,
    ConjureEnumType,
    ConjureFieldDefinition,
    ConjureUnionType,
    OptionalTypeWrapper,
    Service,
)
from requests.adapters import (
    Response,
)
from typing import (
    Any,
    Dict,
    List,
    Optional,
    Set,
)

class api_Empty(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



api_Empty.__name__ = "Empty"
api_Empty.__qualname__ = "Empty"
api_Empty.__module__ = "nominal_api.api"


class api_Granularity(ConjureEnumType):

    PICOSECONDS = 'PICOSECONDS'
    '''PICOSECONDS'''
    NANOSECONDS = 'NANOSECONDS'
    '''NANOSECONDS'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


api_Granularity.__name__ = "Granularity"
api_Granularity.__qualname__ = "Granularity"
api_Granularity.__module__ = "nominal_api.api"


class api_McapChannelLocator(ConjureUnionType):
    """Locator for a channel in an mcap file. Channel name is not guaranteed to be unique, so channel ID should
be used for mcap files with duplicate channel names."""
    _topic: Optional[str] = None
    _id: Optional[int] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'topic': ConjureFieldDefinition('topic', api_McapChannelTopic),
            'id': ConjureFieldDefinition('id', api_McapChannelId)
        }

    def __init__(
            self,
            topic: Optional[str] = None,
            id: Optional[int] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (topic is not None) + (id is not None) != 1:
                raise ValueError('a union must contain a single member')

            if topic is not None:
                self._topic = topic
                self._type = 'topic'
            if id is not None:
                self._id = id
                self._type = 'id'

        elif type_of_union == 'topic':
            if topic is None:
                raise ValueError('a union value must not be None')
            self._topic = topic
            self._type = 'topic'
        elif type_of_union == 'id':
            if id is None:
                raise ValueError('a union value must not be None')
            self._id = id
            self._type = 'id'

    @builtins.property
    def topic(self) -> Optional[str]:
        return self._topic

    @builtins.property
    def id(self) -> Optional[int]:
        return self._id

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, api_McapChannelLocatorVisitor):
            raise ValueError('{} is not an instance of api_McapChannelLocatorVisitor'.format(visitor.__class__.__name__))
        if self._type == 'topic' and self.topic is not None:
            return visitor._topic(self.topic)
        if self._type == 'id' and self.id is not None:
            return visitor._id(self.id)


api_McapChannelLocator.__name__ = "McapChannelLocator"
api_McapChannelLocator.__qualname__ = "McapChannelLocator"
api_McapChannelLocator.__module__ = "nominal_api.api"


class api_McapChannelLocatorVisitor:

    @abstractmethod
    def _topic(self, topic: str) -> Any:
        pass

    @abstractmethod
    def _id(self, id: int) -> Any:
        pass


api_McapChannelLocatorVisitor.__name__ = "McapChannelLocatorVisitor"
api_McapChannelLocatorVisitor.__qualname__ = "McapChannelLocatorVisitor"
api_McapChannelLocatorVisitor.__module__ = "nominal_api.api"


class api_SerializableError(ConjureBeanType):
    """
    A SerializableError is a representation of a ServiceException that exists to send error
results to clients as part of a response object when directly throwing an exception is undesirable.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', str),
            'message': ConjureFieldDefinition('message', OptionalTypeWrapper[str]),
            'error_instance_id': ConjureFieldDefinition('errorInstanceId', str),
            'status_code': ConjureFieldDefinition('statusCode', int)
        }

    __slots__: List[str] = ['_name', '_message', '_error_instance_id', '_status_code']

    def __init__(self, error_instance_id: str, name: str, status_code: int, message: Optional[str] = None) -> None:
        self._name = name
        self._message = message
        self._error_instance_id = error_instance_id
        self._status_code = status_code

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def message(self) -> Optional[str]:
        return self._message

    @builtins.property
    def error_instance_id(self) -> str:
        return self._error_instance_id

    @builtins.property
    def status_code(self) -> int:
        return self._status_code


api_SerializableError.__name__ = "SerializableError"
api_SerializableError.__qualname__ = "SerializableError"
api_SerializableError.__module__ = "nominal_api.api"


class api_SeriesDataType(ConjureEnumType):

    DOUBLE = 'DOUBLE'
    '''DOUBLE'''
    STRING = 'STRING'
    '''STRING'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


api_SeriesDataType.__name__ = "SeriesDataType"
api_SeriesDataType.__qualname__ = "SeriesDataType"
api_SeriesDataType.__module__ = "nominal_api.api"


class api_TimeUnit(ConjureEnumType):

    DAYS = 'DAYS'
    '''DAYS'''
    HOURS = 'HOURS'
    '''HOURS'''
    MINUTES = 'MINUTES'
    '''MINUTES'''
    SECONDS = 'SECONDS'
    '''SECONDS'''
    MILLISECONDS = 'MILLISECONDS'
    '''MILLISECONDS'''
    MICROSECONDS = 'MICROSECONDS'
    '''MICROSECONDS'''
    NANOSECONDS = 'NANOSECONDS'
    '''NANOSECONDS'''
    PICOSECONDS = 'PICOSECONDS'
    '''PICOSECONDS'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


api_TimeUnit.__name__ = "TimeUnit"
api_TimeUnit.__qualname__ = "TimeUnit"
api_TimeUnit.__module__ = "nominal_api.api"


class api_Timestamp(ConjureBeanType):
    """
    Picosecond precision timestamp type, represented by an epoch time in seconds, a nanosecond offset, and
optional picosecond offset.
The nanosecond offset is from the start of the epoch second, so must be less than 1 billion.
The optional picosecond offset is only used for picosecond-precision data sources and is from the start of
the nanosecond, so must be less than 1000.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'seconds': ConjureFieldDefinition('seconds', int),
            'nanos': ConjureFieldDefinition('nanos', int),
            'picos': ConjureFieldDefinition('picos', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_seconds', '_nanos', '_picos']

    def __init__(self, nanos: int, seconds: int, picos: Optional[int] = None) -> None:
        self._seconds = seconds
        self._nanos = nanos
        self._picos = picos

    @builtins.property
    def seconds(self) -> int:
        return self._seconds

    @builtins.property
    def nanos(self) -> int:
        return self._nanos

    @builtins.property
    def picos(self) -> Optional[int]:
        return self._picos


api_Timestamp.__name__ = "Timestamp"
api_Timestamp.__qualname__ = "Timestamp"
api_Timestamp.__module__ = "nominal_api.api"


class attachments_api_Attachment(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', api_rids_AttachmentRid),
            'title': ConjureFieldDefinition('title', str),
            's3_path': ConjureFieldDefinition('s3Path', attachments_api_S3Path),
            'file_type': ConjureFieldDefinition('fileType', str),
            'description': ConjureFieldDefinition('description', str),
            'created_by': ConjureFieldDefinition('createdBy', str),
            'properties': ConjureFieldDefinition('properties', Dict[attachments_api_PropertyName, attachments_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[attachments_api_Label]),
            'created_at': ConjureFieldDefinition('createdAt', str),
            'is_archived': ConjureFieldDefinition('isArchived', bool)
        }

    __slots__: List[str] = ['_rid', '_title', '_s3_path', '_file_type', '_description', '_created_by', '_properties', '_labels', '_created_at', '_is_archived']

    def __init__(self, created_at: str, created_by: str, description: str, file_type: str, is_archived: bool, labels: List[str], properties: Dict[str, str], rid: str, s3_path: str, title: str) -> None:
        self._rid = rid
        self._title = title
        self._s3_path = s3_path
        self._file_type = file_type
        self._description = description
        self._created_by = created_by
        self._properties = properties
        self._labels = labels
        self._created_at = created_at
        self._is_archived = is_archived

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def s3_path(self) -> str:
        return self._s3_path

    @builtins.property
    def file_type(self) -> str:
        return self._file_type

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def created_by(self) -> str:
        return self._created_by

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def created_at(self) -> str:
        return self._created_at

    @builtins.property
    def is_archived(self) -> bool:
        return self._is_archived


attachments_api_Attachment.__name__ = "Attachment"
attachments_api_Attachment.__qualname__ = "Attachment"
attachments_api_Attachment.__module__ = "nominal_api.attachments_api"


class attachments_api_AttachmentService(Service):
    """
    The attachment service provides functionality for creating, updating, and archiving attachments uploaded to S3.
    """

    def create(self, auth_header: str, request: "attachments_api_CreateAttachmentRequest") -> "attachments_api_Attachment":
        """
        Create a new attachment. Assumes the file is already uploaded to S3 through the upload service.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/attachments/v1/attachments'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), attachments_api_Attachment, self._return_none_for_unknown_union_types)

    def get(self, auth_header: str, rid: str) -> "attachments_api_Attachment":
        """
        Get an attachment by its RID.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/attachments/v1/attachments/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), attachments_api_Attachment, self._return_none_for_unknown_union_types)

    def get_batch(self, auth_header: str, request: "attachments_api_GetAttachmentsRequest") -> "attachments_api_GetAttachmentsResponse":
        """
        Get a set of attachments by their RIDs.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/attachments/v1/attachments/batch'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), attachments_api_GetAttachmentsResponse, self._return_none_for_unknown_union_types)

    def get_content(self, auth_header: str, rid: str) -> Any:
        """
        Get the binary content of an attachment.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/octet-stream',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/attachments/v1/attachments/{rid}/content'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            stream=True,
            json=_json)

        _raw = _response.raw
        _raw.decode_content = True
        return _raw

    def update(self, auth_header: str, request: "attachments_api_UpdateAttachmentRequest", rid: str) -> "attachments_api_Attachment":
        """
        Update an attachment. Only the fields that are set in the request will be updated.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/attachments/v1/attachments/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), attachments_api_Attachment, self._return_none_for_unknown_union_types)

    def archive(self, auth_header: str, rid: str) -> None:
        """
        Archive an attachment.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/attachments/v1/attachments/{rid}/archive'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def unarchive(self, auth_header: str, rid: str) -> None:
        """
        Unarchive an attachment.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/attachments/v1/attachments/{rid}/unarchive'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return


attachments_api_AttachmentService.__name__ = "AttachmentService"
attachments_api_AttachmentService.__qualname__ = "AttachmentService"
attachments_api_AttachmentService.__module__ = "nominal_api.attachments_api"


class attachments_api_CreateAttachmentRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            's3_path': ConjureFieldDefinition('s3Path', attachments_api_S3Path),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', str),
            'properties': ConjureFieldDefinition('properties', Dict[attachments_api_PropertyName, attachments_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[attachments_api_Label])
        }

    __slots__: List[str] = ['_s3_path', '_title', '_description', '_properties', '_labels']

    def __init__(self, description: str, labels: List[str], properties: Dict[str, str], s3_path: str, title: str) -> None:
        self._s3_path = s3_path
        self._title = title
        self._description = description
        self._properties = properties
        self._labels = labels

    @builtins.property
    def s3_path(self) -> str:
        return self._s3_path

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels


attachments_api_CreateAttachmentRequest.__name__ = "CreateAttachmentRequest"
attachments_api_CreateAttachmentRequest.__qualname__ = "CreateAttachmentRequest"
attachments_api_CreateAttachmentRequest.__module__ = "nominal_api.attachments_api"


class attachments_api_GetAttachmentsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'attachment_rids': ConjureFieldDefinition('attachmentRids', List[api_rids_AttachmentRid])
        }

    __slots__: List[str] = ['_attachment_rids']

    def __init__(self, attachment_rids: List[str]) -> None:
        self._attachment_rids = attachment_rids

    @builtins.property
    def attachment_rids(self) -> List[str]:
        return self._attachment_rids


attachments_api_GetAttachmentsRequest.__name__ = "GetAttachmentsRequest"
attachments_api_GetAttachmentsRequest.__qualname__ = "GetAttachmentsRequest"
attachments_api_GetAttachmentsRequest.__module__ = "nominal_api.attachments_api"


class attachments_api_GetAttachmentsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'response': ConjureFieldDefinition('response', List[attachments_api_Attachment])
        }

    __slots__: List[str] = ['_response']

    def __init__(self, response: List["attachments_api_Attachment"]) -> None:
        self._response = response

    @builtins.property
    def response(self) -> List["attachments_api_Attachment"]:
        return self._response


attachments_api_GetAttachmentsResponse.__name__ = "GetAttachmentsResponse"
attachments_api_GetAttachmentsResponse.__qualname__ = "GetAttachmentsResponse"
attachments_api_GetAttachmentsResponse.__module__ = "nominal_api.attachments_api"


class attachments_api_Property(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', attachments_api_PropertyName),
            'value': ConjureFieldDefinition('value', attachments_api_PropertyValue)
        }

    __slots__: List[str] = ['_name', '_value']

    def __init__(self, name: str, value: str) -> None:
        self._name = name
        self._value = value

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def value(self) -> str:
        return self._value


attachments_api_Property.__name__ = "Property"
attachments_api_Property.__qualname__ = "Property"
attachments_api_Property.__module__ = "nominal_api.attachments_api"


class attachments_api_SearchAttachmentsQuery(ConjureUnionType):
    _search_text: Optional[str] = None
    _label: Optional[str] = None
    _property: Optional["attachments_api_Property"] = None
    _and_: Optional[List["attachments_api_SearchAttachmentsQuery"]] = None
    _or_: Optional[List["attachments_api_SearchAttachmentsQuery"]] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'search_text': ConjureFieldDefinition('searchText', str),
            'label': ConjureFieldDefinition('label', attachments_api_Label),
            'property': ConjureFieldDefinition('property', attachments_api_Property),
            'and_': ConjureFieldDefinition('and', List[attachments_api_SearchAttachmentsQuery]),
            'or_': ConjureFieldDefinition('or', List[attachments_api_SearchAttachmentsQuery])
        }

    def __init__(
            self,
            search_text: Optional[str] = None,
            label: Optional[str] = None,
            property: Optional["attachments_api_Property"] = None,
            and_: Optional[List["attachments_api_SearchAttachmentsQuery"]] = None,
            or_: Optional[List["attachments_api_SearchAttachmentsQuery"]] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (search_text is not None) + (label is not None) + (property is not None) + (and_ is not None) + (or_ is not None) != 1:
                raise ValueError('a union must contain a single member')

            if search_text is not None:
                self._search_text = search_text
                self._type = 'searchText'
            if label is not None:
                self._label = label
                self._type = 'label'
            if property is not None:
                self._property = property
                self._type = 'property'
            if and_ is not None:
                self._and_ = and_
                self._type = 'and'
            if or_ is not None:
                self._or_ = or_
                self._type = 'or'

        elif type_of_union == 'searchText':
            if search_text is None:
                raise ValueError('a union value must not be None')
            self._search_text = search_text
            self._type = 'searchText'
        elif type_of_union == 'label':
            if label is None:
                raise ValueError('a union value must not be None')
            self._label = label
            self._type = 'label'
        elif type_of_union == 'property':
            if property is None:
                raise ValueError('a union value must not be None')
            self._property = property
            self._type = 'property'
        elif type_of_union == 'and':
            if and_ is None:
                raise ValueError('a union value must not be None')
            self._and_ = and_
            self._type = 'and'
        elif type_of_union == 'or':
            if or_ is None:
                raise ValueError('a union value must not be None')
            self._or_ = or_
            self._type = 'or'

    @builtins.property
    def search_text(self) -> Optional[str]:
        return self._search_text

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def property(self) -> Optional["attachments_api_Property"]:
        return self._property

    @builtins.property
    def and_(self) -> Optional[List["attachments_api_SearchAttachmentsQuery"]]:
        return self._and_

    @builtins.property
    def or_(self) -> Optional[List["attachments_api_SearchAttachmentsQuery"]]:
        return self._or_

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, attachments_api_SearchAttachmentsQueryVisitor):
            raise ValueError('{} is not an instance of attachments_api_SearchAttachmentsQueryVisitor'.format(visitor.__class__.__name__))
        if self._type == 'searchText' and self.search_text is not None:
            return visitor._search_text(self.search_text)
        if self._type == 'label' and self.label is not None:
            return visitor._label(self.label)
        if self._type == 'property' and self.property is not None:
            return visitor._property(self.property)
        if self._type == 'and' and self.and_ is not None:
            return visitor._and(self.and_)
        if self._type == 'or' and self.or_ is not None:
            return visitor._or(self.or_)


attachments_api_SearchAttachmentsQuery.__name__ = "SearchAttachmentsQuery"
attachments_api_SearchAttachmentsQuery.__qualname__ = "SearchAttachmentsQuery"
attachments_api_SearchAttachmentsQuery.__module__ = "nominal_api.attachments_api"


class attachments_api_SearchAttachmentsQueryVisitor:

    @abstractmethod
    def _search_text(self, search_text: str) -> Any:
        pass

    @abstractmethod
    def _label(self, label: str) -> Any:
        pass

    @abstractmethod
    def _property(self, property: "attachments_api_Property") -> Any:
        pass

    @abstractmethod
    def _and(self, and_: List["attachments_api_SearchAttachmentsQuery"]) -> Any:
        pass

    @abstractmethod
    def _or(self, or_: List["attachments_api_SearchAttachmentsQuery"]) -> Any:
        pass


attachments_api_SearchAttachmentsQueryVisitor.__name__ = "SearchAttachmentsQueryVisitor"
attachments_api_SearchAttachmentsQueryVisitor.__qualname__ = "SearchAttachmentsQueryVisitor"
attachments_api_SearchAttachmentsQueryVisitor.__module__ = "nominal_api.attachments_api"


class attachments_api_UpdateAttachmentRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'properties': ConjureFieldDefinition('properties', OptionalTypeWrapper[Dict[attachments_api_PropertyName, attachments_api_PropertyValue]]),
            'labels': ConjureFieldDefinition('labels', OptionalTypeWrapper[List[attachments_api_Label]])
        }

    __slots__: List[str] = ['_title', '_description', '_properties', '_labels']

    def __init__(self, description: Optional[str] = None, labels: Optional[List[str]] = None, properties: Optional[Dict[str, str]] = None, title: Optional[str] = None) -> None:
        self._title = title
        self._description = description
        self._properties = properties
        self._labels = labels

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def properties(self) -> Optional[Dict[str, str]]:
        return self._properties

    @builtins.property
    def labels(self) -> Optional[List[str]]:
        return self._labels


attachments_api_UpdateAttachmentRequest.__name__ = "UpdateAttachmentRequest"
attachments_api_UpdateAttachmentRequest.__qualname__ = "UpdateAttachmentRequest"
attachments_api_UpdateAttachmentRequest.__module__ = "nominal_api.attachments_api"


class authentication_AddToOrganizationRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'user_uuid': ConjureFieldDefinition('userUuid', str),
            'org_uuid': ConjureFieldDefinition('orgUuid', str)
        }

    __slots__: List[str] = ['_user_uuid', '_org_uuid']

    def __init__(self, org_uuid: str, user_uuid: str) -> None:
        self._user_uuid = user_uuid
        self._org_uuid = org_uuid

    @builtins.property
    def user_uuid(self) -> str:
        return self._user_uuid

    @builtins.property
    def org_uuid(self) -> str:
        return self._org_uuid


authentication_AddToOrganizationRequest.__name__ = "AddToOrganizationRequest"
authentication_AddToOrganizationRequest.__qualname__ = "AddToOrganizationRequest"
authentication_AddToOrganizationRequest.__module__ = "nominal_api.authentication"


class authentication_AuthenticationService(Service):
    """
    An internal-use-only service for managing users and org-memberships.
    """

    def create_user(self, auth_header: str, request: "authentication_CreateUserRequest") -> "authentication_User":
        """
        Only admins can create users.
Will throw if a user with the same email already exists.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/authentication/v1/users'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_User, self._return_none_for_unknown_union_types)

    def add_to_organization(self, auth_header: str, request: "authentication_AddToOrganizationRequest") -> None:
        """
        Only admins can add users to organizations.
Will throw if the user is already in the organization.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/authentication/v1/users/add-to-organization'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return


authentication_AuthenticationService.__name__ = "AuthenticationService"
authentication_AuthenticationService.__qualname__ = "AuthenticationService"
authentication_AuthenticationService.__module__ = "nominal_api.authentication"


class authentication_CreateUserRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'display_name': ConjureFieldDefinition('displayName', str),
            'email': ConjureFieldDefinition('email', str),
            'org_uuid': ConjureFieldDefinition('orgUuid', str)
        }

    __slots__: List[str] = ['_display_name', '_email', '_org_uuid']

    def __init__(self, display_name: str, email: str, org_uuid: str) -> None:
        self._display_name = display_name
        self._email = email
        self._org_uuid = org_uuid

    @builtins.property
    def display_name(self) -> str:
        return self._display_name

    @builtins.property
    def email(self) -> str:
        return self._email

    @builtins.property
    def org_uuid(self) -> str:
        return self._org_uuid


authentication_CreateUserRequest.__name__ = "CreateUserRequest"
authentication_CreateUserRequest.__qualname__ = "CreateUserRequest"
authentication_CreateUserRequest.__module__ = "nominal_api.authentication"


class authentication_User(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', str),
            'uuid': ConjureFieldDefinition('uuid', OptionalTypeWrapper[str]),
            'display_name': ConjureFieldDefinition('displayName', str),
            'username': ConjureFieldDefinition('username', OptionalTypeWrapper[str]),
            'email': ConjureFieldDefinition('email', str),
            'org_rid': ConjureFieldDefinition('orgRid', str)
        }

    __slots__: List[str] = ['_rid', '_uuid', '_display_name', '_username', '_email', '_org_rid']

    def __init__(self, display_name: str, email: str, org_rid: str, rid: str, username: Optional[str] = None, uuid: Optional[str] = None) -> None:
        self._rid = rid
        self._uuid = uuid
        self._display_name = display_name
        self._username = username
        self._email = email
        self._org_rid = org_rid

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def uuid(self) -> Optional[str]:
        return self._uuid

    @builtins.property
    def display_name(self) -> str:
        return self._display_name

    @builtins.property
    def username(self) -> Optional[str]:
        return self._username

    @builtins.property
    def email(self) -> str:
        return self._email

    @builtins.property
    def org_rid(self) -> str:
        return self._org_rid


authentication_User.__name__ = "User"
authentication_User.__qualname__ = "User"
authentication_User.__module__ = "nominal_api.authentication"


class authentication_api_AppearanceSetting(ConjureEnumType):

    SYSTEM = 'SYSTEM'
    '''SYSTEM'''
    LIGHT = 'LIGHT'
    '''LIGHT'''
    DARK = 'DARK'
    '''DARK'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


authentication_api_AppearanceSetting.__name__ = "AppearanceSetting"
authentication_api_AppearanceSetting.__qualname__ = "AppearanceSetting"
authentication_api_AppearanceSetting.__module__ = "nominal_api.authentication_api"


class authentication_api_AuthenticationServiceV2(Service):
    """
    This service provides operations for managing user and org profiles/settings.
Its name is a bit of a misnomer.
    """

    def get_my_profile(self, auth_header: str) -> "authentication_api_UserV2":
        """
        Gets the profile of the authenticated user.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/authentication/v2/my/profile'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_api_UserV2, self._return_none_for_unknown_union_types)

    def update_my_profile(self, auth_header: str, update_my_profile_request: "authentication_api_UpdateMyProfileRequest") -> "authentication_api_UserV2":
        """
        Updates the profile of the authenticated user.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(update_my_profile_request)

        _path = '/authentication/v2/my/profile'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_api_UserV2, self._return_none_for_unknown_union_types)

    def get_my_settings(self, auth_header: str) -> "authentication_api_UserSettings":
        """
        Gets the settings of the authenticated user.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/authentication/v2/my/settings'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_api_UserSettings, self._return_none_for_unknown_union_types)

    def update_my_settings(self, auth_header: str, user_settings: "authentication_api_UserSettings") -> "authentication_api_UserSettings":
        """
        Updates the settings of the authenticated user.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(user_settings)

        _path = '/authentication/v2/my/settings'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_api_UserSettings, self._return_none_for_unknown_union_types)

    def get_my_org_settings(self, auth_header: str) -> "authentication_api_OrgSettings":
        """
        Gets the settings of the org of the authenticated user.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/authentication/v2/org/settings'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_api_OrgSettings, self._return_none_for_unknown_union_types)

    def update_my_org_settings(self, auth_header: str, org_settings: "authentication_api_OrgSettings") -> "authentication_api_OrgSettings":
        """
        Updates the settings of the org of the authenticated user.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(org_settings)

        _path = '/authentication/v2/org/settings'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_api_OrgSettings, self._return_none_for_unknown_union_types)

    def search_users_v2(self, auth_header: str, request: "authentication_api_SearchUsersRequest") -> "authentication_api_SearchUsersResponseV2":
        """
        Searches for users by email and displayName.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/authentication/v2/users'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_api_SearchUsersResponseV2, self._return_none_for_unknown_union_types)

    def get_users(self, auth_header: str, user_rids: List[str] = None) -> List["authentication_api_UserV2"]:
        """
        Get users by RID.
        """
        user_rids = user_rids if user_rids is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(user_rids)

        _path = '/authentication/v2/users/batch'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[authentication_api_UserV2], self._return_none_for_unknown_union_types)

    def get_user(self, auth_header: str, user_rid: str) -> "authentication_api_UserV2":
        """
        Gets a user by RID.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'userRid': user_rid,
        }

        _json: Any = None

        _path = '/authentication/v2/users/{userRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authentication_api_UserV2, self._return_none_for_unknown_union_types)


authentication_api_AuthenticationServiceV2.__name__ = "AuthenticationServiceV2"
authentication_api_AuthenticationServiceV2.__qualname__ = "AuthenticationServiceV2"
authentication_api_AuthenticationServiceV2.__module__ = "nominal_api.authentication_api"


class authentication_api_DefaultTimeRangeTypeSetting(ConjureEnumType):

    DEFAULT = 'DEFAULT'
    '''DEFAULT'''
    ABSOLUTE = 'ABSOLUTE'
    '''ABSOLUTE'''
    RELATIVE = 'RELATIVE'
    '''RELATIVE'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


authentication_api_DefaultTimeRangeTypeSetting.__name__ = "DefaultTimeRangeTypeSetting"
authentication_api_DefaultTimeRangeTypeSetting.__qualname__ = "DefaultTimeRangeTypeSetting"
authentication_api_DefaultTimeRangeTypeSetting.__module__ = "nominal_api.authentication_api"


class authentication_api_OrgSettings(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'default_time_range_type': ConjureFieldDefinition('defaultTimeRangeType', OptionalTypeWrapper[authentication_api_DefaultTimeRangeTypeSetting])
        }

    __slots__: List[str] = ['_default_time_range_type']

    def __init__(self, default_time_range_type: Optional["authentication_api_DefaultTimeRangeTypeSetting"] = None) -> None:
        self._default_time_range_type = default_time_range_type

    @builtins.property
    def default_time_range_type(self) -> Optional["authentication_api_DefaultTimeRangeTypeSetting"]:
        return self._default_time_range_type


authentication_api_OrgSettings.__name__ = "OrgSettings"
authentication_api_OrgSettings.__qualname__ = "OrgSettings"
authentication_api_OrgSettings.__module__ = "nominal_api.authentication_api"


class authentication_api_SearchUsersQuery(ConjureUnionType):
    _and_: Optional[List["authentication_api_SearchUsersQuery"]] = None
    _or_: Optional[List["authentication_api_SearchUsersQuery"]] = None
    _exact_match: Optional[str] = None
    _search_text: Optional[str] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'and_': ConjureFieldDefinition('and', List[authentication_api_SearchUsersQuery]),
            'or_': ConjureFieldDefinition('or', List[authentication_api_SearchUsersQuery]),
            'exact_match': ConjureFieldDefinition('exactMatch', str),
            'search_text': ConjureFieldDefinition('searchText', str)
        }

    def __init__(
            self,
            and_: Optional[List["authentication_api_SearchUsersQuery"]] = None,
            or_: Optional[List["authentication_api_SearchUsersQuery"]] = None,
            exact_match: Optional[str] = None,
            search_text: Optional[str] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (and_ is not None) + (or_ is not None) + (exact_match is not None) + (search_text is not None) != 1:
                raise ValueError('a union must contain a single member')

            if and_ is not None:
                self._and_ = and_
                self._type = 'and'
            if or_ is not None:
                self._or_ = or_
                self._type = 'or'
            if exact_match is not None:
                self._exact_match = exact_match
                self._type = 'exactMatch'
            if search_text is not None:
                self._search_text = search_text
                self._type = 'searchText'

        elif type_of_union == 'and':
            if and_ is None:
                raise ValueError('a union value must not be None')
            self._and_ = and_
            self._type = 'and'
        elif type_of_union == 'or':
            if or_ is None:
                raise ValueError('a union value must not be None')
            self._or_ = or_
            self._type = 'or'
        elif type_of_union == 'exactMatch':
            if exact_match is None:
                raise ValueError('a union value must not be None')
            self._exact_match = exact_match
            self._type = 'exactMatch'
        elif type_of_union == 'searchText':
            if search_text is None:
                raise ValueError('a union value must not be None')
            self._search_text = search_text
            self._type = 'searchText'

    @builtins.property
    def and_(self) -> Optional[List["authentication_api_SearchUsersQuery"]]:
        return self._and_

    @builtins.property
    def or_(self) -> Optional[List["authentication_api_SearchUsersQuery"]]:
        return self._or_

    @builtins.property
    def exact_match(self) -> Optional[str]:
        """
        Performs case insensitive exact match on email
        """
        return self._exact_match

    @builtins.property
    def search_text(self) -> Optional[str]:
        """
        Searches email and display name
        """
        return self._search_text

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, authentication_api_SearchUsersQueryVisitor):
            raise ValueError('{} is not an instance of authentication_api_SearchUsersQueryVisitor'.format(visitor.__class__.__name__))
        if self._type == 'and' and self.and_ is not None:
            return visitor._and(self.and_)
        if self._type == 'or' and self.or_ is not None:
            return visitor._or(self.or_)
        if self._type == 'exactMatch' and self.exact_match is not None:
            return visitor._exact_match(self.exact_match)
        if self._type == 'searchText' and self.search_text is not None:
            return visitor._search_text(self.search_text)


authentication_api_SearchUsersQuery.__name__ = "SearchUsersQuery"
authentication_api_SearchUsersQuery.__qualname__ = "SearchUsersQuery"
authentication_api_SearchUsersQuery.__module__ = "nominal_api.authentication_api"


class authentication_api_SearchUsersQueryVisitor:

    @abstractmethod
    def _and(self, and_: List["authentication_api_SearchUsersQuery"]) -> Any:
        pass

    @abstractmethod
    def _or(self, or_: List["authentication_api_SearchUsersQuery"]) -> Any:
        pass

    @abstractmethod
    def _exact_match(self, exact_match: str) -> Any:
        pass

    @abstractmethod
    def _search_text(self, search_text: str) -> Any:
        pass


authentication_api_SearchUsersQueryVisitor.__name__ = "SearchUsersQueryVisitor"
authentication_api_SearchUsersQueryVisitor.__qualname__ = "SearchUsersQueryVisitor"
authentication_api_SearchUsersQueryVisitor.__module__ = "nominal_api.authentication_api"


class authentication_api_SearchUsersRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'query': ConjureFieldDefinition('query', authentication_api_SearchUsersQuery),
            'sort_by': ConjureFieldDefinition('sortBy', OptionalTypeWrapper[authentication_api_SortBy]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_backend_Token]),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_query', '_sort_by', '_next_page_token', '_page_size']

    def __init__(self, query: "authentication_api_SearchUsersQuery", next_page_token: Optional[str] = None, page_size: Optional[int] = None, sort_by: Optional["authentication_api_SortBy"] = None) -> None:
        self._query = query
        self._sort_by = sort_by
        self._next_page_token = next_page_token
        self._page_size = page_size

    @builtins.property
    def query(self) -> "authentication_api_SearchUsersQuery":
        return self._query

    @builtins.property
    def sort_by(self) -> Optional["authentication_api_SortBy"]:
        """
        UPDATED_AT descending by default
        """
        return self._sort_by

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Defaults to 100. Will throw if larger than 1_000.
        """
        return self._page_size


authentication_api_SearchUsersRequest.__name__ = "SearchUsersRequest"
authentication_api_SearchUsersRequest.__qualname__ = "SearchUsersRequest"
authentication_api_SearchUsersRequest.__module__ = "nominal_api.authentication_api"


class authentication_api_SearchUsersResponseV2(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[authentication_api_UserV2]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_backend_Token])
        }

    __slots__: List[str] = ['_results', '_next_page_token']

    def __init__(self, results: List["authentication_api_UserV2"], next_page_token: Optional[str] = None) -> None:
        self._results = results
        self._next_page_token = next_page_token

    @builtins.property
    def results(self) -> List["authentication_api_UserV2"]:
        return self._results

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


authentication_api_SearchUsersResponseV2.__name__ = "SearchUsersResponseV2"
authentication_api_SearchUsersResponseV2.__qualname__ = "SearchUsersResponseV2"
authentication_api_SearchUsersResponseV2.__module__ = "nominal_api.authentication_api"


class authentication_api_SortBy(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'is_descending': ConjureFieldDefinition('isDescending', bool),
            'field': ConjureFieldDefinition('field', authentication_api_SortByField)
        }

    __slots__: List[str] = ['_is_descending', '_field']

    def __init__(self, field: "authentication_api_SortByField", is_descending: bool) -> None:
        self._is_descending = is_descending
        self._field = field

    @builtins.property
    def is_descending(self) -> bool:
        return self._is_descending

    @builtins.property
    def field(self) -> "authentication_api_SortByField":
        return self._field


authentication_api_SortBy.__name__ = "SortBy"
authentication_api_SortBy.__qualname__ = "SortBy"
authentication_api_SortBy.__module__ = "nominal_api.authentication_api"


class authentication_api_SortByField(ConjureEnumType):

    NAME = 'NAME'
    '''NAME'''
    EMAIL = 'EMAIL'
    '''EMAIL'''
    CREATED_AT = 'CREATED_AT'
    '''CREATED_AT'''
    UPDATED_AT = 'UPDATED_AT'
    '''UPDATED_AT'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


authentication_api_SortByField.__name__ = "SortByField"
authentication_api_SortByField.__qualname__ = "SortByField"
authentication_api_SortByField.__module__ = "nominal_api.authentication_api"


class authentication_api_TimezoneSetting(ConjureEnumType):

    LOCAL = 'LOCAL'
    '''LOCAL'''
    UTC = 'UTC'
    '''UTC'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


authentication_api_TimezoneSetting.__name__ = "TimezoneSetting"
authentication_api_TimezoneSetting.__qualname__ = "TimezoneSetting"
authentication_api_TimezoneSetting.__module__ = "nominal_api.authentication_api"


class authentication_api_UpdateMyProfileRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'display_name': ConjureFieldDefinition('displayName', str)
        }

    __slots__: List[str] = ['_display_name']

    def __init__(self, display_name: str) -> None:
        self._display_name = display_name

    @builtins.property
    def display_name(self) -> str:
        return self._display_name


authentication_api_UpdateMyProfileRequest.__name__ = "UpdateMyProfileRequest"
authentication_api_UpdateMyProfileRequest.__qualname__ = "UpdateMyProfileRequest"
authentication_api_UpdateMyProfileRequest.__module__ = "nominal_api.authentication_api"


class authentication_api_UserSettings(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'default_time_range_type': ConjureFieldDefinition('defaultTimeRangeType', OptionalTypeWrapper[authentication_api_DefaultTimeRangeTypeSetting]),
            'appearance': ConjureFieldDefinition('appearance', OptionalTypeWrapper[authentication_api_AppearanceSetting]),
            'timezone': ConjureFieldDefinition('timezone', OptionalTypeWrapper[authentication_api_TimezoneSetting]),
            'time_series_hover_tooltip_concise': ConjureFieldDefinition('timeSeriesHoverTooltipConcise', OptionalTypeWrapper[bool])
        }

    __slots__: List[str] = ['_default_time_range_type', '_appearance', '_timezone', '_time_series_hover_tooltip_concise']

    def __init__(self, appearance: Optional["authentication_api_AppearanceSetting"] = None, default_time_range_type: Optional["authentication_api_DefaultTimeRangeTypeSetting"] = None, time_series_hover_tooltip_concise: Optional[bool] = None, timezone: Optional["authentication_api_TimezoneSetting"] = None) -> None:
        self._default_time_range_type = default_time_range_type
        self._appearance = appearance
        self._timezone = timezone
        self._time_series_hover_tooltip_concise = time_series_hover_tooltip_concise

    @builtins.property
    def default_time_range_type(self) -> Optional["authentication_api_DefaultTimeRangeTypeSetting"]:
        return self._default_time_range_type

    @builtins.property
    def appearance(self) -> Optional["authentication_api_AppearanceSetting"]:
        return self._appearance

    @builtins.property
    def timezone(self) -> Optional["authentication_api_TimezoneSetting"]:
        return self._timezone

    @builtins.property
    def time_series_hover_tooltip_concise(self) -> Optional[bool]:
        return self._time_series_hover_tooltip_concise


authentication_api_UserSettings.__name__ = "UserSettings"
authentication_api_UserSettings.__qualname__ = "UserSettings"
authentication_api_UserSettings.__module__ = "nominal_api.authentication_api"


class authentication_api_UserV2(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', authentication_api_UserRid),
            'org_rid': ConjureFieldDefinition('orgRid', authentication_api_OrgRid),
            'email': ConjureFieldDefinition('email', str),
            'display_name': ConjureFieldDefinition('displayName', str),
            'avatar_url': ConjureFieldDefinition('avatarUrl', str)
        }

    __slots__: List[str] = ['_rid', '_org_rid', '_email', '_display_name', '_avatar_url']

    def __init__(self, avatar_url: str, display_name: str, email: str, org_rid: str, rid: str) -> None:
        self._rid = rid
        self._org_rid = org_rid
        self._email = email
        self._display_name = display_name
        self._avatar_url = avatar_url

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def org_rid(self) -> str:
        return self._org_rid

    @builtins.property
    def email(self) -> str:
        return self._email

    @builtins.property
    def display_name(self) -> str:
        return self._display_name

    @builtins.property
    def avatar_url(self) -> str:
        """
        Avatar URL or a default avatar if the user does not have one.
        """
        return self._avatar_url


authentication_api_UserV2.__name__ = "UserV2"
authentication_api_UserV2.__qualname__ = "UserV2"
authentication_api_UserV2.__module__ = "nominal_api.authentication_api"


class authorization_ApiKey(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', authorization_ApiKeyRid),
            'api_key_name': ConjureFieldDefinition('apiKeyName', str),
            'created_by': ConjureFieldDefinition('createdBy', str),
            'created_at': ConjureFieldDefinition('createdAt', str),
            'expires_at': ConjureFieldDefinition('expiresAt', OptionalTypeWrapper[str]),
            'is_deleted': ConjureFieldDefinition('isDeleted', bool)
        }

    __slots__: List[str] = ['_rid', '_api_key_name', '_created_by', '_created_at', '_expires_at', '_is_deleted']

    def __init__(self, api_key_name: str, created_at: str, created_by: str, is_deleted: bool, rid: str, expires_at: Optional[str] = None) -> None:
        self._rid = rid
        self._api_key_name = api_key_name
        self._created_by = created_by
        self._created_at = created_at
        self._expires_at = expires_at
        self._is_deleted = is_deleted

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def api_key_name(self) -> str:
        return self._api_key_name

    @builtins.property
    def created_by(self) -> str:
        return self._created_by

    @builtins.property
    def created_at(self) -> str:
        return self._created_at

    @builtins.property
    def expires_at(self) -> Optional[str]:
        return self._expires_at

    @builtins.property
    def is_deleted(self) -> bool:
        return self._is_deleted


authorization_ApiKey.__name__ = "ApiKey"
authorization_ApiKey.__qualname__ = "ApiKey"
authorization_ApiKey.__module__ = "nominal_api.authorization"


class authorization_AuthorizationRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rids': ConjureFieldDefinition('rids', List[str])
        }

    __slots__: List[str] = ['_rids']

    def __init__(self, rids: List[str]) -> None:
        self._rids = rids

    @builtins.property
    def rids(self) -> List[str]:
        return self._rids


authorization_AuthorizationRequest.__name__ = "AuthorizationRequest"
authorization_AuthorizationRequest.__qualname__ = "AuthorizationRequest"
authorization_AuthorizationRequest.__module__ = "nominal_api.authorization"


class authorization_AuthorizationService(Service):
    """
    Authorization service manages the permissions for a user
to access resources.
    """

    def authorize(self, auth_header: str, authorize_request: "authorization_AuthorizationRequest") -> List[str]:
        """
        Given a set of resources, returns the set of resources that the
user is authorized to access, per their organization.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(authorize_request)

        _path = '/authorization/v1/authorize'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[str], self._return_none_for_unknown_union_types)

    def register(self, auth_header: str, register_request: "authorization_RegistrationRequest") -> None:
        """
        Marks a resource as belonging to an organization.
Will throw if trying to register a resource that already
belongs to a different organization.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(register_request)

        _path = '/authorization/v1/register'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def check_admin(self, auth_header: str) -> None:
        """
        Given an authenticated session, this endpoint returns a HTTP 204 if the
authenticated user is an admin and HTTP 403 otherwise.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/authorization/v1/checkAdmin'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def is_email_allowed(self, request: "authorization_IsEmailAllowedRequest") -> "authorization_IsEmailAllowedResponse":
        """
        Checks if the email is allowed to register.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/authorization/v1/is-email-allowed'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authorization_IsEmailAllowedResponse, self._return_none_for_unknown_union_types)

    def get_access_token(self, request: "authorization_GetAccessTokenRequest") -> "authorization_GetAccessTokenResponse":
        """
        Provide an OIDC ID and access token to get a Nominal access token,
suitable for making API requests. Its expiry will match that of the
input access token, capped at 24h.
Throws NotAuthorized if either token is invalid or if the OIDC provider
is not known.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/authorization/v1/access-token'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authorization_GetAccessTokenResponse, self._return_none_for_unknown_union_types)

    def create_api_key(self, auth_header: str, request: "authorization_CreateApiKeyRequest") -> "authorization_CreateApiKeyResponse":
        """
        Provide a long-lived API key for making API requests.
The API key is irretrievable after initial creation.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/authorization/v1/api-key'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authorization_CreateApiKeyResponse, self._return_none_for_unknown_union_types)

    def list_api_keys_in_org(self, auth_header: str, request: "authorization_ListApiKeyRequest") -> "authorization_ListApiKeyResponse":
        """
        List all API keys in the organization.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/authorization/v1/api-keys/org'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authorization_ListApiKeyResponse, self._return_none_for_unknown_union_types)

    def list_user_api_keys(self, auth_header: str, request: "authorization_ListApiKeyRequest") -> "authorization_ListApiKeyResponse":
        """
        List all API keys for the user.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/authorization/v1/api-keys/user'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authorization_ListApiKeyResponse, self._return_none_for_unknown_union_types)

    def revoke_api_key(self, auth_header: str, rid: str) -> None:
        """
        Delete an API key.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/authorization/v1/api-key/{rid}/delete'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return


authorization_AuthorizationService.__name__ = "AuthorizationService"
authorization_AuthorizationService.__qualname__ = "AuthorizationService"
authorization_AuthorizationService.__module__ = "nominal_api.authorization"


class authorization_CreateApiKeyRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'api_key_name': ConjureFieldDefinition('apiKeyName', str),
            'expires_after_days': ConjureFieldDefinition('expiresAfterDays', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_api_key_name', '_expires_after_days']

    def __init__(self, api_key_name: str, expires_after_days: Optional[int] = None) -> None:
        self._api_key_name = api_key_name
        self._expires_after_days = expires_after_days

    @builtins.property
    def api_key_name(self) -> str:
        """
        The name of the API key to create.
        """
        return self._api_key_name

    @builtins.property
    def expires_after_days(self) -> Optional[int]:
        """
        The number of days after which the API key will expire.
If omitted, the API key will not expire.
        """
        return self._expires_after_days


authorization_CreateApiKeyRequest.__name__ = "CreateApiKeyRequest"
authorization_CreateApiKeyRequest.__qualname__ = "CreateApiKeyRequest"
authorization_CreateApiKeyRequest.__module__ = "nominal_api.authorization"


class authorization_CreateApiKeyResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'api_key_metadata': ConjureFieldDefinition('apiKeyMetadata', authorization_ApiKey),
            'api_key_value': ConjureFieldDefinition('apiKeyValue', str)
        }

    __slots__: List[str] = ['_api_key_metadata', '_api_key_value']

    def __init__(self, api_key_metadata: "authorization_ApiKey", api_key_value: str) -> None:
        self._api_key_metadata = api_key_metadata
        self._api_key_value = api_key_value

    @builtins.property
    def api_key_metadata(self) -> "authorization_ApiKey":
        return self._api_key_metadata

    @builtins.property
    def api_key_value(self) -> str:
        return self._api_key_value


authorization_CreateApiKeyResponse.__name__ = "CreateApiKeyResponse"
authorization_CreateApiKeyResponse.__qualname__ = "CreateApiKeyResponse"
authorization_CreateApiKeyResponse.__module__ = "nominal_api.authorization"


class authorization_GetAccessTokenFromApiKeyRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'api_key_value': ConjureFieldDefinition('apiKeyValue', str)
        }

    __slots__: List[str] = ['_api_key_value']

    def __init__(self, api_key_value: str) -> None:
        self._api_key_value = api_key_value

    @builtins.property
    def api_key_value(self) -> str:
        return self._api_key_value


authorization_GetAccessTokenFromApiKeyRequest.__name__ = "GetAccessTokenFromApiKeyRequest"
authorization_GetAccessTokenFromApiKeyRequest.__qualname__ = "GetAccessTokenFromApiKeyRequest"
authorization_GetAccessTokenFromApiKeyRequest.__module__ = "nominal_api.authorization"


class authorization_GetAccessTokenRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'id_token': ConjureFieldDefinition('idToken', str),
            'access_token': ConjureFieldDefinition('accessToken', str)
        }

    __slots__: List[str] = ['_id_token', '_access_token']

    def __init__(self, access_token: str, id_token: str) -> None:
        self._id_token = id_token
        self._access_token = access_token

    @builtins.property
    def id_token(self) -> str:
        return self._id_token

    @builtins.property
    def access_token(self) -> str:
        """
        The access token's audience must be for the Nominal API.
        """
        return self._access_token


authorization_GetAccessTokenRequest.__name__ = "GetAccessTokenRequest"
authorization_GetAccessTokenRequest.__qualname__ = "GetAccessTokenRequest"
authorization_GetAccessTokenRequest.__module__ = "nominal_api.authorization"


class authorization_GetAccessTokenResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'access_token': ConjureFieldDefinition('accessToken', str),
            'expires_at_seconds': ConjureFieldDefinition('expiresAtSeconds', int),
            'user_uuid': ConjureFieldDefinition('userUuid', str),
            'org_uuid': ConjureFieldDefinition('orgUuid', str)
        }

    __slots__: List[str] = ['_access_token', '_expires_at_seconds', '_user_uuid', '_org_uuid']

    def __init__(self, access_token: str, expires_at_seconds: int, org_uuid: str, user_uuid: str) -> None:
        self._access_token = access_token
        self._expires_at_seconds = expires_at_seconds
        self._user_uuid = user_uuid
        self._org_uuid = org_uuid

    @builtins.property
    def access_token(self) -> str:
        return self._access_token

    @builtins.property
    def expires_at_seconds(self) -> int:
        return self._expires_at_seconds

    @builtins.property
    def user_uuid(self) -> str:
        return self._user_uuid

    @builtins.property
    def org_uuid(self) -> str:
        return self._org_uuid


authorization_GetAccessTokenResponse.__name__ = "GetAccessTokenResponse"
authorization_GetAccessTokenResponse.__qualname__ = "GetAccessTokenResponse"
authorization_GetAccessTokenResponse.__module__ = "nominal_api.authorization"


class authorization_InternalApiKeyService(Service):
    """
    This internal-only service manages long lived api keys.e
    """

    def get_access_token_from_api_key_value(self, request: "authorization_GetAccessTokenFromApiKeyRequest") -> "authorization_GetAccessTokenResponse":
        """
        Get a Nominal-issued access token from a long-lived API key. Callers should verify that
their api key is formatted properly (i.e. prefixed with "nominal_api_key") before calling this endpoint.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/api-key-internal/v1/access-token'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), authorization_GetAccessTokenResponse, self._return_none_for_unknown_union_types)


authorization_InternalApiKeyService.__name__ = "InternalApiKeyService"
authorization_InternalApiKeyService.__qualname__ = "InternalApiKeyService"
authorization_InternalApiKeyService.__module__ = "nominal_api.authorization"


class authorization_IsEmailAllowedRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'email': ConjureFieldDefinition('email', str)
        }

    __slots__: List[str] = ['_email']

    def __init__(self, email: str) -> None:
        self._email = email

    @builtins.property
    def email(self) -> str:
        return self._email


authorization_IsEmailAllowedRequest.__name__ = "IsEmailAllowedRequest"
authorization_IsEmailAllowedRequest.__qualname__ = "IsEmailAllowedRequest"
authorization_IsEmailAllowedRequest.__module__ = "nominal_api.authorization"


class authorization_IsEmailAllowedResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'allowed': ConjureFieldDefinition('allowed', bool)
        }

    __slots__: List[str] = ['_allowed']

    def __init__(self, allowed: bool) -> None:
        self._allowed = allowed

    @builtins.property
    def allowed(self) -> bool:
        return self._allowed


authorization_IsEmailAllowedResponse.__name__ = "IsEmailAllowedResponse"
authorization_IsEmailAllowedResponse.__qualname__ = "IsEmailAllowedResponse"
authorization_IsEmailAllowedResponse.__module__ = "nominal_api.authorization"


class authorization_ListApiKeyRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'include_deleted': ConjureFieldDefinition('includeDeleted', OptionalTypeWrapper[bool]),
            'include_expired': ConjureFieldDefinition('includeExpired', OptionalTypeWrapper[bool]),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_backend_Token])
        }

    __slots__: List[str] = ['_include_deleted', '_include_expired', '_page_size', '_next_page_token']

    def __init__(self, include_deleted: Optional[bool] = None, include_expired: Optional[bool] = None, next_page_token: Optional[str] = None, page_size: Optional[int] = None) -> None:
        self._include_deleted = include_deleted
        self._include_expired = include_expired
        self._page_size = page_size
        self._next_page_token = next_page_token

    @builtins.property
    def include_deleted(self) -> Optional[bool]:
        """
        If true, include deleted API keys in the response. Defaults to false.
        """
        return self._include_deleted

    @builtins.property
    def include_expired(self) -> Optional[bool]:
        """
        If true, include expired API keys in the response. Defaults to false.
        """
        return self._include_expired

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        The maximum number of API keys to return. Defaults to 100.
        """
        return self._page_size

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


authorization_ListApiKeyRequest.__name__ = "ListApiKeyRequest"
authorization_ListApiKeyRequest.__qualname__ = "ListApiKeyRequest"
authorization_ListApiKeyRequest.__module__ = "nominal_api.authorization"


class authorization_ListApiKeyResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'api_keys': ConjureFieldDefinition('apiKeys', List[authorization_ApiKey]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_backend_Token])
        }

    __slots__: List[str] = ['_api_keys', '_next_page_token']

    def __init__(self, api_keys: List["authorization_ApiKey"], next_page_token: Optional[str] = None) -> None:
        self._api_keys = api_keys
        self._next_page_token = next_page_token

    @builtins.property
    def api_keys(self) -> List["authorization_ApiKey"]:
        return self._api_keys

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


authorization_ListApiKeyResponse.__name__ = "ListApiKeyResponse"
authorization_ListApiKeyResponse.__qualname__ = "ListApiKeyResponse"
authorization_ListApiKeyResponse.__module__ = "nominal_api.authorization"


class authorization_RegistrationRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rids': ConjureFieldDefinition('rids', List[str]),
            'organization_rid': ConjureFieldDefinition('organizationRid', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_rids', '_organization_rid']

    def __init__(self, rids: List[str], organization_rid: Optional[str] = None) -> None:
        self._rids = rids
        self._organization_rid = organization_rid

    @builtins.property
    def rids(self) -> List[str]:
        return self._rids

    @builtins.property
    def organization_rid(self) -> Optional[str]:
        """
        If omitted, will use the user's organizationRid.
        """
        return self._organization_rid


authorization_RegistrationRequest.__name__ = "RegistrationRequest"
authorization_RegistrationRequest.__qualname__ = "RegistrationRequest"
authorization_RegistrationRequest.__module__ = "nominal_api.authorization"


class comments_api_Comment(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', comments_api_CommentRid),
            'parent': ConjureFieldDefinition('parent', comments_api_CommentParent),
            'author_rid': ConjureFieldDefinition('authorRid', str),
            'created_at': ConjureFieldDefinition('createdAt', str),
            'edited_at': ConjureFieldDefinition('editedAt', OptionalTypeWrapper[str]),
            'deleted_at': ConjureFieldDefinition('deletedAt', OptionalTypeWrapper[str]),
            'content': ConjureFieldDefinition('content', str),
            'pinned_by': ConjureFieldDefinition('pinnedBy', OptionalTypeWrapper[str]),
            'pinned_at': ConjureFieldDefinition('pinnedAt', OptionalTypeWrapper[str]),
            'reactions': ConjureFieldDefinition('reactions', List[comments_api_Reaction]),
            'attachments': ConjureFieldDefinition('attachments', List[str])
        }

    __slots__: List[str] = ['_rid', '_parent', '_author_rid', '_created_at', '_edited_at', '_deleted_at', '_content', '_pinned_by', '_pinned_at', '_reactions', '_attachments']

    def __init__(self, attachments: List[str], author_rid: str, content: str, created_at: str, parent: "comments_api_CommentParent", reactions: List["comments_api_Reaction"], rid: str, deleted_at: Optional[str] = None, edited_at: Optional[str] = None, pinned_at: Optional[str] = None, pinned_by: Optional[str] = None) -> None:
        self._rid = rid
        self._parent = parent
        self._author_rid = author_rid
        self._created_at = created_at
        self._edited_at = edited_at
        self._deleted_at = deleted_at
        self._content = content
        self._pinned_by = pinned_by
        self._pinned_at = pinned_at
        self._reactions = reactions
        self._attachments = attachments

    @builtins.property
    def rid(self) -> str:
        """
        Unique resource identifier for the comment
        """
        return self._rid

    @builtins.property
    def parent(self) -> "comments_api_CommentParent":
        """
        The parent of the comment. It can be a resource or another comment.
        """
        return self._parent

    @builtins.property
    def author_rid(self) -> str:
        """
        The user who authored the comment
        """
        return self._author_rid

    @builtins.property
    def created_at(self) -> str:
        """
        The time the comment was created
        """
        return self._created_at

    @builtins.property
    def edited_at(self) -> Optional[str]:
        """
        The time the comment was edited. Empty if the comment has not been edited.
        """
        return self._edited_at

    @builtins.property
    def deleted_at(self) -> Optional[str]:
        """
        The time the comment was deleted. Empty if the comment has not been deleted.
        """
        return self._deleted_at

    @builtins.property
    def content(self) -> str:
        """
        The markdown content of the comment.
        """
        return self._content

    @builtins.property
    def pinned_by(self) -> Optional[str]:
        """
        The user who pinned the comment. Empty if the comment is not pinned.
        """
        return self._pinned_by

    @builtins.property
    def pinned_at(self) -> Optional[str]:
        """
        The time the comment was pinned. Empty if the comment is not pinned.
        """
        return self._pinned_at

    @builtins.property
    def reactions(self) -> List["comments_api_Reaction"]:
        """
        The reactions on the comment
        """
        return self._reactions

    @builtins.property
    def attachments(self) -> List[str]:
        """
        The comment's attachments
        """
        return self._attachments


comments_api_Comment.__name__ = "Comment"
comments_api_Comment.__qualname__ = "Comment"
comments_api_Comment.__module__ = "nominal_api.comments_api"


class comments_api_CommentParent(ConjureUnionType):
    """The parent of a comment. It can be a resource or another comment in the case of a reply."""
    _resource: Optional["comments_api_CommentParentResource"] = None
    _comment: Optional["comments_api_CommentParentComment"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'resource': ConjureFieldDefinition('resource', comments_api_CommentParentResource),
            'comment': ConjureFieldDefinition('comment', comments_api_CommentParentComment)
        }

    def __init__(
            self,
            resource: Optional["comments_api_CommentParentResource"] = None,
            comment: Optional["comments_api_CommentParentComment"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (resource is not None) + (comment is not None) != 1:
                raise ValueError('a union must contain a single member')

            if resource is not None:
                self._resource = resource
                self._type = 'resource'
            if comment is not None:
                self._comment = comment
                self._type = 'comment'

        elif type_of_union == 'resource':
            if resource is None:
                raise ValueError('a union value must not be None')
            self._resource = resource
            self._type = 'resource'
        elif type_of_union == 'comment':
            if comment is None:
                raise ValueError('a union value must not be None')
            self._comment = comment
            self._type = 'comment'

    @builtins.property
    def resource(self) -> Optional["comments_api_CommentParentResource"]:
        return self._resource

    @builtins.property
    def comment(self) -> Optional["comments_api_CommentParentComment"]:
        return self._comment

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, comments_api_CommentParentVisitor):
            raise ValueError('{} is not an instance of comments_api_CommentParentVisitor'.format(visitor.__class__.__name__))
        if self._type == 'resource' and self.resource is not None:
            return visitor._resource(self.resource)
        if self._type == 'comment' and self.comment is not None:
            return visitor._comment(self.comment)


comments_api_CommentParent.__name__ = "CommentParent"
comments_api_CommentParent.__qualname__ = "CommentParent"
comments_api_CommentParent.__module__ = "nominal_api.comments_api"


class comments_api_CommentParentVisitor:

    @abstractmethod
    def _resource(self, resource: "comments_api_CommentParentResource") -> Any:
        pass

    @abstractmethod
    def _comment(self, comment: "comments_api_CommentParentComment") -> Any:
        pass


comments_api_CommentParentVisitor.__name__ = "CommentParentVisitor"
comments_api_CommentParentVisitor.__qualname__ = "CommentParentVisitor"
comments_api_CommentParentVisitor.__module__ = "nominal_api.comments_api"


class comments_api_CommentParentComment(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'comment_rid': ConjureFieldDefinition('commentRid', comments_api_CommentRid)
        }

    __slots__: List[str] = ['_comment_rid']

    def __init__(self, comment_rid: str) -> None:
        self._comment_rid = comment_rid

    @builtins.property
    def comment_rid(self) -> str:
        """
        The resource identifier for the comment that the comment is replying to
        """
        return self._comment_rid


comments_api_CommentParentComment.__name__ = "CommentParentComment"
comments_api_CommentParentComment.__qualname__ = "CommentParentComment"
comments_api_CommentParentComment.__module__ = "nominal_api.comments_api"


class comments_api_CommentParentResource(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'resource_type': ConjureFieldDefinition('resourceType', comments_api_ResourceType),
            'resource_rid': ConjureFieldDefinition('resourceRid', comments_api_ResourceRid)
        }

    __slots__: List[str] = ['_resource_type', '_resource_rid']

    def __init__(self, resource_rid: str, resource_type: "comments_api_ResourceType") -> None:
        self._resource_type = resource_type
        self._resource_rid = resource_rid

    @builtins.property
    def resource_type(self) -> "comments_api_ResourceType":
        """
        The type of resource that the comment is associated with
        """
        return self._resource_type

    @builtins.property
    def resource_rid(self) -> str:
        """
        The resource identifier for the resource that the comment is associated with. For example, a run or a workbook.
        """
        return self._resource_rid


comments_api_CommentParentResource.__name__ = "CommentParentResource"
comments_api_CommentParentResource.__qualname__ = "CommentParentResource"
comments_api_CommentParentResource.__module__ = "nominal_api.comments_api"


class comments_api_CommentsService(Service):
    """
    Comments service manages conversations about resources.
    """

    def get_conversation(self, auth_header: str, resource_rid: str, resource_type: "comments_api_ResourceType") -> "comments_api_Conversation":
        """
        A conversation is a fully resolved comment tree. It includes all comments for the given resource
and all the nested comments/replies to those comments.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceType': resource_type,
            'resourceRid': resource_rid,
        }

        _json: Any = None

        _path = '/comments/v1/conversations/{resourceType}/{resourceRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Conversation, self._return_none_for_unknown_union_types)

    def get_conversation_count(self, auth_header: str, resource_rid: str, resource_type: "comments_api_ResourceType", include_deleted: Optional[bool] = None) -> int:
        """
        Returns the number of comments in a conversation.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'includeDeleted': include_deleted,
        }

        _path_params: Dict[str, Any] = {
            'resourceType': resource_type,
            'resourceRid': resource_rid,
        }

        _json: Any = None

        _path = '/comments/v1/conversations/{resourceType}/{resourceRid}/count'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), int, self._return_none_for_unknown_union_types)

    def get_comment(self, auth_header: str, comment_rid: str) -> "comments_api_Comment":
        """
        Get a comment identified by its RID
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'commentRid': comment_rid,
        }

        _json: Any = None

        _path = '/comments/v1/comments/{commentRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Comment, self._return_none_for_unknown_union_types)

    def create_comment(self, auth_header: str, request: "comments_api_CreateCommentRequest") -> "comments_api_Comment":
        """
        Create a comment on a resource
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/comments/v1/comments'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Comment, self._return_none_for_unknown_union_types)

    def edit_comment(self, auth_header: str, comment_rid: str, request: "comments_api_EditCommentRequest") -> "comments_api_Comment":
        """
        Edit an existing comment
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'commentRid': comment_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/comments/v1/comments/{commentRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Comment, self._return_none_for_unknown_union_types)

    def delete_comment(self, auth_header: str, comment_rid: str) -> "comments_api_Comment":
        """
        Delete an existing comment
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'commentRid': comment_rid,
        }

        _json: Any = None

        _path = '/comments/v1/comments/{commentRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Comment, self._return_none_for_unknown_union_types)

    def pin_comment(self, auth_header: str, comment_rid: str) -> "comments_api_Comment":
        """
        Pin a comment to the top of the conversation
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'commentRid': comment_rid,
        }

        _json: Any = None

        _path = '/comments/v1/comments/{commentRid}/pin'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Comment, self._return_none_for_unknown_union_types)

    def unpin_comment(self, auth_header: str, comment_rid: str) -> "comments_api_Comment":
        """
        Unpin a comment from the top of the conversation
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'commentRid': comment_rid,
        }

        _json: Any = None

        _path = '/comments/v1/comments/{commentRid}/pin'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Comment, self._return_none_for_unknown_union_types)

    def add_reaction(self, auth_header: str, comment_rid: str, type: "comments_api_ReactionType") -> "comments_api_Comment":
        """
        Create a reaction on a comment
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'commentRid': comment_rid,
            'type': type,
        }

        _json: Any = None

        _path = '/comments/v1/comments/{commentRid}/reactions/{type}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Comment, self._return_none_for_unknown_union_types)

    def remove_reaction(self, auth_header: str, comment_rid: str, type: "comments_api_ReactionType") -> "comments_api_Comment":
        """
        Create a reaction on a comment
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'commentRid': comment_rid,
            'type': type,
        }

        _json: Any = None

        _path = '/comments/v1/comments/{commentRid}/reactions/{type}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), comments_api_Comment, self._return_none_for_unknown_union_types)


comments_api_CommentsService.__name__ = "CommentsService"
comments_api_CommentsService.__qualname__ = "CommentsService"
comments_api_CommentsService.__module__ = "nominal_api.comments_api"


class comments_api_Conversation(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'resource_rid': ConjureFieldDefinition('resourceRid', comments_api_ResourceRid),
            'resource_type': ConjureFieldDefinition('resourceType', comments_api_ResourceType),
            'comments': ConjureFieldDefinition('comments', List[comments_api_ConversationNode])
        }

    __slots__: List[str] = ['_resource_rid', '_resource_type', '_comments']

    def __init__(self, comments: List["comments_api_ConversationNode"], resource_rid: str, resource_type: "comments_api_ResourceType") -> None:
        self._resource_rid = resource_rid
        self._resource_type = resource_type
        self._comments = comments

    @builtins.property
    def resource_rid(self) -> str:
        """
        RID for the resource that the conversation is associated with.
        """
        return self._resource_rid

    @builtins.property
    def resource_type(self) -> "comments_api_ResourceType":
        """
        The type of the resource that the conversation is associated with.
        """
        return self._resource_type

    @builtins.property
    def comments(self) -> List["comments_api_ConversationNode"]:
        """
        The comments on the conversation ordered by creation time.
Empty if the comment has no replies.
        """
        return self._comments


comments_api_Conversation.__name__ = "Conversation"
comments_api_Conversation.__qualname__ = "Conversation"
comments_api_Conversation.__module__ = "nominal_api.comments_api"


class comments_api_ConversationNode(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'comment': ConjureFieldDefinition('comment', comments_api_Comment),
            'replies': ConjureFieldDefinition('replies', List[comments_api_ConversationNode])
        }

    __slots__: List[str] = ['_comment', '_replies']

    def __init__(self, comment: "comments_api_Comment", replies: List["comments_api_ConversationNode"]) -> None:
        self._comment = comment
        self._replies = replies

    @builtins.property
    def comment(self) -> "comments_api_Comment":
        """
        The comment
        """
        return self._comment

    @builtins.property
    def replies(self) -> List["comments_api_ConversationNode"]:
        """
        The comments on (aka replies to) the comment ordered by creation time. Empty if the comment has no replies.
        """
        return self._replies


comments_api_ConversationNode.__name__ = "ConversationNode"
comments_api_ConversationNode.__qualname__ = "ConversationNode"
comments_api_ConversationNode.__module__ = "nominal_api.comments_api"


class comments_api_CreateCommentRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'parent': ConjureFieldDefinition('parent', comments_api_CommentParent),
            'content': ConjureFieldDefinition('content', str),
            'attachments': ConjureFieldDefinition('attachments', List[str])
        }

    __slots__: List[str] = ['_parent', '_content', '_attachments']

    def __init__(self, attachments: List[str], content: str, parent: "comments_api_CommentParent") -> None:
        self._parent = parent
        self._content = content
        self._attachments = attachments

    @builtins.property
    def parent(self) -> "comments_api_CommentParent":
        return self._parent

    @builtins.property
    def content(self) -> str:
        """
        The content of the comment. Markdown supported.
        """
        return self._content

    @builtins.property
    def attachments(self) -> List[str]:
        """
        Attachments to the comment.
        """
        return self._attachments


comments_api_CreateCommentRequest.__name__ = "CreateCommentRequest"
comments_api_CreateCommentRequest.__qualname__ = "CreateCommentRequest"
comments_api_CreateCommentRequest.__module__ = "nominal_api.comments_api"


class comments_api_EditCommentRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'content': ConjureFieldDefinition('content', str),
            'attachments': ConjureFieldDefinition('attachments', List[str])
        }

    __slots__: List[str] = ['_content', '_attachments']

    def __init__(self, attachments: List[str], content: str) -> None:
        self._content = content
        self._attachments = attachments

    @builtins.property
    def content(self) -> str:
        """
        The content of the comment. Markdown supported.
        """
        return self._content

    @builtins.property
    def attachments(self) -> List[str]:
        """
        Attachments to the comment.
        """
        return self._attachments


comments_api_EditCommentRequest.__name__ = "EditCommentRequest"
comments_api_EditCommentRequest.__qualname__ = "EditCommentRequest"
comments_api_EditCommentRequest.__module__ = "nominal_api.comments_api"


class comments_api_Reaction(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', comments_api_ReactionRid),
            'user_rid': ConjureFieldDefinition('userRid', str),
            'created_at': ConjureFieldDefinition('createdAt', str),
            'type': ConjureFieldDefinition('type', comments_api_ReactionType)
        }

    __slots__: List[str] = ['_rid', '_user_rid', '_created_at', '_type']

    def __init__(self, created_at: str, rid: str, type: "comments_api_ReactionType", user_rid: str) -> None:
        self._rid = rid
        self._user_rid = user_rid
        self._created_at = created_at
        self._type = type

    @builtins.property
    def rid(self) -> str:
        """
        Unique resource identifier for the reaction
        """
        return self._rid

    @builtins.property
    def user_rid(self) -> str:
        """
        The user who authored the reaction
        """
        return self._user_rid

    @builtins.property
    def created_at(self) -> str:
        """
        The time the reaction was created
        """
        return self._created_at

    @builtins.property
    def type(self) -> "comments_api_ReactionType":
        """
        The type of reaction
        """
        return self._type


comments_api_Reaction.__name__ = "Reaction"
comments_api_Reaction.__qualname__ = "Reaction"
comments_api_Reaction.__module__ = "nominal_api.comments_api"


class comments_api_ReactionType(ConjureEnumType):

    LIKE = 'LIKE'
    '''LIKE'''
    DISLIKE = 'DISLIKE'
    '''DISLIKE'''
    HEART = 'HEART'
    '''HEART'''
    HOORAY = 'HOORAY'
    '''HOORAY'''
    ROCKET = 'ROCKET'
    '''ROCKET'''
    EYES = 'EYES'
    '''EYES'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


comments_api_ReactionType.__name__ = "ReactionType"
comments_api_ReactionType.__qualname__ = "ReactionType"
comments_api_ReactionType.__module__ = "nominal_api.comments_api"


class comments_api_ResourceType(ConjureEnumType):

    RUN = 'RUN'
    '''RUN'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


comments_api_ResourceType.__name__ = "ResourceType"
comments_api_ResourceType.__qualname__ = "ResourceType"
comments_api_ResourceType.__module__ = "nominal_api.comments_api"


class datasource_Property(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', datasource_PropertyName),
            'value': ConjureFieldDefinition('value', datasource_PropertyValue)
        }

    __slots__: List[str] = ['_name', '_value']

    def __init__(self, name: str, value: str) -> None:
        self._name = name
        self._value = value

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def value(self) -> str:
        return self._value


datasource_Property.__name__ = "Property"
datasource_Property.__qualname__ = "Property"
datasource_Property.__module__ = "nominal_api.datasource"


class datasource_TimestampType(ConjureEnumType):

    RELATIVE = 'RELATIVE'
    '''RELATIVE'''
    ABSOLUTE = 'ABSOLUTE'
    '''ABSOLUTE'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


datasource_TimestampType.__name__ = "TimestampType"
datasource_TimestampType.__qualname__ = "TimestampType"
datasource_TimestampType.__module__ = "nominal_api.datasource"


class datasource_api_BatchGetChannelPrefixTreeRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_source_rids': ConjureFieldDefinition('dataSourceRids', List[api_rids_DataSourceRid])
        }

    __slots__: List[str] = ['_data_source_rids']

    def __init__(self, data_source_rids: List[str]) -> None:
        self._data_source_rids = data_source_rids

    @builtins.property
    def data_source_rids(self) -> List[str]:
        return self._data_source_rids


datasource_api_BatchGetChannelPrefixTreeRequest.__name__ = "BatchGetChannelPrefixTreeRequest"
datasource_api_BatchGetChannelPrefixTreeRequest.__qualname__ = "BatchGetChannelPrefixTreeRequest"
datasource_api_BatchGetChannelPrefixTreeRequest.__module__ = "nominal_api.datasource_api"


class datasource_api_BatchGetChannelPrefixTreeResponse(ConjureBeanType):
    """
    If the tree for a data source has not been indexed, it will be omitted from the map.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'channel_prefix_trees': ConjureFieldDefinition('channelPrefixTrees', Dict[api_rids_DataSourceRid, datasource_api_ChannelPrefixTree])
        }

    __slots__: List[str] = ['_channel_prefix_trees']

    def __init__(self, channel_prefix_trees: Dict[str, "datasource_api_ChannelPrefixTree"]) -> None:
        self._channel_prefix_trees = channel_prefix_trees

    @builtins.property
    def channel_prefix_trees(self) -> Dict[str, "datasource_api_ChannelPrefixTree"]:
        return self._channel_prefix_trees


datasource_api_BatchGetChannelPrefixTreeResponse.__name__ = "BatchGetChannelPrefixTreeResponse"
datasource_api_BatchGetChannelPrefixTreeResponse.__qualname__ = "BatchGetChannelPrefixTreeResponse"
datasource_api_BatchGetChannelPrefixTreeResponse.__module__ = "nominal_api.datasource_api"


class datasource_api_ChannelMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', api_Channel),
            'data_source': ConjureFieldDefinition('dataSource', api_rids_DataSourceRid),
            'unit': ConjureFieldDefinition('unit', OptionalTypeWrapper[scout_run_api_Unit]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'data_type': ConjureFieldDefinition('dataType', OptionalTypeWrapper[api_SeriesDataType]),
            'series_rid': ConjureFieldDefinition('seriesRid', datasource_api_SeriesArchetypeRidOrLogicalSeriesRid)
        }

    __slots__: List[str] = ['_name', '_data_source', '_unit', '_description', '_data_type', '_series_rid']

    def __init__(self, data_source: str, name: str, series_rid: "datasource_api_SeriesArchetypeRidOrLogicalSeriesRid", data_type: Optional["api_SeriesDataType"] = None, description: Optional[str] = None, unit: Optional["scout_run_api_Unit"] = None) -> None:
        self._name = name
        self._data_source = data_source
        self._unit = unit
        self._description = description
        self._data_type = data_type
        self._series_rid = series_rid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def data_source(self) -> str:
        return self._data_source

    @builtins.property
    def unit(self) -> Optional["scout_run_api_Unit"]:
        return self._unit

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def data_type(self) -> Optional["api_SeriesDataType"]:
        return self._data_type

    @builtins.property
    def series_rid(self) -> "datasource_api_SeriesArchetypeRidOrLogicalSeriesRid":
        return self._series_rid


datasource_api_ChannelMetadata.__name__ = "ChannelMetadata"
datasource_api_ChannelMetadata.__qualname__ = "ChannelMetadata"
datasource_api_ChannelMetadata.__module__ = "nominal_api.datasource_api"


class datasource_api_ChannelOrPrefix(ConjureUnionType):
    _channel: Optional["datasource_api_ChannelMetadata"] = None
    _prefix: Optional["datasource_api_DataSourcePrefixNode"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'channel': ConjureFieldDefinition('channel', datasource_api_ChannelMetadata),
            'prefix': ConjureFieldDefinition('prefix', datasource_api_DataSourcePrefixNode)
        }

    def __init__(
            self,
            channel: Optional["datasource_api_ChannelMetadata"] = None,
            prefix: Optional["datasource_api_DataSourcePrefixNode"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (channel is not None) + (prefix is not None) != 1:
                raise ValueError('a union must contain a single member')

            if channel is not None:
                self._channel = channel
                self._type = 'channel'
            if prefix is not None:
                self._prefix = prefix
                self._type = 'prefix'

        elif type_of_union == 'channel':
            if channel is None:
                raise ValueError('a union value must not be None')
            self._channel = channel
            self._type = 'channel'
        elif type_of_union == 'prefix':
            if prefix is None:
                raise ValueError('a union value must not be None')
            self._prefix = prefix
            self._type = 'prefix'

    @builtins.property
    def channel(self) -> Optional["datasource_api_ChannelMetadata"]:
        return self._channel

    @builtins.property
    def prefix(self) -> Optional["datasource_api_DataSourcePrefixNode"]:
        return self._prefix

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, datasource_api_ChannelOrPrefixVisitor):
            raise ValueError('{} is not an instance of datasource_api_ChannelOrPrefixVisitor'.format(visitor.__class__.__name__))
        if self._type == 'channel' and self.channel is not None:
            return visitor._channel(self.channel)
        if self._type == 'prefix' and self.prefix is not None:
            return visitor._prefix(self.prefix)


datasource_api_ChannelOrPrefix.__name__ = "ChannelOrPrefix"
datasource_api_ChannelOrPrefix.__qualname__ = "ChannelOrPrefix"
datasource_api_ChannelOrPrefix.__module__ = "nominal_api.datasource_api"


class datasource_api_ChannelOrPrefixVisitor:

    @abstractmethod
    def _channel(self, channel: "datasource_api_ChannelMetadata") -> Any:
        pass

    @abstractmethod
    def _prefix(self, prefix: "datasource_api_DataSourcePrefixNode") -> Any:
        pass


datasource_api_ChannelOrPrefixVisitor.__name__ = "ChannelOrPrefixVisitor"
datasource_api_ChannelOrPrefixVisitor.__qualname__ = "ChannelOrPrefixVisitor"
datasource_api_ChannelOrPrefixVisitor.__module__ = "nominal_api.datasource_api"


class datasource_api_ChannelPrefixTree(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'roots': ConjureFieldDefinition('roots', List[datasource_api_ChannelPrefixTreeNode]),
            'delimiter': ConjureFieldDefinition('delimiter', str)
        }

    __slots__: List[str] = ['_roots', '_delimiter']

    def __init__(self, delimiter: str, roots: List["datasource_api_ChannelPrefixTreeNode"]) -> None:
        self._roots = roots
        self._delimiter = delimiter

    @builtins.property
    def roots(self) -> List["datasource_api_ChannelPrefixTreeNode"]:
        return self._roots

    @builtins.property
    def delimiter(self) -> str:
        return self._delimiter


datasource_api_ChannelPrefixTree.__name__ = "ChannelPrefixTree"
datasource_api_ChannelPrefixTree.__qualname__ = "ChannelPrefixTree"
datasource_api_ChannelPrefixTree.__module__ = "nominal_api.datasource_api"


class datasource_api_ChannelPrefixTreeNode(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'part': ConjureFieldDefinition('part', str),
            'children': ConjureFieldDefinition('children', List[datasource_api_ChannelPrefixTreeNode])
        }

    __slots__: List[str] = ['_part', '_children']

    def __init__(self, children: List["datasource_api_ChannelPrefixTreeNode"], part: str) -> None:
        self._part = part
        self._children = children

    @builtins.property
    def part(self) -> str:
        """
        Should be combined with the ancestor parts and the delimiter to form the full prefix.
        """
        return self._part

    @builtins.property
    def children(self) -> List["datasource_api_ChannelPrefixTreeNode"]:
        return self._children


datasource_api_ChannelPrefixTreeNode.__name__ = "ChannelPrefixTreeNode"
datasource_api_ChannelPrefixTreeNode.__qualname__ = "ChannelPrefixTreeNode"
datasource_api_ChannelPrefixTreeNode.__module__ = "nominal_api.datasource_api"


class datasource_api_ChannelWithAvailableTags(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_source_rid': ConjureFieldDefinition('dataSourceRid', api_rids_DataSourceRid),
            'channel': ConjureFieldDefinition('channel', api_Channel),
            'available_tags': ConjureFieldDefinition('availableTags', Dict[api_TagName, List[api_TagValue]])
        }

    __slots__: List[str] = ['_data_source_rid', '_channel', '_available_tags']

    def __init__(self, available_tags: Dict[str, List[str]], channel: str, data_source_rid: str) -> None:
        self._data_source_rid = data_source_rid
        self._channel = channel
        self._available_tags = available_tags

    @builtins.property
    def data_source_rid(self) -> str:
        return self._data_source_rid

    @builtins.property
    def channel(self) -> str:
        return self._channel

    @builtins.property
    def available_tags(self) -> Dict[str, List[str]]:
        """
        A set of tag keys and their values given the initial set of filters. The initial tag filters
will be included in the map with their corresponding values.
        """
        return self._available_tags


datasource_api_ChannelWithAvailableTags.__name__ = "ChannelWithAvailableTags"
datasource_api_ChannelWithAvailableTags.__qualname__ = "ChannelWithAvailableTags"
datasource_api_ChannelWithAvailableTags.__module__ = "nominal_api.datasource_api"


class datasource_api_ChannelWithTagFilters(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_source_rid': ConjureFieldDefinition('dataSourceRid', api_rids_DataSourceRid),
            'channel': ConjureFieldDefinition('channel', api_Channel),
            'tag_filters': ConjureFieldDefinition('tagFilters', Dict[api_TagName, api_TagValue])
        }

    __slots__: List[str] = ['_data_source_rid', '_channel', '_tag_filters']

    def __init__(self, channel: str, data_source_rid: str, tag_filters: Dict[str, str]) -> None:
        self._data_source_rid = data_source_rid
        self._channel = channel
        self._tag_filters = tag_filters

    @builtins.property
    def data_source_rid(self) -> str:
        return self._data_source_rid

    @builtins.property
    def channel(self) -> str:
        return self._channel

    @builtins.property
    def tag_filters(self) -> Dict[str, str]:
        return self._tag_filters


datasource_api_ChannelWithTagFilters.__name__ = "ChannelWithTagFilters"
datasource_api_ChannelWithTagFilters.__qualname__ = "ChannelWithTagFilters"
datasource_api_ChannelWithTagFilters.__module__ = "nominal_api.datasource_api"


class datasource_api_DataSourcePrefixNode(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'part': ConjureFieldDefinition('part', str),
            'data_source': ConjureFieldDefinition('dataSource', api_rids_DataSourceRid)
        }

    __slots__: List[str] = ['_part', '_data_source']

    def __init__(self, data_source: str, part: str) -> None:
        self._part = part
        self._data_source = data_source

    @builtins.property
    def part(self) -> str:
        """
        Should be combined with the ancestor parts and the delimiter to form the full prefix.
        """
        return self._part

    @builtins.property
    def data_source(self) -> str:
        return self._data_source


datasource_api_DataSourcePrefixNode.__name__ = "DataSourcePrefixNode"
datasource_api_DataSourcePrefixNode.__qualname__ = "DataSourcePrefixNode"
datasource_api_DataSourcePrefixNode.__module__ = "nominal_api.datasource_api"


class datasource_api_GetAvailableTagsForChannelRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'channel_with_tag_filters': ConjureFieldDefinition('channelWithTagFilters', datasource_api_ChannelWithTagFilters),
            'start_time': ConjureFieldDefinition('startTime', scout_run_api_UtcTimestamp),
            'end_time': ConjureFieldDefinition('endTime', scout_run_api_UtcTimestamp)
        }

    __slots__: List[str] = ['_channel_with_tag_filters', '_start_time', '_end_time']

    def __init__(self, channel_with_tag_filters: "datasource_api_ChannelWithTagFilters", end_time: "scout_run_api_UtcTimestamp", start_time: "scout_run_api_UtcTimestamp") -> None:
        self._channel_with_tag_filters = channel_with_tag_filters
        self._start_time = start_time
        self._end_time = end_time

    @builtins.property
    def channel_with_tag_filters(self) -> "datasource_api_ChannelWithTagFilters":
        return self._channel_with_tag_filters

    @builtins.property
    def start_time(self) -> "scout_run_api_UtcTimestamp":
        return self._start_time

    @builtins.property
    def end_time(self) -> "scout_run_api_UtcTimestamp":
        return self._end_time


datasource_api_GetAvailableTagsForChannelRequest.__name__ = "GetAvailableTagsForChannelRequest"
datasource_api_GetAvailableTagsForChannelRequest.__qualname__ = "GetAvailableTagsForChannelRequest"
datasource_api_GetAvailableTagsForChannelRequest.__module__ = "nominal_api.datasource_api"


class datasource_api_GetAvailableTagsForChannelResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'available_tags': ConjureFieldDefinition('availableTags', datasource_api_ChannelWithAvailableTags)
        }

    __slots__: List[str] = ['_available_tags']

    def __init__(self, available_tags: "datasource_api_ChannelWithAvailableTags") -> None:
        self._available_tags = available_tags

    @builtins.property
    def available_tags(self) -> "datasource_api_ChannelWithAvailableTags":
        return self._available_tags


datasource_api_GetAvailableTagsForChannelResponse.__name__ = "GetAvailableTagsForChannelResponse"
datasource_api_GetAvailableTagsForChannelResponse.__qualname__ = "GetAvailableTagsForChannelResponse"
datasource_api_GetAvailableTagsForChannelResponse.__module__ = "nominal_api.datasource_api"


class datasource_api_IndexChannelPrefixTreeRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_source_rid': ConjureFieldDefinition('dataSourceRid', api_rids_DataSourceRid),
            'delimiter': ConjureFieldDefinition('delimiter', str)
        }

    __slots__: List[str] = ['_data_source_rid', '_delimiter']

    def __init__(self, data_source_rid: str, delimiter: str) -> None:
        self._data_source_rid = data_source_rid
        self._delimiter = delimiter

    @builtins.property
    def data_source_rid(self) -> str:
        return self._data_source_rid

    @builtins.property
    def delimiter(self) -> str:
        return self._delimiter


datasource_api_IndexChannelPrefixTreeRequest.__name__ = "IndexChannelPrefixTreeRequest"
datasource_api_IndexChannelPrefixTreeRequest.__qualname__ = "IndexChannelPrefixTreeRequest"
datasource_api_IndexChannelPrefixTreeRequest.__module__ = "nominal_api.datasource_api"


class datasource_api_SearchChannelsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'fuzzy_search_text': ConjureFieldDefinition('fuzzySearchText', str),
            'prefix': ConjureFieldDefinition('prefix', OptionalTypeWrapper[str]),
            'exact_match': ConjureFieldDefinition('exactMatch', List[str]),
            'data_sources': ConjureFieldDefinition('dataSources', List[api_rids_DataSourceRid]),
            'previously_selected_channels': ConjureFieldDefinition('previouslySelectedChannels', OptionalTypeWrapper[Dict[api_rids_DataSourceRid, List[api_Channel]]]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[datasource_Token]),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_fuzzy_search_text', '_prefix', '_exact_match', '_data_sources', '_previously_selected_channels', '_next_page_token', '_page_size']

    def __init__(self, data_sources: List[str], exact_match: List[str], fuzzy_search_text: str, next_page_token: Optional[str] = None, page_size: Optional[int] = None, prefix: Optional[str] = None, previously_selected_channels: Optional[Dict[str, List[str]]] = None) -> None:
        self._fuzzy_search_text = fuzzy_search_text
        self._prefix = prefix
        self._exact_match = exact_match
        self._data_sources = data_sources
        self._previously_selected_channels = previously_selected_channels
        self._next_page_token = next_page_token
        self._page_size = page_size

    @builtins.property
    def fuzzy_search_text(self) -> str:
        return self._fuzzy_search_text

    @builtins.property
    def prefix(self) -> Optional[str]:
        return self._prefix

    @builtins.property
    def exact_match(self) -> List[str]:
        """
        Will return only channels that contain all strings specified as exact matches (case insensitive).
        """
        return self._exact_match

    @builtins.property
    def data_sources(self) -> List[str]:
        return self._data_sources

    @builtins.property
    def previously_selected_channels(self) -> Optional[Dict[str, List[str]]]:
        return self._previously_selected_channels

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Defaults to 1000. Will throw if larger than 1000.
        """
        return self._page_size


datasource_api_SearchChannelsRequest.__name__ = "SearchChannelsRequest"
datasource_api_SearchChannelsRequest.__qualname__ = "SearchChannelsRequest"
datasource_api_SearchChannelsRequest.__module__ = "nominal_api.datasource_api"


class datasource_api_SearchChannelsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[datasource_api_ChannelMetadata]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[datasource_Token])
        }

    __slots__: List[str] = ['_results', '_next_page_token']

    def __init__(self, results: List["datasource_api_ChannelMetadata"], next_page_token: Optional[str] = None) -> None:
        self._results = results
        self._next_page_token = next_page_token

    @builtins.property
    def results(self) -> List["datasource_api_ChannelMetadata"]:
        return self._results

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


datasource_api_SearchChannelsResponse.__name__ = "SearchChannelsResponse"
datasource_api_SearchChannelsResponse.__qualname__ = "SearchChannelsResponse"
datasource_api_SearchChannelsResponse.__module__ = "nominal_api.datasource_api"


class datasource_api_SearchFilteredChannelsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'fuzzy_search_text': ConjureFieldDefinition('fuzzySearchText', str),
            'exact_match': ConjureFieldDefinition('exactMatch', List[str]),
            'data_sources': ConjureFieldDefinition('dataSources', List[api_rids_DataSourceRid]),
            'previously_selected_channels': ConjureFieldDefinition('previouslySelectedChannels', OptionalTypeWrapper[Dict[api_rids_DataSourceRid, List[api_Channel]]]),
            'result_size': ConjureFieldDefinition('resultSize', OptionalTypeWrapper[int]),
            'tags': ConjureFieldDefinition('tags', Dict[api_rids_DataSourceRid, Dict[api_TagName, api_TagValue]]),
            'min_data_updated_time': ConjureFieldDefinition('minDataUpdatedTime', OptionalTypeWrapper[scout_run_api_UtcTimestamp]),
            'max_data_start_time': ConjureFieldDefinition('maxDataStartTime', OptionalTypeWrapper[scout_run_api_UtcTimestamp])
        }

    __slots__: List[str] = ['_fuzzy_search_text', '_exact_match', '_data_sources', '_previously_selected_channels', '_result_size', '_tags', '_min_data_updated_time', '_max_data_start_time']

    def __init__(self, data_sources: List[str], exact_match: List[str], fuzzy_search_text: str, tags: Dict[str, Dict[str, str]], max_data_start_time: Optional["scout_run_api_UtcTimestamp"] = None, min_data_updated_time: Optional["scout_run_api_UtcTimestamp"] = None, previously_selected_channels: Optional[Dict[str, List[str]]] = None, result_size: Optional[int] = None) -> None:
        self._fuzzy_search_text = fuzzy_search_text
        self._exact_match = exact_match
        self._data_sources = data_sources
        self._previously_selected_channels = previously_selected_channels
        self._result_size = result_size
        self._tags = tags
        self._min_data_updated_time = min_data_updated_time
        self._max_data_start_time = max_data_start_time

    @builtins.property
    def fuzzy_search_text(self) -> str:
        return self._fuzzy_search_text

    @builtins.property
    def exact_match(self) -> List[str]:
        """
        Will return only channels that contain all strings specified as exact matches (case insensitive).
        """
        return self._exact_match

    @builtins.property
    def data_sources(self) -> List[str]:
        return self._data_sources

    @builtins.property
    def previously_selected_channels(self) -> Optional[Dict[str, List[str]]]:
        return self._previously_selected_channels

    @builtins.property
    def result_size(self) -> Optional[int]:
        """
        Defaults to 200. Will throw if larger than 200.
        """
        return self._result_size

    @builtins.property
    def tags(self) -> Dict[str, Dict[str, str]]:
        """
        For each data source specified as a key, search will only return channels containing a superset of the 
tags specified for that given datasource. If a data source is present in the dataSources field but not
in this map, or if a data source points to an empty map of tags, it will be searched without tag filters.
        """
        return self._tags

    @builtins.property
    def min_data_updated_time(self) -> Optional["scout_run_api_UtcTimestamp"]:
        """
        If specified, search will only return channels that have had new data after the specified time.
        """
        return self._min_data_updated_time

    @builtins.property
    def max_data_start_time(self) -> Optional["scout_run_api_UtcTimestamp"]:
        """
        If specified, search will only return channels that have data before the specified time.
        """
        return self._max_data_start_time


datasource_api_SearchFilteredChannelsRequest.__name__ = "SearchFilteredChannelsRequest"
datasource_api_SearchFilteredChannelsRequest.__qualname__ = "SearchFilteredChannelsRequest"
datasource_api_SearchFilteredChannelsRequest.__module__ = "nominal_api.datasource_api"


class datasource_api_SearchFilteredChannelsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[datasource_api_ChannelMetadata])
        }

    __slots__: List[str] = ['_results']

    def __init__(self, results: List["datasource_api_ChannelMetadata"]) -> None:
        self._results = results

    @builtins.property
    def results(self) -> List["datasource_api_ChannelMetadata"]:
        return self._results


datasource_api_SearchFilteredChannelsResponse.__name__ = "SearchFilteredChannelsResponse"
datasource_api_SearchFilteredChannelsResponse.__qualname__ = "SearchFilteredChannelsResponse"
datasource_api_SearchFilteredChannelsResponse.__module__ = "nominal_api.datasource_api"


class datasource_api_SearchHierarchicalChannelsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'parent': ConjureFieldDefinition('parent', List[str]),
            'data_sources': ConjureFieldDefinition('dataSources', List[api_rids_DataSourceRid])
        }

    __slots__: List[str] = ['_parent', '_data_sources']

    def __init__(self, data_sources: List[str], parent: List[str]) -> None:
        self._parent = parent
        self._data_sources = data_sources

    @builtins.property
    def parent(self) -> List[str]:
        """
        The parent (represented as a list of parts) to search under. If empty, will return all top-level channels.
        """
        return self._parent

    @builtins.property
    def data_sources(self) -> List[str]:
        return self._data_sources


datasource_api_SearchHierarchicalChannelsRequest.__name__ = "SearchHierarchicalChannelsRequest"
datasource_api_SearchHierarchicalChannelsRequest.__qualname__ = "SearchHierarchicalChannelsRequest"
datasource_api_SearchHierarchicalChannelsRequest.__module__ = "nominal_api.datasource_api"


class datasource_api_SearchHierarchicalChannelsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[datasource_api_ChannelOrPrefix])
        }

    __slots__: List[str] = ['_results']

    def __init__(self, results: List["datasource_api_ChannelOrPrefix"]) -> None:
        self._results = results

    @builtins.property
    def results(self) -> List["datasource_api_ChannelOrPrefix"]:
        return self._results


datasource_api_SearchHierarchicalChannelsResponse.__name__ = "SearchHierarchicalChannelsResponse"
datasource_api_SearchHierarchicalChannelsResponse.__qualname__ = "SearchHierarchicalChannelsResponse"
datasource_api_SearchHierarchicalChannelsResponse.__module__ = "nominal_api.datasource_api"


class datasource_api_SeriesArchetypeRidOrLogicalSeriesRid(ConjureUnionType):
    _series_archetype: Optional[str] = None
    _logical_series: Optional[str] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'series_archetype': ConjureFieldDefinition('seriesArchetype', api_SeriesArchetypeRid),
            'logical_series': ConjureFieldDefinition('logicalSeries', api_LogicalSeriesRid)
        }

    def __init__(
            self,
            series_archetype: Optional[str] = None,
            logical_series: Optional[str] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (series_archetype is not None) + (logical_series is not None) != 1:
                raise ValueError('a union must contain a single member')

            if series_archetype is not None:
                self._series_archetype = series_archetype
                self._type = 'seriesArchetype'
            if logical_series is not None:
                self._logical_series = logical_series
                self._type = 'logicalSeries'

        elif type_of_union == 'seriesArchetype':
            if series_archetype is None:
                raise ValueError('a union value must not be None')
            self._series_archetype = series_archetype
            self._type = 'seriesArchetype'
        elif type_of_union == 'logicalSeries':
            if logical_series is None:
                raise ValueError('a union value must not be None')
            self._logical_series = logical_series
            self._type = 'logicalSeries'

    @builtins.property
    def series_archetype(self) -> Optional[str]:
        return self._series_archetype

    @builtins.property
    def logical_series(self) -> Optional[str]:
        return self._logical_series

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, datasource_api_SeriesArchetypeRidOrLogicalSeriesRidVisitor):
            raise ValueError('{} is not an instance of datasource_api_SeriesArchetypeRidOrLogicalSeriesRidVisitor'.format(visitor.__class__.__name__))
        if self._type == 'seriesArchetype' and self.series_archetype is not None:
            return visitor._series_archetype(self.series_archetype)
        if self._type == 'logicalSeries' and self.logical_series is not None:
            return visitor._logical_series(self.logical_series)


datasource_api_SeriesArchetypeRidOrLogicalSeriesRid.__name__ = "SeriesArchetypeRidOrLogicalSeriesRid"
datasource_api_SeriesArchetypeRidOrLogicalSeriesRid.__qualname__ = "SeriesArchetypeRidOrLogicalSeriesRid"
datasource_api_SeriesArchetypeRidOrLogicalSeriesRid.__module__ = "nominal_api.datasource_api"


class datasource_api_SeriesArchetypeRidOrLogicalSeriesRidVisitor:

    @abstractmethod
    def _series_archetype(self, series_archetype: str) -> Any:
        pass

    @abstractmethod
    def _logical_series(self, logical_series: str) -> Any:
        pass


datasource_api_SeriesArchetypeRidOrLogicalSeriesRidVisitor.__name__ = "SeriesArchetypeRidOrLogicalSeriesRidVisitor"
datasource_api_SeriesArchetypeRidOrLogicalSeriesRidVisitor.__qualname__ = "SeriesArchetypeRidOrLogicalSeriesRidVisitor"
datasource_api_SeriesArchetypeRidOrLogicalSeriesRidVisitor.__module__ = "nominal_api.datasource_api"


class datasource_logset_LogSetService(Service):
    """
    Log sets are a type of datasource which can be used to store log data.
    """

    def create(self, auth_header: str, request: "datasource_logset_api_CreateLogSetRequest") -> "datasource_logset_api_LogSetMetadata":
        """
        Creates a log set, to which logs can be attached using `attach-and-finalize`. The logs within a logset are
not searchable until the logset is finalized.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/data-source/log-set/v1/log-sets'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), datasource_logset_api_LogSetMetadata, self._return_none_for_unknown_union_types)

    def attach_logs_and_finalize(self, auth_header: str, log_set_rid: str, request: "datasource_logset_api_AttachLogsAndFinalizeRequest") -> "datasource_logset_api_LogSetMetadata":
        """
        A logset is not considered readable until it has been finalized. Once finalized, the logset is immutable.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'logSetRid': log_set_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/data-source/log-set/v1/log-set/{logSetRid}/attach-and-finalize'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), datasource_logset_api_LogSetMetadata, self._return_none_for_unknown_union_types)

    def get_log_set_metadata(self, auth_header: str, log_set_rid: str) -> "datasource_logset_api_LogSetMetadata":
        """
        Returns metadata about a log set.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'logSetRid': log_set_rid,
        }

        _json: Any = None

        _path = '/data-source/log-set/v1/log-set/{logSetRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), datasource_logset_api_LogSetMetadata, self._return_none_for_unknown_union_types)

    def batch_get_log_set_metadata(self, auth_header: str, request: List[str] = None) -> List["datasource_logset_api_LogSetMetadata"]:
        """
        Returns metadata about a set of log sets.
        """
        request = request if request is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/data-source/log-set/v1/log-set/multiple'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[datasource_logset_api_LogSetMetadata], self._return_none_for_unknown_union_types)

    def search_log_sets(self, auth_header: str, request: "datasource_logset_api_SearchLogSetsRequest") -> "datasource_logset_api_SearchLogSetsResponse":
        """
        Searches all visible log sets based on log set metadata.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/data-source/log-set/v1/log-sets/get'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), datasource_logset_api_SearchLogSetsResponse, self._return_none_for_unknown_union_types)

    def search_logs(self, auth_header: str, log_set_rid: str, request: "datasource_logset_api_SearchLogsRequest") -> "datasource_logset_api_SearchLogsResponse":
        """
        Searches logs within a single log set.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'logSetRid': log_set_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/data-source/log-set/v1/log-set/{logSetRid}/logs'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), datasource_logset_api_SearchLogsResponse, self._return_none_for_unknown_union_types)


datasource_logset_LogSetService.__name__ = "LogSetService"
datasource_logset_LogSetService.__qualname__ = "LogSetService"
datasource_logset_LogSetService.__module__ = "nominal_api.datasource_logset"


class datasource_logset_api_AttachLogsAndFinalizeRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'logs': ConjureFieldDefinition('logs', List[datasource_logset_api_Log])
        }

    __slots__: List[str] = ['_logs']

    def __init__(self, logs: List["datasource_logset_api_Log"]) -> None:
        self._logs = logs

    @builtins.property
    def logs(self) -> List["datasource_logset_api_Log"]:
        return self._logs


datasource_logset_api_AttachLogsAndFinalizeRequest.__name__ = "AttachLogsAndFinalizeRequest"
datasource_logset_api_AttachLogsAndFinalizeRequest.__qualname__ = "AttachLogsAndFinalizeRequest"
datasource_logset_api_AttachLogsAndFinalizeRequest.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_BasicLogBody(ConjureBeanType):
    """
    A largely unopinionated but very flexible format.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'properties': ConjureFieldDefinition('properties', Dict[str, str]),
            'message': ConjureFieldDefinition('message', str)
        }

    __slots__: List[str] = ['_properties', '_message']

    def __init__(self, message: str, properties: Dict[str, str]) -> None:
        self._properties = properties
        self._message = message

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def message(self) -> str:
        return self._message


datasource_logset_api_BasicLogBody.__name__ = "BasicLogBody"
datasource_logset_api_BasicLogBody.__qualname__ = "BasicLogBody"
datasource_logset_api_BasicLogBody.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_CreateLogSetRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', str),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'origin_metadata': ConjureFieldDefinition('originMetadata', Dict[str, str]),
            'timestamp_type': ConjureFieldDefinition('timestampType', datasource_TimestampType)
        }

    __slots__: List[str] = ['_name', '_description', '_origin_metadata', '_timestamp_type']

    def __init__(self, name: str, origin_metadata: Dict[str, str], timestamp_type: "datasource_TimestampType", description: Optional[str] = None) -> None:
        self._name = name
        self._description = description
        self._origin_metadata = origin_metadata
        self._timestamp_type = timestamp_type

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def origin_metadata(self) -> Dict[str, str]:
        return self._origin_metadata

    @builtins.property
    def timestamp_type(self) -> "datasource_TimestampType":
        return self._timestamp_type


datasource_logset_api_CreateLogSetRequest.__name__ = "CreateLogSetRequest"
datasource_logset_api_CreateLogSetRequest.__qualname__ = "CreateLogSetRequest"
datasource_logset_api_CreateLogSetRequest.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_Log(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'time': ConjureFieldDefinition('time', api_Timestamp),
            'body': ConjureFieldDefinition('body', datasource_logset_api_LogBody)
        }

    __slots__: List[str] = ['_time', '_body']

    def __init__(self, body: "datasource_logset_api_LogBody", time: "api_Timestamp") -> None:
        self._time = time
        self._body = body

    @builtins.property
    def time(self) -> "api_Timestamp":
        return self._time

    @builtins.property
    def body(self) -> "datasource_logset_api_LogBody":
        return self._body


datasource_logset_api_Log.__name__ = "Log"
datasource_logset_api_Log.__qualname__ = "Log"
datasource_logset_api_Log.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_LogBody(ConjureUnionType):
    _basic: Optional["datasource_logset_api_BasicLogBody"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'basic': ConjureFieldDefinition('basic', datasource_logset_api_BasicLogBody)
        }

    def __init__(
            self,
            basic: Optional["datasource_logset_api_BasicLogBody"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (basic is not None) != 1:
                raise ValueError('a union must contain a single member')

            if basic is not None:
                self._basic = basic
                self._type = 'basic'

        elif type_of_union == 'basic':
            if basic is None:
                raise ValueError('a union value must not be None')
            self._basic = basic
            self._type = 'basic'

    @builtins.property
    def basic(self) -> Optional["datasource_logset_api_BasicLogBody"]:
        return self._basic

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, datasource_logset_api_LogBodyVisitor):
            raise ValueError('{} is not an instance of datasource_logset_api_LogBodyVisitor'.format(visitor.__class__.__name__))
        if self._type == 'basic' and self.basic is not None:
            return visitor._basic(self.basic)


datasource_logset_api_LogBody.__name__ = "LogBody"
datasource_logset_api_LogBody.__qualname__ = "LogBody"
datasource_logset_api_LogBody.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_LogBodyVisitor:

    @abstractmethod
    def _basic(self, basic: "datasource_logset_api_BasicLogBody") -> Any:
        pass


datasource_logset_api_LogBodyVisitor.__name__ = "LogBodyVisitor"
datasource_logset_api_LogBodyVisitor.__qualname__ = "LogBodyVisitor"
datasource_logset_api_LogBodyVisitor.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_LogFormat(ConjureEnumType):

    V1 = 'V1'
    '''V1'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


datasource_logset_api_LogFormat.__name__ = "LogFormat"
datasource_logset_api_LogFormat.__qualname__ = "LogFormat"
datasource_logset_api_LogFormat.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_LogHandle(ConjureUnionType):
    _s3: Optional["datasource_logset_api_S3LogHandle"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            's3': ConjureFieldDefinition('s3', datasource_logset_api_S3LogHandle)
        }

    def __init__(
            self,
            s3: Optional["datasource_logset_api_S3LogHandle"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (s3 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if s3 is not None:
                self._s3 = s3
                self._type = 's3'

        elif type_of_union == 's3':
            if s3 is None:
                raise ValueError('a union value must not be None')
            self._s3 = s3
            self._type = 's3'

    @builtins.property
    def s3(self) -> Optional["datasource_logset_api_S3LogHandle"]:
        return self._s3

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, datasource_logset_api_LogHandleVisitor):
            raise ValueError('{} is not an instance of datasource_logset_api_LogHandleVisitor'.format(visitor.__class__.__name__))
        if self._type == 's3' and self.s3 is not None:
            return visitor._s3(self.s3)


datasource_logset_api_LogHandle.__name__ = "LogHandle"
datasource_logset_api_LogHandle.__qualname__ = "LogHandle"
datasource_logset_api_LogHandle.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_LogHandleVisitor:

    @abstractmethod
    def _s3(self, s3: "datasource_logset_api_S3LogHandle") -> Any:
        pass


datasource_logset_api_LogHandleVisitor.__name__ = "LogHandleVisitor"
datasource_logset_api_LogHandleVisitor.__qualname__ = "LogHandleVisitor"
datasource_logset_api_LogHandleVisitor.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_LogSetMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', datasource_logset_api_LogSetRid),
            'created_by': ConjureFieldDefinition('createdBy', str),
            'name': ConjureFieldDefinition('name', str),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'origin_metadata': ConjureFieldDefinition('originMetadata', Dict[str, str]),
            'created_at': ConjureFieldDefinition('createdAt', str),
            'updated_at': ConjureFieldDefinition('updatedAt', str),
            'log_count': ConjureFieldDefinition('logCount', int),
            'timestamp_type': ConjureFieldDefinition('timestampType', datasource_TimestampType)
        }

    __slots__: List[str] = ['_rid', '_created_by', '_name', '_description', '_origin_metadata', '_created_at', '_updated_at', '_log_count', '_timestamp_type']

    def __init__(self, created_at: str, created_by: str, log_count: int, name: str, origin_metadata: Dict[str, str], rid: str, timestamp_type: "datasource_TimestampType", updated_at: str, description: Optional[str] = None) -> None:
        self._rid = rid
        self._created_by = created_by
        self._name = name
        self._description = description
        self._origin_metadata = origin_metadata
        self._created_at = created_at
        self._updated_at = updated_at
        self._log_count = log_count
        self._timestamp_type = timestamp_type

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def created_by(self) -> str:
        return self._created_by

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def origin_metadata(self) -> Dict[str, str]:
        return self._origin_metadata

    @builtins.property
    def created_at(self) -> str:
        return self._created_at

    @builtins.property
    def updated_at(self) -> str:
        return self._updated_at

    @builtins.property
    def log_count(self) -> int:
        return self._log_count

    @builtins.property
    def timestamp_type(self) -> "datasource_TimestampType":
        return self._timestamp_type


datasource_logset_api_LogSetMetadata.__name__ = "LogSetMetadata"
datasource_logset_api_LogSetMetadata.__qualname__ = "LogSetMetadata"
datasource_logset_api_LogSetMetadata.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_S3LogHandle(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'path': ConjureFieldDefinition('path', str),
            'format': ConjureFieldDefinition('format', datasource_logset_api_LogFormat)
        }

    __slots__: List[str] = ['_path', '_format']

    def __init__(self, format: "datasource_logset_api_LogFormat", path: str) -> None:
        self._path = path
        self._format = format

    @builtins.property
    def path(self) -> str:
        return self._path

    @builtins.property
    def format(self) -> "datasource_logset_api_LogFormat":
        return self._format


datasource_logset_api_S3LogHandle.__name__ = "S3LogHandle"
datasource_logset_api_S3LogHandle.__qualname__ = "S3LogHandle"
datasource_logset_api_S3LogHandle.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_SearchLogSetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'token': ConjureFieldDefinition('token', OptionalTypeWrapper[datasource_logset_api_Token]),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_token', '_page_size']

    def __init__(self, page_size: Optional[int] = None, token: Optional[str] = None) -> None:
        self._token = token
        self._page_size = page_size

    @builtins.property
    def token(self) -> Optional[str]:
        return self._token

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        The maximum allowed page size is 1000. Defaults to the maximum if not supplied.
        """
        return self._page_size


datasource_logset_api_SearchLogSetsRequest.__name__ = "SearchLogSetsRequest"
datasource_logset_api_SearchLogSetsRequest.__qualname__ = "SearchLogSetsRequest"
datasource_logset_api_SearchLogSetsRequest.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_SearchLogSetsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'values': ConjureFieldDefinition('values', List[datasource_logset_api_LogSetMetadata]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[datasource_logset_api_Token])
        }

    __slots__: List[str] = ['_values', '_next_page_token']

    def __init__(self, values: List["datasource_logset_api_LogSetMetadata"], next_page_token: Optional[str] = None) -> None:
        self._values = values
        self._next_page_token = next_page_token

    @builtins.property
    def values(self) -> List["datasource_logset_api_LogSetMetadata"]:
        return self._values

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


datasource_logset_api_SearchLogSetsResponse.__name__ = "SearchLogSetsResponse"
datasource_logset_api_SearchLogSetsResponse.__qualname__ = "SearchLogSetsResponse"
datasource_logset_api_SearchLogSetsResponse.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_SearchLogsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'token': ConjureFieldDefinition('token', OptionalTypeWrapper[datasource_logset_api_Token]),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_token', '_page_size']

    def __init__(self, page_size: Optional[int] = None, token: Optional[str] = None) -> None:
        self._token = token
        self._page_size = page_size

    @builtins.property
    def token(self) -> Optional[str]:
        return self._token

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        The maximum allowed page size is 10000. Defaults to the maximum if not supplied.
        """
        return self._page_size


datasource_logset_api_SearchLogsRequest.__name__ = "SearchLogsRequest"
datasource_logset_api_SearchLogsRequest.__qualname__ = "SearchLogsRequest"
datasource_logset_api_SearchLogsRequest.__module__ = "nominal_api.datasource_logset_api"


class datasource_logset_api_SearchLogsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'logs': ConjureFieldDefinition('logs', List[datasource_logset_api_Log]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[datasource_logset_api_Token])
        }

    __slots__: List[str] = ['_logs', '_next_page_token']

    def __init__(self, logs: List["datasource_logset_api_Log"], next_page_token: Optional[str] = None) -> None:
        self._logs = logs
        self._next_page_token = next_page_token

    @builtins.property
    def logs(self) -> List["datasource_logset_api_Log"]:
        return self._logs

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


datasource_logset_api_SearchLogsResponse.__name__ = "SearchLogsResponse"
datasource_logset_api_SearchLogsResponse.__qualname__ = "SearchLogsResponse"
datasource_logset_api_SearchLogsResponse.__module__ = "nominal_api.datasource_logset_api"


class datasource_pagination_api_PageRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'page_token': ConjureFieldDefinition('pageToken', OptionalTypeWrapper[datasource_pagination_api_PageToken]),
            'page_size': ConjureFieldDefinition('pageSize', int)
        }

    __slots__: List[str] = ['_page_token', '_page_size']

    def __init__(self, page_size: int, page_token: Optional[str] = None) -> None:
        self._page_token = page_token
        self._page_size = page_size

    @builtins.property
    def page_token(self) -> Optional[str]:
        return self._page_token

    @builtins.property
    def page_size(self) -> int:
        return self._page_size


datasource_pagination_api_PageRequest.__name__ = "PageRequest"
datasource_pagination_api_PageRequest.__qualname__ = "PageRequest"
datasource_pagination_api_PageRequest.__module__ = "nominal_api.datasource_pagination_api"


class datasource_pagination_api_PageResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[datasource_pagination_api_PageToken])
        }

    __slots__: List[str] = ['_next_page_token']

    def __init__(self, next_page_token: Optional[str] = None) -> None:
        self._next_page_token = next_page_token

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


datasource_pagination_api_PageResponse.__name__ = "PageResponse"
datasource_pagination_api_PageResponse.__qualname__ = "PageResponse"
datasource_pagination_api_PageResponse.__module__ = "nominal_api.datasource_pagination_api"


class event_ArchiveEvent(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'uuid': ConjureFieldDefinition('uuid', str)
        }

    __slots__: List[str] = ['_uuid']

    def __init__(self, uuid: str) -> None:
        self._uuid = uuid

    @builtins.property
    def uuid(self) -> str:
        return self._uuid


event_ArchiveEvent.__name__ = "ArchiveEvent"
event_ArchiveEvent.__qualname__ = "ArchiveEvent"
event_ArchiveEvent.__module__ = "nominal_api.event"


class event_CreateEvent(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'asset_rids': ConjureFieldDefinition('assetRids', List[str]),
            'associations': ConjureFieldDefinition('associations', List[str]),
            'authorization_rid': ConjureFieldDefinition('authorizationRid', str),
            'timestamp': ConjureFieldDefinition('timestamp', api_Timestamp),
            'duration': ConjureFieldDefinition('duration', scout_run_api_Duration),
            'name': ConjureFieldDefinition('name', str),
            'type': ConjureFieldDefinition('type', event_EventType),
            'labels': ConjureFieldDefinition('labels', OptionalTypeWrapper[List[scout_run_api_Label]]),
            'properties': ConjureFieldDefinition('properties', Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue])
        }

    __slots__: List[str] = ['_asset_rids', '_associations', '_authorization_rid', '_timestamp', '_duration', '_name', '_type', '_labels', '_properties']

    def __init__(self, asset_rids: List[str], associations: List[str], authorization_rid: str, duration: "scout_run_api_Duration", name: str, properties: Dict[str, str], timestamp: "api_Timestamp", type: "event_EventType", labels: Optional[List[str]] = None) -> None:
        self._asset_rids = asset_rids
        self._associations = associations
        self._authorization_rid = authorization_rid
        self._timestamp = timestamp
        self._duration = duration
        self._name = name
        self._type = type
        self._labels = labels
        self._properties = properties

    @builtins.property
    def asset_rids(self) -> List[str]:
        return self._asset_rids

    @builtins.property
    def associations(self) -> List[str]:
        return self._associations

    @builtins.property
    def authorization_rid(self) -> str:
        return self._authorization_rid

    @builtins.property
    def timestamp(self) -> "api_Timestamp":
        return self._timestamp

    @builtins.property
    def duration(self) -> "scout_run_api_Duration":
        return self._duration

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def type(self) -> "event_EventType":
        return self._type

    @builtins.property
    def labels(self) -> Optional[List[str]]:
        return self._labels

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties


event_CreateEvent.__name__ = "CreateEvent"
event_CreateEvent.__qualname__ = "CreateEvent"
event_CreateEvent.__module__ = "nominal_api.event"


class event_Event(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'uuid': ConjureFieldDefinition('uuid', str),
            'asset_rids': ConjureFieldDefinition('assetRids', List[str]),
            'authorization_rid': ConjureFieldDefinition('authorizationRid', str),
            'timestamp': ConjureFieldDefinition('timestamp', api_Timestamp),
            'duration': ConjureFieldDefinition('duration', scout_run_api_Duration),
            'name': ConjureFieldDefinition('name', str),
            'type': ConjureFieldDefinition('type', event_EventType),
            'labels': ConjureFieldDefinition('labels', OptionalTypeWrapper[List[scout_run_api_Label]]),
            'properties': ConjureFieldDefinition('properties', Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue])
        }

    __slots__: List[str] = ['_uuid', '_asset_rids', '_authorization_rid', '_timestamp', '_duration', '_name', '_type', '_labels', '_properties']

    def __init__(self, asset_rids: List[str], authorization_rid: str, duration: "scout_run_api_Duration", name: str, properties: Dict[str, str], timestamp: "api_Timestamp", type: "event_EventType", uuid: str, labels: Optional[List[str]] = None) -> None:
        self._uuid = uuid
        self._asset_rids = asset_rids
        self._authorization_rid = authorization_rid
        self._timestamp = timestamp
        self._duration = duration
        self._name = name
        self._type = type
        self._labels = labels
        self._properties = properties

    @builtins.property
    def uuid(self) -> str:
        return self._uuid

    @builtins.property
    def asset_rids(self) -> List[str]:
        """
        A set of asset rids associated with the event.
        """
        return self._asset_rids

    @builtins.property
    def authorization_rid(self) -> str:
        """
        A user authorized for this rid can access the event. 
Can be a workbook rid, asset rid, etc.
        """
        return self._authorization_rid

    @builtins.property
    def timestamp(self) -> "api_Timestamp":
        return self._timestamp

    @builtins.property
    def duration(self) -> "scout_run_api_Duration":
        return self._duration

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def type(self) -> "event_EventType":
        return self._type

    @builtins.property
    def labels(self) -> Optional[List[str]]:
        return self._labels

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties


event_Event.__name__ = "Event"
event_Event.__qualname__ = "Event"
event_Event.__module__ = "nominal_api.event"


class event_EventService(Service):
    """
    An Event is an annotated moment or time range.
The Event Service is responsible for creating and retrieving events for a particular data source.
    """

    def create_event(self, auth_header: str, request: "event_CreateEvent") -> "event_Event":
        """
        Creates an event
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/event/v1/events'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), event_Event, self._return_none_for_unknown_union_types)

    def get_events(self, auth_header: str, request: "event_GetEvents") -> List["event_Event"]:
        """
        Gets a set of events by UUIDs
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/event/v1/get-events'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[event_Event], self._return_none_for_unknown_union_types)

    def update_event(self, auth_header: str, request: "event_UpdateEvent") -> "event_Event":
        """
        Updates the fields of an event. Empty fields are left unchanged
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/event/v1/update-event'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), event_Event, self._return_none_for_unknown_union_types)

    def archive_event(self, auth_header: str, request: "event_ArchiveEvent") -> None:
        """
        Archives an event
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/event/v1/archive-event'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def search_events(self, auth_header: str, list_runs_request: "event_SearchEventsRequest") -> "event_SearchEventsResponse":
        """
        Searches for events that match the given ƒilters.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(list_runs_request)

        _path = '/event/v1/search-events'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), event_SearchEventsResponse, self._return_none_for_unknown_union_types)


event_EventService.__name__ = "EventService"
event_EventService.__qualname__ = "EventService"
event_EventService.__module__ = "nominal_api.event"


class event_EventType(ConjureEnumType):

    INFO = 'INFO'
    '''INFO'''
    FLAG = 'FLAG'
    '''FLAG'''
    ERROR = 'ERROR'
    '''ERROR'''
    SUCCESS = 'SUCCESS'
    '''SUCCESS'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


event_EventType.__name__ = "EventType"
event_EventType.__qualname__ = "EventType"
event_EventType.__module__ = "nominal_api.event"


class event_GetEvents(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'uuids': ConjureFieldDefinition('uuids', List[str])
        }

    __slots__: List[str] = ['_uuids']

    def __init__(self, uuids: List[str]) -> None:
        self._uuids = uuids

    @builtins.property
    def uuids(self) -> List[str]:
        return self._uuids


event_GetEvents.__name__ = "GetEvents"
event_GetEvents.__qualname__ = "GetEvents"
event_GetEvents.__module__ = "nominal_api.event"


class event_SearchEventsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'sort': ConjureFieldDefinition('sort', event_SortOptions),
            'page_size': ConjureFieldDefinition('pageSize', int),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token]),
            'query': ConjureFieldDefinition('query', event_SearchQuery)
        }

    __slots__: List[str] = ['_sort', '_page_size', '_next_page_token', '_query']

    def __init__(self, page_size: int, query: "event_SearchQuery", sort: "event_SortOptions", next_page_token: Optional[str] = None) -> None:
        self._sort = sort
        self._page_size = page_size
        self._next_page_token = next_page_token
        self._query = query

    @builtins.property
    def sort(self) -> "event_SortOptions":
        return self._sort

    @builtins.property
    def page_size(self) -> int:
        """
        Will reject page sizes greater than 10k.
        """
        return self._page_size

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token

    @builtins.property
    def query(self) -> "event_SearchQuery":
        return self._query


event_SearchEventsRequest.__name__ = "SearchEventsRequest"
event_SearchEventsRequest.__qualname__ = "SearchEventsRequest"
event_SearchEventsRequest.__module__ = "nominal_api.event"


class event_SearchEventsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[event_Event]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token])
        }

    __slots__: List[str] = ['_results', '_next_page_token']

    def __init__(self, results: List["event_Event"], next_page_token: Optional[str] = None) -> None:
        self._results = results
        self._next_page_token = next_page_token

    @builtins.property
    def results(self) -> List["event_Event"]:
        return self._results

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


event_SearchEventsResponse.__name__ = "SearchEventsResponse"
event_SearchEventsResponse.__qualname__ = "SearchEventsResponse"
event_SearchEventsResponse.__module__ = "nominal_api.event"


class event_SearchQuery(ConjureUnionType):
    _after: Optional["api_Timestamp"] = None
    _before: Optional["api_Timestamp"] = None
    _asset: Optional[str] = None
    _label: Optional[str] = None
    _property: Optional["scout_run_api_Property"] = None
    _and_: Optional[List["event_SearchQuery"]] = None
    _or_: Optional[List["event_SearchQuery"]] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'after': ConjureFieldDefinition('after', api_Timestamp),
            'before': ConjureFieldDefinition('before', api_Timestamp),
            'asset': ConjureFieldDefinition('asset', scout_rids_api_AssetRid),
            'label': ConjureFieldDefinition('label', scout_run_api_Label),
            'property': ConjureFieldDefinition('property', scout_run_api_Property),
            'and_': ConjureFieldDefinition('and', List[event_SearchQuery]),
            'or_': ConjureFieldDefinition('or', List[event_SearchQuery])
        }

    def __init__(
            self,
            after: Optional["api_Timestamp"] = None,
            before: Optional["api_Timestamp"] = None,
            asset: Optional[str] = None,
            label: Optional[str] = None,
            property: Optional["scout_run_api_Property"] = None,
            and_: Optional[List["event_SearchQuery"]] = None,
            or_: Optional[List["event_SearchQuery"]] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (after is not None) + (before is not None) + (asset is not None) + (label is not None) + (property is not None) + (and_ is not None) + (or_ is not None) != 1:
                raise ValueError('a union must contain a single member')

            if after is not None:
                self._after = after
                self._type = 'after'
            if before is not None:
                self._before = before
                self._type = 'before'
            if asset is not None:
                self._asset = asset
                self._type = 'asset'
            if label is not None:
                self._label = label
                self._type = 'label'
            if property is not None:
                self._property = property
                self._type = 'property'
            if and_ is not None:
                self._and_ = and_
                self._type = 'and'
            if or_ is not None:
                self._or_ = or_
                self._type = 'or'

        elif type_of_union == 'after':
            if after is None:
                raise ValueError('a union value must not be None')
            self._after = after
            self._type = 'after'
        elif type_of_union == 'before':
            if before is None:
                raise ValueError('a union value must not be None')
            self._before = before
            self._type = 'before'
        elif type_of_union == 'asset':
            if asset is None:
                raise ValueError('a union value must not be None')
            self._asset = asset
            self._type = 'asset'
        elif type_of_union == 'label':
            if label is None:
                raise ValueError('a union value must not be None')
            self._label = label
            self._type = 'label'
        elif type_of_union == 'property':
            if property is None:
                raise ValueError('a union value must not be None')
            self._property = property
            self._type = 'property'
        elif type_of_union == 'and':
            if and_ is None:
                raise ValueError('a union value must not be None')
            self._and_ = and_
            self._type = 'and'
        elif type_of_union == 'or':
            if or_ is None:
                raise ValueError('a union value must not be None')
            self._or_ = or_
            self._type = 'or'

    @builtins.property
    def after(self) -> Optional["api_Timestamp"]:
        """
        Filters to start times after this time, inclusive.
        """
        return self._after

    @builtins.property
    def before(self) -> Optional["api_Timestamp"]:
        """
        Filters to start times before this time, exclusive.
        """
        return self._before

    @builtins.property
    def asset(self) -> Optional[str]:
        return self._asset

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def property(self) -> Optional["scout_run_api_Property"]:
        return self._property

    @builtins.property
    def and_(self) -> Optional[List["event_SearchQuery"]]:
        return self._and_

    @builtins.property
    def or_(self) -> Optional[List["event_SearchQuery"]]:
        return self._or_

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, event_SearchQueryVisitor):
            raise ValueError('{} is not an instance of event_SearchQueryVisitor'.format(visitor.__class__.__name__))
        if self._type == 'after' and self.after is not None:
            return visitor._after(self.after)
        if self._type == 'before' and self.before is not None:
            return visitor._before(self.before)
        if self._type == 'asset' and self.asset is not None:
            return visitor._asset(self.asset)
        if self._type == 'label' and self.label is not None:
            return visitor._label(self.label)
        if self._type == 'property' and self.property is not None:
            return visitor._property(self.property)
        if self._type == 'and' and self.and_ is not None:
            return visitor._and(self.and_)
        if self._type == 'or' and self.or_ is not None:
            return visitor._or(self.or_)


event_SearchQuery.__name__ = "SearchQuery"
event_SearchQuery.__qualname__ = "SearchQuery"
event_SearchQuery.__module__ = "nominal_api.event"


class event_SearchQueryVisitor:

    @abstractmethod
    def _after(self, after: "api_Timestamp") -> Any:
        pass

    @abstractmethod
    def _before(self, before: "api_Timestamp") -> Any:
        pass

    @abstractmethod
    def _asset(self, asset: str) -> Any:
        pass

    @abstractmethod
    def _label(self, label: str) -> Any:
        pass

    @abstractmethod
    def _property(self, property: "scout_run_api_Property") -> Any:
        pass

    @abstractmethod
    def _and(self, and_: List["event_SearchQuery"]) -> Any:
        pass

    @abstractmethod
    def _or(self, or_: List["event_SearchQuery"]) -> Any:
        pass


event_SearchQueryVisitor.__name__ = "SearchQueryVisitor"
event_SearchQueryVisitor.__qualname__ = "SearchQueryVisitor"
event_SearchQueryVisitor.__module__ = "nominal_api.event"


class event_SortField(ConjureEnumType):

    START_TIME = 'START_TIME'
    '''START_TIME'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


event_SortField.__name__ = "SortField"
event_SortField.__qualname__ = "SortField"
event_SortField.__module__ = "nominal_api.event"


class event_SortOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'is_descending': ConjureFieldDefinition('isDescending', bool),
            'field': ConjureFieldDefinition('field', event_SortField)
        }

    __slots__: List[str] = ['_is_descending', '_field']

    def __init__(self, field: "event_SortField", is_descending: bool) -> None:
        self._is_descending = is_descending
        self._field = field

    @builtins.property
    def is_descending(self) -> bool:
        return self._is_descending

    @builtins.property
    def field(self) -> "event_SortField":
        return self._field


event_SortOptions.__name__ = "SortOptions"
event_SortOptions.__qualname__ = "SortOptions"
event_SortOptions.__module__ = "nominal_api.event"


class event_UpdateEvent(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'uuid': ConjureFieldDefinition('uuid', str),
            'asset_rids': ConjureFieldDefinition('assetRids', OptionalTypeWrapper[List[str]]),
            'authorization_rid': ConjureFieldDefinition('authorizationRid', OptionalTypeWrapper[str]),
            'timestamp': ConjureFieldDefinition('timestamp', OptionalTypeWrapper[api_Timestamp]),
            'duration': ConjureFieldDefinition('duration', OptionalTypeWrapper[scout_run_api_Duration]),
            'name': ConjureFieldDefinition('name', OptionalTypeWrapper[str]),
            'type': ConjureFieldDefinition('type', OptionalTypeWrapper[event_EventType]),
            'labels': ConjureFieldDefinition('labels', OptionalTypeWrapper[List[scout_run_api_Label]]),
            'properties': ConjureFieldDefinition('properties', OptionalTypeWrapper[Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue]])
        }

    __slots__: List[str] = ['_uuid', '_asset_rids', '_authorization_rid', '_timestamp', '_duration', '_name', '_type', '_labels', '_properties']

    def __init__(self, uuid: str, asset_rids: Optional[List[str]] = None, authorization_rid: Optional[str] = None, duration: Optional["scout_run_api_Duration"] = None, labels: Optional[List[str]] = None, name: Optional[str] = None, properties: Optional[Dict[str, str]] = None, timestamp: Optional["api_Timestamp"] = None, type: Optional["event_EventType"] = None) -> None:
        self._uuid = uuid
        self._asset_rids = asset_rids
        self._authorization_rid = authorization_rid
        self._timestamp = timestamp
        self._duration = duration
        self._name = name
        self._type = type
        self._labels = labels
        self._properties = properties

    @builtins.property
    def uuid(self) -> str:
        return self._uuid

    @builtins.property
    def asset_rids(self) -> Optional[List[str]]:
        return self._asset_rids

    @builtins.property
    def authorization_rid(self) -> Optional[str]:
        return self._authorization_rid

    @builtins.property
    def timestamp(self) -> Optional["api_Timestamp"]:
        return self._timestamp

    @builtins.property
    def duration(self) -> Optional["scout_run_api_Duration"]:
        return self._duration

    @builtins.property
    def name(self) -> Optional[str]:
        return self._name

    @builtins.property
    def type(self) -> Optional["event_EventType"]:
        return self._type

    @builtins.property
    def labels(self) -> Optional[List[str]]:
        return self._labels

    @builtins.property
    def properties(self) -> Optional[Dict[str, str]]:
        return self._properties


event_UpdateEvent.__name__ = "UpdateEvent"
event_UpdateEvent.__qualname__ = "UpdateEvent"
event_UpdateEvent.__module__ = "nominal_api.event"


class ingest_api_AbsoluteTimestamp(ConjureUnionType):
    _iso8601: Optional["ingest_api_Iso8601Timestamp"] = None
    _epoch_of_time_unit: Optional["ingest_api_EpochTimestamp"] = None
    _custom_format: Optional["ingest_api_CustomTimestamp"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'iso8601': ConjureFieldDefinition('iso8601', ingest_api_Iso8601Timestamp),
            'epoch_of_time_unit': ConjureFieldDefinition('epochOfTimeUnit', ingest_api_EpochTimestamp),
            'custom_format': ConjureFieldDefinition('customFormat', ingest_api_CustomTimestamp)
        }

    def __init__(
            self,
            iso8601: Optional["ingest_api_Iso8601Timestamp"] = None,
            epoch_of_time_unit: Optional["ingest_api_EpochTimestamp"] = None,
            custom_format: Optional["ingest_api_CustomTimestamp"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (iso8601 is not None) + (epoch_of_time_unit is not None) + (custom_format is not None) != 1:
                raise ValueError('a union must contain a single member')

            if iso8601 is not None:
                self._iso8601 = iso8601
                self._type = 'iso8601'
            if epoch_of_time_unit is not None:
                self._epoch_of_time_unit = epoch_of_time_unit
                self._type = 'epochOfTimeUnit'
            if custom_format is not None:
                self._custom_format = custom_format
                self._type = 'customFormat'

        elif type_of_union == 'iso8601':
            if iso8601 is None:
                raise ValueError('a union value must not be None')
            self._iso8601 = iso8601
            self._type = 'iso8601'
        elif type_of_union == 'epochOfTimeUnit':
            if epoch_of_time_unit is None:
                raise ValueError('a union value must not be None')
            self._epoch_of_time_unit = epoch_of_time_unit
            self._type = 'epochOfTimeUnit'
        elif type_of_union == 'customFormat':
            if custom_format is None:
                raise ValueError('a union value must not be None')
            self._custom_format = custom_format
            self._type = 'customFormat'

    @builtins.property
    def iso8601(self) -> Optional["ingest_api_Iso8601Timestamp"]:
        return self._iso8601

    @builtins.property
    def epoch_of_time_unit(self) -> Optional["ingest_api_EpochTimestamp"]:
        return self._epoch_of_time_unit

    @builtins.property
    def custom_format(self) -> Optional["ingest_api_CustomTimestamp"]:
        return self._custom_format

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_AbsoluteTimestampVisitor):
            raise ValueError('{} is not an instance of ingest_api_AbsoluteTimestampVisitor'.format(visitor.__class__.__name__))
        if self._type == 'iso8601' and self.iso8601 is not None:
            return visitor._iso8601(self.iso8601)
        if self._type == 'epochOfTimeUnit' and self.epoch_of_time_unit is not None:
            return visitor._epoch_of_time_unit(self.epoch_of_time_unit)
        if self._type == 'customFormat' and self.custom_format is not None:
            return visitor._custom_format(self.custom_format)


ingest_api_AbsoluteTimestamp.__name__ = "AbsoluteTimestamp"
ingest_api_AbsoluteTimestamp.__qualname__ = "AbsoluteTimestamp"
ingest_api_AbsoluteTimestamp.__module__ = "nominal_api.ingest_api"


class ingest_api_AbsoluteTimestampVisitor:

    @abstractmethod
    def _iso8601(self, iso8601: "ingest_api_Iso8601Timestamp") -> Any:
        pass

    @abstractmethod
    def _epoch_of_time_unit(self, epoch_of_time_unit: "ingest_api_EpochTimestamp") -> Any:
        pass

    @abstractmethod
    def _custom_format(self, custom_format: "ingest_api_CustomTimestamp") -> Any:
        pass


ingest_api_AbsoluteTimestampVisitor.__name__ = "AbsoluteTimestampVisitor"
ingest_api_AbsoluteTimestampVisitor.__qualname__ = "AbsoluteTimestampVisitor"
ingest_api_AbsoluteTimestampVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_AsyncHandle(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', str)
        }

    __slots__: List[str] = ['_rid']

    def __init__(self, rid: str) -> None:
        self._rid = rid

    @builtins.property
    def rid(self) -> str:
        return self._rid


ingest_api_AsyncHandle.__name__ = "AsyncHandle"
ingest_api_AsyncHandle.__qualname__ = "AsyncHandle"
ingest_api_AsyncHandle.__module__ = "nominal_api.ingest_api"


class ingest_api_ChannelConfig(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'prefix_tree_delimiter': ConjureFieldDefinition('prefixTreeDelimiter', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_prefix_tree_delimiter']

    def __init__(self, prefix_tree_delimiter: Optional[str] = None) -> None:
        self._prefix_tree_delimiter = prefix_tree_delimiter

    @builtins.property
    def prefix_tree_delimiter(self) -> Optional[str]:
        """
        If set, will construct a prefix tree for channels of the dataset using the given delimiter.
        """
        return self._prefix_tree_delimiter


ingest_api_ChannelConfig.__name__ = "ChannelConfig"
ingest_api_ChannelConfig.__qualname__ = "ChannelConfig"
ingest_api_ChannelConfig.__module__ = "nominal_api.ingest_api"


class ingest_api_CompleteMultipartUploadResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'location': ConjureFieldDefinition('location', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_location']

    def __init__(self, location: Optional[str] = None) -> None:
        self._location = location

    @builtins.property
    def location(self) -> Optional[str]:
        return self._location


ingest_api_CompleteMultipartUploadResponse.__name__ = "CompleteMultipartUploadResponse"
ingest_api_CompleteMultipartUploadResponse.__qualname__ = "CompleteMultipartUploadResponse"
ingest_api_CompleteMultipartUploadResponse.__module__ = "nominal_api.ingest_api"


class ingest_api_CustomTimestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'format': ConjureFieldDefinition('format', str),
            'default_year': ConjureFieldDefinition('defaultYear', OptionalTypeWrapper[int]),
            'default_day_of_year': ConjureFieldDefinition('defaultDayOfYear', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_format', '_default_year', '_default_day_of_year']

    def __init__(self, format: str, default_day_of_year: Optional[int] = None, default_year: Optional[int] = None) -> None:
        self._format = format
        self._default_year = default_year
        self._default_day_of_year = default_day_of_year

    @builtins.property
    def format(self) -> str:
        """
        The format string should be in the format of the `DateTimeFormatter` class in Java.
        """
        return self._format

    @builtins.property
    def default_year(self) -> Optional[int]:
        """
        Default year is accepted as an optional field for cases like IRIG time format and will be overridden by year in time format.
        """
        return self._default_year

    @builtins.property
    def default_day_of_year(self) -> Optional[int]:
        """
        Default day of year is accepted as an optional field for cases like IRIG time format and will be overridden by day of year in time format.
        """
        return self._default_day_of_year


ingest_api_CustomTimestamp.__name__ = "CustomTimestamp"
ingest_api_CustomTimestamp.__qualname__ = "CustomTimestamp"
ingest_api_CustomTimestamp.__module__ = "nominal_api.ingest_api"


class ingest_api_DataflashOpts(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'source': ConjureFieldDefinition('source', ingest_api_IngestSource),
            'target': ConjureFieldDefinition('target', ingest_api_DatasetIngestTarget)
        }

    __slots__: List[str] = ['_source', '_target']

    def __init__(self, source: "ingest_api_IngestSource", target: "ingest_api_DatasetIngestTarget") -> None:
        self._source = source
        self._target = target

    @builtins.property
    def source(self) -> "ingest_api_IngestSource":
        return self._source

    @builtins.property
    def target(self) -> "ingest_api_DatasetIngestTarget":
        return self._target


ingest_api_DataflashOpts.__name__ = "DataflashOpts"
ingest_api_DataflashOpts.__qualname__ = "DataflashOpts"
ingest_api_DataflashOpts.__module__ = "nominal_api.ingest_api"


class ingest_api_DatasetIngestTarget(ConjureUnionType):
    _new: Optional["ingest_api_NewDatasetIngestDestination"] = None
    _existing: Optional["ingest_api_ExistingDatasetIngestDestination"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'new': ConjureFieldDefinition('new', ingest_api_NewDatasetIngestDestination),
            'existing': ConjureFieldDefinition('existing', ingest_api_ExistingDatasetIngestDestination)
        }

    def __init__(
            self,
            new: Optional["ingest_api_NewDatasetIngestDestination"] = None,
            existing: Optional["ingest_api_ExistingDatasetIngestDestination"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (new is not None) + (existing is not None) != 1:
                raise ValueError('a union must contain a single member')

            if new is not None:
                self._new = new
                self._type = 'new'
            if existing is not None:
                self._existing = existing
                self._type = 'existing'

        elif type_of_union == 'new':
            if new is None:
                raise ValueError('a union value must not be None')
            self._new = new
            self._type = 'new'
        elif type_of_union == 'existing':
            if existing is None:
                raise ValueError('a union value must not be None')
            self._existing = existing
            self._type = 'existing'

    @builtins.property
    def new(self) -> Optional["ingest_api_NewDatasetIngestDestination"]:
        return self._new

    @builtins.property
    def existing(self) -> Optional["ingest_api_ExistingDatasetIngestDestination"]:
        return self._existing

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_DatasetIngestTargetVisitor):
            raise ValueError('{} is not an instance of ingest_api_DatasetIngestTargetVisitor'.format(visitor.__class__.__name__))
        if self._type == 'new' and self.new is not None:
            return visitor._new(self.new)
        if self._type == 'existing' and self.existing is not None:
            return visitor._existing(self.existing)


ingest_api_DatasetIngestTarget.__name__ = "DatasetIngestTarget"
ingest_api_DatasetIngestTarget.__qualname__ = "DatasetIngestTarget"
ingest_api_DatasetIngestTarget.__module__ = "nominal_api.ingest_api"


class ingest_api_DatasetIngestTargetVisitor:

    @abstractmethod
    def _new(self, new: "ingest_api_NewDatasetIngestDestination") -> Any:
        pass

    @abstractmethod
    def _existing(self, existing: "ingest_api_ExistingDatasetIngestDestination") -> Any:
        pass


ingest_api_DatasetIngestTargetVisitor.__name__ = "DatasetIngestTargetVisitor"
ingest_api_DatasetIngestTargetVisitor.__qualname__ = "DatasetIngestTargetVisitor"
ingest_api_DatasetIngestTargetVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_DatasetSpec(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_name']

    def __init__(self, name: Optional[str] = None) -> None:
        self._name = name

    @builtins.property
    def name(self) -> Optional[str]:
        return self._name


ingest_api_DatasetSpec.__name__ = "DatasetSpec"
ingest_api_DatasetSpec.__qualname__ = "DatasetSpec"
ingest_api_DatasetSpec.__module__ = "nominal_api.ingest_api"


class ingest_api_DeprecatedNewCsv(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'source': ConjureFieldDefinition('source', ingest_api_IngestSource),
            'name': ConjureFieldDefinition('name', OptionalTypeWrapper[str]),
            'properties': ConjureFieldDefinition('properties', Dict[str, str]),
            'time_column_spec': ConjureFieldDefinition('timeColumnSpec', OptionalTypeWrapper[ingest_api_TimestampMetadata]),
            'channel_config': ConjureFieldDefinition('channelConfig', OptionalTypeWrapper[ingest_api_ChannelConfig])
        }

    __slots__: List[str] = ['_source', '_name', '_properties', '_time_column_spec', '_channel_config']

    def __init__(self, properties: Dict[str, str], source: "ingest_api_IngestSource", channel_config: Optional["ingest_api_ChannelConfig"] = None, name: Optional[str] = None, time_column_spec: Optional["ingest_api_TimestampMetadata"] = None) -> None:
        self._source = source
        self._name = name
        self._properties = properties
        self._time_column_spec = time_column_spec
        self._channel_config = channel_config

    @builtins.property
    def source(self) -> "ingest_api_IngestSource":
        return self._source

    @builtins.property
    def name(self) -> Optional[str]:
        return self._name

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def time_column_spec(self) -> Optional["ingest_api_TimestampMetadata"]:
        return self._time_column_spec

    @builtins.property
    def channel_config(self) -> Optional["ingest_api_ChannelConfig"]:
        return self._channel_config


ingest_api_DeprecatedNewCsv.__name__ = "DeprecatedNewCsv"
ingest_api_DeprecatedNewCsv.__qualname__ = "DeprecatedNewCsv"
ingest_api_DeprecatedNewCsv.__module__ = "nominal_api.ingest_api"


class ingest_api_DeprecatedNewDataSource(ConjureUnionType):
    _csv: Optional["ingest_api_DeprecatedNewCsv"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'csv': ConjureFieldDefinition('csv', ingest_api_DeprecatedNewCsv)
        }

    def __init__(
            self,
            csv: Optional["ingest_api_DeprecatedNewCsv"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (csv is not None) != 1:
                raise ValueError('a union must contain a single member')

            if csv is not None:
                self._csv = csv
                self._type = 'csv'

        elif type_of_union == 'csv':
            if csv is None:
                raise ValueError('a union value must not be None')
            self._csv = csv
            self._type = 'csv'

    @builtins.property
    def csv(self) -> Optional["ingest_api_DeprecatedNewCsv"]:
        return self._csv

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_DeprecatedNewDataSourceVisitor):
            raise ValueError('{} is not an instance of ingest_api_DeprecatedNewDataSourceVisitor'.format(visitor.__class__.__name__))
        if self._type == 'csv' and self.csv is not None:
            return visitor._csv(self.csv)


ingest_api_DeprecatedNewDataSource.__name__ = "DeprecatedNewDataSource"
ingest_api_DeprecatedNewDataSource.__qualname__ = "DeprecatedNewDataSource"
ingest_api_DeprecatedNewDataSource.__module__ = "nominal_api.ingest_api"


class ingest_api_DeprecatedNewDataSourceVisitor:

    @abstractmethod
    def _csv(self, csv: "ingest_api_DeprecatedNewCsv") -> Any:
        pass


ingest_api_DeprecatedNewDataSourceVisitor.__name__ = "DeprecatedNewDataSourceVisitor"
ingest_api_DeprecatedNewDataSourceVisitor.__qualname__ = "DeprecatedNewDataSourceVisitor"
ingest_api_DeprecatedNewDataSourceVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_DeprecatedTimestampMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'series_name': ConjureFieldDefinition('seriesName', str),
            'is_absolute': ConjureFieldDefinition('isAbsolute', bool)
        }

    __slots__: List[str] = ['_series_name', '_is_absolute']

    def __init__(self, is_absolute: bool, series_name: str) -> None:
        self._series_name = series_name
        self._is_absolute = is_absolute

    @builtins.property
    def series_name(self) -> str:
        return self._series_name

    @builtins.property
    def is_absolute(self) -> bool:
        return self._is_absolute


ingest_api_DeprecatedTimestampMetadata.__name__ = "DeprecatedTimestampMetadata"
ingest_api_DeprecatedTimestampMetadata.__qualname__ = "DeprecatedTimestampMetadata"
ingest_api_DeprecatedTimestampMetadata.__module__ = "nominal_api.ingest_api"


class ingest_api_DeprecatedTriggerIngest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'source': ConjureFieldDefinition('source', ingest_api_IngestSource),
            'properties': ConjureFieldDefinition('properties', Dict[str, str]),
            'dataset_name': ConjureFieldDefinition('datasetName', OptionalTypeWrapper[str]),
            'timestamp_metadata': ConjureFieldDefinition('timestampMetadata', OptionalTypeWrapper[ingest_api_DeprecatedTimestampMetadata])
        }

    __slots__: List[str] = ['_source', '_properties', '_dataset_name', '_timestamp_metadata']

    def __init__(self, properties: Dict[str, str], source: "ingest_api_IngestSource", dataset_name: Optional[str] = None, timestamp_metadata: Optional["ingest_api_DeprecatedTimestampMetadata"] = None) -> None:
        self._source = source
        self._properties = properties
        self._dataset_name = dataset_name
        self._timestamp_metadata = timestamp_metadata

    @builtins.property
    def source(self) -> "ingest_api_IngestSource":
        return self._source

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def dataset_name(self) -> Optional[str]:
        return self._dataset_name

    @builtins.property
    def timestamp_metadata(self) -> Optional["ingest_api_DeprecatedTimestampMetadata"]:
        return self._timestamp_metadata


ingest_api_DeprecatedTriggerIngest.__name__ = "DeprecatedTriggerIngest"
ingest_api_DeprecatedTriggerIngest.__qualname__ = "DeprecatedTriggerIngest"
ingest_api_DeprecatedTriggerIngest.__module__ = "nominal_api.ingest_api"


class ingest_api_EpochTimestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'time_unit': ConjureFieldDefinition('timeUnit', api_TimeUnit)
        }

    __slots__: List[str] = ['_time_unit']

    def __init__(self, time_unit: "api_TimeUnit") -> None:
        self._time_unit = time_unit

    @builtins.property
    def time_unit(self) -> "api_TimeUnit":
        return self._time_unit


ingest_api_EpochTimestamp.__name__ = "EpochTimestamp"
ingest_api_EpochTimestamp.__qualname__ = "EpochTimestamp"
ingest_api_EpochTimestamp.__module__ = "nominal_api.ingest_api"


class ingest_api_ErrorResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'error_type': ConjureFieldDefinition('errorType', ingest_api_ErrorType),
            'message': ConjureFieldDefinition('message', str)
        }

    __slots__: List[str] = ['_error_type', '_message']

    def __init__(self, error_type: str, message: str) -> None:
        self._error_type = error_type
        self._message = message

    @builtins.property
    def error_type(self) -> str:
        return self._error_type

    @builtins.property
    def message(self) -> str:
        return self._message


ingest_api_ErrorResult.__name__ = "ErrorResult"
ingest_api_ErrorResult.__qualname__ = "ErrorResult"
ingest_api_ErrorResult.__module__ = "nominal_api.ingest_api"


class ingest_api_ExistingDatasetIngestDestination(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset_rid': ConjureFieldDefinition('datasetRid', str)
        }

    __slots__: List[str] = ['_dataset_rid']

    def __init__(self, dataset_rid: str) -> None:
        self._dataset_rid = dataset_rid

    @builtins.property
    def dataset_rid(self) -> str:
        return self._dataset_rid


ingest_api_ExistingDatasetIngestDestination.__name__ = "ExistingDatasetIngestDestination"
ingest_api_ExistingDatasetIngestDestination.__qualname__ = "ExistingDatasetIngestDestination"
ingest_api_ExistingDatasetIngestDestination.__module__ = "nominal_api.ingest_api"


class ingest_api_InProgressResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



ingest_api_InProgressResult.__name__ = "InProgressResult"
ingest_api_InProgressResult.__qualname__ = "InProgressResult"
ingest_api_InProgressResult.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestDataSource(ConjureUnionType):
    _existing_data_source: Optional[str] = None
    _new_data_source: Optional["ingest_api_DeprecatedNewDataSource"] = None
    _new_data_source_v2: Optional["ingest_api_NewDataSource"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'existing_data_source': ConjureFieldDefinition('existingDataSource', api_rids_DataSourceRid),
            'new_data_source': ConjureFieldDefinition('newDataSource', ingest_api_DeprecatedNewDataSource),
            'new_data_source_v2': ConjureFieldDefinition('newDataSourceV2', ingest_api_NewDataSource)
        }

    def __init__(
            self,
            existing_data_source: Optional[str] = None,
            new_data_source: Optional["ingest_api_DeprecatedNewDataSource"] = None,
            new_data_source_v2: Optional["ingest_api_NewDataSource"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (existing_data_source is not None) + (new_data_source is not None) + (new_data_source_v2 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if existing_data_source is not None:
                self._existing_data_source = existing_data_source
                self._type = 'existingDataSource'
            if new_data_source is not None:
                self._new_data_source = new_data_source
                self._type = 'newDataSource'
            if new_data_source_v2 is not None:
                self._new_data_source_v2 = new_data_source_v2
                self._type = 'newDataSourceV2'

        elif type_of_union == 'existingDataSource':
            if existing_data_source is None:
                raise ValueError('a union value must not be None')
            self._existing_data_source = existing_data_source
            self._type = 'existingDataSource'
        elif type_of_union == 'newDataSource':
            if new_data_source is None:
                raise ValueError('a union value must not be None')
            self._new_data_source = new_data_source
            self._type = 'newDataSource'
        elif type_of_union == 'newDataSourceV2':
            if new_data_source_v2 is None:
                raise ValueError('a union value must not be None')
            self._new_data_source_v2 = new_data_source_v2
            self._type = 'newDataSourceV2'

    @builtins.property
    def existing_data_source(self) -> Optional[str]:
        return self._existing_data_source

    @builtins.property
    def new_data_source(self) -> Optional["ingest_api_DeprecatedNewDataSource"]:
        return self._new_data_source

    @builtins.property
    def new_data_source_v2(self) -> Optional["ingest_api_NewDataSource"]:
        return self._new_data_source_v2

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_IngestDataSourceVisitor):
            raise ValueError('{} is not an instance of ingest_api_IngestDataSourceVisitor'.format(visitor.__class__.__name__))
        if self._type == 'existingDataSource' and self.existing_data_source is not None:
            return visitor._existing_data_source(self.existing_data_source)
        if self._type == 'newDataSource' and self.new_data_source is not None:
            return visitor._new_data_source(self.new_data_source)
        if self._type == 'newDataSourceV2' and self.new_data_source_v2 is not None:
            return visitor._new_data_source_v2(self.new_data_source_v2)


ingest_api_IngestDataSource.__name__ = "IngestDataSource"
ingest_api_IngestDataSource.__qualname__ = "IngestDataSource"
ingest_api_IngestDataSource.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestDataSourceVisitor:

    @abstractmethod
    def _existing_data_source(self, existing_data_source: str) -> Any:
        pass

    @abstractmethod
    def _new_data_source(self, new_data_source: "ingest_api_DeprecatedNewDataSource") -> Any:
        pass

    @abstractmethod
    def _new_data_source_v2(self, new_data_source_v2: "ingest_api_NewDataSource") -> Any:
        pass


ingest_api_IngestDataSourceVisitor.__name__ = "IngestDataSourceVisitor"
ingest_api_IngestDataSourceVisitor.__qualname__ = "IngestDataSourceVisitor"
ingest_api_IngestDataSourceVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestDatasetFileDetails(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset_file_id': ConjureFieldDefinition('datasetFileId', str),
            'dataset_rid': ConjureFieldDefinition('datasetRid', api_rids_DatasetRid)
        }

    __slots__: List[str] = ['_dataset_file_id', '_dataset_rid']

    def __init__(self, dataset_file_id: str, dataset_rid: str) -> None:
        self._dataset_file_id = dataset_file_id
        self._dataset_rid = dataset_rid

    @builtins.property
    def dataset_file_id(self) -> str:
        return self._dataset_file_id

    @builtins.property
    def dataset_rid(self) -> str:
        return self._dataset_rid


ingest_api_IngestDatasetFileDetails.__name__ = "IngestDatasetFileDetails"
ingest_api_IngestDatasetFileDetails.__qualname__ = "IngestDatasetFileDetails"
ingest_api_IngestDatasetFileDetails.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestDestination(ConjureUnionType):
    _new_dataset: Optional["ingest_api_NewDatasetIngestDestination"] = None
    _existing_dataset: Optional["ingest_api_ExistingDatasetIngestDestination"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'new_dataset': ConjureFieldDefinition('newDataset', ingest_api_NewDatasetIngestDestination),
            'existing_dataset': ConjureFieldDefinition('existingDataset', ingest_api_ExistingDatasetIngestDestination)
        }

    def __init__(
            self,
            new_dataset: Optional["ingest_api_NewDatasetIngestDestination"] = None,
            existing_dataset: Optional["ingest_api_ExistingDatasetIngestDestination"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (new_dataset is not None) + (existing_dataset is not None) != 1:
                raise ValueError('a union must contain a single member')

            if new_dataset is not None:
                self._new_dataset = new_dataset
                self._type = 'newDataset'
            if existing_dataset is not None:
                self._existing_dataset = existing_dataset
                self._type = 'existingDataset'

        elif type_of_union == 'newDataset':
            if new_dataset is None:
                raise ValueError('a union value must not be None')
            self._new_dataset = new_dataset
            self._type = 'newDataset'
        elif type_of_union == 'existingDataset':
            if existing_dataset is None:
                raise ValueError('a union value must not be None')
            self._existing_dataset = existing_dataset
            self._type = 'existingDataset'

    @builtins.property
    def new_dataset(self) -> Optional["ingest_api_NewDatasetIngestDestination"]:
        return self._new_dataset

    @builtins.property
    def existing_dataset(self) -> Optional["ingest_api_ExistingDatasetIngestDestination"]:
        return self._existing_dataset

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_IngestDestinationVisitor):
            raise ValueError('{} is not an instance of ingest_api_IngestDestinationVisitor'.format(visitor.__class__.__name__))
        if self._type == 'newDataset' and self.new_dataset is not None:
            return visitor._new_dataset(self.new_dataset)
        if self._type == 'existingDataset' and self.existing_dataset is not None:
            return visitor._existing_dataset(self.existing_dataset)


ingest_api_IngestDestination.__name__ = "IngestDestination"
ingest_api_IngestDestination.__qualname__ = "IngestDestination"
ingest_api_IngestDestination.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestDestinationVisitor:

    @abstractmethod
    def _new_dataset(self, new_dataset: "ingest_api_NewDatasetIngestDestination") -> Any:
        pass

    @abstractmethod
    def _existing_dataset(self, existing_dataset: "ingest_api_ExistingDatasetIngestDestination") -> Any:
        pass


ingest_api_IngestDestinationVisitor.__name__ = "IngestDestinationVisitor"
ingest_api_IngestDestinationVisitor.__qualname__ = "IngestDestinationVisitor"
ingest_api_IngestDestinationVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestDetails(ConjureUnionType):
    _dataset: Optional["ingest_api_IngestDatasetFileDetails"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset': ConjureFieldDefinition('dataset', ingest_api_IngestDatasetFileDetails)
        }

    def __init__(
            self,
            dataset: Optional["ingest_api_IngestDatasetFileDetails"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (dataset is not None) != 1:
                raise ValueError('a union must contain a single member')

            if dataset is not None:
                self._dataset = dataset
                self._type = 'dataset'

        elif type_of_union == 'dataset':
            if dataset is None:
                raise ValueError('a union value must not be None')
            self._dataset = dataset
            self._type = 'dataset'

    @builtins.property
    def dataset(self) -> Optional["ingest_api_IngestDatasetFileDetails"]:
        return self._dataset

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_IngestDetailsVisitor):
            raise ValueError('{} is not an instance of ingest_api_IngestDetailsVisitor'.format(visitor.__class__.__name__))
        if self._type == 'dataset' and self.dataset is not None:
            return visitor._dataset(self.dataset)


ingest_api_IngestDetails.__name__ = "IngestDetails"
ingest_api_IngestDetails.__qualname__ = "IngestDetails"
ingest_api_IngestDetails.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestDetailsVisitor:

    @abstractmethod
    def _dataset(self, dataset: "ingest_api_IngestDatasetFileDetails") -> Any:
        pass


ingest_api_IngestDetailsVisitor.__name__ = "IngestDetailsVisitor"
ingest_api_IngestDetailsVisitor.__qualname__ = "IngestDetailsVisitor"
ingest_api_IngestDetailsVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestMcapRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'sources': ConjureFieldDefinition('sources', List[ingest_api_IngestSource]),
            'channel_config': ConjureFieldDefinition('channelConfig', List[ingest_api_McapChannelConfig]),
            'channels': ConjureFieldDefinition('channels', OptionalTypeWrapper[ingest_api_McapChannels]),
            'properties': ConjureFieldDefinition('properties', Dict[ingest_api_PropertyName, ingest_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[ingest_api_Label]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_sources', '_channel_config', '_channels', '_properties', '_labels', '_title', '_description']

    def __init__(self, channel_config: List["ingest_api_McapChannelConfig"], labels: List[str], properties: Dict[str, str], sources: List["ingest_api_IngestSource"], channels: Optional["ingest_api_McapChannels"] = None, description: Optional[str] = None, title: Optional[str] = None) -> None:
        self._sources = sources
        self._channel_config = channel_config
        self._channels = channels
        self._properties = properties
        self._labels = labels
        self._title = title
        self._description = description

    @builtins.property
    def sources(self) -> List["ingest_api_IngestSource"]:
        """
        List of files in S3 to be ingested. These should be ordered by time, as data will be ingested and 
concatenated across all the files.
Note: only a single files are currently supported, this field is mostly for forward compatibility.
        """
        return self._sources

    @builtins.property
    def channel_config(self) -> List["ingest_api_McapChannelConfig"]:
        return self._channel_config

    @builtins.property
    def channels(self) -> Optional["ingest_api_McapChannels"]:
        """
        Config to define which channels in the mcap should be ingested. The default is to ingest only
channels with config, otherwise the mcap may not be supported.
        """
        return self._channels

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description


ingest_api_IngestMcapRequest.__name__ = "IngestMcapRequest"
ingest_api_IngestMcapRequest.__qualname__ = "IngestMcapRequest"
ingest_api_IngestMcapRequest.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestMcapResponse(ConjureBeanType):
    """
    Returns references to the data ingested from an MCAP file, along with a rid to monitor the progress of the
ingestion.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'outputs': ConjureFieldDefinition('outputs', List[ingest_api_McapIngestionOutput]),
            'async_handle': ConjureFieldDefinition('asyncHandle', ingest_api_AsyncHandle)
        }

    __slots__: List[str] = ['_outputs', '_async_handle']

    def __init__(self, async_handle: "ingest_api_AsyncHandle", outputs: List["ingest_api_McapIngestionOutput"]) -> None:
        self._outputs = outputs
        self._async_handle = async_handle

    @builtins.property
    def outputs(self) -> List["ingest_api_McapIngestionOutput"]:
        return self._outputs

    @builtins.property
    def async_handle(self) -> "ingest_api_AsyncHandle":
        return self._async_handle


ingest_api_IngestMcapResponse.__name__ = "IngestMcapResponse"
ingest_api_IngestMcapResponse.__qualname__ = "IngestMcapResponse"
ingest_api_IngestMcapResponse.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestOptions(ConjureUnionType):
    _dataflash: Optional["ingest_api_DataflashOpts"] = None
    _mcap_protobuf_timeseries: Optional["ingest_api_McapProtobufTimeseriesOpts"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataflash': ConjureFieldDefinition('dataflash', ingest_api_DataflashOpts),
            'mcap_protobuf_timeseries': ConjureFieldDefinition('mcapProtobufTimeseries', ingest_api_McapProtobufTimeseriesOpts)
        }

    def __init__(
            self,
            dataflash: Optional["ingest_api_DataflashOpts"] = None,
            mcap_protobuf_timeseries: Optional["ingest_api_McapProtobufTimeseriesOpts"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (dataflash is not None) + (mcap_protobuf_timeseries is not None) != 1:
                raise ValueError('a union must contain a single member')

            if dataflash is not None:
                self._dataflash = dataflash
                self._type = 'dataflash'
            if mcap_protobuf_timeseries is not None:
                self._mcap_protobuf_timeseries = mcap_protobuf_timeseries
                self._type = 'mcapProtobufTimeseries'

        elif type_of_union == 'dataflash':
            if dataflash is None:
                raise ValueError('a union value must not be None')
            self._dataflash = dataflash
            self._type = 'dataflash'
        elif type_of_union == 'mcapProtobufTimeseries':
            if mcap_protobuf_timeseries is None:
                raise ValueError('a union value must not be None')
            self._mcap_protobuf_timeseries = mcap_protobuf_timeseries
            self._type = 'mcapProtobufTimeseries'

    @builtins.property
    def dataflash(self) -> Optional["ingest_api_DataflashOpts"]:
        return self._dataflash

    @builtins.property
    def mcap_protobuf_timeseries(self) -> Optional["ingest_api_McapProtobufTimeseriesOpts"]:
        return self._mcap_protobuf_timeseries

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_IngestOptionsVisitor):
            raise ValueError('{} is not an instance of ingest_api_IngestOptionsVisitor'.format(visitor.__class__.__name__))
        if self._type == 'dataflash' and self.dataflash is not None:
            return visitor._dataflash(self.dataflash)
        if self._type == 'mcapProtobufTimeseries' and self.mcap_protobuf_timeseries is not None:
            return visitor._mcap_protobuf_timeseries(self.mcap_protobuf_timeseries)


ingest_api_IngestOptions.__name__ = "IngestOptions"
ingest_api_IngestOptions.__qualname__ = "IngestOptions"
ingest_api_IngestOptions.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestOptionsVisitor:

    @abstractmethod
    def _dataflash(self, dataflash: "ingest_api_DataflashOpts") -> Any:
        pass

    @abstractmethod
    def _mcap_protobuf_timeseries(self, mcap_protobuf_timeseries: "ingest_api_McapProtobufTimeseriesOpts") -> Any:
        pass


ingest_api_IngestOptionsVisitor.__name__ = "IngestOptionsVisitor"
ingest_api_IngestOptionsVisitor.__qualname__ = "IngestOptionsVisitor"
ingest_api_IngestOptionsVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestProgressV2(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'start_time': ConjureFieldDefinition('startTime', str),
            'end_time': ConjureFieldDefinition('endTime', OptionalTypeWrapper[str]),
            'ingest_status': ConjureFieldDefinition('ingestStatus', ingest_api_IngestStatusV2),
            'incalculable': ConjureFieldDefinition('incalculable', OptionalTypeWrapper[bool])
        }

    __slots__: List[str] = ['_start_time', '_end_time', '_ingest_status', '_incalculable']

    def __init__(self, ingest_status: "ingest_api_IngestStatusV2", start_time: str, end_time: Optional[str] = None, incalculable: Optional[bool] = None) -> None:
        self._start_time = start_time
        self._end_time = end_time
        self._ingest_status = ingest_status
        self._incalculable = incalculable

    @builtins.property
    def start_time(self) -> str:
        """
        Timestamp at start of ingest
        """
        return self._start_time

    @builtins.property
    def end_time(self) -> Optional[str]:
        """
        Timestamp at end of ingest, empty if still in progress
        """
        return self._end_time

    @builtins.property
    def ingest_status(self) -> "ingest_api_IngestStatusV2":
        """
        Status of ingest, contains error if failed
        """
        return self._ingest_status

    @builtins.property
    def incalculable(self) -> Optional[bool]:
        """
        Whether ingest duration can be reliably calculated
        """
        return self._incalculable


ingest_api_IngestProgressV2.__name__ = "IngestProgressV2"
ingest_api_IngestProgressV2.__qualname__ = "IngestProgressV2"
ingest_api_IngestProgressV2.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'options': ConjureFieldDefinition('options', ingest_api_IngestOptions)
        }

    __slots__: List[str] = ['_options']

    def __init__(self, options: "ingest_api_IngestOptions") -> None:
        self._options = options

    @builtins.property
    def options(self) -> "ingest_api_IngestOptions":
        return self._options


ingest_api_IngestRequest.__name__ = "IngestRequest"
ingest_api_IngestRequest.__qualname__ = "IngestRequest"
ingest_api_IngestRequest.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'details': ConjureFieldDefinition('details', ingest_api_IngestDetails)
        }

    __slots__: List[str] = ['_details']

    def __init__(self, details: "ingest_api_IngestDetails") -> None:
        self._details = details

    @builtins.property
    def details(self) -> "ingest_api_IngestDetails":
        return self._details


ingest_api_IngestResponse.__name__ = "IngestResponse"
ingest_api_IngestResponse.__qualname__ = "IngestResponse"
ingest_api_IngestResponse.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestRunDataSource(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_source': ConjureFieldDefinition('dataSource', ingest_api_IngestDataSource),
            'time_offset_spec': ConjureFieldDefinition('timeOffsetSpec', OptionalTypeWrapper[ingest_api_TimeOffsetSpec])
        }

    __slots__: List[str] = ['_data_source', '_time_offset_spec']

    def __init__(self, data_source: "ingest_api_IngestDataSource", time_offset_spec: Optional["ingest_api_TimeOffsetSpec"] = None) -> None:
        self._data_source = data_source
        self._time_offset_spec = time_offset_spec

    @builtins.property
    def data_source(self) -> "ingest_api_IngestDataSource":
        return self._data_source

    @builtins.property
    def time_offset_spec(self) -> Optional["ingest_api_TimeOffsetSpec"]:
        return self._time_offset_spec


ingest_api_IngestRunDataSource.__name__ = "IngestRunDataSource"
ingest_api_IngestRunDataSource.__qualname__ = "IngestRunDataSource"
ingest_api_IngestRunDataSource.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestRunRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', OptionalTypeWrapper[str]),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', str),
            'start_time': ConjureFieldDefinition('startTime', ingest_api_UtcTimestamp),
            'end_time': ConjureFieldDefinition('endTime', OptionalTypeWrapper[ingest_api_UtcTimestamp]),
            'properties': ConjureFieldDefinition('properties', Dict[ingest_api_PropertyName, ingest_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[ingest_api_Label]),
            'run_prefix': ConjureFieldDefinition('runPrefix', OptionalTypeWrapper[str]),
            'data_sources': ConjureFieldDefinition('dataSources', Dict[ingest_api_DataSourceRefName, ingest_api_IngestRunDataSource])
        }

    __slots__: List[str] = ['_rid', '_title', '_description', '_start_time', '_end_time', '_properties', '_labels', '_run_prefix', '_data_sources']

    def __init__(self, data_sources: Dict[str, "ingest_api_IngestRunDataSource"], description: str, labels: List[str], properties: Dict[str, str], start_time: "ingest_api_UtcTimestamp", title: str, end_time: Optional["ingest_api_UtcTimestamp"] = None, rid: Optional[str] = None, run_prefix: Optional[str] = None) -> None:
        self._rid = rid
        self._title = title
        self._description = description
        self._start_time = start_time
        self._end_time = end_time
        self._properties = properties
        self._labels = labels
        self._run_prefix = run_prefix
        self._data_sources = data_sources

    @builtins.property
    def rid(self) -> Optional[str]:
        """
        If a run with the same rid already exists, the run will be updated.
        """
        return self._rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def start_time(self) -> "ingest_api_UtcTimestamp":
        return self._start_time

    @builtins.property
    def end_time(self) -> Optional["ingest_api_UtcTimestamp"]:
        return self._end_time

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def run_prefix(self) -> Optional[str]:
        """
        for example, SIM, HTL, FLT
        """
        return self._run_prefix

    @builtins.property
    def data_sources(self) -> Dict[str, "ingest_api_IngestRunDataSource"]:
        return self._data_sources


ingest_api_IngestRunRequest.__name__ = "IngestRunRequest"
ingest_api_IngestRunRequest.__qualname__ = "IngestRunRequest"
ingest_api_IngestRunRequest.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestRunResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'run_rid': ConjureFieldDefinition('runRid', ingest_api_RunRid)
        }

    __slots__: List[str] = ['_run_rid']

    def __init__(self, run_rid: str) -> None:
        self._run_rid = run_rid

    @builtins.property
    def run_rid(self) -> str:
        return self._run_rid


ingest_api_IngestRunResponse.__name__ = "IngestRunResponse"
ingest_api_IngestRunResponse.__qualname__ = "IngestRunResponse"
ingest_api_IngestRunResponse.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestService(Service):
    """
    The Ingest Service handles the data ingestion into Nominal/Clickhouse.
    """

    def ingest(self, auth_header: str, trigger_ingest: "ingest_api_IngestRequest") -> "ingest_api_IngestResponse":
        """
        Triggers an ingest job, allowing either creating a new dataset or uploading to an
existing one. This endpoint is meant to supersede all other ingestion endpoints as their functionality
gets migrated to this one.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(trigger_ingest)

        _path = '/ingest/v1/ingest'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), ingest_api_IngestResponse, self._return_none_for_unknown_union_types)

    def deprecated_trigger_ingest(self, auth_header: str, trigger_ingest: "ingest_api_DeprecatedTriggerIngest") -> "ingest_api_TriggeredIngest":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(trigger_ingest)

        _path = '/ingest/v1/trigger-ingest'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), ingest_api_TriggeredIngest, self._return_none_for_unknown_union_types)

    def trigger_ingest(self, auth_header: str, trigger_ingest: "ingest_api_TriggerIngest") -> "ingest_api_TriggeredIngest":
        """
        Triggers an ingest job for the given data source.
The ingest job will be processed asynchronously.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(trigger_ingest)

        _path = '/ingest/v1/trigger-ingest-v2'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), ingest_api_TriggeredIngest, self._return_none_for_unknown_union_types)

    def trigger_file_ingest(self, auth_header: str, trigger_ingest: "ingest_api_TriggerFileIngest") -> "ingest_api_TriggeredIngest":
        """
        Triggers an ingest job of a new file, allowing either creating a new dataset or uploading to an
existing one.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(trigger_ingest)

        _path = '/ingest/v1/trigger-file-ingest'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), ingest_api_TriggeredIngest, self._return_none_for_unknown_union_types)

    def ingest_run(self, auth_header: str, request: "ingest_api_IngestRunRequest") -> "ingest_api_IngestRunResponse":
        """
        Creates a run and ingests data sources to be added to the run.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/ingest/v1/ingest-run'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), ingest_api_IngestRunResponse, self._return_none_for_unknown_union_types)

    def ingest_video(self, auth_header: str, ingest_video: "ingest_api_IngestVideoRequest") -> "ingest_api_IngestVideoResponse":
        """
        Ingests video data from a S3 Nominal upload bucket.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(ingest_video)

        _path = '/ingest/v1/ingest-video'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), ingest_api_IngestVideoResponse, self._return_none_for_unknown_union_types)

    def ingest_mcap(self, auth_header: str, ingest_video: "ingest_api_IngestMcapRequest") -> "ingest_api_IngestMcapResponse":
        """
        Ingests data from mcap files in the S3 Nominal upload bucket.
Currently only supports ingesting video channels.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(ingest_video)

        _path = '/ingest/v1/ingest-mcap'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), ingest_api_IngestMcapResponse, self._return_none_for_unknown_union_types)


ingest_api_IngestService.__name__ = "IngestService"
ingest_api_IngestService.__qualname__ = "IngestService"
ingest_api_IngestService.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestSource(ConjureUnionType):
    _s3: Optional["ingest_api_S3IngestSource"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            's3': ConjureFieldDefinition('s3', ingest_api_S3IngestSource)
        }

    def __init__(
            self,
            s3: Optional["ingest_api_S3IngestSource"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (s3 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if s3 is not None:
                self._s3 = s3
                self._type = 's3'

        elif type_of_union == 's3':
            if s3 is None:
                raise ValueError('a union value must not be None')
            self._s3 = s3
            self._type = 's3'

    @builtins.property
    def s3(self) -> Optional["ingest_api_S3IngestSource"]:
        return self._s3

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_IngestSourceVisitor):
            raise ValueError('{} is not an instance of ingest_api_IngestSourceVisitor'.format(visitor.__class__.__name__))
        if self._type == 's3' and self.s3 is not None:
            return visitor._s3(self.s3)


ingest_api_IngestSource.__name__ = "IngestSource"
ingest_api_IngestSource.__qualname__ = "IngestSource"
ingest_api_IngestSource.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestSourceVisitor:

    @abstractmethod
    def _s3(self, s3: "ingest_api_S3IngestSource") -> Any:
        pass


ingest_api_IngestSourceVisitor.__name__ = "IngestSourceVisitor"
ingest_api_IngestSourceVisitor.__qualname__ = "IngestSourceVisitor"
ingest_api_IngestSourceVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestSourceMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'timestamp_metadata': ConjureFieldDefinition('timestampMetadata', OptionalTypeWrapper[ingest_api_TimestampMetadata])
        }

    __slots__: List[str] = ['_timestamp_metadata']

    def __init__(self, timestamp_metadata: Optional["ingest_api_TimestampMetadata"] = None) -> None:
        self._timestamp_metadata = timestamp_metadata

    @builtins.property
    def timestamp_metadata(self) -> Optional["ingest_api_TimestampMetadata"]:
        return self._timestamp_metadata


ingest_api_IngestSourceMetadata.__name__ = "IngestSourceMetadata"
ingest_api_IngestSourceMetadata.__qualname__ = "IngestSourceMetadata"
ingest_api_IngestSourceMetadata.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestStatus(ConjureEnumType):

    IN_PROGRESS = 'IN_PROGRESS'
    '''IN_PROGRESS'''
    COMPLETED = 'COMPLETED'
    '''COMPLETED'''
    FAILED = 'FAILED'
    '''FAILED'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


ingest_api_IngestStatus.__name__ = "IngestStatus"
ingest_api_IngestStatus.__qualname__ = "IngestStatus"
ingest_api_IngestStatus.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestStatusV2(ConjureUnionType):
    _success: Optional["ingest_api_SuccessResult"] = None
    _error: Optional["ingest_api_ErrorResult"] = None
    _in_progress: Optional["ingest_api_InProgressResult"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'success': ConjureFieldDefinition('success', ingest_api_SuccessResult),
            'error': ConjureFieldDefinition('error', ingest_api_ErrorResult),
            'in_progress': ConjureFieldDefinition('inProgress', ingest_api_InProgressResult)
        }

    def __init__(
            self,
            success: Optional["ingest_api_SuccessResult"] = None,
            error: Optional["ingest_api_ErrorResult"] = None,
            in_progress: Optional["ingest_api_InProgressResult"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (success is not None) + (error is not None) + (in_progress is not None) != 1:
                raise ValueError('a union must contain a single member')

            if success is not None:
                self._success = success
                self._type = 'success'
            if error is not None:
                self._error = error
                self._type = 'error'
            if in_progress is not None:
                self._in_progress = in_progress
                self._type = 'inProgress'

        elif type_of_union == 'success':
            if success is None:
                raise ValueError('a union value must not be None')
            self._success = success
            self._type = 'success'
        elif type_of_union == 'error':
            if error is None:
                raise ValueError('a union value must not be None')
            self._error = error
            self._type = 'error'
        elif type_of_union == 'inProgress':
            if in_progress is None:
                raise ValueError('a union value must not be None')
            self._in_progress = in_progress
            self._type = 'inProgress'

    @builtins.property
    def success(self) -> Optional["ingest_api_SuccessResult"]:
        return self._success

    @builtins.property
    def error(self) -> Optional["ingest_api_ErrorResult"]:
        return self._error

    @builtins.property
    def in_progress(self) -> Optional["ingest_api_InProgressResult"]:
        return self._in_progress

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_IngestStatusV2Visitor):
            raise ValueError('{} is not an instance of ingest_api_IngestStatusV2Visitor'.format(visitor.__class__.__name__))
        if self._type == 'success' and self.success is not None:
            return visitor._success(self.success)
        if self._type == 'error' and self.error is not None:
            return visitor._error(self.error)
        if self._type == 'inProgress' and self.in_progress is not None:
            return visitor._in_progress(self.in_progress)


ingest_api_IngestStatusV2.__name__ = "IngestStatusV2"
ingest_api_IngestStatusV2.__qualname__ = "IngestStatusV2"
ingest_api_IngestStatusV2.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestStatusV2Visitor:

    @abstractmethod
    def _success(self, success: "ingest_api_SuccessResult") -> Any:
        pass

    @abstractmethod
    def _error(self, error: "ingest_api_ErrorResult") -> Any:
        pass

    @abstractmethod
    def _in_progress(self, in_progress: "ingest_api_InProgressResult") -> Any:
        pass


ingest_api_IngestStatusV2Visitor.__name__ = "IngestStatusV2Visitor"
ingest_api_IngestStatusV2Visitor.__qualname__ = "IngestStatusV2Visitor"
ingest_api_IngestStatusV2Visitor.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestVideoDestination(ConjureUnionType):
    _new_video: Optional["ingest_api_NewVideoIngestDestination"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'new_video': ConjureFieldDefinition('newVideo', ingest_api_NewVideoIngestDestination)
        }

    def __init__(
            self,
            new_video: Optional["ingest_api_NewVideoIngestDestination"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (new_video is not None) != 1:
                raise ValueError('a union must contain a single member')

            if new_video is not None:
                self._new_video = new_video
                self._type = 'newVideo'

        elif type_of_union == 'newVideo':
            if new_video is None:
                raise ValueError('a union value must not be None')
            self._new_video = new_video
            self._type = 'newVideo'

    @builtins.property
    def new_video(self) -> Optional["ingest_api_NewVideoIngestDestination"]:
        return self._new_video

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_IngestVideoDestinationVisitor):
            raise ValueError('{} is not an instance of ingest_api_IngestVideoDestinationVisitor'.format(visitor.__class__.__name__))
        if self._type == 'newVideo' and self.new_video is not None:
            return visitor._new_video(self.new_video)


ingest_api_IngestVideoDestination.__name__ = "IngestVideoDestination"
ingest_api_IngestVideoDestination.__qualname__ = "IngestVideoDestination"
ingest_api_IngestVideoDestination.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestVideoDestinationVisitor:

    @abstractmethod
    def _new_video(self, new_video: "ingest_api_NewVideoIngestDestination") -> Any:
        pass


ingest_api_IngestVideoDestinationVisitor.__name__ = "IngestVideoDestinationVisitor"
ingest_api_IngestVideoDestinationVisitor.__qualname__ = "IngestVideoDestinationVisitor"
ingest_api_IngestVideoDestinationVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestVideoRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'sources': ConjureFieldDefinition('sources', List[ingest_api_IngestSource]),
            'properties': ConjureFieldDefinition('properties', Dict[ingest_api_PropertyName, ingest_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[ingest_api_Label]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'timestamps': ConjureFieldDefinition('timestamps', ingest_api_VideoTimestampManifest)
        }

    __slots__: List[str] = ['_sources', '_properties', '_labels', '_title', '_description', '_timestamps']

    def __init__(self, labels: List[str], properties: Dict[str, str], sources: List["ingest_api_IngestSource"], timestamps: "ingest_api_VideoTimestampManifest", description: Optional[str] = None, title: Optional[str] = None) -> None:
        self._sources = sources
        self._properties = properties
        self._labels = labels
        self._title = title
        self._description = description
        self._timestamps = timestamps

    @builtins.property
    def sources(self) -> List["ingest_api_IngestSource"]:
        return self._sources

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def timestamps(self) -> "ingest_api_VideoTimestampManifest":
        return self._timestamps


ingest_api_IngestVideoRequest.__name__ = "IngestVideoRequest"
ingest_api_IngestVideoRequest.__qualname__ = "IngestVideoRequest"
ingest_api_IngestVideoRequest.__module__ = "nominal_api.ingest_api"


class ingest_api_IngestVideoResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'video_rid': ConjureFieldDefinition('videoRid', api_rids_VideoRid),
            'async_handle': ConjureFieldDefinition('asyncHandle', ingest_api_AsyncHandle)
        }

    __slots__: List[str] = ['_video_rid', '_async_handle']

    def __init__(self, async_handle: "ingest_api_AsyncHandle", video_rid: str) -> None:
        self._video_rid = video_rid
        self._async_handle = async_handle

    @builtins.property
    def video_rid(self) -> str:
        return self._video_rid

    @builtins.property
    def async_handle(self) -> "ingest_api_AsyncHandle":
        return self._async_handle


ingest_api_IngestVideoResponse.__name__ = "IngestVideoResponse"
ingest_api_IngestVideoResponse.__qualname__ = "IngestVideoResponse"
ingest_api_IngestVideoResponse.__module__ = "nominal_api.ingest_api"


class ingest_api_InitiateMultipartUploadRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'filename': ConjureFieldDefinition('filename', str),
            'filetype': ConjureFieldDefinition('filetype', str)
        }

    __slots__: List[str] = ['_filename', '_filetype']

    def __init__(self, filename: str, filetype: str) -> None:
        self._filename = filename
        self._filetype = filetype

    @builtins.property
    def filename(self) -> str:
        return self._filename

    @builtins.property
    def filetype(self) -> str:
        return self._filetype


ingest_api_InitiateMultipartUploadRequest.__name__ = "InitiateMultipartUploadRequest"
ingest_api_InitiateMultipartUploadRequest.__qualname__ = "InitiateMultipartUploadRequest"
ingest_api_InitiateMultipartUploadRequest.__module__ = "nominal_api.ingest_api"


class ingest_api_InitiateMultipartUploadResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'upload_id': ConjureFieldDefinition('uploadId', str),
            'key': ConjureFieldDefinition('key', str)
        }

    __slots__: List[str] = ['_upload_id', '_key']

    def __init__(self, key: str, upload_id: str) -> None:
        self._upload_id = upload_id
        self._key = key

    @builtins.property
    def upload_id(self) -> str:
        return self._upload_id

    @builtins.property
    def key(self) -> str:
        return self._key


ingest_api_InitiateMultipartUploadResponse.__name__ = "InitiateMultipartUploadResponse"
ingest_api_InitiateMultipartUploadResponse.__qualname__ = "InitiateMultipartUploadResponse"
ingest_api_InitiateMultipartUploadResponse.__module__ = "nominal_api.ingest_api"


class ingest_api_Iso8601Timestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



ingest_api_Iso8601Timestamp.__name__ = "Iso8601Timestamp"
ingest_api_Iso8601Timestamp.__qualname__ = "Iso8601Timestamp"
ingest_api_Iso8601Timestamp.__module__ = "nominal_api.ingest_api"


class ingest_api_LogTime(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



ingest_api_LogTime.__name__ = "LogTime"
ingest_api_LogTime.__qualname__ = "LogTime"
ingest_api_LogTime.__module__ = "nominal_api.ingest_api"


class ingest_api_McapChannelConfig(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'locator': ConjureFieldDefinition('locator', api_McapChannelLocator),
            'channel_type': ConjureFieldDefinition('channelType', ingest_api_McapChannelConfigType)
        }

    __slots__: List[str] = ['_locator', '_channel_type']

    def __init__(self, channel_type: "ingest_api_McapChannelConfigType", locator: "api_McapChannelLocator") -> None:
        self._locator = locator
        self._channel_type = channel_type

    @builtins.property
    def locator(self) -> "api_McapChannelLocator":
        return self._locator

    @builtins.property
    def channel_type(self) -> "ingest_api_McapChannelConfigType":
        return self._channel_type


ingest_api_McapChannelConfig.__name__ = "McapChannelConfig"
ingest_api_McapChannelConfig.__qualname__ = "McapChannelConfig"
ingest_api_McapChannelConfig.__module__ = "nominal_api.ingest_api"


class ingest_api_McapChannelConfigType(ConjureUnionType):
    _video: Optional["ingest_api_McapVideoChannelConfig"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'video': ConjureFieldDefinition('video', ingest_api_McapVideoChannelConfig)
        }

    def __init__(
            self,
            video: Optional["ingest_api_McapVideoChannelConfig"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (video is not None) != 1:
                raise ValueError('a union must contain a single member')

            if video is not None:
                self._video = video
                self._type = 'video'

        elif type_of_union == 'video':
            if video is None:
                raise ValueError('a union value must not be None')
            self._video = video
            self._type = 'video'

    @builtins.property
    def video(self) -> Optional["ingest_api_McapVideoChannelConfig"]:
        return self._video

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_McapChannelConfigTypeVisitor):
            raise ValueError('{} is not an instance of ingest_api_McapChannelConfigTypeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'video' and self.video is not None:
            return visitor._video(self.video)


ingest_api_McapChannelConfigType.__name__ = "McapChannelConfigType"
ingest_api_McapChannelConfigType.__qualname__ = "McapChannelConfigType"
ingest_api_McapChannelConfigType.__module__ = "nominal_api.ingest_api"


class ingest_api_McapChannelConfigTypeVisitor:

    @abstractmethod
    def _video(self, video: "ingest_api_McapVideoChannelConfig") -> Any:
        pass


ingest_api_McapChannelConfigTypeVisitor.__name__ = "McapChannelConfigTypeVisitor"
ingest_api_McapChannelConfigTypeVisitor.__qualname__ = "McapChannelConfigTypeVisitor"
ingest_api_McapChannelConfigTypeVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_McapChannels(ConjureUnionType):
    _all: Optional["api_Empty"] = None
    _include: Optional[List["api_McapChannelLocator"]] = None
    _exclude: Optional[List["api_McapChannelLocator"]] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'all': ConjureFieldDefinition('all', api_Empty),
            'include': ConjureFieldDefinition('include', List[api_McapChannelLocator]),
            'exclude': ConjureFieldDefinition('exclude', List[api_McapChannelLocator])
        }

    def __init__(
            self,
            all: Optional["api_Empty"] = None,
            include: Optional[List["api_McapChannelLocator"]] = None,
            exclude: Optional[List["api_McapChannelLocator"]] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (all is not None) + (include is not None) + (exclude is not None) != 1:
                raise ValueError('a union must contain a single member')

            if all is not None:
                self._all = all
                self._type = 'all'
            if include is not None:
                self._include = include
                self._type = 'include'
            if exclude is not None:
                self._exclude = exclude
                self._type = 'exclude'

        elif type_of_union == 'all':
            if all is None:
                raise ValueError('a union value must not be None')
            self._all = all
            self._type = 'all'
        elif type_of_union == 'include':
            if include is None:
                raise ValueError('a union value must not be None')
            self._include = include
            self._type = 'include'
        elif type_of_union == 'exclude':
            if exclude is None:
                raise ValueError('a union value must not be None')
            self._exclude = exclude
            self._type = 'exclude'

    @builtins.property
    def all(self) -> Optional["api_Empty"]:
        return self._all

    @builtins.property
    def include(self) -> Optional[List["api_McapChannelLocator"]]:
        return self._include

    @builtins.property
    def exclude(self) -> Optional[List["api_McapChannelLocator"]]:
        return self._exclude

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_McapChannelsVisitor):
            raise ValueError('{} is not an instance of ingest_api_McapChannelsVisitor'.format(visitor.__class__.__name__))
        if self._type == 'all' and self.all is not None:
            return visitor._all(self.all)
        if self._type == 'include' and self.include is not None:
            return visitor._include(self.include)
        if self._type == 'exclude' and self.exclude is not None:
            return visitor._exclude(self.exclude)


ingest_api_McapChannels.__name__ = "McapChannels"
ingest_api_McapChannels.__qualname__ = "McapChannels"
ingest_api_McapChannels.__module__ = "nominal_api.ingest_api"


class ingest_api_McapChannelsVisitor:

    @abstractmethod
    def _all(self, all: "api_Empty") -> Any:
        pass

    @abstractmethod
    def _include(self, include: List["api_McapChannelLocator"]) -> Any:
        pass

    @abstractmethod
    def _exclude(self, exclude: List["api_McapChannelLocator"]) -> Any:
        pass


ingest_api_McapChannelsVisitor.__name__ = "McapChannelsVisitor"
ingest_api_McapChannelsVisitor.__qualname__ = "McapChannelsVisitor"
ingest_api_McapChannelsVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_McapDestination(ConjureUnionType):
    _video_rid: Optional[str] = None
    _dataset_rid: Optional[str] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'video_rid': ConjureFieldDefinition('videoRid', api_rids_VideoRid),
            'dataset_rid': ConjureFieldDefinition('datasetRid', str)
        }

    def __init__(
            self,
            video_rid: Optional[str] = None,
            dataset_rid: Optional[str] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (video_rid is not None) + (dataset_rid is not None) != 1:
                raise ValueError('a union must contain a single member')

            if video_rid is not None:
                self._video_rid = video_rid
                self._type = 'videoRid'
            if dataset_rid is not None:
                self._dataset_rid = dataset_rid
                self._type = 'datasetRid'

        elif type_of_union == 'videoRid':
            if video_rid is None:
                raise ValueError('a union value must not be None')
            self._video_rid = video_rid
            self._type = 'videoRid'
        elif type_of_union == 'datasetRid':
            if dataset_rid is None:
                raise ValueError('a union value must not be None')
            self._dataset_rid = dataset_rid
            self._type = 'datasetRid'

    @builtins.property
    def video_rid(self) -> Optional[str]:
        return self._video_rid

    @builtins.property
    def dataset_rid(self) -> Optional[str]:
        return self._dataset_rid

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_McapDestinationVisitor):
            raise ValueError('{} is not an instance of ingest_api_McapDestinationVisitor'.format(visitor.__class__.__name__))
        if self._type == 'videoRid' and self.video_rid is not None:
            return visitor._video_rid(self.video_rid)
        if self._type == 'datasetRid' and self.dataset_rid is not None:
            return visitor._dataset_rid(self.dataset_rid)


ingest_api_McapDestination.__name__ = "McapDestination"
ingest_api_McapDestination.__qualname__ = "McapDestination"
ingest_api_McapDestination.__module__ = "nominal_api.ingest_api"


class ingest_api_McapDestinationVisitor:

    @abstractmethod
    def _video_rid(self, video_rid: str) -> Any:
        pass

    @abstractmethod
    def _dataset_rid(self, dataset_rid: str) -> Any:
        pass


ingest_api_McapDestinationVisitor.__name__ = "McapDestinationVisitor"
ingest_api_McapDestinationVisitor.__qualname__ = "McapDestinationVisitor"
ingest_api_McapDestinationVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_McapIngestionOutput(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'source': ConjureFieldDefinition('source', ingest_api_McapSource),
            'target': ConjureFieldDefinition('target', ingest_api_McapDestination)
        }

    __slots__: List[str] = ['_source', '_target']

    def __init__(self, source: "ingest_api_McapSource", target: "ingest_api_McapDestination") -> None:
        self._source = source
        self._target = target

    @builtins.property
    def source(self) -> "ingest_api_McapSource":
        return self._source

    @builtins.property
    def target(self) -> "ingest_api_McapDestination":
        return self._target


ingest_api_McapIngestionOutput.__name__ = "McapIngestionOutput"
ingest_api_McapIngestionOutput.__qualname__ = "McapIngestionOutput"
ingest_api_McapIngestionOutput.__module__ = "nominal_api.ingest_api"


class ingest_api_McapProtobufTimeseriesOpts(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'source': ConjureFieldDefinition('source', ingest_api_IngestSource),
            'target': ConjureFieldDefinition('target', ingest_api_DatasetIngestTarget),
            'channel_filter': ConjureFieldDefinition('channelFilter', ingest_api_McapChannels),
            'timestamp_type': ConjureFieldDefinition('timestampType', ingest_api_McapTimestampType)
        }

    __slots__: List[str] = ['_source', '_target', '_channel_filter', '_timestamp_type']

    def __init__(self, channel_filter: "ingest_api_McapChannels", source: "ingest_api_IngestSource", target: "ingest_api_DatasetIngestTarget", timestamp_type: "ingest_api_McapTimestampType") -> None:
        self._source = source
        self._target = target
        self._channel_filter = channel_filter
        self._timestamp_type = timestamp_type

    @builtins.property
    def source(self) -> "ingest_api_IngestSource":
        return self._source

    @builtins.property
    def target(self) -> "ingest_api_DatasetIngestTarget":
        return self._target

    @builtins.property
    def channel_filter(self) -> "ingest_api_McapChannels":
        return self._channel_filter

    @builtins.property
    def timestamp_type(self) -> "ingest_api_McapTimestampType":
        return self._timestamp_type


ingest_api_McapProtobufTimeseriesOpts.__name__ = "McapProtobufTimeseriesOpts"
ingest_api_McapProtobufTimeseriesOpts.__qualname__ = "McapProtobufTimeseriesOpts"
ingest_api_McapProtobufTimeseriesOpts.__module__ = "nominal_api.ingest_api"


class ingest_api_McapSource(ConjureUnionType):
    _single_channel: Optional["api_McapChannelLocator"] = None
    _mcap_file: Optional["ingest_api_IngestSource"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'single_channel': ConjureFieldDefinition('singleChannel', api_McapChannelLocator),
            'mcap_file': ConjureFieldDefinition('mcapFile', ingest_api_IngestSource)
        }

    def __init__(
            self,
            single_channel: Optional["api_McapChannelLocator"] = None,
            mcap_file: Optional["ingest_api_IngestSource"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (single_channel is not None) + (mcap_file is not None) != 1:
                raise ValueError('a union must contain a single member')

            if single_channel is not None:
                self._single_channel = single_channel
                self._type = 'singleChannel'
            if mcap_file is not None:
                self._mcap_file = mcap_file
                self._type = 'mcapFile'

        elif type_of_union == 'singleChannel':
            if single_channel is None:
                raise ValueError('a union value must not be None')
            self._single_channel = single_channel
            self._type = 'singleChannel'
        elif type_of_union == 'mcapFile':
            if mcap_file is None:
                raise ValueError('a union value must not be None')
            self._mcap_file = mcap_file
            self._type = 'mcapFile'

    @builtins.property
    def single_channel(self) -> Optional["api_McapChannelLocator"]:
        return self._single_channel

    @builtins.property
    def mcap_file(self) -> Optional["ingest_api_IngestSource"]:
        return self._mcap_file

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_McapSourceVisitor):
            raise ValueError('{} is not an instance of ingest_api_McapSourceVisitor'.format(visitor.__class__.__name__))
        if self._type == 'singleChannel' and self.single_channel is not None:
            return visitor._single_channel(self.single_channel)
        if self._type == 'mcapFile' and self.mcap_file is not None:
            return visitor._mcap_file(self.mcap_file)


ingest_api_McapSource.__name__ = "McapSource"
ingest_api_McapSource.__qualname__ = "McapSource"
ingest_api_McapSource.__module__ = "nominal_api.ingest_api"


class ingest_api_McapSourceVisitor:

    @abstractmethod
    def _single_channel(self, single_channel: "api_McapChannelLocator") -> Any:
        pass

    @abstractmethod
    def _mcap_file(self, mcap_file: "ingest_api_IngestSource") -> Any:
        pass


ingest_api_McapSourceVisitor.__name__ = "McapSourceVisitor"
ingest_api_McapSourceVisitor.__qualname__ = "McapSourceVisitor"
ingest_api_McapSourceVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_McapTimestampType(ConjureUnionType):
    """LogTime is default timestamp for MCAP messages and should be used in most cases."""
    _log_time: Optional["ingest_api_LogTime"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'log_time': ConjureFieldDefinition('logTime', ingest_api_LogTime)
        }

    def __init__(
            self,
            log_time: Optional["ingest_api_LogTime"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (log_time is not None) != 1:
                raise ValueError('a union must contain a single member')

            if log_time is not None:
                self._log_time = log_time
                self._type = 'logTime'

        elif type_of_union == 'logTime':
            if log_time is None:
                raise ValueError('a union value must not be None')
            self._log_time = log_time
            self._type = 'logTime'

    @builtins.property
    def log_time(self) -> Optional["ingest_api_LogTime"]:
        return self._log_time

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_McapTimestampTypeVisitor):
            raise ValueError('{} is not an instance of ingest_api_McapTimestampTypeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'logTime' and self.log_time is not None:
            return visitor._log_time(self.log_time)


ingest_api_McapTimestampType.__name__ = "McapTimestampType"
ingest_api_McapTimestampType.__qualname__ = "McapTimestampType"
ingest_api_McapTimestampType.__module__ = "nominal_api.ingest_api"


class ingest_api_McapTimestampTypeVisitor:

    @abstractmethod
    def _log_time(self, log_time: "ingest_api_LogTime") -> Any:
        pass


ingest_api_McapTimestampTypeVisitor.__name__ = "McapTimestampTypeVisitor"
ingest_api_McapTimestampTypeVisitor.__qualname__ = "McapTimestampTypeVisitor"
ingest_api_McapTimestampTypeVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_McapVideoChannelConfig(ConjureBeanType):
    """
    Ingest a channel as video. This requires:
* Using the `foxglove.CompressedVideo` schema
* Protobuf representation of the message data
* Messages are in sequential order in the mcap file
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



ingest_api_McapVideoChannelConfig.__name__ = "McapVideoChannelConfig"
ingest_api_McapVideoChannelConfig.__qualname__ = "McapVideoChannelConfig"
ingest_api_McapVideoChannelConfig.__module__ = "nominal_api.ingest_api"


class ingest_api_NewDataSource(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'source': ConjureFieldDefinition('source', ingest_api_IngestSource),
            'properties': ConjureFieldDefinition('properties', Dict[ingest_api_PropertyName, ingest_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[ingest_api_Label]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'name': ConjureFieldDefinition('name', OptionalTypeWrapper[str]),
            'time_column_spec': ConjureFieldDefinition('timeColumnSpec', OptionalTypeWrapper[ingest_api_TimestampMetadata]),
            'channel_config': ConjureFieldDefinition('channelConfig', OptionalTypeWrapper[ingest_api_ChannelConfig])
        }

    __slots__: List[str] = ['_source', '_properties', '_labels', '_description', '_name', '_time_column_spec', '_channel_config']

    def __init__(self, labels: List[str], properties: Dict[str, str], source: "ingest_api_IngestSource", channel_config: Optional["ingest_api_ChannelConfig"] = None, description: Optional[str] = None, name: Optional[str] = None, time_column_spec: Optional["ingest_api_TimestampMetadata"] = None) -> None:
        self._source = source
        self._properties = properties
        self._labels = labels
        self._description = description
        self._name = name
        self._time_column_spec = time_column_spec
        self._channel_config = channel_config

    @builtins.property
    def source(self) -> "ingest_api_IngestSource":
        return self._source

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def name(self) -> Optional[str]:
        return self._name

    @builtins.property
    def time_column_spec(self) -> Optional["ingest_api_TimestampMetadata"]:
        return self._time_column_spec

    @builtins.property
    def channel_config(self) -> Optional["ingest_api_ChannelConfig"]:
        return self._channel_config


ingest_api_NewDataSource.__name__ = "NewDataSource"
ingest_api_NewDataSource.__qualname__ = "NewDataSource"
ingest_api_NewDataSource.__module__ = "nominal_api.ingest_api"


class ingest_api_NewDatasetIngestDestination(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset_name': ConjureFieldDefinition('datasetName', OptionalTypeWrapper[str]),
            'dataset_description': ConjureFieldDefinition('datasetDescription', OptionalTypeWrapper[str]),
            'properties': ConjureFieldDefinition('properties', Dict[ingest_api_PropertyName, ingest_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[ingest_api_Label]),
            'channel_config': ConjureFieldDefinition('channelConfig', OptionalTypeWrapper[ingest_api_ChannelConfig])
        }

    __slots__: List[str] = ['_dataset_name', '_dataset_description', '_properties', '_labels', '_channel_config']

    def __init__(self, labels: List[str], properties: Dict[str, str], channel_config: Optional["ingest_api_ChannelConfig"] = None, dataset_description: Optional[str] = None, dataset_name: Optional[str] = None) -> None:
        self._dataset_name = dataset_name
        self._dataset_description = dataset_description
        self._properties = properties
        self._labels = labels
        self._channel_config = channel_config

    @builtins.property
    def dataset_name(self) -> Optional[str]:
        return self._dataset_name

    @builtins.property
    def dataset_description(self) -> Optional[str]:
        return self._dataset_description

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def channel_config(self) -> Optional["ingest_api_ChannelConfig"]:
        return self._channel_config


ingest_api_NewDatasetIngestDestination.__name__ = "NewDatasetIngestDestination"
ingest_api_NewDatasetIngestDestination.__qualname__ = "NewDatasetIngestDestination"
ingest_api_NewDatasetIngestDestination.__module__ = "nominal_api.ingest_api"


class ingest_api_NewVideoIngestDestination(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'properties': ConjureFieldDefinition('properties', Dict[ingest_api_PropertyName, ingest_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[ingest_api_Label]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_properties', '_labels', '_title', '_description']

    def __init__(self, labels: List[str], properties: Dict[str, str], description: Optional[str] = None, title: Optional[str] = None) -> None:
        self._properties = properties
        self._labels = labels
        self._title = title
        self._description = description

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description


ingest_api_NewVideoIngestDestination.__name__ = "NewVideoIngestDestination"
ingest_api_NewVideoIngestDestination.__qualname__ = "NewVideoIngestDestination"
ingest_api_NewVideoIngestDestination.__module__ = "nominal_api.ingest_api"


class ingest_api_NoTimestampManifest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'starting_timestamp': ConjureFieldDefinition('startingTimestamp', ingest_api_UtcTimestamp),
            'scale_parameter': ConjureFieldDefinition('scaleParameter', OptionalTypeWrapper[ingest_api_ScaleParameter])
        }

    __slots__: List[str] = ['_starting_timestamp', '_scale_parameter']

    def __init__(self, starting_timestamp: "ingest_api_UtcTimestamp", scale_parameter: Optional["ingest_api_ScaleParameter"] = None) -> None:
        self._starting_timestamp = starting_timestamp
        self._scale_parameter = scale_parameter

    @builtins.property
    def starting_timestamp(self) -> "ingest_api_UtcTimestamp":
        return self._starting_timestamp

    @builtins.property
    def scale_parameter(self) -> Optional["ingest_api_ScaleParameter"]:
        """
        A field that specifies that the frame rate of the video does not match the frame rate of the camera | i.e. a slowed down or sped up video. Can specify either the camera frame rate or the absolute end time.
        """
        return self._scale_parameter


ingest_api_NoTimestampManifest.__name__ = "NoTimestampManifest"
ingest_api_NoTimestampManifest.__qualname__ = "NoTimestampManifest"
ingest_api_NoTimestampManifest.__module__ = "nominal_api.ingest_api"


class ingest_api_Part(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'part_number': ConjureFieldDefinition('partNumber', int),
            'etag': ConjureFieldDefinition('etag', str)
        }

    __slots__: List[str] = ['_part_number', '_etag']

    def __init__(self, etag: str, part_number: int) -> None:
        self._part_number = part_number
        self._etag = etag

    @builtins.property
    def part_number(self) -> int:
        return self._part_number

    @builtins.property
    def etag(self) -> str:
        return self._etag


ingest_api_Part.__name__ = "Part"
ingest_api_Part.__qualname__ = "Part"
ingest_api_Part.__module__ = "nominal_api.ingest_api"


class ingest_api_PartWithSize(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'part_number': ConjureFieldDefinition('partNumber', int),
            'etag': ConjureFieldDefinition('etag', str),
            'size': ConjureFieldDefinition('size', int)
        }

    __slots__: List[str] = ['_part_number', '_etag', '_size']

    def __init__(self, etag: str, part_number: int, size: int) -> None:
        self._part_number = part_number
        self._etag = etag
        self._size = size

    @builtins.property
    def part_number(self) -> int:
        return self._part_number

    @builtins.property
    def etag(self) -> str:
        return self._etag

    @builtins.property
    def size(self) -> int:
        return self._size


ingest_api_PartWithSize.__name__ = "PartWithSize"
ingest_api_PartWithSize.__qualname__ = "PartWithSize"
ingest_api_PartWithSize.__module__ = "nominal_api.ingest_api"


class ingest_api_RelativeTimestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'time_unit': ConjureFieldDefinition('timeUnit', api_TimeUnit),
            'offset': ConjureFieldDefinition('offset', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_time_unit', '_offset']

    def __init__(self, time_unit: "api_TimeUnit", offset: Optional[str] = None) -> None:
        self._time_unit = time_unit
        self._offset = offset

    @builtins.property
    def time_unit(self) -> "api_TimeUnit":
        return self._time_unit

    @builtins.property
    def offset(self) -> Optional[str]:
        """
        Starting timestamp to use when indexing the file. This field is required when uploading a new file to an existing dataset.
        """
        return self._offset


ingest_api_RelativeTimestamp.__name__ = "RelativeTimestamp"
ingest_api_RelativeTimestamp.__qualname__ = "RelativeTimestamp"
ingest_api_RelativeTimestamp.__module__ = "nominal_api.ingest_api"


class ingest_api_S3IngestSource(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'path': ConjureFieldDefinition('path', str)
        }

    __slots__: List[str] = ['_path']

    def __init__(self, path: str) -> None:
        self._path = path

    @builtins.property
    def path(self) -> str:
        return self._path


ingest_api_S3IngestSource.__name__ = "S3IngestSource"
ingest_api_S3IngestSource.__qualname__ = "S3IngestSource"
ingest_api_S3IngestSource.__module__ = "nominal_api.ingest_api"


class ingest_api_ScaleParameter(ConjureUnionType):
    _true_frame_rate: Optional[float] = None
    _ending_timestamp: Optional["ingest_api_UtcTimestamp"] = None
    _scale_factor: Optional[float] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'true_frame_rate': ConjureFieldDefinition('trueFrameRate', float),
            'ending_timestamp': ConjureFieldDefinition('endingTimestamp', ingest_api_UtcTimestamp),
            'scale_factor': ConjureFieldDefinition('scaleFactor', float)
        }

    def __init__(
            self,
            true_frame_rate: Optional[float] = None,
            ending_timestamp: Optional["ingest_api_UtcTimestamp"] = None,
            scale_factor: Optional[float] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (true_frame_rate is not None) + (ending_timestamp is not None) + (scale_factor is not None) != 1:
                raise ValueError('a union must contain a single member')

            if true_frame_rate is not None:
                self._true_frame_rate = true_frame_rate
                self._type = 'trueFrameRate'
            if ending_timestamp is not None:
                self._ending_timestamp = ending_timestamp
                self._type = 'endingTimestamp'
            if scale_factor is not None:
                self._scale_factor = scale_factor
                self._type = 'scaleFactor'

        elif type_of_union == 'trueFrameRate':
            if true_frame_rate is None:
                raise ValueError('a union value must not be None')
            self._true_frame_rate = true_frame_rate
            self._type = 'trueFrameRate'
        elif type_of_union == 'endingTimestamp':
            if ending_timestamp is None:
                raise ValueError('a union value must not be None')
            self._ending_timestamp = ending_timestamp
            self._type = 'endingTimestamp'
        elif type_of_union == 'scaleFactor':
            if scale_factor is None:
                raise ValueError('a union value must not be None')
            self._scale_factor = scale_factor
            self._type = 'scaleFactor'

    @builtins.property
    def true_frame_rate(self) -> Optional[float]:
        return self._true_frame_rate

    @builtins.property
    def ending_timestamp(self) -> Optional["ingest_api_UtcTimestamp"]:
        """
        the timestamp corresponding to absolute starting timestamp plus absolute duration of the video.
        """
        return self._ending_timestamp

    @builtins.property
    def scale_factor(self) -> Optional[float]:
        """
        the scale factor can be used to calculate whether media duration differs from a video's | real duration, and if so, the true frame rate of the camera. The video time will thus be scaled | by the ratio of the real duration to media duration, or media frame rate to true frame rate.
        """
        return self._scale_factor

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_ScaleParameterVisitor):
            raise ValueError('{} is not an instance of ingest_api_ScaleParameterVisitor'.format(visitor.__class__.__name__))
        if self._type == 'trueFrameRate' and self.true_frame_rate is not None:
            return visitor._true_frame_rate(self.true_frame_rate)
        if self._type == 'endingTimestamp' and self.ending_timestamp is not None:
            return visitor._ending_timestamp(self.ending_timestamp)
        if self._type == 'scaleFactor' and self.scale_factor is not None:
            return visitor._scale_factor(self.scale_factor)


ingest_api_ScaleParameter.__name__ = "ScaleParameter"
ingest_api_ScaleParameter.__qualname__ = "ScaleParameter"
ingest_api_ScaleParameter.__module__ = "nominal_api.ingest_api"


class ingest_api_ScaleParameterVisitor:

    @abstractmethod
    def _true_frame_rate(self, true_frame_rate: float) -> Any:
        pass

    @abstractmethod
    def _ending_timestamp(self, ending_timestamp: "ingest_api_UtcTimestamp") -> Any:
        pass

    @abstractmethod
    def _scale_factor(self, scale_factor: float) -> Any:
        pass


ingest_api_ScaleParameterVisitor.__name__ = "ScaleParameterVisitor"
ingest_api_ScaleParameterVisitor.__qualname__ = "ScaleParameterVisitor"
ingest_api_ScaleParameterVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_SignPartResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'url': ConjureFieldDefinition('url', str),
            'headers': ConjureFieldDefinition('headers', Dict[str, str])
        }

    __slots__: List[str] = ['_url', '_headers']

    def __init__(self, headers: Dict[str, str], url: str) -> None:
        self._url = url
        self._headers = headers

    @builtins.property
    def url(self) -> str:
        return self._url

    @builtins.property
    def headers(self) -> Dict[str, str]:
        return self._headers


ingest_api_SignPartResponse.__name__ = "SignPartResponse"
ingest_api_SignPartResponse.__qualname__ = "SignPartResponse"
ingest_api_SignPartResponse.__module__ = "nominal_api.ingest_api"


class ingest_api_SkipRowsConfig(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'header_row_index': ConjureFieldDefinition('headerRowIndex', int),
            'data_start_row_index': ConjureFieldDefinition('dataStartRowIndex', int)
        }

    __slots__: List[str] = ['_header_row_index', '_data_start_row_index']

    def __init__(self, data_start_row_index: int, header_row_index: int) -> None:
        self._header_row_index = header_row_index
        self._data_start_row_index = data_start_row_index

    @builtins.property
    def header_row_index(self) -> int:
        return self._header_row_index

    @builtins.property
    def data_start_row_index(self) -> int:
        return self._data_start_row_index


ingest_api_SkipRowsConfig.__name__ = "SkipRowsConfig"
ingest_api_SkipRowsConfig.__qualname__ = "SkipRowsConfig"
ingest_api_SkipRowsConfig.__module__ = "nominal_api.ingest_api"


class ingest_api_SuccessResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



ingest_api_SuccessResult.__name__ = "SuccessResult"
ingest_api_SuccessResult.__qualname__ = "SuccessResult"
ingest_api_SuccessResult.__module__ = "nominal_api.ingest_api"


class ingest_api_TimeOffsetSpec(ConjureUnionType):
    _nanos: Optional["scout_run_api_Duration"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'nanos': ConjureFieldDefinition('nanos', scout_run_api_Duration)
        }

    def __init__(
            self,
            nanos: Optional["scout_run_api_Duration"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (nanos is not None) != 1:
                raise ValueError('a union must contain a single member')

            if nanos is not None:
                self._nanos = nanos
                self._type = 'nanos'

        elif type_of_union == 'nanos':
            if nanos is None:
                raise ValueError('a union value must not be None')
            self._nanos = nanos
            self._type = 'nanos'

    @builtins.property
    def nanos(self) -> Optional["scout_run_api_Duration"]:
        return self._nanos

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_TimeOffsetSpecVisitor):
            raise ValueError('{} is not an instance of ingest_api_TimeOffsetSpecVisitor'.format(visitor.__class__.__name__))
        if self._type == 'nanos' and self.nanos is not None:
            return visitor._nanos(self.nanos)


ingest_api_TimeOffsetSpec.__name__ = "TimeOffsetSpec"
ingest_api_TimeOffsetSpec.__qualname__ = "TimeOffsetSpec"
ingest_api_TimeOffsetSpec.__module__ = "nominal_api.ingest_api"


class ingest_api_TimeOffsetSpecVisitor:

    @abstractmethod
    def _nanos(self, nanos: "scout_run_api_Duration") -> Any:
        pass


ingest_api_TimeOffsetSpecVisitor.__name__ = "TimeOffsetSpecVisitor"
ingest_api_TimeOffsetSpecVisitor.__qualname__ = "TimeOffsetSpecVisitor"
ingest_api_TimeOffsetSpecVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_TimestampManifest(ConjureBeanType):
    """
    The timestamp manifest files will contain a list of absolute timestamps, in nanoseconds, that correspond to 
each frame in a video. Each file should be of type JSON and store a single list, the length of which equals
the number of frames in its corresponding video.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'sources': ConjureFieldDefinition('sources', List[ingest_api_IngestSource])
        }

    __slots__: List[str] = ['_sources']

    def __init__(self, sources: List["ingest_api_IngestSource"]) -> None:
        self._sources = sources

    @builtins.property
    def sources(self) -> List["ingest_api_IngestSource"]:
        return self._sources


ingest_api_TimestampManifest.__name__ = "TimestampManifest"
ingest_api_TimestampManifest.__qualname__ = "TimestampManifest"
ingest_api_TimestampManifest.__module__ = "nominal_api.ingest_api"


class ingest_api_TimestampMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'series_name': ConjureFieldDefinition('seriesName', str),
            'timestamp_type': ConjureFieldDefinition('timestampType', ingest_api_TimestampType)
        }

    __slots__: List[str] = ['_series_name', '_timestamp_type']

    def __init__(self, series_name: str, timestamp_type: "ingest_api_TimestampType") -> None:
        self._series_name = series_name
        self._timestamp_type = timestamp_type

    @builtins.property
    def series_name(self) -> str:
        return self._series_name

    @builtins.property
    def timestamp_type(self) -> "ingest_api_TimestampType":
        return self._timestamp_type


ingest_api_TimestampMetadata.__name__ = "TimestampMetadata"
ingest_api_TimestampMetadata.__qualname__ = "TimestampMetadata"
ingest_api_TimestampMetadata.__module__ = "nominal_api.ingest_api"


class ingest_api_TimestampType(ConjureUnionType):
    _relative: Optional["ingest_api_RelativeTimestamp"] = None
    _absolute: Optional["ingest_api_AbsoluteTimestamp"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'relative': ConjureFieldDefinition('relative', ingest_api_RelativeTimestamp),
            'absolute': ConjureFieldDefinition('absolute', ingest_api_AbsoluteTimestamp)
        }

    def __init__(
            self,
            relative: Optional["ingest_api_RelativeTimestamp"] = None,
            absolute: Optional["ingest_api_AbsoluteTimestamp"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (relative is not None) + (absolute is not None) != 1:
                raise ValueError('a union must contain a single member')

            if relative is not None:
                self._relative = relative
                self._type = 'relative'
            if absolute is not None:
                self._absolute = absolute
                self._type = 'absolute'

        elif type_of_union == 'relative':
            if relative is None:
                raise ValueError('a union value must not be None')
            self._relative = relative
            self._type = 'relative'
        elif type_of_union == 'absolute':
            if absolute is None:
                raise ValueError('a union value must not be None')
            self._absolute = absolute
            self._type = 'absolute'

    @builtins.property
    def relative(self) -> Optional["ingest_api_RelativeTimestamp"]:
        return self._relative

    @builtins.property
    def absolute(self) -> Optional["ingest_api_AbsoluteTimestamp"]:
        return self._absolute

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_TimestampTypeVisitor):
            raise ValueError('{} is not an instance of ingest_api_TimestampTypeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'relative' and self.relative is not None:
            return visitor._relative(self.relative)
        if self._type == 'absolute' and self.absolute is not None:
            return visitor._absolute(self.absolute)


ingest_api_TimestampType.__name__ = "TimestampType"
ingest_api_TimestampType.__qualname__ = "TimestampType"
ingest_api_TimestampType.__module__ = "nominal_api.ingest_api"


class ingest_api_TimestampTypeVisitor:

    @abstractmethod
    def _relative(self, relative: "ingest_api_RelativeTimestamp") -> Any:
        pass

    @abstractmethod
    def _absolute(self, absolute: "ingest_api_AbsoluteTimestamp") -> Any:
        pass


ingest_api_TimestampTypeVisitor.__name__ = "TimestampTypeVisitor"
ingest_api_TimestampTypeVisitor.__qualname__ = "TimestampTypeVisitor"
ingest_api_TimestampTypeVisitor.__module__ = "nominal_api.ingest_api"


class ingest_api_TriggerFileIngest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'source': ConjureFieldDefinition('source', ingest_api_IngestSource),
            'source_metadata': ConjureFieldDefinition('sourceMetadata', ingest_api_IngestSourceMetadata),
            'destination': ConjureFieldDefinition('destination', ingest_api_IngestDestination)
        }

    __slots__: List[str] = ['_source', '_source_metadata', '_destination']

    def __init__(self, destination: "ingest_api_IngestDestination", source: "ingest_api_IngestSource", source_metadata: "ingest_api_IngestSourceMetadata") -> None:
        self._source = source
        self._source_metadata = source_metadata
        self._destination = destination

    @builtins.property
    def source(self) -> "ingest_api_IngestSource":
        """
        Source data for the ingest. Supported file types include:
  * CSV (*.csv)
  * Compressed CSV (*.csv.gz)
  * Parquet (*.parquet)
  * Parquet archives (*.parquet.tar, *.parquet.tar.gz, *.parquet.zip).
      Note that timestamp column must have the same name and format across files.
      Non parquet files will be ignored.
      Each file can contribute to the overall schema, but conflicting types will fail the ingest.
      Conflicting values (same timestamp, column) across files will be de-conflicted based on archive file ordering (first point taken)
        """
        return self._source

    @builtins.property
    def source_metadata(self) -> "ingest_api_IngestSourceMetadata":
        return self._source_metadata

    @builtins.property
    def destination(self) -> "ingest_api_IngestDestination":
        return self._destination


ingest_api_TriggerFileIngest.__name__ = "TriggerFileIngest"
ingest_api_TriggerFileIngest.__qualname__ = "TriggerFileIngest"
ingest_api_TriggerFileIngest.__module__ = "nominal_api.ingest_api"


class ingest_api_TriggerIngest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'source': ConjureFieldDefinition('source', ingest_api_IngestSource),
            'properties': ConjureFieldDefinition('properties', Dict[ingest_api_PropertyName, ingest_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[ingest_api_Label]),
            'dataset_name': ConjureFieldDefinition('datasetName', OptionalTypeWrapper[str]),
            'dataset_description': ConjureFieldDefinition('datasetDescription', OptionalTypeWrapper[str]),
            'timestamp_metadata': ConjureFieldDefinition('timestampMetadata', OptionalTypeWrapper[ingest_api_TimestampMetadata]),
            'channel_config': ConjureFieldDefinition('channelConfig', OptionalTypeWrapper[ingest_api_ChannelConfig])
        }

    __slots__: List[str] = ['_source', '_properties', '_labels', '_dataset_name', '_dataset_description', '_timestamp_metadata', '_channel_config']

    def __init__(self, labels: List[str], properties: Dict[str, str], source: "ingest_api_IngestSource", channel_config: Optional["ingest_api_ChannelConfig"] = None, dataset_description: Optional[str] = None, dataset_name: Optional[str] = None, timestamp_metadata: Optional["ingest_api_TimestampMetadata"] = None) -> None:
        self._source = source
        self._properties = properties
        self._labels = labels
        self._dataset_name = dataset_name
        self._dataset_description = dataset_description
        self._timestamp_metadata = timestamp_metadata
        self._channel_config = channel_config

    @builtins.property
    def source(self) -> "ingest_api_IngestSource":
        return self._source

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def dataset_name(self) -> Optional[str]:
        return self._dataset_name

    @builtins.property
    def dataset_description(self) -> Optional[str]:
        return self._dataset_description

    @builtins.property
    def timestamp_metadata(self) -> Optional["ingest_api_TimestampMetadata"]:
        return self._timestamp_metadata

    @builtins.property
    def channel_config(self) -> Optional["ingest_api_ChannelConfig"]:
        """
        If absent, will default to a channel config that constructs a prefix tree with `.` as the delimiter.
        """
        return self._channel_config


ingest_api_TriggerIngest.__name__ = "TriggerIngest"
ingest_api_TriggerIngest.__qualname__ = "TriggerIngest"
ingest_api_TriggerIngest.__module__ = "nominal_api.ingest_api"


class ingest_api_TriggeredIngest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset_rid': ConjureFieldDefinition('datasetRid', str),
            'dataset_file_id': ConjureFieldDefinition('datasetFileId', OptionalTypeWrapper[str]),
            'async_handle': ConjureFieldDefinition('asyncHandle', OptionalTypeWrapper[ingest_api_AsyncHandle])
        }

    __slots__: List[str] = ['_dataset_rid', '_dataset_file_id', '_async_handle']

    def __init__(self, dataset_rid: str, async_handle: Optional["ingest_api_AsyncHandle"] = None, dataset_file_id: Optional[str] = None) -> None:
        self._dataset_rid = dataset_rid
        self._dataset_file_id = dataset_file_id
        self._async_handle = async_handle

    @builtins.property
    def dataset_rid(self) -> str:
        return self._dataset_rid

    @builtins.property
    def dataset_file_id(self) -> Optional[str]:
        return self._dataset_file_id

    @builtins.property
    def async_handle(self) -> Optional["ingest_api_AsyncHandle"]:
        return self._async_handle


ingest_api_TriggeredIngest.__name__ = "TriggeredIngest"
ingest_api_TriggeredIngest.__qualname__ = "TriggeredIngest"
ingest_api_TriggeredIngest.__module__ = "nominal_api.ingest_api"


class ingest_api_UtcTimestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'seconds_since_epoch': ConjureFieldDefinition('secondsSinceEpoch', int),
            'offset_nanoseconds': ConjureFieldDefinition('offsetNanoseconds', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_seconds_since_epoch', '_offset_nanoseconds']

    def __init__(self, seconds_since_epoch: int, offset_nanoseconds: Optional[int] = None) -> None:
        self._seconds_since_epoch = seconds_since_epoch
        self._offset_nanoseconds = offset_nanoseconds

    @builtins.property
    def seconds_since_epoch(self) -> int:
        return self._seconds_since_epoch

    @builtins.property
    def offset_nanoseconds(self) -> Optional[int]:
        return self._offset_nanoseconds


ingest_api_UtcTimestamp.__name__ = "UtcTimestamp"
ingest_api_UtcTimestamp.__qualname__ = "UtcTimestamp"
ingest_api_UtcTimestamp.__module__ = "nominal_api.ingest_api"


class ingest_api_VideoTimestampManifest(ConjureUnionType):
    _no_manifest: Optional["ingest_api_NoTimestampManifest"] = None
    _timestamp_manifests: Optional["ingest_api_TimestampManifest"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'no_manifest': ConjureFieldDefinition('noManifest', ingest_api_NoTimestampManifest),
            'timestamp_manifests': ConjureFieldDefinition('timestampManifests', ingest_api_TimestampManifest)
        }

    def __init__(
            self,
            no_manifest: Optional["ingest_api_NoTimestampManifest"] = None,
            timestamp_manifests: Optional["ingest_api_TimestampManifest"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (no_manifest is not None) + (timestamp_manifests is not None) != 1:
                raise ValueError('a union must contain a single member')

            if no_manifest is not None:
                self._no_manifest = no_manifest
                self._type = 'noManifest'
            if timestamp_manifests is not None:
                self._timestamp_manifests = timestamp_manifests
                self._type = 'timestampManifests'

        elif type_of_union == 'noManifest':
            if no_manifest is None:
                raise ValueError('a union value must not be None')
            self._no_manifest = no_manifest
            self._type = 'noManifest'
        elif type_of_union == 'timestampManifests':
            if timestamp_manifests is None:
                raise ValueError('a union value must not be None')
            self._timestamp_manifests = timestamp_manifests
            self._type = 'timestampManifests'

    @builtins.property
    def no_manifest(self) -> Optional["ingest_api_NoTimestampManifest"]:
        return self._no_manifest

    @builtins.property
    def timestamp_manifests(self) -> Optional["ingest_api_TimestampManifest"]:
        return self._timestamp_manifests

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, ingest_api_VideoTimestampManifestVisitor):
            raise ValueError('{} is not an instance of ingest_api_VideoTimestampManifestVisitor'.format(visitor.__class__.__name__))
        if self._type == 'noManifest' and self.no_manifest is not None:
            return visitor._no_manifest(self.no_manifest)
        if self._type == 'timestampManifests' and self.timestamp_manifests is not None:
            return visitor._timestamp_manifests(self.timestamp_manifests)


ingest_api_VideoTimestampManifest.__name__ = "VideoTimestampManifest"
ingest_api_VideoTimestampManifest.__qualname__ = "VideoTimestampManifest"
ingest_api_VideoTimestampManifest.__module__ = "nominal_api.ingest_api"


class ingest_api_VideoTimestampManifestVisitor:

    @abstractmethod
    def _no_manifest(self, no_manifest: "ingest_api_NoTimestampManifest") -> Any:
        pass

    @abstractmethod
    def _timestamp_manifests(self, timestamp_manifests: "ingest_api_TimestampManifest") -> Any:
        pass


ingest_api_VideoTimestampManifestVisitor.__name__ = "VideoTimestampManifestVisitor"
ingest_api_VideoTimestampManifestVisitor.__qualname__ = "VideoTimestampManifestVisitor"
ingest_api_VideoTimestampManifestVisitor.__module__ = "nominal_api.ingest_api"


class scout_InternalVersioningService(Service):
    """
    These endpoints are not intended to be used directly by clients, since
they require saving resource-specific state associated with new commits.
    """

    def init_resource_versioning(self, auth_header: str, request: "scout_versioning_api_InitResourceVersioningRequest", resource_rid: str) -> "scout_versioning_api_BranchAndCommit":
        """
        Creates a root commit (no parents) and a "main" branch 
pointing to that commit, for the given resource.
Throws if the resource already has a commit graph.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/versioning/internal/{resourceRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_BranchAndCommit, self._return_none_for_unknown_union_types)

    def save_working_state(self, auth_header: str, branch_name: str, request: "scout_versioning_api_SaveWorkingStateRequest", resource_rid: str) -> "scout_versioning_api_BranchAndCommit":
        """
        Creates a non-permanent commit on the given branch,
Throws if the branch doesn't exist.
Throws if latestCommit is passed and is not the latest commit.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'branchName': branch_name,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/versioning/internal/{resourceRid}/branch/{branchName}/working-state'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_BranchAndCommit, self._return_none_for_unknown_union_types)

    def commit(self, auth_header: str, branch_name: str, request: "scout_versioning_api_CommitRequest", resource_rid: str) -> "scout_versioning_api_BranchAndCommit":
        """
        Creates a new permanent commit on the given branch.
Throws if the branch doesn't exist.
Throws if latestCommit is passed and is not the latest commit.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'branchName': branch_name,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/versioning/internal/{resourceRid}/branch/{branchName}/commit'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_BranchAndCommit, self._return_none_for_unknown_union_types)

    def compact_commits(self, auth_header: str, request: "scout_versioning_api_CompactCommitsRequest", resource_rid: str) -> List[str]:
        """
        Compacts the commit graph for the resource by deleting
working state commits that match the provided strategy.
Persists commits that are pointed to by branches.
Returns the set of commits that were compacted.
Throws if the resource doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/versioning/internal/{resourceRid}/compact-commits'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_versioning_api_CommitId], self._return_none_for_unknown_union_types)


scout_InternalVersioningService.__name__ = "InternalVersioningService"
scout_InternalVersioningService.__qualname__ = "InternalVersioningService"
scout_InternalVersioningService.__module__ = "nominal_api.scout"


class scout_NotebookService(Service):
    """
    NotebookService manages workbooks (formerly known as notebooks).
    """

    def create(self, auth_header: str, request: "scout_notebook_api_CreateNotebookRequest") -> "scout_notebook_api_Notebook":
        """
        Creates a new workbook. The workbook will be associated with the provided run. If the run does not exist, 
a RunNotFound error will be thrown.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/notebook'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_notebook_api_Notebook, self._return_none_for_unknown_union_types)

    def update(self, auth_header: str, request: "scout_notebook_api_UpdateNotebookRequest", rid: str) -> "scout_notebook_api_Notebook":
        """
        Updates the contents of a workbook.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/notebook/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_notebook_api_Notebook, self._return_none_for_unknown_union_types)

    def get(self, auth_header: str, rid: str) -> "scout_notebook_api_Notebook":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v2/notebook/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_notebook_api_Notebook, self._return_none_for_unknown_union_types)

    def batch_get(self, auth_header: str, rids: List[str] = None) -> List["scout_notebook_api_Notebook"]:
        rids = rids if rids is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(rids)

        _path = '/scout/v2/notebook/batch-get'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_notebook_api_Notebook], self._return_none_for_unknown_union_types)

    def batch_get_metadata(self, auth_header: str, rids: List[str] = None) -> List["scout_notebook_api_NotebookMetadataWithRid"]:
        rids = rids if rids is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(rids)

        _path = '/scout/v2/notebook/batch-get-metadata'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_notebook_api_NotebookMetadataWithRid], self._return_none_for_unknown_union_types)

    def update_metadata(self, auth_header: str, request: "scout_notebook_api_UpdateNotebookMetadataRequest", rid: str) -> "scout_notebook_api_NotebookMetadata":
        """
        Updates metadata about a workbook, but not its contents.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/notebook/{rid}/update-metadata'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_notebook_api_NotebookMetadata, self._return_none_for_unknown_union_types)

    def get_all_labels_and_properties(self, auth_header: str) -> "scout_notebook_api_GetAllLabelsAndPropertiesResponse":
        """
        Returns all properties (key value pairs) and labels that have been previously used on workbook. These can
be used to organize workbooks.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/scout/v2/notebook/get-all-labels-properties'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_notebook_api_GetAllLabelsAndPropertiesResponse, self._return_none_for_unknown_union_types)

    def search(self, auth_header: str, request: "scout_notebook_api_SearchNotebooksRequest") -> "scout_notebook_api_SearchNotebooksResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/notebook/search'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_notebook_api_SearchNotebooksResponse, self._return_none_for_unknown_union_types)

    def lock(self, auth_header: str, rid: str) -> None:
        """
        Makes a workbook uneditable.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v2/notebook/{rid}/lock'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def unlock(self, auth_header: str, rid: str) -> None:
        """
        Unlocks a workbook for editing.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v2/notebook/{rid}/unlock'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def archive(self, auth_header: str, rid: str) -> None:
        """
        Archives a workbook, which excludes it from search and hides it from being publicly visible, but does not
permanently delete it. Archived workbooks can be unarchived.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v2/notebook/{rid}/archive'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def unarchive(self, auth_header: str, rid: str) -> None:
        """
        Makes a previously archived workbook searchable.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v2/notebook/{rid}/unarchive'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def delete(self, auth_header: str, rid: str) -> None:
        """
        The workbook will be deleted and is not recoverable. For soft deletion, use archive.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v2/notebook/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return


scout_NotebookService.__name__ = "NotebookService"
scout_NotebookService.__qualname__ = "NotebookService"
scout_NotebookService.__module__ = "nominal_api.scout"


class scout_RunService(Service):
    """
    Runs are collections of channels and metadata from one or more data sources, synchronized over a 
range of real time, which represents a test event or simulation. These API endpoints let you
manage runs in the Nominal app.
    """

    def create_run(self, auth_header: str, details: "scout_run_api_CreateRunRequest") -> "scout_run_api_Run":
        """
        Create a new run in Nominal.

Throws if start is equal to or after end.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(details)

        _path = '/scout/v1/run'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_Run, self._return_none_for_unknown_union_types)

    def update_run(self, auth_header: str, details: "scout_run_api_UpdateRunRequest", rid: str) -> "scout_run_api_Run":
        """
        Updates an existing run based on its RID.

Throws if start is equal to or after end.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(details)

        _path = '/scout/v1/run/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_Run, self._return_none_for_unknown_union_types)

    def add_data_sources_to_run(self, auth_header: str, request: Dict[str, "scout_run_api_CreateRunDataSource"], run_rid: str) -> "scout_run_api_Run":
        """
        Adds datasources to the run in question.

Throws if any of the ref names conflict with existing data sources or each other.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'runRid': run_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/run/{runRid}/data-sources'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_Run, self._return_none_for_unknown_union_types)

    def create_or_update_run(self, auth_header: str, details: "scout_run_api_CreateOrUpdateRunRequest") -> "scout_run_api_Run":
        """
        Updates a run if it exists, otherwise it's created from scratch.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(details)

        _path = '/scout/v1/run/create-or-update'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_Run, self._return_none_for_unknown_union_types)

    def get_run(self, auth_header: str, rid: str) -> "scout_run_api_Run":
        """
        Fetches details about the run in question based on its RID.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/run/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_Run, self._return_none_for_unknown_union_types)

    def get_run_with_data_review_metrics(self, auth_header: str, rid: str) -> "scout_run_api_RunWithDataReviewMetrics":
        """
        Fetches details about the run in question based on its RID, 
including metrics for check and violation review status.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/run/{rid}/with-data-review-metrics'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_RunWithDataReviewMetrics, self._return_none_for_unknown_union_types)

    def get_run_with_data_review_summary(self, auth_header: str, rid: str) -> "scout_run_api_RunWithDataReviewSummary":
        """
        Fetches details about the run in question based on its RID, including a summary of the data review status.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/run/{rid}/with-data-review-summary'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_RunWithDataReviewSummary, self._return_none_for_unknown_union_types)

    def get_run_by_id(self, auth_header: str, get_run_by_id_request: "scout_run_api_GetRunByIdRequest") -> "scout_run_api_Run":
        """
        Fetches a run based on the run number, rather than RID.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(get_run_by_id_request)

        _path = '/scout/v1/run/by-id'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_Run, self._return_none_for_unknown_union_types)

    def get_runs(self, auth_header: str, rids: List[str] = None) -> Dict[str, "scout_run_api_Run"]:
        """
        Fetches a list of run details based on a list of RIDs.
        """
        rids = rids if rids is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(rids)

        _path = '/scout/v1/run/multiple'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), Dict[scout_run_api_RunRid, scout_run_api_Run], self._return_none_for_unknown_union_types)

    def get_runs_by_asset(self, auth_header: str, request: "scout_run_api_GetRunsByAssetRequest") -> "scout_run_api_GetRunsByAssetResponse":
        """
        Fetches the runs with the given asset.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/run/by-asset'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_GetRunsByAssetResponse, self._return_none_for_unknown_union_types)

    def get_all_runs_properties_and_labels(self, auth_header: str) -> "scout_run_api_AllRunsPropertiesAndLabelsResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/scout/v1/all-runs-properties-labels'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_AllRunsPropertiesAndLabelsResponse, self._return_none_for_unknown_union_types)

    def search_runs(self, auth_header: str, list_runs_request: "scout_run_api_SearchRunsRequest") -> "scout_run_api_SearchRunsResponse":
        """
        Searches for runs that match the given ƒilters.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(list_runs_request)

        _path = '/scout/v1/search-runs'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_SearchRunsResponse, self._return_none_for_unknown_union_types)

    def search_runs_with_data_review_metrics(self, auth_header: str, list_runs_request: "scout_run_api_SearchRunsRequest") -> "scout_run_api_SearchRunsWithDataReviewMetricsResponse":
        """
        Searches for runs that match the given filters and 
includes metrics for check and violation review status
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(list_runs_request)

        _path = '/scout/v1/search-runs-with-data-review-metrics'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_SearchRunsWithDataReviewMetricsResponse, self._return_none_for_unknown_union_types)

    def search_runs_with_data_review_summary(self, auth_header: str, list_runs_request: "scout_run_api_SearchRunsRequest") -> "scout_run_api_SearchRunsWithDataReviewSummaryResponse":
        """
        Searches for runs that match the given filters and includes a summary of the data review status.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(list_runs_request)

        _path = '/scout/v1/search-runs-with-data-review-summary'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_SearchRunsWithDataReviewSummaryResponse, self._return_none_for_unknown_union_types)

    def archive_run(self, auth_header: str, rid: str) -> bool:
        """
        Soft-deletes a run. Runs still exist in the database but are no longer visible.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/archive-run/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), bool, self._return_none_for_unknown_union_types)

    def get_data_source_ref_name_and_type_list(self, auth_header: str) -> List["scout_run_api_RefNameAndType"]:
        """
        Returns the list of ref names that are in use across an entire organization.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/scout/v1/data-source-ref-names-and-types'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_run_api_RefNameAndType], self._return_none_for_unknown_union_types)

    def search_channels(self, auth_header: str, request: "scout_run_api_SearchRunChannelsRequest", rid: str) -> "scout_run_api_SearchRunChannelsResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/run/{rid}/search-channels'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_SearchRunChannelsResponse, self._return_none_for_unknown_union_types)

    def get_events(self, auth_header: str, rid: str, next_page_token: Optional[str] = None) -> "scout_run_api_GetEventsForRunPage":
        """
        Fetches a list of events associated with the run.

Returns a maximum page size of 1000.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'nextPageToken': next_page_token,
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/run/{rid}/events'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_run_api_GetEventsForRunPage, self._return_none_for_unknown_union_types)

    def publish_events(self, auth_header: str, request: "scout_run_api_PublishEventsToRunRequest", rid: str) -> None:
        """
        Adds a set of new events to the run.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/run/{rid}/events'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def delete_events_from_run(self, auth_header: str, request: "scout_run_api_DeleteEventsFromRunRequest", rid: str) -> None:
        """
        Deletes events from the run.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/run/{rid}/delete-events'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def update_run_attachment(self, auth_header: str, request: "scout_run_api_UpdateAttachmentsRequest", rid: str) -> None:
        """
        Updates the attachments associated with a run.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/run/{rid}/attachments'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return


scout_RunService.__name__ = "RunService"
scout_RunService.__qualname__ = "RunService"
scout_RunService.__module__ = "nominal_api.scout"


class scout_TemplateService(Service):
    """
    TemplateService manages templates, which are workbooks that
can be re-used across runs. Templates are versioned.
    """

    def create(self, auth_header: str, request: "scout_template_api_CreateTemplateRequest") -> "scout_template_api_Template":
        """
        Creates a new template.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/template'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_template_api_Template, self._return_none_for_unknown_union_types)

    def get(self, auth_header: str, template_rid: str, branch: Optional[str] = None, commit: Optional[str] = None) -> "scout_template_api_Template":
        """
        Must only pass one of (branch, commit). If neither are passed,
the latest commit on the "main" branch is returned.
Throws if the template, branch, or commit doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'branch': branch,
            'commit': commit,
        }

        _path_params: Dict[str, Any] = {
            'templateRid': template_rid,
        }

        _json: Any = None

        _path = '/scout/v1/template/{templateRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_template_api_Template, self._return_none_for_unknown_union_types)

    def save_working_state(self, auth_header: str, request: "scout_template_api_SaveTemplateRequest", template_rid: str, branch: Optional[str] = None) -> "scout_template_api_Template":
        """
        Creates a commit that may be compacted, e.g cleaned up and not exist anymore.
Throws if the template or branch doesn't exist.
Throws if the latest commit doesn't match the provided id.
Throws if you save to an archived template.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'branch': branch,
        }

        _path_params: Dict[str, Any] = {
            'templateRid': template_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/template/{templateRid}/save-working-state'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_template_api_Template, self._return_none_for_unknown_union_types)

    def commit(self, auth_header: str, request: "scout_template_api_CommitTemplateRequest", template_rid: str, branch: Optional[str] = None) -> "scout_template_api_Template":
        """
        Creates a commit with a commit message. 
Throws if the template or branch doesn't exist.
Throws if the latest commit doesn't match the provided id.
Throws if you commit to an archived template.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'branch': branch,
        }

        _path_params: Dict[str, Any] = {
            'templateRid': template_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/template/{templateRid}/commit'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_template_api_Template, self._return_none_for_unknown_union_types)

    def update_metadata(self, auth_header: str, request: "scout_template_api_UpdateMetadataRequest", template_rid: str) -> "scout_template_api_TemplateMetadata":
        """
        Throws if the template doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'templateRid': template_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/template/{templateRid}/metadata'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_template_api_TemplateMetadata, self._return_none_for_unknown_union_types)

    def search_templates(self, auth_header: str, request: "scout_template_api_SearchTemplatesRequest") -> "scout_template_api_SearchTemplatesResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/template/search'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_template_api_SearchTemplatesResponse, self._return_none_for_unknown_union_types)

    def get_all_labels_and_properties(self, auth_header: str) -> "scout_template_api_GetAllLabelsAndPropertiesResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/scout/v1/template/get-all-labels-properties'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_template_api_GetAllLabelsAndPropertiesResponse, self._return_none_for_unknown_union_types)

    def merge_to_main(self, auth_header: str, request: "scout_template_api_MergeToMainRequest", template_rid: str) -> "scout_template_api_Template":
        """
        Merges the given branch to the "main" branch.
Throws if the template or branch doesn't exist.
Throws if the latest commit doesn't match the provided id.
Throws if you merge on an archived template.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'templateRid': template_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/template/{templateRid}/merge-to-main'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_template_api_Template, self._return_none_for_unknown_union_types)


scout_TemplateService.__name__ = "TemplateService"
scout_TemplateService.__qualname__ = "TemplateService"
scout_TemplateService.__module__ = "nominal_api.scout"


class scout_UnitsService(Service):
    """
    The Units Service serves as a comprehensive catalog of the units of measurement supported by scout. Units, by 
default, follow the UCUM convention for representation.
    """

    def get_all_units(self, auth_header: str) -> "scout_units_api_GetUnitsResponse":
        """
        Returns all known units, grouped by the physical property they measure.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/units/v1/units'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_units_api_GetUnitsResponse, self._return_none_for_unknown_union_types)

    def get_unit(self, auth_header: str, unit: str) -> Optional["scout_units_api_Unit"]:
        """
        Returns information for a unit symbol if available. Returns as empty if the provided symbol cannot be parsed.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(unit)

        _path = '/units/v1/units/get-unit'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return None if _response.status_code == 204 else _decoder.decode(_response.json(), OptionalTypeWrapper[scout_units_api_Unit], self._return_none_for_unknown_union_types)

    def get_commensurable_units(self, auth_header: str, unit: str) -> List["scout_units_api_Unit"]:
        """
        Returns the set of units that can be converted to and from the given unit.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(unit)

        _path = '/units/v1/units/commensurable-units'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_units_api_Unit], self._return_none_for_unknown_union_types)


scout_UnitsService.__name__ = "UnitsService"
scout_UnitsService.__qualname__ = "UnitsService"
scout_UnitsService.__module__ = "nominal_api.scout"


class scout_VersioningService(Service):
    """
    This is the external-facing portion of VersioningService which
gives clients access to functionality that doesn't create new
commits. The creation of new commits should be done via the
resource-specific services.
    """

    def create_branch(self, auth_header: str, request: "scout_versioning_api_CreateBranchRequest", resource_rid: str) -> "scout_versioning_api_Branch":
        """
        Creates a mutable pointer to the provided commit.
"Saves"/"commits" can be performed on this pointer.
Throws if the name is already used as a commit
pointer for this resource.
Throws if the provided commit doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/versioning/{resourceRid}/branch'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_Branch, self._return_none_for_unknown_union_types)

    def create_tag(self, auth_header: str, request: "scout_versioning_api_CreateTagRequest", resource_rid: str) -> "scout_versioning_api_Tag":
        """
        Creates an immutable pointer to the provided commit.
Throws if the name is already used as a commit
pointer for this resource.
Throws if the provided commit doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/versioning/{resourceRid}/tag'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_Tag, self._return_none_for_unknown_union_types)

    def get_commit(self, auth_header: str, commit_id: str, resource_rid: str) -> "scout_versioning_api_Commit":
        """
        Throws if the commit doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'commitId': commit_id,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/commit/{commitId}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_Commit, self._return_none_for_unknown_union_types)

    def batch_get_commits(self, auth_header: str, resource_and_commit_ids: List["scout_versioning_api_ResourceAndCommitId"] = None) -> List["scout_versioning_api_Commit"]:
        """
        Filters out resources that are not authorized.
        """
        resource_and_commit_ids = resource_and_commit_ids if resource_and_commit_ids is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(resource_and_commit_ids)

        _path = '/scout/v1/versioning/commit/batch-get'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_versioning_api_Commit], self._return_none_for_unknown_union_types)

    def get_commit_by_branch(self, auth_header: str, branch_name: str, resource_rid: str) -> "scout_versioning_api_Commit":
        """
        Returns the commit pointed to by the branch.
Throws if the branch doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'branchName': branch_name,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/branch/{branchName}/commit'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_Commit, self._return_none_for_unknown_union_types)

    def get_commit_by_tag(self, auth_header: str, resource_rid: str, tag_name: str) -> "scout_versioning_api_Commit":
        """
        Returns the commit pointed to by the tag.
Throws if the tag doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'tagName': tag_name,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/tag/{tagName}/commit'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_Commit, self._return_none_for_unknown_union_types)

    def get_least_common_ancestor(self, auth_header: str, request: "scout_versioning_api_GetLeastCommonAncestorRequest", resource_rid: str) -> str:
        """
        Returns the least common ancestor of the two commits.
Throws if either commit doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/versioning/{resourceRid}/commit/least-common-ancestor'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_CommitId, self._return_none_for_unknown_union_types)

    def get_commit_history(self, auth_header: str, commit_id: str, resource_rid: str, next_page_token: Optional[str] = None, page_size: Optional[int] = None) -> "scout_versioning_api_CommitHistory":
        """
        Returns the commit history sorted by creation time descending.
Excludes working state commits.
Throws if the commit doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'pageSize': page_size,
            'nextPageToken': next_page_token,
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'commitId': commit_id,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/commit/{commitId}/history'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_CommitHistory, self._return_none_for_unknown_union_types)

    def persist_commits(self, auth_header: str, request: List["scout_versioning_api_ResourceAndCommitId"] = None) -> None:
        """
        Persists the commits so that they are not compacted.
This operation is atomic - either all commits are persisted
or none are (in the case of an error).
        """
        request = request if request is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/versioning/commit/persist'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def get_branch(self, auth_header: str, branch_name: str, resource_rid: str) -> "scout_versioning_api_Branch":
        """
        Throws if the branch doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'branchName': branch_name,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/branch/{branchName}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_Branch, self._return_none_for_unknown_union_types)

    def get_branches(self, auth_header: str, resource_rid: str) -> List["scout_versioning_api_Branch"]:
        """
        Returns all branches for the resource in order of
most recently updated.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/branch'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_versioning_api_Branch], self._return_none_for_unknown_union_types)

    def batch_get_branches(self, auth_header: str, resource_and_branches: List["scout_versioning_api_ResourceAndBranchName"] = None) -> List["scout_versioning_api_Branch"]:
        """
        Omits branches that are not authorized.
        """
        resource_and_branches = resource_and_branches if resource_and_branches is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(resource_and_branches)

        _path = '/scout/v1/versioning/branch/batch-get'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_versioning_api_Branch], self._return_none_for_unknown_union_types)

    def get_tag(self, auth_header: str, resource_rid: str, tag_name: str) -> "scout_versioning_api_Tag":
        """
        Throws if the tag doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'tagName': tag_name,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/tag/{tagName}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_versioning_api_Tag, self._return_none_for_unknown_union_types)

    def batch_get_tags(self, auth_header: str, resource_and_commits: List["scout_versioning_api_ResourceAndCommitId"] = None) -> List["scout_versioning_api_Tag"]:
        """
        Omits tags that are not authorized.
        """
        resource_and_commits = resource_and_commits if resource_and_commits is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(resource_and_commits)

        _path = '/scout/v1/versioning/tag/batch-get'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_versioning_api_Tag], self._return_none_for_unknown_union_types)

    def get_tags_by_resource(self, auth_header: str, resource_rid: str) -> List["scout_versioning_api_Tag"]:
        """
        Returns all tags for the resource in order of
most recently created.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/tag'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_versioning_api_Tag], self._return_none_for_unknown_union_types)

    def delete_branch(self, auth_header: str, branch_name: str, resource_rid: str) -> None:
        """
        Deletes the branch pointer.
Throws if the branch doesn't exist.
Throws if you attempt to delete the "main" branch.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'branchName': branch_name,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/branch/{branchName}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def delete_branches(self, auth_header: str, resource_and_branches: List["scout_versioning_api_ResourceAndBranchName"] = None) -> None:
        """
        Deletes the branch pointers.
Throws if any resource or branch is non-existent
or unauthorized.
Throws if any attempt is made to delete "main".
        """
        resource_and_branches = resource_and_branches if resource_and_branches is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(resource_and_branches)

        _path = '/scout/v1/versioning/branch/batch-delete'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def delete_tag(self, auth_header: str, resource_rid: str, tag_name: str) -> None:
        """
        Deletes the tag pointer.
Throws if the tag doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'resourceRid': resource_rid,
            'tagName': tag_name,
        }

        _json: Any = None

        _path = '/scout/v1/versioning/{resourceRid}/tag/{tagName}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return


scout_VersioningService.__name__ = "VersioningService"
scout_VersioningService.__qualname__ = "VersioningService"
scout_VersioningService.__module__ = "nominal_api.scout"


class scout_api_ChannelLocator(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_source_ref': ConjureFieldDefinition('dataSourceRef', scout_api_DataSourceRefName),
            'channel': ConjureFieldDefinition('channel', api_Channel),
            'tags': ConjureFieldDefinition('tags', Dict[api_TagName, api_TagValue])
        }

    __slots__: List[str] = ['_data_source_ref', '_channel', '_tags']

    def __init__(self, channel: str, data_source_ref: str, tags: Dict[str, str]) -> None:
        self._data_source_ref = data_source_ref
        self._channel = channel
        self._tags = tags

    @builtins.property
    def data_source_ref(self) -> str:
        return self._data_source_ref

    @builtins.property
    def channel(self) -> str:
        return self._channel

    @builtins.property
    def tags(self) -> Dict[str, str]:
        return self._tags


scout_api_ChannelLocator.__name__ = "ChannelLocator"
scout_api_ChannelLocator.__qualname__ = "ChannelLocator"
scout_api_ChannelLocator.__module__ = "nominal_api.scout_api"


class scout_asset_api_AddDataScopesToAssetRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_scopes': ConjureFieldDefinition('dataScopes', List[scout_asset_api_CreateAssetDataScope])
        }

    __slots__: List[str] = ['_data_scopes']

    def __init__(self, data_scopes: List["scout_asset_api_CreateAssetDataScope"]) -> None:
        self._data_scopes = data_scopes

    @builtins.property
    def data_scopes(self) -> List["scout_asset_api_CreateAssetDataScope"]:
        """
        The data scopes to add to the asset.

Throws if any existing data scopes have data scope names that conflict with the data scope names
in the request.
        """
        return self._data_scopes


scout_asset_api_AddDataScopesToAssetRequest.__name__ = "AddDataScopesToAssetRequest"
scout_asset_api_AddDataScopesToAssetRequest.__qualname__ = "AddDataScopesToAssetRequest"
scout_asset_api_AddDataScopesToAssetRequest.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_Asset(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', scout_rids_api_AssetRid),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'properties': ConjureFieldDefinition('properties', Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[scout_run_api_Label]),
            'links': ConjureFieldDefinition('links', List[scout_run_api_Link]),
            'data_scopes': ConjureFieldDefinition('dataScopes', List[scout_asset_api_AssetDataScope]),
            'created_by': ConjureFieldDefinition('createdBy', OptionalTypeWrapper[str]),
            'created_at': ConjureFieldDefinition('createdAt', str),
            'updated_at': ConjureFieldDefinition('updatedAt', str),
            'attachments': ConjureFieldDefinition('attachments', List[api_rids_AttachmentRid]),
            'type': ConjureFieldDefinition('type', OptionalTypeWrapper[scout_rids_api_TypeRid]),
            'is_staged': ConjureFieldDefinition('isStaged', bool)
        }

    __slots__: List[str] = ['_rid', '_title', '_description', '_properties', '_labels', '_links', '_data_scopes', '_created_by', '_created_at', '_updated_at', '_attachments', '_type', '_is_staged']

    def __init__(self, attachments: List[str], created_at: str, data_scopes: List["scout_asset_api_AssetDataScope"], is_staged: bool, labels: List[str], links: List["scout_run_api_Link"], properties: Dict[str, str], rid: str, title: str, updated_at: str, created_by: Optional[str] = None, description: Optional[str] = None, type: Optional[str] = None) -> None:
        self._rid = rid
        self._title = title
        self._description = description
        self._properties = properties
        self._labels = labels
        self._links = links
        self._data_scopes = data_scopes
        self._created_by = created_by
        self._created_at = created_at
        self._updated_at = updated_at
        self._attachments = attachments
        self._type = type
        self._is_staged = is_staged

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        """
        Labels associated with the asset. These labels do not have a time dimension.
To associate labels with a range of time, create a time range on the asset with labels.
        """
        return self._labels

    @builtins.property
    def links(self) -> List["scout_run_api_Link"]:
        """
        Links associated with the asset. These links do not have a time dimension.
To associate links with a range of time, create a time range on the asset with links.
        """
        return self._links

    @builtins.property
    def data_scopes(self) -> List["scout_asset_api_AssetDataScope"]:
        """
        The data scopes associated with the asset.
        """
        return self._data_scopes

    @builtins.property
    def created_by(self) -> Optional[str]:
        return self._created_by

    @builtins.property
    def created_at(self) -> str:
        return self._created_at

    @builtins.property
    def updated_at(self) -> str:
        return self._updated_at

    @builtins.property
    def attachments(self) -> List[str]:
        return self._attachments

    @builtins.property
    def type(self) -> Optional[str]:
        return self._type

    @builtins.property
    def is_staged(self) -> bool:
        """
        Auto created assets are considered staged by default.
        """
        return self._is_staged


scout_asset_api_Asset.__name__ = "Asset"
scout_asset_api_Asset.__qualname__ = "Asset"
scout_asset_api_Asset.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_AssetDataScope(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_scope_name': ConjureFieldDefinition('dataScopeName', scout_asset_api_DataScopeName),
            'data_source': ConjureFieldDefinition('dataSource', scout_run_api_DataSource),
            'offset': ConjureFieldDefinition('offset', OptionalTypeWrapper[scout_run_api_Duration]),
            'timestamp_type': ConjureFieldDefinition('timestampType', scout_run_api_WeakTimestampType),
            'series_tags': ConjureFieldDefinition('seriesTags', Dict[scout_asset_api_SeriesTagName, scout_asset_api_SeriesTagValue])
        }

    __slots__: List[str] = ['_data_scope_name', '_data_source', '_offset', '_timestamp_type', '_series_tags']

    def __init__(self, data_scope_name: str, data_source: "scout_run_api_DataSource", series_tags: Dict[str, str], timestamp_type: "scout_run_api_WeakTimestampType", offset: Optional["scout_run_api_Duration"] = None) -> None:
        self._data_scope_name = data_scope_name
        self._data_source = data_source
        self._offset = offset
        self._timestamp_type = timestamp_type
        self._series_tags = series_tags

    @builtins.property
    def data_scope_name(self) -> str:
        """
        The name of the data scope. The name is guaranteed to be be unique within the context of an asset.
        """
        return self._data_scope_name

    @builtins.property
    def data_source(self) -> "scout_run_api_DataSource":
        return self._data_source

    @builtins.property
    def offset(self) -> Optional["scout_run_api_Duration"]:
        return self._offset

    @builtins.property
    def timestamp_type(self) -> "scout_run_api_WeakTimestampType":
        return self._timestamp_type

    @builtins.property
    def series_tags(self) -> Dict[str, str]:
        """
        Filters the data source to series matching these tag values. The filtered set of series should be
the ones that belong to the asset.
        """
        return self._series_tags


scout_asset_api_AssetDataScope.__name__ = "AssetDataScope"
scout_asset_api_AssetDataScope.__qualname__ = "AssetDataScope"
scout_asset_api_AssetDataScope.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_ChannelMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', scout_asset_api_Channel),
            'data_source': ConjureFieldDefinition('dataSource', scout_run_api_DataSource),
            'unit': ConjureFieldDefinition('unit', OptionalTypeWrapper[scout_run_api_Unit]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'data_type': ConjureFieldDefinition('dataType', OptionalTypeWrapper[api_SeriesDataType])
        }

    __slots__: List[str] = ['_name', '_data_source', '_unit', '_description', '_data_type']

    def __init__(self, data_source: "scout_run_api_DataSource", name: str, data_type: Optional["api_SeriesDataType"] = None, description: Optional[str] = None, unit: Optional["scout_run_api_Unit"] = None) -> None:
        self._name = name
        self._data_source = data_source
        self._unit = unit
        self._description = description
        self._data_type = data_type

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def data_source(self) -> "scout_run_api_DataSource":
        return self._data_source

    @builtins.property
    def unit(self) -> Optional["scout_run_api_Unit"]:
        return self._unit

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def data_type(self) -> Optional["api_SeriesDataType"]:
        return self._data_type


scout_asset_api_ChannelMetadata.__name__ = "ChannelMetadata"
scout_asset_api_ChannelMetadata.__qualname__ = "ChannelMetadata"
scout_asset_api_ChannelMetadata.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_CreateAssetDataScope(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_scope_name': ConjureFieldDefinition('dataScopeName', scout_asset_api_DataScopeName),
            'data_source': ConjureFieldDefinition('dataSource', scout_run_api_DataSource),
            'offset': ConjureFieldDefinition('offset', OptionalTypeWrapper[scout_run_api_Duration]),
            'series_tags': ConjureFieldDefinition('seriesTags', Dict[scout_asset_api_SeriesTagName, scout_asset_api_SeriesTagValue])
        }

    __slots__: List[str] = ['_data_scope_name', '_data_source', '_offset', '_series_tags']

    def __init__(self, data_scope_name: str, data_source: "scout_run_api_DataSource", series_tags: Dict[str, str], offset: Optional["scout_run_api_Duration"] = None) -> None:
        self._data_scope_name = data_scope_name
        self._data_source = data_source
        self._offset = offset
        self._series_tags = series_tags

    @builtins.property
    def data_scope_name(self) -> str:
        """
        The name of the data scope. The name is guaranteed to be be unique within the context of an asset.
        """
        return self._data_scope_name

    @builtins.property
    def data_source(self) -> "scout_run_api_DataSource":
        return self._data_source

    @builtins.property
    def offset(self) -> Optional["scout_run_api_Duration"]:
        return self._offset

    @builtins.property
    def series_tags(self) -> Dict[str, str]:
        """
        Filters the data source to series matching these tag values. The filtered set of series should be
the ones that belong to the asset.
        """
        return self._series_tags


scout_asset_api_CreateAssetDataScope.__name__ = "CreateAssetDataScope"
scout_asset_api_CreateAssetDataScope.__qualname__ = "CreateAssetDataScope"
scout_asset_api_CreateAssetDataScope.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_CreateAssetRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'properties': ConjureFieldDefinition('properties', Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[scout_run_api_Label]),
            'links': ConjureFieldDefinition('links', List[scout_run_api_Link]),
            'data_scopes': ConjureFieldDefinition('dataScopes', List[scout_asset_api_CreateAssetDataScope]),
            'attachments': ConjureFieldDefinition('attachments', List[api_rids_AttachmentRid]),
            'type': ConjureFieldDefinition('type', OptionalTypeWrapper[scout_rids_api_TypeRid])
        }

    __slots__: List[str] = ['_title', '_description', '_properties', '_labels', '_links', '_data_scopes', '_attachments', '_type']

    def __init__(self, attachments: List[str], data_scopes: List["scout_asset_api_CreateAssetDataScope"], labels: List[str], links: List["scout_run_api_Link"], properties: Dict[str, str], title: str, description: Optional[str] = None, type: Optional[str] = None) -> None:
        self._title = title
        self._description = description
        self._properties = properties
        self._labels = labels
        self._links = links
        self._data_scopes = data_scopes
        self._attachments = attachments
        self._type = type

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def links(self) -> List["scout_run_api_Link"]:
        return self._links

    @builtins.property
    def data_scopes(self) -> List["scout_asset_api_CreateAssetDataScope"]:
        """
        The data scopes associated with the asset.
        """
        return self._data_scopes

    @builtins.property
    def attachments(self) -> List[str]:
        return self._attachments

    @builtins.property
    def type(self) -> Optional[str]:
        return self._type


scout_asset_api_CreateAssetRequest.__name__ = "CreateAssetRequest"
scout_asset_api_CreateAssetRequest.__qualname__ = "CreateAssetRequest"
scout_asset_api_CreateAssetRequest.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_CreateTypeRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', str),
            'property_configs': ConjureFieldDefinition('propertyConfigs', Dict[scout_run_api_PropertyName, scout_asset_api_PropertyConfig]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'icon_name': ConjureFieldDefinition('iconName', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_name', '_property_configs', '_description', '_icon_name']

    def __init__(self, name: str, property_configs: Dict[str, "scout_asset_api_PropertyConfig"], description: Optional[str] = None, icon_name: Optional[str] = None) -> None:
        self._name = name
        self._property_configs = property_configs
        self._description = description
        self._icon_name = icon_name

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def property_configs(self) -> Dict[str, "scout_asset_api_PropertyConfig"]:
        return self._property_configs

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def icon_name(self) -> Optional[str]:
        return self._icon_name


scout_asset_api_CreateTypeRequest.__name__ = "CreateTypeRequest"
scout_asset_api_CreateTypeRequest.__qualname__ = "CreateTypeRequest"
scout_asset_api_CreateTypeRequest.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_PropertyConfig(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'is_required': ConjureFieldDefinition('isRequired', bool),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_is_required', '_description']

    def __init__(self, is_required: bool, description: Optional[str] = None) -> None:
        self._is_required = is_required
        self._description = description

    @builtins.property
    def is_required(self) -> bool:
        return self._is_required

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description


scout_asset_api_PropertyConfig.__name__ = "PropertyConfig"
scout_asset_api_PropertyConfig.__qualname__ = "PropertyConfig"
scout_asset_api_PropertyConfig.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_RemoveType(ConjureBeanType):
    """
    The request to remove the type from the asset.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_asset_api_RemoveType.__name__ = "RemoveType"
scout_asset_api_RemoveType.__qualname__ = "RemoveType"
scout_asset_api_RemoveType.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchAssetChannelsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'search_text': ConjureFieldDefinition('searchText', str),
            'data_scope_name_filter': ConjureFieldDefinition('dataScopeNameFilter', OptionalTypeWrapper[List[scout_asset_api_DataScopeName]]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token]),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_search_text', '_data_scope_name_filter', '_next_page_token', '_page_size']

    def __init__(self, search_text: str, data_scope_name_filter: Optional[List[str]] = None, next_page_token: Optional[str] = None, page_size: Optional[int] = None) -> None:
        self._search_text = search_text
        self._data_scope_name_filter = data_scope_name_filter
        self._next_page_token = next_page_token
        self._page_size = page_size

    @builtins.property
    def search_text(self) -> str:
        return self._search_text

    @builtins.property
    def data_scope_name_filter(self) -> Optional[List[str]]:
        """
        If not empty, will filter to channels from the selected data scope names.
        """
        return self._data_scope_name_filter

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Defaults to 1000. Will throw if larger than 10000. Default pageSize is 100.
        """
        return self._page_size


scout_asset_api_SearchAssetChannelsRequest.__name__ = "SearchAssetChannelsRequest"
scout_asset_api_SearchAssetChannelsRequest.__qualname__ = "SearchAssetChannelsRequest"
scout_asset_api_SearchAssetChannelsRequest.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchAssetChannelsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[scout_asset_api_ChannelMetadata]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token])
        }

    __slots__: List[str] = ['_results', '_next_page_token']

    def __init__(self, results: List["scout_asset_api_ChannelMetadata"], next_page_token: Optional[str] = None) -> None:
        self._results = results
        self._next_page_token = next_page_token

    @builtins.property
    def results(self) -> List["scout_asset_api_ChannelMetadata"]:
        return self._results

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


scout_asset_api_SearchAssetChannelsResponse.__name__ = "SearchAssetChannelsResponse"
scout_asset_api_SearchAssetChannelsResponse.__qualname__ = "SearchAssetChannelsResponse"
scout_asset_api_SearchAssetChannelsResponse.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchAssetsQuery(ConjureUnionType):
    _search_text: Optional[str] = None
    _label: Optional[str] = None
    _property: Optional["scout_run_api_Property"] = None
    _type_rid: Optional[str] = None
    _and_: Optional[List["scout_asset_api_SearchAssetsQuery"]] = None
    _or_: Optional[List["scout_asset_api_SearchAssetsQuery"]] = None
    _is_staged: Optional[bool] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'search_text': ConjureFieldDefinition('searchText', str),
            'label': ConjureFieldDefinition('label', scout_run_api_Label),
            'property': ConjureFieldDefinition('property', scout_run_api_Property),
            'type_rid': ConjureFieldDefinition('typeRid', scout_rids_api_TypeRid),
            'and_': ConjureFieldDefinition('and', List[scout_asset_api_SearchAssetsQuery]),
            'or_': ConjureFieldDefinition('or', List[scout_asset_api_SearchAssetsQuery]),
            'is_staged': ConjureFieldDefinition('isStaged', bool)
        }

    def __init__(
            self,
            search_text: Optional[str] = None,
            label: Optional[str] = None,
            property: Optional["scout_run_api_Property"] = None,
            type_rid: Optional[str] = None,
            and_: Optional[List["scout_asset_api_SearchAssetsQuery"]] = None,
            or_: Optional[List["scout_asset_api_SearchAssetsQuery"]] = None,
            is_staged: Optional[bool] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (search_text is not None) + (label is not None) + (property is not None) + (type_rid is not None) + (and_ is not None) + (or_ is not None) + (is_staged is not None) != 1:
                raise ValueError('a union must contain a single member')

            if search_text is not None:
                self._search_text = search_text
                self._type = 'searchText'
            if label is not None:
                self._label = label
                self._type = 'label'
            if property is not None:
                self._property = property
                self._type = 'property'
            if type_rid is not None:
                self._type_rid = type_rid
                self._type = 'typeRid'
            if and_ is not None:
                self._and_ = and_
                self._type = 'and'
            if or_ is not None:
                self._or_ = or_
                self._type = 'or'
            if is_staged is not None:
                self._is_staged = is_staged
                self._type = 'isStaged'

        elif type_of_union == 'searchText':
            if search_text is None:
                raise ValueError('a union value must not be None')
            self._search_text = search_text
            self._type = 'searchText'
        elif type_of_union == 'label':
            if label is None:
                raise ValueError('a union value must not be None')
            self._label = label
            self._type = 'label'
        elif type_of_union == 'property':
            if property is None:
                raise ValueError('a union value must not be None')
            self._property = property
            self._type = 'property'
        elif type_of_union == 'typeRid':
            if type_rid is None:
                raise ValueError('a union value must not be None')
            self._type_rid = type_rid
            self._type = 'typeRid'
        elif type_of_union == 'and':
            if and_ is None:
                raise ValueError('a union value must not be None')
            self._and_ = and_
            self._type = 'and'
        elif type_of_union == 'or':
            if or_ is None:
                raise ValueError('a union value must not be None')
            self._or_ = or_
            self._type = 'or'
        elif type_of_union == 'isStaged':
            if is_staged is None:
                raise ValueError('a union value must not be None')
            self._is_staged = is_staged
            self._type = 'isStaged'

    @builtins.property
    def search_text(self) -> Optional[str]:
        return self._search_text

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def property(self) -> Optional["scout_run_api_Property"]:
        return self._property

    @builtins.property
    def type_rid(self) -> Optional[str]:
        return self._type_rid

    @builtins.property
    def and_(self) -> Optional[List["scout_asset_api_SearchAssetsQuery"]]:
        return self._and_

    @builtins.property
    def or_(self) -> Optional[List["scout_asset_api_SearchAssetsQuery"]]:
        return self._or_

    @builtins.property
    def is_staged(self) -> Optional[bool]:
        return self._is_staged

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_asset_api_SearchAssetsQueryVisitor):
            raise ValueError('{} is not an instance of scout_asset_api_SearchAssetsQueryVisitor'.format(visitor.__class__.__name__))
        if self._type == 'searchText' and self.search_text is not None:
            return visitor._search_text(self.search_text)
        if self._type == 'label' and self.label is not None:
            return visitor._label(self.label)
        if self._type == 'property' and self.property is not None:
            return visitor._property(self.property)
        if self._type == 'typeRid' and self.type_rid is not None:
            return visitor._type_rid(self.type_rid)
        if self._type == 'and' and self.and_ is not None:
            return visitor._and(self.and_)
        if self._type == 'or' and self.or_ is not None:
            return visitor._or(self.or_)
        if self._type == 'isStaged' and self.is_staged is not None:
            return visitor._is_staged(self.is_staged)


scout_asset_api_SearchAssetsQuery.__name__ = "SearchAssetsQuery"
scout_asset_api_SearchAssetsQuery.__qualname__ = "SearchAssetsQuery"
scout_asset_api_SearchAssetsQuery.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchAssetsQueryVisitor:

    @abstractmethod
    def _search_text(self, search_text: str) -> Any:
        pass

    @abstractmethod
    def _label(self, label: str) -> Any:
        pass

    @abstractmethod
    def _property(self, property: "scout_run_api_Property") -> Any:
        pass

    @abstractmethod
    def _type_rid(self, type_rid: str) -> Any:
        pass

    @abstractmethod
    def _and(self, and_: List["scout_asset_api_SearchAssetsQuery"]) -> Any:
        pass

    @abstractmethod
    def _or(self, or_: List["scout_asset_api_SearchAssetsQuery"]) -> Any:
        pass

    @abstractmethod
    def _is_staged(self, is_staged: bool) -> Any:
        pass


scout_asset_api_SearchAssetsQueryVisitor.__name__ = "SearchAssetsQueryVisitor"
scout_asset_api_SearchAssetsQueryVisitor.__qualname__ = "SearchAssetsQueryVisitor"
scout_asset_api_SearchAssetsQueryVisitor.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchAssetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'sort': ConjureFieldDefinition('sort', scout_asset_api_SortOptions),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token]),
            'query': ConjureFieldDefinition('query', scout_asset_api_SearchAssetsQuery),
            'archived_statuses': ConjureFieldDefinition('archivedStatuses', OptionalTypeWrapper[List[scout_rids_api_ArchivedStatus]])
        }

    __slots__: List[str] = ['_sort', '_page_size', '_next_page_token', '_query', '_archived_statuses']

    def __init__(self, query: "scout_asset_api_SearchAssetsQuery", sort: "scout_asset_api_SortOptions", archived_statuses: Optional[List["scout_rids_api_ArchivedStatus"]] = None, next_page_token: Optional[str] = None, page_size: Optional[int] = None) -> None:
        self._sort = sort
        self._page_size = page_size
        self._next_page_token = next_page_token
        self._query = query
        self._archived_statuses = archived_statuses

    @builtins.property
    def sort(self) -> "scout_asset_api_SortOptions":
        return self._sort

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Page sizes greater than 10_000 will be rejected. Default pageSize is 100.
        """
        return self._page_size

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token

    @builtins.property
    def query(self) -> "scout_asset_api_SearchAssetsQuery":
        return self._query

    @builtins.property
    def archived_statuses(self) -> Optional[List["scout_rids_api_ArchivedStatus"]]:
        """
        Default search status is NOT_ARCHIVED if none are provided. Allows for including archived assets in search.
        """
        return self._archived_statuses


scout_asset_api_SearchAssetsRequest.__name__ = "SearchAssetsRequest"
scout_asset_api_SearchAssetsRequest.__qualname__ = "SearchAssetsRequest"
scout_asset_api_SearchAssetsRequest.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchAssetsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[scout_asset_api_Asset]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token])
        }

    __slots__: List[str] = ['_results', '_next_page_token']

    def __init__(self, results: List["scout_asset_api_Asset"], next_page_token: Optional[str] = None) -> None:
        self._results = results
        self._next_page_token = next_page_token

    @builtins.property
    def results(self) -> List["scout_asset_api_Asset"]:
        return self._results

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


scout_asset_api_SearchAssetsResponse.__name__ = "SearchAssetsResponse"
scout_asset_api_SearchAssetsResponse.__qualname__ = "SearchAssetsResponse"
scout_asset_api_SearchAssetsResponse.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchTypesQuery(ConjureUnionType):
    _search_text: Optional[str] = None
    _property: Optional[str] = None
    _and_: Optional[List["scout_asset_api_SearchTypesQuery"]] = None
    _or_: Optional[List["scout_asset_api_SearchTypesQuery"]] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'search_text': ConjureFieldDefinition('searchText', str),
            'property': ConjureFieldDefinition('property', scout_run_api_PropertyName),
            'and_': ConjureFieldDefinition('and', List[scout_asset_api_SearchTypesQuery]),
            'or_': ConjureFieldDefinition('or', List[scout_asset_api_SearchTypesQuery])
        }

    def __init__(
            self,
            search_text: Optional[str] = None,
            property: Optional[str] = None,
            and_: Optional[List["scout_asset_api_SearchTypesQuery"]] = None,
            or_: Optional[List["scout_asset_api_SearchTypesQuery"]] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (search_text is not None) + (property is not None) + (and_ is not None) + (or_ is not None) != 1:
                raise ValueError('a union must contain a single member')

            if search_text is not None:
                self._search_text = search_text
                self._type = 'searchText'
            if property is not None:
                self._property = property
                self._type = 'property'
            if and_ is not None:
                self._and_ = and_
                self._type = 'and'
            if or_ is not None:
                self._or_ = or_
                self._type = 'or'

        elif type_of_union == 'searchText':
            if search_text is None:
                raise ValueError('a union value must not be None')
            self._search_text = search_text
            self._type = 'searchText'
        elif type_of_union == 'property':
            if property is None:
                raise ValueError('a union value must not be None')
            self._property = property
            self._type = 'property'
        elif type_of_union == 'and':
            if and_ is None:
                raise ValueError('a union value must not be None')
            self._and_ = and_
            self._type = 'and'
        elif type_of_union == 'or':
            if or_ is None:
                raise ValueError('a union value must not be None')
            self._or_ = or_
            self._type = 'or'

    @builtins.property
    def search_text(self) -> Optional[str]:
        return self._search_text

    @builtins.property
    def property(self) -> Optional[str]:
        return self._property

    @builtins.property
    def and_(self) -> Optional[List["scout_asset_api_SearchTypesQuery"]]:
        return self._and_

    @builtins.property
    def or_(self) -> Optional[List["scout_asset_api_SearchTypesQuery"]]:
        return self._or_

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_asset_api_SearchTypesQueryVisitor):
            raise ValueError('{} is not an instance of scout_asset_api_SearchTypesQueryVisitor'.format(visitor.__class__.__name__))
        if self._type == 'searchText' and self.search_text is not None:
            return visitor._search_text(self.search_text)
        if self._type == 'property' and self.property is not None:
            return visitor._property(self.property)
        if self._type == 'and' and self.and_ is not None:
            return visitor._and(self.and_)
        if self._type == 'or' and self.or_ is not None:
            return visitor._or(self.or_)


scout_asset_api_SearchTypesQuery.__name__ = "SearchTypesQuery"
scout_asset_api_SearchTypesQuery.__qualname__ = "SearchTypesQuery"
scout_asset_api_SearchTypesQuery.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchTypesQueryVisitor:

    @abstractmethod
    def _search_text(self, search_text: str) -> Any:
        pass

    @abstractmethod
    def _property(self, property: str) -> Any:
        pass

    @abstractmethod
    def _and(self, and_: List["scout_asset_api_SearchTypesQuery"]) -> Any:
        pass

    @abstractmethod
    def _or(self, or_: List["scout_asset_api_SearchTypesQuery"]) -> Any:
        pass


scout_asset_api_SearchTypesQueryVisitor.__name__ = "SearchTypesQueryVisitor"
scout_asset_api_SearchTypesQueryVisitor.__qualname__ = "SearchTypesQueryVisitor"
scout_asset_api_SearchTypesQueryVisitor.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchTypesRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'sort': ConjureFieldDefinition('sort', scout_asset_api_SortOptions),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token]),
            'query': ConjureFieldDefinition('query', scout_asset_api_SearchTypesQuery),
            'archived_statuses': ConjureFieldDefinition('archivedStatuses', OptionalTypeWrapper[List[scout_rids_api_ArchivedStatus]])
        }

    __slots__: List[str] = ['_sort', '_page_size', '_next_page_token', '_query', '_archived_statuses']

    def __init__(self, query: "scout_asset_api_SearchTypesQuery", sort: "scout_asset_api_SortOptions", archived_statuses: Optional[List["scout_rids_api_ArchivedStatus"]] = None, next_page_token: Optional[str] = None, page_size: Optional[int] = None) -> None:
        self._sort = sort
        self._page_size = page_size
        self._next_page_token = next_page_token
        self._query = query
        self._archived_statuses = archived_statuses

    @builtins.property
    def sort(self) -> "scout_asset_api_SortOptions":
        return self._sort

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Page sizes greater than 10_000 will be rejected. Default pageSize is 100.
        """
        return self._page_size

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token

    @builtins.property
    def query(self) -> "scout_asset_api_SearchTypesQuery":
        return self._query

    @builtins.property
    def archived_statuses(self) -> Optional[List["scout_rids_api_ArchivedStatus"]]:
        """
        Default search status is NOT_ARCHIVED if none are provided. Allows for including archived assets in search.
        """
        return self._archived_statuses


scout_asset_api_SearchTypesRequest.__name__ = "SearchTypesRequest"
scout_asset_api_SearchTypesRequest.__qualname__ = "SearchTypesRequest"
scout_asset_api_SearchTypesRequest.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SearchTypesResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[scout_asset_api_Type]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token])
        }

    __slots__: List[str] = ['_results', '_next_page_token']

    def __init__(self, results: List["scout_asset_api_Type"], next_page_token: Optional[str] = None) -> None:
        self._results = results
        self._next_page_token = next_page_token

    @builtins.property
    def results(self) -> List["scout_asset_api_Type"]:
        return self._results

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


scout_asset_api_SearchTypesResponse.__name__ = "SearchTypesResponse"
scout_asset_api_SearchTypesResponse.__qualname__ = "SearchTypesResponse"
scout_asset_api_SearchTypesResponse.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SortField(ConjureEnumType):

    CREATED_AT = 'CREATED_AT'
    '''CREATED_AT'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_asset_api_SortField.__name__ = "SortField"
scout_asset_api_SortField.__qualname__ = "SortField"
scout_asset_api_SortField.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_SortOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'is_descending': ConjureFieldDefinition('isDescending', bool),
            'field': ConjureFieldDefinition('field', scout_asset_api_SortField)
        }

    __slots__: List[str] = ['_is_descending', '_field']

    def __init__(self, field: "scout_asset_api_SortField", is_descending: bool) -> None:
        self._is_descending = is_descending
        self._field = field

    @builtins.property
    def is_descending(self) -> bool:
        return self._is_descending

    @builtins.property
    def field(self) -> "scout_asset_api_SortField":
        return self._field


scout_asset_api_SortOptions.__name__ = "SortOptions"
scout_asset_api_SortOptions.__qualname__ = "SortOptions"
scout_asset_api_SortOptions.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_Type(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', scout_rids_api_TypeRid),
            'name': ConjureFieldDefinition('name', str),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'property_configs': ConjureFieldDefinition('propertyConfigs', Dict[scout_run_api_PropertyName, scout_asset_api_PropertyConfig]),
            'created_at': ConjureFieldDefinition('createdAt', str),
            'icon_name': ConjureFieldDefinition('iconName', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_rid', '_name', '_description', '_property_configs', '_created_at', '_icon_name']

    def __init__(self, created_at: str, name: str, property_configs: Dict[str, "scout_asset_api_PropertyConfig"], rid: str, description: Optional[str] = None, icon_name: Optional[str] = None) -> None:
        self._rid = rid
        self._name = name
        self._description = description
        self._property_configs = property_configs
        self._created_at = created_at
        self._icon_name = icon_name

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def property_configs(self) -> Dict[str, "scout_asset_api_PropertyConfig"]:
        return self._property_configs

    @builtins.property
    def created_at(self) -> str:
        return self._created_at

    @builtins.property
    def icon_name(self) -> Optional[str]:
        """
        The name of the icon to display for the type. This name maps to a Lucide icon in the frontend.
        """
        return self._icon_name


scout_asset_api_Type.__name__ = "Type"
scout_asset_api_Type.__qualname__ = "Type"
scout_asset_api_Type.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_UpdateAssetRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'properties': ConjureFieldDefinition('properties', OptionalTypeWrapper[Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue]]),
            'labels': ConjureFieldDefinition('labels', OptionalTypeWrapper[List[scout_run_api_Label]]),
            'links': ConjureFieldDefinition('links', OptionalTypeWrapper[List[scout_run_api_Link]]),
            'data_scopes': ConjureFieldDefinition('dataScopes', OptionalTypeWrapper[List[scout_asset_api_CreateAssetDataScope]]),
            'type': ConjureFieldDefinition('type', OptionalTypeWrapper[scout_asset_api_UpdateOrRemoveAssetType]),
            'is_staged': ConjureFieldDefinition('isStaged', OptionalTypeWrapper[bool])
        }

    __slots__: List[str] = ['_title', '_description', '_properties', '_labels', '_links', '_data_scopes', '_type', '_is_staged']

    def __init__(self, data_scopes: Optional[List["scout_asset_api_CreateAssetDataScope"]] = None, description: Optional[str] = None, is_staged: Optional[bool] = None, labels: Optional[List[str]] = None, links: Optional[List["scout_run_api_Link"]] = None, properties: Optional[Dict[str, str]] = None, title: Optional[str] = None, type: Optional["scout_asset_api_UpdateOrRemoveAssetType"] = None) -> None:
        self._title = title
        self._description = description
        self._properties = properties
        self._labels = labels
        self._links = links
        self._data_scopes = data_scopes
        self._type = type
        self._is_staged = is_staged

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def properties(self) -> Optional[Dict[str, str]]:
        return self._properties

    @builtins.property
    def labels(self) -> Optional[List[str]]:
        return self._labels

    @builtins.property
    def links(self) -> Optional[List["scout_run_api_Link"]]:
        return self._links

    @builtins.property
    def data_scopes(self) -> Optional[List["scout_asset_api_CreateAssetDataScope"]]:
        """
        The data scopes for the asset. This will replace all existing data scopes with the scopes specified.
        """
        return self._data_scopes

    @builtins.property
    def type(self) -> Optional["scout_asset_api_UpdateOrRemoveAssetType"]:
        return self._type

    @builtins.property
    def is_staged(self) -> Optional[bool]:
        return self._is_staged


scout_asset_api_UpdateAssetRequest.__name__ = "UpdateAssetRequest"
scout_asset_api_UpdateAssetRequest.__qualname__ = "UpdateAssetRequest"
scout_asset_api_UpdateAssetRequest.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_UpdateAttachmentsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'attachments_to_add': ConjureFieldDefinition('attachmentsToAdd', List[api_rids_AttachmentRid]),
            'attachments_to_remove': ConjureFieldDefinition('attachmentsToRemove', List[api_rids_AttachmentRid])
        }

    __slots__: List[str] = ['_attachments_to_add', '_attachments_to_remove']

    def __init__(self, attachments_to_add: List[str], attachments_to_remove: List[str]) -> None:
        self._attachments_to_add = attachments_to_add
        self._attachments_to_remove = attachments_to_remove

    @builtins.property
    def attachments_to_add(self) -> List[str]:
        return self._attachments_to_add

    @builtins.property
    def attachments_to_remove(self) -> List[str]:
        return self._attachments_to_remove


scout_asset_api_UpdateAttachmentsRequest.__name__ = "UpdateAttachmentsRequest"
scout_asset_api_UpdateAttachmentsRequest.__qualname__ = "UpdateAttachmentsRequest"
scout_asset_api_UpdateAttachmentsRequest.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_UpdateOrRemoveAssetType(ConjureUnionType):
    """The request to update the type of the asset. The request will replace the existing type with the type
specified in the request if a typeRID is provided. Otherwise, the type will be removed from the asset."""
    _type_rid: Optional[str] = None
    _remove_type: Optional["scout_asset_api_RemoveType"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'type_rid': ConjureFieldDefinition('typeRid', scout_rids_api_TypeRid),
            'remove_type': ConjureFieldDefinition('removeType', scout_asset_api_RemoveType)
        }

    def __init__(
            self,
            type_rid: Optional[str] = None,
            remove_type: Optional["scout_asset_api_RemoveType"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (type_rid is not None) + (remove_type is not None) != 1:
                raise ValueError('a union must contain a single member')

            if type_rid is not None:
                self._type_rid = type_rid
                self._type = 'typeRid'
            if remove_type is not None:
                self._remove_type = remove_type
                self._type = 'removeType'

        elif type_of_union == 'typeRid':
            if type_rid is None:
                raise ValueError('a union value must not be None')
            self._type_rid = type_rid
            self._type = 'typeRid'
        elif type_of_union == 'removeType':
            if remove_type is None:
                raise ValueError('a union value must not be None')
            self._remove_type = remove_type
            self._type = 'removeType'

    @builtins.property
    def type_rid(self) -> Optional[str]:
        return self._type_rid

    @builtins.property
    def remove_type(self) -> Optional["scout_asset_api_RemoveType"]:
        return self._remove_type

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_asset_api_UpdateOrRemoveAssetTypeVisitor):
            raise ValueError('{} is not an instance of scout_asset_api_UpdateOrRemoveAssetTypeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'typeRid' and self.type_rid is not None:
            return visitor._type_rid(self.type_rid)
        if self._type == 'removeType' and self.remove_type is not None:
            return visitor._remove_type(self.remove_type)


scout_asset_api_UpdateOrRemoveAssetType.__name__ = "UpdateOrRemoveAssetType"
scout_asset_api_UpdateOrRemoveAssetType.__qualname__ = "UpdateOrRemoveAssetType"
scout_asset_api_UpdateOrRemoveAssetType.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_UpdateOrRemoveAssetTypeVisitor:

    @abstractmethod
    def _type_rid(self, type_rid: str) -> Any:
        pass

    @abstractmethod
    def _remove_type(self, remove_type: "scout_asset_api_RemoveType") -> Any:
        pass


scout_asset_api_UpdateOrRemoveAssetTypeVisitor.__name__ = "UpdateOrRemoveAssetTypeVisitor"
scout_asset_api_UpdateOrRemoveAssetTypeVisitor.__qualname__ = "UpdateOrRemoveAssetTypeVisitor"
scout_asset_api_UpdateOrRemoveAssetTypeVisitor.__module__ = "nominal_api.scout_asset_api"


class scout_asset_api_UpdateTypeRequest(ConjureBeanType):
    """
    The request to update a type. The request will replace all existing properties with the properties
specified in the request.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', OptionalTypeWrapper[str]),
            'property_configs': ConjureFieldDefinition('propertyConfigs', OptionalTypeWrapper[Dict[scout_run_api_PropertyName, scout_asset_api_PropertyConfig]]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'icon_name': ConjureFieldDefinition('iconName', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_name', '_property_configs', '_description', '_icon_name']

    def __init__(self, description: Optional[str] = None, icon_name: Optional[str] = None, name: Optional[str] = None, property_configs: Optional[Dict[str, "scout_asset_api_PropertyConfig"]] = None) -> None:
        self._name = name
        self._property_configs = property_configs
        self._description = description
        self._icon_name = icon_name

    @builtins.property
    def name(self) -> Optional[str]:
        return self._name

    @builtins.property
    def property_configs(self) -> Optional[Dict[str, "scout_asset_api_PropertyConfig"]]:
        return self._property_configs

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def icon_name(self) -> Optional[str]:
        return self._icon_name


scout_asset_api_UpdateTypeRequest.__name__ = "UpdateTypeRequest"
scout_asset_api_UpdateTypeRequest.__qualname__ = "UpdateTypeRequest"
scout_asset_api_UpdateTypeRequest.__module__ = "nominal_api.scout_asset_api"


class scout_assets_AssetService(Service):
    """
    An asset is a physical entity within an organization, such as a vehicle or ground station. Assets are comprised
of some metadata about the data, as well as a set of data scopes that define the data belonging to the asset. API
endpoints allow for CRUD operations on asset objects.
    """

    def create_asset(self, auth_header: str, details: "scout_asset_api_CreateAssetRequest") -> "scout_asset_api_Asset":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(details)

        _path = '/scout/v1/asset'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_asset_api_Asset, self._return_none_for_unknown_union_types)

    def update_asset(self, auth_header: str, details: "scout_asset_api_UpdateAssetRequest", rid: str) -> "scout_asset_api_Asset":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(details)

        _path = '/scout/v1/asset/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_asset_api_Asset, self._return_none_for_unknown_union_types)

    def add_data_scopes_to_asset(self, asset_rid: str, auth_header: str, request: "scout_asset_api_AddDataScopesToAssetRequest") -> "scout_asset_api_Asset":
        """
        Adds data scopes to an asset.

Throws if the asset already has data scopes with data scope names matching those in the request.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'assetRid': asset_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/asset/{assetRid}/data-sources'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_asset_api_Asset, self._return_none_for_unknown_union_types)

    def get_assets(self, auth_header: str, rids: List[str] = None) -> Dict[str, "scout_asset_api_Asset"]:
        rids = rids if rids is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(rids)

        _path = '/scout/v1/asset/multiple'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), Dict[scout_rids_api_AssetRid, scout_asset_api_Asset], self._return_none_for_unknown_union_types)

    def archive(self, auth_header: str, rid: str) -> None:

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/archive/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def unarchive(self, auth_header: str, rid: str) -> None:

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/unarchive/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def search_assets(self, auth_header: str, search_assets_request: "scout_asset_api_SearchAssetsRequest") -> "scout_asset_api_SearchAssetsResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(search_assets_request)

        _path = '/scout/v1/search-assets'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_asset_api_SearchAssetsResponse, self._return_none_for_unknown_union_types)

    def search_types(self, auth_header: str, search_types_request: "scout_asset_api_SearchTypesRequest") -> "scout_asset_api_SearchTypesResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(search_types_request)

        _path = '/scout/v1/search-types'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_asset_api_SearchTypesResponse, self._return_none_for_unknown_union_types)

    def update_asset_attachments(self, auth_header: str, request: "scout_asset_api_UpdateAttachmentsRequest", rid: str) -> None:
        """
        Update the attachments associated with an asset.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/asset/{rid}/attachments'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def create_type(self, auth_header: str, request: "scout_asset_api_CreateTypeRequest") -> "scout_asset_api_Type":
        """
        Creates a new type.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/type'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_asset_api_Type, self._return_none_for_unknown_union_types)

    def update_type(self, auth_header: str, request: "scout_asset_api_UpdateTypeRequest", type_rid: str) -> "scout_asset_api_Type":
        """
        Updates a type. Will throw unless all assets that reference the type pass the updated type check.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'typeRid': type_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/type/{typeRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_asset_api_Type, self._return_none_for_unknown_union_types)

    def get_types(self, auth_header: str, rids: List[str] = None) -> Dict[str, "scout_asset_api_Type"]:
        rids = rids if rids is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(rids)

        _path = '/scout/v1/type/multiple'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), Dict[scout_rids_api_TypeRid, scout_asset_api_Type], self._return_none_for_unknown_union_types)

    def delete_type(self, auth_header: str, rid: str) -> None:
        """
        Deletes a type. The type must not be referenced by any assets.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/type/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def archive_type(self, auth_header: str, rid: str) -> None:

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/type/archive/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def unarchive_type(self, auth_header: str, rid: str) -> None:

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/type/unarchive/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return


scout_assets_AssetService.__name__ = "AssetService"
scout_assets_AssetService.__qualname__ = "AssetService"
scout_assets_AssetService.__module__ = "nominal_api.scout_assets"


class scout_catalog_AbsoluteTimestamp(ConjureUnionType):
    _iso8601: Optional["scout_catalog_Iso8601Timestamp"] = None
    _epoch_of_time_unit: Optional["scout_catalog_EpochTimestamp"] = None
    _custom_format: Optional["scout_catalog_CustomTimestamp"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'iso8601': ConjureFieldDefinition('iso8601', scout_catalog_Iso8601Timestamp),
            'epoch_of_time_unit': ConjureFieldDefinition('epochOfTimeUnit', scout_catalog_EpochTimestamp),
            'custom_format': ConjureFieldDefinition('customFormat', scout_catalog_CustomTimestamp)
        }

    def __init__(
            self,
            iso8601: Optional["scout_catalog_Iso8601Timestamp"] = None,
            epoch_of_time_unit: Optional["scout_catalog_EpochTimestamp"] = None,
            custom_format: Optional["scout_catalog_CustomTimestamp"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (iso8601 is not None) + (epoch_of_time_unit is not None) + (custom_format is not None) != 1:
                raise ValueError('a union must contain a single member')

            if iso8601 is not None:
                self._iso8601 = iso8601
                self._type = 'iso8601'
            if epoch_of_time_unit is not None:
                self._epoch_of_time_unit = epoch_of_time_unit
                self._type = 'epochOfTimeUnit'
            if custom_format is not None:
                self._custom_format = custom_format
                self._type = 'customFormat'

        elif type_of_union == 'iso8601':
            if iso8601 is None:
                raise ValueError('a union value must not be None')
            self._iso8601 = iso8601
            self._type = 'iso8601'
        elif type_of_union == 'epochOfTimeUnit':
            if epoch_of_time_unit is None:
                raise ValueError('a union value must not be None')
            self._epoch_of_time_unit = epoch_of_time_unit
            self._type = 'epochOfTimeUnit'
        elif type_of_union == 'customFormat':
            if custom_format is None:
                raise ValueError('a union value must not be None')
            self._custom_format = custom_format
            self._type = 'customFormat'

    @builtins.property
    def iso8601(self) -> Optional["scout_catalog_Iso8601Timestamp"]:
        return self._iso8601

    @builtins.property
    def epoch_of_time_unit(self) -> Optional["scout_catalog_EpochTimestamp"]:
        return self._epoch_of_time_unit

    @builtins.property
    def custom_format(self) -> Optional["scout_catalog_CustomTimestamp"]:
        return self._custom_format

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_catalog_AbsoluteTimestampVisitor):
            raise ValueError('{} is not an instance of scout_catalog_AbsoluteTimestampVisitor'.format(visitor.__class__.__name__))
        if self._type == 'iso8601' and self.iso8601 is not None:
            return visitor._iso8601(self.iso8601)
        if self._type == 'epochOfTimeUnit' and self.epoch_of_time_unit is not None:
            return visitor._epoch_of_time_unit(self.epoch_of_time_unit)
        if self._type == 'customFormat' and self.custom_format is not None:
            return visitor._custom_format(self.custom_format)


scout_catalog_AbsoluteTimestamp.__name__ = "AbsoluteTimestamp"
scout_catalog_AbsoluteTimestamp.__qualname__ = "AbsoluteTimestamp"
scout_catalog_AbsoluteTimestamp.__module__ = "nominal_api.scout_catalog"


class scout_catalog_AbsoluteTimestampVisitor:

    @abstractmethod
    def _iso8601(self, iso8601: "scout_catalog_Iso8601Timestamp") -> Any:
        pass

    @abstractmethod
    def _epoch_of_time_unit(self, epoch_of_time_unit: "scout_catalog_EpochTimestamp") -> Any:
        pass

    @abstractmethod
    def _custom_format(self, custom_format: "scout_catalog_CustomTimestamp") -> Any:
        pass


scout_catalog_AbsoluteTimestampVisitor.__name__ = "AbsoluteTimestampVisitor"
scout_catalog_AbsoluteTimestampVisitor.__qualname__ = "AbsoluteTimestampVisitor"
scout_catalog_AbsoluteTimestampVisitor.__module__ = "nominal_api.scout_catalog"


class scout_catalog_AddFileToDataset(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'handle': ConjureFieldDefinition('handle', scout_catalog_Handle),
            'timestamp_metadata': ConjureFieldDefinition('timestampMetadata', OptionalTypeWrapper[scout_catalog_TimestampMetadata])
        }

    __slots__: List[str] = ['_handle', '_timestamp_metadata']

    def __init__(self, handle: "scout_catalog_Handle", timestamp_metadata: Optional["scout_catalog_TimestampMetadata"] = None) -> None:
        self._handle = handle
        self._timestamp_metadata = timestamp_metadata

    @builtins.property
    def handle(self) -> "scout_catalog_Handle":
        return self._handle

    @builtins.property
    def timestamp_metadata(self) -> Optional["scout_catalog_TimestampMetadata"]:
        return self._timestamp_metadata


scout_catalog_AddFileToDataset.__name__ = "AddFileToDataset"
scout_catalog_AddFileToDataset.__qualname__ = "AddFileToDataset"
scout_catalog_AddFileToDataset.__module__ = "nominal_api.scout_catalog"


class scout_catalog_AllPropertiesAndLabelsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'properties': ConjureFieldDefinition('properties', Dict[datasource_PropertyName, List[datasource_PropertyValue]]),
            'labels': ConjureFieldDefinition('labels', List[datasource_Label])
        }

    __slots__: List[str] = ['_properties', '_labels']

    def __init__(self, labels: List[str], properties: Dict[str, List[str]]) -> None:
        self._properties = properties
        self._labels = labels

    @builtins.property
    def properties(self) -> Dict[str, List[str]]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels


scout_catalog_AllPropertiesAndLabelsResponse.__name__ = "AllPropertiesAndLabelsResponse"
scout_catalog_AllPropertiesAndLabelsResponse.__qualname__ = "AllPropertiesAndLabelsResponse"
scout_catalog_AllPropertiesAndLabelsResponse.__module__ = "nominal_api.scout_catalog"


class scout_catalog_Bounds(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'start': ConjureFieldDefinition('start', api_Timestamp),
            'end': ConjureFieldDefinition('end', api_Timestamp),
            'type': ConjureFieldDefinition('type', datasource_TimestampType)
        }

    __slots__: List[str] = ['_start', '_end', '_type']

    def __init__(self, end: "api_Timestamp", start: "api_Timestamp", type: "datasource_TimestampType") -> None:
        self._start = start
        self._end = end
        self._type = type

    @builtins.property
    def start(self) -> "api_Timestamp":
        return self._start

    @builtins.property
    def end(self) -> "api_Timestamp":
        return self._end

    @builtins.property
    def type(self) -> "datasource_TimestampType":
        return self._type


scout_catalog_Bounds.__name__ = "Bounds"
scout_catalog_Bounds.__qualname__ = "Bounds"
scout_catalog_Bounds.__module__ = "nominal_api.scout_catalog"


class scout_catalog_CatalogService(Service):
    """
    The Catalog Service provides the ability to query for information about Datasets that are stored in
the Nominal platform. A Dataset is the Nominal representation of data that has been uploaded to Nominal via
a file, primarily CSV.
    """

    def get_enriched_dataset(self, auth_header: str, dataset_uuid: str) -> "scout_catalog_EnrichedDataset":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetUuid': dataset_uuid,
        }

        _json: Any = None

        _path = '/catalog/v1/datasets/{datasetUuid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_EnrichedDataset, self._return_none_for_unknown_union_types)

    def get_enriched_datasets(self, auth_header: str, get_datasets_request: "scout_catalog_GetDatasetsRequest") -> List["scout_catalog_EnrichedDataset"]:

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(get_datasets_request)

        _path = '/catalog/v1/datasets/multiple'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_catalog_EnrichedDataset], self._return_none_for_unknown_union_types)

    def get_dataset(self, auth_header: str, dataset_uuid: str) -> "scout_catalog_Dataset":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetUuid': dataset_uuid,
        }

        _json: Any = None

        _path = '/catalog/v1/datasets-simple/{datasetUuid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_Dataset, self._return_none_for_unknown_union_types)

    def get_datasets(self, auth_header: str, get_datasets_request: "scout_catalog_GetDatasetsRequest") -> List["scout_catalog_Dataset"]:

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(get_datasets_request)

        _path = '/catalog/v1/datasets-simple/multiple'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_catalog_Dataset], self._return_none_for_unknown_union_types)

    def get_dataset_file(self, auth_header: str, dataset_rid: str, file_id: str) -> "scout_catalog_DatasetFile":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
            'fileId': file_id,
        }

        _json: Any = None

        _path = '/catalog/v1/dataset/{datasetRid}/file/{fileId}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_DatasetFile, self._return_none_for_unknown_union_types)

    def search_datasets_by_text(self, auth_header: str, request: "scout_catalog_SearchDatasetsByTextRequest") -> "scout_catalog_SearchDatasetsByTextResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/catalog/v1/search-datasets'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_SearchDatasetsByTextResponse, self._return_none_for_unknown_union_types)

    def search_datasets(self, auth_header: str, request: "scout_catalog_SearchDatasetsRequest") -> "scout_catalog_SearchDatasetsResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/catalog/v1/search-datasets-v2'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_SearchDatasetsResponse, self._return_none_for_unknown_union_types)

    def update_dataset_ingest_status_v2(self, auth_header: str, details: "scout_catalog_UpdateIngestStatusV2") -> "scout_catalog_IngestStatusV2":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(details)

        _path = '/catalog/v1/datasets/ingest-status-v2'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_IngestStatusV2, self._return_none_for_unknown_union_types)

    def get_ingest_progress_v2(self, auth_header: str, dataset_rid: str) -> "scout_catalog_IngestProgressV2":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
        }

        _json: Any = None

        _path = '/catalog/v1/datasets/{datasetRid}/ingest-progress-v2'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_IngestProgressV2, self._return_none_for_unknown_union_types)

    def get_handle_for_dataset(self, auth_header: str, dataset: str) -> Optional["scout_catalog_Handle"]:

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'dataset': dataset,
        }

        _json: Any = None

        _path = '/catalog/v1/datasets/{dataset}/handle'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return None if _response.status_code == 204 else _decoder.decode(_response.json(), OptionalTypeWrapper[scout_catalog_Handle], self._return_none_for_unknown_union_types)

    def get_series_details(self, auth_header: str, get_series_details: "scout_catalog_GetSeriesDetailsRequest") -> "scout_catalog_SeriesDetails":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(get_series_details)

        _path = '/catalog/v1/series/details'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_SeriesDetails, self._return_none_for_unknown_union_types)

    def create_dataset(self, auth_header: str, details: "scout_catalog_CreateDataset") -> "scout_catalog_EnrichedDataset":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(details)

        _path = '/catalog/v1/datasets'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_EnrichedDataset, self._return_none_for_unknown_union_types)

    def create_or_update_dataset(self, auth_header: str, details: "scout_catalog_CreateDataset") -> "scout_catalog_EnrichedDataset":
        """
        Creates a dataset if the s3 path does not exist, otherwise updates the dataset
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(details)

        _path = '/catalog/v1/datasets/create-or-update'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_EnrichedDataset, self._return_none_for_unknown_union_types)

    def add_file_to_dataset(self, auth_header: str, dataset_rid: str, request: "scout_catalog_AddFileToDataset") -> "scout_catalog_DatasetFile":
        """
        Adds a single file to an existing dataset.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/catalog/v1/datasets/{datasetRid}/add-file'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_DatasetFile, self._return_none_for_unknown_union_types)

    def list_dataset_files(self, auth_header: str, dataset_rid: str, next_page_token: Optional[str] = None) -> "scout_catalog_DatasetFilesPage":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'nextPageToken': next_page_token,
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
        }

        _json: Any = None

        _path = '/catalog/v1/datasets/{datasetRid}/files'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_DatasetFilesPage, self._return_none_for_unknown_union_types)

    def get_dataset_file_uri(self, auth_header: str, dataset_rid: str, file_id: str) -> "scout_catalog_DatasetFileUri":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
            'fileId': file_id,
        }

        _json: Any = None

        _path = '/catalog/v1/datasets/{datasetRid}/{fileId}/uri'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_DatasetFileUri, self._return_none_for_unknown_union_types)

    def mark_file_ingest_successful(self, auth_header: str, dataset_rid: str, file_id: str, request: "scout_catalog_MarkFileIngestSuccessful") -> "scout_catalog_DatasetFile":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
            'fileId': file_id,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/catalog/v1/datasets/{datasetRid}/file/{fileId}/success'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_DatasetFile, self._return_none_for_unknown_union_types)

    def mark_file_ingest_error(self, auth_header: str, dataset_rid: str, file_id: str, request: "scout_catalog_MarkFileIngestError") -> "scout_catalog_DatasetFile":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
            'fileId': file_id,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/catalog/v1/datasets/{datasetRid}/file/{fileId}/error'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_DatasetFile, self._return_none_for_unknown_union_types)

    def update_dataset_metadata(self, auth_header: str, dataset_rid: str, request: "scout_catalog_UpdateDatasetMetadata") -> "scout_catalog_EnrichedDataset":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/catalog/v1/datasets/{datasetRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_EnrichedDataset, self._return_none_for_unknown_union_types)

    def update_bounds(self, auth_header: str, request: "scout_catalog_UpdateBoundsRequest", rid: str) -> "scout_catalog_EnrichedDataset":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/catalog/v1/datasets/{rid}/bounds'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_EnrichedDataset, self._return_none_for_unknown_union_types)

    def delete_dataset_series(self, auth_header: str, dataset_rid: str) -> "scout_catalog_DeleteSeriesResult":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
        }

        _json: Any = None

        _path = '/catalog/v1/series/for-dataset/{datasetRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_DeleteSeriesResult, self._return_none_for_unknown_union_types)

    def archive_dataset(self, auth_header: str, dataset_rid: str) -> None:
        """
        Archives a dataset, which will hide it from search results unless the includeArchived flag is set to true. The
dataset can still be directly accessed by its UUID/rid.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
        }

        _json: Any = None

        _path = '/catalog/v1/datasets/{datasetRid}/archive'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def unarchive_dataset(self, auth_header: str, dataset_rid: str) -> None:
        """
        Undoes the archiving of a dataset.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'datasetRid': dataset_rid,
        }

        _json: Any = None

        _path = '/catalog/v1/datasets/{datasetRid}/unarchive'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def get_all_properties_and_labels(self, auth_header: str) -> "scout_catalog_AllPropertiesAndLabelsResponse":

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/catalog/v1/datasets/all-properties-labels'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_catalog_AllPropertiesAndLabelsResponse, self._return_none_for_unknown_union_types)


scout_catalog_CatalogService.__name__ = "CatalogService"
scout_catalog_CatalogService.__qualname__ = "CatalogService"
scout_catalog_CatalogService.__module__ = "nominal_api.scout_catalog"


class scout_catalog_Channel(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'uuid': ConjureFieldDefinition('uuid', str),
            'name': ConjureFieldDefinition('name', str),
            'dataset_uuid': ConjureFieldDefinition('datasetUuid', str)
        }

    __slots__: List[str] = ['_uuid', '_name', '_dataset_uuid']

    def __init__(self, dataset_uuid: str, name: str, uuid: str) -> None:
        self._uuid = uuid
        self._name = name
        self._dataset_uuid = dataset_uuid

    @builtins.property
    def uuid(self) -> str:
        return self._uuid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def dataset_uuid(self) -> str:
        return self._dataset_uuid


scout_catalog_Channel.__name__ = "Channel"
scout_catalog_Channel.__qualname__ = "Channel"
scout_catalog_Channel.__module__ = "nominal_api.scout_catalog"


class scout_catalog_ChannelConfig(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'prefix_tree_delimiter': ConjureFieldDefinition('prefixTreeDelimiter', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_prefix_tree_delimiter']

    def __init__(self, prefix_tree_delimiter: Optional[str] = None) -> None:
        self._prefix_tree_delimiter = prefix_tree_delimiter

    @builtins.property
    def prefix_tree_delimiter(self) -> Optional[str]:
        """
        If set, will construct a prefix tree for channels of the dataset using the given delimiter.
        """
        return self._prefix_tree_delimiter


scout_catalog_ChannelConfig.__name__ = "ChannelConfig"
scout_catalog_ChannelConfig.__qualname__ = "ChannelConfig"
scout_catalog_ChannelConfig.__module__ = "nominal_api.scout_catalog"


class scout_catalog_ChannelDetails(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'uuid': ConjureFieldDefinition('uuid', str),
            'name': ConjureFieldDefinition('name', str),
            'origin_metadata': ConjureFieldDefinition('originMetadata', Dict[str, str]),
            'csv_index': ConjureFieldDefinition('csvIndex', int),
            'dataset_uuid': ConjureFieldDefinition('datasetUuid', str),
            'handle': ConjureFieldDefinition('handle', scout_catalog_Handle)
        }

    __slots__: List[str] = ['_uuid', '_name', '_origin_metadata', '_csv_index', '_dataset_uuid', '_handle']

    def __init__(self, csv_index: int, dataset_uuid: str, handle: "scout_catalog_Handle", name: str, origin_metadata: Dict[str, str], uuid: str) -> None:
        self._uuid = uuid
        self._name = name
        self._origin_metadata = origin_metadata
        self._csv_index = csv_index
        self._dataset_uuid = dataset_uuid
        self._handle = handle

    @builtins.property
    def uuid(self) -> str:
        return self._uuid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def origin_metadata(self) -> Dict[str, str]:
        return self._origin_metadata

    @builtins.property
    def csv_index(self) -> int:
        return self._csv_index

    @builtins.property
    def dataset_uuid(self) -> str:
        return self._dataset_uuid

    @builtins.property
    def handle(self) -> "scout_catalog_Handle":
        return self._handle


scout_catalog_ChannelDetails.__name__ = "ChannelDetails"
scout_catalog_ChannelDetails.__qualname__ = "ChannelDetails"
scout_catalog_ChannelDetails.__module__ = "nominal_api.scout_catalog"


class scout_catalog_CreateDataset(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', str),
            'handle': ConjureFieldDefinition('handle', OptionalTypeWrapper[scout_catalog_Handle]),
            'metadata': ConjureFieldDefinition('metadata', Dict[str, str]),
            'origin_metadata': ConjureFieldDefinition('originMetadata', scout_catalog_DatasetOriginMetadata),
            'labels': ConjureFieldDefinition('labels', List[datasource_Label]),
            'properties': ConjureFieldDefinition('properties', Dict[datasource_PropertyName, datasource_PropertyValue]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'granularity': ConjureFieldDefinition('granularity', OptionalTypeWrapper[api_Granularity])
        }

    __slots__: List[str] = ['_name', '_handle', '_metadata', '_origin_metadata', '_labels', '_properties', '_description', '_granularity']

    def __init__(self, labels: List[str], metadata: Dict[str, str], name: str, origin_metadata: "scout_catalog_DatasetOriginMetadata", properties: Dict[str, str], description: Optional[str] = None, granularity: Optional["api_Granularity"] = None, handle: Optional["scout_catalog_Handle"] = None) -> None:
        self._name = name
        self._handle = handle
        self._metadata = metadata
        self._origin_metadata = origin_metadata
        self._labels = labels
        self._properties = properties
        self._description = description
        self._granularity = granularity

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def handle(self) -> Optional["scout_catalog_Handle"]:
        return self._handle

    @builtins.property
    def metadata(self) -> Dict[str, str]:
        return self._metadata

    @builtins.property
    def origin_metadata(self) -> "scout_catalog_DatasetOriginMetadata":
        return self._origin_metadata

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def granularity(self) -> Optional["api_Granularity"]:
        """
        Granularity of dataset timestamps. Defaults to nanoseconds.
        """
        return self._granularity


scout_catalog_CreateDataset.__name__ = "CreateDataset"
scout_catalog_CreateDataset.__qualname__ = "CreateDataset"
scout_catalog_CreateDataset.__module__ = "nominal_api.scout_catalog"


class scout_catalog_CustomTimestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'format': ConjureFieldDefinition('format', str),
            'default_year': ConjureFieldDefinition('defaultYear', OptionalTypeWrapper[int]),
            'default_day_of_year': ConjureFieldDefinition('defaultDayOfYear', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_format', '_default_year', '_default_day_of_year']

    def __init__(self, format: str, default_day_of_year: Optional[int] = None, default_year: Optional[int] = None) -> None:
        self._format = format
        self._default_year = default_year
        self._default_day_of_year = default_day_of_year

    @builtins.property
    def format(self) -> str:
        """
        The format string should be in the format of the `DateTimeFormatter` class in Java.
        """
        return self._format

    @builtins.property
    def default_year(self) -> Optional[int]:
        """
        Year is accepted as an optional field for cases like IRIG time format, and will be assumed as current year if not provided.
        """
        return self._default_year

    @builtins.property
    def default_day_of_year(self) -> Optional[int]:
        """
        Default day of year is accepted as an optional field for cases like IRIG time format and will be overridden by day of year in time format.
        """
        return self._default_day_of_year


scout_catalog_CustomTimestamp.__name__ = "CustomTimestamp"
scout_catalog_CustomTimestamp.__qualname__ = "CustomTimestamp"
scout_catalog_CustomTimestamp.__module__ = "nominal_api.scout_catalog"


class scout_catalog_Dataset(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', str),
            'name': ConjureFieldDefinition('name', str),
            'handle': ConjureFieldDefinition('handle', OptionalTypeWrapper[scout_catalog_Handle]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'origin_metadata': ConjureFieldDefinition('originMetadata', scout_catalog_DatasetOriginMetadata),
            'bounds': ConjureFieldDefinition('bounds', OptionalTypeWrapper[scout_catalog_Bounds]),
            'properties': ConjureFieldDefinition('properties', Dict[datasource_PropertyName, datasource_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[datasource_Label]),
            'timestamp_type': ConjureFieldDefinition('timestampType', scout_catalog_WeakTimestampType)
        }

    __slots__: List[str] = ['_rid', '_name', '_handle', '_description', '_origin_metadata', '_bounds', '_properties', '_labels', '_timestamp_type']

    def __init__(self, labels: List[str], name: str, origin_metadata: "scout_catalog_DatasetOriginMetadata", properties: Dict[str, str], rid: str, timestamp_type: "scout_catalog_WeakTimestampType", bounds: Optional["scout_catalog_Bounds"] = None, description: Optional[str] = None, handle: Optional["scout_catalog_Handle"] = None) -> None:
        self._rid = rid
        self._name = name
        self._handle = handle
        self._description = description
        self._origin_metadata = origin_metadata
        self._bounds = bounds
        self._properties = properties
        self._labels = labels
        self._timestamp_type = timestamp_type

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def handle(self) -> Optional["scout_catalog_Handle"]:
        return self._handle

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def origin_metadata(self) -> "scout_catalog_DatasetOriginMetadata":
        return self._origin_metadata

    @builtins.property
    def bounds(self) -> Optional["scout_catalog_Bounds"]:
        return self._bounds

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def timestamp_type(self) -> "scout_catalog_WeakTimestampType":
        return self._timestamp_type


scout_catalog_Dataset.__name__ = "Dataset"
scout_catalog_Dataset.__qualname__ = "Dataset"
scout_catalog_Dataset.__module__ = "nominal_api.scout_catalog"


class scout_catalog_DatasetFile(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'id': ConjureFieldDefinition('id', datasource_DatasetFileId),
            'dataset_rid': ConjureFieldDefinition('datasetRid', api_rids_DatasetRid),
            'name': ConjureFieldDefinition('name', str),
            'handle': ConjureFieldDefinition('handle', scout_catalog_Handle),
            'bounds': ConjureFieldDefinition('bounds', OptionalTypeWrapper[scout_catalog_Bounds]),
            'uploaded_at': ConjureFieldDefinition('uploadedAt', str),
            'ingested_at': ConjureFieldDefinition('ingestedAt', OptionalTypeWrapper[str]),
            'ingest_status': ConjureFieldDefinition('ingestStatus', scout_catalog_IngestStatusV2),
            'timestamp_metadata': ConjureFieldDefinition('timestampMetadata', OptionalTypeWrapper[scout_catalog_TimestampMetadata])
        }

    __slots__: List[str] = ['_id', '_dataset_rid', '_name', '_handle', '_bounds', '_uploaded_at', '_ingested_at', '_ingest_status', '_timestamp_metadata']

    def __init__(self, dataset_rid: str, handle: "scout_catalog_Handle", id: str, ingest_status: "scout_catalog_IngestStatusV2", name: str, uploaded_at: str, bounds: Optional["scout_catalog_Bounds"] = None, ingested_at: Optional[str] = None, timestamp_metadata: Optional["scout_catalog_TimestampMetadata"] = None) -> None:
        self._id = id
        self._dataset_rid = dataset_rid
        self._name = name
        self._handle = handle
        self._bounds = bounds
        self._uploaded_at = uploaded_at
        self._ingested_at = ingested_at
        self._ingest_status = ingest_status
        self._timestamp_metadata = timestamp_metadata

    @builtins.property
    def id(self) -> str:
        return self._id

    @builtins.property
    def dataset_rid(self) -> str:
        return self._dataset_rid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def handle(self) -> "scout_catalog_Handle":
        return self._handle

    @builtins.property
    def bounds(self) -> Optional["scout_catalog_Bounds"]:
        return self._bounds

    @builtins.property
    def uploaded_at(self) -> str:
        """
        Timestamp that the file was received and stored, but not processed or made available to consumers.
        """
        return self._uploaded_at

    @builtins.property
    def ingested_at(self) -> Optional[str]:
        """
        Timestamp that the file is ingested at and made available for processing. If the file has failed to be
ingested for any reason or is still being processed, then this value will be empty.
        """
        return self._ingested_at

    @builtins.property
    def ingest_status(self) -> "scout_catalog_IngestStatusV2":
        return self._ingest_status

    @builtins.property
    def timestamp_metadata(self) -> Optional["scout_catalog_TimestampMetadata"]:
        return self._timestamp_metadata


scout_catalog_DatasetFile.__name__ = "DatasetFile"
scout_catalog_DatasetFile.__qualname__ = "DatasetFile"
scout_catalog_DatasetFile.__module__ = "nominal_api.scout_catalog"


class scout_catalog_DatasetFileUri(ConjureBeanType):
    """
    Pre-signed URI that can be used to download the original file directly. Expires if the download has
not been initiated within 1 minute.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'uri': ConjureFieldDefinition('uri', str)
        }

    __slots__: List[str] = ['_uri']

    def __init__(self, uri: str) -> None:
        self._uri = uri

    @builtins.property
    def uri(self) -> str:
        return self._uri


scout_catalog_DatasetFileUri.__name__ = "DatasetFileUri"
scout_catalog_DatasetFileUri.__qualname__ = "DatasetFileUri"
scout_catalog_DatasetFileUri.__module__ = "nominal_api.scout_catalog"


class scout_catalog_DatasetFilesPage(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'files': ConjureFieldDefinition('files', List[scout_catalog_DatasetFile]),
            'next_page': ConjureFieldDefinition('nextPage', OptionalTypeWrapper[datasource_Token])
        }

    __slots__: List[str] = ['_files', '_next_page']

    def __init__(self, files: List["scout_catalog_DatasetFile"], next_page: Optional[str] = None) -> None:
        self._files = files
        self._next_page = next_page

    @builtins.property
    def files(self) -> List["scout_catalog_DatasetFile"]:
        return self._files

    @builtins.property
    def next_page(self) -> Optional[str]:
        return self._next_page


scout_catalog_DatasetFilesPage.__name__ = "DatasetFilesPage"
scout_catalog_DatasetFilesPage.__qualname__ = "DatasetFilesPage"
scout_catalog_DatasetFilesPage.__module__ = "nominal_api.scout_catalog"


class scout_catalog_DatasetFilter(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', str),
            'value': ConjureFieldDefinition('value', str)
        }

    __slots__: List[str] = ['_name', '_value']

    def __init__(self, name: str, value: str) -> None:
        self._name = name
        self._value = value

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def value(self) -> str:
        return self._value


scout_catalog_DatasetFilter.__name__ = "DatasetFilter"
scout_catalog_DatasetFilter.__qualname__ = "DatasetFilter"
scout_catalog_DatasetFilter.__module__ = "nominal_api.scout_catalog"


class scout_catalog_DatasetOriginMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'path': ConjureFieldDefinition('path', OptionalTypeWrapper[str]),
            'x_series_is_absolute': ConjureFieldDefinition('xSeriesIsAbsolute', OptionalTypeWrapper[bool]),
            'schema_directive_path': ConjureFieldDefinition('schemaDirectivePath', OptionalTypeWrapper[str]),
            'x_series_column_name': ConjureFieldDefinition('xSeriesColumnName', OptionalTypeWrapper[str]),
            'x_series_time_unit': ConjureFieldDefinition('xSeriesTimeUnit', OptionalTypeWrapper[api_TimeUnit]),
            'timestamp_metadata': ConjureFieldDefinition('timestampMetadata', OptionalTypeWrapper[scout_catalog_TimestampMetadata]),
            'channel_config': ConjureFieldDefinition('channelConfig', OptionalTypeWrapper[scout_catalog_ChannelConfig])
        }

    __slots__: List[str] = ['_path', '_x_series_is_absolute', '_schema_directive_path', '_x_series_column_name', '_x_series_time_unit', '_timestamp_metadata', '_channel_config']

    def __init__(self, channel_config: Optional["scout_catalog_ChannelConfig"] = None, path: Optional[str] = None, schema_directive_path: Optional[str] = None, timestamp_metadata: Optional["scout_catalog_TimestampMetadata"] = None, x_series_column_name: Optional[str] = None, x_series_is_absolute: Optional[bool] = None, x_series_time_unit: Optional["api_TimeUnit"] = None) -> None:
        self._path = path
        self._x_series_is_absolute = x_series_is_absolute
        self._schema_directive_path = schema_directive_path
        self._x_series_column_name = x_series_column_name
        self._x_series_time_unit = x_series_time_unit
        self._timestamp_metadata = timestamp_metadata
        self._channel_config = channel_config

    @builtins.property
    def path(self) -> Optional[str]:
        """
        Deprecated in favor of FileOriginMetadata
        """
        return self._path

    @builtins.property
    def x_series_is_absolute(self) -> Optional[bool]:
        """
        Deprecated in favor of FileOriginMetadata
        """
        return self._x_series_is_absolute

    @builtins.property
    def schema_directive_path(self) -> Optional[str]:
        """
        Deprecated in favor of FileOriginMetadata
        """
        return self._schema_directive_path

    @builtins.property
    def x_series_column_name(self) -> Optional[str]:
        """
        Deprecated in favor of FileOriginMetadata
        """
        return self._x_series_column_name

    @builtins.property
    def x_series_time_unit(self) -> Optional["api_TimeUnit"]:
        """
        Deprecated in favor of FileOriginMetadata
        """
        return self._x_series_time_unit

    @builtins.property
    def timestamp_metadata(self) -> Optional["scout_catalog_TimestampMetadata"]:
        """
        Deprecated in favor of FileOriginMetadata
        """
        return self._timestamp_metadata

    @builtins.property
    def channel_config(self) -> Optional["scout_catalog_ChannelConfig"]:
        return self._channel_config


scout_catalog_DatasetOriginMetadata.__name__ = "DatasetOriginMetadata"
scout_catalog_DatasetOriginMetadata.__qualname__ = "DatasetOriginMetadata"
scout_catalog_DatasetOriginMetadata.__module__ = "nominal_api.scout_catalog"


class scout_catalog_DeleteSeriesResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'number_deleted': ConjureFieldDefinition('numberDeleted', int)
        }

    __slots__: List[str] = ['_number_deleted']

    def __init__(self, number_deleted: int) -> None:
        self._number_deleted = number_deleted

    @builtins.property
    def number_deleted(self) -> int:
        return self._number_deleted


scout_catalog_DeleteSeriesResult.__name__ = "DeleteSeriesResult"
scout_catalog_DeleteSeriesResult.__qualname__ = "DeleteSeriesResult"
scout_catalog_DeleteSeriesResult.__module__ = "nominal_api.scout_catalog"


class scout_catalog_EnrichedDataset(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', str),
            'uuid': ConjureFieldDefinition('uuid', str),
            'name': ConjureFieldDefinition('name', str),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'display_name': ConjureFieldDefinition('displayName', str),
            'metadata': ConjureFieldDefinition('metadata', OptionalTypeWrapper[Dict[str, str]]),
            'handle': ConjureFieldDefinition('handle', OptionalTypeWrapper[scout_catalog_Handle]),
            'ingest_date': ConjureFieldDefinition('ingestDate', str),
            'ingest_status': ConjureFieldDefinition('ingestStatus', OptionalTypeWrapper[scout_catalog_IngestStatus]),
            'origin_metadata': ConjureFieldDefinition('originMetadata', scout_catalog_DatasetOriginMetadata),
            'last_ingest_status': ConjureFieldDefinition('lastIngestStatus', scout_catalog_IngestStatusV2),
            'retention_policy': ConjureFieldDefinition('retentionPolicy', scout_catalog_RetentionPolicy),
            'source': ConjureFieldDefinition('source', OptionalTypeWrapper[str]),
            'bounds': ConjureFieldDefinition('bounds', OptionalTypeWrapper[scout_catalog_Bounds]),
            'timestamp_type': ConjureFieldDefinition('timestampType', scout_catalog_WeakTimestampType),
            'labels': ConjureFieldDefinition('labels', List[datasource_Label]),
            'properties': ConjureFieldDefinition('properties', Dict[datasource_PropertyName, datasource_PropertyValue]),
            'granularity': ConjureFieldDefinition('granularity', api_Granularity)
        }

    __slots__: List[str] = ['_rid', '_uuid', '_name', '_description', '_display_name', '_metadata', '_handle', '_ingest_date', '_ingest_status', '_origin_metadata', '_last_ingest_status', '_retention_policy', '_source', '_bounds', '_timestamp_type', '_labels', '_properties', '_granularity']

    def __init__(self, display_name: str, granularity: "api_Granularity", ingest_date: str, labels: List[str], last_ingest_status: "scout_catalog_IngestStatusV2", name: str, origin_metadata: "scout_catalog_DatasetOriginMetadata", properties: Dict[str, str], retention_policy: "scout_catalog_RetentionPolicy", rid: str, timestamp_type: "scout_catalog_WeakTimestampType", uuid: str, bounds: Optional["scout_catalog_Bounds"] = None, description: Optional[str] = None, handle: Optional["scout_catalog_Handle"] = None, ingest_status: Optional["scout_catalog_IngestStatus"] = None, metadata: Optional[Dict[str, str]] = None, source: Optional[str] = None) -> None:
        self._rid = rid
        self._uuid = uuid
        self._name = name
        self._description = description
        self._display_name = display_name
        self._metadata = metadata
        self._handle = handle
        self._ingest_date = ingest_date
        self._ingest_status = ingest_status
        self._origin_metadata = origin_metadata
        self._last_ingest_status = last_ingest_status
        self._retention_policy = retention_policy
        self._source = source
        self._bounds = bounds
        self._timestamp_type = timestamp_type
        self._labels = labels
        self._properties = properties
        self._granularity = granularity

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def uuid(self) -> str:
        return self._uuid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def display_name(self) -> str:
        return self._display_name

    @builtins.property
    def metadata(self) -> Optional[Dict[str, str]]:
        return self._metadata

    @builtins.property
    def handle(self) -> Optional["scout_catalog_Handle"]:
        return self._handle

    @builtins.property
    def ingest_date(self) -> str:
        return self._ingest_date

    @builtins.property
    def ingest_status(self) -> Optional["scout_catalog_IngestStatus"]:
        return self._ingest_status

    @builtins.property
    def origin_metadata(self) -> "scout_catalog_DatasetOriginMetadata":
        return self._origin_metadata

    @builtins.property
    def last_ingest_status(self) -> "scout_catalog_IngestStatusV2":
        return self._last_ingest_status

    @builtins.property
    def retention_policy(self) -> "scout_catalog_RetentionPolicy":
        return self._retention_policy

    @builtins.property
    def source(self) -> Optional[str]:
        return self._source

    @builtins.property
    def bounds(self) -> Optional["scout_catalog_Bounds"]:
        return self._bounds

    @builtins.property
    def timestamp_type(self) -> "scout_catalog_WeakTimestampType":
        return self._timestamp_type

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def granularity(self) -> "api_Granularity":
        return self._granularity


scout_catalog_EnrichedDataset.__name__ = "EnrichedDataset"
scout_catalog_EnrichedDataset.__qualname__ = "EnrichedDataset"
scout_catalog_EnrichedDataset.__module__ = "nominal_api.scout_catalog"


class scout_catalog_EpochTimestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'time_unit': ConjureFieldDefinition('timeUnit', api_TimeUnit)
        }

    __slots__: List[str] = ['_time_unit']

    def __init__(self, time_unit: "api_TimeUnit") -> None:
        self._time_unit = time_unit

    @builtins.property
    def time_unit(self) -> "api_TimeUnit":
        return self._time_unit


scout_catalog_EpochTimestamp.__name__ = "EpochTimestamp"
scout_catalog_EpochTimestamp.__qualname__ = "EpochTimestamp"
scout_catalog_EpochTimestamp.__module__ = "nominal_api.scout_catalog"


class scout_catalog_ErrorResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'error_type': ConjureFieldDefinition('errorType', scout_catalog_ErrorType),
            'message': ConjureFieldDefinition('message', str)
        }

    __slots__: List[str] = ['_error_type', '_message']

    def __init__(self, error_type: str, message: str) -> None:
        self._error_type = error_type
        self._message = message

    @builtins.property
    def error_type(self) -> str:
        return self._error_type

    @builtins.property
    def message(self) -> str:
        return self._message


scout_catalog_ErrorResult.__name__ = "ErrorResult"
scout_catalog_ErrorResult.__qualname__ = "ErrorResult"
scout_catalog_ErrorResult.__module__ = "nominal_api.scout_catalog"


class scout_catalog_GetChannelsForDatasetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset_uuids': ConjureFieldDefinition('datasetUuids', List[str])
        }

    __slots__: List[str] = ['_dataset_uuids']

    def __init__(self, dataset_uuids: List[str]) -> None:
        self._dataset_uuids = dataset_uuids

    @builtins.property
    def dataset_uuids(self) -> List[str]:
        return self._dataset_uuids


scout_catalog_GetChannelsForDatasetsRequest.__name__ = "GetChannelsForDatasetsRequest"
scout_catalog_GetChannelsForDatasetsRequest.__qualname__ = "GetChannelsForDatasetsRequest"
scout_catalog_GetChannelsForDatasetsRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_GetDatasetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset_rids': ConjureFieldDefinition('datasetRids', List[str])
        }

    __slots__: List[str] = ['_dataset_rids']

    def __init__(self, dataset_rids: List[str]) -> None:
        self._dataset_rids = dataset_rids

    @builtins.property
    def dataset_rids(self) -> List[str]:
        return self._dataset_rids


scout_catalog_GetDatasetsRequest.__name__ = "GetDatasetsRequest"
scout_catalog_GetDatasetsRequest.__qualname__ = "GetDatasetsRequest"
scout_catalog_GetDatasetsRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_GetDetailsForChannelsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'channel_uuids': ConjureFieldDefinition('channelUuids', List[str])
        }

    __slots__: List[str] = ['_channel_uuids']

    def __init__(self, channel_uuids: List[str]) -> None:
        self._channel_uuids = channel_uuids

    @builtins.property
    def channel_uuids(self) -> List[str]:
        return self._channel_uuids


scout_catalog_GetDetailsForChannelsRequest.__name__ = "GetDetailsForChannelsRequest"
scout_catalog_GetDetailsForChannelsRequest.__qualname__ = "GetDetailsForChannelsRequest"
scout_catalog_GetDetailsForChannelsRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_GetHandlesForDatasetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset_uuids': ConjureFieldDefinition('datasetUuids', List[str])
        }

    __slots__: List[str] = ['_dataset_uuids']

    def __init__(self, dataset_uuids: List[str]) -> None:
        self._dataset_uuids = dataset_uuids

    @builtins.property
    def dataset_uuids(self) -> List[str]:
        return self._dataset_uuids


scout_catalog_GetHandlesForDatasetsRequest.__name__ = "GetHandlesForDatasetsRequest"
scout_catalog_GetHandlesForDatasetsRequest.__qualname__ = "GetHandlesForDatasetsRequest"
scout_catalog_GetHandlesForDatasetsRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_GetSeriesDetailsByName(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'dataset_rid': ConjureFieldDefinition('datasetRid', str),
            'name': ConjureFieldDefinition('name', str)
        }

    __slots__: List[str] = ['_dataset_rid', '_name']

    def __init__(self, dataset_rid: str, name: str) -> None:
        self._dataset_rid = dataset_rid
        self._name = name

    @builtins.property
    def dataset_rid(self) -> str:
        return self._dataset_rid

    @builtins.property
    def name(self) -> str:
        return self._name


scout_catalog_GetSeriesDetailsByName.__name__ = "GetSeriesDetailsByName"
scout_catalog_GetSeriesDetailsByName.__qualname__ = "GetSeriesDetailsByName"
scout_catalog_GetSeriesDetailsByName.__module__ = "nominal_api.scout_catalog"


class scout_catalog_GetSeriesDetailsByRid(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', str)
        }

    __slots__: List[str] = ['_rid']

    def __init__(self, rid: str) -> None:
        self._rid = rid

    @builtins.property
    def rid(self) -> str:
        return self._rid


scout_catalog_GetSeriesDetailsByRid.__name__ = "GetSeriesDetailsByRid"
scout_catalog_GetSeriesDetailsByRid.__qualname__ = "GetSeriesDetailsByRid"
scout_catalog_GetSeriesDetailsByRid.__module__ = "nominal_api.scout_catalog"


class scout_catalog_GetSeriesDetailsRequest(ConjureUnionType):
    _by_rid: Optional["scout_catalog_GetSeriesDetailsByRid"] = None
    _by_name: Optional["scout_catalog_GetSeriesDetailsByName"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'by_rid': ConjureFieldDefinition('byRid', scout_catalog_GetSeriesDetailsByRid),
            'by_name': ConjureFieldDefinition('byName', scout_catalog_GetSeriesDetailsByName)
        }

    def __init__(
            self,
            by_rid: Optional["scout_catalog_GetSeriesDetailsByRid"] = None,
            by_name: Optional["scout_catalog_GetSeriesDetailsByName"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (by_rid is not None) + (by_name is not None) != 1:
                raise ValueError('a union must contain a single member')

            if by_rid is not None:
                self._by_rid = by_rid
                self._type = 'byRid'
            if by_name is not None:
                self._by_name = by_name
                self._type = 'byName'

        elif type_of_union == 'byRid':
            if by_rid is None:
                raise ValueError('a union value must not be None')
            self._by_rid = by_rid
            self._type = 'byRid'
        elif type_of_union == 'byName':
            if by_name is None:
                raise ValueError('a union value must not be None')
            self._by_name = by_name
            self._type = 'byName'

    @builtins.property
    def by_rid(self) -> Optional["scout_catalog_GetSeriesDetailsByRid"]:
        return self._by_rid

    @builtins.property
    def by_name(self) -> Optional["scout_catalog_GetSeriesDetailsByName"]:
        return self._by_name

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_catalog_GetSeriesDetailsRequestVisitor):
            raise ValueError('{} is not an instance of scout_catalog_GetSeriesDetailsRequestVisitor'.format(visitor.__class__.__name__))
        if self._type == 'byRid' and self.by_rid is not None:
            return visitor._by_rid(self.by_rid)
        if self._type == 'byName' and self.by_name is not None:
            return visitor._by_name(self.by_name)


scout_catalog_GetSeriesDetailsRequest.__name__ = "GetSeriesDetailsRequest"
scout_catalog_GetSeriesDetailsRequest.__qualname__ = "GetSeriesDetailsRequest"
scout_catalog_GetSeriesDetailsRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_GetSeriesDetailsRequestVisitor:

    @abstractmethod
    def _by_rid(self, by_rid: "scout_catalog_GetSeriesDetailsByRid") -> Any:
        pass

    @abstractmethod
    def _by_name(self, by_name: "scout_catalog_GetSeriesDetailsByName") -> Any:
        pass


scout_catalog_GetSeriesDetailsRequestVisitor.__name__ = "GetSeriesDetailsRequestVisitor"
scout_catalog_GetSeriesDetailsRequestVisitor.__qualname__ = "GetSeriesDetailsRequestVisitor"
scout_catalog_GetSeriesDetailsRequestVisitor.__module__ = "nominal_api.scout_catalog"


class scout_catalog_Handle(ConjureUnionType):
    _s3: Optional["scout_catalog_S3Handle"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            's3': ConjureFieldDefinition('s3', scout_catalog_S3Handle)
        }

    def __init__(
            self,
            s3: Optional["scout_catalog_S3Handle"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (s3 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if s3 is not None:
                self._s3 = s3
                self._type = 's3'

        elif type_of_union == 's3':
            if s3 is None:
                raise ValueError('a union value must not be None')
            self._s3 = s3
            self._type = 's3'

    @builtins.property
    def s3(self) -> Optional["scout_catalog_S3Handle"]:
        return self._s3

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_catalog_HandleVisitor):
            raise ValueError('{} is not an instance of scout_catalog_HandleVisitor'.format(visitor.__class__.__name__))
        if self._type == 's3' and self.s3 is not None:
            return visitor._s3(self.s3)


scout_catalog_Handle.__name__ = "Handle"
scout_catalog_Handle.__qualname__ = "Handle"
scout_catalog_Handle.__module__ = "nominal_api.scout_catalog"


class scout_catalog_HandleVisitor:

    @abstractmethod
    def _s3(self, s3: "scout_catalog_S3Handle") -> Any:
        pass


scout_catalog_HandleVisitor.__name__ = "HandleVisitor"
scout_catalog_HandleVisitor.__qualname__ = "HandleVisitor"
scout_catalog_HandleVisitor.__module__ = "nominal_api.scout_catalog"


class scout_catalog_InProgressResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_catalog_InProgressResult.__name__ = "InProgressResult"
scout_catalog_InProgressResult.__qualname__ = "InProgressResult"
scout_catalog_InProgressResult.__module__ = "nominal_api.scout_catalog"


class scout_catalog_IngestProgressV2(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'start_time': ConjureFieldDefinition('startTime', str),
            'end_time': ConjureFieldDefinition('endTime', OptionalTypeWrapper[str]),
            'ingest_status': ConjureFieldDefinition('ingestStatus', scout_catalog_IngestStatusV2),
            'incalculable': ConjureFieldDefinition('incalculable', OptionalTypeWrapper[bool])
        }

    __slots__: List[str] = ['_start_time', '_end_time', '_ingest_status', '_incalculable']

    def __init__(self, ingest_status: "scout_catalog_IngestStatusV2", start_time: str, end_time: Optional[str] = None, incalculable: Optional[bool] = None) -> None:
        self._start_time = start_time
        self._end_time = end_time
        self._ingest_status = ingest_status
        self._incalculable = incalculable

    @builtins.property
    def start_time(self) -> str:
        """
        Timestamp at start of ingest
        """
        return self._start_time

    @builtins.property
    def end_time(self) -> Optional[str]:
        """
        Timestamp at end of ingest, empty if still in progress
        """
        return self._end_time

    @builtins.property
    def ingest_status(self) -> "scout_catalog_IngestStatusV2":
        """
        Status of ingest, contains error if failed
        """
        return self._ingest_status

    @builtins.property
    def incalculable(self) -> Optional[bool]:
        """
        Whether ingest duration can be reliably calculated
        """
        return self._incalculable


scout_catalog_IngestProgressV2.__name__ = "IngestProgressV2"
scout_catalog_IngestProgressV2.__qualname__ = "IngestProgressV2"
scout_catalog_IngestProgressV2.__module__ = "nominal_api.scout_catalog"


class scout_catalog_IngestStatus(ConjureEnumType):

    IN_PROGRESS = 'IN_PROGRESS'
    '''IN_PROGRESS'''
    COMPLETED = 'COMPLETED'
    '''COMPLETED'''
    FAILED = 'FAILED'
    '''FAILED'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_catalog_IngestStatus.__name__ = "IngestStatus"
scout_catalog_IngestStatus.__qualname__ = "IngestStatus"
scout_catalog_IngestStatus.__module__ = "nominal_api.scout_catalog"


class scout_catalog_IngestStatusV2(ConjureUnionType):
    _success: Optional["scout_catalog_SuccessResult"] = None
    _error: Optional["scout_catalog_ErrorResult"] = None
    _in_progress: Optional["scout_catalog_InProgressResult"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'success': ConjureFieldDefinition('success', scout_catalog_SuccessResult),
            'error': ConjureFieldDefinition('error', scout_catalog_ErrorResult),
            'in_progress': ConjureFieldDefinition('inProgress', scout_catalog_InProgressResult)
        }

    def __init__(
            self,
            success: Optional["scout_catalog_SuccessResult"] = None,
            error: Optional["scout_catalog_ErrorResult"] = None,
            in_progress: Optional["scout_catalog_InProgressResult"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (success is not None) + (error is not None) + (in_progress is not None) != 1:
                raise ValueError('a union must contain a single member')

            if success is not None:
                self._success = success
                self._type = 'success'
            if error is not None:
                self._error = error
                self._type = 'error'
            if in_progress is not None:
                self._in_progress = in_progress
                self._type = 'inProgress'

        elif type_of_union == 'success':
            if success is None:
                raise ValueError('a union value must not be None')
            self._success = success
            self._type = 'success'
        elif type_of_union == 'error':
            if error is None:
                raise ValueError('a union value must not be None')
            self._error = error
            self._type = 'error'
        elif type_of_union == 'inProgress':
            if in_progress is None:
                raise ValueError('a union value must not be None')
            self._in_progress = in_progress
            self._type = 'inProgress'

    @builtins.property
    def success(self) -> Optional["scout_catalog_SuccessResult"]:
        return self._success

    @builtins.property
    def error(self) -> Optional["scout_catalog_ErrorResult"]:
        return self._error

    @builtins.property
    def in_progress(self) -> Optional["scout_catalog_InProgressResult"]:
        return self._in_progress

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_catalog_IngestStatusV2Visitor):
            raise ValueError('{} is not an instance of scout_catalog_IngestStatusV2Visitor'.format(visitor.__class__.__name__))
        if self._type == 'success' and self.success is not None:
            return visitor._success(self.success)
        if self._type == 'error' and self.error is not None:
            return visitor._error(self.error)
        if self._type == 'inProgress' and self.in_progress is not None:
            return visitor._in_progress(self.in_progress)


scout_catalog_IngestStatusV2.__name__ = "IngestStatusV2"
scout_catalog_IngestStatusV2.__qualname__ = "IngestStatusV2"
scout_catalog_IngestStatusV2.__module__ = "nominal_api.scout_catalog"


class scout_catalog_IngestStatusV2Visitor:

    @abstractmethod
    def _success(self, success: "scout_catalog_SuccessResult") -> Any:
        pass

    @abstractmethod
    def _error(self, error: "scout_catalog_ErrorResult") -> Any:
        pass

    @abstractmethod
    def _in_progress(self, in_progress: "scout_catalog_InProgressResult") -> Any:
        pass


scout_catalog_IngestStatusV2Visitor.__name__ = "IngestStatusV2Visitor"
scout_catalog_IngestStatusV2Visitor.__qualname__ = "IngestStatusV2Visitor"
scout_catalog_IngestStatusV2Visitor.__module__ = "nominal_api.scout_catalog"


class scout_catalog_Iso8601Timestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_catalog_Iso8601Timestamp.__name__ = "Iso8601Timestamp"
scout_catalog_Iso8601Timestamp.__qualname__ = "Iso8601Timestamp"
scout_catalog_Iso8601Timestamp.__module__ = "nominal_api.scout_catalog"


class scout_catalog_ListDatasetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'include_filters': ConjureFieldDefinition('includeFilters', OptionalTypeWrapper[List[scout_catalog_DatasetFilter]]),
            'exclude_filters': ConjureFieldDefinition('excludeFilters', OptionalTypeWrapper[List[scout_catalog_DatasetFilter]])
        }

    __slots__: List[str] = ['_include_filters', '_exclude_filters']

    def __init__(self, exclude_filters: Optional[List["scout_catalog_DatasetFilter"]] = None, include_filters: Optional[List["scout_catalog_DatasetFilter"]] = None) -> None:
        self._include_filters = include_filters
        self._exclude_filters = exclude_filters

    @builtins.property
    def include_filters(self) -> Optional[List["scout_catalog_DatasetFilter"]]:
        return self._include_filters

    @builtins.property
    def exclude_filters(self) -> Optional[List["scout_catalog_DatasetFilter"]]:
        return self._exclude_filters


scout_catalog_ListDatasetsRequest.__name__ = "ListDatasetsRequest"
scout_catalog_ListDatasetsRequest.__qualname__ = "ListDatasetsRequest"
scout_catalog_ListDatasetsRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_MarkFileIngestError(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'error_result': ConjureFieldDefinition('errorResult', scout_catalog_ErrorResult)
        }

    __slots__: List[str] = ['_error_result']

    def __init__(self, error_result: "scout_catalog_ErrorResult") -> None:
        self._error_result = error_result

    @builtins.property
    def error_result(self) -> "scout_catalog_ErrorResult":
        return self._error_result


scout_catalog_MarkFileIngestError.__name__ = "MarkFileIngestError"
scout_catalog_MarkFileIngestError.__qualname__ = "MarkFileIngestError"
scout_catalog_MarkFileIngestError.__module__ = "nominal_api.scout_catalog"


class scout_catalog_MarkFileIngestSuccessful(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'bounds': ConjureFieldDefinition('bounds', scout_catalog_Bounds),
            'ingested_at': ConjureFieldDefinition('ingestedAt', api_Timestamp)
        }

    __slots__: List[str] = ['_bounds', '_ingested_at']

    def __init__(self, bounds: "scout_catalog_Bounds", ingested_at: "api_Timestamp") -> None:
        self._bounds = bounds
        self._ingested_at = ingested_at

    @builtins.property
    def bounds(self) -> "scout_catalog_Bounds":
        return self._bounds

    @builtins.property
    def ingested_at(self) -> "api_Timestamp":
        """
        The ingestion timestamp is produced by CSV splitter and stored directly in the clickhouse table.
It's produced externally and passed here to handle retries and failures, and must be nanosecond precision.
Two files cannot have the same ingested at timestamp.
        """
        return self._ingested_at


scout_catalog_MarkFileIngestSuccessful.__name__ = "MarkFileIngestSuccessful"
scout_catalog_MarkFileIngestSuccessful.__qualname__ = "MarkFileIngestSuccessful"
scout_catalog_MarkFileIngestSuccessful.__module__ = "nominal_api.scout_catalog"


class scout_catalog_RelativeTimestamp(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'time_unit': ConjureFieldDefinition('timeUnit', api_TimeUnit),
            'offset': ConjureFieldDefinition('offset', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_time_unit', '_offset']

    def __init__(self, time_unit: "api_TimeUnit", offset: Optional[str] = None) -> None:
        self._time_unit = time_unit
        self._offset = offset

    @builtins.property
    def time_unit(self) -> "api_TimeUnit":
        return self._time_unit

    @builtins.property
    def offset(self) -> Optional[str]:
        return self._offset


scout_catalog_RelativeTimestamp.__name__ = "RelativeTimestamp"
scout_catalog_RelativeTimestamp.__qualname__ = "RelativeTimestamp"
scout_catalog_RelativeTimestamp.__module__ = "nominal_api.scout_catalog"


class scout_catalog_RetentionPolicy(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'type': ConjureFieldDefinition('type', scout_catalog_RetentionPolicyType),
            'expiry_days': ConjureFieldDefinition('expiryDays', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_type', '_expiry_days']

    def __init__(self, type: "scout_catalog_RetentionPolicyType", expiry_days: Optional[int] = None) -> None:
        self._type = type
        self._expiry_days = expiry_days

    @builtins.property
    def type(self) -> "scout_catalog_RetentionPolicyType":
        return self._type

    @builtins.property
    def expiry_days(self) -> Optional[int]:
        return self._expiry_days


scout_catalog_RetentionPolicy.__name__ = "RetentionPolicy"
scout_catalog_RetentionPolicy.__qualname__ = "RetentionPolicy"
scout_catalog_RetentionPolicy.__module__ = "nominal_api.scout_catalog"


class scout_catalog_RetentionPolicyType(ConjureEnumType):

    KEEP_UNTIL_EXPIRY = 'KEEP_UNTIL_EXPIRY'
    '''KEEP_UNTIL_EXPIRY'''
    KEEP_FOREVER = 'KEEP_FOREVER'
    '''KEEP_FOREVER'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_catalog_RetentionPolicyType.__name__ = "RetentionPolicyType"
scout_catalog_RetentionPolicyType.__qualname__ = "RetentionPolicyType"
scout_catalog_RetentionPolicyType.__module__ = "nominal_api.scout_catalog"


class scout_catalog_S3Handle(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'bucket': ConjureFieldDefinition('bucket', str),
            'key': ConjureFieldDefinition('key', str)
        }

    __slots__: List[str] = ['_bucket', '_key']

    def __init__(self, bucket: str, key: str) -> None:
        self._bucket = bucket
        self._key = key

    @builtins.property
    def bucket(self) -> str:
        return self._bucket

    @builtins.property
    def key(self) -> str:
        return self._key


scout_catalog_S3Handle.__name__ = "S3Handle"
scout_catalog_S3Handle.__qualname__ = "S3Handle"
scout_catalog_S3Handle.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SearchDatasetsByTextRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'search_text': ConjureFieldDefinition('searchText', OptionalTypeWrapper[str]),
            'include_filters': ConjureFieldDefinition('includeFilters', OptionalTypeWrapper[List[scout_catalog_DatasetFilter]]),
            'exclude_filters': ConjureFieldDefinition('excludeFilters', OptionalTypeWrapper[List[scout_catalog_DatasetFilter]]),
            'page_request': ConjureFieldDefinition('pageRequest', OptionalTypeWrapper[datasource_pagination_api_PageRequest]),
            'include_archived': ConjureFieldDefinition('includeArchived', OptionalTypeWrapper[bool])
        }

    __slots__: List[str] = ['_search_text', '_include_filters', '_exclude_filters', '_page_request', '_include_archived']

    def __init__(self, exclude_filters: Optional[List["scout_catalog_DatasetFilter"]] = None, include_archived: Optional[bool] = None, include_filters: Optional[List["scout_catalog_DatasetFilter"]] = None, page_request: Optional["datasource_pagination_api_PageRequest"] = None, search_text: Optional[str] = None) -> None:
        self._search_text = search_text
        self._include_filters = include_filters
        self._exclude_filters = exclude_filters
        self._page_request = page_request
        self._include_archived = include_archived

    @builtins.property
    def search_text(self) -> Optional[str]:
        return self._search_text

    @builtins.property
    def include_filters(self) -> Optional[List["scout_catalog_DatasetFilter"]]:
        return self._include_filters

    @builtins.property
    def exclude_filters(self) -> Optional[List["scout_catalog_DatasetFilter"]]:
        return self._exclude_filters

    @builtins.property
    def page_request(self) -> Optional["datasource_pagination_api_PageRequest"]:
        """
        The PageRequest.pageToken must be an integer offset. If a PageRequest.pageSize is not provided, the 
default page size (100) will be used.
        """
        return self._page_request

    @builtins.property
    def include_archived(self) -> Optional[bool]:
        """
        Defaults to false if not specified.
        """
        return self._include_archived


scout_catalog_SearchDatasetsByTextRequest.__name__ = "SearchDatasetsByTextRequest"
scout_catalog_SearchDatasetsByTextRequest.__qualname__ = "SearchDatasetsByTextRequest"
scout_catalog_SearchDatasetsByTextRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SearchDatasetsByTextResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[scout_catalog_EnrichedDataset]),
            'page_response': ConjureFieldDefinition('pageResponse', datasource_pagination_api_PageResponse)
        }

    __slots__: List[str] = ['_results', '_page_response']

    def __init__(self, page_response: "datasource_pagination_api_PageResponse", results: List["scout_catalog_EnrichedDataset"]) -> None:
        self._results = results
        self._page_response = page_response

    @builtins.property
    def results(self) -> List["scout_catalog_EnrichedDataset"]:
        return self._results

    @builtins.property
    def page_response(self) -> "datasource_pagination_api_PageResponse":
        return self._page_response


scout_catalog_SearchDatasetsByTextResponse.__name__ = "SearchDatasetsByTextResponse"
scout_catalog_SearchDatasetsByTextResponse.__qualname__ = "SearchDatasetsByTextResponse"
scout_catalog_SearchDatasetsByTextResponse.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SearchDatasetsQuery(ConjureUnionType):
    _search_text: Optional[str] = None
    _exact_match: Optional[str] = None
    _label: Optional[str] = None
    _properties: Optional["datasource_Property"] = None
    _ingest_status: Optional["scout_catalog_IngestStatus"] = None
    _ingested_before_inclusive: Optional[str] = None
    _ingested_after_inclusive: Optional[str] = None
    _archive_status: Optional[bool] = None
    _and_: Optional[List["scout_catalog_SearchDatasetsQuery"]] = None
    _or_: Optional[List["scout_catalog_SearchDatasetsQuery"]] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'search_text': ConjureFieldDefinition('searchText', str),
            'exact_match': ConjureFieldDefinition('exactMatch', str),
            'label': ConjureFieldDefinition('label', datasource_Label),
            'properties': ConjureFieldDefinition('properties', datasource_Property),
            'ingest_status': ConjureFieldDefinition('ingestStatus', scout_catalog_IngestStatus),
            'ingested_before_inclusive': ConjureFieldDefinition('ingestedBeforeInclusive', str),
            'ingested_after_inclusive': ConjureFieldDefinition('ingestedAfterInclusive', str),
            'archive_status': ConjureFieldDefinition('archiveStatus', bool),
            'and_': ConjureFieldDefinition('and', List[scout_catalog_SearchDatasetsQuery]),
            'or_': ConjureFieldDefinition('or', List[scout_catalog_SearchDatasetsQuery])
        }

    def __init__(
            self,
            search_text: Optional[str] = None,
            exact_match: Optional[str] = None,
            label: Optional[str] = None,
            properties: Optional["datasource_Property"] = None,
            ingest_status: Optional["scout_catalog_IngestStatus"] = None,
            ingested_before_inclusive: Optional[str] = None,
            ingested_after_inclusive: Optional[str] = None,
            archive_status: Optional[bool] = None,
            and_: Optional[List["scout_catalog_SearchDatasetsQuery"]] = None,
            or_: Optional[List["scout_catalog_SearchDatasetsQuery"]] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (search_text is not None) + (exact_match is not None) + (label is not None) + (properties is not None) + (ingest_status is not None) + (ingested_before_inclusive is not None) + (ingested_after_inclusive is not None) + (archive_status is not None) + (and_ is not None) + (or_ is not None) != 1:
                raise ValueError('a union must contain a single member')

            if search_text is not None:
                self._search_text = search_text
                self._type = 'searchText'
            if exact_match is not None:
                self._exact_match = exact_match
                self._type = 'exactMatch'
            if label is not None:
                self._label = label
                self._type = 'label'
            if properties is not None:
                self._properties = properties
                self._type = 'properties'
            if ingest_status is not None:
                self._ingest_status = ingest_status
                self._type = 'ingestStatus'
            if ingested_before_inclusive is not None:
                self._ingested_before_inclusive = ingested_before_inclusive
                self._type = 'ingestedBeforeInclusive'
            if ingested_after_inclusive is not None:
                self._ingested_after_inclusive = ingested_after_inclusive
                self._type = 'ingestedAfterInclusive'
            if archive_status is not None:
                self._archive_status = archive_status
                self._type = 'archiveStatus'
            if and_ is not None:
                self._and_ = and_
                self._type = 'and'
            if or_ is not None:
                self._or_ = or_
                self._type = 'or'

        elif type_of_union == 'searchText':
            if search_text is None:
                raise ValueError('a union value must not be None')
            self._search_text = search_text
            self._type = 'searchText'
        elif type_of_union == 'exactMatch':
            if exact_match is None:
                raise ValueError('a union value must not be None')
            self._exact_match = exact_match
            self._type = 'exactMatch'
        elif type_of_union == 'label':
            if label is None:
                raise ValueError('a union value must not be None')
            self._label = label
            self._type = 'label'
        elif type_of_union == 'properties':
            if properties is None:
                raise ValueError('a union value must not be None')
            self._properties = properties
            self._type = 'properties'
        elif type_of_union == 'ingestStatus':
            if ingest_status is None:
                raise ValueError('a union value must not be None')
            self._ingest_status = ingest_status
            self._type = 'ingestStatus'
        elif type_of_union == 'ingestedBeforeInclusive':
            if ingested_before_inclusive is None:
                raise ValueError('a union value must not be None')
            self._ingested_before_inclusive = ingested_before_inclusive
            self._type = 'ingestedBeforeInclusive'
        elif type_of_union == 'ingestedAfterInclusive':
            if ingested_after_inclusive is None:
                raise ValueError('a union value must not be None')
            self._ingested_after_inclusive = ingested_after_inclusive
            self._type = 'ingestedAfterInclusive'
        elif type_of_union == 'archiveStatus':
            if archive_status is None:
                raise ValueError('a union value must not be None')
            self._archive_status = archive_status
            self._type = 'archiveStatus'
        elif type_of_union == 'and':
            if and_ is None:
                raise ValueError('a union value must not be None')
            self._and_ = and_
            self._type = 'and'
        elif type_of_union == 'or':
            if or_ is None:
                raise ValueError('a union value must not be None')
            self._or_ = or_
            self._type = 'or'

    @builtins.property
    def search_text(self) -> Optional[str]:
        return self._search_text

    @builtins.property
    def exact_match(self) -> Optional[str]:
        """
        Performs case insensitive exact match search on the name.
        """
        return self._exact_match

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def properties(self) -> Optional["datasource_Property"]:
        return self._properties

    @builtins.property
    def ingest_status(self) -> Optional["scout_catalog_IngestStatus"]:
        return self._ingest_status

    @builtins.property
    def ingested_before_inclusive(self) -> Optional[str]:
        return self._ingested_before_inclusive

    @builtins.property
    def ingested_after_inclusive(self) -> Optional[str]:
        return self._ingested_after_inclusive

    @builtins.property
    def archive_status(self) -> Optional[bool]:
        return self._archive_status

    @builtins.property
    def and_(self) -> Optional[List["scout_catalog_SearchDatasetsQuery"]]:
        return self._and_

    @builtins.property
    def or_(self) -> Optional[List["scout_catalog_SearchDatasetsQuery"]]:
        return self._or_

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_catalog_SearchDatasetsQueryVisitor):
            raise ValueError('{} is not an instance of scout_catalog_SearchDatasetsQueryVisitor'.format(visitor.__class__.__name__))
        if self._type == 'searchText' and self.search_text is not None:
            return visitor._search_text(self.search_text)
        if self._type == 'exactMatch' and self.exact_match is not None:
            return visitor._exact_match(self.exact_match)
        if self._type == 'label' and self.label is not None:
            return visitor._label(self.label)
        if self._type == 'properties' and self.properties is not None:
            return visitor._properties(self.properties)
        if self._type == 'ingestStatus' and self.ingest_status is not None:
            return visitor._ingest_status(self.ingest_status)
        if self._type == 'ingestedBeforeInclusive' and self.ingested_before_inclusive is not None:
            return visitor._ingested_before_inclusive(self.ingested_before_inclusive)
        if self._type == 'ingestedAfterInclusive' and self.ingested_after_inclusive is not None:
            return visitor._ingested_after_inclusive(self.ingested_after_inclusive)
        if self._type == 'archiveStatus' and self.archive_status is not None:
            return visitor._archive_status(self.archive_status)
        if self._type == 'and' and self.and_ is not None:
            return visitor._and(self.and_)
        if self._type == 'or' and self.or_ is not None:
            return visitor._or(self.or_)


scout_catalog_SearchDatasetsQuery.__name__ = "SearchDatasetsQuery"
scout_catalog_SearchDatasetsQuery.__qualname__ = "SearchDatasetsQuery"
scout_catalog_SearchDatasetsQuery.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SearchDatasetsQueryVisitor:

    @abstractmethod
    def _search_text(self, search_text: str) -> Any:
        pass

    @abstractmethod
    def _exact_match(self, exact_match: str) -> Any:
        pass

    @abstractmethod
    def _label(self, label: str) -> Any:
        pass

    @abstractmethod
    def _properties(self, properties: "datasource_Property") -> Any:
        pass

    @abstractmethod
    def _ingest_status(self, ingest_status: "scout_catalog_IngestStatus") -> Any:
        pass

    @abstractmethod
    def _ingested_before_inclusive(self, ingested_before_inclusive: str) -> Any:
        pass

    @abstractmethod
    def _ingested_after_inclusive(self, ingested_after_inclusive: str) -> Any:
        pass

    @abstractmethod
    def _archive_status(self, archive_status: bool) -> Any:
        pass

    @abstractmethod
    def _and(self, and_: List["scout_catalog_SearchDatasetsQuery"]) -> Any:
        pass

    @abstractmethod
    def _or(self, or_: List["scout_catalog_SearchDatasetsQuery"]) -> Any:
        pass


scout_catalog_SearchDatasetsQueryVisitor.__name__ = "SearchDatasetsQueryVisitor"
scout_catalog_SearchDatasetsQueryVisitor.__qualname__ = "SearchDatasetsQueryVisitor"
scout_catalog_SearchDatasetsQueryVisitor.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SearchDatasetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'query': ConjureFieldDefinition('query', scout_catalog_SearchDatasetsQuery),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int]),
            'token': ConjureFieldDefinition('token', OptionalTypeWrapper[datasource_Token]),
            'sort_options': ConjureFieldDefinition('sortOptions', scout_catalog_SortOptions)
        }

    __slots__: List[str] = ['_query', '_page_size', '_token', '_sort_options']

    def __init__(self, query: "scout_catalog_SearchDatasetsQuery", sort_options: "scout_catalog_SortOptions", page_size: Optional[int] = None, token: Optional[str] = None) -> None:
        self._query = query
        self._page_size = page_size
        self._token = token
        self._sort_options = sort_options

    @builtins.property
    def query(self) -> "scout_catalog_SearchDatasetsQuery":
        return self._query

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Defaults to 100. Will throw if larger than 1_000.
        """
        return self._page_size

    @builtins.property
    def token(self) -> Optional[str]:
        return self._token

    @builtins.property
    def sort_options(self) -> "scout_catalog_SortOptions":
        return self._sort_options


scout_catalog_SearchDatasetsRequest.__name__ = "SearchDatasetsRequest"
scout_catalog_SearchDatasetsRequest.__qualname__ = "SearchDatasetsRequest"
scout_catalog_SearchDatasetsRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SearchDatasetsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[scout_catalog_EnrichedDataset]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[datasource_Token])
        }

    __slots__: List[str] = ['_results', '_next_page_token']

    def __init__(self, results: List["scout_catalog_EnrichedDataset"], next_page_token: Optional[str] = None) -> None:
        self._results = results
        self._next_page_token = next_page_token

    @builtins.property
    def results(self) -> List["scout_catalog_EnrichedDataset"]:
        return self._results

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


scout_catalog_SearchDatasetsResponse.__name__ = "SearchDatasetsResponse"
scout_catalog_SearchDatasetsResponse.__qualname__ = "SearchDatasetsResponse"
scout_catalog_SearchDatasetsResponse.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SeriesDetails(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', str),
            'name': ConjureFieldDefinition('name', str),
            'origin_metadata': ConjureFieldDefinition('originMetadata', Dict[str, str]),
            'csv_index': ConjureFieldDefinition('csvIndex', OptionalTypeWrapper[int]),
            'dataset_rid': ConjureFieldDefinition('datasetRid', str),
            'handle': ConjureFieldDefinition('handle', scout_catalog_Handle)
        }

    __slots__: List[str] = ['_rid', '_name', '_origin_metadata', '_csv_index', '_dataset_rid', '_handle']

    def __init__(self, dataset_rid: str, handle: "scout_catalog_Handle", name: str, origin_metadata: Dict[str, str], rid: str, csv_index: Optional[int] = None) -> None:
        self._rid = rid
        self._name = name
        self._origin_metadata = origin_metadata
        self._csv_index = csv_index
        self._dataset_rid = dataset_rid
        self._handle = handle

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def origin_metadata(self) -> Dict[str, str]:
        return self._origin_metadata

    @builtins.property
    def csv_index(self) -> Optional[int]:
        return self._csv_index

    @builtins.property
    def dataset_rid(self) -> str:
        return self._dataset_rid

    @builtins.property
    def handle(self) -> "scout_catalog_Handle":
        return self._handle


scout_catalog_SeriesDetails.__name__ = "SeriesDetails"
scout_catalog_SeriesDetails.__qualname__ = "SeriesDetails"
scout_catalog_SeriesDetails.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SkipRowsConfig(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'header_row_index': ConjureFieldDefinition('headerRowIndex', int),
            'data_start_row_index': ConjureFieldDefinition('dataStartRowIndex', int)
        }

    __slots__: List[str] = ['_header_row_index', '_data_start_row_index']

    def __init__(self, data_start_row_index: int, header_row_index: int) -> None:
        self._header_row_index = header_row_index
        self._data_start_row_index = data_start_row_index

    @builtins.property
    def header_row_index(self) -> int:
        return self._header_row_index

    @builtins.property
    def data_start_row_index(self) -> int:
        return self._data_start_row_index


scout_catalog_SkipRowsConfig.__name__ = "SkipRowsConfig"
scout_catalog_SkipRowsConfig.__qualname__ = "SkipRowsConfig"
scout_catalog_SkipRowsConfig.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SortField(ConjureEnumType):

    INGEST_DATE = 'INGEST_DATE'
    '''INGEST_DATE'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_catalog_SortField.__name__ = "SortField"
scout_catalog_SortField.__qualname__ = "SortField"
scout_catalog_SortField.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SortOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'is_descending': ConjureFieldDefinition('isDescending', bool),
            'field': ConjureFieldDefinition('field', scout_catalog_SortField)
        }

    __slots__: List[str] = ['_is_descending', '_field']

    def __init__(self, field: "scout_catalog_SortField", is_descending: bool) -> None:
        self._is_descending = is_descending
        self._field = field

    @builtins.property
    def is_descending(self) -> bool:
        return self._is_descending

    @builtins.property
    def field(self) -> "scout_catalog_SortField":
        return self._field


scout_catalog_SortOptions.__name__ = "SortOptions"
scout_catalog_SortOptions.__qualname__ = "SortOptions"
scout_catalog_SortOptions.__module__ = "nominal_api.scout_catalog"


class scout_catalog_SuccessResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_catalog_SuccessResult.__name__ = "SuccessResult"
scout_catalog_SuccessResult.__qualname__ = "SuccessResult"
scout_catalog_SuccessResult.__module__ = "nominal_api.scout_catalog"


class scout_catalog_TimestampMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'series_name': ConjureFieldDefinition('seriesName', str),
            'timestamp_type': ConjureFieldDefinition('timestampType', scout_catalog_TimestampType)
        }

    __slots__: List[str] = ['_series_name', '_timestamp_type']

    def __init__(self, series_name: str, timestamp_type: "scout_catalog_TimestampType") -> None:
        self._series_name = series_name
        self._timestamp_type = timestamp_type

    @builtins.property
    def series_name(self) -> str:
        return self._series_name

    @builtins.property
    def timestamp_type(self) -> "scout_catalog_TimestampType":
        return self._timestamp_type


scout_catalog_TimestampMetadata.__name__ = "TimestampMetadata"
scout_catalog_TimestampMetadata.__qualname__ = "TimestampMetadata"
scout_catalog_TimestampMetadata.__module__ = "nominal_api.scout_catalog"


class scout_catalog_TimestampType(ConjureUnionType):
    _relative: Optional["scout_catalog_RelativeTimestamp"] = None
    _absolute: Optional["scout_catalog_AbsoluteTimestamp"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'relative': ConjureFieldDefinition('relative', scout_catalog_RelativeTimestamp),
            'absolute': ConjureFieldDefinition('absolute', scout_catalog_AbsoluteTimestamp)
        }

    def __init__(
            self,
            relative: Optional["scout_catalog_RelativeTimestamp"] = None,
            absolute: Optional["scout_catalog_AbsoluteTimestamp"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (relative is not None) + (absolute is not None) != 1:
                raise ValueError('a union must contain a single member')

            if relative is not None:
                self._relative = relative
                self._type = 'relative'
            if absolute is not None:
                self._absolute = absolute
                self._type = 'absolute'

        elif type_of_union == 'relative':
            if relative is None:
                raise ValueError('a union value must not be None')
            self._relative = relative
            self._type = 'relative'
        elif type_of_union == 'absolute':
            if absolute is None:
                raise ValueError('a union value must not be None')
            self._absolute = absolute
            self._type = 'absolute'

    @builtins.property
    def relative(self) -> Optional["scout_catalog_RelativeTimestamp"]:
        return self._relative

    @builtins.property
    def absolute(self) -> Optional["scout_catalog_AbsoluteTimestamp"]:
        return self._absolute

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_catalog_TimestampTypeVisitor):
            raise ValueError('{} is not an instance of scout_catalog_TimestampTypeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'relative' and self.relative is not None:
            return visitor._relative(self.relative)
        if self._type == 'absolute' and self.absolute is not None:
            return visitor._absolute(self.absolute)


scout_catalog_TimestampType.__name__ = "TimestampType"
scout_catalog_TimestampType.__qualname__ = "TimestampType"
scout_catalog_TimestampType.__module__ = "nominal_api.scout_catalog"


class scout_catalog_TimestampTypeVisitor:

    @abstractmethod
    def _relative(self, relative: "scout_catalog_RelativeTimestamp") -> Any:
        pass

    @abstractmethod
    def _absolute(self, absolute: "scout_catalog_AbsoluteTimestamp") -> Any:
        pass


scout_catalog_TimestampTypeVisitor.__name__ = "TimestampTypeVisitor"
scout_catalog_TimestampTypeVisitor.__qualname__ = "TimestampTypeVisitor"
scout_catalog_TimestampTypeVisitor.__module__ = "nominal_api.scout_catalog"


class scout_catalog_UnitConfig(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'unit_metadata_key': ConjureFieldDefinition('unitMetadataKey', str)
        }

    __slots__: List[str] = ['_unit_metadata_key']

    def __init__(self, unit_metadata_key: str) -> None:
        self._unit_metadata_key = unit_metadata_key

    @builtins.property
    def unit_metadata_key(self) -> str:
        """
        If set, will extract the series unit from the column metadata of the ingested file.
        """
        return self._unit_metadata_key


scout_catalog_UnitConfig.__name__ = "UnitConfig"
scout_catalog_UnitConfig.__qualname__ = "UnitConfig"
scout_catalog_UnitConfig.__module__ = "nominal_api.scout_catalog"


class scout_catalog_UpdateBoundsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'bounds': ConjureFieldDefinition('bounds', scout_catalog_Bounds)
        }

    __slots__: List[str] = ['_bounds']

    def __init__(self, bounds: "scout_catalog_Bounds") -> None:
        self._bounds = bounds

    @builtins.property
    def bounds(self) -> "scout_catalog_Bounds":
        return self._bounds


scout_catalog_UpdateBoundsRequest.__name__ = "UpdateBoundsRequest"
scout_catalog_UpdateBoundsRequest.__qualname__ = "UpdateBoundsRequest"
scout_catalog_UpdateBoundsRequest.__module__ = "nominal_api.scout_catalog"


class scout_catalog_UpdateDatasetMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', OptionalTypeWrapper[str]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'labels': ConjureFieldDefinition('labels', OptionalTypeWrapper[List[datasource_Label]]),
            'properties': ConjureFieldDefinition('properties', OptionalTypeWrapper[Dict[datasource_PropertyName, datasource_PropertyValue]])
        }

    __slots__: List[str] = ['_name', '_description', '_labels', '_properties']

    def __init__(self, description: Optional[str] = None, labels: Optional[List[str]] = None, name: Optional[str] = None, properties: Optional[Dict[str, str]] = None) -> None:
        self._name = name
        self._description = description
        self._labels = labels
        self._properties = properties

    @builtins.property
    def name(self) -> Optional[str]:
        return self._name

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def labels(self) -> Optional[List[str]]:
        return self._labels

    @builtins.property
    def properties(self) -> Optional[Dict[str, str]]:
        return self._properties


scout_catalog_UpdateDatasetMetadata.__name__ = "UpdateDatasetMetadata"
scout_catalog_UpdateDatasetMetadata.__qualname__ = "UpdateDatasetMetadata"
scout_catalog_UpdateDatasetMetadata.__module__ = "nominal_api.scout_catalog"


class scout_catalog_UpdateIngestStatus(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'status': ConjureFieldDefinition('status', scout_catalog_IngestStatus),
            'dataset_uuid': ConjureFieldDefinition('datasetUuid', str)
        }

    __slots__: List[str] = ['_status', '_dataset_uuid']

    def __init__(self, dataset_uuid: str, status: "scout_catalog_IngestStatus") -> None:
        self._status = status
        self._dataset_uuid = dataset_uuid

    @builtins.property
    def status(self) -> "scout_catalog_IngestStatus":
        return self._status

    @builtins.property
    def dataset_uuid(self) -> str:
        return self._dataset_uuid


scout_catalog_UpdateIngestStatus.__name__ = "UpdateIngestStatus"
scout_catalog_UpdateIngestStatus.__qualname__ = "UpdateIngestStatus"
scout_catalog_UpdateIngestStatus.__module__ = "nominal_api.scout_catalog"


class scout_catalog_UpdateIngestStatusV2(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'status': ConjureFieldDefinition('status', scout_catalog_IngestStatusV2),
            'dataset_uuid': ConjureFieldDefinition('datasetUuid', str)
        }

    __slots__: List[str] = ['_status', '_dataset_uuid']

    def __init__(self, dataset_uuid: str, status: "scout_catalog_IngestStatusV2") -> None:
        self._status = status
        self._dataset_uuid = dataset_uuid

    @builtins.property
    def status(self) -> "scout_catalog_IngestStatusV2":
        return self._status

    @builtins.property
    def dataset_uuid(self) -> str:
        return self._dataset_uuid


scout_catalog_UpdateIngestStatusV2.__name__ = "UpdateIngestStatusV2"
scout_catalog_UpdateIngestStatusV2.__qualname__ = "UpdateIngestStatusV2"
scout_catalog_UpdateIngestStatusV2.__module__ = "nominal_api.scout_catalog"


class scout_catalog_WeakTimestampType(ConjureEnumType):
    """
    If a CSV dataset is still being split, the timestamp type is not known.
    """

    ABSOLUTE = 'ABSOLUTE'
    '''ABSOLUTE'''
    RELATIVE = 'RELATIVE'
    '''RELATIVE'''
    PENDING = 'PENDING'
    '''PENDING'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_catalog_WeakTimestampType.__name__ = "WeakTimestampType"
scout_catalog_WeakTimestampType.__qualname__ = "WeakTimestampType"
scout_catalog_WeakTimestampType.__module__ = "nominal_api.scout_catalog"


class scout_channelvariables_api_ChannelVariable(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'display_name': ConjureFieldDefinition('displayName', OptionalTypeWrapper[str]),
            'variable_name': ConjureFieldDefinition('variableName', scout_channelvariables_api_ChannelVariableName),
            'compute_spec': ConjureFieldDefinition('computeSpec', scout_channelvariables_api_ComputeSpec)
        }

    __slots__: List[str] = ['_display_name', '_variable_name', '_compute_spec']

    def __init__(self, compute_spec: "scout_channelvariables_api_ComputeSpec", variable_name: str, display_name: Optional[str] = None) -> None:
        self._display_name = display_name
        self._variable_name = variable_name
        self._compute_spec = compute_spec

    @builtins.property
    def display_name(self) -> Optional[str]:
        return self._display_name

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def compute_spec(self) -> "scout_channelvariables_api_ComputeSpec":
        return self._compute_spec


scout_channelvariables_api_ChannelVariable.__name__ = "ChannelVariable"
scout_channelvariables_api_ChannelVariable.__qualname__ = "ChannelVariable"
scout_channelvariables_api_ChannelVariable.__module__ = "nominal_api.scout_channelvariables_api"


class scout_channelvariables_api_ComputeSpec(ConjureUnionType):
    _v1: Optional[str] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_channelvariables_api_ComputeSpecV1)
        }

    def __init__(
            self,
            v1: Optional[str] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional[str]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_channelvariables_api_ComputeSpecVisitor):
            raise ValueError('{} is not an instance of scout_channelvariables_api_ComputeSpecVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_channelvariables_api_ComputeSpec.__name__ = "ComputeSpec"
scout_channelvariables_api_ComputeSpec.__qualname__ = "ComputeSpec"
scout_channelvariables_api_ComputeSpec.__module__ = "nominal_api.scout_channelvariables_api"


class scout_channelvariables_api_ComputeSpecVisitor:

    @abstractmethod
    def _v1(self, v1: str) -> Any:
        pass


scout_channelvariables_api_ComputeSpecVisitor.__name__ = "ComputeSpecVisitor"
scout_channelvariables_api_ComputeSpecVisitor.__qualname__ = "ComputeSpecVisitor"
scout_channelvariables_api_ComputeSpecVisitor.__module__ = "nominal_api.scout_channelvariables_api"


class scout_chart_api_DeprecatedDefinitionAndSchemaVersion(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'schema_version': ConjureFieldDefinition('schemaVersion', str),
            'definition': ConjureFieldDefinition('definition', scout_chart_api_JsonString)
        }

    __slots__: List[str] = ['_schema_version', '_definition']

    def __init__(self, definition: str, schema_version: str) -> None:
        self._schema_version = schema_version
        self._definition = definition

    @builtins.property
    def schema_version(self) -> str:
        return self._schema_version

    @builtins.property
    def definition(self) -> str:
        return self._definition


scout_chart_api_DeprecatedDefinitionAndSchemaVersion.__name__ = "DeprecatedDefinitionAndSchemaVersion"
scout_chart_api_DeprecatedDefinitionAndSchemaVersion.__qualname__ = "DeprecatedDefinitionAndSchemaVersion"
scout_chart_api_DeprecatedDefinitionAndSchemaVersion.__module__ = "nominal_api.scout_chart_api"


class scout_chartdefinition_api_AxisDisplayOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'show_title': ConjureFieldDefinition('showTitle', bool),
            'axis_width': ConjureFieldDefinition('axisWidth', OptionalTypeWrapper[float]),
            'scale_type': ConjureFieldDefinition('scaleType', OptionalTypeWrapper[scout_chartdefinition_api_AxisScaleType])
        }

    __slots__: List[str] = ['_show_title', '_axis_width', '_scale_type']

    def __init__(self, show_title: bool, axis_width: Optional[float] = None, scale_type: Optional["scout_chartdefinition_api_AxisScaleType"] = None) -> None:
        self._show_title = show_title
        self._axis_width = axis_width
        self._scale_type = scale_type

    @builtins.property
    def show_title(self) -> bool:
        return self._show_title

    @builtins.property
    def axis_width(self) -> Optional[float]:
        return self._axis_width

    @builtins.property
    def scale_type(self) -> Optional["scout_chartdefinition_api_AxisScaleType"]:
        """
        The scale type of the axis. If not specified, the default is LINEAR.
        """
        return self._scale_type


scout_chartdefinition_api_AxisDisplayOptions.__name__ = "AxisDisplayOptions"
scout_chartdefinition_api_AxisDisplayOptions.__qualname__ = "AxisDisplayOptions"
scout_chartdefinition_api_AxisDisplayOptions.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_AxisDomainType(ConjureEnumType):

    NUMERIC = 'NUMERIC'
    '''NUMERIC'''
    CATEGORICAL = 'CATEGORICAL'
    '''CATEGORICAL'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_chartdefinition_api_AxisDomainType.__name__ = "AxisDomainType"
scout_chartdefinition_api_AxisDomainType.__qualname__ = "AxisDomainType"
scout_chartdefinition_api_AxisDomainType.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_AxisPosition(ConjureEnumType):

    LEFT = 'LEFT'
    '''LEFT'''
    RIGHT = 'RIGHT'
    '''RIGHT'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_chartdefinition_api_AxisPosition.__name__ = "AxisPosition"
scout_chartdefinition_api_AxisPosition.__qualname__ = "AxisPosition"
scout_chartdefinition_api_AxisPosition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_AxisRange(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'start': ConjureFieldDefinition('start', OptionalTypeWrapper[float]),
            'end': ConjureFieldDefinition('end', OptionalTypeWrapper[float])
        }

    __slots__: List[str] = ['_start', '_end']

    def __init__(self, end: Optional[float] = None, start: Optional[float] = None) -> None:
        self._start = start
        self._end = end

    @builtins.property
    def start(self) -> Optional[float]:
        return self._start

    @builtins.property
    def end(self) -> Optional[float]:
        return self._end


scout_chartdefinition_api_AxisRange.__name__ = "AxisRange"
scout_chartdefinition_api_AxisRange.__qualname__ = "AxisRange"
scout_chartdefinition_api_AxisRange.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_AxisScaleType(ConjureEnumType):

    LINEAR = 'LINEAR'
    '''LINEAR'''
    LOG = 'LOG'
    '''LOG'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_chartdefinition_api_AxisScaleType.__name__ = "AxisScaleType"
scout_chartdefinition_api_AxisScaleType.__qualname__ = "AxisScaleType"
scout_chartdefinition_api_AxisScaleType.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_AxisThresholdGroup(ConjureUnionType):
    _line_thresholds: Optional["scout_chartdefinition_api_LineThresholdGroup"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'line_thresholds': ConjureFieldDefinition('lineThresholds', scout_chartdefinition_api_LineThresholdGroup)
        }

    def __init__(
            self,
            line_thresholds: Optional["scout_chartdefinition_api_LineThresholdGroup"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (line_thresholds is not None) != 1:
                raise ValueError('a union must contain a single member')

            if line_thresholds is not None:
                self._line_thresholds = line_thresholds
                self._type = 'lineThresholds'

        elif type_of_union == 'lineThresholds':
            if line_thresholds is None:
                raise ValueError('a union value must not be None')
            self._line_thresholds = line_thresholds
            self._type = 'lineThresholds'

    @builtins.property
    def line_thresholds(self) -> Optional["scout_chartdefinition_api_LineThresholdGroup"]:
        return self._line_thresholds

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_AxisThresholdGroupVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_AxisThresholdGroupVisitor'.format(visitor.__class__.__name__))
        if self._type == 'lineThresholds' and self.line_thresholds is not None:
            return visitor._line_thresholds(self.line_thresholds)


scout_chartdefinition_api_AxisThresholdGroup.__name__ = "AxisThresholdGroup"
scout_chartdefinition_api_AxisThresholdGroup.__qualname__ = "AxisThresholdGroup"
scout_chartdefinition_api_AxisThresholdGroup.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_AxisThresholdGroupVisitor:

    @abstractmethod
    def _line_thresholds(self, line_thresholds: "scout_chartdefinition_api_LineThresholdGroup") -> Any:
        pass


scout_chartdefinition_api_AxisThresholdGroupVisitor.__name__ = "AxisThresholdGroupVisitor"
scout_chartdefinition_api_AxisThresholdGroupVisitor.__qualname__ = "AxisThresholdGroupVisitor"
scout_chartdefinition_api_AxisThresholdGroupVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_AxisThresholdVisualization(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'axis_id': ConjureFieldDefinition('axisId', scout_chartdefinition_api_AxisId),
            'visibility': ConjureFieldDefinition('visibility', bool),
            'thresholds': ConjureFieldDefinition('thresholds', scout_chartdefinition_api_AxisThresholdGroup)
        }

    __slots__: List[str] = ['_axis_id', '_visibility', '_thresholds']

    def __init__(self, axis_id: str, thresholds: "scout_chartdefinition_api_AxisThresholdGroup", visibility: bool) -> None:
        self._axis_id = axis_id
        self._visibility = visibility
        self._thresholds = thresholds

    @builtins.property
    def axis_id(self) -> str:
        return self._axis_id

    @builtins.property
    def visibility(self) -> bool:
        """
        Determines it's current visibility in the time series chart.
        """
        return self._visibility

    @builtins.property
    def thresholds(self) -> "scout_chartdefinition_api_AxisThresholdGroup":
        return self._thresholds


scout_chartdefinition_api_AxisThresholdVisualization.__name__ = "AxisThresholdVisualization"
scout_chartdefinition_api_AxisThresholdVisualization.__qualname__ = "AxisThresholdVisualization"
scout_chartdefinition_api_AxisThresholdVisualization.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_CartesianChartDefinition(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_CartesianChartDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_CartesianChartDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_CartesianChartDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_CartesianChartDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_CartesianChartDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_CartesianChartDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_CartesianChartDefinition.__name__ = "CartesianChartDefinition"
scout_chartdefinition_api_CartesianChartDefinition.__qualname__ = "CartesianChartDefinition"
scout_chartdefinition_api_CartesianChartDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_CartesianChartDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_CartesianChartDefinitionV1") -> Any:
        pass


scout_chartdefinition_api_CartesianChartDefinitionVisitor.__name__ = "CartesianChartDefinitionVisitor"
scout_chartdefinition_api_CartesianChartDefinitionVisitor.__qualname__ = "CartesianChartDefinitionVisitor"
scout_chartdefinition_api_CartesianChartDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_CartesianChartDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'plots': ConjureFieldDefinition('plots', List[scout_chartdefinition_api_CartesianPlot]),
            'events': ConjureFieldDefinition('events', List[scout_chartdefinition_api_Event]),
            'comparison_run_groups': ConjureFieldDefinition('comparisonRunGroups', List[scout_comparisonrun_api_ComparisonRunGroup]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'value_axes': ConjureFieldDefinition('valueAxes', List[scout_chartdefinition_api_ValueAxis])
        }

    __slots__: List[str] = ['_plots', '_events', '_comparison_run_groups', '_title', '_value_axes']

    def __init__(self, comparison_run_groups: List["scout_comparisonrun_api_ComparisonRunGroup"], events: List["scout_chartdefinition_api_Event"], plots: List["scout_chartdefinition_api_CartesianPlot"], value_axes: List["scout_chartdefinition_api_ValueAxis"], title: Optional[str] = None) -> None:
        self._plots = plots
        self._events = events
        self._comparison_run_groups = comparison_run_groups
        self._title = title
        self._value_axes = value_axes

    @builtins.property
    def plots(self) -> List["scout_chartdefinition_api_CartesianPlot"]:
        return self._plots

    @builtins.property
    def events(self) -> List["scout_chartdefinition_api_Event"]:
        return self._events

    @builtins.property
    def comparison_run_groups(self) -> List["scout_comparisonrun_api_ComparisonRunGroup"]:
        return self._comparison_run_groups

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def value_axes(self) -> List["scout_chartdefinition_api_ValueAxis"]:
        return self._value_axes


scout_chartdefinition_api_CartesianChartDefinitionV1.__name__ = "CartesianChartDefinitionV1"
scout_chartdefinition_api_CartesianChartDefinitionV1.__qualname__ = "CartesianChartDefinitionV1"
scout_chartdefinition_api_CartesianChartDefinitionV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_CartesianPlot(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'x_variable_name': ConjureFieldDefinition('xVariableName', scout_channelvariables_api_ChannelVariableName),
            'y_variable_name': ConjureFieldDefinition('yVariableName', scout_channelvariables_api_ChannelVariableName),
            'enabled': ConjureFieldDefinition('enabled', OptionalTypeWrapper[bool]),
            'x_axis_id': ConjureFieldDefinition('xAxisId', scout_chartdefinition_api_AxisId),
            'y_axis_id': ConjureFieldDefinition('yAxisId', scout_chartdefinition_api_AxisId),
            'color': ConjureFieldDefinition('color', scout_api_HexColor)
        }

    __slots__: List[str] = ['_x_variable_name', '_y_variable_name', '_enabled', '_x_axis_id', '_y_axis_id', '_color']

    def __init__(self, color: str, x_axis_id: str, x_variable_name: str, y_axis_id: str, y_variable_name: str, enabled: Optional[bool] = None) -> None:
        self._x_variable_name = x_variable_name
        self._y_variable_name = y_variable_name
        self._enabled = enabled
        self._x_axis_id = x_axis_id
        self._y_axis_id = y_axis_id
        self._color = color

    @builtins.property
    def x_variable_name(self) -> str:
        return self._x_variable_name

    @builtins.property
    def y_variable_name(self) -> str:
        return self._y_variable_name

    @builtins.property
    def enabled(self) -> Optional[bool]:
        return self._enabled

    @builtins.property
    def x_axis_id(self) -> str:
        return self._x_axis_id

    @builtins.property
    def y_axis_id(self) -> str:
        return self._y_axis_id

    @builtins.property
    def color(self) -> str:
        return self._color


scout_chartdefinition_api_CartesianPlot.__name__ = "CartesianPlot"
scout_chartdefinition_api_CartesianPlot.__qualname__ = "CartesianPlot"
scout_chartdefinition_api_CartesianPlot.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ChecklistChartDefinition(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_ChecklistChartDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_ChecklistChartDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_ChecklistChartDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_ChecklistChartDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_ChecklistChartDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_ChecklistChartDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_ChecklistChartDefinition.__name__ = "ChecklistChartDefinition"
scout_chartdefinition_api_ChecklistChartDefinition.__qualname__ = "ChecklistChartDefinition"
scout_chartdefinition_api_ChecklistChartDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ChecklistChartDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_ChecklistChartDefinitionV1") -> Any:
        pass


scout_chartdefinition_api_ChecklistChartDefinitionVisitor.__name__ = "ChecklistChartDefinitionVisitor"
scout_chartdefinition_api_ChecklistChartDefinitionVisitor.__qualname__ = "ChecklistChartDefinitionVisitor"
scout_chartdefinition_api_ChecklistChartDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ChecklistChartDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'selected_checklist_rids': ConjureFieldDefinition('selectedChecklistRids', List[str]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_selected_checklist_rids', '_title']

    def __init__(self, selected_checklist_rids: List[str], title: Optional[str] = None) -> None:
        self._selected_checklist_rids = selected_checklist_rids
        self._title = title

    @builtins.property
    def selected_checklist_rids(self) -> List[str]:
        return self._selected_checklist_rids

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title


scout_chartdefinition_api_ChecklistChartDefinitionV1.__name__ = "ChecklistChartDefinitionV1"
scout_chartdefinition_api_ChecklistChartDefinitionV1.__qualname__ = "ChecklistChartDefinitionV1"
scout_chartdefinition_api_ChecklistChartDefinitionV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_DefaultFill(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'color': ConjureFieldDefinition('color', scout_api_HexColor)
        }

    __slots__: List[str] = ['_color']

    def __init__(self, color: str) -> None:
        self._color = color

    @builtins.property
    def color(self) -> str:
        return self._color


scout_chartdefinition_api_DefaultFill.__name__ = "DefaultFill"
scout_chartdefinition_api_DefaultFill.__qualname__ = "DefaultFill"
scout_chartdefinition_api_DefaultFill.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_EnumRawVisualisation(ConjureBeanType):
    """
    The settings for a raw enum visualisation.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'color_map': ConjureFieldDefinition('colorMap', OptionalTypeWrapper[Dict[str, scout_api_HexColor]])
        }

    __slots__: List[str] = ['_color_map']

    def __init__(self, color_map: Optional[Dict[str, str]] = None) -> None:
        self._color_map = color_map

    @builtins.property
    def color_map(self) -> Optional[Dict[str, str]]:
        return self._color_map


scout_chartdefinition_api_EnumRawVisualisation.__name__ = "EnumRawVisualisation"
scout_chartdefinition_api_EnumRawVisualisation.__qualname__ = "EnumRawVisualisation"
scout_chartdefinition_api_EnumRawVisualisation.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_EnumValueChannel(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variable_name': ConjureFieldDefinition('variableName', scout_channelvariables_api_ChannelVariableName),
            'visualisation': ConjureFieldDefinition('visualisation', scout_chartdefinition_api_EnumValueVisualisation)
        }

    __slots__: List[str] = ['_variable_name', '_visualisation']

    def __init__(self, variable_name: str, visualisation: "scout_chartdefinition_api_EnumValueVisualisation") -> None:
        self._variable_name = variable_name
        self._visualisation = visualisation

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def visualisation(self) -> "scout_chartdefinition_api_EnumValueVisualisation":
        return self._visualisation


scout_chartdefinition_api_EnumValueChannel.__name__ = "EnumValueChannel"
scout_chartdefinition_api_EnumValueChannel.__qualname__ = "EnumValueChannel"
scout_chartdefinition_api_EnumValueChannel.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_EnumValueVisualisation(ConjureUnionType):
    _raw: Optional["scout_chartdefinition_api_EnumRawVisualisation"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'raw': ConjureFieldDefinition('raw', scout_chartdefinition_api_EnumRawVisualisation)
        }

    def __init__(
            self,
            raw: Optional["scout_chartdefinition_api_EnumRawVisualisation"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (raw is not None) != 1:
                raise ValueError('a union must contain a single member')

            if raw is not None:
                self._raw = raw
                self._type = 'raw'

        elif type_of_union == 'raw':
            if raw is None:
                raise ValueError('a union value must not be None')
            self._raw = raw
            self._type = 'raw'

    @builtins.property
    def raw(self) -> Optional["scout_chartdefinition_api_EnumRawVisualisation"]:
        return self._raw

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_EnumValueVisualisationVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_EnumValueVisualisationVisitor'.format(visitor.__class__.__name__))
        if self._type == 'raw' and self.raw is not None:
            return visitor._raw(self.raw)


scout_chartdefinition_api_EnumValueVisualisation.__name__ = "EnumValueVisualisation"
scout_chartdefinition_api_EnumValueVisualisation.__qualname__ = "EnumValueVisualisation"
scout_chartdefinition_api_EnumValueVisualisation.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_EnumValueVisualisationVisitor:

    @abstractmethod
    def _raw(self, raw: "scout_chartdefinition_api_EnumRawVisualisation") -> Any:
        pass


scout_chartdefinition_api_EnumValueVisualisationVisitor.__name__ = "EnumValueVisualisationVisitor"
scout_chartdefinition_api_EnumValueVisualisationVisitor.__qualname__ = "EnumValueVisualisationVisitor"
scout_chartdefinition_api_EnumValueVisualisationVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_Event(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'display_name': ConjureFieldDefinition('displayName', OptionalTypeWrapper[str]),
            'event_uuid': ConjureFieldDefinition('eventUuid', str),
            'is_pinned': ConjureFieldDefinition('isPinned', bool)
        }

    __slots__: List[str] = ['_display_name', '_event_uuid', '_is_pinned']

    def __init__(self, event_uuid: str, is_pinned: bool, display_name: Optional[str] = None) -> None:
        self._display_name = display_name
        self._event_uuid = event_uuid
        self._is_pinned = is_pinned

    @builtins.property
    def display_name(self) -> Optional[str]:
        return self._display_name

    @builtins.property
    def event_uuid(self) -> str:
        return self._event_uuid

    @builtins.property
    def is_pinned(self) -> bool:
        return self._is_pinned


scout_chartdefinition_api_Event.__name__ = "Event"
scout_chartdefinition_api_Event.__qualname__ = "Event"
scout_chartdefinition_api_Event.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_FrequencyChartDefinition(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_FrequencyChartDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_FrequencyChartDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_FrequencyChartDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_FrequencyChartDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_FrequencyChartDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_FrequencyChartDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_FrequencyChartDefinition.__name__ = "FrequencyChartDefinition"
scout_chartdefinition_api_FrequencyChartDefinition.__qualname__ = "FrequencyChartDefinition"
scout_chartdefinition_api_FrequencyChartDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_FrequencyChartDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_FrequencyChartDefinitionV1") -> Any:
        pass


scout_chartdefinition_api_FrequencyChartDefinitionVisitor.__name__ = "FrequencyChartDefinitionVisitor"
scout_chartdefinition_api_FrequencyChartDefinitionVisitor.__qualname__ = "FrequencyChartDefinitionVisitor"
scout_chartdefinition_api_FrequencyChartDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_FrequencyChartDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'plots': ConjureFieldDefinition('plots', List[scout_chartdefinition_api_FrequencyPlot]),
            'events': ConjureFieldDefinition('events', List[scout_chartdefinition_api_Event]),
            'comparison_run_groups': ConjureFieldDefinition('comparisonRunGroups', List[scout_comparisonrun_api_ComparisonRunGroup]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'value_axes': ConjureFieldDefinition('valueAxes', List[scout_chartdefinition_api_ValueAxis])
        }

    __slots__: List[str] = ['_plots', '_events', '_comparison_run_groups', '_title', '_value_axes']

    def __init__(self, comparison_run_groups: List["scout_comparisonrun_api_ComparisonRunGroup"], events: List["scout_chartdefinition_api_Event"], plots: List["scout_chartdefinition_api_FrequencyPlot"], value_axes: List["scout_chartdefinition_api_ValueAxis"], title: Optional[str] = None) -> None:
        self._plots = plots
        self._events = events
        self._comparison_run_groups = comparison_run_groups
        self._title = title
        self._value_axes = value_axes

    @builtins.property
    def plots(self) -> List["scout_chartdefinition_api_FrequencyPlot"]:
        return self._plots

    @builtins.property
    def events(self) -> List["scout_chartdefinition_api_Event"]:
        return self._events

    @builtins.property
    def comparison_run_groups(self) -> List["scout_comparisonrun_api_ComparisonRunGroup"]:
        return self._comparison_run_groups

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def value_axes(self) -> List["scout_chartdefinition_api_ValueAxis"]:
        return self._value_axes


scout_chartdefinition_api_FrequencyChartDefinitionV1.__name__ = "FrequencyChartDefinitionV1"
scout_chartdefinition_api_FrequencyChartDefinitionV1.__qualname__ = "FrequencyChartDefinitionV1"
scout_chartdefinition_api_FrequencyChartDefinitionV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_FrequencyPlot(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variable_name': ConjureFieldDefinition('variableName', scout_channelvariables_api_ChannelVariableName),
            'enabled': ConjureFieldDefinition('enabled', OptionalTypeWrapper[bool]),
            'y_axis_id': ConjureFieldDefinition('yAxisId', scout_chartdefinition_api_AxisId),
            'color': ConjureFieldDefinition('color', scout_api_HexColor),
            'line_style': ConjureFieldDefinition('lineStyle', scout_chartdefinition_api_LineStyle)
        }

    __slots__: List[str] = ['_variable_name', '_enabled', '_y_axis_id', '_color', '_line_style']

    def __init__(self, color: str, line_style: "scout_chartdefinition_api_LineStyle", variable_name: str, y_axis_id: str, enabled: Optional[bool] = None) -> None:
        self._variable_name = variable_name
        self._enabled = enabled
        self._y_axis_id = y_axis_id
        self._color = color
        self._line_style = line_style

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def enabled(self) -> Optional[bool]:
        return self._enabled

    @builtins.property
    def y_axis_id(self) -> str:
        return self._y_axis_id

    @builtins.property
    def color(self) -> str:
        return self._color

    @builtins.property
    def line_style(self) -> "scout_chartdefinition_api_LineStyle":
        return self._line_style


scout_chartdefinition_api_FrequencyPlot.__name__ = "FrequencyPlot"
scout_chartdefinition_api_FrequencyPlot.__qualname__ = "FrequencyPlot"
scout_chartdefinition_api_FrequencyPlot.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoAdditionalVariable(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variable_name': ConjureFieldDefinition('variableName', scout_channelvariables_api_ChannelVariableName),
            'label': ConjureFieldDefinition('label', OptionalTypeWrapper[str]),
            'visualization_option': ConjureFieldDefinition('visualizationOption', OptionalTypeWrapper[scout_chartdefinition_api_GeoSecondaryPlotVisualizationOption])
        }

    __slots__: List[str] = ['_variable_name', '_label', '_visualization_option']

    def __init__(self, variable_name: str, label: Optional[str] = None, visualization_option: Optional["scout_chartdefinition_api_GeoSecondaryPlotVisualizationOption"] = None) -> None:
        self._variable_name = variable_name
        self._label = label
        self._visualization_option = visualization_option

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def visualization_option(self) -> Optional["scout_chartdefinition_api_GeoSecondaryPlotVisualizationOption"]:
        return self._visualization_option


scout_chartdefinition_api_GeoAdditionalVariable.__name__ = "GeoAdditionalVariable"
scout_chartdefinition_api_GeoAdditionalVariable.__qualname__ = "GeoAdditionalVariable"
scout_chartdefinition_api_GeoAdditionalVariable.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoCustomFeature(ConjureUnionType):
    """Additional static objects on the map, such as a point representing a tower"""
    _point: Optional["scout_chartdefinition_api_GeoPoint"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'point': ConjureFieldDefinition('point', scout_chartdefinition_api_GeoPoint)
        }

    def __init__(
            self,
            point: Optional["scout_chartdefinition_api_GeoPoint"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (point is not None) != 1:
                raise ValueError('a union must contain a single member')

            if point is not None:
                self._point = point
                self._type = 'point'

        elif type_of_union == 'point':
            if point is None:
                raise ValueError('a union value must not be None')
            self._point = point
            self._type = 'point'

    @builtins.property
    def point(self) -> Optional["scout_chartdefinition_api_GeoPoint"]:
        return self._point

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_GeoCustomFeatureVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_GeoCustomFeatureVisitor'.format(visitor.__class__.__name__))
        if self._type == 'point' and self.point is not None:
            return visitor._point(self.point)


scout_chartdefinition_api_GeoCustomFeature.__name__ = "GeoCustomFeature"
scout_chartdefinition_api_GeoCustomFeature.__qualname__ = "GeoCustomFeature"
scout_chartdefinition_api_GeoCustomFeature.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoCustomFeatureVisitor:

    @abstractmethod
    def _point(self, point: "scout_chartdefinition_api_GeoPoint") -> Any:
        pass


scout_chartdefinition_api_GeoCustomFeatureVisitor.__name__ = "GeoCustomFeatureVisitor"
scout_chartdefinition_api_GeoCustomFeatureVisitor.__qualname__ = "GeoCustomFeatureVisitor"
scout_chartdefinition_api_GeoCustomFeatureVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoLineStyle(ConjureEnumType):

    POINTS = 'POINTS'
    '''POINTS'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_chartdefinition_api_GeoLineStyle.__name__ = "GeoLineStyle"
scout_chartdefinition_api_GeoLineStyle.__qualname__ = "GeoLineStyle"
scout_chartdefinition_api_GeoLineStyle.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoPlotFromLatLong(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'latitude_variable_name': ConjureFieldDefinition('latitudeVariableName', scout_channelvariables_api_ChannelVariableName),
            'longitude_variable_name': ConjureFieldDefinition('longitudeVariableName', scout_channelvariables_api_ChannelVariableName),
            'secondary_variables': ConjureFieldDefinition('secondaryVariables', OptionalTypeWrapper[List[scout_chartdefinition_api_GeoAdditionalVariable]]),
            'enabled': ConjureFieldDefinition('enabled', OptionalTypeWrapper[bool]),
            'label': ConjureFieldDefinition('label', OptionalTypeWrapper[str]),
            'visualization_options': ConjureFieldDefinition('visualizationOptions', scout_chartdefinition_api_GeoPlotVisualizationOptions)
        }

    __slots__: List[str] = ['_latitude_variable_name', '_longitude_variable_name', '_secondary_variables', '_enabled', '_label', '_visualization_options']

    def __init__(self, latitude_variable_name: str, longitude_variable_name: str, visualization_options: "scout_chartdefinition_api_GeoPlotVisualizationOptions", enabled: Optional[bool] = None, label: Optional[str] = None, secondary_variables: Optional[List["scout_chartdefinition_api_GeoAdditionalVariable"]] = None) -> None:
        self._latitude_variable_name = latitude_variable_name
        self._longitude_variable_name = longitude_variable_name
        self._secondary_variables = secondary_variables
        self._enabled = enabled
        self._label = label
        self._visualization_options = visualization_options

    @builtins.property
    def latitude_variable_name(self) -> str:
        return self._latitude_variable_name

    @builtins.property
    def longitude_variable_name(self) -> str:
        return self._longitude_variable_name

    @builtins.property
    def secondary_variables(self) -> Optional[List["scout_chartdefinition_api_GeoAdditionalVariable"]]:
        """
        optional for backcompatibility
        """
        return self._secondary_variables

    @builtins.property
    def enabled(self) -> Optional[bool]:
        return self._enabled

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def visualization_options(self) -> "scout_chartdefinition_api_GeoPlotVisualizationOptions":
        return self._visualization_options


scout_chartdefinition_api_GeoPlotFromLatLong.__name__ = "GeoPlotFromLatLong"
scout_chartdefinition_api_GeoPlotFromLatLong.__qualname__ = "GeoPlotFromLatLong"
scout_chartdefinition_api_GeoPlotFromLatLong.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoPlotSecondaryVisibilityConfig(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'visible': ConjureFieldDefinition('visible', bool),
            'variable_name': ConjureFieldDefinition('variableName', OptionalTypeWrapper[scout_channelvariables_api_ChannelVariableName])
        }

    __slots__: List[str] = ['_visible', '_variable_name']

    def __init__(self, visible: bool, variable_name: Optional[str] = None) -> None:
        self._visible = visible
        self._variable_name = variable_name

    @builtins.property
    def visible(self) -> bool:
        return self._visible

    @builtins.property
    def variable_name(self) -> Optional[str]:
        return self._variable_name


scout_chartdefinition_api_GeoPlotSecondaryVisibilityConfig.__name__ = "GeoPlotSecondaryVisibilityConfig"
scout_chartdefinition_api_GeoPlotSecondaryVisibilityConfig.__qualname__ = "GeoPlotSecondaryVisibilityConfig"
scout_chartdefinition_api_GeoPlotSecondaryVisibilityConfig.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoPlotVisualizationOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'color': ConjureFieldDefinition('color', scout_api_HexColor),
            'line_style': ConjureFieldDefinition('lineStyle', scout_chartdefinition_api_GeoLineStyle),
            'secondary_color_visualization': ConjureFieldDefinition('secondaryColorVisualization', OptionalTypeWrapper[scout_chartdefinition_api_GeoPlotSecondaryVisibilityConfig])
        }

    __slots__: List[str] = ['_color', '_line_style', '_secondary_color_visualization']

    def __init__(self, color: str, line_style: "scout_chartdefinition_api_GeoLineStyle", secondary_color_visualization: Optional["scout_chartdefinition_api_GeoPlotSecondaryVisibilityConfig"] = None) -> None:
        self._color = color
        self._line_style = line_style
        self._secondary_color_visualization = secondary_color_visualization

    @builtins.property
    def color(self) -> str:
        return self._color

    @builtins.property
    def line_style(self) -> "scout_chartdefinition_api_GeoLineStyle":
        return self._line_style

    @builtins.property
    def secondary_color_visualization(self) -> Optional["scout_chartdefinition_api_GeoPlotSecondaryVisibilityConfig"]:
        """
        If visible, overwrites any existing visualization options on the geo plot.
        """
        return self._secondary_color_visualization


scout_chartdefinition_api_GeoPlotVisualizationOptions.__name__ = "GeoPlotVisualizationOptions"
scout_chartdefinition_api_GeoPlotVisualizationOptions.__qualname__ = "GeoPlotVisualizationOptions"
scout_chartdefinition_api_GeoPlotVisualizationOptions.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoPoint(ConjureBeanType):
    """
    A static coordinate on the map
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'label': ConjureFieldDefinition('label', OptionalTypeWrapper[str]),
            'icon': ConjureFieldDefinition('icon', str),
            'latitude': ConjureFieldDefinition('latitude', float),
            'longitude': ConjureFieldDefinition('longitude', float),
            'variables': ConjureFieldDefinition('variables', List[scout_chartdefinition_api_GeoAdditionalVariable])
        }

    __slots__: List[str] = ['_label', '_icon', '_latitude', '_longitude', '_variables']

    def __init__(self, icon: str, latitude: float, longitude: float, variables: List["scout_chartdefinition_api_GeoAdditionalVariable"], label: Optional[str] = None) -> None:
        self._label = label
        self._icon = icon
        self._latitude = latitude
        self._longitude = longitude
        self._variables = variables

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def icon(self) -> str:
        return self._icon

    @builtins.property
    def latitude(self) -> float:
        return self._latitude

    @builtins.property
    def longitude(self) -> float:
        return self._longitude

    @builtins.property
    def variables(self) -> List["scout_chartdefinition_api_GeoAdditionalVariable"]:
        return self._variables


scout_chartdefinition_api_GeoPoint.__name__ = "GeoPoint"
scout_chartdefinition_api_GeoPoint.__qualname__ = "GeoPoint"
scout_chartdefinition_api_GeoPoint.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoSecondaryPlotVisualizationOption(ConjureUnionType):
    """Specifies how values of a secondary channel should be visualized."""
    _as_colors: Optional["scout_chartdefinition_api_ValueToColorMap"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'as_colors': ConjureFieldDefinition('asColors', scout_chartdefinition_api_ValueToColorMap)
        }

    def __init__(
            self,
            as_colors: Optional["scout_chartdefinition_api_ValueToColorMap"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (as_colors is not None) != 1:
                raise ValueError('a union must contain a single member')

            if as_colors is not None:
                self._as_colors = as_colors
                self._type = 'asColors'

        elif type_of_union == 'asColors':
            if as_colors is None:
                raise ValueError('a union value must not be None')
            self._as_colors = as_colors
            self._type = 'asColors'

    @builtins.property
    def as_colors(self) -> Optional["scout_chartdefinition_api_ValueToColorMap"]:
        return self._as_colors

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_GeoSecondaryPlotVisualizationOptionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_GeoSecondaryPlotVisualizationOptionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'asColors' and self.as_colors is not None:
            return visitor._as_colors(self.as_colors)


scout_chartdefinition_api_GeoSecondaryPlotVisualizationOption.__name__ = "GeoSecondaryPlotVisualizationOption"
scout_chartdefinition_api_GeoSecondaryPlotVisualizationOption.__qualname__ = "GeoSecondaryPlotVisualizationOption"
scout_chartdefinition_api_GeoSecondaryPlotVisualizationOption.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoSecondaryPlotVisualizationOptionVisitor:

    @abstractmethod
    def _as_colors(self, as_colors: "scout_chartdefinition_api_ValueToColorMap") -> Any:
        pass


scout_chartdefinition_api_GeoSecondaryPlotVisualizationOptionVisitor.__name__ = "GeoSecondaryPlotVisualizationOptionVisitor"
scout_chartdefinition_api_GeoSecondaryPlotVisualizationOptionVisitor.__qualname__ = "GeoSecondaryPlotVisualizationOptionVisitor"
scout_chartdefinition_api_GeoSecondaryPlotVisualizationOptionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoVizDefinition(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_GeoVizDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_GeoVizDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_GeoVizDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_GeoVizDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_GeoVizDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_GeoVizDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_GeoVizDefinition.__name__ = "GeoVizDefinition"
scout_chartdefinition_api_GeoVizDefinition.__qualname__ = "GeoVizDefinition"
scout_chartdefinition_api_GeoVizDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoVizDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_GeoVizDefinitionV1") -> Any:
        pass


scout_chartdefinition_api_GeoVizDefinitionVisitor.__name__ = "GeoVizDefinitionVisitor"
scout_chartdefinition_api_GeoVizDefinitionVisitor.__qualname__ = "GeoVizDefinitionVisitor"
scout_chartdefinition_api_GeoVizDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_GeoVizDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'plots': ConjureFieldDefinition('plots', List[scout_chartdefinition_api_GeoPlotFromLatLong]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'custom_features': ConjureFieldDefinition('customFeatures', List[scout_chartdefinition_api_GeoCustomFeature])
        }

    __slots__: List[str] = ['_plots', '_title', '_custom_features']

    def __init__(self, custom_features: List["scout_chartdefinition_api_GeoCustomFeature"], plots: List["scout_chartdefinition_api_GeoPlotFromLatLong"], title: Optional[str] = None) -> None:
        self._plots = plots
        self._title = title
        self._custom_features = custom_features

    @builtins.property
    def plots(self) -> List["scout_chartdefinition_api_GeoPlotFromLatLong"]:
        return self._plots

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def custom_features(self) -> List["scout_chartdefinition_api_GeoCustomFeature"]:
        return self._custom_features


scout_chartdefinition_api_GeoVizDefinitionV1.__name__ = "GeoVizDefinitionV1"
scout_chartdefinition_api_GeoVizDefinitionV1.__qualname__ = "GeoVizDefinitionV1"
scout_chartdefinition_api_GeoVizDefinitionV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_HistogramChartDefinition(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_HistogramChartDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_HistogramChartDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_HistogramChartDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_HistogramChartDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_HistogramChartDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_HistogramChartDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_HistogramChartDefinition.__name__ = "HistogramChartDefinition"
scout_chartdefinition_api_HistogramChartDefinition.__qualname__ = "HistogramChartDefinition"
scout_chartdefinition_api_HistogramChartDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_HistogramChartDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_HistogramChartDefinitionV1") -> Any:
        pass


scout_chartdefinition_api_HistogramChartDefinitionVisitor.__name__ = "HistogramChartDefinitionVisitor"
scout_chartdefinition_api_HistogramChartDefinitionVisitor.__qualname__ = "HistogramChartDefinitionVisitor"
scout_chartdefinition_api_HistogramChartDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_HistogramChartDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'plots': ConjureFieldDefinition('plots', List[scout_chartdefinition_api_HistogramPlot]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'display_settings': ConjureFieldDefinition('displaySettings', scout_chartdefinition_api_HistogramDisplaySettings),
            'numeric_bucket_strategy': ConjureFieldDefinition('numericBucketStrategy', OptionalTypeWrapper[scout_compute_api_NumericHistogramBucketStrategy]),
            'value_axis': ConjureFieldDefinition('valueAxis', OptionalTypeWrapper[scout_chartdefinition_api_ValueAxis])
        }

    __slots__: List[str] = ['_plots', '_title', '_display_settings', '_numeric_bucket_strategy', '_value_axis']

    def __init__(self, display_settings: "scout_chartdefinition_api_HistogramDisplaySettings", plots: List["scout_chartdefinition_api_HistogramPlot"], numeric_bucket_strategy: Optional["scout_compute_api_NumericHistogramBucketStrategy"] = None, title: Optional[str] = None, value_axis: Optional["scout_chartdefinition_api_ValueAxis"] = None) -> None:
        self._plots = plots
        self._title = title
        self._display_settings = display_settings
        self._numeric_bucket_strategy = numeric_bucket_strategy
        self._value_axis = value_axis

    @builtins.property
    def plots(self) -> List["scout_chartdefinition_api_HistogramPlot"]:
        return self._plots

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def display_settings(self) -> "scout_chartdefinition_api_HistogramDisplaySettings":
        return self._display_settings

    @builtins.property
    def numeric_bucket_strategy(self) -> Optional["scout_compute_api_NumericHistogramBucketStrategy"]:
        return self._numeric_bucket_strategy

    @builtins.property
    def value_axis(self) -> Optional["scout_chartdefinition_api_ValueAxis"]:
        return self._value_axis


scout_chartdefinition_api_HistogramChartDefinitionV1.__name__ = "HistogramChartDefinitionV1"
scout_chartdefinition_api_HistogramChartDefinitionV1.__qualname__ = "HistogramChartDefinitionV1"
scout_chartdefinition_api_HistogramChartDefinitionV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_HistogramDisplaySettings(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'stacked': ConjureFieldDefinition('stacked', bool),
            'sort': ConjureFieldDefinition('sort', scout_chartdefinition_api_HistogramSortOrder)
        }

    __slots__: List[str] = ['_stacked', '_sort']

    def __init__(self, sort: "scout_chartdefinition_api_HistogramSortOrder", stacked: bool) -> None:
        self._stacked = stacked
        self._sort = sort

    @builtins.property
    def stacked(self) -> bool:
        return self._stacked

    @builtins.property
    def sort(self) -> "scout_chartdefinition_api_HistogramSortOrder":
        return self._sort


scout_chartdefinition_api_HistogramDisplaySettings.__name__ = "HistogramDisplaySettings"
scout_chartdefinition_api_HistogramDisplaySettings.__qualname__ = "HistogramDisplaySettings"
scout_chartdefinition_api_HistogramDisplaySettings.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_HistogramPlot(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variable_name': ConjureFieldDefinition('variableName', scout_channelvariables_api_ChannelVariableName),
            'enabled': ConjureFieldDefinition('enabled', OptionalTypeWrapper[bool]),
            'color': ConjureFieldDefinition('color', scout_api_HexColor)
        }

    __slots__: List[str] = ['_variable_name', '_enabled', '_color']

    def __init__(self, color: str, variable_name: str, enabled: Optional[bool] = None) -> None:
        self._variable_name = variable_name
        self._enabled = enabled
        self._color = color

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def enabled(self) -> Optional[bool]:
        return self._enabled

    @builtins.property
    def color(self) -> str:
        return self._color


scout_chartdefinition_api_HistogramPlot.__name__ = "HistogramPlot"
scout_chartdefinition_api_HistogramPlot.__qualname__ = "HistogramPlot"
scout_chartdefinition_api_HistogramPlot.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_HistogramSortOrder(ConjureEnumType):
    """
    The order in which the histogram should be sorted.
Default is VALUE_ASCENDING.
    """

    VALUE_ASCENDING = 'VALUE_ASCENDING'
    '''VALUE_ASCENDING'''
    VALUE_DESCENDING = 'VALUE_DESCENDING'
    '''VALUE_DESCENDING'''
    FREQUENCY_ASCENDING = 'FREQUENCY_ASCENDING'
    '''FREQUENCY_ASCENDING'''
    FREQUENCY_DESCENDING = 'FREQUENCY_DESCENDING'
    '''FREQUENCY_DESCENDING'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_chartdefinition_api_HistogramSortOrder.__name__ = "HistogramSortOrder"
scout_chartdefinition_api_HistogramSortOrder.__qualname__ = "HistogramSortOrder"
scout_chartdefinition_api_HistogramSortOrder.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_LineStyle(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_LineStyleV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_LineStyleV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_LineStyleV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_LineStyleV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_LineStyleVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_LineStyleVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_LineStyle.__name__ = "LineStyle"
scout_chartdefinition_api_LineStyle.__qualname__ = "LineStyle"
scout_chartdefinition_api_LineStyle.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_LineStyleVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_LineStyleV1") -> Any:
        pass


scout_chartdefinition_api_LineStyleVisitor.__name__ = "LineStyleVisitor"
scout_chartdefinition_api_LineStyleVisitor.__qualname__ = "LineStyleVisitor"
scout_chartdefinition_api_LineStyleVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_LineStyleV1(ConjureEnumType):

    SOLID = 'SOLID'
    '''SOLID'''
    DASH = 'DASH'
    '''DASH'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_chartdefinition_api_LineStyleV1.__name__ = "LineStyleV1"
scout_chartdefinition_api_LineStyleV1.__qualname__ = "LineStyleV1"
scout_chartdefinition_api_LineStyleV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_LineThreshold(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'value': ConjureFieldDefinition('value', float),
            'label': ConjureFieldDefinition('label', OptionalTypeWrapper[str]),
            'color': ConjureFieldDefinition('color', scout_api_HexColor),
            'line_style': ConjureFieldDefinition('lineStyle', scout_chartdefinition_api_ThresholdLineStyle)
        }

    __slots__: List[str] = ['_value', '_label', '_color', '_line_style']

    def __init__(self, color: str, line_style: "scout_chartdefinition_api_ThresholdLineStyle", value: float, label: Optional[str] = None) -> None:
        self._value = value
        self._label = label
        self._color = color
        self._line_style = line_style

    @builtins.property
    def value(self) -> float:
        return self._value

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def color(self) -> str:
        return self._color

    @builtins.property
    def line_style(self) -> "scout_chartdefinition_api_ThresholdLineStyle":
        return self._line_style


scout_chartdefinition_api_LineThreshold.__name__ = "LineThreshold"
scout_chartdefinition_api_LineThreshold.__qualname__ = "LineThreshold"
scout_chartdefinition_api_LineThreshold.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_LineThresholdGroup(ConjureBeanType):
    """
    Line thresholds are used to mark values or demarcate regions along a single axis.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'shading_config': ConjureFieldDefinition('shadingConfig', scout_chartdefinition_api_ThresholdShadingConfig),
            'lines': ConjureFieldDefinition('lines', List[scout_chartdefinition_api_LineThreshold]),
            'default_fill': ConjureFieldDefinition('defaultFill', OptionalTypeWrapper[scout_chartdefinition_api_DefaultFill])
        }

    __slots__: List[str] = ['_shading_config', '_lines', '_default_fill']

    def __init__(self, lines: List["scout_chartdefinition_api_LineThreshold"], shading_config: "scout_chartdefinition_api_ThresholdShadingConfig", default_fill: Optional["scout_chartdefinition_api_DefaultFill"] = None) -> None:
        self._shading_config = shading_config
        self._lines = lines
        self._default_fill = default_fill

    @builtins.property
    def shading_config(self) -> "scout_chartdefinition_api_ThresholdShadingConfig":
        return self._shading_config

    @builtins.property
    def lines(self) -> List["scout_chartdefinition_api_LineThreshold"]:
        return self._lines

    @builtins.property
    def default_fill(self) -> Optional["scout_chartdefinition_api_DefaultFill"]:
        """
        To supplement a set of line thresholds, the default fill configures how the remaining space (either
above or below) should be colored. Transparent if empty.
        """
        return self._default_fill


scout_chartdefinition_api_LineThresholdGroup.__name__ = "LineThresholdGroup"
scout_chartdefinition_api_LineThresholdGroup.__qualname__ = "LineThresholdGroup"
scout_chartdefinition_api_LineThresholdGroup.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_NumericBarGaugeVisualisation(ConjureBeanType):
    """
    The settings for a bar gauge visualisation.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'low_threshold': ConjureFieldDefinition('lowThreshold', float),
            'high_threshold': ConjureFieldDefinition('highThreshold', float)
        }

    __slots__: List[str] = ['_low_threshold', '_high_threshold']

    def __init__(self, high_threshold: float, low_threshold: float) -> None:
        self._low_threshold = low_threshold
        self._high_threshold = high_threshold

    @builtins.property
    def low_threshold(self) -> float:
        return self._low_threshold

    @builtins.property
    def high_threshold(self) -> float:
        return self._high_threshold


scout_chartdefinition_api_NumericBarGaugeVisualisation.__name__ = "NumericBarGaugeVisualisation"
scout_chartdefinition_api_NumericBarGaugeVisualisation.__qualname__ = "NumericBarGaugeVisualisation"
scout_chartdefinition_api_NumericBarGaugeVisualisation.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_NumericRawVisualisation(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_chartdefinition_api_NumericRawVisualisation.__name__ = "NumericRawVisualisation"
scout_chartdefinition_api_NumericRawVisualisation.__qualname__ = "NumericRawVisualisation"
scout_chartdefinition_api_NumericRawVisualisation.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_NumericValueChannel(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variable_name': ConjureFieldDefinition('variableName', scout_channelvariables_api_ChannelVariableName),
            'visualisation': ConjureFieldDefinition('visualisation', scout_chartdefinition_api_NumericValueVisualisation)
        }

    __slots__: List[str] = ['_variable_name', '_visualisation']

    def __init__(self, variable_name: str, visualisation: "scout_chartdefinition_api_NumericValueVisualisation") -> None:
        self._variable_name = variable_name
        self._visualisation = visualisation

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def visualisation(self) -> "scout_chartdefinition_api_NumericValueVisualisation":
        return self._visualisation


scout_chartdefinition_api_NumericValueChannel.__name__ = "NumericValueChannel"
scout_chartdefinition_api_NumericValueChannel.__qualname__ = "NumericValueChannel"
scout_chartdefinition_api_NumericValueChannel.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_NumericValueVisualisation(ConjureUnionType):
    _raw: Optional["scout_chartdefinition_api_NumericRawVisualisation"] = None
    _bar_gauge: Optional["scout_chartdefinition_api_NumericBarGaugeVisualisation"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'raw': ConjureFieldDefinition('raw', scout_chartdefinition_api_NumericRawVisualisation),
            'bar_gauge': ConjureFieldDefinition('barGauge', scout_chartdefinition_api_NumericBarGaugeVisualisation)
        }

    def __init__(
            self,
            raw: Optional["scout_chartdefinition_api_NumericRawVisualisation"] = None,
            bar_gauge: Optional["scout_chartdefinition_api_NumericBarGaugeVisualisation"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (raw is not None) + (bar_gauge is not None) != 1:
                raise ValueError('a union must contain a single member')

            if raw is not None:
                self._raw = raw
                self._type = 'raw'
            if bar_gauge is not None:
                self._bar_gauge = bar_gauge
                self._type = 'barGauge'

        elif type_of_union == 'raw':
            if raw is None:
                raise ValueError('a union value must not be None')
            self._raw = raw
            self._type = 'raw'
        elif type_of_union == 'barGauge':
            if bar_gauge is None:
                raise ValueError('a union value must not be None')
            self._bar_gauge = bar_gauge
            self._type = 'barGauge'

    @builtins.property
    def raw(self) -> Optional["scout_chartdefinition_api_NumericRawVisualisation"]:
        return self._raw

    @builtins.property
    def bar_gauge(self) -> Optional["scout_chartdefinition_api_NumericBarGaugeVisualisation"]:
        return self._bar_gauge

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_NumericValueVisualisationVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_NumericValueVisualisationVisitor'.format(visitor.__class__.__name__))
        if self._type == 'raw' and self.raw is not None:
            return visitor._raw(self.raw)
        if self._type == 'barGauge' and self.bar_gauge is not None:
            return visitor._bar_gauge(self.bar_gauge)


scout_chartdefinition_api_NumericValueVisualisation.__name__ = "NumericValueVisualisation"
scout_chartdefinition_api_NumericValueVisualisation.__qualname__ = "NumericValueVisualisation"
scout_chartdefinition_api_NumericValueVisualisation.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_NumericValueVisualisationVisitor:

    @abstractmethod
    def _raw(self, raw: "scout_chartdefinition_api_NumericRawVisualisation") -> Any:
        pass

    @abstractmethod
    def _bar_gauge(self, bar_gauge: "scout_chartdefinition_api_NumericBarGaugeVisualisation") -> Any:
        pass


scout_chartdefinition_api_NumericValueVisualisationVisitor.__name__ = "NumericValueVisualisationVisitor"
scout_chartdefinition_api_NumericValueVisualisationVisitor.__qualname__ = "NumericValueVisualisationVisitor"
scout_chartdefinition_api_NumericValueVisualisationVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_RangeRawVisualisation(ConjureBeanType):
    """
    The settings for a raw range visualisation.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'range_color': ConjureFieldDefinition('rangeColor', OptionalTypeWrapper[scout_api_HexColor]),
            'no_range_color': ConjureFieldDefinition('noRangeColor', OptionalTypeWrapper[scout_api_HexColor])
        }

    __slots__: List[str] = ['_range_color', '_no_range_color']

    def __init__(self, no_range_color: Optional[str] = None, range_color: Optional[str] = None) -> None:
        self._range_color = range_color
        self._no_range_color = no_range_color

    @builtins.property
    def range_color(self) -> Optional[str]:
        return self._range_color

    @builtins.property
    def no_range_color(self) -> Optional[str]:
        return self._no_range_color


scout_chartdefinition_api_RangeRawVisualisation.__name__ = "RangeRawVisualisation"
scout_chartdefinition_api_RangeRawVisualisation.__qualname__ = "RangeRawVisualisation"
scout_chartdefinition_api_RangeRawVisualisation.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_RangeValueChannel(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variable_name': ConjureFieldDefinition('variableName', scout_channelvariables_api_ChannelVariableName),
            'visualisation': ConjureFieldDefinition('visualisation', scout_chartdefinition_api_RangeValueVisualisation)
        }

    __slots__: List[str] = ['_variable_name', '_visualisation']

    def __init__(self, variable_name: str, visualisation: "scout_chartdefinition_api_RangeValueVisualisation") -> None:
        self._variable_name = variable_name
        self._visualisation = visualisation

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def visualisation(self) -> "scout_chartdefinition_api_RangeValueVisualisation":
        return self._visualisation


scout_chartdefinition_api_RangeValueChannel.__name__ = "RangeValueChannel"
scout_chartdefinition_api_RangeValueChannel.__qualname__ = "RangeValueChannel"
scout_chartdefinition_api_RangeValueChannel.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_RangeValueVisualisation(ConjureUnionType):
    _raw: Optional["scout_chartdefinition_api_RangeRawVisualisation"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'raw': ConjureFieldDefinition('raw', scout_chartdefinition_api_RangeRawVisualisation)
        }

    def __init__(
            self,
            raw: Optional["scout_chartdefinition_api_RangeRawVisualisation"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (raw is not None) != 1:
                raise ValueError('a union must contain a single member')

            if raw is not None:
                self._raw = raw
                self._type = 'raw'

        elif type_of_union == 'raw':
            if raw is None:
                raise ValueError('a union value must not be None')
            self._raw = raw
            self._type = 'raw'

    @builtins.property
    def raw(self) -> Optional["scout_chartdefinition_api_RangeRawVisualisation"]:
        return self._raw

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_RangeValueVisualisationVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_RangeValueVisualisationVisitor'.format(visitor.__class__.__name__))
        if self._type == 'raw' and self.raw is not None:
            return visitor._raw(self.raw)


scout_chartdefinition_api_RangeValueVisualisation.__name__ = "RangeValueVisualisation"
scout_chartdefinition_api_RangeValueVisualisation.__qualname__ = "RangeValueVisualisation"
scout_chartdefinition_api_RangeValueVisualisation.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_RangeValueVisualisationVisitor:

    @abstractmethod
    def _raw(self, raw: "scout_chartdefinition_api_RangeRawVisualisation") -> Any:
        pass


scout_chartdefinition_api_RangeValueVisualisationVisitor.__name__ = "RangeValueVisualisationVisitor"
scout_chartdefinition_api_RangeValueVisualisationVisitor.__qualname__ = "RangeValueVisualisationVisitor"
scout_chartdefinition_api_RangeValueVisualisationVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ThresholdLineStyle(ConjureEnumType):

    DOTTED = 'DOTTED'
    '''DOTTED'''
    SOLID = 'SOLID'
    '''SOLID'''
    NONE = 'NONE'
    '''NONE'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_chartdefinition_api_ThresholdLineStyle.__name__ = "ThresholdLineStyle"
scout_chartdefinition_api_ThresholdLineStyle.__qualname__ = "ThresholdLineStyle"
scout_chartdefinition_api_ThresholdLineStyle.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ThresholdShadingConfig(ConjureEnumType):
    """
    Specifies how areas of a plot should be shaded in relation to the defined threshold lines. Will eventually
include other shading configurations, like a buffer zone.
    """

    ABOVE = 'ABOVE'
    '''ABOVE'''
    BELOW = 'BELOW'
    '''BELOW'''
    NONE = 'NONE'
    '''NONE'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_chartdefinition_api_ThresholdShadingConfig.__name__ = "ThresholdShadingConfig"
scout_chartdefinition_api_ThresholdShadingConfig.__qualname__ = "ThresholdShadingConfig"
scout_chartdefinition_api_ThresholdShadingConfig.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_TimeSeriesChartDefinition(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_TimeSeriesChartDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_TimeSeriesChartDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_TimeSeriesChartDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_TimeSeriesChartDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_TimeSeriesChartDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_TimeSeriesChartDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_TimeSeriesChartDefinition.__name__ = "TimeSeriesChartDefinition"
scout_chartdefinition_api_TimeSeriesChartDefinition.__qualname__ = "TimeSeriesChartDefinition"
scout_chartdefinition_api_TimeSeriesChartDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_TimeSeriesChartDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_TimeSeriesChartDefinitionV1") -> Any:
        pass


scout_chartdefinition_api_TimeSeriesChartDefinitionVisitor.__name__ = "TimeSeriesChartDefinitionVisitor"
scout_chartdefinition_api_TimeSeriesChartDefinitionVisitor.__qualname__ = "TimeSeriesChartDefinitionVisitor"
scout_chartdefinition_api_TimeSeriesChartDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_TimeSeriesChartDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rows': ConjureFieldDefinition('rows', List[scout_chartdefinition_api_TimeSeriesRow]),
            'comparison_run_groups': ConjureFieldDefinition('comparisonRunGroups', List[scout_comparisonrun_api_ComparisonRunGroup]),
            'events': ConjureFieldDefinition('events', List[scout_chartdefinition_api_Event]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'value_axes': ConjureFieldDefinition('valueAxes', List[scout_chartdefinition_api_ValueAxis]),
            'thresholds': ConjureFieldDefinition('thresholds', OptionalTypeWrapper[List[scout_chartdefinition_api_AxisThresholdVisualization]])
        }

    __slots__: List[str] = ['_rows', '_comparison_run_groups', '_events', '_title', '_value_axes', '_thresholds']

    def __init__(self, comparison_run_groups: List["scout_comparisonrun_api_ComparisonRunGroup"], events: List["scout_chartdefinition_api_Event"], rows: List["scout_chartdefinition_api_TimeSeriesRow"], value_axes: List["scout_chartdefinition_api_ValueAxis"], thresholds: Optional[List["scout_chartdefinition_api_AxisThresholdVisualization"]] = None, title: Optional[str] = None) -> None:
        self._rows = rows
        self._comparison_run_groups = comparison_run_groups
        self._events = events
        self._title = title
        self._value_axes = value_axes
        self._thresholds = thresholds

    @builtins.property
    def rows(self) -> List["scout_chartdefinition_api_TimeSeriesRow"]:
        return self._rows

    @builtins.property
    def comparison_run_groups(self) -> List["scout_comparisonrun_api_ComparisonRunGroup"]:
        return self._comparison_run_groups

    @builtins.property
    def events(self) -> List["scout_chartdefinition_api_Event"]:
        return self._events

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def value_axes(self) -> List["scout_chartdefinition_api_ValueAxis"]:
        return self._value_axes

    @builtins.property
    def thresholds(self) -> Optional[List["scout_chartdefinition_api_AxisThresholdVisualization"]]:
        return self._thresholds


scout_chartdefinition_api_TimeSeriesChartDefinitionV1.__name__ = "TimeSeriesChartDefinitionV1"
scout_chartdefinition_api_TimeSeriesChartDefinitionV1.__qualname__ = "TimeSeriesChartDefinitionV1"
scout_chartdefinition_api_TimeSeriesChartDefinitionV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_TimeSeriesPlot(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variable_name': ConjureFieldDefinition('variableName', scout_channelvariables_api_ChannelVariableName),
            'enabled': ConjureFieldDefinition('enabled', OptionalTypeWrapper[bool]),
            'y_axis_id': ConjureFieldDefinition('yAxisId', scout_chartdefinition_api_AxisId),
            'color': ConjureFieldDefinition('color', scout_api_HexColor),
            'line_style': ConjureFieldDefinition('lineStyle', scout_chartdefinition_api_LineStyle)
        }

    __slots__: List[str] = ['_variable_name', '_enabled', '_y_axis_id', '_color', '_line_style']

    def __init__(self, color: str, line_style: "scout_chartdefinition_api_LineStyle", variable_name: str, y_axis_id: str, enabled: Optional[bool] = None) -> None:
        self._variable_name = variable_name
        self._enabled = enabled
        self._y_axis_id = y_axis_id
        self._color = color
        self._line_style = line_style

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def enabled(self) -> Optional[bool]:
        return self._enabled

    @builtins.property
    def y_axis_id(self) -> str:
        return self._y_axis_id

    @builtins.property
    def color(self) -> str:
        return self._color

    @builtins.property
    def line_style(self) -> "scout_chartdefinition_api_LineStyle":
        return self._line_style


scout_chartdefinition_api_TimeSeriesPlot.__name__ = "TimeSeriesPlot"
scout_chartdefinition_api_TimeSeriesPlot.__qualname__ = "TimeSeriesPlot"
scout_chartdefinition_api_TimeSeriesPlot.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_TimeSeriesRow(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'plots': ConjureFieldDefinition('plots', List[scout_chartdefinition_api_TimeSeriesPlot]),
            'row_flex_size': ConjureFieldDefinition('rowFlexSize', float)
        }

    __slots__: List[str] = ['_title', '_plots', '_row_flex_size']

    def __init__(self, plots: List["scout_chartdefinition_api_TimeSeriesPlot"], row_flex_size: float, title: Optional[str] = None) -> None:
        self._title = title
        self._plots = plots
        self._row_flex_size = row_flex_size

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def plots(self) -> List["scout_chartdefinition_api_TimeSeriesPlot"]:
        return self._plots

    @builtins.property
    def row_flex_size(self) -> float:
        return self._row_flex_size


scout_chartdefinition_api_TimeSeriesRow.__name__ = "TimeSeriesRow"
scout_chartdefinition_api_TimeSeriesRow.__qualname__ = "TimeSeriesRow"
scout_chartdefinition_api_TimeSeriesRow.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ValueAxis(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'id': ConjureFieldDefinition('id', str),
            'title': ConjureFieldDefinition('title', str),
            'display_options': ConjureFieldDefinition('displayOptions', scout_chartdefinition_api_AxisDisplayOptions),
            'range': ConjureFieldDefinition('range', scout_chartdefinition_api_AxisRange),
            'limit': ConjureFieldDefinition('limit', scout_chartdefinition_api_AxisRange),
            'position': ConjureFieldDefinition('position', scout_chartdefinition_api_AxisPosition),
            'domain_type': ConjureFieldDefinition('domainType', scout_chartdefinition_api_AxisDomainType)
        }

    __slots__: List[str] = ['_id', '_title', '_display_options', '_range', '_limit', '_position', '_domain_type']

    def __init__(self, display_options: "scout_chartdefinition_api_AxisDisplayOptions", domain_type: "scout_chartdefinition_api_AxisDomainType", id: str, limit: "scout_chartdefinition_api_AxisRange", position: "scout_chartdefinition_api_AxisPosition", range: "scout_chartdefinition_api_AxisRange", title: str) -> None:
        self._id = id
        self._title = title
        self._display_options = display_options
        self._range = range
        self._limit = limit
        self._position = position
        self._domain_type = domain_type

    @builtins.property
    def id(self) -> str:
        return self._id

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def display_options(self) -> "scout_chartdefinition_api_AxisDisplayOptions":
        return self._display_options

    @builtins.property
    def range(self) -> "scout_chartdefinition_api_AxisRange":
        return self._range

    @builtins.property
    def limit(self) -> "scout_chartdefinition_api_AxisRange":
        return self._limit

    @builtins.property
    def position(self) -> "scout_chartdefinition_api_AxisPosition":
        return self._position

    @builtins.property
    def domain_type(self) -> "scout_chartdefinition_api_AxisDomainType":
        return self._domain_type


scout_chartdefinition_api_ValueAxis.__name__ = "ValueAxis"
scout_chartdefinition_api_ValueAxis.__qualname__ = "ValueAxis"
scout_chartdefinition_api_ValueAxis.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ValueTableChannel(ConjureUnionType):
    _numeric: Optional["scout_chartdefinition_api_NumericValueChannel"] = None
    _enum: Optional["scout_chartdefinition_api_EnumValueChannel"] = None
    _range: Optional["scout_chartdefinition_api_RangeValueChannel"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'numeric': ConjureFieldDefinition('numeric', scout_chartdefinition_api_NumericValueChannel),
            'enum': ConjureFieldDefinition('enum', scout_chartdefinition_api_EnumValueChannel),
            'range': ConjureFieldDefinition('range', scout_chartdefinition_api_RangeValueChannel)
        }

    def __init__(
            self,
            numeric: Optional["scout_chartdefinition_api_NumericValueChannel"] = None,
            enum: Optional["scout_chartdefinition_api_EnumValueChannel"] = None,
            range: Optional["scout_chartdefinition_api_RangeValueChannel"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (numeric is not None) + (enum is not None) + (range is not None) != 1:
                raise ValueError('a union must contain a single member')

            if numeric is not None:
                self._numeric = numeric
                self._type = 'numeric'
            if enum is not None:
                self._enum = enum
                self._type = 'enum'
            if range is not None:
                self._range = range
                self._type = 'range'

        elif type_of_union == 'numeric':
            if numeric is None:
                raise ValueError('a union value must not be None')
            self._numeric = numeric
            self._type = 'numeric'
        elif type_of_union == 'enum':
            if enum is None:
                raise ValueError('a union value must not be None')
            self._enum = enum
            self._type = 'enum'
        elif type_of_union == 'range':
            if range is None:
                raise ValueError('a union value must not be None')
            self._range = range
            self._type = 'range'

    @builtins.property
    def numeric(self) -> Optional["scout_chartdefinition_api_NumericValueChannel"]:
        return self._numeric

    @builtins.property
    def enum(self) -> Optional["scout_chartdefinition_api_EnumValueChannel"]:
        return self._enum

    @builtins.property
    def range(self) -> Optional["scout_chartdefinition_api_RangeValueChannel"]:
        return self._range

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_ValueTableChannelVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_ValueTableChannelVisitor'.format(visitor.__class__.__name__))
        if self._type == 'numeric' and self.numeric is not None:
            return visitor._numeric(self.numeric)
        if self._type == 'enum' and self.enum is not None:
            return visitor._enum(self.enum)
        if self._type == 'range' and self.range is not None:
            return visitor._range(self.range)


scout_chartdefinition_api_ValueTableChannel.__name__ = "ValueTableChannel"
scout_chartdefinition_api_ValueTableChannel.__qualname__ = "ValueTableChannel"
scout_chartdefinition_api_ValueTableChannel.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ValueTableChannelVisitor:

    @abstractmethod
    def _numeric(self, numeric: "scout_chartdefinition_api_NumericValueChannel") -> Any:
        pass

    @abstractmethod
    def _enum(self, enum: "scout_chartdefinition_api_EnumValueChannel") -> Any:
        pass

    @abstractmethod
    def _range(self, range: "scout_chartdefinition_api_RangeValueChannel") -> Any:
        pass


scout_chartdefinition_api_ValueTableChannelVisitor.__name__ = "ValueTableChannelVisitor"
scout_chartdefinition_api_ValueTableChannelVisitor.__qualname__ = "ValueTableChannelVisitor"
scout_chartdefinition_api_ValueTableChannelVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ValueTableDefinition(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_ValueTableDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_ValueTableDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_ValueTableDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_ValueTableDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_ValueTableDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_ValueTableDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_ValueTableDefinition.__name__ = "ValueTableDefinition"
scout_chartdefinition_api_ValueTableDefinition.__qualname__ = "ValueTableDefinition"
scout_chartdefinition_api_ValueTableDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ValueTableDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_ValueTableDefinitionV1") -> Any:
        pass


scout_chartdefinition_api_ValueTableDefinitionVisitor.__name__ = "ValueTableDefinitionVisitor"
scout_chartdefinition_api_ValueTableDefinitionVisitor.__qualname__ = "ValueTableDefinitionVisitor"
scout_chartdefinition_api_ValueTableDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ValueTableDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'channels': ConjureFieldDefinition('channels', List[scout_chartdefinition_api_ValueTableChannel])
        }

    __slots__: List[str] = ['_title', '_channels']

    def __init__(self, channels: List["scout_chartdefinition_api_ValueTableChannel"], title: Optional[str] = None) -> None:
        self._title = title
        self._channels = channels

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def channels(self) -> List["scout_chartdefinition_api_ValueTableChannel"]:
        """
        Each channel to be displayed. Based on the data type, the visualisation
options and settings will be different.
        """
        return self._channels


scout_chartdefinition_api_ValueTableDefinitionV1.__name__ = "ValueTableDefinitionV1"
scout_chartdefinition_api_ValueTableDefinitionV1.__qualname__ = "ValueTableDefinitionV1"
scout_chartdefinition_api_ValueTableDefinitionV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ValueToColorMap(ConjureUnionType):
    """Specifies an assignment of colors across several values."""
    _numeric: Optional[Dict[str, float]] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'numeric': ConjureFieldDefinition('numeric', Dict[scout_api_HexColor, float])
        }

    def __init__(
            self,
            numeric: Optional[Dict[str, float]] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (numeric is not None) != 1:
                raise ValueError('a union must contain a single member')

            if numeric is not None:
                self._numeric = numeric
                self._type = 'numeric'

        elif type_of_union == 'numeric':
            if numeric is None:
                raise ValueError('a union value must not be None')
            self._numeric = numeric
            self._type = 'numeric'

    @builtins.property
    def numeric(self) -> Optional[Dict[str, float]]:
        return self._numeric

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_ValueToColorMapVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_ValueToColorMapVisitor'.format(visitor.__class__.__name__))
        if self._type == 'numeric' and self.numeric is not None:
            return visitor._numeric(self.numeric)


scout_chartdefinition_api_ValueToColorMap.__name__ = "ValueToColorMap"
scout_chartdefinition_api_ValueToColorMap.__qualname__ = "ValueToColorMap"
scout_chartdefinition_api_ValueToColorMap.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_ValueToColorMapVisitor:

    @abstractmethod
    def _numeric(self, numeric: Dict[str, float]) -> Any:
        pass


scout_chartdefinition_api_ValueToColorMapVisitor.__name__ = "ValueToColorMapVisitor"
scout_chartdefinition_api_ValueToColorMapVisitor.__qualname__ = "ValueToColorMapVisitor"
scout_chartdefinition_api_ValueToColorMapVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_VideoVizDefinition(ConjureUnionType):
    _v1: Optional["scout_chartdefinition_api_VideoVizDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_chartdefinition_api_VideoVizDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_chartdefinition_api_VideoVizDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_chartdefinition_api_VideoVizDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_VideoVizDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_VideoVizDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_chartdefinition_api_VideoVizDefinition.__name__ = "VideoVizDefinition"
scout_chartdefinition_api_VideoVizDefinition.__qualname__ = "VideoVizDefinition"
scout_chartdefinition_api_VideoVizDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_VideoVizDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_chartdefinition_api_VideoVizDefinitionV1") -> Any:
        pass


scout_chartdefinition_api_VideoVizDefinitionVisitor.__name__ = "VideoVizDefinitionVisitor"
scout_chartdefinition_api_VideoVizDefinitionVisitor.__qualname__ = "VideoVizDefinitionVisitor"
scout_chartdefinition_api_VideoVizDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_VideoVizDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'events': ConjureFieldDefinition('events', List[scout_chartdefinition_api_Event]),
            'comparison_run_groups': ConjureFieldDefinition('comparisonRunGroups', List[scout_comparisonrun_api_ComparisonRunGroup]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'ref_name': ConjureFieldDefinition('refName', OptionalTypeWrapper[scout_chartdefinition_api_DataSourceRefName])
        }

    __slots__: List[str] = ['_events', '_comparison_run_groups', '_title', '_ref_name']

    def __init__(self, comparison_run_groups: List["scout_comparisonrun_api_ComparisonRunGroup"], events: List["scout_chartdefinition_api_Event"], ref_name: Optional[str] = None, title: Optional[str] = None) -> None:
        self._events = events
        self._comparison_run_groups = comparison_run_groups
        self._title = title
        self._ref_name = ref_name

    @builtins.property
    def events(self) -> List["scout_chartdefinition_api_Event"]:
        return self._events

    @builtins.property
    def comparison_run_groups(self) -> List["scout_comparisonrun_api_ComparisonRunGroup"]:
        return self._comparison_run_groups

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def ref_name(self) -> Optional[str]:
        return self._ref_name


scout_chartdefinition_api_VideoVizDefinitionV1.__name__ = "VideoVizDefinitionV1"
scout_chartdefinition_api_VideoVizDefinitionV1.__qualname__ = "VideoVizDefinitionV1"
scout_chartdefinition_api_VideoVizDefinitionV1.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_VizDefinition(ConjureUnionType):
    _time_series: Optional["scout_chartdefinition_api_TimeSeriesChartDefinition"] = None
    _cartesian: Optional["scout_chartdefinition_api_CartesianChartDefinition"] = None
    _frequency: Optional["scout_chartdefinition_api_FrequencyChartDefinition"] = None
    _histogram: Optional["scout_chartdefinition_api_HistogramChartDefinition"] = None
    _video: Optional["scout_chartdefinition_api_VideoVizDefinition"] = None
    _value_table: Optional["scout_chartdefinition_api_ValueTableDefinition"] = None
    _geo: Optional["scout_chartdefinition_api_GeoVizDefinition"] = None
    _checklist: Optional["scout_chartdefinition_api_ChecklistChartDefinition"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'time_series': ConjureFieldDefinition('timeSeries', scout_chartdefinition_api_TimeSeriesChartDefinition),
            'cartesian': ConjureFieldDefinition('cartesian', scout_chartdefinition_api_CartesianChartDefinition),
            'frequency': ConjureFieldDefinition('frequency', scout_chartdefinition_api_FrequencyChartDefinition),
            'histogram': ConjureFieldDefinition('histogram', scout_chartdefinition_api_HistogramChartDefinition),
            'video': ConjureFieldDefinition('video', scout_chartdefinition_api_VideoVizDefinition),
            'value_table': ConjureFieldDefinition('valueTable', scout_chartdefinition_api_ValueTableDefinition),
            'geo': ConjureFieldDefinition('geo', scout_chartdefinition_api_GeoVizDefinition),
            'checklist': ConjureFieldDefinition('checklist', scout_chartdefinition_api_ChecklistChartDefinition)
        }

    def __init__(
            self,
            time_series: Optional["scout_chartdefinition_api_TimeSeriesChartDefinition"] = None,
            cartesian: Optional["scout_chartdefinition_api_CartesianChartDefinition"] = None,
            frequency: Optional["scout_chartdefinition_api_FrequencyChartDefinition"] = None,
            histogram: Optional["scout_chartdefinition_api_HistogramChartDefinition"] = None,
            video: Optional["scout_chartdefinition_api_VideoVizDefinition"] = None,
            value_table: Optional["scout_chartdefinition_api_ValueTableDefinition"] = None,
            geo: Optional["scout_chartdefinition_api_GeoVizDefinition"] = None,
            checklist: Optional["scout_chartdefinition_api_ChecklistChartDefinition"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (time_series is not None) + (cartesian is not None) + (frequency is not None) + (histogram is not None) + (video is not None) + (value_table is not None) + (geo is not None) + (checklist is not None) != 1:
                raise ValueError('a union must contain a single member')

            if time_series is not None:
                self._time_series = time_series
                self._type = 'timeSeries'
            if cartesian is not None:
                self._cartesian = cartesian
                self._type = 'cartesian'
            if frequency is not None:
                self._frequency = frequency
                self._type = 'frequency'
            if histogram is not None:
                self._histogram = histogram
                self._type = 'histogram'
            if video is not None:
                self._video = video
                self._type = 'video'
            if value_table is not None:
                self._value_table = value_table
                self._type = 'valueTable'
            if geo is not None:
                self._geo = geo
                self._type = 'geo'
            if checklist is not None:
                self._checklist = checklist
                self._type = 'checklist'

        elif type_of_union == 'timeSeries':
            if time_series is None:
                raise ValueError('a union value must not be None')
            self._time_series = time_series
            self._type = 'timeSeries'
        elif type_of_union == 'cartesian':
            if cartesian is None:
                raise ValueError('a union value must not be None')
            self._cartesian = cartesian
            self._type = 'cartesian'
        elif type_of_union == 'frequency':
            if frequency is None:
                raise ValueError('a union value must not be None')
            self._frequency = frequency
            self._type = 'frequency'
        elif type_of_union == 'histogram':
            if histogram is None:
                raise ValueError('a union value must not be None')
            self._histogram = histogram
            self._type = 'histogram'
        elif type_of_union == 'video':
            if video is None:
                raise ValueError('a union value must not be None')
            self._video = video
            self._type = 'video'
        elif type_of_union == 'valueTable':
            if value_table is None:
                raise ValueError('a union value must not be None')
            self._value_table = value_table
            self._type = 'valueTable'
        elif type_of_union == 'geo':
            if geo is None:
                raise ValueError('a union value must not be None')
            self._geo = geo
            self._type = 'geo'
        elif type_of_union == 'checklist':
            if checklist is None:
                raise ValueError('a union value must not be None')
            self._checklist = checklist
            self._type = 'checklist'

    @builtins.property
    def time_series(self) -> Optional["scout_chartdefinition_api_TimeSeriesChartDefinition"]:
        return self._time_series

    @builtins.property
    def cartesian(self) -> Optional["scout_chartdefinition_api_CartesianChartDefinition"]:
        return self._cartesian

    @builtins.property
    def frequency(self) -> Optional["scout_chartdefinition_api_FrequencyChartDefinition"]:
        return self._frequency

    @builtins.property
    def histogram(self) -> Optional["scout_chartdefinition_api_HistogramChartDefinition"]:
        return self._histogram

    @builtins.property
    def video(self) -> Optional["scout_chartdefinition_api_VideoVizDefinition"]:
        return self._video

    @builtins.property
    def value_table(self) -> Optional["scout_chartdefinition_api_ValueTableDefinition"]:
        return self._value_table

    @builtins.property
    def geo(self) -> Optional["scout_chartdefinition_api_GeoVizDefinition"]:
        return self._geo

    @builtins.property
    def checklist(self) -> Optional["scout_chartdefinition_api_ChecklistChartDefinition"]:
        return self._checklist

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_chartdefinition_api_VizDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_chartdefinition_api_VizDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'timeSeries' and self.time_series is not None:
            return visitor._time_series(self.time_series)
        if self._type == 'cartesian' and self.cartesian is not None:
            return visitor._cartesian(self.cartesian)
        if self._type == 'frequency' and self.frequency is not None:
            return visitor._frequency(self.frequency)
        if self._type == 'histogram' and self.histogram is not None:
            return visitor._histogram(self.histogram)
        if self._type == 'video' and self.video is not None:
            return visitor._video(self.video)
        if self._type == 'valueTable' and self.value_table is not None:
            return visitor._value_table(self.value_table)
        if self._type == 'geo' and self.geo is not None:
            return visitor._geo(self.geo)
        if self._type == 'checklist' and self.checklist is not None:
            return visitor._checklist(self.checklist)


scout_chartdefinition_api_VizDefinition.__name__ = "VizDefinition"
scout_chartdefinition_api_VizDefinition.__qualname__ = "VizDefinition"
scout_chartdefinition_api_VizDefinition.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_chartdefinition_api_VizDefinitionVisitor:

    @abstractmethod
    def _time_series(self, time_series: "scout_chartdefinition_api_TimeSeriesChartDefinition") -> Any:
        pass

    @abstractmethod
    def _cartesian(self, cartesian: "scout_chartdefinition_api_CartesianChartDefinition") -> Any:
        pass

    @abstractmethod
    def _frequency(self, frequency: "scout_chartdefinition_api_FrequencyChartDefinition") -> Any:
        pass

    @abstractmethod
    def _histogram(self, histogram: "scout_chartdefinition_api_HistogramChartDefinition") -> Any:
        pass

    @abstractmethod
    def _video(self, video: "scout_chartdefinition_api_VideoVizDefinition") -> Any:
        pass

    @abstractmethod
    def _value_table(self, value_table: "scout_chartdefinition_api_ValueTableDefinition") -> Any:
        pass

    @abstractmethod
    def _geo(self, geo: "scout_chartdefinition_api_GeoVizDefinition") -> Any:
        pass

    @abstractmethod
    def _checklist(self, checklist: "scout_chartdefinition_api_ChecklistChartDefinition") -> Any:
        pass


scout_chartdefinition_api_VizDefinitionVisitor.__name__ = "VizDefinitionVisitor"
scout_chartdefinition_api_VizDefinitionVisitor.__qualname__ = "VizDefinitionVisitor"
scout_chartdefinition_api_VizDefinitionVisitor.__module__ = "nominal_api.scout_chartdefinition_api"


class scout_checklistexecution_api_AssetStreamConfiguration(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'notification_configurations': ConjureFieldDefinition('notificationConfigurations', List[scout_integrations_api_NotificationConfiguration]),
            'evaluation_delay': ConjureFieldDefinition('evaluationDelay', scout_run_api_Duration),
            'recovery_delay': ConjureFieldDefinition('recoveryDelay', scout_run_api_Duration)
        }

    __slots__: List[str] = ['_notification_configurations', '_evaluation_delay', '_recovery_delay']

    def __init__(self, evaluation_delay: "scout_run_api_Duration", notification_configurations: List["scout_integrations_api_NotificationConfiguration"], recovery_delay: "scout_run_api_Duration") -> None:
        self._notification_configurations = notification_configurations
        self._evaluation_delay = evaluation_delay
        self._recovery_delay = recovery_delay

    @builtins.property
    def notification_configurations(self) -> List["scout_integrations_api_NotificationConfiguration"]:
        return self._notification_configurations

    @builtins.property
    def evaluation_delay(self) -> "scout_run_api_Duration":
        return self._evaluation_delay

    @builtins.property
    def recovery_delay(self) -> "scout_run_api_Duration":
        return self._recovery_delay


scout_checklistexecution_api_AssetStreamConfiguration.__name__ = "AssetStreamConfiguration"
scout_checklistexecution_api_AssetStreamConfiguration.__qualname__ = "AssetStreamConfiguration"
scout_checklistexecution_api_AssetStreamConfiguration.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_BatchChecklistLiveStatusRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'requests': ConjureFieldDefinition('requests', List[scout_checklistexecution_api_ChecklistLiveStatusRequest]),
            'computation_range': ConjureFieldDefinition('computationRange', OptionalTypeWrapper[scout_checklistexecution_api_ComputationRange])
        }

    __slots__: List[str] = ['_requests', '_computation_range']

    def __init__(self, requests: List["scout_checklistexecution_api_ChecklistLiveStatusRequest"], computation_range: Optional["scout_checklistexecution_api_ComputationRange"] = None) -> None:
        self._requests = requests
        self._computation_range = computation_range

    @builtins.property
    def requests(self) -> List["scout_checklistexecution_api_ChecklistLiveStatusRequest"]:
        return self._requests

    @builtins.property
    def computation_range(self) -> Optional["scout_checklistexecution_api_ComputationRange"]:
        """
        The time range over which status and last failure were calculated. Failures outside of this time range
will not be included in the results. If not specified, defaults to evaluating from now minus sixty minutes to now.
        """
        return self._computation_range


scout_checklistexecution_api_BatchChecklistLiveStatusRequest.__name__ = "BatchChecklistLiveStatusRequest"
scout_checklistexecution_api_BatchChecklistLiveStatusRequest.__qualname__ = "BatchChecklistLiveStatusRequest"
scout_checklistexecution_api_BatchChecklistLiveStatusRequest.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_BatchChecklistLiveStatusResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'computation_range': ConjureFieldDefinition('computationRange', scout_checklistexecution_api_ComputationRange),
            'checklist_results': ConjureFieldDefinition('checklistResults', List[scout_checklistexecution_api_ChecklistLiveStatusResponse])
        }

    __slots__: List[str] = ['_computation_range', '_checklist_results']

    def __init__(self, checklist_results: List["scout_checklistexecution_api_ChecklistLiveStatusResponse"], computation_range: "scout_checklistexecution_api_ComputationRange") -> None:
        self._computation_range = computation_range
        self._checklist_results = checklist_results

    @builtins.property
    def computation_range(self) -> "scout_checklistexecution_api_ComputationRange":
        """
        The time range over which status and last failure were calculated. Failures outside of this time range
will not be included in the results.
        """
        return self._computation_range

    @builtins.property
    def checklist_results(self) -> List["scout_checklistexecution_api_ChecklistLiveStatusResponse"]:
        return self._checklist_results


scout_checklistexecution_api_BatchChecklistLiveStatusResponse.__name__ = "BatchChecklistLiveStatusResponse"
scout_checklistexecution_api_BatchChecklistLiveStatusResponse.__qualname__ = "BatchChecklistLiveStatusResponse"
scout_checklistexecution_api_BatchChecklistLiveStatusResponse.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_CheckLiveStatusResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'result': ConjureFieldDefinition('result', scout_checklistexecution_api_CheckLiveStatusResult),
            'check_rid': ConjureFieldDefinition('checkRid', scout_rids_api_CheckRid),
            'check_implementation_index': ConjureFieldDefinition('checkImplementationIndex', OptionalTypeWrapper[int])
        }

    __slots__: List[str] = ['_result', '_check_rid', '_check_implementation_index']

    def __init__(self, check_rid: str, result: "scout_checklistexecution_api_CheckLiveStatusResult", check_implementation_index: Optional[int] = None) -> None:
        self._result = result
        self._check_rid = check_rid
        self._check_implementation_index = check_implementation_index

    @builtins.property
    def result(self) -> "scout_checklistexecution_api_CheckLiveStatusResult":
        return self._result

    @builtins.property
    def check_rid(self) -> str:
        return self._check_rid

    @builtins.property
    def check_implementation_index(self) -> Optional[int]:
        """
        Checks can define a single range computation which can evaluate over multiple implementations of a context.
The check implementation index will correspond to the implementation index of the check condition.
        """
        return self._check_implementation_index


scout_checklistexecution_api_CheckLiveStatusResponse.__name__ = "CheckLiveStatusResponse"
scout_checklistexecution_api_CheckLiveStatusResponse.__qualname__ = "CheckLiveStatusResponse"
scout_checklistexecution_api_CheckLiveStatusResponse.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_CheckLiveStatusResult(ConjureUnionType):
    _computed: Optional["scout_checklistexecution_api_ComputedResult"] = None
    _not_computable: Optional["scout_checklistexecution_api_NotComputable"] = None
    _failed_to_resolve: Optional["scout_checklistexecution_api_FailedToResolve"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'computed': ConjureFieldDefinition('computed', scout_checklistexecution_api_ComputedResult),
            'not_computable': ConjureFieldDefinition('notComputable', scout_checklistexecution_api_NotComputable),
            'failed_to_resolve': ConjureFieldDefinition('failedToResolve', scout_checklistexecution_api_FailedToResolve)
        }

    def __init__(
            self,
            computed: Optional["scout_checklistexecution_api_ComputedResult"] = None,
            not_computable: Optional["scout_checklistexecution_api_NotComputable"] = None,
            failed_to_resolve: Optional["scout_checklistexecution_api_FailedToResolve"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (computed is not None) + (not_computable is not None) + (failed_to_resolve is not None) != 1:
                raise ValueError('a union must contain a single member')

            if computed is not None:
                self._computed = computed
                self._type = 'computed'
            if not_computable is not None:
                self._not_computable = not_computable
                self._type = 'notComputable'
            if failed_to_resolve is not None:
                self._failed_to_resolve = failed_to_resolve
                self._type = 'failedToResolve'

        elif type_of_union == 'computed':
            if computed is None:
                raise ValueError('a union value must not be None')
            self._computed = computed
            self._type = 'computed'
        elif type_of_union == 'notComputable':
            if not_computable is None:
                raise ValueError('a union value must not be None')
            self._not_computable = not_computable
            self._type = 'notComputable'
        elif type_of_union == 'failedToResolve':
            if failed_to_resolve is None:
                raise ValueError('a union value must not be None')
            self._failed_to_resolve = failed_to_resolve
            self._type = 'failedToResolve'

    @builtins.property
    def computed(self) -> Optional["scout_checklistexecution_api_ComputedResult"]:
        return self._computed

    @builtins.property
    def not_computable(self) -> Optional["scout_checklistexecution_api_NotComputable"]:
        """
        The check cannot be computed in a streaming context. This would be returned if an overly expensive
computation was attempted, such as checking the rolling median over the last year of data.
        """
        return self._not_computable

    @builtins.property
    def failed_to_resolve(self) -> Optional["scout_checklistexecution_api_FailedToResolve"]:
        """
        The checklist cannot be applied to the data sources.
        """
        return self._failed_to_resolve

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checklistexecution_api_CheckLiveStatusResultVisitor):
            raise ValueError('{} is not an instance of scout_checklistexecution_api_CheckLiveStatusResultVisitor'.format(visitor.__class__.__name__))
        if self._type == 'computed' and self.computed is not None:
            return visitor._computed(self.computed)
        if self._type == 'notComputable' and self.not_computable is not None:
            return visitor._not_computable(self.not_computable)
        if self._type == 'failedToResolve' and self.failed_to_resolve is not None:
            return visitor._failed_to_resolve(self.failed_to_resolve)


scout_checklistexecution_api_CheckLiveStatusResult.__name__ = "CheckLiveStatusResult"
scout_checklistexecution_api_CheckLiveStatusResult.__qualname__ = "CheckLiveStatusResult"
scout_checklistexecution_api_CheckLiveStatusResult.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_CheckLiveStatusResultVisitor:

    @abstractmethod
    def _computed(self, computed: "scout_checklistexecution_api_ComputedResult") -> Any:
        pass

    @abstractmethod
    def _not_computable(self, not_computable: "scout_checklistexecution_api_NotComputable") -> Any:
        pass

    @abstractmethod
    def _failed_to_resolve(self, failed_to_resolve: "scout_checklistexecution_api_FailedToResolve") -> Any:
        pass


scout_checklistexecution_api_CheckLiveStatusResultVisitor.__name__ = "CheckLiveStatusResultVisitor"
scout_checklistexecution_api_CheckLiveStatusResultVisitor.__qualname__ = "CheckLiveStatusResultVisitor"
scout_checklistexecution_api_CheckLiveStatusResultVisitor.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_CheckStatus(ConjureUnionType):
    _pass_: Optional["scout_checklistexecution_api_Pass"] = None
    _fail: Optional["scout_checklistexecution_api_Fail"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'pass_': ConjureFieldDefinition('pass', scout_checklistexecution_api_Pass),
            'fail': ConjureFieldDefinition('fail', scout_checklistexecution_api_Fail)
        }

    def __init__(
            self,
            pass_: Optional["scout_checklistexecution_api_Pass"] = None,
            fail: Optional["scout_checklistexecution_api_Fail"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (pass_ is not None) + (fail is not None) != 1:
                raise ValueError('a union must contain a single member')

            if pass_ is not None:
                self._pass_ = pass_
                self._type = 'pass'
            if fail is not None:
                self._fail = fail
                self._type = 'fail'

        elif type_of_union == 'pass':
            if pass_ is None:
                raise ValueError('a union value must not be None')
            self._pass_ = pass_
            self._type = 'pass'
        elif type_of_union == 'fail':
            if fail is None:
                raise ValueError('a union value must not be None')
            self._fail = fail
            self._type = 'fail'

    @builtins.property
    def pass_(self) -> Optional["scout_checklistexecution_api_Pass"]:
        """
        The check was evaluated and is currently in state PASS.
        """
        return self._pass_

    @builtins.property
    def fail(self) -> Optional["scout_checklistexecution_api_Fail"]:
        """
        The check was evaluated and is currently in state FAIL.
        """
        return self._fail

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checklistexecution_api_CheckStatusVisitor):
            raise ValueError('{} is not an instance of scout_checklistexecution_api_CheckStatusVisitor'.format(visitor.__class__.__name__))
        if self._type == 'pass' and self.pass_ is not None:
            return visitor._pass(self.pass_)
        if self._type == 'fail' and self.fail is not None:
            return visitor._fail(self.fail)


scout_checklistexecution_api_CheckStatus.__name__ = "CheckStatus"
scout_checklistexecution_api_CheckStatus.__qualname__ = "CheckStatus"
scout_checklistexecution_api_CheckStatus.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_CheckStatusVisitor:

    @abstractmethod
    def _pass(self, pass_: "scout_checklistexecution_api_Pass") -> Any:
        pass

    @abstractmethod
    def _fail(self, fail: "scout_checklistexecution_api_Fail") -> Any:
        pass


scout_checklistexecution_api_CheckStatusVisitor.__name__ = "CheckStatusVisitor"
scout_checklistexecution_api_CheckStatusVisitor.__qualname__ = "CheckStatusVisitor"
scout_checklistexecution_api_CheckStatusVisitor.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ChecklistExecutionService(Service):
    """
    The Checklist Execution Service checks the status of checklist executions.
    """

    def checklist_live_status(self, auth_header: str, request: "scout_checklistexecution_api_BatchChecklistLiveStatusRequest") -> "scout_checklistexecution_api_BatchChecklistLiveStatusResponse":
        """
        For each request, get the current status and last failing time (if possible) for
every automatic check against the given run.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/checklist-execution/checklist-live-status'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checklistexecution_api_BatchChecklistLiveStatusResponse, self._return_none_for_unknown_union_types)

    def execute_streaming_checklist(self, auth_header: str, request: "scout_checklistexecution_api_ExecuteChecklistForAssetsRequest") -> None:
        """
        Triggers a checklist to run continuously against assets. Returns a ChecklistAlreadyRunning error if the
checklist is already running for the given asset.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/checklist-execution/execute-streaming-checklist'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def list_streaming_checklist(self, auth_header: str, request: "scout_checklistexecution_api_ListStreamingChecklistRequest") -> "scout_checklistexecution_api_ListStreamingChecklistResponse":
        """
        Lists all running streaming checklists.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/checklist-execution/list-streaming-checklists'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checklistexecution_api_ListStreamingChecklistResponse, self._return_none_for_unknown_union_types)

    def list_streaming_checklist_for_asset(self, auth_header: str, request: "scout_checklistexecution_api_ListStreamingChecklistForAssetRequest") -> "scout_checklistexecution_api_ListStreamingChecklistForAssetResponse":
        """
        Retrieves all streaming checklists for a given asset.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/checklist-execution/list-streaming-checklists-for-asset'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checklistexecution_api_ListStreamingChecklistForAssetResponse, self._return_none_for_unknown_union_types)

    def get_streaming_checklist(self, auth_header: str, checklist_rid: str) -> "scout_checklistexecution_api_StreamingChecklistInfo":
        """
        Retrieves information about a streaming checklist using the provided checklistRid.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'checklistRid': checklist_rid,
        }

        _json: Any = None

        _path = '/scout/v2/checklist-execution/get-streaming-checklist/{checklistRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checklistexecution_api_StreamingChecklistInfo, self._return_none_for_unknown_union_types)

    def stop_streaming_checklist(self, auth_header: str, checklist_rid: str) -> None:
        """
        Stops the execution of a streaming checklist on all assets.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'checklistRid': checklist_rid,
        }

        _json: Any = None

        _path = '/scout/v2/checklist-execution/stop-streaming-checklist/{checklistRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'DELETE',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def stop_streaming_checklist_for_assets(self, auth_header: str, request: "scout_checklistexecution_api_StopStreamingChecklistForAssetsRequest") -> None:
        """
        Stops the execution of a streaming checklist for a given set of assets.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v2/checklist-execution/stop-streaming-checklist-for-assets'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def reload_streaming_checklist(self, auth_header: str, checklist_rid: str) -> None:
        """
        Reloads the state of the streaming checklist for the given checklistRid.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'checklistRid': checklist_rid,
        }

        _json: Any = None

        _path = '/scout/v2/checklist-execution/reload-streaming-checklist/{checklistRid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return


scout_checklistexecution_api_ChecklistExecutionService.__name__ = "ChecklistExecutionService"
scout_checklistexecution_api_ChecklistExecutionService.__qualname__ = "ChecklistExecutionService"
scout_checklistexecution_api_ChecklistExecutionService.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ChecklistLiveStatusRequest(ConjureBeanType):
    """
    If commit is not provided, the latest commit on main will be used. At least one of runRid or assetRid must be provided, but not both.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklist_rid': ConjureFieldDefinition('checklistRid', scout_rids_api_ChecklistRid),
            'commit': ConjureFieldDefinition('commit', OptionalTypeWrapper[scout_versioning_api_CommitId]),
            'run_rid': ConjureFieldDefinition('runRid', OptionalTypeWrapper[scout_run_api_RunRid]),
            'asset_rid': ConjureFieldDefinition('assetRid', OptionalTypeWrapper[scout_rids_api_AssetRid])
        }

    __slots__: List[str] = ['_checklist_rid', '_commit', '_run_rid', '_asset_rid']

    def __init__(self, checklist_rid: str, asset_rid: Optional[str] = None, commit: Optional[str] = None, run_rid: Optional[str] = None) -> None:
        self._checklist_rid = checklist_rid
        self._commit = commit
        self._run_rid = run_rid
        self._asset_rid = asset_rid

    @builtins.property
    def checklist_rid(self) -> str:
        return self._checklist_rid

    @builtins.property
    def commit(self) -> Optional[str]:
        return self._commit

    @builtins.property
    def run_rid(self) -> Optional[str]:
        return self._run_rid

    @builtins.property
    def asset_rid(self) -> Optional[str]:
        return self._asset_rid


scout_checklistexecution_api_ChecklistLiveStatusRequest.__name__ = "ChecklistLiveStatusRequest"
scout_checklistexecution_api_ChecklistLiveStatusRequest.__qualname__ = "ChecklistLiveStatusRequest"
scout_checklistexecution_api_ChecklistLiveStatusRequest.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ChecklistLiveStatusResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklist_rid': ConjureFieldDefinition('checklistRid', scout_rids_api_ChecklistRid),
            'check_results': ConjureFieldDefinition('checkResults', List[scout_checklistexecution_api_CheckLiveStatusResponse])
        }

    __slots__: List[str] = ['_checklist_rid', '_check_results']

    def __init__(self, check_results: List["scout_checklistexecution_api_CheckLiveStatusResponse"], checklist_rid: str) -> None:
        self._checklist_rid = checklist_rid
        self._check_results = check_results

    @builtins.property
    def checklist_rid(self) -> str:
        return self._checklist_rid

    @builtins.property
    def check_results(self) -> List["scout_checklistexecution_api_CheckLiveStatusResponse"]:
        return self._check_results


scout_checklistexecution_api_ChecklistLiveStatusResponse.__name__ = "ChecklistLiveStatusResponse"
scout_checklistexecution_api_ChecklistLiveStatusResponse.__qualname__ = "ChecklistLiveStatusResponse"
scout_checklistexecution_api_ChecklistLiveStatusResponse.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ComputationRange(ConjureBeanType):
    """
    The time range over which status and last failure were calculated. Failures outside of this time range
will not be included in the results.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'start': ConjureFieldDefinition('start', api_Timestamp),
            'end': ConjureFieldDefinition('end', api_Timestamp)
        }

    __slots__: List[str] = ['_start', '_end']

    def __init__(self, end: "api_Timestamp", start: "api_Timestamp") -> None:
        self._start = start
        self._end = end

    @builtins.property
    def start(self) -> "api_Timestamp":
        return self._start

    @builtins.property
    def end(self) -> "api_Timestamp":
        return self._end


scout_checklistexecution_api_ComputationRange.__name__ = "ComputationRange"
scout_checklistexecution_api_ComputationRange.__qualname__ = "ComputationRange"
scout_checklistexecution_api_ComputationRange.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ComputedResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'status': ConjureFieldDefinition('status', scout_checklistexecution_api_CheckStatus),
            'last_failure': ConjureFieldDefinition('lastFailure', scout_checklistexecution_api_LastFailure)
        }

    __slots__: List[str] = ['_status', '_last_failure']

    def __init__(self, last_failure: "scout_checklistexecution_api_LastFailure", status: "scout_checklistexecution_api_CheckStatus") -> None:
        self._status = status
        self._last_failure = last_failure

    @builtins.property
    def status(self) -> "scout_checklistexecution_api_CheckStatus":
        """
        The current status of a check at the time it was evaluated.
        """
        return self._status

    @builtins.property
    def last_failure(self) -> "scout_checklistexecution_api_LastFailure":
        """
        If computed, represents the last time the check failed.
Can also represent that the last failure hasn't been computed, or is currently being computed.
If the check is currently failing, the current failure is returned.
        """
        return self._last_failure


scout_checklistexecution_api_ComputedResult.__name__ = "ComputedResult"
scout_checklistexecution_api_ComputedResult.__qualname__ = "ComputedResult"
scout_checklistexecution_api_ComputedResult.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_Computing(ConjureBeanType):
    """
    The result is currently being computed.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_checklistexecution_api_Computing.__name__ = "Computing"
scout_checklistexecution_api_Computing.__qualname__ = "Computing"
scout_checklistexecution_api_Computing.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ExecuteChecklistForAssetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklist': ConjureFieldDefinition('checklist', scout_rids_api_ChecklistRid),
            'assets': ConjureFieldDefinition('assets', List[scout_rids_api_AssetRid]),
            'notification_configurations': ConjureFieldDefinition('notificationConfigurations', List[scout_integrations_api_NotificationConfiguration]),
            'evaluation_delay': ConjureFieldDefinition('evaluationDelay', scout_run_api_Duration),
            'recovery_delay': ConjureFieldDefinition('recoveryDelay', scout_run_api_Duration)
        }

    __slots__: List[str] = ['_checklist', '_assets', '_notification_configurations', '_evaluation_delay', '_recovery_delay']

    def __init__(self, assets: List[str], checklist: str, evaluation_delay: "scout_run_api_Duration", notification_configurations: List["scout_integrations_api_NotificationConfiguration"], recovery_delay: "scout_run_api_Duration") -> None:
        self._checklist = checklist
        self._assets = assets
        self._notification_configurations = notification_configurations
        self._evaluation_delay = evaluation_delay
        self._recovery_delay = recovery_delay

    @builtins.property
    def checklist(self) -> str:
        return self._checklist

    @builtins.property
    def assets(self) -> List[str]:
        return self._assets

    @builtins.property
    def notification_configurations(self) -> List["scout_integrations_api_NotificationConfiguration"]:
        """
        Checklist violations will be sent to the specified integrations. At least one integration must be specified.
        """
        return self._notification_configurations

    @builtins.property
    def evaluation_delay(self) -> "scout_run_api_Duration":
        """
        Delays the evaluation of the streaming checklist. This is useful for when data is delayed.
        """
        return self._evaluation_delay

    @builtins.property
    def recovery_delay(self) -> "scout_run_api_Duration":
        """
        Specifies the minimum amount of time that must pass before a check can recover from a failure.
Minimum value is 15 seconds.
        """
        return self._recovery_delay


scout_checklistexecution_api_ExecuteChecklistForAssetsRequest.__name__ = "ExecuteChecklistForAssetsRequest"
scout_checklistexecution_api_ExecuteChecklistForAssetsRequest.__qualname__ = "ExecuteChecklistForAssetsRequest"
scout_checklistexecution_api_ExecuteChecklistForAssetsRequest.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_Fail(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_checklistexecution_api_Fail.__name__ = "Fail"
scout_checklistexecution_api_Fail.__qualname__ = "Fail"
scout_checklistexecution_api_Fail.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_FailedToResolve(ConjureBeanType):
    """
    The checklist cannot be applied to the data sources.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variable_name': ConjureFieldDefinition('variableName', str)
        }

    __slots__: List[str] = ['_variable_name']

    def __init__(self, variable_name: str) -> None:
        self._variable_name = variable_name

    @builtins.property
    def variable_name(self) -> str:
        """
        The variable name that couldn't be resolved to a series.
        """
        return self._variable_name


scout_checklistexecution_api_FailedToResolve.__name__ = "FailedToResolve"
scout_checklistexecution_api_FailedToResolve.__qualname__ = "FailedToResolve"
scout_checklistexecution_api_FailedToResolve.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_Failure(ConjureBeanType):
    """
    The time range where a check was failing.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'start_inclusive': ConjureFieldDefinition('startInclusive', OptionalTypeWrapper[api_Timestamp]),
            'end_exclusive': ConjureFieldDefinition('endExclusive', OptionalTypeWrapper[api_Timestamp])
        }

    __slots__: List[str] = ['_start_inclusive', '_end_exclusive']

    def __init__(self, end_exclusive: Optional["api_Timestamp"] = None, start_inclusive: Optional["api_Timestamp"] = None) -> None:
        self._start_inclusive = start_inclusive
        self._end_exclusive = end_exclusive

    @builtins.property
    def start_inclusive(self) -> Optional["api_Timestamp"]:
        return self._start_inclusive

    @builtins.property
    def end_exclusive(self) -> Optional["api_Timestamp"]:
        return self._end_exclusive


scout_checklistexecution_api_Failure.__name__ = "Failure"
scout_checklistexecution_api_Failure.__qualname__ = "Failure"
scout_checklistexecution_api_Failure.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_LastFailure(ConjureUnionType):
    _not_computed: Optional["scout_checklistexecution_api_NotComputed"] = None
    _computing: Optional["scout_checklistexecution_api_Computing"] = None
    _no_previous_failure: Optional["scout_checklistexecution_api_NoPreviousFailure"] = None
    _failure: Optional["scout_checklistexecution_api_Failure"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'not_computed': ConjureFieldDefinition('notComputed', scout_checklistexecution_api_NotComputed),
            'computing': ConjureFieldDefinition('computing', scout_checklistexecution_api_Computing),
            'no_previous_failure': ConjureFieldDefinition('noPreviousFailure', scout_checklistexecution_api_NoPreviousFailure),
            'failure': ConjureFieldDefinition('failure', scout_checklistexecution_api_Failure)
        }

    def __init__(
            self,
            not_computed: Optional["scout_checklistexecution_api_NotComputed"] = None,
            computing: Optional["scout_checklistexecution_api_Computing"] = None,
            no_previous_failure: Optional["scout_checklistexecution_api_NoPreviousFailure"] = None,
            failure: Optional["scout_checklistexecution_api_Failure"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (not_computed is not None) + (computing is not None) + (no_previous_failure is not None) + (failure is not None) != 1:
                raise ValueError('a union must contain a single member')

            if not_computed is not None:
                self._not_computed = not_computed
                self._type = 'notComputed'
            if computing is not None:
                self._computing = computing
                self._type = 'computing'
            if no_previous_failure is not None:
                self._no_previous_failure = no_previous_failure
                self._type = 'noPreviousFailure'
            if failure is not None:
                self._failure = failure
                self._type = 'failure'

        elif type_of_union == 'notComputed':
            if not_computed is None:
                raise ValueError('a union value must not be None')
            self._not_computed = not_computed
            self._type = 'notComputed'
        elif type_of_union == 'computing':
            if computing is None:
                raise ValueError('a union value must not be None')
            self._computing = computing
            self._type = 'computing'
        elif type_of_union == 'noPreviousFailure':
            if no_previous_failure is None:
                raise ValueError('a union value must not be None')
            self._no_previous_failure = no_previous_failure
            self._type = 'noPreviousFailure'
        elif type_of_union == 'failure':
            if failure is None:
                raise ValueError('a union value must not be None')
            self._failure = failure
            self._type = 'failure'

    @builtins.property
    def not_computed(self) -> Optional["scout_checklistexecution_api_NotComputed"]:
        """
        The last failure time has not been computed, and is not currently being computed.
        """
        return self._not_computed

    @builtins.property
    def computing(self) -> Optional["scout_checklistexecution_api_Computing"]:
        """
        The last failure time is currently being computed. This may require evaluating the history
of data for the check, so could be slow depending on the data frequency and lookback period.
        """
        return self._computing

    @builtins.property
    def no_previous_failure(self) -> Optional["scout_checklistexecution_api_NoPreviousFailure"]:
        """
        No previous failure was found for the check.
        """
        return self._no_previous_failure

    @builtins.property
    def failure(self) -> Optional["scout_checklistexecution_api_Failure"]:
        """
        The most recent failure for the check.
        """
        return self._failure

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checklistexecution_api_LastFailureVisitor):
            raise ValueError('{} is not an instance of scout_checklistexecution_api_LastFailureVisitor'.format(visitor.__class__.__name__))
        if self._type == 'notComputed' and self.not_computed is not None:
            return visitor._not_computed(self.not_computed)
        if self._type == 'computing' and self.computing is not None:
            return visitor._computing(self.computing)
        if self._type == 'noPreviousFailure' and self.no_previous_failure is not None:
            return visitor._no_previous_failure(self.no_previous_failure)
        if self._type == 'failure' and self.failure is not None:
            return visitor._failure(self.failure)


scout_checklistexecution_api_LastFailure.__name__ = "LastFailure"
scout_checklistexecution_api_LastFailure.__qualname__ = "LastFailure"
scout_checklistexecution_api_LastFailure.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_LastFailureVisitor:

    @abstractmethod
    def _not_computed(self, not_computed: "scout_checklistexecution_api_NotComputed") -> Any:
        pass

    @abstractmethod
    def _computing(self, computing: "scout_checklistexecution_api_Computing") -> Any:
        pass

    @abstractmethod
    def _no_previous_failure(self, no_previous_failure: "scout_checklistexecution_api_NoPreviousFailure") -> Any:
        pass

    @abstractmethod
    def _failure(self, failure: "scout_checklistexecution_api_Failure") -> Any:
        pass


scout_checklistexecution_api_LastFailureVisitor.__name__ = "LastFailureVisitor"
scout_checklistexecution_api_LastFailureVisitor.__qualname__ = "LastFailureVisitor"
scout_checklistexecution_api_LastFailureVisitor.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ListStreamingChecklistForAssetRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'asset_rid': ConjureFieldDefinition('assetRid', scout_rids_api_AssetRid),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int]),
            'page_token': ConjureFieldDefinition('pageToken', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_asset_rid', '_page_size', '_page_token']

    def __init__(self, asset_rid: str, page_size: Optional[int] = None, page_token: Optional[str] = None) -> None:
        self._asset_rid = asset_rid
        self._page_size = page_size
        self._page_token = page_token

    @builtins.property
    def asset_rid(self) -> str:
        return self._asset_rid

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Page sizes greater than 10_000 will be rejected. Default pageSize is 100.
        """
        return self._page_size

    @builtins.property
    def page_token(self) -> Optional[str]:
        return self._page_token


scout_checklistexecution_api_ListStreamingChecklistForAssetRequest.__name__ = "ListStreamingChecklistForAssetRequest"
scout_checklistexecution_api_ListStreamingChecklistForAssetRequest.__qualname__ = "ListStreamingChecklistForAssetRequest"
scout_checklistexecution_api_ListStreamingChecklistForAssetRequest.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ListStreamingChecklistForAssetResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklists': ConjureFieldDefinition('checklists', List[scout_rids_api_ChecklistRid]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_checklists', '_next_page_token']

    def __init__(self, checklists: List[str], next_page_token: Optional[str] = None) -> None:
        self._checklists = checklists
        self._next_page_token = next_page_token

    @builtins.property
    def checklists(self) -> List[str]:
        return self._checklists

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


scout_checklistexecution_api_ListStreamingChecklistForAssetResponse.__name__ = "ListStreamingChecklistForAssetResponse"
scout_checklistexecution_api_ListStreamingChecklistForAssetResponse.__qualname__ = "ListStreamingChecklistForAssetResponse"
scout_checklistexecution_api_ListStreamingChecklistForAssetResponse.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ListStreamingChecklistRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int]),
            'page_token': ConjureFieldDefinition('pageToken', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_page_size', '_page_token']

    def __init__(self, page_size: Optional[int] = None, page_token: Optional[str] = None) -> None:
        self._page_size = page_size
        self._page_token = page_token

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Page sizes greater than 10_000 will be rejected. Default pageSize is 100.
        """
        return self._page_size

    @builtins.property
    def page_token(self) -> Optional[str]:
        return self._page_token


scout_checklistexecution_api_ListStreamingChecklistRequest.__name__ = "ListStreamingChecklistRequest"
scout_checklistexecution_api_ListStreamingChecklistRequest.__qualname__ = "ListStreamingChecklistRequest"
scout_checklistexecution_api_ListStreamingChecklistRequest.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_ListStreamingChecklistResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklists': ConjureFieldDefinition('checklists', List[scout_rids_api_ChecklistRid]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[str])
        }

    __slots__: List[str] = ['_checklists', '_next_page_token']

    def __init__(self, checklists: List[str], next_page_token: Optional[str] = None) -> None:
        self._checklists = checklists
        self._next_page_token = next_page_token

    @builtins.property
    def checklists(self) -> List[str]:
        return self._checklists

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


scout_checklistexecution_api_ListStreamingChecklistResponse.__name__ = "ListStreamingChecklistResponse"
scout_checklistexecution_api_ListStreamingChecklistResponse.__qualname__ = "ListStreamingChecklistResponse"
scout_checklistexecution_api_ListStreamingChecklistResponse.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_NoPreviousFailure(ConjureBeanType):
    """
    No previous failure was computed for the check.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_checklistexecution_api_NoPreviousFailure.__name__ = "NoPreviousFailure"
scout_checklistexecution_api_NoPreviousFailure.__qualname__ = "NoPreviousFailure"
scout_checklistexecution_api_NoPreviousFailure.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_NotComputable(ConjureBeanType):
    """
    The result cannot be computed in a streaming context.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'message': ConjureFieldDefinition('message', str)
        }

    __slots__: List[str] = ['_message']

    def __init__(self, message: str) -> None:
        self._message = message

    @builtins.property
    def message(self) -> str:
        """
        Why the result couldn't be computed
        """
        return self._message


scout_checklistexecution_api_NotComputable.__name__ = "NotComputable"
scout_checklistexecution_api_NotComputable.__qualname__ = "NotComputable"
scout_checklistexecution_api_NotComputable.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_NotComputed(ConjureBeanType):
    """
    The result has not been computed, and is not currently being computed.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_checklistexecution_api_NotComputed.__name__ = "NotComputed"
scout_checklistexecution_api_NotComputed.__qualname__ = "NotComputed"
scout_checklistexecution_api_NotComputed.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_Pass(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_checklistexecution_api_Pass.__name__ = "Pass"
scout_checklistexecution_api_Pass.__qualname__ = "Pass"
scout_checklistexecution_api_Pass.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_StopStreamingChecklistForAssetsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklist': ConjureFieldDefinition('checklist', scout_rids_api_ChecklistRid),
            'assets': ConjureFieldDefinition('assets', List[scout_rids_api_AssetRid])
        }

    __slots__: List[str] = ['_checklist', '_assets']

    def __init__(self, assets: List[str], checklist: str) -> None:
        self._checklist = checklist
        self._assets = assets

    @builtins.property
    def checklist(self) -> str:
        return self._checklist

    @builtins.property
    def assets(self) -> List[str]:
        return self._assets


scout_checklistexecution_api_StopStreamingChecklistForAssetsRequest.__name__ = "StopStreamingChecklistForAssetsRequest"
scout_checklistexecution_api_StopStreamingChecklistForAssetsRequest.__qualname__ = "StopStreamingChecklistForAssetsRequest"
scout_checklistexecution_api_StopStreamingChecklistForAssetsRequest.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checklistexecution_api_StreamingChecklistInfo(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklist_rid': ConjureFieldDefinition('checklistRid', scout_rids_api_ChecklistRid),
            'asset_configurations': ConjureFieldDefinition('assetConfigurations', Dict[scout_rids_api_AssetRid, scout_checklistexecution_api_AssetStreamConfiguration])
        }

    __slots__: List[str] = ['_checklist_rid', '_asset_configurations']

    def __init__(self, asset_configurations: Dict[str, "scout_checklistexecution_api_AssetStreamConfiguration"], checklist_rid: str) -> None:
        self._checklist_rid = checklist_rid
        self._asset_configurations = asset_configurations

    @builtins.property
    def checklist_rid(self) -> str:
        return self._checklist_rid

    @builtins.property
    def asset_configurations(self) -> Dict[str, "scout_checklistexecution_api_AssetStreamConfiguration"]:
        return self._asset_configurations


scout_checklistexecution_api_StreamingChecklistInfo.__name__ = "StreamingChecklistInfo"
scout_checklistexecution_api_StreamingChecklistInfo.__qualname__ = "StreamingChecklistInfo"
scout_checklistexecution_api_StreamingChecklistInfo.__module__ = "nominal_api.scout_checklistexecution_api"


class scout_checks_api_ArchiveChecklistsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rids': ConjureFieldDefinition('rids', List[scout_rids_api_ChecklistRid])
        }

    __slots__: List[str] = ['_rids']

    def __init__(self, rids: List[str]) -> None:
        self._rids = rids

    @builtins.property
    def rids(self) -> List[str]:
        return self._rids


scout_checks_api_ArchiveChecklistsRequest.__name__ = "ArchiveChecklistsRequest"
scout_checks_api_ArchiveChecklistsRequest.__qualname__ = "ArchiveChecklistsRequest"
scout_checks_api_ArchiveChecklistsRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_BatchGetChecklistMetadataRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rids': ConjureFieldDefinition('rids', List[scout_rids_api_ChecklistRid])
        }

    __slots__: List[str] = ['_rids']

    def __init__(self, rids: List[str]) -> None:
        self._rids = rids

    @builtins.property
    def rids(self) -> List[str]:
        return self._rids


scout_checks_api_BatchGetChecklistMetadataRequest.__name__ = "BatchGetChecklistMetadataRequest"
scout_checks_api_BatchGetChecklistMetadataRequest.__qualname__ = "BatchGetChecklistMetadataRequest"
scout_checks_api_BatchGetChecklistMetadataRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_BatchGetChecklistMetadataResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'metadata': ConjureFieldDefinition('metadata', Dict[scout_rids_api_ChecklistRid, scout_checks_api_ChecklistMetadata])
        }

    __slots__: List[str] = ['_metadata']

    def __init__(self, metadata: Dict[str, "scout_checks_api_ChecklistMetadata"]) -> None:
        self._metadata = metadata

    @builtins.property
    def metadata(self) -> Dict[str, "scout_checks_api_ChecklistMetadata"]:
        return self._metadata


scout_checks_api_BatchGetChecklistMetadataResponse.__name__ = "BatchGetChecklistMetadataResponse"
scout_checks_api_BatchGetChecklistMetadataResponse.__qualname__ = "BatchGetChecklistMetadataResponse"
scout_checks_api_BatchGetChecklistMetadataResponse.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_BatchGetJobReportsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'job_rids': ConjureFieldDefinition('jobRids', List[scout_checks_api_JobRid])
        }

    __slots__: List[str] = ['_job_rids']

    def __init__(self, job_rids: List[str]) -> None:
        self._job_rids = job_rids

    @builtins.property
    def job_rids(self) -> List[str]:
        return self._job_rids


scout_checks_api_BatchGetJobReportsRequest.__name__ = "BatchGetJobReportsRequest"
scout_checks_api_BatchGetJobReportsRequest.__qualname__ = "BatchGetJobReportsRequest"
scout_checks_api_BatchGetJobReportsRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_BatchGetJobReportsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'job_rids_to_job_reports': ConjureFieldDefinition('jobRidsToJobReports', Dict[scout_checks_api_JobRid, scout_checks_api_JobReport])
        }

    __slots__: List[str] = ['_job_rids_to_job_reports']

    def __init__(self, job_rids_to_job_reports: Dict[str, "scout_checks_api_JobReport"]) -> None:
        self._job_rids_to_job_reports = job_rids_to_job_reports

    @builtins.property
    def job_rids_to_job_reports(self) -> Dict[str, "scout_checks_api_JobReport"]:
        return self._job_rids_to_job_reports


scout_checks_api_BatchGetJobReportsResponse.__name__ = "BatchGetJobReportsResponse"
scout_checks_api_BatchGetJobReportsResponse.__qualname__ = "BatchGetJobReportsResponse"
scout_checks_api_BatchGetJobReportsResponse.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_Check(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', scout_rids_api_CheckRid),
            'check_lineage_rid': ConjureFieldDefinition('checkLineageRid', scout_rids_api_CheckLineageRid),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', str),
            'auto_generated_title': ConjureFieldDefinition('autoGeneratedTitle', OptionalTypeWrapper[str]),
            'auto_generated_description': ConjureFieldDefinition('autoGeneratedDescription', OptionalTypeWrapper[str]),
            'priority': ConjureFieldDefinition('priority', scout_checks_api_Priority),
            'chart': ConjureFieldDefinition('chart', OptionalTypeWrapper[scout_rids_api_VersionedVizId]),
            'condition': ConjureFieldDefinition('condition', OptionalTypeWrapper[scout_checks_api_CheckCondition])
        }

    __slots__: List[str] = ['_rid', '_check_lineage_rid', '_title', '_description', '_auto_generated_title', '_auto_generated_description', '_priority', '_chart', '_condition']

    def __init__(self, check_lineage_rid: str, description: str, priority: "scout_checks_api_Priority", rid: str, title: str, auto_generated_description: Optional[str] = None, auto_generated_title: Optional[str] = None, chart: Optional["scout_rids_api_VersionedVizId"] = None, condition: Optional["scout_checks_api_CheckCondition"] = None) -> None:
        self._rid = rid
        self._check_lineage_rid = check_lineage_rid
        self._title = title
        self._description = description
        self._auto_generated_title = auto_generated_title
        self._auto_generated_description = auto_generated_description
        self._priority = priority
        self._chart = chart
        self._condition = condition

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def check_lineage_rid(self) -> str:
        return self._check_lineage_rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def auto_generated_title(self) -> Optional[str]:
        return self._auto_generated_title

    @builtins.property
    def auto_generated_description(self) -> Optional[str]:
        return self._auto_generated_description

    @builtins.property
    def priority(self) -> "scout_checks_api_Priority":
        return self._priority

    @builtins.property
    def chart(self) -> Optional["scout_rids_api_VersionedVizId"]:
        return self._chart

    @builtins.property
    def condition(self) -> Optional["scout_checks_api_CheckCondition"]:
        """
        If omitted, this check represents a manual check.
        """
        return self._condition


scout_checks_api_Check.__name__ = "Check"
scout_checks_api_Check.__qualname__ = "Check"
scout_checks_api_Check.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CheckCondition(ConjureUnionType):
    _num_ranges: Optional["scout_checks_api_NumRangesConditionV1"] = None
    _num_ranges_v2: Optional["scout_checks_api_NumRangesConditionV2"] = None
    _num_ranges_v3: Optional["scout_checks_api_NumRangesConditionV3"] = None
    _parameterized_num_ranges_v1: Optional["scout_checks_api_ParameterizedNumRangesConditionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'num_ranges': ConjureFieldDefinition('numRanges', scout_checks_api_NumRangesConditionV1),
            'num_ranges_v2': ConjureFieldDefinition('numRangesV2', scout_checks_api_NumRangesConditionV2),
            'num_ranges_v3': ConjureFieldDefinition('numRangesV3', scout_checks_api_NumRangesConditionV3),
            'parameterized_num_ranges_v1': ConjureFieldDefinition('parameterizedNumRangesV1', scout_checks_api_ParameterizedNumRangesConditionV1)
        }

    def __init__(
            self,
            num_ranges: Optional["scout_checks_api_NumRangesConditionV1"] = None,
            num_ranges_v2: Optional["scout_checks_api_NumRangesConditionV2"] = None,
            num_ranges_v3: Optional["scout_checks_api_NumRangesConditionV3"] = None,
            parameterized_num_ranges_v1: Optional["scout_checks_api_ParameterizedNumRangesConditionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (num_ranges is not None) + (num_ranges_v2 is not None) + (num_ranges_v3 is not None) + (parameterized_num_ranges_v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if num_ranges is not None:
                self._num_ranges = num_ranges
                self._type = 'numRanges'
            if num_ranges_v2 is not None:
                self._num_ranges_v2 = num_ranges_v2
                self._type = 'numRangesV2'
            if num_ranges_v3 is not None:
                self._num_ranges_v3 = num_ranges_v3
                self._type = 'numRangesV3'
            if parameterized_num_ranges_v1 is not None:
                self._parameterized_num_ranges_v1 = parameterized_num_ranges_v1
                self._type = 'parameterizedNumRangesV1'

        elif type_of_union == 'numRanges':
            if num_ranges is None:
                raise ValueError('a union value must not be None')
            self._num_ranges = num_ranges
            self._type = 'numRanges'
        elif type_of_union == 'numRangesV2':
            if num_ranges_v2 is None:
                raise ValueError('a union value must not be None')
            self._num_ranges_v2 = num_ranges_v2
            self._type = 'numRangesV2'
        elif type_of_union == 'numRangesV3':
            if num_ranges_v3 is None:
                raise ValueError('a union value must not be None')
            self._num_ranges_v3 = num_ranges_v3
            self._type = 'numRangesV3'
        elif type_of_union == 'parameterizedNumRangesV1':
            if parameterized_num_ranges_v1 is None:
                raise ValueError('a union value must not be None')
            self._parameterized_num_ranges_v1 = parameterized_num_ranges_v1
            self._type = 'parameterizedNumRangesV1'

    @builtins.property
    def num_ranges(self) -> Optional["scout_checks_api_NumRangesConditionV1"]:
        return self._num_ranges

    @builtins.property
    def num_ranges_v2(self) -> Optional["scout_checks_api_NumRangesConditionV2"]:
        return self._num_ranges_v2

    @builtins.property
    def num_ranges_v3(self) -> Optional["scout_checks_api_NumRangesConditionV3"]:
        return self._num_ranges_v3

    @builtins.property
    def parameterized_num_ranges_v1(self) -> Optional["scout_checks_api_ParameterizedNumRangesConditionV1"]:
        return self._parameterized_num_ranges_v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_CheckConditionVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_CheckConditionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'numRanges' and self.num_ranges is not None:
            return visitor._num_ranges(self.num_ranges)
        if self._type == 'numRangesV2' and self.num_ranges_v2 is not None:
            return visitor._num_ranges_v2(self.num_ranges_v2)
        if self._type == 'numRangesV3' and self.num_ranges_v3 is not None:
            return visitor._num_ranges_v3(self.num_ranges_v3)
        if self._type == 'parameterizedNumRangesV1' and self.parameterized_num_ranges_v1 is not None:
            return visitor._parameterized_num_ranges_v1(self.parameterized_num_ranges_v1)


scout_checks_api_CheckCondition.__name__ = "CheckCondition"
scout_checks_api_CheckCondition.__qualname__ = "CheckCondition"
scout_checks_api_CheckCondition.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CheckConditionVisitor:

    @abstractmethod
    def _num_ranges(self, num_ranges: "scout_checks_api_NumRangesConditionV1") -> Any:
        pass

    @abstractmethod
    def _num_ranges_v2(self, num_ranges_v2: "scout_checks_api_NumRangesConditionV2") -> Any:
        pass

    @abstractmethod
    def _num_ranges_v3(self, num_ranges_v3: "scout_checks_api_NumRangesConditionV3") -> Any:
        pass

    @abstractmethod
    def _parameterized_num_ranges_v1(self, parameterized_num_ranges_v1: "scout_checks_api_ParameterizedNumRangesConditionV1") -> Any:
        pass


scout_checks_api_CheckConditionVisitor.__name__ = "CheckConditionVisitor"
scout_checks_api_CheckConditionVisitor.__qualname__ = "CheckConditionVisitor"
scout_checks_api_CheckConditionVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CheckContext(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variables': ConjureFieldDefinition('variables', Dict[scout_compute_api_VariableName, scout_checks_api_VariableLocator]),
            'sub_function_variables': ConjureFieldDefinition('subFunctionVariables', Dict[scout_compute_api_FunctionReference, scout_checks_api_CheckContext])
        }

    __slots__: List[str] = ['_variables', '_sub_function_variables']

    def __init__(self, sub_function_variables: Dict[str, "scout_checks_api_CheckContext"], variables: Dict[str, "scout_checks_api_VariableLocator"]) -> None:
        self._variables = variables
        self._sub_function_variables = sub_function_variables

    @builtins.property
    def variables(self) -> Dict[str, "scout_checks_api_VariableLocator"]:
        return self._variables

    @builtins.property
    def sub_function_variables(self) -> Dict[str, "scout_checks_api_CheckContext"]:
        return self._sub_function_variables


scout_checks_api_CheckContext.__name__ = "CheckContext"
scout_checks_api_CheckContext.__qualname__ = "CheckContext"
scout_checks_api_CheckContext.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CheckJobResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'status': ConjureFieldDefinition('status', scout_datareview_api_Status),
            'failures': ConjureFieldDefinition('failures', List[scout_compute_api_Range]),
            'are_failures_decimated': ConjureFieldDefinition('areFailuresDecimated', bool)
        }

    __slots__: List[str] = ['_status', '_failures', '_are_failures_decimated']

    def __init__(self, are_failures_decimated: bool, failures: List["scout_compute_api_Range"], status: "scout_datareview_api_Status") -> None:
        self._status = status
        self._failures = failures
        self._are_failures_decimated = are_failures_decimated

    @builtins.property
    def status(self) -> "scout_datareview_api_Status":
        return self._status

    @builtins.property
    def failures(self) -> List["scout_compute_api_Range"]:
        return self._failures

    @builtins.property
    def are_failures_decimated(self) -> bool:
        return self._are_failures_decimated


scout_checks_api_CheckJobResult.__name__ = "CheckJobResult"
scout_checks_api_CheckJobResult.__qualname__ = "CheckJobResult"
scout_checks_api_CheckJobResult.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CheckJobSpec(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_review_rid': ConjureFieldDefinition('dataReviewRid', scout_datareview_api_DataReviewRid),
            'check_rid': ConjureFieldDefinition('checkRid', scout_rids_api_CheckRid),
            'run_rid': ConjureFieldDefinition('runRid', OptionalTypeWrapper[scout_run_api_RunRid]),
            'check_implementation_index': ConjureFieldDefinition('checkImplementationIndex', OptionalTypeWrapper[int]),
            'check_evaluation_rid': ConjureFieldDefinition('checkEvaluationRid', scout_datareview_api_AutomaticCheckEvaluationRid),
            'check_condition': ConjureFieldDefinition('checkCondition', scout_checks_api_CheckCondition),
            'start': ConjureFieldDefinition('start', api_Timestamp),
            'end': ConjureFieldDefinition('end', api_Timestamp),
            'context': ConjureFieldDefinition('context', scout_compute_api_Context)
        }

    __slots__: List[str] = ['_data_review_rid', '_check_rid', '_run_rid', '_check_implementation_index', '_check_evaluation_rid', '_check_condition', '_start', '_end', '_context']

    def __init__(self, check_condition: "scout_checks_api_CheckCondition", check_evaluation_rid: str, check_rid: str, context: "scout_compute_api_Context", data_review_rid: str, end: "api_Timestamp", start: "api_Timestamp", check_implementation_index: Optional[int] = None, run_rid: Optional[str] = None) -> None:
        self._data_review_rid = data_review_rid
        self._check_rid = check_rid
        self._run_rid = run_rid
        self._check_implementation_index = check_implementation_index
        self._check_evaluation_rid = check_evaluation_rid
        self._check_condition = check_condition
        self._start = start
        self._end = end
        self._context = context

    @builtins.property
    def data_review_rid(self) -> str:
        return self._data_review_rid

    @builtins.property
    def check_rid(self) -> str:
        return self._check_rid

    @builtins.property
    def run_rid(self) -> Optional[str]:
        return self._run_rid

    @builtins.property
    def check_implementation_index(self) -> Optional[int]:
        """
        Checks can define a single range computation which can evaluate over multiple implementations of a context.
The check implementation index will correspond to the implementation index of the check condition.
        """
        return self._check_implementation_index

    @builtins.property
    def check_evaluation_rid(self) -> str:
        return self._check_evaluation_rid

    @builtins.property
    def check_condition(self) -> "scout_checks_api_CheckCondition":
        return self._check_condition

    @builtins.property
    def start(self) -> "api_Timestamp":
        return self._start

    @builtins.property
    def end(self) -> "api_Timestamp":
        return self._end

    @builtins.property
    def context(self) -> "scout_compute_api_Context":
        return self._context


scout_checks_api_CheckJobSpec.__name__ = "CheckJobSpec"
scout_checks_api_CheckJobSpec.__qualname__ = "CheckJobSpec"
scout_checks_api_CheckJobSpec.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ChecklistEntry(ConjureUnionType):
    _check: Optional["scout_checks_api_Check"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'check': ConjureFieldDefinition('check', scout_checks_api_Check)
        }

    def __init__(
            self,
            check: Optional["scout_checks_api_Check"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (check is not None) != 1:
                raise ValueError('a union must contain a single member')

            if check is not None:
                self._check = check
                self._type = 'check'

        elif type_of_union == 'check':
            if check is None:
                raise ValueError('a union value must not be None')
            self._check = check
            self._type = 'check'

    @builtins.property
    def check(self) -> Optional["scout_checks_api_Check"]:
        return self._check

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_ChecklistEntryVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_ChecklistEntryVisitor'.format(visitor.__class__.__name__))
        if self._type == 'check' and self.check is not None:
            return visitor._check(self.check)


scout_checks_api_ChecklistEntry.__name__ = "ChecklistEntry"
scout_checks_api_ChecklistEntry.__qualname__ = "ChecklistEntry"
scout_checks_api_ChecklistEntry.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ChecklistEntryVisitor:

    @abstractmethod
    def _check(self, check: "scout_checks_api_Check") -> Any:
        pass


scout_checks_api_ChecklistEntryVisitor.__name__ = "ChecklistEntryVisitor"
scout_checks_api_ChecklistEntryVisitor.__qualname__ = "ChecklistEntryVisitor"
scout_checks_api_ChecklistEntryVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ChecklistMetadata(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'author_rid': ConjureFieldDefinition('authorRid', scout_rids_api_UserRid),
            'assignee_rid': ConjureFieldDefinition('assigneeRid', scout_rids_api_UserRid),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', str),
            'created_at': ConjureFieldDefinition('createdAt', str),
            'properties': ConjureFieldDefinition('properties', Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[scout_run_api_Label]),
            'last_used': ConjureFieldDefinition('lastUsed', OptionalTypeWrapper[str]),
            'is_archived': ConjureFieldDefinition('isArchived', bool),
            'is_published': ConjureFieldDefinition('isPublished', bool)
        }

    __slots__: List[str] = ['_author_rid', '_assignee_rid', '_title', '_description', '_created_at', '_properties', '_labels', '_last_used', '_is_archived', '_is_published']

    def __init__(self, assignee_rid: str, author_rid: str, created_at: str, description: str, is_archived: bool, is_published: bool, labels: List[str], properties: Dict[str, str], title: str, last_used: Optional[str] = None) -> None:
        self._author_rid = author_rid
        self._assignee_rid = assignee_rid
        self._title = title
        self._description = description
        self._created_at = created_at
        self._properties = properties
        self._labels = labels
        self._last_used = last_used
        self._is_archived = is_archived
        self._is_published = is_published

    @builtins.property
    def author_rid(self) -> str:
        return self._author_rid

    @builtins.property
    def assignee_rid(self) -> str:
        return self._assignee_rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def created_at(self) -> str:
        return self._created_at

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def last_used(self) -> Optional[str]:
        return self._last_used

    @builtins.property
    def is_archived(self) -> bool:
        return self._is_archived

    @builtins.property
    def is_published(self) -> bool:
        return self._is_published


scout_checks_api_ChecklistMetadata.__name__ = "ChecklistMetadata"
scout_checks_api_ChecklistMetadata.__qualname__ = "ChecklistMetadata"
scout_checks_api_ChecklistMetadata.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ChecklistRef(ConjureBeanType):
    """
    A reference to a checklist that may be pinned to a specific commit.
If commit is empty, this refers to "the latest commit on main".
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', scout_rids_api_ChecklistRid),
            'commit': ConjureFieldDefinition('commit', OptionalTypeWrapper[scout_versioning_api_CommitId])
        }

    __slots__: List[str] = ['_rid', '_commit']

    def __init__(self, rid: str, commit: Optional[str] = None) -> None:
        self._rid = rid
        self._commit = commit

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def commit(self) -> Optional[str]:
        return self._commit


scout_checks_api_ChecklistRef.__name__ = "ChecklistRef"
scout_checks_api_ChecklistRef.__qualname__ = "ChecklistRef"
scout_checks_api_ChecklistRef.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ChecklistSearchQuery(ConjureUnionType):
    _and_: Optional[List["scout_checks_api_ChecklistSearchQuery"]] = None
    _or_: Optional[List["scout_checks_api_ChecklistSearchQuery"]] = None
    _search_text: Optional[str] = None
    _label: Optional[str] = None
    _property: Optional["scout_run_api_Property"] = None
    _author_rid: Optional[str] = None
    _assignee_rid: Optional[str] = None
    _is_published: Optional[bool] = None
    _not_: Optional["scout_checks_api_ChecklistSearchQuery"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'and_': ConjureFieldDefinition('and', List[scout_checks_api_ChecklistSearchQuery]),
            'or_': ConjureFieldDefinition('or', List[scout_checks_api_ChecklistSearchQuery]),
            'search_text': ConjureFieldDefinition('searchText', str),
            'label': ConjureFieldDefinition('label', scout_run_api_Label),
            'property': ConjureFieldDefinition('property', scout_run_api_Property),
            'author_rid': ConjureFieldDefinition('authorRid', scout_rids_api_UserRid),
            'assignee_rid': ConjureFieldDefinition('assigneeRid', scout_rids_api_UserRid),
            'is_published': ConjureFieldDefinition('isPublished', bool),
            'not_': ConjureFieldDefinition('not', scout_checks_api_ChecklistSearchQuery)
        }

    def __init__(
            self,
            and_: Optional[List["scout_checks_api_ChecklistSearchQuery"]] = None,
            or_: Optional[List["scout_checks_api_ChecklistSearchQuery"]] = None,
            search_text: Optional[str] = None,
            label: Optional[str] = None,
            property: Optional["scout_run_api_Property"] = None,
            author_rid: Optional[str] = None,
            assignee_rid: Optional[str] = None,
            is_published: Optional[bool] = None,
            not_: Optional["scout_checks_api_ChecklistSearchQuery"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (and_ is not None) + (or_ is not None) + (search_text is not None) + (label is not None) + (property is not None) + (author_rid is not None) + (assignee_rid is not None) + (is_published is not None) + (not_ is not None) != 1:
                raise ValueError('a union must contain a single member')

            if and_ is not None:
                self._and_ = and_
                self._type = 'and'
            if or_ is not None:
                self._or_ = or_
                self._type = 'or'
            if search_text is not None:
                self._search_text = search_text
                self._type = 'searchText'
            if label is not None:
                self._label = label
                self._type = 'label'
            if property is not None:
                self._property = property
                self._type = 'property'
            if author_rid is not None:
                self._author_rid = author_rid
                self._type = 'authorRid'
            if assignee_rid is not None:
                self._assignee_rid = assignee_rid
                self._type = 'assigneeRid'
            if is_published is not None:
                self._is_published = is_published
                self._type = 'isPublished'
            if not_ is not None:
                self._not_ = not_
                self._type = 'not'

        elif type_of_union == 'and':
            if and_ is None:
                raise ValueError('a union value must not be None')
            self._and_ = and_
            self._type = 'and'
        elif type_of_union == 'or':
            if or_ is None:
                raise ValueError('a union value must not be None')
            self._or_ = or_
            self._type = 'or'
        elif type_of_union == 'searchText':
            if search_text is None:
                raise ValueError('a union value must not be None')
            self._search_text = search_text
            self._type = 'searchText'
        elif type_of_union == 'label':
            if label is None:
                raise ValueError('a union value must not be None')
            self._label = label
            self._type = 'label'
        elif type_of_union == 'property':
            if property is None:
                raise ValueError('a union value must not be None')
            self._property = property
            self._type = 'property'
        elif type_of_union == 'authorRid':
            if author_rid is None:
                raise ValueError('a union value must not be None')
            self._author_rid = author_rid
            self._type = 'authorRid'
        elif type_of_union == 'assigneeRid':
            if assignee_rid is None:
                raise ValueError('a union value must not be None')
            self._assignee_rid = assignee_rid
            self._type = 'assigneeRid'
        elif type_of_union == 'isPublished':
            if is_published is None:
                raise ValueError('a union value must not be None')
            self._is_published = is_published
            self._type = 'isPublished'
        elif type_of_union == 'not':
            if not_ is None:
                raise ValueError('a union value must not be None')
            self._not_ = not_
            self._type = 'not'

    @builtins.property
    def and_(self) -> Optional[List["scout_checks_api_ChecklistSearchQuery"]]:
        return self._and_

    @builtins.property
    def or_(self) -> Optional[List["scout_checks_api_ChecklistSearchQuery"]]:
        return self._or_

    @builtins.property
    def search_text(self) -> Optional[str]:
        return self._search_text

    @builtins.property
    def label(self) -> Optional[str]:
        return self._label

    @builtins.property
    def property(self) -> Optional["scout_run_api_Property"]:
        return self._property

    @builtins.property
    def author_rid(self) -> Optional[str]:
        return self._author_rid

    @builtins.property
    def assignee_rid(self) -> Optional[str]:
        return self._assignee_rid

    @builtins.property
    def is_published(self) -> Optional[bool]:
        return self._is_published

    @builtins.property
    def not_(self) -> Optional["scout_checks_api_ChecklistSearchQuery"]:
        return self._not_

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_ChecklistSearchQueryVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_ChecklistSearchQueryVisitor'.format(visitor.__class__.__name__))
        if self._type == 'and' and self.and_ is not None:
            return visitor._and(self.and_)
        if self._type == 'or' and self.or_ is not None:
            return visitor._or(self.or_)
        if self._type == 'searchText' and self.search_text is not None:
            return visitor._search_text(self.search_text)
        if self._type == 'label' and self.label is not None:
            return visitor._label(self.label)
        if self._type == 'property' and self.property is not None:
            return visitor._property(self.property)
        if self._type == 'authorRid' and self.author_rid is not None:
            return visitor._author_rid(self.author_rid)
        if self._type == 'assigneeRid' and self.assignee_rid is not None:
            return visitor._assignee_rid(self.assignee_rid)
        if self._type == 'isPublished' and self.is_published is not None:
            return visitor._is_published(self.is_published)
        if self._type == 'not' and self.not_ is not None:
            return visitor._not(self.not_)


scout_checks_api_ChecklistSearchQuery.__name__ = "ChecklistSearchQuery"
scout_checks_api_ChecklistSearchQuery.__qualname__ = "ChecklistSearchQuery"
scout_checks_api_ChecklistSearchQuery.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ChecklistSearchQueryVisitor:

    @abstractmethod
    def _and(self, and_: List["scout_checks_api_ChecklistSearchQuery"]) -> Any:
        pass

    @abstractmethod
    def _or(self, or_: List["scout_checks_api_ChecklistSearchQuery"]) -> Any:
        pass

    @abstractmethod
    def _search_text(self, search_text: str) -> Any:
        pass

    @abstractmethod
    def _label(self, label: str) -> Any:
        pass

    @abstractmethod
    def _property(self, property: "scout_run_api_Property") -> Any:
        pass

    @abstractmethod
    def _author_rid(self, author_rid: str) -> Any:
        pass

    @abstractmethod
    def _assignee_rid(self, assignee_rid: str) -> Any:
        pass

    @abstractmethod
    def _is_published(self, is_published: bool) -> Any:
        pass

    @abstractmethod
    def _not(self, not_: "scout_checks_api_ChecklistSearchQuery") -> Any:
        pass


scout_checks_api_ChecklistSearchQueryVisitor.__name__ = "ChecklistSearchQueryVisitor"
scout_checks_api_ChecklistSearchQueryVisitor.__qualname__ = "ChecklistSearchQueryVisitor"
scout_checks_api_ChecklistSearchQueryVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ChecklistService(Service):
    """
    The Checklist Service is responsible for managing checklists and checks.
A checklist is a collection of checks and functions that can be executed against a set of data sources.
    """

    def create(self, auth_header: str, request: "scout_checks_api_CreateChecklistRequest") -> "scout_checks_api_VersionedChecklist":
        """
        Creates a new checklist with the provided checks and functions.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_VersionedChecklist, self._return_none_for_unknown_union_types)

    def commit(self, auth_header: str, checklist_rid: str, request: "scout_checks_api_CommitChecklistRequest", branch: Optional[str] = None) -> "scout_checks_api_VersionedChecklist":
        """
        Creates a permanent commit with a commit message. 
Throws if the checklist or branch doesn't exist.
Throws if the latest commit doesn't match the provided id.
Throws if you commit to an archived checklist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'branch': branch,
        }

        _path_params: Dict[str, Any] = {
            'checklistRid': checklist_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists/{checklistRid}/commit'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_VersionedChecklist, self._return_none_for_unknown_union_types)

    def save_working_state(self, auth_header: str, checklist_rid: str, request: "scout_checks_api_SaveChecklistRequest", branch: Optional[str] = None) -> "scout_checks_api_VersionedChecklist":
        """
        Creates a commit that may be compacted, e.g cleaned up and not exist anymore.
Throws if the checklist or branch doesn't exist.
Throws if the latest commit doesn't match the provided id.
Throws if you save to an archived checklist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'branch': branch,
        }

        _path_params: Dict[str, Any] = {
            'checklistRid': checklist_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists/{checklistRid}/save-working-state'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_VersionedChecklist, self._return_none_for_unknown_union_types)

    def merge_to_main(self, auth_header: str, checklist_rid: str, request: "scout_checks_api_MergeToMainRequest") -> "scout_checks_api_VersionedChecklist":
        """
        Merges the given branch to the "main" branch.
Throws if the checklist or branch doesn't exist.
Throws if the latest commit doesn't match the provided id.
Throws if you merge with an archived checklist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'checklistRid': checklist_rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists/{checklistRid}/merge-to-main'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_VersionedChecklist, self._return_none_for_unknown_union_types)

    def update_data_source_ref_names(self, auth_header: str, checklist_rid: str, ref_name_updates: Dict[str, str], branch: Optional[str] = None) -> "scout_checks_api_VersionedChecklist":
        """
        Updates the data source ref names for all checks within a checklist.
Throws if the checklist doesn't exist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'branch': branch,
        }

        _path_params: Dict[str, Any] = {
            'checklistRid': checklist_rid,
        }

        _json: Any = ConjureEncoder().default(ref_name_updates)

        _path = '/scout/v1/checklists/{checklistRid}/update-ref-names'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_VersionedChecklist, self._return_none_for_unknown_union_types)

    def update_metadata(self, auth_header: str, request: "scout_checks_api_UpdateChecklistMetadataRequest", rid: str) -> "scout_checks_api_ChecklistMetadata":
        """
        Updates the metadata of a checklist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists/{rid}/update-metadata'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'PUT',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_ChecklistMetadata, self._return_none_for_unknown_union_types)

    def get(self, auth_header: str, rid: str, branch: Optional[str] = None, commit: Optional[str] = None) -> "scout_checks_api_VersionedChecklist":
        """
        Specify at most one of (branch, commit).
If neither is specified, branch = "main" is the default.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
            'branch': branch,
            'commit': commit,
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/checklists/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_VersionedChecklist, self._return_none_for_unknown_union_types)

    def batch_get(self, auth_header: str, checklist_refs: List["scout_checks_api_PinnedChecklistRef"] = None) -> List["scout_checks_api_VersionedChecklist"]:
        """
        Returns the pinned commit for each provided checklist reference.
        """
        checklist_refs = checklist_refs if checklist_refs is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(checklist_refs)

        _path = '/scout/v1/checklists/batch-get'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_checks_api_VersionedChecklist], self._return_none_for_unknown_union_types)

    def batch_get_metadata(self, auth_header: str, request: "scout_checks_api_BatchGetChecklistMetadataRequest") -> "scout_checks_api_BatchGetChecklistMetadataResponse":
        """
        Returns the metadata for each provided checklist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists/batch-get-metadata'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_BatchGetChecklistMetadataResponse, self._return_none_for_unknown_union_types)

    def search(self, auth_header: str, request: "scout_checks_api_SearchChecklistsRequest") -> "scout_checks_api_VersionedChecklistPage":
        """
        Results will be the latest commit on main for each checklist.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists/search'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_VersionedChecklistPage, self._return_none_for_unknown_union_types)

    def archive(self, auth_header: str, request: "scout_checks_api_ArchiveChecklistsRequest") -> None:
        """
        Archives the provided checklists.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists/archive'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def unarchive(self, auth_header: str, request: "scout_checks_api_UnarchiveChecklistsRequest") -> None:
        """
        Unarchives the provided checklists.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(request)

        _path = '/scout/v1/checklists/unarchive'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        return

    def get_check(self, auth_header: str, rid: str) -> "scout_checks_api_Check":
        """
        Returns the check with the given rid.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
            'rid': rid,
        }

        _json: Any = None

        _path = '/scout/v1/checklists/check/{rid}'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_Check, self._return_none_for_unknown_union_types)

    def batch_get_checks(self, auth_header: str, rids: List[str] = None) -> List["scout_checks_api_Check"]:
        """
        Returns the checks with the given rids.
        """
        rids = rids if rids is not None else []

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = ConjureEncoder().default(rids)

        _path = '/scout/v1/checklists/check/batch-get'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'POST',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), List[scout_checks_api_Check], self._return_none_for_unknown_union_types)

    def get_all_labels_and_properties(self, auth_header: str) -> "scout_checks_api_GetAllLabelsAndPropertiesResponse":
        """
        Returns all labels and properties.
        """

        _headers: Dict[str, Any] = {
            'Accept': 'application/json',
            'Authorization': auth_header,
        }

        _params: Dict[str, Any] = {
        }

        _path_params: Dict[str, Any] = {
        }

        _json: Any = None

        _path = '/scout/v1/checklists/get-all-labels-properties'
        _path = _path.format(**_path_params)

        _response: Response = self._request(
            'GET',
            self._uri + _path,
            params=_params,
            headers=_headers,
            json=_json)

        _decoder = ConjureDecoder()
        return _decoder.decode(_response.json(), scout_checks_api_GetAllLabelsAndPropertiesResponse, self._return_none_for_unknown_union_types)


scout_checks_api_ChecklistService.__name__ = "ChecklistService"
scout_checks_api_ChecklistService.__qualname__ = "ChecklistService"
scout_checks_api_ChecklistService.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ChecklistVariable(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', scout_compute_api_VariableName),
            'display_name': ConjureFieldDefinition('displayName', OptionalTypeWrapper[str]),
            'value': ConjureFieldDefinition('value', scout_checks_api_VariableLocator)
        }

    __slots__: List[str] = ['_name', '_display_name', '_value']

    def __init__(self, name: str, value: "scout_checks_api_VariableLocator", display_name: Optional[str] = None) -> None:
        self._name = name
        self._display_name = display_name
        self._value = value

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def display_name(self) -> Optional[str]:
        return self._display_name

    @builtins.property
    def value(self) -> "scout_checks_api_VariableLocator":
        return self._value


scout_checks_api_ChecklistVariable.__name__ = "ChecklistVariable"
scout_checks_api_ChecklistVariable.__qualname__ = "ChecklistVariable"
scout_checks_api_ChecklistVariable.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CommitChecklistRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'commit_message': ConjureFieldDefinition('commitMessage', str),
            'checks': ConjureFieldDefinition('checks', List[scout_checks_api_UpdateChecklistEntryRequest]),
            'functions': ConjureFieldDefinition('functions', Dict[scout_compute_api_FunctionReference, scout_checks_api_UpdateFunctionEntryRequest]),
            'checklist_variables': ConjureFieldDefinition('checklistVariables', List[scout_checks_api_UnresolvedChecklistVariable]),
            'latest_commit': ConjureFieldDefinition('latestCommit', OptionalTypeWrapper[scout_versioning_api_CommitId])
        }

    __slots__: List[str] = ['_commit_message', '_checks', '_functions', '_checklist_variables', '_latest_commit']

    def __init__(self, checklist_variables: List["scout_checks_api_UnresolvedChecklistVariable"], checks: List["scout_checks_api_UpdateChecklistEntryRequest"], commit_message: str, functions: Dict[str, "scout_checks_api_UpdateFunctionEntryRequest"], latest_commit: Optional[str] = None) -> None:
        self._commit_message = commit_message
        self._checks = checks
        self._functions = functions
        self._checklist_variables = checklist_variables
        self._latest_commit = latest_commit

    @builtins.property
    def commit_message(self) -> str:
        return self._commit_message

    @builtins.property
    def checks(self) -> List["scout_checks_api_UpdateChecklistEntryRequest"]:
        return self._checks

    @builtins.property
    def functions(self) -> Dict[str, "scout_checks_api_UpdateFunctionEntryRequest"]:
        """
        The keys of the map are references that can be used for checks to reference functions in the same request, before the function RIDs have been created.
        """
        return self._functions

    @builtins.property
    def checklist_variables(self) -> List["scout_checks_api_UnresolvedChecklistVariable"]:
        """
        Variables that can be used in checks and functions. Variables are resolved in order of declaration.
If variable `a` depends on variable `b`, then `b` must be defined before `a` in the list.
        """
        return self._checklist_variables

    @builtins.property
    def latest_commit(self) -> Optional[str]:
        """
        If present, will validate that the latest commit matches this id,
and otherwise throw CommitConflict.
        """
        return self._latest_commit


scout_checks_api_CommitChecklistRequest.__name__ = "CommitChecklistRequest"
scout_checks_api_CommitChecklistRequest.__qualname__ = "CommitChecklistRequest"
scout_checks_api_CommitChecklistRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_Completed(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'result': ConjureFieldDefinition('result', scout_checks_api_JobResult),
            'execution_start_time': ConjureFieldDefinition('executionStartTime', str),
            'execution_end_time': ConjureFieldDefinition('executionEndTime', str)
        }

    __slots__: List[str] = ['_result', '_execution_start_time', '_execution_end_time']

    def __init__(self, execution_end_time: str, execution_start_time: str, result: "scout_checks_api_JobResult") -> None:
        self._result = result
        self._execution_start_time = execution_start_time
        self._execution_end_time = execution_end_time

    @builtins.property
    def result(self) -> "scout_checks_api_JobResult":
        return self._result

    @builtins.property
    def execution_start_time(self) -> str:
        return self._execution_start_time

    @builtins.property
    def execution_end_time(self) -> str:
        return self._execution_end_time


scout_checks_api_Completed.__name__ = "Completed"
scout_checks_api_Completed.__qualname__ = "Completed"
scout_checks_api_Completed.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ComputeNodeWithContext(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'series_node': ConjureFieldDefinition('seriesNode', scout_compute_api_ComputeNode),
            'context': ConjureFieldDefinition('context', scout_checks_api_CheckContext)
        }

    __slots__: List[str] = ['_series_node', '_context']

    def __init__(self, context: "scout_checks_api_CheckContext", series_node: "scout_compute_api_ComputeNode") -> None:
        self._series_node = series_node
        self._context = context

    @builtins.property
    def series_node(self) -> "scout_compute_api_ComputeNode":
        return self._series_node

    @builtins.property
    def context(self) -> "scout_checks_api_CheckContext":
        return self._context


scout_checks_api_ComputeNodeWithContext.__name__ = "ComputeNodeWithContext"
scout_checks_api_ComputeNodeWithContext.__qualname__ = "ComputeNodeWithContext"
scout_checks_api_ComputeNodeWithContext.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CreateCheckRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'check_lineage_rid': ConjureFieldDefinition('checkLineageRid', OptionalTypeWrapper[str]),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', str),
            'auto_generated_title': ConjureFieldDefinition('autoGeneratedTitle', OptionalTypeWrapper[str]),
            'auto_generated_description': ConjureFieldDefinition('autoGeneratedDescription', OptionalTypeWrapper[str]),
            'priority': ConjureFieldDefinition('priority', scout_checks_api_Priority),
            'chart': ConjureFieldDefinition('chart', OptionalTypeWrapper[scout_rids_api_VersionedVizId]),
            'condition': ConjureFieldDefinition('condition', OptionalTypeWrapper[scout_checks_api_UnresolvedCheckCondition])
        }

    __slots__: List[str] = ['_check_lineage_rid', '_title', '_description', '_auto_generated_title', '_auto_generated_description', '_priority', '_chart', '_condition']

    def __init__(self, description: str, priority: "scout_checks_api_Priority", title: str, auto_generated_description: Optional[str] = None, auto_generated_title: Optional[str] = None, chart: Optional["scout_rids_api_VersionedVizId"] = None, check_lineage_rid: Optional[str] = None, condition: Optional["scout_checks_api_UnresolvedCheckCondition"] = None) -> None:
        self._check_lineage_rid = check_lineage_rid
        self._title = title
        self._description = description
        self._auto_generated_title = auto_generated_title
        self._auto_generated_description = auto_generated_description
        self._priority = priority
        self._chart = chart
        self._condition = condition

    @builtins.property
    def check_lineage_rid(self) -> Optional[str]:
        """
        Identifies the lineage of checks this check belongs to. If not specified, a new lineage will be created.
This is named checkLineageRid for historical reasons but is actually a UUID.
        """
        return self._check_lineage_rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def auto_generated_title(self) -> Optional[str]:
        return self._auto_generated_title

    @builtins.property
    def auto_generated_description(self) -> Optional[str]:
        return self._auto_generated_description

    @builtins.property
    def priority(self) -> "scout_checks_api_Priority":
        return self._priority

    @builtins.property
    def chart(self) -> Optional["scout_rids_api_VersionedVizId"]:
        return self._chart

    @builtins.property
    def condition(self) -> Optional["scout_checks_api_UnresolvedCheckCondition"]:
        """
        If omitted, this check represents a manual check.
        """
        return self._condition


scout_checks_api_CreateCheckRequest.__name__ = "CreateCheckRequest"
scout_checks_api_CreateCheckRequest.__qualname__ = "CreateCheckRequest"
scout_checks_api_CreateCheckRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CreateChecklistEntryRequest(ConjureUnionType):
    _create_check: Optional["scout_checks_api_CreateCheckRequest"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'create_check': ConjureFieldDefinition('createCheck', scout_checks_api_CreateCheckRequest)
        }

    def __init__(
            self,
            create_check: Optional["scout_checks_api_CreateCheckRequest"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (create_check is not None) != 1:
                raise ValueError('a union must contain a single member')

            if create_check is not None:
                self._create_check = create_check
                self._type = 'createCheck'

        elif type_of_union == 'createCheck':
            if create_check is None:
                raise ValueError('a union value must not be None')
            self._create_check = create_check
            self._type = 'createCheck'

    @builtins.property
    def create_check(self) -> Optional["scout_checks_api_CreateCheckRequest"]:
        return self._create_check

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_CreateChecklistEntryRequestVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_CreateChecklistEntryRequestVisitor'.format(visitor.__class__.__name__))
        if self._type == 'createCheck' and self.create_check is not None:
            return visitor._create_check(self.create_check)


scout_checks_api_CreateChecklistEntryRequest.__name__ = "CreateChecklistEntryRequest"
scout_checks_api_CreateChecklistEntryRequest.__qualname__ = "CreateChecklistEntryRequest"
scout_checks_api_CreateChecklistEntryRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CreateChecklistEntryRequestVisitor:

    @abstractmethod
    def _create_check(self, create_check: "scout_checks_api_CreateCheckRequest") -> Any:
        pass


scout_checks_api_CreateChecklistEntryRequestVisitor.__name__ = "CreateChecklistEntryRequestVisitor"
scout_checks_api_CreateChecklistEntryRequestVisitor.__qualname__ = "CreateChecklistEntryRequestVisitor"
scout_checks_api_CreateChecklistEntryRequestVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CreateChecklistRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'commit_message': ConjureFieldDefinition('commitMessage', str),
            'assignee_rid': ConjureFieldDefinition('assigneeRid', scout_rids_api_UserRid),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', str),
            'functions': ConjureFieldDefinition('functions', Dict[scout_compute_api_FunctionReference, scout_checks_api_CreateFunctionRequest]),
            'checks': ConjureFieldDefinition('checks', List[scout_checks_api_CreateChecklistEntryRequest]),
            'properties': ConjureFieldDefinition('properties', Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue]),
            'labels': ConjureFieldDefinition('labels', List[scout_run_api_Label]),
            'checklist_variables': ConjureFieldDefinition('checklistVariables', List[scout_checks_api_UnresolvedChecklistVariable]),
            'is_published': ConjureFieldDefinition('isPublished', OptionalTypeWrapper[bool])
        }

    __slots__: List[str] = ['_commit_message', '_assignee_rid', '_title', '_description', '_functions', '_checks', '_properties', '_labels', '_checklist_variables', '_is_published']

    def __init__(self, assignee_rid: str, checklist_variables: List["scout_checks_api_UnresolvedChecklistVariable"], checks: List["scout_checks_api_CreateChecklistEntryRequest"], commit_message: str, description: str, functions: Dict[str, "scout_checks_api_CreateFunctionRequest"], labels: List[str], properties: Dict[str, str], title: str, is_published: Optional[bool] = None) -> None:
        self._commit_message = commit_message
        self._assignee_rid = assignee_rid
        self._title = title
        self._description = description
        self._functions = functions
        self._checks = checks
        self._properties = properties
        self._labels = labels
        self._checklist_variables = checklist_variables
        self._is_published = is_published

    @builtins.property
    def commit_message(self) -> str:
        return self._commit_message

    @builtins.property
    def assignee_rid(self) -> str:
        return self._assignee_rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def functions(self) -> Dict[str, "scout_checks_api_CreateFunctionRequest"]:
        """
        The keys of the map are references that can be used for checks to reference functions in the same request, before the function RIDs have been created.
        """
        return self._functions

    @builtins.property
    def checks(self) -> List["scout_checks_api_CreateChecklistEntryRequest"]:
        return self._checks

    @builtins.property
    def properties(self) -> Dict[str, str]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels

    @builtins.property
    def checklist_variables(self) -> List["scout_checks_api_UnresolvedChecklistVariable"]:
        """
        Variables that can be used in checks and functions. Variables are resolved in order of declaration.
If variable `a` depends on variable `b`, then `b` must be defined before `a` in the list.
        """
        return self._checklist_variables

    @builtins.property
    def is_published(self) -> Optional[bool]:
        """
        Default value is true.
        """
        return self._is_published


scout_checks_api_CreateChecklistRequest.__name__ = "CreateChecklistRequest"
scout_checks_api_CreateChecklistRequest.__qualname__ = "CreateChecklistRequest"
scout_checks_api_CreateChecklistRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_CreateFunctionRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'function_lineage_rid': ConjureFieldDefinition('functionLineageRid', OptionalTypeWrapper[scout_rids_api_FunctionLineageRid]),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', str),
            'auto_generated_title': ConjureFieldDefinition('autoGeneratedTitle', OptionalTypeWrapper[str]),
            'auto_generated_description': ConjureFieldDefinition('autoGeneratedDescription', OptionalTypeWrapper[str]),
            'node': ConjureFieldDefinition('node', scout_checks_api_FunctionNode)
        }

    __slots__: List[str] = ['_function_lineage_rid', '_title', '_description', '_auto_generated_title', '_auto_generated_description', '_node']

    def __init__(self, description: str, node: "scout_checks_api_FunctionNode", title: str, auto_generated_description: Optional[str] = None, auto_generated_title: Optional[str] = None, function_lineage_rid: Optional[str] = None) -> None:
        self._function_lineage_rid = function_lineage_rid
        self._title = title
        self._description = description
        self._auto_generated_title = auto_generated_title
        self._auto_generated_description = auto_generated_description
        self._node = node

    @builtins.property
    def function_lineage_rid(self) -> Optional[str]:
        """
        Identifies the lineage of functions this function belongs to. If not specified, a new lineage will be created.
        """
        return self._function_lineage_rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def auto_generated_title(self) -> Optional[str]:
        return self._auto_generated_title

    @builtins.property
    def auto_generated_description(self) -> Optional[str]:
        return self._auto_generated_description

    @builtins.property
    def node(self) -> "scout_checks_api_FunctionNode":
        return self._node


scout_checks_api_CreateFunctionRequest.__name__ = "CreateFunctionRequest"
scout_checks_api_CreateFunctionRequest.__qualname__ = "CreateFunctionRequest"
scout_checks_api_CreateFunctionRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_DeprecatedCheckJobSpec(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_review_rid': ConjureFieldDefinition('dataReviewRid', scout_datareview_api_DataReviewRid),
            'check_rid': ConjureFieldDefinition('checkRid', scout_rids_api_CheckRid),
            'check_evaluation_rid': ConjureFieldDefinition('checkEvaluationRid', scout_datareview_api_AutomaticCheckEvaluationRid),
            'check_condition': ConjureFieldDefinition('checkCondition', scout_checks_api_CheckCondition),
            'start': ConjureFieldDefinition('start', api_Timestamp),
            'end': ConjureFieldDefinition('end', api_Timestamp),
            'context': ConjureFieldDefinition('context', scout_compute_api_deprecated_Context)
        }

    __slots__: List[str] = ['_data_review_rid', '_check_rid', '_check_evaluation_rid', '_check_condition', '_start', '_end', '_context']

    def __init__(self, check_condition: "scout_checks_api_CheckCondition", check_evaluation_rid: str, check_rid: str, context: "scout_compute_api_deprecated_Context", data_review_rid: str, end: "api_Timestamp", start: "api_Timestamp") -> None:
        self._data_review_rid = data_review_rid
        self._check_rid = check_rid
        self._check_evaluation_rid = check_evaluation_rid
        self._check_condition = check_condition
        self._start = start
        self._end = end
        self._context = context

    @builtins.property
    def data_review_rid(self) -> str:
        return self._data_review_rid

    @builtins.property
    def check_rid(self) -> str:
        return self._check_rid

    @builtins.property
    def check_evaluation_rid(self) -> str:
        return self._check_evaluation_rid

    @builtins.property
    def check_condition(self) -> "scout_checks_api_CheckCondition":
        return self._check_condition

    @builtins.property
    def start(self) -> "api_Timestamp":
        return self._start

    @builtins.property
    def end(self) -> "api_Timestamp":
        return self._end

    @builtins.property
    def context(self) -> "scout_compute_api_deprecated_Context":
        return self._context


scout_checks_api_DeprecatedCheckJobSpec.__name__ = "DeprecatedCheckJobSpec"
scout_checks_api_DeprecatedCheckJobSpec.__qualname__ = "DeprecatedCheckJobSpec"
scout_checks_api_DeprecatedCheckJobSpec.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_Failed(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'failure_message': ConjureFieldDefinition('failureMessage', OptionalTypeWrapper[str]),
            'execution_start_time': ConjureFieldDefinition('executionStartTime', OptionalTypeWrapper[str]),
            'execution_end_time': ConjureFieldDefinition('executionEndTime', str)
        }

    __slots__: List[str] = ['_failure_message', '_execution_start_time', '_execution_end_time']

    def __init__(self, execution_end_time: str, execution_start_time: Optional[str] = None, failure_message: Optional[str] = None) -> None:
        self._failure_message = failure_message
        self._execution_start_time = execution_start_time
        self._execution_end_time = execution_end_time

    @builtins.property
    def failure_message(self) -> Optional[str]:
        return self._failure_message

    @builtins.property
    def execution_start_time(self) -> Optional[str]:
        return self._execution_start_time

    @builtins.property
    def execution_end_time(self) -> str:
        return self._execution_end_time


scout_checks_api_Failed.__name__ = "Failed"
scout_checks_api_Failed.__qualname__ = "Failed"
scout_checks_api_Failed.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_Function(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', scout_rids_api_FunctionRid),
            'function_lineage_rid': ConjureFieldDefinition('functionLineageRid', scout_rids_api_FunctionLineageRid),
            'title': ConjureFieldDefinition('title', str),
            'description': ConjureFieldDefinition('description', str),
            'auto_generated_title': ConjureFieldDefinition('autoGeneratedTitle', OptionalTypeWrapper[str]),
            'auto_generated_description': ConjureFieldDefinition('autoGeneratedDescription', OptionalTypeWrapper[str]),
            'function_node': ConjureFieldDefinition('functionNode', scout_checks_api_FunctionNode)
        }

    __slots__: List[str] = ['_rid', '_function_lineage_rid', '_title', '_description', '_auto_generated_title', '_auto_generated_description', '_function_node']

    def __init__(self, description: str, function_lineage_rid: str, function_node: "scout_checks_api_FunctionNode", rid: str, title: str, auto_generated_description: Optional[str] = None, auto_generated_title: Optional[str] = None) -> None:
        self._rid = rid
        self._function_lineage_rid = function_lineage_rid
        self._title = title
        self._description = description
        self._auto_generated_title = auto_generated_title
        self._auto_generated_description = auto_generated_description
        self._function_node = function_node

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def function_lineage_rid(self) -> str:
        return self._function_lineage_rid

    @builtins.property
    def title(self) -> str:
        return self._title

    @builtins.property
    def description(self) -> str:
        return self._description

    @builtins.property
    def auto_generated_title(self) -> Optional[str]:
        return self._auto_generated_title

    @builtins.property
    def auto_generated_description(self) -> Optional[str]:
        return self._auto_generated_description

    @builtins.property
    def function_node(self) -> "scout_checks_api_FunctionNode":
        return self._function_node


scout_checks_api_Function.__name__ = "Function"
scout_checks_api_Function.__qualname__ = "Function"
scout_checks_api_Function.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_FunctionNode(ConjureUnionType):
    _enum: Optional["scout_compute_api_EnumSeries"] = None
    _numeric: Optional["scout_compute_api_NumericSeries"] = None
    _ranges: Optional["scout_compute_api_RangeSeries"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'enum': ConjureFieldDefinition('enum', scout_compute_api_EnumSeries),
            'numeric': ConjureFieldDefinition('numeric', scout_compute_api_NumericSeries),
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_RangeSeries)
        }

    def __init__(
            self,
            enum: Optional["scout_compute_api_EnumSeries"] = None,
            numeric: Optional["scout_compute_api_NumericSeries"] = None,
            ranges: Optional["scout_compute_api_RangeSeries"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (enum is not None) + (numeric is not None) + (ranges is not None) != 1:
                raise ValueError('a union must contain a single member')

            if enum is not None:
                self._enum = enum
                self._type = 'enum'
            if numeric is not None:
                self._numeric = numeric
                self._type = 'numeric'
            if ranges is not None:
                self._ranges = ranges
                self._type = 'ranges'

        elif type_of_union == 'enum':
            if enum is None:
                raise ValueError('a union value must not be None')
            self._enum = enum
            self._type = 'enum'
        elif type_of_union == 'numeric':
            if numeric is None:
                raise ValueError('a union value must not be None')
            self._numeric = numeric
            self._type = 'numeric'
        elif type_of_union == 'ranges':
            if ranges is None:
                raise ValueError('a union value must not be None')
            self._ranges = ranges
            self._type = 'ranges'

    @builtins.property
    def enum(self) -> Optional["scout_compute_api_EnumSeries"]:
        return self._enum

    @builtins.property
    def numeric(self) -> Optional["scout_compute_api_NumericSeries"]:
        return self._numeric

    @builtins.property
    def ranges(self) -> Optional["scout_compute_api_RangeSeries"]:
        return self._ranges

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_FunctionNodeVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_FunctionNodeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'enum' and self.enum is not None:
            return visitor._enum(self.enum)
        if self._type == 'numeric' and self.numeric is not None:
            return visitor._numeric(self.numeric)
        if self._type == 'ranges' and self.ranges is not None:
            return visitor._ranges(self.ranges)


scout_checks_api_FunctionNode.__name__ = "FunctionNode"
scout_checks_api_FunctionNode.__qualname__ = "FunctionNode"
scout_checks_api_FunctionNode.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_FunctionNodeVisitor:

    @abstractmethod
    def _enum(self, enum: "scout_compute_api_EnumSeries") -> Any:
        pass

    @abstractmethod
    def _numeric(self, numeric: "scout_compute_api_NumericSeries") -> Any:
        pass

    @abstractmethod
    def _ranges(self, ranges: "scout_compute_api_RangeSeries") -> Any:
        pass


scout_checks_api_FunctionNodeVisitor.__name__ = "FunctionNodeVisitor"
scout_checks_api_FunctionNodeVisitor.__qualname__ = "FunctionNodeVisitor"
scout_checks_api_FunctionNodeVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_GetAllLabelsAndPropertiesResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'properties': ConjureFieldDefinition('properties', Dict[scout_run_api_PropertyName, List[scout_run_api_PropertyValue]]),
            'labels': ConjureFieldDefinition('labels', List[scout_run_api_Label])
        }

    __slots__: List[str] = ['_properties', '_labels']

    def __init__(self, labels: List[str], properties: Dict[str, List[str]]) -> None:
        self._properties = properties
        self._labels = labels

    @builtins.property
    def properties(self) -> Dict[str, List[str]]:
        return self._properties

    @builtins.property
    def labels(self) -> List[str]:
        return self._labels


scout_checks_api_GetAllLabelsAndPropertiesResponse.__name__ = "GetAllLabelsAndPropertiesResponse"
scout_checks_api_GetAllLabelsAndPropertiesResponse.__qualname__ = "GetAllLabelsAndPropertiesResponse"
scout_checks_api_GetAllLabelsAndPropertiesResponse.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_InProgress(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_checks_api_InProgress.__name__ = "InProgress"
scout_checks_api_InProgress.__qualname__ = "InProgress"
scout_checks_api_InProgress.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_JobReport(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'job_status': ConjureFieldDefinition('jobStatus', scout_checks_api_JobStatus),
            'job_rid': ConjureFieldDefinition('jobRid', scout_checks_api_JobRid),
            'submitted_at': ConjureFieldDefinition('submittedAt', str)
        }

    __slots__: List[str] = ['_job_status', '_job_rid', '_submitted_at']

    def __init__(self, job_rid: str, job_status: "scout_checks_api_JobStatus", submitted_at: str) -> None:
        self._job_status = job_status
        self._job_rid = job_rid
        self._submitted_at = submitted_at

    @builtins.property
    def job_status(self) -> "scout_checks_api_JobStatus":
        return self._job_status

    @builtins.property
    def job_rid(self) -> str:
        return self._job_rid

    @builtins.property
    def submitted_at(self) -> str:
        return self._submitted_at


scout_checks_api_JobReport.__name__ = "JobReport"
scout_checks_api_JobReport.__qualname__ = "JobReport"
scout_checks_api_JobReport.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_JobResult(ConjureUnionType):
    _check_job_result: Optional["scout_checks_api_CheckJobResult"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'check_job_result': ConjureFieldDefinition('checkJobResult', scout_checks_api_CheckJobResult)
        }

    def __init__(
            self,
            check_job_result: Optional["scout_checks_api_CheckJobResult"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (check_job_result is not None) != 1:
                raise ValueError('a union must contain a single member')

            if check_job_result is not None:
                self._check_job_result = check_job_result
                self._type = 'checkJobResult'

        elif type_of_union == 'checkJobResult':
            if check_job_result is None:
                raise ValueError('a union value must not be None')
            self._check_job_result = check_job_result
            self._type = 'checkJobResult'

    @builtins.property
    def check_job_result(self) -> Optional["scout_checks_api_CheckJobResult"]:
        return self._check_job_result

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_JobResultVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_JobResultVisitor'.format(visitor.__class__.__name__))
        if self._type == 'checkJobResult' and self.check_job_result is not None:
            return visitor._check_job_result(self.check_job_result)


scout_checks_api_JobResult.__name__ = "JobResult"
scout_checks_api_JobResult.__qualname__ = "JobResult"
scout_checks_api_JobResult.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_JobResultVisitor:

    @abstractmethod
    def _check_job_result(self, check_job_result: "scout_checks_api_CheckJobResult") -> Any:
        pass


scout_checks_api_JobResultVisitor.__name__ = "JobResultVisitor"
scout_checks_api_JobResultVisitor.__qualname__ = "JobResultVisitor"
scout_checks_api_JobResultVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_JobSpec(ConjureUnionType):
    _check: Optional["scout_checks_api_DeprecatedCheckJobSpec"] = None
    _check_v2: Optional["scout_checks_api_CheckJobSpec"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'check': ConjureFieldDefinition('check', scout_checks_api_DeprecatedCheckJobSpec),
            'check_v2': ConjureFieldDefinition('checkV2', scout_checks_api_CheckJobSpec)
        }

    def __init__(
            self,
            check: Optional["scout_checks_api_DeprecatedCheckJobSpec"] = None,
            check_v2: Optional["scout_checks_api_CheckJobSpec"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (check is not None) + (check_v2 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if check is not None:
                self._check = check
                self._type = 'check'
            if check_v2 is not None:
                self._check_v2 = check_v2
                self._type = 'checkV2'

        elif type_of_union == 'check':
            if check is None:
                raise ValueError('a union value must not be None')
            self._check = check
            self._type = 'check'
        elif type_of_union == 'checkV2':
            if check_v2 is None:
                raise ValueError('a union value must not be None')
            self._check_v2 = check_v2
            self._type = 'checkV2'

    @builtins.property
    def check(self) -> Optional["scout_checks_api_DeprecatedCheckJobSpec"]:
        return self._check

    @builtins.property
    def check_v2(self) -> Optional["scout_checks_api_CheckJobSpec"]:
        return self._check_v2

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_JobSpecVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_JobSpecVisitor'.format(visitor.__class__.__name__))
        if self._type == 'check' and self.check is not None:
            return visitor._check(self.check)
        if self._type == 'checkV2' and self.check_v2 is not None:
            return visitor._check_v2(self.check_v2)


scout_checks_api_JobSpec.__name__ = "JobSpec"
scout_checks_api_JobSpec.__qualname__ = "JobSpec"
scout_checks_api_JobSpec.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_JobSpecVisitor:

    @abstractmethod
    def _check(self, check: "scout_checks_api_DeprecatedCheckJobSpec") -> Any:
        pass

    @abstractmethod
    def _check_v2(self, check_v2: "scout_checks_api_CheckJobSpec") -> Any:
        pass


scout_checks_api_JobSpecVisitor.__name__ = "JobSpecVisitor"
scout_checks_api_JobSpecVisitor.__qualname__ = "JobSpecVisitor"
scout_checks_api_JobSpecVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_JobStatus(ConjureUnionType):
    _in_progress: Optional["scout_checks_api_InProgress"] = None
    _completed: Optional["scout_checks_api_Completed"] = None
    _failed: Optional["scout_checks_api_Failed"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'in_progress': ConjureFieldDefinition('inProgress', scout_checks_api_InProgress),
            'completed': ConjureFieldDefinition('completed', scout_checks_api_Completed),
            'failed': ConjureFieldDefinition('failed', scout_checks_api_Failed)
        }

    def __init__(
            self,
            in_progress: Optional["scout_checks_api_InProgress"] = None,
            completed: Optional["scout_checks_api_Completed"] = None,
            failed: Optional["scout_checks_api_Failed"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (in_progress is not None) + (completed is not None) + (failed is not None) != 1:
                raise ValueError('a union must contain a single member')

            if in_progress is not None:
                self._in_progress = in_progress
                self._type = 'inProgress'
            if completed is not None:
                self._completed = completed
                self._type = 'completed'
            if failed is not None:
                self._failed = failed
                self._type = 'failed'

        elif type_of_union == 'inProgress':
            if in_progress is None:
                raise ValueError('a union value must not be None')
            self._in_progress = in_progress
            self._type = 'inProgress'
        elif type_of_union == 'completed':
            if completed is None:
                raise ValueError('a union value must not be None')
            self._completed = completed
            self._type = 'completed'
        elif type_of_union == 'failed':
            if failed is None:
                raise ValueError('a union value must not be None')
            self._failed = failed
            self._type = 'failed'

    @builtins.property
    def in_progress(self) -> Optional["scout_checks_api_InProgress"]:
        return self._in_progress

    @builtins.property
    def completed(self) -> Optional["scout_checks_api_Completed"]:
        return self._completed

    @builtins.property
    def failed(self) -> Optional["scout_checks_api_Failed"]:
        return self._failed

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_JobStatusVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_JobStatusVisitor'.format(visitor.__class__.__name__))
        if self._type == 'inProgress' and self.in_progress is not None:
            return visitor._in_progress(self.in_progress)
        if self._type == 'completed' and self.completed is not None:
            return visitor._completed(self.completed)
        if self._type == 'failed' and self.failed is not None:
            return visitor._failed(self.failed)


scout_checks_api_JobStatus.__name__ = "JobStatus"
scout_checks_api_JobStatus.__qualname__ = "JobStatus"
scout_checks_api_JobStatus.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_JobStatusVisitor:

    @abstractmethod
    def _in_progress(self, in_progress: "scout_checks_api_InProgress") -> Any:
        pass

    @abstractmethod
    def _completed(self, completed: "scout_checks_api_Completed") -> Any:
        pass

    @abstractmethod
    def _failed(self, failed: "scout_checks_api_Failed") -> Any:
        pass


scout_checks_api_JobStatusVisitor.__name__ = "JobStatusVisitor"
scout_checks_api_JobStatusVisitor.__qualname__ = "JobStatusVisitor"
scout_checks_api_JobStatusVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_MergeToMainRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'branch_name': ConjureFieldDefinition('branchName', scout_versioning_api_BranchName),
            'message': ConjureFieldDefinition('message', str),
            'latest_commit_on_main': ConjureFieldDefinition('latestCommitOnMain', OptionalTypeWrapper[scout_versioning_api_CommitId])
        }

    __slots__: List[str] = ['_branch_name', '_message', '_latest_commit_on_main']

    def __init__(self, branch_name: str, message: str, latest_commit_on_main: Optional[str] = None) -> None:
        self._branch_name = branch_name
        self._message = message
        self._latest_commit_on_main = latest_commit_on_main

    @builtins.property
    def branch_name(self) -> str:
        """
        If "main", the request will throw.
        """
        return self._branch_name

    @builtins.property
    def message(self) -> str:
        return self._message

    @builtins.property
    def latest_commit_on_main(self) -> Optional[str]:
        """
        If present, will validate that the latest commit matches this id,
and otherwise throw CommitConflict.
        """
        return self._latest_commit_on_main


scout_checks_api_MergeToMainRequest.__name__ = "MergeToMainRequest"
scout_checks_api_MergeToMainRequest.__qualname__ = "MergeToMainRequest"
scout_checks_api_MergeToMainRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_NumRangesConditionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_deprecated_RangesNode),
            'function_spec': ConjureFieldDefinition('functionSpec', object),
            'threshold': ConjureFieldDefinition('threshold', int),
            'operator': ConjureFieldDefinition('operator', scout_compute_api_ThresholdOperator),
            'variables': ConjureFieldDefinition('variables', Dict[scout_compute_api_VariableName, scout_api_ChannelLocator])
        }

    __slots__: List[str] = ['_ranges', '_function_spec', '_threshold', '_operator', '_variables']

    def __init__(self, function_spec: Any, operator: "scout_compute_api_ThresholdOperator", ranges: "scout_compute_api_deprecated_RangesNode", threshold: int, variables: Dict[str, "scout_api_ChannelLocator"]) -> None:
        self._ranges = ranges
        self._function_spec = function_spec
        self._threshold = threshold
        self._operator = operator
        self._variables = variables

    @builtins.property
    def ranges(self) -> "scout_compute_api_deprecated_RangesNode":
        return self._ranges

    @builtins.property
    def function_spec(self) -> Any:
        return self._function_spec

    @builtins.property
    def threshold(self) -> int:
        return self._threshold

    @builtins.property
    def operator(self) -> "scout_compute_api_ThresholdOperator":
        return self._operator

    @builtins.property
    def variables(self) -> Dict[str, "scout_api_ChannelLocator"]:
        return self._variables


scout_checks_api_NumRangesConditionV1.__name__ = "NumRangesConditionV1"
scout_checks_api_NumRangesConditionV1.__qualname__ = "NumRangesConditionV1"
scout_checks_api_NumRangesConditionV1.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_NumRangesConditionV2(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_deprecated_RangesNode),
            'function_spec': ConjureFieldDefinition('functionSpec', object),
            'threshold': ConjureFieldDefinition('threshold', int),
            'operator': ConjureFieldDefinition('operator', scout_compute_api_ThresholdOperator),
            'variables': ConjureFieldDefinition('variables', Dict[scout_compute_api_VariableName, scout_checks_api_VariableLocator])
        }

    __slots__: List[str] = ['_ranges', '_function_spec', '_threshold', '_operator', '_variables']

    def __init__(self, function_spec: Any, operator: "scout_compute_api_ThresholdOperator", ranges: "scout_compute_api_deprecated_RangesNode", threshold: int, variables: Dict[str, "scout_checks_api_VariableLocator"]) -> None:
        self._ranges = ranges
        self._function_spec = function_spec
        self._threshold = threshold
        self._operator = operator
        self._variables = variables

    @builtins.property
    def ranges(self) -> "scout_compute_api_deprecated_RangesNode":
        return self._ranges

    @builtins.property
    def function_spec(self) -> Any:
        return self._function_spec

    @builtins.property
    def threshold(self) -> int:
        return self._threshold

    @builtins.property
    def operator(self) -> "scout_compute_api_ThresholdOperator":
        return self._operator

    @builtins.property
    def variables(self) -> Dict[str, "scout_checks_api_VariableLocator"]:
        return self._variables


scout_checks_api_NumRangesConditionV2.__name__ = "NumRangesConditionV2"
scout_checks_api_NumRangesConditionV2.__qualname__ = "NumRangesConditionV2"
scout_checks_api_NumRangesConditionV2.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_NumRangesConditionV3(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_RangeSeries),
            'function_spec': ConjureFieldDefinition('functionSpec', object),
            'threshold': ConjureFieldDefinition('threshold', int),
            'operator': ConjureFieldDefinition('operator', scout_compute_api_ThresholdOperator),
            'variables': ConjureFieldDefinition('variables', Dict[scout_compute_api_VariableName, scout_checks_api_VariableLocator]),
            'function_variables': ConjureFieldDefinition('functionVariables', Dict[scout_compute_api_FunctionReference, scout_checks_api_CheckContext])
        }

    __slots__: List[str] = ['_ranges', '_function_spec', '_threshold', '_operator', '_variables', '_function_variables']

    def __init__(self, function_spec: Any, function_variables: Dict[str, "scout_checks_api_CheckContext"], operator: "scout_compute_api_ThresholdOperator", ranges: "scout_compute_api_RangeSeries", threshold: int, variables: Dict[str, "scout_checks_api_VariableLocator"]) -> None:
        self._ranges = ranges
        self._function_spec = function_spec
        self._threshold = threshold
        self._operator = operator
        self._variables = variables
        self._function_variables = function_variables

    @builtins.property
    def ranges(self) -> "scout_compute_api_RangeSeries":
        return self._ranges

    @builtins.property
    def function_spec(self) -> Any:
        return self._function_spec

    @builtins.property
    def threshold(self) -> int:
        return self._threshold

    @builtins.property
    def operator(self) -> "scout_compute_api_ThresholdOperator":
        return self._operator

    @builtins.property
    def variables(self) -> Dict[str, "scout_checks_api_VariableLocator"]:
        return self._variables

    @builtins.property
    def function_variables(self) -> Dict[str, "scout_checks_api_CheckContext"]:
        """
        The variables to be passed into functions referenced by the check condition. The function reference key
should match the FunctionReference in the function node definition.
        """
        return self._function_variables


scout_checks_api_NumRangesConditionV3.__name__ = "NumRangesConditionV3"
scout_checks_api_NumRangesConditionV3.__qualname__ = "NumRangesConditionV3"
scout_checks_api_NumRangesConditionV3.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_ParameterizedNumRangesConditionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_RangeSeries),
            'implementations': ConjureFieldDefinition('implementations', List[scout_checks_api_CheckContext])
        }

    __slots__: List[str] = ['_ranges', '_implementations']

    def __init__(self, implementations: List["scout_checks_api_CheckContext"], ranges: "scout_compute_api_RangeSeries") -> None:
        self._ranges = ranges
        self._implementations = implementations

    @builtins.property
    def ranges(self) -> "scout_compute_api_RangeSeries":
        return self._ranges

    @builtins.property
    def implementations(self) -> List["scout_checks_api_CheckContext"]:
        """
        Each permutation of check context in the list will be executed as a new a stand-alone check. Violations
produced will be associated with the context of the index.
        """
        return self._implementations


scout_checks_api_ParameterizedNumRangesConditionV1.__name__ = "ParameterizedNumRangesConditionV1"
scout_checks_api_ParameterizedNumRangesConditionV1.__qualname__ = "ParameterizedNumRangesConditionV1"
scout_checks_api_ParameterizedNumRangesConditionV1.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_PinnedChecklistRef(ConjureBeanType):
    """
    A reference to a checklist that is pinned to a specific commit.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', scout_rids_api_ChecklistRid),
            'commit': ConjureFieldDefinition('commit', scout_versioning_api_CommitId)
        }

    __slots__: List[str] = ['_rid', '_commit']

    def __init__(self, commit: str, rid: str) -> None:
        self._rid = rid
        self._commit = commit

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def commit(self) -> str:
        return self._commit


scout_checks_api_PinnedChecklistRef.__name__ = "PinnedChecklistRef"
scout_checks_api_PinnedChecklistRef.__qualname__ = "PinnedChecklistRef"
scout_checks_api_PinnedChecklistRef.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_Priority(ConjureEnumType):

    P0 = 'P0'
    '''P0'''
    P1 = 'P1'
    '''P1'''
    P2 = 'P2'
    '''P2'''
    P3 = 'P3'
    '''P3'''
    P4 = 'P4'
    '''P4'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_checks_api_Priority.__name__ = "Priority"
scout_checks_api_Priority.__qualname__ = "Priority"
scout_checks_api_Priority.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_SaveChecklistRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checks': ConjureFieldDefinition('checks', List[scout_checks_api_UpdateChecklistEntryRequest]),
            'functions': ConjureFieldDefinition('functions', Dict[scout_compute_api_FunctionReference, scout_checks_api_UpdateFunctionEntryRequest]),
            'checklist_variables': ConjureFieldDefinition('checklistVariables', List[scout_checks_api_UnresolvedChecklistVariable]),
            'latest_commit': ConjureFieldDefinition('latestCommit', OptionalTypeWrapper[scout_versioning_api_CommitId])
        }

    __slots__: List[str] = ['_checks', '_functions', '_checklist_variables', '_latest_commit']

    def __init__(self, checklist_variables: List["scout_checks_api_UnresolvedChecklistVariable"], checks: List["scout_checks_api_UpdateChecklistEntryRequest"], functions: Dict[str, "scout_checks_api_UpdateFunctionEntryRequest"], latest_commit: Optional[str] = None) -> None:
        self._checks = checks
        self._functions = functions
        self._checklist_variables = checklist_variables
        self._latest_commit = latest_commit

    @builtins.property
    def checks(self) -> List["scout_checks_api_UpdateChecklistEntryRequest"]:
        return self._checks

    @builtins.property
    def functions(self) -> Dict[str, "scout_checks_api_UpdateFunctionEntryRequest"]:
        """
        The keys of the map are references that can be used for checks to reference functions in the same request, before the function RIDs have been created.
        """
        return self._functions

    @builtins.property
    def checklist_variables(self) -> List["scout_checks_api_UnresolvedChecklistVariable"]:
        """
        Variables that can be used in checks and functions. Variables are resolved in order of declaration.
If variable `a` depends on variable `b`, then `b` must be defined before `a` in the list.
        """
        return self._checklist_variables

    @builtins.property
    def latest_commit(self) -> Optional[str]:
        """
        If present, will validate that the latest commit matches this id,
and otherwise throw CommitConflict.
        """
        return self._latest_commit


scout_checks_api_SaveChecklistRequest.__name__ = "SaveChecklistRequest"
scout_checks_api_SaveChecklistRequest.__qualname__ = "SaveChecklistRequest"
scout_checks_api_SaveChecklistRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_SearchChecklistsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'query': ConjureFieldDefinition('query', scout_checks_api_ChecklistSearchQuery),
            'sort_by': ConjureFieldDefinition('sortBy', OptionalTypeWrapper[scout_checks_api_SortOptions]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token]),
            'page_size': ConjureFieldDefinition('pageSize', OptionalTypeWrapper[int]),
            'archived_statuses': ConjureFieldDefinition('archivedStatuses', OptionalTypeWrapper[List[scout_rids_api_ArchivedStatus]])
        }

    __slots__: List[str] = ['_query', '_sort_by', '_next_page_token', '_page_size', '_archived_statuses']

    def __init__(self, query: "scout_checks_api_ChecklistSearchQuery", archived_statuses: Optional[List["scout_rids_api_ArchivedStatus"]] = None, next_page_token: Optional[str] = None, page_size: Optional[int] = None, sort_by: Optional["scout_checks_api_SortOptions"] = None) -> None:
        self._query = query
        self._sort_by = sort_by
        self._next_page_token = next_page_token
        self._page_size = page_size
        self._archived_statuses = archived_statuses

    @builtins.property
    def query(self) -> "scout_checks_api_ChecklistSearchQuery":
        return self._query

    @builtins.property
    def sort_by(self) -> Optional["scout_checks_api_SortOptions"]:
        """
        If not present, will sort by LAST_USED in descending order.
        """
        return self._sort_by

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token

    @builtins.property
    def page_size(self) -> Optional[int]:
        """
        Defaults to 100. Will throw if larger than 1000.
        """
        return self._page_size

    @builtins.property
    def archived_statuses(self) -> Optional[List["scout_rids_api_ArchivedStatus"]]:
        """
        Default search status is NOT_ARCHIVED if none are provided. Allows for including archived checklists in search.
        """
        return self._archived_statuses


scout_checks_api_SearchChecklistsRequest.__name__ = "SearchChecklistsRequest"
scout_checks_api_SearchChecklistsRequest.__qualname__ = "SearchChecklistsRequest"
scout_checks_api_SearchChecklistsRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_SortField(ConjureEnumType):

    CREATED_AT = 'CREATED_AT'
    '''CREATED_AT'''
    LAST_USED = 'LAST_USED'
    '''LAST_USED'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_checks_api_SortField.__name__ = "SortField"
scout_checks_api_SortField.__qualname__ = "SortField"
scout_checks_api_SortField.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_SortOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'is_descending': ConjureFieldDefinition('isDescending', bool),
            'field': ConjureFieldDefinition('field', scout_checks_api_SortField)
        }

    __slots__: List[str] = ['_is_descending', '_field']

    def __init__(self, field: "scout_checks_api_SortField", is_descending: bool) -> None:
        self._is_descending = is_descending
        self._field = field

    @builtins.property
    def is_descending(self) -> bool:
        return self._is_descending

    @builtins.property
    def field(self) -> "scout_checks_api_SortField":
        return self._field


scout_checks_api_SortOptions.__name__ = "SortOptions"
scout_checks_api_SortOptions.__qualname__ = "SortOptions"
scout_checks_api_SortOptions.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_SubmitJobsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'jobs_to_execute': ConjureFieldDefinition('jobsToExecute', List[scout_checks_api_JobSpec])
        }

    __slots__: List[str] = ['_jobs_to_execute']

    def __init__(self, jobs_to_execute: List["scout_checks_api_JobSpec"]) -> None:
        self._jobs_to_execute = jobs_to_execute

    @builtins.property
    def jobs_to_execute(self) -> List["scout_checks_api_JobSpec"]:
        return self._jobs_to_execute


scout_checks_api_SubmitJobsRequest.__name__ = "SubmitJobsRequest"
scout_checks_api_SubmitJobsRequest.__qualname__ = "SubmitJobsRequest"
scout_checks_api_SubmitJobsRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_SubmitJobsResponse(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'jobs': ConjureFieldDefinition('jobs', List[scout_checks_api_SubmittedJob])
        }

    __slots__: List[str] = ['_jobs']

    def __init__(self, jobs: List["scout_checks_api_SubmittedJob"]) -> None:
        self._jobs = jobs

    @builtins.property
    def jobs(self) -> List["scout_checks_api_SubmittedJob"]:
        return self._jobs


scout_checks_api_SubmitJobsResponse.__name__ = "SubmitJobsResponse"
scout_checks_api_SubmitJobsResponse.__qualname__ = "SubmitJobsResponse"
scout_checks_api_SubmitJobsResponse.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_SubmittedJob(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'job_rid': ConjureFieldDefinition('jobRid', scout_checks_api_JobRid),
            'job_spec': ConjureFieldDefinition('jobSpec', scout_checks_api_JobSpec)
        }

    __slots__: List[str] = ['_job_rid', '_job_spec']

    def __init__(self, job_rid: str, job_spec: "scout_checks_api_JobSpec") -> None:
        self._job_rid = job_rid
        self._job_spec = job_spec

    @builtins.property
    def job_rid(self) -> str:
        return self._job_rid

    @builtins.property
    def job_spec(self) -> "scout_checks_api_JobSpec":
        return self._job_spec


scout_checks_api_SubmittedJob.__name__ = "SubmittedJob"
scout_checks_api_SubmittedJob.__qualname__ = "SubmittedJob"
scout_checks_api_SubmittedJob.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_TimestampLocator(ConjureEnumType):

    RUN_START = 'RUN_START'
    '''RUN_START'''
    RUN_END = 'RUN_END'
    '''RUN_END'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_checks_api_TimestampLocator.__name__ = "TimestampLocator"
scout_checks_api_TimestampLocator.__qualname__ = "TimestampLocator"
scout_checks_api_TimestampLocator.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnarchiveChecklistsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rids': ConjureFieldDefinition('rids', List[scout_rids_api_ChecklistRid])
        }

    __slots__: List[str] = ['_rids']

    def __init__(self, rids: List[str]) -> None:
        self._rids = rids

    @builtins.property
    def rids(self) -> List[str]:
        return self._rids


scout_checks_api_UnarchiveChecklistsRequest.__name__ = "UnarchiveChecklistsRequest"
scout_checks_api_UnarchiveChecklistsRequest.__qualname__ = "UnarchiveChecklistsRequest"
scout_checks_api_UnarchiveChecklistsRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedCheckCondition(ConjureUnionType):
    _num_ranges_v2: Optional["scout_checks_api_UnresolvedNumRangesConditionV2"] = None
    _num_ranges_v3: Optional["scout_checks_api_UnresolvedNumRangesConditionV3"] = None
    _parameterized_num_ranges_v1: Optional["scout_checks_api_UnresolvedParameterizedNumRangesConditionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'num_ranges_v2': ConjureFieldDefinition('numRangesV2', scout_checks_api_UnresolvedNumRangesConditionV2),
            'num_ranges_v3': ConjureFieldDefinition('numRangesV3', scout_checks_api_UnresolvedNumRangesConditionV3),
            'parameterized_num_ranges_v1': ConjureFieldDefinition('parameterizedNumRangesV1', scout_checks_api_UnresolvedParameterizedNumRangesConditionV1)
        }

    def __init__(
            self,
            num_ranges_v2: Optional["scout_checks_api_UnresolvedNumRangesConditionV2"] = None,
            num_ranges_v3: Optional["scout_checks_api_UnresolvedNumRangesConditionV3"] = None,
            parameterized_num_ranges_v1: Optional["scout_checks_api_UnresolvedParameterizedNumRangesConditionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (num_ranges_v2 is not None) + (num_ranges_v3 is not None) + (parameterized_num_ranges_v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if num_ranges_v2 is not None:
                self._num_ranges_v2 = num_ranges_v2
                self._type = 'numRangesV2'
            if num_ranges_v3 is not None:
                self._num_ranges_v3 = num_ranges_v3
                self._type = 'numRangesV3'
            if parameterized_num_ranges_v1 is not None:
                self._parameterized_num_ranges_v1 = parameterized_num_ranges_v1
                self._type = 'parameterizedNumRangesV1'

        elif type_of_union == 'numRangesV2':
            if num_ranges_v2 is None:
                raise ValueError('a union value must not be None')
            self._num_ranges_v2 = num_ranges_v2
            self._type = 'numRangesV2'
        elif type_of_union == 'numRangesV3':
            if num_ranges_v3 is None:
                raise ValueError('a union value must not be None')
            self._num_ranges_v3 = num_ranges_v3
            self._type = 'numRangesV3'
        elif type_of_union == 'parameterizedNumRangesV1':
            if parameterized_num_ranges_v1 is None:
                raise ValueError('a union value must not be None')
            self._parameterized_num_ranges_v1 = parameterized_num_ranges_v1
            self._type = 'parameterizedNumRangesV1'

    @builtins.property
    def num_ranges_v2(self) -> Optional["scout_checks_api_UnresolvedNumRangesConditionV2"]:
        return self._num_ranges_v2

    @builtins.property
    def num_ranges_v3(self) -> Optional["scout_checks_api_UnresolvedNumRangesConditionV3"]:
        return self._num_ranges_v3

    @builtins.property
    def parameterized_num_ranges_v1(self) -> Optional["scout_checks_api_UnresolvedParameterizedNumRangesConditionV1"]:
        return self._parameterized_num_ranges_v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_UnresolvedCheckConditionVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_UnresolvedCheckConditionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'numRangesV2' and self.num_ranges_v2 is not None:
            return visitor._num_ranges_v2(self.num_ranges_v2)
        if self._type == 'numRangesV3' and self.num_ranges_v3 is not None:
            return visitor._num_ranges_v3(self.num_ranges_v3)
        if self._type == 'parameterizedNumRangesV1' and self.parameterized_num_ranges_v1 is not None:
            return visitor._parameterized_num_ranges_v1(self.parameterized_num_ranges_v1)


scout_checks_api_UnresolvedCheckCondition.__name__ = "UnresolvedCheckCondition"
scout_checks_api_UnresolvedCheckCondition.__qualname__ = "UnresolvedCheckCondition"
scout_checks_api_UnresolvedCheckCondition.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedCheckConditionVisitor:

    @abstractmethod
    def _num_ranges_v2(self, num_ranges_v2: "scout_checks_api_UnresolvedNumRangesConditionV2") -> Any:
        pass

    @abstractmethod
    def _num_ranges_v3(self, num_ranges_v3: "scout_checks_api_UnresolvedNumRangesConditionV3") -> Any:
        pass

    @abstractmethod
    def _parameterized_num_ranges_v1(self, parameterized_num_ranges_v1: "scout_checks_api_UnresolvedParameterizedNumRangesConditionV1") -> Any:
        pass


scout_checks_api_UnresolvedCheckConditionVisitor.__name__ = "UnresolvedCheckConditionVisitor"
scout_checks_api_UnresolvedCheckConditionVisitor.__qualname__ = "UnresolvedCheckConditionVisitor"
scout_checks_api_UnresolvedCheckConditionVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedChecklistVariable(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'name': ConjureFieldDefinition('name', scout_compute_api_VariableName),
            'display_name': ConjureFieldDefinition('displayName', OptionalTypeWrapper[str]),
            'value': ConjureFieldDefinition('value', scout_checks_api_UnresolvedVariableLocator)
        }

    __slots__: List[str] = ['_name', '_display_name', '_value']

    def __init__(self, name: str, value: "scout_checks_api_UnresolvedVariableLocator", display_name: Optional[str] = None) -> None:
        self._name = name
        self._display_name = display_name
        self._value = value

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def display_name(self) -> Optional[str]:
        return self._display_name

    @builtins.property
    def value(self) -> "scout_checks_api_UnresolvedVariableLocator":
        return self._value


scout_checks_api_UnresolvedChecklistVariable.__name__ = "UnresolvedChecklistVariable"
scout_checks_api_UnresolvedChecklistVariable.__qualname__ = "UnresolvedChecklistVariable"
scout_checks_api_UnresolvedChecklistVariable.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedComputeNodeWithContext(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'series_node': ConjureFieldDefinition('seriesNode', scout_compute_api_ComputeNode),
            'context': ConjureFieldDefinition('context', scout_checks_api_UnresolvedVariables)
        }

    __slots__: List[str] = ['_series_node', '_context']

    def __init__(self, context: "scout_checks_api_UnresolvedVariables", series_node: "scout_compute_api_ComputeNode") -> None:
        self._series_node = series_node
        self._context = context

    @builtins.property
    def series_node(self) -> "scout_compute_api_ComputeNode":
        return self._series_node

    @builtins.property
    def context(self) -> "scout_checks_api_UnresolvedVariables":
        return self._context


scout_checks_api_UnresolvedComputeNodeWithContext.__name__ = "UnresolvedComputeNodeWithContext"
scout_checks_api_UnresolvedComputeNodeWithContext.__qualname__ = "UnresolvedComputeNodeWithContext"
scout_checks_api_UnresolvedComputeNodeWithContext.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedNumRangesConditionV2(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_deprecated_RangesNode),
            'function_spec': ConjureFieldDefinition('functionSpec', object),
            'threshold': ConjureFieldDefinition('threshold', int),
            'operator': ConjureFieldDefinition('operator', scout_compute_api_ThresholdOperator),
            'variables': ConjureFieldDefinition('variables', Dict[scout_compute_api_VariableName, scout_checks_api_UnresolvedVariableLocator])
        }

    __slots__: List[str] = ['_ranges', '_function_spec', '_threshold', '_operator', '_variables']

    def __init__(self, function_spec: Any, operator: "scout_compute_api_ThresholdOperator", ranges: "scout_compute_api_deprecated_RangesNode", threshold: int, variables: Dict[str, "scout_checks_api_UnresolvedVariableLocator"]) -> None:
        self._ranges = ranges
        self._function_spec = function_spec
        self._threshold = threshold
        self._operator = operator
        self._variables = variables

    @builtins.property
    def ranges(self) -> "scout_compute_api_deprecated_RangesNode":
        return self._ranges

    @builtins.property
    def function_spec(self) -> Any:
        return self._function_spec

    @builtins.property
    def threshold(self) -> int:
        return self._threshold

    @builtins.property
    def operator(self) -> "scout_compute_api_ThresholdOperator":
        return self._operator

    @builtins.property
    def variables(self) -> Dict[str, "scout_checks_api_UnresolvedVariableLocator"]:
        return self._variables


scout_checks_api_UnresolvedNumRangesConditionV2.__name__ = "UnresolvedNumRangesConditionV2"
scout_checks_api_UnresolvedNumRangesConditionV2.__qualname__ = "UnresolvedNumRangesConditionV2"
scout_checks_api_UnresolvedNumRangesConditionV2.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedNumRangesConditionV3(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_RangeSeries),
            'function_spec': ConjureFieldDefinition('functionSpec', object),
            'threshold': ConjureFieldDefinition('threshold', int),
            'operator': ConjureFieldDefinition('operator', scout_compute_api_ThresholdOperator),
            'variables': ConjureFieldDefinition('variables', Dict[scout_compute_api_VariableName, scout_checks_api_UnresolvedVariableLocator]),
            'function_variables': ConjureFieldDefinition('functionVariables', Dict[scout_compute_api_FunctionReference, scout_checks_api_UnresolvedVariables])
        }

    __slots__: List[str] = ['_ranges', '_function_spec', '_threshold', '_operator', '_variables', '_function_variables']

    def __init__(self, function_spec: Any, function_variables: Dict[str, "scout_checks_api_UnresolvedVariables"], operator: "scout_compute_api_ThresholdOperator", ranges: "scout_compute_api_RangeSeries", threshold: int, variables: Dict[str, "scout_checks_api_UnresolvedVariableLocator"]) -> None:
        self._ranges = ranges
        self._function_spec = function_spec
        self._threshold = threshold
        self._operator = operator
        self._variables = variables
        self._function_variables = function_variables

    @builtins.property
    def ranges(self) -> "scout_compute_api_RangeSeries":
        return self._ranges

    @builtins.property
    def function_spec(self) -> Any:
        return self._function_spec

    @builtins.property
    def threshold(self) -> int:
        return self._threshold

    @builtins.property
    def operator(self) -> "scout_compute_api_ThresholdOperator":
        return self._operator

    @builtins.property
    def variables(self) -> Dict[str, "scout_checks_api_UnresolvedVariableLocator"]:
        """
        Default overrides for the variables used in the check condition. These variables can be overridden
at checklist execution time.
        """
        return self._variables

    @builtins.property
    def function_variables(self) -> Dict[str, "scout_checks_api_UnresolvedVariables"]:
        """
        The variables to be passed into functions referenced by the check condition. The function reference key
should match the FunctionReference in the function node definition. These variables can be overridden
at checklist execution time.
        """
        return self._function_variables


scout_checks_api_UnresolvedNumRangesConditionV3.__name__ = "UnresolvedNumRangesConditionV3"
scout_checks_api_UnresolvedNumRangesConditionV3.__qualname__ = "UnresolvedNumRangesConditionV3"
scout_checks_api_UnresolvedNumRangesConditionV3.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedParameterizedNumRangesConditionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_RangeSeries),
            'implementations': ConjureFieldDefinition('implementations', List[scout_checks_api_UnresolvedVariables])
        }

    __slots__: List[str] = ['_ranges', '_implementations']

    def __init__(self, implementations: List["scout_checks_api_UnresolvedVariables"], ranges: "scout_compute_api_RangeSeries") -> None:
        self._ranges = ranges
        self._implementations = implementations

    @builtins.property
    def ranges(self) -> "scout_compute_api_RangeSeries":
        return self._ranges

    @builtins.property
    def implementations(self) -> List["scout_checks_api_UnresolvedVariables"]:
        """
        Each permutation of check context in the list will be executed as a new a stand-alone check. Violations
produced will be associated with the context of the index.
        """
        return self._implementations


scout_checks_api_UnresolvedParameterizedNumRangesConditionV1.__name__ = "UnresolvedParameterizedNumRangesConditionV1"
scout_checks_api_UnresolvedParameterizedNumRangesConditionV1.__qualname__ = "UnresolvedParameterizedNumRangesConditionV1"
scout_checks_api_UnresolvedParameterizedNumRangesConditionV1.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedVariableLocator(ConjureUnionType):
    _checklist_function: Optional[str] = None
    _checklist_variable: Optional[str] = None
    _compute_node: Optional["scout_checks_api_UnresolvedComputeNodeWithContext"] = None
    _series: Optional["scout_api_ChannelLocator"] = None
    _timestamp: Optional["scout_checks_api_TimestampLocator"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklist_function': ConjureFieldDefinition('checklistFunction', scout_compute_api_FunctionReference),
            'checklist_variable': ConjureFieldDefinition('checklistVariable', scout_compute_api_VariableName),
            'compute_node': ConjureFieldDefinition('computeNode', scout_checks_api_UnresolvedComputeNodeWithContext),
            'series': ConjureFieldDefinition('series', scout_api_ChannelLocator),
            'timestamp': ConjureFieldDefinition('timestamp', scout_checks_api_TimestampLocator)
        }

    def __init__(
            self,
            checklist_function: Optional[str] = None,
            checklist_variable: Optional[str] = None,
            compute_node: Optional["scout_checks_api_UnresolvedComputeNodeWithContext"] = None,
            series: Optional["scout_api_ChannelLocator"] = None,
            timestamp: Optional["scout_checks_api_TimestampLocator"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (checklist_function is not None) + (checklist_variable is not None) + (compute_node is not None) + (series is not None) + (timestamp is not None) != 1:
                raise ValueError('a union must contain a single member')

            if checklist_function is not None:
                self._checklist_function = checklist_function
                self._type = 'checklistFunction'
            if checklist_variable is not None:
                self._checklist_variable = checklist_variable
                self._type = 'checklistVariable'
            if compute_node is not None:
                self._compute_node = compute_node
                self._type = 'computeNode'
            if series is not None:
                self._series = series
                self._type = 'series'
            if timestamp is not None:
                self._timestamp = timestamp
                self._type = 'timestamp'

        elif type_of_union == 'checklistFunction':
            if checklist_function is None:
                raise ValueError('a union value must not be None')
            self._checklist_function = checklist_function
            self._type = 'checklistFunction'
        elif type_of_union == 'checklistVariable':
            if checklist_variable is None:
                raise ValueError('a union value must not be None')
            self._checklist_variable = checklist_variable
            self._type = 'checklistVariable'
        elif type_of_union == 'computeNode':
            if compute_node is None:
                raise ValueError('a union value must not be None')
            self._compute_node = compute_node
            self._type = 'computeNode'
        elif type_of_union == 'series':
            if series is None:
                raise ValueError('a union value must not be None')
            self._series = series
            self._type = 'series'
        elif type_of_union == 'timestamp':
            if timestamp is None:
                raise ValueError('a union value must not be None')
            self._timestamp = timestamp
            self._type = 'timestamp'

    @builtins.property
    def checklist_function(self) -> Optional[str]:
        return self._checklist_function

    @builtins.property
    def checklist_variable(self) -> Optional[str]:
        """
        A pointer to a variable in the checklist.
        """
        return self._checklist_variable

    @builtins.property
    def compute_node(self) -> Optional["scout_checks_api_UnresolvedComputeNodeWithContext"]:
        return self._compute_node

    @builtins.property
    def series(self) -> Optional["scout_api_ChannelLocator"]:
        return self._series

    @builtins.property
    def timestamp(self) -> Optional["scout_checks_api_TimestampLocator"]:
        return self._timestamp

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_UnresolvedVariableLocatorVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_UnresolvedVariableLocatorVisitor'.format(visitor.__class__.__name__))
        if self._type == 'checklistFunction' and self.checklist_function is not None:
            return visitor._checklist_function(self.checklist_function)
        if self._type == 'checklistVariable' and self.checklist_variable is not None:
            return visitor._checklist_variable(self.checklist_variable)
        if self._type == 'computeNode' and self.compute_node is not None:
            return visitor._compute_node(self.compute_node)
        if self._type == 'series' and self.series is not None:
            return visitor._series(self.series)
        if self._type == 'timestamp' and self.timestamp is not None:
            return visitor._timestamp(self.timestamp)


scout_checks_api_UnresolvedVariableLocator.__name__ = "UnresolvedVariableLocator"
scout_checks_api_UnresolvedVariableLocator.__qualname__ = "UnresolvedVariableLocator"
scout_checks_api_UnresolvedVariableLocator.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedVariableLocatorVisitor:

    @abstractmethod
    def _checklist_function(self, checklist_function: str) -> Any:
        pass

    @abstractmethod
    def _checklist_variable(self, checklist_variable: str) -> Any:
        pass

    @abstractmethod
    def _compute_node(self, compute_node: "scout_checks_api_UnresolvedComputeNodeWithContext") -> Any:
        pass

    @abstractmethod
    def _series(self, series: "scout_api_ChannelLocator") -> Any:
        pass

    @abstractmethod
    def _timestamp(self, timestamp: "scout_checks_api_TimestampLocator") -> Any:
        pass


scout_checks_api_UnresolvedVariableLocatorVisitor.__name__ = "UnresolvedVariableLocatorVisitor"
scout_checks_api_UnresolvedVariableLocatorVisitor.__qualname__ = "UnresolvedVariableLocatorVisitor"
scout_checks_api_UnresolvedVariableLocatorVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UnresolvedVariables(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variables': ConjureFieldDefinition('variables', Dict[scout_compute_api_VariableName, scout_checks_api_UnresolvedVariableLocator]),
            'sub_function_variables': ConjureFieldDefinition('subFunctionVariables', Dict[scout_compute_api_FunctionReference, scout_checks_api_UnresolvedVariables])
        }

    __slots__: List[str] = ['_variables', '_sub_function_variables']

    def __init__(self, sub_function_variables: Dict[str, "scout_checks_api_UnresolvedVariables"], variables: Dict[str, "scout_checks_api_UnresolvedVariableLocator"]) -> None:
        self._variables = variables
        self._sub_function_variables = sub_function_variables

    @builtins.property
    def variables(self) -> Dict[str, "scout_checks_api_UnresolvedVariableLocator"]:
        return self._variables

    @builtins.property
    def sub_function_variables(self) -> Dict[str, "scout_checks_api_UnresolvedVariables"]:
        return self._sub_function_variables


scout_checks_api_UnresolvedVariables.__name__ = "UnresolvedVariables"
scout_checks_api_UnresolvedVariables.__qualname__ = "UnresolvedVariables"
scout_checks_api_UnresolvedVariables.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UpdateChecklistEntryRequest(ConjureUnionType):
    _create_check: Optional["scout_checks_api_CreateCheckRequest"] = None
    _check: Optional[str] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'create_check': ConjureFieldDefinition('createCheck', scout_checks_api_CreateCheckRequest),
            'check': ConjureFieldDefinition('check', scout_rids_api_CheckRid)
        }

    def __init__(
            self,
            create_check: Optional["scout_checks_api_CreateCheckRequest"] = None,
            check: Optional[str] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (create_check is not None) + (check is not None) != 1:
                raise ValueError('a union must contain a single member')

            if create_check is not None:
                self._create_check = create_check
                self._type = 'createCheck'
            if check is not None:
                self._check = check
                self._type = 'check'

        elif type_of_union == 'createCheck':
            if create_check is None:
                raise ValueError('a union value must not be None')
            self._create_check = create_check
            self._type = 'createCheck'
        elif type_of_union == 'check':
            if check is None:
                raise ValueError('a union value must not be None')
            self._check = check
            self._type = 'check'

    @builtins.property
    def create_check(self) -> Optional["scout_checks_api_CreateCheckRequest"]:
        return self._create_check

    @builtins.property
    def check(self) -> Optional[str]:
        return self._check

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_UpdateChecklistEntryRequestVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_UpdateChecklistEntryRequestVisitor'.format(visitor.__class__.__name__))
        if self._type == 'createCheck' and self.create_check is not None:
            return visitor._create_check(self.create_check)
        if self._type == 'check' and self.check is not None:
            return visitor._check(self.check)


scout_checks_api_UpdateChecklistEntryRequest.__name__ = "UpdateChecklistEntryRequest"
scout_checks_api_UpdateChecklistEntryRequest.__qualname__ = "UpdateChecklistEntryRequest"
scout_checks_api_UpdateChecklistEntryRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UpdateChecklistEntryRequestVisitor:

    @abstractmethod
    def _create_check(self, create_check: "scout_checks_api_CreateCheckRequest") -> Any:
        pass

    @abstractmethod
    def _check(self, check: str) -> Any:
        pass


scout_checks_api_UpdateChecklistEntryRequestVisitor.__name__ = "UpdateChecklistEntryRequestVisitor"
scout_checks_api_UpdateChecklistEntryRequestVisitor.__qualname__ = "UpdateChecklistEntryRequestVisitor"
scout_checks_api_UpdateChecklistEntryRequestVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UpdateChecklistMetadataRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'assignee_rid': ConjureFieldDefinition('assigneeRid', OptionalTypeWrapper[scout_rids_api_UserRid]),
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'description': ConjureFieldDefinition('description', OptionalTypeWrapper[str]),
            'properties': ConjureFieldDefinition('properties', OptionalTypeWrapper[Dict[scout_run_api_PropertyName, scout_run_api_PropertyValue]]),
            'labels': ConjureFieldDefinition('labels', OptionalTypeWrapper[List[scout_run_api_Label]]),
            'is_published': ConjureFieldDefinition('isPublished', OptionalTypeWrapper[bool])
        }

    __slots__: List[str] = ['_assignee_rid', '_title', '_description', '_properties', '_labels', '_is_published']

    def __init__(self, assignee_rid: Optional[str] = None, description: Optional[str] = None, is_published: Optional[bool] = None, labels: Optional[List[str]] = None, properties: Optional[Dict[str, str]] = None, title: Optional[str] = None) -> None:
        self._assignee_rid = assignee_rid
        self._title = title
        self._description = description
        self._properties = properties
        self._labels = labels
        self._is_published = is_published

    @builtins.property
    def assignee_rid(self) -> Optional[str]:
        return self._assignee_rid

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def description(self) -> Optional[str]:
        return self._description

    @builtins.property
    def properties(self) -> Optional[Dict[str, str]]:
        return self._properties

    @builtins.property
    def labels(self) -> Optional[List[str]]:
        return self._labels

    @builtins.property
    def is_published(self) -> Optional[bool]:
        return self._is_published


scout_checks_api_UpdateChecklistMetadataRequest.__name__ = "UpdateChecklistMetadataRequest"
scout_checks_api_UpdateChecklistMetadataRequest.__qualname__ = "UpdateChecklistMetadataRequest"
scout_checks_api_UpdateChecklistMetadataRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UpdateFunctionEntryRequest(ConjureUnionType):
    _create_function: Optional["scout_checks_api_CreateFunctionRequest"] = None
    _function: Optional[str] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'create_function': ConjureFieldDefinition('createFunction', scout_checks_api_CreateFunctionRequest),
            'function': ConjureFieldDefinition('function', scout_rids_api_FunctionRid)
        }

    def __init__(
            self,
            create_function: Optional["scout_checks_api_CreateFunctionRequest"] = None,
            function: Optional[str] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (create_function is not None) + (function is not None) != 1:
                raise ValueError('a union must contain a single member')

            if create_function is not None:
                self._create_function = create_function
                self._type = 'createFunction'
            if function is not None:
                self._function = function
                self._type = 'function'

        elif type_of_union == 'createFunction':
            if create_function is None:
                raise ValueError('a union value must not be None')
            self._create_function = create_function
            self._type = 'createFunction'
        elif type_of_union == 'function':
            if function is None:
                raise ValueError('a union value must not be None')
            self._function = function
            self._type = 'function'

    @builtins.property
    def create_function(self) -> Optional["scout_checks_api_CreateFunctionRequest"]:
        return self._create_function

    @builtins.property
    def function(self) -> Optional[str]:
        return self._function

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_UpdateFunctionEntryRequestVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_UpdateFunctionEntryRequestVisitor'.format(visitor.__class__.__name__))
        if self._type == 'createFunction' and self.create_function is not None:
            return visitor._create_function(self.create_function)
        if self._type == 'function' and self.function is not None:
            return visitor._function(self.function)


scout_checks_api_UpdateFunctionEntryRequest.__name__ = "UpdateFunctionEntryRequest"
scout_checks_api_UpdateFunctionEntryRequest.__qualname__ = "UpdateFunctionEntryRequest"
scout_checks_api_UpdateFunctionEntryRequest.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_UpdateFunctionEntryRequestVisitor:

    @abstractmethod
    def _create_function(self, create_function: "scout_checks_api_CreateFunctionRequest") -> Any:
        pass

    @abstractmethod
    def _function(self, function: str) -> Any:
        pass


scout_checks_api_UpdateFunctionEntryRequestVisitor.__name__ = "UpdateFunctionEntryRequestVisitor"
scout_checks_api_UpdateFunctionEntryRequestVisitor.__qualname__ = "UpdateFunctionEntryRequestVisitor"
scout_checks_api_UpdateFunctionEntryRequestVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_VariableLocator(ConjureUnionType):
    _checklist_variable: Optional[str] = None
    _compute_node: Optional["scout_checks_api_ComputeNodeWithContext"] = None
    _function_rid: Optional[str] = None
    _series: Optional["scout_api_ChannelLocator"] = None
    _timestamp: Optional["scout_checks_api_TimestampLocator"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'checklist_variable': ConjureFieldDefinition('checklistVariable', scout_compute_api_VariableName),
            'compute_node': ConjureFieldDefinition('computeNode', scout_checks_api_ComputeNodeWithContext),
            'function_rid': ConjureFieldDefinition('functionRid', scout_rids_api_FunctionRid),
            'series': ConjureFieldDefinition('series', scout_api_ChannelLocator),
            'timestamp': ConjureFieldDefinition('timestamp', scout_checks_api_TimestampLocator)
        }

    def __init__(
            self,
            checklist_variable: Optional[str] = None,
            compute_node: Optional["scout_checks_api_ComputeNodeWithContext"] = None,
            function_rid: Optional[str] = None,
            series: Optional["scout_api_ChannelLocator"] = None,
            timestamp: Optional["scout_checks_api_TimestampLocator"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (checklist_variable is not None) + (compute_node is not None) + (function_rid is not None) + (series is not None) + (timestamp is not None) != 1:
                raise ValueError('a union must contain a single member')

            if checklist_variable is not None:
                self._checklist_variable = checklist_variable
                self._type = 'checklistVariable'
            if compute_node is not None:
                self._compute_node = compute_node
                self._type = 'computeNode'
            if function_rid is not None:
                self._function_rid = function_rid
                self._type = 'functionRid'
            if series is not None:
                self._series = series
                self._type = 'series'
            if timestamp is not None:
                self._timestamp = timestamp
                self._type = 'timestamp'

        elif type_of_union == 'checklistVariable':
            if checklist_variable is None:
                raise ValueError('a union value must not be None')
            self._checklist_variable = checklist_variable
            self._type = 'checklistVariable'
        elif type_of_union == 'computeNode':
            if compute_node is None:
                raise ValueError('a union value must not be None')
            self._compute_node = compute_node
            self._type = 'computeNode'
        elif type_of_union == 'functionRid':
            if function_rid is None:
                raise ValueError('a union value must not be None')
            self._function_rid = function_rid
            self._type = 'functionRid'
        elif type_of_union == 'series':
            if series is None:
                raise ValueError('a union value must not be None')
            self._series = series
            self._type = 'series'
        elif type_of_union == 'timestamp':
            if timestamp is None:
                raise ValueError('a union value must not be None')
            self._timestamp = timestamp
            self._type = 'timestamp'

    @builtins.property
    def checklist_variable(self) -> Optional[str]:
        """
        A pointer to a checklist level variable.
        """
        return self._checklist_variable

    @builtins.property
    def compute_node(self) -> Optional["scout_checks_api_ComputeNodeWithContext"]:
        return self._compute_node

    @builtins.property
    def function_rid(self) -> Optional[str]:
        return self._function_rid

    @builtins.property
    def series(self) -> Optional["scout_api_ChannelLocator"]:
        return self._series

    @builtins.property
    def timestamp(self) -> Optional["scout_checks_api_TimestampLocator"]:
        return self._timestamp

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_checks_api_VariableLocatorVisitor):
            raise ValueError('{} is not an instance of scout_checks_api_VariableLocatorVisitor'.format(visitor.__class__.__name__))
        if self._type == 'checklistVariable' and self.checklist_variable is not None:
            return visitor._checklist_variable(self.checklist_variable)
        if self._type == 'computeNode' and self.compute_node is not None:
            return visitor._compute_node(self.compute_node)
        if self._type == 'functionRid' and self.function_rid is not None:
            return visitor._function_rid(self.function_rid)
        if self._type == 'series' and self.series is not None:
            return visitor._series(self.series)
        if self._type == 'timestamp' and self.timestamp is not None:
            return visitor._timestamp(self.timestamp)


scout_checks_api_VariableLocator.__name__ = "VariableLocator"
scout_checks_api_VariableLocator.__qualname__ = "VariableLocator"
scout_checks_api_VariableLocator.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_VariableLocatorVisitor:

    @abstractmethod
    def _checklist_variable(self, checklist_variable: str) -> Any:
        pass

    @abstractmethod
    def _compute_node(self, compute_node: "scout_checks_api_ComputeNodeWithContext") -> Any:
        pass

    @abstractmethod
    def _function_rid(self, function_rid: str) -> Any:
        pass

    @abstractmethod
    def _series(self, series: "scout_api_ChannelLocator") -> Any:
        pass

    @abstractmethod
    def _timestamp(self, timestamp: "scout_checks_api_TimestampLocator") -> Any:
        pass


scout_checks_api_VariableLocatorVisitor.__name__ = "VariableLocatorVisitor"
scout_checks_api_VariableLocatorVisitor.__qualname__ = "VariableLocatorVisitor"
scout_checks_api_VariableLocatorVisitor.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_VersionedChecklist(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'rid': ConjureFieldDefinition('rid', scout_rids_api_ChecklistRid),
            'metadata': ConjureFieldDefinition('metadata', scout_checks_api_ChecklistMetadata),
            'commit': ConjureFieldDefinition('commit', scout_versioning_api_Commit),
            'functions': ConjureFieldDefinition('functions', List[scout_checks_api_Function]),
            'checks': ConjureFieldDefinition('checks', List[scout_checks_api_ChecklistEntry]),
            'checklist_variables': ConjureFieldDefinition('checklistVariables', List[scout_checks_api_ChecklistVariable])
        }

    __slots__: List[str] = ['_rid', '_metadata', '_commit', '_functions', '_checks', '_checklist_variables']

    def __init__(self, checklist_variables: List["scout_checks_api_ChecklistVariable"], checks: List["scout_checks_api_ChecklistEntry"], commit: "scout_versioning_api_Commit", functions: List["scout_checks_api_Function"], metadata: "scout_checks_api_ChecklistMetadata", rid: str) -> None:
        self._rid = rid
        self._metadata = metadata
        self._commit = commit
        self._functions = functions
        self._checks = checks
        self._checklist_variables = checklist_variables

    @builtins.property
    def rid(self) -> str:
        return self._rid

    @builtins.property
    def metadata(self) -> "scout_checks_api_ChecklistMetadata":
        return self._metadata

    @builtins.property
    def commit(self) -> "scout_versioning_api_Commit":
        return self._commit

    @builtins.property
    def functions(self) -> List["scout_checks_api_Function"]:
        """
        A list of functions that are available to be used by checks in this checklist.
        """
        return self._functions

    @builtins.property
    def checks(self) -> List["scout_checks_api_ChecklistEntry"]:
        return self._checks

    @builtins.property
    def checklist_variables(self) -> List["scout_checks_api_ChecklistVariable"]:
        """
        Variables that can be used in checks and functions. Variables are resolved in order of declaration.
If variable `a` depends on variable `b`, then `b` must be defined before `a` in the list.
        """
        return self._checklist_variables


scout_checks_api_VersionedChecklist.__name__ = "VersionedChecklist"
scout_checks_api_VersionedChecklist.__qualname__ = "VersionedChecklist"
scout_checks_api_VersionedChecklist.__module__ = "nominal_api.scout_checks_api"


class scout_checks_api_VersionedChecklistPage(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'values': ConjureFieldDefinition('values', List[scout_checks_api_VersionedChecklist]),
            'next_page_token': ConjureFieldDefinition('nextPageToken', OptionalTypeWrapper[scout_api_Token])
        }

    __slots__: List[str] = ['_values', '_next_page_token']

    def __init__(self, values: List["scout_checks_api_VersionedChecklist"], next_page_token: Optional[str] = None) -> None:
        self._values = values
        self._next_page_token = next_page_token

    @builtins.property
    def values(self) -> List["scout_checks_api_VersionedChecklist"]:
        return self._values

    @builtins.property
    def next_page_token(self) -> Optional[str]:
        return self._next_page_token


scout_checks_api_VersionedChecklistPage.__name__ = "VersionedChecklistPage"
scout_checks_api_VersionedChecklistPage.__qualname__ = "VersionedChecklistPage"
scout_checks_api_VersionedChecklistPage.__module__ = "nominal_api.scout_checks_api"


class scout_comparisonnotebook_api_AggregationType(ConjureUnionType):
    _max: Optional["scout_comparisonnotebook_api_Max"] = None
    _min: Optional["scout_comparisonnotebook_api_Min"] = None
    _mean: Optional["scout_comparisonnotebook_api_Mean"] = None
    _standard_deviation: Optional["scout_comparisonnotebook_api_StandardDeviation"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'max': ConjureFieldDefinition('max', scout_comparisonnotebook_api_Max),
            'min': ConjureFieldDefinition('min', scout_comparisonnotebook_api_Min),
            'mean': ConjureFieldDefinition('mean', scout_comparisonnotebook_api_Mean),
            'standard_deviation': ConjureFieldDefinition('standardDeviation', scout_comparisonnotebook_api_StandardDeviation)
        }

    def __init__(
            self,
            max: Optional["scout_comparisonnotebook_api_Max"] = None,
            min: Optional["scout_comparisonnotebook_api_Min"] = None,
            mean: Optional["scout_comparisonnotebook_api_Mean"] = None,
            standard_deviation: Optional["scout_comparisonnotebook_api_StandardDeviation"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (max is not None) + (min is not None) + (mean is not None) + (standard_deviation is not None) != 1:
                raise ValueError('a union must contain a single member')

            if max is not None:
                self._max = max
                self._type = 'max'
            if min is not None:
                self._min = min
                self._type = 'min'
            if mean is not None:
                self._mean = mean
                self._type = 'mean'
            if standard_deviation is not None:
                self._standard_deviation = standard_deviation
                self._type = 'standardDeviation'

        elif type_of_union == 'max':
            if max is None:
                raise ValueError('a union value must not be None')
            self._max = max
            self._type = 'max'
        elif type_of_union == 'min':
            if min is None:
                raise ValueError('a union value must not be None')
            self._min = min
            self._type = 'min'
        elif type_of_union == 'mean':
            if mean is None:
                raise ValueError('a union value must not be None')
            self._mean = mean
            self._type = 'mean'
        elif type_of_union == 'standardDeviation':
            if standard_deviation is None:
                raise ValueError('a union value must not be None')
            self._standard_deviation = standard_deviation
            self._type = 'standardDeviation'

    @builtins.property
    def max(self) -> Optional["scout_comparisonnotebook_api_Max"]:
        return self._max

    @builtins.property
    def min(self) -> Optional["scout_comparisonnotebook_api_Min"]:
        return self._min

    @builtins.property
    def mean(self) -> Optional["scout_comparisonnotebook_api_Mean"]:
        return self._mean

    @builtins.property
    def standard_deviation(self) -> Optional["scout_comparisonnotebook_api_StandardDeviation"]:
        return self._standard_deviation

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_AggregationTypeVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_AggregationTypeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'max' and self.max is not None:
            return visitor._max(self.max)
        if self._type == 'min' and self.min is not None:
            return visitor._min(self.min)
        if self._type == 'mean' and self.mean is not None:
            return visitor._mean(self.mean)
        if self._type == 'standardDeviation' and self.standard_deviation is not None:
            return visitor._standard_deviation(self.standard_deviation)


scout_comparisonnotebook_api_AggregationType.__name__ = "AggregationType"
scout_comparisonnotebook_api_AggregationType.__qualname__ = "AggregationType"
scout_comparisonnotebook_api_AggregationType.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_AggregationTypeVisitor:

    @abstractmethod
    def _max(self, max: "scout_comparisonnotebook_api_Max") -> Any:
        pass

    @abstractmethod
    def _min(self, min: "scout_comparisonnotebook_api_Min") -> Any:
        pass

    @abstractmethod
    def _mean(self, mean: "scout_comparisonnotebook_api_Mean") -> Any:
        pass

    @abstractmethod
    def _standard_deviation(self, standard_deviation: "scout_comparisonnotebook_api_StandardDeviation") -> Any:
        pass


scout_comparisonnotebook_api_AggregationTypeVisitor.__name__ = "AggregationTypeVisitor"
scout_comparisonnotebook_api_AggregationTypeVisitor.__qualname__ = "AggregationTypeVisitor"
scout_comparisonnotebook_api_AggregationTypeVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ChannelVariable(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'display_name': ConjureFieldDefinition('displayName', OptionalTypeWrapper[str]),
            'variable_name': ConjureFieldDefinition('variableName', scout_comparisonnotebook_api_VariableName),
            'value': ConjureFieldDefinition('value', scout_comparisonnotebook_api_ComputeNodeWithContext),
            'data_scope': ConjureFieldDefinition('dataScope', scout_comparisonnotebook_api_ComparisonWorkbookDataScope)
        }

    __slots__: List[str] = ['_display_name', '_variable_name', '_value', '_data_scope']

    def __init__(self, data_scope: "scout_comparisonnotebook_api_ComparisonWorkbookDataScope", value: "scout_comparisonnotebook_api_ComputeNodeWithContext", variable_name: str, display_name: Optional[str] = None) -> None:
        self._display_name = display_name
        self._variable_name = variable_name
        self._value = value
        self._data_scope = data_scope

    @builtins.property
    def display_name(self) -> Optional[str]:
        return self._display_name

    @builtins.property
    def variable_name(self) -> str:
        return self._variable_name

    @builtins.property
    def value(self) -> "scout_comparisonnotebook_api_ComputeNodeWithContext":
        return self._value

    @builtins.property
    def data_scope(self) -> "scout_comparisonnotebook_api_ComparisonWorkbookDataScope":
        return self._data_scope


scout_comparisonnotebook_api_ChannelVariable.__name__ = "ChannelVariable"
scout_comparisonnotebook_api_ChannelVariable.__qualname__ = "ChannelVariable"
scout_comparisonnotebook_api_ChannelVariable.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ColorByVariableConfiguration(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'locator': ConjureFieldDefinition('locator', scout_comparisonnotebook_api_VariableLocator),
            'aggregation_type': ConjureFieldDefinition('aggregationType', scout_comparisonnotebook_api_AggregationType),
            'lower_bound': ConjureFieldDefinition('lowerBound', float),
            'upper_bound': ConjureFieldDefinition('upperBound', float)
        }

    __slots__: List[str] = ['_locator', '_aggregation_type', '_lower_bound', '_upper_bound']

    def __init__(self, aggregation_type: "scout_comparisonnotebook_api_AggregationType", locator: "scout_comparisonnotebook_api_VariableLocator", lower_bound: float, upper_bound: float) -> None:
        self._locator = locator
        self._aggregation_type = aggregation_type
        self._lower_bound = lower_bound
        self._upper_bound = upper_bound

    @builtins.property
    def locator(self) -> "scout_comparisonnotebook_api_VariableLocator":
        return self._locator

    @builtins.property
    def aggregation_type(self) -> "scout_comparisonnotebook_api_AggregationType":
        return self._aggregation_type

    @builtins.property
    def lower_bound(self) -> float:
        return self._lower_bound

    @builtins.property
    def upper_bound(self) -> float:
        return self._upper_bound


scout_comparisonnotebook_api_ColorByVariableConfiguration.__name__ = "ColorByVariableConfiguration"
scout_comparisonnotebook_api_ColorByVariableConfiguration.__qualname__ = "ColorByVariableConfiguration"
scout_comparisonnotebook_api_ColorByVariableConfiguration.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonScatterPlotColoringOption(ConjureUnionType):
    _by_variable: Optional["scout_comparisonnotebook_api_ColorByVariableConfiguration"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'by_variable': ConjureFieldDefinition('byVariable', scout_comparisonnotebook_api_ColorByVariableConfiguration)
        }

    def __init__(
            self,
            by_variable: Optional["scout_comparisonnotebook_api_ColorByVariableConfiguration"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (by_variable is not None) != 1:
                raise ValueError('a union must contain a single member')

            if by_variable is not None:
                self._by_variable = by_variable
                self._type = 'byVariable'

        elif type_of_union == 'byVariable':
            if by_variable is None:
                raise ValueError('a union value must not be None')
            self._by_variable = by_variable
            self._type = 'byVariable'

    @builtins.property
    def by_variable(self) -> Optional["scout_comparisonnotebook_api_ColorByVariableConfiguration"]:
        return self._by_variable

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_ComparisonScatterPlotColoringOptionVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_ComparisonScatterPlotColoringOptionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'byVariable' and self.by_variable is not None:
            return visitor._by_variable(self.by_variable)


scout_comparisonnotebook_api_ComparisonScatterPlotColoringOption.__name__ = "ComparisonScatterPlotColoringOption"
scout_comparisonnotebook_api_ComparisonScatterPlotColoringOption.__qualname__ = "ComparisonScatterPlotColoringOption"
scout_comparisonnotebook_api_ComparisonScatterPlotColoringOption.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonScatterPlotColoringOptionVisitor:

    @abstractmethod
    def _by_variable(self, by_variable: "scout_comparisonnotebook_api_ColorByVariableConfiguration") -> Any:
        pass


scout_comparisonnotebook_api_ComparisonScatterPlotColoringOptionVisitor.__name__ = "ComparisonScatterPlotColoringOptionVisitor"
scout_comparisonnotebook_api_ComparisonScatterPlotColoringOptionVisitor.__qualname__ = "ComparisonScatterPlotColoringOptionVisitor"
scout_comparisonnotebook_api_ComparisonScatterPlotColoringOptionVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonScatterPlotDefinition(ConjureUnionType):
    _v1: Optional["scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_comparisonnotebook_api_ComparisonScatterPlotDefinition.__name__ = "ComparisonScatterPlotDefinition"
scout_comparisonnotebook_api_ComparisonScatterPlotDefinition.__qualname__ = "ComparisonScatterPlotDefinition"
scout_comparisonnotebook_api_ComparisonScatterPlotDefinition.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1") -> Any:
        pass


scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionVisitor.__name__ = "ComparisonScatterPlotDefinitionVisitor"
scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionVisitor.__qualname__ = "ComparisonScatterPlotDefinitionVisitor"
scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1(ConjureBeanType):
    """
    an n x n array of scatter plots
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'range_aggregation': ConjureFieldDefinition('rangeAggregation', OptionalTypeWrapper[scout_comparisonnotebook_api_RangeAggregationDefinition]),
            'variables': ConjureFieldDefinition('variables', List[scout_comparisonnotebook_api_ComparisonScatterPlotVariable]),
            'axes': ConjureFieldDefinition('axes', List[scout_comparisonnotebook_api_ScatterPlotValueAxes]),
            'visualization_options': ConjureFieldDefinition('visualizationOptions', OptionalTypeWrapper[scout_comparisonnotebook_api_ComparisonScatterPlotVizOptions])
        }

    __slots__: List[str] = ['_title', '_range_aggregation', '_variables', '_axes', '_visualization_options']

    def __init__(self, axes: List["scout_comparisonnotebook_api_ScatterPlotValueAxes"], variables: List["scout_comparisonnotebook_api_ComparisonScatterPlotVariable"], range_aggregation: Optional["scout_comparisonnotebook_api_RangeAggregationDefinition"] = None, title: Optional[str] = None, visualization_options: Optional["scout_comparisonnotebook_api_ComparisonScatterPlotVizOptions"] = None) -> None:
        self._title = title
        self._range_aggregation = range_aggregation
        self._variables = variables
        self._axes = axes
        self._visualization_options = visualization_options

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def range_aggregation(self) -> Optional["scout_comparisonnotebook_api_RangeAggregationDefinition"]:
        return self._range_aggregation

    @builtins.property
    def variables(self) -> List["scout_comparisonnotebook_api_ComparisonScatterPlotVariable"]:
        return self._variables

    @builtins.property
    def axes(self) -> List["scout_comparisonnotebook_api_ScatterPlotValueAxes"]:
        return self._axes

    @builtins.property
    def visualization_options(self) -> Optional["scout_comparisonnotebook_api_ComparisonScatterPlotVizOptions"]:
        """
        optional for backcompatibility.
        """
        return self._visualization_options


scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1.__name__ = "ComparisonScatterPlotDefinitionV1"
scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1.__qualname__ = "ComparisonScatterPlotDefinitionV1"
scout_comparisonnotebook_api_ComparisonScatterPlotDefinitionV1.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonScatterPlotVariable(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'x_axis_id': ConjureFieldDefinition('xAxisId', scout_chartdefinition_api_AxisId),
            'y_axis_id': ConjureFieldDefinition('yAxisId', scout_chartdefinition_api_AxisId),
            'locator': ConjureFieldDefinition('locator', scout_comparisonnotebook_api_VariableLocator),
            'aggregation_type': ConjureFieldDefinition('aggregationType', scout_comparisonnotebook_api_AggregationType)
        }

    __slots__: List[str] = ['_x_axis_id', '_y_axis_id', '_locator', '_aggregation_type']

    def __init__(self, aggregation_type: "scout_comparisonnotebook_api_AggregationType", locator: "scout_comparisonnotebook_api_VariableLocator", x_axis_id: str, y_axis_id: str) -> None:
        self._x_axis_id = x_axis_id
        self._y_axis_id = y_axis_id
        self._locator = locator
        self._aggregation_type = aggregation_type

    @builtins.property
    def x_axis_id(self) -> str:
        """
        the axisId when the variable is used as the x variable.
        """
        return self._x_axis_id

    @builtins.property
    def y_axis_id(self) -> str:
        """
        the axisId when the variable is used as the y variable.
        """
        return self._y_axis_id

    @builtins.property
    def locator(self) -> "scout_comparisonnotebook_api_VariableLocator":
        return self._locator

    @builtins.property
    def aggregation_type(self) -> "scout_comparisonnotebook_api_AggregationType":
        return self._aggregation_type


scout_comparisonnotebook_api_ComparisonScatterPlotVariable.__name__ = "ComparisonScatterPlotVariable"
scout_comparisonnotebook_api_ComparisonScatterPlotVariable.__qualname__ = "ComparisonScatterPlotVariable"
scout_comparisonnotebook_api_ComparisonScatterPlotVariable.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonScatterPlotVizOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'coloring': ConjureFieldDefinition('coloring', scout_comparisonnotebook_api_ComparisonScatterPlotColoringOption)
        }

    __slots__: List[str] = ['_coloring']

    def __init__(self, coloring: "scout_comparisonnotebook_api_ComparisonScatterPlotColoringOption") -> None:
        self._coloring = coloring

    @builtins.property
    def coloring(self) -> "scout_comparisonnotebook_api_ComparisonScatterPlotColoringOption":
        return self._coloring


scout_comparisonnotebook_api_ComparisonScatterPlotVizOptions.__name__ = "ComparisonScatterPlotVizOptions"
scout_comparisonnotebook_api_ComparisonScatterPlotVizOptions.__qualname__ = "ComparisonScatterPlotVizOptions"
scout_comparisonnotebook_api_ComparisonScatterPlotVizOptions.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTableColumn(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'locator': ConjureFieldDefinition('locator', scout_comparisonnotebook_api_VariableLocator),
            'visualization_options': ConjureFieldDefinition('visualizationOptions', scout_comparisonnotebook_api_ComparisonTableColumnVisualizationOptions),
            'aggregation_type': ConjureFieldDefinition('aggregationType', OptionalTypeWrapper[scout_comparisonnotebook_api_AggregationType])
        }

    __slots__: List[str] = ['_title', '_locator', '_visualization_options', '_aggregation_type']

    def __init__(self, locator: "scout_comparisonnotebook_api_VariableLocator", visualization_options: "scout_comparisonnotebook_api_ComparisonTableColumnVisualizationOptions", aggregation_type: Optional["scout_comparisonnotebook_api_AggregationType"] = None, title: Optional[str] = None) -> None:
        self._title = title
        self._locator = locator
        self._visualization_options = visualization_options
        self._aggregation_type = aggregation_type

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def locator(self) -> "scout_comparisonnotebook_api_VariableLocator":
        return self._locator

    @builtins.property
    def visualization_options(self) -> "scout_comparisonnotebook_api_ComparisonTableColumnVisualizationOptions":
        return self._visualization_options

    @builtins.property
    def aggregation_type(self) -> Optional["scout_comparisonnotebook_api_AggregationType"]:
        """
        optional for backcompatibility.
        """
        return self._aggregation_type


scout_comparisonnotebook_api_ComparisonTableColumn.__name__ = "ComparisonTableColumn"
scout_comparisonnotebook_api_ComparisonTableColumn.__qualname__ = "ComparisonTableColumn"
scout_comparisonnotebook_api_ComparisonTableColumn.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTableColumnOptions(ConjureEnumType):

    AXIS = 'AXIS'
    '''AXIS'''
    BARS = 'BARS'
    '''BARS'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_comparisonnotebook_api_ComparisonTableColumnOptions.__name__ = "ComparisonTableColumnOptions"
scout_comparisonnotebook_api_ComparisonTableColumnOptions.__qualname__ = "ComparisonTableColumnOptions"
scout_comparisonnotebook_api_ComparisonTableColumnOptions.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTableColumnVisualizationOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'format': ConjureFieldDefinition('format', scout_comparisonnotebook_api_ComparisonTableColumnOptions)
        }

    __slots__: List[str] = ['_format']

    def __init__(self, format: "scout_comparisonnotebook_api_ComparisonTableColumnOptions") -> None:
        self._format = format

    @builtins.property
    def format(self) -> "scout_comparisonnotebook_api_ComparisonTableColumnOptions":
        return self._format


scout_comparisonnotebook_api_ComparisonTableColumnVisualizationOptions.__name__ = "ComparisonTableColumnVisualizationOptions"
scout_comparisonnotebook_api_ComparisonTableColumnVisualizationOptions.__qualname__ = "ComparisonTableColumnVisualizationOptions"
scout_comparisonnotebook_api_ComparisonTableColumnVisualizationOptions.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTableDefinition(ConjureUnionType):
    _v1: Optional["scout_comparisonnotebook_api_ComparisonTableDefinitionV1"] = None
    _v2: Optional["scout_comparisonnotebook_api_ComparisonTableDefinitionV2"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_comparisonnotebook_api_ComparisonTableDefinitionV1),
            'v2': ConjureFieldDefinition('v2', scout_comparisonnotebook_api_ComparisonTableDefinitionV2)
        }

    def __init__(
            self,
            v1: Optional["scout_comparisonnotebook_api_ComparisonTableDefinitionV1"] = None,
            v2: Optional["scout_comparisonnotebook_api_ComparisonTableDefinitionV2"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) + (v2 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'
            if v2 is not None:
                self._v2 = v2
                self._type = 'v2'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'
        elif type_of_union == 'v2':
            if v2 is None:
                raise ValueError('a union value must not be None')
            self._v2 = v2
            self._type = 'v2'

    @builtins.property
    def v1(self) -> Optional["scout_comparisonnotebook_api_ComparisonTableDefinitionV1"]:
        return self._v1

    @builtins.property
    def v2(self) -> Optional["scout_comparisonnotebook_api_ComparisonTableDefinitionV2"]:
        return self._v2

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_ComparisonTableDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_ComparisonTableDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)
        if self._type == 'v2' and self.v2 is not None:
            return visitor._v2(self.v2)


scout_comparisonnotebook_api_ComparisonTableDefinition.__name__ = "ComparisonTableDefinition"
scout_comparisonnotebook_api_ComparisonTableDefinition.__qualname__ = "ComparisonTableDefinition"
scout_comparisonnotebook_api_ComparisonTableDefinition.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTableDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_comparisonnotebook_api_ComparisonTableDefinitionV1") -> Any:
        pass

    @abstractmethod
    def _v2(self, v2: "scout_comparisonnotebook_api_ComparisonTableDefinitionV2") -> Any:
        pass


scout_comparisonnotebook_api_ComparisonTableDefinitionVisitor.__name__ = "ComparisonTableDefinitionVisitor"
scout_comparisonnotebook_api_ComparisonTableDefinitionVisitor.__qualname__ = "ComparisonTableDefinitionVisitor"
scout_comparisonnotebook_api_ComparisonTableDefinitionVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTableDefinitionV1(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'range_aggregations': ConjureFieldDefinition('rangeAggregations', List[scout_comparisonnotebook_api_VariableName]),
            'visualization_options': ConjureFieldDefinition('visualizationOptions', scout_comparisonnotebook_api_ComparisonTableVisualizationOptions),
            'columns': ConjureFieldDefinition('columns', List[scout_comparisonnotebook_api_ComparisonTableColumn])
        }

    __slots__: List[str] = ['_title', '_range_aggregations', '_visualization_options', '_columns']

    def __init__(self, columns: List["scout_comparisonnotebook_api_ComparisonTableColumn"], range_aggregations: List[str], visualization_options: "scout_comparisonnotebook_api_ComparisonTableVisualizationOptions", title: Optional[str] = None) -> None:
        self._title = title
        self._range_aggregations = range_aggregations
        self._visualization_options = visualization_options
        self._columns = columns

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def range_aggregations(self) -> List[str]:
        return self._range_aggregations

    @builtins.property
    def visualization_options(self) -> "scout_comparisonnotebook_api_ComparisonTableVisualizationOptions":
        return self._visualization_options

    @builtins.property
    def columns(self) -> List["scout_comparisonnotebook_api_ComparisonTableColumn"]:
        return self._columns


scout_comparisonnotebook_api_ComparisonTableDefinitionV1.__name__ = "ComparisonTableDefinitionV1"
scout_comparisonnotebook_api_ComparisonTableDefinitionV1.__qualname__ = "ComparisonTableDefinitionV1"
scout_comparisonnotebook_api_ComparisonTableDefinitionV1.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTableDefinitionV2(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'range_aggregation': ConjureFieldDefinition('rangeAggregation', OptionalTypeWrapper[scout_comparisonnotebook_api_RangeAggregationDefinition]),
            'columns': ConjureFieldDefinition('columns', List[scout_comparisonnotebook_api_ComparisonTableColumn])
        }

    __slots__: List[str] = ['_title', '_range_aggregation', '_columns']

    def __init__(self, columns: List["scout_comparisonnotebook_api_ComparisonTableColumn"], range_aggregation: Optional["scout_comparisonnotebook_api_RangeAggregationDefinition"] = None, title: Optional[str] = None) -> None:
        self._title = title
        self._range_aggregation = range_aggregation
        self._columns = columns

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def range_aggregation(self) -> Optional["scout_comparisonnotebook_api_RangeAggregationDefinition"]:
        return self._range_aggregation

    @builtins.property
    def columns(self) -> List["scout_comparisonnotebook_api_ComparisonTableColumn"]:
        return self._columns


scout_comparisonnotebook_api_ComparisonTableDefinitionV2.__name__ = "ComparisonTableDefinitionV2"
scout_comparisonnotebook_api_ComparisonTableDefinitionV2.__qualname__ = "ComparisonTableDefinitionV2"
scout_comparisonnotebook_api_ComparisonTableDefinitionV2.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTableVisualizationOptions(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'aggregation_colors': ConjureFieldDefinition('aggregationColors', Dict[scout_comparisonnotebook_api_VariableName, scout_api_HexColor])
        }

    __slots__: List[str] = ['_aggregation_colors']

    def __init__(self, aggregation_colors: Dict[str, str]) -> None:
        self._aggregation_colors = aggregation_colors

    @builtins.property
    def aggregation_colors(self) -> Dict[str, str]:
        return self._aggregation_colors


scout_comparisonnotebook_api_ComparisonTableVisualizationOptions.__name__ = "ComparisonTableVisualizationOptions"
scout_comparisonnotebook_api_ComparisonTableVisualizationOptions.__qualname__ = "ComparisonTableVisualizationOptions"
scout_comparisonnotebook_api_ComparisonTableVisualizationOptions.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTimeSeriesDisplayOption(ConjureEnumType):
    """
    The display option for the time series plot. A CAROUSEL display option means that each plot grouping 
is displayed one at a time, while a MULTIROW display option means that multiple plot groupings are displayed
simultaneously on different rows.
    """

    CAROUSEL = 'CAROUSEL'
    '''CAROUSEL'''
    MULTIROW = 'MULTIROW'
    '''MULTIROW'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_comparisonnotebook_api_ComparisonTimeSeriesDisplayOption.__name__ = "ComparisonTimeSeriesDisplayOption"
scout_comparisonnotebook_api_ComparisonTimeSeriesDisplayOption.__qualname__ = "ComparisonTimeSeriesDisplayOption"
scout_comparisonnotebook_api_ComparisonTimeSeriesDisplayOption.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTimeSeriesGroupBy(ConjureEnumType):
    """
    The group by option for the time series plot. A RANGE group by option means that all channels on the
same time range are plotted together, while a VARIABLE group by option means that all time ranges 
for the same channel are plotted together.
    """

    RANGE = 'RANGE'
    '''RANGE'''
    VARIABLE = 'VARIABLE'
    '''VARIABLE'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_comparisonnotebook_api_ComparisonTimeSeriesGroupBy.__name__ = "ComparisonTimeSeriesGroupBy"
scout_comparisonnotebook_api_ComparisonTimeSeriesGroupBy.__qualname__ = "ComparisonTimeSeriesGroupBy"
scout_comparisonnotebook_api_ComparisonTimeSeriesGroupBy.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition(ConjureUnionType):
    _v1: Optional["scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'v1': ConjureFieldDefinition('v1', scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1)
        }

    def __init__(
            self,
            v1: Optional["scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (v1 is not None) != 1:
                raise ValueError('a union must contain a single member')

            if v1 is not None:
                self._v1 = v1
                self._type = 'v1'

        elif type_of_union == 'v1':
            if v1 is None:
                raise ValueError('a union value must not be None')
            self._v1 = v1
            self._type = 'v1'

    @builtins.property
    def v1(self) -> Optional["scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1"]:
        return self._v1

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'v1' and self.v1 is not None:
            return visitor._v1(self.v1)


scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition.__name__ = "ComparisonTimeSeriesPlotDefinition"
scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition.__qualname__ = "ComparisonTimeSeriesPlotDefinition"
scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionVisitor:

    @abstractmethod
    def _v1(self, v1: "scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1") -> Any:
        pass


scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionVisitor.__name__ = "ComparisonTimeSeriesPlotDefinitionVisitor"
scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionVisitor.__qualname__ = "ComparisonTimeSeriesPlotDefinitionVisitor"
scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1(ConjureBeanType):
    """
    a cross-range time series plot with 
configurable range and channel groupings
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'title': ConjureFieldDefinition('title', OptionalTypeWrapper[str]),
            'range_aggregation': ConjureFieldDefinition('rangeAggregation', OptionalTypeWrapper[scout_comparisonnotebook_api_RangeAggregationDefinition]),
            'variables': ConjureFieldDefinition('variables', List[scout_comparisonnotebook_api_ComparisonTimeSeriesPlotVariable]),
            'value_axes': ConjureFieldDefinition('valueAxes', List[scout_chartdefinition_api_ValueAxis]),
            'group_by': ConjureFieldDefinition('groupBy', scout_comparisonnotebook_api_ComparisonTimeSeriesGroupBy),
            'display_option': ConjureFieldDefinition('displayOption', scout_comparisonnotebook_api_ComparisonTimeSeriesDisplayOption)
        }

    __slots__: List[str] = ['_title', '_range_aggregation', '_variables', '_value_axes', '_group_by', '_display_option']

    def __init__(self, display_option: "scout_comparisonnotebook_api_ComparisonTimeSeriesDisplayOption", group_by: "scout_comparisonnotebook_api_ComparisonTimeSeriesGroupBy", value_axes: List["scout_chartdefinition_api_ValueAxis"], variables: List["scout_comparisonnotebook_api_ComparisonTimeSeriesPlotVariable"], range_aggregation: Optional["scout_comparisonnotebook_api_RangeAggregationDefinition"] = None, title: Optional[str] = None) -> None:
        self._title = title
        self._range_aggregation = range_aggregation
        self._variables = variables
        self._value_axes = value_axes
        self._group_by = group_by
        self._display_option = display_option

    @builtins.property
    def title(self) -> Optional[str]:
        return self._title

    @builtins.property
    def range_aggregation(self) -> Optional["scout_comparisonnotebook_api_RangeAggregationDefinition"]:
        return self._range_aggregation

    @builtins.property
    def variables(self) -> List["scout_comparisonnotebook_api_ComparisonTimeSeriesPlotVariable"]:
        return self._variables

    @builtins.property
    def value_axes(self) -> List["scout_chartdefinition_api_ValueAxis"]:
        """
        a list of value ranges, where each variable is assigned to a y-axis
        """
        return self._value_axes

    @builtins.property
    def group_by(self) -> "scout_comparisonnotebook_api_ComparisonTimeSeriesGroupBy":
        return self._group_by

    @builtins.property
    def display_option(self) -> "scout_comparisonnotebook_api_ComparisonTimeSeriesDisplayOption":
        return self._display_option


scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1.__name__ = "ComparisonTimeSeriesPlotDefinitionV1"
scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1.__qualname__ = "ComparisonTimeSeriesPlotDefinitionV1"
scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinitionV1.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonTimeSeriesPlotVariable(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'y_axis_id': ConjureFieldDefinition('yAxisId', scout_chartdefinition_api_AxisId),
            'locator': ConjureFieldDefinition('locator', scout_comparisonnotebook_api_VariableLocator)
        }

    __slots__: List[str] = ['_y_axis_id', '_locator']

    def __init__(self, locator: "scout_comparisonnotebook_api_VariableLocator", y_axis_id: str) -> None:
        self._y_axis_id = y_axis_id
        self._locator = locator

    @builtins.property
    def y_axis_id(self) -> str:
        """
        the axisId when the variable is used as the y variable.
        """
        return self._y_axis_id

    @builtins.property
    def locator(self) -> "scout_comparisonnotebook_api_VariableLocator":
        return self._locator


scout_comparisonnotebook_api_ComparisonTimeSeriesPlotVariable.__name__ = "ComparisonTimeSeriesPlotVariable"
scout_comparisonnotebook_api_ComparisonTimeSeriesPlotVariable.__qualname__ = "ComparisonTimeSeriesPlotVariable"
scout_comparisonnotebook_api_ComparisonTimeSeriesPlotVariable.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonWorkbookContent(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'channel_variables': ConjureFieldDefinition('channelVariables', scout_comparisonnotebook_api_ComparisonChannelVariableMap),
            'charts': ConjureFieldDefinition('charts', scout_comparisonnotebook_api_ComparisonVizDefinitionMap)
        }

    __slots__: List[str] = ['_channel_variables', '_charts']

    def __init__(self, channel_variables: Dict[str, "scout_comparisonnotebook_api_ChannelVariable"], charts: Dict[str, "scout_comparisonnotebook_api_VizDefinition"]) -> None:
        self._channel_variables = channel_variables
        self._charts = charts

    @builtins.property
    def channel_variables(self) -> Dict[str, "scout_comparisonnotebook_api_ChannelVariable"]:
        return self._channel_variables

    @builtins.property
    def charts(self) -> Dict[str, "scout_comparisonnotebook_api_VizDefinition"]:
        """
        visualizations. Charts is a legacy term.
        """
        return self._charts


scout_comparisonnotebook_api_ComparisonWorkbookContent.__name__ = "ComparisonWorkbookContent"
scout_comparisonnotebook_api_ComparisonWorkbookContent.__qualname__ = "ComparisonWorkbookContent"
scout_comparisonnotebook_api_ComparisonWorkbookContent.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonWorkbookContext(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variables': ConjureFieldDefinition('variables', Dict[scout_comparisonnotebook_api_VariableName, scout_comparisonnotebook_api_VariableLocator])
        }

    __slots__: List[str] = ['_variables']

    def __init__(self, variables: Dict[str, "scout_comparisonnotebook_api_VariableLocator"]) -> None:
        self._variables = variables

    @builtins.property
    def variables(self) -> Dict[str, "scout_comparisonnotebook_api_VariableLocator"]:
        return self._variables


scout_comparisonnotebook_api_ComparisonWorkbookContext.__name__ = "ComparisonWorkbookContext"
scout_comparisonnotebook_api_ComparisonWorkbookContext.__qualname__ = "ComparisonWorkbookContext"
scout_comparisonnotebook_api_ComparisonWorkbookContext.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonWorkbookDataScope(ConjureUnionType):
    _runs: Optional[List[str]] = None
    _assets: Optional[List[str]] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'runs': ConjureFieldDefinition('runs', List[scout_run_api_RunRid]),
            'assets': ConjureFieldDefinition('assets', List[scout_rids_api_AssetRid])
        }

    def __init__(
            self,
            runs: Optional[List[str]] = None,
            assets: Optional[List[str]] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (runs is not None) + (assets is not None) != 1:
                raise ValueError('a union must contain a single member')

            if runs is not None:
                self._runs = runs
                self._type = 'runs'
            if assets is not None:
                self._assets = assets
                self._type = 'assets'

        elif type_of_union == 'runs':
            if runs is None:
                raise ValueError('a union value must not be None')
            self._runs = runs
            self._type = 'runs'
        elif type_of_union == 'assets':
            if assets is None:
                raise ValueError('a union value must not be None')
            self._assets = assets
            self._type = 'assets'

    @builtins.property
    def runs(self) -> Optional[List[str]]:
        return self._runs

    @builtins.property
    def assets(self) -> Optional[List[str]]:
        return self._assets

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_ComparisonWorkbookDataScopeVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_ComparisonWorkbookDataScopeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'runs' and self.runs is not None:
            return visitor._runs(self.runs)
        if self._type == 'assets' and self.assets is not None:
            return visitor._assets(self.assets)


scout_comparisonnotebook_api_ComparisonWorkbookDataScope.__name__ = "ComparisonWorkbookDataScope"
scout_comparisonnotebook_api_ComparisonWorkbookDataScope.__qualname__ = "ComparisonWorkbookDataScope"
scout_comparisonnotebook_api_ComparisonWorkbookDataScope.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComparisonWorkbookDataScopeVisitor:

    @abstractmethod
    def _runs(self, runs: List[str]) -> Any:
        pass

    @abstractmethod
    def _assets(self, assets: List[str]) -> Any:
        pass


scout_comparisonnotebook_api_ComparisonWorkbookDataScopeVisitor.__name__ = "ComparisonWorkbookDataScopeVisitor"
scout_comparisonnotebook_api_ComparisonWorkbookDataScopeVisitor.__qualname__ = "ComparisonWorkbookDataScopeVisitor"
scout_comparisonnotebook_api_ComparisonWorkbookDataScopeVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ComputeNodeWithContext(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'compute_node': ConjureFieldDefinition('computeNode', scout_compute_api_ComputeNode),
            'context': ConjureFieldDefinition('context', scout_comparisonnotebook_api_ComparisonWorkbookContext),
            'supplemental_context': ConjureFieldDefinition('supplementalContext', scout_comparisonnotebook_api_SupplementalComparisonWorkbookContext)
        }

    __slots__: List[str] = ['_compute_node', '_context', '_supplemental_context']

    def __init__(self, compute_node: "scout_compute_api_ComputeNode", context: "scout_comparisonnotebook_api_ComparisonWorkbookContext", supplemental_context: "scout_comparisonnotebook_api_SupplementalComparisonWorkbookContext") -> None:
        self._compute_node = compute_node
        self._context = context
        self._supplemental_context = supplemental_context

    @builtins.property
    def compute_node(self) -> "scout_compute_api_ComputeNode":
        return self._compute_node

    @builtins.property
    def context(self) -> "scout_comparisonnotebook_api_ComparisonWorkbookContext":
        return self._context

    @builtins.property
    def supplemental_context(self) -> "scout_comparisonnotebook_api_SupplementalComparisonWorkbookContext":
        return self._supplemental_context


scout_comparisonnotebook_api_ComputeNodeWithContext.__name__ = "ComputeNodeWithContext"
scout_comparisonnotebook_api_ComputeNodeWithContext.__qualname__ = "ComputeNodeWithContext"
scout_comparisonnotebook_api_ComputeNodeWithContext.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_Max(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_comparisonnotebook_api_Max.__name__ = "Max"
scout_comparisonnotebook_api_Max.__qualname__ = "Max"
scout_comparisonnotebook_api_Max.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_Mean(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_comparisonnotebook_api_Mean.__name__ = "Mean"
scout_comparisonnotebook_api_Mean.__qualname__ = "Mean"
scout_comparisonnotebook_api_Mean.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_Min(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_comparisonnotebook_api_Min.__name__ = "Min"
scout_comparisonnotebook_api_Min.__qualname__ = "Min"
scout_comparisonnotebook_api_Min.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_RangeAggregationContext(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'variables_to_aggregate': ConjureFieldDefinition('variablesToAggregate', List[scout_comparisonnotebook_api_VariableWithAggregation])
        }

    __slots__: List[str] = ['_variables_to_aggregate']

    def __init__(self, variables_to_aggregate: List["scout_comparisonnotebook_api_VariableWithAggregation"]) -> None:
        self._variables_to_aggregate = variables_to_aggregate

    @builtins.property
    def variables_to_aggregate(self) -> List["scout_comparisonnotebook_api_VariableWithAggregation"]:
        return self._variables_to_aggregate


scout_comparisonnotebook_api_RangeAggregationContext.__name__ = "RangeAggregationContext"
scout_comparisonnotebook_api_RangeAggregationContext.__qualname__ = "RangeAggregationContext"
scout_comparisonnotebook_api_RangeAggregationContext.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_RangeAggregationDefinition(ConjureBeanType):
    """
    Every range aggregation must be defined by a data scope and set of conditions. This type should be used by
every comparison workbook viz definition.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_scope': ConjureFieldDefinition('dataScope', scout_comparisonnotebook_api_ComparisonWorkbookDataScope),
            'condition': ConjureFieldDefinition('condition', scout_comparisonnotebook_api_ComputeNodeWithContext),
            'time_window': ConjureFieldDefinition('timeWindow', OptionalTypeWrapper[scout_comparisonnotebook_api_TimeWindow])
        }

    __slots__: List[str] = ['_data_scope', '_condition', '_time_window']

    def __init__(self, condition: "scout_comparisonnotebook_api_ComputeNodeWithContext", data_scope: "scout_comparisonnotebook_api_ComparisonWorkbookDataScope", time_window: Optional["scout_comparisonnotebook_api_TimeWindow"] = None) -> None:
        self._data_scope = data_scope
        self._condition = condition
        self._time_window = time_window

    @builtins.property
    def data_scope(self) -> "scout_comparisonnotebook_api_ComparisonWorkbookDataScope":
        return self._data_scope

    @builtins.property
    def condition(self) -> "scout_comparisonnotebook_api_ComputeNodeWithContext":
        return self._condition

    @builtins.property
    def time_window(self) -> Optional["scout_comparisonnotebook_api_TimeWindow"]:
        """
        determines the window of data included in the visualization. Currently only applied on assets.
        """
        return self._time_window


scout_comparisonnotebook_api_RangeAggregationDefinition.__name__ = "RangeAggregationDefinition"
scout_comparisonnotebook_api_RangeAggregationDefinition.__qualname__ = "RangeAggregationDefinition"
scout_comparisonnotebook_api_RangeAggregationDefinition.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_ScatterPlotValueAxes(ConjureBeanType):
    """
    specifies the axes configurations for a given x and y axis.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'x_axis': ConjureFieldDefinition('xAxis', scout_chartdefinition_api_ValueAxis),
            'y_axis': ConjureFieldDefinition('yAxis', scout_chartdefinition_api_ValueAxis)
        }

    __slots__: List[str] = ['_x_axis', '_y_axis']

    def __init__(self, x_axis: "scout_chartdefinition_api_ValueAxis", y_axis: "scout_chartdefinition_api_ValueAxis") -> None:
        self._x_axis = x_axis
        self._y_axis = y_axis

    @builtins.property
    def x_axis(self) -> "scout_chartdefinition_api_ValueAxis":
        return self._x_axis

    @builtins.property
    def y_axis(self) -> "scout_chartdefinition_api_ValueAxis":
        return self._y_axis


scout_comparisonnotebook_api_ScatterPlotValueAxes.__name__ = "ScatterPlotValueAxes"
scout_comparisonnotebook_api_ScatterPlotValueAxes.__qualname__ = "ScatterPlotValueAxes"
scout_comparisonnotebook_api_ScatterPlotValueAxes.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_StandardDeviation(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_comparisonnotebook_api_StandardDeviation.__name__ = "StandardDeviation"
scout_comparisonnotebook_api_StandardDeviation.__qualname__ = "StandardDeviation"
scout_comparisonnotebook_api_StandardDeviation.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_SupplementalComparisonWorkbookContext(ConjureUnionType):
    """This is used to allow variables to specify additional context that does not fit well into the general shape of
a compute node. For example, a range aggregation variable represents a bulk computation across several inputs,
whose context is specified here instead."""
    _none: Optional["scout_rids_api_Empty"] = None
    _range_aggregation: Optional["scout_comparisonnotebook_api_RangeAggregationContext"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'none': ConjureFieldDefinition('none', scout_rids_api_Empty),
            'range_aggregation': ConjureFieldDefinition('rangeAggregation', scout_comparisonnotebook_api_RangeAggregationContext)
        }

    def __init__(
            self,
            none: Optional["scout_rids_api_Empty"] = None,
            range_aggregation: Optional["scout_comparisonnotebook_api_RangeAggregationContext"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (none is not None) + (range_aggregation is not None) != 1:
                raise ValueError('a union must contain a single member')

            if none is not None:
                self._none = none
                self._type = 'none'
            if range_aggregation is not None:
                self._range_aggregation = range_aggregation
                self._type = 'rangeAggregation'

        elif type_of_union == 'none':
            if none is None:
                raise ValueError('a union value must not be None')
            self._none = none
            self._type = 'none'
        elif type_of_union == 'rangeAggregation':
            if range_aggregation is None:
                raise ValueError('a union value must not be None')
            self._range_aggregation = range_aggregation
            self._type = 'rangeAggregation'

    @builtins.property
    def none(self) -> Optional["scout_rids_api_Empty"]:
        return self._none

    @builtins.property
    def range_aggregation(self) -> Optional["scout_comparisonnotebook_api_RangeAggregationContext"]:
        return self._range_aggregation

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_SupplementalComparisonWorkbookContextVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_SupplementalComparisonWorkbookContextVisitor'.format(visitor.__class__.__name__))
        if self._type == 'none' and self.none is not None:
            return visitor._none(self.none)
        if self._type == 'rangeAggregation' and self.range_aggregation is not None:
            return visitor._range_aggregation(self.range_aggregation)


scout_comparisonnotebook_api_SupplementalComparisonWorkbookContext.__name__ = "SupplementalComparisonWorkbookContext"
scout_comparisonnotebook_api_SupplementalComparisonWorkbookContext.__qualname__ = "SupplementalComparisonWorkbookContext"
scout_comparisonnotebook_api_SupplementalComparisonWorkbookContext.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_SupplementalComparisonWorkbookContextVisitor:

    @abstractmethod
    def _none(self, none: "scout_rids_api_Empty") -> Any:
        pass

    @abstractmethod
    def _range_aggregation(self, range_aggregation: "scout_comparisonnotebook_api_RangeAggregationContext") -> Any:
        pass


scout_comparisonnotebook_api_SupplementalComparisonWorkbookContextVisitor.__name__ = "SupplementalComparisonWorkbookContextVisitor"
scout_comparisonnotebook_api_SupplementalComparisonWorkbookContextVisitor.__qualname__ = "SupplementalComparisonWorkbookContextVisitor"
scout_comparisonnotebook_api_SupplementalComparisonWorkbookContextVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_TimeWindow(ConjureUnionType):
    _relative: Optional["scout_run_api_Duration"] = None
    _fixed: Optional["scout_comparisonnotebook_api_TimestampedRange"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'relative': ConjureFieldDefinition('relative', scout_run_api_Duration),
            'fixed': ConjureFieldDefinition('fixed', scout_comparisonnotebook_api_TimestampedRange)
        }

    def __init__(
            self,
            relative: Optional["scout_run_api_Duration"] = None,
            fixed: Optional["scout_comparisonnotebook_api_TimestampedRange"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (relative is not None) + (fixed is not None) != 1:
                raise ValueError('a union must contain a single member')

            if relative is not None:
                self._relative = relative
                self._type = 'relative'
            if fixed is not None:
                self._fixed = fixed
                self._type = 'fixed'

        elif type_of_union == 'relative':
            if relative is None:
                raise ValueError('a union value must not be None')
            self._relative = relative
            self._type = 'relative'
        elif type_of_union == 'fixed':
            if fixed is None:
                raise ValueError('a union value must not be None')
            self._fixed = fixed
            self._type = 'fixed'

    @builtins.property
    def relative(self) -> Optional["scout_run_api_Duration"]:
        return self._relative

    @builtins.property
    def fixed(self) -> Optional["scout_comparisonnotebook_api_TimestampedRange"]:
        return self._fixed

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_TimeWindowVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_TimeWindowVisitor'.format(visitor.__class__.__name__))
        if self._type == 'relative' and self.relative is not None:
            return visitor._relative(self.relative)
        if self._type == 'fixed' and self.fixed is not None:
            return visitor._fixed(self.fixed)


scout_comparisonnotebook_api_TimeWindow.__name__ = "TimeWindow"
scout_comparisonnotebook_api_TimeWindow.__qualname__ = "TimeWindow"
scout_comparisonnotebook_api_TimeWindow.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_TimeWindowVisitor:

    @abstractmethod
    def _relative(self, relative: "scout_run_api_Duration") -> Any:
        pass

    @abstractmethod
    def _fixed(self, fixed: "scout_comparisonnotebook_api_TimestampedRange") -> Any:
        pass


scout_comparisonnotebook_api_TimeWindowVisitor.__name__ = "TimeWindowVisitor"
scout_comparisonnotebook_api_TimeWindowVisitor.__qualname__ = "TimeWindowVisitor"
scout_comparisonnotebook_api_TimeWindowVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_TimestampedRange(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'start': ConjureFieldDefinition('start', OptionalTypeWrapper[api_Timestamp]),
            'end': ConjureFieldDefinition('end', OptionalTypeWrapper[api_Timestamp])
        }

    __slots__: List[str] = ['_start', '_end']

    def __init__(self, end: Optional["api_Timestamp"] = None, start: Optional["api_Timestamp"] = None) -> None:
        self._start = start
        self._end = end

    @builtins.property
    def start(self) -> Optional["api_Timestamp"]:
        return self._start

    @builtins.property
    def end(self) -> Optional["api_Timestamp"]:
        return self._end


scout_comparisonnotebook_api_TimestampedRange.__name__ = "TimestampedRange"
scout_comparisonnotebook_api_TimestampedRange.__qualname__ = "TimestampedRange"
scout_comparisonnotebook_api_TimestampedRange.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_VariableLocator(ConjureUnionType):
    _comparison_workbook_variable: Optional[str] = None
    _series: Optional["scout_api_ChannelLocator"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'comparison_workbook_variable': ConjureFieldDefinition('comparisonWorkbookVariable', scout_comparisonnotebook_api_VariableName),
            'series': ConjureFieldDefinition('series', scout_api_ChannelLocator)
        }

    def __init__(
            self,
            comparison_workbook_variable: Optional[str] = None,
            series: Optional["scout_api_ChannelLocator"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (comparison_workbook_variable is not None) + (series is not None) != 1:
                raise ValueError('a union must contain a single member')

            if comparison_workbook_variable is not None:
                self._comparison_workbook_variable = comparison_workbook_variable
                self._type = 'comparisonWorkbookVariable'
            if series is not None:
                self._series = series
                self._type = 'series'

        elif type_of_union == 'comparisonWorkbookVariable':
            if comparison_workbook_variable is None:
                raise ValueError('a union value must not be None')
            self._comparison_workbook_variable = comparison_workbook_variable
            self._type = 'comparisonWorkbookVariable'
        elif type_of_union == 'series':
            if series is None:
                raise ValueError('a union value must not be None')
            self._series = series
            self._type = 'series'

    @builtins.property
    def comparison_workbook_variable(self) -> Optional[str]:
        return self._comparison_workbook_variable

    @builtins.property
    def series(self) -> Optional["scout_api_ChannelLocator"]:
        return self._series

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_VariableLocatorVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_VariableLocatorVisitor'.format(visitor.__class__.__name__))
        if self._type == 'comparisonWorkbookVariable' and self.comparison_workbook_variable is not None:
            return visitor._comparison_workbook_variable(self.comparison_workbook_variable)
        if self._type == 'series' and self.series is not None:
            return visitor._series(self.series)


scout_comparisonnotebook_api_VariableLocator.__name__ = "VariableLocator"
scout_comparisonnotebook_api_VariableLocator.__qualname__ = "VariableLocator"
scout_comparisonnotebook_api_VariableLocator.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_VariableLocatorVisitor:

    @abstractmethod
    def _comparison_workbook_variable(self, comparison_workbook_variable: str) -> Any:
        pass

    @abstractmethod
    def _series(self, series: "scout_api_ChannelLocator") -> Any:
        pass


scout_comparisonnotebook_api_VariableLocatorVisitor.__name__ = "VariableLocatorVisitor"
scout_comparisonnotebook_api_VariableLocatorVisitor.__qualname__ = "VariableLocatorVisitor"
scout_comparisonnotebook_api_VariableLocatorVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_VariableWithAggregation(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'locator': ConjureFieldDefinition('locator', scout_comparisonnotebook_api_VariableLocator),
            'aggregation_type': ConjureFieldDefinition('aggregationType', scout_comparisonnotebook_api_AggregationType)
        }

    __slots__: List[str] = ['_locator', '_aggregation_type']

    def __init__(self, aggregation_type: "scout_comparisonnotebook_api_AggregationType", locator: "scout_comparisonnotebook_api_VariableLocator") -> None:
        self._locator = locator
        self._aggregation_type = aggregation_type

    @builtins.property
    def locator(self) -> "scout_comparisonnotebook_api_VariableLocator":
        return self._locator

    @builtins.property
    def aggregation_type(self) -> "scout_comparisonnotebook_api_AggregationType":
        return self._aggregation_type


scout_comparisonnotebook_api_VariableWithAggregation.__name__ = "VariableWithAggregation"
scout_comparisonnotebook_api_VariableWithAggregation.__qualname__ = "VariableWithAggregation"
scout_comparisonnotebook_api_VariableWithAggregation.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_VizDefinition(ConjureUnionType):
    _table: Optional["scout_comparisonnotebook_api_ComparisonTableDefinition"] = None
    _scatter: Optional["scout_comparisonnotebook_api_ComparisonScatterPlotDefinition"] = None
    _time_series: Optional["scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'table': ConjureFieldDefinition('table', scout_comparisonnotebook_api_ComparisonTableDefinition),
            'scatter': ConjureFieldDefinition('scatter', scout_comparisonnotebook_api_ComparisonScatterPlotDefinition),
            'time_series': ConjureFieldDefinition('timeSeries', scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition)
        }

    def __init__(
            self,
            table: Optional["scout_comparisonnotebook_api_ComparisonTableDefinition"] = None,
            scatter: Optional["scout_comparisonnotebook_api_ComparisonScatterPlotDefinition"] = None,
            time_series: Optional["scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (table is not None) + (scatter is not None) + (time_series is not None) != 1:
                raise ValueError('a union must contain a single member')

            if table is not None:
                self._table = table
                self._type = 'table'
            if scatter is not None:
                self._scatter = scatter
                self._type = 'scatter'
            if time_series is not None:
                self._time_series = time_series
                self._type = 'timeSeries'

        elif type_of_union == 'table':
            if table is None:
                raise ValueError('a union value must not be None')
            self._table = table
            self._type = 'table'
        elif type_of_union == 'scatter':
            if scatter is None:
                raise ValueError('a union value must not be None')
            self._scatter = scatter
            self._type = 'scatter'
        elif type_of_union == 'timeSeries':
            if time_series is None:
                raise ValueError('a union value must not be None')
            self._time_series = time_series
            self._type = 'timeSeries'

    @builtins.property
    def table(self) -> Optional["scout_comparisonnotebook_api_ComparisonTableDefinition"]:
        return self._table

    @builtins.property
    def scatter(self) -> Optional["scout_comparisonnotebook_api_ComparisonScatterPlotDefinition"]:
        return self._scatter

    @builtins.property
    def time_series(self) -> Optional["scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition"]:
        return self._time_series

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonnotebook_api_VizDefinitionVisitor):
            raise ValueError('{} is not an instance of scout_comparisonnotebook_api_VizDefinitionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'table' and self.table is not None:
            return visitor._table(self.table)
        if self._type == 'scatter' and self.scatter is not None:
            return visitor._scatter(self.scatter)
        if self._type == 'timeSeries' and self.time_series is not None:
            return visitor._time_series(self.time_series)


scout_comparisonnotebook_api_VizDefinition.__name__ = "VizDefinition"
scout_comparisonnotebook_api_VizDefinition.__qualname__ = "VizDefinition"
scout_comparisonnotebook_api_VizDefinition.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonnotebook_api_VizDefinitionVisitor:

    @abstractmethod
    def _table(self, table: "scout_comparisonnotebook_api_ComparisonTableDefinition") -> Any:
        pass

    @abstractmethod
    def _scatter(self, scatter: "scout_comparisonnotebook_api_ComparisonScatterPlotDefinition") -> Any:
        pass

    @abstractmethod
    def _time_series(self, time_series: "scout_comparisonnotebook_api_ComparisonTimeSeriesPlotDefinition") -> Any:
        pass


scout_comparisonnotebook_api_VizDefinitionVisitor.__name__ = "VizDefinitionVisitor"
scout_comparisonnotebook_api_VizDefinitionVisitor.__qualname__ = "VizDefinitionVisitor"
scout_comparisonnotebook_api_VizDefinitionVisitor.__module__ = "nominal_api.scout_comparisonnotebook_api"


class scout_comparisonrun_api_ComparisonRun(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'run_rid': ConjureFieldDefinition('runRid', scout_run_api_RunRid),
            'enabled': ConjureFieldDefinition('enabled', bool),
            'offset_override': ConjureFieldDefinition('offsetOverride', OptionalTypeWrapper[scout_comparisonrun_api_Offset])
        }

    __slots__: List[str] = ['_run_rid', '_enabled', '_offset_override']

    def __init__(self, enabled: bool, run_rid: str, offset_override: Optional["scout_comparisonrun_api_Offset"] = None) -> None:
        self._run_rid = run_rid
        self._enabled = enabled
        self._offset_override = offset_override

    @builtins.property
    def run_rid(self) -> str:
        return self._run_rid

    @builtins.property
    def enabled(self) -> bool:
        return self._enabled

    @builtins.property
    def offset_override(self) -> Optional["scout_comparisonrun_api_Offset"]:
        return self._offset_override


scout_comparisonrun_api_ComparisonRun.__name__ = "ComparisonRun"
scout_comparisonrun_api_ComparisonRun.__qualname__ = "ComparisonRun"
scout_comparisonrun_api_ComparisonRun.__module__ = "nominal_api.scout_comparisonrun_api"


class scout_comparisonrun_api_ComparisonRunGroup(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'uuid': ConjureFieldDefinition('uuid', str),
            'name': ConjureFieldDefinition('name', str),
            'offset': ConjureFieldDefinition('offset', scout_comparisonrun_api_Offset),
            'offset_anchor': ConjureFieldDefinition('offsetAnchor', scout_comparisonrun_api_OffsetAnchor),
            'runs': ConjureFieldDefinition('runs', List[scout_comparisonrun_api_ComparisonRun]),
            'color': ConjureFieldDefinition('color', OptionalTypeWrapper[scout_api_HexColor])
        }

    __slots__: List[str] = ['_uuid', '_name', '_offset', '_offset_anchor', '_runs', '_color']

    def __init__(self, name: str, offset: "scout_comparisonrun_api_Offset", offset_anchor: "scout_comparisonrun_api_OffsetAnchor", runs: List["scout_comparisonrun_api_ComparisonRun"], uuid: str, color: Optional[str] = None) -> None:
        self._uuid = uuid
        self._name = name
        self._offset = offset
        self._offset_anchor = offset_anchor
        self._runs = runs
        self._color = color

    @builtins.property
    def uuid(self) -> str:
        return self._uuid

    @builtins.property
    def name(self) -> str:
        return self._name

    @builtins.property
    def offset(self) -> "scout_comparisonrun_api_Offset":
        return self._offset

    @builtins.property
    def offset_anchor(self) -> "scout_comparisonrun_api_OffsetAnchor":
        return self._offset_anchor

    @builtins.property
    def runs(self) -> List["scout_comparisonrun_api_ComparisonRun"]:
        return self._runs

    @builtins.property
    def color(self) -> Optional[str]:
        return self._color


scout_comparisonrun_api_ComparisonRunGroup.__name__ = "ComparisonRunGroup"
scout_comparisonrun_api_ComparisonRunGroup.__qualname__ = "ComparisonRunGroup"
scout_comparisonrun_api_ComparisonRunGroup.__module__ = "nominal_api.scout_comparisonrun_api"


class scout_comparisonrun_api_Offset(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'value': ConjureFieldDefinition('value', int),
            'unit': ConjureFieldDefinition('unit', api_TimeUnit)
        }

    __slots__: List[str] = ['_value', '_unit']

    def __init__(self, unit: "api_TimeUnit", value: int) -> None:
        self._value = value
        self._unit = unit

    @builtins.property
    def value(self) -> int:
        return self._value

    @builtins.property
    def unit(self) -> "api_TimeUnit":
        return self._unit


scout_comparisonrun_api_Offset.__name__ = "Offset"
scout_comparisonrun_api_Offset.__qualname__ = "Offset"
scout_comparisonrun_api_Offset.__module__ = "nominal_api.scout_comparisonrun_api"


class scout_comparisonrun_api_OffsetAnchor(ConjureUnionType):
    _run: Optional["scout_comparisonrun_api_OffsetRunAnchor"] = None
    _series: Optional["scout_comparisonrun_api_OffsetSeriesAnchor"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'run': ConjureFieldDefinition('run', scout_comparisonrun_api_OffsetRunAnchor),
            'series': ConjureFieldDefinition('series', scout_comparisonrun_api_OffsetSeriesAnchor)
        }

    def __init__(
            self,
            run: Optional["scout_comparisonrun_api_OffsetRunAnchor"] = None,
            series: Optional["scout_comparisonrun_api_OffsetSeriesAnchor"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (run is not None) + (series is not None) != 1:
                raise ValueError('a union must contain a single member')

            if run is not None:
                self._run = run
                self._type = 'run'
            if series is not None:
                self._series = series
                self._type = 'series'

        elif type_of_union == 'run':
            if run is None:
                raise ValueError('a union value must not be None')
            self._run = run
            self._type = 'run'
        elif type_of_union == 'series':
            if series is None:
                raise ValueError('a union value must not be None')
            self._series = series
            self._type = 'series'

    @builtins.property
    def run(self) -> Optional["scout_comparisonrun_api_OffsetRunAnchor"]:
        return self._run

    @builtins.property
    def series(self) -> Optional["scout_comparisonrun_api_OffsetSeriesAnchor"]:
        return self._series

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_comparisonrun_api_OffsetAnchorVisitor):
            raise ValueError('{} is not an instance of scout_comparisonrun_api_OffsetAnchorVisitor'.format(visitor.__class__.__name__))
        if self._type == 'run' and self.run is not None:
            return visitor._run(self.run)
        if self._type == 'series' and self.series is not None:
            return visitor._series(self.series)


scout_comparisonrun_api_OffsetAnchor.__name__ = "OffsetAnchor"
scout_comparisonrun_api_OffsetAnchor.__qualname__ = "OffsetAnchor"
scout_comparisonrun_api_OffsetAnchor.__module__ = "nominal_api.scout_comparisonrun_api"


class scout_comparisonrun_api_OffsetAnchorVisitor:

    @abstractmethod
    def _run(self, run: "scout_comparisonrun_api_OffsetRunAnchor") -> Any:
        pass

    @abstractmethod
    def _series(self, series: "scout_comparisonrun_api_OffsetSeriesAnchor") -> Any:
        pass


scout_comparisonrun_api_OffsetAnchorVisitor.__name__ = "OffsetAnchorVisitor"
scout_comparisonrun_api_OffsetAnchorVisitor.__qualname__ = "OffsetAnchorVisitor"
scout_comparisonrun_api_OffsetAnchorVisitor.__module__ = "nominal_api.scout_comparisonrun_api"


class scout_comparisonrun_api_OffsetRunAnchor(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_comparisonrun_api_OffsetRunAnchor.__name__ = "OffsetRunAnchor"
scout_comparisonrun_api_OffsetRunAnchor.__qualname__ = "OffsetRunAnchor"
scout_comparisonrun_api_OffsetRunAnchor.__module__ = "nominal_api.scout_comparisonrun_api"


class scout_comparisonrun_api_OffsetSeriesAnchor(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'series_variable': ConjureFieldDefinition('seriesVariable', scout_channelvariables_api_ChannelVariableName)
        }

    __slots__: List[str] = ['_series_variable']

    def __init__(self, series_variable: str) -> None:
        self._series_variable = series_variable

    @builtins.property
    def series_variable(self) -> str:
        return self._series_variable


scout_comparisonrun_api_OffsetSeriesAnchor.__name__ = "OffsetSeriesAnchor"
scout_comparisonrun_api_OffsetSeriesAnchor.__qualname__ = "OffsetSeriesAnchor"
scout_comparisonrun_api_OffsetSeriesAnchor.__module__ = "nominal_api.scout_comparisonrun_api"


class scout_compute_api_AbsoluteThreshold(ConjureBeanType):
    """
    Threshold defined as a real number corresponding the unit of a series.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'value': ConjureFieldDefinition('value', scout_compute_api_DoubleConstant)
        }

    __slots__: List[str] = ['_value']

    def __init__(self, value: "scout_compute_api_DoubleConstant") -> None:
        self._value = value

    @builtins.property
    def value(self) -> "scout_compute_api_DoubleConstant":
        return self._value


scout_compute_api_AbsoluteThreshold.__name__ = "AbsoluteThreshold"
scout_compute_api_AbsoluteThreshold.__qualname__ = "AbsoluteThreshold"
scout_compute_api_AbsoluteThreshold.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_AfterPersistenceWindow(ConjureBeanType):
    """
    The first point in the output range will be the first point after the condition has been true
for the min duration and min points.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_compute_api_AfterPersistenceWindow.__name__ = "AfterPersistenceWindow"
scout_compute_api_AfterPersistenceWindow.__qualname__ = "AfterPersistenceWindow"
scout_compute_api_AfterPersistenceWindow.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_AggregateEnumSeries(ConjureBeanType):
    """
    Aggregates values with duplicate timestamps in the input series values into a single value using the specified aggregation function.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'input': ConjureFieldDefinition('input', scout_compute_api_EnumSeries),
            'function': ConjureFieldDefinition('function', scout_compute_api_EnumAggregationFunction)
        }

    __slots__: List[str] = ['_input', '_function']

    def __init__(self, function: "scout_compute_api_EnumAggregationFunction", input: "scout_compute_api_EnumSeries") -> None:
        self._input = input
        self._function = function

    @builtins.property
    def input(self) -> "scout_compute_api_EnumSeries":
        return self._input

    @builtins.property
    def function(self) -> "scout_compute_api_EnumAggregationFunction":
        return self._function


scout_compute_api_AggregateEnumSeries.__name__ = "AggregateEnumSeries"
scout_compute_api_AggregateEnumSeries.__qualname__ = "AggregateEnumSeries"
scout_compute_api_AggregateEnumSeries.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_AggregateNumericSeries(ConjureBeanType):
    """
    Aggregates values with duplicate timestamps in the input series values into a single value using the specified aggregation function.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'input': ConjureFieldDefinition('input', scout_compute_api_NumericSeries),
            'function': ConjureFieldDefinition('function', scout_compute_api_NumericAggregationFunction)
        }

    __slots__: List[str] = ['_input', '_function']

    def __init__(self, function: "scout_compute_api_NumericAggregationFunction", input: "scout_compute_api_NumericSeries") -> None:
        self._input = input
        self._function = function

    @builtins.property
    def input(self) -> "scout_compute_api_NumericSeries":
        return self._input

    @builtins.property
    def function(self) -> "scout_compute_api_NumericAggregationFunction":
        return self._function


scout_compute_api_AggregateNumericSeries.__name__ = "AggregateNumericSeries"
scout_compute_api_AggregateNumericSeries.__qualname__ = "AggregateNumericSeries"
scout_compute_api_AggregateNumericSeries.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_AllowNegativeValues(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_compute_api_AllowNegativeValues.__name__ = "AllowNegativeValues"
scout_compute_api_AllowNegativeValues.__qualname__ = "AllowNegativeValues"
scout_compute_api_AllowNegativeValues.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_ArithmeticSeries(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'inputs': ConjureFieldDefinition('inputs', Dict[scout_compute_api_LocalVariableName, scout_compute_api_NumericSeries]),
            'expression': ConjureFieldDefinition('expression', str),
            'interpolation_configuration': ConjureFieldDefinition('interpolationConfiguration', OptionalTypeWrapper[scout_compute_api_InterpolationConfiguration])
        }

    __slots__: List[str] = ['_inputs', '_expression', '_interpolation_configuration']

    def __init__(self, expression: str, inputs: Dict[str, "scout_compute_api_NumericSeries"], interpolation_configuration: Optional["scout_compute_api_InterpolationConfiguration"] = None) -> None:
        self._inputs = inputs
        self._expression = expression
        self._interpolation_configuration = interpolation_configuration

    @builtins.property
    def inputs(self) -> Dict[str, "scout_compute_api_NumericSeries"]:
        return self._inputs

    @builtins.property
    def expression(self) -> str:
        return self._expression

    @builtins.property
    def interpolation_configuration(self) -> Optional["scout_compute_api_InterpolationConfiguration"]:
        """
        Defaults to forward fill interpolation with a 1s interpolation radius
        """
        return self._interpolation_configuration


scout_compute_api_ArithmeticSeries.__name__ = "ArithmeticSeries"
scout_compute_api_ArithmeticSeries.__qualname__ = "ArithmeticSeries"
scout_compute_api_ArithmeticSeries.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_AssetChannel(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'asset_rid': ConjureFieldDefinition('assetRid', scout_compute_api_StringConstant),
            'data_scope_name': ConjureFieldDefinition('dataScopeName', scout_compute_api_StringConstant),
            'channel': ConjureFieldDefinition('channel', scout_compute_api_StringConstant),
            'additional_tags': ConjureFieldDefinition('additionalTags', Dict[str, scout_compute_api_StringConstant])
        }

    __slots__: List[str] = ['_asset_rid', '_data_scope_name', '_channel', '_additional_tags']

    def __init__(self, additional_tags: Dict[str, "scout_compute_api_StringConstant"], asset_rid: "scout_compute_api_StringConstant", channel: "scout_compute_api_StringConstant", data_scope_name: "scout_compute_api_StringConstant") -> None:
        self._asset_rid = asset_rid
        self._data_scope_name = data_scope_name
        self._channel = channel
        self._additional_tags = additional_tags

    @builtins.property
    def asset_rid(self) -> "scout_compute_api_StringConstant":
        return self._asset_rid

    @builtins.property
    def data_scope_name(self) -> "scout_compute_api_StringConstant":
        """
        Used to disambiguate when multiple data scopes within this asset contain channels with the same name.
        """
        return self._data_scope_name

    @builtins.property
    def channel(self) -> "scout_compute_api_StringConstant":
        return self._channel

    @builtins.property
    def additional_tags(self) -> Dict[str, "scout_compute_api_StringConstant"]:
        """
        Tags to filter the channel by, in addition to tag filters defined for a given Asset data scope. Throws on 
collisions with tag keys already defined for the given Asset data scope. Only returns points that match 
both sets of tag filters.
        """
        return self._additional_tags


scout_compute_api_AssetChannel.__name__ = "AssetChannel"
scout_compute_api_AssetChannel.__qualname__ = "AssetChannel"
scout_compute_api_AssetChannel.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_Average(ConjureBeanType):
    """
    A average that gives the same weight to each observation inside the time window
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
        }

    __slots__: List[str] = []



scout_compute_api_Average.__name__ = "Average"
scout_compute_api_Average.__qualname__ = "Average"
scout_compute_api_Average.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BatchComputeUnitResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'results': ConjureFieldDefinition('results', List[scout_compute_api_ComputeUnitResult])
        }

    __slots__: List[str] = ['_results']

    def __init__(self, results: List["scout_compute_api_ComputeUnitResult"]) -> None:
        self._results = results

    @builtins.property
    def results(self) -> List["scout_compute_api_ComputeUnitResult"]:
        return self._results


scout_compute_api_BatchComputeUnitResult.__name__ = "BatchComputeUnitResult"
scout_compute_api_BatchComputeUnitResult.__qualname__ = "BatchComputeUnitResult"
scout_compute_api_BatchComputeUnitResult.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BatchComputeUnitsRequest(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'requests': ConjureFieldDefinition('requests', List[scout_compute_api_ComputeUnitsRequest])
        }

    __slots__: List[str] = ['_requests']

    def __init__(self, requests: List["scout_compute_api_ComputeUnitsRequest"]) -> None:
        self._requests = requests

    @builtins.property
    def requests(self) -> List["scout_compute_api_ComputeUnitsRequest"]:
        return self._requests


scout_compute_api_BatchComputeUnitsRequest.__name__ = "BatchComputeUnitsRequest"
scout_compute_api_BatchComputeUnitsRequest.__qualname__ = "BatchComputeUnitsRequest"
scout_compute_api_BatchComputeUnitsRequest.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BinaryArithmeticOperation(ConjureEnumType):

    ATAN2 = 'ATAN2'
    '''ATAN2'''
    UNKNOWN = 'UNKNOWN'
    '''UNKNOWN'''

    def __reduce_ex__(self, proto):
        return self.__class__, (self.name,)


scout_compute_api_BinaryArithmeticOperation.__name__ = "BinaryArithmeticOperation"
scout_compute_api_BinaryArithmeticOperation.__qualname__ = "BinaryArithmeticOperation"
scout_compute_api_BinaryArithmeticOperation.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BinaryArithmeticSeries(ConjureBeanType):
    """
    Applies a point-wise transformation to a pair of series.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'input1': ConjureFieldDefinition('input1', scout_compute_api_NumericSeries),
            'input2': ConjureFieldDefinition('input2', scout_compute_api_NumericSeries),
            'operation': ConjureFieldDefinition('operation', scout_compute_api_BinaryArithmeticOperation),
            'interpolation_configuration': ConjureFieldDefinition('interpolationConfiguration', OptionalTypeWrapper[scout_compute_api_InterpolationConfiguration])
        }

    __slots__: List[str] = ['_input1', '_input2', '_operation', '_interpolation_configuration']

    def __init__(self, input1: "scout_compute_api_NumericSeries", input2: "scout_compute_api_NumericSeries", operation: "scout_compute_api_BinaryArithmeticOperation", interpolation_configuration: Optional["scout_compute_api_InterpolationConfiguration"] = None) -> None:
        self._input1 = input1
        self._input2 = input2
        self._operation = operation
        self._interpolation_configuration = interpolation_configuration

    @builtins.property
    def input1(self) -> "scout_compute_api_NumericSeries":
        return self._input1

    @builtins.property
    def input2(self) -> "scout_compute_api_NumericSeries":
        return self._input2

    @builtins.property
    def operation(self) -> "scout_compute_api_BinaryArithmeticOperation":
        return self._operation

    @builtins.property
    def interpolation_configuration(self) -> Optional["scout_compute_api_InterpolationConfiguration"]:
        """
        Defaults to forward fill interpolation with a 1s interpolation radius
        """
        return self._interpolation_configuration


scout_compute_api_BinaryArithmeticSeries.__name__ = "BinaryArithmeticSeries"
scout_compute_api_BinaryArithmeticSeries.__qualname__ = "BinaryArithmeticSeries"
scout_compute_api_BinaryArithmeticSeries.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BitAndFunction(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'operand': ConjureFieldDefinition('operand', int)
        }

    __slots__: List[str] = ['_operand']

    def __init__(self, operand: int) -> None:
        self._operand = operand

    @builtins.property
    def operand(self) -> int:
        return self._operand


scout_compute_api_BitAndFunction.__name__ = "BitAndFunction"
scout_compute_api_BitAndFunction.__qualname__ = "BitAndFunction"
scout_compute_api_BitAndFunction.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BitOperationFunction(ConjureUnionType):
    _and_: Optional["scout_compute_api_BitAndFunction"] = None
    _or_: Optional["scout_compute_api_BitOrFunction"] = None
    _xor: Optional["scout_compute_api_BitXorFunction"] = None
    _bit_test: Optional["scout_compute_api_BitTestFunction"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'and_': ConjureFieldDefinition('and', scout_compute_api_BitAndFunction),
            'or_': ConjureFieldDefinition('or', scout_compute_api_BitOrFunction),
            'xor': ConjureFieldDefinition('xor', scout_compute_api_BitXorFunction),
            'bit_test': ConjureFieldDefinition('bitTest', scout_compute_api_BitTestFunction)
        }

    def __init__(
            self,
            and_: Optional["scout_compute_api_BitAndFunction"] = None,
            or_: Optional["scout_compute_api_BitOrFunction"] = None,
            xor: Optional["scout_compute_api_BitXorFunction"] = None,
            bit_test: Optional["scout_compute_api_BitTestFunction"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (and_ is not None) + (or_ is not None) + (xor is not None) + (bit_test is not None) != 1:
                raise ValueError('a union must contain a single member')

            if and_ is not None:
                self._and_ = and_
                self._type = 'and'
            if or_ is not None:
                self._or_ = or_
                self._type = 'or'
            if xor is not None:
                self._xor = xor
                self._type = 'xor'
            if bit_test is not None:
                self._bit_test = bit_test
                self._type = 'bitTest'

        elif type_of_union == 'and':
            if and_ is None:
                raise ValueError('a union value must not be None')
            self._and_ = and_
            self._type = 'and'
        elif type_of_union == 'or':
            if or_ is None:
                raise ValueError('a union value must not be None')
            self._or_ = or_
            self._type = 'or'
        elif type_of_union == 'xor':
            if xor is None:
                raise ValueError('a union value must not be None')
            self._xor = xor
            self._type = 'xor'
        elif type_of_union == 'bitTest':
            if bit_test is None:
                raise ValueError('a union value must not be None')
            self._bit_test = bit_test
            self._type = 'bitTest'

    @builtins.property
    def and_(self) -> Optional["scout_compute_api_BitAndFunction"]:
        return self._and_

    @builtins.property
    def or_(self) -> Optional["scout_compute_api_BitOrFunction"]:
        return self._or_

    @builtins.property
    def xor(self) -> Optional["scout_compute_api_BitXorFunction"]:
        return self._xor

    @builtins.property
    def bit_test(self) -> Optional["scout_compute_api_BitTestFunction"]:
        return self._bit_test

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_compute_api_BitOperationFunctionVisitor):
            raise ValueError('{} is not an instance of scout_compute_api_BitOperationFunctionVisitor'.format(visitor.__class__.__name__))
        if self._type == 'and' and self.and_ is not None:
            return visitor._and(self.and_)
        if self._type == 'or' and self.or_ is not None:
            return visitor._or(self.or_)
        if self._type == 'xor' and self.xor is not None:
            return visitor._xor(self.xor)
        if self._type == 'bitTest' and self.bit_test is not None:
            return visitor._bit_test(self.bit_test)


scout_compute_api_BitOperationFunction.__name__ = "BitOperationFunction"
scout_compute_api_BitOperationFunction.__qualname__ = "BitOperationFunction"
scout_compute_api_BitOperationFunction.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BitOperationFunctionVisitor:

    @abstractmethod
    def _and(self, and_: "scout_compute_api_BitAndFunction") -> Any:
        pass

    @abstractmethod
    def _or(self, or_: "scout_compute_api_BitOrFunction") -> Any:
        pass

    @abstractmethod
    def _xor(self, xor: "scout_compute_api_BitXorFunction") -> Any:
        pass

    @abstractmethod
    def _bit_test(self, bit_test: "scout_compute_api_BitTestFunction") -> Any:
        pass


scout_compute_api_BitOperationFunctionVisitor.__name__ = "BitOperationFunctionVisitor"
scout_compute_api_BitOperationFunctionVisitor.__qualname__ = "BitOperationFunctionVisitor"
scout_compute_api_BitOperationFunctionVisitor.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BitOperationSeries(ConjureBeanType):
    """
    Casts input series values to long before applying the bitwise operation.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'input': ConjureFieldDefinition('input', scout_compute_api_NumericSeries),
            'function': ConjureFieldDefinition('function', scout_compute_api_BitOperationFunction)
        }

    __slots__: List[str] = ['_input', '_function']

    def __init__(self, function: "scout_compute_api_BitOperationFunction", input: "scout_compute_api_NumericSeries") -> None:
        self._input = input
        self._function = function

    @builtins.property
    def input(self) -> "scout_compute_api_NumericSeries":
        return self._input

    @builtins.property
    def function(self) -> "scout_compute_api_BitOperationFunction":
        return self._function


scout_compute_api_BitOperationSeries.__name__ = "BitOperationSeries"
scout_compute_api_BitOperationSeries.__qualname__ = "BitOperationSeries"
scout_compute_api_BitOperationSeries.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BitOrFunction(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'operand': ConjureFieldDefinition('operand', int)
        }

    __slots__: List[str] = ['_operand']

    def __init__(self, operand: int) -> None:
        self._operand = operand

    @builtins.property
    def operand(self) -> int:
        return self._operand


scout_compute_api_BitOrFunction.__name__ = "BitOrFunction"
scout_compute_api_BitOrFunction.__qualname__ = "BitOrFunction"
scout_compute_api_BitOrFunction.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BitTestFunction(ConjureBeanType):
    """
    Returns the bit at the specified index, where the right-most bit has index 0.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'index': ConjureFieldDefinition('index', int)
        }

    __slots__: List[str] = ['_index']

    def __init__(self, index: int) -> None:
        self._index = index

    @builtins.property
    def index(self) -> int:
        return self._index


scout_compute_api_BitTestFunction.__name__ = "BitTestFunction"
scout_compute_api_BitTestFunction.__qualname__ = "BitTestFunction"
scout_compute_api_BitTestFunction.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BitXorFunction(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'operand': ConjureFieldDefinition('operand', int)
        }

    __slots__: List[str] = ['_operand']

    def __init__(self, operand: int) -> None:
        self._operand = operand

    @builtins.property
    def operand(self) -> int:
        return self._operand


scout_compute_api_BitXorFunction.__name__ = "BitXorFunction"
scout_compute_api_BitXorFunction.__qualname__ = "BitXorFunction"
scout_compute_api_BitXorFunction.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BucketedCartesianPlot(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'buckets': ConjureFieldDefinition('buckets', List[scout_compute_api_CartesianBucket])
        }

    __slots__: List[str] = ['_buckets']

    def __init__(self, buckets: List["scout_compute_api_CartesianBucket"]) -> None:
        self._buckets = buckets

    @builtins.property
    def buckets(self) -> List["scout_compute_api_CartesianBucket"]:
        return self._buckets


scout_compute_api_BucketedCartesianPlot.__name__ = "BucketedCartesianPlot"
scout_compute_api_BucketedCartesianPlot.__qualname__ = "BucketedCartesianPlot"
scout_compute_api_BucketedCartesianPlot.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BucketedEnumPlot(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'timestamps': ConjureFieldDefinition('timestamps', List[api_Timestamp]),
            'buckets': ConjureFieldDefinition('buckets', List[scout_compute_api_EnumBucket]),
            'categories': ConjureFieldDefinition('categories', List[str])
        }

    __slots__: List[str] = ['_timestamps', '_buckets', '_categories']

    def __init__(self, buckets: List["scout_compute_api_EnumBucket"], categories: List[str], timestamps: List["api_Timestamp"]) -> None:
        self._timestamps = timestamps
        self._buckets = buckets
        self._categories = categories

    @builtins.property
    def timestamps(self) -> List["api_Timestamp"]:
        """
        The end of the bucket, exclusive.
        """
        return self._timestamps

    @builtins.property
    def buckets(self) -> List["scout_compute_api_EnumBucket"]:
        return self._buckets

    @builtins.property
    def categories(self) -> List[str]:
        return self._categories


scout_compute_api_BucketedEnumPlot.__name__ = "BucketedEnumPlot"
scout_compute_api_BucketedEnumPlot.__qualname__ = "BucketedEnumPlot"
scout_compute_api_BucketedEnumPlot.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BucketedGeoPlot(ConjureUnionType):
    _time_based: Optional["scout_compute_api_TimeBucketedGeoPlot"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'time_based': ConjureFieldDefinition('timeBased', scout_compute_api_TimeBucketedGeoPlot)
        }

    def __init__(
            self,
            time_based: Optional["scout_compute_api_TimeBucketedGeoPlot"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (time_based is not None) != 1:
                raise ValueError('a union must contain a single member')

            if time_based is not None:
                self._time_based = time_based
                self._type = 'timeBased'

        elif type_of_union == 'timeBased':
            if time_based is None:
                raise ValueError('a union value must not be None')
            self._time_based = time_based
            self._type = 'timeBased'

    @builtins.property
    def time_based(self) -> Optional["scout_compute_api_TimeBucketedGeoPlot"]:
        return self._time_based

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_compute_api_BucketedGeoPlotVisitor):
            raise ValueError('{} is not an instance of scout_compute_api_BucketedGeoPlotVisitor'.format(visitor.__class__.__name__))
        if self._type == 'timeBased' and self.time_based is not None:
            return visitor._time_based(self.time_based)


scout_compute_api_BucketedGeoPlot.__name__ = "BucketedGeoPlot"
scout_compute_api_BucketedGeoPlot.__qualname__ = "BucketedGeoPlot"
scout_compute_api_BucketedGeoPlot.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BucketedGeoPlotVisitor:

    @abstractmethod
    def _time_based(self, time_based: "scout_compute_api_TimeBucketedGeoPlot") -> Any:
        pass


scout_compute_api_BucketedGeoPlotVisitor.__name__ = "BucketedGeoPlotVisitor"
scout_compute_api_BucketedGeoPlotVisitor.__qualname__ = "BucketedGeoPlotVisitor"
scout_compute_api_BucketedGeoPlotVisitor.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_BucketedNumericPlot(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'timestamps': ConjureFieldDefinition('timestamps', List[api_Timestamp]),
            'buckets': ConjureFieldDefinition('buckets', List[scout_compute_api_NumericBucket])
        }

    __slots__: List[str] = ['_timestamps', '_buckets']

    def __init__(self, buckets: List["scout_compute_api_NumericBucket"], timestamps: List["api_Timestamp"]) -> None:
        self._timestamps = timestamps
        self._buckets = buckets

    @builtins.property
    def timestamps(self) -> List["api_Timestamp"]:
        """
        The end of the bucket, exclusive.
        """
        return self._timestamps

    @builtins.property
    def buckets(self) -> List["scout_compute_api_NumericBucket"]:
        return self._buckets


scout_compute_api_BucketedNumericPlot.__name__ = "BucketedNumericPlot"
scout_compute_api_BucketedNumericPlot.__qualname__ = "BucketedNumericPlot"
scout_compute_api_BucketedNumericPlot.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_Cartesian(ConjureUnionType):
    _scatter: Optional["scout_compute_api_Scatter"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'scatter': ConjureFieldDefinition('scatter', scout_compute_api_Scatter)
        }

    def __init__(
            self,
            scatter: Optional["scout_compute_api_Scatter"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (scatter is not None) != 1:
                raise ValueError('a union must contain a single member')

            if scatter is not None:
                self._scatter = scatter
                self._type = 'scatter'

        elif type_of_union == 'scatter':
            if scatter is None:
                raise ValueError('a union value must not be None')
            self._scatter = scatter
            self._type = 'scatter'

    @builtins.property
    def scatter(self) -> Optional["scout_compute_api_Scatter"]:
        return self._scatter

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_compute_api_CartesianVisitor):
            raise ValueError('{} is not an instance of scout_compute_api_CartesianVisitor'.format(visitor.__class__.__name__))
        if self._type == 'scatter' and self.scatter is not None:
            return visitor._scatter(self.scatter)


scout_compute_api_Cartesian.__name__ = "Cartesian"
scout_compute_api_Cartesian.__qualname__ = "Cartesian"
scout_compute_api_Cartesian.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_CartesianVisitor:

    @abstractmethod
    def _scatter(self, scatter: "scout_compute_api_Scatter") -> Any:
        pass


scout_compute_api_CartesianVisitor.__name__ = "CartesianVisitor"
scout_compute_api_CartesianVisitor.__qualname__ = "CartesianVisitor"
scout_compute_api_CartesianVisitor.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_CartesianBounds(ConjureBeanType):
    """
    Min/max bounds of an XY Cartesian plot, inclusive.
    """

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'min_x': ConjureFieldDefinition('minX', scout_compute_api_DoubleConstant),
            'max_x': ConjureFieldDefinition('maxX', scout_compute_api_DoubleConstant),
            'min_y': ConjureFieldDefinition('minY', scout_compute_api_DoubleConstant),
            'max_y': ConjureFieldDefinition('maxY', scout_compute_api_DoubleConstant)
        }

    __slots__: List[str] = ['_min_x', '_max_x', '_min_y', '_max_y']

    def __init__(self, max_x: "scout_compute_api_DoubleConstant", max_y: "scout_compute_api_DoubleConstant", min_x: "scout_compute_api_DoubleConstant", min_y: "scout_compute_api_DoubleConstant") -> None:
        self._min_x = min_x
        self._max_x = max_x
        self._min_y = min_y
        self._max_y = max_y

    @builtins.property
    def min_x(self) -> "scout_compute_api_DoubleConstant":
        return self._min_x

    @builtins.property
    def max_x(self) -> "scout_compute_api_DoubleConstant":
        return self._max_x

    @builtins.property
    def min_y(self) -> "scout_compute_api_DoubleConstant":
        return self._min_y

    @builtins.property
    def max_y(self) -> "scout_compute_api_DoubleConstant":
        return self._max_y


scout_compute_api_CartesianBounds.__name__ = "CartesianBounds"
scout_compute_api_CartesianBounds.__qualname__ = "CartesianBounds"
scout_compute_api_CartesianBounds.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_CartesianBucket(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'min_x': ConjureFieldDefinition('minX', float),
            'max_x': ConjureFieldDefinition('maxX', float),
            'min_y': ConjureFieldDefinition('minY', float),
            'max_y': ConjureFieldDefinition('maxY', float),
            'min_timestamp': ConjureFieldDefinition('minTimestamp', api_Timestamp),
            'max_timestamp': ConjureFieldDefinition('maxTimestamp', api_Timestamp),
            'count': ConjureFieldDefinition('count', int)
        }

    __slots__: List[str] = ['_min_x', '_max_x', '_min_y', '_max_y', '_min_timestamp', '_max_timestamp', '_count']

    def __init__(self, count: int, max_timestamp: "api_Timestamp", max_x: float, max_y: float, min_timestamp: "api_Timestamp", min_x: float, min_y: float) -> None:
        self._min_x = min_x
        self._max_x = max_x
        self._min_y = min_y
        self._max_y = max_y
        self._min_timestamp = min_timestamp
        self._max_timestamp = max_timestamp
        self._count = count

    @builtins.property
    def min_x(self) -> float:
        return self._min_x

    @builtins.property
    def max_x(self) -> float:
        return self._max_x

    @builtins.property
    def min_y(self) -> float:
        return self._min_y

    @builtins.property
    def max_y(self) -> float:
        return self._max_y

    @builtins.property
    def min_timestamp(self) -> "api_Timestamp":
        return self._min_timestamp

    @builtins.property
    def max_timestamp(self) -> "api_Timestamp":
        return self._max_timestamp

    @builtins.property
    def count(self) -> int:
        return self._count


scout_compute_api_CartesianBucket.__name__ = "CartesianBucket"
scout_compute_api_CartesianBucket.__qualname__ = "CartesianBucket"
scout_compute_api_CartesianBucket.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_CartesianPlot(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'timestamps': ConjureFieldDefinition('timestamps', List[api_Timestamp]),
            'x_values': ConjureFieldDefinition('xValues', List[float]),
            'y_values': ConjureFieldDefinition('yValues', List[float])
        }

    __slots__: List[str] = ['_timestamps', '_x_values', '_y_values']

    def __init__(self, timestamps: List["api_Timestamp"], x_values: List[float], y_values: List[float]) -> None:
        self._timestamps = timestamps
        self._x_values = x_values
        self._y_values = y_values

    @builtins.property
    def timestamps(self) -> List["api_Timestamp"]:
        return self._timestamps

    @builtins.property
    def x_values(self) -> List[float]:
        return self._x_values

    @builtins.property
    def y_values(self) -> List[float]:
        return self._y_values


scout_compute_api_CartesianPlot.__name__ = "CartesianPlot"
scout_compute_api_CartesianPlot.__qualname__ = "CartesianPlot"
scout_compute_api_CartesianPlot.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_CartesianUnitResult(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'x': ConjureFieldDefinition('x', scout_compute_api_UnitResult),
            'y': ConjureFieldDefinition('y', scout_compute_api_UnitResult)
        }

    __slots__: List[str] = ['_x', '_y']

    def __init__(self, x: "scout_compute_api_UnitResult", y: "scout_compute_api_UnitResult") -> None:
        self._x = x
        self._y = y

    @builtins.property
    def x(self) -> "scout_compute_api_UnitResult":
        return self._x

    @builtins.property
    def y(self) -> "scout_compute_api_UnitResult":
        return self._y


scout_compute_api_CartesianUnitResult.__name__ = "CartesianUnitResult"
scout_compute_api_CartesianUnitResult.__qualname__ = "CartesianUnitResult"
scout_compute_api_CartesianUnitResult.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_ChannelSeries(ConjureUnionType):
    _data_source: Optional["scout_compute_api_DataSourceChannel"] = None
    _asset: Optional["scout_compute_api_AssetChannel"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'data_source': ConjureFieldDefinition('dataSource', scout_compute_api_DataSourceChannel),
            'asset': ConjureFieldDefinition('asset', scout_compute_api_AssetChannel)
        }

    def __init__(
            self,
            data_source: Optional["scout_compute_api_DataSourceChannel"] = None,
            asset: Optional["scout_compute_api_AssetChannel"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (data_source is not None) + (asset is not None) != 1:
                raise ValueError('a union must contain a single member')

            if data_source is not None:
                self._data_source = data_source
                self._type = 'dataSource'
            if asset is not None:
                self._asset = asset
                self._type = 'asset'

        elif type_of_union == 'dataSource':
            if data_source is None:
                raise ValueError('a union value must not be None')
            self._data_source = data_source
            self._type = 'dataSource'
        elif type_of_union == 'asset':
            if asset is None:
                raise ValueError('a union value must not be None')
            self._asset = asset
            self._type = 'asset'

    @builtins.property
    def data_source(self) -> Optional["scout_compute_api_DataSourceChannel"]:
        return self._data_source

    @builtins.property
    def asset(self) -> Optional["scout_compute_api_AssetChannel"]:
        return self._asset

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_compute_api_ChannelSeriesVisitor):
            raise ValueError('{} is not an instance of scout_compute_api_ChannelSeriesVisitor'.format(visitor.__class__.__name__))
        if self._type == 'dataSource' and self.data_source is not None:
            return visitor._data_source(self.data_source)
        if self._type == 'asset' and self.asset is not None:
            return visitor._asset(self.asset)


scout_compute_api_ChannelSeries.__name__ = "ChannelSeries"
scout_compute_api_ChannelSeries.__qualname__ = "ChannelSeries"
scout_compute_api_ChannelSeries.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_ChannelSeriesVisitor:

    @abstractmethod
    def _data_source(self, data_source: "scout_compute_api_DataSourceChannel") -> Any:
        pass

    @abstractmethod
    def _asset(self, asset: "scout_compute_api_AssetChannel") -> Any:
        pass


scout_compute_api_ChannelSeriesVisitor.__name__ = "ChannelSeriesVisitor"
scout_compute_api_ChannelSeriesVisitor.__qualname__ = "ChannelSeriesVisitor"
scout_compute_api_ChannelSeriesVisitor.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_CompactEnumPoint(ConjureBeanType):

    @builtins.classmethod
    def _fields(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'timestamp': ConjureFieldDefinition('timestamp', api_Timestamp),
            'value': ConjureFieldDefinition('value', int)
        }

    __slots__: List[str] = ['_timestamp', '_value']

    def __init__(self, timestamp: "api_Timestamp", value: int) -> None:
        self._timestamp = timestamp
        self._value = value

    @builtins.property
    def timestamp(self) -> "api_Timestamp":
        return self._timestamp

    @builtins.property
    def value(self) -> int:
        return self._value


scout_compute_api_CompactEnumPoint.__name__ = "CompactEnumPoint"
scout_compute_api_CompactEnumPoint.__qualname__ = "CompactEnumPoint"
scout_compute_api_CompactEnumPoint.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_ComputableNode(ConjureUnionType):
    _ranges: Optional["scout_compute_api_SummarizeRanges"] = None
    _series: Optional["scout_compute_api_SummarizeSeries"] = None
    _value: Optional["scout_compute_api_SelectValue"] = None
    _cartesian: Optional["scout_compute_api_SummarizeCartesian"] = None
    _frequency: Optional["scout_compute_api_FrequencyDomain"] = None
    _histogram: Optional["scout_compute_api_Histogram"] = None
    _geo: Optional["scout_compute_api_SummarizeGeo"] = None

    @builtins.classmethod
    def _options(cls) -> Dict[str, ConjureFieldDefinition]:
        return {
            'ranges': ConjureFieldDefinition('ranges', scout_compute_api_SummarizeRanges),
            'series': ConjureFieldDefinition('series', scout_compute_api_SummarizeSeries),
            'value': ConjureFieldDefinition('value', scout_compute_api_SelectValue),
            'cartesian': ConjureFieldDefinition('cartesian', scout_compute_api_SummarizeCartesian),
            'frequency': ConjureFieldDefinition('frequency', scout_compute_api_FrequencyDomain),
            'histogram': ConjureFieldDefinition('histogram', scout_compute_api_Histogram),
            'geo': ConjureFieldDefinition('geo', scout_compute_api_SummarizeGeo)
        }

    def __init__(
            self,
            ranges: Optional["scout_compute_api_SummarizeRanges"] = None,
            series: Optional["scout_compute_api_SummarizeSeries"] = None,
            value: Optional["scout_compute_api_SelectValue"] = None,
            cartesian: Optional["scout_compute_api_SummarizeCartesian"] = None,
            frequency: Optional["scout_compute_api_FrequencyDomain"] = None,
            histogram: Optional["scout_compute_api_Histogram"] = None,
            geo: Optional["scout_compute_api_SummarizeGeo"] = None,
            type_of_union: Optional[str] = None
            ) -> None:
        if type_of_union is None:
            if (ranges is not None) + (series is not None) + (value is not None) + (cartesian is not None) + (frequency is not None) + (histogram is not None) + (geo is not None) != 1:
                raise ValueError('a union must contain a single member')

            if ranges is not None:
                self._ranges = ranges
                self._type = 'ranges'
            if series is not None:
                self._series = series
                self._type = 'series'
            if value is not None:
                self._value = value
                self._type = 'value'
            if cartesian is not None:
                self._cartesian = cartesian
                self._type = 'cartesian'
            if frequency is not None:
                self._frequency = frequency
                self._type = 'frequency'
            if histogram is not None:
                self._histogram = histogram
                self._type = 'histogram'
            if geo is not None:
                self._geo = geo
                self._type = 'geo'

        elif type_of_union == 'ranges':
            if ranges is None:
                raise ValueError('a union value must not be None')
            self._ranges = ranges
            self._type = 'ranges'
        elif type_of_union == 'series':
            if series is None:
                raise ValueError('a union value must not be None')
            self._series = series
            self._type = 'series'
        elif type_of_union == 'value':
            if value is None:
                raise ValueError('a union value must not be None')
            self._value = value
            self._type = 'value'
        elif type_of_union == 'cartesian':
            if cartesian is None:
                raise ValueError('a union value must not be None')
            self._cartesian = cartesian
            self._type = 'cartesian'
        elif type_of_union == 'frequency':
            if frequency is None:
                raise ValueError('a union value must not be None')
            self._frequency = frequency
            self._type = 'frequency'
        elif type_of_union == 'histogram':
            if histogram is None:
                raise ValueError('a union value must not be None')
            self._histogram = histogram
            self._type = 'histogram'
        elif type_of_union == 'geo':
            if geo is None:
                raise ValueError('a union value must not be None')
            self._geo = geo
            self._type = 'geo'

    @builtins.property
    def ranges(self) -> Optional["scout_compute_api_SummarizeRanges"]:
        return self._ranges

    @builtins.property
    def series(self) -> Optional["scout_compute_api_SummarizeSeries"]:
        return self._series

    @builtins.property
    def value(self) -> Optional["scout_compute_api_SelectValue"]:
        return self._value

    @builtins.property
    def cartesian(self) -> Optional["scout_compute_api_SummarizeCartesian"]:
        return self._cartesian

    @builtins.property
    def frequency(self) -> Optional["scout_compute_api_FrequencyDomain"]:
        return self._frequency

    @builtins.property
    def histogram(self) -> Optional["scout_compute_api_Histogram"]:
        return self._histogram

    @builtins.property
    def geo(self) -> Optional["scout_compute_api_SummarizeGeo"]:
        return self._geo

    def accept(self, visitor) -> Any:
        if not isinstance(visitor, scout_compute_api_ComputableNodeVisitor):
            raise ValueError('{} is not an instance of scout_compute_api_ComputableNodeVisitor'.format(visitor.__class__.__name__))
        if self._type == 'ranges' and self.ranges is not None:
            return visitor._ranges(self.ranges)
        if self._type == 'series' and self.series is not None:
            return visitor._series(self.series)
        if self._type == 'value' and self.value is not None:
            return visitor._value(self.value)
        if self._type == 'cartesian' and self.cartesian is not None:
            return visitor._cartesian(self.cartesian)
        if self._type == 'frequency' and self.frequency is not None:
            return visitor._frequency(self.frequency)
        if self._type == 'histogram' and self.histogram is not None:
            return visitor._histogram(self.histogram)
        if self._type == 'geo' and self.geo is not None:
            return visitor._geo(self.geo)


scout_compute_api_ComputableNode.__name__ = "ComputableNode"
scout_compute_api_ComputableNode.__qualname__ = "ComputableNode"
scout_compute_api_ComputableNode.__module__ = "nominal_api.scout_compute_api"


class scout_compute_api_ComputableNodeVisitor:

    @abstractmethod
    def _ranges(self, ranges: "scout_compute_api_SummarizeRanges") -> Any:
        pass

    @abstractmethod
    def _series(self, series: "scout_compute_api_SummarizeSeries") -> Any:
        pass

    @abstractmethod
    def _value(self, value: "scout_compute_api_SelectValue") -> Any:
        pass

    @abstractmethod
    def _cartesian(self, cartesian: "scout_compute_api_SummarizeCartesian") -> Any:
        pass

    @abstractmethod
    def _frequency(self, frequency: "scout_compute_api_FrequencyDomain") -> Any:
        pass

    @abstractmethod
    def _histogram(self, histogram: "scout_compute_api_Histogram") -> Any:
        pass

    @abstractmethod
    def _geo(self, geo: "scout_compute_api_SummarizeGeo") -> Any:
        pass


scout_compute_api_ComputableNodeVisitor.__name__ = "ComputableNodeVisitor"
scout_compute_api_ComputableNodeVisitor.__qualname__ = "ComputableNodeVisitor"
scout_compute_api_ComputableNodeVisitor.__m