"""
Tests for University of Ottawa course data models.

This module tests the Pydantic models for courses, subjects,
prerequisites, and components.
"""

import pytest
from pydantic import ValidationError

from uoapi.course.models import Subject, Course, Prerequisite, Component


class TestSubjectModel:
    """Test Subject model validation and functionality."""

    def test_valid_subject_creation(self):
        """Test creating a valid Subject instance."""
        subject = Subject(
            subject="Computer Science",
            subject_code="CSI",
            link="https://catalogue.uottawa.ca/en/courses/csi/",
        )

        assert subject.subject == "Computer Science"
        assert subject.subject_code == "CSI"
        assert str(subject.link) == "https://catalogue.uottawa.ca/en/courses/csi/"

    def test_subject_with_invalid_url(self):
        """Test Subject validation with invalid URL."""
        with pytest.raises(ValidationError):
            Subject(subject="Computer Science", subject_code="CSI", link="not-a-valid-url")

    def test_subject_required_fields(self):
        """Test that Subject requires all fields."""
        with pytest.raises(ValidationError):
            Subject(subject="Computer Science")  # Missing required fields

    def test_subject_serialization(self):
        """Test Subject serialization to dict."""
        subject = Subject(
            subject="Computer Science",
            subject_code="CSI",
            link="https://catalogue.uottawa.ca/en/courses/csi/",
        )

        data = subject.dict()
        assert data["subject"] == "Computer Science"
        assert data["subject_code"] == "CSI"
        assert data["link"] == "https://catalogue.uottawa.ca/en/courses/csi/"


class TestCourseModel:
    """Test Course model validation and functionality."""

    def test_valid_course_creation(self):
        """Test creating a valid Course instance."""
        course = Course(
            course_code="CSI3140",
            title="World Wide Web Programming",
            credits=3,
            description="Introduction to web programming concepts...",
            components=["LECTURE", "LAB"],
            prerequisites="CSI2520, CSI2101",
            dependencies=[["CSI2520"], ["CSI2101"]],
        )

        assert course.course_code == "CSI3140"
        assert course.title == "World Wide Web Programming"
        assert course.credits == 3
        assert len(course.components) == 2
        assert "LECTURE" in course.components

    def test_course_with_minimal_data(self):
        """Test Course with only required fields."""
        course = Course(
            course_code="CSI1100",
            title="Introduction to Computing",
            credits=3,
            description="Basic computing concepts",
        )

        assert course.course_code == "CSI1100"
        assert course.components == []  # Default empty list
        assert course.prerequisites == ""  # Default empty string
        assert course.dependencies == []  # Default empty list

    def test_course_credits_validation(self):
        """Test Course credits validation (must be >= 0)."""
        # Valid credits
        course = Course(course_code="CSI1100", title="Test Course", credits=0, description="Test")
        assert course.credits == 0

        # Invalid negative credits
        with pytest.raises(ValidationError):
            Course(course_code="CSI1100", title="Test Course", credits=-1, description="Test")

    def test_course_required_fields(self):
        """Test that Course requires all mandatory fields."""
        with pytest.raises(ValidationError):
            Course(course_code="CSI3140")  # Missing required fields

    def test_course_dependencies_structure(self):
        """Test Course dependencies structure validation."""
        course = Course(
            course_code="CSI3140",
            title="Test Course",
            credits=3,
            description="Test description",
            dependencies=[
                ["CSI2520"],  # Required course
                ["CSI2101", "CSI2110"],  # Alternative courses (OR)
            ],
        )

        assert len(course.dependencies) == 2
        assert len(course.dependencies[1]) == 2  # OR group

    def test_course_serialization(self):
        """Test Course serialization to dict."""
        course = Course(
            course_code="CSI3140",
            title="World Wide Web Programming",
            credits=3,
            description="Web programming course",
            components=["LECTURE"],
            prerequisites="CSI2520",
            dependencies=[["CSI2520"]],
        )

        data = course.dict()
        assert data["course_code"] == "CSI3140"
        assert data["credits"] == 3
        assert isinstance(data["components"], list)
        assert isinstance(data["dependencies"], list)


class TestPrerequisiteModel:
    """Test Prerequisite model validation and parsing."""

    def test_prerequisite_creation(self):
        """Test creating a Prerequisite instance."""
        prereq = Prerequisite(content="Prerequisite: CSI2520, CSI2101")
        assert prereq.content == "Prerequisite: CSI2520, CSI2101"

    def test_prerequisite_try_parse_success(self):
        """Test successful prerequisite parsing."""
        # English prerequisite
        prereq = Prerequisite.try_parse("Prerequisite: CSI2520 and CSI2101")
        assert prereq is not None
        assert "Prerequisite" in prereq.content

        # French prerequisite
        prereq_fr = Prerequisite.try_parse("Préalable: CSI2520 et CSI2101")
        assert prereq_fr is not None
        assert "Préalable" in prereq_fr.content

    def test_prerequisite_try_parse_failure(self):
        """Test prerequisite parsing failure."""
        # Should return None for non-prerequisite text
        result = Prerequisite.try_parse("This is just regular course content")
        assert result is None

        result = Prerequisite.try_parse("Course Component: Lecture")
        assert result is None

    def test_prerequisite_try_parse_edge_cases(self):
        """Test prerequisite parsing edge cases."""
        # Empty string
        result = Prerequisite.try_parse("")
        assert result is None

        # Case sensitivity (current implementation is case-sensitive)
        result = Prerequisite.try_parse("prerequisite: CSI2520")
        assert result is None  # Current implementation requires exact case

        # Partial match (requires exact "Prerequisite" or "Préalable")
        result = Prerequisite.try_parse("The Prerequisite is CSI2520")
        assert result is not None


class TestComponentModel:
    """Test Component model validation and parsing."""

    def test_component_creation(self):
        """Test creating a Component instance."""
        component = Component(content="Course Component: Lecture")
        assert component.content == "Course Component: Lecture"

    def test_component_try_parse_success(self):
        """Test successful component parsing."""
        # English component
        comp = Component.try_parse("Course Component: Lecture")
        assert comp is not None
        assert "Course Component" in comp.content

        # French component
        comp_fr = Component.try_parse("Volet : Cours magistral")
        assert comp_fr is not None
        assert "Volet" in comp_fr.content

    def test_component_try_parse_failure(self):
        """Test component parsing failure."""
        # Should return None for non-component text
        result = Component.try_parse("This is course description content")
        assert result is None

        result = Component.try_parse("Prerequisite: CSI2520")
        assert result is None

    def test_component_try_parse_edge_cases(self):
        """Test component parsing edge cases."""
        # Empty string
        result = Component.try_parse("")
        assert result is None

        # Case sensitivity (current implementation is case-sensitive)
        result = Component.try_parse("course component: Laboratory")
        assert result is None  # Current implementation requires exact case

    def test_component_types(self):
        """Test parsing different component types."""
        components = [
            "Course Component: Lecture",
            "Course Component: Laboratory",
            "Course Component: Tutorial",
            "Course Component: Seminar",
        ]

        for comp_text in components:
            comp = Component.try_parse(comp_text)
            assert comp is not None
            assert comp.content == comp_text


class TestModelIntegration:
    """Test integration between different models."""

    def test_course_with_parsed_prerequisites_and_components(self):
        """Test Course creation with parsed prerequisites and components."""
        # Simulate parsing prerequisites and components
        prereq_text = "Prerequisite: CSI2520, CSI2101"
        comp_text = "Course Component: Lecture"

        prereq = Prerequisite.try_parse(prereq_text)
        comp = Component.try_parse(comp_text)

        assert prereq is not None
        assert comp is not None

        # Create course with parsed data
        course = Course(
            course_code="CSI3140",
            title="Web Programming",
            credits=3,
            description="Web development course",
            prerequisites=prereq.content,
            components=["LECTURE"],  # Extracted from component
        )

        assert course.prerequisites == prereq_text
        assert "LECTURE" in course.components

    def test_subject_course_relationship(self):
        """Test relationship between Subject and Course models."""
        subject = Subject(
            subject="Computer Science",
            subject_code="CSI",
            link="https://catalogue.uottawa.ca/en/courses/csi/",
        )

        course = Course(
            course_code="CSI3140",
            title="Web Programming",
            credits=3,
            description="Course from " + subject.subject + " department",
        )

        # Verify course code starts with subject code
        assert course.course_code.startswith(subject.subject_code)
        assert subject.subject in course.description

    def test_model_json_compatibility(self):
        """Test that models can be serialized to JSON-compatible format."""
        import json

        # Test Subject
        subject = Subject(
            subject="Computer Science",
            subject_code="CSI",
            link="https://catalogue.uottawa.ca/en/courses/csi/",
        )
        subject_json = json.dumps(subject.dict())
        assert "Computer Science" in subject_json

        # Test Course
        course = Course(
            course_code="CSI3140", title="Web Programming", credits=3, description="Test course"
        )
        course_json = json.dumps(course.dict())
        assert "CSI3140" in course_json

    @pytest.mark.parametrize(
        "course_code,expected_valid",
        [
            ("CSI3140", True),
            ("MAT1320", True),
            ("PHY1321", True),
            ("", False),  # Empty string
            ("123", False),  # Numbers only
            ("TOOLONG1234", True),  # Long but valid
        ],
    )
    def test_course_code_patterns(self, course_code, expected_valid):
        """Test various course code patterns."""
        if expected_valid:
            # Should create successfully
            course = Course(
                course_code=course_code,
                title="Test Course",
                credits=3,
                description="Test description",
            )
            assert course.course_code == course_code
        else:
            # Should either raise ValidationError or create successfully
            # (Pydantic doesn't validate course code format by default)
            course = Course(
                course_code=course_code,
                title="Test Course",
                credits=3,
                description="Test description",
            )
            # This test shows that basic validation passes
            # For stricter validation, we'd need custom validators


if __name__ == "__main__":
    pytest.main([__file__])
