"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HyperledgerFabricNode = exports.STARTER_INSTANCE_TYPES = exports.InstanceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const cdk = require("aws-cdk-lib");
const managedblockchain = require("aws-cdk-lib/aws-managedblockchain");
const customresources = require("aws-cdk-lib/custom-resources");
const constructs = require("constructs");
const network = require("./network");
const utilities = require("./utilities");
/**
 * Supported instance types for Managed Blockchain nodes
 */
var InstanceType;
(function (InstanceType) {
    InstanceType["BURSTABLE3_SMALL"] = "bc.t3.small";
    InstanceType["BURSTABLE3_MEDIUM"] = "bc.t3.medium";
    InstanceType["BURSTABLE3_LARGE"] = "bc.t3.large";
    InstanceType["BURSTABLE3_XLARGE"] = "bc.t3.xlarge";
    InstanceType["STANDARD5_LARGE"] = "bc.m5.large";
    InstanceType["STANDARD5_XLARGE"] = "bc.m5.xlarge";
    InstanceType["STANDARD5_XLARGE2"] = "bc.m5.2xlarge";
    InstanceType["STANDARD5_XLARGE4"] = "bc.m5.4xlarge";
    InstanceType["COMPUTE5_LARGE"] = "bc.c5.large";
    InstanceType["COMPUTE5_XLARGE"] = "bc.c5.xlarge";
    InstanceType["COMPUTE5_XLARGE2"] = "bc.c5.2xlarge";
    InstanceType["COMPUTE5_XLARGE4"] = "bc.c5.4xlarge";
})(InstanceType = exports.InstanceType || (exports.InstanceType = {}));
/**
 * Valid instance types for starter networks
 */
exports.STARTER_INSTANCE_TYPES = [
    InstanceType.BURSTABLE3_SMALL,
    InstanceType.BURSTABLE3_MEDIUM,
];
/**
 * Creates a Hyperledger Fabric node on an Amazon Managed Blockchain network
 */
class HyperledgerFabricNode extends constructs.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // These cannot be readonly since they have to be set after construction
        // due the race condition documented in https://github.com/aws/aws-cdk/issues/18237.
        this.endpoint = '';
        this.eventEndpoint = '';
        // Collect metadata on the stack
        const region = cdk.Stack.of(this).region;
        // Populate instance variables from input properties, using defaults if values not provided
        if (typeof props === 'undefined')
            props = {};
        this.availabilityZone = props.availabilityZone ?? `${region}a`;
        this.instanceType = props.instanceType ?? InstanceType.BURSTABLE3_SMALL;
        this.enableChaincodeLogging = props.enableChaincodeLogging ?? true;
        this.enableNodeLogging = props.enableNodeLogging ?? true;
        this.networkId = scope.networkId;
        this.memberId = scope.memberId;
        // Ensure the parameters captured above are valid, so we don't
        // need to wait until deployment time to discover an error
        utilities.validateRegion(region);
        utilities.validateAvailabilityZone(region, this.availabilityZone);
        if (scope.networkEdition === network.NetworkEdition.STARTER && !exports.STARTER_INSTANCE_TYPES.includes(this.instanceType)) {
            const starterInstanceTypeList = exports.STARTER_INSTANCE_TYPES.join(', ');
            throw new Error(`Instance type in a starter network must be one of the following: ${starterInstanceTypeList}.`);
        }
        // Build out the Cloudformation construct for the network/member
        const node = new managedblockchain.CfnNode(this, 'Node', {
            networkId: this.networkId,
            memberId: this.memberId,
            nodeConfiguration: {
                availabilityZone: this.availabilityZone,
                instanceType: this.instanceType,
            },
        });
        // Capture data included in the Cloudformation output in instance variables
        this.nodeId = node.getAtt('NodeId').toString();
    }
    /**
     * Build out a list of HyperledgerFabricNode constructs given a list of input property
     * objects; additionally checks to ensure node count is supported given the network type
     */
    static constructNodes(scope, nodeProps) {
        // If no node configurations are provided, create one; the empty object
        // will be populated with defaults when passed to the node constructor
        if (typeof nodeProps === 'undefined')
            nodeProps = [{}];
        const starter = scope.networkEdition === network.NetworkEdition.STARTER;
        if (starter && nodeProps.length > 2) {
            throw new Error('A starter network can have at most 2 nodes per member.');
        }
        if (!starter && nodeProps.length > 3) {
            throw new Error('A standard network can have at most 3 nodes per member.');
        }
        // Construct the node list, using an index value in the identifier
        return Array.from(nodeProps.entries()).map(e => new HyperledgerFabricNode(scope, `Node${e[0]}`, e[1]));
    }
    /**
     * Configure logging for the node via SDK call; this function
     * should be merged back into the constructor once the race condition is solved
     */
    configureLogging(sdkCallPolicy) {
        // This call doesn't really need all the permissions its using in the
        // provided policy, but since the policy must be constructed all at once
        // this is the only way to do it effectively
        const logPublishingConfiguration = {
            Fabric: {
                ChaincodeLogs: {
                    Cloudwatch: { Enabled: this.enableChaincodeLogging },
                },
                PeerLogs: {
                    Cloudwatch: { Enabled: this.enableNodeLogging },
                },
            },
        };
        const configureNodeLogSdkCall = {
            service: 'ManagedBlockchain',
            action: 'updateNode',
            parameters: {
                NetworkId: this.networkId,
                MemberId: this.memberId,
                NodeId: this.nodeId,
                LogPublishingConfiguration: logPublishingConfiguration,
            },
            physicalResourceId: customresources.PhysicalResourceId.of('Id'),
        };
        new customresources.AwsCustomResource(this, 'ConfigureNodeLogResource', {
            policy: sdkCallPolicy,
            onCreate: configureNodeLogSdkCall,
            onUpdate: configureNodeLogSdkCall,
        });
    }
    /**
     * Populate the output properties that must be fetched via SDK call; this function
     * should be merged back into the constructor once the race condition is solved
     */
    fetchData(dataSdkCallPolicy) {
        // This call doesn't really need all the permissions its using in the
        // provided policy, but since the policy must be constructed all at once
        // this is the only way to do it effectively
        const nodeDataSdkCall = {
            service: 'ManagedBlockchain',
            action: 'getNode',
            parameters: { NetworkId: this.networkId, MemberId: this.memberId, NodeId: this.nodeId },
            physicalResourceId: customresources.PhysicalResourceId.of('Id'),
        };
        const nodeData = new customresources.AwsCustomResource(this, 'NodeDataResource', {
            policy: dataSdkCallPolicy,
            onCreate: nodeDataSdkCall,
            onUpdate: nodeDataSdkCall,
        });
        // Grab items out of the above return values and stick them in output properties
        this.endpoint = nodeData.getResponseField('Node.FrameworkAttributes.Fabric.PeerEndpoint');
        this.eventEndpoint = nodeData.getResponseField('Node.FrameworkAttributes.Fabric.PeerEventEndpoint');
    }
}
exports.HyperledgerFabricNode = HyperledgerFabricNode;
_a = JSII_RTTI_SYMBOL_1;
HyperledgerFabricNode[_a] = { fqn: "@cdklabs/cdk-hyperledger-fabric-network.HyperledgerFabricNode", version: "0.8.562" };
//# sourceMappingURL=data:application/json;base64,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