#  Pyrogram - Telegram MTProto API Client Library for Python
#  Copyright (C) 2017-present Dan <https://github.com/delivrance>
#
#  This file is part of Pyrogram.
#
#  Pyrogram is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Lesser General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  Pyrogram is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public License
#  along with Pyrogram.  If not, see <http://www.gnu.org/licenses/>.

import logging

import pyrogram
from pyrogram import raw
from pyrogram import types
from pyrogram.errors import UserMigrate

log = logging.getLogger(__name__)


class SignInBot:
    async def sign_in_bot(
        self: "pyrogram.Client",
        bot_token: str
    ) -> "types.User":
        """Authorize a bot using its bot token generated by BotFather.

        .. include:: /_includes/usable-by/bots.rst

        Parameters:
            bot_token (``str``):
                The bot token generated by BotFather

        Returns:
            :obj:`~pyrogram.types.User`: On success, the bot identity is return in form of a user object.

        Raises:
            BadRequest: In case the bot token is invalid.
        """
        while True:
            try:
                r = await self.invoke(
                    raw.functions.auth.ImportBotAuthorization(
                        flags=0,
                        api_id=self.api_id,
                        api_hash=self.api_hash,
                        bot_auth_token=bot_token
                    )
                )
            except UserMigrate as e:
                dc_option = await self.get_dc_option(e.value, ipv6=self.ipv6)
                await self.session.stop()

                self.session = await self.get_session(
                    dc_id=e.value,
                    server_address=dc_option.ip_address,
                    port=dc_option.port,
                    export_authorization=False,
                    temporary=True
                )

                await self.storage.dc_id(e.value)
                await self.storage.server_address(dc_option.ip_address)
                await self.storage.port(dc_option.port)
                await self.storage.auth_key(self.session.auth_key)
            else:
                await self.storage.user_id(r.user.id)
                await self.storage.is_bot(True)

                return types.User._parse(self, r.user)
