# Library for using the COM-Interface of Zuken E3.series with python
# Important differences to common E3 COM functionality:
# 	-The first, empty element of lists is removed
# 	-As python does not support call-by-reference there are additional returns for [out] parameters
# This file was created for E3.series 26.11 Build 293676 (TLB version 26.01)
# 
# mypy: ignore-errors

import typing
from win32com.client import VARIANT
from win32com.client import CDispatch
import pythoncom
from .tools import _get_default_dbe, _get_default_app, _raw_connect_dbe, _raw_connect_app, _variant_to_dict, _dict_to_variant

DLLDEFAULTVALUE = "-353353"

# -------------------- IBundleInterface--------------------
class Bundle:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Bundle. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current bundle item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current bundle item identifier |
		| 0 | No bundle item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a bundle as the current item.

		:param id: Unique value identifying a bundle item. Since v2011-1010 conductor identifiers are also valid. The conductor's parent bundle will be set as the current item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current bundle item identifier |
		| 0 | No current bundle item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the bundle item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Bundle item name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the bundle item.

		:param name: Internal name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the bundle item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the bundle item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is "<Empty>", defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the bundle item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the bundle item's specified attribute value.

		:param name: Internal name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the bundle item.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the bundle item's specified attributes.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def GetPinCount(self) -> int:
		"""
		Gets the number of conductors contained in the current bundle item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors |
		| 0 | No conductors or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPinCount()

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductors contained in the current bundle item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of conductors passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors in ids |
		| 0 | No conductors or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSymbolCount(self) -> int:
		"""
		Gets the number of symbols representing the current bundle item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of cables, conductors and bundle items |
		| 0 | No cables, conductors and bundle items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSymbolCount()

	def GetSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbols representing the current bundle item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbols passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbols in ids |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsShield(self) -> int:
		"""
		Indicates if the bundle item is a shield.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Bundle item is a shield |
		| 0 | Bundle item is not a shield or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsShield()

	def GetBundleCount(self) -> int:
		"""
		Gets the number of bundle items contained in the current bundle item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of bundle items |
		| 0 | No bundle items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBundleCount()

	def GetBundleIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the bundle items contained in the current bundle item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of bundle items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of bundle items in ids |
		| 0 | No bundle items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetBundleIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetParentBundleId(self) -> int:
		"""
		Gets the  parent bundle item of the current bundle item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Parent bundle item identifier |
		| 0 | Bundle item has no parent bundle or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetParentBundleId()

	def GetRootBundleId(self, bndid:int) -> int:
		"""
		Gets the root  bundle item of a cable, conductor or bundle item.

		:param bndid: Cable, conductor or bundle item identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Root bundle item identifier |
		| 0 | bndid is the identifier of the root bundle item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetRootBundleId(bndid)

	def Create(self, aroundids:list[int]) -> int:
		"""
		Creates a new shield.

		:param aroundids: Array of cable, conductor and bundle identifiers to be contained in the new bundle passed into the function. If successful, an array of cable, conductor and bundle identifiers contained in the new bundle is passed out of the function. This parameter can also hold a single integer value
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new shield |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.Create(aroundids)
		return ret[0]

	def CreateIn(self, cableid:int) -> int:
		"""
		Creates an empty shield in an overbraid cable.

		:param cableid: Cable identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new shield |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateIn(cableid)

	def Capture(self, cabwirids:list[int]) -> int:
		"""
		Moves cables, conductors and bundle items from their current structural position to the current bundle item.

		:param cabwirids: Array of cable, conductor and bundle item identifiers passed into the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current bundle item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.Capture(cabwirids)
		return ret[0]

	def Delete(self) -> int:
		"""
		Deletes the bundle item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Value always returned |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def SetName(self, name:str) -> int:
		"""
		Sets the bundle item's name.

		:param name: New bundle item name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | New bundle item name |
		| "&lt;Empty&gt; | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def IsTwisted(self) -> int:
		"""
		Indicates if the bundle item is a twisted pair.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Bundle item is a twisted pair |
		| 0 | Bundle item is not a twisted pair or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsTwisted()

	def IsBundle(self) -> int:
		"""
		Indicates if  the bundle item is a bundle.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Bundle item is a bundle |
		| 0 | Bundle item is not a bundle or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsBundle()

	def CreateShield(self, aroundids:list[int]) -> int:
		"""
		Creates a new shield.

		:param aroundids: Array of cable, conductor and bundle identifiers to be contained in the new bundle passed into the function. If successful, an array of cable, conductor and bundle identifiers contained in the new bundle is passed out of the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new shield |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.CreateShield(aroundids)
		return ret[0]

	def CreateTwist(self, aroundids:list[int]) -> int:
		"""
		Creates a new twisted pair.

		:param aroundids: Array of cable, conductor and bundle identifiers to be contained in the new bundle passed into the function. If successful, an array of cable, conductor and bundle identifiers contained in the new bundle is passed out of the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new twisted pair |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.CreateTwist(aroundids)
		return ret[0]

	def CreateBundle(self, aroundids:list[int]) -> int:
		"""
		Creates a new bundle.

		:param aroundids: Array of cable, conductor and bundle identifiers to be contained in the new bundle passed into the function. If successful, an array of cable, conductor and bundle identifiers contained in the new bundle is passed out of the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new bundle |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.CreateBundle(aroundids)
		return ret[0]

	def CreateShieldIn(self, cableid:int) -> int:
		"""
		Creates an empty shield in an overbraid cable.

		:param cableid: Cable identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new shield |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateShieldIn(cableid)

	def CreateTwistIn(self, cableid:int) -> int:
		"""
		Creates an empty twisted pair in an overbraid cable.

		:param cableid: Cable identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new twisted pair |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateTwistIn(cableid)

	def CreateBundleIn(self, cableid:int) -> int:
		"""
		Creates an empty bundle in an overbraid cable.

		:param cableid: Cable identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new bundle |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateBundleIn(cableid)

	def PlaceAll(self) -> int:
		"""
		Automatically places the bundle item's symbols.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of placed symbols |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PlaceAll()

	def GetCableCount(self) -> int:
		"""
		Gets the number of cables contained in the current bundle item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of cables |
		| 0 | No cables or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCableCount()

	def GetCableIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the cables contained in the current bundle item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of cables passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of cables in ids |
		| 0 | No cables or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetCableIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAnyCount(self) -> int:
		"""
		Gets the number of cables, conductors and bundle items contained in the current bundle item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of cables, conductors and bundle items |
		| 0 | No cables, conductors and bundle items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAnyCount()

	def GetAnyIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the cables, conductors and bundle items contained in the current bundle item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of cables, conductors and bundle items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of cables, conductors and bundle items in ids |
		| 0 | No cables, conductors and bundle items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAnyIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetCoreCount(self) -> int:
		"""
		Gets the number of conductors contained in the current bundle item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors |
		| 0 | No conductors or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCoreCount()

	def GetCoreIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductors contained in the current bundle item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of conductors passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors in ids |
		| 0 | No conductors or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetCoreIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def ReleaseIDs(self, ids:list[int]) -> int:
		"""
		Moves cables, conductors and bundle items from the current bundle item.

		:param ids: Array of cable, conductor and bundle item identifiers passed into the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | All cable, conductor and bundle item were moved out of the current bundle item |
		| 0 | Not all cable, conductor and bundle item were moved out of the current bundle item or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		ret = self._obj.ReleaseIDs(ids)
		return ret[0]

	def GetOverbraidIdEx(self) -> int:
		"""
		Gets the identifier of the bundle item's overbraid cable.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the overbraid cable |
		| -1 | No project open or bundle item set |
		| -2 | Bundle item is not an overbraid cable element |
		| -3 | Bundle item is an invalid type |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.GetOverbraidIdEx()

	def GetRootOverbraidId(self) -> int:
		"""
		Gets the identifier of the bundle item's root overbraid cable.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the root overbraid cable |
		| -1 | No project open or bundle item set |
		| -2 | Bundle item is not in an overbraid cable element |
		| -3 | Bundle item is an invalid type |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.GetRootOverbraidId()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the bundle item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the bundle item |
		| "&lt;Empty&gt;" | No bundle item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a bundle as the current item.

		:param gid: Global identifier value of a bundle item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current bundle item |
		| "&lt;Empty&gt;" | No current bundle item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current bundle item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current bundle item |
		| "&lt;Empty&gt;" | No bundle item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a bundle as the current item.

		:param guid: Globally unique identifier value of a bundle item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current bundle item |
		| "&lt;Empty&gt;" | No current bundle item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- IAttributeInterface--------------------
class Attribute:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Attribute. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current attribute item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current attribute item identifier |
		| 0 | No attribute item or an error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets an attribute as the current item.

		:param id: Unique value identifying an attribute item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current attribute item identifier |
		| 0 | No current attribute item or an error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the attribute item's display name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's display name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.GetName()

	def GetInternalName(self) -> str:
		"""
		Gets the attribute item's user-defined internal name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's user-defined internal name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.GetInternalName()

	def GetValue(self) -> str:
		"""
		Gets the attribute item's value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's value |
		| "&lt;Empty&gt;" | Attribute item has no value or an error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.GetValue()

	def GetInternalValue(self) -> str:
		"""
		Gets the attribute item's internal value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's internal value |
		| "&lt;Empty&gt;" | Attribute item has no internal value or an error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.GetInternalValue()

	def SetValue(self, value:str) -> int:
		"""
		Sets the attribute item's value.

		:param value: Value to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier value of the attribute item, Value is applied |
		| 0 | Value could not be set |
		| -1 | value exceeds 252 characters in length, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.SetValue(value)

	def Delete(self) -> int:
		"""
		Deletes the current attribute from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes deleted |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.Delete()

	def GetTextIds(self, txttyp:int=0, search_string:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the text items displaying the attribute value.

		:param txttyp: Text type filter. Only text identifiers with this Text Type will be supplied. All text identifiers are supplied if txttyp is 0. The default value 0, defaults to 0
		:type int: int
		:param search_string: Text value filter, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of text identifiers displaying the attribute value passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No text identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		dummy=0
		ret, ids = self._obj.GetTextIds(dummy, txttyp, search_string)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetTextCount(self) -> int:
		"""
		Gets the number of  text items displaying the attribute value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items |
		| 0 | No text items or an error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.GetTextCount()

	def DisplayAttribute(self, id:int=0) -> int:
		"""
		Creates a text item to display the current attribute item's value.

		:param id: Identifier of the symbol or sheet to bind the text item to., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the created text item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.DisplayAttribute(id)

	def GetOwnerId(self) -> int:
		"""
		Gets a unique value identifying the attribute item's owner.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current attribute item's owner identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.GetOwnerId()

	def GetFormattedValue(self) -> str:
		"""
		Gets the  attribute item's formatted value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's formatted value |
		| "&lt;Empty&gt;" | Attribute item has no formatted value or an error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.GetFormattedValue()

	def DisplayValueAt(self, sheetid:int, x:float, y:float, bindid:int=0) -> int:
		"""
		Creates a text item to display the current attribute item's value.

		:param sheetid: Identifier of the sheet to display the text item on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param bindid: Identifier of the placed symbol item to bind the text item to., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the text item created |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 7.20
		"""
		return self._obj.DisplayValueAt(sheetid, x, y, bindid)

	def IsLockChangeable(self) -> bool:
		"""
		Indicates whether the attribute item's lock status can be changed.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Attribute lock status can be changed |
		| 0 | Attribute lock status cannot be changed or an error occurred |

		Available since E3.series TLB-Versions: 14.00
		"""
		return self._obj.IsLockChangeable()

	def GetLockChangeable(self) -> bool:
		"""
		Indicates whether the attribute item's value can be changed.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Attribute value can be changed |
		| 0 | Attribute value cannot be changed or an error occurred |

		Available since E3.series TLB-Versions: 14.00
		"""
		return self._obj.GetLockChangeable()

	def SetLockChangeable(self, lockchangeable:bool) -> int:
		"""
		Sets the attribute item's lock status.

		:param lockchangeable: Attribute's lock status value. If 1, sets the attribute lock status to locked. If 0, sets the attribute lock status to unlocked
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Attribute lock status changed |
		| 1 | No project open or no attribute item set |
		| 2 | Lock status can only be changed for locked attributes |
		| 4 | Invalid attribute owner |
		| 5 | Lock status cannot be changed due to the attribute definition or properties defined |

		Available since E3.series TLB-Versions: 14.00
		"""
		return self._obj.SetLockChangeable(lockchangeable)

	def FormatValue(self, name:str, value:str) -> str:
		"""
		Converts a value to the formatted value if it is assigned to the attribute.

		:param name: Attribute display or internal name
		:type str: str
		:param value: Attribute value
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value as it is displayed in E³.series |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 14.11
		"""
		return self._obj.FormatValue(name, value)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current attribute item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current attribute item |
		| "&lt;Empty&gt;" | No attribute item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets an attribute as the current item.

		:param gid: Global identifier value of an attribute item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current attribute item |
		| "&lt;Empty&gt;" | No current attribute item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IClipboardInterface--------------------
class Clipboard:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Clipboard. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current clipboard item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current clipboard item identifier |
		| 0 | No clipboard item or an error occurred |

		Available since E3.series TLB-Versions: 18.30
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a clipboard item as the current item.

		:param id: Unique value identifying a clipboard item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current clipboard item identifier |
		| 0 | No current clipboard item or an error occurred |

		Available since E3.series TLB-Versions: 18.30
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the clipboard item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Clipboard item name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 18.30
		"""
		return self._obj.GetName()

	def GetAnyIds(self, flags:int) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the item contained in the clipboard item.

		:param flags: Options for filtering the identifiers in. ids
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of item identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of item identifiers in ids |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 18.30
		"""
		dummy=0
		ret, ids = self._obj.GetAnyIds(flags, dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def CommitToProject(self, flags:int, viewnumber:int=0) -> int:
		"""
		Commits the content of the clipboard item to the project.

		:param flags: Merge options for committing the clipboard item to the project
		:type int: int
		:param viewnumber: Device view number., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Content of the clipboard item committed to the project |
		| -1 | No project open or no clipboard item set |
		| -4 | flags is invalid |
		| -5 | flags contains invalid value combinations |
		| -6 | viewnumber is invalid |

		Available since E3.series TLB-Versions: 18.30
		"""
		return self._obj.CommitToProject(flags, viewnumber)

	def GetCollidingIds(self, flags:int) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the colliding items contained in the clipboard item.

		:param flags: Options for filtering the identifiers in. ids
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of item identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of item identifiers in ids |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 18.30
		"""
		dummy=0
		ret, ids = self._obj.GetCollidingIds(flags, dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Delete(self) -> int:
		"""
		Deletes the clipboard item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Clipboard item was deleted from the project |
		| -1 | No project open or no clipboard item set |
		| -2 | Clipboard item may not be deleted |
		| -3 | Error occurred |

		Available since E3.series TLB-Versions: 18.30
		"""
		return self._obj.Delete()

	def DeleteForced(self) -> int:
		"""
		Deletes the clipboard item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Clipboard item was deleted from the project |
		| -1 | No project open or no clipboard item set |
		| -2 | Clipboard item may not be deleted |
		| -3 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.DeleteForced()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current clipboard item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the clipboard item |
		| "&lt;Empty&gt;" | No clipboard item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a clipboard item as the current item.

		:param gid: Global identifier value of a clipboard item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current clipboard item |
		| "&lt;Empty&gt;" | No current clipboard item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current clipboard item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current clipboard item |
		| "&lt;Empty&gt;" | No clipboard item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a clipboard item as the current item.

		:param guid: Globally unique identifier value of a clipboard item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current clipboard item |
		| "&lt;Empty&gt;" | No current clipboard item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- IComponentInterface--------------------
class Component:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Component. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current component item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current component item identifier |
		| 0 | No component item or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a component as the current item.

		:param id: Unique value identifying a component item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current component item identifier |
		| 0 | No current component item or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the component item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Component's name value is supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.GetName()

	def GetVersion(self) -> str:
		"""
		Gets the component item's version.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Component's version value is supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.GetVersion()

	def GetSupplyPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the component item's pins belonging to a supply circuit.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of supply circuit pin identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No supply circuit pin identifiers are supplied or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		dummy=0
		ret, ids = self._obj.GetSupplyPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the component item.

		:param name: Internal name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the component item's  attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the component item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the component item's specified attribute value.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the component item's specified attribute value.

		:param name: Internal name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the component item.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the component item's specified attributes.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.HasAttribute(name)

	def Search(self, name:str, version:str) -> int:
		"""
		Searches for a component item matching the name and version in the project.

		:param name: Name of the component
		:type str: str
		:param version: Version of the component
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Found component identifier |
		| 0 | No component was found or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.Search(name, version)

	def GetModelName(self) -> str:
		"""
		Gets the component item's model name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Model's name value is supplied |
		| "&lt;Empty&gt;" | Component has no model or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.GetModelName()

	def GetValidModelCharacteristics(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets the component item's model characteristics.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | characteristics | tuple[str,...] | Array of model characteristics passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in characteristics |
		| 0 | Component item has no model, no characteristics or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		dummy=0
		ret, characteristics = self._obj.GetValidModelCharacteristics(dummy)
		characteristics = characteristics[1:] if type(characteristics) == tuple and len(characteristics) > 0 else tuple()
		return ret, characteristics

	def GetFormboardSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of Formboard symbols belonging to the component item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of Formboard symbol identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No Formboard symbol identifiers are supplied or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		dummy=0
		ret, ids = self._obj.GetFormboardSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Rename(self, name:str, version:str) -> int:
		"""
		Sets the component item's name and version.

		:param name: New name of the component
		:type str: str
		:param version: New version of the component
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Component is renamed |
		| 0 | Error has occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.Rename(name, version)

	def GetViewDefinitions(self) -> tuple[int, tuple[tuple[int,int],...]]:
		"""
		Gets the component item's symbol identifiers and their view numbers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | viewDefinitions | tuple[tuple[int,int],...] | Array of symbol identifiers and the symbol's user-defined view number passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in viewDefinitions |
		| 0 | Component has no symbols or an error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		dummy=0
		return self._obj.GetViewDefinitions(dummy)

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the component item's pins.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of pin identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No pin identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 17.10
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSubType(self) -> int:
		"""
		Gets the component item's subtype.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Subtype of the component |
		| -1 | No project open |
		| -2 | No component item set |
		| -3 | Component is not a standard device or block type |
		| -4 | Component item does not refer to a component, For example, it could refer to a cable |
		| -5 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetSubType()

	def GetComponentType(self) -> int:
		"""
		Gets the component item's type.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type of the component |
		| -1 | No project open |
		| -2 | No component item set |
		| -3 | Component type is undefined |

		Available since E3.series TLB-Versions: 20.00
		"""
		return self._obj.GetComponentType()

	def GetStateIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the component item's defined states.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of component state identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No state identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 22.10
		"""
		dummy=0
		ret, ids = self._obj.GetStateIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current component item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current component item |
		| "&lt;Empty&gt;" | No component item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a component as the current item.

		:param gid: Global identifier value of a component item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current component item |
		| "&lt;Empty&gt;" | No current component item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IConnectionInterface--------------------
class Connection:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Connection. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current connection item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current connection item identifier |
		| 0 | No connection item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a connection item as the current item.

		:param id: Unique value identifying a connection item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current connection item identifier |
		| 0 | No current connection item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the connection item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Connection item's name value is supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the connection item.

		:param name: Internal name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the connection item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the connection item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the connection item's specified attribute value.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the connection item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the connection item.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the connection item's specified attributes.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def DisplayAttributeValue(self, name:str) -> int:
		"""
		Displays an attribute value from the connection item.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text item identifier displaying the attribute value |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DisplayAttributeValue(name)

	def DisplayAttributeValueAt(self, name:str, sheetid:int, x:float, y:float) -> int:
		"""
		Displays an attribute value from the connection item at a specific position.

		:param name: Name of the attribute
		:type str: str
		:param sheetid: Sheet identifier to display the attribute value on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text item identifier displaying the attribute value |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DisplayAttributeValueAt(name, sheetid, x, y)

	def SetAttributeVisibility(self, name:str, onoff:int) -> int:
		"""
		Sets the visibility status of all text items representing the connection item's specified attribute value.

		:param name: Internal name of the attribute
		:type str: str
		:param onoff: Indicates whether the text items' visibility status should be shown or hidden. If > 0, the text items are shown. If 0, the text items are hidden
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items changed |
		| 0 | No text items found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeVisibility(name, onoff)

	def IsValid(self) -> int:
		"""
		Indicates if  the connection item is valid.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Connection item is valid |
		| 0 | Connection item is invalid or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsValid()

	def GetPinCount(self) -> int:
		"""
		Gets the number of pins associated with the connection item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pins |
		| 0 | No pins or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPinCount()

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the pins associated with the connection item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of pin identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pins in ids |
		| 0 | No pins or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetCoreCount(self) -> int:
		"""
		Gets the number of conductors associated with the connection item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors |
		| 0 | No conductors or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCoreCount()

	def GetCoreIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductors associated with the connection item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of conductors passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors in ids |
		| 0 | No conductors or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetCoreIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetReferenceSymbolCount(self) -> int:
		"""
		Gets the number of reference symbols associated with the connection item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of reference symbols |
		| 0 | No reference symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetReferenceSymbolCount()

	def GetReferenceSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the reference symbols associated with the connection item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of reference symbol identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of reference symbols in ids |
		| 0 | No reference symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetReferenceSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSignalName(self) -> str:
		"""
		Gets the connection item's signal name value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Connection item's signal name value is supplied |
		| "&lt;Empty&gt;" | No signal name value or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSignalName()

	def SetSignalName(self, name:str) -> int:
		"""
		Sets the connection item's signal name value.

		:param name: Signal name value to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signal name applied |
		| 0 | Error occurred |
		| -1 | Signal name applied, Connection item deleted due to the changed connection logic |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetSignalName(name)

	def Highlight(self) -> int:
		"""
		Highlights the connection item on the sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of connection item elements highlighted |
		| 0 | No connection items found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Highlight()

	def GetNetId(self) -> int:
		"""
		Gets the identifier of the net item in which the item is located.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Net item identifier |
		| 0 | Error occurred |
		| -1 | Conductor is placed on the original connection item and on some of the view connection items |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetNetId()

	def GetNetSegmentCount(self) -> int:
		"""
		Gets the number of net segments associated with the connection item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of net segments |
		| 0 | No net segments or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetNetSegmentCount()

	def GetNetSegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the net segments associated with the connection item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of net segments passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of net segments in ids |
		| 0 | No net segments or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetNetSegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetPinGroupsCount(self) -> int:
		"""
		Gets the number of pin groups associated with the connection item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pin groups |
		| 0 | No pin groups or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPinGroupsCount()

	def GetPinGroupsIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the pin groups associated with the connection item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pin groups passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pin groups in ids |
		| 0 | No pin groups or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinGroupsIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetPinGroupCount(self, num:int) -> int:
		"""
		Gets the number of pins in a pin group associated with the connection item.

		:param num: Index of the pin group
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pins |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPinGroupCount(num)

	def GetPinGroupIds(self, num:int) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the pins of a pin group associated with the connection item.

		:param num: Index of the pin group
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of pin identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pins in ids |
		| 0 | No pins or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinGroupIds(num, dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsView(self) -> int:
		"""
		Indicates if the connection item is a view.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Connection item is a view |
		| 0 | Connection item is not a view or an error occurred |
		| -1 | Conductor is placed on the original connection item and on some of the view connection items |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsView()

	def GetViewNumber(self) -> int:
		"""
		Gets the connection item's view number.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Connection item's view number |
		| 0 | Connection item is not a view or an error occurred |
		| -1 | Conductor is placed on the original connection item and on some of the view connection items |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetViewNumber()

	def Create(self, shti:int, pnts:int, x:list[float], y:list[float], PointTypArr:list[int]=pythoncom.Empty) -> int:
		"""
		Creates a new connection path on a sheet.

		:param shti: Identifier of the schematic sheet
		:type int: int
		:param pnts: Number of placement positions
		:type int: int
		:param x: Array of positions on the x-axis
		:type list[float]: list[float]
		:param y: Array of positions on the y-axis
		:type list[float]: list[float]
		:param PointTypArr: Array of position types. If no array is provided, all position types are lines, defaults to pythoncom.Empty
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the last net segment created |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		x = [0.] + x
		y = [0.] + y
		if PointTypArr != pythoncom.Empty:
			PointTypArr = [0] + PointTypArr
		return self._obj.Create(shti, pnts, x, y, PointTypArr)

	def CreateOnFormboard(self, shti:int, pnts:int, x:list[float], y:list[float], PointTypArr:list[int]=pythoncom.Empty) -> typing.Union[tuple[int,...],int]:
		"""
		Creates a new connection path on a Formboard sheet.

		:param shti: Identifier of the Formboard sheet
		:type int: int
		:param pnts: Number of placement positions
		:type int: int
		:param x: Array of positions on the x-axis
		:type list[float]: list[float]
		:param y: Array of positions on the y-axis
		:type list[float]: list[float]
		:param PointTypArr: Array of position types. If no array is provided, all position types are straight lines, defaults to pythoncom.Empty
		:type list[int]: list[int]

		:return:
		Type: typing.Union[tuple[int,...],int]

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the last net segment created |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		x = [0.] + x
		y = [0.] + y
		if PointTypArr != pythoncom.Empty:
			PointTypArr = [0] + PointTypArr
		ret = self._obj.CreateOnFormboard(shti, pnts, x, y, PointTypArr)
		if type(ret) is tuple:
			ret = ret[1:] if type(ret) == tuple and len(ret) > 0 else tuple()
		return ret

	def GetTranslatedSignalName(self) -> str:
		"""
		Gets the connection item's translated signal name value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Connection item's translated signal name value is supplied |
		| "&lt;Empty&gt;" | No translated signal name value or an error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.GetTranslatedSignalName()

	def CreateConnectionBetweenPoints(self, shti:int, x1:float, y1:float, x2:float, y2:float, flags:int=0) -> int:
		"""
		Creates a new connection path between 2 points on a sheet.

		:param shti: Identifier of the schematic sheet
		:type int: int
		:param x1: First position on the x-axis
		:type float: float
		:param y1: First position on the y-axis
		:type float: float
		:param x2: Second position on the x-axis
		:type float: float
		:param y2: Second position on the y-axis
		:type float: float
		:param flags: Not used, defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the last net segment created |
		| -1 | No project open |
		| -2 | shti is invalid |
		| -3 | Sheet is read-only |
		| -4 | Sheet is locked |
		| -5 | Item at first position is locked |
		| -6 | No valid item found at first position |
		| -7 | Item at second position is locked |
		| -8 | Items at the first and second positions cannot be connected due to a signal collision |
		| -9 | No valid item found at second position |
		| -10 | No route could be found between the items at the first and second positions |
		| -11 | The busbar item cannot be connected to a non-busbar item, Available since v2022-23.00 |
		| -12 | The busbar items cannot be connected to each other, Available since v2022-23.00 |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.CreateConnectionBetweenPoints(shti, x1, y1, x2, y2, flags)

	def CreateConnection(self, flags:int, shti:int, pnts:int, x:list[float], y:list[float], PointTypArr:list[int]=pythoncom.Empty) -> tuple[int, tuple[int,...]]:
		"""
		Creates a new connection path on a sheet.

		:param flags: Connection creation options
		:type int: int
		:param shti: Identifier of the schematic sheet
		:type int: int
		:param pnts: Number of placement positions
		:type int: int
		:param x: Array of positions on the x-axis
		:type list[float]: list[float]
		:param y: Array of positions on the y-axis
		:type list[float]: list[float]
		:param PointTypArr: Array of position types. If no array is provided, all position types are straight lines, defaults to pythoncom.Empty
		:type list[int]: list[int]

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | connections | tuple[int,...] | Array of identifiers of the created net segments passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of created connections |
		| -1 | No project open |
		| -2 | flags is invalid |
		| -3 | E³.logic module is not running |
		| -4 | shti is invalid |
		| -5 | Sheet is locked or read-only |
		| -6 | No license available for E³.formboard |
		| -7 | No license available for E³.funtionaldesign |
		| -8 | No license available for E³.topology |
		| -9 | pnts value is greater than the number of points |
		| -10 | PointTypArr array size is smaller than the pnts value |
		| -11 | Connection could not be created |

		Available since E3.series TLB-Versions: 23.00
		"""
		dummy=0
		x = [0.] + x
		y = [0.] + y
		ret, connections = self._obj.CreateConnection(flags, shti, pnts, x, y, dummy, PointTypArr)
		connections = connections[1:] if type(connections) == tuple and len(connections) > 0 else tuple()
		return ret, connections

# -------------------- IExternalDocumentInterface--------------------
class ExternalDocument:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize ExternalDocument. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current external document item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current external document item identifier |
		| 0 | No external document item  or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets an external document as the current item.

		:param id: Unique value identifying an external document item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current external document item identifier |
		| 0 | No current external document item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the external document item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | External document name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def SetName(self, newname:str) -> str:
		"""
		Sets the external document item's name.

		:param newname: New external document name
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | New external document name |
		| "&lt;Empty&gt; | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(newname)

	def Display(self) -> int:
		"""
		Opens the external document item using the system's designated application.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | External document is opened in another application |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Display()

	def Remove(self) -> int:
		"""
		Stops displaying the external document item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | External document item display closed |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Remove()

	def Visible(self) -> int:
		"""
		No functionality provided.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Always returned |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Visible()

	def Delete(self) -> int:
		"""
		Deletes the current external document from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | External document item was deleted or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def Create(self, modi:int, name:str, file:str) -> int:
		"""
		Creates a new external document item.

		:param modi: Not used
		:type int: int
		:param name: Name of the external document
		:type str: str
		:param file: Path and name of the external file
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new external document item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(modi, name, file)

	def Search(self, modi:int, name:str) -> int:
		"""
		Searches for an item and sets it as the current external document item.

		:param modi: Not used
		:type int: int
		:param name: Name of the external document item to be search for
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the found external document item |
		| 0 | No external document item was found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Search(modi, name)

	def Save(self) -> int:
		"""
		Saves the external document item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | External document item is saved |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Save()

	def InsertFile(self, modi:int, name:str, file:str) -> int:
		"""
		Creates a new external document item.

		:param modi: Indicates how the external file should be handled. If 1, a reference to the file is used. If any other value, the file is embedded within the project
		:type int: int
		:param name: Name of the external document
		:type str: str
		:param file: Path and name of the external file
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new external document item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.InsertFile(modi, name, file)

	def GetFile(self) -> str:
		"""
		Gets the path and file name of the external document item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File Name&gt;" | Path and file name of the external document |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetFile()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the external document.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the external document item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the external document item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the external document item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the external document item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the external document item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the external document item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def GetAssignment(self) -> str:
		"""
		Gets the external document item's higher level assignment value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Higher level assignment value supplied |
		| "&lt;Empty&gt;" | No higher level assignment value supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAssignment()

	def SetAssignment(self, newass:str) -> int:
		"""
		Sets the external document item's higher level assignment value.

		:param newass: New higher level assignment value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Higher level assignment value applied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAssignment(newass)

	def GetLocation(self) -> str:
		"""
		Gets the external document item's location value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Location value supplied |
		| "&lt;Empty&gt;" | No location value supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetLocation()

	def SetLocation(self, newloc:str) -> int:
		"""
		Sets the external document item's location value.

		:param newloc: New location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Location value applied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetLocation(newloc)

	def SetCompleteName(self, newnam:str, newass:str, newloc:str) -> int:
		"""
		Sets all elements of the external document item's item designation.

		:param newnam: New external document name value
		:type str: str
		:param newass: New assignment value
		:type str: str
		:param newloc: New location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Item designation values applied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetCompleteName(newnam, newass, newloc)

	def MoveTo(self, position:int, before:int=0) -> int:
		"""
		Moves the external document item to a position in the project tree.

		:param position: Identifier of the item to use as the insertion position of the external document item in the project tree
		:type int: int
		:param before: Indicates whether to insert the external document item before or after. position. If 0, the external document is inserted below position. If 1, the external document is inserted above position., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | External document item moved |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.MoveTo(position, before)

	def CheckOut(self, lock:bool=True) -> int:
		"""
		Checks out the external document item from the multi-user server.

		:param lock: Indicates if the external document is read/write-able or read-only. If 1, the external document is checked out read-only. If 0, the external document is checked out read/write-able., defaults to True
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 2 | Internal error occurred during check out |
		| 1 | Internal error occurred during data update |
		| 0 | External document was checked out or an error occurred |
		| -1 | External document could not be checked out |
		| -2 | External document could not be checked out |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CheckOut(lock)

	def CheckIn(self, unlock:bool=True) -> int:
		"""
		Checks in the external document item to the multi-user server.

		:param unlock: Indicates if the external document is read/write-able or read-only. If 1, the external document is checked in read/write-able. If 0, the external document is checked in read-only., defaults to True
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 2 | Internal error occurred during check in |
		| 1 | Internal error occurred during data update |
		| 0 | External document was checked in or an error occurred |
		| -1 | External document could not be checked in |
		| -2 | External document could not be checked in |
		| -3 | External document was checked in however it was closed without saving any changes, Available since v2018-19.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CheckIn(unlock)

	def IsReadOnly(self) -> bool:
		"""
		Indicates the external document item's read-only status.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | External document item is read-only |
		| 0 | External document item is read/write-able or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsReadOnly()

	def GetOwner(self) -> str:
		"""
		Gets the external document item's owner.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Owner value supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetOwner()

	def SetVisible(self, visible:int=1) -> int:
		"""
		Sets the external document item's visibility within the project.

		:param visible: Visibility status to apply. If 0, the external document is hidden. If 1, the external document is shown. ., defaults to 1
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previous visibility status was shown |
		| 0 | Previous visibility status was hidden or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetVisible(visible)

	def IsVisible(self) -> bool:
		"""
		Indicates whether the external document item is shown or hidden in the project.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | External document item is shown in the project |
		| 0 | External document item is hidden in the project or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsVisible()

	def SetCompleteNameEx(self, newnam:str, newass:str, newloc:str, onlygiven:bool) -> int:
		"""
		Sets all elements of the external document item's item designation.

		:param newnam: New external document name value
		:type str: str
		:param newass: New assignment value
		:type str: str
		:param newloc: New location value
		:type str: str
		:param onlygiven: Indicates whether empty values in. newnam. ,. newass. and. newloc. should be ignored. If 1, empty values are ignored. If 0, empty values are set
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Item designation values applied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.SetCompleteNameEx(newnam, newass, newloc, onlygiven)

	def GetInternalName(self) -> str:
		"""
		Gets the path and name of the external document item's original file.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File Name&gt;" | Original path and file name of the external document |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 16.05
		"""
		return self._obj.GetInternalName()

	def DisplayEx(self, flags:int=0) -> int:
		"""
		Opens the external document item using the system's designated application.

		:param flags: Indicates if the external document is  opened read/write-able or read-only. If 0, the external document is opened read/write-able. If 1, the external document is checked out read-only., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | External document is opened in another application and is read-only |
		| -1 | Error occurred |
		| -2 | External document is opened in another application but is not read-only |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.DisplayEx(flags)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the external document item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the external document item |
		| "&lt;Empty&gt;" | No external document item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets an external document as the current item.

		:param gid: Global identifier value of an external document item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current external document item |
		| "&lt;Empty&gt;" | No current external document item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current external document item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current external document item |
		| "&lt;Empty&gt;" | No external document item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets an external document as the current item.

		:param guid: Globally unique identifier value of an external document item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current external document item |
		| "&lt;Empty&gt;" | No current external document item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- IFieldInterface--------------------
class Field:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Field. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current field item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current field item identifier |
		| 0 | No field item  or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a field as the current item.

		:param id: Unique value identifying a field item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current field item identifier |
		| 0 | No current field item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def SetName(self, name:str) -> int:
		"""
		Sets the field item's name.

		:param name: New field name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Field name applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def GetName(self) -> str:
		"""
		Gets the field item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Field name supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def GetType(self) -> str:
		"""
		Gets the field item's type name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Type name supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetType()

	def GetVersion(self) -> str:
		"""
		Gets the field item's version.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Version supplied |
		| "&lt;Empty&gt;" | Field has no version value or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetVersion()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the field item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the field item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the field item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the field item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the field item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the field item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the field item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def GetTextCount(self) -> int:
		"""
		Gets the number of the field's text items.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items |
		| 0 | No text items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTextCount()

	def GetTextIds(self, txttyp:int=0, search_string:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of identifiers of the field's text items.

		:param txttyp: Text type filter. Only text identifiers with this Text Type will be supplied. All text identifiers are supplied if txttyp is 0. The default value 0, defaults to 0
		:type int: int
		:param search_string: Text value filter, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of text identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No text items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetTextIds(dummy, txttyp, search_string)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSchemaLocation(self) -> tuple[int, float, float, str, str, str]:
		"""
		Gets the field item's position within the project.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Placement position on the x-axis passed out of the function |
		| 2 | y | float | Placement position on the y-axis passed out of the function |
		| 3 | grid | str | Location in the format ". /&lt;sheet&gt;.&lt;grid&gt;. " passed out of the function |
		| 4 | column_value | str | Sheet placement column of the field item passed out of the function |
		| 5 | row_value | str | Sheet placement row of the field item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the field item's sheet |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetSchemaLocation(dummy, dummy, dummy, dummy, dummy)

	def GetArea(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the field item's placement area on the sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xmin | float | First position on the x-axis passed out of the function |
		| 2 | ymin | float | First position on the y-axis passed out of the function |
		| 3 | xmax | float | Second position on the x-axis passed out of the function |
		| 4 | ymax | float | Second position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Placement area values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetArea(dummy, dummy, dummy, dummy)

	def GetGraphId(self) -> int:
		"""
		Gets an identifier of the graphic item representing the boundary of the field item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Graphic item identifier of the field item's boundary |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetGraphId()

	def Jump(self) -> int:
		"""
		Jumps to the field item's sheet and highlights the field item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Always returned |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Jump()

	def Delete(self) -> int:
		"""
		Deletes the current field from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the field item, Unable to complete delete operation |
		| 0 | Field item was deleted or no project open or no field item set |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def GetSymbolCount(self) -> int:
		"""
		Gets the number of the field item's symbols.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbol items |
		| 0 | No symbol items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSymbolCount()

	def GetSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the field item's symbol identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of symbol identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Place(self, shti:int, x1:float, y1:float, x2:float, y2:float, moveall:bool=False) -> int:
		"""
		Creates and places a field item or moves the current field item's position within the project.

		:param shti: Identifier of sheet to place field on
		:type int: int
		:param x1: First placement position on the x-axis
		:type float: float
		:param y1: First placement position on the y-axis
		:type float: float
		:param x2: Second placement position on the x-axis
		:type float: float
		:param y2: Second placement position on the y-axis
		:type float: float
		:param moveall: Indicates whether the items contained within the field should also be moved. If 1, items contained within the field are also moved. If 0, only the field item is moved., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the field item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Place(shti, x1, y1, x2, y2, moveall)

	def GetInsideNetSegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the net segments with routing paths placed within the field item's area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of net segment items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No net segment identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsideNetSegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetCrossingNetSegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the net segments with routing paths entering the field item's area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of net segment items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No net segment identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetCrossingNetSegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetInsideTextIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the text items contained within the field item's area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of text items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No text item identifiers found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsideTextIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetInsideGraphIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the graphic items contained within the field item's area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of graphic items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No graphic item identifiers found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsideGraphIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetInsideSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbol items whose origins are contained within the field item's area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbol items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No symbol item identifiers found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsideSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetParentFieldId(self) -> int:
		"""
		Gets the identifier of the field item's parent field.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the field item's parent field |
		| 0 | Field item has no parent field or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetParentFieldId()

	def GetCrossingFieldIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the fields partially placed within the field item's area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of field items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No field identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetCrossingFieldIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetInsideFieldIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the fields placed directly within the field item's area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of field items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No field identifiers found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsideFieldIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAllInsideFieldIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the fields placed within the field item's area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of field items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No field identifiers found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAllInsideFieldIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetTypeName(self) -> str:
		"""
		Gets the field item's type name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Type name supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTypeName()

	def GetInsidePanelConnectionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductors contained within the field item's area on a panel sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pin items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No pin identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsidePanelConnectionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetCrossingPanelConnectionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductors entering the field item's area on a panel sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of conductor items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No conductor identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetCrossingPanelConnectionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAssignedOptionExpressions(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options) assigned to the field item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressions(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def SetOptionExpressions(self, expressions:list[str]) -> int:
		"""
		Replaces the field item's option expressions.

		:param expressions: Array of strings of all option names, Boolean expressions or alias names of Boolean expressions
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.12
		"""
		return self._obj.SetOptionExpressions(expressions)

	def SetCompleteName(self, newdev:str, newass:str, newloc:str, onlygiven:bool=False) -> int:
		"""
		Sets all elements of the field item's  item designation.

		:param newdev: New device value
		:type str: str
		:param newass: New higher level assignment value
		:type str: str
		:param newloc: New location value
		:type str: str
		:param onlygiven: Indicates whether empty values in. newdev. ,. newass. and. newloc. should be ignored. If 1, empty values are ignored. If 0, empty values are set., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Item designation values applied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.SetCompleteName(newdev, newass, newloc, onlygiven)

	def SetDeviceAssignment(self, newass:str) -> int:
		"""
		Sets the field item's higher level assignment value.

		:param newass: New higher level assignment value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Higher level assignment value applied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.SetDeviceAssignment(newass)

	def GetDeviceAssignment(self) -> str:
		"""
		Gets the field item's higher level assignment value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Higher level assignment value supplied |
		| "&lt;Empty&gt;" | No higher level assignment value supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.GetDeviceAssignment()

	def SetDeviceLocation(self, newloc:str) -> int:
		"""
		Sets the field item's location value.

		:param newloc: New location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Location value applied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.SetDeviceLocation(newloc)

	def GetDeviceLocation(self) -> str:
		"""
		Gets the field item's location value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Location value supplied |
		| "&lt;Empty&gt;" | No location value supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.GetDeviceLocation()

	def SetDeviceName(self, newdev:str) -> int:
		"""
		Sets the field item's device name value.

		:param newdev: New device name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Device name value applied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.SetDeviceName(newdev)

	def GetDeviceName(self) -> str:
		"""
		Gets the field item's device designation value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Device designation value supplied |
		| "&lt;Empty&gt;" | No device designation value supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.GetDeviceName()

	def GetAssignedOptionExpressionsEx(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options)  including variant instances assigned to the field item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.23
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressionsEx(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def GetInterruptBorder(self) -> bool:
		"""
		Gets a value indicating if the field border is interrupted where connection lines cross the field item's boundary.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Field border is interrupted where connection lines cross the field item's boundary |
		| 0 | Field border is not interrupted where connection lines cross the field item's boundary or an error occurred |

		Available since E3.series TLB-Versions: 10.22
		"""
		return self._obj.GetInterruptBorder()

	def SetInterruptBorder(self, interrupt:bool) -> bool:
		"""
		Sets whether the field border is interrupted where connection lines cross the field item's boundary.

		:param interrupt: New field border interruption flag value. If 1, the field item's border is interrupted where connection lines cross the field item's boundary. If 0, the field item's border is not interrupted where connection lines cross the field item's boundary
		:type bool: bool

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 or 0 | Previous interrupt border value or an error occurred |

		Available since E3.series TLB-Versions: 10.22
		"""
		return self._obj.SetInterruptBorder(interrupt)

	def GetInterruptBorderGap(self) -> float:
		"""
		Gets the size of the gap when the field border is interrupted where connection lines cross the field item's boundary.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Size of the gap when the field border is interrupted where connection lines cross the field item's boundary |
		| 0.0 | Size of the gap when the field border is interrupted where connection lines cross the field item's boundary or an error occurred |

		Available since E3.series TLB-Versions: 10.22
		"""
		return self._obj.GetInterruptBorderGap()

	def SetInterruptBorderGap(self, gap:float) -> float:
		"""
		Sets the size of the gap when the field border is interrupted where connection lines cross the field item's boundary.

		:param gap: New size of the interrupt border gap value
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.0 | Previous interrupt border gap value or an error occurred |

		Available since E3.series TLB-Versions: 10.22
		"""
		return self._obj.SetInterruptBorderGap(gap)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the field item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the field item |
		| "&lt;Empty&gt;" | No field item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a field as the current item.

		:param gid: Global identifier value of a field item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current field item |
		| "&lt;Empty&gt;" | No current field item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current field item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current field item |
		| "&lt;Empty&gt;" | No field item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a field as the current item.

		:param guid: Globally unique identifier value of a field item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current field item |
		| "&lt;Empty&gt;" | No current field item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- IGraphInterface--------------------
class Graph:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Graph. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current graphic item identifier |
		| 0 | No graphic item  or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a graphic as the current item.

		:param id: Unique value identifying a graphic item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current graphic item identifier |
		| 0 | No current field item or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetId(id)

	def CreateText(self, shti:int, text:str, x:float, y:float) -> int:
		"""
		Adds a new text graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the text item on
		:type int: int
		:param text: Text value
		:type str: str
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new text item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateText(shti, text, x, y)

	def CreateRotatedText(self, shti:int, text:str, x:float, y:float, rotation:float) -> int:
		"""
		Adds a new text graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the text item on
		:type int: int
		:param text: Text value
		:type str: str
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rotation: Rotation value in degrees
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new text item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateRotatedText(shti, text, x, y, rotation)

	def CreateLine(self, shti:int, x1:float, y1:float, x2:float, y2:float) -> int:
		"""
		Adds a new line graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param x1: First placement position on the x-axis
		:type float: float
		:param y1: First placement position on the y-axis
		:type float: float
		:param x2: Second placement position on the x-axis
		:type float: float
		:param y2: Second placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateLine(shti, x1, y1, x2, y2)

	def CreateRectangle(self, shti:int, x1:float, y1:float, x2:float, y2:float) -> int:
		"""
		Adds a new rectangle graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param x1: First placement position on the x-axis
		:type float: float
		:param y1: First placement position on the y-axis
		:type float: float
		:param x2: Second placement position on the x-axis
		:type float: float
		:param y2: Second placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateRectangle(shti, x1, y1, x2, y2)

	def CreateMeasure(self, shti:int, x1:float, y1:float, x2:float, y2:float) -> int:
		"""
		Adds a new dimension line graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param x1: First placement position on the x-axis
		:type float: float
		:param y1: First placement position on the y-axis
		:type float: float
		:param x2: Second placement position on the x-axis
		:type float: float
		:param y2: Second placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Identifier of the new graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateMeasure(shti, x1, y1, x2, y2)

	def CreatePolygon(self, shti:int, pnts:int, x:list[int], y:list[int]) -> int:
		"""
		Adds a new polygon graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param pnts: Number of placement positions in. x. and. y
		:type int: int
		:param x: Array of placement positions on the x-axis
		:type list[int]: list[int]
		:param y: Array of placement positions on the y-axis
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreatePolygon(shti, pnts, x, y)

	def CreateCircle(self, shti:int, x:float, y:float, radius:float) -> int:
		"""
		Adds a new circle graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param radius: Radius of the circle
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateCircle(shti, x, y, radius)

	def CreateArc(self, shti:int, x:float, y:float, radius:float, start:float, end:float) -> int:
		"""
		Adds a new arc graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param radius: Radius of the circle of the arc
		:type float: float
		:param start: Angle of the circle to begin the arc
		:type float: float
		:param end: Angle of the circle to end the arc
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateArc(shti, x, y, radius, start, end)

	def Place(self, x:float, y:float) -> int:
		"""
		Places or repositions the graphic item.

		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Graphic item placed or repositioned |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.Place(x, y)

	def GetType(self) -> int:
		"""
		Gets a value representing the type of the graphic item.

		:return:
		Type: int **Available enum type:** e3series.types.GraphType.

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type value of the graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetType()

	def GetLevel(self) -> int:
		"""
		Gets the graphic item's display level value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Display level value supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLevel()

	def SetLevel(self, newlev:int) -> int:
		"""
		Sets the graphic item's display level value.

		:param newlev: Display level value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display level value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLevel(newlev)

	def GetGraphCount(self) -> int:
		"""
		Gets the number of graphic items belonging to the group graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | The number of items in the group graphic |
		| 0 | an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetGraphCount()

	def GetGraphIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of graphic items belonging to the group graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of graphic items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetGraphIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Delete(self) -> int:
		"""
		Deletes the graphic item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the graphic item |
		| 0 | Graphic item was deleted or no graphic item was set |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.Delete()

	def GetText(self) -> str:
		"""
		Gets the graphic item's text value including text token translations.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Text value is supplied |
		| "&lt;Empty&gt;" | Text item does not have a value or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetText()

	def SetText(self, newtext:str) -> int:
		"""
		Sets the graphic item's text value.

		:param newtext: Text value to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text value is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetText(newtext)

	def GetTextHeight(self) -> float:
		"""
		Gets the graphic item's font height.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Size of the graphic's font height |
		| 0.0 | Font height could not be supplied |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetTextHeight()

	def SetTextHeight(self, hgt:float) -> int:
		"""
		Sets the graphic item's font height.

		:param hgt: Value of font height to apply
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font height applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetTextHeight(hgt)

	def GetTextMode(self) -> int:
		"""
		Gets the graphic item's text ratio.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..3 | Text ratio value supplied |
		| 0 | Text ratio value could not be supplied |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetTextMode()

	def SetTextMode(self, newmod:int) -> int:
		"""
		Sets the graphic item's text ratio.

		:param newmod: Text ratio value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text ratio value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetTextMode(newmod)

	def GetTextStyle(self) -> int:
		"""
		Gets the graphic item's text font style.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..7 | Font style value supplied |
		| 0 | Regular font style value used or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetTextStyle()

	def SetTextStyle(self, newstl:int) -> int:
		"""
		Sets the text item's font style.

		:param newstl: Font style to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font style value applied |
		| 0 | No font style value used or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetTextStyle(newstl)

	def GetTextFontName(self) -> str:
		"""
		Gets the graphic item's font name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Name of the graphic's font |
		| "Arial" | Name of the graphic's font is Arial or the font name could not be supplied |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetTextFontName()

	def SetTextFontName(self, newname:str) -> int:
		"""
		Sets the graphic item's font.

		:param newname: Name of font to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetTextFontName(newname)

	def GetTextColour(self) -> int:
		"""
		Gets the graphic item's text color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or the operation has failed |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetTextColour()

	def SetTextColour(self, newcol:int) -> int:
		"""
		Sets the graphic item's text color.

		:param newcol: Color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetTextColour(newcol)

	def GetColour(self) -> int:
		"""
		Gets the graphic item's color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or an error occurred |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetColour()

	def SetColour(self, newcol:int) -> int:
		"""
		Sets the graphic item's color.

		:param newcol: Color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetColour(newcol)

	def GetHatchColour(self) -> int:
		"""
		Gets the graphic item's hatch color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Hatch color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or the operation has failed |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetHatchColour()

	def SetHatchColour(self, newcol:int) -> int:
		"""
		Sets the graphic item's hatch color.

		:param newcol: Hatch color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Hatch color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetHatchColour(newcol)

	def GetAssignedOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of options assigned to the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of assigned options passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned option identifiers are supplied or an error occurred |
		| -1 | An option contains a Boolean expression or an alias |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetAssignedOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetInternalText(self) -> str:
		"""
		Gets the graphic item's text value without text token translations.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Text value is supplied |
		| "&lt;Empty&gt;" | Graphic does not have a text value or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetInternalText()

	def CreateImage(self, sheetid:int, xpos:float, ypos:float, xsize:float, ysize:float, filename:str, embed:int=1) -> int:
		"""
		Adds a new image graphic item onto a sheet of the project.

		:param sheetid: Identifier of the sheet to place the graphic item on
		:type int: int
		:param xpos: Placement position on the x-axis
		:type float: float
		:param ypos: Placement position on the y-axis
		:type float: float
		:param xsize: Size of the graphic on the x-axis
		:type float: float
		:param ysize: Size of the graphic on the y-axis
		:type float: float
		:param filename: Path and name of the image file to be used
		:type str: str
		:param embed: If set to 0, a reference to the image is stored in the project. If set to 1, the image is stored internally in the project., defaults to 1
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic image item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateImage(sheetid, xpos, ypos, xsize, ysize, filename, embed)

	def GetImageInfo(self) -> tuple[int, float, float, float, float, str, int]:
		"""
		Gets the image graphic item's values.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xpos | float | Placement position on the x-axis passed out of the function |
		| 2 | ypos | float | Placement position on the y-axis passed out of the function |
		| 3 | xsize | float | Size of the graphic on the x-axis passed out of the function |
		| 4 | ysize | float | Size of the graphic on the y-axis passed out of the function |
		| 5 | filename | str | Path and name of the image file passed out of the function |
		| 6 | embed | int | Image storage status passed out of the function. If value is 0, a reference to the image stored in the project. If value is 1, the image is stored internally in the project |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Items values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetImageInfo(dummy, dummy, dummy, dummy, dummy, dummy)

	def SetImageInfo(self, xpos:float, ypos:float, xsize:float, ysize:float, filename:str="", embed:int=-1) -> int:
		"""
		Sets the image graphic item's values.

		:param xpos: Placement position on the x-axis
		:type float: float
		:param ypos: Placement position on the y-axis
		:type float: float
		:param xsize: Size of the graphic on the x-axis
		:type float: float
		:param ysize: Size of the graphic on the y-axis
		:type float: float
		:param filename: Path and name of the image file to be used. Default value is "<Empty>", defaults to ""
		:type str: str
		:param embed: If set to 0, a reference to the image is stored in the project. If set to 1, the image is stored internally in the project. If set to -1, the current image storage method will be retained., defaults to -1
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New values are applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetImageInfo(xpos, ypos, xsize, ysize, filename, embed)

	def IsRedlined(self) -> int:
		"""
		Gets the graphic's redlined status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | redlined status is on |
		| 0 | redlined status is off |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.IsRedlined()

	def GetLineColour(self) -> int:
		"""
		Gets the graphic item's line color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Line color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or the operation has failed |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLineColour()

	def SetLineColour(self, newcol:int) -> int:
		"""
		Sets the graphic item's line color.

		:param newcol: Line color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Line color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLineColour(newcol)

	def GetLineWidth(self) -> float:
		"""
		Gets the line width of a graphic item.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Line width value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLineWidth()

	def SetLineWidth(self, newwid:float) -> float:
		"""
		Sets the line width of a graphic item.

		:param newwid: Line width value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.0 | Previous line width value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLineWidth(newwid)

	def GetLineStyle(self) -> int:
		"""
		Gets the line style of a graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Line style value, See Line Styles for possible values |
		| 0 or &gt;47 | User defined line style value |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLineStyle()

	def SetLineStyle(self, newstyle:int) -> int:
		"""
		Sets the line style of a graphic item.

		:param newstyle: Line style value. See Line Styles for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Previous line style value |
		| 0 or &gt;47 | Previous user defined line style value |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLineStyle(newstyle)

	def GetHatchLineWidth(self) -> float:
		"""
		Gets the line width of a graphic item's hatch.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Hatch line width value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetHatchLineWidth()

	def SetHatchLineWidth(self, newwid:float) -> float:
		"""
		Sets the line width of a graphic item's hatch.

		:param newwid: Hatch line width value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.0 | Previous hatch line width value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetHatchLineWidth(newwid)

	def GetHatchLineStyle(self) -> int:
		"""
		Gets the line style of a graphic item's hatch.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Hatch line style value, See Line Styles for possible values |
		| 0 or &gt;47 | 0 or &gt;47 |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetHatchLineStyle()

	def SetHatchLineStyle(self, newstyle:int) -> int:
		"""
		Sets the line style of a graphic item's hatch.

		:param newstyle: Hatch line style value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Previous hatch line style value, See Line Styles for possible values |
		| 0 or &gt;47 | Previous hatch line style value |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetHatchLineStyle(newstyle)

	def GetHatchLineDistance(self) -> float:
		"""
		Gets the distance between the lines of a graphic item's hatch.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Hatch line distance value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetHatchLineDistance()

	def SetHatchLineDistance(self, newdist:float) -> float:
		"""
		Sets the distance between the lines of a graphic item's hatch.

		:param newdist: Hatch line distance value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Previous hatch line distance value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetHatchLineDistance(newdist)

	def GetArrows(self) -> int:
		"""
		Gets a value describing how arrows are displayed on the graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Arc data is successfully supplied |
		| 0 | No arrows or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetArrows()

	def SetArrows(self, newarrows:int) -> int:
		"""
		Sets a value specifying how arrows are displayed on the graphic item.

		:param newarrows: Arrows display value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Previous arrows display value |
		| 0 | Previously no arrows displayed or error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetArrows(newarrows)

	def GetHatchPattern(self) -> tuple[int, float, float]:
		"""
		Gets a value representing the used hatch pattern of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | angle1 | float | Angle value in degrees of a hatch line if a line or cross hatch pattern is in use. The value will be 0.0 if neither a line or cross hatch pattern is in use |
		| 2 | angle2 | float | Angle value in degrees of a hatch line if a cross hatch pattern is in use. The value will be 0.0 if a cross hatch pattern is not in use |

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Error occurred |
		| 1 | Solid hatch |
		| 2 | Line hatch |
		| 4 | Cross hatch |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetHatchPattern(dummy, dummy)

	def SetHatchPattern(self, newpat:int, angle1:float, angle2:float) -> int:
		"""
		Sets a value representing the hatch pattern of the graphic item.

		:param newpat: Hatch pattern value to apply
		:type int: int
		:param angle1: Angle value in degrees of a hatch line to apply. Value is only used if newPat is a line or cross hatch pattern value. The value range is from 90.0 (pointing upwards) to 0.0 (pointing rightwards) to -90.0 (pointing downwards)
		:type float: float
		:param angle2: Angle value in degrees of a hatch line to apply. Value is only used if newPat is a cross hatch pattern value. The value range is from 90.0 (pointing upwards) to 0.0 (pointing rightwards) to -90.0 (pointing downwards)
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | No previous hatch pattern value set or error occurred |
		| 1, 2 or 4 | Previous hatch pattern value |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetHatchPattern(newpat, angle1, angle2)

	def GetTypeId(self) -> int:
		"""
		Gets a value representing the type of the graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type value of the graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetTypeId()

	def CreateFromSymbol(self, shti:int, x:float, y:float, rot:str, scale:float, maintaintextsize:bool, symnam:str, symver:str) -> int:
		"""
		Adds a new group graphic item, based on a database symbol, onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Rotation value in degrees. See Rotation (String) for possible values
		:type str: str
		:param scale: Rotation value in degrees
		:type float: float
		:param maintaintextsize: scale. is applied to text graphics if. maintaintextsize. is set to 0. scale is ignored by text graphics if maintaintextsize is greater than 0
		:type bool: bool
		:param symnam: Name of database symbol to use to create the graphic
		:type str: str
		:param symver: Version of database symbol to use to create the graphic
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new group graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateFromSymbol(shti, x, y, rot, scale, maintaintextsize, symnam, symver)

	def GetArc(self) -> tuple[int, float, float, float, float, float]:
		"""
		Gets the arc placement data of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xm | float | Position of the arc center on the x-axis  passed out of the function |
		| 2 | ym | float | Position of the arc center on the y-axis  passed out of the function |
		| 3 | rad | float | Radius of the arc  passed out of the function |
		| 4 | startang | float | Start angle of the arc  passed out of the function |
		| 5 | endang | float | End angle of the arc  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Arc data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetArc(dummy, dummy, dummy, dummy, dummy)

	def GetCircle(self) -> tuple[int, float, float, float]:
		"""
		Gets the circle placement position of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xm | float | Position of the circle center on the x-axis passed out of the function |
		| 2 | ym | float | Position of the circle center on the y-axis passed out of the function |
		| 3 | rad | float | Radius of the circle passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Circle data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetCircle(dummy, dummy, dummy)

	def GetLine(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the line placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x1 | float | First placement position on the x-axis passed out of the function |
		| 2 | y1 | float | First placement position on the y-axis passed out of the function |
		| 3 | x2 | float | Second placement position on the x-axis passed out of the function |
		| 4 | y2 | float | Second placement position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Line data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetLine(dummy, dummy, dummy, dummy)

	def GetPolygon(self) -> tuple[int, int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the polygon placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | npnts | int | Number of placement positions passed out of the function |
		| 2 | xarr | tuple[float,...] | Array of placement positions on the x-axis passed out of the function |
		| 3 | yarr | tuple[float,...] | Array of placement positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Polygon data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, npnts, xarr, yarr = self._obj.GetPolygon(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, npnts, xarr, yarr

	def GetRectangle(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the rectangle placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x1 | float | First placement position on the x-axis passed out of the function |
		| 2 | y1 | float | First placement position on the y-axis passed out of the function |
		| 3 | x2 | float | Second placement position on the x-axis passed out of the function |
		| 4 | y2 | float | Second placement position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Rectangle data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetRectangle(dummy, dummy, dummy, dummy)

	def SaveImage(self, filename:str) -> int:
		"""
		Saves the image graphic item's image data to an external file.

		:param filename: Path and name of the image file
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Image file saved |
		| 0 | No graphic item set or no project open |
		| -1 | Invalid graphic item |
		| -2 | Graphic item is not an image |
		| -3 | filename is an invalid path |
		| -4 | filename has an invalid file extension |
		| -5 | Original image file is not available |
		| -6 | Image could not be saved |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SaveImage(filename)

	def CreateGroup(self, ids:list[int]) -> int:
		"""
		Combines graphic items into a single graphic item.

		:param ids: Array of graphic item identifiers to include in the group
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic group item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.CreateGroup(ids)

	def UnGroup(self) -> tuple[int, tuple[int,...]]:
		"""
		Breaks a group graphic item into its individual graphic item parts.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of graphic item identifiers that were in the group passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of graphic items in ids |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.UnGroup(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetParentID(self) -> int:
		"""
		Gets the identifier of the parent item of a graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of parent item |
		| 0 | Graphic item has no parent or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetParentID()

	def SetParentID(self, id:int) -> int:
		"""
		Sets the graphic item's parent item.

		:param id: Item identifier of the parent
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Previous line width value |
		| 0 | Parent ungrouped or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetParentID(id)

	def CreateCurve(self, shti:int, pnts:int, x:list[float], y:list[float]) -> int:
		"""
		Adds a new curve graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param pnts: Number of placement positions in. x. and. y
		:type int: int
		:param x: Array of placement positions on the x-axis
		:type list[float]: list[float]
		:param y: Array of placement positions on the y-axis
		:type list[float]: list[float]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		x = [0.] + x
		y = [0.] + y
		return self._obj.CreateCurve(shti, pnts, x, y)

	def GetCurve(self) -> tuple[int, int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the curve placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | npnts | int | Number of placement positions passed out of the function |
		| 2 | xarr | tuple[float,...] | Array of placement positions on the x-axis passed out of the function |
		| 3 | yarr | tuple[float,...] | Array of placement positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Curve data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, npnts, xarr, yarr = self._obj.GetCurve(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, npnts, xarr, yarr

	def GetAssignedOptionExpressions(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options) assigned to the graphics item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressions(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def SetOptionExpressions(self, expressions:list[str]) -> int:
		"""
		Replaces the graphic item's option expressions.

		:param expressions: Array of strings of all option names, Boolean expressions or alias names of Boolean expressions
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.12
		"""
		return self._obj.SetOptionExpressions(expressions)

	def GetAssignedOptionExpressionsEx(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options) including variant instances assigned to the graphic item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.23
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressionsEx(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the graphic item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the graphic item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.DeleteAttribute(name)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the graphic item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the graphic item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is "<Empty>", defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the graphics item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetAttributeValue(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the graphic item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.HasAttribute(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the graphic item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def CreateCloud(self, shti:int, pnts:int, x:list[float], y:list[float]) -> int:
		"""
		Adds a new cloud graphic item onto a sheet of the project.

		:param shti: Identifier of the sheet to place the graphic item on
		:type int: int
		:param pnts: Number of placement positions in. x. and. y
		:type int: int
		:param x: Array of placement positions on the x-axis
		:type list[float]: list[float]
		:param y: Array of placement positions on the y-axis
		:type list[float]: list[float]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.70
		"""
		x = [0.] + x
		y = [0.] + y
		return self._obj.CreateCloud(shti, pnts, x, y)

	def GetCloud(self) -> tuple[int, int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the cloud placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | npnts | int | Number of placement positions passed out of the function |
		| 2 | xarr | tuple[float,...] | Array of placement positions on the x-axis passed out of the function |
		| 3 | yarr | tuple[float,...] | Array of placement positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Cloud data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.70
		"""
		dummy=0
		ret, npnts, xarr, yarr = self._obj.GetCloud(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, npnts, xarr, yarr

	def SetRedlined(self, onoff:bool) -> int:
		"""
		Sets the graphic item's redlined status.

		:param onoff: Redlined status value to apply. If 1, the redlined status is set to on. If 0, the redlined status in set to off., defaults to TRUE
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previous redlined status was on |
		| 0 | Previous redlined status was off or an error occurred |

		Available since E3.series TLB-Versions: 17.04
		"""
		return self._obj.SetRedlined(onoff)

	def OptimizeGraphicObjects(self, ids:list[int], mode:int, angle:int) -> tuple[int, list[int]]:
		"""
		Optimizes the graphics objects by reducing the number and modifying them.

		:param ids: Array of graphic identifiers to be optimized passed into the function and passed out optimized
		:type list[int]: list[int]
		:param mode: Type of optimization to apply
		:type int: int
		:param angle: Maximum angle for converting arcs to polygons
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | list[int] | Array of graphic identifiers to be optimized passed into the function and passed out optimized |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Graphic items have been optimized |
		| 0 | No graphic items optimized |
		| -1 | No project open |
		| -2 | Invalid mode value |
		| -3 | angle value out of range |

		Available since E3.series TLB-Versions: 22.00, 21.01, 20.22
		"""
		ret, ids = self._obj.OptimizeGraphicObjects(ids, mode, angle)
		ids = ids[1:] if type(ids) == list and len(ids) > 0 else []
		return ret, ids

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current graphic item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current  graphic item |
		| "&lt;Empty&gt;" | No graphic item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a graphic item as the current item.

		:param gid: Global identifier value of a graphic item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current graphic item |
		| "&lt;Empty&gt;" | No current graphic item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def SendToForeground(self) -> int:
		"""
		Brings the display of the graphic item to the foreground of the sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Graphic item send to foreground |
		| -1 | No project open |
		| -2 | No graphic item set |
		| -3 | Error accessing graphic item |
		| -4 | Error accessing sheet item |
		| -5 | Graphic item cannot be sent to the foreground |
		| -6 | No license available |
		| -7 | Graphic item is locked |

		Available since E3.series TLB-Versions: 24.22
		"""
		return self._obj.SendToForeground()

	def SendToBackground(self) -> int:
		"""
		Brings the display of the graphic item to the background of the sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Graphic item send to background |
		| -1 | No project open |
		| -2 | No graphic item set |
		| -3 | Error accessing graphic item |
		| -4 | Error accessing sheet item |
		| -5 | Graphic item cannot be sent to the background |
		| -6 | No license available |
		| -7 | Graphic item is locked |

		Available since E3.series TLB-Versions: 24.22
		"""
		return self._obj.SendToBackground()

# -------------------- IModuleInterface--------------------
class Module:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Module. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current module item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current module item identifier |
		| 0 | No module item |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a module as the current item.

		:param id: Unique value identifying a module item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current module item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.SetId(id)

	def GetLevel(self) -> int:
		"""
		Gets the level of the current module item within the module hierarchy.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current module level |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetLevel()

	def GetParentModuleId(self) -> int:
		"""
		Gets the identifier of the parent module of the current module item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Parent module item identifier |
		| 0 | No parent module item or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetParentModuleId()

	def GetName(self) -> str:
		"""
		Gets the name of the current module item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Module item name value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the name of the current module item.

		:param name: Name of the module item
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | name assigned to the module item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.SetName(name)

	def GetFileName(self) -> str:
		"""
		Gets the system path and name of the file from which the module item was created.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File&gt;" | System path and name of the module item's associated file |
		| "&lt;Empty&gt;" | Module item's has no associated or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetFileName()

	def GetTypeName(self) -> str:
		"""
		Gets the type name of the current module item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Type&gt;" | Type name |
		| "&lt;Empty&gt;" | No type name or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetTypeName()

	def IsTypeLoadable(self) -> int:
		"""
		Indicates whether the module item's type exists in the module library.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Module type exists in the module library |
		| 0 | Module type does not exist in the module library or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.IsTypeLoadable()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the module item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the module item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the module item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the module item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the module item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the module item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the module item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.HasAttribute(name)

	def GetPortCount(self) -> int:
		"""
		Gets the number of the module ports belonging to the current module item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of module ports |
		| 0 | No  module ports found or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetPortCount()

	def GetPortIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the current module item's module ports.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of module port identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of module port identifiers in ids |
		| 0 | No module ports or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		dummy=0
		ret, ids = self._obj.GetPortIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Search(self, name:str) -> int:
		"""
		Searches for a module item matching the name.

		:param name: Name of the module
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Found module item identifier |
		| 0 | No module item found or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.Search(name)

	def GetParentSheetId(self) -> int:
		"""
		Gets the current module item's parent sheet identifier.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Parent sheet identifier |
		| 0 | Parent sheet is not visible or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetParentSheetId()

	def GetSheetIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the module item's sheets.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of sheet identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of sheet items in ids |
		| 0 | No sheets or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		dummy=0
		ret, ids = self._obj.GetSheetIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetModuleIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the current module item's child modules.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of module identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of module identifiers in ids |
		| 0 | No modules or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		dummy=0
		ret, ids = self._obj.GetModuleIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the module item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the module item |
		| "&lt;Empty&gt;" | No module item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a module as the current item.

		:param gid: Global identifier value of a module item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current module item |
		| "&lt;Empty&gt;" | No module item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IModulePortInterface--------------------
class ModulePort:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize ModulePort. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current module port item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current module port item identifier |
		| 0 | No module port item |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a module port item as the current item.

		:param id: Unique value identifying a module port item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current module port item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the name of the current module port item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Module port item name value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the name of the current module port item.

		:param name: Name of the module port item
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | name assigned to the module port item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.SetName(name)

	def GetSymbolIds(self) -> tuple[int, tuple[int,...], tuple[int,...]]:
		"""
		Gets an array of identifiers of current module port item's symbols placed on the module block and on the module sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | OnBlockId | tuple[int,...] | Identifier of the symbol item placed on the module block passed out of the function |
		| 2 | OnSheetId | tuple[int,...] | Identifier of the symbol item placed on the module sheet passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1..2 | Total number of symbol item identifiers in OnBlockid and OnSheetId |
		| 0 | No symbol item identifiers found or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		dummy=0
		ret, OnBlockId, OnSheetId = self._obj.GetSymbolIds(dummy, dummy)
		OnBlockId = OnBlockId[1:] if type(OnBlockId) == tuple and len(OnBlockId) > 0 else tuple()
		OnSheetId = OnSheetId[1:] if type(OnSheetId) == tuple and len(OnSheetId) > 0 else tuple()
		return ret, OnBlockId, OnSheetId

	def GetModuleId(self) -> int:
		"""
		Gets the identifier of the module item the current module port belongs to.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Module item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetModuleId()

	def IsBus(self) -> int:
		"""
		Indicates whether the module port item is a bus port.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Module port is a bus port |
		| 0 | Module port is not a bus port or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.IsBus()

	def GetSignalId(self) -> int:
		"""
		Gets the identifier of the signal carried by the current module port item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Signal item identifier |
		| 0 | No signal for an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetSignalId()

	def GetBusName(self) -> str:
		"""
		Gets the bus name of the current module port item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Module port item bus name value |
		| "&lt;Empty&gt;" | Module port is not a bus port or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetBusName()

	def SetBusName(self, name:str) -> int:
		"""
		Sets the bus name of the current module port item.

		:param name: Bus name of the module port item
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | name assigned as the bus name |
		| 0 | Module port is not a bus port or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.SetBusName(name)

	def GetSignalIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the signals carried by the current bus module port item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of signals identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of signal items in ids |
		| 0 | No signals or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		dummy=0
		ret, ids = self._obj.GetSignalIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the module port item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the module port item |
		| "&lt;Empty&gt;" | No module port item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a module port as the current item.

		:param gid: Global identifier value of a module port item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current module port item |
		| "&lt;Empty&gt;" | No module port item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current module port item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current module port item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a module port as the current item.

		:param guid: Globally unique identifier value of a module port item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current module port item |
		| "&lt;Empty&gt;" | No module port item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- INetSegmentInterface--------------------
class NetSegment:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize NetSegment. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current net segment item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current net segment item identifier |
		| 0 | No net segment item |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a net segment item as the current item.

		:param id: Unique value identifying a net segment item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current net segment item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the net segment item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Pin name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetName()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the net segment item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the net segment item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the net segment item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the connection net item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the net segment item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the net segment item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the net segment item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.HasAttribute(name)

	def DisplayAttributeValue(self, name:str) -> int:
		"""
		Displays an attribute value from the net segment item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text item identifier displaying the attribute value |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.DisplayAttributeValue(name)

	def DisplayAttributeValueAt(self, name:str, sheetid:int, x:float, y:float) -> int:
		"""
		Displays an attribute value from the net segment item at a specific position.

		:param name: Name of the attribute
		:type str: str
		:param sheetid: Sheet identifier to display the attribute value on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text item identifier displaying the attribute value |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.DisplayAttributeValueAt(name, sheetid, x, y)

	def SetAttributeVisibility(self, name:str, onoff:int) -> int:
		"""
		Sets the visibility status of all text items representing the net segment item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param onoff: Indicates whether the text items' visibility status should be shown or hidden. If > 0, the text items are shown. If 0, the text items are hidden
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items changed |
		| 0 | No text items found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetAttributeVisibility(name, onoff)

	def GetCoreCount(self) -> int:
		"""
		Gets the number of conductors and wires contained within the net segment item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors and wires |
		| 0 | No conductors and wires or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetCoreCount()

	def GetCoreIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductors contained in the current net segment item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of conductors passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors in ids |
		| 0 | No conductors or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetCoreIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSignalName(self) -> str:
		"""
		Gets the net segment item's signal name or bus name value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Pin item's signal name or bus name value is supplied |
		| "&lt;Empty&gt;" | No signal name or bus name value or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetSignalName()

	def SetSignalName(self, name:str) -> int:
		"""
		Sets the net segment item's signal name or bus name value.

		:param name: New signal name or bus name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signal name or bus name applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetSignalName(name)

	def Highlight(self) -> int:
		"""
		Highlights the net segment item on the sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of net segment item elements highlighted |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.Highlight()

	def GetNetId(self) -> int:
		"""
		Gets the identifier of the net item in which the net segment item is located.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Net item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetNetId()

	def GetLength(self) -> float:
		"""
		Gets the length of the net segment item.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Net segment length value |
		| 0.0 | Net segment has no length value or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLength()

	def SetLength(self, newlen:float) -> float:
		"""
		Sets the length of the net segment item.

		:param newlen: New length value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | If the net segment is on a panel, the new length value |
		| 0.0 | If the net segment is on a panel, the new length was not applied, If the new segment is not on a panel, the new length might be applied |
		| -1.0 | No project open |
		| -2.0 | No net segment item is set |
		| -3.0 | newlen is a negative value |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLength(newlen)

	def GetLineSegments(self) -> tuple[int, int, tuple[int,...], tuple[int,...], tuple[int,...]]:
		"""
		Gets the net segment's connect lines' location and type within the project.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | shtid | int | Identifier of the sheet passed out of the function |
		| 2 | xarr | tuple[int,...] | Array of connect line positions on the x-axis passed out of the function |
		| 3 | yarr | tuple[int,...] | Array of connect line positions on the y-axis passed out of the function |
		| 4 | PointTypArr | tuple[int,...] | Array of values indicating if the position represents a spline passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of values within the arrays |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, shtid, xarr, yarr, PointTypArr = self._obj.GetLineSegments(dummy, dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		PointTypArr = PointTypArr[1:] if type(PointTypArr) == tuple and len(PointTypArr) > 0 else tuple()
		return ret, shtid, xarr, yarr, PointTypArr

	def SetLineColour(self, newcol:int) -> int:
		"""
		Sets the net segment item's line color.

		:param newcol: Line color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Line color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLineColour(newcol)

	def SetLineStyle(self, newstl:int) -> int:
		"""
		Sets the line style of a net segment item.

		:param newstl: Line style value. See Line Styles for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Line style value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLineStyle(newstl)

	def SetLineWidth(self, newwid:float) -> int:
		"""
		Gets the  line width of the net segment item.

		:param newwid: New line width value
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Line width value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLineWidth(newwid)

	def SetLineLevel(self, newlev:int) -> int:
		"""
		Sets the value of the net segment's connect line display level.

		:param newlev: New level value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New level value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLineLevel(newlev)

	def GetSymbolCount(self) -> int:
		"""
		Gets the number of symbols contained in the net segment item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbols |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetSymbolCount()

	def GetSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductor indicator symbols placed on the net segment item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbols passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbols in ids |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAssignedOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of options assigned to the net segment item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of assigned options passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned option identifiers are supplied or an error occurred |
		| -1 | An option contains a Boolean expression or an alias |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetAssignedOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Delete(self) -> int:
		"""
		Deletes the net segment item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &lt; 0 | Identifier of the current net segment item |
		| 0 | Net segment item deleted |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.Delete()

	def GetBusName(self) -> str:
		"""
		Gets the net segment item's bus name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Net segment item's bus name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetBusName()

	def SetBusName(self, name:str) -> int:
		"""
		Sets the net segment item's bus name.

		:param name: Name of the new bus
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Bus name applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetBusName(name)

	def GetSignalIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the bus net segment item's signals.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of signals passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 10 | Number of items in ids |
		| 0 | No signals found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetSignalIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsView(self) -> int:
		"""
		Indicates whether the net segment item connects connector view symbols.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Net segment item connects connector view symbols |
		| 0 | Net segment item does not connect connector view symbols or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.IsView()

	def IsBus(self) -> int:
		"""
		Indicates whether the net segment item is part of a bus.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Net segment is part of a bus |
		| 0 | Net segment is not part of a bus |
		| 4711 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.IsBus()

	def IsPanelPath(self) -> int:
		"""
		Indicates whether the net segment item is on a panel sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Net segment is on a panel sheet |
		| 0 | Net segment is not on a panel sheet or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.IsPanelPath()

	def IsOffline(self) -> int:
		"""
		Indicates the net segment item's checked out status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Net segment item is checked out |
		| 0 | Net segment item is not checked out or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.IsOffline()

	def GetLevel(self) -> int:
		"""
		Gets the net segments item's display level value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Display level value supplied |
		| 0 | Error occurred |
		| -1 | Lines of the net segment are placed on different levels |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLevel()

	def SetLevel(self, level:int) -> int:
		"""
		Sets the net segments item's display level value.

		:param level: Display level value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Previous display level value |
		| 0 | Error occurred |
		| -1 | Lines of the net segment are placed on different levels |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLevel(level)

	def GetLineStyle(self) -> int:
		"""
		Gets the line style of a net segment item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Line style value, See Line Styles for possible values |
		| 0 or &gt;47 | 0 or &gt;47 |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLineStyle()

	def GetLineColour(self) -> int:
		"""
		Gets the net segment item's line color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Line color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or the operation has failed |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLineColour()

	def GetLineWidth(self) -> float:
		"""
		Gets the line width of the net segment item.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Line width value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLineWidth()

	def GetNodeIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the net segment item's nodes.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of nodes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No node identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetNodeIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetConnectedSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbols connected to the net segment item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbols passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No symbols are found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetConnectedSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetEffectiveDirection(self) -> tuple[int, int, int]:
		"""
		Gets the source and destination items at either end of the current net segment item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | fromID | int | Identifier of the source item passed out of the function |
		| 2 | toID | int | Identifier of the destination item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Source and destination item identifiers supplied |
		| 0 | Error has occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetEffectiveDirection(dummy, dummy)

	def SetEffectiveDirection(self, toID:int) -> int:
		"""
		Sets the effective direction of the current net segment item.

		:param toID: Identifier of the destination item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Source and destination item identifiers supplied |
		| 0 | Error has occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetEffectiveDirection(toID)

	def GetManufacturingLength(self) -> float:
		"""
		Gets the manufacturing length of the net segment.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Net segment manufacturing length value |
		| 0.0 | Net segment manufacturing length value or anrror occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetManufacturingLength()

	def SetManufacturingLength(self, newval:float) -> float:
		"""
		Sets the manufacturing length of the net segment.

		:param newval: Manufacturing length value
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| 0.0 | Manufacturing length is assigned |
		| -1.0 | No project open |
		| -2.0 | No valid net segment item set |
		| -3.0 | newval is a negative value |
		| -10.0 | No Formboard license available, Available since v2018-19.44, v2019-20.24 and v2020-21.10 |
		| -11.0 | Net segment is not on a Formboard sheet, Available since v2018-19.44, v2019-20.24 and v2020-21.10 |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetManufacturingLength(newval)

	def GetSchemaLength(self) -> float:
		"""
		Gets the displayed length of the net segment item.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Net segment display length value |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetSchemaLength()

	def AdjustSchemaLength(self) -> int:
		"""
		Sets the manufacturing length to the net segment item's displayed length.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Net segment manufacturing length set to the display length value |
		| 0 | No project open, no net segment item set or net segment is not on a formboard sheet |
		| -1 | Displayed length is the same as the manufacturing length |
		| -2 | Manufacturing length is 0 |
		| -3 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.AdjustSchemaLength()

	def GetRotation(self, anchorid:int) -> float:
		"""
		Gets the net segment item's rotation value.

		:param anchorid: Item identifier representing the start position to get the rotation value from
		:type int: int

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0..&lt; 360.0 | Rotation value in degrees supplied |
		| 0.0 | Rotation value supplied or an error occurred |
		| -1 | anchorid is invalid |
		| -2 | anchorid is not connected to the net segment item |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetRotation(anchorid)

	def SetRotation(self, anchorid:int, newval:float) -> float:
		"""
		Sets the net segment item's rotation value.

		:param anchorid: Item identifier representing the start position to set the rotation value to
		:type int: int
		:param newval: New rotation value
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| 0.0..&lt; 360.0 | Previous rotation value in degrees |
		| -1 | No project open, no net segment item set or no E³.formboard license available |
		| -2 | Net segment is not placed on a formboard |
		| -3 | anchorid is invalid |
		| -4 | Rotation is not possible |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetRotation(anchorid, newval)

	def GetOuterDiameter(self) -> float:
		"""
		Gets the outer diameter of wires and cables running through the net segment item.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Outer diameter |
		| 0.0 | No outer diameter or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetOuterDiameter()

	def GetAssignedOptionExpressions(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options) assigned to the net segment item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressions(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def SetOptionExpressions(self, expressions:list[str]) -> int:
		"""
		Replaces the net segment item's option expressions.

		:param expressions: Array of strings of all option names, Boolean expressions or alias names of Boolean expressions
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.12
		"""
		return self._obj.SetOptionExpressions(expressions)

	def GetConnectLineIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the connect lines of the net segment item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbols passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No symbols are found or an error occurred |

		Available since E3.series TLB-Versions: 9.22
		"""
		dummy=0
		ret, ids = self._obj.GetConnectLineIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAssignedOptionExpressionsEx(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options)  including variant instances assigned to the net segment item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.23
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressionsEx(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def SetSchemaLength(self, newval:float) -> float:
		"""
		Sets the displayed length of the net segment item.

		:param newval: New displayed length value
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| 1.0 | Net segment display length value |
		| 0.0 | One of the following errors occurred:, No project openNo project item setNo E³.formboard license availableNet segment item is not located on a formboard sheet |
		| -1.0 | Displayed length is the same as the manufacturing length |
		| -2.0 | newval is invalid |
		| -3.0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetSchemaLength(newval)

	def GetTranslatedSignalName(self) -> str:
		"""
		Gets the net segment item's translated signal name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Translated name |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetTranslatedSignalName()

	def SetIgnoreForCablingTable(self, set:bool) -> int:
		"""
		Sets whether the net segment item's connection is ignored in the cabling table list.

		:param set: Ignore cabling table list status to apply. If 1, the connection is ignored. If 0, the connection is taken into account
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previously the net segment was ignored |
		| 0 | Previously the net segment was taken into account |
		| -1 | No project open |
		| -2 | No net segment item set |
		| -3 | Net segment is part of a formboard or topology net |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.SetIgnoreForCablingTable(set)

	def GetBundleSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of identifers of the net segment's bundle symbols.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of bundle symbol identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of bundle symbol identifiers in ids |
		| 0 | Net segment has no bundle symbols or an error occurred |

		Available since E3.series TLB-Versions: 20.20
		"""
		dummy=0
		ret, ids = self._obj.GetBundleSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsBusbar(self) -> int:
		"""
		Indicates if the net segment item belongs to a busbar.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Net segment belongs to a busbar |
		| 0 | Net segment does not belong to a busbar |
		| -1 | No project open or no net segment item set |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.IsBusbar()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current net segment item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current net segment item |
		| "&lt;Empty&gt;" | No net segment item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a net segment item as the current item.

		:param gid: Global identifier value
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current net segment item |
		| "&lt;Empty&gt;" | No net segment item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IOptionInterface--------------------
class Option:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Option. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current option or variant item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current option or variant item identifier |
		| 0 | No option or variant item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets an option or variant item as the current item.

		:param id: Unique value identifying an option or variant item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current option or variant item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the option item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Option name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the option item's name.

		:param name: New option name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Option item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def Create(self, name:str, parent:int=0, position:int=0, before:int=0) -> int:
		"""
		Creates a new option item.

		:param name: Name of the new option item
		:type str: str
		:param parent: Identifier of the parent option item. If 0, the new option item is placed on the top level., defaults to 0
		:type int: int
		:param position: Identifier of an existing option item the new option item should be placed next to. If 0, the new option item is placed at the end., defaults to 0
		:type int: int
		:param before: Indicates which side of the. position. option item the new option item is placed. If 1, the new option item is placed before position option item. If 0, the new option item is placed after position option item., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the created option item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(name, parent, position, before)

	def Delete(self, _del:int) -> int:
		"""
		Deletes the option item from the project.

		:param _del: Indicates whether items the option item is exclusively assigned to are also removed from the project.. If 0, the items are not deleted. If greater than 0, the items are deleted
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred, Current option or variant item identifier |
		| 0 | Option was deleted, no project open or no option item set |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete(_del)

	def Search(self, name:str) -> int:
		"""
		Searches for a option item matching the name.

		:param name: Name of the option
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Found option identifier |
		| 0 | No option was found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Search(name)

	def IsActive(self) -> int:
		"""
		Get the active status of the current option item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option is active |
		| 0 | Option is inactive or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsActive()

	def Activate(self) -> int:
		"""
		Activates the option item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option was activated |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Activate()

	def Deactivate(self) -> int:
		"""
		Deactivates the option item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option was activated |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Deactivate()

	def Add(self, devi:int) -> int:
		"""
		Adds an item to the current option item.

		:param devi: Identifier of item to add to the option
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item added to the option item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Add(devi)

	def Remove(self, devi:int) -> int:
		"""
		Removes an item from the current option item.

		:param devi: Identifier of item to remove from the option
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item removed from the option item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Remove(devi)

	def GetFullName(self) -> str:
		"""
		Gets the option item's full name and path from the top level of the project tree.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Option name and path |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetFullName()

	def GetInternalName(self) -> str:
		"""
		Gets the option item's name without text token translations.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Option name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetInternalName()

	def GetFullInternalName(self) -> str:
		"""
		Gets the option item's full name and path from the top level of the project tree without text token translations.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Option name and path |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetFullInternalName()

	def GetDescription(self) -> str:
		"""
		Gets the option item's description.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Option description |
		| "&lt;Empty&gt;" | No description or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetDescription()

	def GetInternalDescription(self) -> str:
		"""
		Gets the option item's description without text token translations.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Option description |
		| "&lt;Empty&gt;" | No description or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetInternalDescription()

	def IsAssignable(self) -> int:
		"""
		Indicates whether the current option item can be assigned to other items.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option item can be assigned to other items |
		| 0 | Option item cannot be assigned to other items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsAssignable()

	def IsVariant(self) -> int:
		"""
		Indicates whether the current option item is a variant item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option item is a variant item |
		| 0 | Option item is not a variant item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsVariant()

	def GetOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the child option items belonging to the current option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of option items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of option items in ids |
		| 0 | No child option items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetParentOptionId(self) -> int:
		"""
		Gets the identifier of the parent option item of the current option item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of parent option item |
		| 0 | Current option item is on the top level or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetParentOptionId()

	def GetPropertyFlags(self) -> int:
		"""
		Gets the option item's visibility and activation property values.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Property values |
		| 0 | No property values or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPropertyFlags()

	def SetPropertyFlags(self, newflags:int) -> int:
		"""
		Sets the option item's visibility and activation property values.

		:param newflags: Property value flags
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=0 | Property values |
		| =1 | No property values or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetPropertyFlags(newflags)

	def GetAllDeviceIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the devices assigned to the option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of device identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of device identifiers in ids |
		| 0 | No devices assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAllDeviceIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSheetIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the sheets assigned to the option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of sheet identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of sheet items in ids |
		| 0 | No sheets assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSheetIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbols assigned to the current option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbols passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbols in ids |
		| 0 | No symbols assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the wires and connector pins assigned to the option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of wire and connector pin identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of wires and connector pins in ids |
		| 0 | No wires and connector pins or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetNetSegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the net segments assigned to the option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of net segments passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of net segments in ids |
		| 0 | No net segments or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetNetSegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGraphIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the graphics assigned to the option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of graphic identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No graphic items assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetGraphIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetTextIds(self, txttyp:int=0, search_string:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the text items assigned to the current option item.

		:param txttyp: Text type filter. Only text item identifiers with this Text Type will be supplied. All text item identifiers are supplied if txttyp is 0. The default value 0, defaults to 0
		:type int: int
		:param search_string: Text value filter, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of text item identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No text items assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetTextIds(dummy, txttyp, search_string)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def MoveTo(self, vari:int, before:int=0) -> int:
		"""
		Moves the option item to another position within the project tree.

		:param vari: Identifier of an existing option item in the project tree indicating the position where the current option item is to be moved. If 0, current option item is placed at the end
		:type int: int
		:param before: Indicates where the option is placed relative to the. vari. option., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option item moved |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.MoveTo(vari, before)

	def Highlight(self) -> bool:
		"""
		Highlights the items assigned to the current option item.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Error occurred |
		| 0 | Items assigned to the current option item highlighted |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Highlight()

	def ResetHighlight(self) -> bool:
		"""
		Removed highlights from the items assigned to the current option item.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Error occurred |
		| 0 | Highlights removed from items assigned to the current option item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.ResetHighlight()

	def IsHighlighted(self) -> bool:
		"""
		Indicates whether the items assigned to the current option item are highlighted.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Items assigned to the current option item are highlighted |
		| 0 | Items assigned to the current option item are not highlighted or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsHighlighted()

	def SetDescription(self, newval:str) -> int:
		"""
		Sets the option item's description.

		:param newval: New option description value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current option item identifier |
		| 0 | New description value applied or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.SetDescription(newval)

	def GetFieldIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the fields assigned to the option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of field item identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of field item identifiers in ids |
		| 0 | No field items assigned or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		dummy=0
		ret, ids = self._obj.GetFieldIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsPackage(self) -> int:
		"""
		Indicates whether the current option item is an option package item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option item is an option package item |
		| 0 | Option item is not an option package item or an error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.IsPackage()

	def IsConfiguration(self) -> int:
		"""
		Indicates whether the current option item is a configuration item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option item is a configuration item |
		| 0 | Option item is not a configuration item or an error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.IsConfiguration()

	def CreatePackage(self, name:str, posId:int, before:int) -> int:
		"""
		Creates a new option package item.

		:param name: Name of the new option package item
		:type str: str
		:param posId: Identifier of an existing option item the new option package item should be placed next to
		:type int: int
		:param before: Indicates which side of the. position. option item the new option package item is placed. If 1, the new option package item is placed before position option item. If 0, the new option package item is placed after position option item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the created option package item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.CreatePackage(name, posId, before)

	def CreateConfiguration(self, name:str, posId:int, before:int) -> int:
		"""
		Creates a new configuration item.

		:param name: Name of the new configuration item
		:type str: str
		:param posId: Identifier of an existing option item the new configuration item should be placed next to
		:type int: int
		:param before: Indicates which side of the. position. option item the new configuration item is placed. If 1, the new configuration item is placed before position option item. If 0, the new configuration item is placed after position option item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the created configuration item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.CreateConfiguration(name, posId, before)

	def ActivateAndLockOtherActivations(self) -> int:
		"""
		Activates the current configuration or package item and locks other options.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Configuration or package was activated and other options locked |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.ActivateAndLockOtherActivations()

	def UnLockOtherActivations(self) -> int:
		"""
		Unlocks option items if they were prevented from being activated as a result of the current configuration or package item being activated.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option items are unlocked |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.UnLockOtherActivations()

	def GetInclusiveOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the inclusive option items belonging to the current option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of option items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of option items in ids |
		| 0 | No inclusive option items or an error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		dummy=0
		ret, ids = self._obj.GetInclusiveOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetExclusiveOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the exclusive option items belonging to the current option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of option items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of option items in ids |
		| 0 | No exclusive option items or an error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		dummy=0
		ret, ids = self._obj.GetExclusiveOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def AssignToID(self, parentid:int, how:int) -> int:
		"""
		Assigns the current option item to a parent option item.

		:param parentid: Identifier of the parent option item
		:type int: int
		:param how: Indicates the exclusivity of the option item. If 1, the option item is added to the inclusive list of the parent option item. if 2, the option item is added to the exclusive list of the parent option item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Current option item added to the parent option item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.AssignToID(parentid, how)

	def UnassignFromID(self, parentid:int) -> int:
		"""
		Unassigns the current option item from its parent option.

		:param parentid: Identifier of the parent option item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Current option item removed from the parent option item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.UnassignFromID(parentid)

	def SetXMLVariantID(self, name:str) -> int:
		"""
		Sets the GUID to the current option or variant item.

		:param name: Globally unique identifier value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Globally unique identifier applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.30
		"""
		return self._obj.SetXMLVariantID(name)

	def GetXMLVariantID(self) -> str:
		"""
		Gets the globally unique identifier of the current option or variant item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current option or variant item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.30
		"""
		return self._obj.GetXMLVariantID()

	def IsObjectActive(self, objid:int) -> int:
		"""
		Indicates whether an item is active.

		:param objid: Item identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is active |
		| 0 | Item is inactive or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.IsObjectActive(objid)

	def IsHarnessFamily(self) -> int:
		"""
		Indicates whether the current option item is a cable harness family item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option item is a cable harness family item |
		| 0 | Option item is not a cable harness family item or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.IsHarnessFamily()

	def IsHarnessDerivative(self) -> int:
		"""
		Indicates whether the current option item is a cable harness derivative item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option item is a cable harness derivative item |
		| 0 | Option item is not a cable harness derivative item or an error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.IsHarnessDerivative()

	def CreateHarnessFamily(self, name:str, posId:int, before:int) -> int:
		"""
		Creates a new cable harness family item.

		:param name: Name of the new cable harness family item
		:type str: str
		:param posId: Identifier of an existing option item the new cable harness family item should be placed next to
		:type int: int
		:param before: Indicates which side of the. position. option item the new cable harness family item is placed. If 1, the new cable harness family item is placed before position option item. If 0, the new cable harness family item is placed after position option item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the created cable harness family item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.CreateHarnessFamily(name, posId, before)

	def CreateHarnessDerivative(self, name:str, posId:int, before:int) -> int:
		"""
		Creates a new cable harness derivative item.

		:param name: Name of the new cable harness derivative item
		:type str: str
		:param posId: Identifier of an existing cable harness derivative item the new cable harness derivative item should be placed next to or a harness family item the new cable harness should be placed within
		:type int: int
		:param before: Indicates which side of the. position. existing cable harness derivative item the new cable harness derivative item is placed. If 1, the new cable harness derivative item is placed before position option item. If 0, the new cable harness derivative item is placed after position option item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the created cable harness derivative item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 15.00
		"""
		return self._obj.CreateHarnessDerivative(name, posId, before)

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the option item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the option item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.DeleteAttribute(name)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the option item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the option item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the option item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.GetAttributeValue(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the option item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.HasAttribute(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the option item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def Sort(self, sortMode:int=0) -> int:
		"""
		Sorts the identifiers of the child option items of the current option item.

		:param sortMode: Sorting method., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Option items sorted |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.Sort(sortMode)

	def GetAllDeviantAttributeIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes with deviating values of items assigned to the current option item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 17.04
		"""
		dummy=0
		ret, ids = self._obj.GetAllDeviantAttributeIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsInUseByObject(self) -> int:
		"""
		Indicates whether any items are assigned to the option item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Items are assigned to the option item |
		| 0 | No items are assigned to the option item or an error occurred |

		Available since E3.series TLB-Versions: 19.15
		"""
		return self._obj.IsInUseByObject()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the option or variant item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the option or variant item |
		| "&lt;Empty&gt;" | No option or variant item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets an option or variant as the current item.

		:param gid: Global identifier value of an option or variant item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current option or variant item |
		| "&lt;Empty&gt;" | No option or variant item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current option or variant item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current option or variant item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets an option or variant as the current item.

		:param guid: Globally unique identifier value of an option or variant item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current option or variant item |
		| "&lt;Empty&gt;" | No option or variant item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- IOutlineInterface--------------------
class Outline:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Outline. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current contour item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current contour item identifier |
		| 0 | No contour item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a contour item as the current item.

		:param id: Unique value identifying an contour item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current contour item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the contour item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Contour name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def GetType(self) -> int:
		"""
		Gets a value representing the type of the contour item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type value of the contour item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetType()

	def GetPosition(self) -> tuple[int, float, float, float]:
		"""
		Gets the position of the contour item in the panel.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Position on the x-axis passed out of the function |
		| 2 | y | float | Position on the y-axis passed out of the function |
		| 3 | z | float | Position on the z-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Position values are supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetPosition(dummy, dummy, dummy)

	def GetRadius(self) -> float:
		"""
		Gets the radius of the drill hole contour item.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Drill hole radius value |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetRadius()

	def GetHeight(self) -> float:
		"""
		Gets the height of the contour item in the panel.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Contour height value |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetHeight()

	def GetSymbolCount(self) -> int:
		"""
		Gets the number of symbols representing the contour item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbols |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSymbolCount()

	def GetSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the symbols representing the contour item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of symbol identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbol items in ids |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetTypeId(self) -> int:
		"""
		Gets a value representing the type of the contour item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type value of the contour item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTypeId()

	def GetPathEx(self) -> tuple[int, tuple[float,...], tuple[float,...], tuple[float,...]]:
		"""
		Gets a polygon of points representing the path of a restriction or cutout contour.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xarr | tuple[float,...] | Array of positions on the x-axis passed out of the function |
		| 2 | yarr | tuple[float,...] | Array of positions on the y-axis passed out of the function |
		| 3 | zarr | tuple[float,...] | Array of positions on the z-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of points in xarr, yarr and zarr |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		ret, xarr, yarr, zarr = self._obj.GetPathEx(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		zarr = zarr[1:] if type(zarr) == tuple and len(zarr) > 0 else tuple()
		return ret, xarr, yarr, zarr

	def UseInE3CutOut(self) -> bool:
		"""
		Indicates whether the contour item is used in E³.CutOut.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Contour item is used in E³.CutOut |
		| 0 | Contour item is not used in E³.CutOut or an error occurred |

		Available since E3.series TLB-Versions: 14.11
		"""
		return self._obj.UseInE3CutOut()

	def SetUseInE3CutOut(self, set:bool) -> int:
		"""
		Sets the contour item's usage status value in E³.CutOut.

		:param set: Value to assign to the. <span style="font-style: italic;color: #ff0000;">E³.</span>CutOut. usage. If 1, the contour is used in E³.CutOut. If 0, the contour is not used in E³.CutOut
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | set value is successfully assigned |
		| -1 | No project open |
		| -2 | No item is set |
		| -3 | Item is not a contour |
		| -4 | Contour is not a cutout or drill hole |
		| -5 | Device is locked, Available since v2020-21.01 |
		| -6 | E³.redliner or E³.viewer mode is active, Available since v2025-25.20 and v2023-24.50 |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.SetUseInE3CutOut(set)

	def UseCutOutGraphic(self) -> bool:
		"""
		Indicates whether an alternative graphic is used in E³.CutOut for the contour item.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 2 | Alternative graphic usage is active however E³.CutOut usage is inactive |
		| 1 | Alternative graphic usage and E³.CutOut usage are active |
		| 0 | Alternative graphic usage is inactive |
		| -1 | No project open |
		| -2 | No item is set |
		| -3 | Item is not an contour |
		| -4 | Contour item is not a cutout type or drill hole type |

		Available since E3.series TLB-Versions: 21.10
		"""
		return self._obj.UseCutOutGraphic()

	def SetUseCutOutGraphic(self, set:bool) -> int:
		"""
		Sets the usage status of an alternative graphic being used for the contour item in E³.CutOut.

		:param set: Value to assign to the alternative graphic usage. If 1, the alternative graphic is used in E³.CutOut. If 0, the alternative graphic is not used in E³.CutOut
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | set is successfully assigned |
		| -1 | No project open |
		| -2 | No contour item is set |
		| -3 | Item is not a contour |
		| -4 | Contour is not a cutout or drill hole |
		| -5 | Device is locked |
		| -6 | Contour is dynamic |
		| -7 | E³.redliner or E³.viewer mode is activeAvailable since v2025-25.20 and v2023-24.50 |

		Available since E3.series TLB-Versions: 21.10
		"""
		return self._obj.SetUseCutOutGraphic(set)

	def IsThreadedHole(self) -> int:
		"""
		Indicates if the contour item represents a threaded drill hole.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Drill hole is threaded |
		| 0 | Drill hole is not threaded |
		| -1 | No project open |
		| -2 | No contour item set |
		| -3 | Set item is not a contour |
		| -4 | Contour is not a drill hole |

		Available since E3.series TLB-Versions: 21.31
		"""
		return self._obj.IsThreadedHole()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current contour item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current contour item |
		| "&lt;Empty&gt;" | No contour item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a contour item as the current item.

		:param gid: Global identifier value
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current contour item |
		| "&lt;Empty&gt;" | No contour item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def SetCreatesThreadedHole(self, set:bool, flags:int=0) -> int:
		"""
		Sets whether a drill hole should be threaded on the model's carrier object in  E³.CutOut.

		:param set: Value indicating whether the drill hole is threaded. If 1, the  drill hole is threaded. If 0, the  drill hole is not threaded
		:type bool: bool
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | set is successfully assigned |
		| -1 | No project open |
		| -2 | No contour item is set |
		| -3 | Item is not a contour |
		| -4 | Contour is not a drill hole |
		| -5 | Device is locked |
		| -6 | Drill hole is dynamically created on a mounting rail |
		| -7 | Drill hole defined in the component is read-only |
		| -8 | E³.redliner or E³.viewer mode is active |
		| -9 | flags is invalid |

		Available since E3.series TLB-Versions: 26.00
		"""
		return self._obj.SetCreatesThreadedHole(set, flags)

# -------------------- ISheetInterface--------------------
class Sheet:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Sheet. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetName(self) -> str:
		"""
		Gets the sheet item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Sheet name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> str:
		"""
		Sets the sheet item's name.

		:param name: New name of sheet
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | New sheet name |
		| "&lt;Empty&gt; | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def Remove(self) -> int:
		"""
		Closes the sheet item display window.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheet item display closed |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Remove()

	def Visible(self) -> int:
		"""
		Indicates if the sheet item is displayed.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet is displayed |
		| 0 | Sheet is not displayed or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Visible()

	def Delete(self) -> int:
		"""
		Deletes the current sheet from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheet item was deleted |
		| -1 | Error occurred when checking out the sheet in multi-user |
		| -2 | After trying to check out in multi-user, the sheet is still offline |
		| -3 | Sheet is locked |
		| -4 | Sheet is a region |
		| -5 | Sheet is checked out as read only |
		| -6 | Sheet does not exist, Available since v2015-16.19 |
		| -7 | Sheet is checked out by another multi-user client, Available since v2016-17.33 and v2017-18.13 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def GetDrawingArea(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the coordinate values of the sheet item's drawing area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xmin | float | Lower left position on the x-axis passed out of the function |
		| 2 | ymin | float | Lower left position on the y-axis passed out of the function |
		| 3 | xmax | float | Upper right position on the x-axis passed out of the function |
		| 4 | ymax | float | Upper right position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Drawing area coordinates supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetDrawingArea(dummy, dummy, dummy, dummy)

	def GetTextCount(self) -> int:
		"""
		Gets the number of the sheet's text items.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items |
		| 0 | No text items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTextCount()

	def GetTextIds(self, txttyp:int=0, search_string:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's text identifiers.

		:param txttyp: Text type filter. Only text identifiers with this Text Type will be supplied. All text identifiers are supplied if txttyp is 0., defaults to 0
		:type int: int
		:param search_string: Text value filter. Default value is "<Empty>", defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of text identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No text items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetTextIds(dummy, txttyp, search_string)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSymbolCount(self) -> int:
		"""
		Gets the number of the sheet item's symbols.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbol items |
		| 0 | No symbol items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSymbolCount()

	def GetSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's symbol identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of symbol identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Create(self, modi:int, name:str, symbol:str, position:int, before:int) -> int:
		"""
		Creates a new schematic sheet item.

		:param modi: Identifier of the hierarchical block or module to which the sheet should be inserted
		:type int: int
		:param name: Name of the sheet
		:type str: str
		:param symbol: Border symbol of the sheet
		:type str: str
		:param position: Identifier of the sheet to use as the insertion position of the new sheet in the project
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 0, the new sheet is inserted after position or as the last sheet if position is 0. If 1, the new sheet is inserted before position or as the first sheet if position is 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(modi, name, symbol, position, before)

	def CreatePanel(self, modi:int, name:str, symbol:str, position:int, before:int, refx:float, refy:float, refscale:float) -> int:
		"""
		Creates a new panel sheet item.

		:param modi: Value should be 0 since v2018-19.00
		:type int: int
		:param name: Name of the sheet
		:type str: str
		:param symbol: Border symbol of the sheet
		:type str: str
		:param position: Identifier of the sheet to use as the insertion position of the new sheet in the project
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 0, the new sheet is inserted after position or as the last sheet if position is 0. If 1, the new sheet is inserted before position or as the first sheet if position is 0
		:type int: int
		:param refx: Position on the x-axis of the lower left corner of the sheet region
		:type float: float
		:param refy: Position on the y-axis of the lower left corner of the sheet region
		:type float: float
		:param refscale: Scale factor of the sheet region
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreatePanel(modi, name, symbol, position, before, refx, refy, refscale)

	def GetId(self) -> int:
		"""
		Gets the identifier of the current sheet item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current sheet item identifier |
		| 0 | No sheet item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a sheet as the current item.

		:param id: Unique value identifying a sheet item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current sheet item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def Search(self, modi:int, name:str) -> int:
		"""
		Searches for an item and sets it as the current sheet item.

		:param modi: Identifier of the hierarchical block or module in which to search
		:type int: int
		:param name: Name of the sheet item to search for
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the found sheet item |
		| 0 | No sheet item was found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Search(modi, name)

	def GetGraphCount(self) -> int:
		"""
		Gets the number of the sheet item's graphics.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of graphic items |
		| 0 | No graphics items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetGraphCount()

	def GetGraphIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet item's graphic item identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of graphic identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No graphic items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetGraphIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetModuleCount(self) -> int:
		"""
		Gets the number of the sheet item's modules.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of module items |
		| 0 | No module items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetModuleCount()

	def GetModuleIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's module identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of module identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No modules or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetModuleIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def PlacePart(self, name:str, version:str, x:float, y:float, rot:float) -> int:
		"""
		Places a subcircuit on the sheet item.

		:param name: Subcircuit name from the database or a path to an e3p file
		:type str: str
		:param version: Subcircuit version number from the database
		:type str: str
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Not used
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 9 | Incompatible subcircuit file verison |
		| 8..4 | Error occurred |
		| 3 | Invalid name or version |
		| 1..2 | Error occurred |
		| 0 | Subcircuit placed on sheet |
		| -1 | Subcircuit consists of several sheets and the project setting Ignore sheet border is set (located in Settings → Placement → Import → Sheets) |
		| -2 | Subcircuit contains sheets and the project setting Ignore sheet border is not set (located in Settings → Placement → Import → Sheets) |
		| -3 | Subcircuit is already placed or other objects are placed at x, y |
		| -4 | Sheet is locked, Available since v2018-19.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PlacePart(name, version, x, y, rot)

	def GetNetSegmentCount(self) -> int:
		"""
		Gets the number of the sheet item's net segments.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of net segment items |
		| 0 | No net segment items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetNetSegmentCount()

	def GetNetSegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's net segment identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of net segment identifiers |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No net segments or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetNetSegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def PrintOut(self, scale:float) -> int:
		"""
		Prints the sheet on the currently defined printer.

		:param scale: Re-sizes the print output
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet data sent to printer |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PrintOut(scale)

	def Export(self, format:str, version:int, file:str, flags:int=0) -> int:
		"""
		Exports the sheet content to a file.

		:param format: Format of the export file
		:type str: str
		:param version: Version of the export file
		:type int: int
		:param file: Path and name of the file
		:type str: str
		:param flags: Export options for VRML export files., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheet is exported |
		| 1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Export(format, version, file, flags)

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the sheet item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the sheet item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the sheet item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default value is "<Empty>", defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the sheet item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the sheet item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the sheet item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the sheet item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def ImportDXF(self, filename:str, scale:float, x:float, y:float, rot:int, font:str, flags:int=0) -> int:
		"""
		Imports a DXF or DWG format file's contents on a new sheet in the project.

		:param filename: Import file's path and name
		:type str: str
		:param scale: Scale of the contents
		:type float: float
		:param x: Placement of the file contents on the x-axis of the sheet. Ignored if bit value of 0x0010 is not included in flags
		:type float: float
		:param y: Placement of the file contents on the y-axis of the sheet. Ignored if bit value of 0x0010 is not included in flags
		:type float: float
		:param rot: Not used
		:type int: int
		:param font: Font name of texts on the sheet
		:type str: str
		:param flags: Flag bit values specifying how the file contents should be imported. x0000, defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the sheet |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.ImportDXF(filename, scale, x, y, rot, font, flags)

	def GetAssignment(self) -> str:
		"""
		Gets the sheet item's higher level assignment value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Higher level assignment value supplied |
		| "&lt;Empty&gt;" | No higher level assignment value supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAssignment()

	def SetAssignment(self, newass:str) -> int:
		"""
		Sets the sheet item's higher level assignment value.

		:param newass: New higher level assignment value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Higher level assignment value applied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAssignment(newass)

	def GetLocation(self) -> str:
		"""
		Gets the sheet item's location value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Location value supplied |
		| "&lt;Empty&gt;" | No location value supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetLocation()

	def SetLocation(self, newloc:str) -> int:
		"""
		Sets the sheet item's location value.

		:param newloc: New location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Location value applied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetLocation(newloc)

	def SetCompleteName(self, newnam:str, newass:str, newloc:str) -> int:
		"""
		Sets all elements of the sheet item's  item designation.

		:param newnam: New sheet name value
		:type str: str
		:param newass: New assignment value
		:type str: str
		:param newloc: New location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Item designation values applied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetCompleteName(newnam, newass, newloc)

	def GetOpenNetsegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's net segment identifiers open at one or both ends.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of net segment identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No open net segments or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetOpenNetsegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetEmbeddedSheetIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the identifiers of embedded sheet regions on the sheet item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of embedded sheet identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No embedded sheet items are present or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetEmbeddedSheetIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsEmbedded(self) -> int:
		"""
		Indicates if the sheet item is embedded.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item is embedded |
		| 0 | Sheet item is not embedded or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsEmbedded()

	def GetParentSheetId(self) -> int:
		"""
		Gets the sheet's parent sheet identifier.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Parent sheet identifier |
		| 0 | No parent sheet or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetParentSheetId()

	def ExportImageArea(self, format:str, version:int, file:str, xl:float, yl:float, xr:float, yr:float, width:int, height:int, clrdepth:int, gray:int, dpiX:int, dpiY:int, compressionmode:int) -> int:
		"""
		Exports an area of the sheet item as an image.

		:param format: Image's format
		:type str: str
		:param version: Not used
		:type int: int
		:param file: Path and file name of the export image
		:type str: str
		:param xl: Position of the left border
		:type float: float
		:param yl: Position of the lower border
		:type float: float
		:param xr: Position of the right border
		:type float: float
		:param yr: Position of the upper border
		:type float: float
		:param width: Image width
		:type int: int
		:param height: Image height
		:type int: int
		:param clrdepth: Color bit depth of the image
		:type int: int
		:param gray: Specifies the color mode. If 1, the image is grayscale. If 0, the image is color
		:type int: int
		:param dpiX: Image resolution on the x-axis
		:type int: int
		:param dpiY: Image resolution on the y-axis
		:type int: int
		:param compressionmode: Compression mode depending on. format
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Image is exported |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.ExportImageArea(format, version, file, xl, yl, xr, yr, width, height, clrdepth, gray, dpiX, dpiY, compressionmode)

	def ExportImage(self, format:str, version:int, file:str, dpi:int=0, compressionmode:int=0) -> int:
		"""
		Exports the sheet item as an image.

		:param format: The image's format
		:type str: str
		:param version: Not used
		:type int: int
		:param file: Path and file name of the export image
		:type str: str
		:param dpi: Image resolution. 6, defaults to 0
		:type int: int
		:param compressionmode: Compression mode depending on. format., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Image is exported |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.ExportImage(format, version, file, dpi, compressionmode)

	def ExportImageSelection(self, format:str, version:int, file:str, percentage:int, width:int, height:int, clrdepth:int, gray:int, dpiX:int, dpiY:int, compressionmode:int) -> int:
		"""
		Exports the selected items on a sheet item as an image.

		:param format: Image's format
		:type str: str
		:param version: Currently not used
		:type int: int
		:param file: Path and file name of the export image
		:type str: str
		:param percentage: Specifies the. width. and. height. units. If 1, the  units are in percent. If 0, the units are in pixels
		:type int: int
		:param width: Image width
		:type int: int
		:param height: Image height
		:type int: int
		:param clrdepth: Color bit depth of the image
		:type int: int
		:param gray: Specifies the color mode. If 1, the image is grayscale. If 0, the image is color
		:type int: int
		:param dpiX: Image resolution on the x-axis
		:type int: int
		:param dpiY: Image resolution on the y-axis
		:type int: int
		:param compressionmode: Compression mode depending on. format
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Image is exported |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.ExportImageSelection(format, version, file, percentage, width, height, clrdepth, gray, dpiX, dpiY, compressionmode)

	def SetFormat(self, sym:str, rot:str="") -> int:
		"""
		Sets the  name of the sheet item's sheet format symbol.

		:param sym: Sheet format symbol name
		:type str: str
		:param rot: Sheet rotation and mirroring value. Default value is "<Empty>", defaults to ""
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &lt;&gt; 0 | Format value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetFormat(sym, rot)

	def GetFormat(self) -> str:
		"""
		Gets the  name of the sheet item's sheet format symbol.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Characteristic value supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetFormat()

	def GetPanelConnectionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the panel sheet's connection identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of connection identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No connections or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPanelConnectionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetRedlinedGraphTextIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's redlined text item identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of redlined text identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No redlined text items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetRedlinedGraphTextIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetRedlinedGraphIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's redlined graphic item identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of redlined graphic identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No redlined graphic items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetRedlinedGraphIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetNetIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's local connection net identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of connection net identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No connection nets or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetNetIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def CreateBoard(self, brdi:int, name:str, symbol:str, position:int, before:int, refx:float, refy:float, refscale:float) -> int:
		"""
		Creates a new board sheet item.

		:param brdi: Identifier of the base sheet on which the board sheet should be inserted
		:type int: int
		:param name: Name of the sheet
		:type str: str
		:param symbol: Border symbol of the sheet
		:type str: str
		:param position: Identifier of the sheet to use as the insertion position of the new sheet in the project
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 0, the new sheet is inserted after position or as the last sheet if position is 0. If 1, the new sheet is inserted before position or as the first sheet if position is 0
		:type int: int
		:param refx: Position on the x-axis of the lower left corner of the sheet region
		:type float: float
		:param refy: Position on the y-axis of the lower left corner of the sheet region
		:type float: float
		:param refscale: Scale factor of the sheet region
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateBoard(brdi, name, symbol, position, before, refx, refy, refscale)

	def GetInsideSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's embedded sheet region's symbol identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of symbol identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsideSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetInsideGraphIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's shared sheet regions' graphic item identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of graphic identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No graphic items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsideGraphIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetInsidePanelConnectionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's shared sheet regions' panel connection identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of connection identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No connection or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetInsidePanelConnectionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def ShareWithID(self, id:int) -> int:
		"""
		Shares the sheet region item with another sheet.

		:param id: Identifier of the sheet item to share with
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the base sheet item the sheet region has been shared with |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.ShareWithID(id)

	def IsShared(self) -> int:
		"""
		Indicates if the sheet item is a panel share.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item is shared |
		| 0 | Sheet item is not shared or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsShared()

	def GetBaseId(self) -> int:
		"""
		Gets the shared panel parent sheet of the current panel share sheet item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Shared panel parent sheet identifier |
		| 0 | No shared panel parent sheet or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBaseId()

	def GetSharedIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of sheet identifiers shared with the sheet item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of sheet identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No sheets are shared or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSharedIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetVisibleArea(self) -> tuple[int, tuple[float,...], tuple[float,...], tuple[float,...], tuple[float,...]]:
		"""
		Gets the sheet item's visible area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xmin | tuple[float,...] | Lower left position on the x-axis passed out of the function |
		| 2 | ymin | tuple[float,...] | Lower left position on the y-axis passed out of the function |
		| 3 | xmax | tuple[float,...] | Upper right position on the x-axis passed out of the function |
		| 4 | ymax | tuple[float,...] | Upper right position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Area values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, xmin, ymin, xmax, ymax = self._obj.GetVisibleArea(dummy, dummy, dummy, dummy)
		xmin = xmin[1:] if type(xmin) == tuple and len(xmin) > 0 else tuple()
		ymin = ymin[1:] if type(ymin) == tuple and len(ymin) > 0 else tuple()
		xmax = xmax[1:] if type(xmax) == tuple and len(xmax) > 0 else tuple()
		ymax = ymax[1:] if type(ymax) == tuple and len(ymax) > 0 else tuple()
		return ret, xmin, ymin, xmax, ymax

	def SetVisibleArea(self, xmin:float, ymin:float, xmax:float, ymax:float) -> int:
		"""
		Sets the sheet item's visible area.

		:param xmin: Lower left position on the x-axis
		:type float: float
		:param ymin: Lower left position on the y-axis
		:type float: float
		:param xmax: Upper right position on the x-axis
		:type float: float
		:param ymax: Upper right position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Area values applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetVisibleArea(xmin, ymin, xmax, ymax)

	def IsReadOnly(self) -> bool:
		"""
		Indicates the sheet item's read-only status.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item is read-only |
		| 0 | Sheet item is read-writable or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsReadOnly()

	def GetSchematicTypes(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's schematic type values.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | types | tuple[int,...] | Array of schematic type values passed out of the function. See Schematic for possible values |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in types |
		| 0 | All schematic types are assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, types = self._obj.GetSchematicTypes(dummy)
		types = types[1:] if type(types) == tuple and len(types) > 0 else tuple()
		return ret, types

	def GetAssignedOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of options assigned to the sheet item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of assigned options passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned option identifiers are supplied or an error occurred |
		| -1 | An option contains a Boolean expression or an alias |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAssignedOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def ToGrid(self, xpos:float, ypos:float) -> tuple[int, float, float]:
		"""
		Adjusts  position values to snap to the sheet item's grid.

		:param xpos: Sheet position value on the x-axis. A tentative value is passed into the function. The snapped grid value is passed out of the function
		:type float: float
		:param ypos: Sheet position value on the y-axis. A tentative value is passed into the function. The snapped grid value is passed out of the function
		:type float: float

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xpos | float | Sheet position value on the x-axis. A tentative value is passed into the function. The snapped grid value is passed out of the function |
		| 2 | ypos | float | Sheet position value on the y-axis. A tentative value is passed into the function. The snapped grid value is passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Grid position values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.ToGrid(xpos, ypos)

	def MoveTo(self, position:int, before:int=0) -> int:
		"""
		Moves the sheet item to a position in the project tree.

		:param position: Identifier of the item to use as the insertion position of the new sheet in the project tree
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 1, the sheet is inserted before position. If 0, the sheet is inserted after position., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item moved |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.MoveTo(position, before)

	def GetContentModified(self) -> bool:
		"""
		Indicates whether the sheet item's content has been modified.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet content has been modified |
		| 0 | Sheet content not modified or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetContentModified()

	def SetContentModified(self, value:int) -> bool:
		"""
		Sets the sheet item's modification values.

		:param value: Indicates whether the modification values should be reset or updated. If 0, the modification values are reset to indicate the sheet has not been modified. If <> 0, the modification values are updated
		:type int: int

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Modification value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetContentModified(value)

	def GetDimensionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the sheet item's dimension items.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of dimension identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No dimension items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetDimensionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def SetPanelRegion(self, xoff:float, yoff:float, scale:float) -> int:
		"""
		Sets the panel sheet item's panel region offset and scale values.

		:param xoff: Panel region offset on the x-axis
		:type float: float
		:param yoff: Panel region offset on the y-axis
		:type float: float
		:param scale: Panel region scale ratio
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Ambiguous status |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetPanelRegion(xoff, yoff, scale)

	def GetPanelRegion(self) -> tuple[int, float, float, float]:
		"""
		Gets the panel sheet item's panel region offset and scale values.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xoff | float | Panel region offset on the x-axis passed out of the function |
		| 2 | yoff | float | Panel region offset on the y-axis passed out of the function |
		| 3 | scale | float | Panel region scale ratio passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Ambiguous status |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetPanelRegion(dummy, dummy, dummy)

	def ImportDGN(self, filename:str, scale:float, x:float, y:float, rot:int, font:str, flags:int) -> int:
		"""
		Imports a DGN format file's contents on a new sheet in the project.

		:param filename: Import file's path and name
		:type str: str
		:param scale: Scale of the contents
		:type float: float
		:param x: Placement of the file contents on the x-axis of the sheet
		:type float: float
		:param y: Placement of the file contents on the y-axis of the sheet
		:type float: float
		:param rot: Not used
		:type int: int
		:param font: Font name of texts on the sheet
		:type str: str
		:param flags: Not used
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the sheet |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.ImportDGN(filename, scale, x, y, rot, font, flags)

	def CreateFormboard(self, modi:int, name:str, symbol:str, position:int, before:int, flags:int) -> int:
		"""
		Creates a new formboard sheet item.

		:param modi: Value should be 0 since v2018-19.00
		:type int: int
		:param name: Name of the sheet
		:type str: str
		:param symbol: Border symbol of the sheet
		:type str: str
		:param position: Identifier of the sheet to use as the insertion position of the new sheet in the project
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 0, the new sheet is inserted after position or as the last sheet if position is 0. If 1, the new sheet is inserted before position or as the first sheet if position is 0
		:type int: int
		:param flags: Not used
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateFormboard(modi, name, symbol, position, before, flags)

	def IsFormboard(self) -> bool:
		"""
		Indicates if the item is a formboard type sheet.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a formboard type sheet |
		| 0 | Item is not a formboard type sheet or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsFormboard()

	def GetOwner(self) -> str:
		"""
		Gets the sheet item's owner.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Owner value supplied |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetOwner()

	def IsClipboardPart(self) -> int:
		"""
		Indicates if the sheet item is on the clipboard.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item is on the clipboard |
		| 0 | Sheet item is not on the clipboard or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsClipboardPart()

	def SetCharacteristic(self, characteristic:str) -> int:
		"""
		Sets the sheet item's characteristic value.

		:param characteristic: New characteristic value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Characteristic value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetCharacteristic(characteristic)

	def GetCharacteristic(self) -> str:
		"""
		Gets the sheet item's characteristic value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Characteristic value supplied |
		| "&lt;Empty&gt;" | No characteristic value supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCharacteristic()

	def GetValidCharacteristics(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets an array of the possible characteristics for the sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | characteristics | tuple[str,...] | Array of valid characteristics passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in characteristics |
		| 0 | No valid characteristics or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, characteristics = self._obj.GetValidCharacteristics(dummy)
		characteristics = characteristics[1:] if type(characteristics) == tuple and len(characteristics) > 0 else tuple()
		return ret, characteristics

	def GetHyperlinkTextIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet item's texts with hyperlinks.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of text identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No text items with hyperlinks or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetHyperlinkTextIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGroupIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet item's group item identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of group identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No group items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetGroupIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsTerminalPlan(self) -> int:
		"""
		Indicates if the sheet item is a terminal plan.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item is a terminal plan |
		| 0 | Sheet item is not a terminal plan or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsTerminalPlan()

	def GetWorkingArea(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the coordinate values of the sheet item's working area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xmin | float | Lower left position on the x-axis passed out of the function |
		| 2 | ymin | float | Lower left position on the y-axis passed out of the function |
		| 3 | xmax | float | Upper right position on the x-axis passed out of the function |
		| 4 | ymax | float | Upper right position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Working area coordinates supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetWorkingArea(dummy, dummy, dummy, dummy)

	def SetSchematicTypes(self, types:list[int]) -> int:
		"""
		Sets an array of the sheet's schematic type values.

		:param types: Array of schematic type values
		passed into the function. See Schematic for possible values **Available enum type:** e3series.types.SchematicType.
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of newly assigned schematic types |
		| 0 | All schematic types are assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.SetSchematicTypes(types)
		return ret[0]

	def GetAssignedOptionExpressions(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options) assigned to the sheet item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressions(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def Is2DView(self) -> int:
		"""
		Indicates if  the sheet item defines a 2D view.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item defines a 2D view |
		| 0 | Sheet item does not define a 2D view or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.Is2DView()

	def CreateFunctionalDesign(self, modi:int, name:str, symbol:str, position:int, before:int, flags:int) -> int:
		"""
		Creates a new functional design sheet item.

		:param modi: Identifier of the hierarchical block to which the sheet should be inserted
		:type int: int
		:param name: Name of the sheet
		:type str: str
		:param symbol: Border symbol of the sheet
		:type str: str
		:param position: Identifier of the sheet to use as the insertion position of the new sheet in the project
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 0, the new sheet is inserted after position or as the last sheet if position is 0. If 1, the new sheet is inserted before position or as the first sheet if position is 0
		:type int: int
		:param flags: Not used
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.CreateFunctionalDesign(modi, name, symbol, position, before, flags)

	def IsFunctionalDesign(self) -> bool:
		"""
		Indicates if the item is a functional design type sheet.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a functional design type sheet |
		| 0 | Item is not a functional design type sheet or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.IsFunctionalDesign()

	def GetGetterOptionHandlingMode(self) -> int:
		"""
		Indicates how identifiers and values supplied from e3Sheet object functions are handled according to the active variants/options.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..2 | Option handling mode |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetGetterOptionHandlingMode()

	def SetGetterOptionHandlingMode(self, mode:int) -> int:
		"""
		Sets how identifiers and values supplied from e3Sheet object functions are handled according to the active variants/options.

		:param mode: Option handling mode
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..2 | Previously set option handling mode |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetGetterOptionHandlingMode(mode)

	def SetOptionExpressions(self, expressions:list[str]) -> int:
		"""
		Replaces the sheet item's option expressions.

		:param expressions: Array of strings of all option names, Boolean expressions or alias names of Boolean expressions
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.12
		"""
		return self._obj.SetOptionExpressions(expressions)

	def SetCompleteNameEx(self, newnam:str, newass:str, newloc:str, onlygiven:bool) -> int:
		"""
		Sets all elements of the sheet item's  item designation.

		:param newnam: New sheet name value
		:type str: str
		:param newass: New assignment value
		:type str: str
		:param newloc: New location value
		:type str: str
		:param onlygiven: Indicates whether empty values in. newnam. ,. newass. and. newloc. should be ignored. If 1, empty values are ignored. If 0, empty values are set
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Error occurred |
		| 0 | Item designation values applied or an error occurred |

		Available since E3.series TLB-Versions: 9.20
		"""
		return self._obj.SetCompleteNameEx(newnam, newass, newloc, onlygiven)

	def GetAssignedOptionExpressionsEx(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options)  including variant instances assigned to the sheet item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.23
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressionsEx(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def GetRegionArea(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the sheet item's region area.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xmin | float | First region position on the x-axis passed out of the function |
		| 2 | ymin | float | First region position on the y-axis passed out of the function |
		| 3 | xmax | float | Second region position on the x-axis passed out of the function |
		| 4 | ymax | float | Second region position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Region area values supplied |
		| 0 | No region area or an error occurred |

		Available since E3.series TLB-Versions: 9.30
		"""
		dummy=0
		return self._obj.GetRegionArea(dummy, dummy, dummy, dummy)

	def CreateTopology(self, modi:int, name:str, symbol:str, position:int, before:int, refscale:float) -> int:
		"""
		Creates a new topology sheet item.

		:param modi: Value should be 0 since v2018-19.00
		:type int: int
		:param name: Name of the sheet
		:type str: str
		:param symbol: Border symbol of the sheet
		:type str: str
		:param position: Identifier of the sheet to use as the insertion position of the new sheet in the project
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 0, the new sheet is inserted after position or as the last sheet if position is 0. If 1, the new sheet is inserted before position or as the first sheet if position is 0
		:type int: int
		:param refscale: Scale factor of the sheet region
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.CreateTopology(modi, name, symbol, position, before, refscale)

	def IsTopology(self) -> bool:
		"""
		Indicates if  the item is a topology type sheet.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a topology type sheet |
		| 0 | Item is not a topology type sheet or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsTopology()

	def LockObject(self) -> int:
		"""
		Locks the sheet item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheet item has been locked |
		| -1 | Error occurred |
		| -2 | Sheet item is already locked |
		| -3 | Error occurred |
		| -1009 | No project open or no sheet item set |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.LockObject()

	def UnlockObject(self, password:str) -> int:
		"""
		Unlocks the sheet item.

		:param password: Password needed to unlock the sheet
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheet item has been unlocked |
		| -1 | Error occurred |
		| -2 | Sheet item is already unlocked |
		| -3 | Error occurred |
		| -1009 | No project open or no sheet item set |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.UnlockObject(password)

	def IsLocked(self) -> int:
		"""
		Indicates the sheet item's lock status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item is locked |
		| 0 | Sheet item is not locked or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsLocked()

	def IsPanel(self) -> int:
		"""
		Indicates if the item is a panel type sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a panel type sheet |
		| 0 | Item is not a panel type sheet or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsPanel()

	def SetTopologyRegion(self, xoff:float, yoff:float, scale:float) -> int:
		"""
		Sets the topology sheet item's topology region offset and scale values.

		:param xoff: Topology region offset on the x-axis
		:type float: float
		:param yoff: Topology region offset on the y-axis
		:type float: float
		:param scale: Topology region scale ratio
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Ambiguous status |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetTopologyRegion(xoff, yoff, scale)

	def GetTopologyRegion(self) -> tuple[int, float, float, float]:
		"""
		Gets the topology sheet item's offset and scale values.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xoff | float | Topology region offset on the x-axis passed out of the function |
		| 2 | yoff | float | Topology region offset on the y-axis passed out of the function |
		| 3 | scale | float | Topology region scale ratio passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Ambiguous status |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		return self._obj.GetTopologyRegion(dummy, dummy, dummy)

	def CreateFormboardEx(self, modi:int, name:str, symbol:str, position:int, before:int, refx:float, refy:float, refscale:float) -> int:
		"""
		Creates a new formboard sheet item.

		:param modi: Value should be 0 since v2018-19.00
		:type int: int
		:param name: Name of the sheet
		:type str: str
		:param symbol: Border symbol of the sheet
		:type str: str
		:param position: Identifier of the sheet to use as the insertion position of the new sheet in the project
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 0, the new sheet is inserted after position or as the last sheet if position is 0. If 1, the new sheet is inserted before position or as the first sheet if position is 0
		:type int: int
		:param refx: Position on the x-axis of the lower left corner of the sheet region
		:type float: float
		:param refy: Position on the y-axis of the lower left corner of the sheet region
		:type float: float
		:param refscale: Scale factor of the sheet region
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.30
		"""
		return self._obj.CreateFormboardEx(modi, name, symbol, position, before, refx, refy, refscale)

	def SetSheetRegion(self, xoff:float, yoff:float, scale:float) -> int:
		"""
		Sets the sheet item's region offset and scale values.

		:param xoff: Sheet region offset on the x-axis
		:type float: float
		:param yoff: Sheet region offset on the y-axis
		:type float: float
		:param scale: Sheet region scale ratio
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheet region supplied |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 10.30
		"""
		return self._obj.SetSheetRegion(xoff, yoff, scale)

	def GetSheetRegion(self) -> tuple[int, float, float, float]:
		"""
		Gets the sheet item's region offset and scale values.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xoff | float | Sheet region offset on the x-axis passed out of the function |
		| 2 | yoff | float | Sheet region offset on the y-axis passed out of the function |
		| 3 | scale | float | Sheet region scale ratio passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheet region supplied |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 10.30
		"""
		dummy=0
		return self._obj.GetSheetRegion(dummy, dummy, dummy)

	def GetInsideNetSegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's shared sheet regions' net segment identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of net segment identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No net segments or an error occurred |

		Available since E3.series TLB-Versions: 10.30
		"""
		dummy=0
		ret, ids = self._obj.GetInsideNetSegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsParentSheet(self, flags:int) -> int:
		"""
		Indicates if the item is a parent sheet containing embedded/region sheets of a specific type.

		:param flags: Indicates type of parent sheet
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet item is a parent sheet of the specified sheet types |
		| 0 | Item is not a formboard type sheet or an error occurred |

		Available since E3.series TLB-Versions: 10.30
		"""
		return self._obj.IsParentSheet(flags)

	def PlacePartEx(self, name:str, version:str, flags:int, x:float, y:float, rot:float) -> int:
		"""
		Places a subcircuit on the sheet item.

		:param name: Subcircuit name from the database or a path to an e3p file
		:type str: str
		:param version: Subcircuit version number from the database
		:type str: str
		:param flags: Placement option flags
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Not used
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 9 | Incompatible subcircuit file verison |
		| 8..4 | Error occurred |
		| 3 | Invalid name or version |
		| 1..2 | Error occurred |
		| 0 | Subcircuit placed on sheet |
		| -1 | Subcircuit consists of several sheets and the project setting Ignore sheet border is set (located in Settings → Placement → Import → Sheets) |
		| -2 | Subcircuit contains sheets and the project setting Ignore sheet border is not set (located in Settings → Placement → Import → Sheets) |
		| -3 | Subcircuit is already placed or other objects are placed at x , y |
		| -4 | Sheet is locked, Available since v2018-19.00 |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.PlacePartEx(name, version, flags, x, y, rot)

	def CheckoutReadonlyReferencedSheets(self) -> int:
		"""
		Checks out all sheets referenced using the cross-reference of the sheet item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheets are checked out |
		| -1 | No multi-user project |
		| -2 | No sheet item set |
		| -3 | Sheet is not being accessed |
		| -4 | Multi-user project could not be updated |
		| -5 | Transaction could not be started |
		| -12 | No sheets available for check out |
		| -13 | Sheets could not be checked out |

		Available since E3.series TLB-Versions: 11.80
		"""
		return self._obj.CheckoutReadonlyReferencedSheets()

	def GetOwners(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the sheet's owners.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | owners | tuple[int,...] | Array of owners passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in owners |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 17.16
		"""
		dummy=0
		ret, owners = self._obj.GetOwners(dummy)
		owners = owners[1:] if type(owners) == tuple and len(owners) > 0 else tuple()
		return ret, owners

	def AlignObjects(self, reference:int, ids:list[int], mode:int) -> int:
		"""
		Aligns items on the sheet item.

		:param reference: Identifier of the item to align against
		:type int: int
		:param ids: Array of identifiers of items to align
		:type list[int]: list[int]
		:param mode: Alignment mode to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Items were aligned |
		| -1 | No project open |
		| -2 | No sheet item set |
		| -3 | reference is invalid |
		| -4 | ids contains invalid items |
		| -5 | mode is invalid |
		| -6 | Sheet is locked, Available since v2018-19.00 |
		| -7 | Item to align or panel position is locked, Available since v2023-24.30 |

		Available since E3.series TLB-Versions: 17.70
		"""
		return self._obj.AlignObjects(reference, ids, mode)

	def DisplayEx(self, flags:int=0) -> int:
		"""
		Opens the current sheet item in a new window.

		:param flags: Display options flags., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheet displayed |
		| -1 | Error occurred |
		| -2 | flags is a combination of mutually exclusive values, Available from v2025-25.00 |
		| -3 | Current sheet is not a panel sheet and flags values requires a panel, Available from v2025-25.00 |
		| -4 | Current sheet is not visible due to inactive options, Available from v2025-25.00 |
		| -5 | Current panel sheet is a 3D world, Available from v2025-25.22 |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.DisplayEx(flags)

	def IsLockedByAccessControl(self) -> int:
		"""
		Indicates the sheet item's E³PluginAccessControl lock status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 3 | Object is locked and also indirectly by the sheet |
		| 2 | Object is locked indirectly by the sheet |
		| 1 | Object is locked |
		| 0 | Object is unlocked or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.IsLockedByAccessControl()

	def CreateTopologyEx(self, modi:int, name:str, symbol:str, position:int, before:int, refx:float, refy:float, refscale:float, flags:int=0) -> int:
		"""
		Creates a new topology sheet item.

		:param modi: Identifier of the hierarchical block or module to which the sheet should be inserted
		:type int: int
		:param name: Name of the sheet
		:type str: str
		:param symbol: Border symbol of the sheet
		:type str: str
		:param position: Identifier of the sheet to use as the insertion position of the new sheet in the project
		:type int: int
		:param before: Indicates whether to insert the sheet before or after. position. If 0, the new sheet is inserted after position or as the last sheet if position is 0. If 1, the new sheet is inserted before position or as the first sheet if position is 0
		:type int: int
		:param refx: Position on the x-axis of the lower left corner of the sheet region
		:type float: float
		:param refy: Position on the y-axis of the lower left corner of the sheet region
		:type float: float
		:param refscale: Scale factor of the sheet region
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.CreateTopologyEx(modi, name, symbol, position, before, refx, refy, refscale, flags)

	def GetProduct(self) -> str:
		"""
		Gets the sheet item's product name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Product name supplied |
		| "&lt;Empty&gt;" | No product name or an error occurred |

		Available since E3.series TLB-Versions: 20.00, 19.34
		"""
		return self._obj.GetProduct()

	def SetProduct(self, newproduct:str) -> int:
		"""
		Sets the sheet item's product name.

		:param newproduct: New sheet product name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New sheet product name applied |
		| 0 | Error occurred |
		| -1 | No project open, no sheet item is set or no value set |
		| -3 | Sheet locked |
		| -4 | Sheet is a region |
		| -8 | IEC 61346 setting is inactive |
		| -9 | Sheet is a terminal sheet |

		Available since E3.series TLB-Versions: 20.00, 19.34
		"""
		return self._obj.SetProduct(newproduct)

	def GetDXFSize(self, filename:str, font:str) -> tuple[int, float, float, float]:
		"""
		Gets the size and scale for the sheet item based on the file contents and font.

		:param filename: Path and name of the DXF or DWG file. Since v2022-23.53 and v2023-24.22, the path and name of a DGN file is also accepted
		:type str: str
		:param font: Name of font to apply
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | dx | float | Required width passed out of the function |
		| 2 | dy | float | Required height passed out of the function |
		| 3 | scale | float | Maximum possible scale for the sheet and. font. passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Size and maximum scale supplied |
		| 0 | Size and scale could not be determined |
		| -1 | No project open or no sheet item set |
		| -2 | filename is invalid |
		| -3 | font is invalid |
		| -4 | Error reading DXF file, Available since v2022-23.61 and v2023-24.31 |

		Available since E3.series TLB-Versions: 20.00, 19.01
		"""
		dummy=0
		return self._obj.GetDXFSize(filename, font, dummy, dummy, dummy)

	def SelectIds(self, symbolIds:list[int]) -> tuple[int, list[int]]:
		"""
		Selects symbols on the sheet.

		:param symbolIds: Array of symbol items to select
		:type list[int]: list[int]

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | selectedIds | list[int] | Array of actually selected symbol items |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbols on the sheet are selected |
		| -1 | No project open |
		| -2 | symbolIds is empty |
		| -3 | Previous selection could not be cleared |
		| -4 | No symbol in symbolIds could be selected or no sheet item set |
		| -5 | Not all symbols in symbolIds could be selected |

		Available since E3.series TLB-Versions: 21.00
		"""
		dummy=0
		ret, selectedIds = self._obj.SelectIds(symbolIds, dummy)
		selectedIds = [] if selectedIds is None else selectedIds
		selectedIds = selectedIds[1:] if type(selectedIds) == list and len(selectedIds) > 0 else []
		return ret, selectedIds

	def IsReadOnlyInProject(self) -> bool:
		"""
		Gets the sheet's project read-only status.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet's project read-only status is on |
		| 0 | Sheet's project read-only status is off or an error occurred |

		Available since E3.series TLB-Versions: 21.00, 20.70, 20.03, 19.31
		"""
		return self._obj.IsReadOnlyInProject()

	def SetReadOnlyInProject(self, readonlyproject:bool) -> int:
		"""
		Sets the sheet's project read-only status.

		:param readonlyproject: Project's read-only status. 1 turns the project read-only status to on. 0 turns the project read-only status to off
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet's project read-only status is changed |
		| -1 | No project open or sheet item set |
		| -2 | Sheet is locked |

		Available since E3.series TLB-Versions: 21.00, 20.70, 20.03, 19.31
		"""
		return self._obj.SetReadOnlyInProject(readonlyproject)

	def GetEmbeddedObjectIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the embedded objects on the sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of embedded objects identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No embedded objects are present or an error occurred |

		Available since E3.series TLB-Versions: 21.00
		"""
		dummy=0
		ret, ids = self._obj.GetEmbeddedObjectIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Get2DViewSheetDisplayOnlyBorder(self) -> int:
		"""
		Gets the 2D view's display only border flag value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display only border value is on |
		| 0 | Display only border value is off |
		| -1 | No project is open |
		| -2 | No sheet item is set |
		| -3 | Sheet is not a 2D view |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.Get2DViewSheetDisplayOnlyBorder()

	def Set2DViewSheetDisplayOnlyBorder(self, display_border:bool) -> int:
		"""
		Sets the 2D view's display only border flag value.

		:param display_border: Value to assign to display only border
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | display_border is successfully assigned |
		| -1 | No project open |
		| -2 | No sheet item is set |
		| -3 | Sheet is not a 2D view |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.Set2DViewSheetDisplayOnlyBorder(display_border)

	def GetSlotIdsFrom2DView(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the 2D view sheet item's slot identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of slot identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No slots or an error occurred |

		Available since E3.series TLB-Versions: 21.12
		"""
		dummy=0
		ret, ids = self._obj.GetSlotIdsFrom2DView(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the sheet item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the sheet item |
		| "&lt;Empty&gt;" | No sheet item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a sheet as the current item.

		:param gid: Global identifier value of a sheet item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current sheet item |
		| "&lt;Empty&gt;" | No sheet item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current sheet item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current sheet item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a sheet as the current item.

		:param guid: Globally unique identifier value of a sheet item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current sheet item |
		| "&lt;Empty&gt;" | No sheet item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

	def GetDisplayModelViewAtPosition(self, flags:int=0) -> tuple[int, float]:
		"""
		Gets the display model view at position values for a 2D view sheet item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | view_distance | float | Position value |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display model view at position is active |
		| 0 | Display model view at position is inactive |
		| -1 | No project open |
		| -2 | No sheet item set |
		| -3 | flags is invalid |
		| -4 | Sheet is not a 2D view |

		Available since E3.series TLB-Versions: 24.00
		"""
		dummy=0
		return self._obj.GetDisplayModelViewAtPosition(dummy, flags)

	def SetDisplayModelViewAtPosition(self, display_model_view_at_position:bool, view_distance:float, flags:int=0) -> int:
		"""
		Sets the display model view at position values for a 2D view sheet item.

		:param display_model_view_at_position: Display model view at position value active status. If 0, the display model view is set to inactive. If not 0, display model view is set to active
		:type bool: bool
		:param view_distance: Position value
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display model view at position values are applied |
		| -1 | No project open |
		| -2 | No sheet item set |
		| -3 | flags is invalid |
		| -4 | Sheet is not a 2D view |
		| -5 | Sheet does not contain any slots |
		| -6 | view_distance is invalid |

		Available since E3.series TLB-Versions: 24.00
		"""
		return self._obj.SetDisplayModelViewAtPosition(display_model_view_at_position, view_distance, flags)

	def Get2DViewDisplayInOriginalOrientation(self) -> int:
		"""
		Gets the 2D view's display in original orientation flag value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display in original orientation value is on |
		| 0 | Display in original orientation value is off |
		| -1 | No project is open |
		| -2 | No sheet item is set |
		| -3 | Sheet is not a 2D view |

		Available since E3.series TLB-Versions: 24.00
		"""
		return self._obj.Get2DViewDisplayInOriginalOrientation()

	def Get2DViewSheetDisplaySlots(self) -> int:
		"""
		Indicates whether slots are displayed on the 2D view sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Slots are displayed |
		| 0 | Slots are not displayed |
		| -1 | No project is open |
		| -2 | No sheet item is set |
		| -3 | Sheet is not a 2D view |

		Available since E3.series TLB-Versions: 25.00
		"""
		return self._obj.Get2DViewSheetDisplaySlots()

	def Set2DViewSheetDisplaySlots(self, display_slots:bool) -> int:
		"""
		Sets whether slots are displayed on the 2D view sheet.

		:param display_slots: Value to assign to display only border
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | display_slots is successfully assigned |
		| -1 | No project open |
		| -2 | No sheet item is set |
		| -3 | Sheet is not a 2D view |

		Available since E3.series TLB-Versions: 25.00
		"""
		return self._obj.Set2DViewSheetDisplaySlots(display_slots)

# -------------------- ISignalInterface--------------------
class Signal:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Signal. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current signal item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current signal item identifier |
		| 0 | No signal item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a signal item as the current item.

		:param id: Unique value identifying a signal item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current signal item identifier |
		| 0 | No signal item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the signal item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Signal name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the signal item's name.

		:param name: New name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signal name applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the signal item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the signal item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the signal item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the signal item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the signal item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the signal item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the signal item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def SetAttributeVisibility(self, name:str, onoff:int) -> int:
		"""
		Sets the visibility status of all text items representing the signal item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param onoff: Indicates whether the text items' visibility status should be shown or hidden. If > 0, the text items are shown. If 0, the text items are hidden
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items changed |
		| 0 | No text items found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeVisibility(name, onoff)

	def Search(self, name:str) -> int:
		"""
		Searches for a signal item.

		:param name: Name of the signal item to search for
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the found signal item |
		| 0 | No signal found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Search(name)

	def GetNetSegmentCount(self) -> int:
		"""
		Gets the number of net segments carrying the signal item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of net segments |
		| 0 | No net segments or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetNetSegmentCount()

	def GetNetSegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the net segments carrying the signal item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of net segments passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of net segments in ids |
		| 0 | No net segments or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetNetSegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetPinCount(self) -> int:
		"""
		Gets the number of pins and wires carrying the signal item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pins and wires |
		| 0 | No pins or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPinCount()

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the pins and wires carrying the signal item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins and wires passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No pin and wire identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Highlight(self) -> int:
		"""
		Highlights the signal item occurrences on sheets.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of signal occurrences |
		| 0 | Signal is not placed on a sheet or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Highlight()

	def FindPanelPath(self) -> int:
		"""
		Automatically routes the connections carrying the signal.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signal was automatically routed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.FindPanelPath()

	def DisplayAttributeValueAt(self, name:str, sheetid:int, x:float, y:float) -> int:
		"""
		Displays an attribute value from the signal item at a specific position.

		:param name: Internal name of the attribute
		:type str: str
		:param sheetid: Sheet identifier to display the attribute value on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text item identifier displaying the attribute value |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DisplayAttributeValueAt(name, sheetid, x, y)

	def GetDefaultWires(self) -> tuple[int, tuple[str,...], tuple[str,...]]:
		"""
		Gets the default wire types of the signal item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | wiregroups | tuple[str,...] | Array of wire group name values passed out of the function |
		| 2 | wirenames | tuple[str,...] | Array of wire name values passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in wiregroups and wirenames |
		| 0 | No default wires or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, wiregroups, wirenames = self._obj.GetDefaultWires(dummy, dummy)
		wiregroups = wiregroups[1:] if type(wiregroups) == tuple and len(wiregroups) > 0 else tuple()
		wirenames = wirenames[1:] if type(wirenames) == tuple and len(wirenames) > 0 else tuple()
		return ret, wiregroups, wirenames

	def SetDefaultWires(self, wiregroups:list[str], wirenames:list[str]) -> int:
		"""
		Sets the default wire types of the signal item.

		:param wiregroups: Array of wire group name values passed into the function
		:type list[str]: list[str]
		:param wirenames: Array of wire name values passed into the function
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of wire types applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.SetDefaultWires(wiregroups, wirenames)
		return ret[0]

	def GetCoreCount(self) -> int:
		"""
		Gets the number of conductors and wires carrying the signal item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors and wires |
		| 0 | No conductors and wires or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCoreCount()

	def GetCoreIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductors and wires carrying the signal item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of conductors and wires passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors and wires in ids |
		| 0 | No conductors and wires or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetCoreIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSignalClassId(self) -> int:
		"""
		Gets the identifier of the signal item's signal class.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the signal class item |
		| 0 | Signal not assigned to a signal class or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetSignalClassId()

	def Create(self, name:str) -> int:
		"""
		Creates a signal item.

		:param name: New signal name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the signal item |
		| 0 | Error occurred |
		| -1 | name is invalid |

		Available since E3.series TLB-Versions: 10.30
		"""
		return self._obj.Create(name)

	def GetTranslatedName(self) -> str:
		"""
		Gets the signal item's translated name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Translated name |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetTranslatedName()

	def HighlightCoreLogicLinesOfSignal(self) -> int:
		"""
		Highlights the conductor logic lines of a signal.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Always returned |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.HighlightCoreLogicLinesOfSignal()

	def SetNameFormat(self, name:str) -> int:
		"""
		Sets the signal item's formatted name.

		:param name: New formatted name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Formatted name applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.SetNameFormat(name)

	def GetNameFormat(self) -> str:
		"""
		Gets the signal item's formatted name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Formatted name |
		| "&lt;Empty&gt;" | No formatted name or an error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.GetNameFormat()

	def SetRecalcFormattedName(self, recalculate:int) -> int:
		"""
		Sets the status indicating whether the formatted signal name is automatically recalculated.

		:param recalculate: Indicates whether the formatted signal name is automatically recalculated. If 1, the formatted signal name is automatically recalculated. If 0, the formatted signal name is not automatically recalculated
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New automatic recalculation status applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.SetRecalcFormattedName(recalculate)

	def GetRecalcFormattedName(self) -> int:
		"""
		Gets the status indicating whether the formatted signal name is automatically recalculated.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Formatted signal name is automatically recalculated |
		| 0 | Formatted signal name is not automatically recalculated |

		Available since E3.series TLB-Versions: 16.00
		"""
		return self._obj.GetRecalcFormattedName()

	def AddDefaultWireEx(self, wiregroup:str, wirename:str) -> int:
		"""
		Adds a wire type to the signal item's list of default wire types.

		:param wiregroup: Wire group name value
		:type str: str
		:param wirename: Wire name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Default wire type added |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.AddDefaultWireEx(wiregroup, wirename)

	def DeleteDefaultWireEx(self, wiregroup:str, wirename:str) -> int:
		"""
		Removes a wire type from the signal item's list of default wire types.

		:param wiregroup: Wire group name value
		:type str: str
		:param wirename: Wire name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Default wire type removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.DeleteDefaultWireEx(wiregroup, wirename)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the signal item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the signal item |
		| "&lt;Empty&gt;" | No signal item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a signal item as the current item.

		:param gid: Global identifier value of a signal item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current signal item |
		| "&lt;Empty&gt;" | No signal item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current signal item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current signal item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a signal item as the current item.

		:param guid: Globally unique identifier value of a signal item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current signal item |
		| "&lt;Empty&gt;" | No signal item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- ICavityPartInterface--------------------
class CavityPart:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize CavityPart. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetCavityPartType(self) -> int:
		"""
		Gets a value representing the cavity part item's type.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..3 | Cavity part type |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.GetCavityPartType()

	def GetId(self) -> int:
		"""
		Gets the identifier of the current cavity part item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current cavity part identifier |
		| 0 | No cavity part item or an error occurred |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a cavity part as the current item.

		:param id: Unique value identifying a cavity part item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current cavity part item identifier |
		| 0 | No current cavity part item or an error occurred |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.SetId(id)

	def GetValue(self) -> str:
		"""
		Gets the cavity part item's value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Value&gt;" | Cavity part value |
		| "&lt;Empty&gt;" | No cavity part value or an error occurred |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.GetValue()

	def SetValue(self, value:str) -> int:
		"""
		Sets the cavity part item's value.

		:param value: New cavity part value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | New cavity part value assigned |
		| -1 | No project open or no cavity part item set |
		| -2 | Automatic selection is enabled |
		| -3 | Cavity part item's owner is locked |
		| -4 | Cavity part item cannot be changed |
		| -5 | Model pin outer diameter or cross-section data of the cavity part does not match the conductor data |
		| -6 | Pin is prevented from making physical changes to conductors, Available since v2022-23.00 |
		| -7 | value is too long, Available since v2023-24.50 and v2025-25.20 |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.SetValue(value)

	def GetOwner(self) -> int:
		"""
		Gets the identifier of the pin the cavity part item belongs to.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Pin identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.GetOwner()

	def GetCabwirInfos(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the current cavity part item's cable wire items.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | cabWirs | tuple[int,...] | Array of identifiers of cable wire items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of cable wire items in cabWirs |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 18.00
		"""
		dummy=0
		ret, cabWirs = self._obj.GetCabwirInfos(dummy)
		cabWirs = cabWirs[1:] if type(cabWirs) == tuple and len(cabWirs) > 0 else tuple()
		return ret, cabWirs

	def SetDisableAutomaticSelection(self, DisableAutomaticSelection:bool) -> int:
		"""
		Sets the cavity part item's automatic selection status.

		:param DisableAutomaticSelection: Automatic selection status value. If 1, automatic selection is set to inactive. If 0, automatic selection is set to active
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Automatic selection status value is applied |
		| -1 | No project open or no cavity part item set |
		| -2 | Project settings do not allow changes |
		| -3 | Cavity part item's owner is locked |
		| -4 | Cavity part item cannot be changed |
		| -5 | Pin is prevented from making physical changes to conductors, Available since v2022-23.00 |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.SetDisableAutomaticSelection(DisableAutomaticSelection)

	def GetDisableAutomaticSelection(self) -> int:
		"""
		Gets the cavity part item's automatic selection status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Automatic selection is inactive |
		| 0 | Automatic selection is active |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.GetDisableAutomaticSelection()

	def Create(self, pinid:int, name:str, _type:int) -> int:
		"""
		Creates a cavity part item.

		:param pinid: Identifier of the new cavity part's pin
		:type int: int
		:param name: Value of the new cavity part
		:type str: str
		:param _type: Type of the new cavity part
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new cavity part item |
		| 0 | Error occurred |
		| -1 | No project open |
		| -2 | Invalid pinid value |
		| -3 | Invalid type value |
		| -4 | Pin is locked |
		| -5 | Pin already has a cavity part with the type value |
		| -6 | Pin is prevented from making physical changes to conductors, Available since v2022-23.00 |
		| -7 | pinid represents a busbar pin, Available since v2022-23.00 |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.Create(pinid, name, _type)

	def GetAttributeValue(self, name:str) -> str:
		"""
		Function not implemented yet.

		:param name: Not used
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Empty&gt;" | Always returned |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.GetAttributeValue(name)

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Function not implemented yet.

		:param name: Not used
		:type str: str
		:param value: Not used
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Always returned |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Function not implemented yet.

		:param name: Not used
		:type str: str
		:param value: Not used
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Always returned |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Function not implemented yet.

		:param name: Not used
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Always returned |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.DeleteAttribute(name)

	def IsActive(self) -> int:
		"""
		Indicates the cavity part item's active status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Cavity part is active |
		| 0 | Cavity part is inactive |
		| -1 | No project open or no cavity part item set |
		| -2 | Cavity part item cannot be changed |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.IsActive()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current cavity part item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current cavity part item |
		| "&lt;Empty&gt;" | No cavity part item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a cavity part as the current item.

		:param gid: Global identifier value of a cavity part item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current cavity part item |
		| "&lt;Empty&gt;" | No current cavity part item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- ISymbolInterface--------------------
class Symbol:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Symbol. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current symbol item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current symbol item identifier |
		| 0 | No symbol item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a symbol item as the current item.

		:param id: Unique value identifying a symbol item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current symbol item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def SetName(self, name:str) -> int:
		"""
		Sets the symbol item's name.

		:param name: Symbol name value to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol name |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def GetName(self) -> str:
		"""
		Gets the symbol item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Symbol name |
		| "&lt;Empty&gt;" | Symbol has no name or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def GetVersion(self) -> str:
		"""
		Gets the symbol item's version as it appears in the database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Symbol item's version |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetVersion()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the symbol item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the symbol item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the symbol item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the symbol item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the symbol item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the symbol item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def GetPinCount(self) -> int:
		"""
		Gets the number of the symbol item's connection nodes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of connection nodes |
		| 0 | No connection nodes found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPinCount()

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbol item's connection nodes.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No pin identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetTextCount(self) -> int:
		"""
		Gets the number of text items belonging to the symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items |
		| 0 | No text items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTextCount()

	def GetTextIds(self, txttyp:int=0, search_string:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbol's text items.

		:param txttyp: Text type filter. Only text identifiers with this Text Type will be supplied. All text identifiers are supplied if txttyp is 0. The default value 0, defaults to 0
		:type int: int
		:param search_string: Text value filter, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of text identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No text items are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetTextIds(dummy, txttyp, search_string)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsShield(self) -> int:
		"""
		Indicates if the symbol item is a shield symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a shield symbol |
		| 0 | Item is not a shield symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsShield()

	def IsDynamic(self) -> int:
		"""
		Indicates if the symbol item is a dynamic symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a dynamic symbol |
		| 0 | Item is not a dynamic symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsDynamic()

	def GetSchemaLocation(self) -> tuple[int, float, float, str, str, str]:
		"""
		Gets the symbol item's position within the project.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Placement position on the x-axis passed out of the function |
		| 2 | y | float | Placement position on the y-axis passed out of the function |
		| 3 | grid | str | Location in the format ". /&lt;sheet&gt;.&lt;grid&gt;. " passed out of the function |
		| 4 | column_value | str | Sheet placement column of the symbol item passed out of the function |
		| 5 | row_value | str | Sheet placement row of the symbol item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the symbol item's sheet |
		| 0 | Unplaced symbol or a gate or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetSchemaLocation(dummy, dummy, dummy, dummy, dummy)

	def Load(self, name:str, version:str) -> int:
		"""
		Loads a symbol from the database.

		:param name: Name of the database symbol
		:type str: str
		:param version: Version number of the database symbol
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the symbol item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Load(name, version)

	def GetArea(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the symbol item's defined space requirement from the database.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xmin | float | First position on the x-axis passed out of the function |
		| 2 | ymin | float | First position on the y-axis passed out of the function |
		| 3 | xmax | float | Second position on the x-axis passed out of the function |
		| 4 | ymax | float | Second position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Placement area values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetArea(dummy, dummy, dummy, dummy)

	def Place(self, shti:int, x:float, y:float, rot:str="", scale:float=0, maintaintextsize:bool=False) -> int:
		"""
		Creates and places a symbol item or moves the current symbol item's position within the project.

		:param shti: Identifier of the sheet
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Rotation and mirroring value. See Rotation for possible values.. Default is no rotation or mirroring, defaults to ""
		:type str: str
		:param scale: Scale factor value. Default is no scaling, defaults to 0
		:type float: float
		:param maintaintextsize: Indicates whether symbol text items should also be scaled to the. scale. value. If 1, the text items are not scaled. If 0, the text items are scaled., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the placed symbol item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Place(shti, x, y, rot, scale, maintaintextsize)

	def GetMasterCount(self) -> int:
		"""
		Gets the number of the symbol's master symbol items.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of master symbol identifiers |
		| 0 | No master symbol identifiers found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetMasterCount()

	def GetMasterIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbol item's master symbols.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of master symbols passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No master symbols are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetMasterIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSlaveCount(self) -> int:
		"""
		Gets the number of the symbol's slave symbol items.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of slave symbol identifiers |
		| 0 | No slave symbol identifiers found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSlaveCount()

	def GetSlaveIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbol item's slave symbols.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of slave symbols passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No slave symbols are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSlaveIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Jump(self) -> int:
		"""
		Jumps to the symbol item's schema location and highlights the symbol item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Always returned |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Jump()

	def SetDeviceName(self, name:str) -> int:
		"""
		Sets the symbol item's device designation value.

		:param name: New device designation value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Device designation value assigned |
		| 0 | Device designation value assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetDeviceName(name)

	def SetDeviceAssignment(self, name:str) -> int:
		"""
		Sets the symbol item's higher level assignment value.

		:param name: New higher level assignment value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Higher level assignment value assigned |
		| 0 | Higher level assignment value assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetDeviceAssignment(name)

	def SetDeviceLocation(self, name:str) -> int:
		"""
		Sets the symbol item's location value.

		:param name: New location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Location value assigned |
		| 0 | Location value assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetDeviceLocation(name)

	def Delete(self) -> int:
		"""
		Removes the symbol item from the sheet if it is placed or deletes the symbol item from the project if it is an unplaced internal gate of a dynamic device.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the symbol item: An error occurred |
		| 0 | Symbol item was removed or deleted |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def GetCode(self) -> int:
		"""
		Gets a value representing the internal symbol type code of the symbol item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Internal symbol type value of the symbol item |
		| 0 | Undefined symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCode()

	def GetSymbolCount(self) -> int:
		"""
		Gets the number of symbols placed on the block symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbol items |
		| 0 | No symbol items or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSymbolCount()

	def GetSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbols placed on the block symbol.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbols passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No symbols are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetRotation(self) -> str:
		"""
		Gets the symbol item's rotation and mirroring value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Rotation and mirroring value, See Rotation String for possible values |
		| "0.000000" | Symbol item is not rotated and not mirrored or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetRotation()

	def GetScaling(self) -> float:
		"""
		Gets the symbol item's scaling factor value.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &lt; 0.0 | Scaling factor value |
		| 1.0 | Scaling factor value or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetScaling()

	def GetLevel(self) -> int:
		"""
		Gets the symbol item's display level value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..256 | Display level value supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetLevel()

	def IsConnectorMaster(self) -> int:
		"""
		Indicates if the connector symbol item is a connector master symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Connector symbol is a connector master symbol |
		| 0 | Connector symbol is not a connector master symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsConnectorMaster()

	def GetTypeName(self) -> str:
		"""
		Gets the name of the symbol item's type.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Name of the symbol type |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTypeName()

	def IsConnected(self) -> int:
		"""
		Indicates if the symbol item  has connect lines running from it.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol item  has connect lines running from it |
		| 0 | Symbol item has no connect lines running from it or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsConnected()

	def SetDeviceCompleteName(self, name:str, ass:str, loc:str, onlygiven:bool=True) -> int:
		"""
		Sets all elements of the symbol item's  item designation value.

		:param name: New device designation value
		:type str: str
		:param ass: New higher level assignment value
		:type str: str
		:param loc: New location value
		:type str: str
		:param onlygiven: Indicates whether empty values in. name. ,. ass. and. loc. should be ignored. If 1, empty values are ignored. If 0, empty values are set., defaults to True
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item designation values assigned |
		| 0 | Item designation values assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetDeviceCompleteName(name, ass, loc, onlygiven)

	def SetType(self, name:str, version:str) -> int:
		"""
		Replaces the symbol item in the project for a symbol of the specified type name and version from the database.

		:param name: Symbol type name from database
		:type str: str
		:param version: Symbol type version from database
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the replacement symbol item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetType(name, version)

	def GetType(self) -> str:
		"""
		Gets the name of the symbol item's type as it appears in the database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Symbol type name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetType()

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the symbol item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def PlacePins(self, pinarray:list[int], symname:str, vers:str, shti:int, x:float, y:float, rot:str, scale:float=0) -> int:
		"""
		Creates and places a pin symbol.

		:param pinarray: Array of pin identifiers passed into the function
		:type list[int]: list[int]
		:param symname: Name of the connector symbol in the database
		:type str: str
		:param vers: Version number of the connector symbol in the database
		:type str: str
		:param shti: Identifier of the sheet
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Rotation and mirroring value. See Rotation for possible values
		:type str: str
		:param scale: Scale factor value. Default is no scaling, defaults to 0
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the placed pin symbol |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.PlacePins(pinarray, symname, vers, shti, x, y, rot, scale)
		return ret[0]

	def PlaceDynamic(self, shti:int, x:float, y:float, height:float, width:float, texttemplate:str="", fitsymbol:int=0) -> int:
		"""
		Creates and places a dynamic symbol.

		:param shti: Identifier of the sheet
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param height: Height of the symbol
		:type float: float
		:param width: Width of the symbol
		:type float: float
		:param texttemplate: Text template symbol to use. If set to "<Empty>", the project defined default text template is used. Default value is "<Empty>", defaults to ""
		:type str: str
		:param fitsymbol: Indicates whether any symbol placed on the dynamic symbol should be scaled to fit the dynamic symbol. If 1, symbols placed on the dynamic symbol shall be scaled to fit the dynamic symbol. If 0, symbols placed on the dynamic symbol shall not be scaled to fit the dynamic symbol, defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the placed dynamic symbol |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PlaceDynamic(shti, x, y, height, width, texttemplate, fitsymbol)

	def HasPassWirePins(self) -> int:
		"""
		Indicates if the symbol contains pins with the pass wire value active.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol item does contain pins with the pass wire value active |
		| 0 | Symbol item does not contain pins with the pass wire value active or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasPassWirePins()

	def SetLevel(self, level:int) -> int:
		"""
		Sets the symbol item's display level value.

		:param level: Display level value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..256 | Previous level value |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetLevel(level)

	def SetAsMaster(self, on:int) -> int:
		"""
		Sets the symbol item as a master symbol.

		:param on: Value to set the master status to. If 1, the symbol is set as the master symbol. If 0, the symbol is unset as the master symbol
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previously the symbol was set as the master symbol |
		| 0 | Previously the symbol was not set as the master symbol |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAsMaster(on)

	def IsMaster(self) -> int:
		"""
		Indicates if the symbol item is a master symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a master symbol |
		| 0 | Item is not a master symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsMaster()

	def PlaceBlock(self, shti:int, x:float, y:float, width:float, height:float, parent:int=0) -> int:
		"""
		Creates and places a block symbol item or moves and resizes the current block symbol.

		:param shti: Identifier of the sheet
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param width: Width of the block
		:type float: float
		:param height: Height of the block
		:type float: float
		:param parent: Identifier of the parent block symbol., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the block symbol |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PlaceBlock(shti, x, y, width, height, parent)

	def IsTwisted(self) -> int:
		"""
		Indicates if the symbol item is a twisted pair symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a twisted pair symbol |
		| 0 | Item is not a twisted pair symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsTwisted()

	def IsBundle(self) -> int:
		"""
		Indicates if the symbol item is a bundle.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a bundle symbol |
		| 0 | Item is not a bundle symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsBundle()

	def GetTransformationMatrix(self) -> tuple[int, tuple[tuple[float,...],...]]:
		"""


		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | array | tuple[tuple[float,...],...] |  |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetTransformationMatrix(dummy)

	def GetSheetReferenceInfo(self) -> tuple[int, int, int, str, str]:
		"""
		Gets the signal cross reference symbol item's details.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | inout | int | Reference direction value |
		| 2 | _type | int | Reference type value |
		| 3 | refnam | str | Reference name value |
		| 4 | signam | str | Signal name value |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signal cross reference values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetSheetReferenceInfo(dummy, dummy, dummy, dummy)

	def AddToSheetReference(self, symi:int) -> int:
		"""
		Adds the symbol item's sheet reference to a reference symbol.

		:param symi: Identifier of the reference symbol
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the reference symbol |
		| 0 | The reference could not be added due to logical reasons |
		| &lt; 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddToSheetReference(symi)

	def GetSymbolTypeName(self) -> str:
		"""
		Gets the name of the symbol item's type as it appears in the database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Symbol type name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSymbolTypeName()

	def GetSchematicTypes(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the symbol item's schematic type values.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | types | tuple[int,...] | Array of schematic type values passed out of the function. See Schematic for possible values |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in types |
		| 0 | All schematic types are assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, types = self._obj.GetSchematicTypes(dummy)
		types = types[1:] if type(types) == tuple and len(types) > 0 else tuple()
		return ret, types

	def GetAssignedOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the symbol item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned options are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAssignedOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def PlaceInteractively(self) -> int:
		"""
		Allows the interactive placement or movement of a symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the placed symbol |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PlaceInteractively()

	def AssignTo(self, assignto:int, nonodeassignment:int=0) -> int:
		"""
		Assigns the symbol item to a device, gate or pin item.

		:param assignto: Identifier of the device, gate or pin to assign the symbol item to
		:type int: int
		:param nonodeassignment: Indicates whether free pins of. assignto. may be assigned to the symbol item's pins. If 0, pins may be assigned. If 1, pins may not be assigned., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | value of AssignTo |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AssignTo(assignto, nonodeassignment)

	def PlaceAsGraphic(self, shti:int, x:float, y:float, rot:str, scale:float, maintaintextsize:bool, srcid:int) -> int:
		"""
		Creates and places a graphic item.

		:param shti: Identifier of the sheet
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Rotation and mirroring value. See Rotation for possible values.
		:type str: str
		:param scale: Scale factor value
		:type float: float
		:param maintaintextsize: Indicates whether symbol text items should also be scaled to the. scale. value. If 1, the text items are not scaled. If 0, the text items are scaled
		:type bool: bool
		:param srcid: Identifier of the symbol to copy
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the placed symbol item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PlaceAsGraphic(shti, x, y, rot, scale, maintaintextsize, srcid)

	def GetGateId(self) -> int:
		"""
		Gets the gate identifier of the symbol item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current symbol item identifier |
		| 0 | Symbol item has no gate or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetGateId()

	def SetGateId(self, symid:int) -> int:
		"""
		Sets a gate as the current symbol item.

		:param symid: Unique value identifying a gate item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current gate item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetGateId(symid)

	def SetCharacteristic(self, characteristic:str) -> int:
		"""
		Sets the symbol item's characteristic value.

		:param characteristic: New characteristic value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Characteristic value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetCharacteristic(characteristic)

	def GetCharacteristic(self) -> str:
		"""
		Gets the symbol item's characteristic value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Characteristic value supplied |
		| "&lt;Empty&gt;" | No characteristic value supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCharacteristic()

	def GetValidCharacteristics(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets an array of the possible characteristics for the symbol.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | characteristics | tuple[str,...] | Array of valid characteristics passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in characteristics |
		| 0 | No valid characteristics or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, characteristics = self._obj.GetValidCharacteristics(dummy)
		characteristics = characteristics[1:] if type(characteristics) == tuple and len(characteristics) > 0 else tuple()
		return ret, characteristics

	def IsBlock(self) -> int:
		"""
		Indicates if the symbol item is a block.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a block symbol |
		| 0 | Item is not a block symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsBlock()

	def SetBlockOutlineColour(self, value:int) -> int:
		"""
		Sets the symbol item's block outline color.

		:param value: Block outline color value. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..255 | Previous block outline color value, See Colors for possible values |
		| -1 | Previous block outline color was automatic or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetBlockOutlineColour(value)

	def GetBlockOutlineColour(self) -> int:
		"""
		Gets the symbol item's block outline color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..255 | Block outline color value, See Colors for possible values |
		| -1 | Automatic color or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBlockOutlineColour()

	def SetBlockOutlineStyle(self, value:int) -> int:
		"""
		Sets the symbol item's block outline line style.

		:param value: Outline block line style value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Previous block outline line style value, See Line Styles for possible values |
		| 0 or &gt;47 | Previous block outline line style value |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetBlockOutlineStyle(value)

	def GetBlockOutlineStyle(self) -> int:
		"""
		Gets the symbol item's block outline style.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Block outline style value, See Line Styles for possible values |
		| 0 or &gt;47 | Single line |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBlockOutlineStyle()

	def SetBlockOutlineWidth(self, value:float) -> float:
		"""
		Sets the symbol item's block outline line width.

		:param value: Outline line width value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Previous block outline line width value |
		| -1.0 | No block outline line width value assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetBlockOutlineWidth(value)

	def GetBlockOutlineWidth(self) -> float:
		"""
		Gets the symbol item's block outline width.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Block outline width value |
		| -1.0 | No block outline width assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBlockOutlineWidth()

	def SetBlockHatchPattern(self, value:int, angle1:float, angle2:float) -> int:
		"""
		Sets the symbol item's block hatch pattern.

		:param value: Block hatch pattern to apply
		:type int: int
		:param angle1: Angle value in degrees of a hatch line to apply. Value is only used if value is a line or cross hatch pattern value. The value range is from 90.0 (pointing upwards) to 0.0 (pointing rightwards) to -90.0 (pointing downwards)
		:type float: float
		:param angle2: Angle value in degrees of a hatch line to apply. Value is only used if value is a cross hatch pattern value. The value range is from 90.0 (pointing upwards) to 0.0 (pointing rightwards) to -90.0 (pointing downwards)
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | No previous hatch pattern value set or error occurred |
		| 1, 2 or 4 | Previous hatch pattern value |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetBlockHatchPattern(value, angle1, angle2)

	def GetBlockHatchPattern(self) -> tuple[int, float, float]:
		"""
		Gets the symbol item's block hatch pattern.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | angle1 | float | Angle value in degrees of a hatch line if a line or cross hatch pattern is in use passed out of the function. The value will be 0.0 if neither a line or cross hatch pattern is in use |
		| 2 | angle2 | float | Angle value in degrees of a hatch line if a cross hatch pattern is in use passed out of the function. The value will be 0.0 if a cross hatch pattern is not in use |

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | No block hatch pattern or an error occurred |
		| 1 | Solid hatch |
		| 2 | Line hatch |
		| 4 | Cross hatch |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetBlockHatchPattern(dummy, dummy)

	def SetBlockHatchColour(self, value:int) -> int:
		"""
		Sets the symbol item's block hatch color.

		:param value: Block hatch color value. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..255 | Previous block hatch color value, See Colors for possible values |
		| -1 | Previous block hatch color was automatic or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetBlockHatchColour(value)

	def GetBlockHatchColour(self) -> int:
		"""
		Gets the symbol item's block hatch color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..255 | Block hatch color value, See Colors for possible values |
		| -1 | Automatic color or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBlockHatchColour()

	def SetBlockHatchStyle(self, value:int) -> int:
		"""
		Sets the symbol item's block hatch line style.

		:param value: Hatch line style value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Previous hatch line style value, See Line Styles for possible values |
		| 0 or &gt;47 | Previous hatch line style value |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetBlockHatchStyle(value)

	def GetBlockHatchStyle(self) -> int:
		"""
		Gets the symbol item's block hatch line style.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Block hatch line style value, See Line Styles for possible values |
		| 0 or &gt;47 | 0 or &gt;47 |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBlockHatchStyle()

	def SetBlockHatchWidth(self, value:float) -> float:
		"""
		Sets the symbol item's block hatch line width.

		:param value: Hatch line width value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Previous block hatch line width value |
		| -1.0 | No block hatch line width value assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetBlockHatchWidth(value)

	def GetBlockHatchWidth(self) -> float:
		"""
		Gets the symbol item's block hatch line width.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Block hatch line width value |
		| -1.0 | No block hatch line width assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBlockHatchWidth()

	def SetBlockHatchDistance(self, value:float) -> float:
		"""
		Sets the symbol item's block hatch pattern distance.

		:param value: Block hatch pattern distance value
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Previous block hatch pattern distance value |
		| -1.0 | Previously no block hatch pattern distance value assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetBlockHatchDistance(value)

	def GetBlockHatchDistance(self) -> float:
		"""
		Gets the symbol item's block hatch pattern distance.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Block hatch pattern distance value |
		| -1.0 | No block hatch pattern distance assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBlockHatchDistance()

	def GetGroupId(self) -> int:
		"""
		Gets the identifier of the group containing the symbol item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the containing group |
		| 0 | No containing group or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetGroupId()

	def Ungroup(self) -> int:
		"""
		Removes the symbol item from its group.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Error occurred |
		| 0 | Symbol item is removed from its group |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Ungroup()

	def IsFormboard(self) -> bool:
		"""
		Indicates if the symbol item is a formboard symbol.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a formboard symbol |
		| 0 | Item is not a formboard symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsFormboard()

	def PlaceTable(self, fromid:int, x:float, y:float, rot:str="", scale:float=0, maintaintextsize:bool=False) -> int:
		"""
		Creates and places a table symbol on a formboard sheet.

		:param fromid: Identifier of the placed connector symbol
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Rotation and mirroring value. See Rotation for possible values., defaults to ""
		:type str: str
		:param scale: Scale factor value. Default is no scaling, defaults to 0
		:type float: float
		:param maintaintextsize: Indicates whether symbol text items should also be scaled to the. scale. value. If 1, the text items are not scaled. If 0, the text items are scaled., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the placed table symbol |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PlaceTable(fromid, x, y, rot, scale, maintaintextsize)

	def IsPinView(self) -> int:
		"""
		Indicates if the symbol item is a pin view symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a pin view symbol |
		| 0 | Item is not a pin view symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsPinView()

	def PlacePinView(self, pinid:int, shti:int, symname:str, version:str, x:float, y:float, rot:str, scale:float=0, maintaintextsize:bool=False) -> int:
		"""
		Creates and places a pin view symbol.

		:param pinid: Identifier of the connector pin or block connector pin to create a view for
		:type int: int
		:param shti: Identifier of the sheet to place the symbol on
		:type int: int
		:param symname: Name of the symbol in the database
		:type str: str
		:param version: Version number of the symbol in the database
		:type str: str
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Rotation and mirroring value. See Rotation for possible values.
		:type str: str
		:param scale: Scale factor value. Default is no scaling, defaults to 0
		:type float: float
		:param maintaintextsize: Indicates whether symbol text items should also be scaled to the. scale. value. If 1, the text items are not scaled. If 0, the text items are scaled., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the pin view symbol |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.PlacePinView(pinid, shti, symname, version, x, y, rot, scale, maintaintextsize)

	def GetDynamicParentId(self) -> int:
		"""
		Gets the identifier of the symbol item's dynamic parent symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the dynamic parent symbol |
		| 0 | No dynamic parent symbol found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetDynamicParentId()

	def GetDynamicChildrenIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the subsidiary symbols contained in the dynamic symbol item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No child symbols are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetDynamicChildrenIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsDynamicFixed(self) -> int:
		"""
		Indicates if the symbol item is a fixed dynamic symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a fixed dynamic symbol |
		| 0 | Item is not a fixed dynamic symbol or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsDynamicFixed()

	def SetScaling(self, scale:float) -> float:
		"""
		Sets the symbol item's scaling factor value.

		:param scale: Scaling value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &lt; 0.0 | Previous scaling factor value |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetScaling(scale)

	def SetDBTextSize(self, txtsiz:bool) -> int:
		"""
		Sets a value indicating if the text size factor defined in the database is used for the symbol's text items when scaling.

		:param txtsiz: New text size factor usage value. If 1, the text size factor defined in the database is used for the symbol's text items when scaling. If 0, the text size factor defined in the database is not used for the symbol's text items when scaling
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previously the text size factor defined in the database was used for the symbol's text items when scaling |
		| 0 | Previously the text size factor defined in the database was not used for the symbol's text items when scaling or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetDBTextSize(txtsiz)

	def GetDBTextSize(self) -> bool:
		"""
		Gets a value indicating if the text size factor defined in the database is used for the symbol's text items when scaling.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text size factor defined in the database is used for the symbol's text items when scaling |
		| 0 | Text size factor defined in the database is not used for the symbol's text items when scaling or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetDBTextSize()

	def GetBoundIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of graphic items that have been graphically grouped to the symbol item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of graphic items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No graphic item identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetBoundIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetHyperlinkTextIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the symbol item's texts with hyperlinks.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of text identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No text items with hyperlinks or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetHyperlinkTextIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetDisplayLength(self) -> float:
		"""
		Gets the symbol item's display length.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Symbol display length value |
		| 0.0 | No display length value or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetDisplayLength()

	def GetDisplayWidth(self) -> float:
		"""
		Gets the symbol item's display width.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Symbol display width value |
		| 0.0 | No display width value or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetDisplayWidth()

	def SetDisplayLength(self, length:float) -> int:
		"""
		Sets the symbol item's display length.

		:param length: Display length value
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display length value is assigned |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetDisplayLength(length)

	def SetDisplayWidth(self, width:float) -> int:
		"""
		Sets the symbol item's display width.

		:param width: Display width value
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display width value is assigned |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetDisplayWidth(width)

	def GetPlacedArea(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the symbol item's placement area on the sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xmin | float | First position on the x-axis passed out of the function |
		| 2 | ymin | float | First position on the y-axis passed out of the function |
		| 3 | xmax | float | Second position on the x-axis passed out of the function |
		| 4 | ymax | float | Second position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Placement area values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetPlacedArea(dummy, dummy, dummy, dummy)

	def GetAssignedOptionExpressions(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options) assigned to the symbol item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressions(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def HasNoGraphic(self) -> int:
		"""
		Indicates if the symbol contains at least one graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Symbol contains graphic items |
		| 1 | Symbols missing graphical content or an error occurred |
		| 2 | Symbol graphical content was automatically generated |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.HasNoGraphic()

	def GetDevicePinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the device pins contained in the symbol item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No pin identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetDevicePinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSharedPinGroupState(self) -> int:
		"""
		Indicates if the symbol item is a shared pin group.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol item is a shared pin group |
		| 0 | Symbol is not a shared pin group or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetSharedPinGroupState()

	def SetSharedPinGroupState(self, newval:int) -> int:
		"""
		Converts the symbol item between a normal symbol and a shared pin group.

		:param newval: Shared pin group value to apply. If 1, the symbol item is converted from a normal symbol to a shared pin group symbol. If 0, the symbol item is converted from a shared pin group symbol to a normal symbol
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol item is converted to a shared pin group symbol |
		| 0 | Symbol item is converted to a normal symbol or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetSharedPinGroupState(newval)

	def SetOptionExpressions(self, expressions:list[str]) -> int:
		"""
		Replaces the symbol item's option expressions.

		:param expressions: Array of strings of all option names, Boolean expressions or alias names of Boolean expressions
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.12
		"""
		return self._obj.SetOptionExpressions(expressions)

	def IsProtection(self) -> int:
		"""
		Indicates if the symbol item is a cable protection symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item is a cable protection symbol |
		| 0 | Item is not a cable protection symbol or an error occurred |

		Available since E3.series TLB-Versions: 9.22
		"""
		return self._obj.IsProtection()

	def GetAssignedOptionExpressionsEx(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options)  including variant instances assigned to the symbol item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.23
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressionsEx(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def AssignFunctionalPorts(self, ids:list[int]) -> int:
		"""
		Assigns functional ports defined in an installation space to a dynamic functional unit and assigns the functional unit to the symbol item.

		:param ids: Array of dynamic functional port identifiers from an installation space passed into the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new functional unit |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		ret = self._obj.AssignFunctionalPorts(ids)
		return ret[0]

	def GetTargetObjectId(self) -> int:
		"""
		Gets the identifier of the target item of the attribute text template symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the target item |
		| 0 | No target item found or an error occurred |

		Available since E3.series TLB-Versions: 10.30
		"""
		return self._obj.GetTargetObjectId()

	def GetReferenceTextExtent(self) -> tuple[int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the placement positions of the symbol item's dynamic signal cross reference texts.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xarr | tuple[float,...] | Array of placement positions on the x-axis passed out of the function |
		| 2 | yarr | tuple[float,...] | Array of placement positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Dynamic signal cross reference text placement position values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.40
		"""
		dummy=0
		ret, xarr, yarr = self._obj.GetReferenceTextExtent(dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, xarr, yarr

	def GetReferenceTextExtentSingleLine(self) -> tuple[int, int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the placement positions of the symbol item's dynamic signal cross reference texts.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | nlines | int | Number of dynamic signal cross reference text lines passed out of the function |
		| 2 | xarr | tuple[float,...] | Array of placement positions for each reference text on the x-axis passed out of the function |
		| 3 | yarr | tuple[float,...] | Array of placement positions for each reference text on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Dynamic signal cross reference text placement position values supplied |
		| 0 | Symbol item has no dynamic signal cross references or an error occurred |

		Available since E3.series TLB-Versions: 10.40
		"""
		dummy=0
		return self._obj.GetReferenceTextExtentSingleLine(dummy, dummy, dummy)

	def GetReferenceText(self) -> str:
		"""
		Gets the symbol item's dynamic signal cross references.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Dynamic signal cross references |
		| "&lt;Empty&gt;" | No dynamic cross references or an error occurred |

		Available since E3.series TLB-Versions: 10.41
		"""
		return self._obj.GetReferenceText()

	def GetReferenceTextSingleLine(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets the symbol item's dynamic signal cross references.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | textarr | tuple[str,...] | Array of dynamic signal cross reference values passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of dynamic signal cross references values in textarr |
		| 0 | No dynamic cross references or an error occurred |

		Available since E3.series TLB-Versions: 10.41
		"""
		dummy=0
		ret, textarr = self._obj.GetReferenceTextSingleLine(dummy)
		textarr = textarr[1:] if type(textarr) == tuple and len(textarr) > 0 else tuple()
		return ret, textarr

	def GetSymbolExtend(self) -> tuple[int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the placement area of the symbol item's graphics on the sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xarr | tuple[float,...] | Array of positions on the x-axis passed out of the function |
		| 2 | yarr | tuple[float,...] | Array of positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Area values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.46
		"""
		dummy=0
		ret, xarr, yarr = self._obj.GetSymbolExtend(dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, xarr, yarr

	def GetSchemaLocationShared(self, shtid:int) -> tuple[int, float, float, str, str, str]:
		"""
		Gets the symbol's shared position within the project.

		:param shtid: Identifier of the sheet region
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Placement position on the x-axis passed out of the function |
		| 2 | y | float | Placement position on the y-axis passed out of the function |
		| 3 | grid | str | Location in the format ". /&lt;sheet&gt;.&lt;grid&gt;. " passed out of the function |
		| 4 | column_value | str | Sheet placement column of the symbol item passed out of the function |
		| 5 | row_value | str | Sheet placement row of the symbol item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the model symbol's shared sheet |
		| 0 | No project open or no model symbol item set |
		| -1 | shtid is not a valid sheet identifier |
		| -2 | shtid is not a valid sheet region |
		| -3 | shtid is not a shared sheet |
		| -4 | Symbol is not placed on shtid |

		Available since E3.series TLB-Versions: 11.20
		"""
		dummy=0
		return self._obj.GetSchemaLocationShared(shtid, dummy, dummy, dummy, dummy, dummy)

	def GetPlacedPolygon(self) -> tuple[int, int, tuple[int,...], tuple[int,...]]:
		"""
		Gets the symbol item's placement area on the sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | npnts | int | Number of points making the polygon area passed out of the function |
		| 2 | xarr | tuple[int,...] | Array of positions on the x-axis passed out of the function |
		| 3 | yarr | tuple[int,...] | Array of positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Placement area values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.00
		"""
		dummy=0
		ret, npnts, xarr, yarr = self._obj.GetPlacedPolygon(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, npnts, xarr, yarr

	def RemoveFromSheetReference(self) -> int:
		"""
		Removes a reference symbol item's sheet reference.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet reference removed |
		| 0 | Logical error occurred |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 17.33
		"""
		return self._obj.RemoveFromSheetReference()

	def GetSymbolTypeId(self) -> int:
		"""
		Gets the identifier of the symbol item's type in the database.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;1 | Identifier of the symbol type in the database |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 18.23
		"""
		return self._obj.GetSymbolTypeId()

	def GetTableOneRowForEachCore(self) -> int:
		"""
		Indicates if the table symbol item's display one row for each core value is active.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display one row for each core value is active |
		| 0 | Display one row for each core value is inactive |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetTableOneRowForEachCore()

	def SetTableOneRowForEachCore(self, newval:bool) -> int:
		"""
		Sets the table symbol item's display one row for each core value.

		:param newval: Display one row for each core value to apply. If 1, display one row for each core is activated. If 0, display one row for each core is deactivated
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Display one row for each core value was applied |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetTableOneRowForEachCore(newval)

	def GetTablePinsWithoutCores(self) -> int:
		"""
		Indicates if the table symbol item's display pins without cores value is active.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display pins without cores value is active |
		| 0 | Display pins without cores value is inactive |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetTablePinsWithoutCores()

	def SetTablePinsWithoutCores(self, newval:bool) -> int:
		"""
		Sets the table symbol item's display pins without cores status value.

		:param newval: Display pins without cores status value to apply. If 1, display pins without cores is activated. If 0, display pins without cores is deactivated
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Display pins without cores status value was applied |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetTablePinsWithoutCores(newval)

	def GetTableBreakTableAfter(self) -> int:
		"""
		Indicates if the table symbol item's break table after value is active.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Break table after value is active |
		| 0 | Break table after value is inactive |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetTableBreakTableAfter()

	def SetTableBreakTableAfter(self, newval:bool) -> int:
		"""
		Sets the table symbol item's break table after value.

		:param newval: Break table after value to apply. If 1, break table after value is activated. If 0, break table after value is deactivated
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | newval value is applied |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetTableBreakTableAfter(newval)

	def GetTableBreakTableAfterNumberOfRows(self) -> int:
		"""
		Indicates the number of rows displayed on the table symbol item before a table break occurs.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=2 | Number of table rows before a table break |
		| 0 | Table symbol item's break table after value is inactive |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetTableBreakTableAfterNumberOfRows()

	def SetTableBreakTableAfterNumberOfRows(self, newval:int) -> int:
		"""
		Sets the number of rows displayed on the table symbol item before a table break occurs.

		:param newval: Number of rows value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | New number of rows value was applied |
		| -1 | No symbol item set or the item is not a table symbol |
		| -2 | newval value lies outside the permitted range of 2 to 1000 |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetTableBreakTableAfterNumberOfRows(newval)

	def GetSubType(self) -> int:
		"""
		Gets a value representing the subtype of the symbol item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Subtype value of the symbol item |
		| -1 | No project is open |
		| -2 | No symbol item set |
		| -3 | Symbol is not a standard type |
		| -4 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetSubType()

	def SendToForeground(self) -> int:
		"""
		Moves the placed symbol item to the sheet foreground.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Symbol placed in the foreground |
		| -1 | No project open |
		| -2 | No symbol item set |
		| -3 | Symbol is not placed |
		| -4 | Error occurred |
		| -5 | Error occurred |
		| -6 | Symbol is locked, Available since v2023-24.22 |

		Available since E3.series TLB-Versions: 20.00, 19.01, 18.33, 17.43
		"""
		return self._obj.SendToForeground()

	def SendToBackground(self) -> int:
		"""
		Moves the placed symbol item to the sheet background.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Symbol placed in the background |
		| -1 | No project open |
		| -2 | No symbol item set |
		| -3 | Symbol is not placed |
		| -4 | Error occurred |
		| -5 | Error occurred |
		| -6 | Symbol is locked, Available since v2023-24.22 |

		Available since E3.series TLB-Versions: 20.00, 19.01, 18.33, 17.43
		"""
		return self._obj.SendToBackground()

	def GetSymbolType(self) -> int:
		"""
		Gets a value representing the type of the symbol item.

		:return:
		Type: int **Available enum type:** e3series.types.SymbolType.

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Type value of the symbol item |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 20.00, 19.01
		"""
		return self._obj.GetSymbolType()

	def AssignToConnector(self, AssignToPinIds:list[int], AssignToPinIds_connected:list[int], flags:list[int]) -> int:
		"""
		Assigns connector and block connector symbols to the component.

		:param AssignToPinIds: Array of connector pins to assign to the symbol. Already assigned pins become unassigned so only the transferred pins are assigned to the symbol
		:type list[int]: list[int]
		:param AssignToPinIds_connected: Array of connector pins to assign to the already connected symbol. The mode set by e3Job.SetGetterOptionHandlingMode() is used to determine the connected symbol in the case of multiple connectors due to variants and options assignments. Assigned pins are connected according to the conditions applied when the connection is done interactively by connecting the symbols. The operation is aborted if this is not possible
		:type list[int]: list[int]
		:param flags: Currently not used
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | One of the following results occurred:, The pins are assigned to the connectorNo pins were assigned because no pins were listed in AssignToPinIds or AssignToPinIds_connected |
		| -1 | No project open |
		| -2 | No symbol item set |
		| -3 | The pins could not be assigned to the symbols |
		| -4 | The existing connector is not unique due to variant and option assignments |
		| -5 | The pins assigned to the connector symbols could not be connected |
		| -6 | Internal error |

		Available since E3.series TLB-Versions: 21.00, 20.01, 19.31
		"""
		return self._obj.AssignToConnector(AssignToPinIds, AssignToPinIds_connected, flags)

	def SetSelected(self, newval:bool) -> int:
		"""
		Selects or deselects the symbol.

		:param newval: Indicates selection status of the symbol. If 1, the symbol will be selected. If 0, the symbol will be deselected
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol is selected or deselected |
		| -1 | No project open |
		| -2 | No symbol item set |
		| -3 | Symbol cannot be selected |
		| -4 | Selection exists on another sheet |
		| -5 | Symbol not displayed due to options/variants |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.SetSelected(newval)

	def ModifyBlockSize(self, border:int, delta:float, flags:int=0) -> int:
		"""
		Resizes a placed block symbol.

		:param border: Block border edge on which the block should be modified
		:type int: int
		:param delta: Value the block size should be resized
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Block symbol is resized |
		| -1 | No project open or no symbol item set |
		| -2 | Symbol item is not a block |
		| -3 | Sheet or block symbol is locked |
		| -4 | border value is invalid |
		| -5 | Symbol is rotated at an oblique angle |
		| -6 | Symbol could not be resized due to a collision |

		Available since E3.series TLB-Versions: 21.12
		"""
		return self._obj.ModifyBlockSize(border, delta, flags)

	def SetStateId(self, id:int) -> int:
		"""
		Sets the symbols item's state.

		:param id: Identifier of the state to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol's state applied |
		| -1 | No project open or no symbol item set |
		| -2 | Invalid symbol type |
		| -3 | id is invalid |
		| -4 | Symbol is placed on a locked sheet |
		| -5 | Symbol is a view |
		| -6 | Symbol is not placed on a schematic sheet |
		| -7 | Placed symbol is in an inactive variant or option |

		Available since E3.series TLB-Versions: 22.10
		"""
		return self._obj.SetStateId(id)

	def GetStateId(self) -> int:
		"""
		Gets the symbol item's state identifier.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the symbol's state |
		| 0 | Symbol has no state or an error occurred |

		Available since E3.series TLB-Versions: 22.10
		"""
		return self._obj.GetStateId()

	def GetStateIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the symbols item's defined states.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of symbol state identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No state identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 22.10
		"""
		dummy=0
		ret, ids = self._obj.GetStateIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current symbol item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current symbol item |
		| "&lt;Empty&gt;" | No symbol item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a symbol item as the current item.

		:param gid: Global identifier value of a symbol item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current symbol item |
		| "&lt;Empty&gt;" | No symbol item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- ITextInterface--------------------
class Text:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Text. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current text item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current text item identifier |
		| 0 | No text item |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a text as the current item.

		:param id: Unique value identifying a text item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current text item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetId(id)

	def GetType(self) -> int:
		"""
		Gets the text item's text type.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text type supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetType()

	def GetText(self) -> str:
		"""
		Gets the text item's text value including text token translations.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Text value is supplied |
		| "&lt;Empty&gt;" | Text item does not have a value or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetText()

	def SetText(self, newtext:str) -> int:
		"""
		Sets the text item's text value.

		:param newtext: Text value to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text value is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetText(newtext)

	def GetSchemaLocation(self) -> tuple[int, float, float, str, str, str]:
		"""
		Gets the text item's position within the project.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Placement position on the x-axis passed out of the function |
		| 2 | y | float | Placement position on the y-axis passed out of the function |
		| 3 | grid | str | Location in the format ". /&lt;sheet&gt;.&lt;grid&gt;. " passed out of the function |
		| 4 | column_value | str | Sheet placement column of the text passed out of the function |
		| 5 | row_value | str | Sheet placement row of the text passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the text item's sheet |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetSchemaLocation(dummy, dummy, dummy, dummy, dummy)

	def GetLeftJustifiedSchemaLocation(self) -> tuple[int, float, float, str]:
		"""
		Gets the text item's box’s lower right corner position in text reading direction.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Position on the x-axis passed out of the function |
		| 2 | y | float | Position on the y-axis passed out of the function |
		| 3 | grid | str | Location in the format ". /&lt;sheet&gt;.&lt;grid&gt;. " passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Sheet identifier of the text item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetLeftJustifiedSchemaLocation(dummy, dummy, dummy)

	def GetHeight(self) -> float:
		"""
		Gets the text item's font height.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Size of the text's font height |
		| 0.0 | Font height could not be supplied |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetHeight()

	def SetHeight(self, newval:float) -> int:
		"""
		Sets the text item's font height.

		:param newval: Value of font height to apply
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font height applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetHeight(newval)

	def GetLevel(self) -> int:
		"""
		Gets the text item's display level value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Display level value supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLevel()

	def SetLevel(self, newval:int) -> int:
		"""
		Sets the text item's display level value.

		:param newval: Display level value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display level value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLevel(newval)

	def GetMode(self) -> int:
		"""
		Gets the text item's text ratio.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..3 | Text ratio value supplied |
		| 0 | Text ratio value could not be supplied |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetMode()

	def SetMode(self, newval:int) -> int:
		"""
		Sets the text item's text ratio.

		:param newval: Text ratio value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text ratio value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetMode(newval)

	def GetStyle(self) -> int:
		"""
		Gets the text item's font style.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..7 | Font style value supplied |
		| 0 | Regular font style value used or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetStyle()

	def SetStyle(self, newval:int) -> int:
		"""
		Sets the text item's font style.

		:param newval: Font style to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font style value supplied |
		| 0 | No font style value used or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetStyle(newval)

	def GetColour(self) -> int:
		"""
		Gets the text item's text color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or the operation has failed |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetColour()

	def SetColour(self, newval:int) -> int:
		"""
		Sets the text item's text color.

		:param newval: Color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetColour(newval)

	def GetFontName(self) -> str:
		"""
		Gets the text item's font name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Name of the text's font |
		| "Arial" | Name of the text's font is "Arial" or the font name could not be supplied |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetFontName()

	def SetFontName(self, newname:str) -> int:
		"""
		Sets the text item's font.

		:param newname: Name of font to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetFontName(newname)

	def GetVisibility(self) -> int:
		"""
		Gets the text item's visibility status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text visibility status is shown |
		| 0 | Text visibility status is hidden or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetVisibility()

	def SetVisibility(self, newval:int) -> int:
		"""
		Sets the text item's visibility status.

		:param newval: Visibility status value. If 1, the text is shown. If 0, the text is hidden
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text visibility status value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetVisibility(newval)

	def GetRotation(self) -> float:
		"""
		Gets the text item's rotation value.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0..&lt; 360.0 | Text rotation value in degrees supplied |
		| 0.0 | Text rotation value supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetRotation()

	def GetLanguageID(self) -> int:
		"""
		Gets the text item's translation identifier value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &lt;TranslationId&gt; | The text item's translation identifier |
		| 0 | Text item has no translation identifier or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLanguageID()

	def SetLanguageID(self, newval:int) -> int:
		"""
		Sets the text item's translation identifier value.

		:param newval: Translation identifier to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | No other value is returned |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLanguageID(newval)

	def GetAssignedOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of options assigned to the text item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of assigned options |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned option identifiers are supplied or an error occurred |
		| -1 | An option contains a Boolean expression or an alias |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetAssignedOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAlignment(self) -> int:
		"""
		Gets the text item's current text alignment value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Alignment value could not be supplied |
		| 1 | Left aligned |
		| 2 | Centered |
		| 3 | Right aligned |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetAlignment()

	def SetAlignment(self, newval:int) -> int:
		"""
		Sets the text item's text alignment value.

		:param newval: Alignment value to be set
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Alignment value set |
		| 1 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetAlignment(newval)

	def Delete(self) -> int:
		"""
		Deletes the text item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the text item |
		| 0 | Text item was deleted or no text item was set |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.Delete()

	def GetTypeId(self) -> int:
		"""
		Gets the text item's text type.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text type supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetTypeId()

	def SetSchemaLocation(self, x:float, y:float) -> int:
		"""
		Sets the text item's position.

		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the text item's sheet |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetSchemaLocation(x, y)

	def GetInternalText(self) -> str:
		"""
		Gets the text item's text value without text token translations.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Text value is supplied |
		| "&lt;Empty&gt;" | Text does not have a value or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetInternalText()

	def GetRightJustifiedSchemaLocation(self) -> tuple[int, float, float, str]:
		"""
		Gets the text item's box’s lower right corner position in text reading direction.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Position on the x-axis passed out of the function |
		| 2 | y | float | Position on the y-axis passed out of the function |
		| 3 | grid | str | Location in the format ". /&lt;sheet&gt;.&lt;grid&gt;. " passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Sheet idenifier of the text item |
		| 0 | Error has occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetRightJustifiedSchemaLocation(dummy, dummy, dummy)

	def GetWidth(self) -> float:
		"""
		Gets the text item's box width.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Width of the text supplied |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetWidth()

	def SetBallooning(self, onoff:bool, _type:int) -> int:
		"""
		Sets the text item's ballooning value.

		:param onoff: Indicating if the value should be added or removed. If 1, the value is added. If 0, the value is removed
		:type bool: bool
		:param _type: New ballooning value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Previous ballooning value, See Ballooning for possible values |
		| 0 | Previously no ballooning value set or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetBallooning(onoff, _type)

	def GetBallooning(self) -> int:
		"""
		Gets the text item's ballooning value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Ballooning value applied, See Ballooning for possible values |
		| 0 | Ballooning value could not be supplied or there is no ballooning |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetBallooning()

	def IsRedlined(self) -> int:
		"""
		Gets the text item's redlined status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Redlined status is on |
		| 0 | Redlined status is off |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.IsRedlined()

	def GetAllowedLength(self) -> int:
		"""
		Gets the maximum number of characters allowed for the text item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Maximum text length value |
		| 0 | No maximum text length supplied |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetAllowedLength()

	def SetRotation(self, rotation:float) -> float:
		"""
		Sets the text item's rotation value.

		:param rotation: Rotation value in degrees to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0..&lt; 360.0 | Previous text rotation value in degrees supplied |
		| 0.0 | Previous text rotation value or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetRotation(rotation)

	def GetSingleLine(self) -> int:
		"""
		Gets the text item's flag value determining if multi-line text is displayed on a single line.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text is displayed on a single line |
		| 0 | Text is not displayed on a single line or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetSingleLine()

	def SetSingleLine(self, newval:int) -> int:
		"""
		Sets the text item's flag value determining if multi-line text is displayed on a single line.

		:param newval: Single line value to apply. If 1, mutli-line text is displayed on a single line. If 0, mutli-line text is displayed on a multiple lines
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Single line value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetSingleLine(newval)

	def GetBox(self) -> tuple[int, float, float]:
		"""
		Gets the text item's box width and height.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xsize | float | Width of box passed out of the function |
		| 2 | ysize | float | Height of box passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | xsize and ysize values supplied |
		| 0 | Text is in its original size or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		return self._obj.GetBox(dummy, dummy)

	def SetBox(self, xsize:float, ysize:float) -> int:
		"""
		Sets the text item's box width and height.

		:param xsize: Width of box
		:type float: float
		:param ysize: Height of box
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Box values applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetBox(xsize, ysize)

	def DeleteBox(self) -> int:
		"""
		Deletes the text item's text box.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text box was deleted |
		| 0 | Delete text box operation failed |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.DeleteBox()

	def GetLocking(self) -> bool:
		"""
		Gets the text item's flag value determining if the text position is locked.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text position is locked |
		| 0 | Text position is unlocked or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetLocking()

	def SetLocking(self, newval:bool) -> bool:
		"""
		Gets the text item's flag value determining if the text position is locked.

		:param newval: Position locking value to apply. 1 will lock the text position. 0 will unlock the text position
		:type bool: bool

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text position locking value is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetLocking(newval)

	def GetHyperlinkAddress(self) -> str:
		"""
		Gets the text item's hyperlink address.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Hyperlink address supplied |
		| "&lt;Empty&gt;" | An error occurred or the text does not have a hyperlink address |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.GetHyperlinkAddress()

	def SetHyperlinkAddress(self, newtext:str) -> int:
		"""
		Sets the text item's hyperlink address.

		:param newtext: Hyperlink address to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Hyperlink address applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		return self._obj.SetHyperlinkAddress(newtext)

	def CalculateBoxAt(self, shti:int, text:str, x:float, y:float, rotation:float, height:float, mode:int, style:int, fontname:str, just:int, balloon:int) -> tuple[int, tuple[float,...], tuple[float,...]]:
		"""
		Calculates the placement position of a new text box.

		:param shti: Currently not used
		:type int: int
		:param text: Text of the text field
		:type str: str
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rotation: Rotation of the text field
		:type float: float
		:param height: Size of the text field font
		:type float: float
		:param mode: Ratio of the text field font
		:type int: int
		:param style: Style of the text field font
		:type int: int
		:param fontname: Name of the text field font. "Arial" is set as default if value is empty
		:type str: str
		:param just: Alignment of the text field
		:type int: int
		:param balloon: Balloon style of the text field. See Ballooning for possible values
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | bx | tuple[float,...] | An array of 4 elements representing the corner positions on the x-axis passed out of the function |
		| 2 | by | tuple[float,...] | An array of 4 elements representing the corner positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Calculation was successful |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, bx, by = self._obj.CalculateBoxAt(shti, text, x, y, rotation, height, mode, style, fontname, just, balloon, dummy, dummy)
		bx = bx[1:] if type(bx) == tuple and len(bx) > 0 else tuple()
		by = by[1:] if type(by) == tuple and len(by) > 0 else tuple()
		return ret, bx, by

	def SetOptionExpressions(self, expressions:list[str]) -> int:
		"""
		Replaces the text item's option expressions.

		:param expressions: Array of strings of all option names, Boolean expressions or alias names of Boolean expressions
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions assigned |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.12
		"""
		return self._obj.SetOptionExpressions(expressions)

	def GetAssignedOptionExpressionsEx(self, Term:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets option names and Boolean expressions (combinations of options)  including variant instances assigned to the text item.

		:param Term: Indicates whether aliases should be resolved. If the value is 1, alias names are resolved and the resulting expressions with option names are included in expressions., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | expressions | tuple[str,...] | Array of strings of all option names, Boolean expressions or alias names of Boolean expressions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in expressions |
		| 0 | No assigned option expressions are supplied or an error occurred |

		Available since E3.series TLB-Versions: 9.23
		"""
		dummy=0
		ret, expressions = self._obj.GetAssignedOptionExpressionsEx(dummy, Term)
		expressions = expressions[1:] if type(expressions) == tuple and len(expressions) > 0 else tuple()
		return ret, expressions

	def GetTextExtent(self) -> tuple[int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the dimensions of the text item's text value as a geometric shape.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xarr | tuple[float,...] | Array of dimension values on the x-axis passed out of the function |
		| 2 | yarr | tuple[float,...] | Array of dimension values on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text extent is supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.40
		"""
		dummy=0
		ret, xarr, yarr = self._obj.GetTextExtent(dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, xarr, yarr

	def GetTextExtentSingleLine(self) -> tuple[int, int, tuple[tuple[float,...],...], tuple[tuple[float,...],...]]:
		"""
		Gets the text item's dimension as a geometric shape for each line of text.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | nlines | int | Number of lines passed out of the function |
		| 2 | xarr | tuple[tuple[float,...],...] | Array of lines of dimension values on the x-axis passed out of the function |
		| 3 | yarr | tuple[tuple[float,...],...] | Array of lines of dimension values on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text extent is supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.40
		"""
		dummy=0
		ret, nlines, xarr, yarr = self._obj.GetTextExtentSingleLine(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, nlines, xarr, yarr

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the text item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the text item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.DeleteAttribute(name)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the text item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the text item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default value is "<Empty>", defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the text item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetAttributeValue(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the text item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.HasAttribute(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the text item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def SetRedlined(self, onoff:bool) -> int:
		"""
		Sets the text item's redlined status.

		:param onoff: Redlined status value to apply. If 1, the redlined status is set to on. If 0, the redlined status in set to off., defaults to TRUE
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previous redlined status was on |
		| 0 | Previous redlined status was off or an error occurred |

		Available since E3.series TLB-Versions: 17.04
		"""
		return self._obj.SetRedlined(onoff)

	def GetPictogram(self) -> bool:
		"""
		Gets the text item's flag value determining if the text is displayed in the pictogram language.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text is displayed in the pictogram language |
		| 0 | Text is not displayed in the pictogram language or an error occurred |

		Available since E3.series TLB-Versions: 17.70
		"""
		return self._obj.GetPictogram()

	def SetPictogram(self, newval:bool) -> bool:
		"""
		Sets the text item's flag value determining if the text is displayed in the pictogram language.

		:param newval: Text pictogram value to apply. 1 allows the text to use pictograms. 0 disallows the text to use pictograms
		:type bool: bool

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text pictogram value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 17.70
		"""
		return self._obj.SetPictogram(newval)

	def GetLinearMeasureWithoutUnit(self) -> bool:
		"""
		Gets the text item's flag value determining if the length measurement unit is displayed along with the value.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Length measurement unit is not shown |
		| 0 | Length measurement unit is shown or an error occurred |

		Available since E3.series TLB-Versions: 17.70
		"""
		return self._obj.GetLinearMeasureWithoutUnit()

	def SetLinearMeasureWithoutUnit(self, newval:bool) -> bool:
		"""
		Sets the text item's flag value determining if the length measurement unit is displayed along with the value.

		:param newval: Value for the hiding the length measure unit to apply. 1 hides the measurement unit. 0 shows the measurement unit
		:type bool: bool

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Value is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 17.70
		"""
		return self._obj.SetLinearMeasureWithoutUnit(newval)

	def CalculateBoxHeightEx(self, width:float, text:str, fontName:str, fontSize:float, fontMode:int, fontStyle:int) -> tuple[int, float, float, int]:
		"""
		Calculates the text box rectangle required to hold the given text.

		:param width: Width of the text field
		:type float: float
		:param text: Text of the text field
		:type str: str
		:param fontName: Name of the text field font
		:type str: str
		:param fontSize: Size of the text field font
		:type float: float
		:param fontMode: Ratio of the text field font
		:type int: int
		:param fontStyle: Style of the text field font
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | recHeight | float | Calculated height of the box |
		| 2 | recWidth | float | Calculated width of the box |
		| 3 | lines | int | Number of lines of text in the box |

		| Return Value | Description |
		|:-----:|:-----|
		| 3 | No calculation was performed |
		| 1 | Calculation was performed but an alternative default value was used, For example in the case fontName value was not found |
		| 0 | Calculation was successful |
		| -1 | Error occurred due to invalid width |
		| -2 | Internal error |

		Available since E3.series TLB-Versions: 18.12, 17.33
		"""
		dummy=0
		return self._obj.CalculateBoxHeightEx(width, text, fontName, fontSize, fontMode, fontStyle, dummy, dummy, dummy)

	def SendToForeground(self) -> int:
		"""
		Moves the text item to the foreground.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Text item moved to the foreground |
		| -1 | No project open |
		| -2 | No text item set |
		| -3 | Text item not placed |
		| -4 | Error occurred |
		| -5 | Error occurred |
		| -6 | Text item is locked, Available since v2023-24.22 |

		Available since E3.series TLB-Versions: 21.12
		"""
		return self._obj.SendToForeground()

	def SendToBackground(self) -> int:
		"""
		Moves the text item to the background.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Text item moved to the background |
		| -1 | No project open |
		| -2 | No text item set |
		| -3 | Text item not placed |
		| -4 | Error occurred |
		| -5 | Error occurred |
		| -6 | Text item is locked, Available since v2023-24.22 |

		Available since E3.series TLB-Versions: 21.12
		"""
		return self._obj.SendToBackground()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current text item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current text item |
		| "&lt;Empty&gt;" | No text item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a text item as the current item.

		:param gid: Global identifier value of a text item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current text item |
		| "&lt;Empty&gt;" | No text item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IVariantInterface--------------------
class Variant:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Variant. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current variant item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current variant item identifier |
		| 0 | No variant item |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a variant as the current item.

		:param id: Unique value identifying a variant item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current variant item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the variant item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Variant name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the variant item's name.

		:param name: New variant name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Variant item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.SetName(name)

	def Create(self, name:str) -> int:
		"""
		Creates a new variant item.

		:param name: Path and name of the variant
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the created variant |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.Create(name)

	def Delete(self, _del:int) -> int:
		"""
		Deletes the variant item from the project.

		:param _del: Indicates whether items the variant is exclusively assigned to are also removed from the project.. If 0, the items are not deleted. If greater than 0, the items are deleted
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current variant item identifier |
		| 0 | Variant item was deleted or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.Delete(_del)

	def Search(self, name:str) -> int:
		"""
		Searches for a variant item matching the name.

		:param name: Name of the variant
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Found variant identifier |
		| 0 | No variant was found or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.Search(name)

	def IsActive(self) -> int:
		"""
		Get the active status of the current variant item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Variant is active |
		| 0 | Variant is inactive or an error occurred |

		Available since E3.series TLB-Versions: 3.00
		"""
		return self._obj.IsActive()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the variant item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the variant item |
		| "&lt;Empty&gt;" | No variant item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a variant as the current item.

		:param gid: Global identifier value of a variant item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current variant item |
		| "&lt;Empty&gt;" | No variant item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current variant item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current variant item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a variant as the current item.

		:param guid: Globally unique identifier value of a variant item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current variant item |
		| "&lt;Empty&gt;" | No variant item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- ISlotInterface--------------------
class Slot:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Slot. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current slot item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current slot item identifier |
		| 0 | No slot item |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a slot as the current item.

		:param id: Unique value identifying a slot item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current slot item identifier |
		| 0 | No slot item |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.SetId(id)

	def GetMountType(self) -> str:
		"""
		Gets the mount type description of the slot item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Slot item mount type description |
		| "&lt;Empty&gt;" | Mount type does not have a description or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.GetMountType()

	def GetFixType(self) -> int:
		"""
		Gets a value representing the fix type of the slot item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Error occurred |
		| 1..4 | Fix type value |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.GetFixType()

	def GetName(self) -> str:
		"""
		Gets the name of the slot item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Slot item name value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.GetName()

	def GetPosition(self, point:int) -> tuple[int, float, float, float]:
		"""
		Gets the position of the slot item.

		:param point: Indicates which position to supply. If 2, the second position is supplied. If any other value, the first position is supplied
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Position value of the x-axis passed out of the function |
		| 2 | y | float | Position value of the y-axis passed out of the function |
		| 3 | z | float | Position value of the z-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Position values are supplied |
		| 0 | Error occurred |
		| -4 | Slot area is not rectangular, Available since v2023-24.21 |

		Available since E3.series TLB-Versions: 9.00
		"""
		dummy=0
		return self._obj.GetPosition(point, dummy, dummy, dummy)

	def GetMountedDeviceIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the devices mounted on the slot item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of devices passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No mounted devices are found or an error occurred |

		Available since E3.series TLB-Versions: 19.22
		"""
		dummy=0
		ret, ids = self._obj.GetMountedDeviceIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetDirection(self) -> tuple[int, float, float, float]:
		"""
		Gets the direction of the slot.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Direction value of the x-axis |
		| 2 | y | float | Direction value of the y-axis |
		| 3 | z | float | Direction value of the z-axis |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Direction values are provided |
		| -1 | No project open or slot item set |
		| -2 | Error accessing slot data |

		Available since E3.series TLB-Versions: 21.00
		"""
		dummy=0
		return self._obj.GetDirection(dummy, dummy, dummy)

	def GetRotation(self) -> tuple[int, float]:
		"""
		Gets the rotation of the slot item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | angle | float | Rotation value in degrees |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Rotation value is supplied |
		| -1 | No project open or slot item set |
		| -2 | Error accessing slot data |
		| -3 | Area slots have no rotation value |
		| -4 | Error determining the slot rotation |

		Available since E3.series TLB-Versions: 21.00
		"""
		dummy=0
		return self._obj.GetRotation(dummy)

	def GetDefinedDirection(self) -> tuple[int, float, float, float]:
		"""
		Gets the database defined direction of the slot.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Direction value of the x-axis |
		| 2 | y | float | Direction value of the y-axis |
		| 3 | z | float | Direction value of the z-axis |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Direction values are provided |
		| -1 | No project open or slot item set |

		Available since E3.series TLB-Versions: 21.12
		"""
		dummy=0
		return self._obj.GetDefinedDirection(dummy, dummy, dummy)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current slot item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current slot item |
		| "&lt;Empty&gt;" | No slot item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a slot as the current item.

		:param gid: Global identifier value of a slot item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current slot item |
		| "&lt;Empty&gt;" | No slot item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetSlotName(self, flags:int=0) -> str:
		"""
		Gets the display name of the slot item.

		:param flags: Flag bit values specifying how the display name is supplied., defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Slot item display name value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.GetSlotName(flags)

	def GetAreaPolygon(self, flags:int=0) -> tuple[int, float, float, float]:
		"""
		Gets the polygon positions of the slot's area.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xarr | float | Array of position values on the x-axis passed out of the function |
		| 2 | yarr | float | Array of positions values on the y-axis passed out of the function |
		| 3 | zarr | float | Array of positions values on the z-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Position values are supplied |
		| -1 | No project open or slot item set |
		| -2 | Error accessing slot data |
		| -3 | Slot is not an area type slot |
		| -4 | flags is invalid |

		Available since E3.series TLB-Versions: 24.30
		"""
		dummy=0
		return self._obj.GetAreaPolygon(dummy, dummy, dummy, flags)

	def SetMountType(self, newval:str, flags:int=0) -> int:
		"""
		Sets the mount type description of the slot item.

		:param newval: New mount type description
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Mount type description changed |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |
		| -3 | Slot item is invalid |
		| -4 | Device is locked |
		| -5 | For line and point slots, newval is "&lt;Empty&gt;", Area slots do not have a mount type description |
		| -6 | newval is invalid |

		Available since E3.series TLB-Versions: 26.00
		"""
		return self._obj.SetMountType(newval, flags)

	def GetDefinedRotation(self, flags:int=0) -> tuple[int, float]:
		"""
		Gets the database defined rotation of the slot item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | angle | float | Rotation value in degrees |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Rotation value is supplied |
		| -1 | No project open or slot item set |
		| -2 | Error occurred |
		| -3 | Area slots have no rotation value |
		| -4 | Error occurred |
		| -5 | flags value is invalid |

		Available since E3.series TLB-Versions: 26.01
		"""
		dummy=0
		return self._obj.GetDefinedRotation(dummy, flags)

# -------------------- INetInterface--------------------
class Net:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Net. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current net item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current net item identifier |
		| 0 | No net item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a connection net as the current item.

		:param id: Unique value identifying a connection net item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current connection net item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the connection net item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the connection net item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the connection net item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the connection net item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the connection net item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the connection net item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the connection net item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def SetAttributeVisibility(self, name:str, onoff:int) -> int:
		"""
		Sets the visibility status of all text items representing the connection net item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param onoff: Indicates whether the text items' visibility status should be shown or hidden. If > 0, the text items are shown. If 0, the text items are hidden
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items changed |
		| 0 | No text items found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeVisibility(name, onoff)

	def DisplayAttributeValueAt(self, name:str, sheetid:int, x:float, y:float) -> int:
		"""
		Displays an attribute value from the connection net item at a specific position.

		:param name: Internal name of the attribute
		:type str: str
		:param sheetid: Sheet identifier to display the attribute value on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text item identifier displaying the attribute value |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DisplayAttributeValueAt(name, sheetid, x, y)

	def GetName(self) -> str:
		"""
		Gets the connection net item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Connection net name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def GetParentId(self) -> int:
		"""
		Gets the identifier of the global master connection net of the current local sheet connection net item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the global master connection net item |
		| 0 | No global master connection net found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetParentId()

	def GetNetIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the connection net's local connection net identifiers.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of net identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No nets or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetNetIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetNetSegmentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the net segments of the connection net.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of net segments passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of net segments in ids |
		| 0 | No net segments or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetNetSegmentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsSignalTransferred(self) -> int:
		"""
		Indicates whether signals are transferred to connections for the current connection net item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signals are transferred to connections |
		| 0 | Signals are not transferred to connections or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsSignalTransferred()

	def SetTransferSignal(self, value:int) -> int:
		"""
		Sets the net item's net transfer signal status value.

		:param value: Net transfer signal status value to apply. If 1, the net transfer signal is activated. If 0, the net transfer signal is deactivated
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..0 | Current net transfer signal status value |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetTransferSignal(value)

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the connection net item's pins.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No pin identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetCoreIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the conductors contained in the current connection net item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of conductors passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of conductors in ids |
		| 0 | No conductors or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetCoreIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetHarnessId(self) -> int:
		"""
		Gets the identifier of the net connection item's harness.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the harness |
		| 0 | No harness or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetHarnessId()

	def SplitHarness(self) -> int:
		"""
		Removes the current connection net item from the harness.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new harness for the connection net |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SplitHarness()

	def GetSignalName(self) -> str:
		"""
		Gets the connection net item's signal name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Signal name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 15.10
		"""
		return self._obj.GetSignalName()

	def GetTranslatedSignalName(self) -> str:
		"""
		Gets the connection net item's translated signal name value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Connection net item's translated signal name value is supplied |
		| "&lt;Empty&gt;" | No signal translated name value or an error occurred |

		Available since E3.series TLB-Versions: 15.10
		"""
		return self._obj.GetTranslatedSignalName()

	def SetSignalName(self, name:str) -> int:
		"""
		Sets the connection net item's signal name.

		:param name: New signal name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signal name |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 15.10
		"""
		return self._obj.SetSignalName(name)

	def SetSignalNameOnLocalNet(self, name:str) -> int:
		"""
		Sets the connection net item's signal name.

		:param name: New signal name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Signal name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 15.10
		"""
		return self._obj.SetSignalNameOnLocalNet(name)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the connection net item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the connection net item |
		| "&lt;Empty&gt;" | No connection net item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a connection net as the current item.

		:param gid: Global identifier value of a connection net item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current connection net item |
		| "&lt;Empty&gt;" | No connection net item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current connection net item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current connection net item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a connection net as the current item.

		:param guid: Globally unique identifier value of a connection net item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current connection net item |
		| "&lt;Empty&gt;" | No connection net item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- IUserMenuItemInterface--------------------
class UserMenuItem:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize UserMenuItem. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current menu item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current menu item identifier |
		| 0 | No menu item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a menu item as the current item.

		:param id: Unique value identifying a menu item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current menu item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def Create(self, id:int, text:str, command:str, parameters:str, folder:str, image:str, shortcut:str, visible:int, wait:int, flags:int=1) -> int:
		"""
		Creates a new user menu item.

		:param id: Identifier of the user menu item
		:type int: int
		:param text: Display text and menu position. See Menu Item Text for possible values
		:type str: str
		:param command: Path and name of the user tool
		:type str: str
		:param parameters: Argument values to be passed into the user tool
		:type str: str
		:param folder: Path of the working folder of the user tool
		:type str: str
		:param image: Path and name of a Windows Bitmap (.bmp) file for the display icon
		:type str: str
		:param shortcut: Short cut key combination. See Short Cut for possible values
		:type str: str
		:param visible: Executable window status when run in console execution mode. If 0, the executable window is hidden. If 1, the executable window is shown. If 2, the executable window is minimized. If 3, the executable window is maximized
		:type int: int
		:param wait: Indicates whether the. <span style="font-style: italic;color: #ff0000;">E³.</span>series. process should pause until the user tool execution is finished. Since v2011-10.00 this functionality is no longer available
		:type int: int
		:param flags: User menu item option flags., defaults to 1
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new user menu item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(id, text, command, parameters, folder, image, shortcut, visible, wait, flags)

	def Delete(self) -> int:
		"""
		Deletes the current menu item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Menu item was deleted |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def GetText(self) -> str:
		"""
		Gets the menu item's display text.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Display text |
		| "&lt;Empty&gt;" | No display text or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetText()

	def SetText(self, newval:str) -> int:
		"""
		Sets the menu item's display text.

		:param newval: Display text
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display text applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetText(newval)

	def GetCommand(self) -> str:
		"""
		Gets the path and name of the menu item's user tool.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;User Tool&gt;" | Path and name of the menu item's user tool |
		| "&lt;Empty&gt;" | No user tool path and name or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCommand()

	def SetCommand(self, newval:str) -> int:
		"""
		Sets the path and name of the menu item's user tool.

		:param newval: Path and name of the user tool
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Path and name of the user tool applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetCommand(newval)

	def GetParameters(self) -> str:
		"""
		Gets the menu item's argument values to be passed into the user tool.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Parameters&gt;" | Argument values |
		| "&lt;Empty&gt;" | No argument values or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetParameters()

	def SetParameters(self, newval:str) -> int:
		"""
		Sets the menu item's argument values to be passed into the user tool.

		:param newval: Argument values to be passed into the user tool
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Argument values applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetParameters(newval)

	def GetFolder(self) -> str:
		"""
		Gets the menu item user tool's working folder.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Path&gt;" | Path of the user tool's working folder |
		| "&lt;Empty&gt;" | No working folder or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetFolder()

	def SetFolder(self, newval:str) -> int:
		"""
		Sets the menu item user tool's working folder

		:param newval: Working folder of the user tool to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Working folder value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetFolder(newval)

	def GetImage(self) -> str:
		"""
		Gets the menu item's icon path and file name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File Path&gt;" | Path and file name of the icon image |
		| "&lt;Empty&gt;" | No icon image path and file name or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetImage()

	def SetImage(self, newval:str) -> int:
		"""
		Sets the menu item's icon path and file name.

		:param newval: Path and name of a Windows Bitmap (.bmp) file for the  display icon
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Icon image file is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetImage(newval)

	def GetShortCut(self) -> str:
		"""
		Gets the menu item's short cut key combination.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Short Cut&gt;" | Short cut key combination |
		| "&lt;Empty&gt;" | No short cut key combination or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetShortCut()

	def SetShortCut(self, newval:str) -> int:
		"""
		Sets the menu item's short cut key combination.

		:param newval: Short cut key combination
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Short cut key combination value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetShortCut(newval)

	def GetVisible(self) -> int:
		"""
		Gets the executable window state of the menu item's user tool when run in console execution mode.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 3 | Window state is maximized |
		| 2 | Window state is minimized |
		| 1 | Window state is shown |
		| 0 | Window state is hidden or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetVisible()

	def SetVisible(self, newval:int) -> int:
		"""
		Sets the executable window state of the menu item's user tool when run in console execution mode.

		:param newval: User tool window state
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | User tool window state value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetVisible(newval)

	def GetEnable(self) -> int:
		"""
		Gets the menu item active status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Menu item is active |
		| 0 | Menu item is inactive |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetEnable()

	def SetEnable(self, newval:int) -> int:
		"""
		Sets the menu item active status.

		:param newval: Indicates the active status to apply. If 0, the menu item is inactive. If 1, the menu item is active
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Active status is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetEnable(newval)

	def GetWaitForEndOfCommand(self) -> int:
		"""
		Indicates whether the E³.series process is paused during the execution life time of the user tool.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | E³.series process is paused during the execution life time of the user tool |
		| 0 | E³.series process runs during the execution life time of the user tool, the functionality is no longer supported or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetWaitForEndOfCommand()

	def SetWaitForEndOfCommand(self, newval:int) -> int:
		"""
		Sets whether the E³.series process is paused during the execution life time of the user tool.

		:param newval: Value indicating whether the. <span style="font-style: italic;color: #ff0000;">E³.</span>series. process is paused during the execution life time of the user tool. If 1, the E³.series process is paused during the execution life time of the user tool. If 0, the E³.series process runs during the execution life time of the user tool
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Value applied |
		| 0 | Functionality is no longer supported or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetWaitForEndOfCommand(newval)

	def CreateSeparator(self, text:str) -> int:
		"""
		Creates a new separator menu item.

		:param text: Display text and menu position. See Menu Item Text for possible values
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &lt; 0 | Identifier of the new separator menu item or an error occurred |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateSeparator(text)

	def CreateUserTool(self, text:str, command:str) -> int:
		"""
		Creates a new user tool menu item.

		:param text: Display text and menu position. See Menu Item Text for possible values
		:type str: str
		:param command: Path and name of the user tool
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new user tool menu item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateUserTool(text, command)

	def UpdateUserInterface(self) -> int:
		"""
		Updates the E³.series menus and short cuts.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | E³.series menus and short cuts updated |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.UpdateUserInterface()

	def CreateContextUserTool(self, text:str, command:str) -> int:
		"""
		Creates a new user tool menu item for the context menu.

		:param text: Display text and context menu position. See Menu Item Text for possible values
		:type str: str
		:param command: Path and name of the user tool
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new user tool context menu item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateContextUserTool(text, command)

	def CreateContextSeparator(self, text:str) -> int:
		"""
		Creates a new separator menu item for the context menu.

		:param text: Display text and menu position. See Menu Item Text for possible values
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &lt; 0 | Identifier of the new separator context menu item or an error occurred |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateContextSeparator(text)

	def DeleteContext(self) -> int:
		"""
		Deletes the current context menu item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Context menu item was deleted or an error occurred |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteContext()

	def IsDeleted(self) -> int:
		"""
		Indicates if the system menu item has been deleted.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | System menu item has been deleted |
		| 0 | System menu item has not been deleted |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsDeleted()

	def UnDelete(self) -> int:
		"""
		Restores the deleted menu item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value has been returned since v2011-10.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.UnDelete()

	def GetType(self) -> int:
		"""
		Gets a value representing the type of the menu item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type value of the menu item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetType()

	def GetDescription(self) -> str:
		"""
		Gets the description of the menu item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;User Tool&gt;" | Menu item description |
		| "&lt;Empty&gt;" | No menu item description or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetDescription()

	def SetDescription(self, newval:str) -> int:
		"""
		Sets the description of the menu item.

		:param newval: Description of the menu item
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Description applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetDescription(newval)

	def DeleteUserTool(self) -> int:
		"""
		Deletes the current menu item and associated user tool.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Menu item and user tool were deleted |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.DeleteUserTool()

# -------------------- IStructureNodeInterface--------------------
class StructureNode:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize StructureNode. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current structure node item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current structure node item identifier |
		| 0 | No structure node item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a structure node as the current item.

		:param id: Unique value identifying a structure node item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current structure node item identifier |
		| 0 | No structure node item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetParentId(self) -> int:
		"""
		Gets the identifier of the parent structure node of the current structure node item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the parent structure node |
		| 0 | No parent structure node or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetParentId()

	def GetTypeName(self) -> str:
		"""
		Gets the type of the structure node.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Type&gt;" | Structure node type |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTypeName()

	def GetName(self) -> str:
		"""
		Gets the structure node item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Structure node item's name |
		| "&lt;Empty&gt;" | Unable to supply the name |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the structure node item's name.

		:param name: New name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Structure node item's name |
		| 0 | Error occurred |
		| -3 | Structure node is locked, Available since v2017-18.00 |
		| -4 | Project structure is locked, Available since v2018-19.12 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def Create(self, name:str, typname:str, parentid:int, afterid:int) -> int:
		"""
		Creates a new structure node item.

		:param name: Name of the new structure node item
		:type str: str
		:param typname: Type of the new structure node item
		:type str: str
		:param parentid: Identifier of the parent structure node of the new structure node item
		:type int: int
		:param afterid: Identifier of an existing structure node the new structure node item should be placed after
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new structure node item |
		| 0 | Error occurred |
		| -3 | Structure node is locked, Available since v2017-18.00 |
		| -4 | Project structure is locked, Available since v2018-19.12 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(name, typname, parentid, afterid)

	def Delete(self) -> int:
		"""
		Deletes the current structure node item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Old structure node item identifier |
		| 0 | Error occurred |
		| -1 | Invalid structure node item |
		| -2 | Structure node cannot be deleted |
		| -3 | Structure node is locked, Available since v2017-18.00 |
		| -4 | Project structure is locked, Available since v2018-19.12 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def GetStructureNodeIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the current structure node item's child structure nodes.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of structure nodes identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | Current structure node item contains no child structure nodes or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetStructureNodeIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSheetIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the structure node item's sheets.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of sheet identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | Structure node item contains no sheets or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSheetIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetExternalDocumentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the structure node item's external documents.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of external document identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of external document identifiers in ids |
		| 0 | No external documents or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetExternalDocumentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAllDeviceIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the structure node item's devices.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of device identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of device identifiers in ids |
		| 0 | No devices or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAllDeviceIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the structure node item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the structure node item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default value is "<Empty>", defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the structure node item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the structure node item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the structure node item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the structure node item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def MoveTo(self, parentId:int, afterId:int=0) -> int:
		"""
		Moves the structure node item's position.

		:param parentId: Identifier of the destination structure node to place the current structure node within
		:type int: int
		:param afterId: Identifier of the destination structure node after which to place the current structure node. If 0, the current structure node is placed at the end of the parentId node's content. Available since v2020-21.31 and v2021-22.12:. If 1, the current structure node is placed at the beginning of the parentId node's content., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Structure node item has been moved |
		| -1 | Error occurred |
		| -3 | Structure node item is locked, Available since v2017-18.00 |
		| -4 | Project structure is locked, Available since v2018-19.12 |
		| -5 | parentId or afterId is invalid, Available since v2020-21.31 and v2021-22.12 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.MoveTo(parentId, afterId)

	def SetStructureNodeIds(self, ids:list[int]) -> int:
		"""
		Orders the child structure nodes of the structure node item.

		:param ids: Array of all child structure node identifiers contained in the current structure node in the new order
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Child structure nodes  are reordered in  the structure node |
		| -1 | Error occurred |
		| -3 | Structure node is locked, Available since v2017-18.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.SetStructureNodeIds(ids)
		return ret[0]

	def SetSheetIds(self, ids:list[int]) -> int:
		"""
		Orders the sheets of the structure node item.

		:param ids: Array of all sheet identifiers contained in the current structure node listed in the new order
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Sheets are reordered in  the structure node |
		| -1 | Error occurred |
		| -3 | Structure node is locked, Available since v2017-18.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.SetSheetIds(ids)
		return ret[0]

	def SetExternalDocumentIds(self, ids:list[int]) -> int:
		"""
		Orders the external documents of the structure node item.

		:param ids: Array of all external document identifiers contained in the current structure node in the new order
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | External documents are reordered in  the structure node |
		| -1 | Error occurred |
		| -3 | Structure node is locked, Available since v2017-18.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.SetExternalDocumentIds(ids)
		return ret[0]

	def IsObjectTypeAllowed(self, sheets:bool, devices:bool) -> int:
		"""
		Indicates whether sheets and devices are allowed under the structure node item.

		:param sheets: Indicates whether to query for sheet. If 1, sheets are queried. If 0, sheets are not queried
		:type bool: bool
		:param devices: Indicates whether to query for devices. If 1, devices are queried. If 0, devices are not queried
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Queried item types are allowed under the structure node item |
		| 0 | Queried item types are not allowed under the structure node item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsObjectTypeAllowed(sheets, devices)

	def GetInternalName(self) -> str:
		"""
		Gets	the structure node item's internal name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Structure node item's internal name |
		| "&lt;Empty&gt;" | Error occurred  or the structure node has no internal name |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetInternalName()

	def GetAllSignalIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the structure node item's signals.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of signal identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of signal identifiers in ids |
		| 0 | No signals or an error occurred |

		Available since E3.series TLB-Versions: 10.20
		"""
		dummy=0
		ret, ids = self._obj.GetAllSignalIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Highlight(self, colour:int, width:float) -> int:
		"""
		Highlights the items assigned to the current structure node item.

		:param colour: Color value to apply. See Colors for possible values
		:type int: int
		:param width: Highlight line width
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Error occurred |
		| 0 | Items assigned to the current structure node item are highlighted or an error occurred |
		| -1 | Root level structure node items cannot be highlighted |

		Available since E3.series TLB-Versions: 11.70
		"""
		return self._obj.Highlight(colour, width)

	def ResetHighlight(self) -> int:
		"""
		Resets all highlighted items of the current structure node to their normal display mode.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Highlighting reset for items assigned to the current structure node item or an error occurred |
		| -1 | Root level structure node items cannot be highlighted |

		Available since E3.series TLB-Versions: 11.70
		"""
		return self._obj.ResetHighlight()

	def IsLocked(self) -> int:
		"""
		Indicates the structure node item's locked status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Structure node is unlocked or an error occurred |
		| 1 | Structure node is locked |
		| 2 | Structure node is indirect locked by a sheet |
		| 3 | Structure node is locked and indirect locked by a sheet |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.IsLocked()

	def LockObject(self) -> int:
		"""
		Locks the structure node item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Structure node locked |
		| -1 | Error occurred |
		| -2 | Structure node item is already locked |
		| -3 | Error occurred |
		| -1009 | No project open or no structure node item set |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.LockObject()

	def UnlockObject(self, password:str) -> int:
		"""
		Unlocks the structure node item.

		:param password: Password needed to unlock the structure node
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Structure node item has been unlocked |
		| -1 | Error occurred |
		| -2 | Structure node item is already unlocked |
		| -3 | Error occurred |
		| -1009 | No project open or no structure node item set |

		Available since E3.series TLB-Versions: 18.00
		"""
		return self._obj.UnlockObject(password)

	def DeleteForced(self) -> int:
		"""
		Deletes the current structure node item and its content from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Old structure node item identifier |
		| 0 | Error occurred |
		| -1 | Invalid structure node item |
		| -2 | Structure node cannot be deleted |
		| -3 | Structure node is locked |
		| -4 | Project structure is locked, Available since v2018-19.12 |

		Available since E3.series TLB-Versions: 18.70
		"""
		return self._obj.DeleteForced()

	def IsLockedByAccessControl(self) -> int:
		"""
		Indicates the structure node item's E³PluginAccessControl lock status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 3 | Structure node is locked and also indirectly by the sheet |
		| 2 | Structure node is locked indirectly by the sheet |
		| 1 | Structure node is locked |
		| 0 | Structure node is unlocked or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.IsLockedByAccessControl()

	def GetSheetAndExternalDocumentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the structure node item's sheet and external documents.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of sheet and external document identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of sheet and external document identifiers in ids |
		| 0 | No sheets and external documents or an error occurred |

		Available since E3.series TLB-Versions: 19.22
		"""
		dummy=0
		ret, ids = self._obj.GetSheetAndExternalDocumentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the structure node item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the structure node item |
		| "&lt;Empty&gt;" | No structure node item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a structure node as the current item.

		:param gid: Global identifier value of a structure node item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current structure node item |
		| "&lt;Empty&gt;" | No structure node item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current structure node item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current structure node item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a structure node as the current item.

		:param guid: Globally unique identifier value of a structure node item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current structure node item |
		| "&lt;Empty&gt;" | No structure node item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- ITreeInterface--------------------
class Tree:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Tree. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current tree item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current tree item identifier |
		| 0 | No tree item |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, treeid:int) -> int:
		"""
		Sets a tree item as the current item.

		:param treeid: Unique value identifying a tree item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current tree item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(treeid)

	def GetName(self) -> str:
		"""
		Gets the tree item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Tree item's name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the tree item's name.

		:param name: New tree item name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New tree item name |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def GetVisibleObjectTypes(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of the project tree item's visible object types and the settings about their displayed information.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | type_array | tuple[int,...] | Array containing the object type information passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of object types in type_array |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, type_array = self._obj.GetVisibleObjectTypes(dummy)
		type_array = type_array[1:] if type(type_array) == tuple and len(type_array) > 0 else tuple()
		return ret, type_array

	def SetVisibleObjectTypes(self, type_array:list[int]) -> int:
		"""
		Sets the project tree item's visible object types and the settings about their displayed information.

		:param type_array: Array containing the object type information passed into the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Tree item's visible object types and their display information are applied |
		| 0 | Error occurred |
		| -1 | Tree not found |
		| -4 | type_array is invalid |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.SetVisibleObjectTypes(type_array)
		return ret[0]

	def GetSortingMethod(self) -> tuple[int, int, tuple[tuple[str,int,str,str,int],...], tuple[tuple[str,str,int],...]]:
		"""
		Gets sorting method of elements in the tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | flags | int | Global setting values relevant to the sorting method passed out of the function |
		| 2 | structure | tuple[tuple[str,int,str,str,int],...] | Array describing the levels structure used for sorting passed out of the function |
		| 3 | freetab | tuple[tuple[str,str,int],...] | Array describing the defined keys used for sorting passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sorting method supplied |
		| ≠1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, flags, structure, freetab = self._obj.GetSortingMethod(dummy, dummy, dummy)
		structure = structure[1:] if type(structure) == tuple and len(structure) > 0 else tuple()
		freetab = freetab[1:] if type(freetab) == tuple and len(freetab) > 0 else tuple()
		return ret, flags, structure, freetab

	def SetSortingMethod(self, flags:int, structure:list[tuple[str,int,str,str,int]], freetab:list[tuple[str,str,int]]) -> int:
		"""
		Sets sorting method of elements in the project tree.

		:param flags: Global settings values relevant to the sorting method
		:type int: int
		:param structure: Array describing the levels structure used for sorting passed into the function
		:type list[tuple[str,int,str,str,int]]: list[tuple[str,int,str,str,int]]
		:param freetab: Array describing the defined keys used for sorting passed into the function
		:type list[tuple[str,str,int]]: list[tuple[str,str,int]]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sorting method is applied |
		| ≠1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		structure = [("",0,"","",0)] + structure
		freetab = [("","",0)] + freetab
		ret = self._obj.SetSortingMethod(flags, structure, freetab)
		return ret[0]

	def SetIcon(self, filename:str, index:int=0) -> int:
		"""
		Sets the tree item's display icon.

		:param filename: Path and name of an icon file
		:type str: str
		:param index: Index of an icon within. filename. if it is a file containing icon resources., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New icon is applied |
		| 0 | Tree item cannot be set with a new icon |
		| -1 | No tree item set |
		| -2 | filename is invalid |
		| -3 | Icon cannot be loaded |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetIcon(filename, index)

	def GetSelectedSheetIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of sheets selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of sheets selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No sheets are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedSheetIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedDeviceIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of devices selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of devices selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No devices are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedDeviceIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedCableIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of cables selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of cables selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No cables or cable sub-elements are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedCableIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedTerminalIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of terminals selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of terminals selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No terminals are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedTerminalIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedConnectorIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of connectors selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of connectors selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No connectors are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedConnectorIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedBlockIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of blocks selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of blocks selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No blocks are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedBlockIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedAllDeviceIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of devices of all types selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of devices selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No devices are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedAllDeviceIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of symbols and gates selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbols and gates selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No symbols or gates are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedBundleIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of bundles selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of bundles selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No bundles are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedBundleIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of pins selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No pins are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedStructureNodeIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of structure nodes selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of structure nodes selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No structure nodes are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedStructureNodeIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsVisible(self) -> int:
		"""
		Indicates whether the tree item is displayed.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Tree item is displayed |
		| 0 | Tree item is not displayed or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsVisible()

	def IsActive(self) -> int:
		"""
		Indicates whether the tree item is the currently active tree.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Tree item is active |
		| 0 | Tree item in inactive or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsActive()

	def Create(self, name:str, position:int=0, before:int=0) -> int:
		"""
		Creates a new tree item and places it in the project tree structure.

		:param name: Name of the tree item
		:type str: str
		:param position: Identifier of an existing tree item to use as the insertion position of the new tree item in the project tree structure., defaults to 0
		:type int: int
		:param before: Indicates whether to insert the tree item before or after. position. If 1, the tree item is inserted before position. If 0, the tree item is inserted after position., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new tree item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(name, position, before)

	def Delete(self) -> int:
		"""
		Deletes the current tree item from the project tree structure.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Tree item was deleted or an error occurred |
		| -1 | Tree item not found |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def GetSelectedExternalDocumentIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of external document items selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of external document items selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No external document items are selected or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedExternalDocumentIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetNames(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets the tree item's names in all languages.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | names | tuple[str,...] | Array of tree item names by language passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of name elements in names |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		dummy=0
		ret, names = self._obj.GetNames(dummy)
		names = names[1:] if type(names) == tuple and len(names) > 0 else tuple()
		return ret, names

	def SetNames(self, names:list[str]) -> int:
		"""
		Sets the tree item's names in all languages.

		:param names: Array of tree item names by language passed into the function
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Number of name elements in names |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		ret = self._obj.SetNames(names)
		return ret[0]

	def GetSelectedSheetIdsByFolder(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of sheets in selected folders in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of sheets in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No folders containing sheets are selected or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedSheetIdsByFolder(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedExternalDocumentIdsByFolder(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of external documents in selected folders in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of external documents in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No folders containing external documents are selected or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedExternalDocumentIdsByFolder(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSelectedAllDeviceIdsByFolder(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of devices in selected folders in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of devices in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No folders containing devices are selected or an error occurred |

		Available since E3.series TLB-Versions: 9.10
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedAllDeviceIdsByFolder(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def ViewSignalTree(self, bShowTree:bool) -> int:
		"""
		Gets a unique value identifying the signal tree item and shows or hides it.

		:param bShowTree: Indicates if the signal tree should be displayed. If 1, the signal tree is shown. If 0, the signal tree is hidden
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Signal tree identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.20
		"""
		return self._obj.ViewSignalTree(bShowTree)

	def GetVisibleInfoTypesEx(self) -> tuple[int, tuple[int,...], tuple[int,...], tuple[int,...]]:
		"""
		Gets arrays of views, schematic types and formboard sheets displayed in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | views | tuple[int,...] | Array of displayed view numbers passed out of the function |
		| 2 | schematicTypes | tuple[int,...] | Array of displayed schematic types passed out of the function. See Schematic for possible values |
		| 3 | formboardSheetIds | tuple[int,...] | Array of identifiers of formboard sheets passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Views, schematic type and formboard display information is supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.70
		"""
		dummy=0
		ret, views, schematicTypes, formboardSheetIds = self._obj.GetVisibleInfoTypesEx(dummy, dummy, dummy)
		views = views[1:] if type(views) == tuple and len(views) > 0 else tuple()
		schematicTypes = schematicTypes[1:] if type(schematicTypes) == tuple and len(schematicTypes) > 0 else tuple()
		formboardSheetIds = formboardSheetIds[1:] if type(formboardSheetIds) == tuple and len(formboardSheetIds) > 0 else tuple()
		return ret, views, schematicTypes, formboardSheetIds

	def SetVisibleInfoTypesEx(self, views:list[int], schematicTypes:list[int], formboardSheetIds:list[int]) -> int:
		"""
		Sets the views, schematic types and formboard sheets displayed in the project tree item.

		:param views: Array of displayed view numbers
		:type list[int]: list[int]
		:param schematicTypes: Array of displayed schematic types. See Schematic for possible values
		:type list[int]: list[int]
		:param formboardSheetIds: Array of identifiers of formboard sheets
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Views, schematic type and formboard display information is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 16.70
		"""
		return self._obj.SetVisibleInfoTypesEx(views, schematicTypes, formboardSheetIds)

	def GetTreeType(self) -> int:
		"""
		Gets a value representing the tree item's subtype.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| -1 | Error occurred |
		| 0..2 | Tree item's subtype |

		Available since E3.series TLB-Versions: 17.03
		"""
		return self._obj.GetTreeType()

	def GetSelectedBusbarIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of busbars selected in the project tree item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of busbars selected in the project tree item passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of identifiers in ids |
		| 0 | No busbars or busbar sub-elements are selected or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		dummy=0
		ret, ids = self._obj.GetSelectedBusbarIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

# -------------------- IDimensionInterface--------------------
class Dimension:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Dimension. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def SetId(self, id:int) -> int:
		"""
		Sets a dimension as the current item.

		:param id: Unique value identifying a dimension item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current dimension item identifier |
		| 0 | No current dimension item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetId(self) -> int:
		"""
		Gets the identifier of the current dimension item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current dimension item identifier |
		| 0 | No dimension item  or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def Delete(self) -> int:
		"""
		Deletes the dimension item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the dimension item |
		| 0 | No dimension item was set |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def Create(self, shtid:int, x1:float, y1:float, x2:float, y2:float, flags:int, distance:float, text:str, tx:float, ty:float) -> int:
		"""
		Adds a new dimension item onto a sheet of the project.

		:param shtid: Identifier of the sheet to place the dimension item on
		:type int: int
		:param x1: First placement position on the x-axis
		:type float: float
		:param y1: First placement position on the y-axis
		:type float: float
		:param x2: Second placement position on the x-axis
		:type float: float
		:param y2: Second placement position on the y-axis
		:type float: float
		:param flags: Flags bit values specifying how the dimension item should be created
		:type int: int
		:param distance: Distance of the dimension line from the placement positions, from the reference point or the radius of the dimension item
		:type float: float
		:param text: Defines the text to be displayed
		:type str: str
		:param tx: Placement position of the displayed text on the x-axis
		:type float: float
		:param ty: Placement position of the displayed text on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new dimension item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(shtid, x1, y1, x2, y2, flags, distance, text, tx, ty)

	def GetParameters(self) -> tuple[int, float, float, float, float, float, int, str, float, float]:
		"""
		Gets the properties of the dimension line.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x1 | float | Dimension first point x-axis value |
		| 2 | y1 | float | Dimension first point y-axis value |
		| 3 | x2 | float | Dimension second point x-axis value |
		| 4 | y2 | float | Dimension second point  y-axis value |
		| 5 | distance | float | Dimension line distance |
		| 6 | flags | int | See. Create(). for flag value descriptions. Bit 5 (0x0020) is not returned |
		| 7 | text | str | Dimension display text |
		| 8 | tx | float | Display text position's x-axis value |
		| 9 | ty | float | Display text position's y-axis value |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Sheet identifier of the dimension |
		| 0 | For example no dimension item is set |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetParameters(dummy, dummy, dummy, dummy, dummy, dummy, dummy, dummy, dummy)

	def SetArrowMode(self, value:int) -> int:
		"""
		Sets the dimension's current arrow style value.

		:param value: Arrow style value. See Arrow Style for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 or 2 | Previous assigned arrow style value, See Arrow Style for possible values |
		| 0 | Previous assigned arrow style value or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetArrowMode(value)

	def GetArrowMode(self) -> int:
		"""
		Gets the dimension's current arrow style value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Filled arrow style or an error occurred |
		| 1..5 | Current arrow style, See Arrow Style for possible values |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetArrowMode()

	def SetExtension(self, value:float) -> float:
		"""
		Sets the distance of the dimension item's extension lines beyond the arrows.

		:param value: Distance of the extension lines beyond the arrows to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Previous assigned extension line distance value beyond the arrows |
		| 0.0 | Previous assigned  line distance beyond the arrows or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetExtension(value)

	def GetExtension(self) -> float:
		"""
		Gets the distance of the dimension item's extension lines beyond the arrows.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Extension line distance value beyond the arrows |
		| 0.0 | No extension line distance beyond the arrows or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetExtension()

	def SetLineWidth(self, value:float) -> float:
		"""
		Sets the line width of a dimension item.

		:param value: Line width value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Previous assigned line width value |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetLineWidth(value)

	def GetLineWidth(self) -> float:
		"""
		Gets the line width of a dimension item.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Line width value |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetLineWidth()

	def SetPrecision(self, value:int) -> int:
		"""
		Sets the number of decimal places for displaying generated measurements on the dimension item's text.

		:param value: Number of decimal places to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..4 | Previous number of decimal places assigned |
		| 0 | No decimal places were previously assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetPrecision(value)

	def GetPrecision(self) -> int:
		"""
		Gets the number of decimal places when displaying generated measurements on the dimension item's text.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of decimal places |
		| 0 | No decimal places or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPrecision()

	def SetOffset(self, value:float) -> float:
		"""
		Sets the distance between the dimension item's text and dimension line.

		:param value: Distance between the dimension item's text and dimension line to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &lt; 0.0 or &gt; 0.0 | Previous assigned distance between the text and dimension line |
		| 0.0 | Previous assigned distance between the text and dimension line or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetOffset(value)

	def GetOffset(self) -> float:
		"""
		Gets the distance between the dimension item's text and dimension line.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &lt; 0.0 or &gt; 0.0 | Distance between the  text and dimension line |
		| 0.0 | No distance between the  text and dimension line or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetOffset()

	def SetPrefix(self, value:str) -> str:
		"""
		Sets the dimension item's text prefix.

		:param value: Text prefix value to apply
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Previous assigned text prefix |
		| "&lt;Empty&gt;" | No previous assigned text prefix or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetPrefix(value)

	def GetPrefix(self) -> str:
		"""
		Gets the dimension item's text prefix.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Dimension text prefix |
		| "&lt;Empty&gt;" | No dimension text prefix or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPrefix()

	def SetSuffix(self, value:str) -> str:
		"""
		Sets the dimension item's text suffix.

		:param value: Text suffix value to apply
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Previous assigned text suffix value |
		| "&lt;Empty&gt;" | Previous assigned text suffix value  or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetSuffix(value)

	def GetSuffix(self) -> str:
		"""
		Gets the dimension item's text suffix.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Dimension text suffix |
		| "&lt;Empty&gt;" | No dimension text suffix or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSuffix()

	def SetText(self, text:str, fixed:int) -> int:
		"""
		Sets the dimension item's text value.

		:param text: Text value to apply
		:type str: str
		:param fixed: Indicates whether the text value or the dimension measurement is displayed. If 0, the dimension measurement is displayed. If 1, the text value is displayed
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text value is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetText(text, fixed)

	def GetText(self) -> tuple[int, str, int]:
		"""
		Gets the dimension item's text value.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | text | str | Text value |
		| 2 | fixed | int | Indicates whether the text value or the dimension measurement is displayed. If 0, the dimension measurement is displayed. If 1, the text value is displayed |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text value is supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetText(dummy, dummy)

	def SetLevel(self, value:int) -> int:
		"""
		Sets the dimension item's display level value.

		:param value: Display level value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Previous display level value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetLevel(value)

	def GetLevel(self) -> int:
		"""
		Gets the dimension item's display level value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Display level value supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetLevel()

	def SetColour(self, value:int) -> int:
		"""
		Sets the dimension item's color.

		:param value: Color value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Previous assigned color value, See Colors for possible values |
		| 0 | Previous assigned color was black (RGB: 0, 0, 0 ) or an error occurred |
		| -1 | Previous assigned color was automatic |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetColour(value)

	def GetColour(self) -> int:
		"""
		Gets the dimension item's color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or the operation has failed |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetColour()

	def SetTextFontName(self, text:str) -> str:
		"""
		Sets the dimension item's text font name.

		:param text: Text font name to apply
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Font&gt;" | Previous assigned name of the dimension item's text font |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetTextFontName(text)

	def GetTextFontName(self) -> str:
		"""
		Gets the dimension item's text font name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Name of the dimension item's text font |
		| "Arial" | Name of the dimension item's text font is Arial or the font name could not be supplied |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTextFontName()

	def SetTextStyle(self, value:int) -> int:
		"""
		Sets the dimension item's text font style.

		:param value: Text style to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..31 | Previous assigned text style value |
		| 0 | Previously regular font style value was assigned or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetTextStyle(value)

	def GetTextStyle(self) -> int:
		"""
		Gets the dimension item's text font style.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..31 | Font style value supplied |
		| 0 | Regular font style value used or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTextStyle()

	def SetTextHeight(self, value:float) -> float:
		"""
		Sets the dimension item's text font height.

		:param value: Text height to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Previous assigned text height |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetTextHeight(value)

	def GetTextHeight(self) -> float:
		"""
		Gets the dimension item's text font height.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Size of the text font height |
		| 0.0 | Font height could not be supplied |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTextHeight()

	def SetTextColour(self, value:int) -> int:
		"""
		Sets the dimension item's text color.

		:param value: Color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Previous assigned color value, See Colors for possible values |
		| 0 | Previous assigned color was black (RGB: 0, 0, 0 ) or an error occurred |
		| -1 | Previous assigned color was automatic |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetTextColour(value)

	def GetTextColour(self) -> int:
		"""
		Gets the dimension item's text color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or the operation has failed |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTextColour()

	def GetDimensionedIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the items bound to the dimension item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | anyids | tuple[int,...] | Array of item identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in anyids |
		| 0 | No items bound to the dimension or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, anyids = self._obj.GetDimensionedIds(dummy)
		anyids = anyids[1:] if type(anyids) == tuple and len(anyids) > 0 else tuple()
		return ret, anyids

	def IsAlongPath(self) -> int:
		"""
		Indicates if the dimension item runs along a path.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Dimension item is along a path |
		| 0 | Dimension item is not along a path or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.IsAlongPath()

	def SetHideLongerPart(self, value:bool) -> int:
		"""
		Sets whether the dimension item's start point arrow, start point extension line and dimension line are hidden.

		:param value: Indicates whether the dimension item's start point arrow, start point extension line and dimension line are hidden. If 1, they are hidden. If 0, they are displayed
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previous start point arrow, start point extension line and dimension line were hidden |
		| 0 | Previous start point arrow, start point extension line and dimension line were displayed or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetHideLongerPart(value)

	def GetHideLongerPart(self) -> int:
		"""
		Indicates whether the dimension item's start point arrow, start point extension line and dimension line are hidden.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Start point arrow, start point extension line and dimension line are hidden |
		| 0 | Start point arrow, start point extension line and dimension line are displayed or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetHideLongerPart()

	def SetSuffixSizeFactor(self, value:float) -> float:
		"""
		Sets the dimension item's text suffix size factor.

		:param value: Text suffix size factor to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Previous assigned dimension text suffix size factor |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetSuffixSizeFactor(value)

	def GetSuffixSizeFactor(self) -> float:
		"""
		Gets the dimension item's text suffix size factor.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Dimension text suffix size factor |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetSuffixSizeFactor()

	def SetDisplayAttribute(self, attnam:str) -> int:
		"""
		Sets the user-defined name of the dimension item's display attribute.

		:param attnam: Attribute name to display
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Attribute name assigned |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetDisplayAttribute(attnam)

	def GetDisplayAttribute(self) -> str:
		"""
		Gets the user-defined name of the dimension item's display attribute.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Display attribute's user-defined name |
		| "&lt;Empty&gt;" | No attribute is assigned or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetDisplayAttribute()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the dimension item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the dimension item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.DeleteAttribute(name)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the dimension item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the dimension item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the dimension item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetAttributeValue(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the dimension item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.HasAttribute(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the dimension item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def SetExtensionLineOffset(self, value:float) -> float:
		"""
		Sets the distance between the dimension item's extension lines and the start and end points.

		:param value: Distance between the dimension item's extension lines and the start and end points to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Previous distance between the dimension item's extension lines and the start and end points |
		| 0.0 | Previous distance between the dimension item's extension lines and the start and end points or an error occurred |

		Available since E3.series TLB-Versions: 11.80
		"""
		return self._obj.SetExtensionLineOffset(value)

	def GetExtensionLineOffset(self) -> float:
		"""
		Gets the distance between the dimension item's extension lines and the start and end points.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Distance between the extension lines and the points |
		| 0.0 | No distance between the extension lines and the points or an error occurred |

		Available since E3.series TLB-Versions: 11.80
		"""
		return self._obj.GetExtensionLineOffset()

	def CreateEx(self, dimtyp:int, sheet:int, x:list[float], y:list[float], flags:int=0) -> int:
		"""
		Adds a new running dimension item onto a formboard or panel sheet of the project.

		:param dimtyp: Type of dimension
		:type int: int
		:param sheet: Identifier of the sheet to place the dimension item on
		:type int: int
		:param x: Placement positions on the x-axis
		:type list[float]: list[float]
		:param y: Placement positions on the y-axis
		:type list[float]: list[float]
		:param flags: Not used, defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new dimension item |
		| 0 | Error occurred |
		| -1 | dimtyp is invalid |
		| -2 | sheet is invalid |
		| -3 | Invalid start point |
		| -4 | Invalid placement position |
		| -5 | Sheet is locked                        Available since v2020-21.42, v2021-22.30 and v2022-23.11 |
		| -6 | E³.series is running under a E³.view or E³.viewPlus license                       Available since v2020-21.42, v2021-22.30 and v2022-23.11 |

		Available since E3.series TLB-Versions: 15.82
		"""
		x = [0.] + x
		y = [0.] + y
		return self._obj.CreateEx(dimtyp, sheet, x, y, flags)

	def IsRunningDimension(self) -> int:
		"""
		Indicates if the dimension item is a running dimension.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Dimension item is a running dimension |
		| 0 | Dimension item is not a running dimension or an error occurred |

		Available since E3.series TLB-Versions: 15.82
		"""
		return self._obj.IsRunningDimension()

	def IsPartOfDimension(self) -> int:
		"""
		Indicates if the dimension item is part of a running dimension.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Dimension item is part of a running dimension |
		| 0 | Dimension item is not part of a running dimension or an error occurred |

		Available since E3.series TLB-Versions: 15.82
		"""
		return self._obj.IsPartOfDimension()

	def GetPartIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the part dimensions of the dimension item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of dimension item identifiers |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No dimension part items or an error occurred |

		Available since E3.series TLB-Versions: 15.82
		"""
		dummy=0
		ret, ids = self._obj.GetPartIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def SetRunningDimTextRotation(self, value:int) -> int:
		"""
		Sets the dimension item's text rotation status.

		:param value: Text rotation status value to apply. If 1, text is rotated. If 0, text is not rotated
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Text rotation status is assigned or an error occurred |
		| -1 | Invalid dimension type |

		Available since E3.series TLB-Versions: 15.82
		"""
		return self._obj.SetRunningDimTextRotation(value)

	def GetRunningDimTextRotation(self) -> int:
		"""
		Indicates whether the dimension item's text is rotated.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Dimension text is rotated |
		| 0 | Dimension text is not rotated |
		| -1 | Invalid dimension type |

		Available since E3.series TLB-Versions: 15.82
		"""
		return self._obj.GetRunningDimTextRotation()

	def IsRedlined(self) -> int:
		"""
		Gets the dimension item's redlined status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Redlined status is on |
		| 0 | Redlined status is off |

		Available since E3.series TLB-Versions: 17.04
		"""
		return self._obj.IsRedlined()

	def SetRedlined(self, onoff:bool) -> int:
		"""
		Sets the dimension item's redlined status.

		:param onoff: Redlined status value to apply. If 1, the redlined status is set to on. If 0, the redlined status in set to off., defaults to TRUE
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previous assigned redlined status was on |
		| 0 | Previous assigned redlined status was off or an error occurred |

		Available since E3.series TLB-Versions: 17.04
		"""
		return self._obj.SetRedlined(onoff)

	def GetCenterTextPosition(self) -> int:
		"""
		Gets the value indicating whether the dimension's text position is centered.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text position is centered |
		| 0 | Text position is not centered or an error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetCenterTextPosition()

	def SetCenterTextPosition(self, value:int) -> int:
		"""
		Sets the value indicating whether the dimension's text position is centered.

		:param value: Center text position value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Center text position value assigned |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.SetCenterTextPosition(value)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current dimension item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current  dimension item |
		| "&lt;Empty&gt;" | No dimension item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a dimension as the current item.

		:param gid: Global identifier value of a dimension item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current dimension item |
		| "&lt;Empty&gt;" | No current dimension item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IDllInterface--------------------
class Dll:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Dll. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def Load(self, name:str) -> int:
		"""


		:param name: 
		:type str: str

		:return:
		Type: int

		Available since E3.series TLB-Versions: 6.00
		"""
		return self._obj.Load(name)

	def Call(self, method:str, item1:str=DLLDEFAULTVALUE, item2:str=DLLDEFAULTVALUE, item3:str=DLLDEFAULTVALUE, item4:str=DLLDEFAULTVALUE, item5:str=DLLDEFAULTVALUE) -> int:
		"""


		:param method: 
		:type str: str
		:param item1: , defaults to DLLDEFAULTVALUE
		:type str: str
		:param item2: , defaults to DLLDEFAULTVALUE
		:type str: str
		:param item3: , defaults to DLLDEFAULTVALUE
		:type str: str
		:param item4: , defaults to DLLDEFAULTVALUE
		:type str: str
		:param item5: , defaults to DLLDEFAULTVALUE
		:type str: str

		:return:
		Type: int

		Available since E3.series TLB-Versions: 6.00
		"""
		return self._obj.Call(method, item1, item2, item3, item4, item5)

	def Unload(self) -> int:
		"""


		:return:
		Type: int

		Available since E3.series TLB-Versions: 6.00
		"""
		return self._obj.Unload()

# -------------------- IGroupInterface--------------------
class Group:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Group. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current group item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current group item identifier |
		| 0 | No group item  or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a group item as the current item.

		:param id: Unique value identifying a group item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current group item identifier |
		| 0 | No current group item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def Create(self, ids:list[int], name:str="") -> int:
		"""
		Creates a new group on a sheet of the project.

		:param ids: Identifiers of the items to form the contents of the group passed into the function
		:type list[int]: list[int]
		:param name: Name of the group. Default value is "<Empty>", defaults to ""
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new group item |
		| 0 | Error occurred |
		| -1 | An item in ids is offline |
		| -2 | Sheet is read-only |
		| -3 | An item in ids is already in a group |
		| -4 | Items in ids are located on different sheets |
		| -5 | Items in ids include sheet items and non-sheet items |
		| -6 | ids contains a mix of items on and off a grouped sheet |
		| -10 | name value exceeds maximum length, Available since v2018-19.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.Create(ids, name)
		return ret[0]

	def Delete(self) -> tuple[int, tuple[int,...]]:
		"""
		Deletes the group item from the project.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of Identifiers of the items that formed the contents of the group passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.Delete(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetItems(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the items contained in the group item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of contained items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of item identifiers in ids |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetItems(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def AddItems(self, ids:list[int]) -> int:
		"""
		Adds items to the contents of the group item.

		:param ids: Array of identifiers of items to add passed into the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Items are added to the group item's contents |
		| 1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.AddItems(ids)
		return ret[0]

	def RemoveItems(self, ids:list[int]) -> int:
		"""
		Removes items from the group item.

		:param ids: Array of identifiers of items to remove passed into the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Items are removed from the group item's content |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.RemoveItems(ids)
		return ret[0]

	def GetGroupId(self, id:int) -> int:
		"""
		Gets the identifier of a group item containing the specified item.

		:param id: Identifier of an item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Group item identifier |
		| 0 | Item is not part of a group's content or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetGroupId(id)

	def GetLocation(self) -> tuple[int, float, float, float, str, str, str]:
		"""
		Gets the location within the project of the placed group item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Placement position on the x-axis passed out of the function |
		| 2 | y | float | Placement position on the y-axis passed out of the function |
		| 3 | z | float | Not used and passed out of the function |
		| 4 | grid | str | Reference description of the sheet and grid location passed out of the function |
		| 5 | column_value | str | Grid column value passed out of the function |
		| 6 | row_value | str | Grid row value passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the sheet containing the group |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetLocation(dummy, dummy, dummy, dummy, dummy, dummy)

	def Place(self, shtid:int, x:float, y:float, z:float, rotation:float) -> int:
		"""
		Places or moves a group item on a sheet.

		:param shtid: Identifier of the sheet to place the group item on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param z: Not used
		:type float: float
		:param rotation: Rotation value in degrees to apply
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;7 | Identifier of the group item |
		| 1 | Invalid group item |
		| 2 | Group item could not be moved |
		| 3 | Sheet groups could not be moved |
		| 4 | Sheet is hidden |
		| 5 | Unplaced group item could not be placed |
		| 6 | Required group could not be checked out by MU server |
		| 7 | Sheet group could not be placed |
		| -990120 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Place(shtid, x, y, z, rotation)

	def DeleteContents(self) -> int:
		"""
		Deletes the group item and its contents from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Group item and its contents were deleted or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteContents()

	def GetName(self) -> str:
		"""
		Gets the name of the group item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Group item name value |
		| "&lt;Empty&gt;" | Group item has no name or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the name of the group item.

		:param name: Name of the group item
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | name was assigned to the group item |
		| -1 | Group item was offline |
		| -2 | Sheet is read-only |
		| -3 | Group item is unplaced |
		| -4 | Group subcircuit does not exist |
		| -5 | No group item set |
		| -10 | name value exceeds maximum lengthAvailable since v2018-19.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the group item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeIds(self, name:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the group item.

		:param name: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is "<Empty>", defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, name)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the group item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the group item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the group item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def GetPartName(self) -> str:
		"""
		Gets the file name of the group item's subcircuit.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Group item's subcircuit file name |
		| "&lt;Empty&gt;" | Group item has no subcircuit or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.GetPartName()

	def SetPartName(self, name:str) -> int:
		"""
		Sets the group item's subcircuit.

		:param name: File name of the group item's subcircuit
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | name was assigned as the group item's subcircuit |
		| -1 | Group item was offline |
		| -2 | Sheet is read-only |
		| -3 | Group item is unplaced |
		| -4 | Group subcircuit does not exist |
		| -5 | No group item set |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.SetPartName(name)

	def IsPart(self) -> int:
		"""
		Indicates if the group item defines a subcircuit file.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Group item defines a subcircuit file |
		| 0 | Group item does not define a subcircuit file or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.IsPart()

	def IsSubCircuit(self) -> int:
		"""
		Indicates if the group item defines a database subcircuit.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Group item defines a database subcircuit |
		| 0 | Group item does not define a database subcircuit or an error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.IsSubCircuit()

	def UpdatePart(self) -> int:
		"""
		Updates the group item with the assigned subcircuit.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 2 | Group item is updated however duplicate part identifier values exist, Available since v2018-19.46, v2019-20.28 and v2020-21.12 |
		| 1 | Group item is updated |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.00
		"""
		return self._obj.UpdatePart()

	def Unplace(self) -> int:
		"""
		Removes the group item from its sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Group was unplaced |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.70
		"""
		return self._obj.Unplace()

	def DeleteUnplaced(self) -> int:
		"""
		Deletes the unplaced group item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Group item deleted |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.70
		"""
		return self._obj.DeleteUnplaced()

	def IsUnplaced(self) -> int:
		"""
		Indicates if the group item is unplaced.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Group item is unplaced |
		| 0 | Group item is placed or an error occurred |

		Available since E3.series TLB-Versions: 11.70
		"""
		return self._obj.IsUnplaced()

	def PlaceInteractively(self) -> int:
		"""
		Attaches the group item to the mouse cursor so it can be placed manually.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Group item is attached to the mouse cursor for placement |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.70
		"""
		return self._obj.PlaceInteractively()

	def UpdateDrawingForProjectGeneration(self, flags:int, substitutes:list[tuple[str,str]], allowedTexttypes:list[str]=pythoncom.Empty, allowedAttributenames:list[str]=pythoncom.Empty, resultArray:tuple[typing.Any,...]=0) -> int:
		"""
		Replaces the group item's contents name, text and attribute values.

		:param flags: Replacement criteria
		:type int: int
		:param substitutes: Array of placeholder strings and their replacement strings
		:type list[tuple[str,str]]: list[tuple[str,str]]
		:param allowedTexttypes: Array of. Text Types. for which the replacement is permitted.  (No String Array passed into the function), defaults to pythoncom.Empty
		:type list[str]: list[str]
		:param allowedAttributenames: Array of. Attributes. for which the replacement is permitted.  (No String Array passed into the function), defaults to pythoncom.Empty
		:type list[str]: list[str]
		:param resultArray: Array of errors resulting from the update, defaults to 0
		:type tuple[typing.Any,...]: tuple[typing.Any,...]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 6 | Drawing updated however it contains duplicate part identifier values, Available since v2019-20.28 and v2020-21.12 |
		| 5 | Drawing updated however the XML project structure is missing or differs and not everything could be replaced, resultArray holds the unreplaced items |
		| 4 | Drawing updated however the XML project structure is missing or differs |
		| 2 | Drawing updated but not everything could be replaced, resultArray holds the unreplaced items |
		| 1 | Drawing updated |
		| -1 | No project open, no file found or incorrect version of subcircuit |
		| -2 | Invalid text types defined in allowTexttypesresultArray holds the invalid text types |
		| -3 | Invalid attribute names defined in allowedAttributenamesresultArray holds the invalid attribute names |
		| -4 | substitutes array is invalid |
		| -5 | substitutes contains invalid strings |
		| -6 | Error occurred |
		| -7 | Group cannot be updated |
		| -8 | Group is already defined as a subcircuit in the database |

		Available since E3.series TLB-Versions: 20.00
		"""
		substitutes = [("","")] + substitutes
		return self._obj.UpdateDrawingForProjectGeneration(flags, substitutes, allowedTexttypes, allowedAttributenames, resultArray)

	def GetGroupType(self) -> int:
		"""
		Gets a value representing the type of the group  item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type value of the group item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 22.41, 23.50
		"""
		return self._obj.GetGroupType()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current group item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current group item |
		| "&lt;Empty&gt;" | No group item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a group item as the current item.

		:param gid: Global identifier value of a group item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current group item |
		| "&lt;Empty&gt;" | No current group item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetAnyIds(self, flags:int) -> tuple[int, dict[int,tuple[int,...]]]:
		"""
		Gets identifiers of the items contained in the group item.

		:param flags: Item type filter options
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | anyIds | dict[int,tuple[int,...]] | Dictionary of [. Integer. ]. key. /[. Integer Array. ]. value. pairs representing items in the group. key is the item type. value is the array of item identifiers |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of item types in anyIds |
		| 0 | No items found in the group |
		| -1 | No project open |
		| -2 | No group item set |

		Available since E3.series TLB-Versions: 23.00
		"""
		dummy=0
		ret, anyIds = self._obj.GetAnyIds(flags, dummy)
		anyIds = _variant_to_dict(anyIds)
		for i0 in anyIds.keys():
			anyIds[i0] = anyIds[i0][1:] if type(anyIds[i0]) == tuple and len(anyIds[i0]) > 0 else tuple()
		return ret, anyIds

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current group item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current group item |
		| "&lt;Empty&gt;" | No group item or an error occurred |

		Available since E3.series TLB-Versions: 26.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a group item as the current item.

		:param guid: Globally unique identifier value of a group item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current group item |
		| "&lt;Empty&gt;" | No current group item or an error occurred |

		Available since E3.series TLB-Versions: 26.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- IFunctionalUnitInterface--------------------
class FunctionalUnit:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize FunctionalUnit. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current functional unit item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current functional unit item identifier |
		| 0 | No functional unit item  or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a functional unit as the current item.

		:param id: Unique value identifying a functional unit item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current functional unit item identifier |
		| 0 | No current functional unit item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def Create(self, fdbID:int, name:str, symnam:str, symver:str) -> int:
		"""
		Creates a new functional unit.

		:param fdbID: Identifier of a functional block item
		:type int: int
		:param name: Name of the functional unit
		:type str: str
		:param symnam: Functional unit symbol name
		:type str: str
		:param symver: Functional unit symbol version
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new functional unit item |
		| 0 | No project open |
		| -1 | No E³.topology or E³.functionaldesign license available |
		| -2 | fdbId is not a valid block item identifier |
		| -3 | fdbId is not a valid functional block item identifier |
		| -4 | Dynamic functional unit is already available for the fdbId functional block item |
		| -5 | Symbol does not have a functional unit |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(fdbID, name, symnam, symver)

	def Delete(self) -> int:
		"""
		Deletes the functional unit item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Functional unit deleted or an error occurred |
		| -1 | No E³.topology or E³.functionaldesign license available |
		| -2 | Functional unit is not dynamic |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def GetFunctionalPortIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the functional unit item's functional ports.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of functional port identifiers  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of functional port identifiers in ids |
		| 0 | No functional ports or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetFunctionalPortIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsDynamic(self) -> int:
		"""
		Indicates if the functional unit item is dynamic.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Functional unit is dynamic |
		| 0 | Functional unit is not dynamic or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsDynamic()

	def GetName(self) -> str:
		"""
		Gets the name of the functional unit item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Functional unit item name value |
		| "&lt;Empty&gt;" | Functional unit item has no name or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the functional unit item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the functional unit item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the functional unit item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the functional unit item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the functional unit item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the functional unit item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.HasAttribute(name)

	def SetName(self, name:str) -> int:
		"""
		Sets the functional unit item's name.

		:param name: New functional unit name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New functional unit name applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetName(name)

	def GetDeviceId(self) -> int:
		"""
		Gets the identifier of the device the functional unit is assigned to.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Device identifier |
		| 0 | No device assignment or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetDeviceId()

	def GetSchemaSymbolId(self) -> int:
		"""
		Gets the identifier of the symbol the functional unit is assigned to if no device is assigned.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Symbol identifier |
		| 0 | No schema symbol assigned or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetSchemaSymbolId()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current functional unit item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current functional unit item |
		| "&lt;Empty&gt;" | No functional unit item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a functional unit as the current item.

		:param gid: Global identifier value of a functional unit item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current functional unit item |
		| "&lt;Empty&gt;" | No current functional unit item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IFunctionalPortInterface--------------------
class FunctionalPort:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize FunctionalPort. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current functional port item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current functional port item identifier |
		| 0 | No functional port item  or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a functional port as the current item.

		:param id: Unique value identifying a functional port item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current functional port item identifier |
		| 0 | No current functional port item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def Create(self, fuId:int, name:str) -> int:
		"""
		Creates a new functional port and adds it to a functional unit.

		:param fuId: Identifier of a functional unit item
		:type int: int
		:param name: Name of the functional port
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new functional unit item |
		| 0 | No project open |
		| -1 | No E³.topology or E³.functionaldesign license available |
		| -2 | fuId is not a valid functional unit item identifier |
		| -3 | fdbId is not a dynamic functional unit item identifier |
		| -4 | Port could not be added to the functional unit |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(fuId, name)

	def Delete(self) -> int:
		"""
		Deletes the functional port item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Functional port deleted or an error occurred |
		| -1 | No E³.topology or E³.functionaldesign license available |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def SetSignalName(self, name:str) -> int:
		"""
		Sets the functional port item's signal name value.

		:param name: New functional port signal name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Functional port item's signal name value is supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetSignalName(name)

	def GetSignalName(self) -> str:
		"""
		Gets the functional port item's signal name value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Functional port item's signal name value is supplied |
		| "&lt;Empty&gt;" | No signal name value or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSignalName()

	def SetConnectorName(self, name:str) -> str:
		"""
		Sets the connector name of the functional port item.

		:param name: New functional port item connector name value
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Previous functional port item connector name value |
		| "&lt;Empty&gt;" | No previous functional port item connector or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetConnectorName(name)

	def GetConnectorName(self) -> str:
		"""
		Gets the connector name of the functional port item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Functional port item connector name value |
		| "&lt;Empty&gt;" | Function port has no connector or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetConnectorName()

	def SetPinName(self, name:str) -> str:
		"""
		Sets the pin name of the functional port item.

		:param name: New functional port item pin name value
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Previous functional port item pin name value |
		| "&lt;Empty&gt;" | No previous functional port item pin name value or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetPinName(name)

	def GetPinName(self) -> str:
		"""
		Gets the pin name of the functional port item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Functional port item pin name value |
		| "&lt;Empty&gt;" | No pin name or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPinName()

	def SetConnectorPinID(self, id:int) -> int:
		"""
		Sets the connector pin of the functional port's block pin symbol.

		:param id: Connector pin identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Connector pin value applied |
		| -1 | No E³.topology or E³.functionaldesign license available |
		| -2 | id is invalid |
		| -3 | Another functional port is already assigned to the functional block |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetConnectorPinID(id)

	def GetConnectorPinID(self) -> int:
		"""
		Gets the identifier of the connector pin of the functional port's block pin symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Connector pin identifier |
		| 0 | No connector pin item found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetConnectorPinID()

	def GetName(self) -> str:
		"""
		Gets the name of the functional port item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Functional port item name value |
		| "&lt;Empty&gt;" | Functional port item has no name or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def GetNodeID(self) -> int:
		"""
		Gets the identifier of the node of the functional port's block pin symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Node identifier |
		| 0 | No node item found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetNodeID()

	def SetNodeID(self, id:int) -> int:
		"""
		Assigns the functional port to a block pin symbol.

		:param id: Node identifier of a block pin symbol
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Functional port assigned to the block pin symbol or an error occurred |
		| -1 | No E³.topology or E³.functionaldesign license available |
		| -2 | id is invalid |
		| -3 | Block pin symbol node does not belong to the functional block assigned to the functional port |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetNodeID(id)

	def GetFunctionalUnitId(self) -> int:
		"""
		Gets the identifier of the functional unit the current functional port item belongs to.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Functional unit identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetFunctionalUnitId()

	def SetPinID(self, id:int) -> int:
		"""
		Sets the connector pin of the functional port's block pin symbol.

		:param id: Connector pin identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Connector pin value applied or an error occurred |
		| -1 | No E³.topology or E³.functionaldesign license available |
		| -2 | id is invalid |
		| -3 | Another functional port is already assigned to the functional block |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetPinID(id)

	def GetPinID(self) -> int:
		"""
		Gets the identifier of the connector pin of the functional port's block pin symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Connector pin identifier |
		| 0 | No connector pin item found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetPinID()

	def SetName(self, name:str) -> int:
		"""
		Sets the functional port item's name.

		:param name: New functional port name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | New functional port name applied |
		| -1 | No project open or no functional port item set |
		| -2 | No E³.topology or E³.functionaldesign license available |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetName(name)

	def SetPortType(self, _type:int) -> int:
		"""
		Sets the port type of the current functional port item.

		:param _type: Port type value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Port type value applied |
		| -1 | No project open or no functional port item set |
		| -2 | No E³.topology or E³.functionaldesign license available |
		| -3 | type is invalid |
		| -4 | Port belongs to the pin of a busbar, Available since v2022-23.00 |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetPortType(_type)

	def GetPortType(self) -> int:
		"""
		Gets the port type of the current functional port item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Port type value supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetPortType()

	def SetSignalEquiv(self, signalequiv:int) -> int:
		"""
		Sets the signal equivalence of the functional port.

		:param signalequiv: New signal equivalence value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Signal equivalence value applied |
		| -1 | No project open or no functional port item set |
		| -2 | No E³.topology or E³.functionaldesign license available |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetSignalEquiv(signalequiv)

	def GetSignalEquiv(self) -> int:
		"""
		Gets the signal equivalence of the functional port.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Signal equivalence value |
		| 0 | No signal equivalence or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetSignalEquiv()

	def SetUserDefined(self, _type:int=1) -> int:
		"""
		Sets the user defined status of the current functional port item.

		:param _type: User defined status value to apply. If 1, the port is set to user defined. If 0, the port is set to not user defined., defaults to 1
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | User defined status value applied |
		| -1 | No project open or no functional port item set |
		| -2 | No E³.functionaldesign license available |
		| -3 | Port belongs to the pin of a busbar, Available since v2022-23.00 |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetUserDefined(_type)

	def GetUserDefined(self) -> int:
		"""
		Indicates if the functional port item is user defined.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Functional port item is user defined |
		| 0 | Functional port item is not user defined or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetUserDefined()

	def GetConnectedPorts(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the functional ports connected to the current functional port item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of functional port identifiers  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of functional port identifiers in ids |
		| 0 | No connected functional ports or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		ret, ids = self._obj.GetConnectedPorts(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSignalEqvPorts(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the functional ports with the signal equivalence of the current functional port item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of functional port identifiers  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of functional port identifiers in ids |
		| 0 | No signal equivalence functional ports or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		ret, ids = self._obj.GetSignalEqvPorts(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetTranslatedSignalName(self) -> str:
		"""
		Gets the functional port item's translated signal name value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Functional port item's translated signal name value is supplied |
		| "&lt;Empty&gt;" | No signal name value or an error occurred |

		Available since E3.series TLB-Versions: 11.00
		"""
		return self._obj.GetTranslatedSignalName()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current functional port item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current  functional port item |
		| "&lt;Empty&gt;" | No functional port item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a functional port as the current item.

		:param gid: Global identifier value of a functional port item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current functional port item |
		| "&lt;Empty&gt;" | No current functional port item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IConnectLineInterface--------------------
class ConnectLine:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize ConnectLine. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current connect line item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current connect line item identifier |
		| 0 | No connect line item or an error occurred |

		Available since E3.series TLB-Versions: 9.22
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a connect line as the current item.

		:param id: Unique value identifying a connect line item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current connect line item identifier |
		| 0 | No current connect line item or an error occurred |

		Available since E3.series TLB-Versions: 9.22
		"""
		return self._obj.SetId(id)

	def GetCoordinates(self) -> tuple[int, tuple[float,...], tuple[float,...], tuple[float,...], tuple[int,...]]:
		"""
		Gets the coordinates of the connect lines.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xarr | tuple[float,...] | Array of connect line positions on the x-axis  passed out of the function |
		| 2 | yarr | tuple[float,...] | Array of connect line positions on the y-axis  passed out of the function |
		| 3 | zarr | tuple[float,...] | Not used. Array of 0.0 values passed out of the function |
		| 4 | PointTypArr | tuple[int,...] | Array of values indicating if the position represents a spline passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of values within the arrays |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.22
		"""
		dummy=0
		ret, xarr, yarr, zarr, PointTypArr = self._obj.GetCoordinates(dummy, dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		zarr = zarr[1:] if type(zarr) == tuple and len(zarr) > 0 else tuple()
		PointTypArr = PointTypArr[1:] if type(PointTypArr) == tuple and len(PointTypArr) > 0 else tuple()
		return ret, xarr, yarr, zarr, PointTypArr

	def GetProtectionSymbolIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the cable protection symbols assigned to the connection item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of cable protection symbol identifiers  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No cable protection symbol identifiers or an error occurred |

		Available since E3.series TLB-Versions: 9.22
		"""
		dummy=0
		ret, ids = self._obj.GetProtectionSymbolIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def AddProtectionSymbolId(self, id:int) -> int:
		"""
		Adds a cable protection symbol to the connect line item.

		:param id: Device or cable protection symbol identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Cable protection symbol is placed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 9.22
		"""
		return self._obj.AddProtectionSymbolId(id)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current connect line item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current connect line item |
		| "&lt;Empty&gt;" | No connect line item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a connect line as the current item.

		:param gid: Global identifier value of a connect line item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current connect line item |
		| "&lt;Empty&gt;" | No current component item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- ISignalClassInterface--------------------
class SignalClass:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize SignalClass. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current signal class item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current signal class item identifier |
		| 0 | No signal class item |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a signal class item as the current item.

		:param id: Unique value identifying a signal class item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current signal class item identifier |
		| 0 | No signal class item |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetId(id)

	def Create(self, name:str) -> int:
		"""
		Creates a signal class item.

		:param name: New signal class name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the signal class item |
		| 0 | Error occurred |
		| -1 | name is invalid |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Create(name)

	def Delete(self) -> int:
		"""
		Deletes the current signal class from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Error occurred, Available since v2016-17.42 and v2017-18.31 |
		| 0 | Signal class item was deleted |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Delete()

	def AddSignalId(self, id:int) -> int:
		"""
		Adds a signal to the current signal class item.

		:param id: Signal identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Signal added to signal class |
		| -1..-3 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.AddSignalId(id)

	def RemoveSignalId(self, id:int) -> int:
		"""
		Removes a signal from the current signal class item.

		:param id: Signal identifier
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Error occurred, Available since v2016-17.42 and v2017-18.31 |
		| 0 | Signal removed from signal class |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.RemoveSignalId(id)

	def GetSignalIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the signals within the signal class item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of signals passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No signal identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		ret, ids = self._obj.GetSignalIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetName(self) -> str:
		"""
		Gets the signal class item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Signal class name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the signal class item's name.

		:param name: New name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signal class name applied |
		| 0 | Signal class name applied or an error occurred |
		| -1 | name is invalid |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetName(name)

	def Search(self, name:str) -> int:
		"""
		Searches for a signal class item.

		:param name: Name of the signal class item to search for
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the found signal class item |
		| 0 | No signal class found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Search(name)

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the signal class item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the signal class item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the signal class item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the signal class item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the signal class item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the signal class item.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the signal class item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.HasAttribute(name)

	def DisplayaAttributeValueAt(self, name:str, sheetid:int, x:float, y:float) -> int:
		"""
		Displays an attribute value from the signal class item at a specific position.

		:param name: Name of the attribute
		:type str: str
		:param sheetid: Sheet identifier to display the attribute value on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text item identifier displaying the attribute value |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.DisplayaAttributeValueAt(name, sheetid, x, y)

	def SetAttributeVisibility(self, name:str, onoff:int) -> int:
		"""
		Sets the visibility status of all text items representing the signal class item's specified attribute value.

		:param name: Name of the attribute
		:type str: str
		:param onoff: Indicates whether the text items' visibility status should be shown or hidden. If > 0, the text items are shown. If 0, the text items are hidden
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of text items changed |
		| 0 | No text items found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetAttributeVisibility(name, onoff)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the signal class item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the signal class item |
		| "&lt;Empty&gt;" | No signal class item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets a signal class item as the current item.

		:param gid: Global identifier value of a signal class item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current signal class item |
		| "&lt;Empty&gt;" | No signal class item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current signal class item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current signal class item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> str:
		"""
		Sets a signal class item as the current item.

		:param guid: Globally unique identifier value of a signal class item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current signal class item |
		| "&lt;Empty&gt;" | No signal class item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGUID(guid)

# -------------------- IAttributeDefinitionInterface--------------------
class AttributeDefinition:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize AttributeDefinition. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current attribute definition item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current attribute definition item identifier |
		| 0 | No attribute definition item or an error occurred |

		Available since E3.series TLB-Versions: 15.31
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets an attribute definition as the current item.

		:param id: Unique value identifying an attribute definition item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current attribute definition item identifier |
		| 0 | No current attribute item or an error occurred |

		Available since E3.series TLB-Versions: 15.31
		"""
		return self._obj.SetId(id)

	def Search(self, name:str) -> int:
		"""
		Gets the identifier of the attribute definition.

		:param name: Attribute definition name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the found attribute definition |
		| 0 | No attribute definition found or an error occurred |

		Available since E3.series TLB-Versions: 15.31
		"""
		return self._obj.Search(name)

	def Get(self) -> tuple[int, tuple[tuple[str,str],...]]:
		"""
		Gets the properties of the attribute definition.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | attributeDefinition | tuple[tuple[str,str],...] | Array of attribute properties |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of elements in attributeDefinition |
		| -1 | No project open |
		| -2 | No attribute definition item set |

		Available since E3.series TLB-Versions: 15.31
		"""
		dummy=0
		ret, attributeDefinition = self._obj.Get(dummy)
		attributeDefinition = attributeDefinition[1:] if type(attributeDefinition) == tuple and len(attributeDefinition) > 0 else tuple()
		attributeDefinition = tuple( i0[1:] if type(i0) == tuple and len(i0) > 0 else tuple() for i0 in attributeDefinition)
		return ret, attributeDefinition

	def Set(self, attributeDefinition:list[tuple[str,str]]) -> int:
		"""
		Sets the properties of the attribute definition.

		:param attributeDefinition: Array of attribute properties **Available enum types:** e3series.types.AD_Direction, AD_Owner, AD_Ratio, AD_Type, AD_UniqueValue.
		:type list[tuple[str,str]]: list[tuple[str,str]]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Attribute properties assigned |
		| -1 | No project open |
		| -2 | No attribute definition item set |
		| -3 | Internal error |
		| -101 | Invalid value for "Owner" property |
		| -102 | Invalid value for "Type" property |
		| -103 | Invalid value for "Single instance" property |
		| -104 | Invalid value for "Unique value" property |
		| -105 | Invalid value for "Format" property |
		| -106 | Invalid value for "Size" property |
		| -107 | Invalid value for "Pos x" property |
		| -108 | Invalid value for "Pos y" property |
		| -109 | Invalid value for "Colour" property |
		| -110 | Invalid value for "Ratio" property |
		| -111 | Invalid value for "Direction" property |
		| -112 | Invalid value for "Level" property |
		| -113 | Invalid value for "Visible" property |
		| -114 | Invalid value for "Must exist" property |
		| -115 | Invalid value for "Changeable by script only" property |
		| -116 | Invalid value for "Default value" property |
		| -117 | Invalid value for "List of values" property |
		| -118 | Invalid value for "Changeable when owner is locked" property |
		| -119 | Invalid value for "Allow change of lock behavior" property |
		| -202 | "Type" property exists multiple times |
		| -203 | "Single instance" property exists multiple times |
		| -204 | "Unique value" property exists multiple times |
		| -205 | "Format" property exists multiple times |
		| -206 | "Size" property exists multiple times |
		| -207 | "Pos x" property exists multiple times |
		| -208 | "Pos y" property exists multiple times |
		| -209 | "Colour" property exists multiple times |
		| -210 | "Ratio" property exists multiple times |
		| -211 | "Direction" property exists multiple times |
		| -212 | "Level" property exists multiple times |
		| -213 | "Visible" property exists multiple times |
		| -214 | "Must exist" property exists multiple times |
		| -215 | "Changeable by script only" property exists multiple times |
		| -216 | "Default value" property exists multiple times |
		| -217 | "List of values" property exists multiple times |
		| -218 | "Changeable when owner is locked" property exists multiple times |
		| -219 | "Allow change of lock behavior" property exists multiple times |

		Available since E3.series TLB-Versions: 15.31
		"""
		attributeDefinition = [("","")] + attributeDefinition
		attributeDefinition = [tuple((None,) + i0) for i0 in attributeDefinition]
		return self._obj.Set(attributeDefinition)

	def GetFromDatabase(self) -> tuple[int, dict[str,tuple[tuple[str,str],...]]]:
		"""
		Gets the attribute definitions with all properties from the defined database.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | attributeDefinitions | dict[str,tuple[tuple[str,str],...]] | Dictionary of attribute definitions |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of elements in attributeDefinitions |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 15.31
		"""
		dummy=0
		ret, attributeDefinitions = self._obj.GetFromDatabase(dummy)
		attributeDefinitions = _variant_to_dict(attributeDefinitions)
		for i0 in attributeDefinitions.keys():
			attributeDefinitions[i0] = attributeDefinitions[i0][1:] if type(attributeDefinitions[i0]) == tuple and len(attributeDefinitions[i0]) > 0 else tuple()
			attributeDefinitions[i0] = tuple( i1[1:] if type(i1) == tuple and len(i1) > 0 else tuple() for i1 in attributeDefinitions[i0])
		return ret, attributeDefinitions

	def GetInternalName(self) -> str:
		"""
		Gets the attribute definition's internal name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Attribute definition's internal name |
		| "&lt;Empty&gt;" | Unable to supply the internal name |

		Available since E3.series TLB-Versions: 15.31
		"""
		return self._obj.GetInternalName()

	def GetName(self, languageId:int=0) -> str:
		"""
		Gets the attribute definition's language dependent name.

		:param languageId: Identifier value of the language, defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Attribute definition's translated name |
		| "&lt;Empty&gt;" | Unable to supply the name |

		Available since E3.series TLB-Versions: 15.31
		"""
		return self._obj.GetName(languageId)

	def SetName(self, languageId:int, newName:str) -> int:
		"""
		Sets the attribute definition's language dependent name.

		:param languageId: Id value of the language
		:type int: int
		:param newName: Attribute definition's new name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Name successfully changed |
		| -1 | Internal error |
		| -2 | languageId value is invalid |
		| -3 | newName value exceeds the maximum length |
		| -4 | newName value already exists within the language of another attribute definition |

		Available since E3.series TLB-Versions: 15.31
		"""
		return self._obj.SetName(languageId, newName)

	def Delete(self) -> int:
		"""
		Deletes the attribute definition from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Attribute definition deleted |
		| -1 | No project open |
		| -2 | Attribute definition not set |
		| -3 | Attribute definition in use |
		| -4 | Internal error |

		Available since E3.series TLB-Versions: 15.31
		"""
		return self._obj.Delete()

	def Create(self, name:str, attributeDefinition:list[tuple[str,str]]) -> int:
		"""
		Creates and adds a new attribute definition to the project.

		:param name: Attribute name
		:type str: str
		:param attributeDefinition: Array of attribute properties **Available enum types:** e3series.types.AD_Direction, AD_Owner, AD_Ratio, AD_Type, AD_UniqueValue.
		:type list[tuple[str,str]]: list[tuple[str,str]]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute definition |
		| -1 | No project open |
		| -2 | No attribute definition item set |
		| -3 | Internal error |
		| -4 | name contains invalid characters |
		| -5 | name exceeds maximum length |
		| -101 | Invalid value for "Owner" property |
		| -102 | Invalid value for "Type" property |
		| -103 | Invalid value for "Single instance" property |
		| -104 | Invalid value for "Unique value" property |
		| -105 | Invalid value for "Format" property |
		| -106 | Invalid value for "Size" property |
		| -107 | Invalid value for "Pos x" property |
		| -108 | Invalid value for "Pos y" property |
		| -109 | Invalid value for "Colour" property |
		| -110 | Invalid value for "Ratio" property |
		| -111 | Invalid value for "Direction" property |
		| -112 | Invalid value for "Level" property |
		| -113 | Invalid value for "Visible" property |
		| -114 | Invalid value for "Must exist" property |
		| -115 | Invalid value for "Changeable by script only" property |
		| -116 | Invalid value for "Default value" property |
		| -117 | Invalid value for "List of values" property |
		| -118 | Invalid value for "Changeable when owner is locked" property |
		| -119 | Invalid value for "Allow change of lock behavior" property |
		| -202 | "Type" property exists multiple times |
		| -203 | "Single instance" property exists multiple times |
		| -204 | "Unique value" property exists multiple times |
		| -205 | "Format" property exists multiple times |
		| -206 | "Size" property exists multiple times |
		| -207 | "Pos x" property exists multiple times |
		| -208 | "Pos y" property exists multiple times |
		| -209 | "Colour" property exists multiple times |
		| -210 | "Ratio" property exists multiple times |
		| -211 | "Direction" property exists multiple times |
		| -212 | "Level" property exists multiple times |
		| -213 | "Visible" property exists multiple times |
		| -214 | "Must exist" property exists multiple times |
		| -215 | "Changeable by script only" property exists multiple times |
		| -216 | "Default value" property exists multiple times |
		| -217 | "List of values" property exists multiple times |
		| -218 | "Changeable when owner is locked" property exists multiple times |
		| -219 | "Allow change of lock behavior" property exists multiple times |

		Available since E3.series TLB-Versions: 15.31
		"""
		attributeDefinition = [("","")] + attributeDefinition
		attributeDefinition = [tuple((None,) + i0) for i0 in attributeDefinition]
		return self._obj.Create(name, attributeDefinition)

	def Update(self, flags:int) -> int:
		"""
		Updates the attribute definition in the project from the database.

		:param flags: Not used
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Identifier of the found attribute definition |
		| -1 | No project open |
		| -2 | Attribute definition not set |
		| -3 | Internal error occurred |

		Available since E3.series TLB-Versions: 15.31
		"""
		return self._obj.Update(flags)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current attribute definition item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current attribute definition item |
		| "&lt;Empty&gt;" | No attribute definition item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets an attribute definition as the current item.

		:param gid: Global identifier value of an attribute definition item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current attribute definition item |
		| "&lt;Empty&gt;" | No current attribute definition item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

	def GetValueListName(self, flags:int=0) -> str:
		"""
		Gets the attribute definition's value list name of possible values.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Attribute definition's value list name of possible values |
		| "&lt;Empty&gt;" | No value list is associated with the attribute definition or an error occurred |

		Available since E3.series TLB-Versions: 26.01, 25.34
		"""
		return self._obj.GetValueListName(flags)

	def GetAttributeListValues(self, attributelistname:str, flags:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets the values from an attribute definition value list.

		:param attributelistname: Name of the attribute definition value list
		:type str: str
		:param flags: Translation option flags., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | values | tuple[str,...] | Array of values contained in the attribute definition value list |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of values in values |
		| 0 | No value list is associated with the attributelistname value |
		| -1 | No project open |
		| -2 | flags is invalid |

		Available since E3.series TLB-Versions: 26.01, 25.34
		"""
		dummy=0
		ret, values = self._obj.GetAttributeListValues(attributelistname, dummy, flags)
		values = values[1:] if type(values) == tuple and len(values) > 0 else tuple()
		return ret, values

# -------------------- IEmbeddedObjectInterface--------------------
class EmbeddedObject:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize EmbeddedObject. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current embedded object item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current embedded object item identifier |
		| 0 | No embedded object item  or an error occurred |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets an embedded object as the current item.

		:param id: Unique value identifying an embedded object item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current embedded object item identifier |
		| 0 | No current dimension item or an error occurred |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.SetId(id)

	def Create(self, filename:str, shti:int, x:float, y:float) -> int:
		"""
		Embeds a file's contents onto a sheet of the project.

		:param filename: Path and name of the file to be embedded in the project
		:type str: str
		:param shti: Identifier of the sheet to place the embedded object on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Object is embedded |
		| -2 | No project open |
		| -3 | Transaction pending |
		| -8 | OLE client creation failed |
		| -9 | Open file failed |
		| -12 | shti is invalid |
		| -13 | Sheet is read-only |
		| -14 | Invalid insert position |
		| -15 | No data available |
		| -16 | No image available |
		| -20 | Invalid file type |
		| -21 | 3D view is active |
		| -990120 | Internal error |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.Create(filename, shti, x, y)

	def Delete(self) -> int:
		"""
		Deletes the embedded object item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Embedded object item deleted |
		| -1 | No embedded object item set |
		| -2 | No project open |
		| -5 | Delete operation failed |
		| -11 | Part of read-only sheet |
		| -18 | Process in progress |
		| -990120 | Internal error |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.Delete()

	def Move(self, shti:int, x:float, y:float) -> int:
		"""
		Moves the embedded object item to a new position or sheet.

		:param shti: Sheet to place the embedded object on
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Embedded object is moved |
		| -1 | No embedded object item set |
		| -3 | Transaction pending |
		| -7 | Move operation failed |
		| -11 | Part of read-only sheet |
		| -12 | shti is invalid |
		| -13 | Sheet is read-only |
		| -14 | Invalid insert position |
		| -19 | Not possible to move to another sheet |
		| -990120 | Internal error |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.Move(shti, x, y)

	def Resize(self, width:float, height:float) -> int:
		"""
		Resizes the embedded object item.

		:param width: New width value
		:type float: float
		:param height: New height value
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Embedded object is resized |
		| -1 | No embedded object item set |
		| -3 | Transaction pending |
		| -6 | Resize operation failed |
		| -11 | Part of read-only sheet |
		| -990120 | Internal error |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.Resize(width, height)

	def Open(self) -> int:
		"""
		Opens the embedded object item with external third party software.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Embedded object item is opened externally |
		| -1 | No embedded object item set |
		| -2 | No project open |
		| -3 | Transaction pending |
		| -4 | Open operation failed |
		| -10 | Part of closed sheet |
		| -11 | Part of read-only sheet |
		| -17 | Already open |
		| -990120 | Internal error |

		Available since E3.series TLB-Versions: 21.00
		"""
		return self._obj.Open()

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current embedded object item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current embedded object item |
		| "&lt;Empty&gt;" | No embedded object item |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> str:
		"""
		Sets an embedded object as the current item.

		:param gid: Global identifier value of an embedded object item
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current embedded object item |
		| "&lt;Empty&gt;" | No current embedded object item or an error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.SetGID(gid)

# -------------------- IStateInterface--------------------
class State:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize State. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current state item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current state item identifier |
		| 0 | No state item |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a state as the current item.

		:param id: Unique value identifying a state item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current state item identifier |
		| 0 | No state item |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.SetId(id)

	def GetGID(self) -> str:
		"""
		Gets the global identifier of the current state item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GID&gt;" | Global identifier of the current state item |
		| "0-0-0-0" | No state item |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetGID()

	def SetGID(self, gid:str) -> int:
		"""
		Sets a state as the current item.

		:param gid: Global identifier value of a state item
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current state item identifier |
		| 0 | No state item |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.SetGID(gid)

	def GetGUID(self) -> str:
		"""
		Gets the globally unique identifier of the current state item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current state item |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetGUID()

	def SetGUID(self, guid:str) -> int:
		"""
		Sets a state as the current item.

		:param guid: Globally unique identifier value of a state item
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;GUID&gt;" | Globally unique identifier of the current state item |
		| "&lt;Empty&gt;" | No state item |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.SetGUID(guid)

	def GetName(self) -> str:
		"""
		Gets the state item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | State name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetName()

	def GetAttributeIds(self, name:str) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the state item.

		:param name: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, name)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the state item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetAttributeValue(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the state item's specified attributes.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.HasAttribute(name)

	def GetOwnerId(self) -> int:
		"""
		Gets the identifier of the state item's owner.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Item identifier of the owner |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetOwnerId()

	def GetOwnerType(self) -> int:
		"""
		Gets a value representing the type of the state item's owner.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type value of the owner |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetOwnerType()

	def GetStateType(self) -> int:
		"""
		Gets a value representing the state item's type.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Type value of the state item |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.GetStateType()

	def GetOwnerIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the state item's owners.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of owner items |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 23.00
		"""
		dummy=0
		ret, ids = self._obj.GetOwnerIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

# -------------------- IProjectConfiguratorInterface--------------------
class ProjectConfigurator:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize ProjectConfigurator. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def Delete(self, targetId:int, flags:int=0) -> tuple[int, tuple[int, int, str]]:
		"""
		Deletes items from a field item.

		:param targetId: Field item identifier
		:type int: int
		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | tuple[int, int, str] | List of items that could not be deleted and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Items are deleted |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No E³.series project open |
		| -3 | targetId contains an invalid item identifier value |
		| -4 | targetId contains the identifier of an invalid type of item |
		| -100 | Error occurred |
		| -102 | Field item's sheet is locked |
		| -103 | Field item belongs to an inactive variant or option |

		Available since E3.series TLB-Versions: 26.00
		"""
		dummy=0
		return self._obj.Delete(dummy, targetId, flags)

	def DeleteSheet(self, sheetId:int, flags:int=0) -> tuple[int, tuple[int, int, str]]:
		"""
		Deletes a sheet from the  project.

		:param sheetId: Identifier of sheet item
		:type int: int
		:param flags: Delete option flags., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | tuple[int, int, str] | List of items that could not be deleted and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Sheet or sheet content is deleted |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No E³.series project open |
		| -3 | sheetId contains an invalid identifier value |
		| -4 | sheetId is not a sheet item identifier |
		| -5 | sheetId identifier is an embedded sheet item |
		| -100 | Error occurred |
		| -102 | Sheet is locked |
		| -103 | Sheet belongs to an inactive variant or option |

		Available since E3.series TLB-Versions: 26.00
		"""
		dummy=0
		return self._obj.DeleteSheet(dummy, sheetId, flags)

	def DeleteDevice(self, devId:int, forced:bool, flags:int=0) -> tuple[int, tuple[int, int, str]]:
		"""
		Deletes a device from the project.

		:param devId: Identifier of device item
		:type int: int
		:param forced: If 1, device is deleted even if placed. If 0, device is nor deleted if it is placed.
		:type bool: bool
		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | tuple[int, int, str] | List of items that could not be deleted and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Device is deleted |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No E³.series project open |
		| -3 | devId contains an invalid identifier value |
		| -4 | devId is not a device item identifier |
		| -100 | Error occurred |
		| -103 | Device belongs to an inactive variant or option |
		| -104 | Device is locked |
		| -105 | Device has placed symbols on a sheet |
		| -106 | Wires in cable are in use |
		| -107 | Device is part of a group |
		| -108 | Cable inside the overbraid device is locked |
		| -109 | Device belonging to the block device is locked |
		| -110 | Terminal block of the terminal block device is locked |
		| -111 | Device belonging to the assembly device is locked |

		Available since E3.series TLB-Versions: 26.00
		"""
		dummy=0
		return self._obj.DeleteDevice(dummy, devId, forced, flags)

	def AddAttribute(self, targetId:int, attributeName:str, value:str, objectType1:int, objectType2:int, name:str="", flags:int=0) -> tuple[int, tuple[int, int, str]]:
		"""
		Adds an attribute to an item and items within  if it is a container.

		:param targetId: Identifier of the target item. Target item, and items within the target if it is a container, have the attribute added
		:type int: int
		:param attributeName: Internal name of attribute to add
		:type str: str
		:param value: Value of the attribute to add
		:type str: str
		:param objectType1: Flags values defining item types to add the attribute to. Since v2026-26.10, the flag values only apply if targetId is a field item
		:type int: int
		:param objectType2: Flags values defining item types to add the attribute to. Since v2026-26.10, the flag values only apply if targetId is a field item
		:type int: int
		:param name: Conductor or wire name. Default value is "<Empty>", defaults to ""
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | tuple[int, int, str] | List of items the attribute could not be added to and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Attribute added to items |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No project open |
		| -3 | targetId contains an invalid item identifier value |
		| -4 | targetId contains the identifier of an invalid item type |
		| -102 | Sheet is locked |
		| -103 | Items belong to an inactive variant or option |
		| -112 | Attribute could not be added to all items |
		| -134 | attributeName value is too long, Available since v2026-26.10 |
		| -135 | value value is too long for its text type, Available since v2026-26.10 |
		| -136 | Item being assigned the attribute is locked, Available since v2026-26.10 |

		Available since E3.series TLB-Versions: 26.00
		"""
		dummy=0
		return self._obj.AddAttribute(dummy, targetId, attributeName, value, objectType1, objectType2, name, flags)

	def DeleteAttribute(self, targetId:int, attributeName:str, value:str, objectType1:int, objectType2:int, name:str="", flags:int=0) -> tuple[int, tuple[int, int, str]]:
		"""
		Deletes an attribute from an item and items within  if it is a container.

		:param targetId: Identifier of the target item. Target item, and items within the target if it is a container, have the attribute deleted
		:type int: int
		:param attributeName: Internal name of attribute to delete
		:type str: str
		:param value: Value of the attribute to delete
		:type str: str
		:param objectType1: Flags values defining item types to delete the attribute from. Since v2026-26.10, the flag values only apply if targetId is a field item
		:type int: int
		:param objectType2: Flags values defining item types to delete the attribute from. Since v2026-26.10, the flag values only apply if targetId is a field item
		:type int: int
		:param name: Conductor or wire name. Default value is "<Empty>", defaults to ""
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | tuple[int, int, str] | List of items the attribute could not be deleted from and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Attribute deleted from items |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No project open |
		| -3 | targetId contains an invalid identifier value |
		| -4 | targetId contains the identifier of an invalid item type |
		| -102 | Sheet is locked |
		| -103 | Items belong to an inactive variant or option |
		| -113 | Attribute could not be deleted from all items |

		Available since E3.series TLB-Versions: 26.00
		"""
		dummy=0
		return self._obj.DeleteAttribute(dummy, targetId, attributeName, value, objectType1, objectType2, name, flags)

	def DeleteAndReconnect(self, targetId:int, flags:int=0) -> tuple[int, tuple[int, int, str]]:
		"""
		Deletes items from a field and restores connections passing out of it.

		:param targetId: Field item identifier
		:type int: int
		:param flags: Restore connection option flags., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | tuple[int, int, str] | List of items that could not be deleted and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Items are deleted and connections restored if possible |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No E³.series project open |
		| -3 | targetId contains an invalid identifier value |
		| -4 | targetId contains the identifier of an invalid type of item |
		| -100 | Error occurred |
		| -102 | Field item's sheet is locked |
		| -103 | Field item belongs to an inactive variant or option |

		Available since E3.series TLB-Versions: 26.00
		"""
		dummy=0
		return self._obj.DeleteAndReconnect(dummy, targetId, flags)

	def ChangeComponent(self, targetId:int, newComponent:str, oldComponent:str="", newVersion:str="", oldVersion:str="", flags:int=0) -> tuple[int, list[tuple[int, int, str]]]:
		"""
		Replaces, adds or removes a component to an item or items within a field.

		:param targetId: Identifier of the target item
		:type int: int
		:param newComponent: Name of the replacement component
		:type str: str
		:param oldComponent: Name of the  component to be replaced. Default value is "<Empty>", defaults to ""
		:type str: str
		:param newVersion: Version of the replacement component. Default value is "<Empty>", defaults to ""
		:type str: str
		:param oldVersion: Version of the component to be replaced. Default value is "<Empty>", defaults to ""
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | list[tuple[int, int, str]] | List of items for which the component could not be changed to and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Component changed on items |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No project open |
		| -3 | targetId contains an invalid item identifier value |
		| -4 | targetId contains the identifier of an invalid item type |
		| -104 | Device is locked |
		| -114 | Component does not match the device |
		| -115 | Component type is not supported in the device |
		| -116 | Target device could not be found |
		| -117 | Target device is a terminal strip |
		| -118 | Target device is a connector with inserts |
		| -119 | Target device is a feed-through connector without a component possibility |
		| -120 | Target device is a busbar without a component possibility |
		| -121 | Target device is a part of an assembly or an overbraid |
		| -122 | Target device is a cable duct inlet\outlet |
		| -123 | Target is an unassigned symbol on a block device |
		| -124 | Target device's component cannot be removed because it already has no component |
		| -125 | Target device's component cannot be removed because it is part of a variant |
		| -126 | Error occurred |
		| -127 | Error occurred |

		Available since E3.series TLB-Versions: 26.00
		"""
		dummy=0
		return self._obj.ChangeComponent(dummy, targetId, newComponent, oldComponent, newVersion, oldVersion, flags)

	def SwapSymbol(self, targetId:int, newPin:str, newSymbol:str="", flags:int=0) -> tuple[int, tuple[int, int, str]]:
		"""
		Swaps the pins or gates contained in a field item.

		:param targetId: Field item identifier
		:type int: int
		:param newPin: Name of the pin to be swapped in. For a normal device, a gate with the corresponding pin name is searched for
		:type str: str
		:param newSymbol: Name of the gate on the normal device to be swapped in. Default value is "<Empty>", defaults to ""
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | tuple[int, int, str] | List of items that could not be swapped and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbols are swapped |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No E³.series project open |
		| -3 | targetId contains an invalid identifier value |
		| -4 | targetId is not a field item identifier |
		| -102 | Sheet is locked |
		| -103 | Device belongs to an inactive variant or option |
		| -104 | Device is locked |
		| -128 | Both newPin and newSymbol have no value |
		| -129 | Connector does not have a pin name matching the newPin value |
		| -130 | Device does not have a gate matching the newSymbol value |
		| -131 | Device does not have a gate matching the newSymbol value or the gate does not have a pin matching the newPin value |
		| -132 | Device does not have a gate with a pin matching the newPin value |

		Available since E3.series TLB-Versions: 26.01
		"""
		dummy=0
		return self._obj.SwapSymbol(dummy, targetId, newPin, newSymbol, flags)

	def ChangeSignal(self, targetId:int, oldSignalName:str, newSignalName:str, flags:int=0) -> tuple[int, tuple[int, int, str]]:
		"""
		Replaces, adds or removes signals of pin items within a field.

		:param targetId: Identifier of the target field item
		:type int: int
		:param oldSignalName: Name of the signal to be replaced
		:type str: str
		:param newSignalName: Name of the replacement signal
		:type str: str
		:param flags: Change signal option flags., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | errorMessages | tuple[int, int, str] | List of items for which the component could not be changed to and the associated error information |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Signal changed on items |
		| -1 | No E³.ProjectConfigurator license found |
		| -2 | No project open |
		| -3 | targetId contains an invalid item identifier value |
		| -4 | targetId contains the identifier of an invalid item type |
		| -100 | Error occurred |
		| -102 | Field item's sheet is locked |
		| -103 | Field item belongs to an inactive variant or option |
		| -127 | Error occurred |
		| -137 | oldSignal or newSignal value is too long |
		| -138 | oldSignal or newSignal contains invalid characters |

		Available since E3.series TLB-Versions: 26.01
		"""
		dummy=0
		return self._obj.ChangeSignal(dummy, targetId, oldSignalName, newSignalName, flags)

# -------------------- IDbeComponentInterface--------------------
class DbeComponent:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeComponent. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current component item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current component item identifier |
		| 0 | No component item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a component as the current item.

		:param id: Unique value identifying a component item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current component item identifier |
		| 0 | No current component item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetId(id)

	def Save(self) -> int:
		"""
		Saves the component to the database.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Component is saved to the database |
		| -1 | No E³.DatabaseEditor project open or no component item set |
		| -2 | Database is read-only |
		| -3 | Component could not be saved |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Save()

	def Remove(self, save_changes:bool=False) -> int:
		"""
		Removes the component from the E³.DatabaseEditor project.

		:param save_changes: Indicates whether the component should be saved to the database before being removed from the project. If 1, the component is saved to the database. If 0, the component is not saved to the database., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Component is removed from the E³.DatabaseEditor project |
		| -1 | No E³.DatabaseEditor project open or no component item set |
		| -2 | Database is read-only |
		| -3 | Component could not be removed |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Remove(save_changes)

	def GetAttributeIds(self, end:int=0, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the component item.

		:param end: Attribute location filter., defaults to 0
		:type int: int
		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute identifiers are supplied if attnam is "<Empty>". The default string value is "<Empty>", defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No attributes are found |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, end, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def AddAttributeValue(self, name:str, value:str, end:int=0) -> int:
		"""
		Adds an attribute to the component item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str
		:param end: Where to apply the attribute., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| -1 | No E³.DatabaseEditor project open or no component item set |
		| -2 | Attribute value could not be added |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.AddAttributeValue(name, value, end)

	def GetName(self) -> str:
		"""
		Gets the name of the component item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Component item name value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetName()

	def GetVersion(self) -> str:
		"""
		Gets the version of the component item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Component item version value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetVersion()

	def GetSubType(self) -> int:
		"""
		Gets the component item's subtype.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Subtype of the component |
		| -1 | No project open or no component item set |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetSubType()

	def SetSubType(self, subtype:int) -> int:
		"""
		Sets the component item's subtype.

		:param subtype: Subtype value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Previous subtype value |
		| -1 | No project open or no component item set |
		| -2 | subtype is invalid |
		| -3 | Component is invalid |
		| -4 | subtype is invalid for standard device or block device component, Available since v2019-20.00 |
		| -5 | subtype is invalid for block devices, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetSubType(subtype)

	def GetComponentType(self) -> int:
		"""
		Gets the component item's type.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type of the component |
		| -1 | No project open |
		| -2 | No component item set |
		| -3 | Undefined component type |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.GetComponentType()

	def SetModelName(self, modelName:str, flags:int=0) -> int:
		"""
		Sets the model of the component item.

		:param modelName: Name of model
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Model assigned to component item |
		| -1 | No open project |
		| -2 | No component item set |
		| -3 | modelName value is invalid |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.SetModelName(modelName, flags)

	def GetModelName(self, flags:int=0) -> str:
		"""
		Gets the name of the component item's model name.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Model name |
		| "&lt;Empty&gt;" | Component item has no model or an error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.GetModelName(flags)

# -------------------- IDbeAttributeInterface--------------------
class DbeAttribute:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeAttribute. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current attribute item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current attribute item identifier |
		| 0 | No attribute item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets an attribute as the current item.

		:param id: Unique value identifying an attribute item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current attribute item identifier |
		| 0 | No current attribute item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetId(id)

	def SetValue(self, value:str) -> int:
		"""
		Sets the attribute item's value.

		:param value: Value to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Value is applied |
		| -1 | No E³.DatabaseEditor project open or no attribute item set |
		| -2 | Value could not be set |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetValue(value)

	def Delete(self) -> int:
		"""
		Deletes the attribute item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Attribute item was deleted |
		| -1 | No E³.DatabaseEditor project open or no attribute item set |
		| -3 | Attribute value could not be deleted |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Delete()

	def GetName(self) -> str:
		"""
		Gets the attribute item's name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.GetName()

	def GetInternalName(self) -> str:
		"""
		Gets the attribute item's user-defined internal name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's user-defined internal name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.GetInternalName()

	def GetInternalValue(self) -> str:
		"""
		Gets the  attribute item's internal value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's internal value |
		| "&lt;Empty&gt;" | Attribute item has no internal value or an error occurred |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.GetInternalValue()

	def GetFormattedValue(self) -> str:
		"""
		Gets the attribute item's formatted value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's formatted value |
		| "&lt;Empty&gt;" | Attribute item has no formatted value or an error occurred |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.GetFormattedValue()

	def GetValue(self) -> str:
		"""
		Gets the attribute item's value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute item's value |
		| "&lt;Empty&gt;" | Attribute item has no value or an error occurred |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.GetValue()

# -------------------- IDbeSymbolInterface--------------------
class DbeSymbol:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeSymbol. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current symbol item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current symbol item identifier |
		| 0 | No symbol item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a symbol as the current item.

		:param id: Unique value identifying a symbol item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current symbol item identifier |
		| 0 | No current symbol item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetId(id)

	def Save(self) -> int:
		"""
		Saves the symbol to the database.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol is saved to the database |
		| -1 | No E³.DatabaseEditor project open or no symbol item set |
		| -2 | Database is read-only |
		| -3 | Symbol could not be saved |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Save()

	def Remove(self, save_changes:bool=False) -> int:
		"""
		Removes the symbol from the E³.DatabaseEditor project.

		:param save_changes: Indicates whether the symbol should be saved to the database before being removed from the project. If 1, the symbol is saved to the database. If 0, the symbol is not saved to the database., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol is removed from the E³.DatabaseEditor project |
		| -1 | No E³.DatabaseEditor project open or no symbol item set |
		| -2 | Database is read-only |
		| -3 | Symbol could not be removed |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Remove(save_changes)

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the symbol item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute identifier are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of items in ids |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetGraphicIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the graphics on the symbol sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of graphic item identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of graphic item identifiers in ids |
		| -1 | No E³.DatabaseEditor project open or no symbol item set |
		| -2 | No valid symbol sheet view, Available from v2019-20.22 and v2020-21.01 |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		ret, ids = self._obj.GetGraphicIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetTextIds(self, texttype:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the text items assigned to the symbol item.

		:param texttype: Text type filter. Only text identifiers with this Text Type value will be supplied. All text identifiers are supplied if texttype is set to 0. The default string value is 0, defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of text items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of items in ids |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		ret, ids = self._obj.GetTextIds(dummy, texttype)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def AddAttributeValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the symbol item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| -1 | No E³.DatabaseEditor project open or no symbol item set |
		| -2 | Attribute value could not be added |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.AddAttributeValue(name, value)

	def GetName(self) -> str:
		"""
		Gets the name of the symbol item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Symbol item name value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetName()

	def GetVersion(self) -> str:
		"""
		Gets the version of the symbol item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Symbol item version value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetVersion()

	def GetCharacteristic(self) -> str:
		"""
		Gets the characteristic of the symbol item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Symbol item characteristic value |
		| "&lt;Empty&gt;" | Symbol item has no characteristic or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetCharacteristic()

	def GetSubType(self) -> int:
		"""
		Gets the symbol item's subtype.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Subtype of the symbol |
		| -1 | No project open or no symbol item set |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetSubType()

	def SetSubType(self, subtype:int) -> int:
		"""
		Sets the symbol item's subtype.

		:param subtype: Subtype value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Previous subtype value |
		| -1 | No project open or no symbol item set |
		| -2 | subtype is invalid |
		| -3 | Symbol is invalid |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetSubType(subtype)

	def FitSpaceRequirement(self) -> int:
		"""
		Resizes the surrounding rectangle graphic item to fit around the graphic content of the symbol.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Fitting operation was executed |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.FitSpaceRequirement()

	def GetSymbolType(self) -> int:
		"""
		Gets the symbol item's type.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Type of the symbol |
		| -1 | No project open or no symbol item set |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.GetSymbolType()

	def ImportDXF(self, filename:str, scale:float, x:float, y:float, rot:int, font:str, flags:int=0) -> int:
		"""
		Imports a DXF or DWG format file's content on to the symbol item sheet.

		:param filename: Path and name of the import file
		:type str: str
		:param scale: Resize scale to apply to the import file contents
		:type float: float
		:param x: Placement of the file contents on the x-axis of the symbol sheet. Ignored if bit value of 16 is not included in flags
		:type float: float
		:param y: Placement of the file contents on the y-axis of the symbol sheet. Ignored if bit value of 16 is not included in flags
		:type float: float
		:param rot: Not used
		:type int: int
		:param font: Font name of texts on the symbol sheet
		:type str: str
		:param flags: Flags bit values specifying how the file contents should be imported, defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet item |
		| -1 | No E³.DatabaseEditor project open or no symbol item set |
		| -2 | filename import file does not exist |
		| -3 | rot value is invalid |
		| -4 | font value is invalid |
		| -5 | Import file content exceeds the sheet size |
		| -7 | Error reading DXF file, Available since v2022-23.61 and v2023-24.31 |

		Available since E3.series TLB-Versions: 19.01
		"""
		return self._obj.ImportDXF(filename, scale, x, y, rot, font, flags)

	def GetDXFSize(self, filename:str, font:str) -> tuple[int, float, float, float]:
		"""
		Gets the size and scale for the symbol sheet based on the file contents and font.

		:param filename: Path and name of the DXF or DWG file. Since v2022-23.53 and v2023-24.22, the path and name of a DGN file is also accepted
		:type str: str
		:param font: Name of font to apply
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | dx | float | Required width passed out of the function |
		| 2 | dy | float | Required height passed out of the function |
		| 3 | scale | float | Maximum possible scale for the sheet and. font. passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Size and maximum scale supplied |
		| 0 | Size and scale could not be determined |
		| -1 | No E³.DatabaseEditor project open or no symbol item set |
		| -2 | filename is invalid |
		| -3 | font is invalid |
		| -4 | Error reading DXF file, Available since v2022-23.61 and v2023-24.31 |

		Available since E3.series TLB-Versions: 19.01
		"""
		dummy=0
		return self._obj.GetDXFSize(filename, font, dummy, dummy, dummy)

	def PlaceSymbol(self, symname:str, version:str, x:float, y:float, rot:str, flags:int=0) -> int:
		"""
		Places a symbol on to the symbol item's sheet.

		:param symname: Name of the symbol to place
		:type str: str
		:param version: Version of the symbol to place. If version is invalid, the current version is assumed
		:type str: str
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Rotation value in a counterclockwise direction
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol is placed |
		| -1 | No E³.DatabaseEditor project open or no symbol item set |
		| -3 | Invalid symbol type |
		| -4 | Symbol not found in the database |
		| -5 | Insertion point lies outside of the sheet |
		| -6 | Symbol could not be placed |
		| -7 | Invalid rotation value |

		Available since E3.series TLB-Versions: 19.41
		"""
		return self._obj.PlaceSymbol(symname, version, x, y, rot, flags)

	def GetNodeIds(self, flags:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the nodes on the symbol sheet.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of node item identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of node item identifiers in ids |
		| -1 | No E³.DatabaseEditor project open or no symbol item set |
		| -2 | &gt;flags value is invalid |

		Available since E3.series TLB-Versions: 21.21
		"""
		dummy=0
		ret, ids = self._obj.GetNodeIds(dummy, flags)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

# -------------------- IDbeTextInterface--------------------
class DbeText:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeText. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current text item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current text item identifier |
		| 0 | No text item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a text as the current item.

		:param id: Unique value identifying a text item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current text item identifier |
		| 0 | No current text item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetId(id)

	def Create(self, id:int, texttype:int, x:float, y:float, textvalue:str="") -> int:
		"""
		Adds a new text item onto a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the text item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param texttype: Text type. See Text Types for possible values
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param textvalue: Text value, defaults to ""
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new text item |
		| -1 | No open project or id is invalid |
		| -2 | texttype is invalid |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Create(id, texttype, x, y, textvalue)

	def Delete(self) -> int:
		"""
		Deletes the text from the E³.DatabaseEditor project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text item is deleted |
		| -1 | No project open or no text item was set |
		| -3 | Text could not be deleted |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Delete()

	def GetColour(self) -> int:
		"""
		Gets the text item's text color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or an error occurred |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetColour()

	def SetColour(self, value:int) -> int:
		"""
		Sets the text item's text color.

		:param value: Color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetColour(value)

	def GetAlignment(self) -> int:
		"""
		Gets the text's current alignment value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Alignment value could not be supplied |
		| 1 | Left aligned |
		| 2 | Centered |
		| 3 | Right aligned |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetAlignment()

	def SetAlignment(self, newval:int) -> int:
		"""
		Sets the text's alignment value.

		:param newval: Alignment value to be set
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Alignment value set |
		| 1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetAlignment(newval)

	def GetBallooning(self) -> int:
		"""
		Gets the text's ballooning value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Ballooning value applied, See Ballooning for possible values |
		| 0 | Ballooning value could not be supplied or there is no ballooning |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetBallooning()

	def SetBallooning(self, onoff:bool, _type:int) -> int:
		"""
		Sets the text's ballooning value.

		:param onoff: Indicating if the value should be added or removed. If 1, the value is added. If 0, the value is removed
		:type bool: bool
		:param _type: New ballooning value. See Ballooning for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Previous ballooning value, See Ballooning for possible values |
		| 0 | Previously no ballooning value set or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetBallooning(onoff, _type)

	def GetFontName(self) -> str:
		"""
		Gets the text item's font name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Font&gt;" | Name of the text's font |
		| "Arial" | Name of the text's font is Arial or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetFontName()

	def SetFontName(self, newname:str) -> int:
		"""
		Sets the text item's font name.

		:param newname: Name of font to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetFontName(newname)

	def GetHeight(self) -> float:
		"""
		Gets the text item's font height.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0 | Size of the text's font height |
		| 0.0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetHeight()

	def SetHeight(self, newval:float) -> int:
		"""
		Sets the text item's font height.

		:param newval: Value of font height to apply
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font height applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetHeight(newval)

	def GetLevel(self) -> int:
		"""
		Gets the text item's display level value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Display level value supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetLevel()

	def SetLevel(self, newval:int) -> int:
		"""
		Sets the text item's display level value.

		:param newval: Display level value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display level value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetLevel(newval)

	def GetLocking(self) -> bool:
		"""
		Gets the text item's flag value determining if the text position is locked.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text position is locked |
		| 0 | Text position is unlocked or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetLocking()

	def SetLocking(self, newval:bool) -> bool:
		"""
		Sets the text item's flag value determining if the text position is locked.

		:param newval: Position locking value to apply. 1 will lock the text position. 0 will unlock the text position
		:type bool: bool

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text position locking value is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetLocking(newval)

	def GetMode(self) -> int:
		"""
		Gets the text item's text ratio.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..3 | Text ratio value supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetMode()

	def SetMode(self, newval:int) -> int:
		"""
		Sets the text item's text ratio.

		:param newval: Text ratio value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text ratio value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetMode(newval)

	def GetRotation(self) -> float:
		"""
		Gets the text item's rotation value.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0..&lt; 360.0 | Text rotation value in degrees |
		| 0.0 | Text rotation value  or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetRotation()

	def SetRotation(self, rotation:float) -> float:
		"""
		Sets the text item's rotation value.

		:param rotation: Rotation value in degrees to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0.0..&lt; 360.0 | Previous text rotation value in degrees supplied |
		| 0.0 | Previous text rotation value or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetRotation(rotation)

	def GetSingleLine(self) -> int:
		"""
		Gets the text item's status determining if multi-line text is displayed on a single line.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text is displayed on a single line |
		| 0 | Text is not displayed on a single line or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetSingleLine()

	def SetSingleLine(self, newval:int) -> int:
		"""
		Sets the text item's status determining if multi-line text is displayed on a single line.

		:param newval: Single line value to apply. If 1, mutli-line text is displayed on a single line. If 0, mutli-line text is displayed on multiple lines
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Single line value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetSingleLine(newval)

	def GetStyle(self) -> int:
		"""
		Gets the text item's font style.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..7 | Font style value supplied |
		| 0 | Regular font style value used or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetStyle()

	def SetStyle(self, newval:int) -> int:
		"""
		Sets the text item's font style.

		:param newval: Font style to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Font style value supplied |
		| 0 | No font style value used or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetStyle(newval)

	def GetText(self) -> str:
		"""
		Gets the text item's text value including text token translations.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Text value is supplied |
		| "&lt;Empty&gt;" | Text item does not have a value or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetText()

	def SetText(self, newtext:str) -> int:
		"""
		Sets the text item's text value.

		:param newtext: Text value to apply
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text value is applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetText(newtext)

	def GetType(self) -> int:
		"""
		Gets the text item's text type.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Text type supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetType()

	def SetType(self, texttype:int) -> int:
		"""
		Sets the text item's text type.

		:param texttype: Text type to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text type value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetType(texttype)

	def GetVisibility(self) -> int:
		"""
		Gets the text item's visibility status.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text visibility status is shown |
		| 0 | Text visibility status is hidden or an error occurred |

		Available since E3.series TLB-Versions: 21.11
		"""
		return self._obj.GetVisibility()

	def SetVisibility(self, newval:int) -> int:
		"""
		Sets the text item's visibility status.

		:param newval: Visibility status value. If 1, the text is shown. If 0, the text is hidden
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Text visibility status value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 21.11
		"""
		return self._obj.SetVisibility(newval)

	def GetPosition(self) -> tuple[int, float, float]:
		"""
		Gets the text item's position.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Position on the x-axis passed out of the function |
		| 2 | y | float | Position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Position values are supplied |
		| -1 | No project open |
		| -2 | No text item set |

		Available since E3.series TLB-Versions: 21.21
		"""
		dummy=0
		return self._obj.GetPosition(dummy, dummy)

# -------------------- IDbeGraphInterface--------------------
class DbeGraph:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeGraph. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current graphic item identifier |
		| 0 | No graphic item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a graphic as the current item.

		:param id: Unique value identifying a graphic item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current graphic item identifier |
		| 0 | No current graphic item or an error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetId(id)

	def GetColour(self) -> int:
		"""
		Gets the graphic item's color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or an error occurred |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetColour()

	def SetColour(self, newcol:int) -> int:
		"""
		Sets the graphic item's color.

		:param newcol: Color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetColour(newcol)

	def GetLevel(self) -> int:
		"""
		Gets the graphic item's display level value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Display level value supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetLevel()

	def SetLevel(self, newlev:int) -> int:
		"""
		Sets the graphic item's display level value.

		:param newlev: Display level value to apply
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display level value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetLevel(newlev)

	def SetBlobInfo(self, filename:str) -> int:
		"""
		Sets the blob graphic item's data.

		:param filename: Path and name of the file containing the new blob data
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | New data is applied |
		| -1 | No valid graphic item set or no project open |
		| -2 | filename not found |
		| -3 | filename could not be opened |
		| -4 | filename is empty |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetBlobInfo(filename)

	def CreateArc(self, id:int, x:float, y:float, radius:float, start:float, end:float) -> int:
		"""
		Adds a new arc graphic item on a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param radius: Radius of the circle of the arc
		:type float: float
		:param start: Angle of the circle to begin the arc
		:type float: float
		:param end: Angle of the circle to end the arc
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new arc graphic item |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.CreateArc(id, x, y, radius, start, end)

	def CreateCircle(self, id:int, x:float, y:float, radius:float) -> int:
		"""
		Adds a new circle graphic item onto a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param radius: Radius of the circle
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new circle graphic item |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.CreateCircle(id, x, y, radius)

	def CreatePolygon(self, id:int, pnts:int, x:list[int], y:list[int]) -> int:
		"""
		Adds a new polygon graphic item onto a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param pnts: Number of placement positions in. x. and. y
		:type int: int
		:param x: Array of placement positions on the x-axis
		:type list[int]: list[int]
		:param y: Array of placement positions on the y-axis
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| -1 | No open project or id is invalid |
		| -2 | x or y is an invalid array |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.CreatePolygon(id, pnts, x, y)

	def CreateRectangle(self, id:int, x1:float, y1:float, x2:float, y2:float) -> int:
		"""
		Adds a new rectangle graphic item onto a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param x1: First placement position on the x-axis
		:type float: float
		:param y1: First placement position on the y-axis
		:type float: float
		:param x2: Second placement position on the x-axis
		:type float: float
		:param y2: Second placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.CreateRectangle(id, x1, y1, x2, y2)

	def CreateLine(self, id:int, x1:float, y1:float, x2:float, y2:float) -> int:
		"""
		Adds a new line graphic item onto a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param x1: First placement position on the x-axis
		:type float: float
		:param y1: First placement position on the y-axis
		:type float: float
		:param x2: Second placement position on the x-axis
		:type float: float
		:param y2: Second placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.CreateLine(id, x1, y1, x2, y2)

	def CreateImage(self, id:int, xpos:float, ypos:float, xsize:float, ysize:float, filename:str, embed:int=1) -> int:
		"""
		Adds a new image graphic item onto a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param xpos: Placement position on the x-axis
		:type float: float
		:param ypos: Placement position on the y-axis
		:type float: float
		:param xsize: Size of the graphic on the x-axis
		:type float: float
		:param ysize: Size of the graphic on the y-axis
		:type float: float
		:param filename: Path and name of the image file to be used
		:type str: str
		:param embed: If set to 0, a reference to the image is stored in the project. If set to 1, the image is stored internally in the project., defaults to 1
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new image graphic item |
		| 0 | Error occurred |
		| -1 | No open project or id is an invalid symbol or model identifier |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.CreateImage(id, xpos, ypos, xsize, ysize, filename, embed)

	def CreateCurve(self, id:int, pnts:int, x:list[float], y:list[float]) -> int:
		"""
		Adds a new curve graphic item onto a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param pnts: Number of placement positions in. x. and. y
		:type int: int
		:param x: Array of placement positions on the x-axis
		:type list[float]: list[float]
		:param y: Array of placement positions on the y-axis
		:type list[float]: list[float]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new graphic item |
		| -1 | No open project or id is an invalid symbol or model identifier |
		| -2 | x or y is an invalid array |

		Available since E3.series TLB-Versions: 19.00
		"""
		x = [0.] + x
		y = [0.] + y
		return self._obj.CreateCurve(id, pnts, x, y)

	def CreateCloud(self, id:int, pnts:int, x:list[float], y:list[float]) -> int:
		"""
		Adds a new cloud graphic item onto  a symbol or model of the E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param pnts: Number of placement positions in. x. and. y
		:type int: int
		:param x: Array of placement positions on the x-axis
		:type list[float]: list[float]
		:param y: Array of placement positions on the y-axis
		:type list[float]: list[float]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new cloud graphic item |
		| 0 | Error creating cloud graphic item |
		| -1 | No open project or id is an invalid symbol or model identifier |
		| -2 | x or y is an invalid array |

		Available since E3.series TLB-Versions: 19.00
		"""
		x = [0.] + x
		y = [0.] + y
		return self._obj.CreateCloud(id, pnts, x, y)

	def CreateBlob(self, id:int, filename:str) -> int:
		"""
		Creates a new blob graphic item owned by the symbol or model in a E³.DatabaseEditor project.

		:param id: Identifier of the symbol or model to place the graphic item on. Previous to v2022-23.54 and v2023-24.22, only symbol identifiers are valid
		:type int: int
		:param filename: Path and name of the file containing the blob data
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new blob graphic item |
		| -1 | No open project or id is an invalid symbol or model identifier |
		| -2 | filename not found |
		| -3 | filename could not be opened |
		| -4 | filename is empty |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.CreateBlob(id, filename)

	def SaveBlob(self, filename:str) -> int:
		"""
		Saves a blob graphic item's blob data to an external file.

		:param filename: Path and name of the blob file
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Blob saved to external file |
		| -1 | No valid graphic item set or no project open |
		| -2 | File already exists |
		| -3 | File has no blob or is empty |
		| -4 | File could not be saved |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SaveBlob(filename)

	def Place(self, x:float, y:float) -> int:
		"""
		Places or repositions the graphic item.

		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Graphic item placed or repositioned |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Place(x, y)

	def Delete(self) -> int:
		"""
		Deletes the graphic item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Graphic item was deleted |
		| 1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.Delete()

	def GetTypeId(self) -> int:
		"""
		Gets a value representing the type of the graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Type value of the graphic item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetTypeId()

	def GetArc(self) -> tuple[int, float, float, float, float, float]:
		"""
		Gets the arc placement data of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xm | float | Position of the arc center on the x-axis passed out of the function |
		| 2 | ym | float | Position of the arc center on the y-axis passed out of the function |
		| 3 | rad | float | Radius of the arc passed out of the function |
		| 4 | startang | float | Start angle of the arc passed out of the function |
		| 5 | endang | float | End angle of the arc passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Arc data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		return self._obj.GetArc(dummy, dummy, dummy, dummy, dummy)

	def GetCircle(self) -> tuple[int, float, float, float]:
		"""
		Gets the circle placement position of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xm | float | Position of the circle center on the x-axis passed out of the function |
		| 2 | ym | float | Position of the circle center on the y-axis passed out of the function |
		| 3 | rad | float | Radius of the circle passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Circle data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		return self._obj.GetCircle(dummy, dummy, dummy)

	def GetLine(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the line placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x1 | float | First placement position on the x-axis passed out of the function |
		| 2 | y1 | float | First placement position on the y-axis passed out of the function |
		| 3 | x2 | float | Second placement position on the x-axis passed out of the function |
		| 4 | y2 | float | Second placement position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Line data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		return self._obj.GetLine(dummy, dummy, dummy, dummy)

	def GetPolygon(self) -> tuple[int, int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the polygon placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | npnts | int | Number of placement positions passed out of the function |
		| 2 | xarr | tuple[float,...] | Array of placement positions on the x-axis passed out of the function |
		| 3 | yarr | tuple[float,...] | Array of placement positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Polygon data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		ret, npnts, xarr, yarr = self._obj.GetPolygon(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, npnts, xarr, yarr

	def GetRectangle(self) -> tuple[int, float, float, float, float]:
		"""
		Gets the rectangle placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x1 | float | First placement position on the x-axis passed out of the function |
		| 2 | y1 | float | First placement position on the y-axis passed out of the function |
		| 3 | x2 | float | Second placement position on the x-axis passed out of the function |
		| 4 | y2 | float | Second placement position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Rectangle data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		return self._obj.GetRectangle(dummy, dummy, dummy, dummy)

	def GetCurve(self) -> tuple[int, int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the curve placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | npnts | int | Number of placement positions passed out of the function |
		| 2 | xarr | tuple[float,...] | Array of placement positions on the x-axis passed out of the function |
		| 3 | yarr | tuple[float,...] | Array of placement positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Curve data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		ret, npnts, xarr, yarr = self._obj.GetCurve(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, npnts, xarr, yarr

	def GetCloud(self) -> tuple[int, int, tuple[float,...], tuple[float,...]]:
		"""
		Gets the cloud placement positions of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | npnts | int | Number of placement positions passed out of the function |
		| 2 | xarr | tuple[float,...] | Array of placement positions on the x-axis passed out of the function |
		| 3 | yarr | tuple[float,...] | Array of placement positions on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Cloud data is successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		ret, npnts, xarr, yarr = self._obj.GetCloud(dummy, dummy, dummy)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		return ret, npnts, xarr, yarr

	def GetLineColour(self) -> int:
		"""
		Gets the graphic item's line color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Line color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or an error occurred |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetLineColour()

	def SetLineColour(self, newcol:int) -> int:
		"""
		Sets the graphic item's line color.

		:param newcol: Line color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Line color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetLineColour(newcol)

	def GetLineWidth(self) -> float:
		"""
		Gets the line width of a graphic item.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Line width value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetLineWidth()

	def SetLineWidth(self, newwid:float) -> float:
		"""
		Sets the line width of a graphic item.

		:param newwid: Line width value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.0 | Previous line width value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetLineWidth(newwid)

	def GetLineStyle(self) -> int:
		"""
		Gets the line style of a graphic item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Line style value, See Line Styles for possible values |
		| 0 or &gt;47 | 0 or &gt;47 |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetLineStyle()

	def SetLineStyle(self, newstyle:int) -> int:
		"""
		Sets the line style of a graphic item.

		:param newstyle: Line style value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Previous line style value, See Line Styles for possible values |
		| 0 or &gt;47 | Previous line style value |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetLineStyle(newstyle)

	def SetHatchColour(self, newcol:int) -> int:
		"""
		Sets the graphic item's hatch color.

		:param newcol: Hatch color value to apply. See Colors for possible values
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Hatch color value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetHatchColour(newcol)

	def GetHatchColour(self) -> int:
		"""
		Gets the graphic item's hatch color.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..255 | Hatch color value, See Colors for possible values |
		| 0 | Black (RGB: 0, 0, 0 ) or an error occurred |
		| -1 | Automatic color |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetHatchColour()

	def GetHatchLineWidth(self) -> float:
		"""
		Gets the line width of a graphic item's hatch.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Hatch line width value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetHatchLineWidth()

	def SetHatchLineWidth(self, newwid:float) -> float:
		"""
		Sets the line width of a graphic item's hatch.

		:param newwid: Hatch line width value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.0 | Previous hatch line width value |
		| 0.0 | Error occurred |
		| -1.0 | newwid value is invalid |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetHatchLineWidth(newwid)

	def GetHatchLineStyle(self) -> int:
		"""
		Gets the line style of a graphic item's hatch.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Hatch line style value, See Line Styles for possible values |
		| 0 or &gt;47 | 0 or &gt;47 |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetHatchLineStyle()

	def SetHatchLineStyle(self, newstyle:int) -> int:
		"""
		Sets the line style of a graphic item's hatch.

		:param newstyle: Hatch line style value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..47 | Previous hatch line style value, See Line Styles for possible values |
		| 0 or &gt;47 | Previous hatch line style value |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetHatchLineStyle(newstyle)

	def GetHatchLineDistance(self) -> float:
		"""
		Gets the distance between the lines of a graphic item's hatch.

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Hatch line distance value |
		| -1.0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.GetHatchLineDistance()

	def SetHatchLineDistance(self, newdist:float) -> float:
		"""
		Sets the distance between the lines of a graphic item's hatch.

		:param newdist: Hatch line distance value to apply
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0.1 | Previous hatch line distance value |
		| 0.0 | Error occurred |
		| -1.0 | newdist value is invalid |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetHatchLineDistance(newdist)

	def GetHatchPattern(self) -> tuple[int, float, float]:
		"""
		Gets a value representing the used hatch pattern of the graphic item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | angle1 | float | Angle value in degrees of a hatch line if a line or cross hatch pattern is in use. The value will be 0.0 if neither a line or cross hatch pattern is in use |
		| 2 | angle2 | float | Angle value in degrees of a hatch line if a cross hatch pattern is in use. The value will be 0.0 if a cross hatch pattern is not in use |

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | No hatch pattern or error occurred |
		| 1 | Solid hatch |
		| 2 | Line hatch |
		| 4 | Cross hatch |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		return self._obj.GetHatchPattern(dummy, dummy)

	def SetHatchPattern(self, newpat:int, angle1:float, angle2:float) -> int:
		"""
		Sets a value representing the hatch pattern of the graphic item.

		:param newpat: Hatch pattern value to apply
		:type int: int
		:param angle1: Angle value in degrees of a hatch line to apply. Value is only used if newPat is a line or cross hatch pattern value. The value range is from 90.0 (pointing upwards) to 0.0 (pointing rightwards) to -90.0 (pointing downwards)
		:type float: float
		:param angle2: Angle value in degrees of a hatch line  to apply. Value is only used if newPat is a cross hatch pattern value. The value range is from 90.0 (pointing upwards) to 0.0 (pointing rightwards) to -90.0 (pointing downwards)
		:type float: float

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | No previous hatch pattern value set or error occurred |
		| 1, 2 or 4 | Previous hatch pattern value |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetHatchPattern(newpat, angle1, angle2)

	def GetImageInfo(self) -> tuple[int, float, float, float, float, str, int]:
		"""
		Gets the image graphic item's values.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xpos | float | Placement position on the x-axis passed out of the function |
		| 2 | ypos | float | Placement position on the y-axis passed out of the function |
		| 3 | xsize | float | Size of the graphic on the x-axis passed out of the function |
		| 4 | ysize | float | Size of the graphic on the y-axis passed out of the function |
		| 5 | filename | str | Path and name of the image file passed out of the function |
		| 6 | embed | int | Image storage status passed out of the function. If value is 0, a reference to the image stored in the project. If value is 1, the image is stored internally in the project |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item's values supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		dummy=0
		return self._obj.GetImageInfo(dummy, dummy, dummy, dummy, dummy, dummy)

	def SetImageInfo(self, xpos:float, ypos:float, xsize:float, ysize:float, filename:str="", embed:int=-1) -> int:
		"""
		Sets the image graphic item's values.

		:param xpos: Placement position on the x-axis
		:type float: float
		:param ypos: Placement position on the y-axis
		:type float: float
		:param xsize: Size of the graphic on the x-axis
		:type float: float
		:param ysize: Size of the graphic on the y-axis
		:type float: float
		:param filename: Path and name of the image file to be used, defaults to ""
		:type str: str
		:param embed: If set to 0, a reference to the image is stored in the project. If set to 1, the image is stored internally in the project. If set to -1, the current image storage method will be retained., defaults to -1
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New values are applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 19.00
		"""
		return self._obj.SetImageInfo(xpos, ypos, xsize, ysize, filename, embed)

	def OptimizeGraphicObjects(self, ids:list[int], mode:int, angle:int) -> tuple[int, list[int]]:
		"""
		Optimizes the graphics objects by reducing the number and modifying them.

		:param ids: Array of graphic identifiers to be optimized passed into the function and passed out optimized
		:type list[int]: list[int]
		:param mode: Type of optimization to apply
		:type int: int
		:param angle: Maximum angle for converting arcs to polygons
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | list[int] | Array of graphic identifiers to be optimized passed into the function and passed out optimized |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Graphic items have been optimized |
		| -1 | No project open |
		| -2 | Invalid mode value |
		| -3 | angle value out of range |

		Available since E3.series TLB-Versions: 22.00, 22.00, 21.01, 20.22
		"""
		ret, ids = self._obj.OptimizeGraphicObjects(ids, mode, angle)
		ids = ids[1:] if type(ids) == list and len(ids) > 0 else []
		return ret, ids

	def GetInsideGraphIds(self, flags:int= 0 ) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the graphic items placed within the current graphic item.

		:param flags: Not used., defaults to  0 
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of graphic item identifiers |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=0 | Number of graphic item identifiers in ids |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No graphic item set |
		| -3 | flags value is invalid |
		| -4 | Graphic item does not exist |
		| -5 | Graphic item is placed on an invalid sheet |
		| -6 | Graphic item is not closed or an invalid graphic type |
		| -7 | Error occurred |

		Available since E3.series TLB-Versions: 22.01
		"""
		dummy=0
		ret, ids = self._obj.GetInsideGraphIds(dummy, flags)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def SendToForeground(self) -> int:
		"""
		Brings the display of the graphic item to the foreground of the sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Graphic item send to foreground |
		| -1 | No project open |
		| -2 | No graphic item set |
		| -3 | Error accessing graphic item |
		| -4 | Error accessing sheet item |
		| -5 | Graphic item cannot be sent to the foreground |
		| -6 | No license available |
		| -7 | Graphic item is locked |

		Available since E3.series TLB-Versions: 24.22
		"""
		return self._obj.SendToForeground()

	def SendToBackground(self) -> int:
		"""
		Brings the display of the graphic item to the background of the sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Graphic item send to background |
		| -1 | No project open |
		| -2 | No graphic item set |
		| -3 | Error accessing graphic item |
		| -4 | Error accessing sheet item |
		| -5 | Graphic item cannot be sent to the background |
		| -6 | No license available |
		| -7 | Graphic item is locked |

		Available since E3.series TLB-Versions: 24.22
		"""
		return self._obj.SendToBackground()

# -------------------- IDbeModelInterface--------------------
class DbeModel:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeModel. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current model item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current model item identifier |
		| 0 | No model item or an error occurred |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a model as the current item.

		:param id: Unique value identifying a model item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current model item identifier |
		| 0 | No current model item or an error occurred |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the name of the model item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Model item name value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.GetName()

	def GetCharacteristic(self) -> str:
		"""
		Gets the characteristic of the model item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Model item characteristic value |
		| "&lt;Empty&gt;" | Model item has no characteristic or an error occurred |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.GetCharacteristic()

	def ImportDXF(self, filename:str, scale:float, x:float, y:float, rot:int, font:str, flags:int=0) -> int:
		"""
		Imports a DXF or DWG format file's content on to the model item sheet.

		:param filename: Path and name of the import file
		:type str: str
		:param scale: Resize scale to apply to the import file contents
		:type float: float
		:param x: Placement of the file contents on the x-axis of the model sheet
		:type float: float
		:param y: Placement of the file contents on the y-axis of the model sheet
		:type float: float
		:param rot: Not used
		:type int: int
		:param font: Font name of texts on the model sheet
		:type str: str
		:param flags: Flags bit values specifying how the file contents should be imported, defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new sheet |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -2 | filename import file does not exist |
		| -3 | rot value is invalid |
		| -4 | font value is invalid |
		| -5 | Import file content exceeds the sheet size |
		| -6 | Some manufacturing information specified by flag values could not be imported, Available since v2023-24.11 |
		| -7 | Error reading DXF file, Available since v2022-23.61 and v2023-24.31 |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.ImportDXF(filename, scale, x, y, rot, font, flags)

	def FitSpaceRequirement(self) -> int:
		"""
		Resizes the surrounding rectangle graphic item to fit around the content of each model view.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Fitting operation was executed |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.FitSpaceRequirement()

	def GetDXFSize(self, filename:str, font:str) -> tuple[int, float, float, float]:
		"""
		Gets the size and scale for the model sheet based on the file contents and font.

		:param filename: Path and name of the DXF or DWG file. Since v2022-23.53 and v2023-24.22, the path and name of a DGN file is also accepted
		:type str: str
		:param font: Name of font to apply
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | dx | float | Required width passed out of the function |
		| 2 | dy | float | Required height passed out of the function |
		| 3 | scale | float | Maximum possible scale for the sheet and. font. passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Size and maximum scale supplied |
		| 0 | Size and scale could not be determined |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -2 | filename is invalid |
		| -3 | font is invalid |
		| -4 | Error reading DXF file, Available since v2022-23.61 and v2023-24.31 |

		Available since E3.series TLB-Versions: 19.12
		"""
		dummy=0
		return self._obj.GetDXFSize(filename, font, dummy, dummy, dummy)

	def Save(self) -> int:
		"""
		Saves the model to the database.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Model is saved to the database |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -2 | Database is read-only |
		| -3 | Model could not be saved |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.Save()

	def Remove(self, save_changes:bool=False) -> int:
		"""
		Removes the model from the E³.DatabaseEditor project.

		:param save_changes: Indicates whether the model should be saved to the database before being removed from the project. If 1, the model is saved to the database. If 0, the model is not saved to the database., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Model is removed from the E³.DatabaseEditor project |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -2 | Database is read-only |
		| -3 | Model could not be removed |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.Remove(save_changes)

	def ActivateModelView(self, modelview:int) -> int:
		"""
		Creates or sets a model view to active.

		:param modelview: Model view to create or set to active
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Model view was created or set to active |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 19.12
		"""
		return self._obj.ActivateModelView(modelview)

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the pins belonging to the model item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of items in ids |
		| -1 | No model item set |
		| -4 | No E³.DatabaseEditor project open |

		Available since E3.series TLB-Versions: 20.00
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def PlaceSymbol(self, symname:str, version:str, x:float, y:float, rot:str, flags:int=0) -> int:
		"""
		Places a symbol on the model item sheet.

		:param symname: Name of the symbol to place
		:type str: str
		:param version: Version of the symbol to place. If version is invalid, the current version is assumed
		:type str: str
		:param x: Placement position on the x-axis
		:type float: float
		:param y: Placement position on the y-axis
		:type float: float
		:param rot: Rotation value in a counterclockwise direction
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol is placed |
		| -1 | No project open or no model item set |
		| -3 | Invalid symbol type |
		| -4 | Symbol not found in the database |
		| -5 | Insertion point lies outside of the sheet |
		| -6 | Symbol could not be placed |
		| -7 | Invalid rotation value |

		Available since E3.series TLB-Versions: 21.00, 20.11, 19.41
		"""
		return self._obj.PlaceSymbol(symname, version, x, y, rot, flags)

	def GetGraphicIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the graphics on the model sheet.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of graphic item identifiers passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of graphic item identifiers in ids |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -2 | No valid model sheet view |

		Available since E3.series TLB-Versions: 22.00, 21.01, 20.21
		"""
		dummy=0
		ret, ids = self._obj.GetGraphicIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def ImportStep(self, filename:str, flags:int=0) -> int:
		"""
		Imports a step file's content into the model item.

		:param filename: Path and name of the import file
		:type str: str
		:param flags: Specifies import process options., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | The file is imported |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -2 | filename STEP file does not exist |
		| -3 | flags value is invalid |
		| -4 | Model item's sheet is invalid |
		| -5 | No license for STEP Import available, No longer supported since v2021-22.12 |
		| -6 | 3D-STEP models cannot be imported |
		| -7 | 2D views could not be generated |
		| -9 | Import process aborted by user, Available since v2026-26.00 |
		| -10 | Import process aborted due to timing out, Available since v2026-26.00 |
		| -11 | Steps of the import process skipped by user, Available since v2026-26.00 |
		| -12 | Steps of the import process skipped due to timing out, Available since v2026-26.00 |
		| -13 | STEP Import functionality is disabled through the /noimportstep start-up parameter, Available since v2026-26.10, v2025-25.34 and v2023-24.53 |

		Available since E3.series TLB-Versions: 22.00
		"""
		return self._obj.ImportStep(filename, flags)

	def DisplayOverviewOfExistingViews(self, flags:int=0) -> int:
		"""
		Displays existing 2D views of a model  sheet.

		:param flags: View display flags., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Existing 2D views are displayed |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -2 | No valid view exists |
		| -3 | flags value is invalid |
		| -4 | Sheet is not a 2D view |
		| -5 | Error switching to 3D view, Available since v2023-24.20 |
		| -6 | Error switching to 2D view, Available since v2023-24.20 |
		| -7 | Error occurred, Available since v2023-24.20 |

		Available since E3.series TLB-Versions: 24.00
		"""
		return self._obj.DisplayOverviewOfExistingViews(flags)

	def ExportImage(self, format:str, version:int, file:str, dpi:int=0, compressionmode:int=0, clrdepth:int=24, flags:int=1) -> int:
		"""
		Exports the model sheet as an image.

		:param format: The image's format
		:type str: str
		:param version: Not used
		:type int: int
		:param file: Path and file name of the export image
		:type str: str
		:param dpi: Image resolution. 6, defaults to 0
		:type int: int
		:param compressionmode: Compression mode depending on. format., defaults to 0
		:type int: int
		:param clrdepth: Color bit depth of the export image. 4, defaults to 24
		:type int: int
		:param flags: Model export option flags., defaults to 1
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Image is exported |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -3 | flags value is invalid |
		| -4 | model sheet is invalid |
		| -7 | Error occurred |
		| -8 | Image format is invalid |
		| -9 | Image could not be exported |
		| -12 | Image dimensions including borders could not be calculated |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.ExportImage(format, version, file, dpi, compressionmode, clrdepth, flags)

	def GetDrawingArea(self, flags:int=1) -> tuple[int, float, float, float, float]:
		"""
		Gets the coordinate values of the model item's drawing area.

		:param flags: Area option flags., defaults to 1
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xmin | float | Lower left position on the x-axis passed out of the function |
		| 2 | ymin | float | Lower left position on the y-axis passed out of the function |
		| 3 | xmax | float | Upper right position on the x-axis passed out of the function |
		| 4 | ymax | float | Upper right position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Drawing area coordinates supplied |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -3 | flags value is invalid |
		| -4 | model sheet is invalid |
		| -12 | Image dimensions including borders could not be calculated |

		Available since E3.series TLB-Versions: 24.11
		"""
		dummy=0
		return self._obj.GetDrawingArea(dummy, dummy, dummy, dummy, flags)

	def ExportImageArea(self, format:str, version:int, file:str, xl:float, yl:float, xr:float, yr:float, width:int, height:int, clrdepth:int, gray:int, dpiX:int, dpiY:int, compressionmode:int, flags:int=0) -> int:
		"""
		Exports an area of the model sheet as an image.

		:param format: Image's format
		:type str: str
		:param version: Not used
		:type int: int
		:param file: Path and file name of the export image
		:type str: str
		:param xl: Position of the left border
		:type float: float
		:param yl: Position of the lower border
		:type float: float
		:param xr: Position of the right border
		:type float: float
		:param yr: Position of the upper border
		:type float: float
		:param width: Image width
		:type int: int
		:param height: Image height
		:type int: int
		:param clrdepth: Color bit depth of the image
		:type int: int
		:param gray: Specifies the color mode. If 1, the image is grayscale. If 0, the image is color
		:type int: int
		:param dpiX: Image resolution on the x-axis
		:type int: int
		:param dpiY: Image resolution on the y-axis
		:type int: int
		:param compressionmode: Compression mode depending on. format
		:type int: int
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Image is exported |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -3 | flags value is invalid |
		| -4 | model sheet is invalid |
		| -7 | Error occurred |
		| -8 | Image format is invalid |
		| -9 | Image could not be exported |
		| -10 | xl, yl, xr or yr contains an invalid value |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.ExportImageArea(format, version, file, xl, yl, xr, yr, width, height, clrdepth, gray, dpiX, dpiY, compressionmode, flags)

	def GetAttributeIds(self, attnam:str="", flags:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the model item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute identifiers are supplied if attnam is "<Empty>". The default string value is "<Empty>", defaults to ""
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of items in ids |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -3 | flags is invalid |

		Available since E3.series TLB-Versions: 24.11
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam, flags)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def AddAttributeValue(self, name:str, value:str, flags:int=0) -> int:
		"""
		Adds an attribute to the model item.

		:param name: Name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| -1 | No E³.DatabaseEditor project open or no model item set |
		| -2 | Attribute value could not be added |
		| -3 | flags is invalid |
		| -4 | value value is too long |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.AddAttributeValue(name, value, flags)

	def GetSlotIds(self, flags:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the slots belonging to the model item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of slots passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of items in ids |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags is invalid |

		Available since E3.series TLB-Versions: 24.11
		"""
		dummy=0
		ret, ids = self._obj.GetSlotIds(dummy, flags)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def DeleteModelView(self, flags:int) -> int:
		"""
		Deletes model views from the model item.

		:param flags: Flags bit values specifying which model views to delete
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Model view is deleted or model view does not exist |
		| -1 | No project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |
		| -4 | No views selected for deletion |
		| -5 | A view could not be deleted |
		| -6 | Front view cannot be deleted |

		Available since E3.series TLB-Versions: 24.31
		"""
		return self._obj.DeleteModelView(flags)

	def DeleteStepModel(self, flags:int=0) -> int:
		"""
		Deletes the model item's STEP model.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | STEP&#160;Model is deleted from model item |
		| -1 | No project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |
		| -4 | Model item has no STEP model |
		| -5 | STEP model cannot be deleted |

		Available since E3.series TLB-Versions: 24.31
		"""
		return self._obj.DeleteStepModel(flags)

	def GetMountingDescriptions(self, flags:int=0) -> tuple[int, tuple[str,...]]:
		"""
		Gets the mounting descriptions of the model item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | mountingdescriptions | tuple[str,...] | Array of mounting descriptions passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of  mounting descriptions in mountingdescriptions |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |

		Available since E3.series TLB-Versions: 25.23
		"""
		dummy=0
		ret, mountingdescriptions = self._obj.GetMountingDescriptions(dummy, flags)
		mountingdescriptions = mountingdescriptions[1:] if type(mountingdescriptions) == tuple and len(mountingdescriptions) > 0 else tuple()
		return ret, mountingdescriptions

	def SetMountingDescriptions(self, mountingdescriptions:list[str], flags:int=0) -> int:
		"""
		Sets the mounting descriptions of the model item.

		:param mountingdescriptions: Array of mounting descriptions passed into the function
		:type list[str]: list[str]
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New mounting descriptions are applied |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |

		Available since E3.series TLB-Versions: 25.23
		"""
		ret = self._obj.SetMountingDescriptions(mountingdescriptions, flags)
		return ret[0]

	def GetJustificationPoint(self, flags:int=0) -> tuple[int, float, float, float]:
		"""
		Gets the justification for slot points on the model item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Justification value of the x-axis |
		| 2 | y | float | Justification value of the y-axis |
		| 3 | z | float | Justification value of the z-axis |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Justification values are provided |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |

		Available since E3.series TLB-Versions: 25.23
		"""
		dummy=0
		return self._obj.GetJustificationPoint(dummy, dummy, dummy, flags)

	def GetJustificationLine(self, flags:int=0) -> tuple[int, float, float]:
		"""
		Gets the justification for slot lines on the model item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | y | float | Justification value of the y-axis |
		| 2 | z | float | Justification value of the z-axis |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Justification values are provided |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |

		Available since E3.series TLB-Versions: 25.23
		"""
		dummy=0
		return self._obj.GetJustificationLine(dummy, dummy, flags)

	def GetJustificationArea(self, flags:int=0) -> tuple[int, float]:
		"""
		Gets the justification for slot areas on the model item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | z | float | Justification value of the z-axis |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Justification value is provided |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |

		Available since E3.series TLB-Versions: 25.23
		"""
		dummy=0
		return self._obj.GetJustificationArea(dummy, flags)

	def SetJustificationPoint(self, x:float, y:float, z:float, flags:int=0) -> int:
		"""
		Sets the justification for point slots on the model item.

		:param x: Justification value of the x-axis
		:type float: float
		:param y: Justification value of the y-axis
		:type float: float
		:param z: Justification value of the z-axis
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New justification value is applied |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |
		| -4 | x, y or z value not in range |

		Available since E3.series TLB-Versions: 25.23
		"""
		return self._obj.SetJustificationPoint(x, y, z, flags)

	def SetJustificationLine(self, y:float, z:float, flags:int=0) -> int:
		"""
		Sets the justification for slot lines on the model item.

		:param y: Justification value of the y-axis
		:type float: float
		:param z: Justification value of the z-axis
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New justification value is applied |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |
		| -4 | y or z value not in range |

		Available since E3.series TLB-Versions: 25.23
		"""
		return self._obj.SetJustificationLine(y, z, flags)

	def SetJustificationArea(self, z:float, flags:int=0) -> int:
		"""
		Sets the justification for slot areas on the model item.

		:param z: Justification value of the z-axis
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New justification value is applied |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags value is invalid |
		| -4 | z value not in range |

		Available since E3.series TLB-Versions: 25.23
		"""
		return self._obj.SetJustificationArea(z, flags)

	def GetStepTransformation(self, flags:int=0) -> tuple[int, tuple[float,...]]:
		"""
		Gets the transformation matrix from the  model item's STEP model.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | matrix | tuple[float,...] | Array of STEP model transformation matrix values |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Transformation matrix supplied |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No model item set |
		| -3 | flags is invalid |
		| -4 | Model item has no STEP model |

		Available since E3.series TLB-Versions: 25.31
		"""
		dummy=0
		ret, matrix = self._obj.GetStepTransformation(dummy, flags)
		matrix = matrix[1:] if type(matrix) == tuple and len(matrix) > 0 else tuple()
		return ret, matrix

# -------------------- IDbeModelPinInterface--------------------
class DbeModelPin:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeModelPin. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current model pin item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current model pin item identifier |
		| 0 | No model pin item or an error occurred |

		Available since E3.series TLB-Versions: 20.00
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a model pin as the current item.

		:param id: Unique value identifying a model pin item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current model pin item identifier |
		| 0 | No current model pin item or an error occurred |

		Available since E3.series TLB-Versions: 20.00
		"""
		return self._obj.SetId(id)

	def GetCrimpingRules(self) -> tuple[int, memoryview]:
		"""
		Gets the model pin item's crimping rules.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | rules | memoryview | Array of crimping rules passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of items in rules |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 20.00
		"""
		dummy=0
		return self._obj.GetCrimpingRules(dummy)

	def SetCrimpingRules(self, rules:memoryview) -> int:
		"""
		Sets the model pin item's crimping rules.

		:param rules: Array of crimping rules passed into the function
		:type memoryview: memoryview

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Alignment value set |
		| -1 | No physical pin set |
		| -2 | Invalid Granular Design object! |
		| -3 | Invalid rules array |

		Available since E3.series TLB-Versions: 20.00
		"""
		ret = self._obj.SetCrimpingRules(rules)
		return ret[0]

	def GetRoutingOffset(self, flags:int=0) -> tuple[int, float, float, float]:
		"""
		Gets the current routing offset values of the model pin.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Routing offset on the x-axis passed out of the function |
		| 2 | y | float | Routing offset on the y-axis passed out of the function |
		| 3 | z | float | Routing offset on the z-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Routing offset values supplied |
		| -1 | No project open or no pin item set |
		| -2 | flags is invalid |

		Available since E3.series TLB-Versions: 25.00
		"""
		dummy=0
		return self._obj.GetRoutingOffset(dummy, dummy, dummy, flags)

	def SetRoutingOffset(self, x:float, y:float, z:float, flags:int=0) -> int:
		"""
		Sets the current routing offset values of the model pin.

		:param x: Routing offset on the x-axis
		:type float: float
		:param y: Routing offset on the y-axis
		:type float: float
		:param z: Routing offset on the z-axis
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Routing offset values applied |
		| -1 | No project open or no pin item set |
		| -2 | flags is invalid |

		Available since E3.series TLB-Versions: 25.00
		"""
		return self._obj.SetRoutingOffset(x, y, z, flags)

	def GetPinProperties(self, keyList:str="", flags:int=0) -> str:
		"""
		Gets the model pin properties and their values.

		:param keyList: Comma delimited list of properties to retrieve. If "<Empty>", all properties are retrieved. Default value is "<Empty>", defaults to ""
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Properties supplied in a JSON object containing key-value pairs of pin properties and property values |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 25.31
		"""
		return self._obj.GetPinProperties(keyList, flags)

	def SetPinProperties(self, jsonInput:str, flags:int=0) -> int:
		"""
		Sets the model pin property values.

		:param jsonInput: JSON object containing key-value pairs of pin properties and property values to be changed
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Model pin properties and values applied |
		| -1 | No pin item set |
		| -2 | No project open |
		| -3 | flags is invalid |
		| -4 | jsonInput is invalid |
		| -5 | jsonInput contains invalid keys |
		| -6 | jsonInput contains invalid values |
		| -7 | Error occurred |

		Available since E3.series TLB-Versions: 25.31
		"""
		return self._obj.SetPinProperties(jsonInput, flags)

# -------------------- IDbeNodeInterface--------------------
class DbeNode:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeNode. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current node item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current node item identifier |
		| 0 | No node item or an error occurred |

		Available since E3.series TLB-Versions: 21.21
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a node as the current item.

		:param id: Unique value identifying a node item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current node item identifier |
		| 0 | No current node item or an error occurred |

		Available since E3.series TLB-Versions: 21.21
		"""
		return self._obj.SetId(id)

	def GetDirection(self, flags:int= 0 ) -> int:
		"""
		Gets the direction of the node item.

		:param flags: Not used., defaults to  0 
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Node direction |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No node item set |
		| -3 | flags value is invalid |

		Available since E3.series TLB-Versions: 21.21
		"""
		return self._obj.GetDirection(flags)

	def IsBusPin(self) -> int:
		"""
		Indicates whether the node item is a bus pin.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Node is a bus pin |
		| 0 | Node is not a bus pin |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No node item set |

		Available since E3.series TLB-Versions: 21.21
		"""
		return self._obj.IsBusPin()

	def HasPassWires(self) -> int:
		"""
		Indicates whether the node item has pass wires.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Node has pass wires |
		| 0 | Node does not have pass wires |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No node item set |

		Available since E3.series TLB-Versions: 21.21
		"""
		return self._obj.HasPassWires()

	def GetTextIds(self, texttype:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the text items assigned to the node item.

		:param texttype: Text type filter. Only text identifiers with this Text Type value will be supplied. All text identifiers are supplied if texttype is set to 0. The default string value is  0, defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of text items passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Number of items in ids |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No node item set |

		Available since E3.series TLB-Versions: 21.21
		"""
		dummy=0
		ret, ids = self._obj.GetTextIds(dummy, texttype)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetPosition(self) -> tuple[int, float, float]:
		"""
		Gets the node item's position.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Position on the x-axis passed out of the function |
		| 2 | y | float | Position on the y-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Position values are supplied |
		| -1 | No project open |
		| -2 | No node item set |

		Available since E3.series TLB-Versions: 21.21
		"""
		dummy=0
		return self._obj.GetPosition(dummy, dummy)

	def IsBusbarPin(self) -> int:
		"""
		Indicates whether the node item is a busbar pin.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Node is a busbar pin |
		| 0 | Node is not a busbar pin |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | No node item set |

		Available since E3.series TLB-Versions: 23.00
		"""
		return self._obj.IsBusbarPin()

# -------------------- IDbeSlotInterface--------------------
class DbeSlot:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeSlot. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current slot item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current slot item identifier |
		| 0 | No slot item or an error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a slot as the current item.

		:param id: Unique value identifying a slot item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current slot item identifier |
		| 0 | No current slot item or an error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.SetId(id)

	def GetDirection(self) -> tuple[int, float, float, float]:
		"""
		Gets the direction of the slot item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Direction value of the x-axis |
		| 2 | y | float | Direction value of the y-axis |
		| 3 | z | float | Direction value of the z-axis |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Direction values are provided |
		| -1 | No project open or slot item set |
		| -2 | Error accessing slot data |

		Available since E3.series TLB-Versions: 24.11
		"""
		dummy=0
		return self._obj.GetDirection(dummy, dummy, dummy)

	def GetFixType(self) -> int:
		"""
		Gets a value representing the fix type of the slot item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Error occurred |
		| 1..4 | Fix type value |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.GetFixType()

	def GetMountType(self) -> str:
		"""
		Gets the description of the slot item.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Slot item description value |
		| "&lt;Empty&gt;" | Slot item has no description or an error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.GetMountType()

	def GetPosition(self, point:int) -> tuple[int, float, float, float]:
		"""
		Gets the position of the slot item.

		:param point: Indicates which position to supply. If 1, the first position is supplied. If 2, the second position is supplied
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Position value of the x-axis passed out of the function |
		| 2 | y | float | Position value of the y-axis passed out of the function |
		| 3 | z | float | Position value of the z-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Position values are supplied |
		| -1 | No project open or slot item set |
		| -2 | Error accessing slot data |
		| -3 | point is invalid |
		| -4 | Slot area is not rectangular, Available since v2023-24.21 |

		Available since E3.series TLB-Versions: 24.11
		"""
		dummy=0
		return self._obj.GetPosition(point, dummy, dummy, dummy)

	def GetRotation(self) -> tuple[int, float]:
		"""
		Gets the rotation to be adopted by the device being mounted on the slot item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | angle | float | Rotation value in degrees |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Rotation value is supplied |
		| -1 | No project open or slot item set |
		| -2 | Error accessing slot data |
		| -3 | Area slots have no rotation value |
		| -4 | Error determining the slot rotation |

		Available since E3.series TLB-Versions: 24.11
		"""
		dummy=0
		return self._obj.GetRotation(dummy)

	def GetSlotName(self, flags:int=0) -> str:
		"""
		Gets the display name of the slot item.

		:param flags: Flag bit values specifying how the display name is supplied., defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Slot item display name value |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.GetSlotName(flags)

	def SetSlotName(self, name:str, flags:int=0) -> int:
		"""
		Sets the display name of the slot item.

		:param name: New slot display name
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Display name value applied |
		| -1 | No project open or slot item set |
		| -2 | Error accessing slot data |
		| -3 | name is invalid |
		| -4 | flags is invalid |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.SetSlotName(name, flags)

	def GetAreaPolygon(self, flags:int=0) -> tuple[int, tuple[float,...], tuple[float,...], tuple[float,...]]:
		"""
		Gets the polygon positions of the slot's area.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xarr | tuple[float,...] | Array of position values on the x-axis passed out of the function |
		| 2 | yarr | tuple[float,...] | Array of position values on the y-axis passed out of the function |
		| 3 | zarr | tuple[float,...] | Array of position values on the z-axis passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;= 0 | Position values are supplied |
		| -1 | No project open or slot item set |
		| -2 | Error accessing slot data |
		| -3 | Slot is not an area type slot |
		| -4 | flags is invalid |

		Available since E3.series TLB-Versions: 24.21
		"""
		dummy=0
		ret, xarr, yarr, zarr = self._obj.GetAreaPolygon(dummy, dummy, dummy, flags)
		xarr = xarr[1:] if type(xarr) == tuple and len(xarr) > 0 else tuple()
		yarr = yarr[1:] if type(yarr) == tuple and len(yarr) > 0 else tuple()
		zarr = zarr[1:] if type(zarr) == tuple and len(zarr) > 0 else tuple()
		return ret, xarr, yarr, zarr

	def CreatePoint(self, modelid:int, xpos:float, ypos:float, zpos:float, xdir:float, ydir:float, zdir:float, rotation:float, description:str, name:str, flags:int=0) -> int:
		"""
		Creates a point slot on a model.

		:param modelid: Identifier of the new slot item
		:type int: int
		:param xpos: Position value on the x-axis
		:type float: float
		:param ypos: Position value on the y-axis
		:type float: float
		:param zpos: Position value on the z-axis
		:type float: float
		:param xdir: Slot direction on the x-axis
		:type float: float
		:param ydir: Slot direction on the y-axis
		:type float: float
		:param zdir: Slot direction on the z-axis
		:type float: float
		:param rotation: Rotation of the slot in degrees
		:type float: float
		:param description: Description of the slot
		:type str: str
		:param name: Name of the slot
		:type str: str
		:param flags: Flags for creating the new slot., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new slot item |
		| -1 | modelid is invalid |
		| -2 | flags is invalid |
		| -3 | Direction defined by xdir, ydir and zdir is invalid |
		| -4 | description is an empty string |
		| -7 | Error creating slot |

		Available since E3.series TLB-Versions: 25.12
		"""
		return self._obj.CreatePoint(modelid, xpos, ypos, zpos, xdir, ydir, zdir, rotation, description, name, flags)

	def CreateLine(self, modelid:int, xpos:float, ypos:float, zpos:float, xdir:float, ydir:float, zdir:float, length:float, width:float, rotation:float, description:str, name:str, flags:int=0) -> int:
		"""
		Creates a line slot on a model.

		:param modelid: Item identifier of the model
		:type int: int
		:param xpos: Position value on the x-axis
		:type float: float
		:param ypos: Position value on the y-axis
		:type float: float
		:param zpos: Position value on the z-axis
		:type float: float
		:param xdir: Slot direction on the x-axis
		:type float: float
		:param ydir: Slot direction on the y-axis
		:type float: float
		:param zdir: Slot direction on the z-axis
		:type float: float
		:param length: Length of the slot
		:type float: float
		:param width: Width of the slot
		:type float: float
		:param rotation: Rotation of the slot in degrees
		:type float: float
		:param description: Description of the slot
		:type str: str
		:param name: Name of the slot
		:type str: str
		:param flags: Flags for creating the new slot., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new slot item |
		| -1 | modelid is invalid |
		| -2 | flags is invalid |
		| -3 | Direction defined by xdir, ydir and zdir is invalid |
		| -4 | description is an empty string |
		| -5 | length is invalid |
		| -7 | Error creating slot |

		Available since E3.series TLB-Versions: 25.12
		"""
		return self._obj.CreateLine(modelid, xpos, ypos, zpos, xdir, ydir, zdir, length, width, rotation, description, name, flags)

	def CreateAreaRectangle(self, modelid:int, xpos:float, ypos:float, zpos:float, xdir:float, ydir:float, zdir:float, length:float, width:float, description:str, name:str, flags:int=0) -> int:
		"""
		Creates a rectangle shaped area slot on a model.

		:param modelid: Item identifier of the model
		:type int: int
		:param xpos: Position value on the x-axis
		:type float: float
		:param ypos: Position value on the y-axis
		:type float: float
		:param zpos: Position value on the z-axis
		:type float: float
		:param xdir: Slot direction on the x-axis
		:type float: float
		:param ydir: Slot direction on the y-axis
		:type float: float
		:param zdir: Slot direction on the z-axis
		:type float: float
		:param length: Length of the slot
		:type float: float
		:param width: Width of the slot
		:type float: float
		:param description: Description of the slot
		:type str: str
		:param name: Name of the slot
		:type str: str
		:param flags: Flags for creating the new slot., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new slot item |
		| -1 | modelid is invalid |
		| -2 | flags is invalid |
		| -3 | Direction defined by xdir, ydir and zdir is invalid |
		| -4 | description is an empty string |
		| -5 | length is invalid |
		| -6 | width is invalid |
		| -7 | Error creating slot |

		Available since E3.series TLB-Versions: 25.12
		"""
		return self._obj.CreateAreaRectangle(modelid, xpos, ypos, zpos, xdir, ydir, zdir, length, width, description, name, flags)

	def CreateAreaPolygon(self, modelid:int, xarr:list[float], yarr:list[float], zarr:list[float], xdir:float, ydir:float, zdir:float, description:str, name:str, flags:int=0) -> int:
		"""
		Creates a polygon shaped area slot on a model.

		:param modelid: Item identifier of the model
		:type int: int
		:param xarr: Array of position values on the x-axis
		:type list[float]: list[float]
		:param yarr: Array of position values on the y-axis
		:type list[float]: list[float]
		:param zarr: Array of position values on the z-axis
		:type list[float]: list[float]
		:param xdir: Slot direction on the x-axis
		:type float: float
		:param ydir: Slot direction on the y-axis
		:type float: float
		:param zdir: Slot direction on the z-axis
		:type float: float
		:param description: Description of the slot
		:type str: str
		:param name: Name of the slot
		:type str: str
		:param flags: Flags for creating the new slot., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new slot item |
		| -1 | modelid is invalid |
		| -2 | flags is invalid |
		| -3 | Direction defined by xdir, ydir and zdir is invalid |
		| -4 | xarr, yarr and zarr arrays are different sizes |
		| -5 | xarr, yarr and zarr arrays are too small |
		| -6 | Polygon defined in xarr, yarr and zarr is not closed |
		| -7 | Polygon shaped defined in xarr, yarr and zarr is non-planar |
		| -8 | Polygon shape defined in xarr, yarr and zarr is invalid |
		| -9 | Error creating slot |

		Available since E3.series TLB-Versions: 25.12
		"""
		xarr = [0.] + xarr
		yarr = [0.] + yarr
		zarr = [0.] + zarr
		return self._obj.CreateAreaPolygon(modelid, xarr, yarr, zarr, xdir, ydir, zdir, description, name, flags)

	def Delete(self, flags:int=0) -> int:
		"""
		Deletes the slot item from the model.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Slot was deleted |
		| -1 | No slot item set |
		| -2 | flags is invalid |
		| -3 | model is a mounting rail or cable duct |
		| -4 | Error occurred |

		Available since E3.series TLB-Versions: 25.12
		"""
		return self._obj.Delete(flags)

	def SetRotation(self, rotation:float, flags:int=0) -> int:
		"""
		Sets the rotation to be adopted by the device being mounted on the slot item.

		:param rotation: Rotation value in degrees
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Rotation value is applied |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |
		| -3 | Area slots have no rotation value |
		| -4 | Line slots can only have rotation values of 0.0, +180.0 or -180.0 relative to the current rotation |
		| -5 | Error occurred |

		Available since E3.series TLB-Versions: 26.00
		"""
		return self._obj.SetRotation(rotation, flags)

	def GetDirectionRotation(self, flags:int=0) -> tuple[int, float, float, float]:
		"""
		Gets the direction rotation values of the slot item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | xAxisRotation | float | Direction rotation value around the x-axis in degrees |
		| 2 | yAxisRotation | float | Direction rotation value around the y-axis  in degrees |
		| 3 | zAxisRotation | float | Direction rotation value around the z-axis  in degrees |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Direction rotation values are supplied |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |

		Available since E3.series TLB-Versions: 26.00
		"""
		dummy=0
		return self._obj.GetDirectionRotation(dummy, dummy, dummy, flags)

	def SetDirectionRotation(self, xAxisRotation:float, yAxisRotation:float, zAxisRotation:float, flags:int=0) -> int:
		"""
		Sets the direction rotation values of the slot item.

		:param xAxisRotation: Direction rotation value around the x-axis in degrees
		:type float: float
		:param yAxisRotation: Direction rotation value around the y-axis  in degrees
		:type float: float
		:param zAxisRotation: Direction rotation value around the z-axis  in degrees
		:type float: float
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Direction rotation values are applied |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |
		| -3 | zAxisRotation must be 0.0 for front and back slot directions |
		| -4 | xAxisRotation must be 0.0 for left and right slot directions |
		| -5 | yAxisRotation must be 0.0 for top and bottom slot directions |
		| -6 | Slot direction cannot be changed |
		| -7 | Error occurred |

		Available since E3.series TLB-Versions: 26.00
		"""
		return self._obj.SetDirectionRotation(xAxisRotation, yAxisRotation, zAxisRotation, flags)

	def GetLength(self, flags:int= 0 ) -> tuple[int, float]:
		"""
		Gets the length of the slot item.

		:param flags: Not used., defaults to  0 
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | length | float | Length value in the measurement unit of the model |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Length value is supplied |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |
		| -3 | Error accessing slot data |
		| -4 | Model has an invalid unit of measurement |
		| -5 | Slot item is a point slot |
		| -6 | Error occurred in conversion between internal and model measurement units |

		Available since E3.series TLB-Versions: 26.01
		"""
		dummy=0
		return self._obj.GetLength(dummy, flags)

	def SetLength(self, length:float, flags:int= 0 ) -> int:
		"""
		Sets the length of the slot item.

		:param length: Length value in the measurement unit of the model
		:type float: float
		:param flags: Not used., defaults to  0 
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Length value is applied |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |
		| -3 | Error accessing slot data |
		| -4 | Model has an invalid unit of measurement |
		| -5 | Slot item is a point slot |
		| -6 | Memory allocation error occurred |
		| -7 | Error occurred in conversion between internal and model measurement units |

		Available since E3.series TLB-Versions: 26.01
		"""
		return self._obj.SetLength(length, flags)

	def GetWidth(self, flags:int= 0 ) -> tuple[int, float]:
		"""
		Gets the width of the slot item.

		:param flags: Not used., defaults to  0 
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | width | float | Width value in the measurement unit of the model |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Width value is supplied |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |
		| -3 | Error accessing slot data |
		| -4 | Model has an invalid unit of measurement |
		| -5 | Slot item is a point slot |
		| -6 | Error occurred in conversion between internal and model measurement units |

		Available since E3.series TLB-Versions: 26.01
		"""
		dummy=0
		return self._obj.GetWidth(dummy, flags)

	def SetWidth(self, width:float, flags:int= 0 ) -> int:
		"""
		Sets the width of the slot item.

		:param width: Width value in the measurement unit of the model
		:type float: float
		:param flags: Not used., defaults to  0 
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Width value is applied |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |
		| -3 | Error accessing slot data |
		| -4 | Model has an invalid unit of measurement |
		| -5 | Slot item is a point slot |
		| -6 | Memory allocation error occurred |
		| -7 | Error occurred in conversion between internal and model measurement units |

		Available since E3.series TLB-Versions: 26.01
		"""
		return self._obj.SetWidth(width, flags)

	def GetDirectionEx(self, flags:int=0) -> tuple[int, float, float, float, int]:
		"""
		Gets the direction and facing side of the slot item.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Direction value of the x-axis |
		| 2 | y | float | Direction value of the y-axis |
		| 3 | z | float | Direction value of the z-axis |
		| 4 | side | int | Side from where the slot faces |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Direction values are provided |
		| -1 | No project open or slot item set |
		| -2 | flags value is invalid |

		Available since E3.series TLB-Versions: 26.01
		"""
		dummy=0
		return self._obj.GetDirectionEx(dummy, dummy, dummy, dummy, flags)

# -------------------- IDbeJobInterface--------------------
class DbeJob:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize DbeJob. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def CreateDbeComponentObject(self) -> DbeComponent:
		"""
		Creates  an e3DbeComponent object.

		:return:
		Type: DbeComponent

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeComponent(self._obj.CreateDbeComponentObject())

	def CreateDbeAttributeObject(self) -> DbeAttribute:
		"""
		Creates  an e3DbeAttribute object.

		:return:
		Type: DbeAttribute

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeAttribute(self._obj.CreateDbeAttributeObject())

	def CreateDbeSymbolObject(self) -> DbeSymbol:
		"""
		Creates  an e3DbeSymbol object.

		:return:
		Type: DbeSymbol

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeSymbol(self._obj.CreateDbeSymbolObject())

	def CreateDbeTextObject(self) -> DbeText:
		"""
		Creates  an e3DbeText object.

		:return:
		Type: DbeText

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeText(self._obj.CreateDbeTextObject())

	def CreateDbeGraphObject(self) -> DbeGraph:
		"""
		The Object supplied allows access to the functionality of the e3DbeGraph object.

		:return:
		Type: DbeGraph

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeGraph(self._obj.CreateDbeGraphObject())

	def CreateDbeModelObject(self) -> DbeModel:
		"""
		Creates  an e3DbeModel object.

		:return:
		Type: DbeModel

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeModel(self._obj.CreateDbeModelObject())

	def CreateDbeModelPinObject(self) -> DbeModelPin:
		"""
		Creates  an e3DbeModelPin object.

		:return:
		Type: DbeModelPin

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeModelPin(self._obj.CreateDbeModelPinObject())

	def CreateDbeNodeObject(self) -> DbeNode:
		"""
		Creates  an e3DbeNode object.

		:return:
		Type: DbeNode

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeNode(self._obj.CreateDbeNodeObject())

	def CreateDbeSlotObject(self) -> DbeSlot:
		"""
		Creates  an e3DbeSlot object.

		:return:
		Type: DbeSlot

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeSlot(self._obj.CreateDbeSlotObject())

	def GetId(self) -> int:
		"""
		Gets the identifier of the current project item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current project item identifier |
		| 0 | No project open |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetId()

	def Save(self, flags:int=0) -> int:
		"""
		Saves the E³.DatabaseEditor project to the current path and file name.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Project is saved |
		| -1 | No project open |
		| -2 | flags value is invalid |
		| -3 | Project is read-only |
		| -4 | Error occurred |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.Save(flags)

	def SaveAs(self, name:str, compressed:bool=False, flags:int=0) -> int:
		"""
		Saves the E³.DatabaseEditor project with a new path and file name.

		:param name: File path and name of project
		:type str: str
		:param compressed: Indicates whether the project is compressed. If 1, the project is saved as a compressed file. If 0, the project is saved as an uncompressed file., defaults to False
		:type bool: bool
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Project is saved |
		| -1 | No project open |
		| -2 | flags value is invalid |
		| -3 | Symbol, model or component is open in edit mode |
		| -4 | Error occurred |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.SaveAs(name, compressed, flags)

	def NewModel(self, name:str, baseName:str, flags:int=0) -> int:
		"""
		Creates a model in the E³.DatabaseEditor.

		:param name: Name of the new model. Characteristic information can also be assigned using the format "<ModelName>¶<ModelCharacteristic>"
		:type str: str
		:param baseName: Name of the model to be used as a template. If baseName value is empty, no template is used
		:type str: str
		:param flags: Flags for creating the new model., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New model is created in the E³.DatabaseEditor |
		| -1 | flags is invalid |
		| -2 | baseName model not found |
		| -3 | Model is locked |
		| -4 | Model with name already exists |
		| -7 | name exceeds maximum length |
		| -8 | name contains invalid characters |
		| -9 | No open project |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.NewModel(name, baseName, flags)

	def EditModel(self, name:str, flags:int=0) -> int:
		"""
		Opens the model for editing in the E³.DatabaseEditor.

		:param name: Name of the model to edit
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Opens the model for editing in the E³.DatabaseEditor |
		| -1 | No open project |
		| -2 | Model not found |
		| -3 | Model is locked |
		| -4 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.EditModel(name, flags)

	def DeleteModel(self, name:str, flags:int=0) -> int:
		"""
		Deletes the model from the database.

		:param name: Name of the model to delete
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Model is deleted from the database |
		| -1 | Error occurred |
		| -2 | Model not found |
		| -3 | Model is locked |
		| -5 | Model cannot be deleted |
		| -6 | No project open |
		| -7 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.DeleteModel(name, flags)

	def GetModelIds(self, flags:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the E³.DatabaseEditor project models.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of models in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No models are found |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		dummy=0
		ret, ids = self._obj.GetModelIds(dummy, flags)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetActiveModelId(self) -> int:
		"""
		Gets the model identifier of the current open sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current model identifier |
		| -1 | No E³.DatabaseEditor project open or sheet open |
		| -2 | Active sheet is not a model |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetActiveModelId()

	def NewSymbol(self, name:str, version:str, baseName:str, baseVersion:str, flags:int=0) -> int:
		"""
		Creates a symbol in the E³.DatabaseEditor.

		:param name: Name of the new symbol. Characteristic information can also be assigned using the format "<SymbolName>¶<SymbolCharacteristic>"
		:type str: str
		:param version: Version of the new symbol. If version value is empty, the version is set to 1
		:type str: str
		:param baseName: Name of the symbol to be used as a template. If baseName value is empty, no template is used
		:type str: str
		:param baseVersion: Version of the symbol to be used as a template. If baseVersion value is empty, the version is set to 1
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New symbol is created in the E³.DatabaseEditor |
		| -1 | No open project |
		| -2 | baseName symbol not found |
		| -3 | Symbol is locked |
		| -4 | Symbol with name and version already exists |
		| -5 | version is invalid |
		| -6 | baseVersion is invalid |
		| -7 | name exceeds maximum length |
		| -8 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.NewSymbol(name, version, baseName, baseVersion, flags)

	def EditSymbol(self, name:str, version:str, flags:int=0) -> int:
		"""
		Opens the symbol for editing in the E³.DatabaseEditor.

		:param name: Name of the symbol to edit
		:type str: str
		:param version: Version of the symbol to edit
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Opens the symbol for editing in the E³.DatabaseEditor |
		| -1 | No open project |
		| -2 | Symbol not found |
		| -3 | Symbol is locked |
		| -5 | version is too long |
		| -6 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.EditSymbol(name, version, flags)

	def DeleteSymbol(self, name:str, version:str, flags:int=0) -> int:
		"""
		Deletes the symbol from the database.

		:param name: Name of the symbol to delete
		:type str: str
		:param version: Version of the symbol to delete
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Symbol is deleted from the database |
		| -1 | Error occurred |
		| -2 | Symbol not found |
		| -3 | Symbol is locked |
		| -4 | Symbol cannot be deleted |
		| -5 | version is invalid |
		| -6 | No project open |
		| -7 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.DeleteSymbol(name, version, flags)

	def GetSymbolIds(self, flags:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the E³.DatabaseEditor project symbols.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbols in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No symbols are found |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy, flags)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetActiveSymbolId(self) -> int:
		"""
		Gets the symbol identifier of the current open sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current symbol identifier |
		| -1 | No E³.DatabaseEditor project open or sheet open |
		| -2 | Active sheet is not a symbol |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetActiveSymbolId()

	def NewComponent(self, name:str, version:str, baseName:str, baseVersion:str, flags:int=0) -> int:
		"""
		Adds a component to the database and starts the Component Wizard in the E³.DatabaseEditor.

		:param name: Name of the new component
		:type str: str
		:param version: Version of the new component. If version value is empty, the version is set to "1   "
		:type str: str
		:param baseName: Name of the component to be used as a template. If baseName value is empty, no template is used
		:type str: str
		:param baseVersion: Version of the component to be used as a template. If baseVersion value is empty, the version is set to "1   "
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Starts Component Wizard in the E³.DatabaseEditor |
		| -1 | No open project |
		| -2 | baseName component not found |
		| -3 | Component is locked |
		| -4 | Component with name and version already exists |
		| -5 | version is invalid |
		| -6 | baseVersion is invalid |
		| -7 | name exceeds maximum length |
		| -8 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.NewComponent(name, version, baseName, baseVersion, flags)

	def EditComponent(self, name:str, version:str, flags:int=0) -> int:
		"""
		Opens the component for editing in the E³.DatabaseEditor.

		:param name: Name of the component to edit
		:type str: str
		:param version: Version of the component to edit
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Opens Component Properties dialog in the E³.DatabaseEditor |
		| -1 | No open project |
		| -2 | Component not found |
		| -3 | Component is locked |
		| -5 | version is too long |
		| -6 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.EditComponent(name, version, flags)

	def DeleteComponent(self, name:str, version:str, flags:int=0) -> int:
		"""
		Deletes the component from the database.

		:param name: Name of the component to delete
		:type str: str
		:param version: Version of the component to delete
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Component is deleted from the database |
		| -1 | Error occurred |
		| -2 | Component not found |
		| -3 | Component is locked |
		| -5 | version is invalid |
		| -6 | No open project |
		| -7 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.DeleteComponent(name, version, flags)

	def GetComponentIds(self, flags:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifiers of the E³.DatabaseEditor project components.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of components in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No components are found or an error occurred |
		| -1 | No E³.DatabaseEditor project open |
		| -2 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		dummy=0
		ret, ids = self._obj.GetComponentIds(dummy, flags)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetActiveComponentId(self) -> int:
		"""
		Gets the component identifier of the current open sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current component identifier |
		| -1 | No E³.DatabaseEditor project open or sheet open |
		| -2 | Active sheet is not a component |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetActiveComponentId()

	def Close(self, flags:int=0) -> int:
		"""
		Closes the E³.DatabaseEditor project.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 1 | Error occurred |
		| 1 | Project is closed or no project was open |
		| -1 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.Close(flags)

	def New(self, name:str, flags:int=0) -> int:
		"""
		Creates a new E³.DatabaseEditor project.

		:param name: Path and  name of the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project file
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | E³.DatabaseEditor project was created |
		| 0 | Error occurred |
		| -1 | flags value is invalid |
		| -2 | Project already open |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.New(name, flags)

	def Create(self, name:str, flags:int=0) -> int:
		"""
		Creates a new E³.DatabaseEditor project.

		:param name: Path and name of the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project file
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | E³.DatabaseEditor project was created |
		| 0 | Error occurred |
		| -1 | flags value is invalid |
		| -2 | A project is already open |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.Create(name, flags)

	def Open(self, name:str, flags:int=0) -> int:
		"""
		Opens an E³.DatabaseEditor project or connects to an already open E³.DatabaseEditor project.

		:param name: Path and file name of an. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the opened E³.DatabaseEditor project |
		| 0 | Error occurred |
		| -1 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.Open(name, flags)

	def GetPath(self) -> str:
		"""
		Gets the system path of the project.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Path&gt;" | Project system path |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetPath()

	def GetName(self) -> str:
		"""
		Gets the name of the E³.DatabaseEditor project.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Project name |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetName()

	def GetType(self) -> str:
		"""
		Gets the project's file name extension.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| ".&lt;File Extension&gt;" | Project file name extension |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetType()

	def GetSettingValue(self, name:str) -> typing.Union[str,int]:
		"""
		Gets a setting value.

		:param name: Unique name identifying the setting value to retrieve. See Setting Value Names for possible values
		:type str: str

		:return:
		Type: typing.Union[str,int]

		| Return Value | Description |
		|:-----:|:-----|
		| empty | Error occurred |
		| != empty | Setting value supplied |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetSettingValue(name)

	def SetSettingValue(self, name:str, value:typing.Union[str,int]) -> typing.Union[str,int]:
		"""
		Sets a setting value.

		:param name: Unique name identifying the setting value to modify. See Setting Value Names for possible values
		:type str: str
		:param value: Setting value to apply
		:type typing.Union[str,int]: typing.Union[str,int]

		:return:
		Type: typing.Union[str,int]

		| Return Value | Description |
		|:-----:|:-----|
		| empty | Error occurred |
		| != empty | Previous setting value |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.SetSettingValue(name, value)

	def GetOutbarText(self, index:int) -> tuple[int, tuple[str,...]]:
		"""
		Gets the lines of text displayed in the E³.DatabaseEditor Messages window.

		:param index: Start position of text lines to retrieve
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | lst | tuple[str,...] | Array of displayed text lines passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of displayed text lines |
		| 0 | No displayed text lines or an error occurred |

		Available since E3.series TLB-Versions: 25.32
		"""
		dummy=0
		ret, lst = self._obj.GetOutbarText(index, dummy)
		lst = lst[1:] if type(lst) == tuple and len(lst) > 0 else tuple()
		return ret, lst

	def GetResultText(self, index:int) -> tuple[int, tuple[str,...]]:
		"""
		Gets the lines of text displayed in the E³.DatabaseEditor Results window.

		:param index: Start position of text lines to retrieve
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | lst | tuple[str,...] | Array of displayed text lines passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of displayed text lines |
		| 0 | No displayed text lines or an error occurred |

		Available since E3.series TLB-Versions: 25.32
		"""
		dummy=0
		ret, lst = self._obj.GetResultText(index, dummy)
		lst = lst[1:] if type(lst) == tuple and len(lst) > 0 else tuple()
		return ret, lst

# -------------------- IDbeApplicationInterface--------------------
class DbeApplication:
	def __init__(self, pid: typing.Optional[int]=None) -> None:
		if pid is None:
			pid = _get_default_dbe()
		if pid is None:
			raise RuntimeError('No instance of E3.DatabaseEditor is currently running')
		self._obj = _raw_connect_dbe(pid)

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetName(self) -> str:
		"""
		Gets the current name of E³.DatabaseEditor.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Build&gt;" | Name of E³.DatabaseEditor |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetName()

	def GetVersion(self) -> str:
		"""
		Gets the current version of E³.DatabaseEditor.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Version&gt;" | Version of E³.DatabaseEditor |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetVersion()

	def GetFullName(self) -> str:
		"""
		Gets the current path and name of E³.DatabaseEditor.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Build&gt;" | Path and name of E³.DatabaseEditor |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetFullName()

	def GetId(self) -> int:
		"""
		Gets the identifier of the current E³.DatabaseEditor item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current E³.DatabaseEditor item identifier |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetId()

	def Quit(self) -> int:
		"""
		Stops the script execution.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Script will immediately stop |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Quit()

	def Sleep(self, msec:int) -> int:
		"""
		Temporarily stops the script execution.

		:param msec: Number of milliseconds to sleep
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Script execution temporarily stops |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Sleep(msec)

	def Minimize(self) -> int:
		"""
		Minimizes the E³.DatabaseEditor application window.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | E³.DatabaseEditor application window is minimized |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Minimize()

	def Maximize(self) -> int:
		"""
		Maximizes the E³.DatabaseEditor application window.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | E³.DatabaseEditor application window is maximized |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Maximize()

	def Display(self) -> int:
		"""
		Displays the E³.DatabaseEditor application window.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | E³.DatabaseEditor application window is displayed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Display()

	def ShowNormal(self) -> int:
		"""
		Displays the E³.DatabaseEditor application window in its normal window state.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | E³.DatabaseEditor application window is displayed in its normal window state |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.ShowNormal()

	def PutMessage(self, text:str, item:int=0) -> int:
		"""
		Outputs text to the E³.DatabaseEditor Messages window.

		:param text: Text to be displayed
		:type str: str
		:param item: Identifier of an item contained in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value is always returned |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.PutMessage(text, item)

	def PutInfo(self, ok:int, text:str, item:int=0) -> int:
		"""
		Outputs information text to the E³.DatabaseEditor Messages window.

		:param ok: Indicates whether an additional information dialog should appear. If set to 0, no additional information dialog appears. If set to 1, an additional information dialog appears
		:type int: int
		:param text: Text to be displayed
		:type str: str
		:param item: Identifier of an item contained in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value is always returned |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.PutInfo(ok, text, item)

	def PutWarning(self, ok:int, text:str, item:int=0) -> int:
		"""
		Outputs warning text to the E³.DatabaseEditor Messages window.

		:param ok: Indicates whether an additional warning dialog should appear. If set to 0, no additional warning dialog appears. If set to 1, an additional warning dialog appears
		:type int: int
		:param text: Text to be displayed
		:type str: str
		:param item: Identifier of an item contained in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value is always returned |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.PutWarning(ok, text, item)

	def PutError(self, ok:int, text:str, item:int=0) -> int:
		"""
		Outputs error text to the E³.DatabaseEditor Messages window.

		:param ok: Indicates whether an additional error dialog should appear. If set to 0, no additional error dialog appears. If set to 1, an additional error dialog appears
		:type int: int
		:param text: Text to be displayed
		:type str: str
		:param item: Identifier of an item contained in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value is always returned |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.PutError(ok, text, item)

	def GetTestMark(self, num:int) -> int:
		"""


		:param num: 
		:type int: int

		:return:
		Type: int

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetTestMark(num)

	def SetTestMark(self, num:int, value:int) -> int:
		"""


		:param num: 
		:type int: int
		:param value: 
		:type int: int

		:return:
		Type: int

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetTestMark(num, value)

	def SetPrinterLinewidth(self, linewidth:float) -> float:
		"""
		Sets the  printer line width setting value.

		:param linewidth: Indicates how line widths will be printed. If 0.0, line width on sheets is used but print function scaling parameters are ignored. If -1.0, line width as defined on sheets is used according to print function scaling parameters
		:type float: float

		:return:
		Type: float

		| Return Value | Description |
		|:-----:|:-----|
		| 0.0 or -1.0 | Previous print line width setting value |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetPrinterLinewidth(linewidth)

	def GetInstallationPath(self) -> str:
		"""
		Gets the installation path of E³.DatabaseEditor.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Path&gt;" | Installation path of the E³.DatabaseEditor |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetInstallationPath()

	def GetInstallationLanguage(self) -> int:
		"""
		Gets the installation language of the E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current language identifier |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetInstallationLanguage()

	def EnableLogfile(self, en:int) -> int:
		"""
		Sets whether message data is to be written to the log files.

		:param en: Indicates whether data is written to the log files. If 1, data is written to the log files. If 0, data is not written to the log files
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Enabling of data to be written set. |
		| &lt; 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.EnableLogfile(en)

	def GetComponentDatabase(self) -> str:
		"""
		Gets the connection string of the currently active component database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Connection&gt;" | Connection string of the currently active component database |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetComponentDatabase()

	def GetConfigurationDatabase(self) -> str:
		"""
		Gets the connection string of the currently active configuration database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Connection&gt;" | Connection string of the currently active configuration database |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetConfigurationDatabase()

	def GetSymbolDatabase(self) -> str:
		"""
		Gets the connection string of the currently active symbol database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Database&gt;" | Connection string of the symbol database |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetSymbolDatabase()

	def GetLicense(self, feature:str) -> int:
		"""
		Includes a module or feature license to be adopted by the current instance of E³.DatabaseEditor.

		:param feature: Licensed feature to be used. See Module and Feature License Names for possible values
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Error occurred |
		| 1 | Licensed feature included |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetLicense(feature)

	def FreeLicense(self, feature:str) -> int:
		"""
		Releases a module or feature license adopted by the current instance of E³.DatabaseEditor.

		:param feature: Licensed feature to free up. See Module and Feature License Names for possible values
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Licensed feature could not be found |
		| 1 | Licensed feature freed up |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.FreeLicense(feature)

	def GetInfoCount(self) -> int:
		"""
		Gets the number of information messages already displayed in E³.DatabaseEditor Messages window.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=0 | Number of information messages already displayed in E³.DatabaseEditor Messages window |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetInfoCount()

	def GetWarningCount(self) -> int:
		"""
		Gets the number of warning messages already displayed in E³.DatabaseEditor Messages window.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=0 | Number of warning messages already displayed in  E³.DatabaseEditor Messages window |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetWarningCount()

	def GetErrorCount(self) -> int:
		"""
		Gets the number of error messages already displayed in E³.DatabaseEditor Messages window.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=0 | Number of error messages already displayed in E³.DatabaseEditor Messages window |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetErrorCount()

	def GetScriptArguments(self) -> tuple[str,...]:
		"""
		Gets the arguments passed into the script.

		:return:
		Type: tuple[str,...]

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetScriptArguments()

	def SortArrayByIndex(self, array:list[typing.Any], rows:int, columns:int, sortindex1:int, sortindex2:int) -> tuple[int, list[typing.Any]]:
		"""
		Sorts the contents of a 1 or 2 dimensional array.

		:param array: Array to sort passed in to the function unsorted and passed out sorted
		:type list[typing.Any]: list[typing.Any]
		:param rows: Number of rows in. array
		:type int: int
		:param columns: Number of columns in. array
		:type int: int
		:param sortindex1: Primary element in. array. to sort by
		:type int: int
		:param sortindex2: Secondary element in. array. to sort by
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | array | list[typing.Any] | Array to sort passed in to the function unsorted and passed out sorted |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Error occurred |
		| 0 | Array successfully sorted |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SortArrayByIndex(array, rows, columns, sortindex1, sortindex2)

	def FullName(self) -> str:
		"""
		Gets the current path and name of E³.DatabaseEditor.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| &lt;"Full Name"&gt; | Path and name of E³.DatabaseEditor |
		| &lt;"Empty"&gt; | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.FullName()

	def ScriptArguments(self) -> tuple[str,...]:
		"""
		Gets the arguments passed into the script.

		:return:
		Type: tuple[str,...]

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.ScriptArguments()

	def IsCable(self) -> int:
		"""
		Gets the usage status of the Cable module for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Cable module is in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsCable()

	def IsSchema(self) -> int:
		"""
		Gets the usage status of the Schema module for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Schema module is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsSchema()

	def IsMultiuser(self) -> int:
		"""
		Gets the usage status of the multi-user functionality for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Multi-user functionality is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsMultiuser()

	def IsPanel(self) -> int:
		"""
		Gets the usage status of the Panel module for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Panel module is in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsPanel()

	def IsWire(self) -> int:
		"""
		Gets the usage status of the Wire functionality for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Wire functionality is in use |
		| 0 | Wire functionality is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsWire()

	def IsSmallBusiness(self) -> int:
		"""
		Gets the usage status of the E³.SmallProject version for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | E³.SmallProject version is in use |
		| 0 | E³.SmallProject version is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsSmallBusiness()

	def IsDemo(self) -> int:
		"""
		Gets the usage status of the demonstration version for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Demonstration version is in use |
		| 0 | Demonstration version is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsDemo()

	def IsViewer(self) -> int:
		"""
		Gets the usage status of the Viewer version for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Viewer version is in use |
		| 0 | Viewer version is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsViewer()

	def IsViewPlus(self) -> int:
		"""
		Gets the usage status of the ViewPlus version for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | ViewPlus version is in use |
		| 0 | ViewPlus version is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsViewPlus()

	def IsStudent(self) -> int:
		"""
		Gets the usage status of the Student version for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Student version is in use |
		| 0 | Student version is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsStudent()

	def IsCaddyDemo(self) -> int:
		"""
		Gets the  usage status of the Caddy Demonstration version for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Caddy Demonstration version is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsCaddyDemo()

	def GetBuild(self) -> str:
		"""
		Gets the current build version of E³.DatabaseEditor.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Build&gt;" | Build version of E³.DatabaseEditor |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetBuild()

	def SortArrayByIndexEx(self, array:list[typing.Any], options:list[typing.Any]) -> tuple[int, list[typing.Any]]:
		"""
		Sorts the contents of a 1 or 2 dimensional array.

		:param array: Array to sort passed in to the function unsorted and passed out sorted
		:type list[typing.Any]: list[typing.Any]
		:param options: Array of sorting criteria
		:type list[typing.Any]: list[typing.Any]

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | array | list[typing.Any] | Array to sort passed in to the function unsorted and passed out sorted |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Array successfully sorted |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SortArrayByIndexEx(array, options)

	def GetRegistryVersion(self) -> str:
		"""
		Gets the internal major version of E³.DatabaseEditor.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Version&gt;" | Internal version of E³.DatabaseEditor |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetRegistryVersion()

	def GetLanguageDatabase(self) -> str:
		"""
		Gets the connection string of the currently active language database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Connection&gt;" | Connection string of the currently active language database |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetLanguageDatabase()

	def IsRedliner(self) -> int:
		"""
		Gets the usage status of the Redliner version for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Redliner version is in use |
		| 0 | Redliner version is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsRedliner()

	def ClearOutputWindow(self) -> None:
		"""
		Clears the content of the E³.DatabaseEditor Messages window.

		:return:
		No return value.

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.ClearOutputWindow()

	def AvoidAutomaticClosing(self, avoid:bool=True) -> bool:
		"""
		Sets the ability for E³.DatabaseEditor to remain open following the script process ending.

		:param avoid: Indicates whether. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. should remain open. If 1, E³.DatabaseEditor remains open. If 0, E³.DatabaseEditor may close., defaults to True
		:type bool: bool

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previous status was for E³.DatabaseEditor to remains open |
		| 0 | Previous status was for E³.DatabaseEditor to maybe close |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.AvoidAutomaticClosing(avoid)

	def ScriptFullName(self) -> str:
		"""
		Gets the name and path of the script running internally.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Path&gt;" | Name and path of the script running internally |
		| "&lt;Empty&gt;" | Script is being run externally |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.ScriptFullName()

	def ScriptName(self) -> str:
		"""
		Gets the name of the script running internally.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Path&gt;" | Name of the script running internally |
		| "&lt;Empty&gt;" | Script is being run externally |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.ScriptName()

	def GetPluginObject(self, Plugin:typing.Any) -> typing.Any:
		"""
		Gets or creates an instance of a plugin object.

		:param Plugin: Name of the plugin
		:type typing.Any: typing.Any

		:return:
		Type: typing.Any

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetPluginObject(Plugin)

	def Include(self, text:str) -> int:
		"""
		Executes a script and includes its defined functions for use within the current script.

		:param text: Path and name of the script file
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 5 | Recursive call occurred, Available since v2021 - 22.52, v2022 - 23.50 and v2023-24.10 |
		| 4 | Error occurred |
		| 3 | Error in script code |
		| 2 | Error occurred |
		| 1 | text script file not found or is empty |
		| 0 | Script is executed |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Include(text)

	def GetLogfileName(self, index:int=0) -> str:
		"""
		Gets the current path and file name of the log file.

		:param index: Indicates which log file path is required. If 0, the log file of the messages is returned. If 1, the log file of the results is returned., defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File Path&gt;" | Log file of E³.DatabaseEditor |
		| "&lt;Empty&gt;" | No log file assigned or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetLogfileName(index)

	def SetLogfileName(self, logfile:str, index:int=0) -> int:
		"""
		Sets the current path and file name of the log file.

		:param logfile: Path and name of the new log file
		:type str: str
		:param index: Indicates which log file path is to be set. If 0, the messages log file path is set. If 1, the results log file path is set., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Log file path is set |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetLogfileName(logfile, index)

	def GetWorkspaceName(self) -> str:
		"""
		Gets the current workspace configuration name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Workspace&gt;" | Name of the current workspace configuration |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetWorkspaceName()

	def GetActualDatabase(self) -> str:
		"""
		Gets the current database configuration.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Database&gt;" | Name of the current database configuration |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetActualDatabase()

	def SetActualDatabase(self, dbname:str) -> int:
		"""
		Sets the current database configuration.

		:param dbname: Name of the database configuration
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Current database configuration applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetActualDatabase(dbname)

	def GetDefinedDatabases(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets the configuration names of the defined databases available.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | dbnames | tuple[str,...] | Array of database configuration names |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in dbnames |
		| 0 | No database configurations or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		ret, dbnames = self._obj.GetDefinedDatabases(dummy)
		dbnames = dbnames[1:] if type(dbnames) == tuple and len(dbnames) > 0 else tuple()
		return ret, dbnames

	def GetDefinedDatabaseConnectionStrings(self, dbname:str) -> tuple[int, str, str, str]:
		"""
		Gets the connection strings for a database configuration.

		:param dbname: Name of the database configuration
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | cmp_cs | str | Component database connection string |
		| 2 | sym_cs | str | Symbol database connection string |
		| 3 | cnf_cs | str | Configuration database connection string |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Connection strings successfully supplied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		return self._obj.GetDefinedDatabaseConnectionStrings(dbname, dummy, dummy, dummy)

	def SetDefinedDatabaseConnectionStrings(self, dbname:str, cmp_cs:str, sym_cs:str, cnf_cs:str) -> int:
		"""
		Creates or modifies a database configuration.

		:param dbname: Name of the database configuration
		:type str: str
		:param cmp_cs: Component database connection string
		:type str: str
		:param sym_cs: Symbol database connection string
		:type str: str
		:param cnf_cs: Configuration database connection string
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Connection strings successfully applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetDefinedDatabaseConnectionStrings(dbname, cmp_cs, sym_cs, cnf_cs)

	def SetLanguageDatabase(self, dbname:str) -> int:
		"""
		Sets the currently active language database.

		:param dbname: Connection string for a language database
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Database successfully assigned as the language database |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetLanguageDatabase(dbname)

	def SetTemplateFileDBE(self, templatefilename:str) -> int:
		"""
		Sets the project settings template file.

		:param templatefilename: Path and name of a template file
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Template file settings assigned |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetTemplateFileDBE(templatefilename)

	def GetTemplateFileDBE(self) -> str:
		"""
		Gets the project settings template file.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File&gt;" | Path and name of the template file |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetTemplateFileDBE()

	def IsScriptRunning(self) -> bool:
		"""
		Indicates if instances of E³ API interfaces currently exist.

		:return:
		Type: bool

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Instances of E³ API interfaces exist |
		| 0 | Instances of E³ API interfaces do not exist |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsScriptRunning()

	def SetTriggerReturn(self, value:int) -> int:
		"""
		Sets the current trigger script return value.

		:param value: Trigger return value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1..7 | Previous trigger return value |
		| 0 | No trigger is active |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetTriggerReturn(value)

	def GetTriggerReturn(self) -> int:
		"""
		Gets the current trigger script return value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | No trigger is active |
		| 1..7 | Trigger script return value |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetTriggerReturn()

	def GetComponentDatabaseTableSchema(self) -> str:
		"""
		Gets the table schema of the currently active component database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Schema&gt;" | Table schema of the currently active component database |
		| "&lt;Empty&gt;" | No table schema or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetComponentDatabaseTableSchema()

	def GetConfigurationDatabaseTableSchema(self) -> str:
		"""
		Gets the table schema of the currently active configuration database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Schema&gt;" | Table schema of the currently active configuration database |
		| "&lt;Empty&gt;" | No table schema or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetConfigurationDatabaseTableSchema()

	def GetSymbolDatabaseTableSchema(self) -> str:
		"""
		Gets the table schema of the currently active symbol database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Schema&gt;" | Table schema of the currently active symbol database |
		| "&lt;Empty&gt;" | No table schema or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetSymbolDatabaseTableSchema()

	def GetLanguageDatabaseTableSchema(self) -> str:
		"""
		Gets the table schema of the currently active language database.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Schema&gt;" | Table schema of the currently active language database |
		| "&lt;Empty&gt;" | No table schema or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetLanguageDatabaseTableSchema()

	def GetProcessProperty(self, what:str) -> str:
		"""
		Gets the value of a property of the E³.DatabaseEditor's process.

		:param what: Process property
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Value&gt;" | Select pages setting value |
		| "&lt;Empty&gt;" | Process property has no value or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetProcessProperty(what)

	def IsFluid(self) -> int:
		"""
		Gets the usage status of the Fluid module for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Fluid module is in use |
		| 0 | Fluid module is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsFluid()

	def IsFormboard(self) -> int:
		"""
		Gets the usage status of the Formboard module for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Formboard module is in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsFormboard()

	def GetTrigger(self, name:str) -> tuple[int, str]:
		"""
		Gets a trigger's status and designated script file.

		:param name: Trigger name
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | filename | str | Path of designated script file to be triggered passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Trigger is active |
		| 0 | Trigger is inactive |
		| -2 | name is invalid |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		return self._obj.GetTrigger(name, dummy)

	def SetTrigger(self, name:str, filename:str, active:int) -> int:
		"""
		Sets a trigger's status and designated script file.

		:param name: Trigger name
		:type str: str
		:param filename: Path of designated script file to be triggered
		:type str: str
		:param active: Trigger status
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previous trigger status was active |
		| 0 | Previous trigger status was inactive |
		| -1 | filename is invalid |
		| -2 | name is invalid |
		| -3 | Trigger registry key is missing or set to 0 |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetTrigger(name, filename, active)

	def IsEconomy(self) -> int:
		"""
		Gets the usage status of the economy version for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Economy version is in use |
		| 0 | Economy version is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsEconomy()

	def GetAvailableLanguages(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets the available language names.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | languages | tuple[str,...] | Array of available language names passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in languages |
		| 0 | No available language names are found or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		ret, languages = self._obj.GetAvailableLanguages(dummy)
		languages = languages[1:] if type(languages) == tuple and len(languages) > 0 else tuple()
		return ret, languages

	def GetTranslatedText(self, text:str, language:str) -> str:
		"""
		Gets the text value with all text tokens translated.

		:param text: Text to translate
		:type str: str
		:param language: Translation language
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Translated text or text |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetTranslatedText(text, language)

	def Run(self, filename:str, arguments:list[str]) -> int:
		"""
		Immediately executes a script internally.

		:param filename: Path and name of script file
		:type str: str
		:param arguments: Array of script arguments
		:type list[str]: list[str]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| Any | Return value from the executed script |
		| -1 | Return value from the executed script or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.Run(filename, arguments)

	def SetScriptReturn(self, value:int) -> int:
		"""
		Sets the current script return value.

		:param value: Script return value
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| Any | Previous script return value |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetScriptReturn(value)

	def GetScriptReturn(self) -> int:
		"""
		Gets the current script return value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| Any | Script return value |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetScriptReturn()

	def GetEnableInteractiveDialogs(self) -> int:
		"""
		Gets the setting value enabling interactive dialogs to appear.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Interactive dialogs are set to appear |
		| 0 | Interactive dialogs are not set to appear |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetEnableInteractiveDialogs()

	def SetEnableInteractiveDialogs(self, value:bool) -> int:
		"""
		Sets the setting value enabling interactive dialogs to appear.

		:param value: New enabling interactive dialogs value. If 1, interactive dialogs are enabled. If 0, interactive dialogs are disabled
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Previous value for enabling interactive dialogs enabling interactive dialogs was 1 |
		| 0 | Previous value for enabling interactive dialogs enabling interactive dialogs was 0 |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetEnableInteractiveDialogs(value)

	def IsWireWorks(self) -> int:
		"""
		Gets the usage status of the WireWorks functionality for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | WireWorks functionality is in use |
		| 0 | WireWorks functionality is not in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsWireWorks()

	def SetModalWindow(self, hWnd:int) -> int:
		"""
		Sets the window handle of the modal window in front of the E³.DatabaseEditor.

		:param hWnd: Handle of the window
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Modal window value applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.SetModalWindow(hWnd)

	def GetModalWindow(self) -> int:
		"""
		Gets the window handle of the modal window in front of the E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Handle of the modal window in front of E³.DatabaseEditor |
		| 0 | No modal window is in front of E³.DatabaseEditor or an error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetModalWindow()

	def BeginForeignTask(self) -> int:
		"""
		Changes the status of E³.DatabaseEditor to inactive.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Previous status of E³.DatabaseEditor was active |
		| 1 | Previous status of E³.DatabaseEditor was inactive |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.BeginForeignTask()

	def EndForeignTask(self) -> int:
		"""
		Changes the status of E³.DatabaseEditor to active.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Previous status of E³.DatabaseEditor was active |
		| 1 | Previous status of E³.DatabaseEditor was inactive |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.EndForeignTask()

	def IsFunctionalDesign(self) -> int:
		"""
		Gets the usage status of the Functional Design module for E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Functional Design module is in use |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.IsFunctionalDesign()

	def GetProjectInformation(self, filename:str) -> tuple[int, str, int, int]:
		"""
		Gets a .e3s project's information.

		:param filename: File name of the project. If no project is open, a project file name can be passed into the function. If a project is open, the project file name is passed out of the function
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | filename | str | File name of the project. If no project is open, a project file name can be passed into the function. If a project is open, the project file name is passed out of the function |
		| 2 | _type | int | Type of project passed out of the function |
		| 3 | is_dbe | int | Value passed out of the function indicating if the project is an. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project. If 1, the project is an E³.DatabaseEditor project. If 0, the project is an E³.series project |

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Project information provided |
		| 1 | filename was not found |
		| 2 | File information could not be read |
		| 3 | File is already open |

		Available since E3.series TLB-Versions: 10.00
		"""
		dummy=0
		return self._obj.GetProjectInformation(filename, dummy, dummy)

	def ResetInfoCount(self) -> int:
		"""
		Resets the information count of E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=0 | Information count value before it was reset |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.ResetInfoCount()

	def ResetWarningCount(self) -> int:
		"""
		Resets the warning count of E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=0 | Warning count value before it was reset |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.ResetWarningCount()

	def ResetErrorCount(self) -> int:
		"""
		Resets the error count of E³.DatabaseEditor.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt;=0 | Error count value before it was reset |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.ResetErrorCount()

	def GetLicensePermanent(self, feature:str) -> int:
		"""
		Includes a module or feature license to be adopted by all E³.series processes.

		:param feature: Licensed feature to be used. See Module and Feature License Names for possible values
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Error occurred |
		| 1 | Licensed feature included |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetLicensePermanent(feature)

	def FreeLicensePermanent(self, feature:str) -> int:
		"""
		Releases a module or feature license adopted by E³.series on a permanent basis.

		:param feature: Licensed feature to be used. See Module and Feature License Names for possible values
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Licensed feature could not be found |
		| 1 | Licensed feature freed up |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.FreeLicensePermanent(feature)

	def GetProvider(self, dbname:str) -> str:
		"""
		Gets the database provider.

		:param dbname: Database configuration name
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Provider&gt;" | Name of the database provider |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 10.00
		"""
		return self._obj.GetProvider(dbname)

	def GetPrintCropMarks(self) -> int:
		"""
		Gets the status of the print crop marks.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Print crop marks usage is on |
		| 0 | Print crop marks usage is off |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.GetPrintCropMarks()

	def GetPrintPageNumbers(self) -> int:
		"""
		Gets the print page number setting value.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Page numbers will be printed |
		| 0 | Page numbers will not be printed |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.GetPrintPageNumbers()

	def SetPrintPageNumbers(self, set:bool) -> int:
		"""
		Sets the print page number setting value.

		:param set: Indicates the print page number setting value. If 1, the page numbers are printed. If 0, the page numbers are not printed
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..1 | Previous value of the print page number |

		Available since E3.series TLB-Versions: 10.10
		"""
		return self._obj.SetPrintPageNumbers(set)

	def SetPrintSheetOrder(self, set:int) -> int:
		"""
		Sets the print page order setting value.

		:param set: Indicates the print page order setting
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0..1 | Previous value of the print page order setting |

		Available since E3.series TLB-Versions: 10.20
		"""
		return self._obj.SetPrintSheetOrder(set)

	def SelectComponentFromTable(self) -> tuple[int, str, str]:
		"""
		Gets the interactively selected component name and version from the E³.DatabaseEditor Component Table dialog.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ComponentName | str | Selected component name passed out of the function |
		| 2 | ComponentVersion | str | Selected component version passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Component was selected |
		| 0 | No component was selected |

		Available since E3.series TLB-Versions: 11.70
		"""
		dummy=0
		return self._obj.SelectComponentFromTable(dummy, dummy)

	def GetDatabaseTableSelectedComponents(self) -> tuple[int, tuple[str,...], tuple[str,...]]:
		"""
		Gets the names and versions of the components selected in the E³.DatabaseEditor's Component Table.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ComponentArray | tuple[str,...] | Array of component names |
		| 2 | VersionArray | tuple[str,...] | Array of component versions |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ComponentArray and VersionArray |
		| 0 | No components are selected or an error occurred |

		Available since E3.series TLB-Versions: 11.70
		"""
		dummy=0
		ret, ComponentArray, VersionArray = self._obj.GetDatabaseTableSelectedComponents(dummy, dummy)
		ComponentArray = ComponentArray[1:] if type(ComponentArray) == tuple and len(ComponentArray) > 0 else tuple()
		VersionArray = VersionArray[1:] if type(VersionArray) == tuple and len(VersionArray) > 0 else tuple()
		return ret, ComponentArray, VersionArray

	def GetDatabaseTreeSelectedComponents(self) -> tuple[int, tuple[str,...], tuple[str,...]]:
		"""
		Gets the names and versions of the components selected in the E³.DatabaseEditor Component database tree.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ComponentName | tuple[str,...] | Array of component names |
		| 2 | Version | tuple[str,...] | Array of component versions |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ComponentArray and VersionArray |
		| 0 | No components are selected or an error occurred |

		Available since E3.series TLB-Versions: 11.70
		"""
		dummy=0
		ret, ComponentName, Version = self._obj.GetDatabaseTreeSelectedComponents(dummy, dummy)
		ComponentName = ComponentName[1:] if type(ComponentName) == tuple and len(ComponentName) > 0 else tuple()
		Version = Version[1:] if type(Version) == tuple and len(Version) > 0 else tuple()
		return ret, ComponentName, Version

	def GetDatabaseTreeSelectedSymbols(self) -> tuple[int, tuple[str,...], tuple[str,...]]:
		"""
		Gets the names and versions of the symbols selected in the E³.DatabaseEditor Symbol database tree.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | SymbolName | tuple[str,...] | Array of symbol names |
		| 2 | Version | tuple[str,...] | Array of symbol versions |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in SymbolArray and VersionArray |
		| 0 | No symbols are selected or an error occurred |

		Available since E3.series TLB-Versions: 11.70
		"""
		dummy=0
		ret, SymbolName, Version = self._obj.GetDatabaseTreeSelectedSymbols(dummy, dummy)
		SymbolName = SymbolName[1:] if type(SymbolName) == tuple and len(SymbolName) > 0 else tuple()
		Version = Version[1:] if type(Version) == tuple and len(Version) > 0 else tuple()
		return ret, SymbolName, Version

	def GetDatabaseTreeSelectedModels(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets the names of the models selected in the E³.DatabaseEditor's Model tree.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ModelName | tuple[str,...] | Array of model names |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ModelArray |
		| 0 | No models are selected or an error occurred |

		Available since E3.series TLB-Versions: 11.70
		"""
		dummy=0
		ret, ModelName = self._obj.GetDatabaseTreeSelectedModels(dummy)
		ModelName = ModelName[1:] if type(ModelName) == tuple and len(ModelName) > 0 else tuple()
		return ret, ModelName

	def ClearResultWindow(self) -> None:
		"""
		Clears the content of the E³.DatabaseEditor Results window.

		:return:
		No return value.

		Available since E3.series TLB-Versions: 11.80
		"""
		return self._obj.ClearResultWindow()

	def BringToForeground(self) -> int:
		"""
		Brings the E³.DatabaseEditor application to the foreground of the monitor display.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | E³.DatabaseEditor brought to the foreground |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.90
		"""
		return self._obj.BringToForeground()

	def PutErrorEx(self, flags:int, text:str, item:int, red:int, green:int, blue:int) -> int:
		"""
		Outputs error text to the E³.DatabaseEditor Messages window.

		:param flags: Indicates whether an additional error dialog should appear. If set to 0, no additional error dialog appear. If set to 1, an additional error dialog appear
		:type int: int
		:param text: Text to be displayed
		:type str: str
		:param item: Identifier of an item contained in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project
		:type int: int
		:param red: RGB colorspace's red value for the text color
		:type int: int
		:param green: RGB colorspace's green value for the text color
		:type int: int
		:param blue: RGB colorspace's blue value for the text color
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value is always returned |

		Available since E3.series TLB-Versions: 11.90
		"""
		return self._obj.PutErrorEx(flags, text, item, red, green, blue)

	def PutWarningEx(self, flags:int, text:str, item:int, red:int, green:int, blue:int) -> int:
		"""
		Outputs warning text to the E³.DatabaseEditor Messages window.

		:param flags: Indicates whether an additional warning dialog should appear. If set to 0, no additional warning dialog appears. If set to 1, an additional warning dialog appears
		:type int: int
		:param text: Text to be displayed
		:type str: str
		:param item: Identifier of an item contained in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project
		:type int: int
		:param red: RGB colorspace's red value for the text color
		:type int: int
		:param green: RGB colorspace's green value for the text color
		:type int: int
		:param blue: RGB colorspace's blue value for the text color
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value is always returned |

		Available since E3.series TLB-Versions: 11.90
		"""
		return self._obj.PutWarningEx(flags, text, item, red, green, blue)

	def PutInfoEx(self, flags:int, text:str, item:int, red:int, green:int, blue:int) -> int:
		"""
		Outputs information text to the E³.DatabaseEditor Messages window.

		:param flags: Indicates whether an additional information dialog should appear. If set to 0, no additional information dialog appears. If set to 1, an additional information dialog appears
		:type int: int
		:param text: Text to be displayed
		:type str: str
		:param item: Identifier of an item contained in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project
		:type int: int
		:param red: RGB colorspace's red value for the text color
		:type int: int
		:param green: RGB colorspace's green value for the text color
		:type int: int
		:param blue: RGB colorspace's blue value for the text color
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value is always returned |

		Available since E3.series TLB-Versions: 11.90
		"""
		return self._obj.PutInfoEx(flags, text, item, red, green, blue)

	def PutMessageEx(self, flags:int, text:str, item:int, red:int, green:int, blue:int) -> int:
		"""
		Outputs text to the E³.DatabaseEditor Messages window.

		:param flags: Not used
		:type int: int
		:param text: Text to be displayed
		:type str: str
		:param item: Identifier of an item contained in the. <span style="font-style: italic;color: #ff0000;">E³.</span>DatabaseEditor. project
		:type int: int
		:param red: RGB colorspace's red value for the text color
		:type int: int
		:param green: RGB colorspace's green value for the text color
		:type int: int
		:param blue: RGB colorspace's blue value for the text color
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | This value is always returned |

		Available since E3.series TLB-Versions: 11.90
		"""
		return self._obj.PutMessageEx(flags, text, item, red, green, blue)

	def ActivateOutputWindow(self, windowId:int) -> int:
		"""
		Displays an output window in the E³.DatabaseEditor.

		:param windowId: Identifier of the output window to display
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the previously displayed output window |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 11.90
		"""
		return self._obj.ActivateOutputWindow(windowId)

	def SetChildWindowState(self, state:int) -> int:
		"""
		Sets the child windows' state.

		:param state: Child window state
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Child window state is changed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 17.70
		"""
		return self._obj.SetChildWindowState(state)

	def ShowPluginWindow(self, bShowPluginWindow:bool, guid:str) -> int:
		"""
		Displays or hides an E³.DatabaseEditor plugin window.

		:param bShowPluginWindow: Indicates whether the plugin window is displayed. If 1, the plugin window is displayed. If 0, the plugin window is not displayed
		:type bool: bool
		:param guid: GUID of the plugin window
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Value is always returned |

		Available since E3.series TLB-Versions: 17.70
		"""
		return self._obj.ShowPluginWindow(bShowPluginWindow, guid)

	def ShowWindow(self, windowId:int, show:bool) -> int:
		"""
		Displays or hides an E³.DatabaseEditor window.

		:param windowId: Identifier of the window
		:type int: int
		:param show: Indicates whether the window is displayed. If 1, the window is displayed. If 0, the window is not displayed
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Window display status is changed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 17.70
		"""
		return self._obj.ShowWindow(windowId, show)

	def SaveWorkspaceConfiguration(self, name:str="") -> int:
		"""
		Creates or overwrites a workspace configuration.

		:param name: Name of the workspace configuration, defaults to ""
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Work configuration is saved |
		| -1 | No E³.DatabaseEditor window |
		| -2 | No active workspace configuration |
		| -3 | Workspace configuration cannot be set to active |
		| -4 | Workspace configuration could not be saved |
		| -12 | E³.DatabaseEditor is in full screen mode |

		Available since E3.series TLB-Versions: 18.10
		"""
		return self._obj.SaveWorkspaceConfiguration(name)

	def DeleteWorkspaceConfiguration(self, name:str="") -> int:
		"""
		Deletes a workspace configuration.

		:param name: Name of the workspace configuration, defaults to ""
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Work configuration is saved |
		| -1 | No E³.DatabaseEditor window |
		| -2 | No active workspace configuration |
		| -3 | Workspace configuration cannot be set to active |
		| -4 | Workspace configuration could not be saved |
		| -5 | Default workspace configuration cannot be deleted |
		| -10 | Workspace configuration is not contained in the XML file |
		| -12 | E³.DatabaseEditor is in full screen mode |

		Available since E3.series TLB-Versions: 18.10
		"""
		return self._obj.DeleteWorkspaceConfiguration(name)

	def RestoreWorkspaceConfiguration(self, name:str="") -> int:
		"""
		Restores the workspace configuration.

		:param name: Name of the workspace configuration, defaults to ""
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Name of the current workspace |
		| -1 | No E³.DatabaseEditor window |
		| -2 | No active workspace configuration |
		| -3 | Workspace configuration cannot be set to active |
		| -4 | Workspace configuration could not be changed |
		| -6 | Reference XML file could not be loaded |
		| -10 | Workspace configuration is not contained in the XML file |
		| -12 | E³.DatabaseEditor is in full screen mode |

		Available since E3.series TLB-Versions: 18.10
		"""
		return self._obj.RestoreWorkspaceConfiguration(name)

	def GetWorkspaceConfigurations(self, path:str="") -> tuple[int, tuple[str,...]]:
		"""
		Gets an array of workspace configuration names.

		:param path: XML file to extract workspace configuration names from. Default value is <"Empty">, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | names | tuple[str,...] | Array of workspace configuration names passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| -6 | path file could not be loaded |
		| -7 | path file cannot be found |
		| -8 | XML file in %APPDATA% could not be loaded |
		| -12 | E³.DatabaseEditor is in full screen mode |

		Available since E3.series TLB-Versions: 18.10
		"""
		dummy=0
		ret, names = self._obj.GetWorkspaceConfigurations(dummy, path)
		names = names[1:] if type(names) == tuple and len(names) > 0 else tuple()
		return ret, names

	def LoadWorkspaceConfigurationFromFile(self, name:str, path:str) -> int:
		"""
		Loads a workspace configuration from an XML file and adopts the configuration into the E³.DatabaseEditor instance.

		:param name: Name of the workspace configuration
		:type str: str
		:param path: Name of the configuration file with complete path specification
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Current workspace configuration overwritten from file |
		| -1 | No E³.DatabaseEditor window |
		| -6 | path file could not be loaded |
		| -7 | path file could not be found |
		| -8 | XML file in %APPDATA% could not be loaded |
		| -9 | Workspace configuration could not be set as current workspace configuration |
		| -10 | Workspace configuration is not contained in the XML file |
		| -12 | E³.DatabaseEditor is in full screen mode |

		Available since E3.series TLB-Versions: 18.10
		"""
		return self._obj.LoadWorkspaceConfigurationFromFile(name, path)

	def GetCurrentWorkspaceConfiguration(self) -> str:
		"""
		Gets the current workspace configuration name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Workspace&gt;" | Name of the current workspace configuration |

		Available since E3.series TLB-Versions: 18.10
		"""
		return self._obj.GetCurrentWorkspaceConfiguration()

	def NewComponentWithPreconditions(self, name:str, version:str, baseName:str, baseVersion:str, preconditions:dict[str,str], flags:int=0) -> int:
		"""
		Adds a component to the database and starts the Component Wizard in the E³.DatabaseEditor.

		:param name: Name of the new component
		:type str: str
		:param version: Version of the new component. If version value is empty, the version is set to "1   "
		:type str: str
		:param baseName: Name of the component to be used as a template. If baseName value is empty, no template is used
		:type str: str
		:param baseVersion: Version of the component to be used as a template. If baseVersion value is empty, the version is set to "1   "
		:type str: str
		:param preconditions: Dictionary of [. String. ]. key. /[. String. ]. value. pairs representing preconditions. key is the access value and should be set to "SubCircuitFileName". value is a path to a part file (.e3p) containing subcircuits
		:type dict[str,str]: dict[str,str]
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Starts Component Wizard in the E³.DatabaseEditor |
		| -2 | baseName component not found |
		| -3 | Component is locked |
		| -4 | Component with name and version already exists |
		| -5 | version is invalid |
		| -6 | baseVersion is invalid |
		| -7 | name exceeds maximum length |
		| -8 | preconditions is empty |
		| -9 | preconditions contains no known preconditions |
		| -10 | preconditions contains invalid preconditions |

		Available since E3.series TLB-Versions: 20.12, 19.42
		"""
		preconditions = _dict_to_variant(preconditions)
		return self._obj.NewComponentWithPreconditions(name, version, baseName, baseVersion, preconditions, flags)

	def EditComponentWithPreconditions(self, name:str, version:str, preconditions:dict[str,str], flags:int=0) -> int:
		"""
		Opens the component for editing in the E³.DatabaseEditor.

		:param name: Name of the component to edit
		:type str: str
		:param version: Version of the component to edit. If version value is empty, the version is set to "1   "
		:type str: str
		:param preconditions: Dictionary of [. String. ]. key. /[. String. ]. value. pairs representing preconditions. key is the access value and should be set to "SubCircuitFileName". value is a path to a part file (.e3p) containing subcircuits
		:type dict[str,str]: dict[str,str]
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Opens Component Properties dialog in the E³.DatabaseEditor |
		| -2 | Component not found |
		| -3 | Component is locked |
		| -5 | version is too long |
		| -8 | preconditions is empty |
		| -9 | preconditions contains no known preconditions |
		| -10 | preconditions contains invalid precondition |

		Available since E3.series TLB-Versions: 20.12, 19.42
		"""
		preconditions = _dict_to_variant(preconditions)
		return self._obj.EditComponentWithPreconditions(name, version, preconditions, flags)

	def CreateDXFfromSTEP(self, stepFile:str, outputDirectory:str, dxfVersion:int, views:int=1, color:int=-1, flags:int=0) -> int:
		"""
		Creates AutoCad DXF files from a step file model.

		:param stepFile: Path and name of the STEP file
		:type str: str
		:param outputDirectory: Target system folder to save the DXF files to
		:type str: str
		:param dxfVersion: DXF format version
		:type int: int
		:param views: Model views to be created., defaults to 1
		:type int: int
		:param color: Graphics color in the DXF file. See Colors for possible values., defaults to -1
		:type int: int
		:param flags: Specifies import process options., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | DXF files are created |
		| -1 | stepFile file does not exist |
		| -2 | outputDirectory folder does not exist |
		| -3 | views is invalid |
		| -4 | color is invalid |
		| -5 | flags is invalid |
		| -6 | Error occurred importing the STEP file or exporting DXF files |
		| -7 | dxfVersion is invalid |
		| -8 | Import process aborted by user, Available since v2026-26.00 |
		| -9 | Import process aborted due to timing out, Available since v2026-26.00 |
		| -10 | Steps of the import process skipped by user, Available since v2026-26.00 |
		| -11 | Steps of the import process skipped due to timing out, Available since v2026-26.00 |

		Available since E3.series TLB-Versions: 22.41
		"""
		return self._obj.CreateDXFfromSTEP(stepFile, outputDirectory, dxfVersion, views, color, flags)

	def SuppressMessages(self, suppress:bool, flags:int=0) -> int:
		"""
		Suppresses messages appearing in the E³.DatabaseEditor Messages window.

		:param suppress: Indicates if messages are suppressed. If 1, the messages  are suppressed. If 0, the messages are not suppressed
		:type bool: bool
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Value always returned |

		Available since E3.series TLB-Versions: 23.21
		"""
		return self._obj.SuppressMessages(suppress, flags)

	def SetConfigFile(self, processType:int, filepath:str, flags:int=0) -> str:
		"""
		Sets the current path and file name of the configuration file for importing or exporting files.

		:param processType: Type of the import or export file described in the configuration file **Available enum type:** e3series.types.ConfigFileType.
		:type int: int
		:param filepath: Path and name of the configuration file
		:type str: str
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File Path&gt;" | Previous path and name of the configuration file |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.SetConfigFile(processType, filepath, flags)

	def GetConfigFile(self, processType:int, flags:int=0) -> str:
		"""
		Gets the current path and file name of the configuration file for importing or exporting files.

		:param processType: Type of the import or export files described in the configuration file **Available enum type:** e3series.types.ConfigFileType.
		:type int: int
		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File Path&gt;" | Path and name of the configuration file |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.GetConfigFile(processType, flags)

	def GetComponentList(self, additionalAttributes:tuple[str,...]=pythoncom.Empty, flags:int=0) -> tuple[int, tuple[tuple[typing.Union[str,int],...],...]]:
		"""
		Gets a list of components and component information from the component database.

		:param additionalAttributes: Array of database data column names of the. ComponentData. database table and attribute values to be supplied in. list, defaults to pythoncom.Empty
		:type list[str,...]: list[str,...]
		:param flags: Flags bit values specifying how the list of components is filtered., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | result | tuple[tuple[typing.Union[str,int],...],...] | List of components and component information from the component database passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of components in list |
		| 0 | No components are found or an error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		dummy=0
		return self._obj.GetComponentList(dummy, additionalAttributes, flags)

	def GetModelList(self, additionalAttributes:tuple[str,...]=0, flags:int=0) -> tuple[int, tuple[tuple[typing.Union[str,int],...],...]]:
		"""
		Gets a list of models and model information from the component database.

		:param additionalAttributes: Array of database data values and attributes to be supplied in. list. Array of database data column names of the ModelData database table and attribute values to be supplied in list, defaults to 0
		:type tuple[str,...]: tuple[str,...]
		:param flags: Not used., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | result | tuple[tuple[typing.Union[str,int],...],...] | List of models and model information from the component database passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of models in list |
		| 0 | No models are found or an error occurred |

		Available since E3.series TLB-Versions: 24.11
		"""
		dummy=0
		return self._obj.GetModelList(dummy, additionalAttributes, flags)

	def RemoveUndoInformation(self, flags:int=0) -> int:
		"""
		Removes all E³.DatabaseEditor undo information for the project.

		:param flags: Not used., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Undo information removed |
		| -1 | No project open |
		| -2 | flags value is invalid |

		Available since E3.series TLB-Versions: 24.11
		"""
		return self._obj.RemoveUndoInformation(flags)

	def CreateDbeJobObject(self) -> DbeJob:
		"""
		Creates an e3DbeJob object.

		:return:
		Type: DbeJob

		Available since E3.series TLB-Versions: 24.41
		"""
		return DbeJob(self._obj.CreateDbeJobObject())

	def GetJobCount(self) -> int:
		"""
		Indicates if a project is open.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | A project is open |
		| 0 | No project is open |

		Available since E3.series TLB-Versions: 24.41
		"""
		return self._obj.GetJobCount()

	def GetJobIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets the identifier of the project.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array containing the identifier of the open project passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | A project is open |
		| 0 | No project is open |

		Available since E3.series TLB-Versions: 24.41
		"""
		dummy=0
		ret, ids = self._obj.GetJobIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

# -------------------- IDeviceInterface--------------------
class Device:
	def __init__(self, obj: typing.Any) -> None:
		try:
			obj
		except AttributeError:
			raise OSError("Cannot initialize Device. Use Create-methods of other objects to create an instance.")
		self._obj = obj

	def __del__(self) -> None:
		try:
			del self._obj
		except:
			pass	# If there is no object there is no need to delete it

	def GetId(self) -> int:
		"""
		Gets the identifier of the current device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current device item identifier |
		| 0 | No device item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetId()

	def SetId(self, id:int) -> int:
		"""
		Sets a device as the current item.

		:param id: Unique value identifying a device item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Current device item identifier |
		| 0 | No current device item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetId(id)

	def GetName(self) -> str:
		"""
		Gets the device item's device designation.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Device designation |
		| "&lt;Empty&gt;" | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetName()

	def SetName(self, name:str) -> int:
		"""
		Sets the device item's device designation.

		:param name: Device designation value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Device designation applied or an error occurred |
		| 0 | Device locked |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetName(name)

	def GetAssignment(self) -> str:
		"""
		Gets the device item's higher level assignment value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Higher level assignment value supplied |
		| "&lt;Empty&gt;" | No higher level assignment value supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAssignment()

	def SetAssignment(self, name:str) -> int:
		"""
		Sets the device item's higher level assignment value.

		:param name: New higher level assignment value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Higher level assignment value applied or an error occurred |
		| 0 | Device locked |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAssignment(name)

	def GetLocation(self) -> str:
		"""
		Gets the device item's location value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Location value supplied |
		| "&lt;Empty&gt;" | No location value supplied or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetLocation()

	def SetLocation(self, name:str) -> int:
		"""
		Sets the device item's location value.

		:param name: New location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Location value supplied or an error occurred |
		| 0 | Device locked |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetLocation(name)

	def GetFileName(self) -> str:
		"""
		Gets the system path and name of the block device item's linked file.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;File&gt;" | System path and name of the block device's associated file |
		| "&lt;Empty&gt;" | Block device's has no associated file or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetFileName()

	def SetFileName(self, name:str) -> int:
		"""
		Sets the system path and name of the block device item's file.

		:param name: New file path and name
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New file name applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetFileName(name)

	def GetComponentName(self) -> str:
		"""
		Gets the name of the device item's component.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Component name |
		| "&lt;Empty&gt;" | No component or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetComponentName()

	def GetComponentVersion(self) -> str:
		"""
		Gets the version of the device item's component.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Version&gt;" | Component version |
		| "&lt;Empty&gt;" | No component or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetComponentVersion()

	def SetComponentName(self, name:str, version:str) -> int:
		"""
		Sets the device item's component type.

		:param name: New component name
		:type str: str
		:param version: New component version
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | New component name and version applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetComponentName(name, version)

	def GetCounterpartComponentName(self) -> str:
		"""
		Gets the name of the device item's mating connector component name.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Mating connector component name |
		| "&lt;Empty&gt;" | No mating connector component or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCounterpartComponentName()

	def GetCounterpartComponentVersion(self) -> str:
		"""
		Gets the name of the device item's mating connector component version.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Name&gt;" | Mating connector component version |
		| "&lt;Empty&gt;" | No mating connector component or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCounterpartComponentVersion()

	def AddAttibuteValue(self, name:str, value:str) -> int:
		"""
		Adds an attribute to the device item.

		:param name: Internal name of the new attribute
		:type str: str
		:param value: Value of the new attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new attribute |
		| 0 | Error occurred |
		| -1 | value string too long, Available since v2019-20.00 |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.AddAttibuteValue(name, value)

	def GetAttributeCount(self) -> int:
		"""
		Gets the number of the device item's attributes.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeCount()

	def GetAttributeIds(self, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to the device item.

		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAttributeIds(dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAttributeValue(self, name:str) -> str:
		"""
		Gets the device item's specified attribute value.

		:param name: Name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAttributeValue(name)

	def SetAttributeValue(self, name:str, value:str) -> int:
		"""
		Sets the device item's specified attribute value.

		:param name: Internal name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetAttributeValue(name, value)

	def DeleteAttribute(self, name:str) -> int:
		"""
		Removes an attribute from the device item.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteAttribute(name)

	def HasAttribute(self, name:str) -> int:
		"""
		Gets the number of the device item's specified attributes.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasAttribute(name)

	def GetComponentAttributeValue(self, name:str) -> str:
		"""
		Gets the specified attribute value of the device item's component.

		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetComponentAttributeValue(name)

	def GetViewCount(self) -> int:
		"""
		Gets the number of view connectors of the current block, connector or normal device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of views |
		| 0 | No views or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetViewCount()

	def GetViewIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the view devices of the current connector, block or normal device item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of view device items  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No view device identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetViewIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def IsView(self) -> int:
		"""
		Indicates if the device item represents a view of an original device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device item is the original device or an error occurred |
		| 1 | Device item is a view |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsView()

	def GetViewNumber(self) -> int:
		"""
		Gets the view number of the current view device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | View number of the view device |
		| 0 | Device has no view number or an error occurred |
		| -1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetViewNumber()

	def GetOriginalId(self) -> int:
		"""
		Gets the identifier of the original device of the view device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the original device |
		| 0 | No original device or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetOriginalId()

	def GetPinCount(self) -> int:
		"""
		Gets the number of pins or wires contained in the current device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pins or wires |
		| 0 | No pins and wires or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetPinCount()

	def GetPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the device item's pins or wires.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins or wires  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No pin identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAllPinCount(self) -> int:
		"""
		Gets the number of pins belonging to the device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pins |
		| 0 | No pins or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAllPinCount()

	def GetAllPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the supply pins anywhere within the device item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of pins  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No pin identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAllPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSupplyPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the device item's supply pins.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of supply pins  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No supply pin identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSupplyPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetNoconnPinIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of identifiers of the device item's pins with the "Not Connected" (**NC**) signal.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of pin identifiers  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of pin items in ids |
		| 0 | No pins with the "Not Connected" (**NC**) signal or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetNoconnPinIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetEndAttributeCount(self, which:int) -> int:
		"""
		Gets the number of attributes assigned to a part of the cable item.

		:param which: Cable part
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetEndAttributeCount(which)

	def GetEndAttributeIds(self, which:int, attnam:str="") -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the attributes assigned to a part of the cable item.

		:param which: Cable part
		:type int: int
		:param attnam: Attribute name filter. Only attribute identifiers with this attribute name will be supplied. All attribute names are supplied if attnam is an empty string. The default string value is empty, defaults to ""
		:type str: str

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of attributes  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned attribute identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetEndAttributeIds(which, dummy, attnam)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetEndAttributeValue(self, which:int, name:str) -> str:
		"""
		Gets a cable item part's specified attribute value.

		:param which: Cable part
		:type int: int
		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Attribute value supplied |
		| "&lt;Empty&gt;" | Attribute value could not be found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetEndAttributeValue(which, name)

	def SetEndAttributeValue(self, which:int, name:str, value:str) -> int:
		"""
		Sets the cable item part's specified attribute value.

		:param which: Cable part
		:type int: int
		:param name: Name of the attribute
		:type str: str
		:param value: Value of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of attribute |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetEndAttributeValue(which, name, value)

	def DeleteEndAttribute(self, which:int, name:str) -> int:
		"""
		Removes an attribute from a part of the cable item.

		:param which: Cable part
		:type int: int
		:param name: Name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Attribute was removed |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.DeleteEndAttribute(which, name)

	def HasEndAttribute(self, which:int, name:str) -> int:
		"""
		Gets the number of the cable item part's specified attributes.

		:param which: Cable part
		:type int: int
		:param name: Internal name of the attribute
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of attributes found |
		| 0 | No attribute found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasEndAttribute(which, name)

	def IsTerminal(self) -> int:
		"""
		Indicates if the device item is a terminal or a terminal strip.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a terminal item or an error occurred |
		| 1 | Device is a terminal |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsTerminal()

	def IsTerminalBlock(self) -> int:
		"""
		Indicates if the device item is a terminal strip.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a terminal strip or an error occurred |
		| 1 | Device is a terminal strip |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsTerminalBlock()

	def GetTerminalBlockId(self) -> int:
		"""
		Gets the identifier of the current device item's parent terminal strip device.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the parent terminal strip device |
		| 0 | No parent terminal strip device or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetTerminalBlockId()

	def SetTerminalSequence(self, ids:list[int]) -> int:
		"""
		Sets the terminal sequence in a terminal strip.

		:param ids: Array of device identifiers representing the terminal sequence passed into the function
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Terminal sequence was applied |
		| 1 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.SetTerminalSequence(ids)
		return ret[0]

	def GetSymbolCount(self, get_mode:int=0) -> int:
		"""
		Gets the number of symbols contained in the device item.

		:param get_mode: Symbol items type filter., defaults to 0
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbols |
		| 0 | No symbols or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSymbolCount(get_mode)

	def GetSymbolIds(self, get_mode:int=0) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of identifiers of symbol items of the device item.

		:param get_mode: Symbol items type filter., defaults to 0
		:type int: int

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of symbol items  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of symbol item identifiers in ids |
		| 0 | No symbol item identifiers found with respect to the get_mode filter or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetSymbolIds(dummy, get_mode)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetBundleCount(self) -> int:
		"""
		Gets the number of bundles contained in the cable device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of bundles |
		| 0 | No bundles or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBundleCount()

	def GetBundleIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the bundles of the cable device item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of bundle items  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No bundle identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetBundleIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def Create(self, name:str, assignment:str, location:str, comp:str, vers:str, after:int) -> int:
		"""
		Creates a new device item.

		:param name: Device designation value
		:type str: str
		:param assignment: Higher level assignment value
		:type str: str
		:param location: Location value
		:type str: str
		:param comp: Component name in the database
		:type str: str
		:param vers: Version in the database
		:type str: str
		:param after: Identifier of an existing terminal device. Only relevant for creating a terminal device
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new device item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Create(name, assignment, location, comp, vers, after)

	def CreateView(self, _from:int, level:int, blockid:int=0, databaseDeviceView:str="") -> int:
		"""
		Creates a view for the current device item.

		:param _from: Identifier of a device item from which the view is created
		:type int: int
		:param level: View number
		:type int: int
		:param blockid: Identifier of a block device item on which the connector view is to be placed., defaults to 0
		:type int: int
		:param databaseDeviceView: Database device view number. Default value is "<Empty>" and in this case a connector pin view is created for normal devices, defaults to ""
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new view |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateView(_from, level, blockid, databaseDeviceView)

	def IsDevice(self) -> int:
		"""
		Indicates if the device item is a normal device.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a normal device or an error occurred |
		| 1 | Device is a normal device |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsDevice()

	def CreateCable(self, name:str, assignment:str, location:str) -> int:
		"""
		Creates a new cable device item.

		:param name: Device designation value
		:type str: str
		:param assignment: Higher level assignment value
		:type str: str
		:param location: Location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new cable device item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateCable(name, assignment, location)

	def IsCable(self) -> int:
		"""
		Indicates if the device item is a cable or overbraid.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a cable or overbraid or an error occurred |
		| 1 | Device is a cable or overbraid |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsCable()

	def IsWireGroup(self) -> int:
		"""
		Indicates if the device item is a wire group.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a wire group or an error occurred |
		| 1 | Device is a wire group |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsWireGroup()

	def CreateConnector(self, name:str, assignment:str, location:str) -> int:
		"""
		Creates a new connector device item.

		:param name: Device designation value
		:type str: str
		:param assignment: Higher level assignment value
		:type str: str
		:param location: Location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new connector device item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateConnector(name, assignment, location)

	def IsConnector(self) -> int:
		"""
		Indicates if the device item is a connector.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a connector or an error occurred |
		| 1 | Device is a connector |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsConnector()

	def IsBlock(self) -> int:
		"""
		Indicates if the device item is a block.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a block or an error occurred |
		| 1 | Device is a block |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsBlock()

	def Delete(self) -> int:
		"""
		Deletes the device item from the project.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Device was deleted from the project |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Delete()

	def GetMasterPinName(self) -> str:
		"""
		Gets the terminal device item's master pin name value.

		:return:
		Type: str

		| Return Value | Description |
		|:-----:|:-----|
		| "&lt;Text&gt;" | Master pin name |
		| "&lt;Empty&gt;" | No master pin name or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetMasterPinName()

	def SetMasterPinName(self, name:str) -> int:
		"""
		Sets the terminal device item's master pin name value.

		:param name: New master pin name value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Master pin name |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetMasterPinName(name)

	def Jump(self) -> int:
		"""
		Jumps to the device item's location on the panel sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Value always returned |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Jump()

	def Search(self, name:str, assignment:str, location:str) -> int:
		"""
		Searches for a device item in the project and sets it as the current item.

		:param name: Device designation of device to search for
		:type str: str
		:param assignment: Higher level assignment of device to search for
		:type str: str
		:param location: Location of device to search for
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Device found |
		| 0 | No device was found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.Search(name, assignment, location)

	def GetConnectorCount(self) -> int:
		"""
		Gets the number of connectors contained in the block device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of connectors |
		| 0 | No connectors or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetConnectorCount()

	def GetConnectorIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the connector device items within the block device item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of connector device items  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No connector device identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetConnectorIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetBlockId(self) -> int:
		"""
		Gets the identifier of the current device item's parent block device.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the parent block device |
		| 0 | No parent block device or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetBlockId()

	def GetPanelLocation(self) -> tuple[int, float, float, float, float, float]:
		"""
		Gets the device item's model position on the panel.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | x | float | Panel placement position on the x-axis |
		| 2 | y | float | Panel placement position on the y-axis |
		| 3 | z | float | Panel placement position on the z-axis |
		| 4 | rot | float | Rotation value |
		| 5 | pivot | float | Tilt value  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Model's placement values are supplied |
		| 0 | Model is placed not axis parallel (using Euler Angles rotation ) or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		return self._obj.GetPanelLocation(dummy, dummy, dummy, dummy, dummy)

	def IsMount(self) -> int:
		"""
		Indicates if the device item is a mounting rail.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a mounting rail or an error occurred |
		| 1 | Device is a mounting rail |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsMount()

	def IsCableDuct(self) -> int:
		"""
		Indicates if the device item is a cable duct.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not a cable duct or an error occurred |
		| 1 | Device is a cable duct |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsCableDuct()

	def GetMountedCount(self) -> int:
		"""
		Gets the number of devices mounted on the current  device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of devices |
		| 0 | No devices or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetMountedCount()

	def GetMountedIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the devices mounted on the current device item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of mounted device items  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No mounted device identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetMountedIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetCarrierId(self) -> int:
		"""
		Gets the identifier of the device that the device item is mounted on in a panel sheet.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the carrier device |
		| 0 | No carrier device or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetCarrierId()

	def GetOutlineCount(self) -> int:
		"""
		Gets the number of contours contained in the connector, terminal or normal device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of contours |
		| 0 | No contours or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetOutlineCount()

	def GetOutlineIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets an array of identifiers of the device item's contours.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of contour identifiers  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No contours or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetOutlineIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetValidComponentCodes(self) -> tuple[int, tuple[str,...]]:
		"""
		Gets an array of names of components in the database with the same structure as the device item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | names | tuple[str,...] | Array of component names  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of component names in names |
		| 0 | No matching component names found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, names = self._obj.GetValidComponentCodes(dummy)
		names = names[1:] if type(names) == tuple and len(names) > 0 else tuple()
		return ret, names

	def IsAssembly(self) -> int:
		"""
		Indicates if the device item is an assembly.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not an assembly or an error occurred |
		| 1 | Device is an assembly |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsAssembly()

	def IsAssemblyPart(self) -> int:
		"""
		Indicates if the device item is part of an assembly.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 0 | Device is not part of an assembly or an error occurred |
		| 1 | Device is part of an assembly |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.IsAssemblyPart()

	def GetAssemblyId(self) -> int:
		"""
		Gets the identifier of the current device item's parent assembly device.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the parent assembly device |
		| 0 | No parent assembly device or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetAssemblyId()

	def GetDeviceCount(self) -> int:
		"""
		Gets the number of devices contained within the current terminal strip or assembly master device item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of devices |
		| 0 | No devices or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetDeviceCount()

	def GetDeviceIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of the device items within the terminal strip or assembly device item structure.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of device items  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No device identifiers are found or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetDeviceIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetAssignedOptionIds(self) -> tuple[int, tuple[int,...]]:
		"""
		Gets identifiers of options assigned to the device item.

		:return:

		Type: Tuple
		| Index | Name | Type | Description |
		|:-----:|:-----|:-----|:-----|
		| 0 | ReturnValue | int | See table return value |
		| 1 | ids | tuple[int,...] | Array of identifiers of assigned options  passed out of the function |

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Number of items in ids |
		| 0 | No assigned option identifiers are supplied or an error occurred |
		| -1 | An option contains a Boolean expression or an alias |

		Available since E3.series TLB-Versions: 8.50
		"""
		dummy=0
		ret, ids = self._obj.GetAssignedOptionIds(dummy)
		ids = ids[1:] if type(ids) == tuple and len(ids) > 0 else tuple()
		return ret, ids

	def GetSupplyId(self) -> int:
		"""
		Gets the identifier of the current device item's supply item.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the supply item |
		| 0 | No supply item or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.GetSupplyId()

	def SetSupplyId(self, id:int) -> int:
		"""
		Sets the current device item's supply item.

		:param id: Identifier of new supply item
		:type int: int

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the device item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetSupplyId(id)

	def HasNoconn(self) -> int:
		"""
		Indicates if the device item has pins with the "Not Connected" (**NC**) signal.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Device item has pins with the "Not Connected" (**NC**) signal |
		| 0 | Device item has no pins with the "Not Connected" (**NC**) signal or an error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.HasNoconn()

	def SetCompleteName(self, name:str, assignment:str, location:str, onlygiven:bool=False) -> int:
		"""
		Sets all elements of the device item's item designation.

		:param name: New device designation value
		:type str: str
		:param assignment: New higher level assignment value
		:type str: str
		:param location: New location value
		:type str: str
		:param onlygiven: Indicates whether empty values in. name. ,. assignment. and. location. should be ignored. If 1, empty values are ignored. If 0, empty values are set., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Item designation values applied |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.SetCompleteName(name, assignment, location, onlygiven)

	def CreateAssembly(self, ids:list[int]) -> int:
		"""
		Creates a new assembly device item.

		:param ids: Identifiers of the devices to include in the new assembly passed into the function. Can also hold a single integer value.
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new assembly device item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.CreateAssembly(ids)
		return ret[0]

	def AddToAssembly(self, ids:list[int], position:int=0, before:bool=False) -> int:
		"""
		Adds devices to the assembly device item.

		:param ids: Array of identifiers of devices to be added to the assembly passed into the function.
		:type list[int]: list[int]
		:param position: Identifier of an existing device in the assembly indicating the position where the new devices are added. If 0, the new devices are placed at the end., defaults to 0
		:type int: int
		:param before: Indicates whether the devices are placed before or after the. position. device. If 1, the new devices are placed before position. If 0, the new devices are placed after position., defaults to False
		:type bool: bool

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the assembly device |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.AddToAssembly(ids, position, before)
		return ret[0]

	def RemoveFromAssembly(self, ids:list[int]) -> int:
		"""
		Removes devices from the current assembly device item.

		:param ids: Array of identifiers of devices to be removed from the assembly passed into the function.
		:type list[int]: list[int]

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the assembly device |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		ret = self._obj.RemoveFromAssembly(ids)
		return ret[0]

	def CreateConnectorOnBlock(self, blkid:int, name:str, assignment:str, location:str) -> int:
		"""
		Creates a new connector device item on a block.

		:param blkid: Block identifier to add connector to
		:type int: int
		:param name: Device designation value
		:type str: str
		:param assignment: Higher level assignment value
		:type str: str
		:param location: Location value
		:type str: str

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| &gt; 0 | Identifier of the new connector device item |
		| 0 | Error occurred |

		Available since E3.series TLB-Versions: 8.50
		"""
		return self._obj.CreateConnectorOnBlock(blkid, name, assignment, location)

	def HasPassWirePins(self) -> int:
		"""
		Indicates if the device item has pins with the pass wire property.

		:return:
		Type: int

		| Return Value | Description |
		|:-----:|:-----|
		| 1 | Device item has pins with the pass wire property |
		| 0 | Device item has no pins with the pass wire property or an error occurred |

		Available since E3.series TLB-V