"""
Adaptive Wang-Mendel Method for Iterative Fuzzy Rule Optimization
==================================================================

This module implements an adaptive version of the Wang-Mendel algorithm that
iteratively refines both the fuzzy rule base and the membership functions.

Algorithm:
1. Generate fuzzy rule base using Wang-Mendel on training data
2. Evaluate performance (calculate loss/error on training set)
3. Use metaheuristics to optimize membership function parameters
4. Update fuzzy sets with optimized parameters
5. Return to step 1 and regenerate rules with new membership functions

This creates a feedback loop where membership functions are adjusted to
improve the rule base generated by Wang-Mendel.
"""

import numpy as np
from typing import Dict, List, Tuple, Optional, Literal, Callable
from copy import deepcopy
import warnings

# Assuming these imports work in your package structure
from ..core.fuzzification import LinguisticVariable, FuzzySet
from ..inference.systems import MamdaniSystem
from .wang_mendel import WangMendelLearning
from .metaheuristics import PSO, DE, GA, get_optimizer


class AdaptiveWangMendel:
    """
    Adaptive Wang-Mendel Learning with Metaheuristic Optimization.

    This class implements an iterative refinement process:
    - Wang-Mendel generates rules based on current membership functions
    - Metaheuristics optimize membership function parameters
    - Process repeats, allowing co-evolution of rules and membership functions

    Parameters:
        system: Initial Mamdani system with variables and terms configured
        X: Training input data (n_samples, n_features)
        y: Training output data (n_samples, n_outputs) or (n_samples,)
        task: 'auto', 'regression', or 'classification'
        optimizer: Instance of metaheuristic optimizer (PSO, DE, GA)
                   or string ('pso', 'de', 'ga')
        n_cycles: Number of adaptation cycles (default: 5)
        loss_func: Custom loss function f(y_true, y_pred) -> float
                   If None, uses MSE for regression, accuracy for classification
        early_stopping: Stop if improvement < tolerance for patience cycles
        tolerance: Minimum improvement threshold (default: 1e-4)
        patience: Number of cycles without improvement before stopping (default: 3)
        scale_classification: Scale classification outputs (default: True)
        verbose: Print progress information (default: False)

    Attributes:
        history: List of loss values per cycle
        best_system: Best system configuration found
        best_loss: Best loss achieved
        best_cycle: Cycle where best loss was achieved

    Example:
        >>> from metaheuristics import PSO
        >>> system = MamdaniSystem()
        >>> # ... configure system ...
        >>> optimizer = PSO(n_particles=20, n_iterations=50)
        >>> awm = AdaptiveWangMendel(
        ...     system, X_train, y_train,
        ...     optimizer=optimizer,
        ...     n_cycles=10,
        ...     verbose=True
        ... )
        >>> awm.fit()
        >>> predictions = awm.predict(X_test)
    """

    def __init__(self,
                 system,  # MamdaniSystem
                 X: np.ndarray,
                 y: np.ndarray,
                 task: Literal['auto', 'regression', 'classification'] = 'auto',
                 optimizer = None,  # BaseOptimizer or str
                 n_cycles: int = 5,
                 loss_func: Optional[Callable] = None,
                 early_stopping: bool = True,
                 tolerance: float = 1e-4,
                 patience: int = 3,
                 scale_classification: bool = True,
                 verbose: bool = False):

        self.initial_system = deepcopy(system)
        self.system = system
        self.X = np.atleast_2d(X)
        self.y = np.atleast_2d(y) if y.ndim == 1 else y
        self.task = task
        self.n_cycles = n_cycles
        self.loss_func = loss_func
        self.early_stopping = early_stopping
        self.tolerance = tolerance
        self.patience = patience
        self.scale_classification = scale_classification
        self.verbose = verbose

        # Setup optimizer
        if optimizer is None:
            # Import inside to avoid circular imports
            from ..optimization.metaheuristics import PSO
            self.optimizer = PSO(n_particles=20, n_iterations=50)
            if verbose:
                print("⚙️  No optimizer specified, using default PSO")
        elif isinstance(optimizer, str):
            from ..optimization.metaheuristics import get_optimizer
            self.optimizer = get_optimizer(optimizer)
        else:
            self.optimizer = optimizer

        # Variable names
        self.input_names = list(system.input_variables.keys())
        self.output_names = list(system.output_variables.keys())

        # History tracking
        self.history: List[float] = []
        self.best_system = None
        self.best_loss = float('inf')
        self.best_cycle = 0
        self.best_wm = None

        # Store initial membership function parameters
        self._store_initial_params()

    def _store_initial_params(self):
        """Store initial membership function parameters for reference."""
        self.initial_params = {}

        # Store input variable parameters
        for var_name, var in self.system.input_variables.items():
            self.initial_params[('input', var_name)] = {}
            for term_name, fuzzy_set in var.terms.items():
                # Assuming triangular/trapezoidal with .params attribute
                self.initial_params[('input', var_name)][term_name] =                     deepcopy(fuzzy_set.params) if hasattr(fuzzy_set, 'params') else None

        # Store output variable parameters
        for var_name, var in self.system.output_variables.items():
            self.initial_params[('output', var_name)] = {}
            for term_name, fuzzy_set in var.terms.items():
                self.initial_params[('output', var_name)][term_name] =                     deepcopy(fuzzy_set.params) if hasattr(fuzzy_set, 'params') else None

    def _encode_membership_params(self) -> np.ndarray:
        """
        Encode all membership function parameters into a 1D vector.
        Only encodes parameters that can be optimized (e.g., triangular: [a, b, c]).

        Returns:
            Parameter vector
        """
        params = []

        # Encode input variables
        for var_name in self.input_names:
            var = self.system.input_variables[var_name]
            for term_name in var.terms.keys():
                fuzzy_set = var.terms[term_name]
                if hasattr(fuzzy_set, 'params'):
                    params.extend(fuzzy_set.params)

        # Encode output variables
        for var_name in self.output_names:
            var = self.system.output_variables[var_name]
            for term_name in var.terms.keys():
                fuzzy_set = var.terms[term_name]
                if hasattr(fuzzy_set, 'params'):
                    params.extend(fuzzy_set.params)

        return np.array(params)

    def _decode_membership_params(self, param_vector: np.ndarray):
        """
        Decode parameter vector and update membership functions.

        Parameters:
            param_vector: Optimized parameter vector
        """
        idx = 0

        # Decode input variables
        for var_name in self.input_names:
            var = self.system.input_variables[var_name]
            for term_name in var.terms.keys():
                fuzzy_set = var.terms[term_name]
                if hasattr(fuzzy_set, 'params'):
                    n_params = len(fuzzy_set.params)
                    new_params = param_vector[idx:idx+n_params]
                    fuzzy_set.params = new_params.tolist()
                    idx += n_params

        # Decode output variables
        for var_name in self.output_names:
            var = self.system.output_variables[var_name]
            for term_name in var.terms.keys():
                fuzzy_set = var.terms[term_name]
                if hasattr(fuzzy_set, 'params'):
                    n_params = len(fuzzy_set.params)
                    new_params = param_vector[idx:idx+n_params]
                    fuzzy_set.params = new_params.tolist()
                    idx += n_params

    def _compute_bounds(self) -> np.ndarray:
        """
        Compute optimization bounds for membership function parameters.
        Uses variable universes to set reasonable bounds.

        Returns:
            Bounds array (n_params, 2) with [min, max] for each parameter
        """
        bounds = []

        # Bounds for input variables
        for var_name in self.input_names:
            var = self.system.input_variables[var_name]
            universe_min, universe_max = var.universe
            margin = (universe_max - universe_min) * 0.1

            for term_name in var.terms.keys():
                fuzzy_set = var.terms[term_name]
                if hasattr(fuzzy_set, 'params'):
                    n_params = len(fuzzy_set.params)
                    for _ in range(n_params):
                        bounds.append([universe_min - margin, universe_max + margin])

        # Bounds for output variables
        for var_name in self.output_names:
            var = self.system.output_variables[var_name]
            universe_min, universe_max = var.universe
            margin = (universe_max - universe_min) * 0.1

            for term_name in var.terms.keys():
                fuzzy_set = var.terms[term_name]
                if hasattr(fuzzy_set, 'params'):
                    n_params = len(fuzzy_set.params)
                    for _ in range(n_params):
                        bounds.append([universe_min - margin, universe_max + margin])

        return np.array(bounds)

    def _default_loss(self, y_true: np.ndarray, y_pred: np.ndarray) -> float:
        """Default loss function (MSE for regression, 1-accuracy for classification)."""
        if self.task == 'classification' or (self.task == 'auto' and y_true.shape[1] > 1):
            # Classification: use 1 - accuracy (to minimize)
            y_true_labels = np.argmax(y_true, axis=1)
            y_pred_labels = np.argmax(y_pred, axis=1) if y_pred.ndim > 1 else y_pred
            accuracy = np.mean(y_true_labels == y_pred_labels)
            return 1.0 - accuracy
        else:
            # Regression: use MSE
            return np.mean((y_true - y_pred) ** 2)

    def _evaluate_system(self, wm_learner) -> float:
        """
        Evaluate current system on training data.

        Parameters:
            wm_learner: Trained WangMendelLearning instance

        Returns:
            Loss value
        """
        try:
            predictions = wm_learner.predict(self.X)

            if self.loss_func is not None:
                loss = self.loss_func(self.y, predictions)
            else:
                # For classification, predictions are class indices, need to convert
                if wm_learner.is_classification:
                    # Convert predictions to one-hot for loss calculation
                    y_pred_onehot = np.zeros_like(self.y)
                    y_pred_onehot[np.arange(len(predictions)), predictions] = 1
                    loss = self._default_loss(self.y, y_pred_onehot)
                else:
                    loss = self._default_loss(self.y, predictions)

            return loss
        except Exception as e:
            if self.verbose:
                warnings.warn(f"Error evaluating system: {e}")
            return float('inf')

    def _objective_function(self, param_vector: np.ndarray) -> float:
        """
        Objective function for metaheuristic optimization.

        Parameters:
            param_vector: Membership function parameters

        Returns:
            Loss value to minimize
        """
        # Decode parameters and update system
        self._decode_membership_params(param_vector)

        # Clear existing rules
        self.system.rules = []

        # Step 1: Generate rules with Wang-Mendel using current membership functions
        try:
            wm = WangMendelLearning(
                self.system, 
                self.X, 
                self.y,
                task=self.task,
                scale_classification=self.scale_classification,
                verbose_init=False
            )
            wm.fit(verbose=False)

            # Step 2: Evaluate performance
            loss = self._evaluate_system(wm)

            return loss

        except Exception as e:
            if self.verbose:
                warnings.warn(f"Error in objective function: {e}")
            return float('inf')

    def fit(self) -> 'AdaptiveWangMendel':
        """
        Execute adaptive Wang-Mendel algorithm.

        Iteratively:
        1. Generate rules with Wang-Mendel
        2. Evaluate performance
        3. Optimize membership functions with metaheuristics
        4. Repeat

        Returns:
            Self (trained model)
        """
        if self.verbose:
            print("=" * 70)
            print("🔄 ADAPTIVE WANG-MENDEL ALGORITHM")
            print("=" * 70)
            print(f"📊 Dataset: {self.X.shape[0]} samples, "
                  f"{self.X.shape[1]} inputs, {self.y.shape[1]} outputs")
            print(f"🎯 Task: {self.task.upper()}")
            print(f"🔧 Optimizer: {self.optimizer.__class__.__name__}")
            print(f"🔁 Cycles: {self.n_cycles}")
            print(f"⏱️  Early stopping: {'ON' if self.early_stopping else 'OFF'}")
            if self.early_stopping:
                print(f"   Tolerance: {self.tolerance}, Patience: {self.patience}")
            print("=" * 70)

        no_improvement_count = 0

        for cycle in range(self.n_cycles):
            if self.verbose:
                print(f"
{'='*70}")
                print(f"🔄 CYCLE {cycle + 1}/{self.n_cycles}")
                print(f"{'='*70}")

            # Reset system to current best (or initial for first cycle)
            if cycle == 0:
                self.system = deepcopy(self.initial_system)
            else:
                self.system = deepcopy(self.best_system)

            # Step 1: Initial Wang-Mendel with current membership functions
            if self.verbose:
                print("
📝 Step 1: Generating rules with Wang-Mendel...")

            self.system.rules = []
            wm = WangMendelLearning(
                self.system,
                self.X,
                self.y,
                task=self.task,
                scale_classification=self.scale_classification,
                verbose_init=False
            )
            wm.fit(verbose=self.verbose)

            # Step 2: Evaluate initial performance
            if self.verbose:
                print("
📊 Step 2: Evaluating performance...")

            initial_loss = self._evaluate_system(wm)

            if self.verbose:
                print(f"   Initial loss: {initial_loss:.6f}")

            # Step 3: Optimize membership functions
            if self.verbose:
                print("
🔧 Step 3: Optimizing membership functions...")

            # Get current parameters and bounds
            initial_params = self._encode_membership_params()
            bounds = self._compute_bounds()

            if self.verbose:
                print(f"   Parameters to optimize: {len(initial_params)}")
                print(f"   Running {self.optimizer.__class__.__name__}...")

            # Run optimization
            best_params, best_loss_opt, opt_history = self.optimizer.optimize(
                self._objective_function,
                bounds,
                minimize=True,
                verbose=self.verbose
            )

            # Step 4: Update system with optimized parameters
            if self.verbose:
                print(f"
✅ Cycle {cycle + 1} completed!")
                print(f"   Initial loss: {initial_loss:.6f}")
                print(f"   Optimized loss: {best_loss_opt:.6f}")
                print(f"   Improvement: {initial_loss - best_loss_opt:.6f} "
                      f"({((initial_loss - best_loss_opt) / initial_loss * 100):.2f}%)")

            # Update history
            self.history.append(best_loss_opt)

            # Check if this is the best system so far
            if best_loss_opt < self.best_loss:
                improvement = self.best_loss - best_loss_opt
                self.best_loss = best_loss_opt
                self.best_cycle = cycle

                # Decode best parameters and save system
                self._decode_membership_params(best_params)
                self.system.rules = []
                wm_best = WangMendelLearning(
                    self.system,
                    self.X,
                    self.y,
                    task=self.task,
                    scale_classification=self.scale_classification,
                    verbose_init=False
                )
                wm_best.fit(verbose=False)
                self.best_system = deepcopy(self.system)
                self.best_wm = wm_best

                no_improvement_count = 0

                if self.verbose:
                    print(f"   🌟 New best system! (Loss: {self.best_loss:.6f})")
            else:
                no_improvement_count += 1

                if self.verbose:
                    print(f"   No improvement ({no_improvement_count}/{self.patience})")

            # Early stopping check
            if self.early_stopping and no_improvement_count >= self.patience:
                if self.verbose:
                    print(f"
⏹️  Early stopping: No improvement for {self.patience} cycles")
                break

        # Final summary
        if self.verbose:
            print(f"
{'='*70}")
            print("✅ TRAINING COMPLETED")
            print(f"{'='*70}")
            print(f"📈 Best loss: {self.best_loss:.6f} (Cycle {self.best_cycle + 1})")
            print(f"📊 Total cycles: {len(self.history)}")
            print(f"📉 Loss reduction: {self.history[0] - self.best_loss:.6f} "
                  f"({((self.history[0] - self.best_loss) / self.history[0] * 100):.2f}%)")
            print(f"{'='*70}")

        # Ensure system is set to best configuration
        self.system = self.best_system

        return self

    def predict(self, X: np.ndarray) -> np.ndarray:
        """
        Make predictions using best trained system.

        Parameters:
            X: Input array (n_samples, n_features)

        Returns:
            Predictions (format depends on task type)
        """
        if self.best_wm is None:
            raise ValueError("Model must be trained before making predictions. Call fit() first.")

        return self.best_wm.predict(X)

    def predict_proba(self, X: np.ndarray) -> np.ndarray:
        """
        Return class probabilities (classification only).

        Parameters:
            X: Input array (n_samples, n_features)

        Returns:
            Probability matrix (n_samples, n_classes)
        """
        if self.best_wm is None:
            raise ValueError("Model must be trained before making predictions. Call fit() first.")

        return self.best_wm.predict_proba(X)

    def get_history(self) -> Dict:
        """
        Get training history and statistics.

        Returns:
            Dictionary with training history
        """
        return {
            'losses': self.history,
            'best_loss': self.best_loss,
            'best_cycle': self.best_cycle,
            'n_cycles_completed': len(self.history),
            'final_improvement': self.history[0] - self.best_loss if self.history else 0,
            'relative_improvement': ((self.history[0] - self.best_loss) / self.history[0] * 100) 
                                   if self.history and self.history[0] != 0 else 0
        }

    def plot_history(self):
        """Plot training history (requires matplotlib)."""
        try:
            import matplotlib.pyplot as plt

            plt.figure(figsize=(10, 6))
            plt.plot(range(1, len(self.history) + 1), self.history, 
                    marker='o', linewidth=2, markersize=8)
            plt.axhline(y=self.best_loss, color='r', linestyle='--', 
                       label=f'Best: {self.best_loss:.6f} (Cycle {self.best_cycle + 1})')
            plt.xlabel('Cycle', fontsize=12)
            plt.ylabel('Loss', fontsize=12)
            plt.title('Adaptive Wang-Mendel Training History', fontsize=14, fontweight='bold')
            plt.grid(True, alpha=0.3)
            plt.legend(fontsize=10)
            plt.tight_layout()
            plt.show()
        except ImportError:
            print("Matplotlib not available. Install it to plot history.")
