import sys
import tempfile
from pathlib import Path
from unittest.mock import MagicMock, patch
import pytest
from loguru import logger
from palabra_ai.util.logger import (
    Library, 
    set_logging, 
    DEBUG, INFO, WARNING, ERROR, SUCCESS,
    debug, info, warning, error, critical, exception, trace, success
)


class TestLibrary:
    """Test Library class"""
    
    def test_init_defaults(self):
        """Test Library initialization with defaults"""
        lib = Library()
        
        assert lib.name == "palabra_ai"
        assert lib.level == INFO
        assert lib.handlers == []
    
    def test_init_custom_values(self):
        """Test Library initialization with custom values"""
        lib = Library(name="test_lib", level=DEBUG)
        
        assert lib.name == "test_lib"
        assert lib.level == DEBUG
        assert lib.handlers == []
    
    def test_call_method(self):
        """Test __call__ method for setting level"""
        lib = Library()
        lib(DEBUG)
        
        assert lib.level == DEBUG
    
    def test_set_level_debug(self):
        """Test set_level with debug=True"""
        lib = Library()
        lib.set_level(silent=False, debug=True)
        
        assert lib.level == DEBUG
    
    def test_set_level_silent(self):
        """Test set_level with silent=True"""
        lib = Library()
        lib.set_level(silent=True, debug=False)
        
        assert lib.level == SUCCESS
    
    def test_set_level_normal(self):
        """Test set_level with both flags False"""
        lib = Library()
        lib.set_level(silent=False, debug=False)
        
        assert lib.level == INFO
    
    def test_is_library_record_true(self):
        """Test _is_library_record with library record"""
        lib = Library(name="test_lib")
        record = {"name": "test_lib.module"}
        
        assert lib._is_library_record(record) is True
    
    def test_is_library_record_false(self):
        """Test _is_library_record with non-library record"""
        lib = Library(name="test_lib")
        record = {"name": "other_module"}
        
        assert lib._is_library_record(record) is False
    
    def test_is_library_record_no_name(self):
        """Test _is_library_record with no name field"""
        lib = Library(name="test_lib")
        record = {}
        
        assert lib._is_library_record(record) is False
    
    def test_should_log_non_library(self):
        """Test should_log for non-library records"""
        lib = Library()
        record = {"name": "other_module", "level": MagicMock(no=ERROR)}
        
        assert lib.should_log(record) is True
    
    def test_should_log_library_above_level(self):
        """Test should_log for library record above threshold"""
        lib = Library(level=INFO)
        mock_level = MagicMock()
        mock_level.no = ERROR
        record = {"name": "palabra_ai.module", "level": mock_level}
        
        assert lib.should_log(record) is True
    
    def test_should_log_library_below_level(self):
        """Test should_log for library record below threshold"""
        lib = Library(level=WARNING)
        mock_level = MagicMock()
        mock_level.no = INFO
        record = {"name": "palabra_ai.module", "level": mock_level}
        
        assert lib.should_log(record) is False
    
    def test_create_console_filter_pass(self):
        """Test create_console_filter with passing record"""
        lib = Library()
        original_filter = lambda r: True
        combined_filter = lib.create_console_filter(original_filter)
        
        # Mock should_log to return True
        with patch.object(lib, 'should_log', return_value=True):
            record = {"test": "data"}
            assert combined_filter(record) is True
    
    def test_create_console_filter_fail_lib(self):
        """Test create_console_filter with library filter failing"""
        lib = Library()
        original_filter = lambda r: True
        combined_filter = lib.create_console_filter(original_filter)
        
        # Mock should_log to return False
        with patch.object(lib, 'should_log', return_value=False):
            record = {"test": "data"}
            assert combined_filter(record) is False
    
    def test_create_console_filter_fail_original(self):
        """Test create_console_filter with original filter failing"""
        lib = Library()
        original_filter = lambda r: False
        combined_filter = lib.create_console_filter(original_filter)
        
        # Mock should_log to return True
        with patch.object(lib, 'should_log', return_value=True):
            record = {"test": "data"}
            assert combined_filter(record) is False
    
    def test_create_console_filter_no_original(self):
        """Test create_console_filter with no original filter"""
        lib = Library()
        combined_filter = lib.create_console_filter(None)
        
        # Mock should_log to return True
        with patch.object(lib, 'should_log', return_value=True):
            record = {"test": "data"}
            assert combined_filter(record) is True
    
    def test_create_file_filter(self):
        """Test create_file_filter"""
        lib = Library(name="test_lib")
        file_filter = lib.create_file_filter()
        
        # Test with library record
        lib_record = {"name": "test_lib.module"}
        assert file_filter(lib_record) is True
        
        # Test with non-library record
        other_record = {"name": "other.module"}
        assert file_filter(other_record) is False
    
    def test_cleanup_handlers_success(self):
        """Test cleanup_handlers with successful removal"""
        lib = Library()
        lib.handlers = [1, 2, 3]
        
        with patch.object(logger, 'remove') as mock_remove:
            lib.cleanup_handlers()
            
            assert mock_remove.call_count == 3
            assert lib.handlers == []
    
    def test_cleanup_handlers_with_errors(self):
        """Test cleanup_handlers with some removal errors"""
        lib = Library()
        lib.handlers = [1, 2, 3]
        
        def side_effect(handler_id):
            if handler_id == 2:
                raise ValueError("Handler not found")
        
        with patch.object(logger, 'remove', side_effect=side_effect):
            lib.cleanup_handlers()
            
            assert lib.handlers == []
    
    def test_setup_console_handler_modify_existing(self):
        """Test setup_console_handler with existing default handler"""
        lib = Library()
        
        # Mock existing handler
        mock_handler = MagicMock()
        mock_handler._filter = lambda r: True
        mock_handlers = {0: mock_handler}
        
        with patch.object(logger._core, 'handlers', mock_handlers):
            lib.setup_console_handler()
            
            # Should modify existing handler filter
            assert mock_handler._filter is not None
            # Handlers list should remain empty (no new handler added)
            assert lib.handlers == []
    
    def test_setup_console_handler_create_new(self):
        """Test setup_console_handler with no existing handler"""
        lib = Library()
        
        with patch.object(logger._core, 'handlers', {}), \
             patch.object(logger, 'add', return_value=999) as mock_add:
            
            lib.setup_console_handler()
            
            mock_add.assert_called_once()
            assert 999 in lib.handlers
    
    def test_setup_file_handler_no_file(self):
        """Test setup_file_handler with no log file"""
        lib = Library()
        
        lib.setup_file_handler(None)
        
        assert lib.handlers == []
    
    def test_setup_file_handler_with_file(self):
        """Test setup_file_handler with log file"""
        lib = Library()
        log_file = Path("/tmp/test.log")
        
        with patch.object(logger, 'add', return_value=123) as mock_add:
            lib.setup_file_handler(log_file)
            
            mock_add.assert_called_once()
            call_args, call_kwargs = mock_add.call_args
            assert call_args[0] == str(log_file.absolute())
            assert call_kwargs['level'] == DEBUG
            assert 123 in lib.handlers


class TestSetLogging:
    """Test set_logging function"""
    
    def test_set_logging_debug(self):
        """Test set_logging with debug mode"""
        from io import StringIO
        text_io = StringIO()
        with patch('palabra_ai.util.logger._lib') as mock_lib:
            set_logging(silent=False, debug=True, text_io=text_io, log_file=None)
            
            mock_lib.set_level.assert_called_once_with(False, True)
            mock_lib.cleanup_handlers.assert_called_once()
            mock_lib.setup_console_handler.assert_called_once()
            mock_lib.setup_textio_handler.assert_called_once_with(text_io)
            # setup_file_handler is not called when log_file is None
            mock_lib.setup_file_handler.assert_not_called()
    
    def test_set_logging_with_file(self):
        """Test set_logging with log file"""
        from io import StringIO
        text_io = StringIO()
        log_file = Path("/tmp/test.log")
        
        with patch('palabra_ai.util.logger._lib') as mock_lib:
            set_logging(silent=True, debug=False, text_io=text_io, log_file=log_file)
            
            mock_lib.set_level.assert_called_once_with(True, False)
            mock_lib.cleanup_handlers.assert_called_once()
            mock_lib.setup_console_handler.assert_called_once()
            mock_lib.setup_textio_handler.assert_called_once_with(text_io)
            mock_lib.setup_file_handler.assert_called_once_with(log_file)


class TestLoggerExports:
    """Test exported logger functions"""
    
    def test_direct_exports_exist(self):
        """Test that all direct exports exist and are callable"""
        exports = [debug, info, warning, error, critical, exception, trace, success]
        
        for func in exports:
            assert callable(func)
    
    def test_constants_exist(self):
        """Test that logging constants are defined"""
        assert DEBUG == 10
        assert INFO == 20
        assert SUCCESS == 25
        assert WARNING == 30
        assert ERROR == 40
        assert isinstance(DEBUG, int)
        assert isinstance(INFO, int)
        assert isinstance(WARNING, int)
        assert isinstance(ERROR, int)
        assert isinstance(SUCCESS, int)