"""
2025.10.26: change designate column type from Annotated to Optional and raw type.
"""

# import packages from default or pip library
from datetime import datetime, timedelta, timezone
from sqlmodel import SQLModel, Field

# import packages from this framework
from settings import AUTHENTICATION


# settings
TOKEN_VALID_TIME: int = AUTHENTICATION.get("token").get("token_valid_time")

# define classes
class AccessTokenMixin(SQLModel):
    access_token: str = Field(primary_key=True,
                              nullable=False,
                              unique=True)


class RefreshTokenMixin(SQLModel):
    refresh_token: str = Field(nullable=False,
                               unique=True)


class TokenValidDateTimeMixin(SQLModel):
    """
    This class is for token management.
    it can be stored on your DB as a whitelist.
    """

    issued_dt: datetime= Field(nullable=False)
    expiration_dt: datetime = Field(nullable=False)

    def __setattr__(self, name: str, value: datetime):
        if name in ["issued_at", "expiration_dt",]:
            if hasattr(self, name) and getattr(self, name) is not None:
                raise ValueError(f"[Warning] it is not allowed to edit value of column '{name}' in '{self.__class__}' directly.")

        super().__setattr__(name, value)
        return None

    def set_token_datetime(self) -> None:
        if self.issued_dt is None:
            super().__setattr__(name="issued_at", value=datetime.now(tz=timezone.utc))

        if self.expiration_dt is None:
            super().__setattr__(name="expiration_dt", value=datetime.now(tz=timezone.utc) + timedelta(minutes=TOKEN_VALID_TIME))

        return None


class JWTTokens(RefreshTokenMixin, AccessTokenMixin):
    token_type: str = Field(nullable=False,
                            default="Bearer")


# define function to control SQL event
# please follow the procedure below
# use decorator 'event.listens_for()' for event handling function
# event handler function must have 3 args: mapper, connection and target)
# in your model file, import listens_for from 'sqlalchemy.event'
# event.listens_for(target=TARGET_TABLE_CLASS_NAME, identifier=['before_insert', 'before_update',...])
def _fill_token_datetime_field(mapper, connection, target) -> None:
    target.set_token_datetime()
    return None
