"""
2025.10.26: change designate column type from Annotated to Optional and raw type.
"""

# import packages from default or pip library
import hashlib
from datetime import datetime, timezone
from sqlmodel import SQLModel, Field

# import packages from this framework
from settings import PASSWORD_POLICIES


# password encryption algorithm
ENCRYPT_TYPE: str = PASSWORD_POLICIES.get("encrypt_type")


# define Class for Common SQLModel
class LastSigninDateTimeMixin(SQLModel):
    last_signin_dt: datetime = Field(nullable=True, default=None)

    def __setattr__(self, name, value) -> None:
        if name == "last_signin_dt":
            if hasattr(self, name) and getattr(self, name) is not None:
                raise ValueError(f"[Warning] it is not allowed to edit value of column '{name}' in '{self.__class__}' directly.")

        super().__setattr__(name, value)
        return None

    def update_signin_dt(self) -> None:
        super().__setattr__(name="last_signin_dt", value=datetime.now(tz=timezone.utc))
        return None


class SigninFailMixin(SQLModel):
    is_active: bool = Field(nullable=False, default=False)
    signin_fail: int = Field(nullable=False, default=0, ge=0, le=5)

    def activate_user(self) -> None:
        super().__setattr__(name="is_active", value=True)
        self.__init_signin_fail_count()
        return None

    def deactivate_user(self) -> None:
        super().__setattr__(name="is_active", value=False)
        return None

    def add_signin_fail_count(self, limit: int) -> None:
        if self.signin_fail < limit:
            super().__setattr__(name="signin_fail", value=self.signin_fail + 1)

            if self.signin_fail == limit:
                self.deactivate_user()

        return None

    def __init_signin_fail_count(self) -> None:
        super().__setattr__(name="signin_fail", value=0)
        return None


class IsAdminMixin(SQLModel):
    is_admin: bool = Field(nullable=False, default=False)

    def grant_admin(self) -> None:
        super().__setattr__(name="is_admin", value=True)
        return None

    def revoke_admin(self) -> None:
        super().__setattr__(name="is_admin", value=False)
        return None


class PasswordMixin(SQLModel):
    password: str = Field(nullable=False, unique=False)

    def check_password(self, password: str) -> bool:
        return self.password == encrypt_password(password=password)

    def change_password(self, old_pwd: str, new_pwd: str) -> bool:
        if self.password != encrypt_password(password=old_pwd):
            return False

        super().__setattr__(name="password", value=encrypt_password(password=new_pwd))
        return True

    def encrypt_password(self, enc_type:str=ENCRYPT_TYPE) -> None:
        super().__setattr__(name="password", value=encrypt_password(password=self.password, enc_type=enc_type))
        return None


class UsernameMixin(SQLModel):
    username: str = Field(nullable=False, unique=True)


# define function for users
def encrypt_password(password:str, enc_type:str=ENCRYPT_TYPE) -> str:
    if enc_type not in hashlib.algorithms_available:
        raise ValueError(f"'{enc_type}' is not supported hash method in this application. Password will be encrypted with default hash.")

    return getattr(hashlib, enc_type)(string=password.encode("utf-8")).hexdigest()


# define function to control SQL event
# please follow the procedure below
# use decorator 'event.listens_for()' for event handling function
# event handler function must have 3 args: mapper, connection and target)
# in your model file, import listens_for from 'sqlalchemy.event'
# event.listens_for(target=TARGET_TABLE_CLASS_NAME, identifier=['before_insert', 'before_update',...])

