"""Table schemas."""

__all__ = [
    'ALIGN',
    'VARIANT'
]

import agglovar

import polars as pl

ALIGN: dict[str, pl.DataType] = {
    'chrom': pl.String,
    'pos': pl.Int64,
    'end': pl.Int64,
    'align_index': pl.Int32,
    'filter': pl.List(pl.String),
    'qry_id': pl.String,
    'qry_pos': pl.Int64,
    'qry_end': pl.Int64,
    'qry_order': pl.Int32,
    'rg': pl.String,
    'mapq': pl.UInt8,
    'is_rev': pl.Boolean,
    'flags': pl.UInt16,
    'align_ops': pl.Struct({
        'op_code': pl.List(pl.UInt8),
        'op_len': pl.List(pl.Int32)
    })
}
"""Schema of alignment tables excluding features (added by ailgn.feature)."""

VARIANT: dict[str, pl.DataType] = {
    'chrom': agglovar.schema.VARIANT['chrom'],
    'pos': agglovar.schema.VARIANT['pos'],
    'end': agglovar.schema.VARIANT['end'],
    'id': agglovar.schema.VARIANT['id'],
    'vartype': agglovar.schema.VARIANT['vartype'],
    'varsubtype': agglovar.schema.VARIANT['vartype'],  # Same as vartype, not defined in agglovar
    'varlen': agglovar.schema.VARIANT['varlen'],
    'ref': agglovar.schema.VARIANT['ref'],
    'alt': agglovar.schema.VARIANT['alt'],
    'filter': pl.List(pl.String),
    'qry_id': pl.String,
    'qry_pos': pl.Int64,
    'qry_end': pl.Int64,
    'qry_rev': pl.Boolean,
    'call_source': pl.String,
    'var_score': pl.Float32,
    'outer_ref': pl.Struct({
        'chrom': pl.String,
        'pos': pl.Int64,
        'end': pl.Int64,
    }),
    'outer_qry': pl.Struct({
        'chrom': pl.String,
        'pos': pl.Int64,
        'end': pl.Int64,
    }),
    'templ_res': pl.Boolean,
    'align_index': pl.List(pl.Int32),
    'align_index_anchor': pl.Struct({'left': pl.Int32, 'right': pl.Int32}),
    'ref_templ': pl.List(
        pl.Struct({
            'chrom': pl.String,
            'pos': pl.Int64,
            'end': pl.Int64,
            'is_rev': pl.Boolean,
        })
    ),
    'seg_n': pl.Int32,
    'trace_qry': pl.String,
    'trace_ref': pl.String,
    'align_gap': pl.Int32,
    'var_index': pl.UInt32,
    'hom_ref': pl.Struct({'up': pl.Int32, 'dn': pl.Int32}),
    'hom_qry': pl.Struct({'up': pl.Int32, 'dn': pl.Int32}),
    'derived': pl.List(pl.String),
    'discord': pl.List(pl.String),
    'inner': pl.List(pl.String),
    'inv_dup': pl.Boolean,
    'seq': agglovar.schema.VARIANT['seq'],
}
"""
Schema for variant tables.

Coordinates:
All positions (pos, end, qry_pos, and qry_end) are 0-based, half-open (BED-like coordinates).
Query coordinates are in the original query sequence coordinates even if it was reverse-complemented during alignment
(differs from SAM/BAM). By preserving the original query coordinates, it is possible to extract a variant locus
directly from the assembly sequence without firsst translating the coordinates or extracting and transforming a whole
assembly sequence (e.g. use pysam or samtools faidx to extract the variant locus, then reverse-complement the result if
if necessary). This also allows variant calls to be directly used as annotations on the assembly sequence without
transforming the coordinates.

Outer inversions:
Inversions have inner- and outer- coordinates. Inner coordinates identify the uniquely-inverted region of an inversion,
and these coordinates are found in pos, end, qry_pos, and qry_end. Outer coordinates identify the inversion including
inverted repeats flanking the uniquely inverted region, and these coordinates are found in outer_ref and outer_qry.
The outer coordinates, if defined, should encapsulated the inner coordinates. Some variant callers use the outer
coordinates to define the inversion locus. The true breakpoint is typically somewhere between the inner- and
outer-coordinates.

Fields:

    * chrom: Reference chromosome.
    * pos: Reference position (see notes above - Coordinates).
    * end: End position (see notes above - Coordinates).
    * id: Variant ID. Must be unique in the final variant table, intermediate tables may have duplicate IDs before
        variant integration resolves them.
    * vartype: Variant type, uppercase. Should be one of "SNV", "INS", "DEL", "INV", "CPX".
    * varsubtype: Variant subtype, uppercase. "TD" indicates a tandem duplication.
    * varlen: Length of the variant. Must be defined for all but SNV variants, which are 1 bp by definition.
    * ref: Reference base for SNVs.
    * alt: Alternate base for SNVs.
    * filter: List of filters applied to this variant. An empty list indicates that the variant passed all filters (i.e.
        a "PASS" variant). Filters should be uppercase. Null values should not be generated by PAV, but can be assumed
        to mean that filter status is unknown and should not be interpreted as a "PASS" variant.
    * qry_id: Query ID. Name of the assembly sequence variant was derived from.
    * qry_pos: Query position (see notes above - Coordinates).
    * qry_end: Query end position (see notes above - Coordinates).
    * qry_rev: If the sequence was reverse-complemented during alignment (see notes above - Coordinates).
    * call_source: A short string in all caps indicating how the variant was identified. "INTRA" is intra-alignment
        variant calling, "INTER" is inter-alignment variant calling from split alignment records.
    * var_score: Variant score based on a scoring model.
    * outer_ref: Outer reference region for inversions (see notes above - Outer inversions).
    * outer_qry: Outer query region for inversions (see notes above - Outer inversions).
    * align_index: List of "align_index" values from the alignment table this variant was derived from. Intra-alignment
        variants will have a single value. Inter-alignment variants will have 0 or more values and will exlude anchors.
        Indexes will be in order of the alignment chain from left to right in reference orientation.
    * align_index_anchor: For intra-alignment variants, this is a pair of "align_index" values from the alignment table
        identifying the alignments anchoring this variant, but not part of the variant itself. Anchors are in reference
        order with the first anchor left of the variant and second anchor right of the variant. A null value for either
        anchor value indicates a missing anchor, such as a variant called anchoring on one side. Inter-alignment
        variants should always have a  list, and intra-alignment variants should either exclude this field or have a
        null value for the field (i.e. no list) if they appear in a table with inter-alignment variants.
    * ref_templ: List templated sites (see notes above - Coordinates). Each templated site is a reference coordinate
        that part of the variant was templated from. For insertions, this may be a single region that was duplicated
        (insertion call itself indicates where it was inserted). For complex variants, this may be a list of regions
        that were duplicated or rearranged to form the variant.
    * seg_n: Number of distinct segments in a complex variant.
    * align_gap: Difference in the number of bases between the query and reference sequences for inter-alignment
        inversions. Large numbers suggest a more complex arrangement than a balanced inversion.
    * var_index: A unique index for variants within a callset.
    * derived: ID of a variant this variant was derived from (where applicable). If non-null, then "DERIVED" must also
        be present in "filter".
    * discord: IDs of variants this variant was discordant with. For example, a small variant inside a larger deletion
        will have the deletion variant ID in this list). If not empty, then "DISCORD" must also be present in "filter".
    * inner: IDs of a variant this variant was inside of. These are typically small variants inside aligned segments
        that are part of a larger variant (i.e. duplicated region of an insertion or complex event). If not empty, then
        "INNER" must also be present in "filter".
    * inv_dup: DUP variant is inverted relative to the reference. Only be defined for DUP variants (null otherwise).
    * seq: Variant sequence in reference orientation. Null if not defined (never "*" or other placeholders).
"""
