from typing import Optional, List
from datetime import datetime, date, time
from uuid import UUID
from sqlmodel import select
from sqlalchemy.orm import selectinload,joinedload
from healthdatalayer.models import MedicalVisit
from healthdatalayer.config.db import engines, get_session

class MedicalVisitRepository:
    def __init__(self, tenant: str):
        self.tenant = tenant
        if tenant not in engines:
            raise ValueError(f"Tenant {tenant} is not configured")
    
    def create_command(self, medical_visit: MedicalVisit) -> MedicalVisit:
        with get_session(self.tenant) as session:
            session.add(medical_visit)
            session.commit()
            session.refresh(medical_visit)
            return medical_visit
    
    def get_by_id_command(self, medical_visit_id: UUID, load_relations: bool = False) -> Optional[MedicalVisit]:
        with get_session(self.tenant) as session:
            
            if load_relations:
                statement = select(MedicalVisit).where(MedicalVisit.medical_visit_id == medical_visit_id).options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
                medical_visit = session.exec(statement).first()
               
                return medical_visit
            else:
                return session.get(MedicalVisit, medical_visit_id)
    
    def list_all_command(self, active_only: bool = True, load_relations: bool = False)->List[MedicalVisit]:
        with get_session(self.tenant) as session:
            statement = select(MedicalVisit)
            
            if load_relations:
                
                statement = select(MedicalVisit).options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
                if active_only:
                    statement = statement.where(MedicalVisit.is_active == True)
                medical_visits = session.exec(statement).all()
              
                return medical_visits
            
            statement = select(MedicalVisit)
            return session.exec(statement).all()
    
    def get_by_client_id_command(self, client_id: UUID, active_only: bool = True, load_relations: bool = False) -> List[MedicalVisit]:
        with get_session(self.tenant) as session:
            statement = select(MedicalVisit).where(MedicalVisit.client_id == client_id)
            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visits = session.exec(statement).all()
            return medical_visits
        
    def get_by_collaborator_id_command(self, collaborator_id: UUID, active_only: bool = True, load_relations: bool = False) -> List[MedicalVisit]:
        with get_session(self.tenant) as session:
            statement = select(MedicalVisit).where(MedicalVisit.collaborator_id == collaborator_id)
            
            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visits = session.exec(statement).all()
            return medical_visits
        
    def get_by_next_followup_id_command(self, next_followup_id: UUID, active_only: bool = True, load_relations: bool = False) -> List[MedicalVisit]:
        with get_session(self.tenant) as session:
            statement = select(MedicalVisit).where(MedicalVisit.next_followup_visit_id == next_followup_id)
            
            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visits = session.exec(statement).all()
            return medical_visits

    def get_by_daterange_command(self, start_date: datetime, end_date: datetime, active_only: bool = True, load_relations: bool = False) -> List[MedicalVisit]:
        with get_session(self.tenant) as session:
            statement = select(MedicalVisit).where(
                    MedicalVisit.visit_date >= start_date
                )
            
            if end_date is not None:
                statement = statement.where(MedicalVisit.visit_date <= end_date)
            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visits = session.exec(statement).all()
            return medical_visits
        
    def get_by_targetdate_command(self, target_date: datetime, active_only: bool = True, load_relations: bool = False) -> List[MedicalVisit]:
        with get_session(self.tenant) as session:

            if isinstance(target_date, str):
                try:
                    target_date = datetime.strptime(target_date, "%Y-%m-%d")
                except ValueError:
                    raise ValueError("Invalid date format")

            elif isinstance(target_date, date) and not isinstance(target_date, datetime):
                target_date = datetime.combine(target_date, time.min)

            start_of_day = datetime.combine(target_date.date(), datetime.min.time())
            end_of_day = datetime.combine(target_date.date(), datetime.max.time())

            statement = select(MedicalVisit).where(
                    MedicalVisit.visit_date >= start_of_day,
                    MedicalVisit.visit_date <= end_of_day
                )

            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visits = session.exec(statement).all()
            return medical_visits

    def get_by_collaboratorid_and_daterange_command(self, collaborator_id: UUID, start_date: datetime, end_date: datetime, active_only: bool = True, load_relations: bool = False) -> List[MedicalVisit]:
        with get_session(self.tenant) as session:
            statement = select(MedicalVisit).where(
                    MedicalVisit.collaborator_id == collaborator_id,
                    MedicalVisit.visit_date >= start_date
                )
            
            if end_date is not None:
                statement = statement.where(MedicalVisit.visit_date <= end_date)
            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visits = session.exec(statement).all()
            return medical_visits
        
    def get_by_collaboratorid_and_targetdate_command(self, collaborator_id: UUID, target_date: datetime, active_only: bool = True, load_relations: bool = False) -> List[MedicalVisit]:
        with get_session(self.tenant) as session:

            if isinstance(target_date, str):
                try:
                    target_date = datetime.strptime(target_date, "%Y-%m-%d")
                except ValueError:
                    raise ValueError("Invalid date format")

            elif isinstance(target_date, date) and not isinstance(target_date, datetime):
                target_date = datetime.combine(target_date, time.min)

            start_of_day = datetime.combine(target_date.date(), datetime.min.time())
            end_of_day = datetime.combine(target_date.date(), datetime.max.time())

            statement = select(MedicalVisit).where(
                    MedicalVisit.collaborator_id == collaborator_id,
                    MedicalVisit.visit_date >= start_of_day,
                    MedicalVisit.visit_date <= end_of_day
                )

            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visits = session.exec(statement).all()
            return medical_visits
    
    def get_by_collaboratorid_and_specificdatetime_command(self, collaborator_id: UUID, specific_datetime: datetime, active_only: bool = True, load_relations: bool = False) -> Optional[MedicalVisit]:
        with get_session(self.tenant) as session:

            statement = select(MedicalVisit).where(
                    MedicalVisit.collaborator_id == collaborator_id,
                    MedicalVisit.visit_date == specific_datetime
                )

            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visit = session.exec(statement).first()
            return medical_visit
        
    def get_first_by_clientid_command(self, client_id: UUID, active_only: bool = True, load_relations: bool = False) -> Optional[MedicalVisit]:
        with get_session(self.tenant) as session:

            statement = select(MedicalVisit).where(MedicalVisit.client_id == client_id).order_by(MedicalVisit.visit_date.asc())

            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visit = session.exec(statement).first()
            return medical_visit
    
    def get_last_by_clientid_command(self, client_id: UUID, active_only: bool = True, load_relations: bool = False) -> Optional[MedicalVisit]:
        with get_session(self.tenant) as session:

            statement = select(MedicalVisit).where(MedicalVisit.client_id == client_id).order_by(MedicalVisit.visit_date.desc())

            if active_only:
                statement = statement.where(MedicalVisit.is_active == True)
            if load_relations:
                statement = statement.options(
                    joinedload(MedicalVisit.client),
                    joinedload(MedicalVisit.collaborator),
                    joinedload(MedicalVisit.speciality),
                    selectinload(MedicalVisit.medical_diagnosis_visits),
                    selectinload(MedicalVisit.medical_recipe_visits),
                    selectinload(MedicalVisit.organ_system_reviews),
                    selectinload(MedicalVisit.physical_exams)
                )
            
            medical_visit = session.exec(statement).first()
            return medical_visit
    
    def update_command(self, medical_visit: MedicalVisit) -> MedicalVisit:
        with get_session(self.tenant) as session:
            existing_medical_visit = session.get(MedicalVisit, medical_visit.medical_visit_id)
            if not existing_medical_visit:
                raise ValueError(f"medical_visit with id {medical_visit.medical_visit_id} does not exist")
            
            for key, value in medical_visit.dict(exclude_unset=True).items():
                setattr(existing_medical_visit, key, value)
            
            bd_medical_visit =  session.merge(existing_medical_visit)
            session.commit()
            session.refresh(bd_medical_visit)
            return bd_medical_visit
        
    def delete_command(self, medical_visit_id: UUID, soft_delete: bool = False)->None:
        with get_session(self.tenant) as session:
            existing_bridge = session.get(MedicalVisit, medical_visit_id)
            if not existing_bridge:
                raise ValueError(f"MedicalVisit with id {medical_visit_id} does not exist")

            if soft_delete:
                existing_bridge.is_active = False
                session.add(existing_bridge)
            else:
                session.delete(existing_bridge)

            session.commit()
    
    def exists_by_collaboratoid_and_targetdate_command(self, collaborator_id: UUID, target_date: datetime) -> bool:
        with get_session(self.tenant) as session:
            statement = select(MedicalVisit).where(MedicalVisit.collaborator_id == collaborator_id, MedicalVisit.visit_date == target_date)
            result = session.exec(statement).first()
            return result is not None
    