"""
Manage entities in Microsoft Purview using modular Click-based commands.

Usage:
  entity create                Create a new entity
  entity read                  Read an entity
  entity update                Update an entity
  entity delete                Delete an entity
  entity --help                Show this help message and exit

Options:
  -h --help                    Show this help message and exit
"""

import json
import click
from rich.console import Console

console = Console()


@click.group()
@click.pass_context
def entity(ctx):
    """
    Manage entities in Microsoft Purview.
    """
    pass


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--ignore-relationships", is_flag=True, help="Whether to ignore relationship attributes"
)
@click.option(
    "--min-ext-info",
    is_flag=True,
    help="Whether to return minimal information for referred entities",
)
@click.pass_context
def read(ctx, guid, ignore_relationships, min_ext_info):
    """Read entity information by GUID"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity read command[/yellow]")
            console.print(f"[dim]GUID: {guid}[/dim]")
            console.print(f"[dim]Ignore Relationships: {ignore_relationships}[/dim]")
            console.print(f"[dim]Min Ext Info: {min_ext_info}[/dim]")
            console.print("[green][OK] Mock entity read completed successfully[/green]")
            return

        args = {
            "--guid": guid,
            "--ignoreRelationships": ignore_relationships,
            "--minExtInfo": min_ext_info,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityRead(args)

        if result:
            console.print("[green][OK] Entity read completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity read completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity read: {str(e)}[/red]")


@entity.command()
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing entity data",
)
@click.pass_context
def create(ctx, payload_file):
    """Create a new entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity create command[/yellow]")
            console.print(f"[dim]Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity create completed successfully[/green]")
            return

        args = {"--payloadFile": payload_file}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityCreate(args)

        if result:
            console.print("[green][OK] Entity create completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity create completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity create: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.pass_context
def delete(ctx, guid):
    """Delete an entity by GUID"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity delete command[/yellow]")
            console.print(f"[dim]GUID: {guid}[/dim]")
            console.print("[green][OK] Mock entity delete completed successfully[/green]")
            return

        args = {"--guid": guid}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityDelete(args)

        if result:
            console.print("[green][OK] Entity delete completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity delete completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity delete: {str(e)}[/red]")


@entity.command()
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing bulk entity data",
)
@click.pass_context
def bulk_create(ctx, payload_file):
    """Create multiple entities in bulk"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-create command[/yellow]")
            console.print(f"[dim]Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity bulk-create completed successfully[/green]")
            return

        args = {"--payloadFile": payload_file}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityBulkCreateOrUpdate(args)

        if result:
            console.print("[green][OK] Entity bulk-create completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity bulk-create completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-create: {str(e)}[/red]")


@entity.command(name="bulk-update")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing entities to update/create (same shape as bulk-create).",
)
@click.pass_context
def bulk_update(ctx, payload_file):
    """Bulk update/create entities from a JSON payload file (uses qualifiedName to match existing entities)."""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-update command[/yellow]")
            console.print(f"[dim]Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity bulk-update completed successfully[/green]")
            return

        args = {"--payloadFile": payload_file}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityBulkCreateOrUpdate(args)

        if result:
            console.print("[green][OK] Entity bulk-update completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity bulk-update completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-update: {str(e)}[/red]")


# === BULK OPERATIONS ===


@entity.command()
@click.option(
    "--guid", required=True, multiple=True, help="Entity GUIDs to read (can specify multiple)"
)
@click.option(
    "--ignore-relationships", is_flag=True, help="Whether to ignore relationship attributes"
)
@click.option(
    "--min-ext-info",
    is_flag=True,
    help="Whether to return minimal information for referred entities",
)
@click.pass_context
def bulk_read(ctx, guid, ignore_relationships, min_ext_info):
    """Read multiple entities by their GUIDs"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-read command[/yellow]")
            console.print(f"[dim]GUIDs: {', '.join(guid)}[/dim]")
            console.print("[green][OK] Mock entity bulk-read completed successfully[/green]")
            return

        args = {
            "--guid": list(guid),
            "--ignoreRelationships": ignore_relationships,
            "--minExtInfo": min_ext_info,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityReadBulk(args)

        if result:
            console.print("[green][OK] Entity bulk-read completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity bulk-read completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-read: {str(e)}[/red]")


@entity.command()
@click.option(
    "--guid", required=True, multiple=True, help="Entity GUIDs to delete (can specify multiple)"
)
@click.pass_context
def bulk_delete(ctx, guid):
    """Delete multiple entities by their GUIDs"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-delete command[/yellow]")
            console.print(f"[dim]GUIDs: {', '.join(guid)}[/dim]")
            console.print("[green][OK] Mock entity bulk-delete completed successfully[/green]")
            return

        args = {"--guid": list(guid)}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityDeleteBulk(args)

        if result:
            console.print("[green][OK] Entity bulk-delete completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity bulk-delete completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-delete: {str(e)}[/red]")


# === UNIQUE ATTRIBUTE OPERATIONS ===


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.option(
    "--ignore-relationships", is_flag=True, help="Whether to ignore relationship attributes"
)
@click.option(
    "--min-ext-info",
    is_flag=True,
    help="Whether to return minimal information for referred entities",
)
@click.pass_context
def read_by_attribute(ctx, type_name, qualified_name, ignore_relationships, min_ext_info):
    """Read entity by unique attributes"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity read-by-attribute command[/yellow]")
            console.print(f"[dim]Type: {type_name}, Qualified Name: {qualified_name}[/dim]")
            console.print("[green][OK] Mock entity read-by-attribute completed successfully[/green]")
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": qualified_name,
            "--ignoreRelationships": ignore_relationships,
            "--minExtInfo": min_ext_info,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityReadUniqueAttribute(args)

        if result:
            console.print("[green][OK] Entity read-by-attribute completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity read-by-attribute completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity read-by-attribute: {str(e)}[/red]")


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option(
    "--qualified-name", required=True, multiple=True, help="Qualified names (can specify multiple)"
)
@click.option(
    "--ignore-relationships", is_flag=True, help="Whether to ignore relationship attributes"
)
@click.option(
    "--min-ext-info",
    is_flag=True,
    help="Whether to return minimal information for referred entities",
)
@click.pass_context
def bulk_read_by_attribute(ctx, type_name, qualified_name, ignore_relationships, min_ext_info):
    """Read multiple entities by unique attributes"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-read-by-attribute command[/yellow]")
            console.print(
                f"[dim]Type: {type_name}, Qualified Names: {', '.join(qualified_name)}[/dim]"
            )
            console.print(
                "[green][OK] Mock entity bulk-read-by-attribute completed successfully[/green]"
            )
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": list(qualified_name),
            "--ignoreRelationships": ignore_relationships,
            "--minExtInfo": min_ext_info,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityReadBulkUniqueAttribute(args)

        if result:
            console.print("[green][OK] Entity bulk-read-by-attribute completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity bulk-read-by-attribute completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-read-by-attribute: {str(e)}[/red]")


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.pass_context
def delete_by_attribute(ctx, type_name, qualified_name):
    """Delete entity by unique attributes"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity delete-by-attribute command[/yellow]")
            console.print(f"[dim]Type: {type_name}, Qualified Name: {qualified_name}[/dim]")
            console.print("[green][OK] Mock entity delete-by-attribute completed successfully[/green]")
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": qualified_name,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityDeleteUniqueAttribute(args)

        if result:
            console.print("[green][OK] Entity delete-by-attribute completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity delete-by-attribute completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity delete-by-attribute: {str(e)}[/red]")


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing entity data",
)
@click.pass_context
def update_by_attribute(ctx, type_name, qualified_name, payload_file):
    """Update entity by unique attributes"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity update-by-attribute command[/yellow]")
            console.print(f"[dim]Type: {type_name}, Qualified Name: {qualified_name}[/dim]")
            console.print("[green][OK] Mock entity update-by-attribute completed successfully[/green]")
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": qualified_name,
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityPartialUpdateByUniqueAttribute(args)

        if result:
            console.print("[green][OK] Entity update-by-attribute completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity update-by-attribute completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity update-by-attribute: {str(e)}[/red]")


# === HEADER OPERATIONS ===


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.pass_context
def read_header(ctx, guid):
    """Read entity header information by GUID"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity read-header command[/yellow]")
            console.print(f"[dim]GUID: {guid}[/dim]")
            console.print("[green][OK] Mock entity read-header completed successfully[/green]")
            return

        args = {"--guid": [guid]}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityReadHeader(args)

        if result:
            console.print("[green][OK] Entity read-header completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity read-header completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity read-header: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option("--attr-name", required=True, help="The name of the attribute to update")
@click.option("--attr-value", required=True, help="The new value for the attribute")
@click.pass_context
def update_attribute(ctx, guid, attr_name, attr_value):
    """Update a specific attribute of an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity update-attribute command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Attribute: {attr_name}, Value: {attr_value}[/dim]")
            console.print("[green][OK] Mock entity update-attribute completed successfully[/green]")
            return

        args = {
            "--guid": [guid],
            "--attrName": attr_name,
            "--attrValue": attr_value,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityPartialUpdateAttribute(args)

        if result:
            console.print("[green][OK] Entity update-attribute completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity update-attribute completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity update-attribute: {str(e)}[/red]")


# === CLASSIFICATION OPERATIONS ===


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option("--classification-name", required=True, help="The name of the classification")
@click.pass_context
def read_classification(ctx, guid, classification_name):
    """Read specific classification for an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity read-classification command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Classification: {classification_name}[/dim]")
            console.print("[green][OK] Mock entity read-classification completed successfully[/green]")
            return

        args = {
            "--guid": [guid],
            "--classificationName": classification_name,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityReadClassification(args)

        if result:
            console.print("[green][OK] Entity read-classification completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity read-classification completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity read-classification: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.pass_context
def read_classifications(ctx, guid):
    """Read all classifications for an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity read-classifications command[/yellow]")
            console.print(f"[dim]GUID: {guid}[/dim]")
            console.print(
                "[green][OK] Mock entity read-classifications completed successfully[/green]"
            )
            return

        args = {"--guid": [guid]}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityReadClassifications(args)

        if result:
            console.print("[green][OK] Entity read-classifications completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity read-classifications completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity read-classifications: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing classification data",
)
@click.pass_context
def add_classifications(ctx, guid, payload_file):
    """Add classifications to an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity add-classifications command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity add-classifications completed successfully[/green]")
            return

        args = {
            "--guid": [guid],
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityAddClassifications(args)

        if result:
            console.print("[green][OK] Entity add-classifications completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity add-classifications completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity add-classifications: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing classification data",
)
@click.pass_context
def update_classifications(ctx, guid, payload_file):
    """Update classifications on an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity update-classifications command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Payload File: {payload_file}[/dim]")
            console.print(
                "[green][OK] Mock entity update-classifications completed successfully[/green]"
            )
            return

        args = {
            "--guid": [guid],
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityUpdateClassifications(args)

        if result:
            console.print("[green][OK] Entity update-classifications completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity update-classifications completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity update-classifications: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--classification-name", required=True, help="The name of the classification to remove"
)
@click.pass_context
def remove_classification(ctx, guid, classification_name):
    """Remove classification from an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity remove-classification command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Classification: {classification_name}[/dim]")
            console.print(
                "[green][OK] Mock entity remove-classification completed successfully[/green]"
            )
            return

        args = {
            "--guid": [guid],
            "--classificationName": classification_name,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityDeleteClassification(args)

        if result:
            console.print("[green][OK] Entity remove-classification completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity remove-classification completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity remove-classification: {str(e)}[/red]")


# === CLASSIFICATION OPERATIONS BY UNIQUE ATTRIBUTE ===


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing classification data",
)
@click.pass_context
def add_classifications_by_attribute(ctx, type_name, qualified_name, payload_file):
    """Add classifications by unique attribute"""
    try:
        if ctx.obj.get("mock"):
            console.print(
                "[yellow]🎭 Mock: entity add-classifications-by-attribute command[/yellow]"
            )
            console.print(f"[dim]Type: {type_name}, Qualified Name: {qualified_name}[/dim]")
            console.print(
                "[green][OK] Mock entity add-classifications-by-attribute completed successfully[/green]"
            )
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": qualified_name,
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityAddClassificationsByUniqueAttribute(args)

        if result:
            console.print(
                "[green][OK] Entity add-classifications-by-attribute completed successfully[/green]"
            )
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity add-classifications-by-attribute completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(
            f"[red][X] Error executing entity add-classifications-by-attribute: {str(e)}[/red]"
        )


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing classification data",
)
@click.pass_context
def update_classifications_by_attribute(ctx, type_name, qualified_name, payload_file):
    """Update classifications by unique attribute"""
    try:
        if ctx.obj.get("mock"):
            console.print(
                "[yellow]🎭 Mock: entity update-classifications-by-attribute command[/yellow]"
            )
            console.print(f"[dim]Type: {type_name}, Qualified Name: {qualified_name}[/dim]")
            console.print(
                "[green][OK] Mock entity update-classifications-by-attribute completed successfully[/green]"
            )
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": qualified_name,
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityUpdateClassificationsByUniqueAttribute(args)

        if result:
            console.print(
                "[green][OK] Entity update-classifications-by-attribute completed successfully[/green]"
            )
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity update-classifications-by-attribute completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(
            f"[red][X] Error executing entity update-classifications-by-attribute: {str(e)}[/red]"
        )


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.option(
    "--classification-name", required=True, help="The name of the classification to remove"
)
@click.pass_context
def remove_classification_by_attribute(ctx, type_name, qualified_name, classification_name):
    """Remove classification by unique attribute"""
    try:
        if ctx.obj.get("mock"):
            console.print(
                "[yellow]🎭 Mock: entity remove-classification-by-attribute command[/yellow]"
            )
            console.print(
                f"[dim]Type: {type_name}, Qualified Name: {qualified_name}, Classification: {classification_name}[/dim]"
            )
            console.print(
                "[green][OK] Mock entity remove-classification-by-attribute completed successfully[/green]"
            )
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": qualified_name,
            "--classificationName": classification_name,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityDeleteClassificationByUniqueAttribute(args)

        if result:
            console.print(
                "[green][OK] Entity remove-classification-by-attribute completed successfully[/green]"
            )
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity remove-classification-by-attribute completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(
            f"[red][X] Error executing entity remove-classification-by-attribute: {str(e)}[/red]"
        )


# === BULK CLASSIFICATION OPERATIONS ===


@entity.command()
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing bulk classification data",
)
@click.pass_context
def bulk_add_classification(ctx, payload_file):
    """Add classification to multiple entities in bulk"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-add-classification command[/yellow]")
            console.print(f"[dim]Payload File: {payload_file}[/dim]")
            console.print(
                "[green][OK] Mock entity bulk-add-classification completed successfully[/green]"
            )
            return

        args = {"--payloadFile": payload_file}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityAddClassification(args)

        if result:
            console.print("[green][OK] Entity bulk-add-classification completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity bulk-add-classification completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-add-classification: {str(e)}[/red]")


@entity.command()
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing bulk classification data",
)
@click.pass_context
def bulk_set_classifications(ctx, payload_file):
    """Set classifications on multiple entities in bulk"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-set-classifications command[/yellow]")
            console.print(f"[dim]Payload File: {payload_file}[/dim]")
            console.print(
                "[green][OK] Mock entity bulk-set-classifications completed successfully[/green]"
            )
            return

        args = {"--payloadFile": payload_file}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityBulkSetClassifications(args)

        if result:
            console.print("[green][OK] Entity bulk-set-classifications completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity bulk-set-classifications completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-set-classifications: {str(e)}[/red]")


# === LABEL OPERATIONS ===


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing label data",
)
@click.pass_context
def add_labels(ctx, guid, payload_file):
    """Add labels to an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity add-labels command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity add-labels completed successfully[/green]")
            return

        args = {
            "--guid": [guid],
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityAddLabels(args)

        if result:
            console.print("[green][OK] Entity add-labels completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity add-labels completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity add-labels: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing label data",
)
@click.pass_context
def set_labels(ctx, guid, payload_file):
    """Set labels on an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity set-labels command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity set-labels completed successfully[/green]")
            return

        args = {
            "--guid": [guid],
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entitySetLabels(args)

        if result:
            console.print("[green][OK] Entity set-labels completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity set-labels completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity set-labels: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing label data",
)
@click.pass_context
def remove_labels(ctx, guid, payload_file):
    """Remove labels from an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity remove-labels command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity remove-labels completed successfully[/green]")
            return

        args = {
            "--guid": [guid],
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityRemoveLabels(args)

        if result:
            console.print("[green][OK] Entity remove-labels completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity remove-labels completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity remove-labels: {str(e)}[/red]")


# === LABEL OPERATIONS BY UNIQUE ATTRIBUTE ===


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing label data",
)
@click.pass_context
def add_labels_by_attribute(ctx, type_name, qualified_name, payload_file):
    """Add labels by unique attribute"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity add-labels-by-attribute command[/yellow]")
            console.print(f"[dim]Type: {type_name}, Qualified Name: {qualified_name}[/dim]")
            console.print(
                "[green][OK] Mock entity add-labels-by-attribute completed successfully[/green]"
            )
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": qualified_name,
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityAddLabelsByUniqueAttribute(args)

        if result:
            console.print("[green][OK] Entity add-labels-by-attribute completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity add-labels-by-attribute completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity add-labels-by-attribute: {str(e)}[/red]")


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing label data",
)
@click.pass_context
def set_labels_by_attribute(ctx, type_name, qualified_name, payload_file):
    """Set labels by unique attribute"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity set-labels-by-attribute command[/yellow]")
            console.print(f"[dim]Type: {type_name}, Qualified Name: {qualified_name}[/dim]")
            console.print(
                "[green][OK] Mock entity set-labels-by-attribute completed successfully[/green]"
            )
            return

        args = {
            "--typeName": type_name,
            "--qualifiedName": qualified_name,
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entitySetLabelsByUniqueAttribute(args)

        if result:
            console.print("[green][OK] Entity set-labels-by-attribute completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity set-labels-by-attribute completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity set-labels-by-attribute: {str(e)}[/red]")


@entity.command()
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing bulk label data",
)
@click.pass_context
def bulk_remove_labels(ctx, payload_file):
    """Remove labels from multiple entities in bulk (by GUID)"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-remove-labels command[/yellow]")
            console.print(f"[dim]Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity bulk-remove-labels completed successfully[/green]")
            return
        args = {"--payloadFile": payload_file}
        from purviewcli.client._entity import Entity
        entity_client = Entity()
        result = entity_client.entityBulkRemoveLabels(args)
        if result:
            console.print("[green][OK] Entity bulk-remove-labels completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity bulk-remove-labels completed with no result[/yellow]")
    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-remove-labels: {str(e)}[/red]")


@entity.command()
@click.option("--type-name", required=True, help="The name of the entity type")
@click.option("--qualified-name", required=True, help="The qualified name of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing label data",
)
@click.pass_context
def bulk_remove_labels_by_attribute(ctx, type_name, qualified_name, payload_file):
    """Remove labels from multiple entities in bulk (by unique attribute)"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-remove-labels-by-attribute command[/yellow]")
            console.print(f"[dim]Type: {type_name}, Qualified Name: {qualified_name}, Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity bulk-remove-labels-by-attribute completed successfully[/green]")
            return
        args = {"--typeName": type_name, "--qualifiedName": qualified_name, "--payloadFile": payload_file}
        from purviewcli.client._entity import Entity
        entity_client = Entity()
        result = entity_client.entityBulkRemoveLabelsByUniqueAttribute(args)
        if result:
            console.print("[green][OK] Entity bulk-remove-labels-by-attribute completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity bulk-remove-labels-by-attribute completed with no result[/yellow]")
    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-remove-labels-by-attribute: {str(e)}[/red]")


# === BUSINESS METADATA OPERATIONS ===


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing business metadata",
)
@click.option(
    "--is-overwrite", is_flag=True, help="Whether to overwrite existing business metadata"
)
@click.pass_context
def add_business_metadata(ctx, guid, payload_file, is_overwrite):
    """Add or update business metadata to an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity add-business-metadata command[/yellow]")
            console.print(f"[dim]GUID: {guid}, Overwrite: {is_overwrite}[/dim]")
            console.print(
                "[green][OK] Mock entity add-business-metadata completed successfully[/green]"
            )
            return

        args = {
            "--guid": [guid],
            "--payloadFile": payload_file,
            "--isOverwrite": is_overwrite,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityAddOrUpdateBusinessMetadata(args)

        if result:
            console.print("[green][OK] Entity add-business-metadata completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity add-business-metadata completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity add-business-metadata: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option("--bm-name", required=True, help="The business metadata name")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing business metadata attributes",
)
@click.pass_context
def add_business_metadata_attributes(ctx, guid, bm_name, payload_file):
    """Add or update business metadata attributes"""
    try:
        if ctx.obj.get("mock"):
            console.print(
                "[yellow]🎭 Mock: entity add-business-metadata-attributes command[/yellow]"
            )
            console.print(f"[dim]GUID: {guid}, BM Name: {bm_name}[/dim]")
            console.print(
                "[green][OK] Mock entity add-business-metadata-attributes completed successfully[/green]"
            )
            return

        args = {
            "--guid": [guid],
            "--bmName": bm_name,
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityAddOrUpdateBusinessMetadataAttributes(args)

        if result:
            console.print(
                "[green][OK] Entity add-business-metadata-attributes completed successfully[/green]"
            )
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity add-business-metadata-attributes completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(
            f"[red][X] Error executing entity add-business-metadata-attributes: {str(e)}[/red]"
        )


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document specifying business metadata to remove",
)
@click.pass_context
def remove_business_metadata(ctx, guid, payload_file):
    """Remove business metadata from an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity remove-business-metadata command[/yellow]")
            console.print(f"[dim]GUID: {guid}[/dim]")
            console.print(
                "[green][OK] Mock entity remove-business-metadata completed successfully[/green]"
            )
            return

        args = {
            "--guid": [guid],
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityRemoveBusinessMetadata(args)

        if result:
            console.print("[green][OK] Entity remove-business-metadata completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity remove-business-metadata completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity remove-business-metadata: {str(e)}[/red]")


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.option("--bm-name", required=True, help="The business metadata name")
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document specifying attributes to remove",
)
@click.pass_context
def remove_business_metadata_attributes(ctx, guid, bm_name, payload_file):
    """Remove business metadata attributes"""
    try:
        if ctx.obj.get("mock"):
            console.print(
                "[yellow]🎭 Mock: entity remove-business-metadata-attributes command[/yellow]"
            )
            console.print(f"[dim]GUID: {guid}, BM Name: {bm_name}[/dim]")
            console.print(
                "[green][OK] Mock entity remove-business-metadata-attributes completed successfully[/green]"
            )
            return

        args = {
            "--guid": [guid],
            "--bmName": bm_name,
            "--payloadFile": payload_file,
        }

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityRemoveBusinessMetadataAttributes(args)

        if result:
            console.print(
                "[green][OK] Entity remove-business-metadata-attributes completed successfully[/green]"
            )
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity remove-business-metadata-attributes completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(
            f"[red][X] Error executing entity remove-business-metadata-attributes: {str(e)}[/red]"
        )


@entity.command(name="import-business-metadata")
@click.option(
    "--bm-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid business metadata CSV file",
)
@click.pass_context
def import_business_metadata(ctx, bm_file):
    """Import business metadata in bulk from CSV"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity import-business-metadata command[/yellow]")
            console.print(f"[dim]BM File: {bm_file}[/dim]")
            console.print(
                "[green][OK] Mock entity import-business-metadata completed successfully[/green]"
            )
            return

        args = {"--bmFile": bm_file}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityImportBusinessMetadata(args)

        if result:
            console.print("[green][OK] Entity import-business-metadata completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity import-business-metadata completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(f"[red][X] Error executing entity import-business-metadata: {str(e)}[/red]")


@entity.command()
@click.pass_context
def get_business_metadata_template(ctx):
    """Get sample template for business metadata"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity get-business-metadata-template command[/yellow]")
            console.print(
                "[green][OK] Mock entity get-business-metadata-template completed successfully[/green]"
            )
            return

        args = {}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityGetBusinessMetadataTemplate(args)

        if result:
            console.print(
                "[green][OK] Entity get-business-metadata-template completed successfully[/green]"
            )
            console.print(json.dumps(result, indent=2))
        else:
            console.print(
                "[yellow][!] Entity get-business-metadata-template completed with no result[/yellow]"
            )

    except Exception as e:
        console.print(
            f"[red][X] Error executing entity get-business-metadata-template: {str(e)}[/red]"
        )


# === COLLECTION OPERATIONS ===


@entity.command()
@click.option(
    "--payload-file",
    required=True,
    type=click.Path(exists=True),
    help="File path to a valid JSON document containing entities to move and target collection",
)
@click.pass_context
def move_to_collection(ctx, payload_file):
    """Move entities to a target collection"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity move-to-collection command[/yellow]")
            console.print(f"[dim]Payload File: {payload_file}[/dim]")
            console.print("[green][OK] Mock entity move-to-collection completed successfully[/green]")
            return

        args = {"--payloadFile": payload_file}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityMoveEntitiesToCollection(args)

        if result:
            console.print("[green][OK] Entity move-to-collection completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity move-to-collection completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity move-to-collection: {str(e)}[/red]")


# === SAMPLE OPERATIONS ===


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.pass_context
def read_sample(ctx, guid):
    """Get sample data for an entity"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity read-sample command[/yellow]")
            console.print(f"[dim]GUID: {guid}[/dim]")
            console.print("[green][OK] Mock entity read-sample completed successfully[/green]")
            return

        args = {"--guid": [guid]}

        from purviewcli.client._entity import Entity

        entity_client = Entity()
        result = entity_client.entityReadSample(args)

        if result:
            console.print("[green][OK] Entity read-sample completed successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity read-sample completed with no result[/yellow]")

    except Exception as e:
        console.print(f"[red][X] Error executing entity read-sample: {str(e)}[/red]")


@entity.command()
@click.option("--csv-file", required=True, type=click.Path(exists=True), help="CSV file with GUID and classificationName columns")
@click.option("--batch-size", default=100, help="Batch size for API calls")
@click.pass_context
def bulk_classify_csv(ctx, csv_file, batch_size):
    """Bulk classify entities from a CSV file (guid, classificationName columns)"""
    import pandas as pd
    import tempfile
    from purviewcli.client._entity import Entity
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-classify-csv command[/yellow]")
            console.print(f"[dim]CSV File: {csv_file}[/dim]")
            console.print("[green][OK] Mock entity bulk-classify-csv completed successfully[/green]")
            return

        df = pd.read_csv(csv_file)
        if "guid" not in df.columns or "classificationName" not in df.columns:
            console.print("[red][X] CSV must contain 'guid' and 'classificationName' columns[/red]")
            return
        entity_client = Entity()
        total = len(df)
        success, failed = 0, 0
        errors = []
        for i in range(0, total, batch_size):
            batch = df.iloc[i:i+batch_size]
            payload = {
                "entities": [
                    {
                        "guid": str(row["guid"]),
                        "classifications": [{"typeName": str(row["classificationName"])}]
                    }
                    for _, row in batch.iterrows()
                ]
            }
            with tempfile.NamedTemporaryFile(mode="w", suffix=".json", delete=False) as tmpf:
                import json
                json.dump(payload, tmpf, indent=2)
                tmpf.flush()
                payload_file = tmpf.name
            try:
                args = {"--payloadFile": payload_file}
                result = entity_client.entityAddClassification(args)
                if result and (not isinstance(result, dict) or result.get("status") != "error"):
                    success += len(batch)
                else:
                    failed += len(batch)
                    errors.append(f"Batch {i//batch_size+1}: {result}")
            except Exception as e:
                failed += len(batch)
                errors.append(f"Batch {i//batch_size+1}: {str(e)}")
            finally:
                import os
                os.remove(payload_file)
        console.print(f"[green][OK] Bulk classification completed. Success: {success}, Failed: {failed}[/green]")
        if errors:
            console.print("[red]Errors:[/red]")
            for err in errors:
                console.print(f"[red]- {err}[/red]")
    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-classify-csv: {str(e)}[/red]")


# === BULK ENTITY CSV OPERATIONS ===

@entity.command()
@click.option("--csv-file", required=True, type=click.Path(exists=True), help="CSV file with entity attributes (typeName, qualifiedName, ...)")
@click.option("--batch-size", default=100, help="Batch size for API calls")
@click.option("--dry-run", is_flag=True, help="Preview entities to be created without making changes")
@click.option("--error-csv", type=click.Path(), help="CSV file to write failed rows (optional)")
@click.pass_context
def bulk_create_csv(ctx, csv_file, batch_size, dry_run, error_csv):
    """Bulk create entities from a CSV file (typeName, qualifiedName, ... columns)"""
    import pandas as pd
    import tempfile
    import os
    from purviewcli.client._entity import Entity
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-create-csv command[/yellow]")
            console.print(f"[dim]CSV File: {csv_file}[/dim]")
            console.print("[green][OK] Mock entity bulk-create-csv completed successfully[/green]")
            return

        df = pd.read_csv(csv_file)
        if "typeName" not in df.columns or "qualifiedName" not in df.columns:
            console.print("[red][X] CSV must contain at least 'typeName' and 'qualifiedName' columns[/red]")
            return
        entity_client = Entity()
        total = len(df)
        success, failed = 0, 0
        errors = []
        failed_rows = []
        for i in range(0, total, batch_size):
            batch = df.iloc[i:i+batch_size]
            # Map each row to the correct Purview entity format
            from purviewcli.client._entity import map_flat_entity_to_purview_entity
            entities = [map_flat_entity_to_purview_entity(row) for _, row in batch.iterrows()]
            payload = {"entities": entities}
            if dry_run:
                console.print(f"[blue]DRY RUN: Would create batch {i//batch_size+1} with {len(batch)} entities[/blue]")
                continue
            with tempfile.NamedTemporaryFile(mode="w", suffix=".json", delete=False) as tmpf:
                import json
                json.dump(payload, tmpf, indent=2)
                tmpf.flush()
                payload_file = tmpf.name
            try:
                args = {"--payloadFile": payload_file}
                result = entity_client.entityCreateBulk(args)
                if result and (not isinstance(result, dict) or result.get("status") != "error"):
                    success += len(batch)
                else:
                    failed += len(batch)
                    errors.append(f"Batch {i//batch_size+1}: {result}")
                    failed_rows.extend(batch.to_dict(orient="records"))
            except Exception as e:
                failed += len(batch)
                errors.append(f"Batch {i//batch_size+1}: {str(e)}")
                failed_rows.extend(batch.to_dict(orient="records"))
            finally:
                os.remove(payload_file)
        console.print(f"[green]SUCCESS: Bulk create completed. Success: {success}, Failed: {failed}[/green]")
        if errors:
            console.print("[red]Errors:[/red]")
            for err in errors:
                console.print(f"[red]- {err}[/red]")
        if error_csv and failed_rows:
            pd.DataFrame(failed_rows).to_csv(error_csv, index=False)
            console.print(f"[yellow]WARNING: Failed rows written to {error_csv}[/yellow]")
    except Exception as e:
        console.print(f"[red]ERROR: Error executing entity bulk-create-csv: {str(e)}[/red]")


@entity.command()
@click.option("--csv-file", required=True, type=click.Path(exists=True), help="CSV file with GUID and attributes to update")
@click.option("--batch-size", default=100, help="Batch size for API calls")
@click.option("--dry-run", is_flag=True, help="Preview entities to be updated without making changes")
@click.option("--error-csv", type=click.Path(), help="CSV file to write failed rows (optional)")
@click.pass_context
def bulk_update_csv(ctx, csv_file, batch_size, dry_run, error_csv):
    """Bulk update entities from a CSV file (guid, attributes...)"""
    import pandas as pd
    import tempfile
    import os
    import json
    from purviewcli.client._entity import Entity
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-update-csv command[/yellow]")
            console.print(f"[dim]CSV File: {csv_file}[/dim]")
            console.print("[green][OK] Mock entity bulk-update-csv completed successfully[/green]")
            return

        df = pd.read_csv(csv_file)
        if df.empty:
            console.print("[yellow]No rows found in CSV. Exiting.[/yellow]")
            return

        entity_client = Entity()
        total = len(df)
        success, failed = 0, 0
        errors = []
        failed_rows = []

        # Determine mode:
        # - If CSV has both 'typeName' and 'qualifiedName' -> map rows to Purview entities and call bulk create-or-update
        # - Else if CSV has 'guid' -> build guid-based payloads (preferred for partial attribute updates)
        has_type_qn = ("typeName" in df.columns and "qualifiedName" in df.columns)
        has_guid = "guid" in df.columns

        for i in range(0, total, batch_size):
            batch = df.iloc[i : i + batch_size]

            if has_type_qn:
                # Map flat rows to Purview entity objects using helper
                from purviewcli.client._entity import map_flat_entity_to_purview_entity

                entities = [map_flat_entity_to_purview_entity(row) for _, row in batch.iterrows()]
                payload = {"entities": entities}

                if dry_run:
                    console.print(f"[blue]DRY RUN: Would bulk-create/update batch {i//batch_size+1} with {len(batch)} entities[/blue]")
                    continue

                with tempfile.NamedTemporaryFile(mode="w", suffix=".json", delete=False, encoding="utf-8") as tmpf:
                    json.dump(payload, tmpf, indent=2)
                    tmpf.flush()
                    payload_file = tmpf.name

                try:
                    args = {"--payloadFile": payload_file}
                    result = entity_client.entityCreateBulk(args)
                    if result and (not isinstance(result, dict) or result.get("status") != "error"):
                        success += len(batch)
                    else:
                        failed += len(batch)
                        errors.append(f"Batch {i//batch_size+1}: {result}")
                        failed_rows.extend(batch.to_dict(orient="records"))
                except Exception as e:
                    failed += len(batch)
                    errors.append(f"Batch {i//batch_size+1}: {str(e)}")
                    failed_rows.extend(batch.to_dict(orient="records"))
                finally:
                    try:
                        os.remove(payload_file)
                    except Exception:
                        pass

            elif has_guid:
                # Build guid-based updates. If the CSV contains only guid + attr columns, we'll attempt to perform
                # partial attribute updates by calling entityPartialUpdateAttribute where possible.
                # If a row contains multiple attributes, we will call entityCreateBulk with a payload containing
                # the guid and attributes (server supports bulk create-or-update by guid in some endpoints).

                # Normalize rows into dicts
                rows = [row.to_dict() for _, row in batch.iterrows()]

                # Attempt to detect single-attribute update pattern: columns [guid, attrName, attrValue]
                if set(["guid", "attrName", "attrValue"]).issubset(set(batch.columns)):
                    # perform per-guid partial updates in batch
                    for r in rows:
                        guid = str(r.get("guid"))
                        attr_name = r.get("attrName")
                        attr_value = r.get("attrValue")
                        if pd.isna(guid) or pd.isna(attr_name):
                            failed += 1
                            failed_rows.append(r)
                            continue
                        if dry_run:
                            console.print(f"[blue]DRY RUN: Would update GUID {guid} set {attr_name}={attr_value}[/blue]")
                            success += 1
                            continue
                        try:
                            args = {"--guid": [guid], "--attrName": attr_name, "--attrValue": attr_value}
                            result = entity_client.entityPartialUpdateAttribute(args)
                            if result and (not isinstance(result, dict) or result.get("status") != "error"):
                                success += 1
                            else:
                                failed += 1
                                errors.append(f"GUID {guid}: {result}")
                                failed_rows.append(r)
                        except Exception as e:
                            failed += 1
                            errors.append(f"GUID {guid}: {str(e)}")
                            failed_rows.append(r)

                else:
                    # Fallback: call bulk create-or-update with guid included in each entity object.
                    # Map each row into an entity dict keeping non-null columns.
                    entities = []
                    for r in rows:
                        if pd.isna(r.get("guid")):
                            failed_rows.append(r)
                            failed += 1
                            continue
                        ent = {k: v for k, v in r.items() if pd.notnull(v)}
                        # ensure guid is string under top-level 'guid' field for server bulk endpoints
                        ent["guid"] = str(ent.get("guid"))
                        entities.append(ent)

                    if not entities:
                        continue

                    payload = {"entities": entities}
                    if dry_run:
                        console.print(f"[blue]DRY RUN: Would bulk-update (by guid) batch {i//batch_size+1} with {len(entities)} entities[/blue]")
                        success += len(entities)
                        continue

                    with tempfile.NamedTemporaryFile(mode="w", suffix=".json", delete=False, encoding="utf-8") as tmpf:
                        json.dump(payload, tmpf, indent=2)
                        tmpf.flush()
                        payload_file = tmpf.name

                    try:
                        args = {"--payloadFile": payload_file}
                        # Use the create-or-update bulk endpoint - server will use guid when present
                        result = entity_client.entityCreateBulk(args)
                        if result and (not isinstance(result, dict) or result.get("status") != "error"):
                            success += len(entities)
                        else:
                            failed += len(entities)
                            errors.append(f"Batch {i//batch_size+1}: {result}")
                            failed_rows.extend(batch.to_dict(orient="records"))
                    except Exception as e:
                        failed += len(entities)
                        errors.append(f"Batch {i//batch_size+1}: {str(e)}")
                        failed_rows.extend(batch.to_dict(orient="records"))
                    finally:
                        try:
                            os.remove(payload_file)
                        except Exception:
                            pass

            else:
                console.print(f"[red][X] CSV must contain either (typeName and qualifiedName) or guid column[/red]")
                return

        console.print(f"[green][OK] Bulk update completed. Success: {success}, Failed: {failed}[/green]")
        if errors:
            console.print("[red]Errors:[/red]")
            for err in errors:
                console.print(f"[red]- {err}[/red]")
        if error_csv and failed_rows:
            pd.DataFrame(failed_rows).to_csv(error_csv, index=False)
            console.print(f"[yellow]WARNING: Failed rows written to {error_csv}[/yellow]")
    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-update-csv: {str(e)}[/red]")


@entity.command()
@click.option("--csv-file", required=True, type=click.Path(exists=True), help="CSV file with GUIDs to delete")
@click.option("--batch-size", default=100, help="Batch size for API calls")
@click.option("--dry-run", is_flag=True, help="Preview entities to be deleted without making changes")
@click.option("--error-csv", type=click.Path(), help="CSV file to write failed rows (optional)")
@click.pass_context
def bulk_delete_csv(ctx, csv_file, batch_size, dry_run, error_csv):
    """Bulk delete entities from a CSV file (guid column)"""
    import pandas as pd
    import os
    from purviewcli.client._entity import Entity
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity bulk-delete-csv command[/yellow]")
            console.print(f"[dim]CSV File: {csv_file}[/dim]")
            console.print("[green][OK] Mock entity bulk-delete-csv completed successfully[/green]")
            return

        df = pd.read_csv(csv_file)
        if "guid" not in df.columns:
            console.print("[red][X] CSV must contain 'guid' column[/red]")
            return
        entity_client = Entity()
        total = len(df)
        success, failed = 0, 0
        errors = []
        failed_rows = []
        for i in range(0, total, batch_size):
            batch = df.iloc[i:i+batch_size]
            guids = [str(row["guid"]) for _, row in batch.iterrows() if pd.notnull(row["guid"])]
            if dry_run:
                console.print(f"[blue]DRY RUN: Would delete batch {i//batch_size+1} with {len(guids)} entities[/blue]")
                continue
            try:
                args = {"--guid": guids}
                result = entity_client.entityDeleteBulk(args)
                if result and (not isinstance(result, dict) or result.get("status") != "error"):
                    success += len(guids)
                else:
                    failed += len(guids)
                    errors.append(f"Batch {i//batch_size+1}: {result}")
                    failed_rows.extend(batch.to_dict(orient="records"))
            except Exception as e:
                failed += len(guids)
                errors.append(f"Batch {i//batch_size+1}: {str(e)}")
                failed_rows.extend(batch.to_dict(orient="records"))
        console.print(f"[green][OK] Bulk delete completed. Success: {success}, Failed: {failed}[/green]")
        if errors:
            console.print("[red]Errors:[/red]")
            for err in errors:
                console.print(f"[red]- {err}[/red]")
        if error_csv and failed_rows:
            pd.DataFrame(failed_rows).to_csv(error_csv, index=False)
            console.print(f"[yellow][X] Failed rows written to {error_csv}[/yellow]")
    except Exception as e:
        console.print(f"[red][X] Error executing entity bulk-delete-csv: {str(e)}[/red]")


# === AUDIT OPERATIONS ===


@entity.command()
@click.option("--guid", required=True, help="The globally unique identifier of the entity")
@click.pass_context
def audit(ctx, guid):
    """Get audit events for an entity by GUID"""
    try:
        if ctx.obj.get("mock"):
            console.print("[yellow]🎭 Mock: entity audit command[/yellow]")
            console.print(f"[dim]GUID: {guid}[/dim]")
            console.print("[green][OK] Mock entity audit completed successfully[/green]")
            return
        args = {"--guid": guid}
        from purviewcli.client._entity import Entity
        entity_client = Entity()
        result = entity_client.entityReadAudit(args)
        if result:
            console.print("[green][OK] Entity audit events retrieved successfully[/green]")
            console.print(json.dumps(result, indent=2))
        else:
            console.print("[yellow][!] Entity audit completed with no result[/yellow]")
    except Exception as e:
        console.print(f"[red][X] Error executing entity audit: {str(e)}[/red]")


@entity.command()
@click.option('--type-name', required=False, help='Filter by entity typeName (e.g., DataSet, DataProduct)')
@click.option('--limit', default=100, help='Maximum number of entities to return')
def list(type_name, limit):
    """List entities in Microsoft Purview."""
    try:
        from purviewcli.client._search import Search
        search_client = Search()
        
        # Create search query payload with proper filter structure
        search_payload = {
            "keywords": "*",
            "limit": limit,
        }
        
        # Only add filter if type_name is specified
        if type_name:
            search_payload["filter"] = {
                "entityType": type_name  # Send as string, not array
            }
        # If no type specified, don't include filter at all
        
        # Convert to args format expected by searchQuery
        search_args = {
            "--payloadFile": None,
            "--payload": json.dumps(search_payload)
        }
        
        results = search_client.searchQuery(search_args)
        from rich.console import Console
        console = Console()
        console.print(json.dumps(results, indent=2))
    except Exception as e:
        from rich.console import Console
        console = Console()
        console.print(f"[red][X] Error executing entity list: {str(e)}[/red]")


@entity.command("bulk-delete-optimized")
@click.argument("guids", nargs=-1, required=True)
@click.option("--bulk-size", type=int, default=50, 
              help="Assets per bulk delete request (Microsoft recommended: 50)")
@click.option("--max-parallel", type=int, default=10, 
              help="Maximum parallel deletion jobs")
@click.option("--throttle-ms", type=int, default=200, 
              help="Throttle delay between API calls (milliseconds)")
@click.option("--batch-throttle-ms", type=int, default=800, 
              help="Throttle delay between batches (milliseconds)")
@click.option("--dry-run", is_flag=True, 
              help="Show what would be deleted without actually deleting")
@click.option("--continuous", is_flag=True, 
              help="Continue until all assets in collection are deleted")
@click.option("--collection-name", 
              help="Collection name for continuous deletion mode")
@click.pass_context
def bulk_delete_optimized(ctx, guids, bulk_size, max_parallel, throttle_ms, 
                         batch_throttle_ms, dry_run, continuous, collection_name):
    """
    Optimized bulk delete with mathematical precision (equivalent to Remove-PurviewAsset-Batch.ps1)
    
    Features:
    - Mathematical optimization for perfect efficiency
    - Parallel processing with controlled throttling
    - Continuous deletion mode for large collections
    - Reliable counting and progress tracking
    - Microsoft's recommended 50 assets per bulk request
    """
    try:
        from rich.console import Console
        import math
        
        console = Console()

        # Mathematical optimization display
        if len(guids) > 0:
            total_assets = len(guids)
            assets_per_job = math.ceil(total_assets / max_parallel)
            api_calls_per_job = math.ceil(assets_per_job / bulk_size)
            total_api_calls = api_calls_per_job * max_parallel
            
            console.print(f"[blue][*] Mathematical Optimization Analysis:[/blue]")
            console.print(f"   [INFO] Total Assets: {total_assets}")
            console.print(f"   🔄 Parallel Jobs: {max_parallel}")
            console.print(f"   📦 Assets per Job: {assets_per_job}")
            console.print(f"   🚀 Bulk Size: {bulk_size}")
            console.print(f"   📞 API Calls per Job: {api_calls_per_job}")
            console.print(f"   📈 Total API Calls: {total_api_calls}")
            
            # Check for perfect division (like PowerShell mathematical optimization)
            if total_assets % (max_parallel * bulk_size) == 0:
                console.print(f"[green]✨ Perfect mathematical division achieved! Zero waste.[/green]")
            else:
                waste_assets = (total_api_calls * bulk_size) - total_assets
                console.print(f"[yellow][!] Mathematical waste: {waste_assets} empty slots in final requests[/yellow]")

        if continuous and collection_name:
            deleted_count = _continuous_collection_deletion(
                ctx, collection_name, bulk_size, max_parallel, 
                throttle_ms, batch_throttle_ms, dry_run
            )
        else:
            deleted_count = _execute_optimized_bulk_delete(
                ctx, list(guids), bulk_size, max_parallel, 
                throttle_ms, batch_throttle_ms, dry_run
            )
        
        console.print(f"[green][OK] {'Would delete' if dry_run else 'Successfully deleted'} {deleted_count} assets[/green]")

    except Exception as e:
        from rich.console import Console
        console = Console()
        console.print(f"[red][X] Error in bulk-delete-optimized: {str(e)}[/red]")


@entity.command("bulk-delete-from-collection")
@click.argument("collection-name")
@click.option("--bulk-size", type=int, default=50, 
              help="Assets per bulk delete request (Microsoft recommended: 50)")
@click.option("--max-parallel", type=int, default=10, 
              help="Maximum parallel deletion jobs")
@click.option("--batch-size", type=int, default=1000, 
              help="Assets to process per batch cycle")
@click.option("--throttle-ms", type=int, default=200, 
              help="Throttle delay between API calls (milliseconds)")
@click.option("--dry-run", is_flag=True, 
              help="Show what would be deleted without actually deleting")
@click.confirmation_option(prompt="Are you sure you want to delete all assets in this collection?")
@click.pass_context
def bulk_delete_from_collection(ctx, collection_name, bulk_size, max_parallel, 
                               batch_size, throttle_ms, dry_run):
    """
    Delete all assets from a collection using continuous deletion strategy        
    Features:
    - Continuous deletion until collection is empty
    - Mathematical optimization for each batch
    - Progress tracking and estimation
    - Handles 500K+ assets efficiently
    """
    try:
        from rich.console import Console
        
        console = Console()
        console.print(f"[blue]🎯 Starting continuous deletion for collection: {collection_name}[/blue]")
        
        deleted_count = _continuous_collection_deletion(
            ctx, collection_name, bulk_size, max_parallel, 
            throttle_ms, 800, dry_run, batch_size
        )
        
        console.print(f"[green][OK] Collection cleanup complete: {'Would delete' if dry_run else 'Deleted'} {deleted_count} total assets[/green]")

    except Exception as e:
        from rich.console import Console
        console = Console()
        console.print(f"[red][X] Error in bulk-delete-from-collection: {str(e)}[/red]")


@entity.command("count-assets")
@click.argument("collection-name")
@click.option("--by-type", is_flag=True, help="Group count by asset type")
@click.option("--include-relationships", is_flag=True, help="Include relationship counts")
@click.pass_context
def count_assets(ctx, collection_name, by_type, include_relationships):
    """
    Count assets in a collection with detailed breakdown
    
    """
    try:
        from rich.console import Console
        from rich.table import Table
        
        console = Console()
        console.print(f"[blue][INFO] Counting assets in collection: {collection_name}[/blue]")
        
        # Get asset count using search API
        total_count = _get_collection_asset_count(collection_name)
        
        console.print(f"[green][OK] Total assets: {total_count}[/green]")

        if by_type:
            type_counts = _get_asset_type_breakdown(collection_name)
            _display_type_breakdown(type_counts)

        if include_relationships:
            rel_count = _get_relationship_count(collection_name)
            console.print(f"[blue]🔗 Total relationships: {rel_count}[/blue]")

    except Exception as e:
        from rich.console import Console
        console = Console()
        console.print(f"[red][X] Error in count-assets: {str(e)}[/red]")


@entity.command("analyze-performance")
@click.option("--bulk-size", type=int, default=50, help="Bulk size to analyze")
@click.option("--max-parallel", type=int, default=10, help="Parallel jobs to analyze")
@click.option("--asset-count", type=int, default=1000, help="Total assets for analysis")
@click.pass_context
def analyze_performance(ctx, bulk_size, max_parallel, asset_count):
    """
    Analyze bulk deletion performance with mathematical optimization
    """
    try:
        from rich.console import Console
        from rich.table import Table
        import math
        
        console = Console()
        console.print("[blue]📈 Performance Analysis[/blue]")
        
        # Mathematical calculations (from PowerShell scripts)
        assets_per_job = math.ceil(asset_count / max_parallel)
        api_calls_per_job = math.ceil(assets_per_job / bulk_size)
        total_api_calls = api_calls_per_job * max_parallel
        
        # Time estimations (based on PowerShell measurements)
        avg_api_time_ms = 1500  # Average API call time
        throttle_time_ms = 200  # Throttle between calls
        total_time_per_call = avg_api_time_ms + throttle_time_ms
        
        estimated_time_seconds = (total_api_calls * total_time_per_call) / 1000
        estimated_time_minutes = estimated_time_seconds / 60
        estimated_time_hours = estimated_time_minutes / 60

        # Create performance table
        table = Table(title="Performance Analysis")
        table.add_column("Metric", style="cyan")
        table.add_column("Value", style="green")
        table.add_column("Details", style="yellow")

        table.add_row("Total Assets", f"{asset_count:,}", "Assets to process")
        table.add_row("Parallel Jobs", f"{max_parallel}", "Concurrent deletion jobs")
        table.add_row("Bulk Size", f"{bulk_size}", "Assets per API call")
        table.add_row("Assets per Job", f"{assets_per_job}", f"{asset_count} ÷ {max_parallel}")
        table.add_row("API Calls per Job", f"{api_calls_per_job}", f"{assets_per_job} ÷ {bulk_size}")
        table.add_row("Total API Calls", f"{total_api_calls}", f"{api_calls_per_job} × {max_parallel}")
        table.add_row("Estimated Time", f"{estimated_time_hours:.1f} hours", f"{estimated_time_minutes:.1f} minutes")
        
        # Efficiency calculation
        theoretical_minimum_calls = math.ceil(asset_count / bulk_size)
        efficiency = (theoretical_minimum_calls / total_api_calls) * 100
        table.add_row("Efficiency", f"{efficiency:.1f}%", f"{theoretical_minimum_calls} minimum calls")

        console.print(table)

        # Recommendations (from PowerShell optimization experience)
        console.print("\n[blue]💡 Optimization Recommendations:[/blue]")
        
        if asset_count % (max_parallel * bulk_size) == 0:
            console.print("[green][OK] Perfect mathematical division - optimal configuration![/green]")
        else:
            # Calculate optimal configurations
            optimal_configs = _calculate_optimal_configs(asset_count, bulk_size)
            console.print("[yellow]💡 Consider these optimal configurations:[/yellow]")
            for config in optimal_configs[:3]:
                console.print(f"   • {config['parallel']} parallel jobs: {config['efficiency']:.1f}% efficiency")

    except Exception as e:
        from rich.console import Console
        console = Console()
        console.print(f"[red][X] Error in analyze-performance: {str(e)}[/red]")


# === ENHANCED BULK OPERATION FUNCTIONS ===

def _execute_optimized_bulk_delete(ctx, guids, bulk_size, max_parallel, throttle_ms, batch_throttle_ms, dry_run):
    """
    Execute optimized bulk delete with parallel processing
    (Core logic from PowerShell Remove-PurviewAsset-Batch.ps1)
    """
    from rich.console import Console
    from rich.progress import Progress, SpinnerColumn, TextColumn, BarColumn, TaskProgressColumn
    import concurrent.futures
    import math
    import time
    
    console = Console()
    
    if not guids:
        return 0

    total_assets = len(guids)
    deleted_count = 0

    if dry_run:
        console.print(f"[yellow][*] DRY RUN: Would delete {total_assets} assets[/yellow]")
        return total_assets

    from purviewcli.client._entity import Entity
    entity_client = Entity()

    # Split GUIDs into job batches
    assets_per_job = math.ceil(total_assets / max_parallel)
    job_batches = []
    
    for i in range(max_parallel):
        start_idx = i * assets_per_job
        end_idx = min(start_idx + assets_per_job, total_assets)
        if start_idx < total_assets:
            job_batches.append(guids[start_idx:end_idx])

    console.print(f"[blue]🚀 Starting {len(job_batches)} parallel deletion jobs...[/blue]")

    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        BarColumn(),
        TaskProgressColumn(),
        console=console
    ) as progress:
        
        task = progress.add_task("[red]Deleting assets...", total=total_assets)
        
        # Execute parallel deletions
        with concurrent.futures.ThreadPoolExecutor(max_workers=max_parallel) as executor:
            future_to_batch = {
                executor.submit(_delete_batch_job, entity_client, batch, bulk_size, throttle_ms, i): batch
                for i, batch in enumerate(job_batches)
            }
            
            for future in concurrent.futures.as_completed(future_to_batch):
                batch = future_to_batch[future]
                try:
                    batch_deleted = future.result()
                    deleted_count += batch_deleted
                    progress.update(task, advance=batch_deleted)
                    
                    # Batch throttle
                    if batch_throttle_ms > 0:
                        time.sleep(batch_throttle_ms / 1000)
                        
                except Exception as e:
                    console.print(f"[red][X] Batch deletion failed: {str(e)}[/red]")

    return deleted_count


def _delete_batch_job(entity_client, guid_batch, bulk_size, throttle_ms, job_id):
    """
    Execute a single batch job (parallel worker function)
    """
    import time
    
    deleted_in_job = 0
    
    # Split batch into bulk delete chunks
    for i in range(0, len(guid_batch), bulk_size):
        bulk_guids = guid_batch[i:i + bulk_size]
        
        try:
            # Execute bulk delete API call
            args = {"--guid": bulk_guids}
            result = entity_client.entityDeleteBulk(args)
            
            if result:
                deleted_in_job += len(bulk_guids)
            
            # Throttle between API calls
            if throttle_ms > 0 and i + bulk_size < len(guid_batch):
                time.sleep(throttle_ms / 1000)
                
        except Exception as e:
            from rich.console import Console
            console = Console()
            console.print(f"[red][X] Job {job_id} bulk delete failed: {str(e)}[/red]")
    
    return deleted_in_job


def _continuous_collection_deletion(ctx, collection_name, bulk_size, max_parallel, throttle_ms, batch_throttle_ms, dry_run, batch_size=1000):
    """
    Continuous deletion strategy for large collections
    """
    from rich.console import Console
    
    console = Console()
    total_deleted = 0
    iteration = 1

    console.print(f"[blue]🔄 Starting continuous deletion for collection: {collection_name}[/blue]")

    while True:
        console.print(f"\n[blue]📅 Iteration {iteration}: Finding assets to delete...[/blue]")
        
        # Get next batch of assets from collection
        asset_guids = _get_collection_assets_batch(collection_name, batch_size)
        
        if not asset_guids:
            console.print("[green][OK] No more assets found - collection is clean![/green]")
            break
        
        found_count = len(asset_guids)
        console.print(f"[blue][INFO] Found {found_count} assets in iteration {iteration}[/blue]")
        
        if dry_run:
            console.print(f"[yellow][*] DRY RUN: Would delete {found_count} assets[/yellow]")
            total_deleted += found_count
        else:
            # Execute optimized deletion for this batch
            deleted_in_iteration = _execute_optimized_bulk_delete(
                ctx, asset_guids, bulk_size, max_parallel, 
                throttle_ms, batch_throttle_ms, False
            )
            
            total_deleted += deleted_in_iteration
            console.print(f"[green][OK] Iteration {iteration}: Deleted {deleted_in_iteration}/{found_count} assets[/green]")
            console.print(f"[blue]📈 Running total: {total_deleted} assets deleted[/blue]")
        
        iteration += 1
        
        # Break after reasonable number of iterations in dry-run
        if dry_run and iteration > 5:
            console.print("[yellow][*] DRY RUN: Simulated 5 iterations[/yellow]")
            break

    return total_deleted


def _get_collection_assets_batch(collection_name, batch_size):
    """
    Get a batch of asset GUIDs from a collection
    (Would integrate with search API)
    """
    # Placeholder - would use search API to get actual asset GUIDs
    # For testing, return mock data that decreases over iterations
    import random
    mock_count = random.randint(0, min(batch_size, 100))
    return [f"mock-guid-{i}" for i in range(mock_count)]


def _get_collection_asset_count(collection_name):
    """Get total asset count for a collection"""
    # Placeholder - would use search API
    return 1500  # Mock count


def _get_asset_type_breakdown(collection_name):
    """Get asset count breakdown by type"""
    # Placeholder - would use search API with type filters
    return {
        "DataSet": 450,
        "Table": 320,
        "Column": 580,
        "Process": 150
    }


def _get_relationship_count(collection_name):
    """Get relationship count for collection"""
    # Placeholder - would use relationship API
    return 2340


def _display_type_breakdown(type_counts):
    """Display asset type breakdown in a table"""
    from rich.table import Table
    from rich.console import Console
    
    console = Console()
    table = Table(title="Asset Type Breakdown")
    table.add_column("Asset Type", style="cyan")
    table.add_column("Count", style="green")
    table.add_column("Percentage", style="yellow")

    total = sum(type_counts.values())
    
    for asset_type, count in sorted(type_counts.items(), key=lambda x: x[1], reverse=True):
        percentage = (count / total) * 100 if total > 0 else 0
        table.add_row(asset_type, f"{count:,}", f"{percentage:.1f}%")
    
    table.add_row("[bold]Total[/bold]", f"[bold]{total:,}[/bold]", "[bold]100.0%[/bold]")
    console.print(table)


def _calculate_optimal_configs(asset_count, bulk_size):
    """
    Calculate optimal parallel job configurations
    (Mathematical optimization from PowerShell)
    """
    import math
    
    configs = []
    
    for parallel_jobs in range(1, 21):  # Test 1-20 parallel jobs
        assets_per_job = math.ceil(asset_count / parallel_jobs)
        api_calls_per_job = math.ceil(assets_per_job / bulk_size)
        total_api_calls = api_calls_per_job * parallel_jobs
        
        theoretical_minimum = math.ceil(asset_count / bulk_size)
        efficiency = (theoretical_minimum / total_api_calls) * 100
        
        configs.append({
            'parallel': parallel_jobs,
            'efficiency': efficiency,
            'total_calls': total_api_calls,
            'waste': total_api_calls - theoretical_minimum
        })
    
    # Sort by efficiency (descending)
    return sorted(configs, key=lambda x: x['efficiency'], reverse=True)


# Make the entity group available for import
__all__ = ["entity"]
