"""Defs for Validators"""

# pylint: disable=line-too-long

from __future__ import annotations

import functools
import hashlib
from dataclasses import dataclass, field
from enum import Enum
from typing import ClassVar, Dict, List, Optional, Tuple, Union

from ruamel.yaml import CommentedMap

from opendapi.defs import OpenDAPIEntity
from opendapi.utils import (
    build_location_without_repo_from_fullpath,
    convert_ruemelyaml_commentedmap_to_dict,
    has_underlying_model_changed,
)


class ValidationError(Exception):
    """Exception raised for validation errors"""


class MultiValidationError(ValidationError):
    """Exception raised for multiple validation errors"""

    def __init__(self, errors: List[str], prefix_message: str = None):
        self.errors = errors
        self.prefix_message = prefix_message

    def __str__(self):
        return (
            f"\n\n{self.prefix_message}\n\n"
            + f"Found {len(self.errors)} errors:\n\n"
            + "\n\n".join(self.errors)
        )


class FileSet(Enum):
    """Enum for the file set"""

    ORIGINAL = "original"
    GENERATED = "generated"
    MERGED = "merged"


class IntegrationType(Enum):
    """Enum for the integration type"""

    STATIC = "static"
    RUNTIME = "runtime"
    DBT = "dbt"


@dataclass
class CollectedFile:  # pylint: disable=too-many-instance-attributes
    """class for the collect result"""

    original: Optional[Dict | CommentedMap]
    generated: Optional[Dict | CommentedMap]
    merged: Optional[Dict | CommentedMap]
    filepath: str
    commit_sha: Optional[str]
    entity: OpenDAPIEntity
    root_dir: str
    additional_metadata_from_generated: Optional[Dict] = None
    generate_skipped: bool = False

    @property
    def server_filepath(self) -> str:
        """Return the server filepath"""
        return build_location_without_repo_from_fullpath(self.root_dir, self.filepath)

    def _ensure_dict(self, value: Dict | CommentedMap | None) -> Dict | None:
        """Ensure the value is a dict"""
        return (
            convert_ruemelyaml_commentedmap_to_dict(value)
            if isinstance(value, CommentedMap)
            else value
        )

    @property
    def for_server(self) -> Dict:
        """Return the object as a dict"""
        # NOTE: this loses out on comment info, but the server doesnt use it nor
        #       write anyway
        return {
            "original": self._ensure_dict(self.original),
            "generated": self._ensure_dict(self.generated),
            "merged": self._ensure_dict(self.merged),
            "filepath": self.server_filepath,
            "commit_sha": self.commit_sha,
            "entity": self.entity.value,
            "additional_metadata_from_generated": self.additional_metadata_from_generated,
            "generate_skipped": self.generate_skipped,
        }

    @property
    def as_json(self) -> Dict:
        """Return the object as a dict"""
        # NOTE: this loses comment info, later should check if we need it still
        #       for local writing
        return {
            "original": self._ensure_dict(self.original),
            "generated": self._ensure_dict(self.generated),
            "merged": self._ensure_dict(self.merged),
            "filepath": self.filepath,
            "commit_sha": self.commit_sha,
            "entity": self.entity.value,
            "root_dir": self.root_dir,
            "additional_metadata_from_generated": self.additional_metadata_from_generated,
            "generate_skipped": self.generate_skipped,
        }

    @classmethod
    def from_dict(cls, dict_: Dict) -> CollectedFile:
        """Create a CollectedFile from a json object"""
        return cls(
            original=dict_["original"],
            generated=dict_["generated"],
            merged=dict_["merged"],
            filepath=dict_["filepath"],
            commit_sha=dict_["commit_sha"],
            entity=OpenDAPIEntity(dict_["entity"]),
            root_dir=dict_["root_dir"],
            additional_metadata_from_generated=dict_.get(
                "additional_metadata_from_generated"
            ),
            generate_skipped=dict_.get("generate_skipped", False),
        )

    def reconcile(  # pylint: disable=too-many-branches
        self,
        other: Optional[CollectedFile],
    ) -> CollectedFile:
        """
        Reconcile two CollectedFiles, choosing actually generated values.
        This is useful for multi-runtime setups, where some runtimes may
        run and be skipped, or run the same generation multiple times.
        """
        if not other:
            return self

        if self.filepath != other.filepath:
            raise ValueError("Cannot reconcile files with different filepaths")

        if self.entity is not other.entity:
            raise ValueError("Cannot reconcile files with different entities")

        if self.commit_sha != other.commit_sha:
            raise ValueError("Cannot reconcile files with different commit SHAs")

        if self.original != other.original:
            raise ValueError("Cannot reconcile files with different original contents")

        # choose the appropriate generated
        if self.generate_skipped == other.generate_skipped:
            if (
                self.generated != other.generated
                or self.additional_metadata_from_generated
                != other.additional_metadata_from_generated
            ):
                raise ValueError(
                    "Cannot reconcile files with different generated contents or metadatas"
                )
            generate_skipped = self.generate_skipped
            generated = self.generated
            additional_metadata_from_generated = self.additional_metadata_from_generated
        elif self.generate_skipped:
            generate_skipped = other.generate_skipped
            generated = other.generated
            additional_metadata_from_generated = (
                other.additional_metadata_from_generated
            )
        else:
            generate_skipped = self.generate_skipped
            generated = self.generated
            additional_metadata_from_generated = self.additional_metadata_from_generated

        # do the same for merged
        if self.generate_skipped == other.generate_skipped:
            if self.merged != other.merged:
                raise ValueError(
                    "Cannot reconcile files with different merged contents"
                )
            merged = self.merged
        elif self.generate_skipped:
            merged = other.merged
        else:
            merged = self.merged

        return CollectedFile(
            original=self.original,
            generated=generated,
            merged=merged,
            filepath=self.filepath,
            commit_sha=self.commit_sha,
            entity=self.entity,
            additional_metadata_from_generated=additional_metadata_from_generated,
            generate_skipped=generate_skipped,
            root_dir=self.root_dir,
        )


class CICDFlowType(str, Enum):
    """Enum for the type of pull request entity"""

    GOVERNED_DELETION = "governed_deletion"
    GOVERNED_ANALYZE = "governed_analyze"
    GOVERNED_DO_NOT_ANALYZE = "governed_do_not_analyze"
    GOVERNED_DO_NOT_ANALYZE_REQUIRES_PRE = "governed_do_not_analyze_requires_pre"
    UNGOVERNED_OBSERVE = "ungoverned_observe"
    UNGOVERNED_DO_NOT_OBSERVE = "ungoverned_do_not_observe"

    @staticmethod
    def portal_relevant_types() -> frozenset[CICDFlowType]:
        """Get the types for the portal"""
        return frozenset(
            (
                CICDFlowType.GOVERNED_DELETION,
                CICDFlowType.GOVERNED_ANALYZE,
                CICDFlowType.GOVERNED_DO_NOT_ANALYZE_REQUIRES_PRE,
            )
        )


# NOTE: dbt mainline may have some weird considerations due to skips(?) - I know we do this for base, need
#       to think about head...
# NOTE: dbt generated none means we whitelist all of them for now...
IGNORE_DELETION_FILEPATH_HASHES = {
    "9ab265827e2284f5c3469aad47ce1a034597c24afa6c7b41b2fe0f00f599c06a1f7f4a5f021d71f33dc9b20bbe05e9e1300e0a53279d2d3377fb3023c7ee66ef",
    "558e3342b783d9e93305a88313da697bf547682009f5c2bac46024a6fe20aa3605cb2449c1f139c04942e399d772320c6bdd4307356870bdc3d2af41208cb16f",
    "82531c9c38bfd8c727eaee70a26b3450e734052421c44e0d4ef3485fe5d5b44b3381faeef122d8cc8e00d34fac03137db39697b57ea60560710d7c59d697395b",
    "bd77813cc6fd7390866039d19a6d93bd3fe586b9f96dd28edf95e6ec191e44f60348ed5c671a6dabd9f5f338bdfedbd2ef2b1c9beeed44a5fb65f2c48fbc6b15",
    "ef0b9b89d1d198db0175b9c9df36d797b4f0262d273263d7a1c09e31bbc2d99a460d7a29890c8adea4baea4abb7b98fe3eaf61d2cffe8be6e25e1799dca1d87d",
    "11d3594f2811acd15a6e6c5da224d15e5121ca410f0f0c3996cebc67c10f70a027a398e4e54a8c679d3ec28f7c4ae019f15bccbc633b332c74d1f38a3f9c562f",
    "4168e87ae4514a77e57bf3b06230f9424e0684b0e8550cd561779fd76b8e7e49723d39049a1634476b63a755f954c58bf68b19076e1757c8e465d45d1ca2c037",
    "13d05c52cb74dd228b47f561259aafa4b3a9067a51f86198d7688f0ee5a1cd81ec7871a7b7eb7a8dfb64d7d8da9625026a8cc673f8b63ebfde121313fc403d77",
    "4edfeab1454d28f6ec248052a2c865d5d18fadc3684a22a8a6f27342cc8cdd85666fc5d4ab7b8f3d699700a7656807e583848ae6ebb02c672e605626ee40d11d",
    "9049122c68e6ef1a439d3316b66bdbd1311f5bf83f883f25270e865be1c7723cbb9dc9327cc9ad914ecf22a74c9b12562e282c40d6a8ca905ddcd00463765b2f",
    "ff04ca6dad8a2dd50ff064223d18df9c0c8faa21c2fc9a2e8e70fc358ddf2c7d2281564c320a9c1fa140b8b7df709dae8ffc0ea9bd2f9b8e3148afc36bd66ac7",
    "714eca5e22999f1df34d56699c632df81d223548a1d777d1ff6a77c3965ca8d4b47c0da85793d520c62d35fde7eec716b5546fef4ac7c6fce526f62d23cf7404",
    "c47555871587d6ab98489b554d2efa05073332c99405654a621cba1d5e08f222342b9000e1f72c8afbffd419c85d77262955ac053b0067e261a7e726738b3f6c",
    "78c5305d3b6e1673ddd7308b2dd2dcc6ccb1308992ac0abcbd6dbf7f0936d626c47cfd285ba4b4ba2a9bdbd635877e18973301d0863e82a4a23368a141f81567",
    "c3612c41ed22dde79686983886d9823f7c272d50866373fb83478f87abc735cbab879a1939de0e753501f65f80cfc447e633e978ad4841337baf96bab42c5a17",
    "aad784308ebb296fccc15ca405369d8abd1c96a49c2d7bbf48927ae3332b8bb6651f7507b982091153451e7b52968ac3328bc983786f09c122bfff26939c8b9d",
    "d95521a7611d63737d05f803b4addb8cdc57790f99cfe27d3a891773ea2f9bcbc61aaee04fe7bff407b2f81e2530d20559d3c2e5a2b14d4be2201b898e2b8680",
    "f4164ca1ed1ad113ba65876362a9213807eae02dd1ba30eea239ffa23457af98fc0eeecb029b3171e79be8daf90eed7366e2cc5cc5383837844214b9e81c150d",
    "99dcd05635eb6ccde60247246f255dd328559fa119898d533141466ef41f4ef9d1f51ecf973f3f2a31999af847ee411421b4ae98ec9cb29ae00ae772fc15225d",
    "d19c7a0266d1830745b48cbeb876f667f81b311ccad7c1594fe0cb0d2ff70abaffb327694a951f27e57415eb08eb816e961f625e1b234e49f4a6a573a0ca0548",
    "e73d309129d0c6bc1370e4ed66479aa04a9c736fa79ae774de918cd251c7292e5e347d5761b415a741eb57479b4dbddc988992e19a474286d08e198edef71150",
    "b1fb0328f6fee9facc9f731561780a8dde77c154aade890be80ec8742de1488b10e0fa52e3182c2e6e182bf658d7798633d0129b8795261ba29ebe24a49ce615",
    "11b7ec116a29fd8d5a31775e5ec6b3ada5696d5a1a247ca6558aeee0b6c1bb4d450e6a3efbc282cbe05a28a78bf9d03868e4fe85a604b34dc28607f1dd6bc826",
    "ae2a9c386f714e8fa1aaa6f765c11ad1f1b30547a27c0f72f06ffff3e0257d8b901a529123db3fcf616de3ba5c98040262c6c2f12edc45428fc7ee4f5053a57f",
    "de7929ea7b718ed122db06c7d11f27b16159c2b07c6140b6ccdc413c49db5ebc1415a365a4c3463f9f9e01ec63801398ea3ea17fa0e83ab5ed71faca310ea84e",
    "c6cf7a0081c589f01036b0e7fa18b5478dc41f1ff4c72f223e5d994fa56cdf4c18b48f54d31a4d9fc67de7b734ba71abafc2d85f6828e4a3df4e32741a79bed5",
    "20a2d26d5b6dfe984fef9de38aff3f1f512215464dbb13b7fbbef0508b9d562c156323794430c9c70d5467128d49dc8f9056cef57c57a98d3c9fa32621ac5cbb",
    "d8ba46ad122d93d7b82202b473f27ed8c7c7d11ce0ea4aa867ff4b4ac3bc5092900cb2404618fec76b24535684afb8ec00b2483a42d724216f02c6d4a55f13f1",
    "4f05705c4fe9c780a536f0656cc17c2edc84d98eccf7a94b3421d3b7f552e664743b45af2324e0b53652ec3dfd2e79799ba62655ca68d2017f9594af6a22b478",
    "c069bcdf001adb2c65e972f5249dc0a2df6dee2086fc30c2652a297cab40c95ced6aeac6be6e34a31a37a93b98045e8ab52ee2ee9f2e4fbf8baef2dddce80f51",
    "89540d74e628a939a3c28a1d939128be365b997d6c50aab57c626b6138b11a89b6a87fe5be895962c004d3040b40b3fdd60e90f22993d0eb0a94daf3ff25b5d3",
    "288f06501e494420465ce873024cdf3bd41238bc8c4501735f050deb1f29522eeadb281502db622ed61ba236023d7d05de5b6c48e4b1b5a42470691f965e38ef",
    "978fc733687ecbc1bf9ddcc301dc1eea051f42d9b2a4e3ca4922b2b98b40b2e5c6f77dc9f805b64f12ce53ab5c77a0172c3228fa2886635dd2484cf16de5005d",
    "48fc825aa8c1e5641b2e7fc42cb8774f868531fdb21c5700e84af245e647fbb1fd82b78377af694a2d868d409d6ef2214d5e7a2b64127309bc53ca45b3fd0aa8",
    "98a06a0a3788de6688a2a52996aa47dde78f135e1f1d96a41e5c14d9e32ee42cc24183b57b6433ffddd5e6d3662645a0562d90aa4bf1a95c0b4575f8740c48d4",
    "567ee543250782e8bb44c9cb7090e235df0087e36fdfd9e2353935eb2d7ff51e773733c47ebb5409a9665fd266b63db45b1d3b684f84632d6ba8f737f974957b",
    "3313784296ec9b5b739bcb00efdd658c23738822adddc2f85d166edf40d760df51e783390c82856f52dcd6e89539f942af8aa6b09b5622ac2a988d86f9427ebe",
    "bbfc4cd7f2ef90b38c1bdcdedf472ed07e79f3db2771bbf321bff5ec1fdd1d65566f4d8d30ea8e54b14f4f6f1d5e2cff8f72014aceedeef3257c3c16f7c2b821",
    "272fb2f86b83be032651f5c260f164971d5b1972478155b6444e46a64c8713375be1438816dff61752bcc2c50fef7c4c17039d6827c0ce2f4f0e638fcd4a19f3",
    "10cd3f8c332576ed12c0a5a5c42b8c19636df35889c188e9940cedb3e428833fe00161ebc36f52814420aeee864aa7dde18736207a7fe82fc38ee43a1df93d7f",
    "77ec2a253e4571e029573defa63fd219796f292a38ec5c077e66fe443d4b88a95f24955c45289c4e54b36f1dc78421ed782d242a35e9937798b8d15dd0e62ad4",
    "5caaa0970c58f15a3bd29afb921945a2d4fd64a02fd3e69bad0cb485b43023799a2b341bcf118d0dc0e66a4fb9e7a4c381b76c5a6b7c8c22bc7eba220c15eab7",
    "4b9ab6786b879a5143d3f9e472f4019ac8348a7892c9a25bc983eab2b57f5767b9b0ac8499ba01315aea191c495b160bbfeea463d432c3621cd02497578922f8",
    "eb064186a9b55e81c41add5dedf15ca5d470baf21c50abdbaeb045d311330702ebe23d4425709b9f1b3ef8c4f95f80c360986f7a8768ee394cbad01ed025417d",
    "8d92df82c7165bc2088f4086e279003520bd85631b69c42a60913ff85bad53ba3444108b5b73db7d55d15e58412d2c181d5ad3855a185af37fb029f862113802",
    "a12366bdd384c62e6b0c6be4bd9f0064ce609a9693335fee7c75cdc14b455fa2599ad71e8444a6f9290bababa12f05b4caed092adc371aff730774af25ce2062",
    "25ab7b53e94a6c4f2fa894f4b562966c76dbe509c164c2bad5ff5c701c5db0226456bb97a355c488f6daf2408e95b3fba61a7bee76379dab6f59f67990a8bb36",
    "ee0470778ec4c6c7aedfc417eb01f0a341f57257fce8d08dbd6afc00d6de77e6b0154b11c6ccbf1452904d86639be14bdec905edf2feb7ccb6afbb478887afe3",
    "f4a3ae25aee8cb71348e33f41688a22bd7b433bf1c52bf48957e83b6d1c479a28fb127b4ae845d12b1399079fd4a029ff519c10b2410812191bc035c2f287a35",
    "2f0297e18a3306ded88638ba8502acde47d115eb2f3425134aba3f57d57bd3d07ace3c8215609f6c3cd0798dccdb0ccc4e2eeb7ca42f7c7d81a83ac0c9f3a7c0",
    "ba019f6a1e1cc0cc8740d62691c71abcafcf50e3a40d35bbc1de97248fe961596151d6907a54376a9aa7c38caf311b7a3de5b9d6223960eb9486fd621cfb4368",
    "ae3b591fc9df8e92f7179baee86824fe2f04d369f93bcadac89dc0ded4b8f32a1522f308011f1ad7e96799927c6a24559f7b3601b05a60277cb0fd2490c4e0f4",
    "e93c9b141be899e1651b4965c653afd79f31af21ffeeaa5b56a039cc99eab9cd7d22487f6d2a708da14e2d34a7026fade7e8aa842bb664148d63f62bce1d27b4",
    "6e26c7c67b29ac37ec2620b01ed1400715928bf798b4f86b1ec44692efc5e65fdb6584f8429918f3d17009e85bb85ebff2c982b596578bbcbbeea7fc78b1de36",
    "ec8efa670959c1424d94e819aea0056c3445f75dcc9f8f234de6456571abb3f49506754622f0e8c1354111f68bca1b9db5d7b08e1e481a6f34f997f54112a80c",
    "c5201c122d090f0ba215e6437c03179b97add4b188e2f892c4dc533a2c1b7122cbb59f04488d339c7f3f4c98859a0085d7f4342c521dcddd271e746ddd5d6511",
    "b5fb0460667ed8cb02960d4f0641d02e161b16718a948a5bbfee0f260aaca000d0115a3ae08d0c9c24c24204c85a0bc6f53487d3ea331e47c67a0dcb2d0dcecf",
    "c551ea672cf8762d1cb67e65ee19b57eb7f49cbdaf0b179a765cd598a17e4c8331f95a1aa1eced18b7e1ec1fb66392fe0cb78d2e0916e38acdcb202e83c00d36",
    "4c95d8b4529788ab556e18e03cb66114861cdd5ba3c6a5cb7d1f1436165ac5ca1601fef95086977b6bb8a654863c6f719811d0da5c291cba014e826e148cb9c7",
    "c3fb3b46ee6e165b9c6d7d3102ef9e13180d1c526274453ab4936603512114d4643d19bfce84e59f23be4c413ed01bc72d6e2b0019397bdf2b785aa5ed2c0801",
    "1dd562e62b602878f69cc5a73bd2022e93f5b116631004337b61a62685147c3f81dbc7e37e5d1788304c58f902ce6cc73eb17ea14745bda9279c6f4fcc27260c",
    "a546dc17b1f27c3cd96a0f90d7124d785766b738f13c378a18845569be6cca32da6f2802e16259db39a31ec0df37a82ca3fba03ba7e8c683901d5a8dbd6cfb9d",
    "784811b12e5f0585c0b1058182e498d34582fab133f99210e4cade086952921a85f94e59cefd79ad35cef61ec480a113071ef4bd04fe1d7f65d806150a2c5cdd",
    "f081f1d20a0d57959398ad15b54541e6281c2122fe1e0c0e20387391f6ac0a80d9a06383eaafff3c2ffb4ccabd8e23a084667328aae6e89d33fae5fc8f8b1d25",
    "d4491b8749e6f78c2735d9df4587d3b7aefe0be3a01af1065d99704b213e1715b08595d3351bfb690c7bbbd38fa8d3a2754a6747aa11cecf66f1adb168814fb0",
    "c54aca5c47e6b0b1a24553f9ce9063b8a41efa6477c05c8092586c766b79e230100da588bb69fc8f87ddfb6e564f735294bd7ee0aa2459845a13f7050cb0cea7",
    "cf1e4c83f0e1e7e7172523bd06a27f6c1c7cf2979843c52da4c5b9089e9f97254c36263ce6531da34d33f853ea13f75f7f5a5e6118181fa7b6915aa4b6d9e1f2",
    "5a4f78040ace4f47d25d5a6fd4e93036517ed57b24409602a553b621fe36d223eb3012203236b89f0ffa9d555ad99af42b099b43cd3c94e8a8feebe2c38a35c0",
    "ab7c5c7551cd7585f14b1d4701bfd5f1f305876b6e153a8464aec537838a2346a4d8bd4c4bbe6a5b0bdb406cca6845efac887052f3bc0f4219d285755b00a51f",
    "a83393772d43f4d1b7aa8018bcdfd105eb17bda885403f76bd199969e3e3df9c6807fbeb73b61c49676cbaa2db0990c20b94d5152cf1bb0f31d0fdfe9805ee50",
    "39552ed59b0c9e69350db9b236a87acc943f7ca3bdd724f9cde691145f15366600bf57a308529490902ba12436b00f49368e5870187ff9aaabb0fd482bdbe6cb",
    "85bcf825b7e7a3f7687d5baca82b67e50ea39bd179a7e0ea9263acc7e61e921598baf427e5f6cbe5a62de0da07067c010d82778b53b9833034c4405d9f405c12",
    "3ef7dcab75cae58869f73b247cd5ece5892c20189d0863bfb5be9950139e8f2debeabcd5cdbaf93969f022496851119a4aa4a7624b97c4a76c70b2181a2dbb6d",
    "5a121dd9b476544f5c22af042aaa1f143a0055e0813be1c88749c85f7ed4fc6e50b7c5de5e43c05699b5ba715d922e32f07f79b9e71ed8fd9ad99e336d637f25",
    "81b1c116265a7acfc12eb9b2254f20cc76a5739e0ea94a39bc627b41030d166da8604ec40eb4fee4a432605ce77fc3480db4cf27c525707d0415a892fca841ab",
    "aea091f2d2c87991d397dc1dce784134e1c4309c57e176d5747b69a0315230b5874909927b6acd0f16a6250dec7b2152c7bfb3ec1e529a24ad6b6d218ccc7040",
    "bb3d3af90a1acc1b3c5355276cc5df68e90bde6ebaf9e7dfc42c1b907ac784603b3f8da9eab76ac3c47c436733f4fd032fc391fa5d1205a3f26af98b02834b4f",
    "f7c18e0669dbb4ddad00aa5550476b3d7e9d2320321fea7f280bf01575757cc2ec0eb4e35e5c170549fdebd724b9024e7c9650b03b221e0354c4b63f46e44e10",
    "20d0fc555edcb5deaf456031eb21d6fd44651b128b911731c74e609b17287c4396c9c66ab6a970d2730da885a304bea6a398962d712e1b2814494028f7e6b375",
    "2b78db21ef1120713bff31f697d918877db5d29b50cae7fe15ae66157588ed98971f1f3474e965df27ba067a020360b1e940f7d86227a4f9a9b2b6bf5ab6257e",
    "ea7f08ef00f9cb2e3a03832c53237f42d0a8ec4722be9dabfb0dae93ff4e1846fd84ef8e0471a1a1df345ef1885d219b80b5fec580598b696987509363b41dff",
    "d42dc43af52a1f131ee9946cf360530de77a69d91e662acd1988452440adf11118913bc07df5ffa83e331f14d5bc64aa1a3bb1db1ddc940dcb25f9ad7b400f74",
    "45fd83ac7e1a14e3c3b0af4da536d17b1f2609dd8f50581f2d16c89e3c936b0e473b9064100c48025eb733f4e9a2dac04bec88fbb213d161525556a240c527b7",
    "dab32ce0bb141870e01398d113758c6cd4aa1f6f1cd317613780340e8b3aed41750e864be1cc07db52768e9503a9a8d43db32fd5650ea900214dabda0d787ecc",
    "e2ed4fa04051dc273c02c61267fb653ffec67744cc988a5451c96b9466c220bd6e0e952abca81514dc8f7ca4de98ab1c6eb64436813ecfb15a8e01d549d35c14",
    "4b04c1a6f652b26f4de3428086eb4e0d28e0117efc77b1c16589e23c9bd92c73da78283654c2b4ee16181d17a8d4d59a69b75b3d612702808b98180969b052f1",
    "10c91329fdf7db329eff51b8cfac890ec04323e78b0ed7d188f3520ff6adbac708a07d423095fd7b1219a877b27ab4567518b736976b1fa10b115a0234200c5d",
    "835f1804d29666cbf75e7ba284885968a120870252806629629988f24174c6d4c65a14d41fd8257dcd8489f98f1463ccf37c6b59821cb8d7e9117375408b794c",
    "33263173e67dbb2af9fd9cfd238042d9e307c714e0a219a40ff75613d830cca8bf6de73db5255cb77703e7a38397b9863f967516ef85444a2d1f0d36722263eb",
    "0aaf0ff6631afc9773b43ee4ac75dd27a066cf1652fcec963826fd938cc339a977a84bf727ca8fbbd37adde1ebd3a688df03c24a1dd2ecefaa046d38d547122e",
    "e7c25c410025c8c9317414877e46668866b94dfe0f06a8ff461d13e593e39883e4ad3a5487b22193aa93698164e53c255475f08dd7e5a4bb52d8322321c1a80b",
    "fb3789e3531464e9541ca0cbf7bc6309e6638ca07bec69309564122ff6f27781213e2e04844d111c5b98901af793376646710828c6e672fcdb807cdca6ef4f4a",
    "91be3d445771a5c023a15e496c6d8e78ccddad2a1d7658a35d92c92c5d3d7dbfef1368acd78ae12942e75521407da2dfef34c270208478a55ce532231529cf2f",
    "ed14faa4d16e5a3bb88d24050bb7cf3861c42700997b0b5eb27b5754cb9c5d58787e31968f5ea89dd3b6feaf79337198c0b74dc7ed1f51179421cc3df2ecfed3",
    "0187cd4a8030571b0921ceaf4bccd961b92c338620ec668c15b318cf8ab3695f278965280fa9ed28de2f74b42bab1de7e156f54ea9fffc928e9eafe9f410428e",
    "6ecedda44f90574dfa9ab735bfaf50371d101f7e1fb2dd8cd81388f1099f286905a72121cc1a99064f9b4045c27efce2c74d72f315c505b7d6217ee5cdd8cdff",
    "601a653a3fea453fe13a64612e4b384551104aa20959813625ce2338aa3785e33afdb9a123c1d0077d28787b5df83df88ff83a6380852157eb36ec8a71a69b2c",
    "79f5958567d654bd625d89413556578f9a67afed4d5aa0a8af0860969a1221e9b0004d61a5af9bf6e112c3f3d4cb92906e8a89ec0f0068b47fb7279374ed67f0",
    "26633ab59f7e6fe4f9d73c9fbe1d485b04842a83ff7a9211c14a8678850c7668d0b81c7c34718140b59a9b10feac549a4d6074848a9847937978d24c0137f52b",
    "a8820f108bd2feee3efce67efce5944fa2467f8abd571d66f329d188a07e265c69cfe377b227e4325607ce15f9ff70c4e4d4c5fb60b3b076a163c1fff641b218",
    "cade2e2ad744ce1fd15118196ea6d90888e32ccf4108ef952d463b46bd260540e66575daa96e696b029496972b598c6a6486cc37b00b43d6177257087466736e",
    "424be9e3938d04b46a78334d22f9c52a9892f59cc8624699ec401b1fe9ad8f1fbef2fcd57c23fb0a4833113279f912cfed59f4e7fc0c64feb7f69c78f01048d2",
    "3b3d45bba2f8de646274b5d3b96e3862a04300fbee5a155f86341684d82855e99cebac4b9fc16080170339071a043e8f78eea2c84a24695ab6879bebed076654",
    "3d30186f83952edf3a9e5c9b10c8758b46eab0249ea17733b4fab5607ef4a2ecb8da1792a3ace64ff03107d435d918ef0d41cf17947dde2c101751f26794f2f3",
    "0d2097b786ffe41546e064c6a9e0b3e1f1590fe990cb3ac614c42dc61f1aca07b697842d0dbedaa66e1769f420d032e17829799e788d4b340bd51af9c62e75f7",
    "a17962e2789b4a70851d9ce1245aecada118aa48156f2f03a461549feaa5a41c2a2a9e2888d4f5e55dfdccd15ceb0e2a462f1d436c9b32b8cd11501cfeb7a683",
    "4b92c5a5cbaecb2b96d6cdda3c05e1e2b3838f0084926df0ad036d9e35e942eb5daf264a9378eb163dff828c7aa132aacb36bac3f275de9d74eb685c5994683c",
    "43d86ab4f5ce75054c3bcb229c8b390266ea4cf86aa8f5ea506b0b424e397c1ed59e0248004d91d6feb3e855faa429ebf571fb790a6a7df7c84219738be395e3",
    "e89716971b165fc12df3d46adec54ae11fe482867b0e0a509c90647d2cc01394b311b8b85178f00518b4982c8dd99c9e0b1a998595eb00514e5528c3530d00f1",
    "a0a294d24cd3d0b8f0543e5410f42a417ee6ced1ff023e637ffcdac27637a70e1706000806054b7f16c475e7218a476f72a257496c39b9db0c4c9b1f8a943fd5",
    "234a1174188989714871a1bd3c0258537717da6b280332ffa1786c110c9f678471f029ae9f805477c2af9801409ca6955d3f36ec6053909766e9f3d1f0aedc60",
    "38d0bb51c8da210812ab70897954360a0c558d396b5d6aac4bdfdbc7e9ac1179d8d54fe4a2adc10efc8963cd303d35e89b8574252fa83f0ef16b8a5a8a4fbafa",
    "0a17f774be28e571a3b30710e8e3ebf1528072fe2673458d936b0c6f15dd69e8d88fb1a6563bd1013d0907a5aee17eecba69d1b04fe58c8c5725f9c5deb07eec",
    "27ec3e51d9e10715b69dc03017e5b4561262fc308fcab6c429cfa12dd19ba34c171cd94e7a155c4ee864f01682842506d6ec6da28652f2514982ecfa41cf62a8",
    "0f60fdeca9ec618cccfc2654d4b4fd2ddf42410a74b778154adf37f3dd942f5e65c2030fb4d1e00fa00d309dddc224e6edba8d94961b3af1843fc80d60619bf9",
    "183b5260246d84103a052e0c5ff87672c95af3a993276e1ac29a562ba2cb9c526fb140db4bd6de6bcc5730b129e87625ff7424c2534890114493a41ca50f280d",
    "981c0a40896355e92077638fe07895c43ba4740b09c820ec025bec16dd88e9e463ae490ce87dd6b6b865950d2e7b19f40377445a83f51a9fb6f16a51fa78c457",
    "d7c61e0d264564203b4e98667421ddc15f6e1c0d51988881cf6b623a984da4a10029a7307560324ea0cfd8376e6a7b0fb4ba403cf53617123c96bccdcf779696",
    "c3a50011082e5f196570cadf429f921fc16703fba35094f1082b4e4dfd07e98fd8c749c98abf7cf91b870c93d512271291112d05fde6008c787a8a9f2a7d4fae",
    "5f462c446e49b70dedf2a643e8673d79ce2934ff70d44595289eb7033650e4cea2957af077103904b26fff64f4e146bc1d6ec186381afe331f19e2180f65ebed",
    "3afe4595f6840560a62179ab895e1ea6e66773c3a9f8dceb6b7b3ae8093771c23d958a58f6ebfba7e3db818aec000ce665425fddb8946cfdc4ecd90894e203f2",
    "156cdb76cf2a384d5414f5702fd2c21a338c3c562d7c246bbe250a3056ba17aa051647194283b80b0957b68ab4d623df35560b7ed770a65acf6ebbbb0e06d98d",
    "6b839a26e282e5321d38b01fbbc6be15f0198692231514aa706e8b0e94aed8c3935f08b70c0510a522832673fb497dba3d9ea86f836ce4dc5bfa418225c8977f",
    "6dc4667ac6a56e56a051e4069961e8b23806c602414397a5bc48376b4e53798dfd24fb554ffa6ed83caba3736e57e471eec98277fe50968be41a33e2ee729470",
    "7aedf5db94324c53b08b25552b2d47f9d772b7f915feb2b7f13746396e81d7cf47d13d13677b3f05824f0dba042bf3669188e455caf24f81aab1983573bd6a9c",
    "15c5bfe95b25add85a3e01f60ca9edc859abcff2fc895b13bf01af42de4a0d8f302d8c034816e79956dfaffaf22593a8fd46fbde801e8ebe75996804a321de98",
    "830752836c1e77811ff01d80f548761979803874d766c04a2afc87711efcdaf5e5579deb51990736a6e15f0c716b4416f5aaf5031926003cedcf3bb93610e17a",
    "47a2ff514c8d2443943db9b32abef3dea9216503b7d33220ee95438221ffaef5e2b3d7da228364ada3966da41560f81ab894327bd9e6b9007faf0b7b32ad4c01",
    "f4d450f2739a04405a94390a5acdfd01a03976d5d8c02ae5a15330b04a8225a6df4913eed76fb81e6b1d9406012806fd34acca4217d258a83b0b724508a74ef1",
    "a2a863d8516105cc117dd36c870c8d351ec68e7382ec20783b720a2fcd5766fe1aac141b7b4810009d5d2444feee41b3d65ddb8f93843b9fe6d5f8077aade3d4",
    "621ed67c8bd01ac96b23485e701c376f2683a98a7b17633a03835fdbf4d1109c202122352a57a2219688223f9b7c27815c81e7b3397795bbb7c4b819c2dc67ce",
    "ad9a87937b2eb5ed4b2889271cc7bc3b094b9672c2c59ca9046c03fd94d1c49ff7809a25fd1ee8aece275cc1bd17c3aef98f9a7682c6003f54c4a70e705a97c0",
    "21e1301b7f3a3badadfa8879471812ea3a4e434866379b60fa3bb7b08d7f51f7abb093ca077e2b1a85b238534d7ce420ededfe1ffb92844b4f0376b8b69a0328",
    "b6a481374e425df2d31231d155bc2ed0466ee4ad0033d77c59c06af8c9673f2975e5d109908fb3f1fa81d7525bf78777b8585923ef43eec03900def3d194eaee",
    "388c75f644ab98336e114602a330237197798b3b3216d581a5256c375b770d0ca9a33ce62aefe4fc493c9c45fadb3bb60d4d148cd0d33a172a2abd3c7d4d6518",
    "ba84e5b0dbcdd23e144209bf31d3b2041b6a0f08977b4de7b5c797b217ab5cc26632358ece1da506bc183cc7c9dc90fc8804f8c210fcec790d5c39b1a1dae60c",
    "2fdb80de507391dd166b213a4373c7f99573dc90ab68eace325fba95fceb54c7cf54fc07fb99dfbd6d21ee2c8d79f0d54872bb329e38583a532b7e2287ce7268",
    "a383ac5a34a973e3f8b6d26278592a763d25f3d55d73cb3f4f8b65e074de746777acf50c3b8645a09fa6b65c3e482dae7c1bc59ee7f69e24a3b346c7c7616186",
    "2ffbf7708acf846a9deac7988f1344b32a9e02e6064860e002e4491d248483c96fdf385001054bad686838d3b9897f80b059ee6a911f6b2d7106f1afb26c13c7",
    "2e8f52513fde00aecc404c693100cfd6a4501628003b36de9b416c5f13b97756fd871032330d5d2c4c3a50f64d2ca58851bd8e9ac0e9e80e99e65375970c96ab",
    "f202e124cfac3e5a8f7beee485fcf9740cb15fe3934a9b39944ee8ef7d405deebd2b4381b5c2c4619a5102c65c2560015035e2b8e40c942650ab2aa7d4a8e4d5",
    "ed98aa75c5884d107a3ee5a8437c9f99ce597cd85d284700e1ee038795a8ec0ada93c270a079247072017e65abe8818906434bb57c85f9b335c28813c76e7879",
    "c8b0384378dc19c99b16bce9edb3a5de995f5cd523f0f89e671648ae669e949bbbbc38c204df348f2ede89fcf7e642e76551116cec5b4e0519b30ca999b2d969",
    "3b6ab46ae2c64b73d3852a56563637093d8771b5a90e5b2aaedfe7d8f05764b986eaff49a40b9742e695c861047137a044e25ed00ba5c6cd4d987742dc11926e",
    "76a88fded905a92dba8037a4e7d8ccd30e1eca7f4e30697683ac7c02323ebdf9e7d6d324cd9a6220f586d0b565537f8da3df037b4350d9a572717caf45c85dd0",
    "1bc7cce416f4952a55844e5a165decb745d91768ef911b948b5cf290080a963ced03d6f25dba32a60c81379c0f4697116b4dec77648b0c6f4b6241bafcef2fce",
    "f16e8b0dfd06dc1ad6fb6d577ddefc8126f46b8530f99fd5f7c8acf88db87d7a6bcbd50052e3eb02132498028469f5eac8a6ac5efdca4df9724b674dae21f25a",
    "81ebfb37a2802d243ce4df77d8ca8a11e584cf815a0c6f3cadf0e96c73bc4d5fa2c4ae94331f4c59e3468897c5422d7dafafde7cd768c97c47e8a43895227cd2",
    "65b325e8b4bc85eea57f9013f8809675e2c886b24a45862dac92dc2003447a2e7e759a0550030c195264148f56dadb6f50073c84bbde0c5197ee75000c72703a",
    "6cd9d455ad626ddb7e40a0e74fc4ceda0a60370eaf705abeaeb0ed0756ba7998a583cfab666c16edcabb6b2e794f11f4c805f8d426ef6b7cf95f216d20d2004b",
    "80d734dc886e5113dfbb938a8af7a68741c55b31b1bb8f679f66f02999395fabe6e984537bbfa61d7b046aa520511f5facd1473e2debfd81d6e82df411dfd0db",
    "40bc287d2e0dc3c5b59cf216669dd0371e63a82c71d558cef2501c0bd4bd0ed327f868d0b6f3ab3e187c90e4d49f124ae75ead6447925c2e04ca6033cffa085d",
    "68aaaf1dc15640f11fffd4731ce8200787870d574eb7c929c77ac04c6a424c6fb57586a46876eb8e38c7714bc2de3594a1cfcd0154f08ad3a233bb6b3d434b94",
    "79967c2e5fcbf2bb7584311c9b77d13e6cc59452016b1e04fd9549c1eac04a63a13c6cf86445312392c5b25974e5eef1da0ddf5b11b2c7335c477a5b6175e3a5",
    "8ab0ae405331f771ee7577e174207be45bab1c6817374af421f670222a3bc1df27b38cd731fbe9b131c1c237587504bd850fd02345e71ebdc324ef785dceb0b6",
    "6537bd1feb47a300e8fde8f9222b7a594e518c7826a6748f08aa450a3c92d6664f2aa171032e640e84709581004c600c8c40ec2d56756a5607364ba803612d66",
    "749e1d129821be84c47a46087bb17cb08d0769a27fb37417315e61399e0d72c460465dac3f436854c45417200521b8e68cbbaac51cae3f264310f269cedba7ef",
    "137a78fd7c6da5a41dec52f1cd82591224816bf0bcccb476de47d809d74df731c9193db10e01c876c3a4cb28356bf171b586c85ea34afbeac42cdcd912f6baf2",
    "b77e66908b3e1cb2a93ca82e975d69f118ef0cef5c4437f2d22bae61fc2bba8f0e04aaba51b7670e4381ce6b538ab52aee638c1d61229594e45b72fb12f27cbf",
    "1effd0974ea6a04b510c5ae33ca5b2417505d1321d0eab290cdedace7aafb715c6e3ea32f4f6f48217e66c833cc90707c3625e89f19cb83a6ad63e4e68d74b4d",
    "7079fdfd974cf874524c7b4db7816a9d2955d27cdf890c6bb02bb0928b2d18115f698e2de7da4ccca05ed8ae0f3ac062462fb7796775cb5f237c4ba90883f833",
    "2b5713b28ca6cb74146f2169ee63d65c5bae1a026bde9f49cc2dac9f4b1f19061764315b0670e7e14ae3a1aaad0ad35841d819730aff534405e32017990bdfd3",
    "85c024d57fcf582b6b97e9cb50320dbc50c1d6f459589f0199402d92e745873a0a6177c1b8ff5885aeb222d4f99fa2ba1d90df933fc8aba5f925c88f176d5e31",
    "33164e9163abdacff9f4842212e8419d2258626625f3ca0a589a95183ebd140e63f8ae08f77b69152f98ceac1d848163004e0799d75fc3da76dfe9b9f24f2141",
    "8087359edb6b25c0557a5a57c0c6a8d953b027e10628c79a10ecd4ab50327f2d75915bc9a444c0b6230e1bdc71e3092fbfd57885b7ceb56f980784c1b95a684e",
    "b45865f88ff56bceaec455701ada08526c1a97721c3f5c21ea603bfb11cd6923b3b242a77270f8f679fec69e83a2faeb0adcddd4c913dc029e5a1c65515d811d",
    "ffa85c73e44435b0d048a8a1e39cdc17eee9ce7d9b9c196b5f90aebc088c6143b63c306d8ba0a231a2347eee3bda723b83bc16d5baad1c99b561023d69a85c70",
    "d54b69bbeda6143f272520ea44971f41decfb185e0c2817f10ada2ed7c450f9cdd39168bc70671aee6a3defdd3ebbf9db0403177f5799895127bccaa26030d6d",
    "d4689c9c50011440e5740f7ce2e6d6d99c3a905e13454f986e74d4983f43a35b6b8e817b1b524d1a3dcfc1ecd598a612131519b2111a81314163a5d713f3a25d",
    "f79f6ba135bffa88f5e729060c40bf4cd0c7cfe98ff84c5f2f9c88e75e6436e995ab4189aa9a39921c5391fd2f6d69cea561cc64e8aa3eab49c5370beb9e61a6",
    "63f07753a8bdf16490f34f1e0d5bbb639f5047cf485cbe58c3b61ad25a336f475759268aff9309ea8c9e7a421a8b7e05e98b99fa919dc783ce08de0f8314d8ca",
    "2a3689b14255bfcb3ee2b0fd60b6a5a6561678efdc2f0e2debefc731faffb31e007b945059ae88dfd3bfb4ca7c49ec5dee4092ac636944f0b98fc027151ecdf1",
    "b301637276d16fbc5d543ed422af9221cc18fc1ae4567980dd1e39d0bf5b233f5d41497a62495c8625e95fb947dc91cf50e7a9354931404a031695bfce64a186",
    "bf5e59857e61317b74d088ac7913755699a1045d365c5b570ace01fab4533cddae36f1abd5d4172dfdfd84d3fa61d9f21cc52458220157a4dfce7842cddf3ead",
    "d196a1fd171875cbc5d3bab2ae61f14b1929b6aa44ce076dafec1b9452c083734c4536c5ceb78c9a5ccde1fbdf44c19738928ce6aea7289d3b011874ad6d9f8c",
    "6ec4365b304473eff7edcab831eda8c11fae0c0b8195093dddce181c58ce85a2433b49387e52777d703b094be654ac2f3e784eb06dc06d0ecff410106ddca492",
    "9f94b8cab6cfdec8557d9b7f866e3aaecceace54ee7f603ed633f23eca506d783b970cc3f1c12324e583e0c69b5bb3536949112fea839c5e16ca0daa186d7eb3",
    "b7cee18f739f39f01cb357054c2e8b5e9aff15ab7eedeed331bd9e42fc22fe3c8f664a6307428b1800ea42fd7d203e37175b019469c176e3fd2e6f5d140bdb81",
    "6dffa67340ada87de5d1f65b9ce91bfdfe9365ae9f612bdef68f2b5375756860a7b00415ba1ce926edc62dc919f59020700cb4250844c0430358519b3848e15f",
    "a71f69e72e7d61f23d9bb062793b513044f278f7a994f0323e10351e74e0ef979415b9b8b4f67fc249bb516a1e822455420c83e3fc22044b85f929ec58990e8a",
    "60062543a051f732abf4b2c83c2e990db7b7d15d43e620a73a63f4453ba687d14ae36ae9fb65ad581d080123ab8b5cfa8d2f264aa79d3579de4345f6410c8804",
    "b873822b3dc05803863e7c20c3e10a11a3e9e8c4d4ed85423b0fb5d1de428b848ddfce0339166a20252c85eac64656bfe0b4fac28234b1954336a2e75341c351",
    "0340d0689279b9196a4c35b71fdd4e5dc4f0632068725712d9b0eb534c81f57cbd527c758bc8fa2a5ae8ab9b50ffe8dacc4465be4302d29eeecce625d8df7973",
    "5510023d38863e2978517a234381cf66052c1e61a7015d21af394c6e9437b4283de7c3f1fa8b2654be497e22133df85800a0833927e375b6e508a88a2c21d6db",
    "824f9c4d76db4b4ba236d4f35206cfed4e0615a1321423423ea6c91ef4f352cc30779f38956c8828b622dfa505a88f8a36519837d279dcd0b1c2a0b81629e895",
    "ebb7e68ff4b0ec15c0fb71960303037e698ebabf6e2cb5f912e674a9c9c42e897e1f944597ce7e2db180ef4056564643cea6e2458efe62191b2909b3fb3375fd",
    "f901e482f13a07baf68cc05e7d01f2ec70c2c285a322204a78c449389665889d42b2d3f796fd2f710a0460dd7d4db0b2795c4771723422506c7ea76301c589f9",
    "506b957032905971a1bdbcd720e7b053cde60b342ffc370b22c09f34f21bc6aab783f1002c7bf911649b3c670d870931018ce6c704cd85292d09ac3b0ba0a1a1",
    "6236bf4ad3b7afbed7b5fc7c7bba5bdd5e1283b7f7e2c72cff3cc00bdb0f4e151167ae86257c48ed2b7082a1de0a68cdb9bcf678034fc80f13adb00360e58af1",
    "60ee7f10a4a35720d9fbc48de8c3a34efc08d8846fb093f1a29de38a5cf98f3c28b258991c632227266f277da6ddaef4e07a98f5ddbacd461c4a6cd43ec124b3",
    "3b00d6132900acdd1e976b374f59a0960eb4cc1707b1125702a51a3c68829fc819ba48a1d2d3179a5dd54f4b918800b162c7a897c1dd5660bc21a770dc2f0563",
    "40eb123ac9a995638652f35e0a823227be157c2129a1c71918d1e78d805e4c38284d3b1efa8c8ac03e62e62dd7a8b3370d89e52daba96e8bdcd985468af5b6ca",
    "2c7fe0075cc90f8971db01f1e96eee04c267c0c951bef0559dd33cd29c0022a993c7683794c0a521be88906c2eaa7b81a58d34daa300eb88dece59d95aecaa84",
    "57888cb22ac6de5189b61ffc53c88444e2708fcc88935b46bdcc181163df3c1f488876633f8772536927d23de029c52b3e7567f2dcff36c2cffa39ec6d3064a6",
    "8a5db8eaf3b38ee64caf56949cee105f4f3296b686b39f0e6fd2c8f9700b86a42b88b4e2fda2512c24a21cd1727b2d495696de7d138871a00c3153a8945c638c",
    "7aaf604333a1986b9a4104c095eebdcd776b65ec77e7e394bcaf0d8fb8676798a60647791fb42846c4b294e9b670cd65d537d4c87fc9f33435ff34dda0d98121",
    "a0230ab66733706c8b72e914b4ff84125e4a00164b0bc6635e779f0ac4e272b95815b9e9d4b8556fd0e5389b2436f353b1a68895cee8f91dc181459baf984bf1",
    "af37eedc8372333ac09595c2273b3056e42ccda518972ffcc9018bbf01b8e691c5480e532100114a1d54336db6203f6e633d8151f0456f3878e9b1a71613aa97",
    "ec558fb5be0f5107d43b9ded266d6496b949fa3f4dcd68c300b18677e3e2f3298078e030d8a3d7b9985c5579ca8e329e641d951529c9fe0323bade155f21258b",
    "a4a977957d1b29b0afffe99e088d25675c341bf209a65d5f0b808a1be2e6f88b0655cbb10c07bb4bd1fa916e9d43b5e2fb8d5fb8a62a4b62d3149909fdce6831",
    "892cb4fffcdcaf030ed1fadc16897be549ff2ea069e24028a77b7614e0f7d86d98bb45b3edfa90fb18568da68f4476bc36da2623c8f37ebd97dce80953497097",
    "ccd4252e2c6ec435ebc46cf3da9ff6b7ecb5ee1b58cb6a3529bea7327b59f6411316314df09b58729a607d1a87cf1421936dedfd5a38d2dc504a1294c484330e",
    "e6a3cdc06dcc2cd9998ef400511c3c05277e78c20a3651a18186d2ce36604f036288674c3985dc083d0df675b717fca4a03d357d9c7b3f99ddb73954fafe731e",
    "022ef891593005662de744b92cbc42b5d6bafe781d2d0c3d199ae7bc8cbb4ec8a84b647c827c2ddd322d857ac6bf5ba3bd94ee63e972f362066a426ecbed858f",
    "411add9be360b501706fa0acf0a533f6ff5cb8c4f01e1d0a12224c1d31b01a9235e2ed6efe5698f24e278ba4247ff1fd3160c39756b5c24682c0ebc0c4f508c9",
    "bff9d0f90557100c2a521d1794257489594558782a7064b5fd837b8f11863263a2439628e46cd0b62d880ddf8a862e8d8ad225fba4bbc3447567a2d7bb69064e",
    "9ec7b0145fbec9d654eff5d1b5271aa7b2574e1f0f713b35f3e9f7841a339758e9cce3305a65b8a06a042e3c0cd2c6dafdcc711a3061fd228a3a44105ce95add",
    "5ab76f5d84d150545302d049cd4a4c4ac9e362696d5767b39453f254054e47f406219c4c152dcb87080379166f1548245b51eb152d0dcea56ea8b770488e87d5",
    "deeb823a02393ab0766a946991d5ee037630b81a941b2d860f567b8a98b0bca023ca6bced8b4fb51058e1d10c348b5ded5671888fe7d1f3636f57415695c4bd6",
    "c7c855bbdc6981b797266dd17cdcb9df47294d573765d61e7867c78e5e708afd3ad163ff4f542a3616368a2479ce9d29992b9c35cd49d0f0e5cde33b865e4664",
    "0972e1740ce493bc41b5a56a595a028c27d9b1c18c171a0cbf9f16e1e4f8e175bde16f523d6099d9085ce1befb711331dac356b8bd69f7ac47e9c8b0380d5879",
    "818612fceb6c6951dc9f02b933a1c6b1ed46dc239af7945ced8d992f89cbccd78b6ae86bf47663421e1e77a83eb0364f64c65fef4c7a09633e457d89d3b3e11b",
    "c94b09c08d52c1b20e5d17f0db6c933bca7d3ddbb1d85f797d709a927c485e3b8968187fccbe859e21d0f7eda55137353798f6dec1b79113dbd32a311cbe7a33",
    "290e086652daefdef7244fb96bb0a7b05e8ce280dccab2fbe95b69d97e4a78df644b79b0d34a08fd7d05b847bcb65b8e08a8491d92a8c2e54765b2107d02a823",
    "59e48fff193a7686412c737999d76256f8ed911816cf2405de1d98dc70545dcd8c9df56a16e22f84570c0ae9c0f1211bc644436a7a3e1bc51102b860334b329a",
    "e614a5550e9773320574a78702be4c09f5d9748dc27e0b79090e4180ad0a278b8c4a77990ddb79c779dafe15a8eff5a6033391e966c1756439d86b3811d72877",
    "d29cf7f7c0f386a1c27f095a92dd30e435d5f013a16b4e135d3c7c57c9aaab0adeffddb46a7458678a36e8ca94a4d52b608b324b858a7800af7a7bed74e7c89e",
    "2359441d27c90fa7dfafd9e35ef12341863775e1b59ff0484a8a019cec4881eb2717c001bbad98b77ae113bb8784bdfd8b6ddcee18da066d7b4a69d8a187af96",
    "fc8dfa4f1cc1db729289a09c59d25a21032f940a7b6b88310f42aac0ae534ce3a4e16820395771f5b07e829754754237adf947a0123c877e3f42e7bc03f12cf9",
    "3e67eaf8b47d407c2e86a4cadaeb3eef37692f8bdeb4afcd21f83cfa2993f473afa24d0fca5c69ea4b9ac274ea072b56a0d87994ff1e6bd6b75dbc2e8ec60cd3",
    "b54bff303d442d829844440682a81e0ad7c7dd3ca4dd5139a4f03a2823788f4ab49dd5e2215d3d6f3df231ba1537b57215662cda1dff242784f5c6a5e90600ad",
    "55cfbc7efcc66924858f36d243e9c0381d8814b2ada076f54db00862c1789e225c84a18fc040d90bebb54f55bd4d8dbfd7f4232ad3e44f17ed9f2dbeb03df80a",
    "208de70d8bf5b473d1392435a06f277dab0fc4f6f1dad2731055d1a63e5910a61ae22082d10a9d733da21850044cf9b86f1c91b98b5fef88e49b88270b891bcf",
    "b0433c1faabd945e79ab5d12c00491f466d9fd081ccf14f0ec3420bc7deaf320c2d7111f5d640b0ec91c96abf98cb19976f715954b278ec52b33f63660ec2e5d",
    "187e0f02a09a8be13923f68db3e1a07d1605695d9c375568a7fd65c987d2be9de4934874ab838e500c6f507cd138c866adb623de16a3fef67b81f6d6e015eaa7",
    "5846a72fd10aaeac911bab510d916dd1d04971bb4c30fc684921069db866de412221b194dba1bb2f1e22a7bbe86f726e8838866b39eee783dbb9199eb5c7b4c8",
    "c7aff8b21a90153cb9b2b68f901d24ca4ae39e4cbd4f135427aacd97b3bfff22d1d15f5547390b6aea3dc3f93dc4c16960b31d1c258f40a5a5af515c9cbc48de",
    "8ff5172473e5a37010cc746c5d009432457f7dd722b4222653b68e92883cffba40e71fe94e12c21fb0f95478ab830dc69c8cff351ee1f41aef5d2d949fadf585",
    "b2e18e7065cbd99bc66be88d720c8f7f7da1898c93c652499177812b907eb972addde97e44ab89f35cc53b4966bf2dae23cf968c627528924ca467652e09ad66",
    "5ebfe0e969b4ccdca5e3768b2d14b30d33cf63730d0f10c3f118bc0fb787a9adf08e125de3124d2d9497dcf1e2e477fd239993db028e7f335bdb903b0dbc0a39",
    "f1c09edd20781562736f673dbae6234d48b53807dcbbf1321e37af23c5f8132389eddf522d212fef5c3446ab0ccf20ce471f0db1b052bdd28177738ccc24a231",
    "c0e9b8c985914a10c51585de971b9d14f32f0f64ff6e6a4ff09d868324b63c1c9844d4a700a8343c2b0a9f9be28a013c976b195ff7d0ac65aeb7570c776492ce",
    "9a3eb8a8a4545eb99439074325450fd3bb0ea42338b18fbb82db265d136b29caffc37a144d29ff881bcb671b33150145eab18a10d166ca73408bd9eb0714e059",
    "f95dfeca421a04ce88cb25b8083865886ee7d41b614ae1fefd73c74e6d139e7c744ab0dacfe90d8898a6349fb952d3a0426b2a6748219173c3c49230315b3e64",
    "eb8fc4d4787183f9030a459fe8682e00d00c5f0ea519fc9887aec6d4b65795fdbf437d3b42b96390c565c4057b556d8bd2fd9db788464d5f6b0144edb43ffdd0",
    "7ffa8db24a7f92285105474a177abc57e8170efc4e142e2dbdd28686d658c461bf10abdbb2a8e52b4f7db12af4f3a8734413ea0331fa4ca1bda161da181cd632",
    "87b7892be5ee65c37b8fd3e6fa29aa723b59e9f4947031938e0058ba1c582d951889cfb8c3d4fa6b733700ae987752b1fcd37e0bc1d44705dec8763559d1afc6",
    "9f456993beb302f10c3dab17ae1369a2b881cfc35e40a440d404e5cedc35c3295eb952abbc0e604c9dd1501c904cfdb2c8279e351aefb88b3b2444e2d31f822f",
    "f30a1e21edcd48cac01dd5eb25978c635973f372e03adf8e75b5036b0c59f817c5cf1611e27d69690025208e093d10fd64bb69ea6aa6f00484dedead295cb333",
    "5031dfe841f544d1a24503db99b895a230773fbcdbf730cae49caf404f9f7d7c05143a218be32c0a2a0645ac8ec39626be2d6b603cf824f89991bfedb0b79386",
    "70cea751ce879e6490089e4d13f56ca0fd703f42e1a1428282d165d5d285f36909c5631f8edd689bb362592221ecc90740a57dbbd51843f2991bf2e3fbb5537e",
    "fd1ca5d46d53f0a187bfec564c4f6c3caebb50a4f9424e9711e155b040f051d6c6e89ed37339e4a35d5610f2fefc2fb5c8a0a016abc24cb1156e1055947c4504",
    "61d108e3108a87c99a8b63f614449554ddb00b1132f8b1b635dabb4d8952fe07f9202c3d8019b11b24b6eecd1bb7aa09234131a1ac90cb64fb8eef624c3bc65e",
    "32e43bb6d929d080eabdfd50270f6e9ae57889edd6919f8aee9e2a27fce45fa5fcd4387446e14d507cee6c0852520fc0c4a9cd736f3f28cf6c0dca4197da93c1",
    "54b01dbbdc0389024abd45081a24975978adbb2da8eec47dd477bff5d4d00552b186bd3e1b6cf1b97bcba163403d55462852f35a23f773c84a91b4efe99e74b1",
    "0076346498d6208539371e3aa6d4bac0d179514b3ad9c3179f74b696a8b7ce40318302539bd023fd8dbd68d09b2a4eebee2f10747f1bd2be711e7e7ba8d86d85",
    "7df0703e8629c05b9aea24bcc65c9ff9bafb5b3a73bd33ac2d637186112af63a8cd9da2c7448a9d550e80114074931063af4c8174f120047ef672698f10f032a",
    "bcdf176728b5e4e0c02e23a87ffca3e495517064cb16b048632ddefb38d0bf59a565e61c29c5e1ca06357458494efcb933301d61774cff9a385ca2da62489089",
    "838258bf836bb272760845edbfbf6e482860bb28aba57d476b1623e3bfdfb081487c65d19b92a4320f7c29e52bce60459c22efd6252a32dcf0602d9caca7d94c",
    "522112a0ef49fa674562a9d906d3b4184d1ff098305bbb2abac61fe90057dc476de9bebe4a35f07eb3af3752aa2e494458f3846c4ffd2084d398d255c81bad2c",
    "5f9637049e65cb01d7eb77103305246586c618b0367410f620b64b17ac4ced0768d9ccf722a6b47a63421b44239ef61be2dd4f100f85d5d33b58a4f3432fa000",
    "76a361a4ea1d12416e9ca734f403c89deef655cd881e0e63b46e75df48cffcf6e0d81de40f1381e6255c5eafa618775c9b9ed40182899df3707970464f5ca12d",
    "15bc2c39a6dd79eb7644ef9640d9fb53b4863bb4a88da7f6d96a46ae31df16a6e9b6f4ecaed46e4fa1df87f2ee163817601cc6d84dd494cb9d6807fd021546e8",
    "16d810abddaa61623aa0ef21014bac309ceefeb7d495e3ccc14bd3d847b96017e369bdc42d538889a4c6d827f8d9f012393e8220dcaa30a5d54b291dd2ed7041",
    "62d3ae3b7f64bebb0ad78ddfe3a07e23802f31aeed00108cfef4f221f8608a95df0a192088b6d85f0ddf9bb4365d1973217a8e2f0b23435a179b173897fd819a",
    "0d95fbb475598cda8d369b6cb226dde8c63bfd52656de16c92de47d8e0029b4e9b288c9c0e466900c96a3d6f08609474a27585d00450876111806500a55f9312",
    "02285f40a791ffbe3a01c1e58feb2f7e5ee8121b4ec6e5d2ef63df7f3e7268e73b8421197091d76fcdc41ee05c0e30aa55b8c21606038c445d63b72878b230aa",
    "9a47f960579040ed2016daa00cbe04e0d120dda95f54ab2f3fc99cb5dbeaba8e61dbbcf1af9daca30f22696277012b2401ab2546aca6805d7e12771398cf5b04",
    "66740dd8a12eae45e33d6703ae97a0bd14c8b2787d7bbab3ff804ca9c829470b784c75a8adc3d61b8f769551e362b34450ce55038ff4729d9022e7ebc7767fd8",
    "3003e755ffff424fd2c053d4b214bf06caafbb2c2184158221eeaa26946350758729cd1144d0d079fff1ca83ee6e26fb8c78393c86f4b55e0bf55f8894716ee9",
    "2f143105e92f795f01c098f0f2ccefe6899f9693ce11c8927f6b13a2f401c4e2cdd8c64cce2e7c3f9be47dd2f946c0ccc78f42f10a747be8f12c0fb120412a30",
    "b5754de67d7eddf11cb0f170c88a7315884b6d7b7f3db0379a73f3ef3bdfed8363ab6f19e4818d905410f13076b621dcc87cef84c38bc0da96dc96b27c61c8ab",
    "b74869793c1bafa4b7b9b66ebdf8bfa15f9b6f0a684ef190413a3b158141f8a274e04074171f7f51f255e739c6b0a4abafc117b099259f4f106dbdbdad822151",
    "6f731ff24deb736c151c5ba792f9e1c511f59af41110dc0819cdc31838b124c8667b24c5bc41d50429c8dc3600a0d31c43921f31c429b8666d76a34a4f57b7da",
    "f5cf649d8d96c47031a31753b278b2ec70c37a9998d22a9b519d045df3b917b307218f72823ae1e64017c01e30d32f64fe8638149d29229a0e9c440159560823",
    "0c5e2757509baa6ab89dbae020920bd65bed776d367095dfeff61712224a031630c874b33b8e882532092648d7a35e9732a7ec9d4c3c263154e333ae895bb5f4",
    "229a59b65bb9654b5fbf182ae27c4b3e8d083224698976bb39f845b255f06fcfb9808adb592a87a40d36a5e05962b548ad12a94037c8a314763b09f62f741aff",
    "9ea8c790bb6ff8f5ad8552dd79d6762ee991312186be4ac879aa7b3aa90225f7067fa293b08189cf37b741d49966044ab79b15302fc132c35a669277c2592372",
    "993585916e72119ceb45153c397fb7cff0d45b56d8d0adc9ee7a043a07691ecd76874a74230473527e9e443702a685a0e22011e25c3802733e50dfac0db45f1c",
    "6598125ce57a04d340e0f317d460cff3d63c8a4f5df480bb6bbca0c729c474319299dfe9fd187c943ad0b2263dbe884e1f47ffef94df6ba87fe0727158f5c1e7",
    "edd1803ce26a383cc504aa771ef486c78abfed2b670d4110101a54fec2c5cab21b545a46c4648fb3a6a21a3bc3e133ff078aefa353dd7ec85d315366d4bf546f",
    "e9673a503179cb2bc944e851ea923c604543210cfc90e3b7adb2310a972a678e660a925ec927aa931fce7fad862ce2bfb200f76130e4f07bc79b1414d330ddae",
    "686435f51e296e4ee50c876412409ee04f133ebad0274f4e0a58ed659d3b3e9af49bf84d8a0591d05c7333affe88eaea73f3aba5dfc8d65ef294bd74913853cb",
    "d7fad732a9979d77b05d394a3b6c76db8fa4157255384a939a76c690dbe8c5a1a737ccaec160fbd1a9a1c98368ddf4e3918517e268fd1f88ef0164d2d1d96052",
    "b8e63c1180c6eaff13b39e14dfdb2dcf5b3159b49b94a3a568a9398b78b39ebbe863e4440a29464b3216481e0bcd2f430adb01e69cc6af0929026daa4f5bc899",
    "b6fca8bbe8d5f6e825a3575384873f4257746a779510eed77c317f21a37cd15b70014fd715c2852ad375fb6f5664727bd842a9546f1d4d3fdd4d5be2526f2ba8",
    "1316eb7e7bbdfdc82e2ea8941750ed1ba09e6fe514f265114a70f609ea62fc3c943a8ca489e6cfca0b519bec14df8bf944cd8704042535e816e5116a51b6cfc9",
    "d0398a14633d4f3ec6499d5a668aedc5f2a6cfd6e58479c2f43bcdb261fa6370b9a3eb86b05eca7eecffea0261483c03d3c8bda1ec4c6bd0555f7e124dec2cd4",
    "2c0140a2b8ecc381239c5c4eaea5c7167b5f6ca49e1d84f1f6b96c7dd93d66deb1f2d280f1de9870cc9095c870c70a7d551961ff04e2628b8c238e1b401d4f4d",
    "9df17fc27122495cd0059a7528882a6b2190c55ec31bdecb0c13bd838273f246feaff48727cf32bc33708a27c81980ec118c78f8dcf8a4302e987ef960adde9d",
    "96d684e9081b50e54f7b7b3ae905157117e5fbf5db614deda0bb8d3bb7c0db93c9fb7ea3b1dc4435c1114dbad6d39d8589acc9778c77eed0b9bcba8f5a82c31b",
    "d37cbec3c9664f422ed1919787de543cbcb9d0f6bdd366dc0f3138914505ee45a0948adf149475df7b4b44fac4cb38dab9b8105fada7aac177cb676ffd0c8b90",
    "5a2837f868d649ac3998548e380be427264846375e26abcc9e6c4375d3d2446b5b1075d8228e44b2f6628c4e0fff10d1c44e3dd9cf47dd2cb277d65f0dcbbc2e",
    "b60fa780934b76dcffbda0bfe1ffee7c5e3f299ef63d053877c835ad510480747e2fbaeedf2d45de84ec5c8b11c4528decfad032b6ed1c5ee5103cf39dfe96ea",
    "2d08893b2652d5ba8e76c8a39b6edb738af41cf08f7947cd89d19cd40d8fb81e331eabf4566b0376e19338c48dbc51a5d7b1b386534209ead7e91a69bb02d80d",
    "47c634463dcd48914a582e7fc2040236fc72113c7b1686f9fb79b17e5ee0a97e45e7940960cbc9725977d5d3a672a86a3b8e92b737258dc51fd3703f289c4170",
    "722e4ce9ca6535e0879c8c7a23a233d0a5cc1f24b0c213c47e830d8ecfd5896256f1ff9a19484230cae591c6ba2830a7a9ad7bfc80ec2afbbff9b0b79cc83139",
    "f8e4d0b494be3b2eead6dfaefb866cebbd0a285675bdcce0c9e9c0361b87306d89d3df2bdbcfc20bed9476c9844c89717f605278fcffe608ca2e813a4413a9e6",
    "7c0b6a2851dbd69277ce30bece043cda9dc28bdb2904a43fdf93452c3d7a6f1030b75c45dc02840ecb509b76d4e1f42d871cd88869fa94659d6d08ede3355b7a",
    "7a818bb0d56508a2c8e16a953e6868c506d2db31f8cd423e81d8b332fba108cff1ca536cafa56d1eb38a42bba31b13e9012f96bdd7e9e6f865eb893c0f848400",
    "2be35831ae1e004e65adb37eeee6ea1542bbc65408da092ce1862ade360220a89a013ec6ce0d800580b1c472b4ec19ae04e62570acc464e9e482cc1efc26f7bc",
    "384496fc5e0a335637d25adc4de784f6dd070683c13765e2c8aa2a4bf806d27a6430aa630cceeeb731f6b5a6071c483ee9ddbaba64975e304f2da1d4c8ce6ce8",
    "0672f4723c66fcbf7c00e77f8be11dfa1d181f12cf6d7c8e2e0bb5521b6ce3c883acbdfd40f729a99bed00bb7a668b732a9b4fe038dbf920e7ae8a594568b0fe",
    "ebeb942265c275261ba4c60fc3b6f1bc00c16ad2e7388a834d92f1eb6a7158f6b4b16bc3804e80368dbee5e7a600ad30c102c665dfad29316f0132fa8d9e5ab4",
    "af8a7292e69e65e904072b86b1abf95733782bab77d45313de289ee09d408869d24b58f2b86383d65e9c5147baef3309b9d19623df5905acd691ca81c2e87a5e",
    "3e16225d06e94dcdb2da229e68d160c0f6001e001acc26499502b4d5e8ad8913fe2bbf88312dfa68003ca827fe7f3ba7941e897ba522fb8d633579dadbd1db84",
    "cbc30da9f8ec98a933ff774e80918f251e6a3feab525d944d1adb2f46186e96498bc592f72353f870391714fd326a607b27a48867974dc2d118cb5164155b5cf",
    "38930a3c6dd629a1f840553f402e55ac8e749bf22292bea9c49f3267b55f9524cca743173be75bc2e412ff2367fa742a6a3d7ee9d175712ca7fcbe98dbd73b24",
    "48f124c3e414f62329d340b0fd6b8ff957962db54f672f73a846c21693c0dfd3631b69fd7c3788efdb3c7871fbf5deccface180079e187cf731bc41152eb30f1",
    "ca6e2b9bf52dbf1abf317df2f724f8d11acc0338dc2c05895884ba394812a34567576f128a8a43493f607fe29aa57078ccf4959444eb748f61c9adb900112cef",
    "8b7451c6bd7af3f1cd01d30befc14cd2d34c8da55c2261c66aff8c93da26cba3cc221bdf980290bf99fa9ea8e14b18377258207eafa36bf59765badf8e9b7c15",
    "3001f5ec0fb5a79185cb99f7ea16b68cbf97afe59d49b642efee213a3b1f78b9fe2120bad4555d01e8154bf3d092884ccb81a876f4ae071afe7f985e1cac2ab3",
    "eea210ef8d6119d0ff3fcbd54637b244edc614d81333fd0ecb065fb2d753cbfab60990112d3b487eafed88b8c8d616c3be1b9152464efb68c3a103182d814ef1",
    "60133c6a2756d06338cb45dc527d35aac87ce63e00fcc6068213493636525a60fef362dba33289e67bbd829291cd07e042422e41ac87140783ddd56a982eb97d",
    "1e87e018b263499003290304b571657eb72c03defa93715bb8218e68ba3a802d9dd4bea8c944c40476e8cfd9ba6b89b19d70aef43030162c1ef133af208a56f3",
    "c1a039b0fa323cedf07f29e0d86bc5d8632cfa2c4ab5b6bfd2449da0db3e54fcd8af9594edf1d7ea61ab8f5ee5b2c6d80c5df9385af5e0de608d717aebafc1b3",
    "5e2ff2f32fd5de82577d43f21525695e825bc209de46bb28ce8b54052840ee852cd21a178540182792a367d041dd4303357b22fb894db99fc9946a3e20d2756c",
    "34cea9cf44ea0b8ba3316ad0e79c5d98e49b1f206fdfcdd5a93d00a7a6f0e4069b60d070e7014362b5da0ee73fbb16de961f29ec425ea31ee83ecc3a0d9cb060",
    "d3bfe9b8e6bc047fcf41deb7a49c81c7449bbd09377da4393802f0011c68fa1326e7c22f194d96442a87bec106077cbc547d79e3c48fa7f3b35fdf370ded00b6",
    "62b43167cab1a6a58f0a0ea1d53c172e6fce4724d9fe1df0c90f5180f48432de96b4e885b7c24c0094e69c6a1c3f6c5a8a2046d5588625d9c29e6c4f42f3ed0a",
    "b78fd183d52fcff6cafd00ded505e535cdf46574fefb6eaaeca0dd4ad171c875a9c912719535980cf691773020c08b17b42b589de57c9ed068b51bc7354b72f8",
    "a00a55f77be10eaeffe0fd08eb020adc8a3e77abf561fab4e3d3391130108662fb9d1a7f3c09ddd880495c2d45985723980a769d5f111e03c9ebeb1b4db10cf0",
    "c065998bf900bdf1e67e24c4fc5469342ab7f40631ab075231c8d229d923d1199a935b8ab9a43d3d4240bcafab515722f55462f0c10a9c36145ee4179754245a",
    "371b15ed0e0fc556fa45d590d1b7c6818fd982e4573ae5650d061b294fea4d32c013bfcf525b3f9838e1c425cf73385e2ebc221b96b143b8841f1832d09fddc0",
    "86b7451203980038ca2781233cd1de3834d5973bb6b90d1533361db55971636e43659232849b9e96dbcb959f3f491f81d9d634af80d92881fffe818b6b63789e",
    "e80d48c3cd01cd972b2a0740ad8a138df75e1c4922d4692610c6498e24bab7b105b3faf75ed2bc9adb284997b485135f8a4caa6d319484bc2a6e1b93e3b3c9c7",
    "f87f7627f429b0a3bf2f987ddd7fdaae91cf67ce808876f534948051ac06fd894524f15ae00a2836fa3bde70810e8c37363167dea9262d0d981ce1419fcbcb38",
    "dbe9fba91c213f91ec0148eff81f12d01b3265b2d7400044c68bd66dd1920604f14a029ae575c1e9a8e6b4413146b67aa6d6b29d85066d65791acc3bc5f4d70a",
    "b0df5680e623bc6a8ce15c3b608680f86e214e4921827f59caf6288314b83e26fe940901dcc53d48b5d9eef4b87c3899445ec22de4d674988f2eb178dbed78ce",
    "f786414375186fb4aabd9cd6e50fc3ac919a986344924cb4d8cf6d1c18e4cbf46bb92bfb3e8f39b4182b3fffbb0c557bdd40f34a3b58a0b8f3d878254a2e518e",
    "5afcbe5323713fac774e052dc29502a5a5018679f031b1cbc9fddaf332fdadb3b4a46b18a47ed49716ac0c79171bbd599700e62165cb0320382579e7bf6aecb1",
    "0f744ae99cfc30ae2e3e8e168ae02f6023687dacf4195dd6d2dcc98f478955507127d5424092733cd6d4d8f1baac37f83ae8c8d5cf470b1b6b90a7bc0e3ea56a",
    "6064dc13e281931c3ada23046df1f94cacb75c62bd326be197d21c29d19c875c054ae7b12a6f0dbf99344cfdd02451447f6672439a80b0399ed4d768bc0b4f89",
    "20a39dfcc2a5a829c68d6a894e24312395dc741ce08827a5065c07b77ef214c7ef7220e9df6174195d0822a173dd38689fef19886f1aec3900379c7a4c32db0c",
    "e05fedd5bc5615baff75bb1fee940512719ab37a8d91b4f4e9603964b21d282b30bbccb65d97bb54f347d66c78d17879785469cad981bb6e99b307e89d964ce0",
    "01191cc00f4106b2059d6fca9fe4cae5a32f2583d2f7027cf94abeb92c63cd6daeb2a185b9d24fc226e7913907ddbd169cd1f53a5bd3b979c1f3eda00c7e5efe",
    "1023191e5da131b17ddafae925f89313def9e6c0c062ccdd83096ffd2ee515353ddee06c90e5ef3bb72db0977221d88354322b6b43b54a50cc4e618f80f1df2f",
    "87d9932eb3751e4c786dad1b82719dbf7152e91777e128f501b328fafb1f65163c7634ba6470159aad467e25bbeb5391d2e87b4c8e7dbc640ad9c4d4e0dbd198",
    "771800f74df1fd4a2a4d5ae9d05575c485965e7445b84d523059da527c60140ca670b3cbd329b79aab0cc92d256dbb91d54128a2303b13742ee134f23b15792f",
    "4ed43895949ba4d63f99a0205c3db6cbe33c1afc50a1f9fb833df76d023dd976cc0b1cf33cd3a9b50946d487b507a20cf211a0d0ee11d11d8f5a1e3a97691672",
    "642eb2f431f4e191eda63c6ad965bcd334dd51fd301c83f81962f64de983034935469e61d4445ee1b3f6ddaeb6471a5b527c359469a02a6605d057e5b2a57de1",
    "e6376188304b06e4a70b01708c3a70cc45c97a6a0de0fa35905e5d773106767b32f818a13bc19a3629d0d867e5aef464cc82d2a1cd50db20964566cb0402e81e",
    "8d55843b7171c28d07829776da8660780a2ceebf8c3912742dc728726abe6be15955bb508cdcab295bdd95920c61a076216cc8026d4d364952d68a15ca6c40fc",
    "f523169fd4d95b4e65fb62e236947858c63fe0b54bb4039f4ffbabf4590624e8221d0f39573b94fdfe6193d4479bf1a9b308e72e5f73c41c547871fd8d0c9701",
    "d8fca1e0fd350743d2a16956c0bf26db0b220c359d46a762c45396dc0591f184060e4ff71126860c1db3f30ab24b5ce6249856d331442d90f2e202b2092f3eb3",
    "9b17e6e63ca0d0f7b9edecf546ca7fdef3ddb027a71f5bb1fa87e7ddd86a3d215c370732f86450873714a39f5fde6d83cabe502eff873eedb1af345b4ef40195",
    "5272e6a9854d5f868fd4d899afa9ee24052f6128dacaf1a18eac37db250056bd4b89f26984874a399d9aa278247c4d995f1000d27868894e400cb5c6c18cc2f7",
    "b2d122e7c51d366dcf19a8e1bd8b24db6e6ba2dc9239b6543366ab841d77fd9bc034cadeab1b31637c4536719d33a5869d57cb3ee2d53206f91cfa1bc3f3266a",
    "8ed54370307edf8fc97546dc77072708a5adf611fb446379d1f863918e69f923dc79de0cae49d6bf70973d14879fbcdebd4ec6563145f169c7e7b1dff988049c",
    "d45aec4a53fda8ced2e065a6f07fe1436336f5fccb6f4b3bb35c59bdf42beff542640317d2e5470413647c280503c5b004c00695e5f04c4b4c39a3bc781490e9",
    "1e9186fb08f53a4730fb6d71b1a6c5200af98726fd2c1f27e278710217cd59145af73f9c3438370a5ed2c60e815f49b9deeedc0ff568fe6c91a15ba6b1360985",
    "46cb528527c5749b37e5926bd83e225696577b7a115ae22ead7d50e17d5b8a04ab838129c63764b74e706b6110046d854a44145bff5e58e0821422ce77fba16e",
    "b4127ffab1fbdae677c516a97dd7241a05db6d924648a94aa0fd6065b84fa68e50553687b1af4c0f5bb2879fee43655b550795331c80792abb2f841dce045a62",
    "3970607ab00eaff9fabd45d02b8e3b87d9fbcfa4f52cbed702bec42c7cd34ac17a712d9f2b1b8fa4766592c6bb3ea6d7a18530b1235f6e6bad0eacb202d33dee",
    "f55177d3d33381203dbdeeb1108624736d77ab5d76f72f431c1e2cd3a96e024ea1aa7cd9c9ed224bb7e3074a6b35111cd8093cc72b93659bde686b1f09a6a130",
    "72c2ded75a3402edfad25a49492b4b24ca833d9d4129f939d848ab0bf1da1078b837b9a235ef42d92731ec28b931b832e778288934c3413e42335f0c0df6ad7f",
    "d74ad8ceb0b3dd0db7439c78c98db4fbbe553b8f305deb1f5c1ccd488efbfb02a220f089bb2f4966a367233c12bfdb7c3d11a0c46020f5341d6b37874f13794d",
    "740c7b1ad52c916d8b6211b56b0a448daa42976acb2886b6cee428f7e80cadc9e0017a5c84533592da76422430ea965c5ad6fadb6fbfbeb5fea15d9fdd8edb65",
    "e616c6a8f638d632158957db96ffd10f60740141ebff7bbc0faaa70e475fa8c7932e39e321f36ee3c4ace4ff624130b435f8971d340ed757da07c14a108f3a94",
    "3a58a5ffcd5b30b4e0e013214a473b28bb02eebb74f9342bc5e0f506adbbd286faf6d8348ae71041b4721863f7e0657c2e84883c3ceb52e9d206cd51839f8895",
    "0c309440b5b261751837cb521f3a56716078f53f74d3901312f6d466b3e6f1472654dccb0783c8db7985e94a6860a0cb22aa6826070c6404b2b8549f547c7b50",
    "873cd0f3643eda68d0322a34b46a30aef45ccd3f98a5f8cea9bb8dd24fc56fcf9d2590cdfd5ee4d6c844f4fc2085b097058341532048682134818d296f7a57e9",
    "4b8105961b5dfd59e9f1fe62f31f0a2db2c948e67ee010bd82eacf682715a08e5213d9dbdd912862366f68ad5856ed8ca7e69041de35c970a23194f20e0babdd",
    "c07899c4afd5f84c4946bfd2ac8514c41822ee0fb314da5f204ca7948ac3f48f821a301e2cb4458c7375325a5e17c0e3fa78edb6ab41a18f5e87326fbf8b65d8",
    "c660fd56b1d8e4e84fe120f7d3022636f23d6bf4d316554d7877a138f16d6aa26813b23b5c4f23b3f4cebcca2b5e0b7b0851ca722e8957e1040b98b31c636772",
    "46fc3b3ae8ae875aff707181c86e796a19c03fef09b6a44c0f2ffc55f85cdd1c2947a20d7051e3fd8ff692ef946abbfa5b35779bec507c6851f31db9c23e32c8",
    "92c8ed98567e861dca93fb74a012b493ff801a4960d8bc41cd63e64dc14927ef03db60942bf5822e4e8a1c6b4b70ea80d9d7c6d325e8df3551c40e3c2eab45ea",
    "fcaa20f63f3291baa06995017a0082a2efb36580941d7c12979383dbb339125603d7617a07b886c5224978b3934850dfb89eacfd51fac67e7a8cf60bf48b7dfb",
    "3fbb3c0d21920ad873cd86888b5369fae0a1b655bf348d6941c4d525158377c46af14be915dac7d830e9f76579d92434f44ed30761ca10914eb9678f8501c02c",
    "c3c2d3e3dab0c2cfb03b18ed3cc7c2fad82e4f76c1d4168b823e5373da4bac026364606ad669fd50dcf74331e6adfdb7507bdcedf53fca6d84a33357a5cf4074",
    "775aa5f15a9aec49e8dd3b89bafd60334e624d30eed06fae6bcf1726a802238df8943e921b75dc5b948320fcc794894c11b4a5a9af23c610c18dd79e9aeefaad",
    "38d709ac2e6fcf02b321837ee780524ca0c3bd555d3ef272aea5fef813a76ff885c94b2d67e5794fd8972fa8951d6e6dc22695824f72fb97461473b7fddd785b",
    "a32a764e09a58facaa330d128d7540e44fdbe88cbc68275b19142df510364488cb62b79d0b1be15b58af162ad8726164a1f3a755709ba324e1f69dd54f67ca36",
    "beb820245d8cb7f10c0aa55f013616cbe71a8419932021a2afbd2a04f7f89f5af2127c7c6b4f19ad44c334657d40da48985826e530d450d69c4c48eae16f4923",
    "e5efa8d9c2b2e464cb3a0dc0d33d9fdb200ffd07fc71532bd00d31b93b58c557972815a0909beeb829f4310c0ff3d29dde32b669554da15287639e8803cae9ea",
    "ea1418ddca33ef7c8e595cdd31817f87613985be4e34f0ce6150f1a3fef1c15be7b1f062cafc97a39a3c848f1eb79846630546561c6741326057946cae4e15c5",
    "524ff5cfbe967007d0e86391fcbfe59f811d8ee5cf6b8aee320621d44ff44714f06b58dc579a3792d9e68f435096b9793c347a1b88ada527214294615828ae9c",
    "8de656886f4bf03d9849366706bd464521521e565304d3f05ef8908a6db618c3b2c42da051b1622758c60a57ac94069d9d549e095265e1a33bcc33b6b2eeb8d0",
    "9322a39587a8bd5cd0dfb6a0926922497063fe4ad13f8c19fe8ec306e0b1cd452e96af47cd791bcd919db6fcfa1ee1891925b174d6100aa0a30cdbd0f61a2d33",
    "3864a075c2db4b9db723977496adb72d2576ec23ea39e3dcb105ab4f9d1b8c9d9fcfe7d62360aa0f496bf68ad4fd27bcaa742915805a888299fe3b53ec21ec1b",
    "b25657aff716a1789fe0e43ce35049a87298e8a479fba264011f6eef44be30f9ea8a85b8a339466a21c65331226cb59eeab131b8e0700fc9f633c287f4778fd4",
    "8543db1dbf2545ca55e69c83abe501150c6a8e64e48208a7fb80c013cc2dff096b857fdf370477747585d19b79643cb72e9d42c47bfce9c5a01ff63e840506d8",
    "450c03b1599861833ffefdfe45f8a93a61ab551e2344331aeebded5b4204d9f9378f30735bdb5f201395ee0f633371d7ff26f72fdd8407ec3cbec605de65bebc",
    "7e0d5df770c043049483751e168a0e22d25d28dc4b6d66c7d1bdb47f4ee2a6704c0e1da7d45762bde4ed49eb340eeada98e1d5caf8dd6c31d5f4178892a6060b",
    "31715221ac8e5f52e7049b0ab921607f15524b51e55dfbbcc7f024578a488b0d1695d6a54f0580565b4aa734417a0ebcc25e60ddc59b79146d49e3e482955a16",
    "3fb7f7671925126df4ec8e76032c5ffe93dc987498485aac64ac0210a62b79f3c30a75be18407781a305fbb34376ce5d8081c72782f0fdee73f0dc0cb3d69ff3",
    "b095a880942b6f4e82e867650b70eaf60d5856d94cea825cfe028d84dd386aa5182a41b0f8213b87d58bf9d9e8613d15d59c1d5536a8315a19f8e5717b2bc610",
    "27dbd2564eef970dd6462e42071dc1ceff74526a8b9330c971d0ec53dfb480a758b89fefd14eb4c76f99eddf11da7141389c270896f1ea263125a70d19276731",
    "db9c9d861edff12b03e2dad25dd5da8223bffdc949f8b8d437c678e59adefce4771cf5e94a94c8ec4062bb342c1213b28e73a4c512a5558f7c0ec0658aae65d4",
    "d89913185866539d140737fbb84f6f9888148d4183e3f60fd948d1ec1e455289fbc15a2e6bf8bfbf073bc3ad5ab9a540956c58c3a56d2a59a15d65334519b912",
    "f727bbb651fddd2954ffbe86f4649150d49a94cef7016f926d88b157083f16ad484b382f3285277b885ddc08e1ab4a842660ee219d4850ec04e7513e0a4788cc",
    "182c382acf765b35af4188b9eb443e75f7bfaa91bf695b36fc70f6f7d9eb5ad4411a54d8c3bd3d8685577cbe1194123ad5975513a5f60f087fac55f86ed6ce2f",
    "b180fcf479338e839e2ab93a95da7430c38ead9fbbc0a2afc9d6738c6b9b122eafb5360343322d0a3b672631b7ceb1f88b206bc6f4c0ffd8dac00c6ee3078987",
    "0fba97bd306f65a2d4f77a47343595767109775393fda8fcdffab0e6764fe06175b9d1a5a13aeb210484112de78ca288db297b36cf5e71d24f60e3f7ce0b080b",
    "7d719250b4ef88983cf2c5fc2084fcfd2d5b025888bb18ddfea3fe49fb818421144702b97478ab870d5f543617722042d1b55da579592e405b319828d77e95f7",
    "e71afcdb35583e69906a708fdb4dc81dea9a43525176ef00c357156e08ff964712f3e4155664d23272da9de97b28a8dcf6fb88e431c14ecb3c4748d03ab01165",
    "4f555ac1d23e14c5d6b6e5e4cbce4d371145759289070238c8974790c59ac739f8b3c143b294e2cb5bc5925f52d553e9a7831bf36741e272e9f6bd3c1ab552d9",
    "3e76b25ea8237b07da98fd2002247f737091b5cca3eb9e1094b95f36a7b104ad2be8b7907f07ef358f2d63ab19d16470dfd8bf496d511c2dc7fe79774da5e0ac",
    "558b523f5f5e0cd06818650788120757eb21b9a1f1766d53b5d716fd40b7d379ae70a6eaab8340c7a2ed59098b91dc7fb49f71e36f512cb06fd7b4a21b12f70f",
    "cf39fcf7092ec232eb168c9441c3ab0a53ec46cd463360411a174c350363dc2a80aff91c0d37b78390e039e67df3032e54a7a4050b6edf8f7a5c9ad851d782c7",
    "a0a020ef352f94389ebfbd97307154a7267b01ed506c63a698f72a66ef73173f5d166c67aaaf581c760461641429c359b8cbffbc4dfebaa3d729cd0ad6b2631d",
    "bfe020ef0419b9564c2167a8abb5b558ee13cff4d16423e543c692f6c91acec37880860c19da080491789d6206d86009f77120fad3a2302efd7df2f9c9fbbd59",
    "585096bb092c81e628c8986dfcd560c8c4ebc4564d6e5e4f4fe09b6add61cc6b19a883ac4d588e487f89a3ab4f29c46240321a4bde956fa172589f25133dac4e",
    "2d8fe2d4498e59df1ddf5c8e8fec2d6f58a8f3b2fe5b8192952f5a2ab8eeb08f6f3bcb5b1689803b3ad61797f4fb644333386030d4edc068da35cc03fc7cdc99",
    "85f0ad34958b969a8b31879407bf9e3ded2b89eb2f2c931b28fb92f356bb6356164ebbb3a66a1addf7fd0e3f9527f249932e8fefe1b765c5a92b5cd73e3ebf65",
    "c40a64d1a82bbe1c2dec2f322eec6ec1eb84f620daea3334118c62dac0ba57109960d8726cb7688720cac5d03923da805411310c55c2a2b4e93e5d4aec4e9a2a",
    "9caf9fec6974c55e9c34a8148597b92a0ad09df7ac861e613c8dca8d7dd0a0f95f1ad8b59ae5d16ccea671d0462b7a949fb080d8cd7a9006c5832197c02ea290",
    "018f1e59b1a675ba10ba8c5f0a9be47699a0ccbe20b290eeebb83717375839c8780d12660f09b00f20a87dee8aa52ddc6853cbcf1faeefe0574026983db42c17",
    "2475328f43b2c0d7c495b4668b79f01569e04d4e7e4ec28d300ff3f8e0d52e02ec660731d43583d1007efe77ecf998aeb4cd24331d4cfef7bf867f4007149dde",
    "36c695bbb416ca267c0cdd5f4f2c632c93fa697754b66a79b930c2d65b364ec4ddd5c9abda1d8dc48608fbf68f466bbd35f7bba0d27082f25ad546fe2d616123",
    "f3d2dd8323593c6fca845dbcb94639abf9fea9701723dac3fd9865ab5e263098ca6df7847e9c1010847ad5fc022b3ecb1a07cc0b3ff7ce16d93f4b2513aec8bd",
    "238e2e1b8745ccaa6afe9e52757143021cbca02bb0347c69408444400ccfa3184951ae6d31821f9658747231a12c913a8e3fa45c949bae9d69d527a645d52aaa",
    "82fa23ca4f4cb15db7d3c968281d698d26c4c166a686b2ea62d7aa30a8e1ca63865a0f2d831961fb390fd37750f1619236f263d984d2901e627a288352fa0228",
    "c3f92b25088f26b76714bf7501fae6832ad6886a362c1a680dfeb1e8010b86ae2bf77ab9fabcd24831d9d76d1062520c5e6bb61f4f846e7439aebcb7464f6b80",
    "e484a8d555a30aa2036f6328a15bc724b1272d883d166856e423191a983afc0e4b40a27e7697ae4a1fa9f5bd31540c03bc718b5791eabaa829a3f55cc141b9f1",
    "d5c1ded749c1370ee2a1b7551badba256b7ecadf364fa16173860785eba4d38f46aa13c1c205928a6f1e3c03f068af71342ac6271257f8200802ea879430754a",
    "6ce447f8832a1b9c8b95acf78eeeb82b6ce39ff180ee55acfb300f9c45f54c37017c84908077d28ab780d3ec08ad2116464fa4bb23fdb4e8977a68cc20838937",
    "3d37c2d4acffa5dbb75d94f4e0f482c9b3aa2a188e9125bd8d8998974ea3c21c607423a93127d5f03e95007862193322b6ed4f3994473b74f2f8d73b9a4ca6a5",
    "73670dfd3cd21500b177e1d913598d8671a835dd039654602d669b605d87918d142ae83deb358c70f4c511d40b0b94ceada6df1ff1be108a058a9959b0510ed8",
    "3bcbfce17a4ee91c4fb7bea12cd27b19afb3ef2eac2fe4ae165103f1b53a5a4599a948bf262ae3bf213fc15fd7ae692601622ce560f5d6ffd9ca84ffa1707cea",
    "54598b90bc722d3452e1822bfc2ed3c77a528c10f256648a5d9b31929030d38aa710fa446f971286109cee419500d4d9e08a6434bf5edea06c655f2bb9da9d09",
    "4c45a8415fca5deeb9856b32a3d2dbbc58b95c551107c529dd8bfd2f15f174209af644b507ec618de42c3edb71ce261293ff5d513b72ebb7a85d116b23d8b206",
    "87ee23fb1cde678aa04e275ffe865c84e80c1b221c3e975d763108c7d5b9abb8b3c5661600f12639a998dfcabae1f75e49a4db676c2d76f7b67dedb864ee2315",
    "29a3ea39155187b7eda7fd424b629447751b38a0a5615b83a76497cd8666864da8618e9b7d19ad7361e7f820adb79b90034be7c901d7498a654668633f26901d",
    "ab6eef639b102a5ebaab225a65ea92219893de1855914aef4d7ccaa124fbdfbbcbd4a604dbb5fe7698527d9f4a17476f81d25fb0ebed93d6532346a259505b57",
    "be67661453a1cf692749f8f26e49a1f9b4d8f3d76f6c257a601b8af17141118159721a2cee340ec27476c1c704afd42a59da153a7c860c25fb736eda549978b7",
    "64aa876d4c865533ea91a919febeeecd229669aca774b679aa313e152932d6a914e085e3a44f15a8fe45a1ba725f02855afc581380adc8e9b6d22801ef2a3c98",
    "de163658532dd1b671d5c88fc64a6a22ced09560df23c4fe131021810bdf5fa9ef5d57658286ae0a11e8db6609ecaed55df0461febc7c8f5b49f061fc5376867",
    "27834e43e817c21bde5482a12209f7a6ad1bd90e7dd0643a1f02e390c3f609e96a34690b0e40db35f2e6f3d7192b56754f430a1fd8e55def8c7db5bc514d5cc9",
    "3a04300f41af218ff6f84925cc9d44b9f8bae64f5bc5d966fcd02e637d7a4a56d6d1adb67e103291aa4cf80be877ed7b9f62bb4aff2c090c519fd9d8d6715561",
    "2332200d45c6cc3bfd91e0d9ca3d1348d8e644f99615cee4d28df5564352df15183e31058b8237888ec8c13ab2acee99f2705a6e790ee8b5b6acc21d0d64ec15",
    "5c45b8601a99e4cd61bcbca9065b72bbba827af5803cc50f006962122940371898098a241fbfcb6b7cd9ee9a35ec9a907acaf30ab4cd3d547c10ed0d7e0d8771",
    "a39051cc2ef9a342cf59982544c7e2f7fe7da5feb93a517279dc1cf547d4f54003f3e8b59e5c43700ba7e4a6dacf69a8a66c145bd817e7ba2c1e435ff0823191",
    "42726f408e3ba0633a331d31b9acb43f90e2e2b11cf754b95c2559b4f9b6be9452cb0d3bf1fc4d3cbf9c236848e2084d00083a14314e468dffb07f5e30a124a1",
    "f5ae28b716b2c7d6af5f6b083a8e60134017cfe23a66f7fe88752013281bb63be80be858681fea24b0757725a0dedccff29f7cff80ca9d22ef8848cd44ff7678",
    "b2680b58fe6553bb4d3f0e5aae32cdeeb4faff055e339748e6e955c86b840b1625d26dd2a2c5376869ed7523ef2c3d461e096b72637831345d77321b16c411f1",
    "bf1fb741403b4893e2a25e72c1746940cd613a44a3c346d63500d7ff24350936b7ed5122a4101ed44f92b9af2d6b7d8b0336c44cf9c680ed9d744c83e0994fe7",
    "8f5fdd7a96fbded47d36bb762bcfe85f0f017422260058f5fc91c22a2056206a5a85ce1ad5f83427b2f93c50d803913eff0c8bb8da75583d7af96f572cc9da32",
    "5b459841c55745a7e3b9f5220ca3d2e249e9caec5c5de3063bf106b328296545ae18605dd594371f1c7de3eb28dc4793da8978876779a0d72ed5ed68bb46dd2c",
    "9b67e7053643cea1da3bd70dd09926510ff3d932ebbb2889a2f2d363d972f90c340d1a8fe85ed34061c0b4190f8437d86cae69c7de5ff61ebbf38ec4db8f95de",
    "b95a82b661a50c7ee50e86aeb9972fa08869e7cdc58fa5cc6f581f31c320ad270dc45ed89591aac2e28fd43812d6da28f5991e4d01ea2ca73cb1835e405aa0b0",
    "fd271d8703a5a5bece0354abb1cde1b646701139b4823e64bbd3ef1b7381b3f44e49633fed61dc23227306be9502b96d51f5ac955531dba43b980717c4861108",
    "0ee6248470a61219304ce00c6434c716f802a8f8631ae82d0c18406349d9752233fea96b227399d01a4f3cab4623f06c575539c00e273afc9002ebad14b79fb4",
    "91529522edad2aa142086543c6972a21f718c0cb3293c8acd1ba2f43e51349dc5bb9db9db4007afda2fe8fa59f367e8755868d41f4e1dc123e11ca5333081935",
    "8f51c5c6e953b45c07064b2a72062e4e55f1dc3aa515a29d8a84c7f23d10e55e77d972d31474a0cd649fd3eb7b5732c4ec706c573997af03f5d4134b6e76890e",
    "62cb1e3203affdde867dfcb14ef0159e42f4fa0373a14e94e649209bda53f1bf677348ba15f1f154095a1d42e05a53a3246fd933036c70c7cace215dc3c996d6",
    "a4d8a0e42bf62a1ea00ffbbd9fbd50bc5b311486ca5606387198eadcb6918f36eee6c689433213ea91afd18db02a9b6b862ac41ded99ec27b61cd445162e21b9",
    "97f242a9a5025ee0083ac238f32c15ed709ff33e6eb08a0707b55c5f6940cb0a28b959f0c64c659d43a5a7620edf39632ca5f3d3bd4398606c14ebce73e843d7",
    "3afe81ad50661c698db2de07b7ccf2038ec43dfc23f44cf97f8d6563e3bddb236ddf8bb0609748b4a1b9f928910a7a8910ec7bcc9d72b519968c8049f5b1910c",
    "d09fd2c50c1a71a0828137399abca0fe7d011fa6614bd615e9962ecc1a409eb2d3082ed9b4a6c60eb9821c7222dfe5fc497b8531f0b5c1bccf1ca8d1192bd670",
    "3a870490ae0dfa72954ae066ad626fe0c63ce0b3c32e0def95f654863376ab97b66127ee1be55ba7551e2c166060477f2005f229d9870d0dca665cd652ba44bd",
    "4a056306462f3ce887a34f476c956c80c6dfaf47597a25a76b40d144336529bd33ce1ac42d45cc084fb3688989bbc51f64e6e4d5ec909cf414fecec09a25508d",
    "75943204fb08d1a7bbb51e63a9a0d369182809bcb73844be57f63c36803046fd7be9daffc2f29f2169de52bb3dd14249873a8987163b288e26586eef8651bc90",
    "f77e420eab2394b11b9b8d614e3ece5b343af5ce9067e7385a8c19ac6ab93b199da4e7cecb934bedf6c51efae5e86cd19a5f9848745e831c1fc3cc3b4082272c",
    "b39b976b2ad20f535df332d6c4227217a16b1d852f27b401796962a8a613a96b52f95bb190e749b75fa9df3369636f63d2765ae2535cb48eeede5c5eb6c3990b",
    "06e11a90495498cad93ed5669e2821e857d61411b4e558945201c1ae09d3270d0db1e30e62ce496adcba793d4c428c3dc9d30bda32bdad4e259ff1e12b288455",
    "cf84f8345fc01d41e03947e95c085fe6979e95e36d042df147acc570894664c3afc5da2e7ed7104e97c65ea1b6656a908ada7d77945d8dbb7d5e7f0d369c3311",
    "3d0c675690d74302ac00510f2ac4d3ecdfbc7bdc65844973136f534655fdc97e42ceb9f90d1880ad397131ca587fb31594e8102c84a5910c66c474e2550eb7f7",
    "fff0605d99c80a6b4bdb5193cc61e4001b1d4b5bc1ebafdb96fb1c7b4ba387baddff4b35c5eb8a73b3e86083092170ab037a8023d10f71c9af93fb61368b6ead",
    "2702d543d91fbf736874378f16df97ef13d2814a426546a75bc5a1cd53346f39d78533f626a06e008176900c79cae06ed1f48ee142a454176821f3200cdc116d",
    "f1f87640339ab6496778772a3938b29812e98e7e41bced49724a4e78b83a2042ae1e006c666f250900d3a6665d546db999cbc520ede7019c3916fd76357ad1c8",
    "cad3de4ae250484d50af7c27def83377e2b64461ee771c42913ea245d5f6fa3e821bb27e14e8911ba9297e877dbf9f396df43a626aac9b5fe898cbba1a013a85",
    "165f7f8d309c4ec0a7e4f7f1d0c4b16e34d3dc184bbc6d76cc027e36e45f01c636f66e1628184bce7ae9a5f48995d8ed1badc007d9547e322ccd8f619761711b",
    "1fc94a1de2f01eb38d0f0e9c8289cc39843b3676f9a851aa2f01d8b64914f60532fc3224676340da0627f5d2e4bd90ce7a75965e5feaf53a828e67098247b6eb",
    "1e3089274c49d527e564e206106b719a5c586ae989d212f16e7f4756c69bed2d961f933e66230bfad3e8e499751c87c0a6d8415026720829947dd8bad827402e",
    "24b54c7d60ce4289e982fd5d2635d3cf2a8ebd42500140582c264a6ca0b8b755d7cb395da9512502ad623ad5953b5d51852e82f777ef15a79b03c72e9bc60eb1",
    "27f493e31c94266247e82c1bc6c50d47ad7c5d418aaa5d2d837ea15d201fee46d582027f0e07dcb1bf9bddd970a1aa82f1365ad88df634bf4cf9520faa8f8bde",
    "5275d41a8f8947ee9e5da8cecad1fabfea8b0e2dfc401ceb82135fecf5d858206b9ff312aa7c2cbecda763fd9929e15ed47d5256caf6d5fa3de2c7fa97cd51c3",
    "2631703abace072a07891084a2e32a884756a56a6892b24170b495196dd9c37a802df0c1bc2078d45c58908b33083d7de3eb8e7c052ca67ee47b87ff73297b6c",
    "1ed6df8572c29be2144f06de8eb34e4a0bfcb049dba78ef28e4a44825ef15bea05bb0eaa64b102d4fb50b279a09470cbbf76b7441223e960b92533d7c4ce3bde",
    "d8afbd4ba91eabbcee46aae03a34c9ddad94d4fb36bc1a1ab0b267e1b3371aaf3bb425189e175d1355027d1a4adb2f345fd913ad8525097fe30b82f76969f706",
    "b63ef90ed5b9849cd00caeb4a7944962a06044af8826385740c875996aedff0d3745201b5774e7fcb1fb7a9f04abf43140cd05b47e54257e1a570a14407dda52",
    "5ecd6a363e02857c64f644223bec4b776b18d24960bc49bf4ec0ca6638d966bba4eb728f06b0dfd27d8b54c0cc0b9fff4d9dee39a79543fad19e45a0ffb5f048",
    "781e7a09b9ac4a9a5c507e6a4a23267c99b6988048aa3cb5f379424cf7ca1a88665a0fa7fe49720a2844d11f71e2bf6fb7f46dd942a6c1ec650c1ad896f53899",
    "51c3a3d0d2ea9f327d1550fd88d5da4c5c0d2aa909f6c99d03b3b15e9ee9974c03344c933a76c2124e9fb217fb3ed50cebeb13b28f6b7fd00376e23ef869fe4f",
    "a77fa03b3f5fbde920c225fdd7b688c49b3701fd32ac2a116e577b7c3e0ff87d940f826de8a797da113e59405771b8cafd1b0c8fe66dd8287e6a1ef7ec45a94c",
    "76b928cba43eb1874792941a40e2fdf043d4b1d45e833320a24a6866d37ffcd17b2c7a355119c854d28d847b50680395f49eda5547b4e8511eeb7a1fab60c7a8",
    "7a6effdbf968bdd3c17dd727ac3cfe6f3bff076a5b49be09940e6febe2b7ab6cf6551b30fcf38c2382f836693196ff63941a38385be0abfc4b93eaa8e30f13dc",
    "2508d88927c77d58ca2effd5ee26d5c8f9afdfb21094fa6e43d2a9f627ed5932f7d47ebc2bf3fd7424b5cd264f5ff56c44d4346e7ac94149967481c8b609a27f",
    "82278b497c5318368d931b9bf017478e126c72fe56a74c5817dc48938e48d39d4e4b2108596b56b61d7d0e3109ee030a61eb7c6319c32d19ead2b6a5c99dee36",
    "5b2b264dca528be73e3319f11df1651c63950df9fac2579f2f102144c78367f7e728e77a85eb610519edffaa0cdc7cbba9d4c50d9cfab8546b5bbd6534153881",
    "998563e90c3efaef7facb06c8a1626e667e1b0e0f268c1edcbcbd3dc461a7c35fac9dd6acd67fecbdc5988fecd85819431c2cfd151c5f62d7bdad9d536bdd5e9",
    "a5082886fe89a2c888f0f1f96f4f14ea207931f4f99297a7418f8dfe489e1d19ee7473feec89369cd8a2a03e7f1dd2285628cd4162511457ddcb1547aae64a49",
    "ea5437f620462a9b98c96b7ad3be665cf148b4e941283eecedfb988ce1d57ca0c30c7e7cda7ae1f6f3b648e5f72f9f9270a1e2977f433e8a32b3e2687f09b935",
    "2e6a20e596839d08327769aa49197f01b89486ad6078c6d921dd32497713ac9af0e3b288a7c3b6ed3f07a1c3f94afe2e429c3d378eea69f3c3c322741b31c779",
    "fc8e2c0689374bcca386e99cc6c82a51026f399d8ef597a7ab39ac66cecec4a19268bbc0e53589fc25cfc90d31e8a29b1caaa9f2e54c7ff6f5bf480837ee04fd",
    "d650c19a86ca7d7c12154d5b6b1c11e7c750c2ded367a294262b0fc2ad75183adce0896c6594f71312438ab1546490e11b4864b6f688b32a668f29c574ac9650",
    "447c2d62b10421a7affe85b338517b613d5f1594deaab689d3025db9e25755c74b33a904cc5c2b3fd1079241c805ea9928962be9b87cc4a8393011f19c367138",
    "d411518829160409fc4407d703dd4096a48eb43a2baf3aef8a11e9384dedaaad206bde17922e286ab04f78f22ce7f391bda45e5d809e86af780a0d11cb681c34",
    "5248f4f6509c5e79bc892626fd60741b362b0fc7ba252f62ca07e5e702fd7e86db28c2dcb8e9300ceebfa8faa3408c48a587824b6638d9767fb76d64273bc675",
    "2db6c04f5b354ff35defb3573e23010094886f8889360be23c61d22b49462027dedb543c1aec84f77d6fac276b46abccda0502eff2b4df7c7108a1482b149ef6",
    "b639d5e516ec393681c12da197b4c97e1c34a7b50ac855a2bf748be16d54167842a12d147c06ac5b72cde30ba2dc6309705fb509333b7e6ba2ac8bf3e70f2af6",
    "a18d6f40b35a033fab208dec4e04d8e9d1852c3d9447ab7aea41c9ac24a8d790ec5bda39683d87cf94fd5a13dc7c1f0e932656d616d38ac70e2b0ad08f4fd6a6",
    "12dfd3b5240e9fdb1283953470524c9f6ead28a55efff70078d33afcf9129264b31a7ee73909613fdc42556be6a985398c0c61ec108a15ce4189755fe720947a",
    "b6f581604fefda0339257b39a32c86038a7ac781d0e71678578c93563c74f6ba88d37c030ed21e0c322e4cec765f2e722b17b6b3cc8c71b970ba81124d9b4512",
    "c4b76b736560c4b10fa892b87659fec80c3566ddbecfa8b68381735f0f0d950b9f800e8e80a39594702ce1010589c23e396b2725bb36cf50cba1efc4b270b1eb",
    "f3af352ca433774a3b0a85cae5bb9efb292e02ac82bc7e2a5b7d280036f6ae5c18367a0b2a03916405187aea55fbb771698777e5da695d9b5f8615e7b6c380af",
    "5edeb585da856749e29d90062bd5680059161816c4c4b22d623b8a2913f7cd412d156cf60411ac8a3a452880f41a86511761ec1861991609b54dc58dbe32a09a",
    "565e1e72c9a846974f926eb86348a973079b1827e6716b299194972a6a5dcb57e393e9b78fe3e461a1c30c95cf36d3e1d9af63dff1cccdf7a5a620faa041fff9",
    "6dacd3664180150d7a271e56f47ee6110dc5deb1ebf9ee562711010972cb8f91d8fa27358d5fefd204c7cfec22370c9ed0fd1e6d2684730ec4373783e4143840",
    "8d81fc30cea1def59abedd3e1a22ee8791d8ea0a6fc9f36da4032bbc65b3189164b873605f1990f449a900595edba8e67663c7914ee3c3f1321383fec25295a4",
    "77e88ef49c0a56b4f2c79bfecc9b5d89c29148365e3e6b44ed008835dcf3b417b645830d66d98a70b05b1074cf16c4cf2ecdbd5eabd2174f8edd1e0d51ea5e57",
    "aa3cfd3157c6e56a74acc985edf234dee2b385b7ed9440681f5a30a2d563efcd1bb824fc5300dc42469125ee370c83d93ee6f2836b41c6f76a80dba90249a3f2",
    "c41787f9b9d3c783b61639f5be63a3f707cc37b109f1eab566a75691a7207ee550192aa2a85b11f0eaec44eeaa135d13bf20757b02c1baec31e0c0597d1efcdd",
    "91a4dd9ea967ff177d8855614cb1e888997293cacbb8cff4a48f914676311f9232478769b35198aad5cf2e5792e65aa751808928785ce01378b3953561646e24",
    "8d764e06f67643121f9f28f90ae379e4e8ef1c6e91056ce9a509b6a72e5c8d1cd1c255279e5eacff4c14e9873e27493e6e808df97c03b2096e157f620be18ce6",
    "676d59f823e852206c3c44187cefadee5b544af26da1ff257f0e96100f74435cd20331f35edb6c39791e403eebb0fb6d26dda1e0b94dd9f8da509d1c5de5d5cb",
    "8fbb9489a651b1643bb3b6dfda728056fda1e3d1b01c7b5b3ec8eaa45a18ca0f8ae074d77f8b3139327b918df635c33fdc7786f3a71c5d55559099c1d677bcda",
    "fbfb5150527b363f73607a50573196ebdd1174a7ff00369c46304bb3796499a550bc86a0790d7fb511a53181fd99266efc119c1e9d67d0144190e67898e04b22",
    "612f2218ceadb9c862f7e8e445412cf61d4a1d0d8327ecbf45e16ae2dd31fbfc49409bff6349779a919b2022d706f60e4c1f350f054e88ab6ffabe9cd945eb82",
    "dd00d6c8c07dd9121a1903b49354dc78c7e6f167f5ba415fe679dfe1a5715bb7951579cd6f22e67b531b5dc541be36b4337de9dfa72a23b41262030d406242fc",
    "80fb701e392cb37c85c30cccd66f3af6818ea51d78879ee72e2ae4d044d538c07cc9f136658dcc175aba75a8c2ce9d9a7b550039351e3a110bc16dc661fda9d0",
    "268b6c927688c42222910fbb5b770dfa075c616c7b4114fbff8b76caa84aad969aacb2e09f58877ede01693e1e20c4cb6ead6082eec2164ae8c22cc9ffa8620b",
    "6e575698c074a158c08111d4b283481f781656754c78ef299f9dbe0cf7960365d1fb9e7948a069bcdda4f6c20a3cd39310794bb13cf654881cb79b42bbdadd72",
    "58bf60ac0995c54d8a6f1157d458b619a74bbabc053d35a343d0b5cc6574d45f88b085deac0051de23557b1eb8f2deb7642ba75181de1081572ac9751e44219a",
    "7e35f7a4ee5e6a63832f3fbca170f4d04acf359cfd702f02f02509c1348527a88824576c8c8bb53c1237a46608b56cce69a7334da587442ca3f6f28e3b3673a6",
    "2fef67a766271dd371aa2d622708215d53a8920a210aa54d1a21daa4d26bf7830be033de0dbfb9d43623c09fbb6a9e4df77de9d6dbe279cfa92de0cbb1d9977b",
    "4f82be7d201e69e0fbf5650624ba44b5ebd8e4fe924b8422de0a16b9bede8bb221e2fb15679742e77b7302af75fe7110207a5ce5092fb7e7239ea52520c5dd40",
    "f9382922020bf2f7760bf4869385a77899da78a7a08545a7cc4fd07b851fd058c49dbd019a38ffda9d697c8a36943aafc1e78cd3cf96bce68765abe84ba1911b",
    "ea530dbf760115126cd19f325332d6325a6b3e20e81ab3eea361d4e6e14fe52dba6e1b56e2426824867306e9f9e329325ada7c1f97cf163a4ff8137767238977",
    "27c0e6179f8d9ff55da7473dafc47956881987f64c0efe96df321286835cf89e7e5ff91948fcba553c7ae20b506a599b5b54cd40efc94c29e5b0f5a820d679fb",
    "77bec61a77dbc8c1d971d1b462f8bcca4b44b6b275ab5b6ea7682b063e445b8236ef38ab9da0b149168e6ecbd63337001f36df6d76ec9940ab1258ad887985c8",
    "9f80473dbb0e3ead486de4fbfb486c6d5ff965949ce66cae7b79775fa7bede751f0dd365f5e4c66f5aa7676aaf4420ed722e5a28d1e12409ef21a7403013aef0",
    "0176623cc352c34b3b1554ab37239814f876ad040011f1c6a9588579579e53f1b4e4ec1710ff696bb7b09e0bff6f6b814243644e0a05b5f7a7883831b0d7c221",
    "412db1dd5f50030092da1457ec2b832b5ae4069ad1f591b2b89013aee0999781c7e923e81e33c0e8462ab49abde377ee2c82b6d2c7d498f12dadf3d2470c0b1b",
    "497d7cd698aaa9d30dd6ec29b19891bd51d01288b70de721c2b7e699e9186108e58e571f48e1839f458e5afbd02684f2fd18fe92378e35b08211225adc806a68",
    "b37cd9afbe65b2367dfe86ef3c6dc436b760640634d98e3d2cf761ad29d06938c959b623c9f6d9ab4633f39a8f744d14502656c0f8f4a33521b977636bd568f6",
    "c8dcd942b985bac25d4371cc21c1ccc3ce6f438924c141091836ac3c59540050b73a95f6e71d10f0f93ae011f0271816114843e2ad62fbef01b536e5fc0a17bb",
    "7c739528a23bd0453d7d0ec6710cce54f6f2bffdb60e291d7671bbe6fce768b1678b3678f75f71bcb62a2b50b22fa9177945be48e8de08a551cde519cb63ff99",
    "9ae3d2e84ba5881610d65e8fd921b1f3c2989a841f5dc1342ed7a67182fe14e476446eb608bc2643cc040df8726a0b28fcab4d205221ea180b3f393937fc13c3",
    "6ac51248bf6f704d1299945d0d33fed10d60782557ee63b25978697a078d2e3e0c30b4957b5f71b47c5e299d986b325ed66d2326a8f6134b1b8027d805f47b0d",
    "36dd34eb6a215ae6f9b14eac1bb25e013b671e5b8cbfd70fd3e8e5cbf0c2e58b50ad4d48abb234e605ce5eb0b1a9229be47a9aa91904bb57eafa91a04e0be4d9",
    "814d6bda17415830cfc523aa92fea3a779264cc4dfa0c58b0345e33898922356d6317f5ea55498a4d5ba67d1ae6ef3c25118e8037e1aaa31c82da81099621ba7",
    "981a6f936553154c981666de38b6edb1db700b9f43bf56db49bf587540241c54170c44844be80621ada77caf94a89912d3a893d7199f08dc9128b58c1f5765f2",
    "1e6b99e79296ba46514d61c5f7c8fdec369c778f059f7ec58be6992dee3e022992cb501bc728c0c4ffd8099d049583d7073875522601920df5f205ecc1c82c6d",
    "3710ba07169c285f21a217cd65ad48b500d3cfc01a4cae31a7010e6d6c660af1af3bcdfcc7f562a708dd6aa295a517e08ad5874fdfbe908b1b3fc7ad2323d74d",
    "ca5b72cfb4f5f4087a99f1ed9f78208249123632d0a75d2b474af9c285272288f2742a86d584dc16e4e11109332fd9a83c31baa3865fdbc9fe9c650460e50ea0",
    "71d174209b544afa8c2db75592b53e4a52e6c25b2e05319270f46b0877b82cea13f1edbc70e6fc693b96a28916543766307b991be886dedefd04e9e8b4573652",
    "4c36648440cb2fc4f852d5e445003d1e95f1f776b4483d51a4427220d432b9174033a1ed162249aff2a1f5a54e15c5cf27c666b3bcc1093cf1afb471eb5af139",
    "ecc777152ee8377ed0e9cf5b64b08786dd2bb1e729090edf79f5d90325b6c969911735c2f4a7c166e2f9d9824b2d77df005754a05b26ced1c51d581e7c9871b1",
    "6e3a0c8474b8d9c053379d3d0549267c5f931b213a9ee93504b3ae04fcb95bdd91feb721b7d29c30caff3feb8e2e0b00bd3a63046a70e0e6d64e13a817d9a176",
    "44adfef6c341f0acbf0e0af8c101048f1e2fb702f225d3e56d69a18700d80794ced5d75dec80bf5e186d7d948a4a8ff76ac3fd6f60a8db30e2dbdb4471d9656a",
    "11b7656210cce9a9f95021c83d8a124f30f1271cc9ee28b8b8e0bbdbcb17190f0e4e949851be23d96a7e91c5ecd978ef25293dcaf914dc861fce1bc62c35024d",
    "f10a21876c69b2fb309b58ae63d0beb1582b8a52e2851efe4480ce7347a26676374f4698b8e38c1c09ac84dc6f2c0fc5d1632411261d54d17764cf2bac091723",
    "fb21dbe85a19bdfd634b5e21540e6239aa8dc6185e987f2ec18c7fcdd5cc95f6f17ebd3659b85e6fa54e554fedac6545f96ef443f20e55e2c90c3f375decb069",
    "a8fb62e2512684f37ec83b0ba233e22c6d05e0bf989210f44001822c3adeb938a69ad5a72adfd5824f4b9407e6cd453c82d49d95d10b297a85d18cdc4770cfe9",
    "d9a182da0959006b0b148a5a021c10549d8f81c1cad29145e1eb6768fb1a701731ce1df523fd2174915898c4959ff275bc708ea32724711db3f35876d41e8ec4",
    "1057429a4a88d1b56d6a9fcd979379daa62d5128265d7f47fbc34d8ea302d19d0eb8c1dba2d7367043ad4b24a55b6adb15c755fa9cb2dcd40a79c8e959d7905d",
    "442b61a301a8a7e2585a30ce289f8a4ce316b19d1a2b2191b6391b882d12ecd61d715b23a0367c413700cbd98177551c0bfb43a18e455096b3959b7e50f36dc1",
    "ae788eebe3360edbb19fb876344a4d354f422c4ab869c8a2b1539cff8421c7a8633c396a9a799eca9e7d6bbf749dd67c67e8f09341636f9e273de13e5b4b9d0c",
    "949868c70c88cf5b0a700f58a6b48590e4d32206bbbef337743e6e406eaff0a9a9255b707ecd0026551939ff7f6c9371240ff4bc06e57e43d883ddb102fed16f",
    "b08eda85a47306f9380487e067f8dc9f6580341d85a615b0a7941b6b4af3788410ee0055212fa3b3481385dc4bb7f6e3d3ecde28803b819a8d6d1f5d8f3b6a60",
    "383a305b856e11580bc975e63a322bc9fb3cff4784ca3f88dde224c2558ac881a6b844131b153f12cc854eefbcd680eaea9274e957e70ede3c9d6e6bafd9d18b",
    "1232e00e244ec9b87ff8b981614e9f2d2a7e684f504a1468e438887b960251afeb08f22d553bdc76a8efad7394dfba96560a07202b2b16067eef3a55345d022f",
    "144524aea659d974fab2b637525eeef19200778c377172363581584777bfe8ff610363ab4ed7c23b40fe3ec9640420be93a72dac911e1afd988b2252bc2b865c",
    "0cbda27e1dd8932805b7d1e86ef0b584e7f3081af7e4fc320d6866fc1ca42891131c0e14e52cd16198ed1f68ebc223ff686f828ae2aa9747459a8066a9a9d37c",
    "98d03cfa68379df25a09befe5ff1c997c88814341372352435a9c6f2f3af53c937b8957aed81e2aa04fd809695a9b3949e2f236de5b4001822e61512aea2b957",
    "86327d9dccc9edac1269641a64a3f7194dbe3b4a922f811c7b624507e0652fc579f3b7fd69ac4ac351f19fb2b6e307c49ec9f7e1a3418f95767d34c29808806f",
    "8bc2b592d8530205d02f1e0d24f87f3bed3437fe5d7abab163214507732f29dc8f3fccbef5b2c281b26bec8e302716c7a54f60cfcf6bd28619608494e6493755",
    "805dd8ba728e43c5a735476ab02a0675b50cfbe06d534500d262a7ca82188b85d0f1fbe742074cecdcf33b75714ad223816f2aed116a61e8629bb7ebfdab68e8",
    "35ad38d3614aefaa5db361e7a92340050d5306d531a92c6517a734d6a492274246ee6c3c1d2ecc7bc086fa475c803139e24998d33d268b5f66e6a42d47888ae5",
    "bc46f8b1b38f37fece51465b387809eeb3cd408bfd489a468b023c0a1fb81c09943e6379d7b4b5f5b79783e3f443808b6d12ad5b4be5a2a37870132ba3f04f3b",
    "5bdd7ec1b0a3eab674867eaa6ebe8d030bbcd0ec2539d15f1c06e1fd9541fd96377067b1d1268cf774bff3fb07c9a202ea45a31b8573ea25c03ca1099b5697d3",
    "57dc9c7b0c607e0360a0856105fed74fafb5ff08353db273605556e1c48baeb9523dbb99c98c1762df819304e09077f04f37a8b5d562284a744e3110b750b713",
    "298d8ea4aed627cef9f84b6404bcd2237fe11f9b27d6f4570025c7e0517a26d370530dfa886c2b6276bd3b274268a5eb79e7ad54582de76c2c0044b77bf92456",
    "41aed3df0a91b59e3be0da2a9ef686404e577d74bbbfffea2ba4b49a1251756e6d05db041870797a17fa98e98d5276773601bd041ff9145620dc2527e42ea09a",
    "e95665b72099ac7be3a0419484dca78197f7e22947ffb3fb2eb15fa646122f63640f621ce446df87563dd769bbce217a18b745d95a8420f09dbf825fdd8ed793",
    "d99939b65c98363cf7bbef14b0b5c50205d1f47d0d6fd987bb685354aa7f6ab190cad0d12b7c220827410aa50fb5391864c2af1de72da926a40be592797fe9df",
    "e514f4831a196c57c859fbae7cb6ea3165196ad7160fff3fd8ed8413eebeaabfc5d274b416f8e37371c58d68011b27d208a68e4d112c15a691ff967c39bafa6b",
    "8a9a49fa371a20194af602a11ab598f08aa95f1c69af24762479b810ad3076af2a0ac7aa96c5b65563f1119e6b4bb98f563825b136c76bff8416bff275491a6b",
    "285c58bd8625ee27756505943e2646ac8dd89ef8715ccd93aebf84f9375b85de3c3e20599ac293886f186e71e37c704a968cc842fba0dc13ad1aa890521783aa",
    "dde5a0d7007be293d4d4cd97035f50b6626a6882dbe389f5bd54a0338d0f39f913f9f0e5a4ab6aa07fc175a129e0c1b8078956fd1ec2858e7f113ec1608f476f",
    "a8f88e623e29a9e40b1a7d7e3cb7717c4824a913c98d9e9eb99173489ad5d9c192769ab4aeedf4edf14b69ab99c549aca4108c27b9a66a161c58dd05f2659f80",
    "f36060823182e4da3f03247bcd0bac6adc2ea8cd05dcdaf7124e044e2dedaea1ae54e47b50f71b67307708701b4b113d241f9e3e4d89bf3473d9ddfc0de80236",
    "677aed1948156039dba2703fa54c0b0f35f106c436883de0bc305951dfa3b889047308730036db6414a0ac0eb96b6bcab86d8e56beea9c2e7a7abc8fe0349932",
    "25153bf7dab0e9c7e82fdf4b5735d69c1dc692a1a6994f2ae446fca1c5066639c92103826d066a6b9a14dcfe2e0a96048484f47cdc490342961c937e078dfecb",
    "292bb7a80edc7b5181ed0b25c6e869daf6b9040207be7394217bcd32ccf14ee4870bce01249deadec7477d48bbafbe9d0658f1604a0f026fe991c68712ccde91",
    "88a0fb48c559adfc2a62c1a075685b5fc217a67b5a270ea476fc0a6c4e7632a4078e73d554bbc0494d8991ffe8e455f1d969b965b9cc571935c971144107f70a",
    "12c5ed85233a7a81d7100671fb78e5669ba31172591294c10507bf83d7b81464869e5f8d927f8778b0e1bed012c7f5155f247acafac1463e730fc353fbe4c5ee",
    "aa254ebdaac2a7a506e83c0b9c8f3a1d34fa8b1d1d57e14c1a38876ac483bc6b26fb935c780151e2ddd28e7a22daeb7d38d879719b6ae3c127c3259fef1b155d",
    "6eb3ac9c879c138eac6418d5703f501555f295153de66a59849cdb2812032d4e6f9d0592cc6479b0f9a29d3f0928c33ea72e629e92dbf9a9e24c130f7117fe96",
    "2fdb3686db4d4b6f89c18af20e870d9f1d67eee0fb7edd3cea0815791eb446bc08c35fd021db48d0bf69eb3d99b6af956173acddec839b91dc16096d71b82405",
    "d50077a1bc3acfe4edf99da0e5848e2836074c07355ef6946ac9c8c24b9d217280397a3eb2bc572a82daf3d7ae2f3be1be4e10bcee28a0cdfee0c5ab6a031141",
    "78a5a2f9d39d4b0d3b8349e83340018ee8f2aa5c39691c78a8a16f071db8161b975352a8348a5ed2f378d5143aa60242d4be853fe1592e23602c23e9ec9cbd10",
    "737f9b1b2764b9c6e8b740a9b13be4c01f946dd82ee8eb6d9ebbed7da82a78810096146dda0f1cc3c39b0ee7172bdf8c9ba9fa0e668daa5ac224fe931146ad4b",
    "8935541970885da8c3d2f1490b6a91cee86e3a9e716af8ffeb8a526818b94dcd2665011a67dcfd9896e897137a98833c43f12e7dc63525c3ecfe06c860450639",
    "5a4cd3a916bc870b5c8b4d51ae9f764c84e0fc5dc2aefca96797bd275390366015df1f926f0aee1126fc31bff796be782dc70361481f5db9aaab1ff5f779b2cd",
    "aa520071c984c7f42c4e2e44ed88773e5fa615613d61762e9e03cbfb4a181d17ebe19e9e24ec225968e50db61a63595b5ebfda93dc308f03755f9fe3fbc78a3b",
    "76d5898803ee2ff0f842f785726a21ebcab21965442bd6f2abc5f8fe110907184d4389796f7d545d6092d4b1f01ecbb4672e6fc0b950e7907b8a8e358b194288",
    "3c4f89f38abd645b5bc21c87b73313457a89e5049699399d7bdfeaea6ee89cc8bb716e6de5f11e95928504697da82e81f94c96fa228512055ed827f4f153b7ab",
    "c96119247bd15170189ea5c7d6840c835cd90e7995b8cf48430385e2c52e452d69ea89c14db46c21b7cce7ec9c6bda64c5f03d95b057434c9f65127c76ff317b",
    "698049f5f4f20e07bb1e0a446faac505b90369cbe60219b61aa7aee79cb8ae49bcad904c04b850351d8fe45667906472b9152947a74a5573ff3172c2e5543bd1",
    "19c3ed2bfa5015449decd63b17a608d687b510db122174583845c811b28455749787d58cf1da1c4975269b732a75b9ca954176ab7a3abe28dd7b7c7c0e6aa4e0",
    "4b41cdf5482bd3c5a29c145030e7c8a26dc2a4286d38c896be7671732e8329016cf1330ef72ee826badb5a2be09b27ed61f358930e47ec04a2a7c22b38875df1",
    "357af365249ed16fe701eeae4e964c09c4be5160d94cecf4c48e384c399bfabdf61ccc699eae8ddb7b66f8d10a4b14dc116a2c83e3806164abb64e6a13299504",
    "8bf3c92dda25bfeefaa8a17498ac709a5294e8a75797ee72e1cd0ae022597060edfef88b31d37f9251579d79cb4b5bb251e3303dcdc244473d5c076d0615a908",
    "3053c43aa58127f95ab9268bb956abdab16660398a4b5d6eab1393771019f72d031f5706dbc0b824b867139249751e678658e959d9d60fdda84cd1049c73daeb",
    "7899441e379f65d32ce80fb355017a90347b319a61ea71221290e045d258a04a9c5a5f804d438536969360220b53714d297d1c436533e6ecb310317e1ea862b0",
    "62663a72f3c31637908dd6781caeb67b434490183ee4c94bb5cc939054ce3987a989f2971bf3ad7dbd29c3cbdac8368f9699e7f56b746a8bc03574f1e5a6c573",
    "19326696227c6c1b899c320d26a4f3c72774e51e1fb63a97a61ffdcd3876ccb4e16b96a5868fdb327b96065be2718d63c75dc02bc1f222ad7a49aed244511284",
    "7ff92f2211f4cd49fb8a3560d6f74ce2fc2ae87188355537a13fc5e25a04eca3f0e727ccfbe7fd968a8f92a8082e7ce8acce3adb102930609b63dfd886e79572",
    "6ece4551039a2835c5ba9ee517eafb0a596b834e44673fba794e3bf0c0c840ac817990e463e7eb3cc9f08d876891eb5daab84cb78d6dee24f4f7b152fdd29ef8",
    "8c3f0480827e1d022336142ffa53e94e86facd68d20702f042f6837ece973701e76623eff54a324af807e35e7b03d43afcdf1446e0c338268646d3ad9badf211",
    "8d58ba7713a87bf936a3f55afc8ab2d2643ff79cdcbf77b2d542ae3239aa7cf24ecec6a8592f23e2ecac62cb8e578341ee5f60a638b24f5d4874f14192e6f56f",
    "8c3d444bd6ac73f03fe69faf8f525763ee014340ebb33af6734d683ad9d501c48b8f51e63bce4a231d5106aae25e1ae33dc56f3c62e951783cb8d16762ba49b0",
    "3ae0e45d6891eb7009fc4c035364a148114338fb74c8844c837c0abefd6a761cfcc646f5c55eca1607e1a231894d15ed624932b0ffe5428f00ea8fc7308f7208",
    "6342d038547a0495e2a1f4cef4feb6977327f7d5597f97027e7df51a0f85ed7262d6fbdfd4d6c16500db0afacf289d689369845f803be4d2f7cc068f50cfc379",
    "bb03634e4f5b33e36a36ff9d582cee388a40aa7cd4d3d4bd1ea80732b9524ed821cebb84fca677d19cbf47f1b702e861018c9871e099a60b61201e7fa40703c2",
    "3b93e42a0ce550c25abe87fe28820279d67be46b2f8dc10064e7400a5dc9252a6618d69b5fb66d21d11687f25e09eb4baa44fbf0b942aede8e4b5ea6db65be3b",
    "c4426058c52981d40aed9de5e0234058ea333c89b4d8861dcdc358a330741d0b3f91d0f7d6bc68b3a9fc9f04466c094bfc0336cba17044fd1050604aede4c43d",
    "61f14e48612cdc72de15543b0977dafd95c7c1b4baa4909ca40b9635caedb5b22a8152c5f3bc7a83346e093d2862a5da6b1bdd5391d1c18a92e009401cdd33cc",
    "ee153062792bf077decba8e8fe73ebc75a8424550e83c3de29dc1e8cc274cbe311460917b0e57b7a466e7d9c79c6bb46d68b28c71ae5b5c943acd7986503849e",
    "490f54ba6769886bd6c07b913ce0dc2e53bb859ae7edd0665284650198ad1b7f71125f0ebda315ff2c7a34cc735f0d793364ea9318600ce384b5245cbb004f9a",
    "bfed8cf8ab02e57d1c4ad512ed021e4333e5b9fdb9663b20e64e17880363bb081d5d1049736ca49c7b37d3e688180f775a9d88ff3d985df0af62f2b577d031a0",
    "9e40809e29e0b223ad5f9a56c1a9e4db4db24a215b2cd8df6231940cb6ca2bef975dd7e4cb4f226c7e68a3024cae6df2626372c2e279b6010b123ae963ccb422",
    "fded248b86116c7b84a175eb18b99ed0c8f4b290cc660f6eb542f1f3ba4ea9431d4b4079fd957f540edd41d59e2a1370209d9bb9a634436c55bf8462168bb2b4",
    "01858cb0e099cae844c3bc22acef43af46fa24537f64193482b13c91dca5a723f73bb588d797907c7bfcb812b016061829e70081e260ca8aac9866ec3511ac8e",
    "fb851882d2bf6898008f7c8fe68a566c7cfdf11718cb2cf097ba3b480ce10346ead1ed1812d48b0aad742780141811c8104985409f497e23513f257e277e7434",
    "a80d3f31bf56f81412f66e189fbccde8b77628c63f7fa7f17e3501b10ade70a9f7d87f7b401ef832977b6be531106221c00bb92163419432ea5922a6a01b8587",
    "4890ee5f1aac767caac5b8402443de3d9531411faf3a2fdd3972d5498cc81ceb653c18015b229aedcf7e6c75e8f6dd735223437d19225f08cd45c4acb0d573b5",
    "950b0ac0b347474551760f90f71d6fcecbc0236863208e6cff0af0bdb3486cbede97e07515d9989932fbae1b3f523e3a61d59c668c832b78458a0f4b51bd503d",
    "d1cd9924b804e3f7b6a98e52d419167a58247fffe0c78c3ded328933dee01e8fb2d49e2d36b16a6642c2798dc4213e2b7f14272bb4876bd76d92964a067f7348",
    "b6a811fb39ed551525f68f10abc657d0845316970ecb787b454cfba511c61da89ecda41e52c8a8945aab094142b25288ec02577b0a07041aa43716ad7e7fdafd",
    "5115ee763d618b5fb0c09269e5a84bdf3eaef4fa9e8cfadf82bd93eb25f8dff5147f121aa1d24db866577fb3c9aee47d6fcf6a5f45145244b9a886f337cf0869",
    "186ea7711b9ae37740de14f1835c0775337c299a6e711132693f9def41ce97f1138f4643f39ebfbea742f68be13c44fe06c6a09323fda9e786bba3f60729bf05",
    "c1962575e0d7ee83762c8461da338c6c14f999cc7165c6098f218959ecd28faea88b3a170e6d67db5f82bfe2ddac7adb0fcebce2729133e10794b5f2b1f6a67a",
    "d39a1313cbdc36bb3c15944a9377b30e888393ab0126ff98bf6a641812a0c27e0b7f637192e9c2ea24b3e26ad04dfa632bd0b526c93d2b7cbf143eee15163b56",
    "dfe02fd6ebb227b51436af98a0943bec944e7e8f11831527524d72f11420ab260d7356ea68b1cbd0c2c4a334a44c46ae67490203084baaeef6d97357a3cc3b83",
    "dfb04b41fc182e28e2c4b63de91b9dfa59d6429efeb2da0316952d63f1594031676c7ae3e091ace8cb8cd7411871b9ccdb77bb0449a6adef62fce7fa5247ead9",
    "cb6e3ff27b5121a6d96260d2c7a6e75ffc1fca1e70b667c7ea78973f721619702554be97268678ff6d24c524121aa184a704fe1b11c3f950b1c40b18f5b9f9e0",
    "742e650e27699ca3fbb4aaaaeda248fe0557785df985561be55ba96f76647e40b829f8512614dc31024cad9eb37c0fe442e0fdc47900a1e78db85f5c804b0ea1",
    "30f883afee9e89b1ef0421fd981a90adc39d8bf1316f7474f8db4fe24b7d811e1edc75d00c42c16df4a9f10f80ba80b7eeb711dec1977fb40403ea0d66169e8f",
    "2bcb320735c9420ea1cbf63ed7439cbc325b2230f5a526bdc64266b2d54858597f14c99d34e9dbc0c039deee223349a5197e4c34632063bd7b84064669b62428",
    "c411cfa89cec849e0f97ff6d463243c4db47b5a8a82d2bfbe0f7e5d54f3a781d8b16b626eee259cd737600b06c0759f6c5794a3064bc420399442f78db97827a",
    "c2571d138a1f02cf7aeae4d74238ce67bfc092acc0a655720b3a7078970db22caf8753f8c18b4a12863f50a530d251acb90dc66b052435e83859b274ebc7d9a2",
    "1861f8b6c8df5c3e93039f13749e198a21a07c021a54be474b4b8eeecc8f3339b46e3e8050d03e153a961ca7e311971481722862b26c343a81bde72954434268",
    "8d2a11471d0102a422860ece2c2738b73039f093a61a71ee5ee71a321fcb8e7f4bfb99ed07f980957e1961c75a55bfd6ac1749bd756d36b0c182ceef4146a641",
    "ffab6fa5043d0dbcb53cd7d9c3596d19100d7f4999b66b9216ea2ee958ababd29526462e142ea90810353e1fcacb97fd0b2c3fef755c46ac1d6279804871f08c",
    "0df7800b582d70be03cd48b0d9cb693a0ce56af12c0464e9461fb5deeb790f37cb73232ae5d5e1d77c57b32b78da631392b58459db597b9e118e87f2b7ea9736",
    "bf3e7f60d3d4124d30c7d2cbfffd3cdb8f791ecfad1968f223aa0843eaa2466f422c4ca66f6f74a3b2a2d09521aa7c56017b0aa747e53abd619b4dccfe0fb1bb",
    "6c94bf277ed33d20de8efee053269d041362f78b2d3de83be6dd2d23bf6141fd9df18a83ed9692cf0125bf3d4b536c6c4b04280a9e7ae512bad10a46ed7be3db",
    "b0305d65ce4e9f157c582fc9b51fb12fe65de4c12ff407f8bc474da168f227c4f70f32838a53066a79e593cdc4d59239079efbb3faa3b79f43af96c0b182cfb7",
    "5d7f3ba0e921e0bd8507b8986e96ec4c55c290879463a6d6027527064f2fcc2ad4ebdc63af7dd595bad50f066734a45a686cd3907375d4890e07c2db7ed3a391",
    "6dcd0dd1b5ea54debfc526074854825226bb14ac846978d8a5555209a4df80618db67d769e394232525a1960973626d36171557095b996f7e5c75e40b3980d0f",
    "4fded824fff7c496725784de7458aaf6f39243a370ee66b86941f9cca0abc5b97c937725e758faf95b1cbfc8eba5e1d0eeacb518da73df84c23c23b3dcb13176",
    "3220b1884303920f349db81b178e3d14d409d400e082025010ce7ad82f8b7b9aefef02af9714651533d94cdb081671ea3aa301ecee98aa360996ede10b01d262",
    "31cfb42e4a8adde6a1d833940396849ec26b2912f312ef6e8fced13f9a24a0aa63b5f41289470fe4a1e2fa2089cb0c08d83e7f3b947df44c42f337fe86949909",
    "82e9f7dbe7df25b8c88e3a515f68e4d144ebc37ceb6b37a707efcbc8f0bb99e831c530d871ba546e65e1aab4f2e5c8399653441ead06d28e4a5925d31acef795",
    "f99b3127b71c2e54ce11b6db5a86c742002daeb181f454f248051d909cb0b9676177d96931e34113cca16a553a5e71ce1ef1b5520e9f7179fb2b1edb8e2d853c",
    "47f43654fb25ac765f93bc5195659a2598aa0ce34b1af1e5c5adaca0c3767ecb1e3a1c134e77bd687f25efc6c4ac57a09c30b728686a9b87413f4055878732bf",
    "7c2de083c60714ff2a00a21b0dbc0473d0d6f9ea85e3fe16ff2cadfbcc412ed7125953ec8d5f0fc6a9ad91151555c3c7bb100ada1338057d78b96384e4b098e5",
    "5e5745e8f45f882f1cabe7f9257de2e3d5de2c40a80582ff014e4a4774bfdc9f54be5b6ee82ecaa48bdac3206aa8066368fa4b1ade239983cc5df92b4305004e",
    "9e8ae9118aa87c35edcb658d6db28f6a69f58ff1c12249ab90ee0541c8c449e21c065bfbe06b8f5fc53ae3f3d18168df126a4f5c4bd2589da4fb456d02461981",
    "058bfa5d9153360cfb7d18f4fd5e5bc87b6e31136c38c0d0c607279e8ac420d2cf079b409915096df0d206e1d5cc6610aca6a49fc1b6dc4a886625a573ecb71f",
    "6be65d28e52d6749682caa0a4355b3f29b52aad32658799e417e609aae1a133b6448243039b100516e9c075963e7a524c560fdb42a4eea7454cae2a00035f6ac",
    "9c838b4dd740df03cbf8bc9dbf2043e24eeb6b07521f8e734554093c921868150693edf1b7d61f72f26496096ff727c2935e910428351e7a54b6c9045f1dface",
    "b188350d81e7429fe76fa63ec144f3972725eb54ac92af92ca0ba8f50d03e1a11725c36db287560565e604fdfd41e00c2ee8987dce9948231097ac8e1c616051",
    "2ea1c4b2b39198f82c8eb93248055bce0571d5134e3834399094eb81eb25bb81a579a555fbd65213820f3624a191cca0fcc95e08eace434e3c0f7bb10f00394f",
    "8ef74a1221b1eb94b5dfb3d8fc1302fe9bd3062b460eb6480450796a89adbc54d9e16eef2477a3efbe06062ee05651d8328fc304c4d3c455b99b3a7999a3bdea",
    "569426018856881ac08867cb986c6955774b6d9151dfec9ee7378bc235f12fb3a0af18206fb0aff4abeb0448b51acf69bd8602ff8a612176474363324315e017",
    "a6fffb03a285555f8f8d36f8a1bfb6b97bea881723b71a877004cafbf75b7d13c223a7cf5d438c1515ede3ee0bb16a3238b3a776bdad68eea7cd70ecbe34c20f",
    "e97d888e1bda3400dac8f3e8ce86500f8a3132174b5f8e571132ee88ceddc970a9f3032e2eee83183098293a2b276c2f6c292b557519f3993dd4e85b7fc3c14d",
    "b5431b14b3505915a85d997c2e2ecc078d481abe1b4a51cd262bb31c7220e92dd15775dc68570554b38bb9b0fcac8e9242fa30902522386690685397bffbde0c",
    "a0fbce3e8ca4517ab1e081683bbb391f45b673a5876ab008ffa7eb3e4e9be6732e6299aaf3d080caaa28afd1d74aba962e74d7aefcd98d386ead95ceae36820c",
    "27040d29004b9ac5360a14f101165680b92961a480aa343a93d6796a083cae9bd29a2ff6b9389fbd07e226e4b5cb4e6e099b4ae03b85954ba1b30650d8ff0eba",
    "fb828acfc9d303ea7bd2d232b372906af0899ce452c5f701b46ac116513ded31bcbbbac60ec0b057666d792d55884dd48f9054b514f826b7ba1d8367274313ce",
    "5b67490ba85cd9bf0e92678c4b398491eb0957b9fc9260fda00cd2bc9cd58b3922440c005d3b50a92ea440204634f953128b6de877fb92cdb68c2fc5394b4544",
    "b386ac33ed37ff35c008a94d1c57fb60d90b2a22b210eddd7a753aac76a07387b3ee7325df5c327024eb34a0811078fbb87e186671a6b7a5039c0541f82643da",
    "cbf5def8fb900af90d84e9f987299270b74632dcb00891b5402a53c5443b581df1fe745f5f417063121dc1fbd232668af37f3c3b862f18294e58079f9fed9640",
    "a1a2ea92f22fe5011a4d2963e8c1f7dc9e0d793fb7160c4bb2fb05f0b200aa0440e78d88ab0813c6a3ea6a6ba50dbff8753a6b2af63d6a511e121a9d1a56832d",
    "aeabd527aec3f1687f21e707663548c4bb7a72a6796f1990ee3c9d28173099e82cb789cb30eef1f2994c0caa9318f5d4e93b514b4afe70a6f5add4be37b20d17",
    "5de34e4615f088ce166ff09eb1f3fb95e463e1676cf5ef5d446ec790e2d71f37288e0c62c7cd088f7d72964252bdd2ac4490511c05027a3a2ace927a62c1a22d",
    "ad70f2aa989266cd8dc51578dee9e8bb5370fb09bdca8f86c6b0a6246446f8afea8eb9067d9066a25c2fc44a76ef9d7f207629963b543ad2cfd62179d21a0cb9",
    "f4ef0b39acb57e2c9286779a10a2d95037642f2ada75c7eac7d261d0852b6cf52b738e49fa4e8e9d6e0a7a9d55c6439b4492834ff21f633e9ff0f8c2c971c76a",
    "e6c4c3b28292efc3f716acda388ecd3fa877d67103f7bd4015748cfcc997179062ddbf0c4d3fcb2208e3eadb6c032e88bc6b98b94974bd53a48765e221b1fe2b",
    "8b49df91810a74e87eb9489786ecb7361326e36addb3c6d1cb0972d67d5f3de3882eef580e178945faa5c64686e9150b6d5f3170e6eefa3c559f4fd6962a2b48",
    "3c7abd496df46e33fb4abbb18ccf227025f74aee08a4818a46f51303fe5529bd597f647f27d0886843fa5e228e4c53f6b4023e1a160e276f5ab9e6beb9404abc",
    "4f4a23381ca27196826b9d5c6b95b5d9a8b573699cabb14a8091a4c70db0110c2ca418ae0366f58de491549489a2ce18801a01594220bf78cb792ef9ab32828d",
    "5f7c16594384e6a08683850a3ca92898e86159b6ab0e2f9b7281bd369bbd5f99c4b6bd6744025f1e71c62fcb4526e359eebcbfedf659e38c790d0c951246f4fe",
    "32925b06af1bd4cb1a5b17d90c19c158695930114c94e2ad1da71f5d86e4642246f7ca6519f3db4acbe4e234ac9e99146f0b1b8bfc50ed160f7591e08987c4fa",
    "47c48cf8c3a099988513ee548dcd4ab9544b2b795793c9d435c561f2fa123895043f73e1fb4dfc3307ab74bd8f979802a7c2083a73a99147388cffca2c3b5d9d",
    "3c2d81767bb3541273571c1672263b6039d65e3fd9528a3d1bd2f54cc6d5178719927e77d352fe6984106a2df0390ecf33df6b7d9c809b0c22b1f4c8802b5bb3",
    "cd7f59920c30641a1529ca5a1cff9f990461856fa9c74e080274025d656f7787276eedecfeeb909d3b9ec53344a0c8034fb3ad05aec40253ae77e5a9e8270741",
    "e2a5ba6824fa18675bbc4aa78efbef5982572e89f580fb30206e756b5018b0fcc33314a0e40602c2ad1f89462f05139525cd77a7e45efb7b621693fef1554acb",
    "a9b8e8f1f5a010ce1e879aaae48d769375c4959967dc44c010041dd80f6c95fd628240b9307f4a5973988370c73c7081d50ea0f1893580ba0d78dd37c758c5df",
    "21ba9fba51af27e25ad4ec6be940e9ce339eb62758fe239b8133b903bc14b668abcd1ba65c57fedd2983260f6176f7f3d59d91919949b89a89c9e3ea5a79a566",
    "a70e77b0743eb922c97e27d8c42f678112728af8268cef0715d19629e830d99e55481dc54f47dc2b94d85a0dac45c86b0978ffb2b6c3f1c210e7856dacb07e44",
    "f58c99831bb172e0d354dd55f0718e6e87ef8b0ebdb53909c73dc0c033db17c9c88495c9c42e4a7cd9a32e33658153e96919ff7129812f4aa80190b2f38beaad",
    "0b2c0a96650b2729b6480a28a8cd32306b3ae5e571b16295be6685054d499254a91d4a8cef1020a7ef6d6c83fead9780cd4fef2891d35de1f763d781d117eda1",
    "8e5517ad5cc5eb145810593f93f5e0d329e97cad3d81b2a5dc3a73332b12ac8c877667281e68a679fd22f749f70509cbf511013ff5f6352016dc0e24330015bd",
    "eceab31493ea80223759f9b33f8d778839bd1f1af0fdc06b49fa4282e1de6197ef04b8efd30a9de81652506bc0728240d120e15976ceeb5dd23b3fecaf9de077",
    "64d84761231a84a89e8ba8636b9e39385c2dfb68c202940645232ee73987a2aac49ab6311dbc44fd683c9238934c90a637932c3eaa8738cb4ffe3d706dabf71f",
    "0141f2840ddae9fa74ceb873486741530db217b527282c3adc040782721d73dc0d6892e20ce597f08a2ae3abb11452a30679dedcac5e0a700a8e70f870f4bd48",
    "4856a066d4aec775ab7e9aba857dfb612eaa0d1961876de037dccb5e4e85f6f8e71e5cce90956a5702ca86bb52793c784545716b03e48adaa070eab40b91d747",
    "9105cce4a19740aa6414f985bbd71fc3576f0f6e4ed50d2fcef1536d969385ac8b53733422c3a7ddfd93262754fb439ad502771659bebfbf1d2d14293dd0ba8f",
    "5205717a7b112a476cf3db37972c0fc4000d78e4d6305c0bd956442a347d16d98e2301975cbb129661dfdd4f4974945a70496e350f657b15018a0253d492931f",
    "7fa01306f5a04e9f4819fc76fc510aab71abe6f79fc3cdae7e5c1bfb80234b72f4c76471e7eb9fda33f96d9d77ab3c4a6743d4e1747881260c20289c87cb4377",
    "ab25ddbb2b6557a1b0c2a316786c3562ef29face34cba29e925091e1404aafcce79a5fa179675dcd523e539af583038424ea749f108ef6b41fbc0999de80e4d2",
    "0fb6224baba47cdc5dd2ad1078381336ddeb0ba1b03623c0b12d51f655d7f9fc5011428d2bbf780b2d7fa34d945d82c2eb29cfa4790fa54f628334c566aadaad",
    "b66f72a070ad1555f35a0a7768eca8fb010ddd168a77bd9e9263d89128640253498bb62841bc29a722dc358370b0f06e4783d206b6056b8c9e2c96e37bcec6d2",
    "0b9716b864fc07ea17af66e9e210eba6bc8c42a57e71220069c083700ab4291b815970a8a06fbc1118d38e2836705dfa669276390cadc13bc63031f63ae7abee",
    "046546b3c5364ff8d5346111841480f8369b5c778bff91b67239206633d227c833c523d961da9d46e322d184c87278eae32f434fd0c0e7ddbea5f5c9e2c26ba3",
    "71f3291038a05e11537ce023e006f6930aec03ce4e8d1a9612d7ee985ace57ce747be5ecf7b92fe93a7732463af014333c9834bf31279d2602f2c22170ad0c50",
    "d117235448a6b351d51ebbce95da89c0974f1bd0958f6bf94ef3708c18f342852b6001fbb247babcbe1ac4a4bc22eb1be1794ce63979e328905779c75e675460",
    "02c666340853bc48103668fe86f49a2a2aa56d760c1c6cbe22482413a63cce1d48bcf8e0df915ce49405b2eebb14fbfc1bd00c4a62d43e1338c63be20d1ae252",
    "572d8c6424d04626cf0ea8414969df44e42684b12f6d730eccfae94ac7a22d1d677ed69822af8c121b9856db5e158e0937caeb5d64b0bd761ba735e3371a1d2f",
    "699ae1e54db2886911c0f7da9119b48b197e38437abcc1c58c3f3c2f05a5d23eed5590911f14727a76189485eb2c17bfdffb64aa90c3afa9566f645c8d66f425",
    "e014e87074e627b8b25274fd7d0ae23a0b5bdac9a63ac748b285695ee247d94517f2575f09e02c588b89b74ddfee7716897e7dc2b993d6efa04470d4961e7393",
    "f12ef235ecf42a109ce1fdc507789975cc38725cc0de46c11d96f0d1bef4c0e79a261e61088105b09b9294e3e26c3c2f0be26ba9f91d94bc1e2bfc8af9b087ed",
    "b7f94bac4caf4953c0f3b4a9fa812ada98bfca21e5624aa2f1a756b9d587e08509232e29b076bfa99f841f6b471ac0a42aecd5d9202cc448265db7acb66c6c56",
    "263605d509172171145ed381bb4a8e9fef0ab205f917c6e157c5e7ec7a4e3e88f0b1f2d081d609d9ad4ea20b6987540f172db965039348cd0fd012c7b1990ef4",
    "03b6f1ba56ffef1b45ae8325829972c1d4881a0699e033479512d01b8b16982ec225a525bb7ecd9caf88411c339414da128b0792675c1041cc37c9a7555191bb",
    "f2af7d765193522dd496082a5b4d17ac2565f64f95372010a75c4d47b7a5bc6b70229c6fb214cc77a97185df3ae207b5a99c2da0f9904633cf71c86d7c4d11c7",
    "bd4afbf4b555f9c5211666915d295d0e513f924c18b0b9c00542990b946b7d3be57e46ec6f825ae9e55fe032c767ae11773d106628bbb5c61cf279bea12d9ea0",
    "e76cd5cd176ef27f55815cdb28c50cad15ef96171b7e113725e20f4325e24a2991bf7306d13dc270818e73245eb6743b0d3f201035cb3a70ccc8cab419224394",
    "ecdc4d3fb7e1d964a136fd00f2e6b0c6856d357f4b82bc608ac08faf2cb44d72d86198d25ebab945b6c40ab9cb8aa7f7a61c1250919b8176f320de276d32a32a",
    "334d3fb3e34f50fe8a77de9349d640b46ea966c92c05cf7eb7d704d1ac8a3e441895e6ad7b12ccf62863f844042673e7ac351cd4ba9beaf6f22e2ce26637a9b1",
    "7daa0203f555765f9e8fd2dc0fea74db118ab788532faeda959aa9e2c394bb3691e4f99d96360e311237f0c91be4ee1a7acf2e5a901c314afba00c0c7bb61985",
    "d5d9c48505cee3a54033dc1905a384d2d944073356967e70f5d433c878911ab7549890e75e659ea030950fdbc183d5147bd70d0bc1455859914585a904a388b2",
    "857b68e7b8a698e24063764002123ac00bb3db29bdc9e694c3a639fa030ea5c8397b6e9a85dbc78d21af96a501ed052a36b3f419f1ca5d2ff59b21bd1c6bb06d",
    "9074e207918ad12ffea33c88213d955e7f36740e113bee5988b28b34ad5f900efe385555161961c525b278ee991372aaca31a5b37981b40b1087bfa3ad41e980",
    "65dc2fb17c92554be8071fee29d5317edd5fca9008e4fb28024c6b42a3e9a10d254ea016b8313c8c06d1a2f3e7616794218149f05998a475669279275cfe467e",
    "55f8f0017b7855b7ef7928df9e3a5a7b37064be4cb78621aed0f3df37f1c8e2add3c454a1aba99c8f57d5e6c3430411dde14dca2a800a09c9d857caa521733ca",
    "ce6cefb1b339e19824f17feddf8c87da587c7c0b6d64f3c4f609bc95f43f01d3e782bf903a86e53696b20175f27978336d9d31e836a8ac7259dc0f64bf533ed2",
    "a2f7b1cb261968c90d60a87dc6504fea507c69581a6e77e08c259ed9307c0d0d7855f4891ea4ee2c295602206c73126b38112776631ac57fc593d05b3ed792a1",
    "0e0d1b98b857307214a5193bb1a2427e875a32a99f583dc634043882729d26ae6d7ec40b526c6ccc0c9172dd5d88d414f9a3d449798af43fd236cc8b52530506",
    "e97fb8e15c3aa2e626f45582fdbbf726dac086ca38bbe3152920e6035108b96f33da49797fc5831c99844f6cf69841b1f50aba03f0d177cb619e0686c435b647",
    "463289d29766f182339ca6a69ef309efa93c7aa7683050d1e7b15e7142ba47d334cd9b65dd1bc01b1be06a8cb153e858d0936405d67fc877ea5a3961247abe48",
    "e5fb8d9d3b2bcc972ec748dc32f257b203358d1963109425a0a8f4dbbfc80ef80f9ed2a821f1d072e8b04551112a33aab905c26a54bde0ea98d24270f253d9f7",
    "9d3ac38a903f6770d613c0008268cecb9674ff59e80411db2825e237420d017e31dc7d1133a3d7e790b70226fe8276ae5ac8530f5e75eb57f502573a50b0d3ed",
    "c2df5da54e500b6f4804072471622ada36f4dd2803c8383e058583831c632f497d9b9fc37f4f4fef69c57c4edf1bbc26ceae7ada54542edefef5e425ce917f0a",
    "61919b45017d41d3332ad284d697411ec1fe2a3fd0222c3840c49ba955e215da6eb8d0c2faecb05300814bd076b8be9b5c36039fef29221cfb3bfb6ed68ebada",
    "e14f1162f7fd61d148b226b3fde19c019e9bfdfbc3421a1c0ff695d5a0d0ec0e0fcdb9a2ec25928ddf4b54ea7d07876ef72b81c182c20c552b734007a099d1d3",
    "ea4a0d28d6dd11a41cf3c7813f0fdac9201497d0556ece02085df983457da7f138e8ed72af3d0ce1fd7f6d2a06cea0db183b789467dec274f36457fa6ae18552",
    "70825f9a75911e96b52c5d5243aa833713b1d8dfa05ac63dda18259dd3b3e879173ba88ba6cb4a13fe845e061ad05e872b638de0885e199234bf251b3a37ffa7",
    "8d817a2d00ed3734225279cdd2204a62976235fdfc7066d0d41c1b261742ee4e903451162c080317b880f34840686adfb306467df89d1a61aa1df276f38b5e1b",
    "60bf5519a52ee56b8749156e7bd857e7209ab708eb4f62846f4a57a7b59461ddbac6808fd0c571e65243ca29361f142f401ade5ffa822c903cf6b888b9b770d4",
    "a45512c65cca3206e669a01133854a23231f4eba9b7bab2a1b223dcb6e6c3fc26e8d839cd209e2b0782ca62a1634a05d30c0ada6dc39aa1ad9e8a5ebd6557578",
    "0fd283a6e61bb6c526856abf07a6d7a71ef238c530ce42ca8b7da42adeeadb6caf064fcc19979a132c08b8bb72329dd2251dd79ef504583766436d5412ca1c63",
    "d7f324a9c852dd4cb9f1a9b064a2c29eb31d971530093bcef93863cdf7405292d1fbd1cca890933e8c96fc26ce2aecee1a4da07fb8cfa6a43d7621b0c4ed2b8d",
    "1a5709733ccd500c96331ebd0def85fd24965d8f015f22871a31eb7efb1b9d20d998e524bc193314e874c7f7304ca65303ceec831441750083894831d56018f9",
    "d40b53c906405d0878a53fba9cb93aca36556ff1780b2ed63d2f8a437d47b1bf257f22759c930c7b17effd6e2eb30ea26aa27e9088988720a5863f9924a521b1",
    "5496129a8a924f9896d062227d32369d2020bd7f1d2b32088994e68715c330501b81550329f67783787f476ed38f43cfe8cfcf2484d09526d604228e68b3d592",
    "cf465d368613da34ed4e544e4a20a5b65633050010f0400dc2313553da95c9d8d3ef529a9b1a7f1d54ca85146265e4772d793373b251b1097aa52bdbddca17ae",
    "76b2e1b8dcac7f4d0f4b45a8c19fd9b0422fd59d03392657cc2a856237854f7975fe2ce8c6a52bb46ebdbf9db2eedc2a5147ef62e97e8ca6b942cda967b6fb7d",
    "b4c1147985ca40cd77228e54c3a5b749a93c71d70cd7966c31e0770205f7b22c2ad29041c376522125dc219727f8de0e5750afc5ac315c7e003d295565d7f7e6",
    "78d3a58e255b9501fe60c28805fdc63065725c25b756dd43b20749d0290f5943bb7b8289ef58a6fc929bd77535dcf9f986d2f02f22667382bb5c8d8a355928b8",
    "52dcb9a7024c482fdca119f61e7cbeacf7680642fb8e724081c06977e94c58ca796923f31f135e556730294305c0f8203ee6bcc5aa2bf980fded21510346efdb",
    "f044948ca4bb0c05b87fd6debabb6c3e66939fab2570aa67cbf1f415ab65ae73a5b4c81dc6ecba761ed01f6b7feaa147dc6f5b3460b5d21619e5df3de4b35a2a",
    "e822b458ea2ef8eadec30c717c1104da2d0d3d0308db0f16788887772b901bb2a660e7455af9c32a866a11a056b0991f2eb9bb982d440dcbaa2d87a75ed56042",
    "81b16b16793fcacaa9c43084f376c0bbd69a9809bd779eee171491ec2a4ff2771ea86a75a947692d6f8be318370421c893cde41c2f41986e07e759c1ad6125b2",
    "59e528675b2f3c47d295523caf0e8d2e0875d8b62256ee246756021bfc3af43e8375fee47580c13182f99e73077a9ab9db4840ea2c4bb1ef5eadcf452d65f5a1",
    "8e9273a03c0bc59fae3775c335f2ea929e29eada32d04c97394dd33b483fc0147f7ab85d80467927d2db868316e6e2862da31885865365e896d3f11f61843f11",
    "d971af0641185d278d89858d05e2fe6f595bbf4e55a7cba3f32bb2780d006c5cb13b6c7db41ed6193c87bc43ee1014bd9ccf5e7b7d52e1e01e86d3fa1fa4f1ee",
    "65a291117b9ba4eb76d37f5181cd43b957c859f7f429b4669d3bc3bb8e9e551828a5c7df2bbf58bda3205e4e0c95f3e6f364808663b5184e1fbaed85a42eb253",
    "c3c61c5435de327e12d58ed82a665bc7b68ad502f5372de312d11ea53ac9bd634016ed3710b8e8d5665781a2329c2aa4c1bfce17c6f6cddaf34ce130db26998b",
    "e816bb089947296343813b001a220d1de29cf48a2b4b7281f34015417f21b499bbc1bbeb32b94e1a346532bbf0c74552bfe4e0740d469c0b325cbf8def579994",
    "23d6585864bde2ad686eee009cfe94cf7f17dcc1ccb3c3fd71e2f0d4dc676c02553be2ba959f99c89c8264622bd151497165b3927c30d6a42b86f3714d800542",
    "1ffa117ce00e8650b7c9f5d4067e1814911dd65cf88981e900fb24664667478d207f5559b58fb6b74109a6f1e19254ca3ce1f482d8ba45dc371dd5ea5dc31dc4",
    "ab241949279fe21c272ccc2778f468903be4d37ed91a58c979a84d12d39f97e4da0860b425d1cd32f914ccebea063f5154b2d4f98494dfe44441b251c2f8e138",
    "ef30bd7ed795848ac656073c3636d9b3caaf3a3e074f4f58923ad907bc60debab82b88f6da3f45df8963316b88c7a6c9018ca5ea5811d2ac6956b565ffdf15a4",
    "2a471545f1af70e036b58273ccbb8f2d0cc158fedfbec7913475ca4c0dada5d264db21d0a9be9e82a496f48f3376a991ad15eb749e95dd6508b9f942f006bc35",
    "f0e821ed855519d97bb90f88257beb2862bbe7547ac9ad5622bcd003bf751bfa3e45463ad5a69aab1a8ad8edc2ec55d3bfb439d7e0f2184b3840d0de48184ebd",
    "57a117c28f464368efa42b1fa61dadfeaa5ebfe70e5670b54f8777d174b7e0320f31f7de107837ff9324c916a69700dbb85ebf22841f5199257a035ed8203fee",
    "70fb4e51ec24e15e6433ae47b8a88a476f6d9f91a05bfbfeda514d442488ea5b5a1e8d0ae79923461d17b17dde53749cab842bff6f75269c3a63bbd45eb7d97f",
    "f3ae92dbae6847f929cad5e01b137db92d5dfdbe06f4f3ae8115aa277540a72dc19be1cd4755fd308724838389dfad5de879e5bd96f9e74a5f5eee6462907fb9",
    "88c265ce99b6b956235156dbf9d36f55f9bef42258991688a2d883ffdc10a3310924840b5c94a244567433930030ca0fcbc55f2ac9f97db38c26f7e87f4bf860",
    "87d14267ef991c2568422bf9f0cd956768a869a4278c2bc80b4062eff44dbef07efed33c17da86273961752441ffb2eec15e905e3d08acc29963110e1a224888",
    "33315171417818f811d909323d65f4acdfa253ab3dc64c70adfe1e8746510670f8a05b31449c4723d5ec3d9311dcd303dc63d2ff87ba7857cc77a84bbd56b247",
    "8855c36ebc60660e1096fb51f3ca1d15cd9cebcf50f85191bda6b9c6b4516414d35f77b7fa9232e0d3bb4416ef518dbe1cd82ab7e468ee5a129461f490e789da",
    "b20dcd5081e659445221a38d39c8decb791057e6d6d68a8a0144bb891c95996db69a1f39e32f5eb53b18340d8ea3b960737a5f3795e8aaea94ea34a08a4f02ed",
    "b67f86df457d6f6f9dd9fd8db45c6f9e9126dc77879261c9224228c5333d48fa404f91750bbaf41de81fcdc8dbb91fd57898c4847dbbed2eaea5f1d2e8e0d2b3",
    "195d39f4f5a48a3c0e3924e62aaa95aa0cd1365d1f5521db04c33309f06b80babb88ba215a5eea3b7d913362c1ae8d8ff8daa3c7d71354966753214d4682b761",
    "d296754f7436974fccef6ab8f858a3c839b6632a4a9ade0bf76e75f82cc9bdb74029ab01abef0d9b870f309256270a58d27aa39f3be4aac0b2c6d26c94cfd645",
    "4ccf643be24b26896a157a13af9940e1a737163cafb045e8efb577386707bea56d15bb880f143702d35261a3a75c90aaf777299eaf4fa9274ff5101df599a9db",
    "67c2b5cbaa36a7a0bcd6501993db0dd02dca9b9d3d348d81c26c92582449c7867c47a3421153de7add542a50b7c2920b9f57125d2c5fdd5f4af34ffdc1e113a5",
    "c088ba1af196c757ae41b963d76b5cf98440f868e1788722a8fc930949bea2180243bca910a63b771dfef4e64084bbfdbc74e1c8f9f7eb68932c168deb5e9e56",
    "2f80cbc9c3b5e0b25fdab74d0167729c75bd16986c9ff3ae08c801320c626fe0d8fec911cb0c063bd5a141a2e2f2bf2bde46a6a63e21fbfc293a84a3536a8b72",
    "b32d8761ad6bbe98aaa16a9e73082adff57c436c563fe08c0c0e8892c30ec302d420fc3691517e16966c81da4b6ceaba1528c34a13d36eba3a15cd858fde49ca",
}


@dataclass
class OpenDAPIEntityCICDMetadata:
    """Metadata for OpenDAPI entities"""

    entity: OpenDAPIEntity
    # NOTE: make sure this is formatted for the server!
    filepath: str
    # NOTE: decide if here post init we make sure that
    # head cannot be skipped generate if base wasnt?
    # I think that makes sense...
    base_collect: CollectedFile | None
    head_collect: CollectedFile | None
    integration_mode: str

    @functools.cached_property
    def entity_changed_from_base(self) -> bool:
        """Check if the model is changed from the model state at base."""
        # for a non-dapi file, any change to the **file** state from base
        # is considered a change
        if self.entity is not OpenDAPIEntity.DAPI:
            return (self.base_collect and self.base_collect.original) != (
                self.head_collect and self.head_collect.merged
            )

        # we note that the file can remain after the entity is deleted
        # in that case any file differences arent actual entity changes
        base_generated = self.base_collect.generated if self.base_collect else None
        head_generated = self.head_collect.generated if self.head_collect else None
        if not base_generated and not head_generated:
            return False

        # if it exists in one but not the other, then surely it was changed
        if bool(base_generated) != bool(head_generated):
            return True

        # entity exists at both, so we compare if there were any meaningful changes.
        # note that we compare merged - but prune it - so that nullability
        # changes are considered changes
        base_merged = self.base_collect.merged if self.base_collect else None
        head_merged = self.head_collect.merged if self.head_collect else None

        return has_underlying_model_changed(base_merged, head_merged)

    @property
    def entity_is_new(self) -> bool:
        """Check if the model is new"""
        # we do these gymnastics since the dapi file can remain after the entity
        # is deleted
        base_generated = self.base_collect.generated if self.base_collect else None
        return self.entity_changed_from_base and not base_generated

    @property
    def entity_is_deprecated(self) -> bool:
        """Check if the model is deprecated"""
        head_generated = self.head_collect.generated if self.head_collect else None
        return self.entity_changed_from_base and not head_generated

    # NOTE: entity cannot be changed from head, since we do not
    #       ever change anything at the ORM level

    @property
    def file_git_tracked(self) -> bool:
        """
        Check if the file is tracked by git
        determined if the file ever existed
        """
        return bool(
            (self.head_collect and self.head_collect.original is not None)
            or (self.base_collect and self.base_collect.original is not None)
        )

    @functools.cached_property
    def _flow_type_and_reason(self) -> Tuple[CICDFlowType, str]:
        """Check if the entity should be processed"""
        # in shadow mode, we only run CICD if the dapi was onboarded
        # or is being onboarded
        # note about possibly doing something else for enrich
        # during shadow for a diff experience
        in_shadow_mode = self.integration_mode == "shadow"
        if in_shadow_mode and not self.file_git_tracked:
            if self.entity_changed_from_base:
                # there is no need to observe non dapis
                if self.entity is OpenDAPIEntity.DAPI:
                    return (
                        CICDFlowType.UNGOVERNED_OBSERVE,
                        "SHADOW_MODE_AND_FILE_NOT_TRACKED_DAPI_CHANGED",
                    )

                return (
                    CICDFlowType.UNGOVERNED_DO_NOT_OBSERVE,
                    "SHADOW_MODE_ENTITY_CHANGED_AND_NOT_DAPI",
                )

            return (
                CICDFlowType.UNGOVERNED_DO_NOT_OBSERVE,
                "SHADOW_MODE_AND_FILE_NOT_TRACKED_ENTITY_UNCHANGED",
            )

        # base_collect surely exists here, meaning that either the entity and or the file were
        # removed at head_collect - so this is the deletion flow.
        if not self.head_collect:
            return CICDFlowType.GOVERNED_DELETION, "ENTITY_AND_FILE_NOT_AT_HEAD"

        # it doesn't matter what the base_collect status is, if there is a file
        # without an entity, we should delete it
        # (either if the entity was just deleted and need to clean up the dapi
        #  or if the file was just created manually without an entity)
        # NOTE: this means that generated state must be there for the file to be allowed to exist
        #       - including for non-dapis
        if not self.head_collect.generated:
            filepath_hash = hashlib.blake2b(self.filepath.encode()).hexdigest()
            if filepath_hash in IGNORE_DELETION_FILEPATH_HASHES:
                return (
                    CICDFlowType.GOVERNED_DO_NOT_ANALYZE,
                    "ENTITY_DOES_NOT_EXIST_FILE_EXISTS_SHOULD_NOT_BE_DELETED",
                )

            return (
                CICDFlowType.GOVERNED_DELETION,
                "ENTITY_DOES_NOT_EXIST_FILE_EXISTS_SHOULD_BE_DELETED",
            )

        # first we handle non DAPI cases
        # any file changes from be processed
        if self.entity is not OpenDAPIEntity.DAPI:
            # if the **file** didnt exist and now exists in any capacity,
            # we need to process.
            # note that we consider file changes, since we **always** generate
            # a single opendapi file if there arent any for non dapis
            if (
                not (self.base_collect and self.base_collect.original)
                and self.head_collect.merged
            ):
                return CICDFlowType.GOVERNED_ANALYZE, "NON_DAPI_FILE_CREATED"

            # if the file changed from base, we need to process
            if (
                self.base_collect and self.base_collect.original
            ) != self.head_collect.merged:
                return CICDFlowType.GOVERNED_ANALYZE, "NON_DAPI_FILE_CHANGED_FROM_BASE"

            # if the file changed from head, we need to process
            if self.head_collect.merged != self.head_collect.original:
                return CICDFlowType.GOVERNED_ANALYZE, "NON_DAPI_FILE_CHANGED_FROM_HEAD"

            return (
                CICDFlowType.GOVERNED_DO_NOT_ANALYZE_REQUIRES_PRE,
                "NON_DAPI_FILE_NO_CHANGE",
            )

        # NOTE: head_collect.generated exists at this point
        # we consider if the model itself still exists

        # there was a meaningful change to the model.
        # we need to process
        if self.entity_changed_from_base:
            return CICDFlowType.GOVERNED_ANALYZE, "MODEL_EXISTS_AND_ENTITY_CHANGED"

        # there was no meaningful change to the model, but
        # there is no dapi. we need to create one.
        if not self.head_collect.original:
            return CICDFlowType.GOVERNED_ANALYZE, "MODEL_EXISTS_AND_NO_FILE"

        # there was no meaningful change to the model, the dapi exists,
        # but the dapi materially differs from the model, meaning it is out of sync.
        # note that this only applies to actual models.
        # we need to process
        # note that we compare against merged, since we want to respect nullability changes
        if has_underlying_model_changed(
            self.head_collect.merged,
            self.head_collect.original,
        ):
            return CICDFlowType.GOVERNED_ANALYZE, "MODEL_EXISTS_AND_FILE_OUT_OF_SYNC"

        # no meaningful change to the model and the dapi is synced.
        # in this scenario, we just need to make sure that manual
        # changes are sane.
        # either the file was just added - which is the case above - or
        # the file was manually changed, in both cases we need to process
        # NOTE: we could check if dapi materially changed, but any change to the file
        #       we want to go through feature validation for possible organic onboarding,
        #       so we send it - if its a benign change its non blocking anyway
        if self.head_collect.original != (
            self.base_collect and self.base_collect.original
        ):
            return CICDFlowType.GOVERNED_ANALYZE, "MODEL_EXISTS_AND_FILE_CHANGE"

        return CICDFlowType.GOVERNED_DO_NOT_ANALYZE, "MODEL_EXISTS_AND_NO_CHANGE"

    @functools.cached_property
    def flow_type(self) -> CICDFlowType:
        """Check if the entity should be processed"""
        flow_type, _ = self._flow_type_and_reason
        return flow_type

    @property
    def flow_reason(self) -> str:
        """Get the reason for processing the entity."""
        _, reason = self._flow_type_and_reason
        return reason

    @functools.cached_property
    def orm_unsynced_at_base(self) -> bool:
        """
        Check if the ORM is unsynced at base.

        Can be due to the model not yet being onboarded,
        or if the metadata suggestions were not applied.
        """
        return (
            self.entity is OpenDAPIEntity.DAPI
            and bool(self.base_collect)
            and has_underlying_model_changed(
                self.base_collect.merged,
                self.base_collect.original,
            )
        )

    @functools.cached_property
    def orphaned_file_at_base(self) -> bool:
        """
        Check if the file is orphaned at base.
        """
        return bool(
            self.entity is OpenDAPIEntity.DAPI
            and self.base_collect
            and not self.base_collect.generated
        )

    @property
    def legacy_required_for_minimal_cicd(self) -> bool:
        """Check if the entity should be processed for CICD"""
        # we only process changed dapis
        if self.entity is OpenDAPIEntity.DAPI:
            return self.flow_type in (CICDFlowType.GOVERNED_ANALYZE,)

        # but all other entities are required for CICD -
        # this is more or less if HEAD MERGED exists but more clear
        return self.flow_type in (
            CICDFlowType.GOVERNED_ANALYZE,
            CICDFlowType.GOVERNED_DO_NOT_ANALYZE_REQUIRES_PRE,
        )

    @property
    def required_for_minimal_cicd(self) -> bool:
        """Check if the entity should be processed for CICD"""
        # we only process changed dapis
        if self.entity is OpenDAPIEntity.DAPI:
            return self.flow_type in (
                CICDFlowType.GOVERNED_DELETION,
                CICDFlowType.GOVERNED_ANALYZE,
                CICDFlowType.UNGOVERNED_OBSERVE,
            )

        # but all other entities are required for CICD -
        # this is more or less if HEAD MERGED exists but more clear
        return self.flow_type in (
            CICDFlowType.GOVERNED_ANALYZE,
            CICDFlowType.GOVERNED_DO_NOT_ANALYZE_REQUIRES_PRE,
        )

    @property
    def for_server(self) -> dict:
        """Convert to dict."""

        return {
            "entity": self.entity.value,
            "filepath": self.filepath,
            "base_collect": self.base_collect.for_server if self.base_collect else None,
            "head_collect": self.head_collect.for_server if self.head_collect else None,
            "entity_changed_from_base": self.entity_changed_from_base,
            "entity_is_new": self.entity_is_new,
            "entity_is_deprecated": self.entity_is_deprecated,
            "file_git_tracked": self.file_git_tracked,
            "integration_mode": self.integration_mode,
            "orm_unsynced_at_base": self.orm_unsynced_at_base,
            "required_for_minimal_cicd_v2": self.required_for_minimal_cicd,
            "flow_type": self.flow_type.value,
            "flow_reason": self.flow_reason,
            "orphaned_file_at_base": self.orphaned_file_at_base,
            # legacy
            "required_for_minimal_cicd": self.legacy_required_for_minimal_cicd,
            "should_be_processed": self.flow_type is CICDFlowType.GOVERNED_ANALYZE,
            "should_be_processed_reason": self.flow_reason,
        }


@dataclass
class MergeKeyCompositeIDParams:
    """
    Class to store required and not required portions of a UUID
    """

    class NoIDFoundError(Exception):  # pylint: disable=too-few-public-methods
        """
        Exception raised when no ID can be found
        """

    class IgnoreListIndexType:  # pylint: disable=too-few-public-methods
        """
        Helper class if you want to match path keys that occur at any index
        """

        def __eq__(self, other):
            return isinstance(other, int)

    IGNORE_LIST_INDEX: ClassVar[IgnoreListIndexType] = IgnoreListIndexType()

    class NotSetType:  # pylint: disable=too-few-public-methods
        """represents not set"""

    NOT_SET: ClassVar[NotSetType] = NotSetType()

    # paths to values in the dict
    # for which we must have explicit terminal values
    required: List[List[str]]

    # paths to values in the dict
    # for which we need not reach the terminal value
    optional: List[List[str]] = field(default_factory=list)

    def __post_init__(self):
        """
        additional validation
        """
        # ensure that a returned UUID is not the empty tuple or the item itself
        if not self.required or any(
            not path_to_key_el for path_to_key_el in self.required
        ):
            raise ValueError(
                "Not having required portions for a UUID leads to nonsensical merging"
            )

    def get_id_if_matched(self, itm: dict) -> Optional[
        Tuple[
            Tuple[
                Tuple[str, ...],
                Union[
                    str, int, float, bool, None, MergeKeyCompositeIDParams.NotSetType
                ],
            ],
            ...,
        ]
    ]:
        """
        Returns an ID if we can create one from the given key, meaning that
        all required ID portions are fetched.
        """

        # we will build up the ID
        id_ = []

        # iterate through all paths to ID elements, which may be nested
        # the assumption is that each intermediate portion until we reach the terminal
        # element will be a dict, and the last element is a primitive
        # if we cannot reach the terminal element for a required ID portion,
        # we return None, since that means that an ID cannot be constructed
        for path_to_key_element in self.required:
            cur = itm
            for path_el in path_to_key_element:
                if path_el not in cur:
                    return None
                cur = cur[path_el]
            # we make sure to add the path_to_key_element to the ID, so that we can
            # disambiguate between different keys that have the same terminal value
            id_.append((tuple(path_to_key_element), cur))

        # do similar thing to above, but in this case not reaching a terminal element is OK.
        # to disambiguate between None being a terminal value VS it not being present, in the event
        # we do not reach a terminal element we add NOT_SET to the ID
        for path_to_key_element in self.optional:
            cur = itm
            for path_el in path_to_key_element:
                if path_el not in cur:
                    cur = self.NOT_SET
                    break
                cur = cur[path_el]
            id_.append((tuple(path_to_key_element), cur))

        # make hashable
        return tuple(id_)

    @staticmethod
    def safe_get_key_and_id(
        itm: Dict, merge_keys: List[MergeKeyCompositeIDParams]
    ) -> Optional[
        Tuple[
            MergeKeyCompositeIDParams,
            Tuple[
                Tuple[str, ...],
                Tuple[
                    Union[
                        str,
                        int,
                        float,
                        bool,
                        None,
                        MergeKeyCompositeIDParams.NotSetType,
                    ],
                    ...,
                ],
            ],
        ],
    ]:
        """
        Given a list of merge keys, return the first key and id for that we can create
        from itm, returning None if no match
        """
        return next(
            ((key, id_) for key in merge_keys if (id_ := key.get_id_if_matched(itm))),
            None,
        )

    @classmethod
    def get_key_and_id(
        cls, itm: Dict, merge_keys: List[MergeKeyCompositeIDParams]
    ) -> Tuple[
        MergeKeyCompositeIDParams,
        Tuple[
            Tuple[str, ...],
            Tuple[
                Union[
                    str,
                    int,
                    float,
                    bool,
                    None,
                    MergeKeyCompositeIDParams.NotSetType,
                ],
                ...,
            ],
        ],
    ]:
        """
        Given a list of merge keys, return the first key and id for that we can create
        from itm, raising MergeKeyCompositeIDParams.NoIDFoundError if no match
        """
        key_and_id = cls.safe_get_key_and_id(itm, merge_keys)
        if not key_and_id:
            raise cls.NoIDFoundError(
                f"No ID found in {itm} for any of the keys {merge_keys}"
            )
        return key_and_id
