"""malware_detection backend base module to implement."""

from abc import ABC, abstractmethod

from django.utils.module_loading import import_string


class BaseBackend(ABC):
    """Base class for all malware detection backends."""

    def __init__(self, *, callback_path: str, max_processing_files: int = 15, **kwargs):
        """Initialize the backend."""
        self.callback_path = callback_path

        if max_processing_files < 0:
            raise ValueError("max_processing_files must be greater than or equal to 0")

        self.max_processing_files = max_processing_files

    @property
    def callback(self):
        """Get the callback function."""
        return import_string(self.callback_path)

    @abstractmethod
    def analyse_file(self, file_path: str, **kwargs) -> None:
        """Analyse a file and call the callback with the result."""

    @abstractmethod
    def launch_next_analysis(self) -> None:
        """Launch the next analysis."""
