"""malware detection backendhandler."""

from django.conf import settings
from django.core.exceptions import ImproperlyConfigured
from django.utils.functional import cached_property
from django.utils.module_loading import import_string

from lasuite.malware_detection.exceptions import MalwareDetectionInvalidBackendError


class MalwareDetectionHandler:
    """Malware detection handler managing the backend instantiation."""

    def __init__(self, backend=None):
        """Initialize the malware detection handler."""
        # backend is an optional dict of malware detection backend definitions
        # (structured like settings.MALWARE_DETECTION).
        self._backend = backend
        self._malware_detection = None

    @cached_property
    def backend(self):
        """Put in cache the backend properties from the settings."""
        if self._backend is None:
            try:
                self._backend = settings.MALWARE_DETECTION.copy()
            except AttributeError as e:
                raise ImproperlyConfigured("settings.MALWARE_DETECTION is not configured") from e
        return self._backend

    def __call__(self):
        """Create if not existing the backend and then return it."""
        if self._malware_detection is None:
            self._malware_detection = self.create_malware_detection(self.backend)
        return self._malware_detection

    def create_malware_detection(self, params):
        """Instantiate and configure the malware detection backend."""
        params = params.copy()
        backend = params.pop("BACKEND")
        parameters = params.pop("PARAMETERS", {})
        try:
            klass = import_string(backend)
        except ImportError as e:
            raise MalwareDetectionInvalidBackendError(f"Could not find backend {backend!r}: {e}") from e
        return klass(**parameters)
