"""malware detection models."""

import json
import uuid

from django.db import models


class MalwareDetectionStatus(models.TextChoices):
    """Malware detection status."""

    PENDING = "pending", "Pending"
    PROCESSING = "processing", "Processing"


class JsonUUIDEncoder(json.JSONEncoder):
    """JSON encoder for UUIDs."""

    def default(self, obj):
        """Encode UUIDs."""
        if isinstance(obj, uuid.UUID):
            return str(obj)
        return super().default(obj)


class MalwareDetection(models.Model):
    """Malware detection model."""

    id = models.UUIDField(
        help_text="primary key for the record as UUID",
        primary_key=True,
        default=uuid.uuid4,
        editable=False,
    )
    created_at = models.DateTimeField(
        help_text="date and time at which a record was created",
        auto_now_add=True,
        editable=False,
    )
    updated_at = models.DateTimeField(
        help_text="date and time at which a record was last updated",
        auto_now=True,
        editable=False,
    )
    path = models.CharField(
        help_text="path to the file",
        max_length=255,
        unique=True,
    )
    status = models.CharField(
        help_text="status of the detection",
        max_length=255,
        choices=MalwareDetectionStatus.choices,
        default=MalwareDetectionStatus.PENDING,
    )
    parameters = models.JSONField(
        help_text="parameters for the detection",
        default=dict,
        null=True,
        blank=True,
        encoder=JsonUUIDEncoder,
    )

    def __str__(self):
        """Return a string representation of the model."""
        return f"file {self.path} with status {self.status}"
