#
# SPDX-FileCopyrightText: Copyright (c) 2025 provide.io llc. All rights reserved.
# SPDX-License-Identifier: Apache-2.0
#

"""TODO: Add module docstring."""

from __future__ import annotations

from decimal import Decimal
from typing import Any

# pyvider-cty/src/pyvider/cty/conversion/raw_to_cty.py
import unicodedata

import attrs
from provide.foundation.errors import error_boundary

from pyvider.cty.conversion._utils import _attrs_to_dict_safe
from pyvider.cty.conversion.inference_cache import (
    get_container_schema_cache,
    get_structural_key_cache,
    with_inference_cache,
)
from pyvider.cty.types import CtyType
from pyvider.cty.values import CtyValue


def _extract_container_children(container: Any) -> list[Any]:
    """Extract child elements from a container for cache key generation."""
    children: list[Any] = []
    if isinstance(container, dict):
        children.extend(container.values())
    elif isinstance(container, list | tuple | set | frozenset):
        children.extend(container)
    return children


def _generate_container_cache_key(
    container: Any, structural_cache: dict[int, tuple[Any, ...]]
) -> tuple[Any, ...]:
    """Generate a cache key for a container based on its type and contents.

    Uses value-based keys for small containers with primitives to avoid
    race conditions from Python's object interning.
    """
    if isinstance(container, dict):
        # For small dicts containing only primitives, use value-based keys
        # to avoid race conditions from interned objects sharing IDs
        if len(container) <= 5 and all(
            isinstance(v, (bool, int, float, str, bytes, type(None))) for v in container.values()
        ):
            sorted_items = sorted(container.items(), key=lambda item: repr(item[0]))
            return (dict, frozenset((k, v) for k, v in sorted_items))

        # For larger or complex dicts, use existing structural cache approach
        sorted_items = sorted(container.items(), key=lambda item: repr(item[0]))
        return (
            dict,
            frozenset((k, structural_cache[id(v)]) for k, v in sorted_items),
        )
    elif isinstance(container, list):
        # For lists containing only primitives, use value-based keys to prevent race conditions
        # Increased threshold to handle test cases with larger datasets
        if len(container) <= 100 and all(
            isinstance(v, (bool, int, float, str, bytes, type(None))) for v in container
        ):
            return (list, tuple(container))
        return (list, tuple(structural_cache[id(v)] for v in container))
    elif isinstance(container, tuple):
        # For tuples containing only primitives, use value-based keys to prevent race conditions
        if len(container) <= 100 and all(
            isinstance(v, (bool, int, float, str, bytes, type(None))) for v in container
        ):
            return (tuple, container)
        return (tuple, tuple(structural_cache[id(v)] for v in container))
    elif isinstance(container, set | frozenset):
        # For sets containing only primitives, use value-based keys to prevent race conditions
        if len(container) <= 100 and all(
            isinstance(v, (bool, int, float, str, bytes, type(None))) for v in container
        ):
            sorted_items = sorted(list(container), key=repr)
            return (frozenset, frozenset(sorted_items))

        # Sort elements by their string representation for deterministic order.
        sorted_items = sorted(list(container), key=repr)
        return (frozenset, frozenset(structural_cache[id(v)] for v in sorted_items))
    else:
        return (type(container),)


def _process_container_children(
    current_item: Any,
    work_stack: list[Any],
    post_process_stack: list[Any],
    structural_cache: dict[int, tuple[Any, ...]],
    visited_ids: set[int],
) -> None:
    """Process a container item and add its children to the work stack."""
    item_id = id(current_item)

    if item_id in visited_ids:
        return

    visited_ids.add(item_id)

    # Placeholder is essential for cycle detection.
    structural_cache[item_id] = (type(current_item), item_id, "placeholder")
    post_process_stack.append(current_item)

    children = _extract_container_children(current_item)
    work_stack.extend(children)


def _get_structural_cache_key(value: Any) -> tuple[Any, ...]:
    """
    Iteratively generates a stable, structural cache key from a raw Python object,
    using a context-aware cache to handle object cycles and repeated sub-objects.
    Includes thread identity to ensure complete isolation between concurrent operations.
    """
    import threading

    structural_cache = get_structural_key_cache()
    if structural_cache is None:
        # Fallback for when no cache is available (thread safety mode)
        return (type(value), id(value))

    work_stack: list[Any] = [value]
    post_process_stack: list[Any] = []
    visited_ids: set[int] = set()

    # Process all items to build cache entries
    while work_stack:
        current_item = work_stack.pop()
        item_id = id(current_item)

        if item_id in structural_cache:
            continue

        if not isinstance(current_item, dict | list | tuple | set | frozenset):
            # For primitive values, use value-based cache keys to avoid race conditions
            # from shared object IDs (e.g., interned integers, strings)
            if isinstance(current_item, (bool, int, float, str, bytes, type(None))):
                structural_cache[item_id] = (type(current_item).__name__, current_item)
            else:
                structural_cache[item_id] = (type(current_item),)
            continue

        _process_container_children(
            current_item, work_stack, post_process_stack, structural_cache, visited_ids
        )

    # Build the final keys from the bottom up
    while post_process_stack:
        container = post_process_stack.pop()
        container_id = id(container)
        key = _generate_container_cache_key(container, structural_cache)
        structural_cache[container_id] = key

    # Include thread identity in the final cache key for complete isolation
    thread_id = threading.get_ident()
    base_key = structural_cache.get(id(value), (type(value),))
    return (thread_id, base_key)


@with_inference_cache
def infer_cty_type_from_raw(value: Any) -> CtyType[Any]:  # noqa: C901
    """
    Infers the most specific CtyType from a raw Python value.
    This function uses an iterative approach with a work stack to avoid recursion limits
    and leverages a context-aware cache for performance and thread-safety.
    """
    with error_boundary(
        context={
            "operation": "cty_type_inference",
            "value_type": type(value).__name__,
            "is_attrs_class": attrs.has(type(value)) if hasattr(value, "__class__") else False,
            "value_repr": str(value)[:100] if value is not None else "None",  # Truncated for safety
        }
    ):
        from pyvider.cty.types import (
            CtyBool,
            CtyDynamic,
            CtyList,
            CtyMap,
            CtyNumber,
            CtyObject,
            CtySet,
            CtyString,
            CtyTuple,
            CtyType,
        )

        if isinstance(value, CtyValue) or value is None:
            return CtyDynamic()

        if isinstance(value, CtyType):
            return CtyDynamic()

        if attrs.has(type(value)):
            value = _attrs_to_dict_safe(value)

    container_cache = get_container_schema_cache()

    # If no cache is available (e.g., in worker threads for thread safety),
    # proceed without caching
    structural_key = None
    if container_cache is not None:
        structural_key = _get_structural_cache_key(value)
        if structural_key in container_cache:
            return container_cache[structural_key]

    POST_PROCESS = object()
    work_stack: list[Any] = [value]
    results: dict[int, CtyType[Any]] = {}
    processing: set[int] = set()

    while work_stack:
        current_item = work_stack.pop()

        if current_item is POST_PROCESS:
            container = work_stack.pop()
            container_id = id(container)
            processing.remove(container_id)

            if isinstance(container, dict) and all(isinstance(k, str) for k in container):
                container = {unicodedata.normalize("NFC", k): v for k, v in container.items()}

            child_values = container.values() if isinstance(container, dict) else container
            child_types = [
                (v.type if isinstance(v, CtyValue) else results.get(id(v), CtyDynamic())) for v in child_values
            ]

            inferred_schema: CtyType[Any]
            if isinstance(container, dict):
                if not container:
                    inferred_schema = CtyObject({})
                elif not all(isinstance(k, str) for k in container):
                    unified = _unify_types(set(child_types))
                    inferred_schema = CtyMap(element_type=unified)
                else:
                    attr_types = dict(zip(container.keys(), child_types, strict=True))
                    inferred_schema = CtyObject(attribute_types=attr_types)
            elif isinstance(container, tuple):
                inferred_schema = CtyTuple(element_types=tuple(child_types))
            elif isinstance(container, list | set):
                unified = _unify_types(set(child_types))
                inferred_schema = (
                    CtyList(element_type=unified)
                    if isinstance(container, list)
                    else CtySet(element_type=unified)
                )
            else:
                inferred_schema = CtyDynamic()

            results[container_id] = inferred_schema
            continue

        if attrs.has(type(current_item)) and not isinstance(current_item, CtyType):
            try:
                current_item = _attrs_to_dict_safe(current_item)
            except TypeError:
                results[id(current_item)] = CtyDynamic()
                continue

        if current_item is None:
            continue
        item_id = id(current_item)
        if item_id in results or item_id in processing:
            continue
        if isinstance(current_item, CtyValue):
            results[item_id] = current_item.type
            continue

        if not isinstance(current_item, dict | list | tuple | set):
            if isinstance(current_item, bool):
                results[item_id] = CtyBool()
            elif isinstance(current_item, int | float | Decimal):
                results[item_id] = CtyNumber()
            elif isinstance(current_item, str | bytes):
                results[item_id] = CtyString()
            else:
                results[item_id] = CtyDynamic()
            continue

        structural_key = _get_structural_cache_key(current_item)
        if container_cache is not None and structural_key in container_cache:
            results[item_id] = container_cache[structural_key]
            continue

        processing.add(item_id)
        work_stack.extend([current_item, POST_PROCESS])
        work_stack.extend(
            reversed(list(current_item.values() if isinstance(current_item, dict) else current_item))
        )

    final_type = results.get(id(value), CtyDynamic())

    # Cache the result if caching is available
    if container_cache is not None:
        final_structural_key = _get_structural_cache_key(value)
        container_cache[final_structural_key] = final_type

    return final_type


def _unify_types(types: set[CtyType[Any]]) -> CtyType[Any]:
    """Unifies a set of CtyTypes into a single representative type."""
    from pyvider.cty.conversion.explicit import unify

    return unify(types)


# 🌊🪢🔚
