"""
Document visualizer for layout analysis and OCR results
"""

import logging
import os
from typing import Any

import cv2
import numpy as np
from PIL import Image, ImageDraw, ImageFont, features

from ..constants import PALETTE
from .base import BaseVisualizer


class DocumentVisualizer(BaseVisualizer):
    """Document layout and OCR visualization"""

    # Default color palette for different element types
    DEFAULT_PALETTE = [
        (255, 0, 0),  # Red
        (0, 255, 0),  # Green
        (0, 0, 255),  # Blue
        (255, 255, 0),  # Yellow
        (255, 0, 255),  # Magenta
        (0, 255, 255),  # Cyan
        (128, 0, 0),  # Dark Red
        (0, 128, 0),  # Dark Green
        (0, 0, 128),  # Dark Blue
        (128, 128, 0),  # Olive
        (128, 0, 128),  # Purple
        (0, 128, 128),  # Teal
        (192, 192, 192),  # Silver
        (128, 128, 128),  # Gray
        (255, 165, 0),  # Orange
    ]

    def __init__(self):
        super().__init__()
        self.logger = logging.getLogger(__name__)
        self.palette = PALETTE

    def _convert_pdf_page_to_image(
        self,
        pdf_path: str,
        page_index: int = 0,
        dpi: int = 200,
        target_size: tuple[int, int] | None = None,
    ) -> np.ndarray:
        """
        Convert specified PDF page to image using pypdfium2

        Args:
            pdf_path: PDF file path
            page_index: Page index (0-based)
            dpi: Image resolution (default 200)
            target_size: Target image size (width, height) to match original parsing

        Returns:
            Converted image as numpy array in BGR format

        Raises:
            ImportError: If pypdfium2 is not installed
            ValueError: If PDF file cannot be opened or page index is invalid
        """
        # Import pypdfium2 at the beginning to ensure it's available
        import pypdfium2

        try:
            # Open PDF document
            doc = pypdfium2.PdfDocument(pdf_path)

            # Check if page index is valid
            if page_index < 0 or page_index >= len(doc):
                doc.close()
                raise ValueError(
                    f"Page index {page_index} is out of range. PDF has {len(doc)} pages.",
                )

            if target_size is not None:
                # Use target size to match original parsing dimensions
                target_width, target_height = target_size

                # Validate target size (check for None and invalid values)
                if (
                    target_width is None
                    or target_height is None
                    or target_width <= 0
                    or target_height <= 0
                    or target_width > 20000
                    or target_height > 20000
                ):
                    self.logger.warning(
                        "Invalid target size: %s, falling back to DPI-based conversion",
                        target_size,
                    )
                    target_size = None
                else:
                    # Get specified page
                    page = doc[page_index]

                    # Get original page dimensions
                    original_width = page.get_width()
                    original_height = page.get_height()

                    # Validate original dimensions (check for None and invalid values)
                    if (
                        original_width is None
                        or original_height is None
                        or original_width <= 0
                        or original_height <= 0
                    ):
                        self.logger.warning(
                            "Invalid original page dimensions: %s x %s",
                            original_width,
                            original_height,
                        )
                        target_size = None
                    else:
                        # Calculate scale to match target size exactly
                        # This ensures coordinate alignment between original parsing and visualization
                        scale_x = target_width / original_width
                        scale_y = target_height / original_height

                        # Use uniform scaling to maintain aspect ratio
                        scale = min(scale_x, scale_y)

                        # Validate scale factor
                        if scale <= 0 or scale > 10:  # Reasonable scale range
                            self.logger.warning(
                                "Invalid scale factor: %s, falling back to DPI-based conversion",
                                scale,
                            )
                            target_size = None
                        else:
                            try:
                                # Render page with calculated scale
                                bitmap = page.render(scale=scale)
                                pil_image = bitmap.to_pil()

                                # Calculate the actual size after scaling
                                actual_width = int(original_width * scale)
                                actual_height = int(original_height * scale)

                                # For coordinate alignment, we need to ensure the final image size matches target_size exactly
                                # This is critical for proper bounding box alignment
                                if (
                                    actual_width != target_width
                                    or actual_height != target_height
                                ):
                                    # Calculate scaling factors for final adjustment
                                    final_scale_x = target_width / actual_width
                                    final_scale_y = target_height / actual_height

                                    # Use uniform scaling to maintain aspect ratio
                                    final_scale = min(final_scale_x, final_scale_y)

                                    # Calculate new size
                                    new_width = int(actual_width * final_scale)
                                    new_height = int(actual_height * final_scale)

                                    # Resize the image
                                    pil_image = pil_image.resize(
                                        (new_width, new_height),
                                        Image.Resampling.LANCZOS,
                                    )

                                    # If we still don't match exactly, crop or pad as needed
                                    if (
                                        new_width != target_width
                                        or new_height != target_height
                                    ):
                                        # Create a new image with target size
                                        final_image = Image.new(
                                            "RGB",
                                            (target_width, target_height),
                                            (255, 255, 255),
                                        )

                                        # Calculate position to center the image
                                        x_offset = (target_width - new_width) // 2
                                        y_offset = (target_height - new_height) // 2

                                        # Paste the resized image onto the final image
                                        final_image.paste(
                                            pil_image,
                                            (x_offset, y_offset),
                                        )
                                        pil_image = final_image

                                # Log the coordinate mapping for debugging
                                self.logger.info(
                                    "PDF coordinate mapping: original(%dx%d) -> target(%dx%d)",
                                    original_width,
                                    original_height,
                                    target_width,
                                    target_height,
                                )
                                self.logger.info(
                                    "Scale factor: %.4f, Final scale: %.4f",
                                    scale,
                                    final_scale if "final_scale" in locals() else 1.0,
                                )
                                self.logger.info("Final image size: %s", pil_image.size)

                            except Exception as e:
                                self.logger.warning(
                                    "Error in target size conversion: %s, falling back to DPI-based conversion",
                                    e,
                                )
                                target_size = None

            # If target_size is None (either not provided or failed), use DPI-based conversion
            if target_size is None:
                renderer = doc.render(
                    pypdfium2.PdfBitmap.to_pil,
                    scale=dpi / 72,
                )
                images = list(renderer)

                # Get the specific page image
                if page_index >= len(images):
                    doc.close()
                    raise ValueError(
                        f"Page index {page_index} is out of range. PDF has {len(images)} pages.",
                    )

                pil_image = images[page_index]

            img_array = np.array(pil_image.convert("RGB"))
            img_array = img_array[:, :, ::-1]  # RGB to BGR

            doc.close()
            return img_array

        except Exception as e:
            if "doc" in locals():
                doc.close()
            raise ValueError(f"Failed to convert PDF page to image: {e}") from e

    def _is_pdf_file(self, file_path: str) -> bool:
        """
        Check if file is a PDF file

        Args:
            file_path: File path

        Returns:
            True if PDF file, False otherwise
        """
        if not os.path.exists(file_path):
            return False

        # Check file extension
        _, ext = os.path.splitext(file_path.lower())
        if ext == ".pdf":
            return True

        # Check file header (more reliable method)
        try:
            with open(file_path, "rb") as f:
                header = f.read(4)
                return header == b"%PDF"
        except Exception:
            return False

    def _get_original_image_size(self, results: Any) -> tuple[int, int] | None:
        """
        Extract original image size from parsing results with improved accuracy and stability

        Args:
            results: Document analysis results

        Returns:
            Tuple of (width, height) if found, None otherwise
        """
        try:
            all_coords = []

            # Collect all coordinates from different element types with validation
            # Words (most precise for OCR results)
            if hasattr(results, "words") and results.words:
                for word in results.words:
                    if hasattr(word, "points") and word.points:
                        for point in word.points:
                            if (
                                len(point) >= 2
                                and isinstance(point[0], (int, float))
                                and isinstance(point[1], (int, float))
                            ):
                                all_coords.append((float(point[0]), float(point[1])))
                    elif hasattr(word, "box") and word.box and len(word.box) >= 4:
                        all_coords.extend(
                            [
                                (float(word.box[0]), float(word.box[1])),  # x1, y1
                                (float(word.box[2]), float(word.box[3])),  # x2, y2
                            ],
                        )

            # Paragraphs
            if hasattr(results, "paragraphs") and results.paragraphs:
                for para in results.paragraphs:
                    if hasattr(para, "box") and para.box and len(para.box) >= 4:
                        all_coords.extend(
                            [
                                (float(para.box[0]), float(para.box[1])),  # x1, y1
                                (float(para.box[2]), float(para.box[3])),  # x2, y2
                            ],
                        )

            # Tables
            if hasattr(results, "tables") and results.tables:
                for table in results.tables:
                    if hasattr(table, "box") and table.box and len(table.box) >= 4:
                        all_coords.extend(
                            [
                                (float(table.box[0]), float(table.box[1])),  # x1, y1
                                (float(table.box[2]), float(table.box[3])),  # x2, y2
                            ],
                        )

            # Figures
            if hasattr(results, "figures") and results.figures:
                for figure in results.figures:
                    if hasattr(figure, "box") and figure.box and len(figure.box) >= 4:
                        all_coords.extend(
                            [
                                (float(figure.box[0]), float(figure.box[1])),  # x1, y1
                                (float(figure.box[2]), float(figure.box[3])),  # x2, y2
                            ],
                        )

            if all_coords:
                # Filter out invalid coordinates
                valid_coords = [
                    (x, y)
                    for x, y in all_coords
                    if x >= 0 and y >= 0 and x < 10000 and y < 10000
                ]

                if not valid_coords:
                    self.logger.warning("No valid coordinates found")
                    return None

                # Find the actual image boundaries
                min_x = min(coord[0] for coord in valid_coords)
                min_y = min(coord[1] for coord in valid_coords)
                max_x = max(coord[0] for coord in valid_coords)
                max_y = max(coord[1] for coord in valid_coords)

                # For PDF documents, coordinates are typically absolute pixel coordinates
                # The max coordinates should represent the full image dimensions
                # Add a small margin to ensure we capture the full image
                margin = max(
                    10,
                    (max_x - min_x + max_y - min_y) * 0.01,
                )  # Dynamic margin
                final_width = int(max_x + margin)
                final_height = int(max_y + margin)

                # Validate the detected size
                if (
                    final_width > 0
                    and final_height > 0
                    and final_width < 20000
                    and final_height < 20000
                ):
                    self.logger.info(
                        "Detected original image size: %dx%d from coordinates",
                        final_width,
                        final_height,
                    )
                    self.logger.info(
                        "Coordinate range: x=[%.1f, %.1f], y=[%.1f, %.1f]",
                        min_x,
                        max_x,
                        min_y,
                        max_y,
                    )
                    self.logger.info(
                        "Valid coordinates: %d/%d",
                        len(valid_coords),
                        len(all_coords),
                    )
                    return (final_width, final_height)
                else:
                    self.logger.warning(
                        "Detected size seems invalid: %s x %s",
                        final_width,
                        final_height,
                    )

            return None

        except Exception as e:
            self.logger.warning(
                "Could not extract original image size from results: %s",
                e,
            )
            return None

    def visualize(
        self,
        img,
        results,
        mode="ocr",
    ) -> np.ndarray:
        """
        Main visualization method

        Args:
            img: Input image
            data: Document analysis results or image with results
            mode: Visualization mode - "reading_order", "layout_detail", "layout_rough", "ocr", "confidence"
        Returns:
            Visualized image as numpy array
        """
        if mode == "layout":
            out = self.visualize_layout_detail(img, results)
            return self.visualize_reading_order(out, results)
        elif mode == "ocr":
            return self.visualize_ocr(img, results)
        elif mode == "confidence":
            return self.visualize_confidence_scores(img, results)

    def visualize_reading_order(
        self,
        img: np.ndarray,
        results: Any,
        line_color: tuple[int, int, int] = (0, 0, 255),
        tip_size: int = 10,
        visualize_figure_letter: bool = False,
    ) -> np.ndarray:
        """
        Visualize reading order of document elements

        Args:
            img: Input image
            results: Document analysis results
            line_color: Color for reading order arrows
            tip_size: Size of arrow tips
            visualize_figure_letter: Whether to visualize figure lettering

        Returns:
            Image with reading order visualization
        """
        try:
            return self.reading_order_visualizer(
                img,
                results,
                line_color,
                tip_size,
                visualize_figure_letter,
            )
        except Exception as e:
            self.logger.error(
                "Error in reading order visualization: %s",
                e,
            )
            return img

    def visualize_layout_detail(self, img: np.ndarray, results: Any) -> np.ndarray:
        """
        Detailed layout visualization

        Args:
            img: Input image
            results: Document analysis results

        Returns:
            Image with detailed layout visualization
        """
        try:
            return self.layout_visualizer_detail(results, img)
        except Exception as e:
            self.logger.error(
                "Error in layout detail visualization: %s",
                e,
            )
            return img

    def visualize_ocr(
        self,
        img: np.ndarray,
        results: Any,
        font_path: str = None,
        det_score: np.ndarray = None,
        vis_heatmap: bool = False,
        font_size: int = 12,
        font_color: tuple[int, int, int] = (255, 0, 0),
        line_color: tuple[int, int, int] = (0, 255, 0),
    ) -> np.ndarray:
        """
        OCR visualization

        Args:
            img: Input image
            results: Document analysis results containing words
            font_path: Path to font file
            det_score: Detection score heatmap
            vis_heatmap: Whether to visualize heatmap
            font_size: Font size
            font_color: Font color
            line_color: Line color for bounding boxes

        Returns:
            Image with OCR visualization
        """
        try:
            # Use default font path if not provided
            if not font_path:
                try:
                    from ..font_manager import get_font_path

                    font_path = get_font_path()
                    self.logger.info("Using default font: %s", font_path)
                except Exception as e:
                    self.logger.warning(
                        "No font path provided and default font not available: %s",
                        e,
                    )
                    return img

            # Extract words from results
            words = []
            if hasattr(results, "words") and results.words:
                words = results.words
            elif hasattr(results, "texts") and results.texts:
                # If results has texts instead of words, convert them
                words = results.texts
            else:
                self.logger.warning("No words found in results for OCR visualization")
                return img

            return self.ocr_visualizer(
                words,
                img,
                font_path,
                det_score,
                vis_heatmap,
                font_size,
                font_color,
                line_color,
            )
        except Exception as e:
            self.logger.error(
                "Error in OCR visualization: %s",
                e,
            )
            return img

    def _draw_reading_order_arrows(
        self,
        img: np.ndarray,
        elements: list[Any],
        line_color: tuple[int, int, int],
        tip_size: int,
    ) -> np.ndarray:
        """Draw reading order arrows between elements"""
        out = img.copy()

        for i, element in enumerate(elements):
            if i == 0:
                continue

            prev_element = elements[i - 1]
            cur_x1, cur_y1, cur_x2, cur_y2 = element.box
            prev_x1, prev_y1, prev_x2, prev_y2 = prev_element.box

            cur_center = (
                cur_x1 + (cur_x2 - cur_x1) / 2,
                cur_y1 + (cur_y2 - cur_y1) / 2,
            )
            prev_center = (
                prev_x1 + (prev_x2 - prev_x1) / 2,
                prev_y1 + (prev_y2 - prev_y1) / 2,
            )

            arrow_length = np.linalg.norm(np.array(cur_center) - np.array(prev_center))
            tip_length = tip_size / arrow_length if arrow_length > 0 else 0

            cv2.arrowedLine(
                out,
                (int(prev_center[0]), int(prev_center[1])),
                (int(cur_center[0]), int(cur_center[1])),
                line_color,
                2,
                tipLength=tip_length,
            )

        return out

    def _visualize_element(
        self,
        img: np.ndarray,
        category: str,
        elements: list[Any],
    ) -> np.ndarray:
        """Visualize specific element category  visualize_element"""
        out = img.copy()
        categories = [
            "paragraphs",
            "tables",
            "figures",
            "section_headings",
            "page_header",
            "page_footer",
            "picture",
            "logo",
            "code",
            "seal",
            "list_item",
            "caption",
            "inline_formula",
            "display_formula",
            "index",
        ]

        for element in elements:
            try:
                # Extract box - handle both object and dict formats
                if hasattr(element, "box"):
                    box = element.box
                elif isinstance(element, dict) and "box" in element:
                    box = element["box"]
                else:
                    self.logger.warning(
                        "Element has no box attribute: %s",
                        type(element),
                    )
                    continue

                # Extract role - handle both object and dict formats
                role = None
                if category != "tables":
                    if hasattr(element, "role"):
                        role = element.role
                    elif isinstance(element, dict) and "role" in element:
                        role = element["role"]

                # Color selection logic  exactly
                color_index = categories.index(category)
                if role is None:
                    role = ""
                else:
                    color_index = categories.index(role)
                    role = f"({role})"

                color = self.palette[color_index % len(self.palette)]
                x1, y1, x2, y2 = tuple(map(int, box))
                out = cv2.rectangle(out, (x1, y1), (x2, y2), color, 2)
                out = cv2.putText(
                    out,
                    category + role,
                    (x1, y1),
                    cv2.FONT_HERSHEY_SIMPLEX,
                    0.5,
                    (0, 0, 0),
                    2,
                )

                # Handle captions for tables and figures
                if category in ["tables", "figures"]:
                    caption = None
                    if hasattr(element, "caption"):
                        caption = element.caption
                    elif isinstance(element, dict) and "caption" in element:
                        caption = element["caption"]

                    if caption is not None:
                        caption_box = None
                        if hasattr(caption, "box"):
                            caption_box = caption.box
                        elif isinstance(caption, dict) and "box" in caption:
                            caption_box = caption["box"]

                        if caption_box is not None:
                            color_index = categories.index("caption")
                            color = self.palette[color_index % len(self.palette)]
                            x1, y1, x2, y2 = tuple(map(int, caption_box))
                            out = cv2.rectangle(out, (x1, y1), (x2, y2), color, 2)
                            out = cv2.putText(
                                out,
                                "caption",
                                (x1, y1),
                                cv2.FONT_HERSHEY_SIMPLEX,
                                0.5,
                                (0, 0, 0),
                                2,
                            )

                    # Handle figure paragraphs
                    if category == "figures":
                        paragraphs = None
                        if hasattr(element, "paragraphs"):
                            paragraphs = element.paragraphs
                        elif isinstance(element, dict) and "paragraphs" in element:
                            paragraphs = element["paragraphs"]

                        if paragraphs is not None:
                            for paragraph in paragraphs:
                                try:
                                    para_box = None
                                    if hasattr(paragraph, "box"):
                                        para_box = paragraph.box
                                    elif (
                                        isinstance(paragraph, dict)
                                        and "box" in paragraph
                                    ):
                                        para_box = paragraph["box"]

                                    if para_box is not None:
                                        color_index = categories.index("paragraphs")
                                        color = self.palette[
                                            color_index % len(self.palette)
                                        ]
                                        x1, y1, x2, y2 = tuple(map(int, para_box))
                                        out = cv2.rectangle(
                                            out,
                                            (x1, y1),
                                            (x2, y2),
                                            color,
                                            2,
                                        )
                                        out = cv2.putText(
                                            out,
                                            "paragraphs",
                                            (x1, y1),
                                            cv2.FONT_HERSHEY_SIMPLEX,
                                            0.5,
                                            (0, 0, 0),
                                            2,
                                        )
                                except Exception as e:
                                    self.logger.warning(
                                        "Error processing figure paragraph: %s",
                                        e,
                                    )
                                    continue
            except Exception as e:
                self.logger.warning(
                    "Error processing element in %s: %s",
                    category,
                    e,
                )
                continue

        return out

    def _visualize_table(self, img: np.ndarray, table: Any) -> np.ndarray:
        """Visualize table structure  table_visualizer exactly"""
        out = img.copy()
        cells = table.cells
        for cell in cells:
            box = cell.box
            row = cell.row
            col = cell.col
            row_span = cell.row_span
            col_span = cell.col_span

            text = f"[{row}, {col}] ({row_span}x{col_span})"

            x1, y1, x2, y2 = map(int, box)
            out = cv2.rectangle(out, (x1, y1), (x2, y2), (255, 0, 255), 2)
            out = cv2.putText(
                out,
                text,
                (x1, y1),
                cv2.FONT_HERSHEY_SIMPLEX,
                0.5,
                (255, 0, 0),
                2,
            )

        return out

    def visualize_captions(
        self,
        img: np.ndarray,
        results: Any,
        show_text: bool = True,
        font_size: float = 0.5,
        box_color: tuple[int, int, int] = (255, 128, 0),
        text_color: tuple[int, int, int] = (255, 128, 0),
    ) -> np.ndarray:
        """
        Visualize captions for tables and figures

        Args:
            img: Input image
            results: Document analysis results
            show_text: Whether to show caption text
            font_size: Font size for caption text
            box_color: Color for caption boxes
            text_color: Color for caption text

        Returns:
            Image with caption visualization
        """
        try:
            out = img.copy()

            # Process table captions
            if hasattr(results, "tables"):
                for table in results.tables:
                    if hasattr(table, "caption") and table.caption:
                        out = self._draw_caption(
                            out,
                            table.caption,
                            show_text,
                            font_size,
                            box_color,
                            text_color,
                        )

            # Process figure captions
            if hasattr(results, "figures"):
                for figure in results.figures:
                    if hasattr(figure, "caption") and figure.caption:
                        out = self._draw_caption(
                            out,
                            figure.caption,
                            show_text,
                            font_size,
                            box_color,
                            text_color,
                        )

            return out
        except Exception as e:
            self.logger.error(
                "Error in caption visualization: %s",
                e,
            )
            return img

    def _draw_caption(
        self,
        img: np.ndarray,
        caption: Any,
        show_text: bool,
        font_size: float,
        box_color: tuple[int, int, int],
        text_color: tuple[int, int, int],
    ) -> np.ndarray:
        """Draw a single caption"""
        out = img.copy()

        # Extract caption box
        caption_box = None
        if hasattr(caption, "box"):
            caption_box = caption.box
        elif isinstance(caption, dict) and "box" in caption:
            caption_box = caption["box"]

        # Extract caption text
        caption_text = None
        if hasattr(caption, "contents"):
            caption_text = caption.contents
        elif isinstance(caption, dict) and "contents" in caption:
            caption_text = caption.get("contents", "")
        elif isinstance(caption, str):
            caption_text = caption

        if caption_box is not None:
            x1, y1, x2, y2 = tuple(map(int, caption_box))

            # Draw caption box
            cv2.rectangle(out, (x1, y1), (x2, y2), box_color, 2)

            # Draw caption text if requested
            if show_text and caption_text:
                # Split long text into multiple lines
                max_width = x2 - x1
                words = caption_text.split()
                lines = []
                current_line = []

                for word in words:
                    test_line = " ".join(current_line + [word])
                    text_size = cv2.getTextSize(
                        test_line,
                        cv2.FONT_HERSHEY_SIMPLEX,
                        font_size,
                        1,
                    )[0]

                    if text_size[0] <= max_width or not current_line:
                        current_line.append(word)
                    else:
                        lines.append(" ".join(current_line))
                        current_line = [word]

                if current_line:
                    lines.append(" ".join(current_line))

                # Draw each line
                line_height = int(
                    cv2.getTextSize("Test", cv2.FONT_HERSHEY_SIMPLEX, font_size, 1)[0][
                        1
                    ]
                    * 1.5,
                )
                for i, line in enumerate(lines[:3]):  # Limit to 3 lines
                    y_pos = y1 + (i + 1) * line_height
                    if y_pos < y2:
                        cv2.putText(
                            out,
                            line,
                            (x1 + 5, y_pos),
                            cv2.FONT_HERSHEY_SIMPLEX,
                            font_size,
                            text_color,
                            1,
                        )

        return out

    def visualize_confidence_scores(
        self,
        img_or_path: np.ndarray | str,
        results: Any,
        show_ocr_confidence: bool = True,
        show_detection_confidence: bool = False,
    ) -> np.ndarray:
        """
        Visualize confidence scores for different elements

        Args:
            img_or_path: Input image (numpy array) or path to image file
            results: Document analysis results
            show_ocr_confidence: Whether to show OCR confidence scores
            show_detection_confidence: Whether to show detection confidence scores

        Returns:
            Image with confidence scores visualization
        """
        try:
            # Handle both image array and image path
            if isinstance(img_or_path, str):
                img = cv2.imread(img_or_path)
                if img is None:
                    raise ValueError(f"Could not load image from path: {img_or_path}")
            else:
                img = img_or_path

            out = img.copy()

            if hasattr(results, "words"):
                for word in results.words:
                    if not hasattr(word, "points"):
                        continue

                    points = word.points
                    # Convert points to bounding box
                    x_coords = [p[0] for p in points]
                    y_coords = [p[1] for p in points]
                    x1, y1 = int(min(x_coords)), int(min(y_coords))
                    x2, y2 = int(max(x_coords)), int(max(y_coords))

                    # Show OCR confidence (rec_score) - Recognition confidence
                    if show_ocr_confidence:
                        rec_confidence = None
                        if hasattr(word, "rec_score"):
                            rec_confidence = word.rec_score
                        elif hasattr(word, "confidence"):
                            rec_confidence = word.confidence

                        if rec_confidence is not None:
                            # Color based on OCR confidence
                            if rec_confidence > 0.8:
                                # Green for high confidence
                                color = (0, 255, 0)
                            elif rec_confidence > 0.6:
                                # Yellow for medium confidence
                                color = (0, 255, 255)
                            else:
                                color = (0, 0, 255)  # Red for low confidence

                            cv2.rectangle(out, (x1, y1), (x2, y2), color, 2)
                            cv2.putText(
                                out,
                                f"OCR:{rec_confidence:.2f}",
                                (x1, y1 - 5),
                                cv2.FONT_HERSHEY_SIMPLEX,
                                0.3,
                                color,
                                1,
                            )

                    # Show detection confidence (det_score) - Detection confidence
                    if show_detection_confidence:
                        det_confidence = None
                        if hasattr(word, "det_score"):
                            det_confidence = word.det_score

                        if det_confidence is not None:
                            # Color based on detection confidence (different color scheme)
                            if det_confidence > 0.8:
                                color = (
                                    255,
                                    0,
                                    255,
                                )  # Magenta for high detection confidence
                            elif det_confidence > 0.6:
                                color = (
                                    255,
                                    165,
                                    0,
                                )  # Orange for medium detection confidence
                            else:
                                color = (
                                    0,
                                    0,
                                    128,
                                )  # Dark blue for low detection confidence

                            # Draw detection confidence with different line style
                            cv2.rectangle(
                                out,
                                (x1 + 2, y1 + 2),
                                (x2 - 2, y2 - 2),
                                color,
                                1,
                            )
                            cv2.putText(
                                out,
                                f"DET:{det_confidence:.2f}",
                                (x1, y2 + 15),
                                cv2.FONT_HERSHEY_SIMPLEX,
                                0.3,
                                color,
                                1,
                            )

            return out
        except Exception as e:
            self.logger.error(
                "Error in confidence scores visualization: %s",
                e,
            )
            return img

    # =============================================================================
    # Visualization methods integrated as class methods
    # =============================================================================

    def _reading_order_visualizer(self, img, elements, line_color, tip_size):
        """Internal function for drawing reading order arrows"""
        out = img.copy()
        for i, element in enumerate(elements):
            if i == 0:
                continue

            prev_element = elements[i - 1]
            cur_x1, cur_y1, cur_x2, cur_y2 = element.box
            prev_x1, prev_y1, prev_x2, prev_y2 = prev_element.box

            cur_center = (
                cur_x1 + (cur_x2 - cur_x1) / 2,
                cur_y1 + (cur_y2 - cur_y1) / 2,
            )
            prev_center = (
                prev_x1 + (prev_x2 - prev_x1) / 2,
                prev_y1 + (prev_y2 - prev_y1) / 2,
            )

            arrow_length = np.linalg.norm(np.array(cur_center) - np.array(prev_center))

            # tipLength を計算（矢印長さに対する固定サイズの割合）
            tip_length = tip_size / arrow_length if arrow_length > 0 else 0

            cv2.arrowedLine(
                out,
                (int(prev_center[0]), int(prev_center[1])),
                (int(cur_center[0]), int(cur_center[1])),
                line_color,
                2,
                tipLength=tip_length,
            )
        return out

    def reading_order_visualizer(
        self,
        img,
        results,
        line_color=(0, 0, 255),
        tip_size=10,
        visualize_figure_letter=False,
    ):
        """Visualize reading order of document elements"""
        elements = results.paragraphs + results.tables + results.figures
        elements = sorted(elements, key=lambda x: x.order)

        out = self._reading_order_visualizer(img, elements, line_color, tip_size)

        if visualize_figure_letter:
            for figure in results.figures:
                out = self._reading_order_visualizer(
                    out,
                    figure.paragraphs,
                    line_color=(0, 255, 0),
                    tip_size=5,
                )

        return out

    def det_visualizer(
        self,
        img,
        quads,
        preds=None,
        vis_heatmap=False,
        line_color=(0, 255, 0),
    ):
        """Detection visualizer"""
        out = img.copy()
        h, w = out.shape[:2]
        if vis_heatmap:
            preds = preds["binary"][0]
            binary = preds.detach().cpu().numpy()
            binary = binary.squeeze(0)
            binary = (binary * 255).astype(np.uint8)
            binary = cv2.resize(binary, (w, h), interpolation=cv2.INTER_LINEAR)
            heatmap = cv2.applyColorMap(binary, cv2.COLORMAP_JET)
            out = cv2.addWeighted(out, 0.5, heatmap, 0.5, 0)

        for quad in quads:
            quad = np.array(quad).astype(np.int32)
            out = cv2.polylines(out, [quad], True, line_color, 1)
        return out

    def ocr_visualizer(
        self,
        words,
        img,
        font_path,
        det_score=None,
        vis_heatmap=False,
        font_size=12,
        font_color=(255, 0, 0),
        line_color=(0, 255, 0),
    ):
        """OCR visualizer"""
        out = img.copy()
        if vis_heatmap and det_score is not None:
            w, h = img.shape[1], img.shape[0]
            det_score = (det_score * 255).astype(np.uint8)
            det_score = cv2.resize(det_score, (w, h), interpolation=cv2.INTER_LINEAR)
            heatmap = cv2.applyColorMap(det_score, cv2.COLORMAP_JET)
            out = cv2.addWeighted(out, 0.5, heatmap, 0.5, 0)

        pillow_img = Image.fromarray(out)
        draw = ImageDraw.Draw(pillow_img)
        font = ImageFont.truetype(font_path, font_size)

        has_raqm = features.check_feature(feature="raqm")
        if not has_raqm:
            self.logger.warning(
                "libraqm is not installed. Vertical text rendering is not supported. Rendering horizontally instead.",
            )

        for word in words:
            poly = word.points
            text = word.content
            direction = word.direction

            poly_line = [tuple(point) for point in poly]
            draw.polygon(poly_line, outline=line_color, fill=None)

            if direction == "horizontal" or not has_raqm:
                x_offset = 0
                y_offset = -font_size

                pos_x = poly[0][0] + x_offset
                pox_y = poly[0][1] + y_offset
                draw.text(
                    (pos_x, pox_y),
                    text,
                    font=font,
                    fill=font_color,
                )
            else:
                x_offset = -font_size
                y_offset = 0

                pos_x = poly[0][0] + x_offset
                pox_y = poly[0][1] + y_offset
                draw.text(
                    (pos_x, pox_y),
                    text,
                    font=font,
                    fill=font_color,
                    direction="ttb",
                )

        return np.array(pillow_img)

    def visualize_element(self, img, category, elements):
        """Visualize elements"""
        out = img.copy()
        categories = [
            "paragraphs",
            "tables",
            "figures",
            "section_headings",
            "page_header",
            "page_footer",
            "picture",
            "logo",
            "code",
            "seal",
            "list_item",
            "caption",
            "inline_formula",
            "display_formula",
            "index",
        ]

        for element in elements:
            box = element.box
            role = None

            if category != "tables":
                role = element.role

            color_index = categories.index(category)
            if role is None:
                role = ""
            else:
                color_index = categories.index(role)
                role = f"({role})"

            color = self.palette[color_index % len(self.palette)]
            x1, y1, x2, y2 = tuple(map(int, box))
            out = cv2.rectangle(out, (x1, y1), (x2, y2), color, 2)
            out = cv2.putText(
                out,
                category + role,
                (x1, y1),
                cv2.FONT_HERSHEY_SIMPLEX,
                0.5,
                (0, 0, 0),
                2,
            )

            if category in ["tables", "figures"]:
                caption = None
                if hasattr(element, "caption"):
                    caption = element.caption
                elif isinstance(element, dict) and "caption" in element:
                    caption = element["caption"]

                if caption is not None:
                    caption_box = None
                    if hasattr(caption, "box"):
                        caption_box = caption.box
                    elif isinstance(caption, dict) and "box" in caption:
                        caption_box = caption["box"]

                    if caption_box is not None:
                        color_index = categories.index("caption")
                        color = self.palette[color_index % len(self.palette)]
                        x1, y1, x2, y2 = tuple(map(int, caption_box))
                        out = cv2.rectangle(out, (x1, y1), (x2, y2), color, 2)
                        out = cv2.putText(
                            out,
                            "caption",
                            (x1, y1),
                            cv2.FONT_HERSHEY_SIMPLEX,
                            0.5,
                            (0, 0, 0),
                            2,
                        )

                if category == "figures":
                    paragraphs = None
                    if hasattr(element, "paragraphs"):
                        paragraphs = element.paragraphs
                    elif isinstance(element, dict) and "paragraphs" in element:
                        paragraphs = element["paragraphs"]

                    if paragraphs is not None:
                        for paragraph in paragraphs:
                            try:
                                para_box = None
                                if hasattr(paragraph, "box"):
                                    para_box = paragraph.box
                                elif isinstance(paragraph, dict) and "box" in paragraph:
                                    para_box = paragraph["box"]

                                if para_box is not None:
                                    color_index = categories.index("paragraphs")
                                    color = self.palette[
                                        color_index % len(self.palette)
                                    ]
                                    x1, y1, x2, y2 = tuple(map(int, para_box))
                                    out = cv2.rectangle(
                                        out,
                                        (x1, y1),
                                        (x2, y2),
                                        color,
                                        2,
                                    )
                                    out = cv2.putText(
                                        out,
                                        "paragraphs",
                                        (x1, y1),
                                        cv2.FONT_HERSHEY_SIMPLEX,
                                        0.5,
                                        (0, 0, 0),
                                        2,
                                    )
                            except Exception as e:
                                self.logger.warning(
                                    "Error processing figure paragraph: %s",
                                    e,
                                )
                                continue

        return out

    def layout_visualizer_detail(self, results, img):
        """Detailed layout visualizer"""
        out = img.copy()
        # results_dict = results.dict()
        out = self.visualize_element(out, "paragraphs", results.paragraphs)
        out = self.visualize_element(out, "tables", results.tables)
        out = self.visualize_element(out, "figures", results.figures)

        for table in results.tables:
            out = self.table_visualizer(out, table)

        return out

    def layout_visualizer_rough(self, results, img):
        """Rough layout visualizer"""
        out = img.copy()
        results_dict = results.dict()
        for idx, (category, preds) in enumerate(results_dict.items()):
            for element in preds:
                box = element["box"]
                role = element["role"]
                role = "" if role is None else f"({role})"

                color = self.palette[idx % len(self.palette)]
                x1, y1, x2, y2 = tuple(map(int, box))
                out = cv2.rectangle(out, (x1, y1), (x2, y2), color, 2)
                out = cv2.putText(
                    out,
                    category + role,
                    (x1, y1),
                    cv2.FONT_HERSHEY_SIMPLEX,
                    0.5,
                    color,
                    2,
                )

        return out

    def table_visualizer(self, img, table):
        """Table visualizer"""
        out = img.copy()
        cells = table.cells
        for cell in cells:
            box = cell.box
            row = cell.row
            col = cell.col
            row_span = cell.row_span
            col_span = cell.col_span

            text = f"[{row}, {col}] ({row_span}x{col_span})"

            x1, y1, x2, y2 = map(int, box)
            out = cv2.rectangle(out, (x1, y1), (x2, y2), (255, 0, 255), 2)
            out = cv2.putText(
                out,
                text,
                (x1, y1),
                cv2.FONT_HERSHEY_SIMPLEX,
                0.5,
                (255, 0, 0),
                2,
            )

        return out

    def rec_visualizer(
        self,
        img,
        outputs,
        font_path,
        font_size=12,
        font_color=(255, 0, 0),
    ):
        """Recognition visualizer"""
        out = img.copy()
        pillow_img = Image.fromarray(out)
        draw = ImageDraw.Draw(pillow_img)
        has_raqm = features.check_feature(feature="raqm")
        if not has_raqm:
            self.logger.warning(
                "libraqm is not installed. Vertical text rendering is not supported. Rendering horizontally instead.",
            )

        for pred, quad, direction, score in zip(
            outputs.contents,
            outputs.points,
            outputs.directions,
            outputs.scores,
            strict=True,
        ):
            quad = np.array(quad).astype(np.int32)
            font = ImageFont.truetype(font_path, font_size)

            pred = f"{pred} ({score:.3f})"

            if direction == "horizontal" or not has_raqm:
                x_offset = 0
                y_offset = -font_size

                pos_x = quad[0][0] + x_offset
                pox_y = quad[0][1] + y_offset
                draw.text((pos_x, pox_y), pred, font=font, fill=font_color)
            else:
                x_offset = -font_size
                y_offset = 0

                pos_x = quad[0][0] + x_offset
                pox_y = quad[0][1] + y_offset
                draw.text(
                    (pos_x, pox_y),
                    pred,
                    font=font,
                    fill=font_color,
                    direction="ttb",
                )

        out = np.array(pillow_img)
        return out
